#!/bin/sh
#
# Script to automatically install all Wine Staging patches
#
# Copyright (C) 2015-2017 Sebastian Lackner
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#

# Show usage information
usage()
{
	echo ""
	echo "Usage: ./patchinstall.sh [DESTDIR=path] [--all] [-W patchset] [patchset ...]"
	echo ""
	echo "Autogenerated script to apply all Wine Staging patches on your Wine"
	echo "source tree."
	echo ""
	echo "Configuration:"
	echo "  DESTDIR=path         Specify the path to the wine source tree"
	echo "  --all                Select all patches"
	echo "  --force-autoconf     Run autoreconf and tools/make_requests after each patch"
	echo "  --help               Display this help and exit"
	echo "  --no-autoconf        Do not run autoreconf and tools/make_requests"
	echo "  --upstream-commit    Print the upstream Wine commit SHA1 and exit"
	echo "  --version            Show version information and exit"
	echo "  -W patchset          Exclude a specific patchset"
	echo ""
	echo "Backends:"
	echo "  --backend=patch      Use regular 'patch' utility to apply patches (default)"
	echo "  --backend=eapply     Use 'eapply' to apply patches (Gentoo only)"
	echo "  --backend=epatch     Use 'epatch' to apply patches (Gentoo only, deprecated)"
	echo "  --backend=git-am     Use 'git am' to apply patches"
	echo "  --backend=git-apply  Use 'git apply' to apply patches"
	echo "  --backend=stg        Import the patches using stacked git"
	echo ""
}

# Get the upstream commit sha
upstream_commit()
{
	echo "9b12eae9ea445bbdeae196312fb65d20cdf27745"
}

# Show version information
version()
{
	echo "Wine Staging 6.0-rc5"
	echo "Copyright (C) 2014-2019 the Wine Staging project authors."
	echo "Copyright (C) 2018-2020 Alistair Leslie-Hughes"
	echo ""
	echo "Patchset to be applied on upstream Wine:"
	echo "  commit $(upstream_commit)"
	echo ""
}

# Critical error, abort
abort()
{
	printf '%s\n' "ERROR: $1" >&2
	exit 1
}

# Show a warning
warning()
{
	printf '%s\n' "WARNING: $1" >&2
}

# Enable or disable all patchsets
patch_enable_all ()
{
	enable_Compiler_Warnings="$1"
	enable_Pipelight="$1"
	enable_Staging="$1"
	enable_advapi32_LsaLookupPrivilegeName="$1"
	enable_api_ms_win_Stub_DLLs="$1"
	enable_atl_AtlAxDialogBox="$1"
	enable_bcrypt_ECDHSecretAgreement="$1"
	enable_cmd_launch_association="$1"
	enable_comctl32_Listview_DrawItem="$1"
	enable_comctl32_rebar_capture="$1"
	enable_comctl32_version_6="$1"
	enable_comdlg32_lpstrFileTitle="$1"
	enable_crypt32_CMS_Certificates="$1"
	enable_cryptext_CryptExtOpenCER="$1"
	enable_d3d11_Deferred_Context="$1"
	enable_d3drm_IDirect3D3_support="$1"
	enable_d3dx9_32bpp_Alpha_Channel="$1"
	enable_d3dx9_36_BumpLuminance="$1"
	enable_d3dx9_36_CloneEffect="$1"
	enable_d3dx9_36_D3DXDisassembleShader="$1"
	enable_d3dx9_36_D3DXOptimizeVertices="$1"
	enable_d3dx9_36_D3DXSHProjectCubeMap="$1"
	enable_d3dx9_36_D3DXStubs="$1"
	enable_d3dx9_36_DDS="$1"
	enable_d3dx9_36_Filter_Warnings="$1"
	enable_d3dx9_36_UpdateSkinnedMesh="$1"
	enable_dbghelp_Debug_Symbols="$1"
	enable_ddraw_Device_Caps="$1"
	enable_ddraw_IDirect3DTexture2_Load="$1"
	enable_ddraw_Rendering_Targets="$1"
	enable_ddraw_Silence_FIXMEs="$1"
	enable_ddraw_Texture_Wrong_Caps="$1"
	enable_ddraw_version_check="$1"
	enable_dinput_SetActionMap_genre="$1"
	enable_dinput_axis_recalc="$1"
	enable_dinput_joy_mappings="$1"
	enable_dinput_reconnect_joystick="$1"
	enable_dinput_remap_joystick="$1"
	enable_dsound_EAX="$1"
	enable_dsound_Fast_Mixer="$1"
	enable_dwrite_FontFallback="$1"
	enable_dxdiag_new_dlls="$1"
	enable_eventfd_synchronization="$1"
	enable_explorer_Video_Registry_Key="$1"
	enable_fonts_Missing_Fonts="$1"
	enable_gdi32_rotation="$1"
	enable_gdiplus_Performance_Improvements="$1"
	enable_imagehlp_BindImageEx="$1"
	enable_imm32_message_on_focus="$1"
	enable_include_winsock="$1"
	enable_inseng_Implementation="$1"
	enable_iphlpapi_System_Ping="$1"
	enable_kernel32_CopyFileEx="$1"
	enable_kernel32_Debugger="$1"
	enable_kernel32_FindFirstFile="$1"
	enable_kernel32_Job_Tests="$1"
	enable_kernel32_Processor_Group="$1"
	enable_kernel32_SetProcessDEPPolicy="$1"
	enable_krnl386_exe16_GDT_LDT_Emulation="$1"
	enable_krnl386_exe16_Invalid_Console_Handles="$1"
	enable_libs_Unicode_Collation="$1"
	enable_loader_KeyboardLayouts="$1"
	enable_mfplat_streaming_support="$1"
	enable_mmsystem_dll16_MIDIHDR_Refcount="$1"
	enable_mountmgr_DosDevices="$1"
	enable_mscoree_CorValidateImage="$1"
	enable_mshtml_HTMLLocation_put_hash="$1"
	enable_mshtml_TranslateAccelerator="$1"
	enable_msi_msi_vcl_get_cost="$1"
	enable_msvcrt_Math_Precision="$1"
	enable_msxml3_FreeThreadedXMLHTTP60="$1"
	enable_ntdll_APC_Performance="$1"
	enable_ntdll_Activation_Context="$1"
	enable_ntdll_ApiSetMap="$1"
	enable_ntdll_Builtin_Prot="$1"
	enable_ntdll_CriticalSection="$1"
	enable_ntdll_DOS_Attributes="$1"
	enable_ntdll_Dealloc_Thread_Stack="$1"
	enable_ntdll_Exception="$1"
	enable_ntdll_FileDispositionInformation="$1"
	enable_ntdll_FileFsFullSizeInformation="$1"
	enable_ntdll_ForceBottomUpAlloc="$1"
	enable_ntdll_HashLinks="$1"
	enable_ntdll_Heap_Improvements="$1"
	enable_ntdll_Hide_Wine_Exports="$1"
	enable_ntdll_Interrupt_0x2e="$1"
	enable_ntdll_Junction_Points="$1"
	enable_ntdll_Manifest_Range="$1"
	enable_ntdll_NtAccessCheck="$1"
	enable_ntdll_NtAlertThreadByThreadId="$1"
	enable_ntdll_NtDevicePath="$1"
	enable_ntdll_NtQueryEaFile="$1"
	enable_ntdll_NtQuerySection="$1"
	enable_ntdll_NtQueryVirtualMemory="$1"
	enable_ntdll_NtSetLdtEntries="$1"
	enable_ntdll_Pipe_SpecialCharacters="$1"
	enable_ntdll_ProcessQuotaLimits="$1"
	enable_ntdll_RtlQueryPackageIdentity="$1"
	enable_ntdll_RtlQueryProcessPlaceholderCompatibilityMode="$1"
	enable_ntdll_RtlQueryRegistryValuesEx="$1"
	enable_ntdll_Serial_Port_Detection="$1"
	enable_ntdll_Status_Mapping="$1"
	enable_ntdll_Syscall_Emulation="$1"
	enable_ntdll_SystemCodeIntegrityInformation="$1"
	enable_ntdll_WRITECOPY="$1"
	enable_ntdll_Zero_mod_name="$1"
	enable_ntdll_aarch_TEB="$1"
	enable_ntdll_ext4_case_folder="$1"
	enable_ntdll_set_full_cpu_context="$1"
	enable_ntdll_x86_64_SegDs="$1"
	enable_ntoskrnl_Stubs="$1"
	enable_nvapi_Stub_DLL="$1"
	enable_nvcuda_CUDA_Support="$1"
	enable_nvcuvid_CUDA_Video_Support="$1"
	enable_nvencodeapi_Video_Encoder="$1"
	enable_odbccp32_SQLWriteDSNToIni="$1"
	enable_oleaut32_CreateTypeLib="$1"
	enable_oleaut32_Load_Save_EMF="$1"
	enable_oleaut32_OLEPictureImpl_SaveAsFile="$1"
	enable_oleaut32_OleLoadPicture="$1"
	enable_oleaut32_OleLoadPictureFile="$1"
	enable_opencl_version_1_2="$1"
	enable_packager_DllMain="$1"
	enable_pdh_PdhLookupPerfNameByIndex_processor="$1"
	enable_programs_findstr="$1"
	enable_programs_systeminfo="$1"
	enable_quartz_MediaSeeking_Positions="$1"
	enable_riched20_Class_Tests="$1"
	enable_riched20_IText_Interface="$1"
	enable_server_FileEndOfFileInformation="$1"
	enable_server_File_Permissions="$1"
	enable_server_Inherited_ACLs="$1"
	enable_server_Key_State="$1"
	enable_server_Object_Types="$1"
	enable_server_PeekMessage="$1"
	enable_server_Realtime_Priority="$1"
	enable_server_Registry_Notifications="$1"
	enable_server_Signal_Thread="$1"
	enable_server_Stored_ACLs="$1"
	enable_server_unix_name="$1"
	enable_setupapi_DiskSpaceList="$1"
	enable_setupapi_SPFILENOTIFY_FILEINCABINET="$1"
	enable_shdocvw_ParseURLFromOutsideSource_Tests="$1"
	enable_shell32_ACE_Viewer="$1"
	enable_shell32_Context_Menu="$1"
	enable_shell32_IconCache="$1"
	enable_shell32_NewMenu_Interface="$1"
	enable_shell32_Progress_Dialog="$1"
	enable_shell32_SFGAO_HASSUBFOLDER="$1"
	enable_shell32_SHFileOperation_Move="$1"
	enable_shell32_SHGetStockIconInfo="$1"
	enable_shell32_Toolbar_Bitmaps="$1"
	enable_shell32_UnixFS="$1"
	enable_shlwapi_AssocGetPerceivedType="$1"
	enable_shlwapi_SHAddDataBlock="$1"
	enable_shlwapi_UrlCanonicalize="$1"
	enable_shlwapi_UrlCombine="$1"
	enable_stdole32_idl_Typelib="$1"
	enable_stdole32_tlb_SLTG_Typelib="$1"
	enable_tasklist_basics="$1"
	enable_user32_DM_SETDEFID="$1"
	enable_user32_Dialog_Paint_Event="$1"
	enable_user32_DrawTextExW="$1"
	enable_user32_FlashWindowEx="$1"
	enable_user32_GetSystemMetrics="$1"
	enable_user32_Implement_CascadeWindows="$1"
	enable_user32_InternalGetWindowIcon="$1"
	enable_user32_LR_LOADFROMFILE="$1"
	enable_user32_ListBox_Size="$1"
	enable_user32_LoadKeyboardLayoutEx="$1"
	enable_user32_MessageBox_WS_EX_TOPMOST="$1"
	enable_user32_Mouse_Message_Hwnd="$1"
	enable_user32_QueryDisplayConfig="$1"
	enable_user32_Refresh_MDI_Menus="$1"
	enable_user32_ScrollWindowEx="$1"
	enable_user32_message_order="$1"
	enable_user32_msgbox_Support_WM_COPY_mesg="$1"
	enable_user32_rawinput_hid="$1"
	enable_user32_rawinput_mouse="$1"
	enable_user32_rawinput_mouse_experimental="$1"
	enable_user32_recursive_activation="$1"
	enable_user32_window_activation="$1"
	enable_uxtheme_CloseThemeClass="$1"
	enable_version_VerQueryValue="$1"
	enable_widl_SLTG_Typelib_Support="$1"
	enable_widl_winrt_support="$1"
	enable_windows_gaming_input_dll="$1"
	enable_windows_globalization_dll="$1"
	enable_windows_media_speech_dll="$1"
	enable_windows_networking_connectivity_dll="$1"
	enable_windowscodecs_GIF_Encoder="$1"
	enable_windowscodecs_TIFF_Support="$1"
	enable_wine_inf_Directory_ContextMenuHandlers="$1"
	enable_wine_inf_Dummy_CA_Certificate="$1"
	enable_wine_inf_Performance="$1"
	enable_wineboot_DriveSerial="$1"
	enable_wineboot_HKEY_DYN_DATA="$1"
	enable_wineboot_ProxySettings="$1"
	enable_wineboot_drivers_etc_Stubs="$1"
	enable_winecfg_Libraries="$1"
	enable_winecfg_Staging="$1"
	enable_wined3d_Accounting="$1"
	enable_wined3d_CSMT_Main="$1"
	enable_wined3d_Indexed_Vertex_Blending="$1"
	enable_wined3d_SWVP_shaders="$1"
	enable_wined3d_Silence_FIXMEs="$1"
	enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM="$1"
	enable_wined3d_mesa_texture_download="$1"
	enable_wined3d_unset_flip_gdi="$1"
	enable_wined3d_wined3d_guess_gl_vendor="$1"
	enable_wined3d_zero_inf_shaders="$1"
	enable_winedbg_Process_Arguments="$1"
	enable_winedevice_Default_Drivers="$1"
	enable_winemapi_user_xdg_mail="$1"
	enable_winemenubuilder_Desktop_Icon_Path="$1"
	enable_winemenubuilder_integration="$1"
	enable_wineps_drv_PostScript_Fixes="$1"
	enable_winepulse_PulseAudio_Support="$1"
	enable_winex11_CandidateWindowPos="$1"
	enable_winex11_MWM_Decorations="$1"
	enable_winex11_UpdateLayeredWindow="$1"
	enable_winex11_Vulkan_support="$1"
	enable_winex11_WM_WINDOWPOSCHANGING="$1"
	enable_winex11_Window_Style="$1"
	enable_winex11_XEMBED="$1"
	enable_winex11__NET_ACTIVE_WINDOW="$1"
	enable_winex11_ime_check_thread_data="$1"
	enable_winex11_key_translation="$1"
	enable_winex11_wglShareLists="$1"
	enable_winex11_drv_Query_server_position="$1"
	enable_winex11_drv_mouse_coorrds="$1"
	enable_wininet_Cleanup="$1"
	enable_winmm_Delay_Import_Depends="$1"
	enable_winmm_mciSendCommandA="$1"
	enable_wintab32_improvements="$1"
	enable_wintrust_WTHelperGetProvCertFromChain="$1"
	enable_wow64cpu_Wow64Transition="$1"
	enable_wpcap_Dynamic_Linking="$1"
	enable_ws2_32_APC_Performance="$1"
	enable_ws2_32_Connect_Time="$1"
	enable_ws2_32_getsockopt="$1"
	enable_wtsapi32_EnumerateProcesses="$1"
	enable_xactengine_initial="$1"
	enable_xactengine2_dll="$1"
	enable_xactengine3_7_PrepareWave="$1"
}

# Enable or disable a specific patchset
patch_enable ()
{
	case "$1" in
		Compiler_Warnings)
			enable_Compiler_Warnings="$2"
			;;
		Pipelight)
			enable_Pipelight="$2"
			;;
		Staging)
			enable_Staging="$2"
			;;
		advapi32-LsaLookupPrivilegeName)
			enable_advapi32_LsaLookupPrivilegeName="$2"
			;;
		api-ms-win-Stub_DLLs)
			enable_api_ms_win_Stub_DLLs="$2"
			;;
		atl-AtlAxDialogBox)
			enable_atl_AtlAxDialogBox="$2"
			;;
		bcrypt-ECDHSecretAgreement)
			enable_bcrypt_ECDHSecretAgreement="$2"
			;;
		cmd-launch-association)
			enable_cmd_launch_association="$2"
			;;
		comctl32-Listview_DrawItem)
			enable_comctl32_Listview_DrawItem="$2"
			;;
		comctl32-rebar-capture)
			enable_comctl32_rebar_capture="$2"
			;;
		comctl32-version_6)
			enable_comctl32_version_6="$2"
			;;
		comdlg32-lpstrFileTitle)
			enable_comdlg32_lpstrFileTitle="$2"
			;;
		crypt32-CMS_Certificates)
			enable_crypt32_CMS_Certificates="$2"
			;;
		cryptext-CryptExtOpenCER)
			enable_cryptext_CryptExtOpenCER="$2"
			;;
		d3d11-Deferred_Context)
			enable_d3d11_Deferred_Context="$2"
			;;
		d3drm-IDirect3D3-support)
			enable_d3drm_IDirect3D3_support="$2"
			;;
		d3dx9-32bpp_Alpha_Channel)
			enable_d3dx9_32bpp_Alpha_Channel="$2"
			;;
		d3dx9_36-BumpLuminance)
			enable_d3dx9_36_BumpLuminance="$2"
			;;
		d3dx9_36-CloneEffect)
			enable_d3dx9_36_CloneEffect="$2"
			;;
		d3dx9_36-D3DXDisassembleShader)
			enable_d3dx9_36_D3DXDisassembleShader="$2"
			;;
		d3dx9_36-D3DXOptimizeVertices)
			enable_d3dx9_36_D3DXOptimizeVertices="$2"
			;;
		d3dx9_36-D3DXSHProjectCubeMap)
			enable_d3dx9_36_D3DXSHProjectCubeMap="$2"
			;;
		d3dx9_36-D3DXStubs)
			enable_d3dx9_36_D3DXStubs="$2"
			;;
		d3dx9_36-DDS)
			enable_d3dx9_36_DDS="$2"
			;;
		d3dx9_36-Filter_Warnings)
			enable_d3dx9_36_Filter_Warnings="$2"
			;;
		d3dx9_36-UpdateSkinnedMesh)
			enable_d3dx9_36_UpdateSkinnedMesh="$2"
			;;
		dbghelp-Debug_Symbols)
			enable_dbghelp_Debug_Symbols="$2"
			;;
		ddraw-Device_Caps)
			enable_ddraw_Device_Caps="$2"
			;;
		ddraw-IDirect3DTexture2_Load)
			enable_ddraw_IDirect3DTexture2_Load="$2"
			;;
		ddraw-Rendering_Targets)
			enable_ddraw_Rendering_Targets="$2"
			;;
		ddraw-Silence_FIXMEs)
			enable_ddraw_Silence_FIXMEs="$2"
			;;
		ddraw-Texture_Wrong_Caps)
			enable_ddraw_Texture_Wrong_Caps="$2"
			;;
		ddraw-version-check)
			enable_ddraw_version_check="$2"
			;;
		dinput-SetActionMap-genre)
			enable_dinput_SetActionMap_genre="$2"
			;;
		dinput-axis-recalc)
			enable_dinput_axis_recalc="$2"
			;;
		dinput-joy-mappings)
			enable_dinput_joy_mappings="$2"
			;;
		dinput-reconnect-joystick)
			enable_dinput_reconnect_joystick="$2"
			;;
		dinput-remap-joystick)
			enable_dinput_remap_joystick="$2"
			;;
		dsound-EAX)
			enable_dsound_EAX="$2"
			;;
		dsound-Fast_Mixer)
			enable_dsound_Fast_Mixer="$2"
			;;
		dwrite-FontFallback)
			enable_dwrite_FontFallback="$2"
			;;
		dxdiag-new-dlls)
			enable_dxdiag_new_dlls="$2"
			;;
		eventfd_synchronization)
			enable_eventfd_synchronization="$2"
			;;
		explorer-Video_Registry_Key)
			enable_explorer_Video_Registry_Key="$2"
			;;
		fonts-Missing_Fonts)
			enable_fonts_Missing_Fonts="$2"
			;;
		gdi32-rotation)
			enable_gdi32_rotation="$2"
			;;
		gdiplus-Performance-Improvements)
			enable_gdiplus_Performance_Improvements="$2"
			;;
		imagehlp-BindImageEx)
			enable_imagehlp_BindImageEx="$2"
			;;
		imm32-message_on_focus)
			enable_imm32_message_on_focus="$2"
			;;
		include-winsock)
			enable_include_winsock="$2"
			;;
		inseng-Implementation)
			enable_inseng_Implementation="$2"
			;;
		iphlpapi-System_Ping)
			enable_iphlpapi_System_Ping="$2"
			;;
		kernel32-CopyFileEx)
			enable_kernel32_CopyFileEx="$2"
			;;
		kernel32-Debugger)
			enable_kernel32_Debugger="$2"
			;;
		kernel32-FindFirstFile)
			enable_kernel32_FindFirstFile="$2"
			;;
		kernel32-Job_Tests)
			enable_kernel32_Job_Tests="$2"
			;;
		kernel32-Processor_Group)
			enable_kernel32_Processor_Group="$2"
			;;
		kernel32-SetProcessDEPPolicy)
			enable_kernel32_SetProcessDEPPolicy="$2"
			;;
		krnl386.exe16-GDT_LDT_Emulation)
			enable_krnl386_exe16_GDT_LDT_Emulation="$2"
			;;
		krnl386.exe16-Invalid_Console_Handles)
			enable_krnl386_exe16_Invalid_Console_Handles="$2"
			;;
		libs-Unicode_Collation)
			enable_libs_Unicode_Collation="$2"
			;;
		loader-KeyboardLayouts)
			enable_loader_KeyboardLayouts="$2"
			;;
		mfplat-streaming-support)
			enable_mfplat_streaming_support="$2"
			;;
		mmsystem.dll16-MIDIHDR_Refcount)
			enable_mmsystem_dll16_MIDIHDR_Refcount="$2"
			;;
		mountmgr-DosDevices)
			enable_mountmgr_DosDevices="$2"
			;;
		mscoree-CorValidateImage)
			enable_mscoree_CorValidateImage="$2"
			;;
		mshtml-HTMLLocation_put_hash)
			enable_mshtml_HTMLLocation_put_hash="$2"
			;;
		mshtml-TranslateAccelerator)
			enable_mshtml_TranslateAccelerator="$2"
			;;
		msi-msi_vcl_get_cost)
			enable_msi_msi_vcl_get_cost="$2"
			;;
		msvcrt-Math_Precision)
			enable_msvcrt_Math_Precision="$2"
			;;
		msxml3-FreeThreadedXMLHTTP60)
			enable_msxml3_FreeThreadedXMLHTTP60="$2"
			;;
		ntdll-APC_Performance)
			enable_ntdll_APC_Performance="$2"
			;;
		ntdll-Activation_Context)
			enable_ntdll_Activation_Context="$2"
			;;
		ntdll-ApiSetMap)
			enable_ntdll_ApiSetMap="$2"
			;;
		ntdll-Builtin_Prot)
			enable_ntdll_Builtin_Prot="$2"
			;;
		ntdll-CriticalSection)
			enable_ntdll_CriticalSection="$2"
			;;
		ntdll-DOS_Attributes)
			enable_ntdll_DOS_Attributes="$2"
			;;
		ntdll-Dealloc_Thread_Stack)
			enable_ntdll_Dealloc_Thread_Stack="$2"
			;;
		ntdll-Exception)
			enable_ntdll_Exception="$2"
			;;
		ntdll-FileDispositionInformation)
			enable_ntdll_FileDispositionInformation="$2"
			;;
		ntdll-FileFsFullSizeInformation)
			enable_ntdll_FileFsFullSizeInformation="$2"
			;;
		ntdll-ForceBottomUpAlloc)
			enable_ntdll_ForceBottomUpAlloc="$2"
			;;
		ntdll-HashLinks)
			enable_ntdll_HashLinks="$2"
			;;
		ntdll-Heap_Improvements)
			enable_ntdll_Heap_Improvements="$2"
			;;
		ntdll-Hide_Wine_Exports)
			enable_ntdll_Hide_Wine_Exports="$2"
			;;
		ntdll-Interrupt-0x2e)
			enable_ntdll_Interrupt_0x2e="$2"
			;;
		ntdll-Junction_Points)
			enable_ntdll_Junction_Points="$2"
			;;
		ntdll-Manifest_Range)
			enable_ntdll_Manifest_Range="$2"
			;;
		ntdll-NtAccessCheck)
			enable_ntdll_NtAccessCheck="$2"
			;;
		ntdll-NtAlertThreadByThreadId)
			enable_ntdll_NtAlertThreadByThreadId="$2"
			;;
		ntdll-NtDevicePath)
			enable_ntdll_NtDevicePath="$2"
			;;
		ntdll-NtQueryEaFile)
			enable_ntdll_NtQueryEaFile="$2"
			;;
		ntdll-NtQuerySection)
			enable_ntdll_NtQuerySection="$2"
			;;
		ntdll-NtQueryVirtualMemory)
			enable_ntdll_NtQueryVirtualMemory="$2"
			;;
		ntdll-NtSetLdtEntries)
			enable_ntdll_NtSetLdtEntries="$2"
			;;
		ntdll-Pipe_SpecialCharacters)
			enable_ntdll_Pipe_SpecialCharacters="$2"
			;;
		ntdll-ProcessQuotaLimits)
			enable_ntdll_ProcessQuotaLimits="$2"
			;;
		ntdll-RtlQueryPackageIdentity)
			enable_ntdll_RtlQueryPackageIdentity="$2"
			;;
		ntdll-RtlQueryProcessPlaceholderCompatibilityMode)
			enable_ntdll_RtlQueryProcessPlaceholderCompatibilityMode="$2"
			;;
		ntdll-RtlQueryRegistryValuesEx)
			enable_ntdll_RtlQueryRegistryValuesEx="$2"
			;;
		ntdll-Serial_Port_Detection)
			enable_ntdll_Serial_Port_Detection="$2"
			;;
		ntdll-Status_Mapping)
			enable_ntdll_Status_Mapping="$2"
			;;
		ntdll-Syscall_Emulation)
			enable_ntdll_Syscall_Emulation="$2"
			;;
		ntdll-SystemCodeIntegrityInformation)
			enable_ntdll_SystemCodeIntegrityInformation="$2"
			;;
		ntdll-WRITECOPY)
			enable_ntdll_WRITECOPY="$2"
			;;
		ntdll-Zero_mod_name)
			enable_ntdll_Zero_mod_name="$2"
			;;
		ntdll-aarch-TEB)
			enable_ntdll_aarch_TEB="$2"
			;;
		ntdll-ext4-case-folder)
			enable_ntdll_ext4_case_folder="$2"
			;;
		ntdll-set_full_cpu_context)
			enable_ntdll_set_full_cpu_context="$2"
			;;
		ntdll-x86_64_SegDs)
			enable_ntdll_x86_64_SegDs="$2"
			;;
		ntoskrnl-Stubs)
			enable_ntoskrnl_Stubs="$2"
			;;
		nvapi-Stub_DLL)
			enable_nvapi_Stub_DLL="$2"
			;;
		nvcuda-CUDA_Support)
			enable_nvcuda_CUDA_Support="$2"
			;;
		nvcuvid-CUDA_Video_Support)
			enable_nvcuvid_CUDA_Video_Support="$2"
			;;
		nvencodeapi-Video_Encoder)
			enable_nvencodeapi_Video_Encoder="$2"
			;;
		odbccp32-SQLWriteDSNToIni)
			enable_odbccp32_SQLWriteDSNToIni="$2"
			;;
		oleaut32-CreateTypeLib)
			enable_oleaut32_CreateTypeLib="$2"
			;;
		oleaut32-Load_Save_EMF)
			enable_oleaut32_Load_Save_EMF="$2"
			;;
		oleaut32-OLEPictureImpl_SaveAsFile)
			enable_oleaut32_OLEPictureImpl_SaveAsFile="$2"
			;;
		oleaut32-OleLoadPicture)
			enable_oleaut32_OleLoadPicture="$2"
			;;
		oleaut32-OleLoadPictureFile)
			enable_oleaut32_OleLoadPictureFile="$2"
			;;
		opencl-version_1_2)
			enable_opencl_version_1_2="$2"
			;;
		packager-DllMain)
			enable_packager_DllMain="$2"
			;;
		pdh-PdhLookupPerfNameByIndex-processor)
			enable_pdh_PdhLookupPerfNameByIndex_processor="$2"
			;;
		programs-findstr)
			enable_programs_findstr="$2"
			;;
		programs-systeminfo)
			enable_programs_systeminfo="$2"
			;;
		quartz-MediaSeeking_Positions)
			enable_quartz_MediaSeeking_Positions="$2"
			;;
		riched20-Class_Tests)
			enable_riched20_Class_Tests="$2"
			;;
		riched20-IText_Interface)
			enable_riched20_IText_Interface="$2"
			;;
		server-FileEndOfFileInformation)
			enable_server_FileEndOfFileInformation="$2"
			;;
		server-File_Permissions)
			enable_server_File_Permissions="$2"
			;;
		server-Inherited_ACLs)
			enable_server_Inherited_ACLs="$2"
			;;
		server-Key_State)
			enable_server_Key_State="$2"
			;;
		server-Object_Types)
			enable_server_Object_Types="$2"
			;;
		server-PeekMessage)
			enable_server_PeekMessage="$2"
			;;
		server-Realtime_Priority)
			enable_server_Realtime_Priority="$2"
			;;
		server-Registry_Notifications)
			enable_server_Registry_Notifications="$2"
			;;
		server-Signal_Thread)
			enable_server_Signal_Thread="$2"
			;;
		server-Stored_ACLs)
			enable_server_Stored_ACLs="$2"
			;;
		server-unix_name)
			enable_server_unix_name="$2"
			;;
		setupapi-DiskSpaceList)
			enable_setupapi_DiskSpaceList="$2"
			;;
		setupapi-SPFILENOTIFY_FILEINCABINET)
			enable_setupapi_SPFILENOTIFY_FILEINCABINET="$2"
			;;
		shdocvw-ParseURLFromOutsideSource_Tests)
			enable_shdocvw_ParseURLFromOutsideSource_Tests="$2"
			;;
		shell32-ACE_Viewer)
			enable_shell32_ACE_Viewer="$2"
			;;
		shell32-Context_Menu)
			enable_shell32_Context_Menu="$2"
			;;
		shell32-IconCache)
			enable_shell32_IconCache="$2"
			;;
		shell32-NewMenu_Interface)
			enable_shell32_NewMenu_Interface="$2"
			;;
		shell32-Progress_Dialog)
			enable_shell32_Progress_Dialog="$2"
			;;
		shell32-SFGAO_HASSUBFOLDER)
			enable_shell32_SFGAO_HASSUBFOLDER="$2"
			;;
		shell32-SHFileOperation_Move)
			enable_shell32_SHFileOperation_Move="$2"
			;;
		shell32-SHGetStockIconInfo)
			enable_shell32_SHGetStockIconInfo="$2"
			;;
		shell32-Toolbar_Bitmaps)
			enable_shell32_Toolbar_Bitmaps="$2"
			;;
		shell32-UnixFS)
			enable_shell32_UnixFS="$2"
			;;
		shlwapi-AssocGetPerceivedType)
			enable_shlwapi_AssocGetPerceivedType="$2"
			;;
		shlwapi-SHAddDataBlock)
			enable_shlwapi_SHAddDataBlock="$2"
			;;
		shlwapi-UrlCanonicalize)
			enable_shlwapi_UrlCanonicalize="$2"
			;;
		shlwapi-UrlCombine)
			enable_shlwapi_UrlCombine="$2"
			;;
		stdole32.idl-Typelib)
			enable_stdole32_idl_Typelib="$2"
			;;
		stdole32.tlb-SLTG_Typelib)
			enable_stdole32_tlb_SLTG_Typelib="$2"
			;;
		tasklist-basics)
			enable_tasklist_basics="$2"
			;;
		user32-DM_SETDEFID)
			enable_user32_DM_SETDEFID="$2"
			;;
		user32-Dialog_Paint_Event)
			enable_user32_Dialog_Paint_Event="$2"
			;;
		user32-DrawTextExW)
			enable_user32_DrawTextExW="$2"
			;;
		user32-FlashWindowEx)
			enable_user32_FlashWindowEx="$2"
			;;
		user32-GetSystemMetrics)
			enable_user32_GetSystemMetrics="$2"
			;;
		user32-Implement-CascadeWindows)
			enable_user32_Implement_CascadeWindows="$2"
			;;
		user32-InternalGetWindowIcon)
			enable_user32_InternalGetWindowIcon="$2"
			;;
		user32-LR_LOADFROMFILE)
			enable_user32_LR_LOADFROMFILE="$2"
			;;
		user32-ListBox_Size)
			enable_user32_ListBox_Size="$2"
			;;
		user32-LoadKeyboardLayoutEx)
			enable_user32_LoadKeyboardLayoutEx="$2"
			;;
		user32-MessageBox_WS_EX_TOPMOST)
			enable_user32_MessageBox_WS_EX_TOPMOST="$2"
			;;
		user32-Mouse_Message_Hwnd)
			enable_user32_Mouse_Message_Hwnd="$2"
			;;
		user32-QueryDisplayConfig)
			enable_user32_QueryDisplayConfig="$2"
			;;
		user32-Refresh_MDI_Menus)
			enable_user32_Refresh_MDI_Menus="$2"
			;;
		user32-ScrollWindowEx)
			enable_user32_ScrollWindowEx="$2"
			;;
		user32-message-order)
			enable_user32_message_order="$2"
			;;
		user32-msgbox-Support-WM_COPY-mesg)
			enable_user32_msgbox_Support_WM_COPY_mesg="$2"
			;;
		user32-rawinput-hid)
			enable_user32_rawinput_hid="$2"
			;;
		user32-rawinput-mouse)
			enable_user32_rawinput_mouse="$2"
			;;
		user32-rawinput-mouse-experimental)
			enable_user32_rawinput_mouse_experimental="$2"
			;;
		user32-recursive-activation)
			enable_user32_recursive_activation="$2"
			;;
		user32-window-activation)
			enable_user32_window_activation="$2"
			;;
		uxtheme-CloseThemeClass)
			enable_uxtheme_CloseThemeClass="$2"
			;;
		version-VerQueryValue)
			enable_version_VerQueryValue="$2"
			;;
		widl-SLTG_Typelib_Support)
			enable_widl_SLTG_Typelib_Support="$2"
			;;
		widl-winrt-support)
			enable_widl_winrt_support="$2"
			;;
		windows.gaming.input-dll)
			enable_windows_gaming_input_dll="$2"
			;;
		windows.globalization-dll)
			enable_windows_globalization_dll="$2"
			;;
		windows.media.speech.dll)
			enable_windows_media_speech_dll="$2"
			;;
		windows.networking.connectivity.dll)
			enable_windows_networking_connectivity_dll="$2"
			;;
		windowscodecs-GIF_Encoder)
			enable_windowscodecs_GIF_Encoder="$2"
			;;
		windowscodecs-TIFF_Support)
			enable_windowscodecs_TIFF_Support="$2"
			;;
		wine.inf-Directory_ContextMenuHandlers)
			enable_wine_inf_Directory_ContextMenuHandlers="$2"
			;;
		wine.inf-Dummy_CA_Certificate)
			enable_wine_inf_Dummy_CA_Certificate="$2"
			;;
		wine.inf-Performance)
			enable_wine_inf_Performance="$2"
			;;
		wineboot-DriveSerial)
			enable_wineboot_DriveSerial="$2"
			;;
		wineboot-HKEY_DYN_DATA)
			enable_wineboot_HKEY_DYN_DATA="$2"
			;;
		wineboot-ProxySettings)
			enable_wineboot_ProxySettings="$2"
			;;
		wineboot-drivers_etc_Stubs)
			enable_wineboot_drivers_etc_Stubs="$2"
			;;
		winecfg-Libraries)
			enable_winecfg_Libraries="$2"
			;;
		winecfg-Staging)
			enable_winecfg_Staging="$2"
			;;
		wined3d-Accounting)
			enable_wined3d_Accounting="$2"
			;;
		wined3d-CSMT_Main)
			enable_wined3d_CSMT_Main="$2"
			;;
		wined3d-Indexed_Vertex_Blending)
			enable_wined3d_Indexed_Vertex_Blending="$2"
			;;
		wined3d-SWVP-shaders)
			enable_wined3d_SWVP_shaders="$2"
			;;
		wined3d-Silence_FIXMEs)
			enable_wined3d_Silence_FIXMEs="$2"
			;;
		wined3d-WINED3DFMT_B8G8R8X8_UNORM)
			enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM="$2"
			;;
		wined3d-mesa_texture_download)
			enable_wined3d_mesa_texture_download="$2"
			;;
		wined3d-unset-flip-gdi)
			enable_wined3d_unset_flip_gdi="$2"
			;;
		wined3d-wined3d_guess_gl_vendor)
			enable_wined3d_wined3d_guess_gl_vendor="$2"
			;;
		wined3d-zero-inf-shaders)
			enable_wined3d_zero_inf_shaders="$2"
			;;
		winedbg-Process_Arguments)
			enable_winedbg_Process_Arguments="$2"
			;;
		winedevice-Default_Drivers)
			enable_winedevice_Default_Drivers="$2"
			;;
		winemapi-user-xdg-mail)
			enable_winemapi_user_xdg_mail="$2"
			;;
		winemenubuilder-Desktop_Icon_Path)
			enable_winemenubuilder_Desktop_Icon_Path="$2"
			;;
		winemenubuilder-integration)
			enable_winemenubuilder_integration="$2"
			;;
		wineps.drv-PostScript_Fixes)
			enable_wineps_drv_PostScript_Fixes="$2"
			;;
		winepulse-PulseAudio_Support)
			enable_winepulse_PulseAudio_Support="$2"
			;;
		winex11-CandidateWindowPos)
			enable_winex11_CandidateWindowPos="$2"
			;;
		winex11-MWM_Decorations)
			enable_winex11_MWM_Decorations="$2"
			;;
		winex11-UpdateLayeredWindow)
			enable_winex11_UpdateLayeredWindow="$2"
			;;
		winex11-Vulkan_support)
			enable_winex11_Vulkan_support="$2"
			;;
		winex11-WM_WINDOWPOSCHANGING)
			enable_winex11_WM_WINDOWPOSCHANGING="$2"
			;;
		winex11-Window_Style)
			enable_winex11_Window_Style="$2"
			;;
		winex11-XEMBED)
			enable_winex11_XEMBED="$2"
			;;
		winex11-_NET_ACTIVE_WINDOW)
			enable_winex11__NET_ACTIVE_WINDOW="$2"
			;;
		winex11-ime-check-thread-data)
			enable_winex11_ime_check_thread_data="$2"
			;;
		winex11-key_translation)
			enable_winex11_key_translation="$2"
			;;
		winex11-wglShareLists)
			enable_winex11_wglShareLists="$2"
			;;
		winex11.drv-Query_server_position)
			enable_winex11_drv_Query_server_position="$2"
			;;
		winex11.drv-mouse-coorrds)
			enable_winex11_drv_mouse_coorrds="$2"
			;;
		wininet-Cleanup)
			enable_wininet_Cleanup="$2"
			;;
		winmm-Delay_Import_Depends)
			enable_winmm_Delay_Import_Depends="$2"
			;;
		winmm-mciSendCommandA)
			enable_winmm_mciSendCommandA="$2"
			;;
		wintab32-improvements)
			enable_wintab32_improvements="$2"
			;;
		wintrust-WTHelperGetProvCertFromChain)
			enable_wintrust_WTHelperGetProvCertFromChain="$2"
			;;
		wow64cpu-Wow64Transition)
			enable_wow64cpu_Wow64Transition="$2"
			;;
		wpcap-Dynamic_Linking)
			enable_wpcap_Dynamic_Linking="$2"
			;;
		ws2_32-APC_Performance)
			enable_ws2_32_APC_Performance="$2"
			;;
		ws2_32-Connect_Time)
			enable_ws2_32_Connect_Time="$2"
			;;
		ws2_32-getsockopt)
			enable_ws2_32_getsockopt="$2"
			;;
		wtsapi32-EnumerateProcesses)
			enable_wtsapi32_EnumerateProcesses="$2"
			;;
		xactengine-initial)
			enable_xactengine_initial="$2"
			;;
		xactengine2-dll)
			enable_xactengine2_dll="$2"
			;;
		xactengine3_7-PrepareWave)
			enable_xactengine3_7_PrepareWave="$2"
			;;
		*)
			return 1
			;;
	esac
	return 0
}

# Default settings
patch_enable_all 0
enable_autoconf=1
backend="patch"

# Find location of patches
patchdir="$(cd "$(dirname "$0")" && pwd)"
if test ! -f "$patchdir/patchinstall.sh"; then
	if test -f ./patchinstall.sh; then
		patchdir="$(pwd)"
	else
		abort "Failed to find patch directory."
	fi
fi

# Parse commandline arguments
if test "$#" -eq 0; then
	abort "No commandline arguments given, don't know what to do."
fi

while test "$#" -gt 0; do
	case "$1" in
		DESTDIR=*)
			DESTDIR="${1#*=}"
			shift
			;;

		--all)
			patch_enable_all 1
			shift
			;;

		--backend=*)
			backend="${1#*=}"
			shift
			;;

		--force-autoconf)
			enable_autoconf=2
			shift
			;;

		--help)
			usage
			exit 0
			;;

		--no-autoconf)
			enable_autoconf=0
			shift
			;;

		--upstream-commit)
			upstream_commit
			exit 0
			;;

		--version)
			version
			exit 0
			;;

		-W)
			# Disable patchset
			if ! patch_enable "$2" 2; then
				abort "Wrong usage of -W commandline argument, expected patchname."
			fi
			shift
			shift
			;;

		*)
			# Enable patchset
			if ! patch_enable "$1" 1; then
				abort "Unknown commandline argument $1."
			fi
			shift
			;;
	esac
done

# Determine DESTDIR if not explicitly specified
if test -z "$DESTDIR" -a -f ./tools/make_requests; then
	DESTDIR="$(pwd)"

elif test ! -f "$DESTDIR/tools/make_requests"; then
	abort "DESTDIR does not point to the Wine source tree."
fi

# Change directory to DESTDIR, eapply/epatch depends on that
if ! cd "$DESTDIR"; then
	abort "Unable to change directory to $DESTDIR."
fi

# Helper to update configure / the wineserver protocol if required
if ! command -v diff >/dev/null 2>&1 ||
   ! command -v grep >/dev/null 2>&1 ||
   ! command -v cmp >/dev/null 2>&1; then

	update_configure()
	{
		autoreconf -f
	}

	update_protocol()
	{
		./tools/make_requests
	}

else

	update_configure()
	{
		_file="./configure"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! autoreconf -f; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Shifting by 62 bits is undefined behaviour when off_t is 32-bit, see also
		# https://launchpad.net/ubuntu/+source/autoconf/2.69-6 - the bug is still
		# present in some other distros (including Archlinux).
		_large_off_old="^#define LARGE_OFF_T (((off_t) 1 << 62) - 1 + ((off_t) 1 << 62))\$"
		_large_off_new="#define LARGE_OFF_T ((((off_t) 1 << 31) << 31) - 1 + (((off_t) 1 << 31) << 31))"
		sed -i'' -e "s|$_large_off_old|$_large_off_new|g" "$_file"
		unset _large_off_old _large_off_new

		# Restore original timestamp when nothing changed
		if ! cmp "$_file.old" "$_file" >/dev/null; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}

	update_protocol()
	{
		_file="./include/wine/server_protocol.h"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! ./tools/make_requests; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Restore original timestamp when nothing changed
		if diff -u "$_file.old" "$_file" |
		   grep -v "^[+-]#define SERVER_PROTOCOL_VERSION" |
		   grep -v "^\(+++\|---\)" | grep -q "^[+-]"; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}
fi


# Most backends will try to use git, either directly or indirectly.
# Unfortunately this does not work when "$DESTDIR" points to a
# subdirectory of a git tree, which has the effect that no patches
# are applied, but the exitcode is zero. To avoid broken builds we
# will workaround this issue or abort.
test ! -e ".git" && git rev-parse --git-dir >/dev/null 2>&1
workaround_git_bug="$?"

# Apply the patches using gitapply.sh, a small wrapper around 'patch'
if test "$backend" = "patch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! "$patchdir/gitapply.sh" $gitapply_args < "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# 'eapply/epatch' backend - used on Gentoo
elif test "$backend" = "eapply" -o "$backend" = "epatch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if ! command -v "$backend"  >/dev/null 2>&1 || \
	   ! command -v ebegin      >/dev/null 2>&1 || \
	   ! command -v eend        >/dev/null 2>&1 || \
	   ! command -v nonfatal    >/dev/null 2>&1; then
		abort "Shell functions $backend/ebegin/eend/nonfatal not found. You have to source this script from your ebuild."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		_shortname="$(basename "$1")"
		if grep -q "^GIT binary patch" "$1"; then
			ebegin "Applying $_shortname"
			"$patchdir/gitapply.sh" $gitapply_args < "$1"
			if ! eend $?; then
				exit 1
			fi
		else
			# we are run from a subshell, so we can't call die
			if ! nonfatal "$backend" "$1"; then
				exit 1
			fi
		fi
		unset _shortname
	}

# GIT backend - apply patches using 'git am'
elif test "$backend" = "git" -o "$backend" = "git-am"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-am' not possible when DESTDIR points to a git subdirectory."
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! git am "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
		if test "$enable_autoconf" -gt 1; then
			_do_commit=0

			# Run 'autoreconf -f' if required
			if git show --pretty=format: --name-only | grep -q "^\(configure.ac\|aclocal.m4\)$"; then
				if ! update_configure; then
					abort "'autoreconf -f' failed."
				fi
				git add ./configure
				git add ./include/config.h.in
				_do_commit=1
			fi

			# Run './tools/make_requests' if required
			if git show --pretty=format: --name-only | grep -q "^server/"; then
				if ! update_protocol; then
					abort "'./tools/make_requests' failed."
				fi
				git add ./include/wine/server_protocol.h
				git add ./server/trace.c
				git add ./server/request.h
				_do_commit=1
			fi

			if test "$_do_commit" -ne 0; then
				if ! git commit --amend --reuse-message HEAD; then
					abort "Failed to include autogenerated changes in commit."
				fi
			fi

			unset _do_commit
		fi
	}

# Git apply backend
elif test "$backend" = "git-apply"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-apply' not possible when DESTDIR points to a git subdirectory."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! git apply "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# Stacked GIT backend - import the patches (mainly for developers)
elif test "$backend" = "stg"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'stg' not possible when DESTDIR points to a git subdirectory."
	fi

	# Only import the regular patches, no autogenerated ones -
	# moreover, don't run autoreconf or ./tools/make_requests.
	enable_autoconf=0

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		_shortname="$(basename "$1")"
		if ! printf '%s\n' "staging/$_shortname" | cat - "$1" | stg import; then
			abort "Failed to apply patch, aborting!"
		fi
		unset _shortname
	}

else
	abort "Selected backend $backend not supported."
fi

patch_apply()
{
	patch_apply_file "$patchdir/$1"
}


if test "$enable_winex11_WM_WINDOWPOSCHANGING" -eq 1; then
	if test "$enable_winex11__NET_ACTIVE_WINDOW" -gt 1; then
		abort "Patchset winex11-_NET_ACTIVE_WINDOW disabled, but winex11-WM_WINDOWPOSCHANGING depends on that."
	fi
	enable_winex11__NET_ACTIVE_WINDOW=1
fi

if test "$enable_wined3d_Indexed_Vertex_Blending" -eq 1; then
	if test "$enable_wined3d_SWVP_shaders" -gt 1; then
		abort "Patchset wined3d-SWVP-shaders disabled, but wined3d-Indexed_Vertex_Blending depends on that."
	fi
	enable_wined3d_SWVP_shaders=1
fi

if test "$enable_wineboot_ProxySettings" -eq 1; then
	if test "$enable_wineboot_DriveSerial" -gt 1; then
		abort "Patchset wineboot-DriveSerial disabled, but wineboot-ProxySettings depends on that."
	fi
	if test "$enable_wineboot_drivers_etc_Stubs" -gt 1; then
		abort "Patchset wineboot-drivers_etc_Stubs disabled, but wineboot-ProxySettings depends on that."
	fi
	enable_wineboot_DriveSerial=1
	enable_wineboot_drivers_etc_Stubs=1
fi

if test "$enable_windows_networking_connectivity_dll" -eq 1; then
	if test "$enable_windows_globalization_dll" -gt 1; then
		abort "Patchset windows.globalization-dll disabled, but windows.networking.connectivity.dll depends on that."
	fi
	enable_windows_globalization_dll=1
fi

if test "$enable_windows_globalization_dll" -eq 1; then
	if test "$enable_windows_gaming_input_dll" -gt 1; then
		abort "Patchset windows.gaming.input-dll disabled, but windows.globalization-dll depends on that."
	fi
	enable_windows_gaming_input_dll=1
fi

if test "$enable_windows_gaming_input_dll" -eq 1; then
	if test "$enable_windows_media_speech_dll" -gt 1; then
		abort "Patchset windows.media.speech.dll disabled, but windows.gaming.input-dll depends on that."
	fi
	enable_windows_media_speech_dll=1
fi

if test "$enable_windows_media_speech_dll" -eq 1; then
	if test "$enable_widl_winrt_support" -gt 1; then
		abort "Patchset widl-winrt-support disabled, but windows.media.speech.dll depends on that."
	fi
	enable_widl_winrt_support=1
fi

if test "$enable_user32_window_activation" -eq 1; then
	if test "$enable_user32_recursive_activation" -gt 1; then
		abort "Patchset user32-recursive-activation disabled, but user32-window-activation depends on that."
	fi
	enable_user32_recursive_activation=1
fi

if test "$enable_user32_rawinput_mouse_experimental" -eq 1; then
	if test "$enable_user32_rawinput_mouse" -gt 1; then
		abort "Patchset user32-rawinput-mouse disabled, but user32-rawinput-mouse-experimental depends on that."
	fi
	enable_user32_rawinput_mouse=1
fi

if test "$enable_user32_rawinput_hid" -eq 1; then
	if test "$enable_user32_rawinput_mouse" -gt 1; then
		abort "Patchset user32-rawinput-mouse disabled, but user32-rawinput-hid depends on that."
	fi
	enable_user32_rawinput_mouse=1
fi

if test "$enable_user32_rawinput_mouse" -eq 1; then
	if test "$enable_winex11_drv_mouse_coorrds" -gt 1; then
		abort "Patchset winex11.drv-mouse-coorrds disabled, but user32-rawinput-mouse depends on that."
	fi
	enable_winex11_drv_mouse_coorrds=1
fi

if test "$enable_stdole32_tlb_SLTG_Typelib" -eq 1; then
	if test "$enable_widl_SLTG_Typelib_Support" -gt 1; then
		abort "Patchset widl-SLTG_Typelib_Support disabled, but stdole32.tlb-SLTG_Typelib depends on that."
	fi
	enable_widl_SLTG_Typelib_Support=1
fi

if test "$enable_shell32_ACE_Viewer" -eq 1; then
	if test "$enable_shell32_Progress_Dialog" -gt 1; then
		abort "Patchset shell32-Progress_Dialog disabled, but shell32-ACE_Viewer depends on that."
	fi
	enable_shell32_Progress_Dialog=1
fi

if test "$enable_shell32_Progress_Dialog" -eq 1; then
	if test "$enable_kernel32_CopyFileEx" -gt 1; then
		abort "Patchset kernel32-CopyFileEx disabled, but shell32-Progress_Dialog depends on that."
	fi
	if test "$enable_shell32_SHFileOperation_Move" -gt 1; then
		abort "Patchset shell32-SHFileOperation_Move disabled, but shell32-Progress_Dialog depends on that."
	fi
	enable_kernel32_CopyFileEx=1
	enable_shell32_SHFileOperation_Move=1
fi

if test "$enable_server_Inherited_ACLs" -eq 1; then
	if test "$enable_server_Stored_ACLs" -gt 1; then
		abort "Patchset server-Stored_ACLs disabled, but server-Inherited_ACLs depends on that."
	fi
	enable_server_Stored_ACLs=1
fi

if test "$enable_server_Stored_ACLs" -eq 1; then
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but server-Stored_ACLs depends on that."
	fi
	if test "$enable_server_File_Permissions" -gt 1; then
		abort "Patchset server-File_Permissions disabled, but server-Stored_ACLs depends on that."
	fi
	enable_ntdll_DOS_Attributes=1
	enable_server_File_Permissions=1
fi

if test "$enable_server_File_Permissions" -eq 1; then
	if test "$enable_ntdll_Junction_Points" -gt 1; then
		abort "Patchset ntdll-Junction_Points disabled, but server-File_Permissions depends on that."
	fi
	enable_ntdll_Junction_Points=1
fi

if test "$enable_oleaut32_OLEPictureImpl_SaveAsFile" -eq 1; then
	if test "$enable_oleaut32_Load_Save_EMF" -gt 1; then
		abort "Patchset oleaut32-Load_Save_EMF disabled, but oleaut32-OLEPictureImpl_SaveAsFile depends on that."
	fi
	enable_oleaut32_Load_Save_EMF=1
fi

if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	if test "$enable_nvcuvid_CUDA_Video_Support" -gt 1; then
		abort "Patchset nvcuvid-CUDA_Video_Support disabled, but nvencodeapi-Video_Encoder depends on that."
	fi
	enable_nvcuvid_CUDA_Video_Support=1
fi

if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	if test "$enable_nvapi_Stub_DLL" -gt 1; then
		abort "Patchset nvapi-Stub_DLL disabled, but nvcuvid-CUDA_Video_Support depends on that."
	fi
	enable_nvapi_Stub_DLL=1
fi

if test "$enable_ntdll_NtQueryVirtualMemory" -eq 1; then
	if test "$enable_ntdll_ForceBottomUpAlloc" -gt 1; then
		abort "Patchset ntdll-ForceBottomUpAlloc disabled, but ntdll-NtQueryVirtualMemory depends on that."
	fi
	if test "$enable_ntdll_Junction_Points" -gt 1; then
		abort "Patchset ntdll-Junction_Points disabled, but ntdll-NtQueryVirtualMemory depends on that."
	fi
	if test "$enable_ntdll_NtDevicePath" -gt 1; then
		abort "Patchset ntdll-NtDevicePath disabled, but ntdll-NtQueryVirtualMemory depends on that."
	fi
	enable_ntdll_ForceBottomUpAlloc=1
	enable_ntdll_Junction_Points=1
	enable_ntdll_NtDevicePath=1
fi

if test "$enable_ntdll_NtDevicePath" -eq 1; then
	if test "$enable_ntdll_Pipe_SpecialCharacters" -gt 1; then
		abort "Patchset ntdll-Pipe_SpecialCharacters disabled, but ntdll-NtDevicePath depends on that."
	fi
	enable_ntdll_Pipe_SpecialCharacters=1
fi

if test "$enable_ntdll_NtAlertThreadByThreadId" -eq 1; then
	if test "$enable_server_Object_Types" -gt 1; then
		abort "Patchset server-Object_Types disabled, but ntdll-NtAlertThreadByThreadId depends on that."
	fi
	enable_server_Object_Types=1
fi

if test "$enable_ntdll_Builtin_Prot" -eq 1; then
	if test "$enable_ntdll_WRITECOPY" -gt 1; then
		abort "Patchset ntdll-WRITECOPY disabled, but ntdll-Builtin_Prot depends on that."
	fi
	enable_ntdll_WRITECOPY=1
fi

if test "$enable_ntdll_WRITECOPY" -eq 1; then
	if test "$enable_ntdll_ForceBottomUpAlloc" -gt 1; then
		abort "Patchset ntdll-ForceBottomUpAlloc disabled, but ntdll-WRITECOPY depends on that."
	fi
	enable_ntdll_ForceBottomUpAlloc=1
fi

if test "$enable_kernel32_Processor_Group" -eq 1; then
	if test "$enable_api_ms_win_Stub_DLLs" -gt 1; then
		abort "Patchset api-ms-win-Stub_DLLs disabled, but kernel32-Processor_Group depends on that."
	fi
	enable_api_ms_win_Stub_DLLs=1
fi

if test "$enable_kernel32_CopyFileEx" -eq 1; then
	if test "$enable_ntdll_FileDispositionInformation" -gt 1; then
		abort "Patchset ntdll-FileDispositionInformation disabled, but kernel32-CopyFileEx depends on that."
	fi
	enable_ntdll_FileDispositionInformation=1
fi

if test "$enable_eventfd_synchronization" -eq 1; then
	if test "$enable_ntdll_Junction_Points" -gt 1; then
		abort "Patchset ntdll-Junction_Points disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_PeekMessage" -gt 1; then
		abort "Patchset server-PeekMessage disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_Realtime_Priority" -gt 1; then
		abort "Patchset server-Realtime_Priority disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_Signal_Thread" -gt 1; then
		abort "Patchset server-Signal_Thread disabled, but eventfd_synchronization depends on that."
	fi
	enable_ntdll_Junction_Points=1
	enable_server_PeekMessage=1
	enable_server_Realtime_Priority=1
	enable_server_Signal_Thread=1
fi

if test "$enable_ntdll_Junction_Points" -eq 1; then
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but ntdll-Junction_Points depends on that."
	fi
	if test "$enable_ntdll_NtQueryEaFile" -gt 1; then
		abort "Patchset ntdll-NtQueryEaFile disabled, but ntdll-Junction_Points depends on that."
	fi
	enable_ntdll_DOS_Attributes=1
	enable_ntdll_NtQueryEaFile=1
fi

if test "$enable_dxdiag_new_dlls" -eq 1; then
	if test "$enable_winedevice_Default_Drivers" -gt 1; then
		abort "Patchset winedevice-Default_Drivers disabled, but dxdiag-new-dlls depends on that."
	fi
	enable_winedevice_Default_Drivers=1
fi

if test "$enable_winedevice_Default_Drivers" -eq 1; then
	if test "$enable_ntoskrnl_Stubs" -gt 1; then
		abort "Patchset ntoskrnl-Stubs disabled, but winedevice-Default_Drivers depends on that."
	fi
	enable_ntoskrnl_Stubs=1
fi

if test "$enable_dsound_EAX" -eq 1; then
	if test "$enable_dsound_Fast_Mixer" -gt 1; then
		abort "Patchset dsound-Fast_Mixer disabled, but dsound-EAX depends on that."
	fi
	enable_dsound_Fast_Mixer=1
fi

if test "$enable_dinput_SetActionMap_genre" -eq 1; then
	if test "$enable_dinput_joy_mappings" -gt 1; then
		abort "Patchset dinput-joy-mappings disabled, but dinput-SetActionMap-genre depends on that."
	fi
	enable_dinput_joy_mappings=1
fi

if test "$enable_ddraw_version_check" -eq 1; then
	if test "$enable_ddraw_Device_Caps" -gt 1; then
		abort "Patchset ddraw-Device_Caps disabled, but ddraw-version-check depends on that."
	fi
	enable_ddraw_Device_Caps=1
fi

if test "$enable_ddraw_Texture_Wrong_Caps" -eq 1; then
	if test "$enable_ddraw_Rendering_Targets" -gt 1; then
		abort "Patchset ddraw-Rendering_Targets disabled, but ddraw-Texture_Wrong_Caps depends on that."
	fi
	enable_ddraw_Rendering_Targets=1
fi

if test "$enable_d3d11_Deferred_Context" -eq 1; then
	if test "$enable_nvapi_Stub_DLL" -gt 1; then
		abort "Patchset nvapi-Stub_DLL disabled, but d3d11-Deferred_Context depends on that."
	fi
	enable_nvapi_Stub_DLL=1
fi

if test "$enable_nvapi_Stub_DLL" -eq 1; then
	if test "$enable_nvcuda_CUDA_Support" -gt 1; then
		abort "Patchset nvcuda-CUDA_Support disabled, but nvapi-Stub_DLL depends on that."
	fi
	enable_nvcuda_CUDA_Support=1
fi


# Patchset Compiler_Warnings
# |
# | Modified files:
# |   *	dlls/d2d1/bitmap.c, dlls/d2d1/brush.c, dlls/d2d1/dc_render_target.c, dlls/d2d1/geometry.c,
# | 	dlls/d2d1/hwnd_render_target.c, dlls/d2d1/state_block.c, dlls/d3d11/view.c, dlls/d3d8/texture.c, dlls/d3d9/texture.c,
# | 	dlls/ddraw/viewport.c, dlls/dwrite/font.c, dlls/dwrite/layout.c, dlls/msxml3/schema.c, dlls/oleaut32/oleaut.c,
# | 	dlls/rpcrt4/cstub.c, dlls/vbscript/vbdisp.c, dlls/windowscodecs/info.c, dlls/wsdapi/msgparams.c, include/wine/list.h,
# | 	include/wine/rbtree.h, include/winnt.h
# |
if test "$enable_Compiler_Warnings" -eq 1; then
	patch_apply Compiler_Warnings/0001-windowscodecs-Avoid-implicit-cast-of-interface-point.patch
	patch_apply Compiler_Warnings/0021-d2d1-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0022-d3d11-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0023-d3d8-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0024-d3d9-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0025-ddraw-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0026-dwrite-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0027-msxml3-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0028-oleaut32-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0029-rpcrt4-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0030-vbscript-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0031-include-Check-element-type-in-CONTAINING_RECORD-and-.patch
	patch_apply Compiler_Warnings/0032-wsdapi-Avoid-implicit-cast-of-interface-pointer.patch
fi

# Patchset Pipelight
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/user32/tests/msg.c, dlls/winex11.drv/init.c, dlls/winex11.drv/opengl.c,
# | 	dlls/winex11.drv/x11drv.h
# |
if test "$enable_Pipelight" -eq 1; then
	patch_apply Pipelight/0001-winex11-Implement-X11DRV_FLUSH_GDI_DISPLAY-ExtEscape-c.patch
	patch_apply Pipelight/0002-user32-Decrease-minimum-SetTimer-interval-to-5-ms.patch
	patch_apply Pipelight/0004-winex11.drv-Indicate-direct-rendering-through-OpenGL.patch
fi

# Patchset Staging
# |
# | Modified files:
# |   *	Makefile.in, dlls/ntdll/Makefile.in, dlls/ntdll/loader.c
# |
if test "$enable_Staging" -eq 1; then
	patch_apply Staging/0001-kernel32-Add-winediag-message-to-show-warning-that-t.patch
	patch_apply Staging/0002-winelib-Append-Staging-at-the-end-of-the-version-s.patch
fi

# Patchset advapi32-LsaLookupPrivilegeName
# |
# | Modified files:
# |   *	dlls/advapi32/lsa.c, dlls/advapi32/tests/lsa.c
# |
if test "$enable_advapi32_LsaLookupPrivilegeName" -eq 1; then
	patch_apply advapi32-LsaLookupPrivilegeName/0001-advapi32-Fix-error-code-when-calling-LsaOpenPolicy-f.patch
	patch_apply advapi32-LsaLookupPrivilegeName/0002-advapi32-Use-TRACE-for-LsaOpenPolicy-LsaClose.patch
fi

# Patchset api-ms-win-Stub_DLLs
# |
# | Modified files:
# |   *	configure.ac, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/Makefile.in, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/ext-
# | 	ms-win-appmodel-usercontext-l1-1-0.spec, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/main.c, dlls/ext-ms-win-xaml-
# | 	pal-l1-1-0/Makefile.in, dlls/ext-ms-win-xaml-pal-l1-1-0/ext-ms-win-xaml-pal-l1-1-0.spec, dlls/ext-ms-win-xaml-
# | 	pal-l1-1-0/main.c, dlls/iertutil/Makefile.in, dlls/iertutil/iertutil.spec, dlls/iertutil/main.c,
# | 	dlls/uiautomationcore/Makefile.in, dlls/uiautomationcore/uia_main.c
# |
if test "$enable_api_ms_win_Stub_DLLs" -eq 1; then
	patch_apply api-ms-win-Stub_DLLs/0006-iertutil-Add-dll-and-add-stub-for-ordinal-811.patch
	patch_apply api-ms-win-Stub_DLLs/0009-ext-ms-win-xaml-pal-l1-1-0-Add-dll-and-add-stub-for-.patch
	patch_apply api-ms-win-Stub_DLLs/0010-ext-ms-win-appmodel-usercontext-l1-1-0-Add-dll-and-a.patch
	patch_apply api-ms-win-Stub_DLLs/0012-ext-ms-win-xaml-pal-l1-1-0-Add-stub-for-GetThemeServ.patch
	patch_apply api-ms-win-Stub_DLLs/0027-uiautomationcore-Add-dll-and-stub-some-functions.patch
fi

# Patchset atl-AtlAxDialogBox
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32749] atl: Implemented AtlAxDialogBox[A,W]
# |
# | Modified files:
# |   *	dlls/atl/atl_ax.c
# |
if test "$enable_atl_AtlAxDialogBox" -eq 1; then
	patch_apply atl-AtlAxDialogBox/0001-atl-Implement-AtlAxDialogBox-A-W.patch
fi

# Patchset bcrypt-ECDHSecretAgreement
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47699] Multiple games fail to connect to online services (missing BCryptSecretAgreement / BCryptDeriveKey
# | 	implementation)
# |
# | Modified files:
# |   *	configure.ac, dlls/bcrypt/Makefile.in, dlls/bcrypt/bcrypt_internal.h, dlls/bcrypt/bcrypt_main.c, dlls/bcrypt/gcrypt.c,
# | 	dlls/bcrypt/gnutls.c, dlls/bcrypt/macos.c, dlls/bcrypt/tests/bcrypt.c, dlls/bcrypt/unixlib.c
# |
if test "$enable_bcrypt_ECDHSecretAgreement" -eq 1; then
	patch_apply bcrypt-ECDHSecretAgreement/0001-bcrypt-Allow-multiple-backends-to-coexist.patch
	patch_apply bcrypt-ECDHSecretAgreement/0002-bcrypt-Implement-BCryptSecretAgreement-with-libgcryp.patch
	patch_apply bcrypt-ECDHSecretAgreement/0003-bcrypt-Implement-BCRYPT_KDF_HASH.patch
fi

# Patchset cmd-launch-association
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18154] cmd: Support for launching programs based on file association
# |
# | Modified files:
# |   *	programs/cmd/builtins.c, programs/cmd/tests/test_builtins.cmd, programs/cmd/tests/test_builtins.cmd.exp,
# | 	programs/cmd/wcmdmain.c
# |
if test "$enable_cmd_launch_association" -eq 1; then
	patch_apply cmd-launch-association/0001-cmd-Support-for-launching-programs-based-on-file-ass.patch
	patch_apply cmd-launch-association/0002-cmd-ftype-failed-to-clear-file-associations.patch
fi

# Patchset comctl32-Listview_DrawItem
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38941] Preserve custom colors between subitems for listview control
# |
# | Modified files:
# |   *	dlls/comctl32/listview.c
# |
if test "$enable_comctl32_Listview_DrawItem" -eq 1; then
	patch_apply comctl32-Listview_DrawItem/0001-comctl32-Preserve-custom-colors-between-subitems.patch
fi

# Patchset comctl32-rebar-capture
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#14750] comctl32: Fixed rebar behaviour when there's capture and no drag.
# |
# | Modified files:
# |   *	dlls/comctl32/rebar.c
# |
if test "$enable_comctl32_rebar_capture" -eq 1; then
	patch_apply comctl32-rebar-capture/0001-comctl32-Fixed-rebar-behaviour-when-there-s-capture-.patch
fi

# Patchset comctl32-version_6
# |
# | Modified files:
# |   *	dlls/comctl32/comctl32.h, dlls/comctl32/comctl32.rc, include/commctrl.h
# |
if test "$enable_comctl32_version_6" -eq 1; then
	patch_apply comctl32-version_6/0001-comctl32-Bump-version-to-6.0.patch
fi

# Patchset comdlg32-lpstrFileTitle
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38400] Postpone setting lpstrFileTitle in GetSaveFileNameW
# |   *	[#35200] Postpone setting lpstrFileTitle in GetSaveFileNameA
# |
# | Modified files:
# |   *	dlls/comdlg32/filedlg.c
# |
if test "$enable_comdlg32_lpstrFileTitle" -eq 1; then
	patch_apply comdlg32-lpstrFileTitle/0001-comdlg32-Postpone-setting-ofn-lpstrFileTitle-to-work.patch
fi

# Patchset crypt32-CMS_Certificates
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34388] Skip unknown item when decoding a CMS certificate
# |
# | Modified files:
# |   *	dlls/crypt32/decode.c
# |
if test "$enable_crypt32_CMS_Certificates" -eq 1; then
	patch_apply crypt32-CMS_Certificates/0001-crypt32-Skip-unknown-item-when-decoding-a-CMS-certif.patch
fi

# Patchset cryptext-CryptExtOpenCER
# |
# | Modified files:
# |   *	configure, configure.ac, dlls/cryptext/Makefile.in, dlls/cryptext/cryptext.spec, dlls/cryptext/cryptext_main.c,
# | 	dlls/cryptext/tests/Makefile.in, dlls/cryptext/tests/cryptext.c
# |
if test "$enable_cryptext_CryptExtOpenCER" -eq 1; then
	patch_apply cryptext-CryptExtOpenCER/0001-cryptext-Implement-CryptExtOpenCER.patch
fi

# Patchset nvcuda-CUDA_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37664] MediaCoder needs CUDA for video encoding
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuda/Makefile.in, dlls/nvcuda/internal.c, dlls/nvcuda/nvcuda.c, dlls/nvcuda/nvcuda.h,
# | 	dlls/nvcuda/nvcuda.rc, dlls/nvcuda/nvcuda.spec, dlls/nvcuda/tests/Makefile.in, dlls/nvcuda/tests/nvcuda.c,
# | 	include/Makefile.in, include/cuda.h
# |
if test "$enable_nvcuda_CUDA_Support" -eq 1; then
	patch_apply nvcuda-CUDA_Support/0001-include-Add-cuda.h.h.patch
	patch_apply nvcuda-CUDA_Support/0002-nvcuda-Add-stub-dll.patch
	patch_apply nvcuda-CUDA_Support/0003-nvcuda-First-implementation.patch
	patch_apply nvcuda-CUDA_Support/0004-nvcuda-Implement-new-functions-added-in-CUDA-6.5.patch
	patch_apply nvcuda-CUDA_Support/0005-nvcuda-Properly-wrap-undocumented-ContextStorage-int.patch
	patch_apply nvcuda-CUDA_Support/0006-nvcuda-Emulate-two-d3d9-initialization-functions.patch
	patch_apply nvcuda-CUDA_Support/0007-nvcuda-Properly-wrap-stream-callbacks-by-forwarding-.patch
	patch_apply nvcuda-CUDA_Support/0008-nvcuda-Add-support-for-CUDA-7.0.patch
	patch_apply nvcuda-CUDA_Support/0009-nvcuda-Implement-cuModuleLoad-wrapper-function.patch
	patch_apply nvcuda-CUDA_Support/0010-nvcuda-Search-for-dylib-library-on-Mac-OS-X.patch
	patch_apply nvcuda-CUDA_Support/0011-nvcuda-Add-semi-stub-for-cuD3D10GetDevice.patch
fi

# Patchset nvapi-Stub_DLL
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35062] Fix graphical corruption in FarCry 3 with NVIDIA drivers
# |   *	[#43862] CS:GO fails to start when nvapi cannot be initialized
# |
# | Modified files:
# |   *	configure.ac, dlls/d3d11/device.c, dlls/nvapi/Makefile.in, dlls/nvapi/nvapi.c, dlls/nvapi/nvapi.spec,
# | 	dlls/nvapi/tests/Makefile.in, dlls/nvapi/tests/nvapi.c, dlls/nvapi64/Makefile.in, dlls/nvapi64/nvapi64.spec,
# | 	include/Makefile.in, include/nvapi.h, include/wine/wined3d.h
# |
if test "$enable_nvapi_Stub_DLL" -eq 1; then
	patch_apply nvapi-Stub_DLL/0001-nvapi-First-implementation.patch
	patch_apply nvapi-Stub_DLL/0002-nvapi-Add-stubs-for-NvAPI_EnumLogicalGPUs-and-undocu.patch
	patch_apply nvapi-Stub_DLL/0003-nvapi-Add-NvAPI_GetPhysicalGPUsFromLogicalGPU.patch
	patch_apply nvapi-Stub_DLL/0004-nvapi-Add-stub-for-NvAPI_EnumPhysicalGPUs.patch
	patch_apply nvapi-Stub_DLL/0005-nvapi-Add-stubs-for-NvAPI_GPU_GetFullName.patch
	patch_apply nvapi-Stub_DLL/0006-nvapi-Explicity-return-NULL-for-0x33c7358c-and-0x593.patch
	patch_apply nvapi-Stub_DLL/0007-nvapi-Add-stub-for-NvAPI_DISP_GetGDIPrimaryDisplayId.patch
	patch_apply nvapi-Stub_DLL/0008-nvapi-Add-stub-for-EnumNvidiaDisplayHandle.patch
	patch_apply nvapi-Stub_DLL/0009-nvapi-Add-stub-for-NvAPI_SYS_GetDriverAndBranchVersi.patch
	patch_apply nvapi-Stub_DLL/0010-nvapi-Add-stub-for-NvAPI_Unload.patch
	patch_apply nvapi-Stub_DLL/0011-nvapi-Add-stub-for-NvAPI_D3D_GetCurrentSLIState.patch
	patch_apply nvapi-Stub_DLL/0012-nvapi-tests-Use-structure-to-list-imports.patch
	patch_apply nvapi-Stub_DLL/0013-nvapi-Add-stub-for-NvAPI_GetLogicalGPUFromDisplay.patch
	patch_apply nvapi-Stub_DLL/0014-nvapi-Add-stub-for-NvAPI_D3D_GetObjectHandleForResou.patch
	patch_apply nvapi-Stub_DLL/0015-nvapi-Add-stub-for-NvAPI_D3D9_RegisterResource.patch
	patch_apply nvapi-Stub_DLL/0016-nvapi-Improve-NvAPI_D3D_GetCurrentSLIState.patch
	patch_apply nvapi-Stub_DLL/0017-nvapi-Implement-NvAPI_GPU_Get-Physical-Virtual-Frame.patch
	patch_apply nvapi-Stub_DLL/0018-nvapi-Add-stub-for-NvAPI_GPU_GetGpuCoreCount.patch
	patch_apply nvapi-Stub_DLL/0019-nvapi-Implement-NvAPI_D3D11_SetDepthBoundsTest.patch
	patch_apply nvapi-Stub_DLL/0020-nvapi-Implement-NvAPI_D3D11_CreateDevice-and-NvAPI_D.patch
fi

# Patchset d3d11-Deferred_Context
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support, nvapi-Stub_DLL
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42191] Add semi-stub for D3D11 deferred context implementation
# |   *	[#43743] No 3D graphics in Wolcen: Lords of Mayhem
# |   *	[#44089] Correcly align the mapinfo buffer.
# |   *	[#41636] d3d11: Implement ID3D11Device2 GetImmediateContext1
# |
# | Modified files:
# |   *	dlls/d3d11/device.c, dlls/d3d11/tests/d3d11.c, dlls/wined3d/buffer.c, dlls/wined3d/resource.c, dlls/wined3d/texture.c,
# | 	dlls/wined3d/wined3d.spec, dlls/wined3d/wined3d_private.h, include/wine/wined3d.h
# |
if test "$enable_d3d11_Deferred_Context" -eq 1; then
	patch_apply d3d11-Deferred_Context/0001-d3d11-Add-stub-deferred-rendering-context.patch
	patch_apply d3d11-Deferred_Context/0002-wined3d-Add-wined3d_resource_map_info-function.patch
	patch_apply d3d11-Deferred_Context/0003-d3d11-Initial-implementation-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0004-d3d11-Implement-CSSetShader-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0005-d3d11-Implement-CSSetConstantBuffers-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0006-d3d11-Implement-Dispatch-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0007-d3d11-Implement-CSSetUnorderedAccessViews-for-deferr.patch
	patch_apply d3d11-Deferred_Context/0008-d3d11-Implement-ClearRenderTargetView-for-deferred-c.patch
	patch_apply d3d11-Deferred_Context/0009-d3d11-Implement-Draw-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0010-d3d11-Implement-ClearDepthStencilView-for-deferred-c.patch
	patch_apply d3d11-Deferred_Context/0011-d3d11-Implement-GSSetShader-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0012-d3d11-Implement-GSSetConstantBuffers-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0013-d3d11-Implement-CSSetShaderResources-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0014-d3d11-Implement-GSSetShaderResources-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0015-d3d11-Implement-HSSetShaderResources-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0016-d3d11-Implement-VSSetShaderResources-for-deferred-co.patch
	patch_apply d3d11-Deferred_Context/0017-d3d11-Implement-CSSetSamplers-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0018-d3d11-Implement-GSSetSamplers-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0019-d3d11-Implement-HSSetSamplers-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0020-d3d11-Implement-VSSetSamplers-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0021-d3d11-Implement-Begin-and-End-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0022-d3d11-Implement-CopyResource-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0023-d3d11-Implement-SetResourceMinLOD-for-deferred-conte.patch
	patch_apply d3d11-Deferred_Context/0024-d3d11-Implement-CopySubresourceRegion-for-deferred-c.patch
	patch_apply d3d11-Deferred_Context/0025-d3d11-Implement-ResolveSubresource-for-deferred-cont.patch
	patch_apply d3d11-Deferred_Context/0026-d3d11-Implement-CopyStructureCount-for-deferred-cont.patch
	patch_apply d3d11-Deferred_Context/0027-d3d11-Implement-DrawAuto-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0028-d3d11-Implement-DrawInstanced-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0029-d3d11-Implement-DrawInstancedIndirect-for-deferred-c.patch
	patch_apply d3d11-Deferred_Context/0030-d3d11-Implement-DrawIndexedInstancedIndirect-for-def.patch
	patch_apply d3d11-Deferred_Context/0031-d3d11-Implement-ClearUnorderedAccessViewUint-for-def.patch
	patch_apply d3d11-Deferred_Context/0032-d3d11-Implement-ClearUnorderedAccessViewFloat-for-de.patch
	patch_apply d3d11-Deferred_Context/0033-d3d11-Implement-RsSetScissorRects-for-deferred-conte.patch
	patch_apply d3d11-Deferred_Context/0034-d3d11-Implement-OMSetRenderTargetsAndUnorderedAccess.patch
	patch_apply d3d11-Deferred_Context/0035-d3d11-Implement-SOSetTargets-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0036-d3d11-Implement-GenerateMips-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0037-d3d11-Implement-DispatchIndirect-for-deferred-contex.patch
	patch_apply d3d11-Deferred_Context/0038-d3d11-Implement-SetPredication-for-deferred-contexts.patch
	patch_apply d3d11-Deferred_Context/0039-d3d11-Implement-d3d11_deferred_context_UpdateSubreso.patch
	patch_apply d3d11-Deferred_Context/0040-d3d11-Implement-restoring-of-state-after-executing-a.patch
	patch_apply d3d11-Deferred_Context/0041-d3d11-Allow-NULL-pointer-for-initial-count-in-d3d11_.patch
	patch_apply d3d11-Deferred_Context/0042-d3d11-Correctly-align-map-info-buffer.patch
	patch_apply d3d11-Deferred_Context/0043-d3d11-tests-Add-a-basic-test-for-drawing-with-deferr.patch
	patch_apply d3d11-Deferred_Context/0044-d3d11-Support-ID3D11DeviceContext1-for-deferred-cont.patch
	patch_apply d3d11-Deferred_Context/0045-d3d11-Implement-ID3D11Device2-GetImmediateContext1.patch
fi

# Patchset d3drm-IDirect3D3-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39346] Support IDirect3D3 when creating device.
# |
# | Modified files:
# |   *	dlls/d3drm/device.c
# |
if test "$enable_d3drm_IDirect3D3_support" -eq 1; then
	patch_apply d3drm-IDirect3D3-support/0001-d3drm-Support-IDirect3D3-when-creating-device.patch
fi

# Patchset d3dx9-32bpp_Alpha_Channel
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48563] Runaway: A Twist of Fate renders its cursor incorrectly
# |
# | Modified files:
# |   *	dlls/d3dx9_36/surface.c, dlls/d3dx9_36/tests/surface.c
# |
if test "$enable_d3dx9_32bpp_Alpha_Channel" -eq 1; then
	patch_apply d3dx9-32bpp_Alpha_Channel/0001-d3dx9-Return-D3DFMT_A8R8G8B8-in-D3DXGetImageInfoFrom.patch
fi

# Patchset d3dx9_36-BumpLuminance
# |
# | Modified files:
# |   *	dlls/d3dx9_36/util.c
# |
if test "$enable_d3dx9_36_BumpLuminance" -eq 1; then
	patch_apply d3dx9_36-BumpLuminance/0002-d3dx9_36-Add-format-description-for-X8L8V8U8-for-for.patch
fi

# Patchset d3dx9_36-CloneEffect
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44635] Improve stub for ID3DXEffectImpl_CloneEffect
# |
# | Modified files:
# |   *	dlls/d3dx9_36/effect.c, dlls/d3dx9_36/tests/effect.c
# |
if test "$enable_d3dx9_36_CloneEffect" -eq 1; then
	patch_apply d3dx9_36-CloneEffect/0001-d3dx9_36-Improve-stub-for-ID3DXEffectImpl_CloneEffec.patch
fi

# Patchset d3dx9_36-D3DXDisassembleShader
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46649] Multiple applications need D3DXDisassembleShader() implementation (Tom Clancy's Rainbow Six: Vegas 2, The Void)
# |
# | Modified files:
# |   *	dlls/d3dx9_36/shader.c, dlls/d3dx9_36/tests/shader.c
# |
if test "$enable_d3dx9_36_D3DXDisassembleShader" -eq 1; then
	patch_apply d3dx9_36-D3DXDisassembleShader/0004-d3dx9_36-Implement-D3DXDisassembleShader.patch
	patch_apply d3dx9_36-D3DXDisassembleShader/0005-d3dx9_36-tests-Add-initial-tests-for-D3DXDisassemble.patch
	patch_apply d3dx9_36-D3DXDisassembleShader/0006-d3dx9_36-tests-Add-additional-tests-for-special-case.patch
fi

# Patchset d3dx9_36-D3DXOptimizeVertices
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47776] Timeshift needs D3DXOptimizeVertices()
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/mesh.c, dlls/d3dx9_36/tests/mesh.c, dlls/d3dx9_37/d3dx9_37.spec,
# | 	dlls/d3dx9_38/d3dx9_38.spec, dlls/d3dx9_39/d3dx9_39.spec, dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec,
# | 	dlls/d3dx9_42/d3dx9_42.spec, dlls/d3dx9_43/d3dx9_43.spec
# |
if test "$enable_d3dx9_36_D3DXOptimizeVertices" -eq 1; then
	patch_apply d3dx9_36-D3DXOptimizeVertices/0002-d3dx9_36-Add-semi-stub-for-D3DXOptimizeVertices.patch
fi

# Patchset d3dx9_36-D3DXSHProjectCubeMap
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46284] Add D3DXSHProjectCubeMap stub
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/math.c, dlls/d3dx9_37/d3dx9_37.spec, dlls/d3dx9_38/d3dx9_38.spec, dlls/d3dx9_39/d3dx9_39.spec,
# | 	dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec, dlls/d3dx9_42/d3dx9_42.spec, dlls/d3dx9_43/d3dx9_43.spec,
# | 	include/d3dx9math.h
# |
if test "$enable_d3dx9_36_D3DXSHProjectCubeMap" -eq 1; then
	patch_apply d3dx9_36-D3DXSHProjectCubeMap/0001-d3dx9_-Add-D3DXSHProjectCubeMap-stub.patch
fi

# Patchset d3dx9_36-D3DXStubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41697] Add stub for D3DXComputeTangent
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/mesh.c, dlls/d3dx9_37/d3dx9_37.spec, dlls/d3dx9_38/d3dx9_38.spec,
# | 	dlls/d3dx9_39/d3dx9_39.spec, dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec, dlls/d3dx9_42/d3dx9_42.spec,
# | 	dlls/d3dx9_43/d3dx9_43.spec
# |
if test "$enable_d3dx9_36_D3DXStubs" -eq 1; then
	patch_apply d3dx9_36-D3DXStubs/0003-d3dx9-Implement-D3DXComputeTangent.patch
fi

# Patchset d3dx9_36-DDS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26898] Support for DDS file format in D3DXSaveTextureToFileInMemory
# |
# | Modified files:
# |   *	dlls/d3dx9_36/d3dx9_private.h, dlls/d3dx9_36/surface.c, dlls/d3dx9_36/texture.c
# |
if test "$enable_d3dx9_36_DDS" -eq 1; then
	patch_apply d3dx9_36-DDS/0001-d3dx9_36-Add-support-for-FOURCC-surface-to-save_dds_.patch
	patch_apply d3dx9_36-DDS/0002-d3dx9_36-Improve-D3DXSaveTextureToFile-to-save-simpl.patch
fi

# Patchset d3dx9_36-Filter_Warnings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33770] D3DCompileShader should filter specific warning messages
# |
# | Modified files:
# |   *	dlls/d3dx9_36/shader.c
# |
if test "$enable_d3dx9_36_Filter_Warnings" -eq 1; then
	patch_apply d3dx9_36-Filter_Warnings/0001-d3dx9_36-Filter-out-D3DCompile-warning-messages-that.patch
fi

# Patchset d3dx9_36-UpdateSkinnedMesh
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32572] Support for ID3DXSkinInfoImpl_UpdateSkinnedMesh
# |
# | Modified files:
# |   *	dlls/d3dx9_36/skin.c, dlls/d3dx9_36/tests/mesh.c
# |
if test "$enable_d3dx9_36_UpdateSkinnedMesh" -eq 1; then
	patch_apply d3dx9_36-UpdateSkinnedMesh/0001-d3dx9_36-Implement-ID3DXSkinInfoImpl_UpdateSkinnedMe.patch
fi

# Patchset dbghelp-Debug_Symbols
# |
# | Modified files:
# |   *	dlls/dbghelp/Makefile.in, dlls/dbghelp/elf_module.c
# |
if test "$enable_dbghelp_Debug_Symbols" -eq 1; then
	patch_apply dbghelp-Debug_Symbols/0001-dbghelp-Always-check-for-debug-symbols-in-BINDIR.patch
fi

# Patchset ddraw-Device_Caps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37019] Don't set HWTRANSFORMANDLIGHT flag on d3d7 RGB device
# |   *	[#27002] Properly initialize caps->dwZBufferBitDepths in ddraw7_GetCaps
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c, dlls/ddraw/tests/ddraw7.c
# |
if test "$enable_ddraw_Device_Caps" -eq 1; then
	patch_apply ddraw-Device_Caps/0001-ddraw-Don-t-set-HWTRANSFORMANDLIGHT-flag-on-d3d7-RGB.patch
	patch_apply ddraw-Device_Caps/0002-ddraw-Set-dwZBufferBitDepth-in-ddraw7_GetCaps.patch
fi

# Patchset ddraw-IDirect3DTexture2_Load
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48537] Prezzie Hunt fails to launch
# |
# | Modified files:
# |   *	dlls/ddraw/surface.c, dlls/ddraw/tests/d3d.c, dlls/ddraw/tests/ddraw2.c
# |
if test "$enable_ddraw_IDirect3DTexture2_Load" -eq 1; then
	patch_apply ddraw-IDirect3DTexture2_Load/0001-ddraw-Allow-size-and-format-conversions-in-IDirect3D.patch
	patch_apply ddraw-IDirect3DTexture2_Load/0002-ddraw-tests-Add-more-tests-for-IDirect3DTexture2-Loa.patch
fi

# Patchset ddraw-Rendering_Targets
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34906] Use video memory for rendering targets if possible
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c, dlls/ddraw/ddraw_private.h, dlls/ddraw/device.c, dlls/ddraw/surface.c
# |
if test "$enable_ddraw_Rendering_Targets" -eq 1; then
	patch_apply ddraw-Rendering_Targets/0001-ddraw-Create-rendering-targets-in-video-memory-if-po.patch
fi

# Patchset ddraw-Silence_FIXMEs
# |
# | Modified files:
# |   *	dlls/ddraw/executebuffer.c
# |
if test "$enable_ddraw_Silence_FIXMEs" -eq 1; then
	patch_apply ddraw-Silence_FIXMEs/0001-ddraw-Silence-noisy-FIXME-about-unimplemented-D3DPRO.patch
fi

# Patchset ddraw-Texture_Wrong_Caps
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ddraw-Rendering_Targets
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46948] Allow setting texture without DDSCAPS_TEXTURE for software device
# |
# | Modified files:
# |   *	dlls/ddraw/device.c, dlls/ddraw/tests/ddraw4.c
# |
if test "$enable_ddraw_Texture_Wrong_Caps" -eq 1; then
	patch_apply ddraw-Texture_Wrong_Caps/0001-ddraw-Allow-setting-texture-without-DDSCAPS_TEXTURE-.patch
fi

# Patchset ddraw-version-check
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ddraw-Device_Caps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#19153] Resident Evil 1 fails to start (needs IDirect3D3::EnumDevices() to return a device named "RGB Emulation")
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c
# |
if test "$enable_ddraw_version_check" -eq 1; then
	patch_apply ddraw-version-check/0001-ddraw-Return-correct-devices-based-off-requested-Dir.patch
fi

# Patchset dinput-joy-mappings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34108] dinput: Improve support for user Joystick configuration.
# |
# | Modified files:
# |   *	dlls/dinput/config.c, dlls/dinput/device.c, dlls/dinput/device_private.h, dlls/dinput/dinput_main.c,
# | 	dlls/dinput/joystick.c, dlls/dinput/keyboard.c, dlls/dinput/mouse.c, dlls/dinput8/tests/device.c
# |
if test "$enable_dinput_joy_mappings" -eq 1; then
	patch_apply dinput-joy-mappings/0001-dinput-Load-users-Joystick-mappings.patch
	patch_apply dinput-joy-mappings/0002-dinput-Allow-empty-Joystick-mappings.patch
	patch_apply dinput-joy-mappings/0003-dinput-Support-username-in-Config-dialog.patch
	patch_apply dinput-joy-mappings/0004-dinput-Dont-allow-Fixed-actions-to-be-changed.patch
fi

# Patchset dinput-SetActionMap-genre
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	dinput-joy-mappings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47326] dinput: Allow mapping of controls based of genre type.
# |
# | Modified files:
# |   *	dlls/dinput/device.c, dlls/dinput/dinput_main.c
# |
if test "$enable_dinput_SetActionMap_genre" -eq 1; then
	patch_apply dinput-SetActionMap-genre/0001-dinput-Allow-mapping-of-controls-based-of-Genre-type.patch
	patch_apply dinput-SetActionMap-genre/0002-dinput-Improved-tracing-of-Semantic-value.patch
fi

# Patchset dinput-axis-recalc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41317] dinput: Recalculated Axis after deadzone change.
# |
# | Modified files:
# |   *	dlls/dinput/joystick.c
# |
if test "$enable_dinput_axis_recalc" -eq 1; then
	patch_apply dinput-axis-recalc/0001-dinput-Recalculated-Axis-after-deadzone-change.patch
fi

# Patchset dinput-reconnect-joystick
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34297] dinput: Allow reconnecting to disconnected joysticks
# |
# | Modified files:
# |   *	dlls/dinput/joystick_linuxinput.c
# |
if test "$enable_dinput_reconnect_joystick" -eq 1; then
	patch_apply dinput-reconnect-joystick/0001-dinput-Allow-reconnecting-to-disconnected-joysticks.patch
fi

# Patchset dinput-remap-joystick
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35815] dinput: Allow remapping of joystick buttons.
# |
# | Modified files:
# |   *	dlls/dinput/joystick.c, dlls/dinput/joystick_linux.c, dlls/dinput/joystick_linuxinput.c, dlls/dinput/joystick_osx.c,
# | 	dlls/dinput/joystick_private.h
# |
if test "$enable_dinput_remap_joystick" -eq 1; then
	patch_apply dinput-remap-joystick/0001-dinput-Allow-remapping-of-joystick-buttons.patch
fi

# Patchset dsound-Fast_Mixer
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30639] Audio stuttering and performance drops in multiple applications
# |
# | Modified files:
# |   *	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/mixer.c
# |
if test "$enable_dsound_Fast_Mixer" -eq 1; then
	patch_apply dsound-Fast_Mixer/0001-dsound-Add-a-linear-resampler-for-use-with-a-large-n.patch
fi

# Patchset dsound-EAX
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	dsound-Fast_Mixer
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42886] Mushroom Wars - Has no sounds.
# |
# | Modified files:
# |   *	dlls/dsound/Makefile.in, dlls/dsound/buffer.c, dlls/dsound/dsound.c, dlls/dsound/dsound_eax.h,
# | 	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/eax.c, dlls/dsound/mixer.c
# |
if test "$enable_dsound_EAX" -eq 1; then
	patch_apply dsound-EAX/0001-dsound-Apply-filters-before-sound-is-multiplied-to-s.patch
	patch_apply dsound-EAX/0002-dsound-Add-EAX-v1-constants-and-structs.patch
	patch_apply dsound-EAX/0003-dsound-Report-that-we-support-EAX-v1.patch
	patch_apply dsound-EAX/0004-dsound-Add-EAX-propset-stubs.patch
	patch_apply dsound-EAX/0005-dsound-Add-EAX-presets.patch
	patch_apply dsound-EAX/0006-dsound-Support-getting-and-setting-EAX-properties.patch
	patch_apply dsound-EAX/0007-dsound-Support-getting-and-setting-EAX-buffer-proper.patch
	patch_apply dsound-EAX/0008-dsound-Add-EAX-init-and-free-stubs.patch
	patch_apply dsound-EAX/0009-dsound-Feed-data-through-EAX-function.patch
	patch_apply dsound-EAX/0010-dsound-Allocate-EAX-delay-lines.patch
	patch_apply dsound-EAX/0011-dsound-Add-EAX-VerbPass-stub.patch
	patch_apply dsound-EAX/0012-dsound-Implement-EAX-lowpass-filter.patch
	patch_apply dsound-EAX/0013-dsound-Add-delay-line-EAX-functions.patch
	patch_apply dsound-EAX/0014-dsound-Implement-EAX-early-reflections.patch
	patch_apply dsound-EAX/0015-dsound-Implement-EAX-decorrelator.patch
	patch_apply dsound-EAX/0016-dsound-Implement-EAX-late-reverb.patch
	patch_apply dsound-EAX/0017-dsound-Implement-EAX-late-all-pass-filter.patch
	patch_apply dsound-EAX/0018-dsound-Various-improvements-to-EAX-support.patch
	patch_apply dsound-EAX/0019-dsound-Allow-disabling-of-EAX-support-in-the-registr.patch
	patch_apply dsound-EAX/0020-dsound-Add-stub-support-for-DSPROPSETID_EAX20_Listen.patch
	patch_apply dsound-EAX/0021-dsound-Add-stub-support-for-DSPROPSETID_EAX20_Buffer.patch
	patch_apply dsound-EAX/0022-dsound-Enable-EAX-by-default.patch
fi

# Patchset dwrite-FontFallback
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44052] - Support for font fallback.
# |
# | Modified files:
# |   *	dlls/dwrite/analyzer.c, dlls/dwrite/layout.c, dlls/dwrite/tests/layout.c
# |
if test "$enable_dwrite_FontFallback" -eq 1; then
	patch_apply dwrite-FontFallback/0001-dwrite-Test-IDWriteTextFormat-with-nonexistent-font.patch
	patch_apply dwrite-FontFallback/0002-dwrite-Test-GetMetrics-with-custom-fontcollection.patch
	patch_apply dwrite-FontFallback/0004-dwrite-Use-font-fallback-when-mapping-characters.patch
	patch_apply dwrite-FontFallback/0005-dwrite-Use-MapCharacters-for-non-visual-characters.patch
	patch_apply dwrite-FontFallback/0006-dwrite-Use-MapCharacters-for-dummy-line-metrics.patch
fi

# Patchset ntoskrnl-Stubs
# |
# | Modified files:
# |   *	dlls/ntoskrnl.exe/ntoskrnl.c, dlls/ntoskrnl.exe/ntoskrnl.exe.spec
# |
if test "$enable_ntoskrnl_Stubs" -eq 1; then
	patch_apply ntoskrnl-Stubs/0009-ntoskrnl.exe-Implement-MmMapLockedPages-and-MmUnmapL.patch
	patch_apply ntoskrnl-Stubs/0011-ntoskrnl.exe-Add-IoGetDeviceAttachmentBaseRef-stub.patch
fi

# Patchset winedevice-Default_Drivers
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntoskrnl-Stubs
# |
# | Modified files:
# |   *	configure.ac, dlls/dxgkrnl.sys/Makefile.in, dlls/dxgkrnl.sys/dxgkrnl.sys.spec, dlls/dxgkrnl.sys/main.c,
# | 	dlls/dxgmms1.sys/Makefile.in, dlls/dxgmms1.sys/dxgmms1.sys.spec, dlls/dxgmms1.sys/main.c,
# | 	dlls/ntoskrnl.exe/tests/driver.c, dlls/win32k.sys/Makefile.in, dlls/win32k.sys/main.c, dlls/win32k.sys/win32k.sys.spec,
# | 	loader/wine.inf.in, programs/winedevice/device.c, tools/make_specfiles
# |
if test "$enable_winedevice_Default_Drivers" -eq 1; then
	patch_apply winedevice-Default_Drivers/0001-win32k.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0002-dxgkrnl.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0003-dxgmms1.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0004-programs-winedevice-Load-some-common-drivers-and-fix.patch
fi

# Patchset dxdiag-new-dlls
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntoskrnl-Stubs, winedevice-Default_Drivers
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50293] : native dxdiag complains about missing
# |
# | Modified files:
# |   *	configure.ac, dlls/d3dpmesh/Makefile.in, dlls/d3dpmesh/d3dpmesh.spec, dlls/d3dpmesh/d3dpmesh_main.c,
# | 	dlls/d3dpmesh/version.rc, dlls/diactfrm/Makefile.in, dlls/diactfrm/diactfrm.spec, dlls/diactfrm/diactfrm_main.c,
# | 	dlls/diactfrm/version.rc, dlls/dimap/Makefile.in, dlls/dimap/dimap.spec, dlls/dimap/dimap_main.c, dlls/dimap/version.rc,
# | 	dlls/dpmodemx/Makefile.in, dlls/dpmodemx/dpmodemx.spec, dlls/dpmodemx/dpmodemx_main.c, dlls/dpmodemx/version.rc,
# | 	dlls/dpnhupnp/Makefile.in, dlls/dpnhupnp/dpnhupnp.spec, dlls/dpnhupnp/dpnhupnp_main.c, dlls/dpnhupnp/version.rc,
# | 	dlls/dpvacm/Makefile.in, dlls/dpvacm/dpvacm.spec, dlls/dpvacm/dpvacm_main.c, dlls/dpvacm/version.rc,
# | 	dlls/dpvvox/Makefile.in, dlls/dpvvox/dpvvox.spec, dlls/dpvvox/dpvvox_main.c, dlls/dpvvox/version.rc,
# | 	dlls/dsdmoprp/Makefile.in, dlls/dsdmoprp/dsdmoprp.spec, dlls/dsdmoprp/dsdmoprp_main.c, dlls/dsdmoprp/version.rc,
# | 	dlls/dsound3d/Makefile.in, dlls/dsound3d/dsound3d.spec, dlls/dsound3d/dsound3d_main.c, dlls/dsound3d/version.rc,
# | 	dlls/dx7vb/Makefile.in, dlls/dx7vb/dx7vb.spec, dlls/dx7vb/dx7vb_main.c, dlls/dx7vb/version.rc,
# | 	dlls/dxapi.sys/Makefile.in, dlls/dxapi.sys/dxapi.sys.spec, dlls/dxapi.sys/main.c, dlls/dxapi.sys/version.rc,
# | 	dlls/encapi/Makefile.in, dlls/encapi/encapi.spec, dlls/encapi/encapi_main.c, dlls/encapi/version.rc,
# | 	dlls/gcdef/Makefile.in, dlls/gcdef/gcdef.spec, dlls/gcdef/gcdef_main.c, dlls/gcdef/version.rc, dlls/qdv/Makefile.in,
# | 	dlls/qdv/qdv.spec, dlls/qdv/qdv_main.c, dlls/qdv/version.rc, dlls/qedwipes/Makefile.in, dlls/qedwipes/qedwipes.spec,
# | 	dlls/qedwipes/qedwipes_main.c, dlls/qedwipes/version.rc, loader/wine.inf.in
# |
if test "$enable_dxdiag_new_dlls" -eq 1; then
	patch_apply dxdiag-new-dlls/0001-d3dpmesh-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0002-diactfrm-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0003-dimap-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0004-dpmodemx-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0005-dpnhupnp-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0006-dpvacm-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0007-dpvvox-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0008-dsdmoprp-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0009-dsound3d-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0010-dxapi.sys-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0011-dx7vb-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0012-encapi-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0013-gcdef-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0014-qdv-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0015-qedwipes-add-stub-dll.patch
	patch_apply dxdiag-new-dlls/0016-dpmodemx-Add-stub-SPInit.patch
fi

# Patchset ntdll-DOS_Attributes
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#9158] Support for DOS hidden/system file attributes
# |   *	[#15679] cygwin symlinks not working in wine
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/tests/directory.c, dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_DOS_Attributes" -eq 1; then
	patch_apply ntdll-DOS_Attributes/0001-ntdll-Implement-retrieving-DOS-attributes-in-fd_-get.patch
	patch_apply ntdll-DOS_Attributes/0003-ntdll-Implement-storing-DOS-attributes-in-NtSetInfor.patch
	patch_apply ntdll-DOS_Attributes/0004-ntdll-Implement-storing-DOS-attributes-in-NtCreateFi.patch
	patch_apply ntdll-DOS_Attributes/0005-libport-Add-support-for-Mac-OS-X-style-extended-attr.patch
	patch_apply ntdll-DOS_Attributes/0006-libport-Add-support-for-FreeBSD-style-extended-attri.patch
	patch_apply ntdll-DOS_Attributes/0007-ntdll-Perform-the-Unix-style-hidden-file-check-withi.patch
	patch_apply ntdll-DOS_Attributes/0008-ntdll-Always-store-SAMBA_XATTR_DOS_ATTRIB-when-path-.patch
fi

# Patchset ntdll-NtQueryEaFile
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_NtQueryEaFile" -eq 1; then
	patch_apply ntdll-NtQueryEaFile/0001-ntdll-Improve-stub-of-NtQueryEaFile.patch
fi

# Patchset ntdll-Junction_Points
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12401] NET Framework 2.0, 3.0, 4.0 installers and other apps that make use of GAC API for managed assembly
# | 	installation on NTFS filesystems need reparse point/junction API support
# | 	(FSCTL_SET_REPARSE_POINT/FSCTL_GET_REPARSE_POINT)
# |   *	[#44948] Multiple apps (Spine (Mod starter for Gothic), MS Office 365 installer) need CreateSymbolicLinkW implementation
# |
# | Modified files:
# |   *	configure.ac, dlls/kernel32/path.c, dlls/kernel32/tests/path.c, dlls/kernelbase/file.c, dlls/msvcp120/tests/msvcp120.c,
# | 	dlls/msvcp140/tests/msvcp140.c, dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c, include/Makefile.in, include/ntifs.h,
# | 	include/winternl.h, programs/cmd/builtins.c, programs/cmd/directory.c, server/fd.c, server/file.c, server/protocol.def
# |
if test "$enable_ntdll_Junction_Points" -eq 1; then
	patch_apply ntdll-Junction_Points/0001-ntdll-Add-support-for-junction-point-creation.patch
	patch_apply ntdll-Junction_Points/0002-ntdll-Add-support-for-reading-junction-points.patch
	patch_apply ntdll-Junction_Points/0003-ntdll-Add-support-for-deleting-junction-points.patch
	patch_apply ntdll-Junction_Points/0004-ntdll-Add-a-test-for-junction-point-advertisement.patch
	patch_apply ntdll-Junction_Points/0005-server-Add-support-for-deleting-junction-points-with.patch
	patch_apply ntdll-Junction_Points/0007-ntdll-Add-support-for-absolute-symlink-creation.patch
	patch_apply ntdll-Junction_Points/0008-ntdll-Add-support-for-reading-absolute-symlinks.patch
	patch_apply ntdll-Junction_Points/0009-ntdll-Add-support-for-deleting-symlinks.patch
	patch_apply ntdll-Junction_Points/0010-ntdll-Add-support-for-relative-symlink-creation.patch
	patch_apply ntdll-Junction_Points/0011-ntdll-Add-support-for-reading-relative-symlinks.patch
	patch_apply ntdll-Junction_Points/0012-ntdll-Add-support-for-file-symlinks.patch
	patch_apply ntdll-Junction_Points/0013-ntdll-Allow-creation-of-dangling-reparse-points-to-n.patch
	patch_apply ntdll-Junction_Points/0014-ntdll-Correctly-report-file-symbolic-links-as-files.patch
	patch_apply ntdll-Junction_Points/0015-kernel32-Set-error-code-when-attempting-to-delete-fi.patch
	patch_apply ntdll-Junction_Points/0016-server-Properly-handle-file-symlink-deletion.patch
	patch_apply ntdll-Junction_Points/0017-ntdll-Always-report-symbolic-links-as-containing-zer.patch
	patch_apply ntdll-Junction_Points/0018-ntdll-Find-dangling-symlinks-quickly.patch
	patch_apply ntdll-Junction_Points/0019-kernel32-Implement-CreateSymbolicLink-A-W-with-ntdll.patch
	patch_apply ntdll-Junction_Points/0020-kernel32-Add-reparse-support-to-FindNextFile.patch
	patch_apply ntdll-Junction_Points/0021-wcmd-Display-reparse-point-type-in-directory-listing.patch
	patch_apply ntdll-Junction_Points/0022-wcmd-Show-reparse-point-target-in-directory-listing.patch
	patch_apply ntdll-Junction_Points/0023-wcmd-Add-junction-point-support-to-mklink.patch
	patch_apply ntdll-Junction_Points/0024-server-Fix-obtaining-information-about-a-symlink.patch
	patch_apply ntdll-Junction_Points/0025-ntdll-Allow-set_file_times_precise-to-work-on-repars.patch
	patch_apply ntdll-Junction_Points/0026-kernelbase-Convert-FILE_FLAG_OPEN_REPARSE_POINT-for-.patch
	patch_apply ntdll-Junction_Points/0027-server-Implement-FILE_OPEN_REPARSE_POINT-option.patch
	patch_apply ntdll-Junction_Points/0028-ntdll-Succeed-with-no-data-for-NtReadFile-on-reparse.patch
	patch_apply ntdll-Junction_Points/0029-ntdll-Support-reparse-point-properties-in-fd_get_fil.patch
	patch_apply ntdll-Junction_Points/0030-ntdll-Add-support-for-FileAttributeTagInformation.patch
fi

# Patchset server-PeekMessage
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28884] GetMessage should remove already seen messages with higher priority
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, server/queue.c
# |
if test "$enable_server_PeekMessage" -eq 1; then
	patch_apply server-PeekMessage/0001-server-Fix-handling-of-GetMessage-after-previous-Pee.patch
fi

# Patchset server-Realtime_Priority
# |
# | Modified files:
# |   *	server/Makefile.in, server/main.c, server/scheduler.c, server/thread.c, server/thread.h
# |
if test "$enable_server_Realtime_Priority" -eq 1; then
	patch_apply server-Realtime_Priority/0001-wineserver-Draft-to-implement-priority-levels-throug.patch
fi

# Patchset server-Signal_Thread
# |
# | Modified files:
# |   *	server/thread.c, server/thread.h
# |
if test "$enable_server_Signal_Thread" -eq 1; then
	patch_apply server-Signal_Thread/0001-server-Do-not-signal-thread-until-it-is-really-gone.patch
fi

# Patchset eventfd_synchronization
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, server-PeekMessage, server-Realtime_Priority, server-
# | 	Signal_Thread
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36692] Many multi-threaded applications have poor performance due to heavy use of synchronization primitives
# |
# | Modified files:
# |   *	README.esync, configure, configure.ac, dlls/kernel32/tests/sync.c, dlls/ntdll/Makefile.in, dlls/ntdll/unix/esync.c,
# | 	dlls/ntdll/unix/esync.h, dlls/ntdll/unix/loader.c, dlls/ntdll/unix/server.c, dlls/ntdll/unix/sync.c,
# | 	dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c, dlls/rpcrt4/rpc_server.c, include/config.h.in,
# | 	server/Makefile.in, server/async.c, server/atom.c, server/change.c, server/clipboard.c, server/completion.c,
# | 	server/console.c, server/debugger.c, server/device.c, server/directory.c, server/esync.c, server/esync.h,
# | 	server/event.c, server/fd.c, server/file.c, server/file.h, server/handle.c, server/hook.c, server/mailslot.c,
# | 	server/main.c, server/mapping.c, server/mutex.c, server/named_pipe.c, server/object.h, server/process.c,
# | 	server/process.h, server/protocol.def, server/queue.c, server/registry.c, server/request.c, server/semaphore.c,
# | 	server/serial.c, server/signal.c, server/sock.c, server/symlink.c, server/thread.c, server/thread.h, server/timer.c,
# | 	server/token.c, server/winstation.c
# |
if test "$enable_eventfd_synchronization" -eq 1; then
	patch_apply eventfd_synchronization/0001-configure-Check-for-sys-eventfd.h-ppoll-and-shm_open.patch
	patch_apply eventfd_synchronization/0002-server-Create-server-objects-for-eventfd-based-synch.patch
	patch_apply eventfd_synchronization/0003-ntdll-Create-eventfd-based-objects-for-semaphores.patch
	patch_apply eventfd_synchronization/0004-ntdll-Implement-NtReleaseSemaphore.patch
	patch_apply eventfd_synchronization/0005-ntdll-Implement-NtClose.patch
	patch_apply eventfd_synchronization/0006-ntdll-Implement-NtWaitForMultipleObjects.patch
	patch_apply eventfd_synchronization/0007-ntdll-server-Implement-NtCreateEvent.patch
	patch_apply eventfd_synchronization/0008-ntdll-Implement-NtSetEvent.patch
	patch_apply eventfd_synchronization/0009-ntdll-Implement-NtResetEvent.patch
	patch_apply eventfd_synchronization/0010-ntdll-Implement-waiting-on-manual-reset-events.patch
	patch_apply eventfd_synchronization/0011-server-Add-an-object-operation-to-grab-the-esync-fil.patch
	patch_apply eventfd_synchronization/0012-server-Add-a-request-to-get-the-eventfd-file-descrip.patch
	patch_apply eventfd_synchronization/0013-server-Create-eventfd-file-descriptors-for-process-o.patch
	patch_apply eventfd_synchronization/0014-ntdll-server-Implement-waiting-on-server-bound-objec.patch
	patch_apply eventfd_synchronization/0015-server-Create-eventfd-file-descriptors-for-event-obj.patch
	patch_apply eventfd_synchronization/0016-server-Allow-re-setting-esync-events-on-the-server-s.patch
	patch_apply eventfd_synchronization/0017-ntdll-Try-again-if-poll-returns-EINTR.patch
	patch_apply eventfd_synchronization/0018-server-Create-eventfd-file-descriptors-for-thread-ob.patch
	patch_apply eventfd_synchronization/0019-rpcrt4-Avoid-closing-the-server-thread-handle-while-.patch
	patch_apply eventfd_synchronization/0020-server-Create-eventfd-file-descriptors-for-message-q.patch
	patch_apply eventfd_synchronization/0021-server-ntdll-Implement-message-waits.patch
	patch_apply eventfd_synchronization/0022-server-Create-eventfd-descriptors-for-device-manager.patch
	patch_apply eventfd_synchronization/0023-ntdll-server-Implement-NtCreateMutant.patch
	patch_apply eventfd_synchronization/0024-ntdll-Implement-NtReleaseMutant.patch
	patch_apply eventfd_synchronization/0025-ntdll-Implement-waiting-on-mutexes.patch
	patch_apply eventfd_synchronization/0026-ntdll-Implement-wait-all.patch
	patch_apply eventfd_synchronization/0027-esync-Add-a-README.patch
	patch_apply eventfd_synchronization/0028-ntdll-Implement-NtSignalAndWaitForSingleObject.patch
	patch_apply eventfd_synchronization/0029-ntdll-Implement-NtOpenSemaphore.patch
	patch_apply eventfd_synchronization/0030-ntdll-Implement-NtOpenEvent.patch
	patch_apply eventfd_synchronization/0031-ntdll-Implement-NtOpenMutant.patch
	patch_apply eventfd_synchronization/0032-server-Implement-esync_map_access.patch
	patch_apply eventfd_synchronization/0033-server-Implement-NtDuplicateObject.patch
	patch_apply eventfd_synchronization/0034-server-Create-eventfd-descriptors-for-timers.patch
	patch_apply eventfd_synchronization/0035-ntdll-server-Implement-alertable-waits.patch
	patch_apply eventfd_synchronization/0036-esync-Update-README.patch
	patch_apply eventfd_synchronization/0037-kernel32-tests-Mark-some-existing-tests-as-failing-u.patch
	patch_apply eventfd_synchronization/0038-kernel32-tests-Add-some-semaphore-tests.patch
	patch_apply eventfd_synchronization/0039-kernel32-tests-Add-some-event-tests.patch
	patch_apply eventfd_synchronization/0040-kernel32-tests-Add-some-mutex-tests.patch
	patch_apply eventfd_synchronization/0041-kernel32-tests-Add-some-tests-for-wait-timeouts.patch
	patch_apply eventfd_synchronization/0042-kernel32-tests-Zigzag-test.patch
	patch_apply eventfd_synchronization/0043-ntdll-Implement-NtQuerySemaphore.patch
	patch_apply eventfd_synchronization/0044-ntdll-Implement-NtQueryEvent.patch
	patch_apply eventfd_synchronization/0045-ntdll-Implement-NtQueryMutant.patch
	patch_apply eventfd_synchronization/0046-server-Create-eventfd-descriptors-for-pseudo-fd-obje.patch
	patch_apply eventfd_synchronization/0047-esync-Update-README.patch
	patch_apply eventfd_synchronization/0048-esync-Add-note-about-file-limits-not-being-raised-wh.patch
	patch_apply eventfd_synchronization/0049-ntdll-Try-to-avoid-poll-for-uncontended-objects.patch
	patch_apply eventfd_synchronization/0050-ntdll-server-Try-to-avoid-poll-for-signaled-events.patch
	patch_apply eventfd_synchronization/0051-esync-Update-README.patch
	patch_apply eventfd_synchronization/0052-ntdll-Implement-NtPulseEvent.patch
	patch_apply eventfd_synchronization/0053-esync-Update-README.patch
	patch_apply eventfd_synchronization/0054-server-Create-esync-file-descriptors-for-true-file-o.patch
	patch_apply eventfd_synchronization/0055-ntdll-server-Abandon-esync-mutexes-on-thread-exit.patch
	patch_apply eventfd_synchronization/0056-server-Create-esync-file-descriptors-for-console-ser.patch
fi

# Patchset explorer-Video_Registry_Key
# |
# | Modified files:
# |   *	dlls/advapi32/tests/registry.c, programs/explorer/desktop.c
# |
if test "$enable_explorer_Video_Registry_Key" -eq 1; then
	patch_apply explorer-Video_Registry_Key/0001-explorer-Create-CurrentControlSet-Control-Video-regi.patch
fi

# Patchset fonts-Missing_Fonts
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32323] Implement an Arial replacement font
# |   *	[#32342] Implement a Times New Roman replacement font
# |   *	[#20456] Implement a Courier New replacement font
# |   *	[#13829] Implement a Microsoft Yahei replacement font
# |
# | Modified files:
# |   *	COPYING.arial, COPYING.cour, COPYING.msyh, COPYING.times, LICENSE, fonts/Makefile.in, fonts/arial.sfd, fonts/arial.ttf,
# | 	fonts/cour.sfd, fonts/cour.ttf, fonts/msyh.sfd, fonts/msyh.ttf, fonts/times.sfd, fonts/times.ttf
# |
if test "$enable_fonts_Missing_Fonts" -eq 1; then
	patch_apply fonts-Missing_Fonts/0001-fonts-Add-Liberation-Sans-as-an-Arial-replacement.patch
	patch_apply fonts-Missing_Fonts/0002-fonts-Add-Liberation-Serif-as-an-Times-New-Roman-rep.patch
	patch_apply fonts-Missing_Fonts/0003-fonts-Add-Liberation-Mono-as-an-Courier-New-replacem.patch
	patch_apply fonts-Missing_Fonts/0004-fonts-Add-WenQuanYi-Micro-Hei-as-a-Microsoft-Yahei-r.patch
	patch_apply fonts-Missing_Fonts/0005-Add-licenses-for-fonts-as-separate-files.patch
fi

# Patchset gdi32-rotation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34579] gdi32: fix for rotated Arc, ArcTo, Chord and Pie drawing problem
# |   *	[#35331] gdi32: fix for rotated ellipse
# |
# | Modified files:
# |   *	dlls/gdi32/dibdrv/graphics.c, dlls/gdi32/gdi_private.h
# |
if test "$enable_gdi32_rotation" -eq 1; then
	patch_apply gdi32-rotation/0001-gdi32-fix-for-rotated-Arc-ArcTo-Chord-and-Pie-drawin.patch
	patch_apply gdi32-rotation/0002-gdi32-fix-for-rotated-ellipse.patch
fi

# Patchset gdiplus-Performance-Improvements
# |
# | Modified files:
# |   *	dlls/gdiplus/graphics.c
# |
if test "$enable_gdiplus_Performance_Improvements" -eq 1; then
	patch_apply gdiplus-Performance-Improvements/0001-gdiplus-Change-the-order-of-x-y-loops-in-the-scaler.patch
	patch_apply gdiplus-Performance-Improvements/0002-gdiplus-Change-multiplications-by-additions-in-the-x.patch
	patch_apply gdiplus-Performance-Improvements/0003-gdiplus-Remove-ceilf-floorf-calls-from-bilinear-scal.patch
	patch_apply gdiplus-Performance-Improvements/0004-gdiplus-Prefer-using-pre-multiplied-ARGB-data-in-the.patch
fi

# Patchset imagehlp-BindImageEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#3591] Support for BindImageEx
# |
# | Modified files:
# |   *	dlls/imagehlp/modify.c, dlls/imagehlp/tests/image.c
# |
if test "$enable_imagehlp_BindImageEx" -eq 1; then
	patch_apply imagehlp-BindImageEx/0001-imagehlp-Implement-parts-of-BindImageEx-to-make-free.patch
fi

# Patchset imm32-message_on_focus
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31157] imm32: Only generate 'WM_IME_SETCONTEXT' message if window has focus.
# |
# | Modified files:
# |   *	dlls/imm32/imm.c
# |
if test "$enable_imm32_message_on_focus" -eq 1; then
	patch_apply imm32-message_on_focus/0001-imm32-Only-generate-WM_IME_SETCONTEXT-message-if-win.patch
fi

# Patchset include-winsock
# |
# | Modified files:
# |   *	include/winsock.h
# |
if test "$enable_include_winsock" -eq 1; then
	patch_apply include-winsock/0001-include-Always-define-hton-ntoh-macros.patch
fi

# Patchset inseng-Implementation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39456] Implement CIF reader and download functionality in inseng.dll
# |
# | Modified files:
# |   *	dlls/inseng/Makefile.in, dlls/inseng/icif.c, dlls/inseng/inf.c, dlls/inseng/inseng.spec, dlls/inseng/inseng_main.c,
# | 	dlls/inseng/inseng_private.h, include/inseng.idl
# |
if test "$enable_inseng_Implementation" -eq 1; then
	patch_apply inseng-Implementation/0001-inseng-Implement-CIF-reader-and-download-functions.patch
fi

# Patchset iphlpapi-System_Ping
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8332] Fallback to system ping command when CAP_NET_RAW is not available
# |
# | Modified files:
# |   *	dlls/iphlpapi/icmp.c
# |
if test "$enable_iphlpapi_System_Ping" -eq 1; then
	patch_apply iphlpapi-System_Ping/0001-iphlpapi-Fallback-to-system-ping-when-ICMP-permissio.patch
fi

# Patchset ntdll-FileDispositionInformation
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, server/fd.c
# |
if test "$enable_ntdll_FileDispositionInformation" -eq 1; then
	patch_apply ntdll-FileDispositionInformation/0001-ntdll-tests-Added-tests-to-set-disposition-on-file-w.patch
	patch_apply ntdll-FileDispositionInformation/0002-server-Do-not-allow-to-set-disposition-on-file-which.patch
fi

# Patchset kernel32-CopyFileEx
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-FileDispositionInformation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22692] Add support for CopyFileEx progress callback
# |   *	[#22690] Allow to cancel a file operation via progress callback
# |
# | Modified files:
# |   *	dlls/kernel32/tests/file.c, dlls/kernelbase/file.c
# |
if test "$enable_kernel32_CopyFileEx" -eq 1; then
	patch_apply kernel32-CopyFileEx/0001-kernel32-Add-support-for-progress-callback-in-CopyFi.patch
fi

# Patchset kernel32-Debugger
# |
# | Modified files:
# |   *	dlls/kernelbase/debug.c
# |
if test "$enable_kernel32_Debugger" -eq 1; then
	patch_apply kernel32-Debugger/0001-kernel32-Always-start-debugger-on-WinSta0.patch
fi

# Patchset kernel32-FindFirstFile
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22635] Strip invalid characters from mask in FindFirstFileExW
# |
# | Modified files:
# |   *	dlls/kernel32/tests/file.c, dlls/kernelbase/file.c
# |
if test "$enable_kernel32_FindFirstFile" -eq 1; then
	patch_apply kernel32-FindFirstFile/0001-kernel32-Strip-invalid-characters-from-mask-in-FindF.patch
	patch_apply kernel32-FindFirstFile/0002-kernel32-tests-Add-tests-for-FindFirstFileA-with-inv.patch
fi

# Patchset kernel32-Job_Tests
# |
# | Modified files:
# |   *	dlls/kernel32/tests/process.c
# |
if test "$enable_kernel32_Job_Tests" -eq 1; then
	patch_apply kernel32-Job_Tests/0001-kernel32-tests-Add-tests-for-job-object-accounting.patch
fi

# Patchset kernel32-Processor_Group
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	api-ms-win-Stub_DLLs
# |
# | Modified files:
# |   *	dlls/api-ms-win-core-kernel32-legacy-l1-1-0/api-ms-win-core-kernel32-legacy-l1-1-0.spec, dlls/api-ms-win-core-
# | 	kernel32-legacy-l1-1-1/api-ms-win-core-kernel32-legacy-l1-1-1.spec, dlls/kernel32/kernel32.spec,
# | 	dlls/kernel32/process.c, dlls/kernel32/tests/process.c, dlls/kernelbase/thread.c
# |
if test "$enable_kernel32_Processor_Group" -eq 1; then
	patch_apply kernel32-Processor_Group/0001-kernel32-Implement-some-processor-group-functions.patch
	patch_apply kernel32-Processor_Group/0002-kernel32-Add-stub-for-SetThreadIdealProcessorEx.patch
fi

# Patchset kernel32-SetProcessDEPPolicy
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24125] kernel32: Implement SetProcessDEPPolicy.
# |
# | Modified files:
# |   *	dlls/kernel32/process.c
# |
if test "$enable_kernel32_SetProcessDEPPolicy" -eq 1; then
	patch_apply kernel32-SetProcessDEPPolicy/0001-kernel32-Implement-SetProcessDEPPolicy.patch
	patch_apply kernel32-SetProcessDEPPolicy/0002-kernel32-Implement-GetSystemDEPPolicy.patch
	patch_apply kernel32-SetProcessDEPPolicy/0003-kernel32-Make-system-DEP-policy-affect-GetProcessDEP.patch
fi

# Patchset krnl386.exe16-GDT_LDT_Emulation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30237] Implement emulation of GDT and LDT access in Win98 mode
# |
# | Modified files:
# |   *	dlls/krnl386.exe16/instr.c
# |
if test "$enable_krnl386_exe16_GDT_LDT_Emulation" -eq 1; then
	patch_apply krnl386.exe16-GDT_LDT_Emulation/0001-krnl386.exe16-Emulate-GDT-and-LDT-access.patch
fi

# Patchset krnl386.exe16-Invalid_Console_Handles
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7106] Translate all invalid console handles into usable DOS handles
# |
# | Modified files:
# |   *	dlls/krnl386.exe16/file.c
# |
if test "$enable_krnl386_exe16_Invalid_Console_Handles" -eq 1; then
	patch_apply krnl386.exe16-Invalid_Console_Handles/0001-krnl386.exe16-Really-translate-all-invalid-console-h.patch
fi

# Patchset libs-Unicode_Collation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#5163] Microsoft Office XP 2002 installer reports error 25003 (installation source corrupted), custom action 'CADpc'
# | 	returns 1603
# |   *	[#10767] Fix comparison of punctuation characters in lstrcmp
# |   *	[#32490] Graphical issues in Inquisitor
# |
# | Modified files:
# |   *	dlls/kernel32/tests/locale.c, dlls/kernelbase/locale.c
# |
if test "$enable_libs_Unicode_Collation" -eq 1; then
	patch_apply libs-Unicode_Collation/0001-kernelbase-Implement-sortkey-generation-on-official-.patch
	patch_apply libs-Unicode_Collation/0002-kernelbase-Implement-sortkey-punctuation.patch
	patch_apply libs-Unicode_Collation/0003-kernelbase-Implement-sortkey-for-Japanese-characters.patch
	patch_apply libs-Unicode_Collation/0004-kernelbase-Implement-sortkey-expansion.patch
	patch_apply libs-Unicode_Collation/0005-kernelbase-Implement-sortkey-language-support.patch
	patch_apply libs-Unicode_Collation/0006-kernelbase-Implement-CompareString-functions.patch
fi

# Patchset loader-KeyboardLayouts
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47439] loader: Add Keyboard Layouts registry enteries.
# |
# | Modified files:
# |   *	dlls/user32/driver.c, dlls/user32/tests/input.c, loader/wine.inf.in
# |
if test "$enable_loader_KeyboardLayouts" -eq 1; then
	patch_apply loader-KeyboardLayouts/0001-loader-Add-Keyboard-Layouts-registry-enteries.patch
	patch_apply loader-KeyboardLayouts/0002-user32-Improve-GetKeyboardLayoutList.patch
fi

# Patchset mfplat-streaming-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49692] Multiple applications need a Media Foundation media source implementation
# |
# | Modified files:
# |   *	dlls/mf/Makefile.in, dlls/mf/handler.c, dlls/mf/handler.h, dlls/mf/main.c, dlls/mf/session.c, dlls/mf/tests/mf.c,
# | 	dlls/mfplat/tests/mfplat.c, dlls/mfplat/tests/test.mp4, dlls/mfreadwrite/reader.c, dlls/mfreadwrite/tests/mfplat.c,
# | 	dlls/mfreadwrite/tests/resource.rc, dlls/mfreadwrite/tests/test.mp4, dlls/winegstreamer/Makefile.in,
# | 	dlls/winegstreamer/audioconvert.c, dlls/winegstreamer/colorconvert.c, dlls/winegstreamer/gst_cbs.c,
# | 	dlls/winegstreamer/gst_cbs.h, dlls/winegstreamer/gst_private.h, dlls/winegstreamer/media_source.c,
# | 	dlls/winegstreamer/mf_decode.c, dlls/winegstreamer/mfplat.c, dlls/winegstreamer/winegstreamer_classes.idl,
# | 	include/mfidl.idl, tools/make_makefiles, tools/makedep.c
# |
if test "$enable_mfplat_streaming_support" -eq 1; then
	patch_apply mfplat-streaming-support/0001-winegstreamer-Correct-mistaken-enum-value-in-Process.patch
	patch_apply mfplat-streaming-support/0003-winegstreamer-Implement-Process-Input-Output-for-aud.patch
	patch_apply mfplat-streaming-support/0004-winegstreamer-Implement-Get-Input-Output-StreamInfo-.patch
	patch_apply mfplat-streaming-support/0005-winegstreamer-Implement-Get-Attributes-functions-for.patch
	patch_apply mfplat-streaming-support/0007-winegstreamer-Introduce-color-conversion-transform.patch
	patch_apply mfplat-streaming-support/0008-winegstreamer-Register-the-color-conversion-transfor.patch
	patch_apply mfplat-streaming-support/0009-winegstreamer-Implement-GetInputAvailableType-for-co.patch
	patch_apply mfplat-streaming-support/0010-winegstreamer-Implement-SetInputType-for-color-conve.patch
	patch_apply mfplat-streaming-support/0011-winegstreamer-Implement-GetOutputAvailableType-for-c.patch
	patch_apply mfplat-streaming-support/0012-winegstreamer-Implement-SetOutputType-for-color-conv.patch
	patch_apply mfplat-streaming-support/0013-winegstreamer-Implement-Process-Input-Output-for-col.patch
	patch_apply mfplat-streaming-support/0014-winegstreamer-Implement-ProcessMessage-for-color-con.patch
	patch_apply mfplat-streaming-support/0015-winegstreamer-Implement-Get-Input-Output-StreamInfo-.patch
	patch_apply mfplat-streaming-support/0016-winegstreamer-Implement-Get-Attributes-functions-for.patch
	patch_apply mfplat-streaming-support/0017-winegstreamer-Implement-Get-Input-Output-CurrentType.patch
	patch_apply mfplat-streaming-support/0018-winegstreamer-Implement-IMFMediaSource-Stop.patch
	patch_apply mfplat-streaming-support/0019-winegstreamer-Set-MF_MT_ALL_SAMPLES_INDEPENDENT-attr.patch
	patch_apply mfplat-streaming-support/0020-mf-Add-invalid-connect-method-test.patch
	patch_apply mfplat-streaming-support/0021-Allow-for-compressed-types.patch
	patch_apply mfplat-streaming-support/0022-mf-session-Unconditionally-deliver-NULL-EOS-samples.patch
	patch_apply mfplat-streaming-support/0023-mf-session-Request-more-samples-when-a-transform-nee.patch
	patch_apply mfplat-streaming-support/0024-HACK-Flush-decoder-when-changing-times.patch
	patch_apply mfplat-streaming-support/0025-winegstreamer-Add-IMFSeekInfo-GetNearestKeyFrames-st.patch
	patch_apply mfplat-streaming-support/0026-winegstreamer-Fixup-raw-audio-caps-to-be-compatible-.patch
	patch_apply mfplat-streaming-support/0027-winegstreamer-Set-MF_PD_MIME_TYPE-on-source-s-presen.patch
	patch_apply mfplat-streaming-support/0028-winegstreamer-Insert-parser-into-pipeline-to-rectify.patch
	patch_apply mfplat-streaming-support/0029-winegstreamer-Translate-H.264-caps-to-attributes.patch
	patch_apply mfplat-streaming-support/0030-winegstreamer-Translate-WMV-caps-to-attributes.patch
	patch_apply mfplat-streaming-support/0031-winegstreamer-Translate-AAC-caps-to-attributes.patch
	patch_apply mfplat-streaming-support/0032-winegstreamer-Translate-MPEG-4-Section-2-caps-to-att.patch
	patch_apply mfplat-streaming-support/0033-winegstreamer-Translate-WMA-caps-to-attributes.patch
	patch_apply mfplat-streaming-support/0034-winegstreamer-Translate-H.264-attributes-to-caps.patch
	patch_apply mfplat-streaming-support/0035-winegstreamer-Translate-WMV-attributes-to-caps.patch
	patch_apply mfplat-streaming-support/0036-winegstreamer-Translate-AAC-attributes-to-caps.patch
	patch_apply mfplat-streaming-support/0037-winegstreamer-Translate-MPEG-4-Section-2-attributes-.patch
	patch_apply mfplat-streaming-support/0038-winegstreamer-Translate-WMA-attributes-to-caps.patch
	patch_apply mfplat-streaming-support/0039-tools-Add-support-for-multiple-parent-directories.patch
	patch_apply mfplat-streaming-support/0040-mf-Introduce-handler-helper.patch
	patch_apply mfplat-streaming-support/0041-winegstreamer-Implement-decoder-MFT-on-gstreamer.patch
	patch_apply mfplat-streaming-support/0042-mfreadwrite-Select-all-streams-when-creating-a-sourc.patch
	patch_apply mfplat-streaming-support/0043-Miscellaneous.patch
	patch_apply mfplat-streaming-support/0044-WMV.patch
	patch_apply mfplat-streaming-support/0045-Expose-PCM-output-type-on-AAC-decoder.patch
	patch_apply mfplat-streaming-support/0046-Improve-tests.patch
	patch_apply mfplat-streaming-support/0047-Revert-Improve-tests.patch
	patch_apply mfplat-streaming-support/0048-Report-streams-backwards-and-only-select-one-of-each.patch
	patch_apply mfplat-streaming-support/0049-winegstreamer-Introduce-MPEG-4-Section-2-video-decod.patch
	patch_apply mfplat-streaming-support/0050-winegstreamer-Introduce-WMA-audio-decoder.patch
	patch_apply mfplat-streaming-support/0051-winegstreamer-Implement-MF_SD_LANGUAGE.patch
	patch_apply mfplat-streaming-support/0052-winegstreamer-Force-audio-mpeg-to-decode-in-source.patch
	patch_apply mfplat-streaming-support/0060-winegstreamer-Support-eAVEncH264VProfile_Constrained.patch
fi

# Patchset mmsystem.dll16-MIDIHDR_Refcount
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40024] Fix multiple issues in mmsystem.dll16 when translating MIDI messages
# |
# | Modified files:
# |   *	dlls/mmsystem.dll16/message16.c
# |
if test "$enable_mmsystem_dll16_MIDIHDR_Refcount" -eq 1; then
	patch_apply mmsystem.dll16-MIDIHDR_Refcount/0001-mmsystem.dll16-Refcount-midihdr-to-work-around-buggy.patch
	patch_apply mmsystem.dll16-MIDIHDR_Refcount/0002-mmsystem.dll16-Translate-MidiIn-messages.patch
fi

# Patchset mountmgr-DosDevices
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38235] Fix device paths in HKLM\SYSTEM\MountedDevices
# |
# | Modified files:
# |   *	dlls/mountmgr.sys/device.c, dlls/mountmgr.sys/mountmgr.c, dlls/mountmgr.sys/mountmgr.h
# |
if test "$enable_mountmgr_DosDevices" -eq 1; then
	patch_apply mountmgr-DosDevices/0001-mountmgr.sys-Write-usable-device-paths-into-HKLM-SYS.patch
fi

# Patchset mscoree-CorValidateImage
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38662] Implement mscoree._CorValidateImage for mono runtime
# |
# | Modified files:
# |   *	dlls/mscoree/mscoree_main.c
# |
if test "$enable_mscoree_CorValidateImage" -eq 1; then
	patch_apply mscoree-CorValidateImage/0001-mscoree-Implement-_CorValidateImage.patch
fi

# Patchset mshtml-HTMLLocation_put_hash
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32967] Add IHTMLLocation::hash property's getter implementation
# |
# | Modified files:
# |   *	dlls/mshtml/htmllocation.c, dlls/mshtml/tests/htmldoc.c
# |
if test "$enable_mshtml_HTMLLocation_put_hash" -eq 1; then
	patch_apply mshtml-HTMLLocation_put_hash/0001-mshtml-Add-IHTMLLocation-hash-property-s-getter-impl.patch
fi

# Patchset mshtml-TranslateAccelerator
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37058] mshtml: Improve IOleInPlaceActiveObject TranslateAccelerator
# |
# | Modified files:
# |   *	dlls/mshtml/oleobj.c
# |
if test "$enable_mshtml_TranslateAccelerator" -eq 1; then
	patch_apply mshtml-TranslateAccelerator/0001-mshtml-Improve-IOleInPlaceActiveObject-TranslateAcce.patch
fi

# Patchset msi-msi_vcl_get_cost
# |
# | Modified files:
# |   *	dlls/msi/dialog.c
# |
if test "$enable_msi_msi_vcl_get_cost" -eq 1; then
	patch_apply msi-msi_vcl_get_cost/0001-msi-Do-not-sign-extend-after-multiplying.patch
fi

# Patchset msvcrt-Math_Precision
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37149] Calculate msvcrt exponential math operations with higher precision
# |
# | Modified files:
# |   *	dlls/msvcrt/unixlib.c
# |
if test "$enable_msvcrt_Math_Precision" -eq 1; then
	patch_apply msvcrt-Math_Precision/0001-msvcrt-Calculate-sinh-cosh-exp-pow-with-higher-preci.patch
fi

# Patchset msxml3-FreeThreadedXMLHTTP60
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49740] msxml3: Implement FreeThreadedXMLHTTP60.
# |
# | Modified files:
# |   *	dlls/msxml3/Makefile.in, dlls/msxml3/factory.c, dlls/msxml3/httprequest.c, dlls/msxml3/msxml_private.h,
# | 	dlls/msxml3/tests/httpreq.c, dlls/msxml3/tests/saxreader.c, dlls/msxml3/tests/schema.c, dlls/msxml3/uuid.c,
# | 	include/msxml2.idl, include/msxml6.idl
# |
if test "$enable_msxml3_FreeThreadedXMLHTTP60" -eq 1; then
	patch_apply msxml3-FreeThreadedXMLHTTP60/0001-include-Remove-interfaces-already-define-in-msxml6.i.patch
	patch_apply msxml3-FreeThreadedXMLHTTP60/0002-include-Add-IXMLHTTPRequest2-3-interfaces.patch
	patch_apply msxml3-FreeThreadedXMLHTTP60/0003-msxml3-Implement-FreeThreadedXMLHTTP60.patch
fi

# Patchset ntdll-APC_Performance
# |
# | Modified files:
# |   *	dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_APC_Performance" -eq 1; then
	patch_apply ntdll-APC_Performance/0001-ntdll-Reuse-old-async-fileio-structures-if-possible.patch
fi

# Patchset ntdll-Activation_Context
# |
# | Modified files:
# |   *	dlls/kernel32/tests/actctx.c, dlls/ntdll/actctx.c
# |
if test "$enable_ntdll_Activation_Context" -eq 1; then
	patch_apply ntdll-Activation_Context/0001-ntdll-Fix-return-value-for-missing-ACTIVATION_CONTEX.patch
fi

# Patchset ntdll-ApiSetMap
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44658] Add dummy apiset to PEB struct
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, include/Makefile.in, include/apiset.h, include/winternl.h
# |
if test "$enable_ntdll_ApiSetMap" -eq 1; then
	patch_apply ntdll-ApiSetMap/0001-ntdll-Add-dummy-apiset-to-PEB.patch
fi

# Patchset ntdll-ForceBottomUpAlloc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48175] AION (64 bit) - crashes in crysystem.dll.CryFree() due to high memory pointers allocated
# |   *	[#46568] 64-bit msxml6.dll from Microsoft Core XML Services 6.0 redist package fails to load (Wine doesn't respect
# | 	44-bit user-mode VA limitation from Windows < 8.1)
# |
# | Modified files:
# |   *	dlls/ntdll/unix/virtual.c
# |
if test "$enable_ntdll_ForceBottomUpAlloc" -eq 1; then
	patch_apply ntdll-ForceBottomUpAlloc/0001-ntdll-Increase-step-after-failed-map-attempt-in-try_.patch
	patch_apply ntdll-ForceBottomUpAlloc/0002-ntdll-Increase-free-ranges-view-block-size-on-64-bit.patch
	patch_apply ntdll-ForceBottomUpAlloc/0003-ntdll-Force-virtual-memory-allocation-order.patch
	patch_apply ntdll-ForceBottomUpAlloc/0004-ntdll-Exclude-natively-mapped-areas-from-free-areas-.patch
fi

# Patchset ntdll-WRITECOPY
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-ForceBottomUpAlloc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29384] Multiple applications expect correct handling of WRITECOPY memory protection (Voobly fails to launch Age of
# | 	Empires II, MSYS2)
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/kernel32/tests/virtual.c, dlls/ntdll/unix/loader.c, dlls/ntdll/unix/server.c,
# | 	dlls/ntdll/unix/signal_arm.c, dlls/ntdll/unix/signal_arm64.c, dlls/ntdll/unix/signal_i386.c,
# | 	dlls/ntdll/unix/signal_x86_64.c, dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c,
# | 	dlls/psapi/tests/psapi_main.c
# |
if test "$enable_ntdll_WRITECOPY" -eq 1; then
	patch_apply ntdll-WRITECOPY/0001-ntdll-Trigger-write-watches-before-passing-userdata-.patch
	patch_apply ntdll-WRITECOPY/0003-ntdll-Setup-a-temporary-signal-handler-during-proces.patch
	patch_apply ntdll-WRITECOPY/0004-ntdll-Properly-handle-PAGE_WRITECOPY-protection.-try.patch
	patch_apply ntdll-WRITECOPY/0005-ntdll-Track-if-a-WRITECOPY-page-has-been-modified.patch
	patch_apply ntdll-WRITECOPY/0006-ntdll-Support-WRITECOPY-on-x64.patch
	patch_apply ntdll-WRITECOPY/0007-ntdll-Report-unmodified-WRITECOPY-pages-as-shared.patch
	patch_apply ntdll-WRITECOPY/0008-ntdll-Fallback-to-copy-pages-for-WRITECOPY.patch
	patch_apply ntdll-WRITECOPY/0009-kernel32-tests-psapi-tests-Update-tests.patch
fi

# Patchset ntdll-Builtin_Prot
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-ForceBottomUpAlloc, ntdll-WRITECOPY
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44650] Fix holes in ELF mappings
# |
# | Modified files:
# |   *	dlls/ntdll/unix/virtual.c, dlls/psapi/tests/psapi_main.c
# |
if test "$enable_ntdll_Builtin_Prot" -eq 1; then
	patch_apply ntdll-Builtin_Prot/0001-ntdll-Fix-holes-in-ELF-mappings.patch
fi

# Patchset ntdll-CriticalSection
# |
# | Modified files:
# |   *	dlls/ntdll/heap.c, dlls/ntdll/ntdll_misc.h, dlls/ntdll/threadpool.c
# |
if test "$enable_ntdll_CriticalSection" -eq 1; then
	patch_apply ntdll-CriticalSection/0002-ntdll-Add-inline-versions-of-RtlEnterCriticalSection.patch
	patch_apply ntdll-CriticalSection/0003-ntdll-Use-fast-CS-functions-for-heap-locking.patch
	patch_apply ntdll-CriticalSection/0004-ntdll-Use-fast-CS-functions-for-threadpool-locking.patch
fi

# Patchset ntdll-Dealloc_Thread_Stack
# |
# | Modified files:
# |   *	dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c
# |
if test "$enable_ntdll_Dealloc_Thread_Stack" -eq 1; then
	patch_apply ntdll-Dealloc_Thread_Stack/0001-ntdll-Do-not-allow-to-allocate-thread-stack-for-curr.patch
fi

# Patchset ntdll-Exception
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44819] Throw second DBG_PRINTEXCEPTION_C when debugging.
# |
# | Modified files:
# |   *	dlls/kernelbase/debug.c, dlls/ntdll/tests/exception.c
# |
if test "$enable_ntdll_Exception" -eq 1; then
	patch_apply ntdll-Exception/0002-ntdll-OutputDebugString-should-throw-the-exception-a.patch
fi

# Patchset ntdll-FileFsFullSizeInformation
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_FileFsFullSizeInformation" -eq 1; then
	patch_apply ntdll-FileFsFullSizeInformation/0001-ntdll-Add-support-for-FileFsFullSizeInformation-clas.patch
fi

# Patchset ntdll-HashLinks
# |
# | Modified files:
# |   *	dlls/kernel32/tests/loader.c, dlls/ntdll/loader.c, include/winternl.h
# |
if test "$enable_ntdll_HashLinks" -eq 1; then
	patch_apply ntdll-HashLinks/0001-ntdll-Implement-HashLinks-field-in-LDR-module-data.patch
	patch_apply ntdll-HashLinks/0002-ntdll-Use-HashLinks-when-searching-for-a-dll-using-t.patch
fi

# Patchset ntdll-Heap_Improvements
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43224] Improvement for heap allocation performance
# |
# | Modified files:
# |   *	dlls/ntdll/heap.c
# |
if test "$enable_ntdll_Heap_Improvements" -eq 1; then
	patch_apply ntdll-Heap_Improvements/0001-ntdll-Add-helper-function-to-delete-free-blocks.patch
	patch_apply ntdll-Heap_Improvements/0002-ntdll-Improve-heap-allocation-performance.patch
fi

# Patchset ntdll-Hide_Wine_Exports
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38656] Add support for hiding wine version information from applications
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, dlls/ntdll/ntdll_misc.h
# |
if test "$enable_ntdll_Hide_Wine_Exports" -eq 1; then
	patch_apply ntdll-Hide_Wine_Exports/0001-ntdll-Add-support-for-hiding-wine-version-informatio.patch
fi

# Patchset ntdll-Interrupt-0x2e
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42647] Implement stub handler for int 0x2e
# |
# | Modified files:
# |   *	dlls/ntdll/unix/signal_i386.c
# |
if test "$enable_ntdll_Interrupt_0x2e" -eq 1; then
	patch_apply ntdll-Interrupt-0x2e/0001-ntdll-Catch-windows-int-0x2e-syscall-on-i386.patch
fi

# Patchset ntdll-Manifest_Range
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18889] ntdll: Support ISOLATIONAWARE_MANIFEST_RESOURCE_ID range
# |
# | Modified files:
# |   *	dlls/ntdll/actctx.c
# |
if test "$enable_ntdll_Manifest_Range" -eq 1; then
	patch_apply ntdll-Manifest_Range/0001-ntdll-Support-ISOLATIONAWARE_MANIFEST_RESOURCE_ID-ra.patch
fi

# Patchset ntdll-NtAccessCheck
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/ntdll/unix/security.c
# |
if test "$enable_ntdll_NtAccessCheck" -eq 1; then
	patch_apply ntdll-NtAccessCheck/0001-ntdll-Improve-invalid-paramater-handling-in-NtAccess.patch
fi

# Patchset server-Object_Types
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44629] Process Hacker can't enumerate handles
# |   *	[#45374] Yet Another Process Monitor (.NET 2.0 app) reports System.AccessViolationException
# |
# | Modified files:
# |   *	dlls/ntdll/tests/info.c, dlls/ntdll/tests/om.c, dlls/ntdll/unix/file.c, dlls/ntdll/unix/system.c, include/winternl.h,
# | 	server/completion.c, server/directory.c, server/event.c, server/file.c, server/handle.c, server/mailslot.c,
# | 	server/main.c, server/mapping.c, server/mutex.c, server/named_pipe.c, server/object.c, server/object.h,
# | 	server/process.c, server/protocol.def, server/registry.c, server/semaphore.c, server/symlink.c, server/thread.c,
# | 	server/timer.c, server/token.c, server/winstation.c
# |
if test "$enable_server_Object_Types" -eq 1; then
	patch_apply server-Object_Types/0001-ntdll-Implement-SystemExtendedHandleInformation-in-N.patch
	patch_apply server-Object_Types/0002-ntdll-Implement-ObjectTypesInformation-in-NtQueryObj.patch
	patch_apply server-Object_Types/0003-server-Register-types-during-startup.patch
	patch_apply server-Object_Types/0004-server-Rename-ObjectType-to-Type.patch
	patch_apply server-Object_Types/0008-ntdll-Set-TypeIndex-for-ObjectTypeInformation-in-NtQ.patch
	patch_apply server-Object_Types/0009-ntdll-Set-object-type-for-System-Extended-HandleInfo.patch
	patch_apply server-Object_Types/0010-ntdll-Mimic-object-type-behavior-for-different-windo.patch
fi

# Patchset ntdll-NtAlertThreadByThreadId
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	server-Object_Types
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50292] Process-local synchronization objects use private interfaces into the Unix library
# |
# | Modified files:
# |   *	dlls/ntdll/Makefile.in, dlls/ntdll/critsection.c, dlls/ntdll/ntdll.spec, dlls/ntdll/ntdll_misc.h, dlls/ntdll/sync.c,
# | 	dlls/ntdll/tests/Makefile.in, dlls/ntdll/tests/om.c, dlls/ntdll/tests/sync.c, dlls/ntdll/thread.c,
# | 	dlls/ntdll/unix/loader.c, dlls/ntdll/unix/sync.c, dlls/ntdll/unix/thread.c, dlls/ntdll/unix/unix_private.h,
# | 	dlls/ntdll/unix/virtual.c, dlls/ntdll/unixlib.h, include/winbase.h, include/winternl.h
# |
if test "$enable_ntdll_NtAlertThreadByThreadId" -eq 1; then
	patch_apply ntdll-NtAlertThreadByThreadId/0001-ntdll-tests-Move-some-tests-to-a-new-sync.c-file.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0002-ntdll-tests-Add-some-tests-for-Rtl-resources.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0003-ntdll-Use-a-separate-mutex-to-lock-the-TEB-list.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0004-ntdll-Implement-NtAlertThreadByThreadId-and-NtWaitFo.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0005-ntdll-tests-Add-basic-tests-for-thread-id-alert-func.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0006-ntdll-Implement-thread-id-alerts-on-top-of-futexes-i.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0007-ntdll-Implement-thread-id-alerts-on-top-of-Mach-sema.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0009-ntdll-Reimplement-Win32-futexes-on-top-of-thread-ID-.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0010-ntdll-Merge-critsection.c-into-sync.c.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0011-ntdll-Reimplement-the-critical-section-fast-path-on-.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0012-ntdll-Get-rid-of-the-direct-futex-path-for-condition.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0013-ntdll-Reimplement-SRW-locks-on-top-of-Win32-futexes.patch
fi

# Patchset ntdll-Pipe_SpecialCharacters
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28995] Allow special characters in pipe names
# |
# | Modified files:
# |   *	dlls/kernel32/tests/pipe.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_Pipe_SpecialCharacters" -eq 1; then
	patch_apply ntdll-Pipe_SpecialCharacters/0001-ntdll-Allow-special-characters-in-pipe-names.patch
fi

# Patchset ntdll-NtDevicePath
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-Pipe_SpecialCharacters
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37487] Resolve \\SystemRoot\\ prefix when opening files
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_NtDevicePath" -eq 1; then
	patch_apply ntdll-NtDevicePath/0001-ntdll-Implement-opening-files-through-nt-device-path.patch
fi

# Patchset ntdll-NtQuerySection
# |
# | Modified files:
# |   *	dlls/kernel32/tests/virtual.c
# |
if test "$enable_ntdll_NtQuerySection" -eq 1; then
	patch_apply ntdll-NtQuerySection/0002-kernel32-tests-Add-tests-for-NtQuerySection.patch
fi

# Patchset ntdll-NtQueryVirtualMemory
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-ForceBottomUpAlloc, ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, ntdll-
# | 	Pipe_SpecialCharacters, ntdll-NtDevicePath
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23999] Implement MemorySectionName class in NtQueryVirtualMemory
# |   *	[#27248] Implement K32GetMappedFileName
# |
# | Modified files:
# |   *	dlls/kernelbase/debug.c, dlls/ntdll/tests/info.c, dlls/ntdll/unix/file.c, dlls/ntdll/unix/process.c,
# | 	dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c, dlls/psapi/tests/psapi_main.c, server/mapping.c,
# | 	server/protocol.def
# |
if test "$enable_ntdll_NtQueryVirtualMemory" -eq 1; then
	patch_apply ntdll-NtQueryVirtualMemory/0003-ntdll-Implement-NtQueryVirtualMemory-MemorySectionNa.patch
	patch_apply ntdll-NtQueryVirtualMemory/0004-ntdll-tests-Add-tests-for-NtQueryVirtualMemory-Memor.patch
	patch_apply ntdll-NtQueryVirtualMemory/0005-ntdll-tests-Add-test-to-ensure-section-name-is-full-.patch
	patch_apply ntdll-NtQueryVirtualMemory/0006-ntdll-Allow-to-query-section-names-from-other-proces.patch
	patch_apply ntdll-NtQueryVirtualMemory/0007-kernel32-Implement-K32GetMappedFileName.-v2.patch
	patch_apply ntdll-NtQueryVirtualMemory/0008-ntdll-Resolve-drive-symlinks-before-returning-sectio.patch
	patch_apply ntdll-NtQueryVirtualMemory/0009-ntdll-Fixup-builtin-dll-names-in-get_section_name.patch
	patch_apply ntdll-NtQueryVirtualMemory/0010-server-Fix-process-access-rights-in-get_mapping_file.patch
fi

# Patchset ntdll-NtSetLdtEntries
# |
# | Modified files:
# |   *	dlls/kernel32/tests/thread.c, dlls/ntdll/unix/signal_i386.c, libs/wine/ldt.c
# |
if test "$enable_ntdll_NtSetLdtEntries" -eq 1; then
	patch_apply ntdll-NtSetLdtEntries/0001-ntdll-Implement-NtSetLdtEntries.patch
	patch_apply ntdll-NtSetLdtEntries/0002-libs-wine-Allow-to-modify-reserved-LDT-entries.patch
fi

# Patchset ntdll-ProcessQuotaLimits
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44812] Multiple applications need NtQueryInformationProcess 'ProcessQuotaLimits' class support (MSYS2, ProcessHacker
# | 	2.x)
# |
# | Modified files:
# |   *	dlls/ntdll/unix/process.c
# |
if test "$enable_ntdll_ProcessQuotaLimits" -eq 1; then
	patch_apply ntdll-ProcessQuotaLimits/0001-ntdll-Add-fake-data-implementation-for-ProcessQuotaL.patch
fi

# Patchset ntdll-RtlQueryPackageIdentity
# |
# | Modified files:
# |   *	dlls/ntdll/tests/Makefile.in, dlls/ntdll/tests/rtl.c
# |
if test "$enable_ntdll_RtlQueryPackageIdentity" -eq 1; then
	patch_apply ntdll-RtlQueryPackageIdentity/0003-ntdll-tests-Add-basic-tests-for-RtlQueryPackageIdent.patch
fi

# Patchset ntdll-RtlQueryProcessPlaceholderCompatibilityMode
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50026] : ntdll: Add stub for RtlQueryProcessPlaceholderCompatibilityMode
# |
# | Modified files:
# |   *	dlls/ntdll/ntdll.spec, dlls/ntdll/rtl.c, dlls/ntoskrnl.exe/ntoskrnl.exe.spec, include/ddk/ntifs.h
# |
if test "$enable_ntdll_RtlQueryProcessPlaceholderCompatibilityMode" -eq 1; then
	patch_apply ntdll-RtlQueryProcessPlaceholderCompatibilityMode/0001-ntdll-Add-stub-for-RtlQueryProcessPlaceholderCompati.patch
fi

# Patchset ntdll-RtlQueryRegistryValuesEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46969] ntdll: Implement RtlQueryRegistryValuesEx
# |
# | Modified files:
# |   *	dlls/ntdll/ntdll.spec, dlls/ntdll/reg.c
# |
if test "$enable_ntdll_RtlQueryRegistryValuesEx" -eq 1; then
	patch_apply ntdll-RtlQueryRegistryValuesEx/0001-ntdll-Implement-RtlQueryRegistryValuesEx.patch
fi

# Patchset ntdll-Serial_Port_Detection
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39793] Do a device check before returning a default serial port name
# |
# | Modified files:
# |   *	dlls/mountmgr.sys/device.c
# |
if test "$enable_ntdll_Serial_Port_Detection" -eq 1; then
	patch_apply ntdll-Serial_Port_Detection/0001-ntdll-Do-a-device-check-before-returning-a-default-s.patch
fi

# Patchset ntdll-Status_Mapping
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_Status_Mapping" -eq 1; then
	patch_apply ntdll-Status_Mapping/0001-ntdll-Return-STATUS_INVALID_DEVICE_REQUEST-when-tryi.patch
fi

# Patchset ntdll-Syscall_Emulation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48291] Detroit: Become Human crashes on launch
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/unix/signal_x86_64.c, tools/winebuild/import.c
# |
if test "$enable_ntdll_Syscall_Emulation" -eq 1; then
	patch_apply ntdll-Syscall_Emulation/0001-ntdll-Support-x86_64-syscall-emulation.patch
fi

# Patchset ntdll-SystemCodeIntegrityInformation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49192] ntdll: NtQuerySystemInformation support SystemCodeIntegrityInformation
# |
# | Modified files:
# |   *	dlls/ntdll/unix/system.c, include/winternl.h
# |
if test "$enable_ntdll_SystemCodeIntegrityInformation" -eq 1; then
	patch_apply ntdll-SystemCodeIntegrityInformation/0001-ntdll-NtQuerySystemInformation-support-SystemCodeInt.patch
fi

# Patchset ntdll-Zero_mod_name
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c
# |
if test "$enable_ntdll_Zero_mod_name" -eq 1; then
	patch_apply ntdll-Zero_mod_name/0001-ntdll-Initialize-mod_name-to-zero.patch
fi

# Patchset ntdll-aarch-TEB
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38780] AArch64 platforms: register X18 (TEB) must remain reserved for Wine to run 64-bit ARM Windows applications.
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/loader.c, dlls/ntdll/relay.c
# |
if test "$enable_ntdll_aarch_TEB" -eq 1; then
	patch_apply ntdll-aarch-TEB/0001-configure-Avoid-clobbering-x18-on-arm64-within-wine.patch
	patch_apply ntdll-aarch-TEB/0002-ntdll-Always-restore-TEB-to-x18-on-aarch-64-on-retur.patch
fi

# Patchset ntdll-ext4-case-folder
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47099] Support for EXT4 case folding per directory.
# |
# | Modified files:
# |   *	dlls/ntdll/unix/server.c
# |
if test "$enable_ntdll_ext4_case_folder" -eq 1; then
	patch_apply ntdll-ext4-case-folder/0002-ntdll-server-Mark-drive_c-as-case-insensitive-when-c.patch
fi

# Patchset ntdll-set_full_cpu_context
# |
# | Modified files:
# |   *	dlls/ntdll/unix/signal_i386.c
# |
if test "$enable_ntdll_set_full_cpu_context" -eq 1; then
	patch_apply ntdll-set_full_cpu_context/0001-ntdll-Add-back-SS-segment-prefixes-in-set_full_cpu_c.patch
fi

# Patchset ntdll-x86_64_SegDs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47970] Legends of Runeterra crashes at launch
# |
# | Modified files:
# |   *	dlls/ntdll/unix/signal_x86_64.c
# |
if test "$enable_ntdll_x86_64_SegDs" -eq 1; then
	patch_apply ntdll-x86_64_SegDs/0001-ntdll-Report-SegDs-to-be-identical-to-SegSs-on-x86_6.patch
fi

# Patchset nvcuvid-CUDA_Video_Support
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support, nvapi-Stub_DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuvid/Makefile.in, dlls/nvcuvid/nvcuvid.c, dlls/nvcuvid/nvcuvid.spec, include/Makefile.in,
# | 	include/cuviddec.h, include/nvcuvid.h
# |
if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	patch_apply nvcuvid-CUDA_Video_Support/0001-nvcuvid-First-implementation.patch
fi

# Patchset nvencodeapi-Video_Encoder
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support, nvapi-Stub_DLL, nvcuvid-CUDA_Video_Support
# |
# | Modified files:
# |   *	configure.ac, dlls/nvencodeapi/Makefile.in, dlls/nvencodeapi/nvencodeapi.c, dlls/nvencodeapi/nvencodeapi.spec,
# | 	dlls/nvencodeapi64/Makefile.in, dlls/nvencodeapi64/nvencodeapi64.spec, include/Makefile.in, include/nvencodeapi.h
# |
if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	patch_apply nvencodeapi-Video_Encoder/0001-nvencodeapi-First-implementation.patch
	patch_apply nvencodeapi-Video_Encoder/0002-nvencodeapi-Add-debian-specific-paths-to-native-libr.patch
	patch_apply nvencodeapi-Video_Encoder/0003-nvencodeapi-Add-support-for-version-6.0.patch
fi

# Patchset odbccp32-SQLWriteDSNToIni
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50150] : odbccp32: Implement SQLWriteDSNToIni/W
# |
# | Modified files:
# |   *	dlls/odbccp32/odbccp32.c, dlls/odbccp32/tests/misc.c
# |
if test "$enable_odbccp32_SQLWriteDSNToIni" -eq 1; then
	patch_apply odbccp32-SQLWriteDSNToIni/0001-odbccp32-Implement-SQLWriteDSNToIni-W.patch
fi

# Patchset oleaut32-CreateTypeLib
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8780] Forward CreateTypeLib to CreateTypeLib2
# |
# | Modified files:
# |   *	dlls/oleaut32/typelib.c
# |
if test "$enable_oleaut32_CreateTypeLib" -eq 1; then
	patch_apply oleaut32-CreateTypeLib/0001-oleaut32-Implement-semi-stub-for-CreateTypeLib.patch
fi

# Patchset oleaut32-Load_Save_EMF
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40523] Implement support for loading and saving EMF to IPicture interface
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_Load_Save_EMF" -eq 1; then
	patch_apply oleaut32-Load_Save_EMF/0001-oleaut32-tests-Add-some-tests-for-loading-and-saving.patch
	patch_apply oleaut32-Load_Save_EMF/0002-oleaut32-Add-support-for-loading-and-saving-EMF-to-I.patch
fi

# Patchset oleaut32-OLEPictureImpl_SaveAsFile
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	oleaut32-Load_Save_EMF
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8532] Implement a better stub for IPicture::SaveAsFile
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OLEPictureImpl_SaveAsFile" -eq 1; then
	patch_apply oleaut32-OLEPictureImpl_SaveAsFile/0002-oleaut32-Implement-a-better-stub-for-IPicture-SaveAs.patch
	patch_apply oleaut32-OLEPictureImpl_SaveAsFile/0003-oleaut32-Implement-SaveAsFile-for-PICTYPE_ENHMETAFIL.patch
fi

# Patchset oleaut32-OleLoadPicture
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39474] Create DIB section in OleLoadPicture
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OleLoadPicture" -eq 1; then
	patch_apply oleaut32-OleLoadPicture/0001-oleaut32-OleLoadPicture-should-create-a-DIB-section-.patch
	patch_apply oleaut32-OleLoadPicture/0002-oleaut32-Make-OleLoadPicture-load-DIBs-using-WIC-dec.patch
fi

# Patchset oleaut32-OleLoadPictureFile
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OleLoadPictureFile" -eq 1; then
	patch_apply oleaut32-OleLoadPictureFile/0001-oleaut32-Do-not-reimplement-OleLoadPicture-in-OleLoa.patch
	patch_apply oleaut32-OleLoadPictureFile/0002-oleaut32-Factor-out-stream-creation-from-OleLoadPict.patch
	patch_apply oleaut32-OleLoadPictureFile/0003-oleaut32-Implement-OleLoadPictureFile.patch
fi

# Patchset opencl-version_1_2
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46470] opencl: Add support for OpenCL 1.2.
# |
# | Modified files:
# |   *	configure, configure.ac, dlls/opencl/opencl.c, dlls/opencl/opencl.spec, include/config.h.in
# |
if test "$enable_opencl_version_1_2" -eq 1; then
	patch_apply opencl-version_1_2/0001-opencl-Add-OpenCL-1.0-function-pointer-loader.patch
	patch_apply opencl-version_1_2/0002-opencl-Use-function-pointer-instead-of-call-the-func.patch
	patch_apply opencl-version_1_2/0003-opencl-Add-OpenCL-1.1-implementation.patch
	patch_apply opencl-version_1_2/0004-opencl-Add-OpenCL-1.2-implementation.patch
	patch_apply opencl-version_1_2/0005-opencl-Expose-all-extensions-list-to-wine.patch
fi

# Patchset packager-DllMain
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43472] Prefer native version of packager.dll
# |
# | Modified files:
# |   *	dlls/packager/packager_main.c
# |
if test "$enable_packager_DllMain" -eq 1; then
	patch_apply packager-DllMain/0001-packager-Prefer-native-version.patch
fi

# Patchset pdh-PdhLookupPerfNameByIndex-processor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33018] pdh: Support the 'Processor' object string.
# |
# | Modified files:
# |   *	dlls/pdh/pdh_main.c, dlls/pdh/tests/pdh.c
# |
if test "$enable_pdh_PdhLookupPerfNameByIndex_processor" -eq 1; then
	patch_apply pdh-PdhLookupPerfNameByIndex-processor/0001-pdh-Support-the-Processor-object-string.patch
fi

# Patchset programs-findstr
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35254] findstr: Add minimal implementation
# |
# | Modified files:
# |   *	programs/findstr/Makefile.in, programs/findstr/findstr.rc, programs/findstr/main.c, programs/findstr/resources.h
# |
if test "$enable_programs_findstr" -eq 1; then
	patch_apply programs-findstr/0001-findstr-add-basic-functionality-also-support-literal.patch
fi

# Patchset programs-systeminfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42027] systeminfo: Add basic functionality.
# |
# | Modified files:
# |   *	programs/systeminfo/Makefile.in, programs/systeminfo/main.c
# |
if test "$enable_programs_systeminfo" -eq 1; then
	patch_apply programs-systeminfo/0001-systeminfo-add-basic-functionality.patch
fi

# Patchset quartz-MediaSeeking_Positions
# |
# | Modified files:
# |   *	dlls/strmbase/pospass.c
# |
if test "$enable_quartz_MediaSeeking_Positions" -eq 1; then
	patch_apply quartz-MediaSeeking_Positions/0001-strmbase-Fix-MediaSeekingPassThru_GetPositions-retur.patch
fi

# Patchset riched20-Class_Tests
# |
# | Modified files:
# |   *	dlls/riched20/tests/editor.c
# |
if test "$enable_riched20_Class_Tests" -eq 1; then
	patch_apply riched20-Class_Tests/0001-riched20-tests-Add-a-test-to-see-what-richedit-class.patch
fi

# Patchset riched20-IText_Interface
# |
# | Modified files:
# |   *	dlls/riched20/richole.c, dlls/riched20/tests/richole.c
# |
if test "$enable_riched20_IText_Interface" -eq 1; then
	patch_apply riched20-IText_Interface/0003-riched20-Stub-for-ITextPara-interface-and-implement-.patch
	patch_apply riched20-IText_Interface/0010-riched20-Silence-repeated-FIXMEs-triggered-by-Adobe-.patch
fi

# Patchset server-FileEndOfFileInformation
# |
# | Modified files:
# |   *	dlls/ntdll/unix/file.c, server/fd.c, server/protocol.def
# |
if test "$enable_server_FileEndOfFileInformation" -eq 1; then
	patch_apply server-FileEndOfFileInformation/0001-ntdll-Set-EOF-on-file-which-has-a-memory-mapping-sho.patch
	patch_apply server-FileEndOfFileInformation/0002-server-Growing-files-which-are-mapped-to-memory-shou.patch
fi

# Patchset server-File_Permissions
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44691] Improve mapping of DACL to file permissions
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/ntdll/tests/file.c, server/fd.c, server/file.c
# |
if test "$enable_server_File_Permissions" -eq 1; then
	patch_apply server-File_Permissions/0001-server-Improve-STATUS_CANNOT_DELETE-checks-for-direc.patch
	patch_apply server-File_Permissions/0002-server-Allow-to-open-files-without-any-permission-bi.patch
	patch_apply server-File_Permissions/0003-server-When-creating-new-directories-temporarily-giv.patch
	patch_apply server-File_Permissions/0004-advapi32-tests-Add-tests-for-ACL-inheritance-in-Crea.patch
	patch_apply server-File_Permissions/0005-advapi32-tests-Add-ACL-inheritance-tests-for-creatin.patch
	patch_apply server-File_Permissions/0006-ntdll-tests-Added-tests-for-open-behaviour-on-readon.patch
	patch_apply server-File_Permissions/0007-server-FILE_WRITE_ATTRIBUTES-should-succeed-for-read.patch
	patch_apply server-File_Permissions/0008-server-Improve-mapping-of-DACL-to-file-permissions.patch
fi

# Patchset server-Stored_ACLs
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, server-File_Permissions
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33576] Support for stored file ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, server/change.c, server/file.c, server/file.h, server/object.c, server/object.h
# |
if test "$enable_server_Stored_ACLs" -eq 1; then
	patch_apply server-Stored_ACLs/0001-server-Unify-the-storage-of-security-attributes-for-.patch
	patch_apply server-Stored_ACLs/0002-server-Unify-the-retrieval-of-security-attributes-fo.patch
	patch_apply server-Stored_ACLs/0003-server-Add-a-helper-function-set_sd_from_token_inter.patch
	patch_apply server-Stored_ACLs/0004-server-Temporarily-store-the-full-security-descripto.patch
	patch_apply server-Stored_ACLs/0005-server-Store-file-security-attributes-with-extended-.patch
	patch_apply server-Stored_ACLs/0006-server-Convert-return-of-file-security-masks-with-ge.patch
	patch_apply server-Stored_ACLs/0007-server-Retrieve-file-security-attributes-with-extend.patch
fi

# Patchset server-Inherited_ACLs
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, server-File_Permissions, server-Stored_ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, server/file.c
# |
if test "$enable_server_Inherited_ACLs" -eq 1; then
	patch_apply server-Inherited_ACLs/0001-server-Inherit-security-attributes-from-parent-direc.patch
fi

# Patchset server-Key_State
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31899] Implement locking and synchronization of key states
# |   *	[#35907] Fix caps lock state issues with multiple processes
# |
# | Modified files:
# |   *	server/queue.c
# |
if test "$enable_server_Key_State" -eq 1; then
	patch_apply server-Key_State/0001-server-Introduce-a-helper-function-to-update-the-thr.patch
	patch_apply server-Key_State/0002-server-Implement-locking-and-synchronization-of-keys.patch
fi

# Patchset server-Registry_Notifications
# |
# | Modified files:
# |   *	dlls/ntdll/tests/reg.c, server/registry.c
# |
if test "$enable_server_Registry_Notifications" -eq 1; then
	patch_apply server-Registry_Notifications/0001-server-Allow-multiple-registry-notifications-for-the.patch
	patch_apply server-Registry_Notifications/0002-server-Introduce-refcounting-for-registry-notificati.patch
fi

# Patchset server-unix_name
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46070] Basemark Web 3.0 Desktop Launcher crashes
# |
# | Modified files:
# |   *	server/fd.c, server/file.c, server/file.h
# |
if test "$enable_server_unix_name" -eq 1; then
	patch_apply server-unix_name/0001-server-Try-to-retrieve-the-unix-name-on-handles-crea.patch
fi

# Patchset setupapi-DiskSpaceList
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50337] Roland Zenology Pro (VST3 plugin) used with carla-bridge fails to save files
# |
# | Modified files:
# |   *	dlls/setupapi/diskspace.c, dlls/setupapi/queue.c, dlls/setupapi/setupapi.spec, dlls/setupapi/setupapi_private.h,
# | 	dlls/setupapi/stubs.c, dlls/setupapi/tests/diskspace.c, include/setupapi.h
# |
if test "$enable_setupapi_DiskSpaceList" -eq 1; then
	patch_apply setupapi-DiskSpaceList/0001-setupapi-Rewrite-DiskSpaceList-logic-using-lists.patch
	patch_apply setupapi-DiskSpaceList/0002-setupapi-Implement-SetupAddToDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0003-setupapi-Implement-SetupQueryDrivesInDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0004-setupapi-Ignore-deletion-of-added-files-in-SetupAddT.patch
	patch_apply setupapi-DiskSpaceList/0005-setupapi-ImplementSetupAddSectionToDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0006-setupapi-Implement-SetupAddInstallSectionToDiskSpace.patch
fi

# Patchset setupapi-SPFILENOTIFY_FILEINCABINET
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42827] Fix Param2 in SPFILENOTIFY_FILEINCABINET handler
# |
# | Modified files:
# |   *	dlls/setupapi/setupcab.c, dlls/setupapi/tests/setupcab.c
# |
if test "$enable_setupapi_SPFILENOTIFY_FILEINCABINET" -eq 1; then
	patch_apply setupapi-SPFILENOTIFY_FILEINCABINET/0001-setupapi-tests-Add-more-tests-for-SPFILENOTIFY_FILEI.patch
	patch_apply setupapi-SPFILENOTIFY_FILEINCABINET/0002-setupapi-Fix-CabinetName-passed-to-SPFILENOTIFY_CABI.patch
	patch_apply setupapi-SPFILENOTIFY_FILEINCABINET/0003-setupapi-tests-Add-tests-for-cabinet-name-passed-to-.patch
	patch_apply setupapi-SPFILENOTIFY_FILEINCABINET/0004-setupapi-Fix-parameters-of-SPFILENOTIFY_FILEINCABINE.patch
fi

# Patchset shdocvw-ParseURLFromOutsideSource_Tests
# |
# | Modified files:
# |   *	dlls/shdocvw/shdocvw_main.c, dlls/shdocvw/tests/shdocvw.c
# |
if test "$enable_shdocvw_ParseURLFromOutsideSource_Tests" -eq 1; then
	patch_apply shdocvw-ParseURLFromOutsideSource_Tests/0001-shdocvw-Check-precisely-ParseURLFromOutsideSourceX-r.patch
fi

# Patchset shell32-SHFileOperation_Move
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#25207] SHFileOperation with FO_MOVE should create new directory on Vista+
# |
# | Modified files:
# |   *	dlls/shell32/shlfileop.c, dlls/shell32/tests/shlfileop.c
# |
if test "$enable_shell32_SHFileOperation_Move" -eq 1; then
	patch_apply shell32-SHFileOperation_Move/0001-shell32-Fix-SHFileOperation-FO_MOVE-for-creating-sub.patch
fi

# Patchset shell32-Progress_Dialog
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-FileDispositionInformation, kernel32-CopyFileEx, shell32-SHFileOperation_Move
# |
# | Modified files:
# |   *	dlls/shell32/shell32.rc, dlls/shell32/shlfileop.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_Progress_Dialog" -eq 1; then
	patch_apply shell32-Progress_Dialog/0001-shell32-Correct-indentation-in-shfileop.c.patch
	patch_apply shell32-Progress_Dialog/0002-shell32-Pass-FILE_INFORMATION-into-SHNotify-function.patch
	patch_apply shell32-Progress_Dialog/0003-shell32-Implement-file-operation-progress-dialog.patch
	patch_apply shell32-Progress_Dialog/0004-shell32-Show-animation-during-SHFileOperation.patch
fi

# Patchset shell32-ACE_Viewer
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-FileDispositionInformation, kernel32-CopyFileEx, shell32-SHFileOperation_Move, shell32-Progress_Dialog
# |
# | Modified files:
# |   *	dlls/aclui/Makefile.in, dlls/aclui/aclui.rc, dlls/aclui/aclui_main.c, dlls/aclui/resource.h, dlls/aclui/user_icons.bmp,
# | 	dlls/shell32/Makefile.in, dlls/shell32/shell32.rc, dlls/shell32/shlview_cmenu.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_ACE_Viewer" -eq 1; then
	patch_apply shell32-ACE_Viewer/0001-aclui-Add-basic-ACE-viewer.patch
	patch_apply shell32-ACE_Viewer/0002-shell32-Add-security-property-tab.patch
fi

# Patchset shell32-Context_Menu
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34319] Add support for Paste in context menu
# |   *	[#34322] Fix implementation of Cut file operation
# |   *	[#34321] Fix Cut/Copy/Paste keyboard shortcuts in Total Commander
# |
# | Modified files:
# |   *	dlls/shell32/brsfolder.c, dlls/shell32/clipboard.c, dlls/shell32/dataobject.c, dlls/shell32/recyclebin.c,
# | 	dlls/shell32/shell32.rc, dlls/shell32/shell32_main.h, dlls/shell32/shellfolder.h, dlls/shell32/shfldr_fs.c,
# | 	dlls/shell32/shfldr_unixfs.c, dlls/shell32/shlview.c, dlls/shell32/shlview_cmenu.c
# |
if test "$enable_shell32_Context_Menu" -eq 1; then
	patch_apply shell32-Context_Menu/0001-shell32-Fix-copying-of-files-when-using-a-context-me.patch
	patch_apply shell32-Context_Menu/0002-shell32-Set-return-value-correctly-in-DoPaste.patch
	patch_apply shell32-Context_Menu/0003-shell32-Implement-insert-paste-for-item-context-menu.patch
	patch_apply shell32-Context_Menu/0005-shell32-Add-support-for-setting-getting-PREFERREDDRO.patch
	patch_apply shell32-Context_Menu/0006-shell32-Add-parameter-to-ISFHelper-DeleteItems-to-al.patch
	patch_apply shell32-Context_Menu/0007-shell32-Remove-source-files-when-using-cut-in-the-co.patch
	patch_apply shell32-Context_Menu/0008-shell32-Recognize-cut-copy-paste-string-verbs-in-ite.patch
fi

# Patchset shell32-IconCache
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45696] shell32: Generate icons from available icons if required.
# |
# | Modified files:
# |   *	dlls/shell32/iconcache.c
# |
if test "$enable_shell32_IconCache" -eq 1; then
	patch_apply shell32-IconCache/0001-shell32-iconcache-Generate-icons-from-available-icons-.patch
fi

# Patchset shell32-NewMenu_Interface
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24812] Implement shell32 NewMenu class with new folder item
# |
# | Modified files:
# |   *	dlls/shell32/Makefile.in, dlls/shell32/shell32_classes.idl, dlls/shell32/shell32_main.h, dlls/shell32/shellnew.c,
# | 	dlls/shell32/shellole.c, dlls/shell32/tests/shlview.c
# |
if test "$enable_shell32_NewMenu_Interface" -eq 1; then
	patch_apply shell32-NewMenu_Interface/0001-shell32-Implement-NewMenu-with-new-folder-item.patch
fi

# Patchset shell32-SFGAO_HASSUBFOLDER
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24851] Only set SFGAO_HASSUBFOLDER when there are really subfolders
# |
# | Modified files:
# |   *	dlls/shell32/shfldr_unixfs.c, dlls/shell32/shlfolder.c
# |
if test "$enable_shell32_SFGAO_HASSUBFOLDER" -eq 1; then
	patch_apply shell32-SFGAO_HASSUBFOLDER/0001-shell32-Set-SFGAO_HASSUBFOLDER-correctly-for-unixfs.patch
	patch_apply shell32-SFGAO_HASSUBFOLDER/0002-shell32-Set-SFGAO_HASSUBFOLDER-correctly-for-normal-.patch
fi

# Patchset shell32-SHGetStockIconInfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45868] Improve semi-stub SHGetStockIconInfo
# |
# | Modified files:
# |   *	dlls/shell32/iconcache.c, dlls/shell32/tests/shelllink.c
# |
if test "$enable_shell32_SHGetStockIconInfo" -eq 1; then
	patch_apply shell32-SHGetStockIconInfo/0001-shell32-Improve-semi-stub-SHGetStockIconInfo-try-fin.patch
fi

# Patchset shell32-Toolbar_Bitmaps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40236] Add missing resources for IE6 buttons
# |
# | Modified files:
# |   *	dlls/shell32/Makefile.in, dlls/shell32/resources/ietoolbar.bmp, dlls/shell32/resources/ietoolbar.svg,
# | 	dlls/shell32/resources/ietoolbar_small.bmp, dlls/shell32/resources/ietoolbar_small.svg, dlls/shell32/shell32.rc
# |
if test "$enable_shell32_Toolbar_Bitmaps" -eq 1; then
	patch_apply shell32-Toolbar_Bitmaps/0001-shell32-Add-toolbar-bitmaps-compatible-with-IE6.patch
	patch_apply shell32-Toolbar_Bitmaps/0002-shell32-Add-more-Tango-icons-to-the-IE-toolbar.patch
fi

# Patchset shell32-UnixFS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43109] Do not use unixfs for devices without mountpoint
# |
# | Modified files:
# |   *	dlls/shell32/shfldr_desktop.c, dlls/shell32/tests/shlfolder.c
# |
if test "$enable_shell32_UnixFS" -eq 1; then
	patch_apply shell32-UnixFS/0001-shell32-Do-not-use-unixfs-for-devices-without-mountp.patch
fi

# Patchset shlwapi-AssocGetPerceivedType
# |
# | Modified files:
# |   *	dlls/shlwapi/assoc.c, dlls/shlwapi/tests/assoc.c
# |
if test "$enable_shlwapi_AssocGetPerceivedType" -eq 1; then
	patch_apply shlwapi-AssocGetPerceivedType/0001-shlwapi-tests-Add-tests-for-AssocGetPerceivedType.patch
	patch_apply shlwapi-AssocGetPerceivedType/0002-shlwapi-Implement-AssocGetPerceivedType.patch
fi

# Patchset shlwapi-SHAddDataBlock
# |
# | Modified files:
# |   *	dlls/shlwapi/clist.c, dlls/shlwapi/tests/clist.c
# |
if test "$enable_shlwapi_SHAddDataBlock" -eq 1; then
	patch_apply shlwapi-SHAddDataBlock/0001-shlwapi-Fix-the-return-value-of-SHAddDataBlock.patch
fi

# Patchset shlwapi-UrlCanonicalize
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23166] shlwapi: Support ./ in UrlCanonicalize.
# |
# | Modified files:
# |   *	dlls/kernelbase/path.c, dlls/shlwapi/tests/url.c
# |
if test "$enable_shlwapi_UrlCanonicalize" -eq 1; then
	patch_apply shlwapi-UrlCanonicalize/0001-shlwapi-Support-.-in-UrlCanonicalize.patch
fi

# Patchset shlwapi-UrlCombine
# |
# | Modified files:
# |   *	dlls/kernelbase/path.c, dlls/shlwapi/tests/url.c
# |
if test "$enable_shlwapi_UrlCombine" -eq 1; then
	patch_apply shlwapi-UrlCombine/0001-shlwapi-tests-Add-additional-tests-for-UrlCombine-and-.patch
	patch_apply shlwapi-UrlCombine/0002-shlwapi-UrlCombineW-workaround-for-relative-paths.patch
fi

# Patchset stdole32.idl-Typelib
# |
# | Modified files:
# |   *	dlls/stdole32.tlb/std_ole_v1.idl, include/Makefile.in, include/stdole32.idl
# |
if test "$enable_stdole32_idl_Typelib" -eq 1; then
	patch_apply stdole32.idl-Typelib/0001-include-Make-stdole32.idl-a-public-component.patch
fi

# Patchset widl-SLTG_Typelib_Support
# |
# | Modified files:
# |   *	dlls/oleaut32/typelib.c, dlls/oleaut32/typelib.h, tools/widl/Makefile.in, tools/widl/register.c, tools/widl/typelib.h,
# | 	tools/widl/widl.c, tools/widl/widl.h, tools/widl/write_sltg.c
# |
if test "$enable_widl_SLTG_Typelib_Support" -eq 1; then
	patch_apply widl-SLTG_Typelib_Support/0001-widl-Add-initial-implementation-of-SLTG-typelib-gene.patch
	patch_apply widl-SLTG_Typelib_Support/0002-widl-Add-support-for-structures.patch
	patch_apply widl-SLTG_Typelib_Support/0003-widl-Properly-align-name-table-entries.patch
	patch_apply widl-SLTG_Typelib_Support/0004-widl-More-accurately-report-variable-descriptions-da.patch
	patch_apply widl-SLTG_Typelib_Support/0005-widl-Calculate-size-of-instance-for-structures.patch
	patch_apply widl-SLTG_Typelib_Support/0006-widl-Write-correct-typekind-to-the-SLTG-typeinfo-blo.patch
	patch_apply widl-SLTG_Typelib_Support/0007-widl-Write-SLTG-blocks-according-to-the-index-order.patch
	patch_apply widl-SLTG_Typelib_Support/0008-widl-Write-correct-syskind-by-SLTG-typelib-generator.patch
	patch_apply widl-SLTG_Typelib_Support/0009-widl-Add-support-for-VT_VOID-and-VT_VARIANT-to-SLTG-.patch
	patch_apply widl-SLTG_Typelib_Support/0010-widl-Add-support-for-VT_USERDEFINED-to-SLTG-typelib-.patch
	patch_apply widl-SLTG_Typelib_Support/0011-widl-Factor-out-SLTG-tail-initialization.patch
	patch_apply widl-SLTG_Typelib_Support/0012-widl-Add-support-for-recursive-type-references-to-SL.patch
	patch_apply widl-SLTG_Typelib_Support/0013-widl-Add-support-for-interfaces-to-SLTG-typelib-gene.patch
	patch_apply widl-SLTG_Typelib_Support/0014-widl-Add-support-for-inherited-interfaces-to-SLTG-ty.patch
	patch_apply widl-SLTG_Typelib_Support/0015-widl-Make-automatic-dispid-generation-scheme-better-.patch
	patch_apply widl-SLTG_Typelib_Support/0016-widl-Create-library-block-index-right-after-the-Comp.patch
	patch_apply widl-SLTG_Typelib_Support/0017-widl-Fix-generation-of-resources-containing-an-old-t.patch
	patch_apply widl-SLTG_Typelib_Support/0018-widl-Add-oldtlb-switch-in-usage-message.patch
	patch_apply widl-SLTG_Typelib_Support/0019-widl-Avoid-relying-on-side-effects-when-marking-func.patch
	patch_apply widl-SLTG_Typelib_Support/0020-widl-Set-the-lowest-bit-in-the-param-name-to-indicat.patch
	patch_apply widl-SLTG_Typelib_Support/0021-oleaut32-Fix-logic-for-deciding-whether-type-descrip.patch
	patch_apply widl-SLTG_Typelib_Support/0022-widl-Add-support-for-function-parameter-flags-to-SLT.patch
	patch_apply widl-SLTG_Typelib_Support/0023-oleaut32-Implement-decoding-of-SLTG-help-strings.patch
	patch_apply widl-SLTG_Typelib_Support/0024-oleaut32-Add-support-for-decoding-SLTG-function-help.patch
	patch_apply widl-SLTG_Typelib_Support/0025-oleaut32-Add-support-for-decoding-SLTG-variable-help.patch
	patch_apply widl-SLTG_Typelib_Support/0026-widl-Minor-cosmetic-clean-up.patch
fi

# Patchset stdole32.tlb-SLTG_Typelib
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-SLTG_Typelib_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#3689] Compile stdole32.tlb in SLTG typelib format
# |
# | Modified files:
# |   *	dlls/stdole32.tlb/Makefile.in
# |
if test "$enable_stdole32_tlb_SLTG_Typelib" -eq 1; then
	patch_apply stdole32.tlb-SLTG_Typelib/0020-stdole32.tlb-Compile-typelib-with-oldtlb.patch
fi

# Patchset tasklist-basics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48596] Tasklist add basic functionality.
# |
# | Modified files:
# |   *	programs/tasklist/tasklist.c
# |
if test "$enable_tasklist_basics" -eq 1; then
	patch_apply tasklist-basics/0001-tasklist.exe-add-minimal-functionality.patch
fi

# Patchset user32-DM_SETDEFID
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22764] Use root dialog for DM_SETDEFID/DM_GETDEFID in DefDlgProc
# |
# | Modified files:
# |   *	dlls/user32/defdlg.c, dlls/user32/dialog.c, dlls/user32/tests/win.c
# |
if test "$enable_user32_DM_SETDEFID" -eq 1; then
	patch_apply user32-DM_SETDEFID/0001-user32-Do-not-initialize-dialog-info-for-every-windo.patch
	patch_apply user32-DM_SETDEFID/0002-user32-Use-root-dialog-for-DM_SETDEFID-DM_GETDEFID-i.patch
	patch_apply user32-DM_SETDEFID/0003-user32-tests-Add-a-bunch-of-tests-for-DM_SETDEFID-DM.patch
fi

# Patchset user32-Dialog_Paint_Event
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35652] Send WM_PAINT event during dialog creation
# |
# | Modified files:
# |   *	dlls/user32/dialog.c
# |
if test "$enable_user32_Dialog_Paint_Event" -eq 1; then
	patch_apply user32-Dialog_Paint_Event/0001-user32-Call-UpdateWindow-during-DIALOG_CreateIndirec.patch
fi

# Patchset user32-DrawTextExW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22109] Fix handling of invert_y in DrawTextExW
# |
# | Modified files:
# |   *	dlls/user32/text.c
# |
if test "$enable_user32_DrawTextExW" -eq 1; then
	patch_apply user32-DrawTextExW/0001-user32-Fix-handling-of-invert_y-in-DrawTextExW.patch
fi

# Patchset user32-FlashWindowEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43124] FlashWindowEx: WM_NCACTIVATE behavior is incorrect
# |
# | Modified files:
# |   *	dlls/user32/tests/win.c, dlls/user32/win.c
# |
if test "$enable_user32_FlashWindowEx" -eq 1; then
	patch_apply user32-FlashWindowEx/0001-user32-Improve-FlashWindowEx-message-and-return-valu.patch
fi

# Patchset user32-GetSystemMetrics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18732] Make it possible to change media center / tablet pc status
# |
# | Modified files:
# |   *	dlls/user32/sysparams.c
# |
if test "$enable_user32_GetSystemMetrics" -eq 1; then
	patch_apply user32-GetSystemMetrics/0001-user32-Allow-changing-the-tablet-media-center-status.patch
fi

# Patchset user32-Implement-CascadeWindows
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45968] user32: Implement CascadeWindows
# |   *	[#46197] user32: Implement TileWindows
# |
# | Modified files:
# |   *	dlls/user32/mdi.c
# |
if test "$enable_user32_Implement_CascadeWindows" -eq 1; then
	patch_apply user32-Implement-CascadeWindows/0001-user32-Implement-CascadeWindows.patch
	patch_apply user32-Implement-CascadeWindows/0002-user32-Implement-TileWindows.patch
fi

# Patchset user32-InternalGetWindowIcon
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47915] user32: AddInternalGetWindowIcon stub.
# |
# | Modified files:
# |   *	dlls/user32/user32.spec, dlls/user32/win.c
# |
if test "$enable_user32_InternalGetWindowIcon" -eq 1; then
	patch_apply user32-InternalGetWindowIcon/0001-user32-AddInternalGetWindowIcon-stub.patch
fi

# Patchset user32-LR_LOADFROMFILE
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24963] Workaround for Windows 3.1 apps which call LoadImage(LR_LOADFROMFILE) with a resource id
# |
# | Modified files:
# |   *	dlls/user32/cursoricon.c
# |
if test "$enable_user32_LR_LOADFROMFILE" -eq 1; then
	patch_apply user32-LR_LOADFROMFILE/0001-user32-Add-a-workaround-for-Windows-3.1-apps-which-c.patch
fi

# Patchset user32-ListBox_Size
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38142] Fix calculation of listbox size when horizontal scrollbar is present
# |
# | Modified files:
# |   *	dlls/user32/listbox.c
# |
if test "$enable_user32_ListBox_Size" -eq 1; then
	patch_apply user32-ListBox_Size/0001-user32-Fix-calculation-of-listbox-size-when-horizont.patch
fi

# Patchset user32-LoadKeyboardLayoutEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28170] user32: Added LoadKeyboardLayoutEx stub.
# |
# | Modified files:
# |   *	dlls/user32/input.c, dlls/user32/user32.spec
# |
if test "$enable_user32_LoadKeyboardLayoutEx" -eq 1; then
	patch_apply user32-LoadKeyboardLayoutEx/0001-user32-Added-LoadKeyboardLayoutEx-stub.patch
fi

# Patchset user32-MessageBox_WS_EX_TOPMOST
# |
# | Modified files:
# |   *	dlls/user32/msgbox.c, dlls/user32/tests/dialog.c
# |
if test "$enable_user32_MessageBox_WS_EX_TOPMOST" -eq 1; then
	patch_apply user32-MessageBox_WS_EX_TOPMOST/0001-user32-tests-Add-some-tests-to-see-when-MessageBox-g.patch
	patch_apply user32-MessageBox_WS_EX_TOPMOST/0002-user32-MessageBox-should-be-topmost-when-MB_SYSTEMMO.patch
fi

# Patchset user32-Mouse_Message_Hwnd
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22458] Fix issues with inserting symbols by clicking on center in Word 2007 & 2010
# |   *	[#12007] Fix issues with dragging layers between images in Adobe Photoshop 7.0
# |   *	[#9512] Make sure popups don't block access to objects underneath in DVDPro
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/user32/tests/input.c, dlls/winex11.drv/bitblt.c, server/protocol.def, server/window.c
# |
if test "$enable_user32_Mouse_Message_Hwnd" -eq 1; then
	patch_apply user32-Mouse_Message_Hwnd/0001-user32-Try-harder-to-find-a-target-for-mouse-message.patch
	patch_apply user32-Mouse_Message_Hwnd/0002-user32-tests-Add-tests-for-clicking-through-layered-.patch
	patch_apply user32-Mouse_Message_Hwnd/0003-user32-tests-Add-tests-for-window-region-of-layered-.patch
	patch_apply user32-Mouse_Message_Hwnd/0004-user32-tests-Add-tests-for-DC-region.patch
	patch_apply user32-Mouse_Message_Hwnd/0005-server-Add-support-for-a-layered-window-region.-v2.patch
fi

# Patchset user32-QueryDisplayConfig
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49583] Implement QueryDisplayConfig
# |
# | Modified files:
# |   *	dlls/user32/sysparams.c
# |
if test "$enable_user32_QueryDisplayConfig" -eq 1; then
	patch_apply user32-QueryDisplayConfig/0001-user32-Implement-QueryDisplayConfig.patch
fi

# Patchset user32-Refresh_MDI_Menus
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#21855] Refresh MDI menus when DefMDIChildProc(WM_SETTEXT) is called
# |
# | Modified files:
# |   *	dlls/user32/mdi.c
# |
if test "$enable_user32_Refresh_MDI_Menus" -eq 1; then
	patch_apply user32-Refresh_MDI_Menus/0001-user32-Refresh-MDI-menus-when-DefMDIChildProcW-WM_SE.patch
fi

# Patchset user32-ScrollWindowEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37706] Fix return value of ScrollWindowEx for invisible windows
# |
# | Modified files:
# |   *	dlls/user32/painting.c
# |
if test "$enable_user32_ScrollWindowEx" -eq 1; then
	patch_apply user32-ScrollWindowEx/0001-user32-Fix-return-value-of-ScrollWindowEx-for-invisi.patch
fi

# Patchset user32-message-order
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40262] Correct order of windows messages.
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, dlls/user32/winpos.c
# |
if test "$enable_user32_message_order" -eq 1; then
	patch_apply user32-message-order/0001-user32-Fix-messages-sent-on-a-window-without-WS_CHIL.patch
fi

# Patchset user32-msgbox-Support-WM_COPY-mesg
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#17205] Support WM_COPY in MessageBox dialogs.
# |
# | Modified files:
# |   *	dlls/user32/msgbox.c, dlls/user32/tests/dialog.c
# |
if test "$enable_user32_msgbox_Support_WM_COPY_mesg" -eq 1; then
	patch_apply user32-msgbox-Support-WM_COPY-mesg/0001-user32-msgbox-Support-WM_COPY-Message.patch
	patch_apply user32-msgbox-Support-WM_COPY-mesg/0002-user32-msgbox-Use-a-windows-hook-to-trap-Ctrl-C.patch
fi

# Patchset winex11.drv-mouse-coorrds
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46309] winex11.drv: Use root-relative coordinates for events, if possible.
# |
# | Modified files:
# |   *	dlls/winex11.drv/mouse.c
# |
if test "$enable_winex11_drv_mouse_coorrds" -eq 1; then
	patch_apply winex11.drv-mouse-coorrds/0001-winex11.drv-mouse-Use-root-relative-coordinates-for-ev.patch
fi

# Patchset user32-rawinput-mouse
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11.drv-mouse-coorrds
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42631] Mouse drift, jump or don't react to small slow movements in Unity-engine games and Fallout 4 (partly fixed in
# | 	Unity games, have walkaround in Fallout4 )
# |   *	[#42675] Overwatch: Phantom mouse input / view pulled up to ceiling
# |
# | Modified files:
# |   *	dlls/user32/input.c, dlls/user32/user32.spec, dlls/wineandroid.drv/keyboard.c, dlls/wineandroid.drv/window.c,
# | 	dlls/winemac.drv/ime.c, dlls/winemac.drv/keyboard.c, dlls/winemac.drv/mouse.c, dlls/winex11.drv/event.c,
# | 	dlls/winex11.drv/keyboard.c, dlls/winex11.drv/mouse.c, dlls/winex11.drv/window.c, dlls/winex11.drv/x11drv.h,
# | 	dlls/winex11.drv/x11drv_main.c, include/winuser.h, server/protocol.def, server/queue.c
# |
if test "$enable_user32_rawinput_mouse" -eq 1; then
	patch_apply user32-rawinput-mouse/0006-server-Add-send_hardware_message-flags-for-rawinput-.patch
	patch_apply user32-rawinput-mouse/0007-user32-Add-__wine_send_input-flags-to-hint-raw-input.patch
	patch_apply user32-rawinput-mouse/0008-winex11.drv-Advertise-XInput2-version-2.1-support.patch
	patch_apply user32-rawinput-mouse/0009-winex11.drv-Keep-track-of-pointer-and-device-button-.patch
	patch_apply user32-rawinput-mouse/0010-winex11.drv-Listen-to-RawMotion-and-RawButton-events.patch
	patch_apply user32-rawinput-mouse/0011-winex11.drv-Move-header-order.patch
fi

# Patchset user32-rawinput-hid
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11.drv-mouse-coorrds, user32-rawinput-mouse
# |
# | Modified files:
# |   *	dlls/hidclass.sys/device.c, dlls/hidclass.sys/hid.h, dlls/hidclass.sys/pnp.c, dlls/user32/message.c,
# | 	dlls/user32/rawinput.c, dlls/user32/user_private.h, server/protocol.def, server/queue.c, server/trace.c
# |
if test "$enable_user32_rawinput_hid" -eq 1; then
	patch_apply user32-rawinput-hid/0001-server-Add-process-argument-to-find_rawinput_device.patch
	patch_apply user32-rawinput-hid/0002-server-Allow-extra-data-for-hardware_msg_data-messag.patch
	patch_apply user32-rawinput-hid/0003-server-Make-it-possible-to-queue-rawinput-message-on.patch
	patch_apply user32-rawinput-hid/0004-server-Add-HID-input-message-type-to-send_hardware_m.patch
	patch_apply user32-rawinput-hid/0005-hidclass.sys-Send-input-message-to-server-when-HID-r.patch
fi

# Patchset user32-rawinput-mouse-experimental
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11.drv-mouse-coorrds, user32-rawinput-mouse
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45882] - Raw Input should use untransformed mouse values (affects Overwatch, several Source games).
# |
# | Modified files:
# |   *	dlls/user32/rawinput.c, dlls/winex11.drv/mouse.c, dlls/winex11.drv/x11drv.h, dlls/winex11.drv/x11drv_main.c,
# | 	server/queue.c
# |
if test "$enable_user32_rawinput_mouse_experimental" -eq 1; then
	patch_apply user32-rawinput-mouse-experimental/0001-winex11.drv-Add-support-for-absolute-RawMotion-event.patch
	patch_apply user32-rawinput-mouse-experimental/0002-winex11.drv-Send-relative-RawMotion-events-unprocess.patch
	patch_apply user32-rawinput-mouse-experimental/0003-winex11.drv-Accumulate-mouse-movement-to-avoid-round.patch
fi

# Patchset user32-recursive-activation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46274] user32: Prevent a recursive loop with the activation messages.
# |
# | Modified files:
# |   *	dlls/user32/focus.c, dlls/user32/tests/msg.c, dlls/user32/win.h
# |
if test "$enable_user32_recursive_activation" -eq 1; then
	patch_apply user32-recursive-activation/0001-user32-focus-Prevent-a-recursive-loop-with-the-activ.patch
	patch_apply user32-recursive-activation/0002-user32-tests-Test-a-recursive-activation-loop-on-WM_.patch
fi

# Patchset user32-window-activation
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	user32-recursive-activation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47507] Send a WM_ACTIVATE message after restoring a minimized window.
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, dlls/user32/winpos.c
# |
if test "$enable_user32_window_activation" -eq 1; then
	patch_apply user32-window-activation/0001-user32-Send-a-WM_ACTIVATE-message-after-restoring-a-.patch
fi

# Patchset uxtheme-CloseThemeClass
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41729] Protect CloseThemeData() from invalid input
# |
# | Modified files:
# |   *	dlls/uxtheme/msstyles.c, dlls/uxtheme/msstyles.h, dlls/uxtheme/tests/system.c
# |
if test "$enable_uxtheme_CloseThemeClass" -eq 1; then
	patch_apply uxtheme-CloseThemeClass/0001-uxtheme-Protect-CloseThemeData-from-invalid-input.patch
fi

# Patchset version-VerQueryValue
# |
# | Modified files:
# |   *	dlls/version/tests/info.c
# |
if test "$enable_version_VerQueryValue" -eq 1; then
	patch_apply version-VerQueryValue/0001-version-Test-for-VerQueryValueA-try-2.patch
fi

# Patchset widl-winrt-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49998] widl - Support WinRT idls
# |
# | Modified files:
# |   *	include/Makefile.in, include/windows.foundation.idl, include/windows.media.speechsynthesis.idl, tools/widl/expr.c,
# | 	tools/widl/hash.c, tools/widl/hash.h, tools/widl/header.c, tools/widl/parser.l, tools/widl/parser.y,
# | 	tools/widl/typegen.c, tools/widl/typelib.c, tools/widl/typetree.c, tools/widl/typetree.h, tools/widl/utils.c,
# | 	tools/widl/utils.h, tools/widl/widltypes.h
# |
if test "$enable_widl_winrt_support" -eq 1; then
	patch_apply widl-winrt-support/0001-include-Add-windows.media.speechsynthesis.idl-draft.patch
	patch_apply widl-winrt-support/0002-widl-Support-WinRT-marshaling_behavior-attribute-par.patch
	patch_apply widl-winrt-support/0003-widl-Support-WinRT-mta-threading-attribute-parsing.patch
	patch_apply widl-winrt-support/0004-widl-Support-WinRT-exclusiveto-attribute-parsing.patch
	patch_apply widl-winrt-support/0005-widl-Support-WinRT-runtimeclass-type.patch
	patch_apply widl-winrt-support/0006-widl-Support-WinRT-eventadd-eventremove-attributes.patch
	patch_apply widl-winrt-support/0007-widl-Support-WinRT-flags-attribute-parsing.patch
	patch_apply widl-winrt-support/0008-widl-Support-using-qualified-names-for-interfaces.patch
	patch_apply widl-winrt-support/0009-widl-Support-WinRT-static-attribute-parsing.patch
	patch_apply widl-winrt-support/0010-widl-Support-WinRT-requires-keyword.patch
	patch_apply widl-winrt-support/0011-widl-Support-WinRT-activatable-attribute.patch
	patch_apply widl-winrt-support/0012-widl-Support-WinRT-parameterized-type-parsing.patch
	patch_apply widl-winrt-support/0013-widl-Introduce-new-strappend-helper.patch
	patch_apply widl-winrt-support/0014-widl-Support-partially-specialized-parameterized-typ.patch
	patch_apply widl-winrt-support/0015-widl-Support-WinRT-parameterized-interface-type.patch
	patch_apply widl-winrt-support/0016-widl-Support-WinRT-delegate-type.patch
	patch_apply widl-winrt-support/0017-widl-Support-WinRT-parameterized-delegate-type.patch
	patch_apply widl-winrt-support/0018-widl-Compute-signatures-for-parameterized-types.patch
	patch_apply widl-winrt-support/0019-widl-Compute-uuids-for-parameterized-types.patch
	patch_apply widl-winrt-support/0020-widl-Generate-helper-macros-for-WinRT-implementation.patch
	patch_apply widl-winrt-support/0021-include-Add-IVectorView-HSTRING-declaration-to-windo.patch
fi

# Patchset windows.media.speech.dll
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-winrt-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49740] windows.media.speech: New DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/windows.media.speech.dll/Makefile.in, dlls/windows.media.speech.dll/windows.media.speech.spec,
# | 	dlls/windows.media.speech.dll/windows.media.speech_main.c, loader/wine.inf.in
# |
if test "$enable_windows_media_speech_dll" -eq 1; then
	patch_apply windows.media.speech.dll/0001-windows.media.speech-Add-stub-dll.patch
	patch_apply windows.media.speech.dll/0002-windows.media.speech-Implement-IInstalledVoicesStati.patch
	patch_apply windows.media.speech.dll/0003-windows.media.speech-Implement-IInstalledVoicesStati.patch
	patch_apply windows.media.speech.dll/0004-windows.media.speech-Fake-empty-IInstalledVoicesStat.patch
fi

# Patchset windows.gaming.input-dll
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-winrt-support, windows.media.speech.dll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49756] windows.gaming.input: New DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/windows.gaming.input.dll/Makefile.in, dlls/windows.gaming.input.dll/windows.gaming.input.spec,
# | 	dlls/windows.gaming.input.dll/windows.gaming.input_main.c, include/Makefile.in, include/asyncinfo.idl,
# | 	include/windows.foundation.idl, include/windows.gaming.input.forcefeedback.idl, include/windows.gaming.input.idl,
# | 	include/windows.system.idl, loader/wine.inf.in
# |
if test "$enable_windows_gaming_input_dll" -eq 1; then
	patch_apply windows.gaming.input-dll/0001-windows.gaming.input-Add-stub-dll.patch
	patch_apply windows.gaming.input-dll/0002-windows.gaming.input-Implement-IGamepadStatics-stubs.patch
	patch_apply windows.gaming.input-dll/0003-windows.gaming.input-Implement-IGamepadStatics-Gamep.patch
	patch_apply windows.gaming.input-dll/0004-windows.gaming.input-Fake-empty-IGamepadStatics-Game.patch
	patch_apply windows.gaming.input-dll/0005-windows.gaming.input-Fake-IEventHandler_Gamepad-supp.patch
	patch_apply windows.gaming.input-dll/0006-windows.gaming.input-Implement-IRawGameControllerSta.patch
	patch_apply windows.gaming.input-dll/0007-windows.gaming.input-Implement-IRawGameControllerSta.patch
	patch_apply windows.gaming.input-dll/0008-windows.gaming.input-Fake-empty-IRawGameControllerSt.patch
	patch_apply windows.gaming.input-dll/0009-windows.gaming.input-Fake-IEventHandler_RawGameContr.patch
fi

# Patchset windows.globalization-dll
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-winrt-support, windows.media.speech.dll, windows.gaming.input-dll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49740] windows.globalization: New DLL
# |   *	[#49998] windows.globalization: New DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/windows.globalization.dll/Makefile.in, dlls/windows.globalization.dll/windows.globalization.spec,
# | 	dlls/windows.globalization.dll/windows.globalization_main.c, include/Makefile.in, include/windows.globalization.idl,
# | 	include/windows.system.userprofile.idl, loader/wine.inf.in
# |
if test "$enable_windows_globalization_dll" -eq 1; then
	patch_apply windows.globalization-dll/0001-windows.globalization-Add-stub-dll.patch
	patch_apply windows.globalization-dll/0002-windows.globalization-Implement-IGlobalizationPrefer.patch
	patch_apply windows.globalization-dll/0003-windows.globalization-Implement-IGlobalizationPrefer.patch
	patch_apply windows.globalization-dll/0004-windows.globalization-Implement-IGlobalizationPrefer.patch
	patch_apply windows.globalization-dll/0005-windows.globalization-Fake-empty-IGlobalizationPrefe.patch
fi

# Patchset windows.networking.connectivity.dll
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-winrt-support, windows.media.speech.dll, windows.gaming.input-dll, windows.globalization-dll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46534] windows.networking.connectivity: New DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/windows.networking.connectivity.dll/Makefile.in,
# | 	dlls/windows.networking.connectivity.dll/windows.networking.connectivity.spec,
# | 	dlls/windows.networking.connectivity.dll/windows.networking.connectivity_main.c, loader/wine.inf.in
# |
if test "$enable_windows_networking_connectivity_dll" -eq 1; then
	patch_apply windows.networking.connectivity.dll/0001-windows.networking.connectivity-Add-stub-dll.patch
	patch_apply windows.networking.connectivity.dll/0002-windows.networking.connectivity-Implement-IActivatio.patch
	patch_apply windows.networking.connectivity.dll/0003-windows.networking.connectivity-Implement-INetworkIn.patch
fi

# Patchset windowscodecs-GIF_Encoder
# |
# | Modified files:
# |   *	dlls/windowscodecs/tests/converter.c
# |
if test "$enable_windowscodecs_GIF_Encoder" -eq 1; then
	patch_apply windowscodecs-GIF_Encoder/0007-windowscodecs-tests-Add-IWICBitmapEncoderInfo-test.patch
fi

# Patchset windowscodecs-TIFF_Support
# |
# | Modified files:
# |   *	dlls/gdiplus/image.c, dlls/gdiplus/tests/image.c, dlls/windowscodecs/metadatahandler.c
# |
if test "$enable_windowscodecs_TIFF_Support" -eq 1; then
	patch_apply windowscodecs-TIFF_Support/0015-windowscodecs-Tolerate-partial-reads-in-the-IFD-meta.patch
	patch_apply windowscodecs-TIFF_Support/0016-gdiplus-Add-support-for-more-image-color-formats.patch
	patch_apply windowscodecs-TIFF_Support/0017-gdiplus-tests-Add-some-tests-for-loading-TIFF-images.patch
fi

# Patchset wine.inf-Directory_ContextMenuHandlers
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29523] Add 'New' context menu handler entry for directories
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_Directory_ContextMenuHandlers" -eq 1; then
	patch_apply wine.inf-Directory_ContextMenuHandlers/0001-wine.inf-Add-New-context-menu-handler-entry-for-dire.patch
fi

# Patchset wine.inf-Dummy_CA_Certificate
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_Dummy_CA_Certificate" -eq 1; then
	patch_apply wine.inf-Dummy_CA_Certificate/0001-wine.inf.in-Add-invalid-dummy-certificate-to-CA-cert.patch
fi

# Patchset wine.inf-Performance
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33661] Add performance library registry keys needed by MS SQL Server Management Studio Express 2008 R2
# |   *	[#33037] Visual Studio 6 can't be installed with WinVer >= Win2K (missing HKEY_PERFORMANCE_DATA 230, process object)
# |
# | Modified files:
# |   *	dlls/advapi32/tests/registry.c, loader/wine.inf.in
# |
if test "$enable_wine_inf_Performance" -eq 1; then
	patch_apply wine.inf-Performance/0001-wine.inf-Add-registry-keys-for-Windows-Performance-L.patch
	patch_apply wine.inf-Performance/0002-wine.inf-Add-Counters-to-the-perflib-key-as-an-alias.patch
	patch_apply wine.inf-Performance/0003-advapi32-tests-Add-test-for-perflib-registry-key.patch
fi

# Patchset wineboot-DriveSerial
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#17823] Assign a drive serial number during prefix creation/update
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_DriveSerial" -eq 1; then
	patch_apply wineboot-DriveSerial/0001-wineboot-Assign-a-drive-serial-number-during-prefix-.patch
fi

# Patchset wineboot-HKEY_DYN_DATA
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7115] Need for Speed 3 installer requires devices in HKEY_DYN_DATA
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_HKEY_DYN_DATA" -eq 1; then
	patch_apply wineboot-HKEY_DYN_DATA/0001-wineboot-Add-some-generic-hardware-in-HKEY_DYN_DATA-.patch
fi

# Patchset wineboot-drivers_etc_Stubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12076] Create stub files for system32/drivers/etc/{services,hosts,networks,protocol}
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_drivers_etc_Stubs" -eq 1; then
	patch_apply wineboot-drivers_etc_Stubs/0001-wineboot-Init-system32-drivers-etc-host-networks-pro.patch
fi

# Patchset wineboot-ProxySettings
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	wineboot-DriveSerial, wineboot-drivers_etc_Stubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42024] Create ProxyEnable key on wineprefix update
# |
# | Modified files:
# |   *	programs/wineboot/Makefile.in, programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_ProxySettings" -eq 1; then
	patch_apply wineboot-ProxySettings/0001-wineboot-Initialize-proxy-settings-registry-key.patch
fi

# Patchset winecfg-Libraries
# |
# | Modified files:
# |   *	programs/winecfg/libraries.c
# |
if test "$enable_winecfg_Libraries" -eq 1; then
	patch_apply winecfg-Libraries/0001-winecfg-Double-click-in-dlls-list-to-edit-item-s-ove.patch
fi

# Patchset winecfg-Staging
# |
# | Modified files:
# |   *	programs/winecfg/Makefile.in, programs/winecfg/main.c, programs/winecfg/resource.h, programs/winecfg/staging.c,
# | 	programs/winecfg/winecfg.h, programs/winecfg/winecfg.rc
# |
if test "$enable_winecfg_Staging" -eq 1; then
	patch_apply winecfg-Staging/0001-winecfg-Add-staging-tab-for-CSMT.patch
	patch_apply winecfg-Staging/0002-winecfg-Add-checkbox-to-enable-disable-vaapi-GPU-dec.patch
	patch_apply winecfg-Staging/0003-winecfg-Add-checkbox-to-enable-disable-EAX-support.patch
	patch_apply winecfg-Staging/0004-winecfg-Add-checkbox-to-enable-disable-HideWineExpor.patch
	patch_apply winecfg-Staging/0005-winecfg-Add-option-to-enable-disable-GTK3-theming.patch
fi

# Patchset wined3d-Accounting
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c, dlls/wined3d/device.c, dlls/wined3d/wined3d_gl.h
# |
if test "$enable_wined3d_Accounting" -eq 1; then
	patch_apply wined3d-Accounting/0001-wined3d-Use-real-values-for-memory-accounting-on-NVI.patch
fi

# Patchset wined3d-CSMT_Main
# |
# | Modified files:
# |   *	dlls/wined3d/cs.c, dlls/wined3d/device.c, dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_CSMT_Main" -eq 1; then
	patch_apply wined3d-CSMT_Main/0045-wined3d-Improve-wined3d_cs_emit_update_sub_resource.patch
fi

# Patchset wined3d-SWVP-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8051] Sims 2 demo exits prematurely
# |
# | Modified files:
# |   *	dlls/d3d8/directx.c, dlls/d3d9/d3d9_private.h, dlls/d3d9/device.c, dlls/d3d9/directx.c, dlls/d3d9/tests/device.c,
# | 	dlls/d3d9/tests/visual.c, dlls/wined3d/adapter_gl.c, dlls/wined3d/device.c, dlls/wined3d/glsl_shader.c,
# | 	dlls/wined3d/shader.c, dlls/wined3d/shader_sm1.c, dlls/wined3d/state.c, dlls/wined3d/stateblock.c,
# | 	dlls/wined3d/wined3d_private.h, include/wine/wined3d.h
# |
if test "$enable_wined3d_SWVP_shaders" -eq 1; then
	patch_apply wined3d-SWVP-shaders/0001-wined3d-Use-UBO-for-vertex-shader-float-constants-if.patch
	patch_apply wined3d-SWVP-shaders/0002-d3d9-Support-SWVP-vertex-shader-float-constants-limi.patch
	patch_apply wined3d-SWVP-shaders/0003-wined3d-Report-actual-vertex-shader-float-constants-.patch
	patch_apply wined3d-SWVP-shaders/0004-wined3d-Support-SWVP-vertex-shader-constants-limit-i.patch
	patch_apply wined3d-SWVP-shaders/0005-wined3d-Support-SWVP-mode-vertex-shaders.patch
fi

# Patchset wined3d-Indexed_Vertex_Blending
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	wined3d-SWVP-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39057] Support for Indexed Vertex Blending
# |
# | Modified files:
# |   *	dlls/d3d9/tests/visual.c, dlls/wined3d/adapter_gl.c, dlls/wined3d/cs.c, dlls/wined3d/device.c, dlls/wined3d/directx.c,
# | 	dlls/wined3d/glsl_shader.c, dlls/wined3d/state.c, dlls/wined3d/utils.c, dlls/wined3d/vertexdeclaration.c,
# | 	dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_Indexed_Vertex_Blending" -eq 1; then
	patch_apply wined3d-Indexed_Vertex_Blending/0001-d3d9-tests-Add-test-for-indexed-vertex-blending.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0002-d3d9-tests-Test-normal-calculation-when-indexed-vert.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0003-d3d9-tests-Check-MaxVertexBlendMatrixIndex-capabilit.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0004-wined3d-Allow-higher-world-matrix-states.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0005-wined3d-Support-indexed-vertex-blending.patch
fi

# Patchset wined3d-Silence_FIXMEs
# |
# | Modified files:
# |   *	dlls/wined3d/surface.c
# |
if test "$enable_wined3d_Silence_FIXMEs" -eq 1; then
	patch_apply wined3d-Silence_FIXMEs/0004-wined3d-Print-FIXME-only-once-in-surface_cpu_blt.patch
fi

# Patchset wined3d-WINED3DFMT_B8G8R8X8_UNORM
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44888] Implement WINED3DFMT_B8G8R8X8_UNORM to WINED3DFMT_L8_UNORM conversion
# |
# | Modified files:
# |   *	dlls/wined3d/surface.c
# |
if test "$enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM" -eq 1; then
	patch_apply wined3d-WINED3DFMT_B8G8R8X8_UNORM/0001-wined3d-Implement-WINED3DFMT_B8G8R8X8_UNORM-to-WINED.patch
fi

# Patchset wined3d-mesa_texture_download
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45901] Avoid GPU synchronization due to GPU-CPU transfer (Overwatch)
# |
# | Modified files:
# |   *	dlls/wined3d/texture.c
# |
if test "$enable_wined3d_mesa_texture_download" -eq 1; then
	patch_apply wined3d-mesa_texture_download/0001-wined3d-Use-glReadPixels-for-RT-texture-download.patch
fi

# Patchset wined3d-unset-flip-gdi
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47419] wined3d: Dont set DDSCAPS_FLIP for gdi renderer.
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c, dlls/wined3d/adapter_vk.c, dlls/wined3d/directx.c
# |
if test "$enable_wined3d_unset_flip_gdi" -eq 1; then
	patch_apply wined3d-unset-flip-gdi/0001-wined3d-Dont-set-DDSCAPS_FLIP-for-gdi-renderer.patch
fi

# Patchset wined3d-wined3d_guess_gl_vendor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42538] Add check for GL_VENDOR = "Brian Paul" to detect Mesa
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c
# |
if test "$enable_wined3d_wined3d_guess_gl_vendor" -eq 1; then
	patch_apply wined3d-wined3d_guess_gl_vendor/0001-wined3d-Also-check-for-Brian-Paul-to-detect-Mesa-gl_.patch
fi

# Patchset wined3d-zero-inf-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34266] wined3d: Add a setting to workaround 0 * inf problem in shader models 1-3.
# |   *	[#45375] Halo Online: Weird black display problems.
# |
# | Modified files:
# |   *	dlls/wined3d/glsl_shader.c, dlls/wined3d/wined3d_main.c, dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_zero_inf_shaders" -eq 1; then
	patch_apply wined3d-zero-inf-shaders/0001-wined3d-Add-a-setting-to-workaround-0-inf-problem-in.patch
fi

# Patchset winedbg-Process_Arguments
# |
# | Modified files:
# |   *	programs/winedbg/info.c
# |
if test "$enable_winedbg_Process_Arguments" -eq 1; then
	patch_apply winedbg-Process_Arguments/0001-programs-winedbg-Print-process-arguments-in-info-thr.patch
fi

# Patchset winemapi-user-xdg-mail
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11770] - use xdg-email if it's available.
# |
# | Modified files:
# |   *	dlls/winemapi/Makefile.in, dlls/winemapi/sendmail.c, dlls/winemapi/winemapi_private.h, dlls/winemapi/xdg-email.c
# |
if test "$enable_winemapi_user_xdg_mail" -eq 1; then
	patch_apply winemapi-user-xdg-mail/0001-winemapi-Directly-use-xdg-email-if-available-enablin.patch
fi

# Patchset winemenubuilder-Desktop_Icon_Path
# |
# | Modified files:
# |   *	programs/winemenubuilder/Makefile.in, programs/winemenubuilder/winemenubuilder.c
# |
if test "$enable_winemenubuilder_Desktop_Icon_Path" -eq 1; then
	patch_apply winemenubuilder-Desktop_Icon_Path/0001-winemenubuilder-Create-desktop-shortcuts-with-absolu.patch
fi

# Patchset winemenubuilder-integration
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41275] Winemenubuilder should respect existing defaults for filetype associations
# |   *	[#22904] Register URL protocol handlers under Linux
# |
# | Modified files:
# |   *	dlls/mshtml/mshtml.inf, loader/wine.inf.in, programs/winemenubuilder/winemenubuilder.c
# |
if test "$enable_winemenubuilder_integration" -eq 1; then
	patch_apply winemenubuilder-integration/0001-winemenubuilder-Blacklist-desktop-integration-for-ce.patch
fi

# Patchset wineps.drv-PostScript_Fixes
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#4836] Various improvements for wineps.drv for Adobe PageMaker compatibility
# |
# | Modified files:
# |   *	dlls/wineps.drv/download.c, dlls/wineps.drv/escape.c, dlls/wineps.drv/psdrv.h
# |
if test "$enable_wineps_drv_PostScript_Fixes" -eq 1; then
	patch_apply wineps.drv-PostScript_Fixes/0004-wineps.drv-Add-support-for-GETFACENAME-and-DOWNLOADF.patch
fi

# Patchset winepulse-PulseAudio_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37042] Implement exclusive mode in PulseAudio backend
# |
# | Modified files:
# |   *	dlls/winepulse.drv/Makefile.in, dlls/winepulse.drv/mmdevdrv.c
# |
if test "$enable_winepulse_PulseAudio_Support" -eq 1; then
	patch_apply winepulse-PulseAudio_Support/0001-winepulse.drv-Use-a-separate-mainloop-and-ctx-for-pu.patch
	patch_apply winepulse-PulseAudio_Support/0003-winepulse-expose-audio-devices-directly-to-programs.patch
	patch_apply winepulse-PulseAudio_Support/0005-winepulse-implement-GetPropValue.patch
	patch_apply winepulse-PulseAudio_Support/0006-winepulse-fetch-actual-program-name-if-possible.patch
	patch_apply winepulse-PulseAudio_Support/0007-winepulse-return-PKEY_AudioEndpoint_PhysicalSpeakers.patch
fi

# Patchset winex11-CandidateWindowPos
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30938] Update a XIM candidate position when cursor location changes
# |
# | Modified files:
# |   *	dlls/user32/caret.c, dlls/user32/driver.c, dlls/user32/user_private.h, dlls/winex11.drv/winex11.drv.spec,
# | 	dlls/winex11.drv/xim.c
# |
if test "$enable_winex11_CandidateWindowPos" -eq 1; then
	patch_apply winex11-CandidateWindowPos/0001-winex11.drv-Update-a-candidate-window-s-position-wit.patch
fi

# Patchset winex11-MWM_Decorations
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42117] Avoid double captions observed under some WMs
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_MWM_Decorations" -eq 1; then
	patch_apply winex11-MWM_Decorations/0001-winex11.drv-Don-t-use-MWM_DECOR_RESIZEH-window-resiz.patch
	patch_apply winex11-MWM_Decorations/0002-winex11.drv-Don-t-add-MWM_DECOR_BORDER-to-windows-wi.patch
fi

# Patchset winex11-UpdateLayeredWindow
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33943] Fix alpha blending in X11DRV_UpdateLayeredWindow
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_UpdateLayeredWindow" -eq 1; then
	patch_apply winex11-UpdateLayeredWindow/0001-winex11-Fix-alpha-blending-in-X11DRV_UpdateLayeredWi.patch
fi

# Patchset winex11-Vulkan_support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44775] Allow vulkan support to be detected at runtime.
# |
# | Modified files:
# |   *	dlls/winex11.drv/vulkan.c
# |
if test "$enable_winex11_Vulkan_support" -eq 1; then
	patch_apply winex11-Vulkan_support/0001-winex11-Specify-a-default-vulkan-driver-if-one-not-f.patch
fi

# Patchset winex11-_NET_ACTIVE_WINDOW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#2155] Forward activate window requests to WM using _NET_ACTIVE_WINDOW
# |
# | Modified files:
# |   *	dlls/user32/driver.c, dlls/user32/focus.c, dlls/user32/user_private.h, dlls/winex11.drv/event.c,
# | 	dlls/winex11.drv/window.c, dlls/winex11.drv/winex11.drv.spec, dlls/winex11.drv/x11drv.h, dlls/winex11.drv/x11drv_main.c
# |
if test "$enable_winex11__NET_ACTIVE_WINDOW" -eq 1; then
	patch_apply winex11-_NET_ACTIVE_WINDOW/0001-winex11.drv-Add-support-for-_NET_ACTIVE_WINDOW.patch
	patch_apply winex11-_NET_ACTIVE_WINDOW/0002-user32-Before-asking-a-WM-to-activate-a-window-make-.patch
fi

# Patchset winex11-WM_WINDOWPOSCHANGING
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11-_NET_ACTIVE_WINDOW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34594] Fix handling of WM_WINDOWPOS{CHANGING,CHANGED} for deactivated topmost window
# |
# | Modified files:
# |   *	dlls/winex11.drv/event.c
# |
if test "$enable_winex11_WM_WINDOWPOSCHANGING" -eq 1; then
	patch_apply winex11-WM_WINDOWPOSCHANGING/0001-winex11.drv-Send-WM_WINDOWPOSCHANGING-WM_WINDOWPOSCH.patch
fi

# Patchset winex11-Window_Style
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37876] Fix handling of window attributes for WS_EX_LAYERED | WS_EX_COMPOSITED
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_Window_Style" -eq 1; then
	patch_apply winex11-Window_Style/0001-winex11-Fix-handling-of-window-attributes-for-WS_EX_.patch
fi

# Patchset winex11-XEMBED
# |
# | Modified files:
# |   *	dlls/winex11.drv/event.c
# |
if test "$enable_winex11_XEMBED" -eq 1; then
	patch_apply winex11-XEMBED/0001-winex11-Enable-disable-windows-when-they-are-un-mapped.patch
fi

# Patchset winex11-ime-check-thread-data
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46263] Final Fantasy XI crashes after accepting EULA when using Ashita
# |   *	[#28861] Final Fantasy XI hangs after character selection
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_ime_check_thread_data" -eq 1; then
	patch_apply winex11-ime-check-thread-data/0001-winex11.drv-handle-missing-thread-data-in-X11DRV_get_ic.patch
fi

# Patchset winex11-key_translation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30984] Improve key translation.
# |   *	[#45605] Letter keys doesn't work in DirectX aplications
# |
# | Modified files:
# |   *	dlls/winex11.drv/keyboard.c
# |
if test "$enable_winex11_key_translation" -eq 1; then
	patch_apply winex11-key_translation/0001-winex11-Match-keyboard-in-Unicode.patch
	patch_apply winex11-key_translation/0002-winex11-Fix-more-key-translation.patch
	patch_apply winex11-key_translation/0003-winex11.drv-Fix-main-Russian-keyboard-layout.patch
fi

# Patchset winex11-wglShareLists
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11436] Do not fail when a used context is passed to wglShareLists
# |   *	[#25419] Fix broken textures in XIII Century: Death or Glory
# |
# | Modified files:
# |   *	dlls/opengl32/tests/opengl.c, dlls/winex11.drv/opengl.c
# |
if test "$enable_winex11_wglShareLists" -eq 1; then
	patch_apply winex11-wglShareLists/0001-winex11.drv-Only-warn-about-used-contexts-in-wglShar.patch
fi

# Patchset winex11.drv-Query_server_position
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#15346] winex11.drv: Query the X server for the actual rect of the window before unmapping it
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_drv_Query_server_position" -eq 1; then
	patch_apply winex11.drv-Query_server_position/0001-winex11.drv-window-Query-the-X-server-for-the-actual.patch
fi

# Patchset wininet-Cleanup
# |
# | Modified files:
# |   *	dlls/wininet/http.c, dlls/wininet/tests/http.c
# |
if test "$enable_wininet_Cleanup" -eq 1; then
	patch_apply wininet-Cleanup/0001-wininet-tests-Add-more-tests-for-cookies.patch
	patch_apply wininet-Cleanup/0002-wininet-tests-Test-auth-credential-reusage-with-host.patch
	patch_apply wininet-Cleanup/0003-wininet-tests-Check-cookie-behaviour-when-overriding.patch
	patch_apply wininet-Cleanup/0004-wininet-Strip-filename-if-no-path-is-set-in-cookie.patch
	patch_apply wininet-Cleanup/0005-wininet-Replacing-header-fields-should-fail-if-they-.patch
fi

# Patchset winmm-Delay_Import_Depends
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37983] Jedi Knight: Dark Forces II crashes with winmm set to native
# |
# | Modified files:
# |   *	dlls/winmm/Makefile.in
# |
if test "$enable_winmm_Delay_Import_Depends" -eq 1; then
	patch_apply winmm-Delay_Import_Depends/0001-winmm-Delay-import-ole32-msacm32-to-workaround-bug-w.patch
fi

# Patchset winmm-mciSendCommandA
# |
# | Modified files:
# |   *	dlls/winmm/mci.c
# |
if test "$enable_winmm_mciSendCommandA" -eq 1; then
	patch_apply winmm-mciSendCommandA/0001-winmm-Do-not-crash-in-Win-9X-mode-when-an-invalid-de.patch
fi

# Patchset wintab32-improvements
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11846] Improve pressure sensitivity.
# |   *	[#15443] Improve Wacom Bambo drawing support
# |   *	[#18517] Improve eraser from working.
# |
# | Modified files:
# |   *	dlls/winex11.drv/wintab.c, dlls/wintab32/context.c
# |
if test "$enable_wintab32_improvements" -eq 1; then
	patch_apply wintab32-improvements/0002-wintab32-Set-lcSysExtX-Y-for-the-first-index-of-WTI_.patch
	patch_apply wintab32-improvements/0003-winex11-Handle-negative-orAltitude-values.patch
	patch_apply wintab32-improvements/0004-winex11.drv-Support-multiplex-categories-WTI_DSCTXS-.patch
	patch_apply wintab32-improvements/0005-winex11-Support-WTI_STATUS-in-WTInfo.patch
fi

# Patchset wintrust-WTHelperGetProvCertFromChain
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44061] Check Parameter in WTHelperGetProvCertFromChain
# |
# | Modified files:
# |   *	dlls/wintrust/tests/softpub.c, dlls/wintrust/wintrust_main.c
# |
if test "$enable_wintrust_WTHelperGetProvCertFromChain" -eq 1; then
	patch_apply wintrust-WTHelperGetProvCertFromChain/0001-wintrust-Add-parameter-check-in-WTHelperGetProvCertF.patch
fi

# Patchset wow64cpu-Wow64Transition
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45567] League of Legends 8.12+ fails to start a game (anticheat engine, validation of WoW64 syscall dispatcher)
# |
# | Modified files:
# |   *	configure, configure.ac, dlls/ntdll/loader.c, dlls/ntdll/ntdll.spec, dlls/wow64cpu/Makefile.in,
# | 	dlls/wow64cpu/wow64cpu.spec, dlls/wow64cpu/wow64cpu_main.c
# |
if test "$enable_wow64cpu_Wow64Transition" -eq 1; then
	patch_apply wow64cpu-Wow64Transition/0001-wow64cpu-Add-stub-dll.patch
	patch_apply wow64cpu-Wow64Transition/0002-ntdll-Add-a-stub-implementation-of-Wow64Transition.patch
fi

# Patchset wpcap-Dynamic_Linking
# |
# | Modified files:
# |   *	configure.ac, dlls/wpcap/Makefile.in, dlls/wpcap/wpcap.c
# |
if test "$enable_wpcap_Dynamic_Linking" -eq 1; then
	patch_apply wpcap-Dynamic_Linking/0001-wpcap-Load-libpcap-dynamically.patch
fi

# Patchset ws2_32-APC_Performance
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c
# |
if test "$enable_ws2_32_APC_Performance" -eq 1; then
	patch_apply ws2_32-APC_Performance/0001-ws2_32-Reuse-old-async-ws2_async_io-structures-if-po.patch
fi

# Patchset ws2_32-Connect_Time
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c, server/protocol.def, server/sock.c
# |
if test "$enable_ws2_32_Connect_Time" -eq 1; then
	patch_apply ws2_32-Connect_Time/0001-ws2_32-Implement-returning-the-proper-time-with-SO_C.patch
fi

# Patchset ws2_32-getsockopt
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8606] Divide values returned by SO_RCVBUF and SO_SNDBUF getsockopt options by two
# |
# | Modified files:
# |   *	dlls/ws2_32/socket.c, dlls/ws2_32/tests/sock.c
# |
if test "$enable_ws2_32_getsockopt" -eq 1; then
	patch_apply ws2_32-getsockopt/0001-ws2_32-Divide-values-returned-by-SO_RCVBUF-and-SO_SN.patch
fi

# Patchset wtsapi32-EnumerateProcesses
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29903] Support for WTSEnumerateProcessesW
# |
# | Modified files:
# |   *	dlls/wtsapi32/tests/wtsapi.c, dlls/wtsapi32/wtsapi32.c
# |
if test "$enable_wtsapi32_EnumerateProcesses" -eq 1; then
	patch_apply wtsapi32-EnumerateProcesses/0001-wtsapi32-Partial-implementation-of-WTSEnumerateProce.patch
fi

# Patchset xactengine-initial
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31476] Support Bully Scholarship Edition xactengine3_1.dll.
# |   *	[#38615] DSA: Drakensang Demo fails on IXACTEngine::Initialize
# |   *	[#41030] Pac-Man Museum requires xactengine3_7
# |   *	[#41045] Captain Morgane requires xactengine3_4
# |   *	[#48684] BlazBlue: Calamity Trigger requires for xactengine 3.3 interface.
# |
# | Modified files:
# |   *	dlls/x3daudio1_7/Makefile.in, dlls/xactengine3_7/tests/Makefile.in, dlls/xactengine3_7/tests/globals.xgs,
# | 	dlls/xactengine3_7/tests/rsrc.rc, dlls/xactengine3_7/tests/xact3.c
# |
if test "$enable_xactengine_initial" -eq 1; then
	patch_apply xactengine-initial/0001-x3daudio1_7-Create-import-library.patch
	patch_apply xactengine-initial/0003-xactengine3_7-tests-Add-Global-settings-test.patch
fi

# Patchset xactengine2-dll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41048] xactengine2_0: Roller Coaster Rampage
# |   *	[#49671] xactengine2_4: Supreme Commander demo
# |   *	[#49668] xactengine2_7: Two Worlds Epic Edition
# |   *	[#41468] xactengine2_9: Supreme Commander Forged Alliance
# |
# | Modified files:
# |   *	configure, configure.ac, dlls/xactengine2_0/Makefile.in, dlls/xactengine2_0/xactengine2_0.spec,
# | 	dlls/xactengine2_1/Makefile.in, dlls/xactengine2_1/xactengine2_1.spec, dlls/xactengine2_10/Makefile.in,
# | 	dlls/xactengine2_10/xact2_classes.idl, dlls/xactengine2_10/xact2_dll.c, dlls/xactengine2_10/xactengine2_10.spec,
# | 	dlls/xactengine2_2/Makefile.in, dlls/xactengine2_2/xactengine2_2.spec, dlls/xactengine2_3/Makefile.in,
# | 	dlls/xactengine2_3/xactengine2_3.spec, dlls/xactengine2_4/Makefile.in, dlls/xactengine2_4/xactengine2_4.spec,
# | 	dlls/xactengine2_5/Makefile.in, dlls/xactengine2_5/xactengine2_5.spec, dlls/xactengine2_6/Makefile.in,
# | 	dlls/xactengine2_6/xactengine2_6.spec, dlls/xactengine2_7/Makefile.in, dlls/xactengine2_7/xactengine2_7.spec,
# | 	dlls/xactengine2_8/Makefile.in, dlls/xactengine2_8/xactengine2_8.spec, dlls/xactengine2_9/Makefile.in,
# | 	dlls/xactengine2_9/xactengine2_9.spec, include/Makefile.in, include/xact.h, include/xact2wb.h
# |
if test "$enable_xactengine2_dll" -eq 1; then
	patch_apply xactengine2-dll/0001-include-Add-xact2wb.h.patch
	patch_apply xactengine2-dll/0002-include-Add-xact.h.patch
	patch_apply xactengine2-dll/0003-xactengine2_10-Add-new-dll.patch
	patch_apply xactengine2-dll/0004-xactengine2_10-Implement-IXACTEngine-interface.patch
	patch_apply xactengine2-dll/0005-xactengine2_10-Implement-IXACTSoundBank-interface.patch
	patch_apply xactengine2-dll/0006-xactengine2_10-Implement-IXACTCue-Interface.patch
	patch_apply xactengine2-dll/0007-xactengine2_10-Implement-IXACTWaveBank-interface.patch
	patch_apply xactengine2-dll/0008-xactengine2_10-Implement-IXACTEngine-CreateStreaming.patch
	patch_apply xactengine2-dll/0009-xactengine2_10-Implement-IXACTWave-interface.patch
	patch_apply xactengine2-dll/0010-xactengine2_10-Implement-IXACTSoundBank-Play-functio.patch
	patch_apply xactengine2-dll/0011-xactengine2_10-Implement-IXACTWaveBank-Play-function.patch
	patch_apply xactengine2-dll/0012-xactengine2_10-Implement-IXACTEngine-Un-RegisterNoti.patch
	patch_apply xactengine2-dll/0013-xactengine2_9-New-Dll.patch
	patch_apply xactengine2-dll/0014-xactengine2_8-New-Dll.patch
	patch_apply xactengine2-dll/0015-xactengine2_7-New-Dll.patch
	patch_apply xactengine2-dll/0016-xactengine2_6-New-Dll.patch
	patch_apply xactengine2-dll/0017-xactengine2_5-New-Dll.patch
	patch_apply xactengine2-dll/0018-xactengine2_4-New-Dll.patch
	patch_apply xactengine2-dll/0019-xactengine2_3-New-Dll.patch
	patch_apply xactengine2-dll/0020-xactengine2_2-New-Dll.patch
	patch_apply xactengine2-dll/0021-xactengine2_1-New-Dll.patch
	patch_apply xactengine2-dll/0022-xactengine2_0-New-Dll.patch
fi

# Patchset xactengine3_7-PrepareWave
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49689] xactengine3_7: Implement IXACT3Engine PrepareWave
# |
# | Modified files:
# |   *	dlls/xactengine3_7/xact_dll.c
# |
if test "$enable_xactengine3_7_PrepareWave" -eq 1; then
	patch_apply xactengine3_7-PrepareWave/0001-xactengine3_7-Implement-IXACT3Engine-PrepareWave.patch
	patch_apply xactengine3_7-PrepareWave/0002-xactengine3_7-Implement-IXACT3Engine-PrepareStreamin.patch
	patch_apply xactengine3_7-PrepareWave/0003-xactengine3_7-Implement-IXACT3Engine-PrepareInMemory.patch
fi

if test "$enable_autoconf" -eq 1; then
	if ! update_configure; then
		abort "'autoreconf -f' failed."
	fi
	if ! update_protocol; then
		abort "'./tools/make_requests' failed."
	fi
fi
# Success
exit 0
