# -*- coding: utf-8 -*-
#
# Copyright © 2008-2013  Red Hat, Inc. All rights reserved.
# Copyright © 2008-2013  Luke Macken <lmacken@redhat.com>
# Copyright © 2008  Kushal Das <kushal@fedoraproject.org>
# Copyright © 2012-2015  Tails Developers <tails@boum.org>
#
# This copyrighted material is made available to anyone wishing to use, modify,
# copy, or redistribute it subject to the terms and conditions of the GNU
# General Public License v.2.  This program is distributed in the hope that it
# will be useful, but WITHOUT ANY WARRANTY expressed or implied, including the
# implied warranties of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.  You should have
# received a copy of the GNU General Public License along with this program; if
# not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
# Floor, Boston, MA 02110-1301, USA. Any Red Hat trademarks that are
# incorporated in the source code or documentation are not subject to the GNU
# General Public License and may only be used or replicated with the express
# permission of Red Hat, Inc.
#
# Author(s): Luke Macken <lmacken@redhat.com>
#            Kushal Das <kushal@fedoraproject.org>
#            Tails Developers <tails@boum.org>

"""
A cross-platform graphical interface for the Tails Installer
"""

import os
import sys
import logging
import threading
import urlparse
import traceback

from time import sleep
from datetime import datetime

from gi.repository import Gdk, GLib, Gtk

from tails_installer import TailsInstallerCreator, TailsInstallerError, _
from tails_installer.config import config
from tails_installer.source import SourceError, LocalIsoSource
from tails_installer.source import RunningLiveSystemSource
from tails_installer.releases import releases, get_fedora_releases
from tails_installer.utils import _to_unicode, _format_bytes_in_gb, _get_datadir
from tails_installer.utils import is_running_from_tails
from urlgrabber.grabber import URLGrabber, URLGrabError
from urlgrabber.progress import BaseMeter

MAX_FAT16 = 2047
MAX_FAT32 = 3999
MAX_EXT = 2097152

# FIXME: port to Gtk.Application

class ReleaseDownloader(threading.Thread):
    def __init__(self, release, progress, proxies, parent):
        threading.Thread.__init__(self)
        self.release = release
        self.progress = progress
        self.proxies = proxies
        self.parent = parent
        for rel in releases:
            if rel['name'] == str(release):
                self.url = rel['url']
                break
        else:
            raise TailsInstallerError(_("Unknown release: %s") % release)

    def run(self):
        GLib.idle_add(self.parent.update_log,
                      _("Downloading %s...") % os.path.basename(self.url))
        grabber = URLGrabber(progress_obj=self.progress, proxies=self.proxies)
        home = os.getenv('HOME', 'USERPROFILE')
        filename = os.path.basename(urlparse.urlparse(self.url).path)
        for folder in ('Downloads', 'My Documents'):
            if os.path.isdir(os.path.join(home, folder)):
                filename = os.path.join(home, folder, filename)
                break
        try:
            iso = grabber.urlgrab(self.url, reget='simple')
        except URLGrabError, e:
            GLib.idle_add(self.parent.download_complete, e.strerror)
        else:
            GLib.idle_add(self.parent.download_complete, iso)


class DownloadProgress(BaseMeter):
    """ An urlgrabber BaseMeter class.

    This class is called automatically by urlgrabber with our download details.
    This class then sends signals to our main dialog window to update the
    progress bar.
    """
    def __init__(self, parent):
        BaseMeter.__init__(self)
        self.parent = parent
        self.size = 0

    def start(self, filename=None, url=None, basename=None, size=None,
              now=None, text=None):
        self.size = size

    def update(self, amount_read, now=None):
        """ Update our download progressbar.

        :read: the number of bytes read so far
        """
        GLib.idle_add(parent.progress, float(amount_read) / self.size)

    def end(self, amount_read):
        self.update(amount_read)


class ProgressThread(threading.Thread):
    """ A thread that monitors the progress of Live USB creation.

    This thread periodically checks the amount of free space left on the
    given drive and sends a signal to our main dialog window to update the
    progress bar.
    """
    totalsize = 0
    orig_free = 0
    drive = None
    get_free_bytes = None

    def __init__(self, parent):
        threading.Thread.__init__(self)
        self.parent = parent
        self.terminate = False

    def set_data(self, size, drive, freebytes):
        self.totalsize = size / 1024
        self.drive = drive
        self.get_free_bytes = freebytes
        self.orig_free = self.get_free_bytes()

    def run(self):
        while not self.terminate:
            free = self.get_free_bytes()
            if free is None:
                break
            value = (self.orig_free - free) / 1024
            GLib.idle_add(self.parent.progress, float(value) / self.totalsize)
            if value >= self.totalsize:
                break
            sleep(3)

    def stop(self):
        self.terminate = True

    def __del__(self):
        GLib.idle_add(self.parent.progress, 1.0)
        threading.Thread.__del__(self)

class TailsInstallerThread(threading.Thread):

    def __init__(self, live, progress, parent):
        threading.Thread.__init__(self)
        self.progress = progress
        self.live = live
        self.parent = parent
        self.maximum = 0

    def status(self, text):
        GLib.idle_add(self.parent.status, text)

    def rescan_devices(self, force_partitions=False):
        self._waiting_detection = True

        def detection_done():
            self._waiting_detection = False

        self.live.detect_supported_drives(
            callback=detection_done,
            force_partitions=force_partitions)

        while self._waiting_detection:
            self.sleep(1)

    def installation_complete(self):
        GLib.idle_add(self.parent.on_installation_complete, None)

    def run(self):
        self.handler = TailsInstallerLogHandler(self.status)
        self.live.log.addHandler(self.handler)
        self.now = datetime.now()
        self.live.save_full_drive()
        try:
            if self.parent.opts.partition:
                self.live.unmount_device()
                if not self.live.can_read_partition_table():
                    self.live.log.info('Clearing unreadable partition table.')
                    self.live.clear_all_partition_tables()
                self.live.partition_device()
                self.rescan_devices(force_partitions=True)
                self.live.switch_drive_to_system_partition()
                self.live.format_device()
                self.live.mount_device()

            self.live.verify_filesystem()
            if not self.live.drive['uuid'] and not self.live.label:
                self.status(_("Error: Cannot set the label or obtain "
                              "the UUID of your device.  Unable to continue."))
                self.live.log.removeHandler(self.handler)
                return

            self.live.check_free_space()

            if not self.parent.opts.noverify:
                # If we know about this ISO, and it's SHA1 -- verify it
                release = self.live.source.get_release()
                if release and ('sha1' in release or 'sha256' in release):
                    if not self.live.verify_iso_sha1(progress=self):
                        self.live.log.removeHandler(self.handler)
                        return

            # Setup the progress bar
            self.progress.set_data(size=self.live.totalsize,
                                   drive=self.live.drive['device'],
                                   freebytes=self.live.get_free_bytes)
            self.progress.start()

            self.live.extract_iso()
            self.live.read_extracted_mbr()
            self.live.create_persistent_overlay()
            self.live.update_configs()

            if not self.parent.opts.partition and self.live.is_partition_GPT():
                self.live.update_system_partition_properties()
            self.live.install_bootloader()
            # self.live.bootable_partition()

            if self.parent.opts.device_checksum:
                self.live.calculate_device_checksum(progress=self)
            if self.parent.opts.liveos_checksum:
                self.live.calculate_liveos_checksum()

            self.progress.stop()

            # Flush all filesystem buffers and unmount
            self.live.flush_buffers()
            self.live.unmount_device()

            if self.parent.opts.partition:
                self.live.switch_back_to_full_drive()

            self.live.reset_mbr()
            self.live.flush_buffers()

            duration = str(datetime.now() - self.now).split('.')[0]
            self.status(_("Installation complete! (%s)") % duration)
            self.installation_complete()

        except Exception, e:
            self.status(e.args[0])
            self.status(_("Tails installation failed!"))
            self.live.log.exception(unicode(e))
            self.live.log.debug(traceback.format_exc())

        self.live.log.removeHandler(self.handler)
        self.progress.stop()

    def set_max_progress(self, maximum):
        self.maximum = maximum

    def update_progress(self, value):
        GLib.idle_add(self.parent.progress, float(value) / maximum)

    def __del__(self):
        self.wait()


class TailsInstallerLogHandler(logging.Handler):

    def __init__(self, cb):
        logging.Handler.__init__(self)
        self.cb = cb

    def emit(self, record):
        if record.levelname in ('INFO', 'ERROR', 'WARN'):
            self.cb(record.msg)


class TailsInstallerWindow(Gtk.ApplicationWindow):
    """ Our main dialog class """

    def __init__(self, app=None, opts=None, args=None):
        Gtk.ApplicationWindow.__init__(self, application=app)

        self.opts = opts
        self.args = args
        self.in_process = False
        self.signals_connected = []
        self.source_available = False
        self.target_available = False
        self.target_selected  = False
        self.persistence = False

        self._build_ui()

        self.update_start_button()
        if self.opts.clone or config['download']['enabled']:
            self.source_available = True
        if self.opts.clone:
            self.__box_source.set_visible(False)
        if not config['download']['enabled']:
            self.__box_source_dl.set_visible(False)
        # FIXME: this may be useful when we'll support windows
        #if sys.platform == 'win32':
        #    self.driveBox.setGeometry(QtCore.QRect(10, 20, 145, 22))
        #    self.refreshDevicesButton = QtGui.QPushButton(self.targetGroupBox)
        #    self.refreshDevicesButton.setGeometry(QtCore.QRect(156, 16, 30, 26))
        #    self.refreshDevicesButton.setText("")
        #    icon = QtGui.QIcon()
        #    icon.addPixmap(QtGui.QPixmap(":/refresh.png"), QtGui.QIcon.Normal, QtGui.QIcon.Off)
        #    self.refreshDevicesButton.setIcon(icon)
        #    self.refreshDevicesButton.setFlat(True)
        #    self.refreshDevicesButton.setObjectName("refreshDevicesButton")
        self.update_start_button()
        self.live = TailsInstallerCreator(opts=opts)
        if self.opts.clone:
            self.live.source = RunningLiveSystemSource(
                path=config['running_liveos_mountpoint'])
        if not self.opts.clone:
            self.populate_releases()
        self.populate_devices()
        self.downloader = None
        self.progress_thread = ProgressThread(parent=self)
        self.download_progress = DownloadProgress(parent=self)
        self.live_thread = TailsInstallerThread(live=self.live,
                                         progress=self.progress_thread,
                                         parent=self)
        self.live.connect_drive_monitor(self.populate_devices)
        self.confirmed = False
        self.delete_existing_liveos_confirmed = False

        # Intercept all tails_installer.INFO log messages, and display them in the gui
        self.handler = TailsInstallerLogHandler(lambda x: self.append_to_log(x))
        self.live.log.addHandler(self.handler)
        if not self.opts.verbose:
            self.live.log.removeHandler(self.live.handler)

        # If an ISO was specified on the command line, use it.
        if args:
            for arg in self.args:
                if arg.lower().endswith('.iso') and os.path.exists(arg):
                    self.select_source_iso(arg)

        # Determine if we have admin rights
        if not self.opts.unprivileged and not self.live.is_admin():
            self.live.log.error(_('Warning: This tool needs to be run as an '
                'Administrator. To do this, right click on the icon and open '
                'the Properties. Under the Compatibility tab, check the "Run '
                'this program as an administrator" box.'))

        # Show the UI
        self.show()

    def _build_ui(self):
        # Set windows properties
        self.set_deletable(True)
        self.connect('delete-event', Gtk.main_quit)
        self.set_title(_("Tails Installer"))

        # Import window content from UI file
        builder = Gtk.Builder.new_from_file(
                os.path.join(_get_datadir(), 'tails-installer.ui'))
        self.__box_installer = builder.get_object('box_installer')
        self.__image_header = builder.get_object('image_header')
        self.__infobar = builder.get_object('infobar')
        self.__label_infobar_title = builder.get_object('label_infobar_title')
        self.__label_infobar_details = builder.get_object('label_infobar_details')
        self.__box_source = builder.get_object('box_source')
        self.__box_source_file = builder.get_object('box_source_file')
        self.__filechooserbutton_source_file = builder.get_object('liststore_source')
        self.__box_source_dl = builder.get_object('box_source_dl')
        self.__liststore_source_dl = builder.get_object('liststore_source_dl')
        self.__combobox_source_dl = builder.get_object('combobox_source_dl')
        self.__box_target = builder.get_object('box_target')
        self.__combobox_target = builder.get_object('combobox_target')
        self.__liststore_target = builder.get_object('liststore_target')
        self.__textview_log = builder.get_object('textview_log')
        self.__progressbar = builder.get_object('progressbar_progress')
        self.__button_start = builder.get_object('button_start')

        self.add(self.__box_installer)
        builder.connect_signals(self)

        # Add a cell renderer to the comboboxes
        cell = Gtk.CellRendererText()
        self.__combobox_target.pack_start(cell, True)
        self.__combobox_target.add_attribute(cell, 'text', 0)

        cell = Gtk.CellRendererText()
        self.__combobox_source_dl.pack_start(cell, True)
        self.__combobox_source_dl.add_attribute(cell, 'text', 0)

        # Add image header
        self.__image_header.set_from_file(
                os.path.join(_get_datadir(), config['branding']['header']))
        rgba = Gdk.RGBA()
        rgba.parse(config['branding']['color'])
        self.__image_header.override_background_color(Gtk.StateFlags.NORMAL, rgba)

    def on_source_file_set(self, filechooserbutton):
        self.select_source_iso(filechooserbutton.get_filename())

    def on_start_clicked(self, button):
        self.begin()

    def on_infobar_response(self, infobar, response):
        self.__infobar.set_visible(False)
        self.__label_infobar_title.set_text("")
        self.__label_infobar_details.set_text("")

    def append_to_log(self, text):
        if not text.endswith('\n'):
            text = text + '\n'
        text_buffer = self.__textview_log.get_buffer()
        text_buffer.insert(text_buffer.get_end_iter(), text)
        self.__textview_log.scroll_to_iter(text_buffer.get_end_iter(), 0, False, 0, 0)

    def update_start_button(self):
        if self.source_available and self.target_available:
            self.__button_start.set_sensitive(True)
        else:
            self.__button_start.set_sensitive(False)

    def populate_devices(self, *args, **kw):
        if self.in_process or self.target_selected:
            return

        def add_devices():
            self.__liststore_target.clear()
            if not len(self.live.drives):
                self.__infobar.set_message_type(Gtk.MessageType.INFO)
                self.__label_infobar_title.set_text(
                        _("No device suitable to install Tails could be found"))
                self.__label_infobar_details.set_text(
                        _("Please plug a USB flash drive or SD card of at least %0.1f GB.")
                        % ((config['min_system_partition_size'] +
                            config['min_persistence_partition_size']) / 1000.))
                self.__infobar.set_visible(True)
                self.target_available = False
                self.update_start_button()
                return
            else:
                self.__infobar.set_visible(False)
            self.live.log.debug('drives: %s' % self.live.drives)
            for device, info in self.live.drives.items():
                # Skip the device that is the source of the copy
                if (
                        self.live.source and
                        self.live.source.dev and (
                            info['udi'] == self.live.source.dev or
                            info['parent_udi'] == self.live.source.dev
                        )
                    ):
                    self.live.log.debug('Skipping source device: %s' % info['device'])
                    continue
                # Skip the running device
                if self.live.running_device() in [info['udi'], info['parent_udi']]:
                    self.live.log.debug('Skipping running device: %s' % info['device'])
                    continue
                # Skip LUKS-encrypted partitions
                if info['fstype'] and info['fstype'] == 'crypto_LUKS':
                    self.live.log.debug('Skipping LUKS-encrypted partition: %s' % info['device'])
                    continue
                size = _format_bytes_in_gb(info['parent_size']
                                           if info['parent_size']
                                           else info['size'])
                details = (_("%(size)s %(label)s") % {
                                'label': info['label'],
                                'size': size
                            }
                           if info['label']
                           else size)
                pretty_name = _("%(vendor)s %(model)s (%(details)s) - %(device)s") % {
                    'device':  info['device'],
                    'vendor':  info['vendor'],
                    'model':   info['model'],
                    'details': details
                }
                # Skip too small devices, but inform the user
                if not info['is_device_big_enough']:
                    message =_('The device "%(pretty_name)s"'
                               ' is too small to install'
                               ' Tails (at least %(size)s GB is required).') %  {
                               'pretty_name':  pretty_name,
                               'size'  :  (float(config['min_system_partition_size']
                                       + config['min_persistence_partition_size'])
                                       / 1000)
                               }
                    self.status(message)
                    continue
                # Skip devices without a Tails installation
                if not self.opts.partition and not self.live.device_can_be_upgraded(info):
                    if is_running_from_tails():
                        action = _('\"Clone & Install\"')
                    else:
                        action = _('\"Install from ISO\"')
                    message = _('It is impossible to upgrade the device %(pretty_name)s'
                               ' because it was not created using Tails Installer.'
                               ' You should instead use %(action)s to upgrade Tails'
                               ' on this device.') % {
                               'pretty_name' : pretty_name,
                               'action'      : action
                    }
                    self.status(message)
                    continue
                self.__liststore_target.append([pretty_name, device])
                self.target_available = True
                self.__combobox_target.set_active(0)
                self.update_start_button()

        try:
            self.live.detect_supported_drives(callback=add_devices)
        except TailsInstallerError, e:
            self.__infobar.set_message_type(Gtk.MessageType.ERROR)
            self.__label_infobar_title.set_text(
                    _("An error happened while installing Tails"))
            self.__label_infobar_details.set_text(e.args[0])
            self.__infobar.set_visible(True)
            self.append_to_log(e.args[0])
            self.target_available = False
            self.update_start_button()

    def populate_releases(self):
       for release in [release['name'] for release in releases]:
            self.__liststore_source_dl.append([release, None])

    def refresh_releases(self):
        self.live.log.info(_('Refreshing releases...'))
        fedora_releases = get_fedora_releases()
        self.__liststore_source_dl.clear()
        for release in [release['name'] for release in fedora_releases]:
            self.__liststore_source_dl.add(release)
        self.live.log.info(_('Releases updated!'))

    def progress(self, value):
        self.__progressbar.set_fraction(value)

    def status(self, text):
        if isinstance(text, Exception):
            text = text.args[0]
        elif isinstance(text, int):
            text = str(text)
        self.append_to_log(text)

    def enable_widgets(self, enabled=True):
        if enabled:
            self.update_start_button()
        else:
            self.__button_start.set_sensitive(False)
        self.__box_source.set_sensitive(enabled)
        self.__combobox_source_dl.set_sensitive(enabled)
        self.__combobox_target.set_sensitive(enabled and not self.target_selected)
        # FIXME: this may be useful when we'll support windows
        #if hasattr(self, 'refreshDevicesButton'):
        #    self.refreshDevicesButton.setEnabled(enabled)
        #if hasattr(self, 'refreshReleasesButton'):
        #    self.refreshReleasesButton.setEnabled(enabled)
        self.in_process = not enabled

    def get_selected_drive(self):
        drive = self.__liststore_target.get(
                self.__combobox_target.get_active_iter(), 1)[0]
        if drive:
            return _to_unicode(drive)

    def on_installation_complete(self, data=None):
        # FIXME: replace content by a specific page
        dialog = Gtk.MessageDialog(parent=self,
                                   flags=Gtk.DialogFlags.DESTROY_WITH_PARENT,
                                   message_type=Gtk.MessageType.INFO,
                                   buttons=Gtk.ButtonsType.CLOSE,
                                   message_format=_("Installation complete!"))
        dialog.format_secondary_text(_("Installation was completed. Press OK "
                                       "to close this program."))
        dialog.run()
        self.close()

    def show_confirmation_dialog(self, title, message):
        dialog = Gtk.MessageDialog(parent=self,
                                   flags=Gtk.DialogFlags.DESTROY_WITH_PARENT,
                                   message_type=Gtk.MessageType.QUESTION,
                                   buttons=Gtk.ButtonsType.YES_NO,
                                   message_format=title)
        dialog.format_secondary_text(message)
        reply = dialog.run()
        dialog.hide()
        if (reply == Gtk.ResponseType.YES):
            return True
        else:
            self.target_selected = None
            self.enable_widgets(True)
            return False

    def begin(self):
        """ Begin the Tails installation process.

        This method is called when the "Install Tails" button is clicked.
        """
        self.enable_widgets(False)
        if not self.target_selected:
            self.live.drive = self.get_selected_drive()
        self.target_selected = True
        for signal_match in self.signals_connected:
            signal_match.remove()

        # Unmount the device if needed
        if self.live.drive['mount']:
            self.live.dest = self.live.drive['mount']
            self.live.unmount_device()

        if not self.opts.partition:
            try:
                self.live.mount_device()
            except TailsInstallerError, e:
                self.status(e.args[0])
                self.enable_widgets(True)
                return
            except OSError, e:
                self.status(_('Unable to mount device'))
                self.enable_widgets(True)
                return

        if self.opts.partition:
            if not self.confirmed:
                if self.show_confirmation_dialog(
                        _("Please confirm your device selection"),
                        _("You are going to install Tails on the "
                          "%(size)s %(vendor)s %(model)s device (%(device)s). "
                          "All data on the selected device will be lost. "
                          "Continue?") %
                          {'vendor': self.live.drive['vendor'],
                           'model':  self.live.drive['model'],
                           'device': self.live.drive['device'],
                           'size':   _format_bytes_in_gb(self.live.drive['size'])
                          }):
                    self.confirmed = True
                else:
                    return
            else:
                # The user has confirmed that they wish to partition their device,
                # let's go on
                self.confirmed = False
        else:
            msg = (_("You are going to upgrade Tails on the %(parent_size)s "
                     "%(vendor)s %(model)s device (%(device)s). "
                     "Any persistent volume on this device will remain unchanged. "
                     "Continue?") % {
                         'vendor': self.live.drive['vendor'],
                         'model':  self.live.drive['model'],
                         'device': self.live.drive['device'],
                         'parent_size': _format_bytes_in_gb(self.live.drive['parent_size'])
                     }
                  )
            if self.show_confirmation_dialog(_("Please confirm your device selection"), msg):
                # The user has confirmed that they wish to overwrite their
                # existing Live OS.  Here we delete it first, in order to
                # accurately calculate progress.
                self.delete_existing_liveos_confirmed = False
                try:
                    self.live.delete_liveos()
                except TailsInstallerError, e:
                    self.status(e.args[0])
                    #self.live.unmount_device()
                    self.enable_widgets(True)
                    return
            else:
                return

        # Remove the log handler, because our live thread will register its own
        self.live.log.removeHandler(self.handler)

        # If we are running in clone mode, move on.
        if self.live.opts.clone:
            self.enable_widgets(False)
            self.live_thread.start()
        # If the user has selected an ISO, use it.
        elif self.live.source.__class__ == LocalIsoSource:
            self.enable_widgets(False)
            self.live_thread.start()
        # Else, download an ISO.
        elif config['download']['enabled']:
            self.downloader = ReleaseDownloader(
                    self.downloadCombo.currentText(),
                    progress=self.download_progress,
                    proxies=self.live.get_proxies(),
                    parent=self)
            self.downloader.start()
        else:
            raise NotImplementedError

    def download_complete(self, iso):
        """ Called by our ReleaseDownloader thread upon completion.

        Upon success, the thread passes in the filename of the downloaded
        release.  If the 'iso' argument is not an existing file, then
        it is assumed that the download failed and 'iso' should contain
        the error message.
        """
        if os.path.exists(iso):
            self.status(_("Download complete!"))
            self.live.source = LocalIsoSource(path=iso)
            self.live_thread.start()
        else:
            self.status(_("Download failed: " + iso))
            self.status(_("You can try again to resume your download"))
            self.enable_widgets(True)

    def on_source_file_set(self, filechooserbutton):
        self.select_source_iso(filechooserbutton.get_filename())

    def select_source_iso(self, isofile):
        if not os.access(isofile, os.R_OK):
            self.status(_("The selected file is unreadable. "
                          "Please fix its permissions or select another file."))
            return False
        try:
            self.live.source = LocalIsoSource(path=isofile)
        except Exception, e:
            self.status(_("Unable to use the selected file.  "
                          "You may have better luck if you move your ISO "
                          "to the root of your drive (ie: C:\)"))
            self.live.log.exception(e.args[0])
            return False

        self.live.log.info(_("%(filename)s selected")
                             % {'filename': os.path.basename(self.live.source.path)})
        self.source_available = True
        self.update_start_button()

    def terminate(self):
        """ Final clean up """
        self.live.terminate()
