/*
 * Copyright (c) 2012-2017 Etnaviv Project
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *    Wladimir J. van der Laan <laanwj@gmail.com>
 */

#include "etnaviv_rs.h"

#include "etnaviv_clear_blit.h"
#include "etnaviv_context.h"
#include "etnaviv_emit.h"
#include "etnaviv_format.h"
#include "etnaviv_resource.h"
#include "etnaviv_screen.h"
#include "etnaviv_surface.h"
#include "etnaviv_tiling.h"
#include "etnaviv_translate.h"
#include "etnaviv_util.h"

#include "pipe/p_defines.h"
#include "pipe/p_state.h"
#include "util/u_blitter.h"
#include "util/u_inlines.h"
#include "util/u_memory.h"
#include "util/u_surface.h"

#include "hw/common.xml.h"
#include "hw/state.xml.h"
#include "hw/state_3d.xml.h"

#include <assert.h>

void
etna_compile_rs_state(struct etna_context *ctx, struct compiled_rs_state *cs,
                      const struct rs_state *rs)
{
   memset(cs, 0, sizeof(*cs));

   /* TILED and SUPERTILED layout have their strides multiplied with 4 in RS */
   unsigned source_stride_shift = COND(rs->source_tiling != ETNA_LAYOUT_LINEAR, 2);
   unsigned dest_stride_shift = COND(rs->dest_tiling != ETNA_LAYOUT_LINEAR, 2);

   /* tiling == ETNA_LAYOUT_MULTI_TILED or ETNA_LAYOUT_MULTI_SUPERTILED? */
   int source_multi = COND(rs->source_tiling & ETNA_LAYOUT_BIT_MULTI, 1);
   int dest_multi = COND(rs->dest_tiling & ETNA_LAYOUT_BIT_MULTI, 1);

   /* Vivante RS needs widths to be a multiple of 16 or bad things
    * happen, such as scribbing over memory, or the GPU hanging,
    * even for non-tiled formats.  As this is serious, use abort().
    */
   if (rs->width & ETNA_RS_WIDTH_MASK)
      abort();

   /* TODO could just pre-generate command buffer, would simply submit to one memcpy */
   cs->RS_CONFIG = VIVS_RS_CONFIG_SOURCE_FORMAT(rs->source_format) |
                   COND(rs->downsample_x, VIVS_RS_CONFIG_DOWNSAMPLE_X) |
                   COND(rs->downsample_y, VIVS_RS_CONFIG_DOWNSAMPLE_Y) |
                   COND(rs->source_tiling & 1, VIVS_RS_CONFIG_SOURCE_TILED) |
                   VIVS_RS_CONFIG_DEST_FORMAT(rs->dest_format) |
                   COND(rs->dest_tiling & 1, VIVS_RS_CONFIG_DEST_TILED) |
                   COND(rs->swap_rb, VIVS_RS_CONFIG_SWAP_RB) |
                   COND(rs->flip, VIVS_RS_CONFIG_FLIP);

   cs->RS_SOURCE_STRIDE = (rs->source_stride << source_stride_shift) |
                          COND(rs->source_tiling & 2, VIVS_RS_SOURCE_STRIDE_TILING) |
                          COND(source_multi, VIVS_RS_SOURCE_STRIDE_MULTI);

   /* Initially all pipes are set to the base address of the source and
    * destination buffer respectively. This will be overridden below as
    * necessary for the multi-pipe, multi-tiled case.
    */
   for (unsigned pipe = 0; pipe < ctx->specs.pixel_pipes; ++pipe) {
      cs->source[pipe].bo = rs->source;
      cs->source[pipe].offset = rs->source_offset;
      cs->source[pipe].flags = ETNA_RELOC_READ;

      cs->dest[pipe].bo = rs->dest;
      cs->dest[pipe].offset = rs->dest_offset;
      cs->dest[pipe].flags = ETNA_RELOC_WRITE;

      cs->RS_PIPE_OFFSET[pipe] = VIVS_RS_PIPE_OFFSET_X(0) | VIVS_RS_PIPE_OFFSET_Y(0);
   }

   cs->RS_DEST_STRIDE = (rs->dest_stride << dest_stride_shift) |
                        COND(rs->dest_tiling & 2, VIVS_RS_DEST_STRIDE_TILING) |
                        COND(dest_multi, VIVS_RS_DEST_STRIDE_MULTI);

   if (ctx->specs.pixel_pipes == 1 || ctx->specs.single_buffer) {
      cs->RS_WINDOW_SIZE = VIVS_RS_WINDOW_SIZE_WIDTH(rs->width) |
                           VIVS_RS_WINDOW_SIZE_HEIGHT(rs->height);
   } else if (ctx->specs.pixel_pipes == 2) {
      assert((rs->height & 7) == 0); /* GPU hangs happen if height not 8-aligned */

      if (source_multi)
         cs->source[1].offset = rs->source_offset + rs->source_stride * rs->source_padded_height / 2;

      if (dest_multi)
         cs->dest[1].offset = rs->dest_offset + rs->dest_stride * rs->dest_padded_height / 2;

      cs->RS_WINDOW_SIZE = VIVS_RS_WINDOW_SIZE_WIDTH(rs->width) |
                           VIVS_RS_WINDOW_SIZE_HEIGHT(rs->height / 2);
      cs->RS_PIPE_OFFSET[1] = VIVS_RS_PIPE_OFFSET_X(0) | VIVS_RS_PIPE_OFFSET_Y(rs->height / 2);
   } else {
      abort();
   }

   cs->RS_DITHER[0] = rs->dither[0];
   cs->RS_DITHER[1] = rs->dither[1];
   cs->RS_CLEAR_CONTROL = VIVS_RS_CLEAR_CONTROL_BITS(rs->clear_bits) | rs->clear_mode;
   cs->RS_FILL_VALUE[0] = rs->clear_value[0];
   cs->RS_FILL_VALUE[1] = rs->clear_value[1];
   cs->RS_FILL_VALUE[2] = rs->clear_value[2];
   cs->RS_FILL_VALUE[3] = rs->clear_value[3];
   cs->RS_EXTRA_CONFIG = VIVS_RS_EXTRA_CONFIG_AA(rs->aa) |
                         VIVS_RS_EXTRA_CONFIG_ENDIAN(rs->endian_mode);

   /* If source the same as destination, and the hardware supports this,
    * do an in-place resolve to fill in unrendered tiles.
    */
   if (ctx->specs.single_buffer && rs->source == rs->dest &&
         rs->source_offset == rs->dest_offset &&
         rs->source_format == rs->dest_format &&
         rs->source_tiling == rs->dest_tiling &&
         (rs->source_tiling & ETNA_LAYOUT_BIT_SUPER) &&
         rs->source_stride == rs->dest_stride &&
         !rs->downsample_x && !rs->downsample_y &&
         !rs->swap_rb && !rs->flip &&
         !rs->clear_mode && rs->source_padded_width) {
      /* Total number of tiles (same as for autodisable) */
      cs->RS_KICKER_INPLACE = rs->tile_count;
   }
   cs->source_ts_valid = rs->source_ts_valid;
}

/* modify the clear bits value in the compiled RS state */
static void
etna_modify_rs_clearbits(struct compiled_rs_state *cs, uint32_t clear_bits)
{
   cs->RS_CLEAR_CONTROL &= ~VIVS_RS_CLEAR_CONTROL_BITS__MASK;
   cs->RS_CLEAR_CONTROL |= VIVS_RS_CLEAR_CONTROL_BITS(clear_bits);
}

#define EMIT_STATE(state_name, src_value) \
   etna_coalsence_emit(stream, &coalesce, VIVS_##state_name, src_value)

#define EMIT_STATE_FIXP(state_name, src_value) \
   etna_coalsence_emit_fixp(stream, &coalesce, VIVS_##state_name, src_value)

#define EMIT_STATE_RELOC(state_name, src_value) \
   etna_coalsence_emit_reloc(stream, &coalesce, VIVS_##state_name, src_value)

/* submit RS state, without any processing and no dependence on context
 * except TS if this is a source-to-destination blit. */
static void
etna_submit_rs_state(struct etna_context *ctx,
                     const struct compiled_rs_state *cs)
{
   struct etna_screen *screen = etna_screen(ctx->base.screen);
   struct etna_cmd_stream *stream = ctx->stream;
   struct etna_coalesce coalesce;

   if (cs->RS_KICKER_INPLACE && !cs->source_ts_valid)
      /* Inplace resolve is no-op if TS is not configured */
      return;

   ctx->stats.rs_operations++;

   if (cs->RS_KICKER_INPLACE) {
      etna_cmd_stream_reserve(stream, 6);
      etna_coalesce_start(stream, &coalesce);
      /* 0/1 */ EMIT_STATE(RS_EXTRA_CONFIG, cs->RS_EXTRA_CONFIG);
      /* 2/3 */ EMIT_STATE(RS_SOURCE_STRIDE, cs->RS_SOURCE_STRIDE);
      /* 4/5 */ EMIT_STATE(RS_KICKER_INPLACE, cs->RS_KICKER_INPLACE);
      etna_coalesce_end(stream, &coalesce);
   } else if (screen->specs.pixel_pipes == 1) {
      etna_cmd_stream_reserve(stream, 22);
      etna_coalesce_start(stream, &coalesce);
      /* 0/1 */ EMIT_STATE(RS_CONFIG, cs->RS_CONFIG);
      /* 2   */ EMIT_STATE_RELOC(RS_SOURCE_ADDR, &cs->source[0]);
      /* 3   */ EMIT_STATE(RS_SOURCE_STRIDE, cs->RS_SOURCE_STRIDE);
      /* 4   */ EMIT_STATE_RELOC(RS_DEST_ADDR, &cs->dest[0]);
      /* 5   */ EMIT_STATE(RS_DEST_STRIDE, cs->RS_DEST_STRIDE);
      /* 6/7 */ EMIT_STATE(RS_WINDOW_SIZE, cs->RS_WINDOW_SIZE);
      /* 8/9 */ EMIT_STATE(RS_DITHER(0), cs->RS_DITHER[0]);
      /*10   */ EMIT_STATE(RS_DITHER(1), cs->RS_DITHER[1]);
      /*11 - pad */
      /*12/13*/ EMIT_STATE(RS_CLEAR_CONTROL, cs->RS_CLEAR_CONTROL);
      /*14   */ EMIT_STATE(RS_FILL_VALUE(0), cs->RS_FILL_VALUE[0]);
      /*15   */ EMIT_STATE(RS_FILL_VALUE(1), cs->RS_FILL_VALUE[1]);
      /*16   */ EMIT_STATE(RS_FILL_VALUE(2), cs->RS_FILL_VALUE[2]);
      /*17   */ EMIT_STATE(RS_FILL_VALUE(3), cs->RS_FILL_VALUE[3]);
      /*18/19*/ EMIT_STATE(RS_EXTRA_CONFIG, cs->RS_EXTRA_CONFIG);
      /*20/21*/ EMIT_STATE(RS_KICKER, 0xbeebbeeb);
      etna_coalesce_end(stream, &coalesce);
   } else if (screen->specs.pixel_pipes == 2) {
      etna_cmd_stream_reserve(stream, 34); /* worst case - both pipes multi=1 */
      etna_coalesce_start(stream, &coalesce);
      /* 0/1 */ EMIT_STATE(RS_CONFIG, cs->RS_CONFIG);
      /* 2/3 */ EMIT_STATE(RS_SOURCE_STRIDE, cs->RS_SOURCE_STRIDE);
      /* 4/5 */ EMIT_STATE(RS_DEST_STRIDE, cs->RS_DEST_STRIDE);
      /* 6/7 */ EMIT_STATE_RELOC(RS_PIPE_SOURCE_ADDR(0), &cs->source[0]);
      if (cs->RS_SOURCE_STRIDE & VIVS_RS_SOURCE_STRIDE_MULTI) {
         /*8 */ EMIT_STATE_RELOC(RS_PIPE_SOURCE_ADDR(1), &cs->source[1]);
         /*9 - pad */
      }
      /*10/11*/ EMIT_STATE_RELOC(RS_PIPE_DEST_ADDR(0), &cs->dest[0]);
      if (cs->RS_DEST_STRIDE & VIVS_RS_DEST_STRIDE_MULTI) {
         /*12*/ EMIT_STATE_RELOC(RS_PIPE_DEST_ADDR(1), &cs->dest[1]);
         /*13 - pad */
      }
      /*14/15*/ EMIT_STATE(RS_PIPE_OFFSET(0), cs->RS_PIPE_OFFSET[0]);
      /*16   */ EMIT_STATE(RS_PIPE_OFFSET(1), cs->RS_PIPE_OFFSET[1]);
      /*17 - pad */
      /*18/19*/ EMIT_STATE(RS_WINDOW_SIZE, cs->RS_WINDOW_SIZE);
      /*20/21*/ EMIT_STATE(RS_DITHER(0), cs->RS_DITHER[0]);
      /*22   */ EMIT_STATE(RS_DITHER(1), cs->RS_DITHER[1]);
      /*23 - pad */
      /*24/25*/ EMIT_STATE(RS_CLEAR_CONTROL, cs->RS_CLEAR_CONTROL);
      /*26   */ EMIT_STATE(RS_FILL_VALUE(0), cs->RS_FILL_VALUE[0]);
      /*27   */ EMIT_STATE(RS_FILL_VALUE(1), cs->RS_FILL_VALUE[1]);
      /*28   */ EMIT_STATE(RS_FILL_VALUE(2), cs->RS_FILL_VALUE[2]);
      /*29   */ EMIT_STATE(RS_FILL_VALUE(3), cs->RS_FILL_VALUE[3]);
      /*30/31*/ EMIT_STATE(RS_EXTRA_CONFIG, cs->RS_EXTRA_CONFIG);
      /*32/33*/ EMIT_STATE(RS_KICKER, 0xbeebbeeb);
      etna_coalesce_end(stream, &coalesce);
   } else {
      abort();
   }
}

/* Generate clear command for a surface (non-fast clear case) */
void
etna_rs_gen_clear_surface(struct etna_context *ctx, struct etna_surface *surf,
                          uint32_t clear_value)
{
   struct etna_resource *dst = etna_resource(surf->base.texture);
   uint32_t format = translate_rs_format(surf->base.format);

   if (format == ETNA_NO_MATCH) {
      BUG("etna_rs_gen_clear_surface: Unhandled clear fmt %s", util_format_name(surf->base.format));
      format = RS_FORMAT_A8R8G8B8;
      assert(0);
   }

   /* use tiled clear if width is multiple of 16 */
   bool tiled_clear = (surf->surf.padded_width & ETNA_RS_WIDTH_MASK) == 0 &&
                      (surf->surf.padded_height & ETNA_RS_HEIGHT_MASK) == 0;

   etna_compile_rs_state( ctx, &surf->clear_command, &(struct rs_state) {
      .source_format = format,
      .dest_format = format,
      .dest = dst->bo,
      .dest_offset = surf->surf.offset,
      .dest_stride = surf->surf.stride,
      .dest_padded_height = surf->surf.padded_height,
      .dest_tiling = tiled_clear ? dst->layout : ETNA_LAYOUT_LINEAR,
      .dither = {0xffffffff, 0xffffffff},
      .width = surf->surf.padded_width, /* These must be padded to 16x4 if !LINEAR, otherwise RS will hang */
      .height = surf->surf.padded_height,
      .clear_value = {clear_value},
      .clear_mode = VIVS_RS_CLEAR_CONTROL_MODE_ENABLED1,
      .clear_bits = 0xffff
   });
}

static void
etna_blit_clear_color_rs(struct pipe_context *pctx, struct pipe_surface *dst,
                      const union pipe_color_union *color)
{
   struct etna_context *ctx = etna_context(pctx);
   struct etna_surface *surf = etna_surface(dst);
   uint32_t new_clear_value = etna_clear_blit_pack_rgba(surf->base.format, color->f);

   if (surf->surf.ts_size) { /* TS: use precompiled clear command */
      ctx->framebuffer.TS_COLOR_CLEAR_VALUE = new_clear_value;

      if (VIV_FEATURE(ctx->screen, chipMinorFeatures1, AUTO_DISABLE)) {
         /* Set number of color tiles to be filled */
         etna_set_state(ctx->stream, VIVS_TS_COLOR_AUTO_DISABLE_COUNT,
                        surf->surf.padded_width * surf->surf.padded_height / 16);
         ctx->framebuffer.TS_MEM_CONFIG |= VIVS_TS_MEM_CONFIG_COLOR_AUTO_DISABLE;
      }

      surf->level->ts_valid = true;
      ctx->dirty |= ETNA_DIRTY_TS | ETNA_DIRTY_DERIVE_TS;
   } else if (unlikely(new_clear_value != surf->level->clear_value)) { /* Queue normal RS clear for non-TS surfaces */
      /* If clear color changed, re-generate stored command */
      etna_rs_gen_clear_surface(ctx, surf, new_clear_value);
   }

   etna_submit_rs_state(ctx, &surf->clear_command);

   surf->level->clear_value = new_clear_value;
   resource_written(ctx, surf->base.texture);
   etna_resource(surf->base.texture)->seqno++;
}

static void
etna_blit_clear_zs_rs(struct pipe_context *pctx, struct pipe_surface *dst,
                   unsigned buffers, double depth, unsigned stencil)
{
   struct etna_context *ctx = etna_context(pctx);
   struct etna_surface *surf = etna_surface(dst);
   uint32_t new_clear_value = translate_clear_depth_stencil(surf->base.format, depth, stencil);
   uint32_t new_clear_bits = 0, clear_bits_depth, clear_bits_stencil;

   /* Get the channels to clear */
   switch (surf->base.format) {
   case PIPE_FORMAT_Z16_UNORM:
      clear_bits_depth = 0xffff;
      clear_bits_stencil = 0;
      break;
   case PIPE_FORMAT_X8Z24_UNORM:
   case PIPE_FORMAT_S8_UINT_Z24_UNORM:
      clear_bits_depth = 0xeeee;
      clear_bits_stencil = 0x1111;
      break;
   default:
      clear_bits_depth = clear_bits_stencil = 0xffff;
      break;
   }

   if (buffers & PIPE_CLEAR_DEPTH)
      new_clear_bits |= clear_bits_depth;
   if (buffers & PIPE_CLEAR_STENCIL)
      new_clear_bits |= clear_bits_stencil;
   /* FIXME: when tile status is enabled, this becomes more complex as
    * we may separately clear the depth from the stencil.  In this case,
    * we want to resolve the surface, and avoid using the tile status.
    * We may be better off recording the pending clear operation,
    * delaying the actual clear to the first use.  This way, we can merge
    * consecutive clears together. */
   if (surf->surf.ts_size) { /* TS: use precompiled clear command */
      /* Set new clear depth value */
      ctx->framebuffer.TS_DEPTH_CLEAR_VALUE = new_clear_value;
      if (VIV_FEATURE(ctx->screen, chipMinorFeatures1, AUTO_DISABLE)) {
         /* Set number of depth tiles to be filled */
         etna_set_state(ctx->stream, VIVS_TS_DEPTH_AUTO_DISABLE_COUNT,
                        surf->surf.padded_width * surf->surf.padded_height / 16);
         ctx->framebuffer.TS_MEM_CONFIG |= VIVS_TS_MEM_CONFIG_DEPTH_AUTO_DISABLE;
      }

      surf->level->ts_valid = true;
      ctx->dirty |= ETNA_DIRTY_TS | ETNA_DIRTY_DERIVE_TS;
   } else {
      if (unlikely(new_clear_value != surf->level->clear_value)) { /* Queue normal RS clear for non-TS surfaces */
         /* If clear depth value changed, re-generate stored command */
         etna_rs_gen_clear_surface(ctx, surf, new_clear_value);
      }
      /* Update the channels to be cleared */
      etna_modify_rs_clearbits(&surf->clear_command, new_clear_bits);
   }

   etna_submit_rs_state(ctx, &surf->clear_command);

   surf->level->clear_value = new_clear_value;
   resource_written(ctx, surf->base.texture);
   etna_resource(surf->base.texture)->seqno++;
}

static void
etna_clear_rs(struct pipe_context *pctx, unsigned buffers,
           const union pipe_color_union *color, double depth, unsigned stencil)
{
   struct etna_context *ctx = etna_context(pctx);

   /* Flush color and depth cache before clearing anything.
    * This is especially important when coming from another surface, as
    * otherwise it may clear part of the old surface instead. */
   etna_set_state(ctx->stream, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
   etna_stall(ctx->stream, SYNC_RECIPIENT_RA, SYNC_RECIPIENT_PE);

   /* Preparation: Flush the TS if needed. This must be done after flushing
    * color and depth, otherwise it can result in crashes */
   bool need_ts_flush = false;
   if ((buffers & PIPE_CLEAR_COLOR) && ctx->framebuffer_s.nr_cbufs) {
      struct etna_surface *surf = etna_surface(ctx->framebuffer_s.cbufs[0]);
      if (surf->surf.ts_size)
         need_ts_flush = true;
   }
   if ((buffers & PIPE_CLEAR_DEPTHSTENCIL) && ctx->framebuffer_s.zsbuf != NULL) {
      struct etna_surface *surf = etna_surface(ctx->framebuffer_s.zsbuf);

      if (surf->surf.ts_size)
         need_ts_flush = true;
   }

   if (need_ts_flush)
      etna_set_state(ctx->stream, VIVS_TS_FLUSH_CACHE, VIVS_TS_FLUSH_CACHE_FLUSH);

   /* No need to set up the TS here as RS clear operations (in contrast to
    * resolve and copy) do not require the TS state.
    */
   if (buffers & PIPE_CLEAR_COLOR) {
      for (int idx = 0; idx < ctx->framebuffer_s.nr_cbufs; ++idx) {
         etna_blit_clear_color_rs(pctx, ctx->framebuffer_s.cbufs[idx],
                               &color[idx]);
      }
   }

   /* Flush the color and depth caches before each RS clear operation
    * This fixes a hang on GC600. */
   if (buffers & PIPE_CLEAR_DEPTHSTENCIL && buffers & PIPE_CLEAR_COLOR)
      etna_set_state(ctx->stream, VIVS_GL_FLUSH_CACHE,
                     VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);

   if ((buffers & PIPE_CLEAR_DEPTHSTENCIL) && ctx->framebuffer_s.zsbuf != NULL)
      etna_blit_clear_zs_rs(pctx, ctx->framebuffer_s.zsbuf, buffers, depth, stencil);

   etna_stall(ctx->stream, SYNC_RECIPIENT_RA, SYNC_RECIPIENT_PE);
}

static bool
etna_manual_blit(struct etna_resource *dst, struct etna_resource_level *dst_lev,
                 unsigned int dst_offset, struct etna_resource *src,
                 struct etna_resource_level *src_lev, unsigned int src_offset,
                 const struct pipe_blit_info *blit_info)
{
   void *smap, *srow, *dmap, *drow;
   size_t tile_size;

   assert(src->layout == ETNA_LAYOUT_TILED);
   assert(dst->layout == ETNA_LAYOUT_TILED);
   assert(src->base.nr_samples == 0);
   assert(dst->base.nr_samples == 0);

   tile_size = util_format_get_blocksize(blit_info->src.format) * 4 * 4;

   smap = etna_bo_map(src->bo);
   if (!smap)
      return false;

   dmap = etna_bo_map(dst->bo);
   if (!dmap)
      return false;

   srow = smap + src_offset;
   drow = dmap + dst_offset;

   etna_bo_cpu_prep(src->bo, DRM_ETNA_PREP_READ);
   etna_bo_cpu_prep(dst->bo, DRM_ETNA_PREP_WRITE);

   for (int y = 0; y < blit_info->src.box.height; y += 4) {
      memcpy(drow, srow, tile_size * blit_info->src.box.width);
      srow += src_lev->stride * 4;
      drow += dst_lev->stride * 4;
   }

   etna_bo_cpu_fini(dst->bo);
   etna_bo_cpu_fini(src->bo);

   return true;
}

static inline size_t
etna_compute_tileoffset(const struct pipe_box *box, enum pipe_format format,
                        size_t stride, enum etna_surface_layout layout)
{
   size_t offset;
   unsigned int x = box->x, y = box->y;
   unsigned int blocksize = util_format_get_blocksize(format);

   switch (layout) {
   case ETNA_LAYOUT_LINEAR:
      offset = y * stride + x * blocksize;
      break;
   case ETNA_LAYOUT_MULTI_TILED:
      y >>= 1;
      /* fall-through */
   case ETNA_LAYOUT_TILED:
      assert(!(x & 0x03) && !(y & 0x03));
      offset = (y & ~0x03) * stride + blocksize * ((x & ~0x03) << 2);
      break;
   case ETNA_LAYOUT_MULTI_SUPERTILED:
      y >>= 1;
      /* fall-through */
   case ETNA_LAYOUT_SUPER_TILED:
      assert(!(x & 0x3f) && !(y & 0x3f));
      offset = (y & ~0x3f) * stride + blocksize * ((x & ~0x3f) << 6);
      break;
   default:
      unreachable("invalid resource layout");
   }

   return offset;
}

static inline void
etna_get_rs_alignment_mask(const struct etna_context *ctx,
                           const enum etna_surface_layout layout,
                           unsigned int *width_mask, unsigned int *height_mask)
{
   unsigned int h_align, w_align;

   if (layout & ETNA_LAYOUT_BIT_SUPER) {
      w_align = h_align = 64;
   } else {
      w_align = ETNA_RS_WIDTH_MASK + 1;
      h_align = ETNA_RS_HEIGHT_MASK + 1;
   }

   h_align *= ctx->screen->specs.pixel_pipes;

   *width_mask = w_align - 1;
   *height_mask = h_align -1;
}

static bool
etna_try_rs_blit(struct pipe_context *pctx,
                 const struct pipe_blit_info *blit_info)
{
   struct etna_context *ctx = etna_context(pctx);
   struct etna_resource *src = etna_resource(blit_info->src.resource);
   struct etna_resource *dst = etna_resource(blit_info->dst.resource);
   struct compiled_rs_state copy_to_screen;
   uint32_t ts_mem_config = 0;
   int msaa_xscale = 1, msaa_yscale = 1;

   /* Ensure that the level is valid */
   assert(blit_info->src.level <= src->base.last_level);
   assert(blit_info->dst.level <= dst->base.last_level);

   if (!translate_samples_to_xyscale(src->base.nr_samples, &msaa_xscale, &msaa_yscale, NULL))
      return FALSE;

   /* The width/height are in pixels; they do not change as a result of
    * multi-sampling. So, when blitting from a 4x multisampled surface
    * to a non-multisampled surface, the width and height will be
    * identical. As we do not support scaling, reject different sizes. */
   if (blit_info->dst.box.width != blit_info->src.box.width ||
       blit_info->dst.box.height != blit_info->src.box.height) {
      DBG("scaling requested: source %dx%d destination %dx%d",
          blit_info->src.box.width, blit_info->src.box.height,
          blit_info->dst.box.width, blit_info->dst.box.height);
      return FALSE;
   }

   /* No masks - RS can't copy specific channels */
   unsigned mask = util_format_get_mask(blit_info->dst.format);
   if ((blit_info->mask & mask) != mask) {
      DBG("sub-mask requested: 0x%02x vs format mask 0x%02x", blit_info->mask, mask);
      return FALSE;
   }

   unsigned src_format = etna_compatible_rs_format(blit_info->src.format);
   unsigned dst_format = etna_compatible_rs_format(blit_info->dst.format);
   if (translate_rs_format(src_format) == ETNA_NO_MATCH ||
       translate_rs_format(dst_format) == ETNA_NO_MATCH ||
       blit_info->scissor_enable ||
       blit_info->dst.box.depth != blit_info->src.box.depth ||
       blit_info->dst.box.depth != 1) {
      return FALSE;
   }

   unsigned w_mask, h_mask;

   etna_get_rs_alignment_mask(ctx, src->layout, &w_mask, &h_mask);
   if ((blit_info->src.box.x & w_mask) || (blit_info->src.box.y & h_mask))
      return FALSE;

   etna_get_rs_alignment_mask(ctx, dst->layout, &w_mask, &h_mask);
   if ((blit_info->dst.box.x & w_mask) || (blit_info->dst.box.y & h_mask))
      return FALSE;

   /* Ensure that the Z coordinate is sane */
   if (dst->base.target != PIPE_TEXTURE_CUBE)
      assert(blit_info->dst.box.z == 0);
   if (src->base.target != PIPE_TEXTURE_CUBE)
      assert(blit_info->src.box.z == 0);

   assert(blit_info->src.box.z < src->base.array_size);
   assert(blit_info->dst.box.z < dst->base.array_size);

   struct etna_resource_level *src_lev = &src->levels[blit_info->src.level];
   struct etna_resource_level *dst_lev = &dst->levels[blit_info->dst.level];

   /* we may be given coordinates up to the padded width to avoid
    * any alignment issues with different tiling formats */
   assert((blit_info->src.box.x + blit_info->src.box.width) * msaa_xscale <= src_lev->padded_width);
   assert((blit_info->src.box.y + blit_info->src.box.height) * msaa_yscale <= src_lev->padded_height);
   assert(blit_info->dst.box.x + blit_info->dst.box.width <= dst_lev->padded_width);
   assert(blit_info->dst.box.y + blit_info->dst.box.height <= dst_lev->padded_height);

   unsigned src_offset = src_lev->offset +
                         blit_info->src.box.z * src_lev->layer_stride +
                         etna_compute_tileoffset(&blit_info->src.box,
                                                 blit_info->src.format,
                                                 src_lev->stride,
                                                 src->layout);
   unsigned dst_offset = dst_lev->offset +
                         blit_info->dst.box.z * dst_lev->layer_stride +
                         etna_compute_tileoffset(&blit_info->dst.box,
                                                 blit_info->dst.format,
                                                 dst_lev->stride,
                                                 dst->layout);

   if (src_lev->padded_width <= ETNA_RS_WIDTH_MASK ||
       dst_lev->padded_width <= ETNA_RS_WIDTH_MASK ||
       src_lev->padded_height <= ETNA_RS_HEIGHT_MASK ||
       dst_lev->padded_height <= ETNA_RS_HEIGHT_MASK)
      goto manual;

   /* If the width is not aligned to the RS width, but is within our
    * padding, adjust the width to suite the RS width restriction.
    * Note: the RS width/height are converted to source samples here. */
   unsigned int width = blit_info->src.box.width * msaa_xscale;
   unsigned int height = blit_info->src.box.height * msaa_yscale;
   unsigned int w_align = ETNA_RS_WIDTH_MASK + 1;
   unsigned int h_align = (ETNA_RS_HEIGHT_MASK + 1) * ctx->specs.pixel_pipes;

   if (width & (w_align - 1) && width >= src_lev->width * msaa_xscale && width >= dst_lev->width)
      width = align(width, w_align);

   if (height & (h_align - 1) && height >= src_lev->height * msaa_yscale && height >= dst_lev->height)
      height = align(height, h_align);

   /* The padded dimensions are in samples */
   if (width > src_lev->padded_width ||
       width > dst_lev->padded_width * msaa_xscale ||
       height > src_lev->padded_height ||
       height > dst_lev->padded_height * msaa_yscale ||
       width & (w_align - 1) || height & (h_align - 1))
      goto manual;

   if (src->base.nr_samples > 1) {
      uint32_t msaa_format = translate_msaa_format(src_format);
      assert(msaa_format != ETNA_NO_MATCH);
      ts_mem_config |= VIVS_TS_MEM_CONFIG_COLOR_COMPRESSION | msaa_format;
   }

   /* Always flush color and depth cache together before resolving. This works
    * around artifacts that appear in some cases when scanning out a texture
    * directly after it has been rendered to, such as rendering an animated web
    * page in a QtWebEngine based WebView on GC2000. The artifacts look like
    * the texture sampler samples zeroes instead of texture data in a small,
    * irregular triangle in the lower right of each browser tile quad. Other
    * attempts to avoid these artifacts, including a pipeline stall before the
    * color flush or a TS cache flush afterwards, or flushing multiple times,
    * with stalls before and after each flush, have shown no effect. */
   if (src->base.bind & PIPE_BIND_RENDER_TARGET ||
       src->base.bind & PIPE_BIND_DEPTH_STENCIL) {
      etna_set_state(ctx->stream, VIVS_GL_FLUSH_CACHE,
		     VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
      etna_stall(ctx->stream, SYNC_RECIPIENT_RA, SYNC_RECIPIENT_PE);

      if (src->levels[blit_info->src.level].ts_size &&
          src->levels[blit_info->src.level].ts_valid)
         etna_set_state(ctx->stream, VIVS_TS_FLUSH_CACHE, VIVS_TS_FLUSH_CACHE_FLUSH);
   }

   /* Set up color TS to source surface before blit, if needed */
   bool source_ts_valid = false;
   if (src->levels[blit_info->src.level].ts_size &&
       src->levels[blit_info->src.level].ts_valid) {
      struct etna_reloc reloc;
      unsigned ts_offset =
         src_lev->ts_offset + blit_info->src.box.z * src_lev->ts_layer_stride;

      etna_set_state(ctx->stream, VIVS_TS_MEM_CONFIG,
                     VIVS_TS_MEM_CONFIG_COLOR_FAST_CLEAR | ts_mem_config);

      memset(&reloc, 0, sizeof(struct etna_reloc));
      reloc.bo = src->ts_bo;
      reloc.offset = ts_offset;
      reloc.flags = ETNA_RELOC_READ;
      etna_set_state_reloc(ctx->stream, VIVS_TS_COLOR_STATUS_BASE, &reloc);

      memset(&reloc, 0, sizeof(struct etna_reloc));
      reloc.bo = src->bo;
      reloc.offset = src_lev->offset +
                     blit_info->src.box.z * src_lev->layer_stride;
      reloc.flags = ETNA_RELOC_READ;
      etna_set_state_reloc(ctx->stream, VIVS_TS_COLOR_SURFACE_BASE, &reloc);

      etna_set_state(ctx->stream, VIVS_TS_COLOR_CLEAR_VALUE,
                     src->levels[blit_info->src.level].clear_value);

      source_ts_valid = true;
   } else {
      etna_set_state(ctx->stream, VIVS_TS_MEM_CONFIG, ts_mem_config);
   }
   ctx->dirty |= ETNA_DIRTY_TS;

   /* Kick off RS here */
   etna_compile_rs_state(ctx, &copy_to_screen, &(struct rs_state) {
      .source_format = translate_rs_format(src_format),
      .source_tiling = src->layout,
      .source = src->bo,
      .source_offset = src_offset,
      .source_stride = src_lev->stride,
      .source_padded_width = src_lev->padded_width,
      .source_padded_height = src_lev->padded_height,
      .source_ts_valid = source_ts_valid,
      .dest_format = translate_rs_format(dst_format),
      .dest_tiling = dst->layout,
      .dest = dst->bo,
      .dest_offset = dst_offset,
      .dest_stride = dst_lev->stride,
      .dest_padded_height = dst_lev->padded_height,
      .downsample_x = msaa_xscale > 1,
      .downsample_y = msaa_yscale > 1,
      .swap_rb = translate_rb_src_dst_swap(src->base.format, dst->base.format),
      .dither = {0xffffffff, 0xffffffff}, // XXX dither when going from 24 to 16 bit?
      .clear_mode = VIVS_RS_CLEAR_CONTROL_MODE_DISABLED,
      .width = width,
      .height = height,
      .tile_count = src_lev->layer_stride / 64
   });

   etna_submit_rs_state(ctx, &copy_to_screen);
   resource_written(ctx, &dst->base);
   dst->seqno++;
   dst->levels[blit_info->dst.level].ts_valid = false;
   ctx->dirty |= ETNA_DIRTY_DERIVE_TS;

   return TRUE;

manual:
   if (src->layout == ETNA_LAYOUT_TILED && dst->layout == ETNA_LAYOUT_TILED) {
      if ((src->status & ETNA_PENDING_WRITE) ||
          (dst->status & ETNA_PENDING_WRITE))
         pctx->flush(pctx, NULL, 0);
      return etna_manual_blit(dst, dst_lev, dst_offset, src, src_lev, src_offset, blit_info);
   }

   return FALSE;
}

static void
etna_blit_rs(struct pipe_context *pctx, const struct pipe_blit_info *blit_info)
{
   /* This is a more extended version of resource_copy_region */
   /* TODO Some cases can be handled by RS; if not, fall back to rendering or
    * even CPU copy block of pixels from info->src to info->dst
    * (resource, level, box, format);
    * function is used for scaling, flipping in x and y direction (negative
    * width/height), format conversion, mask and filter and even a scissor rectangle
    *
    * What can the RS do for us:
    *   convert between tiling formats (layouts)
    *   downsample 2x in x and y
    *   convert between a limited number of pixel formats
    *
    * For the rest, fall back to util_blitter
    * XXX this goes wrong when source surface is supertiled. */
   struct etna_context *ctx = etna_context(pctx);
   struct pipe_blit_info info = *blit_info;

   if (info.src.resource->nr_samples > 1 &&
       info.dst.resource->nr_samples <= 1 &&
       !util_format_is_depth_or_stencil(info.src.resource->format) &&
       !util_format_is_pure_integer(info.src.resource->format)) {
      DBG("color resolve unimplemented");
      return;
   }

   if (etna_try_rs_blit(pctx, blit_info))
      return;

   if (util_try_blit_via_copy_region(pctx, blit_info))
      return;

   if (info.mask & PIPE_MASK_S) {
      DBG("cannot blit stencil, skipping");
      info.mask &= ~PIPE_MASK_S;
   }

   if (!util_blitter_is_blit_supported(ctx->blitter, &info)) {
      DBG("blit unsupported %s -> %s",
          util_format_short_name(info.src.resource->format),
          util_format_short_name(info.dst.resource->format));
      return;
   }

   etna_blit_save_state(ctx);
   util_blitter_blit(ctx->blitter, &info);
}

void
etna_clear_blit_rs_init(struct pipe_context *pctx)
{
   DBG("etnaviv: Using RS blit engine");
   pctx->clear = etna_clear_rs;
   pctx->blit = etna_blit_rs;
}
