// license:BSD-3-Clause
// copyright-holders:Carl
// TODO: SCSI, requires NCR5380 BSY IRQs
#include "emu.h"
#include "cpu/i86/i186.h"
#include "machine/wd_fdc.h"
#include "machine/mc68681.h"
#include "bus/rs232/rs232.h"
#include "machine/nscsi_hd.h"
#include "machine/ncr5380n.h"
#include "imagedev/floppy.h"
#include "formats/pc_dsk.h"
#include "formats/naslite_dsk.h"

class lb186_state : public driver_device
{
public:
	lb186_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag)
		, m_maincpu(*this, "maincpu")
		, m_fdc(*this, "fdc")
		, m_scsi(*this, "scsibus:7:ncr5380")
	{
	}

	void lb186(machine_config &config);

private:
	DECLARE_WRITE8_MEMBER(sio_out_w);
	DECLARE_WRITE8_MEMBER(drive_sel_w);
	DECLARE_READ8_MEMBER(scsi_dack_r);
	DECLARE_WRITE8_MEMBER(scsi_dack_w);
	DECLARE_FLOPPY_FORMATS( floppy_formats );
	static void ncr5380(device_t *device);
	void lb186_io(address_map &map);
	void lb186_map(address_map &map);

	required_device<i80186_cpu_device> m_maincpu;
	required_device<wd1772_device> m_fdc;
	required_device<ncr5380n_device> m_scsi;
};

WRITE8_MEMBER(lb186_state::scsi_dack_w)
{
	m_scsi->dma_w(data);
}

READ8_MEMBER(lb186_state::scsi_dack_r)
{
	return m_scsi->dma_r();
}

WRITE8_MEMBER(lb186_state::sio_out_w)
{
	if(!BIT(data, 5))
		m_fdc->soft_reset();
	m_maincpu->tmrin1_w(BIT(data, 3) ? ASSERT_LINE : CLEAR_LINE);
}

WRITE8_MEMBER(lb186_state::drive_sel_w)
{
	m_fdc->dden_w(BIT(data, 5));

	floppy_image_device *floppy;
	char devname[8];
	unsigned int drive = data & 0xf;
	switch(drive)
	{
		case 0:
			return;
		case 1:
			drive = 0;
			break;
		case 2:
			drive = 1;
			break;
		case 4:
			drive = 2;
			break;
		case 8:
			drive = 3;
			break;
		default:
			logerror("More than one drive enabled!\n");
			return;
	}

	sprintf(devname, "%d", drive);
	floppy = m_fdc->subdevice<floppy_connector>(devname)->get_device();
	m_fdc->set_floppy(floppy);
	floppy->ss_w(BIT(data, 4));
}

void lb186_state::lb186_map(address_map &map)
{
	map(0x00000, 0x3ffff).ram(); // fixed 256k for now
	map(0xfc000, 0xfffff).rom().region("bios", 0);
}

void lb186_state::lb186_io(address_map &map)
{
	map.unmap_value_high();
	map(0x1000, 0x101f).rw("duart", FUNC(scn2681_device::read), FUNC(scn2681_device::write)).umask16(0x00ff);
	map(0x1080, 0x108f).rw(m_scsi, FUNC(ncr5380n_device::read), FUNC(ncr5380n_device::write)).umask16(0x00ff);
	map(0x1100, 0x1107).rw(m_fdc, FUNC(wd1772_device::read), FUNC(wd1772_device::write)).umask16(0x00ff);
	map(0x1180, 0x1180).rw(FUNC(lb186_state::scsi_dack_r), FUNC(lb186_state::scsi_dack_w));
	map(0x1200, 0x1200).w(FUNC(lb186_state::drive_sel_w));
}

static void lb186_floppies(device_slot_interface &device)
{
	device.option_add("525dd", FLOPPY_525_DD);
}

void lb186_state::ncr5380(device_t *device)
{
	devcb_base *devcb;
	(void)devcb;
	downcast<ncr5380n_device &>(*device).set_clock(10000000);
	downcast<ncr5380n_device &>(*device).irq_handler().set(":maincpu", FUNC(i80186_cpu_device::int1_w));
	downcast<ncr5380n_device &>(*device).drq_handler().set(":maincpu", FUNC(i80186_cpu_device::drq0_w));
}

static void scsi_devices(device_slot_interface &device)
{
	device.option_add("harddisk", NSCSI_HARDDISK);
	device.option_add_internal("ncr5380", NCR5380N);
}

FLOPPY_FORMATS_MEMBER( lb186_state::floppy_formats )
	FLOPPY_PC_FORMAT,
	FLOPPY_NASLITE_FORMAT
FLOPPY_FORMATS_END

void lb186_state::lb186(machine_config &config)
{
	I80186(config, m_maincpu, 16_MHz_XTAL / 2);
	m_maincpu->set_addrmap(AS_PROGRAM, &lb186_state::lb186_map);
	m_maincpu->set_addrmap(AS_IO, &lb186_state::lb186_io);

	scn2681_device &duart(SCN2681(config, "duart", 3.6864_MHz_XTAL));
	duart.irq_cb().set(m_maincpu, FUNC(i80186_cpu_device::int0_w));
	duart.a_tx_cb().set("rs232_1", FUNC(rs232_port_device::write_txd));
	duart.b_tx_cb().set("rs232_2", FUNC(rs232_port_device::write_txd));
	duart.outport_cb().set(FUNC(lb186_state::sio_out_w));

	rs232_port_device &rs232_1(RS232_PORT(config, "rs232_1", default_rs232_devices, "terminal"));
	rs232_1.rxd_handler().set("duart", FUNC(scn2681_device::rx_a_w));
	rs232_port_device &rs232_2(RS232_PORT(config, "rs232_2", default_rs232_devices, nullptr));
	rs232_2.rxd_handler().set("duart", FUNC(scn2681_device::rx_b_w));

	WD1772(config, m_fdc, 16_MHz_XTAL / 2);
	m_fdc->intrq_wr_callback().set(m_maincpu, FUNC(i80186_cpu_device::int2_w));
	m_fdc->drq_wr_callback().set(m_maincpu, FUNC(i80186_cpu_device::drq0_w));
	FLOPPY_CONNECTOR(config, "fdc:0", lb186_floppies, "525dd", lb186_state::floppy_formats);
	FLOPPY_CONNECTOR(config, "fdc:1", lb186_floppies, nullptr, lb186_state::floppy_formats);
	FLOPPY_CONNECTOR(config, "fdc:2", lb186_floppies, nullptr, lb186_state::floppy_formats);
	FLOPPY_CONNECTOR(config, "fdc:3", lb186_floppies, nullptr, lb186_state::floppy_formats);

	NSCSI_BUS(config, "scsibus");
	NSCSI_CONNECTOR(config, "scsibus:0", scsi_devices, "harddisk");
	NSCSI_CONNECTOR(config, "scsibus:1", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:2", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:3", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:4", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:5", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:6", scsi_devices, nullptr);
	NSCSI_CONNECTOR(config, "scsibus:7", scsi_devices, "ncr5380", true).set_option_machine_config("ncr5380", lb186_state::ncr5380);
}

ROM_START( lb186 )
	ROM_REGION(0x4000, "bios", 0)
	ROM_LOAD16_BYTE("a75515_v3.35.rom", 0x0000, 0x2000, CRC(245824fb) SHA1(b39ed91d421513f5912fdbc290aaa3f1b7d4f1e0))
	ROM_LOAD16_BYTE("a75516_v3.35.rom", 0x0001, 0x2000, CRC(9d9a5e22) SHA1(070be31c622f50508e8cbdb797c79978b6a4b8f6))
ROM_END

COMP( 1985, lb186, 0, 0, lb186, 0, lb186_state, empty_init, "Ampro Computers", "Little Board/186", MACHINE_NO_SOUND_HW )
