/********************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2021 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************

   Fotoxx image edit - Refine menu functions

   m_voodoo1               automatic image enhancement with limited smarts
   m_voodoo2               automatic image enhancement with limited smarts
   m_edit_dist             edit brightness distribution, rebalance bright/dark areas
   m_flatten_dist          flatten brightness distribution
   m_localcon              increase local contrast
   m_gradients             magnify brightness gradients to improve details
   m_gretinex              rescale pixel RGB brightness range - entire image
   m_lretinex              rescale pixel RGB brightness range - within local areas
   m_saturation            adjust saturation based on pixel brightness
   m_color_profile         convert from one color profile to another
   m_vignette              highlight selected image region
   m_area_rescale          rescale while leaving selected areas unchanged

*********************************************************************************/

#define EX extern                                                                //  enable extern declarations
#include "fotoxx.h"                                                              //  (variables in fotoxx.h are refs)

/********************************************************************************/

//  automatic image enhancement without user guidance

float       voodoo1_brdist[256];                                                 //  pixel count per brightness level
int         voodoo_01, voodoo_99;                                                //  1% and 99% brightness levels (0 - 255)
void *      voodoo1_thread(void *);
editfunc    EFvoodoo1;


void m_voodoo1(GtkWidget *, cchar *menu)
{
   F1_help_topic = "voodoo 1";

   Plog(1,"m_voodoo1 \n");

   EFvoodoo1.menuname = "Voodoo 1";
   EFvoodoo1.menufunc = m_voodoo1;
   EFvoodoo1.Farea = 1;                                                          //  select area ignored
   EFvoodoo1.Fscript = 1;                                                        //  scripting supported
   EFvoodoo1.threadfunc = voodoo1_thread;

   if (! edit_setup(EFvoodoo1)) return;                                          //  setup edit
   thread_signal();                                                              //  start update thread
   edit_done(0);                                                                 //  edit done
   return;
}
   

//  thread function - use multiple working threads

void * voodoo1_thread(void *)
{
   void  * voodoo1_wthread(void *arg);

   int         px, py, ii;
   int         ww = E1pxm->ww, hh = E1pxm->hh;
   float       bright1;
   float       *pix1;
   
   for (ii = 0; ii < 256; ii++)                                                  //  clear brightness distribution data
      voodoo1_brdist[ii] = 0;

   for (py = 0; py < hh; py++)                                                   //  compute brightness distribution
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright1 = PIXBRIGHT(pix1);
      voodoo1_brdist[int(bright1)]++;
   }

   for (ii = 1; ii < 256; ii++)                                                  //  cumulative brightness distribution
      voodoo1_brdist[ii] += voodoo1_brdist[ii-1];                                //   0 ... (ww * hh)

   voodoo_01 = 0;
   voodoo_99 = 255;

   for (ii = 0; ii < 256; ii++)                                                  //  compute index values (0 - 255)
   {                                                                             //    for darkest 1% and brightest 1%
      if (voodoo1_brdist[ii] < 0.01 * ww * hh) voodoo_01 = ii;
      if (voodoo1_brdist[ii] < 0.99 * ww * hh) voodoo_99 = ii;
   }

   for (ii = 0; ii < 256; ii++)
      voodoo1_brdist[ii] = voodoo1_brdist[ii]                                    //  multiplier per brightness level
                    / (ww * hh) * 256.0 / (ii + 1);                              //  ( = 1 for a flat distribution)

   do_wthreads(voodoo1_wthread,NWT);

   CEF->Fmods++;
   CEF->Fsaved = 0;
   return 0;
}

void * voodoo1_wthread(void *arg)                                                //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py;
   float       *pix1, *pix3;
   float       bright1, bright2, bright3;
   float       R1, G1, B1, R3, G3, B3;
   float       flat1 = 0.3;                                                      //  strength of distribution flatten
   float       flat2 = 1.0 - flat1;
   float       sat1 = 0.3, sat2;                                                 //  strength of saturation increase
   float       f1, f2;
   float       expand = 256.0 / (voodoo_99 - voodoo_01 + 1);                     //  brightness range expander
   float       max$;

   for (py = index; py < E1pxm->hh; py += NWT)                                   //  voodoo brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];
      G1 = pix1[1];
      B1 = pix1[2];

      bright2 = 0.25 * R1 + 0.65 * G1 + 0.10 * B1;                               //  input brightness, 0 - 256
      bright2 = (bright2 - voodoo_01) * expand;                                  //  expand to clip low / high 1%
      if (bright2 < 0) bright2 = 0;
      if (bright2 > 255) bright2 = 255;

      bright1 = voodoo1_brdist[int(bright2)];                                    //  factor for flat output brightness
      bright3 = flat1 * bright1 + flat2;                                         //  attenuate

      f1 = (256.0 - bright2) / 256.0;                                            //  bright2 = 0 - 256  >>  f1 = 1 - 0
      f2 = 1.0 - f1;                                                             //  prevent banding in bright areas
      bright3 = f1 * bright3 + f2;                                               //  tends to 1.0 for brighter pixels

      R3 = R1 * bright3;                                                         //  blend new and old brightness
      G3 = G1 * bright3;
      B3 = B1 * bright3;

      bright3 = 0.333 * (R3 + G3 + B3);                                          //  mean color brightness
      sat2 = sat1 * (256.0 - bright3) / 256.0;                                   //  bright3 = 0 - 256  >>  sat2 = sat1 - 0
      R3 = R3 + sat2 * (R3 - bright3);                                           //  amplified color, max for dark pixels
      G3 = G3 + sat2 * (G3 - bright3);
      B3 = B3 + sat2 * (B3 - bright3);

      RGBFIX(R3,G3,B3)

      pix3[0] = R3;
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  1-button enhancement via flatten brightness distribution

void m_voodoo2(GtkWidget *, cchar *menu)                                         //  21.40
{
   void flatten_dist_func(int NZ, int flatten, int deband1, int deband2);

   F1_help_topic = "voodoo 2";
   flatten_dist_func(60,40,50,70); 
   return;
}   


/********************************************************************************/

//  edit brightness distribution

namespace edit_dist_names
{
   int      ww, hh;
   float    LC, HC;                                                              //  low, high cutoff levels
   float    LF, MF, HF;                                                          //  low, mid, high flatten parms
   float    LS, MS, HS;                                                          //  low, mid, high stretch parms
   float    BB[1000];                                                            //  adjusted B for input B 0-999

   int    dialog_event(zdialog* zd, cchar *event);
   void   compute_BB();
   void * thread(void *);
   void * wthread(void *);

   editfunc    EFedit_dist;
   char        edit_hist[100];
}


//  menu function

void m_edit_dist(GtkWidget *, cchar *menu)
{
   using namespace edit_dist_names;

   cchar  *title = "Edit Brightness Distribution";

   F1_help_topic = "edit dist";

   Plog(1,"m_edit_dist \n");

   EFedit_dist.menuname = "Edit Dist";
   EFedit_dist.menufunc = m_edit_dist;
   EFedit_dist.FprevReq = 1;                                                     //  preview
   EFedit_dist.Farea = 2;                                                        //  select area usable
   EFedit_dist.Frestart = 1;                                                     //  restart allowed
   EFedit_dist.Fscript = 1;                                                      //  scripting supported 
   EFedit_dist.threadfunc = thread;
   if (! edit_setup(EFedit_dist)) return;                                        //  setup edit

/***
          __________________________________________
         |       Edit Brightness Distribution       |
         |                                          |
         | Low Cutoff   ==========[]==============  |
         | High Cutoff  ==============[]==========  |
         | Low Flatten  =========[]===============  |
         | Mid Flatten  ============[]============  |
         | High Flatten ==============[]==========  |
         | Low Stretch  ================[]========  |
         | Mid Stretch  =============[]===========  |
         | High Stretch =========[]===============  |
         |                                          |
         |                  [Reset] [ OK ] [Cancel] |
         |__________________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,"Reset","OK","Cancel",null);
   EFedit_dist.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");

   zdialog_add_widget(zd,"label","labLC","vb1","Low Cutoff");
   zdialog_add_widget(zd,"label","labHC","vb1","High Cutoff");
   zdialog_add_widget(zd,"label","labLF","vb1","Low Flatten");
   zdialog_add_widget(zd,"label","labMF","vb1","Mid Flatten");
   zdialog_add_widget(zd,"label","labHF","vb1","High Flatten");
   zdialog_add_widget(zd,"label","labLS","vb1","Low Stretch");
   zdialog_add_widget(zd,"label","labMS","vb1","Mid Stretch");
   zdialog_add_widget(zd,"label","labHS","vb1","High Stretch");

   zdialog_add_widget(zd,"hscale2","LC","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","HC","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","LF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","MF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","HF","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","LS","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","MS","vb2","0|1.0|0.002|0","expand");
   zdialog_add_widget(zd,"hscale2","HS","vb2","0|1.0|0.002|0","expand");
   
   zdialog_resize(zd,300,0);
   zdialog_run(zd,dialog_event,"save");                                          //  run dialog - parallel

   LC = HC = LF = MF = HF = LS = MS = HS = 0.0;
   compute_BB();
   
   return;
}


//  dialog event and completion function

int edit_dist_names::dialog_event(zdialog *zd, cchar *event)
{
   using namespace edit_dist_names;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();                                                           //  get full size image
      compute_BB();
      thread_signal();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         LC = HC = LF = MF = HF = LS = MS = HS = 0.0;
         zdialog_stuff(zd,"LC",LC);
         zdialog_stuff(zd,"HC",HC);
         zdialog_stuff(zd,"LF",LF);
         zdialog_stuff(zd,"MF",MF);
         zdialog_stuff(zd,"HF",HF);
         zdialog_stuff(zd,"LS",LS);
         zdialog_stuff(zd,"MS",MS);
         zdialog_stuff(zd,"HS",HS);
         compute_BB();
         edit_reset();
      }
      else if (zd->zstat == 2 && CEF->Fmods) {                                   //  done
         edit_fullsize();                                                        //  get full size image
         compute_BB();
         thread_signal();
         m_RGB_dist(0,"kill");                                                   //  kill RGB distribution graph
         edit_addhist("LC:%.3f HC:%.3f LF:%.3f MF:%.3f HF:%.3f "
                      "LS:%.3f MS:%.3f HS:%.3f",LC,HC,LF,MF,HF,LS,MS,HS);        //  edit params > edit hist               21.50
         edit_done(0);                                                           //  commit edit
      }
      else {
         edit_cancel(0);                                                         //  cancel - discard edit
         m_RGB_dist(0,"kill");                                                   //  kill RGB distribution graph
      }

      return 1;
   }

   if (strmatch(event,"blendwidth"))                                             //  select area blendwidth change
      thread_signal();
   
   if (zstrstr("LC HC LF MF HF LS MS HS apply",event))
   {
      zdialog_fetch(zd,"LC",LC);
      zdialog_fetch(zd,"HC",HC);
      zdialog_fetch(zd,"LF",LF);
      zdialog_fetch(zd,"MF",MF);
      zdialog_fetch(zd,"HF",HF);
      zdialog_fetch(zd,"LS",LS);
      zdialog_fetch(zd,"MS",MS);
      zdialog_fetch(zd,"HS",HS);
      compute_BB();
      thread_signal();
   }

   return 1;
}


//  compute flattened brightness levels for preview size or full size image
//  FB[B] = flattened brightness level for brightness B, scaled 0-1000

void edit_dist_names::compute_BB()
{
   using namespace edit_dist_names;

   int      ii, npix, py, px, iB;
   float    *pix1;
   float    B, LC2, HC2;
   float    FB[1000];
   float    LF2, MF2, HF2, LS2, MS2, HS2;
   float    LFB, MFB, HFB, LSB, MSB, HSB;
   float    LFW, MFW, HFW, LSW, MSW, HSW, TWB;

   ww = E1pxm->ww;
   hh = E1pxm->hh;

   for (ii = 0; ii < 1000; ii++)                                                 //  clear brightness distribution data
      FB[ii] = 0;

   if (sa_stat == 3)                                                             //  process selected area
   {
      for (ii = npix = 0; ii < ww * hh; ii++)
      {
         if (! sa_pixmap[ii]) continue;
         py = ii / ww;
         px = ii - py * ww;
         pix1 = PXMpix(E1pxm,px,py);
         B = 1000.0 * (pix1[0] + pix1[1] + pix1[2]) / 768.0;
         FB[int(B)]++;
         npix++;
      }

      for (ii = 1; ii < 1000; ii++)                                              //  cumulative brightness distribution
         FB[ii] += FB[ii-1];                                                     //   0 ... npix

      for (ii = 0; ii < 1000; ii++)
         FB[ii] = FB[ii] / npix * 999.0;                                         //  flattened brightness level
   }

   else                                                                          //  process whole image
   {
      for (py = 0; py < hh; py++)                                                //  compute brightness distribution
      for (px = 0; px < ww; px++)
      {
         pix1 = PXMpix(E1pxm,px,py);
         B = 1000.0 * (pix1[0] + pix1[1] + pix1[2]) / 768.0;
         FB[int(B)]++;
      }

      for (ii = 1; ii < 1000; ii++)                                              //  cumulative brightness distribution
         FB[ii] += FB[ii-1];                                                     //   0 ... (ww * hh)
   
      for (ii = 0; ii < 1000; ii++)
         FB[ii] = FB[ii] / (ww * hh) * 999.0;                                    //  flattened brightness level
   }

   LC2 = 500 * LC;                                                               //  low cutoff, 0 ... 500
   HC2 = 1000 - 500 * HC;                                                        //  high cutoff, 1000 ... 500

   for (iB = 0; iB < 1000; iB++)                                                 //  loop brightness 0 - 1000   
   {
      B = iB;

      if (LC2 > 0 || HC2 < 1000) {                                               //  stretch to cutoff limits
         if (B < LC2) B = 0;
         else if (B > HC2) B = 999;
         else B = 1000.0 * (B - LC2) / (HC2 - LC2);
      }
      
      LF2 = LF * (1000 - B) / 1000;                                              //  low flatten  LF ... 0
      LF2 = LF2 * LF2;
      LFB = LF2 * FB[iB] + (1.0 - LF2) * B;
      
      LS2 = LS * (1000 - B) / 1000;                                              //  low stretch  LS ... 0
      LS2 = LS2 * LS2;
      LSB = B * (1 - LS2);
      
      MF2 = MF * (500 - fabsf(B - 500)) / 500;                                   //  mid flatten  0 ... MF ... 0
      MF2 = sqrtf(MF2);
      MFB = MF2 * FB[iB] + (1.0 - MF2) * B;

      MS2 = MS * (B - 500) / 500;                                                //  mid stretch  -MS ... 0 ... MS
      MSB = B * (1 + 0.5 * MS2);
      
      HF2 = HF * B / 1000;                                                       //  high flatten  0 ... HF
      HF2 = HF2 * HF2;
      HFB = HF2 * FB[iB] + (1.0 - HF2) * B;

      HS2 = HS * B / 1000;                                                       //  high stretch  0 ... HS
      HS2 = HS2 * HS2;
      HSB = B * (1 + HS2);
      
      LFW = fabsf(B - LFB) / (B + 1);                                            //  weight of each component
      LSW = fabsf(B - LSB) / (B + 1);
      MFW = fabsf(B - MFB) / (B + 1);
      MSW = fabsf(B - MSB) / (B + 1);
      HFW = fabsf(B - HFB) / (B + 1);
      HSW = fabsf(B - HSB) / (B + 1);
      
      TWB = LFW + LSW + MFW + MSW + HFW + HSW;                                   //  add weighted components
      if (TWB == 0) BB[iB] = B;
      else BB[iB] = (LFW * LFB + LSW * LSB
                   + MFW * MFB + MSW * MSB 
                   + HFW * HFB + HSW * HSB) / TWB;
   }
   
   return;
}


//  adjust brightness distribution thread function

void * edit_dist_names::thread(void *)
{
   using namespace edit_dist_names;

   do_wthreads(wthread,NWT);

   CEF->Fmods++;                                                                 //  image modified, not saved
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return 0;
}


//  worker thread for each CPU processor core

void * edit_dist_names::wthread(void *arg)
{
   using namespace edit_dist_names;

   int         index = *((int *) (arg));
   int         iB, px, py, ii, dist = 0;
   float       B, *pix1, *pix3;
   float       dold, dnew;
   float       R1, G1, B1, R3, G3, B3;
   float       max$;
   
   for (py = index; py < E1pxm->hh; py += NWT)                                   //  flatten brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      
      B = (pix1[0] + pix1[1] + pix1[2]) / 768.0 * 1000.0;                        //  pixel brightness scaled 0-1000
      iB = int(B);
      if (B > 0) B = BB[iB] / B;
      
      R1 = pix1[0];
      G1 = pix1[1];
      B1 = pix1[2];
      
      R3 = B * R1;
      G3 = B * G1;
      B3 = B * B1;

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         dnew = sa_blendfunc(dist);                                              //    blend edge
         dold = 1.0 - dnew;
         R3 = dnew * R3 + dold * R1;
         G3 = dnew * G3 + dold * G1;
         B3 = dnew * B3 + dold * B1;
      }
      
      RGBFIX(R3,G3,B3)

      pix3[0] = R3;
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  flatten brightness distribution based on the distribution of nearby zones

namespace flatten_dist_names
{
   int      Zinit;                                 //  zone initialization needed
   float    flatten;                               //  flatten amount, 0 - 100
   float    deband1, deband2;                      //  deband dark, bright, 0 - 100
   int      Eww, Ehh;                              //  image dimensions
   int      NZ;                                    //  no. of image zones
   int      pNZ;                                   //  prior image zones
   int      Zsize, Zrows, Zcols;                   //  zone parameters
   float    *Br;                                   //  Br[py][px]  pixel brightness
   int      *Zxlo, *Zylo, *Zxhi, *Zyhi;            //  Zxlo[ii] etc.  zone ii pixel range
   int      *Zcen;                                 //  Zcen[ii][2]  zone ii center (py,px)
   int16    *Zn;                                   //  Zn[py][px][9]  9 nearest zones for pixel: 0-999 (-1 = none)
   uint8    *Zw;                                   //  Zw[py][px][9]  zone weights for pixel: 0-100 = 1.0
   float    *Zff;                                  //  Zff[ii][1000]  zone ii flatten factors

   editfunc    EFflatten_dist;

   int    dialog_event(zdialog* zd, cchar *event);
   void   doflatten();
   void   calczones();
   void   initzones();
   void * thread(void *);
   void * wthread(void *);
}


//  menu function

void m_flatten_dist(GtkWidget *, cchar *menu)
{
   using namespace flatten_dist_names;

   cchar  *title = "Flatten Brightness Distribution";

   F1_help_topic = "flatten dist";
   
   Plog(1,"m_flatten_dist \n");

   if (! curr_file) return; 

   Eww = Fpxb->ww;
   Ehh = Fpxb->hh;

   EFflatten_dist.menuname = "Flatten Dist";
   EFflatten_dist.menufunc = m_flatten_dist;
   EFflatten_dist.FprevReq = 1;                                                  //  use preview image
   EFflatten_dist.Farea = 2;                                                     //  select area usable
   EFflatten_dist.Frestart = 1;                                                  //  restartable
   EFflatten_dist.Fpaintedits = 1;                                               //  use with paint edits OK
   EFflatten_dist.Fscript = 1;                                                   //  scripting supported
   EFflatten_dist.threadfunc = thread;
   if (! edit_setup(EFflatten_dist)) return;                                     //  setup edit

   Eww = E0pxm->ww;
   Ehh = E0pxm->hh;

/***
          ______________________________________
         |   Flatten Brightness Distribution    |
         |                                      |
         | Zones [ 123 ]  [Apply]               |
         | Flatten  =========[]===============  |
         | Deband Dark =========[]============  |
         | Deband Bright ==========[]=========  |
         |                                      |
         |             [Reset] [ OK ] [Cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,"Reset","OK","Cancel",null);
   EFflatten_dist.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labreg","hb1","Zones","space=5");
   zdialog_add_widget(zd,"zspin","zones","hb1","1|999|1|100");                   //  zone range 1-999
   zdialog_add_widget(zd,"button","apply","hb1","Apply","space=10");
   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","labflatten","hb2","Flatten","space=5");
   zdialog_add_widget(zd,"hscale2","flatten","hb2","0|100|1|0","expand");
   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"label","labdeband1","hb3","Deband Dark","space=5");
   zdialog_add_widget(zd,"hscale2","deband1","hb3","0|100|1|0","expand");
   zdialog_add_widget(zd,"hbox","hb4","dialog");
   zdialog_add_widget(zd,"label","labdeband2","hb4","Deband Bright","space=5");
   zdialog_add_widget(zd,"hscale2","deband2","hb4","0|100|1|0","expand");
   
   zdialog_resize(zd,300,0);
   zdialog_run(zd,dialog_event,"save");                                          //  run dialog - parallel

   NZ = pNZ = 40;                                                                //  default zone count
   calczones();                                                                  //  adjust to fit image
   flatten = deband1 = deband2 = 0;                                              //  dialog controls = neutral
   Zinit = 1;                                                                    //  zone initialization needed
   Br = 0;                                                                       //  no memory allocated
   return;
}


//  dialog event and completion function

int flatten_dist_names::dialog_event(zdialog *zd, cchar *event)
{
   using namespace flatten_dist_names;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      Zinit = 1;
      doflatten();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog alive
         zdialog_stuff(zd,"flatten",0);
         zdialog_stuff(zd,"deband1",0);
         zdialog_stuff(zd,"deband2",0);
         edit_reset();
         return 1;
      }

      else if (zd->zstat == 2) {                                                 //  done
         edit_fullsize();                                                        //  get full size image
         Zinit = 1;                                                              //  recalculate zones
         doflatten();                                                            //  flatten full image
         edit_addhist("Zones:%d flatten:%.0f deb.D:%.0f deb.B:%.0f",             //  edit parms > edit hist                21.50
                                 NZ,flatten,deband1,deband2);
         edit_done(0);                                                           //  commit edit
      }

      else edit_cancel(0);                                                       //  discard edit

      if (Br) {
         zfree(Br);                                                              //  free memory
         zfree(Zn);
         zfree(Zw);
         zfree(Zxlo);
         zfree(Zylo);
         zfree(Zxhi);
         zfree(Zyhi);
         zfree(Zcen);
         zfree(Zff);
         Br = 0;
      }

      return 1;
   }

   if (strmatch(event,"apply")) {                                                //  [apply]  (also from script)
      dialog_event(zd,"zones");
      dialog_event(zd,"flatten");
      dialog_event(zd,"deband1");
      dialog_event(zd,"deband2");
      doflatten();
   }

   if (zstrstr("blendwidth paint",event))                                        //  area blendwidth change or mouse paint
      doflatten();

   if (strmatch(event,"zones")) {                                                //  get zone count input
      zdialog_fetch(zd,"zones",NZ);
      if (NZ == pNZ) return 1;
      calczones();                                                               //  adjust to fit image
      zdialog_stuff(zd,"zones",NZ);                                              //  update dialog
      Zinit = 1;                                                                 //  zone initialization needed
   }

   if (strmatch(event,"flatten")) {
      zdialog_fetch(zd,"flatten",flatten);                                       //  get slider values
      doflatten();
   }

   if (strmatch(event,"deband1")) {
      zdialog_fetch(zd,"deband1",deband1);
      doflatten();
   }

   if (strmatch(event,"deband2")) {
      zdialog_fetch(zd,"deband2",deband2);
      doflatten();
   }

   return 1;
}


//  perform the flatten calculations and update the image

void flatten_dist_names::doflatten()
{
   using namespace flatten_dist_names;

   thread_signal();
   thread_wait();                                                                //  no overlap window update and threads
   return;
}


//  recalculate zone count based on what fits the image dimensions
//  done only when the user zone count input changes
//  outputs: NZ, Zrows, Zcols

void flatten_dist_names::calczones()
{
   using namespace flatten_dist_names;

   int      gNZ, dNZ;

   Eww = E1pxm->ww;                                                              //  image dimensions
   Ehh = E1pxm->hh;

   gNZ = NZ;                                                                     //  new zone count goal
   dNZ = NZ - pNZ;                                                               //  direction of change

   while (true)
   {
      Zsize = sqrtf(Eww * Ehh / gNZ);                                            //  approx. zone size
      Zrows = Ehh / Zsize + 0.5;                                                 //  get appropriate rows and cols
      Zcols = Eww / Zsize + 0.5;
      if (Zrows < 1) Zrows = 1;
      if (Zcols < 1) Zcols = 1;
      NZ = Zrows * Zcols;                                                        //  NZ matching rows x cols

      if (dNZ > 0 && NZ <= pNZ) {                                                //  no increase, try again
         if (NZ >= 999) break;
         gNZ++;
         continue;
      }

      if (dNZ < 0 && NZ >= pNZ) {                                                //  no decrease, try again
         if (NZ <= 20) break;
         gNZ--;
         continue;
      }

      if (dNZ == 0) break;
      if (dNZ > 0 && NZ > pNZ) break;
      if (dNZ < 0 && NZ < pNZ) break;
   }

   pNZ = NZ;                                                                     //  final zone count
   dNZ = 0;
   return;
}


//  build up the zones data when NZ or the image size changes
//  (preview or full size image)

void flatten_dist_names::initzones()
{
   using namespace flatten_dist_names;

   int      px, py, cx, cy;
   int      rx, ii, jj, kk;
   int      zww, zhh, row, col;
   float    *pix1, bright;
   float    weight[9], sumweight;
   float    D, Dthresh;

   if (Br) {
      zfree(Br);                                                                 //  free memory
      zfree(Zn);
      zfree(Zw);
      zfree(Zxlo);
      zfree(Zylo);
      zfree(Zxhi);
      zfree(Zyhi);
      zfree(Zcen);
      zfree(Zff);
      Br = 0;
   }

   Eww = E1pxm->ww;                                                              //  image dimensions
   Ehh = E1pxm->hh;

   Br = (float *) zmalloc(Eww * Ehh * sizeof(float),"flatten_dist");             //  allocate memory
   Zn = (int16 *) zmalloc(Eww * Ehh * 9 * sizeof(int16),"flatten_dist");
   Zw = (uint8 *) zmalloc(Eww * Ehh * 9 * sizeof(uint8),"flatten_dist");
   Zxlo = (int *) zmalloc(NZ * sizeof(int),"flatten_dist");
   Zylo = (int *) zmalloc(NZ * sizeof(int),"flatten_dist");
   Zxhi = (int *) zmalloc(NZ * sizeof(int),"flatten_dist");
   Zyhi = (int *) zmalloc(NZ * sizeof(int),"flatten_dist");
   Zcen = (int *) zmalloc(NZ * 2 * sizeof(int),"flatten_dist");
   Zff = (float *) zmalloc(NZ * 1000 * sizeof(float),"flatten_dist");

   for (py = 0; py < Ehh; py++)                                                  //  get initial pixel brightness levels
   for (px = 0; px < Eww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright = PIXBRIGHT(pix1);
      ii = py * Eww + px;
      Br[ii] = bright;
   }

   zww = Eww / Zcols;                                                            //  actual zone size
   zhh = Ehh / Zrows;

   for (row = 0; row < Zrows; row++)
   for (col = 0; col < Zcols; col++)                                             //  set low/high bounds per zone
   {
      ii = row * Zcols + col;
      Zxlo[ii] = col * zww;
      Zylo[ii] = row * zhh;
      Zxhi[ii] = Zxlo[ii] + zww;
      Zyhi[ii] = Zylo[ii] + zhh;
      Zcen[2*ii] = Zylo[ii] + zhh/2;
      Zcen[2*ii+1] = Zxlo[ii] + zww/2;
   }

   for (ii = 0; ii < NZ; ii++)                                                   //  compute brightness distributiion
   {                                                                             //    for each zone
      for (jj = 0; jj < 1000; jj++)
         Zff[1000*ii+jj] = 0;

      for (py = Zylo[ii]; py < Zyhi[ii]; py++)                                   //  brightness distribution
      for (px = Zxlo[ii]; px < Zxhi[ii]; px++)
      {
         pix1 = PXMpix(E1pxm,px,py);
         kk = PIXBRIGHT(pix1);
         if (kk > 255) kk = 255;
         bright = 3.906 * kk;                                                    //  1000 / 256 * kk
         Zff[1000*ii+int(bright)]++;
      }

      for (jj = 1; jj < 1000; jj++)                                              //  cumulative brightness distribution
         Zff[1000*ii+jj] += Zff[1000*ii+jj-1];

      for (jj = 0; jj < 1000; jj++)                                              //  multiplier per brightness level
         Zff[1000*ii+jj] = Zff[1000*ii+jj] / (zww*zhh) * 1000 / (jj+1);          //    to make distribution flat
   }

   for (py = 0; py < Ehh; py++)                                                  //  set 9 nearest zones per pixel
   for (px = 0; px < Eww; px++)
   {
      rx = (py * Eww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      row = py / zhh;                                                            //  zone containing pixel
      col = px / zww;

      ii = 0;
      for (jj = row-1; jj <= row+1; jj++)                                        //  loop 3x3 surrounding zones
      for (kk = col-1; kk <= col+1; kk++)
      {
         if (jj >= 0 && jj < Zrows && kk >= 0 && kk < Zcols)                     //  zone is not off the edge
            Zn[rx+ii] = jj * Zcols + kk;
         else Zn[rx+ii] = -1;                                                    //  edge row/col: missing neighbor
         ii++;
      }
   }

   if (zww < zhh)                                                                //  pixel to zone distance threshold
      Dthresh = 1.5 * zww;                                                       //  area influence = 0 beyond this distance
   else Dthresh = 1.5 * zhh;

   for (py = 0; py < Ehh; py++)                                                  //  set zone weights per pixel
   for (px = 0; px < Eww; px++)
   {
      rx = (py * Eww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      for (ii = 0; ii < 9; ii++)                                                 //  distance to each zone center
      {
         jj = Zn[rx+ii];
         if (jj == -1) {                                                         //  missing zone
            weight[ii] = 0;                                                      //  weight = 0
            continue;
         }
         cy = Zcen[2*jj];
         cx = Zcen[2*jj+1];
         D = sqrtf((py-cy)*(py-cy) + (px-cx)*(px-cx));                           //  distance from pixel to zone center
         D = D / Dthresh;
         if (D > 1) D = 1;                                                       //  zone influence reaches zero at Dthresh
         weight[ii] = 1.0 - D;
      }

      sumweight = 0;
      for (ii = 0; ii < 9; ii++)                                                 //  zone weights based on distance
         sumweight += weight[ii];

      for (ii = 0; ii < 9; ii++)                                                 //  normalize weights, sum = 1.0
         Zw[rx+ii] = 100 * weight[ii] / sumweight;                               //  0-100 = 1.0
   }

   return;
}


//  adjust brightness distribution thread function

void * flatten_dist_names::thread(void *)
{
   using namespace flatten_dist_names;

   if (Zinit) initzones();                                                       //  reinitialize zones
   Zinit = 0;

   get_edit_pixels_init(NWT,0);                                                  //  initz. pixel loop                     21.44
   do_wthreads(wthread,NWT);

   if (! flatten) CEF->Fmods = 0;                                                //  no modification
   else {
      CEF->Fmods++;                                                              //  image modified, not saved
      CEF->Fsaved = 0;
   }

   if (! Fpaintedits) Fpaint2();                                                 //  update window

   return 0;
}


//  worker thread for each CPU processor core

void * flatten_dist_names::wthread(void *arg)
{
   using namespace flatten_dist_names;

   int         index = *((int *) (arg));
   int         px, py, rx, ii, jj, Fend;
   float       *pix1, *pix3;
   float       fnew1, fnew2, fnew, fold;
   float       R1, G1, B1, R3, G3, B3, R9, G9, B9;
   float       FF, weight, bright;
   float       blend, max$;

   while (true)
   {
      Fend = get_edit_pixels(index,px,py,blend);                                 //  21.44
      if (Fend) break;
      if (blend == 0) continue;

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];
      G1 = pix1[1];
      B1 = pix1[2];

      R3 = pix3[0];                                                              //  current output image RGB
      G3 = pix3[1];
      B3 = pix3[2];

      bright = 0.976 * R1 + 2.539 * G1 + 0.39 * B1;                              //  brightness scaled 0-999.9
      
      rx = (py * Eww + px) * 9;                                                  //  index for 9 nearest zones to px/py

      FF = 0;
      for (ii = 0; ii < 9; ii++) {                                               //  loop 9 nearest zones
         weight = Zw[rx+ii];
         if (weight > 0) {                                                       //  0-100 = 1.0
            jj = Zn[rx+ii];
            FF += 0.01 * weight * Zff[1000*jj+int(bright)];                      //  sum weight * flatten factor
         }
      }

      R9 = FF * R1;                                                              //  fully flattened brightness
      G9 = FF * G1;
      B9 = FF * B1;
      
      fnew1 = 1 - 0.01 * deband1 * 0.001 * (1000 - bright);                      //  attenuate dark pixels
      fnew2 = 1 - 0.01 * deband2 * 0.001 * bright;                               //  attenuate bright pixels
      fnew = fnew1 * fnew2;

      fnew = fnew * 0.01 * flatten;                                              //  amount to flatten, 0 to 1
      fold = 1.0 - fnew;                                                         //  amount to retain, 1 to 0

      R9 = fnew * R9 + fold * R1;                                                //  blend new and old brightness
      G9 = fnew * G9 + fold * G1;
      B9 = fnew * B9 + fold * B1;

      RGBFIX(R9,G9,B9)

      if (Fpaintedits)                                                           //  gradual edit within mouse circle      21.44
      {
         if (blend > 0)
         {                                                                       //  increase edit 
            R3 = blend * R9 + (1-blend) * R3;
            G3 = blend * G9 + (1-blend) * G3;
            B3 = blend * B9 + (1-blend) * B3;
         }
         
         else if (blend < 0)
         {                                                                       //  decrease edit
            R3 = -blend * R1 + (1+blend) * R3;
            G3 = -blend * G1 + (1+blend) * G3;
            B3 = -blend * B1 + (1+blend) * B3;
         }
      }

      else                                                                       //  full edit for image or area           21.44
      {
         R3 = blend * R9 + (1-blend) * R1;
         G3 = blend * G9 + (1-blend) * G1;
         B3 = blend * B9 + (1-blend) * B1;
      }

      pix3[0] = R3;
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


//  callable function for flatten brightness distribution
//  used by Voodoo2

void flatten_dist_func(int _NZ, int _flatten, int _deband1, int _deband2)
{
   using namespace flatten_dist_names;

   if (! curr_file) return; 

   Eww = Fpxb->ww;
   Ehh = Fpxb->hh;

   EFflatten_dist.menuname = "Voodoo 2";
   EFflatten_dist.menufunc = m_flatten_dist;
   EFflatten_dist.Farea = 2;                                                     //  select area usable
   EFflatten_dist.Frestart = 1;                                                  //  restartable
   EFflatten_dist.threadfunc = thread;

   if (! edit_setup(EFflatten_dist)) return;                                     //  setup edit

   Eww = E0pxm->ww;
   Ehh = E0pxm->hh;

   NZ = pNZ = _NZ;                                                               //  args --> namespace
   flatten = _flatten;
   deband1 = _deband1;
   deband2 = _deband2;

   Zinit = 1;                                                                    //  zone initialization needed
   Br = 0;                                                                       //  no memory allocated

   calczones();                                                                  //  adjust zones to fit image
   doflatten();                                                                  //  flatten image
   edit_done(0);

   zfree(Br);                                                                    //  free memory
   zfree(Zn);
   zfree(Zw);
   zfree(Zxlo);
   zfree(Zylo);
   zfree(Zxhi);
   zfree(Zyhi);
   zfree(Zcen);
   zfree(Zff);
   Br = 0;
   
   return;
}


/********************************************************************************

   Local Contrast function
   Rescale pixel brightness based on surrounding area mean brightness.

   loop all pixels
      B = pixel brightness value
      M = average of pixels in neighborhood (zone) 
      R = M / B
      B = 255 * POW(B/255,R)     new pixel brightness

*********************************************************************************/

namespace localcon_names
{
   int         Zrad1, Zrad2;                                                     //  inner, outer zone radius
   float       *Zbrite1, *Zbrite2;                                               //  inner, outer zone mean brightness
   float       Fpower;                                                           //  function power, 0-1
   float       Fbrite;                                                           //  brightness increase, 0-1
   float       Fcolor;                                                           //  color increase, 0-1
   int         Fnewrad, Fnewpower, Fnewbrite, Fnewcolor;                         //  flags, new inputs available
   int         Eww, Ehh;                                                         //  E1/E3 image size
   editfunc    EFlocalcon;                                                       //  edit function struct
}


//  menu function

void m_localcon(GtkWidget *, cchar *menu)                                        //  21.40 - reduce halo effects 
{
   using namespace localcon_names;

   int    localcon_dialog_event(zdialog *zd, cchar *event);
   void * localcon_thread(void *);

   F1_help_topic = "local contrast";
   
   Plog(1,"m_localcon \n");

   cchar *title = "Local Contrast";
   
   EFlocalcon.menuname = "Local Contrast";
   EFlocalcon.menufunc = m_localcon;
   EFlocalcon.Farea = 2;                                                         //  select area usable
   EFlocalcon.Frestart = 1;                                                      //  allow restart
   EFlocalcon.Fscript = 1;                                                       //  scripting supported
   EFlocalcon.Fpaintedits = 1;                                                   //  use with paint edits OK
   EFlocalcon.threadfunc = localcon_thread;                                      //  thread function

   if (! edit_setup(EFlocalcon)) return;                                         //  setup edit

   Eww = E1pxm->ww;                                                              //  E1/E3 image size
   Ehh = E1pxm->hh;
   
   int cc = Eww * Ehh * sizeof(float);
   Zbrite1 = (float *) zmalloc(cc,"localcon");                                   //  memory for brightness means, 2 zones
   Zbrite2 = (float *) zmalloc(cc,"localcon");
   
/***
          ________________________________
         |        Local Contrast          |
         |                                |
         | Power   =============[]======  |
         | Radius  ========[]===========  |
         | Brighten  ===========[]======  |
         | Color  ==========[]==========  |
         |                                |
         |        [Reset] [ OK ] [Cancel] |
         |________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,"Reset","OK","Cancel",null);
   EFlocalcon.zd = zd;

   zdialog_add_widget(zd,"hbox","hbpow","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labpow","hbpow","Power","space=5");
   zdialog_add_widget(zd,"hscale2","power","hbpow","0|1|0.01|0","expand");       //  power range, 0-1
   zdialog_add_widget(zd,"hbox","hbrad","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labrad","hbrad","Radius","space=5");
   zdialog_add_widget(zd,"hscale2","Zrad2","hbrad","20|100|2|10","expand");      //  Zrad2 range 20-100 pixels
   zdialog_add_widget(zd,"hbox","hbbrite","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labbrite","hbbrite","Brighten","space=5");
   zdialog_add_widget(zd,"hscale2","brite","hbbrite","0|1|0.01|0","expand");     //  brighten range, 0-1
   zdialog_add_widget(zd,"hbox","hbcolor","dialog",0,"space=2");
   zdialog_add_widget(zd,"label","labcolor","hbcolor","Color","space=5");
   zdialog_add_widget(zd,"hscale2","color","hbcolor","0|1|0.01|0","expand");     //  color range, 0-1

   Zrad2 = 20;                                                                   //  initial values
   Zrad1 = Zrad2 / 2;
   Fnewrad = 1;
   Fpower = 0;
   Fnewpower = 0;
   Fbrite = 0;
   Fnewbrite = 0;
   Fcolor = 0;
   Fnewcolor = 0;
   
   zdialog_resize(zd,200,0);
   zdialog_run(zd,localcon_dialog_event,"save");                                 //  run dialog - parallel

   thread_signal();
   return;
}


//  dialog event and completion function

int localcon_dialog_event(zdialog *zd, cchar *event)
{
   using namespace localcon_names;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel

   if (strmatch(event,"apply")) event = "power";                                 //  from script

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog alive
         edit_reset();
         zdialog_stuff(zd,"power",0);
         zdialog_stuff(zd,"brite",0);
         zdialog_stuff(zd,"color",0);
         zdialog_stuff(zd,"Zrad2",20);
         Zrad2 = 20;                                                             //  initial values
         Zrad1 = Zrad2 / 2;
         Fnewrad = 1;
         Fpower = 0;
         Fnewpower = 0;
         Fbrite = 0;
         Fnewbrite = 0;
         Fcolor = 0;
         Fnewcolor = 0;
         thread_signal();
         return 1;
      }
      else if (zd->zstat == 2) {                                                 //  done - commit edit
         edit_addhist("pow:%.3f rad:%d brite:%.3f color:%.3f",
                                 Fpower,Zrad2,Fbrite,Fcolor);                    //  edit parms > edit hist                21.50
         edit_done(0);
      }
      else edit_cancel(0);                                                       //  discard edit
      zfree(Zbrite1);                                                            //  free memory
      zfree(Zbrite2);
      return 1;
   }

   if (strmatch(event,"Zrad2")) {                                                //  radius input
      zdialog_fetch(zd,"Zrad2",Zrad2);                                           //  outer radius
      Zrad1 = Zrad2 / 2;                                                         //  inner radius
      Fnewrad = 1;
   }

   if (strmatch(event,"power")) {                                                //  power input
      zdialog_fetch(zd,"power",Fpower);
      Fnewpower = 1;
   }

   if (strmatch(event,"brite")) {                                                //  brighten input
      zdialog_fetch(zd,"brite",Fbrite);
      Fnewbrite = 1;
   }

   if (strmatch(event,"color")) {                                                //  color input
      zdialog_fetch(zd,"color",Fcolor);
      Fnewcolor = 1;
   }
   
   if (zstrstr("blendwidth paint",event))                                        //  area blendwidth change or mouse paint
      Fnewpower = 1;

   thread_signal();
   return 1;
}


//  main thread function

void * localcon_thread(void *)
{
   using namespace localcon_names;

   void * localcon_wthread_rad1(void *);
   void * localcon_wthread_rad2(void *);
   void * localcon_wthread_power(void *);

   if (! Fnewrad && ! Fnewpower && ! Fnewbrite && ! Fnewcolor)                   //  no new inputs
      return 0;
   
   while (Fnewrad) 
   {
      progress_reset(Eww * Ehh * 3);                                             //  initz. progress counter
      Fnewrad = 0;
      get_edit_pixels_init(NWT,0);                                               //  initz. pixel loop                     21.44
      do_wthreads(localcon_wthread_rad1,NWT);
      get_edit_pixels_init(NWT,0);
      do_wthreads(localcon_wthread_rad2,NWT);
      progress_reset(0);
      if (! Fpaintedits) Fpaint2();
      Fnewpower++;
   }
   
   while (Fnewpower) {
      Fnewpower = 0;
      get_edit_pixels_init(NWT,0);
      do_wthreads(localcon_wthread_power,NWT);
      if (! Fpaintedits) Fpaint2();
   }

   while (Fnewbrite) {
      Fnewbrite = 0;
      get_edit_pixels_init(NWT,0);
      do_wthreads(localcon_wthread_power,NWT);
      if (! Fpaintedits) Fpaint2();
   }

   while (Fnewcolor) {
      Fnewcolor = 0;
      get_edit_pixels_init(NWT,0);
      do_wthreads(localcon_wthread_power,NWT);
      if (! Fpaintedits) Fpaint2();
   }

   CEF->Fmods++;                                                                 //  image modified
   CEF->Fsaved = 0;                                                              //  not saved

   return 0;
}


//  worker thread function
//  Compute mean brightness for zone around each pixel in brightness map image.
//  Compute for both Zrad1 (inner) and Zrad2 (outer) zones.

void * localcon_wthread_rad1(void *arg)
{
   using namespace localcon_names;

   int      index = *((int *) arg);
   int      px1, py1, px2, py2, px3;
   int      pxlo, pxhi, pylo, pyhi;
   int      ii, step;
   float    *pix;
   double   sum = 0;
   int      count = 0;
   
   step = 1;

   for (py1 = index; py1 < Ehh; py1 += NWT)
   for (px1 = 0; px1 < Eww; px1++)   
   {
      pylo = py1 - Zrad1;                                                        //  inner zone y-range
      if (pylo < 0) pylo = 0;
      pyhi = py1 + Zrad1;
      if (pyhi > Ehh) pyhi = Ehh;

      pxlo = px1 - Zrad1;                                                        //  inner zone x-range
      if (pxlo < 0) pxlo = 0;
      pxhi = px1 + Zrad1;
      if (pxhi > Eww) pxhi = Eww;

      if (px1 == 0) {                                                            //  start new row, first zone for row
         sum = count = 0;
         for (py2 = pylo; py2 < pyhi; py2 += step)
         for (px2 = pxlo; px2 < pxhi; px2 += step) {
            if (sa_stat == 3) {                                                  //  exclude pixels outside select area    22.1
               ii = Eww * py2 + px2;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px2,py2);                                         //  accumulate brightness sum
            sum += 0.001302 * (pix[0] + pix[1] + pix[2]);
            count++;
         }
         continue;                                                               //  next x column, next zone
      }
      
      if (pxlo > 0) {
         px3 = pxlo - 1;                                                         //  subtract px3 column from zone
         for (py2 = pylo; py2 < pyhi; py2 += step) {                             //  (left column removed from zone)
            if (sa_stat == 3) {                                                  //  22.1
               ii = Eww * py2 + px3;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px3,py2);
            sum = sum - 0.001302 * (pix[0] + pix[1] + pix[2]);
            count--;
         }
      }
      
      if (pxhi < Eww) {                                                          //  add px3 column to zone
         px3 = pxhi;                                                             //  (right column added to zone)
         for (py2 = pylo; py2 < pyhi; py2 += step) {
            if (sa_stat == 3) {                                                  //  22.1
               ii = Eww * py2 + px3;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px3,py2);
            sum = sum + 0.001302 * (pix[0] + pix[1] + pix[2]);
            count++;
         }
      }
      
      ii = py1 * Eww + px1;
      if (count > 0) Zbrite1[ii] = sum / count;                                  //  zone mean brightness
      else Zbrite1[ii] = 0;
      
      progress_add(index,1);                                                     //  track progress
   }

   return 0;                                                                     //  exit thread
}

void * localcon_wthread_rad2(void *arg)
{
   using namespace localcon_names;

   int      index = *((int *) arg);
   int      px1, py1, px2, py2, px3;
   int      pxlo, pxhi, pylo, pyhi;
   int      ii, step;
   float    *pix;
   double   sum = 0;
   int      count = 0;
   
   step = 1;
   if (Zrad2 > 50) step = 2;

   for (py1 = index; py1 < Ehh; py1 += NWT)
   for (px1 = 0; px1 < Eww; px1++)   
   {
      pylo = py1 - Zrad2;
      if (pylo < 0) pylo = 0;
      pyhi = py1 + Zrad2;
      if (pyhi > Ehh) pyhi = Ehh;

      pxlo = px1 - Zrad2;
      if (pxlo < 0) pxlo = 0;
      pxhi = px1 + Zrad2;
      if (pxhi > Eww) pxhi = Eww;

      if (px1 == 0) {
         sum = count = 0;
         for (py2 = pylo; py2 < pyhi; py2 += step)
         for (px2 = pxlo; px2 < pxhi; px2 += step) {
            if (sa_stat == 3) {
               ii = Eww * py2 + px2;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px2,py2);
            sum += 0.001302 * (pix[0] + pix[1] + pix[2]);
            count++;
         }
         continue;
      }
      
      if (pxlo > 0) {
         px3 = pxlo - 1;
         for (py2 = pylo; py2 < pyhi; py2 += step) {
            if (sa_stat == 3) {
               ii = Eww * py2 + px3;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px3,py2);
            sum = sum - 0.001302 * (pix[0] + pix[1] + pix[2]);
            count--;
         }
      }
      
      if (pxhi < Eww) {
         px3 = pxhi;
         for (py2 = pylo; py2 < pyhi; py2 += step) {
            if (sa_stat == 3) {
               ii = Eww * py2 + px3;
               if (sa_pixmap[ii] < 1) continue;
            }
            pix = PXMpix(E1pxm,px3,py2);
            sum = sum + 0.001302 * (pix[0] + pix[1] + pix[2]);
            count++;
         }
      }
      
      ii = py1 * Eww + px1;
      if (count > 0) Zbrite2[ii] = sum / count;
      else Zbrite2[ii] = 0;
      
      progress_add(index,2);
   }

   return 0;                                                                     //  exit thread
}


//  worker thread function
//  calculate new pixel brightness levels based on
//    pixel brightness compared to local area brightness

void * localcon_wthread_power(void *arg)
{
   using namespace localcon_names;

   int      index = *((int *) arg);
   int      px1, py1, Fend, ii;
   float    Bp, Bz1, Bz2, Blo, Bhi, BR;
   float    R1, G1, B1, R3, G3, B3, R9, G9, B9;
   float    *pix1, *pix3;
   float    pixbrite;
   float    blend, max$;
   
   while (true)
   {
      Fend = get_edit_pixels(index,px1,py1,blend);                               //  21.44
      if (Fend) break;
      if (blend == 0) continue;

      pix1 = PXMpix(E1pxm,px1,py1);                                              //  input pixel
      pix3 = PXMpix(E3pxm,px1,py1);                                              //  output pixel

      R1 = pix1[0];                                                              //  input image RGB values
      G1 = pix1[1];
      B1 = pix1[2];
      
      R3 = pix3[0];                                                              //  current output image RGB
      G3 = pix3[1];
      B3 = pix3[2];

      Bp = 0.001302 * (R1 + G1 + B1);                                            //  pixel brightness, 0-1
      if (Bp == 0) continue;
      
      ii = py1 * Eww + px1;
      Bz1 = Zbrite1[ii];                                                         //  inner zone mean brightness, 0-1
      Bz2 = Zbrite2[ii];                                                         //  outer zone mean brightness, 0-1
      
      if (Bz1 <= Bz2) { Blo = Bz1; Bhi = Bz2; }
      else { Blo = Bz2; Bhi = Bz1; }
      
      if (Bp <= Blo) BR = Blo / Bp;
      else if (Bp >= Bhi) BR = Bhi / Bp;
      else BR = 1.0;

      BR = pow(Bp,BR*BR);                                                        //  new pixel brightness

      if (BR < 1 && Fbrite > 0)                                                  //  brighten darker areas
         BR = BR + Fbrite * pow(1.0-BR,0.3);                                     //  Fbrite, 0-1
      
      R9 = BR * R1;                                                              //  modified RGB values
      G9 = BR * G1;
      B9 = BR * B1;

      if (Fcolor) {                                                              //  + saturation
         pixbrite = 0.333 * (R9 + G9 + B9);                                      //  pixel brightness, 0-255
         R9 = R9 + Fcolor * (R9 - pixbrite);                                     //  Fcolor, 0-1
         G9 = G9 + Fcolor * (G9 - pixbrite);
         B9 = B9 + Fcolor * (B9 - pixbrite);
      }
      
      RGBFIX(R9,G9,B9);

      if (Fpaintedits)                                                           //  gradual edit within mouse circle      21.44
      {
         if (blend > 0)
         {                                                                       //  increase edit 
            blend = Fpower * blend;
            R3 = blend * R9 + (1-blend) * R3;
            G3 = blend * G9 + (1-blend) * G3;
            B3 = blend * B9 + (1-blend) * B3;
         }
         
         else if (blend < 0)
         {                                                                       //  decrease edit
            R3 = -blend * R1 + (1+blend) * R3;
            G3 = -blend * G1 + (1+blend) * G3;
            B3 = -blend * B1 + (1+blend) * B3;
         }
      }

      else                                                                       //  full edit for image or area           21.44
      {
         blend = Fpower * blend;
         R3 = blend * R9 + (1-blend) * R1;
         G3 = blend * G9 + (1-blend) * G1;
         B3 = blend * B9 + (1-blend) * B1;
      }

      pix3[0] = R3;                                                              //  set output pixel
      pix3[1] = G3;
      pix3[2] = B3;
   }
   
   return 0;                                                                     //  exit thread
}


/********************************************************************************

   Magnify Gradients function
   improve detail by magnifying brightness gradients

   methodology:
   get brightness gradients for each pixel in 4 directions: SE SW NE NW
   amplify gradients using the edit curve: new gradient = F(old gradient)
   integrate 4 new brightness surfaces from the amplified gradients:
     - pixel brightness = prior pixel brightness + amplified gradient
     - the Amplify control varies amplification from zero to maximum
   new pixel brightness = average from 4 calculated brightness surfaces

*********************************************************************************/

namespace gradients_names
{
   float       *brmap1, *brmap2;
   float       *brmap4[4];
   int         Fnewmap;
   int         contrast99;
   float       amplify;
   int         Eww, Ehh;
   editfunc    EFgradients;

   void   gradients_initz(zdialog *zd);
   int    gradients_dialog_event(zdialog *zd, cchar *event);
   void   gradients_curvedit(int);
   void * gradients_thread(void *);
   void * gradients_wthread1(void *arg);
   void * gradients_wthread2(void *arg);

}

//  menu function

void m_gradients(GtkWidget *, cchar *menu)
{
   using namespace gradients_names;

   cchar    *title = "Magnify Gradients";
   
   F1_help_topic = "gradients";

   Plog(1,"m_gradients \n");

   EFgradients.menuname = "Gradients";
   EFgradients.menufunc = m_gradients;
   EFgradients.Farea = 2;                                                        //  select area usable
   EFgradients.Frestart = 1;                                                     //  restart allowed
   EFgradients.Fpaintedits = 1;                                                  //  use with paint edits OK
   EFgradients.Fscript = 1;                                                      //  scripting supported
   EFgradients.threadfunc = gradients_thread;
   if (! edit_setup(EFgradients)) return;                                        //  setup: no preview, select area OK

/***
          ________________________________
         |      Magnify Gradients         |
         |  ____________________________  |
         | |                            | |
         | |                            | |
         | |    curve drawing area      | |
         | |                            | |
         | |____________________________| |
         |  low       contrast      high  |
         |                                |
         |  Amplify ========[]==========  |
         |                                |
         |  Curve File: [Open] [Save]     |
         |                                |
         |                [ OK ] [Cancel] |
         |________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,"OK","Cancel",null);
   EFgradients.zd = zd;

   zdialog_add_widget(zd,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcL","hb1","low","space=4");
   zdialog_add_widget(zd,"label","labcM","hb1","Contrast","expand");
   zdialog_add_widget(zd,"label","labcH","hb1","high","space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcon","hb2","Amplify","space=5");
   zdialog_add_widget(zd,"hscale","amplify","hb2","0|1|0.005|0","expand");
   zdialog_add_widget(zd,"label","ampval","hb2",0,"space=5");

   zdialog_add_widget(zd,"hbox","hbcf","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labcf","hbcf","Curve File","space=5");
   zdialog_add_widget(zd,"button","loadcurve","hbcf","Open","space=5");
   zdialog_add_widget(zd,"button","savecurve","hbcf","Save","space=5");

   GtkWidget *frame = zdialog_gtkwidget(zd,"frame");                             //  set up curve edit
   spldat *sd = splcurve_init(frame,gradients_curvedit);
   EFgradients.sd = sd;

   sd->Nspc = 1;
   sd->fact[0] = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 2;                                                               //  initial curve anchor points
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.20;
   sd->apx[0][1] = 0.99;
   sd->apy[0][1] = 0.20;

   splcurve_generate(sd,0);                                                      //  generate curve data

   gradients_initz(zd);                                                          //  initialize brightness maps
   
   zdialog_resize(zd,250,300);
   zdialog_run(zd,gradients_dialog_event,"save");                                //  run dialog - parallel

   amplify = 0;
   Fnewmap = 1;
   thread_signal();

   return;
}


//  initialization for new image file
//  set brmap1[] and contrast99

void gradients_names::gradients_initz(zdialog *zd)
{
   using namespace gradients_names;

   int         ii, cc, px, py;
   float       b1, *pix1;
   int         jj, sum, limit, condist[100];

   Eww = E1pxm->ww;                                                              //  image dimensions
   Ehh = E1pxm->hh;

   cc = Eww * Ehh * sizeof(float);                                               //  allocate brightness map memory
   brmap1 = (float *) zmalloc(cc,"gradients");
   brmap2 = (float *) zmalloc(cc,"gradients");
   for (ii = 0; ii < 4; ii++)
      brmap4[ii] = (float *) zmalloc(cc,"gradients");

   for (py = 0; py < Ehh; py++)                                                  //  map initial image brightness
   for (px = 0; px < Eww; px++)
   {
      ii = py * Eww + px;
      pix1 = PXMpix(E1pxm,px,py);
      b1 = 0.333 * (pix1[0] + pix1[1] + pix1[2]);                                //  pixel brightness 0-255.9
      if (b1 < 1) b1 = 1;
      brmap1[ii] = b1;
   }

   for (ii = 0; ii < 100; ii++)
      condist[ii] = 0;

   for (py = 1; py < Ehh; py++)                                                  //  map contrast distribution
   for (px = 1; px < Eww; px++)
   {
      ii = py * Eww + px;
      jj = 0.388 * fabsf(brmap1[ii] - brmap1[ii-1]);                             //  horiz. contrast, ranged 0 - 99
      if (jj > 99) jj = 99;                                                      //  trap bad image data
      condist[jj]++;
      jj = 0.388 * fabsf(brmap1[ii] - brmap1[ii-Eww]);                           //  vertical
      if (jj > 99) jj = 99;
      condist[jj]++;
   }

   sum = 0;
   limit = 0.99 * 2 * (Eww-1) * (Ehh-1);                                         //  find 99th percentile contrast

   for (ii = 0; ii < 100; ii++) {
      sum += condist[ii];
      if (sum > limit) break;
   }

   contrast99 = 255.0 * ii / 100.0;                                              //  0 to 255
   if (contrast99 < 4) contrast99 = 4;                                           //  rescale low-contrast image

   return;
}


//  dialog event and completion callback function

int gradients_names::gradients_dialog_event(zdialog *zd, cchar *event)
{
   using namespace gradients_names;

   spldat   *sd = EFgradients.sd;    
   char     text[8];
   char     *file, *pp;
   FILE     *fid;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  cancel

   if (zd->zstat) {                                                              //  dialog complete
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      zfree(brmap1);                                                             //  free memory
      zfree(brmap2);
      brmap1 = brmap2 = 0;
      for (int ii = 0; ii < 4; ii++) {
         zfree(brmap4[ii]);
         brmap4[ii] = 0;
      }
      return 1;
   }
   
   if (strmatch(event,"apply"))                                                  //  from script
      gradients_dialog_event(zd,"amplify");

   if (zstrstr("blendwidth paint",event))                                        //  area blend change or mouse paint
      thread_signal();                       

   if (strmatch(event,"amplify")) {                                              //  slider value
      zdialog_fetch(zd,"amplify",amplify);
      snprintf(text,8,"%.2f",amplify);                                           //  numeric feedback
      zdialog_stuff(zd,"ampval",text);
      Fnewmap = 1;
      thread_signal();                                                           //  trigger update thread
   }

   if (strmatch(event,"loadcurve"))                                              //  load saved curve
   {
      file = zgetfile("load curve from a file",MWIN,"file",saved_curves_folder);
      if (! file) return 1;
      fid = fopen(file,"r");
      zfree(file);
      if (! fid) return 1;
      splcurve_load(sd,fid);
      fclose(fid);
      Fnewmap = 1;
      thread_signal();
      return 1;
   }

   if (strmatch(event,"savecurve"))                                              //  save curve to file
   {
      file = zgetfile("save curve to a file",MWIN,"save",saved_curves_folder);
      if (! file) return 1;
      pp = zstrdup(file,"gradients",8);
      zfree(file);
      file = pp;
      pp = strrchr(file,'/');                                                    //  force .curve extension
      if (pp) pp = strrchr(pp,'.');
      if (pp) strcpy(pp,".curve");
      else strcat(file,".curve");
      fid = fopen(file,"w");
      zfree(file);
      if (! fid) return 1;
      splcurve_save(sd,fid);
      fclose(fid);
      return 1;
   }

   return 1;
}


//  this function is called when the curve is edited

void gradients_names::gradients_curvedit(int)
{
   Fnewmap = 1;
   thread_signal();
   return;
}


//  thread function

void * gradients_names::gradients_thread(void *)
{
   using namespace gradients_names;

   if (Fnewmap)                                                                  //  recalculate gradients
      do_wthreads(gradients_wthread1,4);
   Fnewmap = 0;

   get_edit_pixels_init(NWT,0);                                                  //  initz. pixel loop                     21.44
   do_wthreads(gradients_wthread2,NWT);                                          //  update image

   CEF->Fmods++;
   CEF->Fsaved = 0;
   if (amplify == 0) CEF->Fmods = 0;

   if (! Fpaintedits) Fpaint2();                                                 //  update window
   return 0;
}


//  calculate brmap2[] and brmap4[] from brmap1[], 
//    contrast curve, amplify, contrast99

void * gradients_names::gradients_wthread1(void *arg)
{
   using namespace gradients_names;

   int         bii;
   int         ii, kk, pii;
   int         px, px1, px2, pxinc;
   int         py, py1, py2, pyinc;
   float       b1, b4, xval, yval, grad;
   float       amp, con99;
   spldat      *sd = EFgradients.sd;    

   con99 = contrast99;                                                           //  99th percentile contrast
   con99 = 1.0 / con99;                                                          //  inverted
   amp = pow(amplify,0.5);                                                       //  amplification, 0 to 1

   bii = *((int *) arg);                                                         //  thread 0...3

   if (bii == 0) {                                                               //  direction SE
      px1 = 1; 
      px2 = Eww; 
      pxinc = 1;
      py1 = 1; 
      py2 = Ehh; 
      pyinc = 1;
      pii = - 1 - Eww;
   }

   else if (bii == 1) {                                                          //  direction SW
      px1 = Eww-2; 
      px2 = 0; 
      pxinc = -1;
      py1 = 1; 
      py2 = Ehh; 
      pyinc = 1;
      pii = + 1 - Eww;
   }

   else if (bii == 2) {                                                          //  direction NE
      px1 = 1; 
      px2 = Eww; 
      pxinc = 1;
      py1 = Ehh-2; 
      py2 = 0; 
      pyinc = -1;
      pii = - 1 + Eww;
   }

   else {   /* bii == 3 */                                                       //  direction NW
      px1 = Eww-2; 
      px2 = 0; 
      pxinc = -1;
      py1 = Ehh-2; 
      py2 = 0; 
      pyinc = -1;
      pii = + 1 + Eww;
   }

   for (ii = 0; ii < Eww * Ehh; ii++)                                            //  initial brightness map
      brmap4[bii][ii] = brmap1[ii];

   for (py = py1; py != py2; py += pyinc)                                        //  loop all image pixels
   for (px = px1; px != px2; px += pxinc)
   {
      ii = py * Eww + px;

      b1 = brmap1[ii];                                                           //  this pixel brightness
      grad = b1 - brmap1[ii+pii];                                                //   - prior pixel --> gradient

      xval = fabsf(grad) * con99;                                                //  gradient scaled 0 to 1+
      kk = 1000.0 * xval;
      if (kk > 999) kk = 999;
      yval = 1.0 + 5.0 * sd->yval[0][kk];                                        //  amplifier = 1...6
      grad = grad * yval;                                                        //  magnified gradient

      b4 = brmap4[bii][ii+pii] + grad;                                           //  pixel brightness = prior + gradient
      b4 = (1.0 - amp) * b1 + amp * b4;                                          //  apply amplifier: b4 range = b1 --> b4

      brmap4[bii][ii] = b4;                                                      //  new pixel brightness
   }

   for (py = bii; py < Ehh; py += 4)                                             //  loop all image pixels
   for (px = 0; px < Eww; px++)
   {
      ii = py * Eww + px;

      b4 = brmap4[0][ii] + brmap4[1][ii]                                         //  new brightness = average of four
         + brmap4[2][ii] + brmap4[3][ii];                                        //    calculated brightness surfaces
      b4 = 0.25 * b4;

      if (b4 < 1) b4 = 1;

      brmap2[ii] = b4;
   }

   return 0;
}


//  calculate new pixel brightness values from brightness maps

void * gradients_names::gradients_wthread2(void *arg)
{
   using namespace gradients_names;

   float       *pix1, *pix3;
   int         index, ii, px, py, Fend;
   float       b1, b2, bf;
   float       R1, G1, B1, R3, G3, B3, R9, G9, B9;;
   float       blend, max$;

   index = *((int *) arg);

   while (true)
   {
      Fend = get_edit_pixels(index,px,py,blend);                                 //  21.44
      if (Fend) break;
      if (blend == 0) continue;

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];                                                              //  input RGB
      G1 = pix1[1];
      B1 = pix1[2];

      R3 = pix3[0];                                                              //  current output image RGB
      G3 = pix3[1];
      B3 = pix3[2];

      ii = py * Eww + px;
      b1 = brmap1[ii];                                                           //  initial pixel brightness
      b2 = brmap2[ii];                                                           //  calculated new brightness

      bf = b2 / b1;                                                              //  brightness ratio

      R9 = bf * R1;                                                              //  output RGB
      G9 = bf * G1;
      B9 = bf * B1;
      
      RGBFIX(R9,G9,B9)
      
      if (Fpaintedits)                                                           //  gradual edit within mouse circle      21.44
      {
         if (blend > 0)
         {                                                                       //  increase edit 
            R3 = blend * R9 + (1-blend) * R3;
            G3 = blend * G9 + (1-blend) * G3;
            B3 = blend * B9 + (1-blend) * B3;
         }
         
         else if (blend < 0)
         {                                                                       //  decrease edit
            R3 = -blend * R1 + (1+blend) * R3;
            G3 = -blend * G1 + (1+blend) * G3;
            B3 = -blend * B1 + (1+blend) * B3;
         }
      }

      else                                                                       //  full edit for image or area           21.44
      {
         R3 = blend * R9 + (1-blend) * R1;
         G3 = blend * G9 + (1-blend) * G1;
         B3 = blend * B9 + (1-blend) * B1;
      }

      pix3[0] = R3;
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  Global Retinex function
//  Rescale RGB values based on entire image - eliminate color caste and reduce fog/haze.

namespace gretinex_names 
{
   editfunc    EFgretinex;                                                       //  edit function data
   int         E3ww, E3hh;
   cchar       *thread_command;
   VOL int     Fcancel;

   float       Rdark, Gdark, Bdark;
   float       Rbrite, Gbrite, Bbrite;
   float       Rmpy, Gmpy, Bmpy;
   float       pRdark, pGdark, pBdark;                                           //  prior values 
   float       pRbrite, pGbrite, pBbrite;
   float       pRmpy, pGmpy, pBmpy;
   float       blend, reducebright;
   int         Fbrightpoint, Fdarkpoint;
}


//  menu function

void m_gretinex(GtkWidget *, cchar *menu)
{
   using namespace gretinex_names;

   int    gretinex_dialog_event(zdialog *zd, cchar *event);
   void   gretinex_mousefunc();
   void * gretinex_thread(void *);

   F1_help_topic = "global retinex";

   Plog(1,"m_gretinex \n");

   EFgretinex.menuname = "Global Retinex";
   EFgretinex.menufunc = m_gretinex;
   EFgretinex.Farea = 2;                                                         //  select area usable
   EFgretinex.Frestart = 1;                                                      //  allow restart
   EFgretinex.Fscript = 1;                                                       //  scripting supported
   EFgretinex.threadfunc = gretinex_thread;                                      //  thread function
   EFgretinex.mousefunc = gretinex_mousefunc;                                    //  mouse function

   if (! edit_setup(EFgretinex)) return;                                         //  setup edit

   E3ww = E3pxm->ww;                                                             //  image size
   E3hh = E3pxm->hh;
   E9pxm = 0;
   
/***
          ______________________________________
         |           Global Retinex             |
         |                                      |
         |               Red  Green  Blue  All  |
         |  Dark Point:  [__]  [__]  [__]  [__] |        0 ... 255      neutral point is 0
         | Bright Point: [__]  [__]  [__]  [__] |        0 ... 255      neutral point is 255
         |  Multiplyer:  [__]  [__]  [__]  [__] |        0 ... 5        neutral point is 1
         |                                      |
         | [auto] brightness rescale            |
         | [x] bright point  [x] dark point     |        mouse click spots
         | blend: =======[]===================  |
         | reduce bright: ==========[]========  |
         |                                      |
         |             [reset] [ OK ] [cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new("Global Retinex",Mwin,"Reset","OK","Cancel",null); 
   EFgretinex.zd = zd;
   
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","space","hb1","","space=2");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=2");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=2");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=2");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"homog");
   zdialog_add_widget(zd,"label","space","hb1","","space=2");
   zdialog_add_widget(zd,"vbox","vb5","hb1",0,"homog");
   
   zdialog_add_widget(zd,"label","space","vb1","");
   zdialog_add_widget(zd,"label","labdark","vb1","Dark Point");
   zdialog_add_widget(zd,"label","labbrite","vb1","Bright Point");
   zdialog_add_widget(zd,"label","labmpy","vb1","Multiplyer");

   zdialog_add_widget(zd,"label","labred","vb2","Red");
   zdialog_add_widget(zd,"zspin","Rdark","vb2","0|255|1|0","size=3");
   zdialog_add_widget(zd,"zspin","Rbrite","vb2","0|255|1|255","size=3");
   zdialog_add_widget(zd,"zspin","Rmpy","vb2","0.1|5|0.01|1","size=3");

   zdialog_add_widget(zd,"label","labgreen","vb3","Green");
   zdialog_add_widget(zd,"zspin","Gdark","vb3","0|255|1|0","size=3");
   zdialog_add_widget(zd,"zspin","Gbrite","vb3","0|255|1|255","size=3");
   zdialog_add_widget(zd,"zspin","Gmpy","vb3","0.1|5|0.01|1","size=3");

   zdialog_add_widget(zd,"label","labred","vb4","Blue");
   zdialog_add_widget(zd,"zspin","Bdark","vb4","0|255|1|0","size=3");
   zdialog_add_widget(zd,"zspin","Bbrite","vb4","0|255|1|255","size=3");
   zdialog_add_widget(zd,"zspin","Bmpy","vb4","0.1|5|0.01|1","size=3");

   zdialog_add_widget(zd,"label","laball","vb5","All");
   zdialog_add_widget(zd,"zspin","dark+-","vb5","-1|+1|1|0","size=3");
   zdialog_add_widget(zd,"zspin","brite+-","vb5","-1|+1|1|0","size=3");
   zdialog_add_widget(zd,"zspin","mpy+-","vb5","-1|+1|1|0","size=3");

   zdialog_add_widget(zd,"hbox","hbauto","dialog");
   zdialog_add_widget(zd,"button","autoscale","hbauto","Auto","space=3");
   zdialog_add_widget(zd,"label","labauto","hbauto","brightness rescale","space=5");

   zdialog_add_widget(zd,"hbox","hbclicks","dialog");
   zdialog_add_widget(zd,"check","brightpoint","hbclicks","click bright point","space=3");
   zdialog_add_widget(zd,"check","darkpoint","hbclicks","click dark point","space=5");
   
   zdialog_add_widget(zd,"hbox","hbb","dialog");
   zdialog_add_widget(zd,"label","labb","hbb","blend","space=5");
   zdialog_add_widget(zd,"hscale","blend","hbb","0|1.0|0.01|1.0","expand");
   zdialog_add_widget(zd,"hbox","hbrd","dialog");
   zdialog_add_widget(zd,"label","labrd","hbrd","reduce bright","space=5");
   zdialog_add_widget(zd,"hscale","reduce bright","hbrd","0|1.0|0.01|0.0","expand");

   zdialog_run(zd,gretinex_dialog_event,"save");                                 //  run dialog - parallel
   zdialog_send_event(zd,"reset");
   return;
}


//  dialog event and completion function

int gretinex_dialog_event(zdialog *zd, cchar *event)
{
   using namespace gretinex_names;
   
   void  gretinex_mousefunc();

   int      adddark, addbrite, addmpy;
   int      Fchange = 0;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"focus")) return 1;                                        //  stop loss of button event (?)
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   if (strmatch(event,"reset")) zd->zstat = 1;                                   //  initz. 
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         Rdark = Gdark = Bdark = 0;                                              //  set neutral values
         Rbrite = Gbrite = Bbrite = 255;
         pRdark = pGdark = pBdark = 0;                                           //  prior values = same
         pRbrite = pGbrite = pBbrite = 255;                                      //  (no change)
         Rmpy = Gmpy = Bmpy = 1.0;
         Fbrightpoint = Fdarkpoint = 0;
         blend = 1.0;
         reducebright = 0.0;
         Fcancel = 0;

         zdialog_stuff(zd,"Rdark",0);                                            //  stuff neutral values into dialog
         zdialog_stuff(zd,"Gdark",0);
         zdialog_stuff(zd,"Bdark",0);
         zdialog_stuff(zd,"Rbrite",255);
         zdialog_stuff(zd,"Gbrite",255);
         zdialog_stuff(zd,"Bbrite",255);
         zdialog_stuff(zd,"Rmpy",1.0);
         zdialog_stuff(zd,"Gmpy",1.0);
         zdialog_stuff(zd,"Bmpy",1.0);
         zdialog_stuff(zd,"brightpoint",0);
         zdialog_stuff(zd,"darkpoint",0);

         edit_reset();
         if (E9pxm) PXM_free(E9pxm);                                             //  free memory
         E9pxm = 0;
         return 1;
      }

      else if (zd->zstat == 2) {                                                 //  done
         edit_done(0);                                                           //  commit edit
         if (E9pxm) PXM_free(E9pxm);                                             //  free memory
         E9pxm = 0;
         return 1;
      }

      else {
         Fcancel = 1;                                                            //  kill thread
         edit_cancel(0);                                                         //  discard edit
         if (E9pxm) PXM_free(E9pxm);                                             //  free memory
         E9pxm = 0;
         return 1;
      }
   }

   Fbrightpoint = Fdarkpoint = 0;                                                //  disable mouse

   if (strmatch(event,"autoscale"))                                              //  auto set dark and bright points
   {
      edit_reset();

      thread_command = "autoscale";                                              //  get dark/bright limits from
      thread_signal();                                                           //    darkest/brightest pixels
      thread_wait();
      if (Fcancel) return 1;

      zdialog_stuff(zd,"Rdark",Rdark);                                           //  update dialog widgets
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
      zdialog_stuff(zd,"Rbrite",Rbrite);
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
      zdialog_stuff(zd,"Rmpy",1.0);
      zdialog_stuff(zd,"Gmpy",1.0);
      zdialog_stuff(zd,"Bmpy",1.0);
      zdialog_stuff(zd,"brightpoint",0);
      zdialog_stuff(zd,"darkpoint",0);

      pRdark = Rdark;                                                            //  prior values = same
      pGdark = Gdark;
      pBdark = Bdark;
      pRbrite = Rbrite; 
      pGbrite = Gbrite; 
      pBbrite = Bbrite;
      pRmpy = Rmpy;
      pGmpy = Gmpy;
      pBmpy = Bmpy;
      return 1;
   }
   
   if (strmatch(event,"brightpoint")) {
      zdialog_fetch(zd,"brightpoint",Fbrightpoint);
      if (Fbrightpoint) {
         zdialog_stuff(zd,"darkpoint",0);
         Fdarkpoint = 0;
      }
   }

   if (strmatch(event,"darkpoint")) {
      zdialog_fetch(zd,"darkpoint",Fdarkpoint);
      if (Fdarkpoint) {
         zdialog_stuff(zd,"brightpoint",0);
         Fbrightpoint = 0;
      }
   }
   
   if (zstrstr("brightpoint darkpoint",event)) {                                 //  brightpoint or darkpoint
      takeMouse(gretinex_mousefunc,dragcursor);                                  //     connect mouse function
      return 1;
   }
   else {
      zdialog_stuff(zd,"brightpoint",0);                                         //  reset zdialog checkboxes
      zdialog_stuff(zd,"darkpoint",0);
   }
   
   adddark = addbrite = addmpy = 0;
   if (strmatch(event,"dark+-")) zdialog_fetch(zd,"dark+-",adddark);             //  All button
   if (strmatch(event,"brite+-")) zdialog_fetch(zd,"brite+-",addbrite);
   if (strmatch(event,"mpy+-")) zdialog_fetch(zd,"mpy+-",addmpy);

   zdialog_stuff(zd,"dark+-",0);                                                 //  reset to zero
   zdialog_stuff(zd,"brite+-",0);
   zdialog_stuff(zd,"mpy+-",0);
   
   if (adddark) {
      zdialog_fetch(zd,"Rdark",Rdark);                                           //  increment dark levels
      zdialog_fetch(zd,"Gdark",Gdark);
      zdialog_fetch(zd,"Bdark",Bdark);
      Rdark += adddark;
      Gdark += adddark;
      Bdark += adddark;
      if (Rdark < 0) Rdark = 0;
      if (Gdark < 0) Gdark = 0;
      if (Bdark < 0) Bdark = 0;
      if (Rdark > 255) Rdark = 255;
      if (Gdark > 255) Gdark = 255;
      if (Bdark > 255) Bdark = 255;
      zdialog_stuff(zd,"Rdark",Rdark);
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
   }

   if (addbrite) {                                                               //  increment bright levels
      zdialog_fetch(zd,"Rbrite",Rbrite);
      zdialog_fetch(zd,"Gbrite",Gbrite);
      zdialog_fetch(zd,"Bbrite",Bbrite);
      Rbrite += addbrite;
      Gbrite += addbrite;
      Bbrite += addbrite;
      if (Rbrite < 0) Rbrite = 0;
      if (Gbrite < 0) Gbrite = 0;
      if (Bbrite < 0) Bbrite = 0;
      if (Rbrite > 255) Rbrite = 255;
      if (Gbrite > 255) Gbrite = 255;
      if (Bbrite > 255) Bbrite = 255;
      zdialog_stuff(zd,"Rbrite",Rbrite);
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
   }

   if (addmpy) {                                                                 //  increment mpy factors
      zdialog_fetch(zd,"Rmpy",Rmpy);
      zdialog_fetch(zd,"Gmpy",Gmpy);
      zdialog_fetch(zd,"Bmpy",Bmpy);
      Rmpy += 0.01 * addmpy;
      Gmpy += 0.01 * addmpy;
      Bmpy += 0.01 * addmpy;
      if (Rmpy < 0.1) Rmpy = 0.1;
      if (Gmpy < 0.1) Gmpy = 0.1;
      if (Bmpy < 0.1) Bmpy = 0.1;
      if (Rmpy > 5) Rmpy = 5;
      if (Gmpy > 5) Gmpy = 5;
      if (Bmpy > 5) Bmpy = 5;
      zdialog_stuff(zd,"Rmpy",Rmpy);
      zdialog_stuff(zd,"Gmpy",Gmpy);
      zdialog_stuff(zd,"Bmpy",Bmpy);
   }
   
   zdialog_fetch(zd,"Rdark",Rdark);                                              //  get all params
   zdialog_fetch(zd,"Gdark",Gdark);
   zdialog_fetch(zd,"Bdark",Bdark);
   zdialog_fetch(zd,"Rbrite",Rbrite);
   zdialog_fetch(zd,"Gbrite",Gbrite);
   zdialog_fetch(zd,"Bbrite",Bbrite);
   zdialog_fetch(zd,"Rmpy",Rmpy);
   zdialog_fetch(zd,"Gmpy",Gmpy);
   zdialog_fetch(zd,"Bmpy",Bmpy);
   
   Fchange = 0;
   if (Rdark != pRdark) Fchange = 1;                                             //  detect changed params
   if (Gdark != pGdark) Fchange = 1;
   if (Bdark != pBdark) Fchange = 1;
   if (Rbrite != pRbrite) Fchange = 1;
   if (Gbrite != pGbrite) Fchange = 1;
   if (Bbrite != pBbrite) Fchange = 1;
   if (Rmpy != pRmpy) Fchange = 1;
   if (Gmpy != pGmpy) Fchange = 1;
   if (Bmpy != pBmpy) Fchange = 1;
   
   pRdark = Rdark;                                                               //  remember values for change detection
   pGdark = Gdark;
   pBdark = Bdark;
   pRbrite = Rbrite; 
   pGbrite = Gbrite; 
   pBbrite = Bbrite; 
   pRmpy = Rmpy; 
   pGmpy = Gmpy; 
   pBmpy = Bmpy; 

   if (Fchange) {                                                                //  global params changed
      thread_command = "global";
      thread_signal();                                                           //  update image
      Fchange = 0;
   }
   
   if (zstrstr("blend, reduce bright",event)) {                                  //  blend params changed
      zdialog_fetch(zd,"blend",blend);
      zdialog_fetch(zd,"reduce bright",reducebright);
      thread_command = "blend";
      thread_signal();                                                           //  update image
   }

   return 1;
}


//  get dark point or bright point from mouse click position

void gretinex_mousefunc()
{
   using namespace gretinex_names;
   
   int         px, py, dx, dy;
   float       red, green, blue;
   float       *ppix;
   char        mousetext[60];
   zdialog     *zd = EFgretinex.zd;
   
   if (! zd) {
      freeMouse();
      return;
   }

   if (! Fbrightpoint && ! Fdarkpoint) {
      freeMouse();
      return;
   }

   if (! LMclick) return;
   LMclick = 0;

   px = Mxclick;                                                                 //  mouse click position
   py = Myclick;

   if (px < 2) px = 2;                                                           //  pull back from edge
   if (px > E3pxm->ww-3) px = E3pxm->ww-3;
   if (py < 2) py = 2;
   if (py > E3pxm->hh-3) py = E3pxm->hh-3;

   red = green = blue = 0;

   for (dy = -1; dy <= 1; dy++)                                                  //  3x3 block around mouse position
   for (dx = -1; dx <= 1; dx++)
   {
      ppix = PXMpix(E1pxm,px+dx,py+dy);                                          //  input image
      red += ppix[0];
      green += ppix[1];
      blue += ppix[2];
   }

   red = red / 9.0;                                                              //  mean RGB levels
   green = green / 9.0;
   blue = blue / 9.0;

   snprintf(mousetext,60,"3x3 pixels RGB: %.0f %.0f %.0f \n",red,green,blue);
   poptext_mouse(mousetext,10,10,0,3);
   
   if (Fbrightpoint) {                                                           //  click pixel is new bright point
      Rbrite= red;
      Gbrite = green;
      Bbrite = blue;
      zdialog_stuff(zd,"Rbrite",Rbrite);                                         //  stuff values into dialog
      zdialog_stuff(zd,"Gbrite",Gbrite);
      zdialog_stuff(zd,"Bbrite",Bbrite);
   }

   if (Fdarkpoint) {                                                             //  click pixel is new dark point
      Rdark = red;
      Gdark = green;
      Bdark = blue;
      zdialog_stuff(zd,"Rdark",Rdark);                                           //  stuff values into dialog
      zdialog_stuff(zd,"Gdark",Gdark);
      zdialog_stuff(zd,"Bdark",Bdark);
   }
   
   if (Fbrightpoint || Fdarkpoint) {
      thread_command = "global";
      thread_signal();                                                           //  trigger image update
      thread_wait();     
   }
   
   return;
}


//  thread function - multiple working threads to update image

void * gretinex_thread(void *)
{
   using namespace gretinex_names;

   void  * gretinex_wthread1(void *arg);                                         //  worker threads
   void  * gretinex_wthread2(void *arg);
   void  * gretinex_wthread3(void *arg);
   
   if (strmatch(thread_command,"autoscale")) {
      do_wthreads(gretinex_wthread1,1);                                          //  worker thread for autoscale
      thread_command = "global";
   }
   
   if (strmatch(thread_command,"global"))
   {
      do_wthreads(gretinex_wthread2,NWT);                                        //  worker thread for image RGB rescale

      if (E9pxm) PXM_free(E9pxm);                                                //  E9 image = global retinex output
      E9pxm = PXM_copy(E3pxm);

      thread_command = "blend";                                                  //  auto blend after global retinex
   }
   
   if (strmatch(thread_command,"blend"))
   {
      if (! E9pxm) E9pxm = PXM_copy(E3pxm);                                      //  stop thread crash
      do_wthreads(gretinex_wthread3,NWT);                                        //  worker thread for image blend
   }

   CEF->Fmods++;                                                                 //  image modified
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return 0;
}


//  worker thread function - autoscale retinex

void * gretinex_wthread1(void *)
{
   using namespace gretinex_names;
   
   int      ii, dist = 0;
   int      px, py, dx, dy;
   int      red, green, blue;
   float    *pix1;

   Rdark = Gdark = Bdark = 255;
   Rbrite = Gbrite = Bbrite = 0;
   Rmpy = Gmpy = Bmpy = 1.0;
   Fbrightpoint = Fdarkpoint = 0;

   for (py = 1; py < E1pxm->hh-1; py++)
   for (px = 1; px < E1pxm->ww-1; px++)
   {
      if (Fcancel) break;                                                        //  user kill

      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      red = green = blue = 0;

      for (dy = -1; dy <= 1; dy++)                                               //  3x3 block around mouse position
      for (dx = -1; dx <= 1; dx++)
      {
         pix1 = PXMpix(E1pxm,px+dx,py+dy);                                       //  input image
         red += pix1[0];
         green += pix1[1];
         blue += pix1[2];
      }

      red = red / 9.0;                                                           //  mean RGB levels
      green = green / 9.0;
      blue = blue / 9.0;
      
      if (red < Rdark) Rdark = red;                                              //  update limits
      if (green < Gdark) Gdark = green;
      if (blue < Bdark) Bdark = blue;
      if (red > Rbrite) Rbrite = red;
      if (green > Gbrite) Gbrite = green;
      if (blue > Bbrite) Bbrite = blue;
   }

   return 0;
}


//  worker thread function - scale image RGB values

void * gretinex_wthread2(void *arg)
{
   using namespace gretinex_names;
   
   int      ii, index, dist = 0;
   int      px, py;
   float    R1, G1, B1, R3, G3, B3;
   float    f1, f2;
   float    *pix1, *pix3;
   float    max$;

   index = *((int *) arg);
   
   for (py = index; py < E3pxm->hh; py += NWT)                                   //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (Fcancel) break;                                                        //  user kill

      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];                                                              //  input RGB values
      G1 = pix1[1];
      B1 = pix1[2];
      
      R1 = 255 * (R1 - Rdark) / (Rbrite - Rdark);                                //  rescale for full 0-255 range
      G1 = 255 * (G1 - Gdark) / (Gbrite - Gdark);
      B1 = 255 * (B1 - Bdark) / (Bbrite - Bdark);

      if (R1 < 0) R1 = 0;
      if (G1 < 0) G1 = 0;
      if (B1 < 0) B1 = 0;

      R3 = R1 * Rmpy;
      G3 = G1 * Gmpy;
      B3 = B1 * Bmpy;

      RGBFIX(R3,G3,B3)

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }

      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


//  worker thread function - blend input and output images, attenuate bright pixels

void * gretinex_wthread3(void *arg)
{
   using namespace gretinex_names;

   int      index = *((int *) arg);
   int      px, py;
   int      ii, dist = 0;
   float    *pix1, *pix2, *pix3;
   float    R1, G1, B1, R2, G2, B2, R3, G3, B3;
   float    F1, F2, Lmax;
   
   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (Fcancel) break;                                                        //  user kill

      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }
      
      pix1 = PXMpix(E1pxm,px,py);                                                //  input image pixel
      pix2 = PXMpix(E9pxm,px,py);                                                //  global retinex image pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output image pixel

      R1 = pix1[0];                                                              //  input color - original image
      G1 = pix1[1];
      B1 = pix1[2];

      R2 = pix2[0];                                                              //  input color - retinex image
      G2 = pix2[1];
      B2 = pix2[2];

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         F1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         F2 = 1.0 - F1;
         R2 = F1 * R2 + F2 * R1;
         G2 = F1 * G2 + F2 * G1;
         B2 = F1 * B2 + F2 * B1;
      }

      Lmax = R1;                                                                 //  max. RGB input
      if (G1 > Lmax) Lmax = G1;
      if (B1 > Lmax) Lmax = B1;
      
      F1 = blend;                                                                //  0 ... 1  >>  E1 ... E3
      F2 = reducebright;                                                         //  0 ... 1
      F1 = F1 * (1.0 - F2 * Lmax / 255.0);                                       //  reduce F1 for high F2 * Lmax

      R3 = F1 * R2 + (1.0 - F1) * R1;                                            //  output RGB = blended inputs
      G3 = F1 * G2 + (1.0 - F1) * G1;
      B3 = F1 * B2 + (1.0 - F1) * B1;
      
      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  Local Retinex function
//  Rescale RGB values within local areas: increase local contrast and color.

namespace lretinex_names 
{
   editfunc    EFlretinex;                                                       //  edit function data
   cchar       *thread_command;
   int         E1ww, E1hh;
   float       blend, reducedark, reducebright;
   
   int         maxzones = 2000;
   int         Nzones = 100, Pzones = 0;                                         //  zone count, 1-2000
   int         zsize, zrows, zcols, zww, zhh;                                    //  zone data

   typedef struct {                                                              //  zone structure
      int      cx, cy;                                                           //  zone center in image
      float    minR, minG, minB;                                                 //  RGB minimum values in zone
      float    maxR, maxG, maxB;                                                 //  RGB mazimum values in zone
   }           zone_t;

   zone_t      *zones = 0;                                                       //  up to 2000 zones
   int16       *zoneindex = 0;                                                   //  zoneindex[ii][z] pixel ii, 9 zones
   int16       *zoneweight = 0;                                                  //  zoneweight[ii][z] zone weights
}


//  menu function

void m_lretinex(GtkWidget *, cchar *menu)
{
   using namespace lretinex_names;

   int    lretinex_dialog_event(zdialog *zd, cchar *event);
   void * lretinex_thread(void *);

   F1_help_topic = "local retinex";

   Plog(1,"m_lretinex \n");

   EFlretinex.menuname = "Local Retinex";
   EFlretinex.menufunc = m_lretinex;
   EFlretinex.Farea = 2;                                                         //  select area usable
   EFlretinex.Frestart = 1;                                                      //  allow restart
   EFlretinex.Fscript = 1;                                                       //  scripting supported
   EFlretinex.threadfunc = lretinex_thread;                                      //  thread function

   if (! edit_setup(EFlretinex)) return;                                         //  setup edit

   E1ww = E1pxm->ww;                                                             //  image size
   E1hh = E1pxm->hh;

   E9pxm = PXM_copy(E1pxm);
   
/***
          ______________________________________
         |            Local Retinex             |
         |                                      |
         | zone count: [___]  [apply]           |
         | blend: =======[]===================  |
         | reduce dark: ==========[]==========  |
         | reduce bright: ==========[]========  |
         |                                      |
         |             [reset] [ OK ] [cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new("Local Retinex",Mwin,"Reset","OK","Cancel",null); 
   EFlretinex.zd = zd;
   
   zdialog_add_widget(zd,"hbox","hbz","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labzs","hbz","zone count:","space=5");
   zdialog_add_widget(zd,"zspin","zone count","hbz","20|2000|10|100");           //  max. zones
   zdialog_add_widget(zd,"button","apply","hbz","Apply","space=3");
   zdialog_add_widget(zd,"hbox","hbb","dialog");
   zdialog_add_widget(zd,"label","labb","hbb","blend","space=5");
   zdialog_add_widget(zd,"hscale2","blend","hbb","0|1.0|0.01|1.0","expand");
   zdialog_add_widget(zd,"hbox","hbrd","dialog");
   zdialog_add_widget(zd,"label","labrd","hbrd","reduce dark","space=5");
   zdialog_add_widget(zd,"hscale2","reduce dark","hbrd","0|1.0|0.01|0.0","expand");
   zdialog_add_widget(zd,"hbox","hbrl","dialog");
   zdialog_add_widget(zd,"label","labrd","hbrl","reduce bright","space=5");
   zdialog_add_widget(zd,"hscale2","reduce bright","hbrl","0|1.0|0.01|0.0","expand");

   zdialog_run(zd,lretinex_dialog_event,"save");                                 //  run dialog - parallel
   zdialog_send_event(zd,"reset");
   return;
}


//  dialog event and completion function

int lretinex_dialog_event(zdialog *zd, cchar *event)
{
   using namespace lretinex_names;

   int lretinex_zonesetup(zdialog *zd);
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   if (strmatch(event,"reset")) zd->zstat = 1;                                   //  initz. 
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         blend = 1.0;
         reducedark = 0.0;
         reducebright = 0.0;
         edit_reset();
         PXM_free(E9pxm);
         E9pxm = PXM_copy(E1pxm);
         return 1;
      }

      else if (zd->zstat == 2) {                                                 //  done
         edit_addhist("zones:%d blend:%.3f red.D:%.3f red.B:%.3f",
                              Nzones,blend,reducedark,reducebright);             //  edit parms > edit hist                21.50
         edit_done(0);                                                           //  commit edit
         PXM_free(E9pxm);
         if (zones) zfree(zones);
         if (zoneindex) zfree(zoneindex);
         if (zoneweight) zfree(zoneweight);
         zones = 0;
         zoneindex = 0;
         zoneweight = 0;
         return 1;
      }

      else {
         edit_cancel(0);                                                         //  discard edit
         PXM_free(E9pxm);
         if (zones) zfree(zones);
         if (zoneindex) zfree(zoneindex);
         if (zoneweight) zfree(zoneweight);
         zones = 0;
         zoneindex = 0;
         zoneweight = 0;
         return 1;
      }
   }

   if (strmatch(event,"apply")) {                                                //  [apply]
      if (! lretinex_zonesetup(zd)) {                                            //  setup zones
         edit_cancel(0);                                                         //  failed (insufficient memory)
         PXM_free(E9pxm);
         return 1;
      }
      thread_command = "apply";
      thread_signal();                                                           //  update image
      thread_wait();     
   }

   if (strmatch(event,"blend")) {                                                //  blend param changed
      zdialog_fetch(zd,"blend",blend);
      thread_command = "blend";
      thread_signal();                                                           //  update image
      thread_wait();     
   }

   if (zstrstr(event,"reduce")) {                                                //  reduce param changed
      zdialog_fetch(zd,"reduce dark",reducedark);
      zdialog_fetch(zd,"reduce bright",reducebright);
      thread_command = "blend";
      thread_signal();                                                           //  update image
      thread_wait();     
   }

   return 1;
}


//  setup new zone parameters and allocate memory

int lretinex_zonesetup(zdialog *zd)
{
   using namespace lretinex_names;

   int      goal, dirc;
   int      row, col, xlo, ylo;
   int      ii, cx, cy;
   int64    nn;
   size_t   reqcc;

   Pzones = Nzones;                                                              //  prior count
   zdialog_fetch(zd,"zone count",Nzones);                                        //  new count

   goal = Nzones;                                                                //  new zone count goal
   dirc = Nzones - Pzones;                                                       //  direction of change
   if (dirc == 0) dirc = 1;

   while (true)
   {
      zsize = sqrt(E1ww * E1hh / goal);                                          //  approx. zone size
      zrows = E1hh / zsize;                                                      //  get appropriate rows and cols
      zcols = E1ww / zsize;
      if (zrows < 1) zrows = 1;
      if (zcols < 1) zcols = 1;
      Nzones = zrows * zcols;                                                    //  matching rows x cols

      if (dirc > 0 && Nzones <= Pzones) {                                        //  no increase, try again
         if (Nzones == maxzones) break;
         goal++;
         continue;
      }

      if (dirc < 0 && Nzones >= Pzones) {                                        //  no decrease, try again
         if (Nzones == 1) break;
         goal--;
         continue;
      }

      if (dirc > 0 && Nzones > Pzones) break;                                    //  done
      if (dirc < 0 && Nzones < Pzones) break;
   }
   
   zdialog_stuff(zd,"zone count",Nzones);                                        //  update zdialog widget

   if (zones) zfree(zones);                                                      //  allocate zone memory
   if (zoneindex) zfree(zoneindex);                                              //  allocate pixel zone index
   if (zoneweight) zfree(zoneweight);                                            //  allocate pixel zone weight
   zones = 0;
   zoneindex = 0;
   zoneweight = 0;
   
   zww = E1ww / zcols;                                                           //  zone width, height
   zhh = E1hh / zrows;
   nn = E1ww * E1hh * 9;                                                         //  9 neighbor zones per pixel

   reqcc = Nzones * sizeof(zone_t) + 2 * nn * sizeof(int16);                     //  check large memory is available
   if (! zmalloc_test(reqcc)) {
      reqcc = reqcc / MEGA;
      zmessageACK(Mwin,"cannot allocate %d MB memory",reqcc);
      return 0;
   }

   zones = (zone_t *) zmalloc(Nzones * sizeof(zone_t),"local retx");             //  allocate zone memory
   zoneindex = (int16 *) zmalloc(nn * sizeof(int16),"local retx");
   zoneweight = (int16 *) zmalloc(nn * sizeof(int16),"local retx");

   for (row = 0; row < zrows; row++)                                             //  loop all zones
   for (col = 0; col < zcols; col++)
   {
      xlo = col * zww;                                                           //  zone low pixel
      ylo = row * zhh;
      cx = xlo + zww/2;                                                          //  zone center pixel
      cy = ylo + zhh/2;
      ii = row * zcols + col;                                                    //  zone index
      zones[ii].cx = cx;                                                         //  zone center
      zones[ii].cy = cy;
   }

   return 1;
}


//  thread function - multiple working threads to update image

void * lretinex_thread(void *)
{
   using namespace lretinex_names;

   void * lretinex_wthread1(void *arg);
   void * lretinex_wthread2(void *arg);
   void * lretinex_wthread3(void *arg);
   
   if (strmatch(thread_command,"apply"))                                         //  update zones and image
   {
      do_wthreads(lretinex_wthread1,NWT);                                        //  compute zone RGB levels and weights
      do_wthreads(lretinex_wthread2,NWT);                                        //  compute new pixel RGB values
      thread_command = "blend";                                                  //  auto blend after
   }
   
   if (strmatch(thread_command,"blend"))
      do_wthreads(lretinex_wthread3,NWT);                                        //  blend input and retinex images

   CEF->Fmods++;                                                                 //  image modified
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return 0;
}


//  worker thread to compute zone RGB levels and weights per image pixel

void * lretinex_wthread1(void *arg)
{
   using namespace lretinex_names;

   int      index = *((int *) arg);
   int      px, py, row, col;
   int      xlo, xhi, ylo, yhi, cx, cy;
   int      ii, jj, kk;
   float    minR, minG, minB, maxR, maxG, maxB;
   float    dist, maxdist, weight, sumweight;
   float    *pix;

   for (row = index; row < zrows; row += NWT)                                    //  loop all zones
   for (col = 0; col < zcols; col++)
   {
      xlo = col * zww;                                                           //  zone low pixel
      ylo = row * zhh;
      xhi = xlo + zww;                                                           //  zone high pixel
      yhi = ylo + zhh;

      minR = minG = minB = 256;
      maxR = maxG = maxB = 0;

      for (py = ylo; py < yhi; py++)                                             //  find min/max RGB in zone
      for (px = xlo; px < xhi; px++)
      {
         pix = PXMpix(E1pxm,px,py);
         if (pix[0] < minR) minR = pix[0];
         if (pix[1] < minG) minG = pix[1];
         if (pix[2] < minB) minB = pix[2];
         if (pix[0] > maxR) maxR = pix[0];
         if (pix[1] > maxG) maxG = pix[1];
         if (pix[2] > maxB) maxB = pix[2];
      }
      
      ii = row * zcols + col;                                                    //  zone index
      zones[ii].minR = minR;                                                     //  zone RGB range
      zones[ii].minG = minG;
      zones[ii].minB = minB;
      zones[ii].maxR = maxR;
      zones[ii].maxG = maxG;
      zones[ii].maxB = maxB;
   }

   for (py = index; py < E1hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E1ww; px++)
   {
      row = py / zhh;                                                            //  zone containing pixel
      col = px / zww;

      ii = (py * E1ww + px) * 9;                                                 //  base zone index for pixel
      
      for (jj = row-1; jj <= row+1; jj++)                                        //  loop 9 neighboring zones
      for (kk = col-1; kk <= col+1; kk++)
      {
         if (jj >= 0 && jj < zrows && kk >= 0 && kk < zcols)                     //  neighbor zone number
            zoneindex[ii] = jj * zcols + kk;                                     //    --> pixel zone index            
         else zoneindex[ii] = -1;                                                //  pixel zone on edge, missing neighbor
         ii++;
      }
   }
   
   if (zww < zhh) maxdist = 1.5 * zww;                                           //  based on 3x3 zones
   else maxdist = 1.5 * zhh;

   for (py = index; py < E1hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E1ww; px++)
   {
      row = py / zhh;                                                            //  zone containing pixel
      col = px / zww;

      ii = (py * E1ww + px) * 9;                                                 //  base zone index for pixel
      
      for (jj = 0; jj < 9; jj++)                                                 //  loop neighbor zones
      {
         kk = zoneindex[ii+jj];
         if (kk < 0) {                                                           //  neighbor missing 
            zoneweight[ii+jj] = 0;
            continue;
         }
         cx = zones[kk].cx;                                                      //  zone center
         cy = zones[kk].cy;
         dist = sqrtf((px-cx)*(px-cx) + (py-cy)*(py-cy));                        //  distance from (px,py)
         weight = 1.0 - dist / maxdist;                                          //  dist 0..max --> weight 1..0
         if (weight < 0) weight = 0;
         zoneweight[ii+jj] = 1000.0 * weight;                                    //  scale 1.0 = 1000
      }

      sumweight = 0;
      for (jj = 0; jj < 9; jj++)                                                 //  get sum of zone weights
         sumweight += zoneweight[ii+jj];
      
      for (jj = 0; jj < 9; jj++)                                                 //  make weights add up to 1.0
         zoneweight[ii+jj] = 1000.0 * zoneweight[ii+jj] / sumweight;             //  1000 = 1.0
   }

   return 0;
}


//  worker thread to compute new image RGB values

void * lretinex_wthread2(void *arg)
{
   using namespace lretinex_names;

   int      index = *((int *) arg);
   int      px, py, ii, jj, kk, dist = 0;
   float    minR, minG, minB, maxR, maxG, maxB;
   float    R1, G1, B1, R2, G2, B2;
   float    weight, f1, f2;
   float    *pix1, *pix2;
   float    max$;
   
   for (py = index; py < E1hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E1ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      minR = minG = minB = 0;
      maxR = maxG = maxB = 0;
      
      ii = (py * E1ww + px) * 9;                                                 //  base zone index for pixel
      
      for (jj = 0; jj < 9; jj++)                                                 //  loop neighbor zones
      {
         kk = zoneindex[ii+jj];                                                  //  zone number
         if (kk < 0) continue;
         weight = 0.001 * zoneweight[ii+jj];                                     //  zone weight (1000 = 1.0)
         minR += weight * zones[kk].minR;                                        //  sum weighted RGB range
         minG += weight * zones[kk].minG;                                        //    for neighbor zones
         minB += weight * zones[kk].minB;
         maxR += weight * zones[kk].maxR;
         maxG += weight * zones[kk].maxG;
         maxB += weight * zones[kk].maxB;
      }
      
      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix2 = PXMpix(E9pxm,px,py);                                                //  output pixel

      R1 = pix1[0];
      G1 = pix1[1];
      B1 = pix1[2];
      
      R2 = 255 * (R1 - minR) / (maxR - minR + 1);                                //  avoid poss. divide by zero
      G2 = 255 * (G1 - minG) / (maxG - minG + 1);
      B2 = 255 * (B1 - minB) / (maxB - minB + 1);

      RGBFIX(R2,G2,B2)

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R2 = f1 * R2 + f2 * R1;
         G2 = f1 * G2 + f2 * G1;
         B2 = f1 * B2 + f2 * B1;
      }

      pix2[0] = R2;
      pix2[1] = G2;
      pix2[2] = B2;
   }

   return 0;
}


//  worker thread to blend input image with retinex image
//  and attenuate bright pixels

void * lretinex_wthread3(void *arg)
{
   using namespace lretinex_names;

   int      index = *((int *) arg);
   int      px, py;
   int      ii, dist = 0;
   float    *pix1, *pix2, *pix3;
   float    R1, G1, B1, R2, G2, B2, R3, G3, B3;
   float    F1, F2, Lmax;
   
   for (py = index; py < E1hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E1ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E1ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input image pixel
      pix2 = PXMpix(E9pxm,px,py);                                                //  retinex image pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output image pixel

      R1 = pix1[0];                                                              //  input color - original image
      G1 = pix1[1];
      B1 = pix1[2];

      R2 = pix2[0];                                                              //  input color - retinex image
      G2 = pix2[1];
      B2 = pix2[2];

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         F1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         F2 = 1.0 - F1;
         R2 = F1 * R2 + F2 * R1;
         G2 = F1 * G2 + F2 * G1;
         B2 = F1 * B2 + F2 * B1;
      }

      Lmax = R1;                                                                 //  max. RGB input
      if (G1 > Lmax) Lmax = G1;
      if (B1 > Lmax) Lmax = B1;
      Lmax = Lmax / 255.0;                                                       //  scale 0-1
      
      F1 = blend;                                                                //  0 ... 1  >>  E1 ... E3

      F2 = reducedark;
      F1 = F1 * (1.0 - F2 * (1.0 - Lmax));                                       //  reduce F1 for high F2 * (1 - Lmax)

      F2 = reducebright;                                                         //  0 ... 1
      F1 = F1 * (1.0 - F2 * Lmax);                                               //  reduce F1 for high F2 * Lmax

      R3 = F1 * R2 + (1.0 - F1) * R1;                                            //  output RGB = blended inputs
      G3 = F1 * G2 + (1.0 - F1) * G1;
      B3 = F1 * B2 + (1.0 - F1) * B1;
      
      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  Saturation
//  Adjust color saturation for entire image (slider),                           //  21.40
//  or set saturation based on pixel brightness (curve).

namespace saturation_names
{
   editfunc    EFsaturation;
   float       colorsat;                                                         //  saturation input, -1 ... +1
}


//  menu function

void m_saturation(GtkWidget *, cchar *menu)
{
   using namespace saturation_names;

   int    saturation_dialog_event(zdialog* zd, cchar *event);
   void   saturation_curvedit(int spc);
   void * saturation_thread(void *);

   GtkWidget   *drawwin_scale;

   F1_help_topic = "saturation";

   Plog(1,"m_saturation \n");

   EFsaturation.menuname = "Saturation";
   EFsaturation.menufunc = m_saturation;
   EFsaturation.FprevReq = 1;                                                    //  use preview
   EFsaturation.Farea = 2;                                                       //  select area usable
   EFsaturation.Frestart = 1;                                                    //  restart allowed
   EFsaturation.Fpaintedits = 1;                                                 //  use with paint edits OK
   EFsaturation.Fscript = 1;                                                     //  scripting supported
   EFsaturation.threadfunc = saturation_thread;
   if (! edit_setup(EFsaturation)) return;                                       //  setup edit

/***
       _______________________________________________
      |                 Saturation                    |
      |  ___________________________________________  |
      | |                                           | |
      | |             curve edit area               | |
      | |    saturation by brightness level         | |
      | |                                           | |
      | |                                           | |
      | | ----------------------------------------- | |                          //  editable function-line
      | |                                           | |
      | |                                           | |
      | |                                           | |
      | |___________________________________________| |
      | |___________________________________________| |                          //  brightness scale: black --> white
      |                                               |
      |  Saturation  =================[]============  |                          //  saturation, B/W ... max-color
      |                                               |
      |                       [Reset] [ OK ] [Cancel] |
      |_______________________________________________|

***/

   zdialog *zd = zdialog_new("Saturation",Mwin,"Reset","OK","Cancel",null);
   EFsaturation.zd = zd;

   zdialog_add_widget(zd,"frame","frameH","dialog",0,"expand");                  //  edit-curve and distribution graph
   zdialog_add_widget(zd,"frame","frameB","dialog");                             //  black to white brightness scale

   zdialog_add_widget(zd,"hbox","hbsat","dialog",0,"space=5|expand");
   zdialog_add_widget(zd,"label","labsat","hbsat","Saturation");
   zdialog_add_widget(zd,"hscale","colorsat","hbsat","-1.0|1.0|0.01|0.0","expand");

   GtkWidget *frameH = zdialog_gtkwidget(zd,"frameH");                           //  setup edit curves
   spldat *sd = splcurve_init(frameH,saturation_curvedit);
   EFsaturation.sd = sd;

   sd->Nscale = 1;                                                               //  horizontal line, neutral value
   sd->xscale[0][0] = 0.00;
   sd->yscale[0][0] = 0.50;
   sd->xscale[1][0] = 1.00;
   sd->yscale[1][0] = 0.50;

   sd->nap[0] = 3;                                                               //  initial curve is neutral
   sd->vert[0] = 0;
   sd->apx[0][0] = 0.01;                                                         //  horizontal line
   sd->apy[0][0] = 0.50;
   sd->apx[0][1] = 0.50;
   sd->apy[0][1] = 0.50;                                                         //  curve 0 = overall brightness
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.50;
   splcurve_generate(sd,0);
   sd->mod[0] = 0;                                                               //  mark curve unmodified

   sd->Nspc = 1;                                                                 //  1 curve
   sd->fact[0] = 1;                                                              //  curve 0 active 
   
   GtkWidget *frameB = zdialog_gtkwidget(zd,"frameB");                           //  setup brightness scale drawing area
   drawwin_scale = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(frameB),drawwin_scale);
   gtk_widget_set_size_request(drawwin_scale,100,12);
   G_SIGNAL(drawwin_scale,"draw",brightness_scale,0);

   colorsat = 0;                                                                 //  neutral saturation

   zdialog_resize(zd,350,300);
   zdialog_run(zd,saturation_dialog_event,"save");                               //  run dialog - parallel

   return;
}


//  dialog event and completion callback function

int saturation_dialog_event(zdialog *zd, cchar *event)
{
   using namespace saturation_names;

   spldat      *sd = EFsaturation.sd;    
   float       sat0, dsat, dy;
   float       Fapply = 0;
   int         ii;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  cancel
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel
   if (strmatch(event,"apply")) Fapply = 1;                                      //  from script

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();                                                           //  get full size image
      thread_signal();
      return 1;
   }

   if (zd->zstat == 1)                                                           //  [reset]
   {
      zd->zstat = 0;                                                             //  keep dialog active

      zdialog_stuff(zd,"colorsat",0);                                            //  neutral saturation
      colorsat = 0;

      sd->nap[0] = 3;                                                            //  curve is neutral
      sd->vert[0] = 0;
      sd->apx[0][0] = 0.01;
      sd->apy[0][0] = 0.50;
      sd->apx[0][1] = 0.50;
      sd->apy[0][1] = 0.50;
      sd->apx[0][2] = 0.99;
      sd->apy[0][2] = 0.50;
      splcurve_generate(sd,0);
      sd->mod[0] = 0;                                                            //  mark curve unmodified

      gtk_widget_queue_draw(sd->drawarea);                                       //  redraw curves

      edit_reset();
      return 1;
   }

   if (zd->zstat == 2)                                                           //  [ OK ]
   {
      freeMouse();
      if (CEF->Fmods) {
         edit_fullsize();                                                        //  get full size image
         thread_signal();                                                        //  apply changes
         edit_done(0);                                                           //  complete edit
      }

      else edit_cancel(0);                                                       //  no change
      return 1;
   }
   
   if (zd->zstat < 0 || zd->zstat > 2) {                                         //  [cancel] or [x]
      edit_cancel(0);
      return 1;
   }
   
   if (strmatch(event,"colorsat"))                                               //  saturation slider -1 ... +1
   {
      sat0 = colorsat;                                                           //  old value
      zdialog_fetch(zd,"colorsat",colorsat);                                     //  new value
      dsat = colorsat - sat0;                                                    //  change in saturation, + -

      for (ii = 0; ii < sd->nap[0]; ii++)                                        //  update curve 0 nodes from slider
      {
         dy = sd->apy[0][ii] + 0.5 * dsat;                                       //  increment saturation
         if (dy < 0) dy = 0;
         if (dy > 1) dy = 1;
         sd->apy[0][ii] = dy;
      }

      splcurve_generate(sd,0);                                                   //  regenerate curve 0
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve 0
      Fapply = 1;
   }
   
   if (zstrstr("blendwidth paint",event)) Fapply = 1;                            //  area edge blend or mouse paint

   if (Fapply) thread_signal();                                                  //  update the image

   return 1;
}


//  this function is called when a curve is edited

void saturation_curvedit(int spc)
{
   using namespace saturation_names;
   thread_signal();
   return;
}


//  thread function

void * saturation_thread(void *arg)
{
   using namespace saturation_names;

   void * saturation_wthread(void *);

   get_edit_pixels_init(NWT,0);                                                  //  initz. pixel loop                     21.44

   do_wthreads(saturation_wthread,NWT);

   CEF->Fmods++;                                                                 //  image3 modified
   CEF->Fsaved = 0;

   if (! Fpaintedits) Fpaint2();                                                 //  update window
   return 0;
}


void * saturation_wthread(void *arg)                                             //  worker thread function
{
   using namespace saturation_names;

   int         index = *((int *) arg);
   int         ii, Fend, px, py;
   float       *pix1, *pix3;
   float       R1, G1, B1, R3, G3, B3, R9, G9, B9;
   float       pixbrite, colorsat;
   float       coeff = 1000.0 / 256.0;
   float       blend, max$;
   spldat      *sd = EFsaturation.sd;    

   while (true)                                                                  //  loop all edit pixels
   {
      Fend = get_edit_pixels(index,px,py,blend);                                 //  21.44
      if (Fend) break;
      if (blend == 0) continue;
      
      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = R9 = pix1[0];                                                         //  input RGB values
      G1 = G9 = pix1[1];
      B1 = B9 = pix1[2];

      R3 = pix3[0];                                                              //  current output image RGB
      G3 = pix3[1];
      B3 = pix3[2];

      //  apply saturation curve

      if (sd->mod[0])                                                            //  curve was modified
      {
         pixbrite = 0.333 * (R3 + G3 + B3);                                      //  pixel brightness, 0 to 255.9
         ii = coeff * pixbrite;                                                  //  "all" curve index 0-999
         if (ii < 0) ii = 0;
         if (ii > 999) ii = 999;
         colorsat = 2.0 * sd->yval[0][ii] - 1.0;                                 //  -1 ... 0 ... +1

         R9 = R9 + colorsat * (R9 - pixbrite);                                   //  colorsat is -1 ... +1
         G9 = G9 + colorsat * (G9 - pixbrite);
         B9 = B9 + colorsat * (B9 - pixbrite);
      }

      RGBFIX(R9,G9,B9)

      if (Fpaintedits)                                                           //  gradual edit within mouse circle      21.44
      {
         if (blend > 0)
         {
            R3 = blend * R9 + (1-blend) * R3;                                    //  increase edit 
            G3 = blend * G9 + (1-blend) * G3;
            B3 = blend * B9 + (1-blend) * B3;
         }
         
         else if (blend < 0)
         {
            R3 = -blend * R1 + (1+blend) * R3;                                   //  decrease edit
            G3 = -blend * G1 + (1+blend) * G3;
            B3 = -blend * B1 + (1+blend) * B3;
         }
      }

      else                                                                       //  full edit for image or area           21.44
      {
         R3 = blend * R9 + (1-blend) * R1;                                       //  increase edit 
         G3 = blend * G9 + (1-blend) * G1;
         B3 = blend * B9 + (1-blend) * B1;
      }

      pix3[0] = R3;                                                              //  set output pixel
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  convert color profile of current image

editfunc    EFcolorprof;
char        ICCprofilename[100];                                                 //  new color profile name 
char        colorprof1[200] = "/usr/share/color/icc/colord/AdobeRGB1998.icc";
char        colorprof2[200] = "/usr/share/color/icc/colord/sRGB.icc";


//  menu function

void m_color_profile(GtkWidget *, cchar *menu)
{
   int colorprof_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   int         err;
   cchar       *exifkey[2], *exifdata[2];

   F1_help_topic = "color profile";

   Plog(1,"m_color_profile \n");

   m_viewmode(0,"F");                                                            //  file view mode

   EFcolorprof.menuname = "Color Profile";
   EFcolorprof.menufunc = m_color_profile;
   EFcolorprof.Frestart = 1;                                                     //  allow restart
   EFcolorprof.Fscript = 1;                                                      //  scripting supported
   if (! edit_setup(EFcolorprof)) return;                                        //  setup edit
   
   *ICCprofilename = 0;                                                          //  no color profile change

/***
       ________________________________________________________
      |       Change Color Profile                             |
      |                                                        |
      |  input profile  [___________________________] [Browse] |
      |  output profile [___________________________] [Browse] |
      |                                                        |
      |                                [Apply] [ OK ] [Cancel] |
      |________________________________________________________|

***/

   zd = zdialog_new("Change Color Profile",Mwin,"Apply","OK","Cancel",null);
   EFcolorprof.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lab1","hb1","input profile","space=5");
   zdialog_add_widget(zd,"zentry","prof1","hb1",0,"expand|size=30");
   zdialog_add_widget(zd,"button","butt1","hb1","Browse","space=5");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab2","hb2","output profile","space=5");
   zdialog_add_widget(zd,"zentry","prof2","hb2",0,"expand|size=30");
   zdialog_add_widget(zd,"button","butt2","hb2","Browse","space=5");

   zdialog_stuff(zd,"prof1",colorprof1);
   zdialog_stuff(zd,"prof2",colorprof2);

   zdialog_run(zd,colorprof_dialog_event,"save");                                //  run dialog, parallel

   zdialog_wait(zd);                                                             //  wait for completion
   if (! *ICCprofilename) return;                                                //  no color profile change
   
   m_file_save_version(0,0);                                                     //  save as new version and re-open
   zshell(0,"rm -f %s/undo_*",temp_folder);                                      //  remove undo/redo files
   URS_pos = URS_max = 0;                                                        //  reset undo/redo stack

   exifkey[0] = exif_colorprof2_key;                                             //  remove embedded color profile
   exifdata[0] = "";
   exifkey[1] = exif_colorprof1_key;                                             //  set new color profile name
   exifdata[1] = ICCprofilename;
   err = exif_put(curr_file,exifkey,exifdata,2);
   if (err) zmessageACK(Mwin,"Unable to change EXIF color profile");

   zmessageACK(Mwin,"automatic new version created");
   return;
}


//  dialog event and completion callback function

int colorprof_dialog_event(zdialog *zd, cchar *event)
{
   cchar    *title = "color profile";
   char     *file;
   float    *fpix1, *fpix2;
   float    f256 = 1.0 / 256.0;
   uint     Npix, nn;
   
   cmsHTRANSFORM  cmsxform;
   cmsHPROFILE    cmsprof1, cmsprof2;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"apply")) zd->zstat = 1;                                   //  from script
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   
   if (strmatch(event,"butt1")) {
      zdialog_fetch(zd,"prof1",colorprof1,200);                                  //  select input profile
      file = zgetfile(title,MWIN,"file",colorprof1);
      if (! file) return 1;
      zdialog_stuff(zd,"prof1",file);
      zfree(file);
   }

   if (strmatch(event,"butt2")) {
      zdialog_fetch(zd,"prof2",colorprof2,200);                                  //  select output profile
      file = zgetfile(title,MWIN,"file",colorprof2);
      if (! file) return 1;
      zdialog_stuff(zd,"prof2",file);
      zfree(file);
   }

   if (! zd->zstat) return 1;                                                    //  wait for user completion

   if (zd->zstat == 1) zd->zstat = 0;                                            //  [apply] - keep dialog open

   if (zd->zstat)
   {
      if (zd->zstat == 2 && CEF->Fmods) edit_done(0);                            //  commit edit
      else {
         edit_cancel(0);                                                         //  discard edit
         *ICCprofilename = 0;                                                    //  no ICC profile change
      }
      return 1;
   }

   zdialog_fetch(zd,"prof1",colorprof1,200);                                     //  [apply] - get final profiles
   zdialog_fetch(zd,"prof2",colorprof2,200);

   cmsprof1 = cmsOpenProfileFromFile(colorprof1,"r");
   if (! cmsprof1) {
      zmessageACK(Mwin,"unknown cms profile %s",colorprof1);
      return 1;
   }

   cmsprof2 = cmsOpenProfileFromFile(colorprof2,"r");
   if (! cmsprof2) {
      zmessageACK(Mwin,"unknown cms profile %s",colorprof2);
      return 1;
   }

   //  calculate the color space transformation table

   zadd_locked(Ffuncbusy,+1);
   zmainsleep(0.2);

   cmsxform = cmsCreateTransform(cmsprof1,TYPE_RGB_FLT,cmsprof2,TYPE_RGB_FLT,INTENT_PERCEPTUAL,0);
   if (! cmsxform) {
      zmessageACK(Mwin,"cmsCreateTransform() failed");
      zadd_locked(Ffuncbusy,-1);
      return 1;
   }

   fpix1 = E0pxm->pixels;                                                        //  input and output pixels
   fpix2 = E3pxm->pixels;
   Npix = E0pxm->ww * E0pxm->hh;

   for (uint ii = 0; ii < 3 * Npix; ii++)                                        //  rescale to range 0 - 0.9999
      fpix2[ii] = f256 * fpix1[ii];

   while (Npix)                                                                  //  convert image pixels
   {
      zmainloop(20);                                                             //  keep GTK alive
      nn = Npix;
      if (nn > 100000) nn = 100000;                                              //  do 100K per call
      cmsDoTransform(cmsxform,fpix2,fpix2,nn);                                   //  speed: 3 megapixels/sec for 3 GHz CPU
      fpix2 += nn * 3;
      Npix -= nn;
   }

   fpix2 = E3pxm->pixels;
   Npix = E0pxm->ww * E0pxm->hh;
   for (uint ii = 0; ii < 3 * Npix; ii++) {                                      //  rescale back to 0 - 255.99
      fpix2[ii] = fpix2[ii] * 256.0;
      if (fpix2[ii] > 255.9) fpix2[ii] = 255.9;
      if (fpix2[ii] < 0) fpix2[ii] = 0;                                          //  compensate cms bug
   }

   cmsInfoType it = (cmsInfoType) 0;
   cmsGetProfileInfoASCII(cmsprof2,it,"en","US",ICCprofilename,100);             //  new color profile name

   cmsDeleteTransform(cmsxform);                                                 //  free resources
   cmsCloseProfile(cmsprof1);
   cmsCloseProfile(cmsprof2);

   zadd_locked(Ffuncbusy,-1);
   CEF->Fmods++;                                                                 //  image is modified
   CEF->Fsaved = 0;
   Fpaint2();                                                                    //  update window image

   return 1;
}


/********************************************************************************

   Vignette function

   1. Change the brightness from center to edge using a curve.
   2. Change the color from center to edge using a color and a curve.
      (the pixel varies between original RGB and selected color)
   3. Mouse click or drag on image sets a new vignette center.

*********************************************************************************/

void  vign_mousefunc();

editfunc    EFvignette;
uint8       vignette_RGB[3] = { 0, 0, 255 };
int         vignette_spc;
float       vign_cx, vign_cy;
float       vign_rad;


void m_vignette(GtkWidget *, cchar *menu)
{
   int      Vign_dialog_event(zdialog *zd, cchar *event);
   void     Vign_curvedit(int);
   void *   Vign_thread(void *);

   cchar    *title = "Vignette";

   F1_help_topic = "vignette";

   Plog(1,"m_vignette \n");

   EFvignette.menuname = "Vignette";
   EFvignette.Farea = 2;                                                         //  select area usable
   EFvignette.FprevReq = 1;                                                      //  use preview image
   EFvignette.threadfunc = Vign_thread;                                          //  thread function
   EFvignette.mousefunc = vign_mousefunc;                                        //  mouse function
   if (! edit_setup(EFvignette)) return;                                         //  setup edit

/***
          ___________________________________
         |  _______________________________  |
         | |                               | |
         | |                               | |
         | |    curve drawing area         | |
         | |                               | |
         | |                               | |
         | |_______________________________| |
         |  center                     edge  |
         |                                   |
         |  (o) Brightness  (o) Color [___]  |
         |  Curve File: [ Open ] [ Save ]    |
         |                                   |
         |                   [ OK ] [Cancel] |
         |___________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,"OK","Cancel",null);
   EFvignette.zd = zd;

   zdialog_add_widget(zd,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labcenter","hb1","Center","space=4");
   zdialog_add_widget(zd,"label","space","hb1",0,"expand");
   zdialog_add_widget(zd,"label","labedge","hb1","Edge","space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","RBbrite","hb2","Brightness","space=5");
   zdialog_add_widget(zd,"radio","RBcolor","hb2","Color","space=5");
   zdialog_add_widget(zd,"colorbutt","color","hb2","0|0|255");

   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcurve","hb3","Curve File","space=5");
   zdialog_add_widget(zd,"button","load","hb3","Open","space=5");
   zdialog_add_widget(zd,"button","save","hb3","Save","space=5");

   vignette_RGB[0] = vignette_RGB[1] = 0;                                        //  initial color = blue
   vignette_RGB[2] = 255;

   vign_cx = E3pxm->ww / 2;                                                      //  initial vignette center
   vign_cy = E3pxm->hh / 2;

   vign_rad = vign_cx * vign_cx + vign_cy * vign_cy;                             //  radius = distance to corners
   vign_rad = sqrtf(vign_rad);

   zdialog_stuff(zd,"RBbrite",1);                                                //  default curve = brightness

   GtkWidget *frame = zdialog_gtkwidget(zd,"frame");                             //  set up curve edit
   spldat *sd = splcurve_init(frame,Vign_curvedit);
   EFvignette.sd = sd;

   sd->Nspc = 2;                                                                 //  2 curves

   sd->vert[0] = 0;                                                              //  curve 0 = brightness curve
   sd->nap[0] = 2;
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.5;
   sd->apx[0][1] = 0.99;
   sd->apy[0][1] = 0.5;
   splcurve_generate(sd,0);

   sd->vert[1] = 0;                                                              //  curve 1 = color curve
   sd->nap[1] = 2;
   sd->apx[1][0] = 0.01;
   sd->apy[1][0] = 0.01;
   sd->apx[1][1] = 0.99;
   sd->apy[1][1] = 0.01;
   splcurve_generate(sd,1);

   vignette_spc = 0;                                                             //  initial curve = brightness
   sd->fact[0] = 1;
   sd->fact[1] = 0; 

   zdialog_run(zd,Vign_dialog_event,"save");                                     //  run dialog - parallel

   takeMouse(vign_mousefunc,dragcursor);                                         //  connect mouse function
   return;
}


//  dialog event and completion callback function

int Vign_dialog_event(zdialog *zd, cchar *event)
{
   void     Vign_curvedit(int);

   spldat   *sd = EFvignette.sd;    
   int      ii;
   char     color[20];
   char     *file, *pp;
   cchar    *ppc;
   FILE     *fid;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      thread_signal();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  done
         thread_wait();                                                          //  insure thread done
         float R = 1.0 * E0pxm->ww / E3pxm->ww;
         vign_cx = R * vign_cx;                                                  //  scale geometries to full size
         vign_cy = R * vign_cy;
         vign_rad = R * vign_rad;
         edit_fullsize();                                                        //  get full size image
         thread_signal();
         edit_done(0);                                                           //  commit edit
      }
      else edit_cancel(0);                                                       //  discard edit
      return 1;
   }

   if (strmatch(event,"focus"))                                                  //  toggle mouse capture
      takeMouse(vign_mousefunc,dragcursor);                                      //  connect mouse function

   if (strmatchN(event,"RB",2)) {                                                //  new choice of curve
      sd->fact[0] = sd->fact[1] = 0; 
      ii = strmatchV(event,"RBbrite","RBcolor",null);
      vignette_spc = ii = ii - 1;
      sd->fact[ii] = 1;                                                          //  active curve
      splcurve_generate(sd,ii);                                                  //  regenerate curve
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve
      thread_signal();
   }

   if (strmatch(event,"blendwidth")) thread_signal();

   if (strmatch(event,"color")) {                                                //  change color
      zdialog_fetch(zd,"color",color,19);                                        //  get color from color wheel
      ppc = substring(color,"|",1);
      if (ppc) vignette_RGB[0] = atoi(ppc);
      ppc = substring(color,"|",2);
      if (ppc) vignette_RGB[1] = atoi(ppc);
      ppc = substring(color,"|",3);
      if (ppc) vignette_RGB[2] = atoi(ppc);
      thread_signal();                                                           //  trigger update thread
   }

   if (strmatch(event,"load"))                                                   //  load saved curve
   {
      file = zgetfile("load curve from a file",MWIN,"file",saved_curves_folder);
      if (! file) return 1;
      fid = fopen(file,"r");
      zfree(file);
      if (! fid) return 1;
      splcurve_load(sd,fid);
      fclose(fid);
      Vign_curvedit(0);
      thread_signal();
      return 1;
   }

   if (strmatch(event,"save"))                                                   //  save curve to file
   {
      file = zgetfile("save curve to a file",MWIN,"save",saved_curves_folder);
      if (! file) return 1;
      pp = zstrdup(file,"vignette",8);
      zfree(file);
      file = pp;
      pp = strrchr(file,'/');                                                    //  force .curve extension
      if (pp) pp = strrchr(pp,'.');
      if (pp) strcpy(pp,".curve");
      else strcat(file,".curve");
      fid = fopen(file,"w");
      zfree(file);
      if (! fid) return 1;
      splcurve_save(sd,fid);
      fclose(fid);
      return 1;
   }

   return 1;
}


//  get mouse position and set new center for vignette

void vign_mousefunc()                                                            //  mouse function
{
   if (! LMclick && ! Mdrag) return;
   LMclick = 0;

   vign_cx = Mxposn;                                                             //  new vignette center = mouse position
   vign_cy = Myposn;

   Mxdrag = Mydrag = 0;

   thread_signal();                                                              //  trigger image update
   return;
}


//  this function is called when the curve is edited

void Vign_curvedit(int)
{
   thread_signal();                                                              //  update image
   return;
}


//  thread function

void * Vign_thread(void *)
{
   void * Vign_wthread(void *arg);

   do_wthreads(Vign_wthread,NWT);                                                //  worker threads

   CEF->Fmods++;
   CEF->Fsaved = 0;

   Fpaint2();
   return 0;
}


//  working thread

void * Vign_wthread(void *arg)
{
   float       *pix1, *pix3;
   int         index, ii, kk, px, py, dist = 0;
   float       cx, cy, rad, radx, rady, f1, f2, xval, yval;
   float       R1, G1, B1, R3, G3, B3;
   float       max$;
   spldat      *sd = EFvignette.sd;

   cx = vign_cx;                                                                 //  vignette center (mouse)
   cy = vign_cy;

   index = *((int *) arg);

   for (py = index; py < E3pxm->hh; py += NWT)                                   //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      ii = py * E3pxm->ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];                                                              //  input RGB
      G1 = pix1[1];
      B1 = pix1[2];

      radx = px - cx;                                                            //  distance from vignette center
      rady = py - cy;
      rad = sqrtf(radx*radx + rady*rady);                                        //  (px,py) distance from center

      xval = rad / vign_rad;                                                     //  scale 0 to 1.0
      kk = 999.0 * xval;                                                         //  scale 0 to 999
      if (kk > 999) kk = 999;                                                    //  beyond radius

      yval = sd->yval[0][kk];                                                    //  brightness curve y-value 0 to 1.0
      if (yval > 1.0) yval = 1.0;
      yval = 2.0 * yval;                                                         //  0 to 2.0

      R3 = yval * R1;                                                            //  adjust brightness
      G3 = yval * G1;
      B3 = yval * B1;

      yval = sd->yval[1][kk];                                                    //  color curve y-value 0 to 1.0
      if (yval > 1.0) yval = 1.0;
      f1 = yval;                                                                 //  0 to 1.0   new color
      f2 = 1.0 - f1;                                                             //  1.0 to 0   old color

      R3 = f1 * vignette_RGB[0] + f2 * R3;                                       //  mix input and vignette color
      G3 = f1 * vignette_RGB[1] + f2 * G3;
      B3 = f1 * vignette_RGB[2] + f2 * B3;

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }

      RGBFIX(R3,G3,B3)
      
      pix3[0] = R3;
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  Rescale an image while leaving selected areas unchanged.

namespace area_rescale_names
{
   editfunc    EFarea_rescale;

   int      Fsetups = 0;
   int      dragx, dragy;
   int      E3ww, E3hh;
   char     *sqrow, *sqcol;
   int      Nsqrow, Nsqcol;
   int      *npx, *npy;

   int    dialog_event(zdialog *zd, cchar *event);
   void   setups();
   void   cleanups();
   void   mousefunc();
   void   warpfunc();
   void   *warpthread(void *);
}


//  menu function

void m_area_rescale(GtkWidget *, cchar *menu)
{
   using namespace area_rescale_names;

   cchar  *message = " Select areas to remain unchanged. \n"
                     " Pull image from upper left corner. \n"
                     " When finished, press [ OK ].";

   F1_help_topic = "area rescale";

   Plog(1,"m_area_rescale \n");

   EFarea_rescale.menuname = "Area Rescale";
   EFarea_rescale.Farea = 2;                                                     //  select area usable
   EFarea_rescale.mousefunc = mousefunc;                                         //  mouse function
   if (! edit_setup(EFarea_rescale)) return;                                     //  setup edit

   PXM_addalpha(E0pxm);
   PXM_addalpha(E1pxm);
   PXM_addalpha(E3pxm);

   zdialog *zd = zdialog_new("Area Rescale",Mwin,"Proceed","OK","Cancel",null);
   EFarea_rescale.zd = zd;
   zdialog_add_widget(zd,"label","lab1","dialog",message,"space=3");

   zdialog_run(zd,dialog_event,"save");                                          //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int area_rescale_names::dialog_event(zdialog * zd, cchar *event)
{
   using namespace area_rescale_names;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   
   if (! zd->zstat) return 1;                                                    //  wait for completion
   
   if (zd->zstat == 1) {                                                         //  [proceed]
      zd->zstat = 0;                                                             //  keep dialog active
      if (sa_stat != 3)
         zmessageACK(Mwin,"select areas first");
      else setups();                                                             //  start drag and warp
      return 1;
   }
   
   if (zd->zstat != 2 || dragx + dragy == 0) {                                   //  [cancel] or no change
      edit_cancel(0);
      cleanups();
      return 1;
   }

   edit_done(0);                                                                 //  [ OK ]
   cleanups();
   return 1;
}


//  do setups based on select area data

void area_rescale_names::setups()
{
   int      ii, spx, spy, sum;

   cleanups();                                                                   //  free prior if any

   dragx = dragy = 0;                                                            //  no drag data

   E3ww = E3pxm->ww;                                                             //  image dimensions
   E3hh = E3pxm->hh;

   sqrow = (char *) zmalloc(E3hh,"area rescale");                                //  maps squishable rows/cols
   sqcol = (char *) zmalloc(E3ww,"area rescale");
   memset(sqrow,1,E3hh);                                                         //  mark all rows/cols squishable
   memset(sqcol,1,E3ww);
   
   for (spy = 0; spy < E3hh; spy++)                                              //  loop all source pixels
   for (spx = 0; spx < E3ww; spx++)
   {
      ii = spy * E3ww + spx;                                                     //  pixel within area?
      if (sa_pixmap[ii]) sqrow[spy] = sqcol[spx] = 0;                            //  mark row/col non-squishable
   }
   
   Nsqrow = Nsqcol = 0;
   for (spy = 0; spy < E3hh; spy++)                                              //  count total squishable rows/cols
      Nsqrow += sqrow[spy];
   for (spx = 0; spx < E3ww; spx++)
      Nsqcol += sqcol[spx];

   npx = (int *) zmalloc(E3ww * sizeof(int),"area rescale");                     //  count of squishable rows/cols
   npy = (int *) zmalloc(E3hh * sizeof(int),"area rescale");                     //    predeeding a given row/col
   
   for (sum = spx = 0; spx < E3ww; spx++)
   {
      if (sqcol[spx]) sum++;
      npx[spx] = sum;
   }

   for (sum = spy = 0; spy < E3hh; spy++)
   {
      if (sqrow[spy]) sum++;
      npy[spy] = sum;                                                            //  squishable rows < spy
   }

   Fsetups = 1;
   sa_clear();                                                                   //  clear area
   takeMouse(mousefunc,dragcursor);                                              //  connect mouse function
   return;
}


//  free allocated memory

void area_rescale_names::cleanups()
{
   if (! Fsetups) return;
   Fsetups = 0;
   zfree(sqrow);
   zfree(sqcol);   
   zfree(npx);
   zfree(npy);
   return;
}


//  mouse function

void area_rescale_names::mousefunc()
{
   using namespace area_rescale_names;
   
   float    R;
   
   if (Mxdrag || Mydrag)                                                         //  mouse drag underway
   {
      R = 1.0 * Mxdown / E3ww;                                                   //  ignore drag not from NW corner
      if (R > 0.2) return;
      R = 1.0 * Mydown / E3hh;
      if (R > 0.2) return;
      dragx = Mxdrag - Mxdown;                                                   //  drag amount
      dragy = Mydrag - Mydown;
      warpfunc();                                                                //  drag image
      Mxdrag = Mydrag = 0;
   }

   return;
}


//  warp function

void area_rescale_names::warpfunc()
{
   using namespace area_rescale_names;

   do_wthreads(warpthread,NWT);                                                  //  worker threads

   CEF->Fmods++;
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return;
}


//  warp thread

void * area_rescale_names::warpthread(void *arg)
{
   using namespace area_rescale_names;

   int      index = *((int *) (arg));
   int      spx, spy, dpx, dpy;
   float    Rx, Ry;
   float    *spix, *dpix;
   int      nc = E1pxm->nc, pcc = nc * sizeof(float);
   
   for (spy = index; spy < E3hh; spy += NWT)                                     //  loop all source pixels
   for (spx = 0; spx < E3ww; spx++)
   {
      if (spx < dragx || spy < dragy) {                                          //  pixels < dragx/dragy:
         spix = PXMpix(E3pxm,spx,spy);                                           //    black, transparent
         memset(spix,0,pcc);
      }

      Rx = 1.0 * npx[spx] / Nsqcol;                                              //  squishable pixel ratios, 0-1
      Ry = 1.0 * npy[spy] / Nsqrow;
      
      dpx = spx + dragx * (1.0 - Rx);                                            //  destination pixel
      dpy = spy + dragy * (1.0 - Ry);

      if (dpx < 0 || dpx > E3ww-1) continue;                                     //  necessary, why? 
      if (dpy < 0 || dpy > E3hh-1) continue;
      
      dpix = PXMpix(E3pxm,dpx,dpy);                                              //  source pixel >> destination pixel
      spix = PXMpix(E1pxm,spx,spy);
      memcpy(dpix,spix,pcc);
   }

   return 0;
}


