/********************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2020 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************

   Fotoxx image edit - Repair menu functions

   m_sharpen               sharpen an image
   m_fix_motionblur        sharpen blur from camera motion
   m_fix_motionblur_RL     same, using Richardson-Lucy algorithm
   m_blur                  blur an image
   m_blur_background       blur background (called via m_blur())
   m_add_motionblur        add motion blur to an image
   m_denoise               remove noise from an image
   m_redeyes               remove red-eyes from flash photos
   m_smart_erase           replace pixels inside selected areas with background
   m_adjust_RGB            adjust brightness/color using RGB or CMY colors
   m_adjust_HSL            adjust color using HSL model
   m_match_colors          adjust image colors to match those in a chosen image
   m_brite_ramp            adjust brightness graduaally across the image
   m_remove_dust           remove dust specs from an image
   m_chromatic1            fix lateral CA (color bands increasing with radius)
   m_chromatic2            fix axial CA (color bands on dark/bright edges)

*********************************************************************************/

#define EX extern                                                                //  enable extern declarations
#include "fotoxx.h"                                                              //  (variables in fotoxx.h are refs)

/********************************************************************************/


//  image sharpen functions

namespace sharpen_names
{
   int      E3ww, E3hh;                                                          //  image dimensions
   int      UM_radius, UM_amount, UM_thresh;
   int      GR_amount, GR_thresh;
   int      KH_radius;
   int      MD_radius, MD_dark, MD_light, *MD_britemap;
   float    RL_radius;
   int      RL_iters;
   int      FMB, FMBRL;                                                          //  fix motion blur, 2 methods            21.0
   char     sharp_function[8] = "";

   int      brhood_radius;
   float    brhood_kernel[200][200];                                             //  up to radius = 99
   char     brhood_method;                                                       //  g = gaussian, f = flat distribution
   float    *brhood_brightness;                                                  //  neighborhood brightness per pixel

   VOL int  sharp_cancel, brhood_cancel;                                         //  avoid GCC optimizing code away

   editfunc    EFsharp;
   char        editparms[100] = "";
}


//  menu function

void m_sharpen(GtkWidget *, cchar *menu)
{
   using namespace sharpen_names;

   int    sharp_dialog_event(zdialog *zd, cchar *event);
   void * sharp_thread(void *);
   int    ii;

   F1_help_topic = "sharpen";

   EFsharp.menuname = "Sharpen";
   EFsharp.menufunc = m_sharpen;
   EFsharp.Farea = 2;                                                            //  select area usable
   EFsharp.threadfunc = sharp_thread;                                            //  thread function
   EFsharp.Frestart = 1;                                                         //  allow restart
   EFsharp.Fpaint = 1;                                                           //  use with paint edits OK
   EFsharp.Fscript = 1;                                                          //  scripting supported
   if (! edit_setup(EFsharp)) return;                                            //  setup edit

   E3ww = E3pxm->ww;                                                             //  image dimensions
   E3hh = E3pxm->hh;

/***
          _________________________________________
         |                 Sharpen                 |
         |                                         |
         |  [_] unsharp mask        radius  [__]   |
         |                          amount  [__]   |
         |                        threshold [__]   |
         |                                         |
         |  [_] gradient            amount  [__]   |
         |                        threshold [__]   |
         |                                         |
         |  [_] Kuwahara            radius  [__]   |
         |                                         |
         |  [_] median diff         radius  [__]   |
         |                           dark   [__]   |
         |                           light  [__]   |
         |                                         |
         |  [_] Richardson-Lucy     radius  [__]   |                             //  21.0
         |                       iterations [__]   |
         |                                         |
         |  [_] fix motion blur                    |                             //  21.0
         |  [_] fix motion blur (RL)               |                             //  21.0
         |                                         |
         |        [reset] [apply] [ OK ] [cancel]  |
         |_________________________________________|

***/

   zdialog *zd = zdialog_new("Sharpen",Mwin,Breset,Bapply,BOK,Bcancel,null);
   EFsharp.zd = zd;

   zdialog_add_widget(zd,"hbox","hbum","dialog",0,"space=5");                    //  unsharp mask
   zdialog_add_widget(zd,"vbox","vb21","hbum",0,"space=2");
   zdialog_add_widget(zd,"label","space","hbum",0,"expand");
   zdialog_add_widget(zd,"vbox","vb22","hbum",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vb23","hbum",0,"homog|space=2");
   zdialog_add_widget(zd,"check","UM","vb21","unsharp mask","space=5");
   zdialog_add_widget(zd,"label","lab21","vb22",Bradius);
   zdialog_add_widget(zd,"label","lab22","vb22",Bamount);
   zdialog_add_widget(zd,"label","lab23","vb22",Bthresh);
   zdialog_add_widget(zd,"zspin","radiusUM","vb23","1|20|1|2");
   zdialog_add_widget(zd,"zspin","amountUM","vb23","1|200|1|100");
   zdialog_add_widget(zd,"zspin","threshUM","vb23","1|100|1|0");

   zdialog_add_widget(zd,"hsep","sep3","dialog");                                //  gradient
   zdialog_add_widget(zd,"hbox","hbgr","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb31","hbgr",0,"space=2");
   zdialog_add_widget(zd,"label","space","hbgr",0,"expand");
   zdialog_add_widget(zd,"vbox","vb32","hbgr",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vb33","hbgr",0,"homog|space=2");
   zdialog_add_widget(zd,"check","GR","vb31","gradient","space=5");
   zdialog_add_widget(zd,"label","lab32","vb32",Bamount);
   zdialog_add_widget(zd,"label","lab33","vb32",Bthresh);
   zdialog_add_widget(zd,"zspin","amountGR","vb33","1|400|1|100");
   zdialog_add_widget(zd,"zspin","threshGR","vb33","1|100|1|0");

   zdialog_add_widget(zd,"hsep","sep4","dialog");                                //  kuwahara
   zdialog_add_widget(zd,"hbox","hbku","dialog",0,"space=5");
   zdialog_add_widget(zd,"check","KH","hbku","Kuwahara","space=3");
   zdialog_add_widget(zd,"label","space","hbku",0,"expand");
   zdialog_add_widget(zd,"label","lab42","hbku",Bradius,"space=3");
   zdialog_add_widget(zd,"zspin","radiusKH","hbku","1|9|1|1");

   zdialog_add_widget(zd,"hsep","sep5","dialog");                                //  median diff
   zdialog_add_widget(zd,"hbox","hbmd","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb51","hbmd",0,"space=2");
   zdialog_add_widget(zd,"label","space","hbmd",0,"expand");
   zdialog_add_widget(zd,"vbox","vb52","hbmd",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vb53","hbmd",0,"homog|space=2");
   zdialog_add_widget(zd,"check","MD","vb51","median diff","space=5");
   zdialog_add_widget(zd,"label","lab51","vb52",Bradius);
   zdialog_add_widget(zd,"label","lab52","vb52","dark");
   zdialog_add_widget(zd,"label","lab53","vb52","light");
   zdialog_add_widget(zd,"zspin","radiusMD","vb53","1|20|1|3");
   zdialog_add_widget(zd,"zspin","darkMD","vb53","0|50|1|1");
   zdialog_add_widget(zd,"zspin","lightMD","vb53","0|50|1|1");

   zdialog_add_widget(zd,"hsep","sep6","dialog");                                //  Richardson-Lucy                       21.0
   zdialog_add_widget(zd,"hbox","hbrl","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb61","hbrl",0,"space=2");
   zdialog_add_widget(zd,"label","space","hbrl",0,"expand");
   zdialog_add_widget(zd,"vbox","vb62","hbrl",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vb63","hbrl",0,"homog|space=2");
   zdialog_add_widget(zd,"check","RL","vb61","Richardson-Lucy","space=5");
   zdialog_add_widget(zd,"label","lab62","vb62",Bradius);
   zdialog_add_widget(zd,"label","lab63","vb62","Iterations");
   zdialog_add_widget(zd,"zspin","radiusRL","vb63","1|9|0.1|1");
   zdialog_add_widget(zd,"zspin","itersRL","vb63","1|100|1|10");

   zdialog_add_widget(zd,"hsep","sep7","dialog");                                //  fix motion blur                       21.0
   zdialog_add_widget(zd,"hbox","hbmb","dialog",0,"space=5");
   zdialog_add_widget(zd,"check","FMB","hbmb","fix motion blur","space=2");
   zdialog_add_widget(zd,"hbox","hbmbrl","dialog");
   zdialog_add_widget(zd,"check","FMBRL","hbmbrl","fix motion blur (RL)","space=2");

   zdialog_restore_inputs(zd);
   
   zdialog_fetch(zd,"UM",ii);                                                    //  set function from checkboxes
   if (ii) strcpy(sharp_function,"UM");
   zdialog_fetch(zd,"GR",ii);
   if (ii) strcpy(sharp_function,"GR");
   zdialog_fetch(zd,"KH",ii);
   if (ii) strcpy(sharp_function,"KH");
   zdialog_fetch(zd,"MD",ii);
   if (ii) strcpy(sharp_function,"MD");
   zdialog_fetch(zd,"RL",ii);                                                    //  21.0
   if (ii) strcpy(sharp_function,"RL");
   zdialog_fetch(zd,"FMB",ii);                                                   //  21.0
   if (ii) strcpy(sharp_function,"FMB");
   zdialog_fetch(zd,"FMBRL",ii);                                                 //  21.0
   if (ii) strcpy(sharp_function,"FMBRL");

   zdialog_run(zd,sharp_dialog_event,"save");                                    //  run dialog - parallel

   return;
}


//  dialog event and completion callback function

int sharp_dialog_event(zdialog *zd, cchar *event)                                //  reworked for script files
{
   using namespace sharpen_names;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"focus")) return 1;

   zdialog_fetch(zd,"radiusUM",UM_radius);                                       //  get all parameters
   zdialog_fetch(zd,"amountUM",UM_amount);
   zdialog_fetch(zd,"threshUM",UM_thresh);
   zdialog_fetch(zd,"amountGR",GR_amount);
   zdialog_fetch(zd,"threshGR",GR_thresh);
   zdialog_fetch(zd,"radiusKH",KH_radius);
   zdialog_fetch(zd,"radiusMD",MD_radius);
   zdialog_fetch(zd,"radiusRL",RL_radius);                                       //  21.0
   zdialog_fetch(zd,"itersRL",RL_iters);
   zdialog_fetch(zd,"darkMD",MD_dark);
   zdialog_fetch(zd,"lightMD",MD_light);
   zdialog_fetch(zd,"FMB",FMB);                                                  //  21.0
   zdialog_fetch(zd,"FMBRL",FMBRL);                                              //  21.0

   if (strmatch(event,"apply")) zd->zstat = 2;                                   //  from script file 
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  from f_open()
   
   sharp_cancel = 0;

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;
         edit_reset();
         return 1;
      }

      if (zd->zstat == 2) {                                                      //  apply
         zd->zstat = 0;
         edit_reset();

         if (strmatch(sharp_function,"FMB")) {                                   //  fix motion blur, new function         21.0
            sharp_cancel = 1;
            edit_cancel(0);
            m_fix_motionblur(0,0);
            return 1;
         }

         if (strmatch(sharp_function,"FMBRL")) {                                 //  fix motion blur, new function         21.0
            sharp_cancel = 1;                                                    //  (Richardson-Lucy algorithm)
            edit_cancel(0);
            m_fix_motionblur_RL(0,0);
            return 1;
         }

         if (*sharp_function) signal_thread();                                   //  start thread function
         else zmessageACK(Mwin,Bnoselection);                                    //  no choice made
         return 1;
      }

      if (zd->zstat == 3) {
         EFsharp.editparms = editparms;                                          //  log exif hist data                    21.0
         edit_done(0);                                                           //  done
         return 1;
      }

      sharp_cancel = 1;                                                          //  cancel or [x]
      edit_cancel(0);                                                            //  discard edit
      return 1;
   }

   if (strmatch(event,"blendwidth")) signal_thread();

   if (strstr("UM GR KH MD RL FMB FMBRL",event))
   {
      zdialog_stuff(zd,"UM",0);                                                  //  make checkboxes like radio buttons
      zdialog_stuff(zd,"GR",0);
      zdialog_stuff(zd,"KH",0);
      zdialog_stuff(zd,"MD",0);
      zdialog_stuff(zd,"RL",0);                                                  //  21.0
      zdialog_stuff(zd,"FMB",0);                                                 //  21.0
      zdialog_stuff(zd,"FMBRL",0);                                               //  21.0
      zdialog_stuff(zd,event,1);
      strcpy(sharp_function,event);                                              //  set chosen method
   }

   return 1;
}


//  sharpen image thread function

void * sharp_thread(void *)
{
   using namespace sharpen_names;

   int sharp_UM(void);
   int sharp_GR(void);
   int sharp_KH(void);
   int sharp_MD(void);
   int sharp_RL(void);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

      if (strmatch(sharp_function,"UM")) sharp_UM();
      if (strmatch(sharp_function,"GR")) sharp_GR();
      if (strmatch(sharp_function,"KH")) sharp_KH();
      if (strmatch(sharp_function,"MD")) sharp_MD();
      if (strmatch(sharp_function,"RL")) sharp_RL();

      CEF->Fmods++;
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();
   }

   return 0;
}


// ------------------------------------------------------------------------------

//  image sharpen function using unsharp mask

int sharp_UM()
{
   using namespace sharpen_names;

   void  britehood(int radius, char method);                                     //  compute neighborhood brightness
   void * sharp_UM_wthread(void *arg);

   int      cc;

   cc = E3ww * E3hh * sizeof(float);
   brhood_brightness = (float *) zmalloc(cc);

   britehood(UM_radius,'f');
   if (sharp_cancel) return 1;

   if (sa_stat == 3) progressmon_reset(sa_Npixel);                               //  initz. progress counter
   else  progressmon_reset(E3ww * E3hh);

   do_wthreads(sharp_UM_wthread,NWT);                                            //  worker threads

   progressmon_reset(0);

   zfree(brhood_brightness);
   
   snprintf(editparms,100,"unsharp mask %d %d",UM_amount,UM_thresh);             //  record edit hist                      21.0
   return 1;
}


void * sharp_UM_wthread(void *arg)                                               //  worker thread function
{
   using namespace sharpen_names;

   int         index = *((int *) arg);
   int         px, py, ii, dist = 0;
   float       amount, thresh, bright;
   float       mean, incr, ratio, f1, f2;
   float       red1, green1, blue1, red3, green3, blue3;
   float       *pix1, *pix3;
   float       max$;

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image3 pixels
   for (px = 0; px < E3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      amount = 0.01 * UM_amount;                                                 //  0.0 to 2.0
      thresh = 0.4 * UM_thresh;                                                  //  0 to 40 (256 max. possible)

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      bright = PIXBRIGHT(pix1);
      if (bright < 1) continue;                                                  //  effectively black
      ii = py * E3ww + px;
      mean = brhood_brightness[ii];

      incr = (bright - mean);
      if (fabsf(incr) < thresh) continue;                                        //  omit low-contrast pixels

      incr = incr * amount;                                                      //  0.0 to 2.0
      if (bright + incr > 255) incr = 255 - bright;
      ratio = (bright + incr) / bright;
      if (ratio < 0) ratio = 0;

      red1 = pix1[0];                                                            //  input RGB
      green1 = pix1[1];
      blue1 = pix1[2];

      red3 = ratio * red1;                                                       //  output RGB
      green3 = ratio * green1;
      blue3 = ratio * blue1;

      RGBFIX(red3,green3,blue3)                                                  //  21.0

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;

      busy_add(index,1);                                                         //  track progress
      if (sharp_cancel) break;
   }

   return 0;
}


// ------------------------------------------------------------------------------

//  sharpen image by increasing brightness gradient

int sharp_GR()
{
   using namespace sharpen_names;

   void * sharp_GR_wthread(void *arg);

   if (sa_stat == 3) progressmon_reset(sa_Npixel);                               //  initz. progress counter
   else  progressmon_reset(E3ww * E3hh);

   do_wthreads(sharp_GR_wthread,NWT);                                            //  worker threads

   progressmon_reset(0); 

   snprintf(editparms,100,"gradient %d %d",GR_amount,GR_thresh);                 //  record edit hist                      21.0
   return 1;
}


//  callable sharp_GR() used by rotate function
//  returns E3 = sharpened E3

void sharp_GR_callable(int amount, int thresh)
{
   using namespace sharpen_names;

   PXM *PXMtemp = E1pxm;                                                         //  save E1
   E1pxm = PXM_copy(E3pxm);                                                      //  copy E3 > E1
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;
   GR_amount = amount;
   GR_thresh = thresh;
   sharp_GR();                                                                   //  E3 = sharpened E1
   PXM_free(E1pxm);
   E1pxm = PXMtemp;                                                              //  restore org. E1
   return;
}


void * sharp_GR_wthread(void *arg)                                               //  worker thread function 
{
   using namespace sharpen_names;

   float       *pix1, *pix3;
   int         ii, px, py, dist = 0;
   int         nc = E1pxm->nc;
   float       amount, thresh;
   float       b1, b1x, b1y, b3x, b3y, b3, bf, f1, f2;
   float       red1, green1, blue1, red3, green3, blue3;
   float       max$;

   int         index = *((int *) arg);

   amount = 1 + 0.01 * GR_amount;                                                //  1.0 - 5.0
   thresh = GR_thresh;                                                           //  0 - 100

   for (py = index + 1; py < E3hh; py += NWT)                                    //  loop all image pixels
   for (px = 1; px < E3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      b1 = PIXBRIGHT(pix1);                                                      //  pixel brightness, 0 - 256
      if (b1 == 0) continue;                                                     //  black, don't change
      b1x = b1 - PIXBRIGHT(pix1-nc);                                             //  horiz. brightness gradient
      b1y = b1 - PIXBRIGHT(pix1-nc * E3ww);                                      //  vertical
      f1 = fabsf(b1x) + fabsf(b1y);                                              //  bugfix                                21.0

      if (f1 < thresh)                                                           //  moderate brightness change for
         f1 = f1 / thresh;                                                       //    pixels below threshold gradient
      else  f1 = 1.0;
      f2 = 1.0 - f1;

      b1x = b1x * amount;                                                        //  amplified gradient
      b1y = b1y * amount;

      b3x = PIXBRIGHT(pix1-nc) + b1x;                                            //  + prior pixel brightness
      b3y = PIXBRIGHT(pix1-nc * E3ww) + b1y;                                     //  = new brightness
      b3 = 0.5 * (b3x + b3y);

      b3 = f1 * b3 + f2 * b1;                                                    //  possibly moderated

      bf = b3 / b1;                                                              //  ratio of brightness change
      if (bf < 0) bf = 0;
      if (bf > 4) bf = 4;

      red1 = pix1[0];                                                            //  input RGB
      green1 = pix1[1];
      blue1 = pix1[2];

      red3 = bf * red1;                                                          //  output RGB
      green3 = bf * green1;
      blue3 = bf * blue1;

      RGBFIX(red3,green3,blue3)                                                  //  21.0

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;

      busy_add(index,1);                                                         //  track progress
      if (sharp_cancel) break;
   }

   return 0;
}


// ------------------------------------------------------------------------------

//  sharpen edges using the Kuwahara algorithm

int sharp_KH()
{
   using namespace sharpen_names;

   void * sharp_KH_wthread(void *arg);
   
   if (sa_stat == 3) progressmon_reset(sa_Npixel);                               //  initz. progress counter
   else  progressmon_reset(E3ww * E3hh);

   do_wthreads(sharp_KH_wthread,NWT);                                            //  worker threads

   progressmon_reset(0); 

   snprintf(editparms,100,"kuwahara %d",KH_radius);                              //  record edit hist                      21.0
   return 1;
}


void * sharp_KH_wthread(void *arg)                                               //  worker thread function
{
   using namespace sharpen_names;

   float       *pix1, *pix3;
   int         px, py, qx, qy, rx, ry;
   int         ii, rad, N, dist = 0;
   float       red, green, blue, red2, green2, blue2;
   float       vmin, vall, vred, vgreen, vblue;
   float       red3, green3, blue3;
   float       f1, f2;

   int      index = *((int *) arg);

   rad = KH_radius;                                                              //  user input radius
   N = (rad + 1) * (rad + 1);

   for (py = index + rad; py < E3hh-rad; py += NWT)                              //  loop all image pixels
   for (px = rad; px < E3ww-rad; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel is outside area
      }

      vmin = 99999;
      red3 = green3 = blue3 = 0;

      for (qy = py - rad; qy <= py; qy++)                                        //  loop all surrounding neighborhoods
      for (qx = px - rad; qx <= px; qx++)
      {
         red = green = blue = 0;
         red2 = green2 = blue2 = 0;

         for (ry = qy; ry <= qy + rad; ry++)                                     //  loop all pixels in neighborhood
         for (rx = qx; rx <= qx + rad; rx++)
         {
            pix1 = PXMpix(E1pxm,rx,ry);
            red += pix1[0];                                                      //  compute mean RGB and mean RGB**2
            red2 += pix1[0] * pix1[0];
            green += pix1[1];
            green2 += pix1[1] * pix1[1];
            blue += pix1[2];
            blue2 += pix1[2] * pix1[2];
         }

         red = red / N;                                                          //  mean RGB of neighborhood
         green = green / N;
         blue = blue / N;

         vred = red2 / N - red * red;                                            //  variance RGB
         vgreen = green2 / N - green * green;
         vblue = blue2 / N - blue * blue;

         vall = vred + vgreen + vblue;                                           //  save RGB values with least variance
         if (vall < vmin) {
            vmin = vall;
            red3 = red;
            green3 = green;
            blue3 = blue;
         }
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  if select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm,px,py);                                             //  input pixel
         red3 = f1 * red3 + f2 * pix1[0];
         green3 = f1 * green3 + f2 * pix1[1];
         blue3 = f1 * blue3 + f2 * pix1[2];
      }

      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;

      busy_add(index,1);                                                         //  track progress
      if (sharp_cancel) break;
   }

   return 0;
}


// ------------------------------------------------------------------------------

//  sharpen edges using the median difference algorithm

int sharp_MD()
{
   using namespace sharpen_names;

   void * sharp_MD_wthread(void *arg);
   
   int      px, py, ii;
   float    *pix1;
   
   MD_britemap = (int *) zmalloc(E3ww * E3hh * sizeof(int));
   
   for (py = 0; py < E3hh; py++)                                                 //  loop all pixels
   for (px = 0; px < E3ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);                                                //  initz. pixel brightness map
      ii = py * E3ww + px;
      MD_britemap[ii] = PIXBRIGHT(pix1);
   }

   if (sa_stat == 3) progressmon_reset(sa_Npixel);                               //  initz. progress counter
   else  progressmon_reset(E3ww * E3hh);

   do_wthreads(sharp_MD_wthread,NWT);

   progressmon_reset(0); 

   zfree(MD_britemap);

   snprintf(editparms,100,"median diff %d %d %d",MD_radius,MD_dark,MD_light);    //  record edit hist                      21.0
   return 1;
}


void * sharp_MD_wthread(void *arg)                                               //  worker thread function
{
   using namespace sharpen_names;
   
   int         index = *((int *) arg);
   int         rad, dark, light, *britemap;
   int         ii, px, py, dist = 0;
   int         dy, dx, ns;
   float       R, G, B, R2, G2, B2;
   float       F, f1, f2;
   float       *pix1, *pix3;
   int         bright, median;
   int         bsortN[1681];                                                     //  radius <= 20 (41 x 41 pixels)
   
   rad = MD_radius;                                                              //  parameters from dialog
   dark = MD_dark;
   light = MD_light;
   britemap = MD_britemap;

   for (py = index+rad; py < E3hh-rad; py += NWT)                                //  loop all image3 pixels
   for (px = rad; px < E3ww-rad; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  source pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  target pixel

      ns = 0;

      for (dy = py-rad; dy <= py+rad; dy++)                                      //  loop surrounding pixels
      for (dx = px-rad; dx <= px+rad; dx++)                                      //  get brightness values
      {
         ii = dy * E3ww + dx;
         bsortN[ns] = britemap[ii];
         ns++;
      }

      HeapSort(bsortN,ns);                                                       //  sort the pixels
      median = bsortN[ns/2];                                                     //  median brightness
      
      R = pix3[0];
      G = pix3[1];
      B = pix3[2];
      
      bright = PIXBRIGHT(pix3);
      
      if (bright < median) {
         F = 1.0 - 0.1 * dark * (median - bright) / (median + 50);
         R2 = R * F;
         G2 = G * F;
         B2 = B * F;
         if (R2 > 0 && G2 > 0 && B2 > 0) {
            R = R2;
            G = G2;
            B = B2;
         }
      }
      
      if (bright > median) {
         F = 1.0 + 0.03 * light * (bright - median) / (median + 50);
         R2 = R * F;
         G2 = G * F;
         B2 = B * F;
         if (R2 < 255 && G2 < 255 && B2 < 255) {
            R = R2;
            G = G2;
            B = B2;
         }
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R = f1 * R + f2 * pix1[0];
         G = f1 * G + f2 * pix1[1];
         B = f1 * B + f2 * pix1[2];
      }
      
      pix3[0] = R;
      pix3[1] = G;
      pix3[2] = B;

      busy_add(index,1);                                                         //  track progress
      if (sharp_cancel) break;
   }

   return 0;
}


// ------------------------------------------------------------------------------

//  Sharpen image using the Richardson-Lucy deconvolution algorithm
//  (best for camera focus error - point source blur disc is uniform)

int sharp_RL()                                                                   //  21.0
{
   using namespace sharpen_names;

   void * sharp_RL_wthread(void *arg);

   do_wthreads(sharp_RL_wthread,3);                                              //  3 threads for 3 RGB colors
   snprintf(editparms,100,"Richardson-Lucy %.1f %d",RL_radius,RL_iters);         //  record edit hist                      21.0
   return 1;
}


void * sharp_RL_wthread(void *arg)                                               //  worker thread function
{
   using namespace sharpen_names;
   
   void RLdecon(PXM *pxm, int rgb, float frad, int iter);

   int   rgb = *((int *) arg);

   progressmon_reset(6 * RL_iters);

   RLdecon(E3pxm,rgb,RL_radius,RL_iters);
   
   progressmon_reset(0);

   return 0;
}


//  Deblur out-of-focus image using Richardson-Lucy deconvolution
//    pxm     input image in PXM format (float RGB values)
//    rgb     RGB channel to deblur (0/1/2)
//    frad    input blur disc radius (found by trial and error)
//    iter    R-L algorithm iterations to use
//  Code below follows notation in Richardson-Lucy Wikipedia topic. 

void RLdecon(PXM *pxm, int rgb, float frad, int iter)
{
   int      ww, hh, marg;                                                        //  image dimensions
   int      px, py, rx, ry, sx, sy;
   int      ii, jj, dist = 0;
   
   float    *pix;                                                                //  pixel within input image
   float    *Di;                                                                 //  initial pixel values (blurred)
   float    *Uj;                                                                 //  current pixel values (deblurred
   float    *Ci;                                                                 //  R-L factor
   float    Pij;                                                                 //  R-L factor
   float    f1, f2;

   float    PSF[20][20];                                                         //  1x blur disc, radius < 10
   float    PSF2[200][200];                                                      //  9x blur disc, radius < 100
   int      irad, irad2, B, B2;
   float    frad2, R, S;

   ww = pxm->ww;                                                                 //  image dimensions
   hh = pxm->hh;

   Di = (float *) malloc(ww * hh * sizeof(float));                               //  allocate memory
   Uj = (float *) malloc(ww * hh * sizeof(float));
   Ci = (float *) malloc(ww * hh * sizeof(float));
   
//  compute 9x blur disc (point spread function PSF)
//  shrink to 1x size for higher precision

   irad = frad + 0.5;                                                            //  disc radius, next greater integer
   B = 2 * irad + 1;                                                             //  disc container array, B x B
   B2 = 9 * B;                                                                   //  9x container size
   irad2 = B2 / 2;                                                               //  container irad
   frad2 = 9 * frad;                                                             //  container frad
   
   for (ry = -irad2; ry <= +irad2; ry++)                                         //  build 9x disc
   for (rx = -irad2; rx <= +irad2; rx++)
   {
      R = sqrtf(rx * rx + ry * ry);
      if (R > frad2 + 0.5)                                                       //  pixel fully outside radius
         PSF2[99+rx][99+ry] = 0;
      else if (R < frad2 - 0.5)                                                  //  pixel fully inside radius
         PSF2[99+rx][99+ry] = 1.0;
      else                                                                       //  pixel strides radius
         PSF2[99+rx][99+ry] = frad2 + 0.5 - R;
   }

   for (ry = -irad; ry <= +irad; ry++)                                           //  loop 1x disc
   for (rx = -irad; rx <= +irad; rx++)
      PSF[9+rx][9+ry] = 0;                                                       //  clear to zeros
   
   for (ry = -irad2; ry <= +irad2; ry++)                                         //  loop 9x disc
   for (rx = -irad2; rx <= +irad2; rx++)
   {
      sx = (rx + irad2) / 9 - irad;                                              //  corresp. 1x disc cell
      sy = (ry + irad2) / 9 - irad;
      PSF[9+sx][9+sy] += PSF2[99+rx][99+ry];                                     //  aggregate 9x disc into 1x disc
   }

   S = 0;
   for (ry = -irad; ry <= +irad; ry++)                                           //  loop pixels in 1x disc
   for (rx = -irad; rx <= +irad; rx++)
      S += PSF[9+rx][9+ry];                                                      //  sum pixel values

   for (ry = -irad; ry <= +irad; ry++)                                           //  loop pixels in 1x disc
   for (rx = -irad; rx <= +irad; rx++)
      PSF[9+rx][9+ry] /= S;                                                      //  normalize to sum 1.0

/***  print blur disc

   if (rgb == 0) {
      printf("\nfrad: %.1f  irad: %d \n",frad,irad);
      for (ry = -irad; ry <= +irad; ry++) {
         for (rx = -irad; rx <= +irad; rx++)
            printf("%8.3f",PSF[9+rx][9+ry]);
         printf("\n");
      }
      printf("\n");
   }

***/

//  initialize Di and Uj from input image rgb values   

   for (py = 0; py < hh; py++)
   for (px = 0; px < ww; px++)
   {
      ii = py * ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel outside area
      }

      pix = PXMpix(pxm,px,py);
      Di[ii] = Uj[ii] = pix[rgb];
   }

//  loop algorithm iterations
   
   for (int tt = 0; tt < iter; tt++)
   {
   
//  compute Ci for each pixel i

      for (py = irad; py < hh-irad; py++)                                        //  loop all pixels i
      for (px = irad; px < ww-irad; px++)                                        //  (omit pixels within irad of edges)
      {
         ii = py * ww + px;

         if (sa_stat == 3) {                                                     //  select area active
            dist = sa_pixmap[ii];
            if (! dist) continue;                                                //  pixel outside area
         }

         Ci[ii] = 0;
         for (ry = -irad; ry <= +irad; ry++)                                     //  loop contributing pixels j
         for (rx = -irad; rx <= +irad; rx++) {
            Pij = PSF[9+rx][9+ry];                                               //  contribution factor 
            jj = ii + ry * ww + rx;
            Ci[ii] += Pij * Uj[jj];                                              //  aggregate
         }

         if (Ci[ii] <= 0) Ci[ii] = 1;
         if (isnan(Ci[ii])) Ci[ii] = 1;
      }

      busy_add(rgb,1);                                                           //  track progress (rgb = thread)

//  compute new Uj for each pixel j

      for (py = irad; py < hh-irad; py++)                                        //  loop all pixels j
      for (px = irad; px < ww-irad; px++)
      {
         jj = py * ww + px;

         if (sa_stat == 3) {                                                     //  select area active
            dist = sa_pixmap[jj];
            if (! dist) continue;                                                //  pixel outside area
         }

         S = 0;
         for (ry = -irad; ry <= +irad; ry++)                                     //  loop all pixels i that pixel j
         for (rx = -irad; rx <= +irad; rx++) {                                   //    contributes to 
            ii = jj + ry * ww + rx;
            Pij = PSF[9+rx][9+ry];                                               //  contribution factor
            S += Di[ii] / Ci[ii] * Pij;                                          //  Di / Ci * Pij
         }

         Uj[jj] = Uj[jj] * S;                                                    //  new Uj
      }

      busy_add(rgb,1);                                                           //  track progress (rgb = thread)
   }

   marg = 5 * irad + 0.3 * (irad * iter);                                        //  21.32

   for (py = marg; py < hh-marg; py++)                                           //  replace input image rgb values
   for (px = marg; px < ww-marg; px++)
   {
      ii = py * ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel outside area
      }

      if (isnan(Uj[ii])) Uj[ii] = 0;
      if (Uj[ii] < 0) Uj[ii] = 0;                                                //  modified pixel
      if (Uj[ii] > 255) Uj[ii] = 255;

      pix = PXMpix(pxm,px,py);                                                   //  output image pixel
      f1 = 1.0;
      f2 = 0.0;
      
      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
      }

      pix[rgb] = f1 * Uj[ii] + f2 * Di[ii]; 
   }
      
   free(Uj);
   free(Di);
   free(Ci);

   return;
}


// ------------------------------------------------------------------------------

//  Compute the mean brightness of all pixel neighborhoods,
//  using a Gaussian or a flat distribution for the weightings.
//  If a select area is active, only inside pixels are calculated.
//  The flat method is 10-100x faster than the Gaussian method.

void britehood(int radius, char method)
{
   using namespace sharpen_names;

   void * brhood_wthread(void *arg);

   int      rad, radflat2, dx, dy;
   float    kern;

   brhood_radius = radius;
   brhood_method = method;

   if (brhood_method == 'g')                                                     //  compute Gaussian kernel
   {                                                                             //  (not currently used)
      rad = brhood_radius;
      radflat2 = rad * rad;

      for (dy = -rad; dy <= rad; dy++)
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx*dx + dy*dy <= radflat2)                                          //  cells within radius
            kern = exp( - (dx*dx + dy*dy) / radflat2);
         else kern = 0;                                                          //  outside radius
         brhood_kernel[dy+rad][dx+rad] = kern;
      }
   }

   if (sa_stat == 3) progressmon_reset(sa_Npixel);                               //  initz. progress counter
   else  progressmon_reset(E3ww * E3hh);

   do_wthreads(brhood_wthread,NWT);                                              //  worker threads

   progressmon_reset(0);
   return;
}


//  worker thread function

void * brhood_wthread(void *arg)
{
   using namespace sharpen_names;

   int      index = *((int *) arg);
   int      rad = brhood_radius;
   int      ii, px, py, qx, qy, Fstart;
   float    kern, bsum, bsamp, bmean;
   float    *pixel;

   if (brhood_method == 'g')                                                     //  use round gaussian distribution
   {
      for (py = index; py < E3hh; py += NWT)
      for (px = 0; px < E3ww; px++)
      {
         if (sa_stat == 3 && sa_mode != mode_image) {                            //  select area, not whole image
            ii = py * E3ww + px;                                                 //    use only inside pixels
            if (! sa_pixmap[ii]) continue;
         }

         bsum = bsamp = 0;

         for (qy = py-rad; qy <= py+rad; qy++)                                   //  computed weighted sum of brightness
         for (qx = px-rad; qx <= px+rad; qx++)                                   //    for pixels in neighborhood
         {
            if (qy < 0 || qy > E3hh-1) continue;
            if (qx < 0 || qx > E3ww-1) continue;
            kern = brhood_kernel[qy+rad-py][qx+rad-px];
            pixel = PXMpix(E1pxm,qx,qy);
            bsum += PIXBRIGHT(pixel) * kern;                                     //  sum brightness * weight
            bsamp += kern;                                                       //  sum weights
         }

         bmean = bsum / bsamp;                                                   //  mean brightness
         ii = py * E3ww + px;
         brhood_brightness[ii] = bmean;                                          //  pixel value

         busy_add(index,1);                                                      //  track progress
         if (sharp_cancel) break;
      }
   }

   if (brhood_method == 'f')                                                     //  use square flat distribution
   {
      Fstart = 1;
      bsum = bsamp = 0;

      for (py = index; py < E3hh; py += NWT)
      for (px = 0; px < E3ww; px++)
      {
         if (sa_stat == 3 && sa_mode != mode_image) {                            //  select area, not whole image
            ii = py * E3ww + px;                                                 //     compute only inside pixels
            if (! sa_pixmap[ii]) {
               Fstart = 1;
               continue;
            }
         }

         if (px == 0) Fstart = 1;

         if (Fstart)
         {
            Fstart = 0;
            bsum = bsamp = 0;

            for (qy = py-rad; qy <= py+rad; qy++)                                //  add up all columns
            for (qx = px-rad; qx <= px+rad; qx++)
            {
               if (qy < 0 || qy > E3hh-1) continue;
               if (qx < 0 || qx > E3ww-1) continue;
               pixel = PXMpix(E1pxm,qx,qy);
               bsum += PIXBRIGHT(pixel);
               bsamp += 1;
            }
         }
         else
         {
            qx = px-rad-1;                                                       //  subtract first-1 column
            if (qx >= 0) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E3hh-1) continue;
                  pixel = PXMpix(E1pxm,qx,qy);
                  bsum -= PIXBRIGHT(pixel);
                  bsamp -= 1;
               }
            }
            qx = px+rad;                                                         //  add last column
            if (qx < E3ww) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E3hh-1) continue;
                  pixel = PXMpix(E1pxm,qx,qy);
                  bsum += PIXBRIGHT(pixel);
                  bsamp += 1;
               }
            }
         }

         bmean = bsum / bsamp;                                                   //  mean brightness
         ii = py * E3ww + px;
         brhood_brightness[ii] = bmean;

         busy_add(index,1);                                                      //  track progress
         if (sharp_cancel) break;
      }
   }

   return 0;
}


/********************************************************************************/

//  remove blur from camera motion

namespace  fix_motionblur_names
{
   editfunc    EFfixmotionblur;
   int         Eww, Ehh;                                                         //  image dimensions
   float       span;                                                             //  blur span, pixels
   float       angle;                                                            //  blur angle, 0-180 deg.
   int         supring;                                                          //  suppress ringing, 0-9
   int         supnoise;                                                         //  suppress noise, 0-9
   char        editparms[40];
   PXM         *E2pxm;
}


//  menu function

void m_fix_motionblur(GtkWidget *, const char *menu)                             //  21.0
{
   using namespace fix_motionblur_names;

   void   fix_motionblur_mousefunc();
   int    fix_motionblur_dialog_event(zdialog* zd, const char *event);
   void * fix_motionblur_thread(void *);

   cchar *hintmess = "Shift + drag mouse across image \n"
                     " to indicate blur direction";

   EFfixmotionblur.menufunc = m_fix_motionblur;
   EFfixmotionblur.menuname = "Fix Motion Blur";
   EFfixmotionblur.Farea = 2;                                                    //  select area usable
   EFfixmotionblur.threadfunc = fix_motionblur_thread;                           //  thread function
   EFfixmotionblur.mousefunc = fix_motionblur_mousefunc;                         //  mouse function

   if (! edit_setup(EFfixmotionblur)) return;                                    //  setup edit
   
   Eww = E3pxm->ww;                                                              //  image dimensions
   Ehh = E3pxm->hh;

/***
          ___________________________________
         |         Fix Motion Blur           |
         |                                   |
         | Shift + drag mouse across image   |
         |  to indicate blur direction       |
         |                                   |
         |  Blur Span (pixels)   [___]       |
         | Blur Angle (degrees)  [___]       |
         |   Suppress Ringing    [___]       |
         |    Suppress Noise     [___]       |
         |                                   |
         |           [Reset] [ OK ] [Cancel] |
         |___________________________________|
         
***/

   zdialog *zd = zdialog_new("Fix Motion Blur",Mwin,Breset,BOK,Bcancel,null);
   CEF->zd = zd;
   
   zdialog_add_widget(zd,"label","labhint","dialog",hintmess,"space=5");
   
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=4|homog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=8");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=4|homog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");
   zdialog_add_widget(zd,"label","labspan","vb1","Blur Span (pixels)");
   zdialog_add_widget(zd,"label","labangle","vb1","Blur Angle (degrees)");
   zdialog_add_widget(zd,"label","labsupR","vb1","Suppress Ringing");
   zdialog_add_widget(zd,"label","labsupN","vb1","Suppress Noise");
   zdialog_add_widget(zd,"zspin","span","vb2","0|40|0.1|1");
   zdialog_add_widget(zd,"zspin","angle","vb2","-180|180|0.1|0");
   zdialog_add_widget(zd,"zspin","supring","vb2","0|9|1|0");
   zdialog_add_widget(zd,"zspin","supnoise","vb2","0|9|1|0");

   zdialog_restore_inputs(zd);
   zdialog_run(zd,fix_motionblur_dialog_event);                                  //  run dialog - parallel
   zdialog_send_event(zd,"init");
   return;
}


//  dialog event and completion function

int fix_motionblur_dialog_event(zdialog *zd, const char *event)
{
   using namespace fix_motionblur_names;
   
   void  fix_motionblur_mousefunc();
   
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  [reset]
         zd->zstat = 0;                                                          //  keep dialog active
         edit_reset();
         zdialog_stuff(zd,"span",0);
         zdialog_stuff(zd,"angle",0);
         zdialog_stuff(zd,"supring",0);
         zdialog_stuff(zd,"supnoise",0);
         span = angle = supring = supnoise = 0;
      }

      else if (zd->zstat == 2) {                                                 //  [ OK ] commit edit 
         snprintf(editparms,40,"%.1f %.1f %d %d",span,angle,supring,supnoise);   //  log exif hist data                    21.0
         EFfixmotionblur.editparms = editparms;
         edit_done(0);
      }
      else edit_cancel(0);                                                       //  [cancel] or [x] discard edit
      return 1;
   }
   
   if (strmatch(event,"focus")) {
      takeMouse(fix_motionblur_mousefunc,0);                                     //  connect mouse
      return 1;
   }

   if (! strstr("init span angle supring supnoise",event))
      return 1;

   zdialog_fetch(zd,"angle",angle);                                              //  get all inputs
   if (angle < 0) {
      angle = 180 + angle;                                                       //  -1 --> 179
      zdialog_stuff(zd,"angle",angle);
   }
   if (angle >= 180) {                                                           //  180 --> 0
      angle = angle - 180;
      zdialog_stuff(zd,"angle",angle);
   }

   zdialog_fetch(zd,"span",span);
   zdialog_fetch(zd,"supring",supring);
   zdialog_fetch(zd,"supnoise",supnoise);
   
   signal_thread();                                                              //  process
   return 1;
}


//  mouse function - capture mouse drag direction and set rotate angle

void fix_motionblur_mousefunc()
{
   using namespace fix_motionblur_names;

   int         dx, dy;
   float       R;
   
   if (! KBshiftkey) {
      takeMouse(fix_motionblur_mousefunc,0);                                     //  set normal cursor
      return;
   }

   takeMouse(fix_motionblur_mousefunc,dragcursor);                               //  set drag cursor

   if (! Mxdrag && ! Mydrag) return;
   
   dx = Mxdrag - Mxdown;                                                         //  drag vector
   dy = Mydrag - Mydown;
   Mxdrag = Mydrag = 0;

   R = sqrtf(dx*dx + dy*dy);                                                     //  get angle of drag
   angle = RAD * acosf(dx/R);
   if (dy > 0) angle = 180 - angle;                                              //  top quadrant only, 0-180 deg.
   if (angle == 180) angle = 0;
   if (CEF) zdialog_stuff(CEF->zd,"angle",angle);

   return;
}


//  thread function

void * fix_motionblur_thread(void *)
{
   using namespace fix_motionblur_names;

   void  * fix_motionblur_wthread1(void *arg);                                   //  worker threads
   void  * fix_motionblur_wthread2(void *arg);
   void  * fix_motionblur_wthread3(void *arg);
      
   float    rotate;
   int      mww, mhh;
   
   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      zsleep(0.5);                                                               //  consolidate spin button input storm
      CEF->thread_pend = 0;

      paintlock(1);                                                              //  prevent window updates

      if (angle <= 90) rotate = angle;                                           //  avoid upside-down result
      else rotate = angle - 180;
      E2pxm = PXM_rotate(E1pxm,rotate);                                          //  rotate to make blur angle = 0
      
      do_wthreads(fix_motionblur_wthread1,NWT);                                  //  fill black margins from edge pixels
      
      PXM_free(E3pxm);
      E3pxm = PXM_copy(E2pxm);                                                   //  E2 = E3 = E9 = rotated E1
      E9pxm = PXM_copy(E2pxm);

      progressmon_reset(E3pxm->hh);                                              //  progress monitor = rows

      do_wthreads(fix_motionblur_wthread2,NWT);                                  //  remove motion blur

      progressmon_reset(0);

      PXM_free(E9pxm);
      PXM_free(E2pxm);
      E2pxm = PXM_rotate(E3pxm,-rotate);                                         //  un-rotate
      mww = (E2pxm->ww - Eww) / 2;
      mhh = (E2pxm->hh - Ehh) / 2;                                               //  cut-off margins from rotate
      PXM_free(E3pxm);
      E3pxm = PXM_copy_area(E2pxm,mww,mhh,Eww,Ehh);
      PXM_free(E2pxm);
      E2pxm = 0;
      
      if (sa_stat == 3)                                                          //  copy input image outside select area
         do_wthreads(fix_motionblur_wthread3,NWT);

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;                                                           //  not saved

      paintlock(0);
      Fpaint2();
   }

   return 0;                                                                     //  not executed, stop warning
}


//  propagate image edge pixels into the black margins created from rotation

void * fix_motionblur_wthread1(void *arg)
{
   using namespace fix_motionblur_names;
   
   int      index = *((int *) arg);
   int      Eww = E2pxm->ww;                                                     //  rotated and larger image
   int      Ehh = E2pxm->hh;
   int      px, py, pxL;
   float    *pix2, R, G, B;

   if (angle == 0) return 0;
   
   for (py = index; py < Ehh; py += NWT)                                         //  loop image rows
   {
      for (px = 0; px < Eww; px++)                                               //  loop columns
      {
         pix2 = PXMpix(E2pxm,px,py);
         if (pix2[0] > 0 || pix2[1] > 0 || pix2[2] > 0) break;                   //  find first non-black pixel in row
      }
      
      pxL = px + 2;                                                              //  use next pixel
      if (pxL < Eww) {
         pix2 = PXMpix(E2pxm,pxL,py);                                            //  get RGB values
         R = pix2[0];
         G = pix2[1];
         B = pix2[2];

         for (px = 0; px < pxL; px++) {                                          //  fill black pixels with RGB
            pix2 = PXMpix(E2pxm,px,py);
            pix2[0] = R;
            pix2[1] = G;
            pix2[2] = B;
         }
      }

      for (px = Eww-1; px > 0; px--)                                             //  find last non-black pixel in row
      {
         pix2 = PXMpix(E2pxm,px,py);
         if (pix2[0] > 0 || pix2[1] > 0 || pix2[2] > 0) break;
      }
      
      pxL = px - 2;                                                              //  use previous pixel
      if (pxL > 0) {
         pix2 = PXMpix(E2pxm,pxL,py);                                            //  get RGB values
         R = pix2[0];
         G = pix2[1];
         B = pix2[2];

         for (px = pxL+1; px < Eww; px++) {                                      //  fill black pixels with RGB
            pix2 = PXMpix(E2pxm,px,py);
            pix2[0] = R;
            pix2[1] = G;
            pix2[2] = B;
         }
      }
   }
   
   return 0;
}


//  fix motion blur worker thread function
//  E2 = E3 = rotated blurred image so that blur angle = 0
//  span = N: each blurred pixel is average of N input pixels

void * fix_motionblur_wthread2(void *arg)
{
   using namespace fix_motionblur_names;

   int      index = *((int *) arg);
   int      px, py, ii;
   float    *pix2, *pix3, *pixN;
   float    fspan, R, G, B;
   int      Eww = E3pxm->ww;                                                     //  rotated and larger image
   int      Ehh = E3pxm->hh;
   int      rgb, nc, pcc;
   int      lspan, ispan;
   float    v0, v1, v2;
   float    F1, F2, con;
   float    *pix9, *pix9a, *pix9b;

   if (span < 1) return 0;
   lspan = span * 2;
   ispan = span + 0.5;
   nc = E3pxm->nc;                                                               //  channels (3 or 4) 
   pcc = nc * sizeof(float);                                                     //  pixel size

   for (py = index; py < Ehh; py += NWT)                                         //  loop image rows
   {
      busy_add(index,1);                                                         //  count progress

      for (px = lspan; px < Eww-lspan; px++)                                     //  loop columns left to right
      {
         pix3 = PXMpix(E3pxm,px,py);                                             //  input/output pixel

         R = (span+1) * pix3[0];
         G = (span+1) * pix3[1];
         B = (span+1) * pix3[2];
         
         for (ii = 1; ii <= span; ii++) 
         {
            pixN = PXMpix(E3pxm,px-ii,py);
            R -= pixN[0];
            G -= pixN[1];
            B -= pixN[2];
         }
         
         if (ii > span) {
            fspan = span + 1 - ii;
            pixN = PXMpix(E3pxm,px-ii,py);
            R -= fspan * pixN[0];
            G -= fspan * pixN[1];
            B -= fspan * pixN[2];
         }
         
         if (R < 0) R = 0;
         if (G < 0) G = 0;
         if (B < 0) B = 0;

         if (R > 255) R = 255;
         if (G > 255) G = 255;
         if (B > 255) B = 255;

         F2 = 0.1 * supring;                                                     //  suppress ringing
         F1 = 1.0 - F2;
         R = F1 * R + F2 * pix3[0];
         G = F1 * G + F2 * pix3[1];
         B = F1 * B + F2 * pix3[2];

         pix3[0] = R;                                                            //  output pixel
         pix3[1] = G;
         pix3[2] = B;
      }

      for (px = Eww-lspan; px > lspan; px--)                                     //  loop columns right to left
      {
         pix2 = PXMpix(E2pxm,px,py);                                             //  input/output pixel

         R = (span+1) * pix2[0];
         G = (span+1) * pix2[1];
         B = (span+1) * pix2[2];
         
         for (ii = 1; ii <= span; ii++) 
         {
            pixN = PXMpix(E2pxm,px+ii,py);
            R -= pixN[0];
            G -= pixN[1];
            B -= pixN[2];
         }

         if (ii > span) {
            fspan = span + 1 - ii;
            pixN = PXMpix(E2pxm,px+ii,py);
            R -= fspan * pixN[0];
            G -= fspan * pixN[1];
            B -= fspan * pixN[2];
         }

         if (R < 0) R = 0;
         if (G < 0) G = 0;
         if (B < 0) B = 0;

         if (R > 255) R = 255;
         if (G > 255) G = 255;
         if (B > 255) B = 255;

         F2 = 0.1 * supring;                                                     //  suppress ringing
         F1 = 1.0 - F2;
         R = F1 * R + F2 * pix2[0];
         G = F1 * G + F2 * pix2[1];
         B = F1 * B + F2 * pix2[2];

         pix2[0] = R;                                                            //  output pixel
         pix2[1] = G;
         pix2[2] = B;
      }

      for (px = 0; px < Eww-span; px++)
      {
         pix3 = PXMpix(E3pxm,px,py);                                             //  merge left and right scans
         pix2 = PXMpix(E2pxm,px+ispan,py);
         pix3[0] = 0.5 * (pix3[0] + pix2[0]);
         pix3[1] = 0.5 * (pix3[1] + pix2[1]);
         pix3[2] = 0.5 * (pix3[2] + pix2[2]);
      }
      
      for (ii = 0; ii < supnoise; ii++)                                          //  suppress noise
      for (px = 0; px < Eww-2; px++)
      for (rgb = 0; rgb < 3; rgb++)
      {
         pix3 = PXMpix(E3pxm,px,py);
         v0 = pix3[rgb];
         v1 = pix3[rgb+nc];
         v2 = pix3[rgb+nc+nc];
         if (v1 < v0 && v1 < v2)
            v1 = 0.5 * (v0 + v2);
         else if (v1 > v0 && v1 > v2)
            v1 = 0.5 * (v0 + v2);
         pix3[rgb+nc] = v1;
      }
      
      for (px = Eww-1; px > ispan; px--)                                         //  fix position shift
      {
         pix2 = PXMpix(E3pxm,px,py);
         pix3 = PXMpix(E3pxm,px-ispan/2,py);
         memcpy(pix2,pix3,pcc);
      }         

      for (px = 1; px < Eww-1; px++)                                             //  suppress ringing
      for (rgb = 0; rgb < 3; rgb++)
      {
         pix9 = PXMpix(E9pxm,px,py);                                             //  blurred input image
         pix9a = pix9 - nc;
         pix9b = pix9 + nc;
         con = span * 0.0039 * fabsf(pix9a[rgb] - pix9b[rgb]);                   //  0 - 1 = max. contrast
         if (con > 1.0) con = 1.0;
         F1 = 0.1 * supring;                                                     //  0 - 1 = max. suppression
         F2 = F1 * (1.0 - con);                                                  //  min. contrast --> max. suppression
         pix3 = PXMpix(E3pxm,px,py);                                             //  sharpened output image
         pix3[rgb] = F2 * pix9[rgb] + (1.0 - F2) * pix3[rgb];
      }         
   }

   return 0;                                                                     //  exit thread
}


//  fix motion blur worker thread function
//  replace output with input image in areas outside the select area

void * fix_motionblur_wthread3(void *arg)
{
   using namespace fix_motionblur_names;

   int      index = *((int *) arg);
   int      px, py, ii, dist;
   float    *pix1, *pix3, f1, f2;
   int      Eww = E3pxm->ww;
   int      Ehh = E3pxm->hh;
   int      pcc = E1pxm->nc * sizeof(float);

   for (py = index; py < Ehh; py += NWT)
   for (px = 0; px < Eww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      ii = py * Eww + px;
      dist = sa_pixmap[ii];
      if (dist) {
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix1[0];
         pix3[1] = f1 * pix3[1] + f2 * pix1[1];
         pix3[2] = f1 * pix3[2] + f2 * pix1[2];
      }

      else memcpy(pix3,pix1,pcc);
   }
   
   return 0;
}


/********************************************************************************/

//  remove blur from camera motion - Richardson-Lucy method

namespace  fix_motionblur_RL_names
{
   editfunc    EFfixmotionblur;
   int         Eww, Ehh;                                                         //  image dimensions
   int         span;                                                             //  blur span, pixels
   int         angle;                                                            //  blur angle, 0-180 deg.
   int         iter;                                                             //  algorithm iterations
   int         supring;                                                          //  suppress ringing
   PXM         *E2pxm;
   char        editparms[40];
}


//  menu function

void m_fix_motionblur_RL(GtkWidget *, const char *menu)                          //  21.0
{
   using namespace fix_motionblur_RL_names;

   void   fix_motionblur_RL_mousefunc();
   int    fix_motionblur_RL_dialog_event(zdialog* zd, const char *event);
   void * fix_motionblur_RL_thread(void *);

   cchar *hintmess = "Shift + drag mouse across image \n"
                     " to indicate blur direction";

   EFfixmotionblur.menufunc = m_fix_motionblur_RL;
   EFfixmotionblur.menuname = "Fix Motion Blur (RL)";
   EFfixmotionblur.Farea = 2;                                                    //  select area usable
   EFfixmotionblur.threadfunc = fix_motionblur_RL_thread;                        //  thread function
   EFfixmotionblur.mousefunc = fix_motionblur_RL_mousefunc;                      //  mouse function

   if (! edit_setup(EFfixmotionblur)) return;                                    //  setup edit
   
   Eww = E3pxm->ww;                                                              //  image dimensions
   Ehh = E3pxm->hh;

/***
          ___________________________________
         |       Fix Motion Blur (RL)        |
         |                                   |
         | Shift + drag mouse across image   |
         |  to indicate blur direction       |
         |                                   |
         |  Blur Span (pixels)   [___]       |
         | Blur Angle (degrees)  [___]       |
         | Algorithm Iterations  [___]       |
         |   Suppress Ringing    [___]       |
         |                                   |
         |     [Reset] [Apply] [OK] [Cancel] |
         |___________________________________|
         
***/

   zdialog *zd = zdialog_new("Fix Motion Blur (RL)",Mwin,Breset,Bapply,BOK,Bcancel,null);
   CEF->zd = zd;
   
   zdialog_add_widget(zd,"label","labhint","dialog",hintmess,"space=5");
   
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=4|homog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=8");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=4|homog");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=3");
   zdialog_add_widget(zd,"label","labspan","vb1","Blur Span (pixels)");
   zdialog_add_widget(zd,"label","labangle","vb1","Blur Angle (degrees)");
   zdialog_add_widget(zd,"label","labiter","vb1","Algorithm Iterations");
   zdialog_add_widget(zd,"label","labsup","vb1","Suppress Ringing");
   zdialog_add_widget(zd,"zspin","span","vb2","0|40|1|0");
   zdialog_add_widget(zd,"zspin","angle","vb2","-180|180|1|0");
   zdialog_add_widget(zd,"zspin","iter","vb2","0|100|1|0");
   zdialog_add_widget(zd,"zspin","supring","vb2","0|9|1|0");

   zdialog_restore_inputs(zd);
   zdialog_run(zd,fix_motionblur_RL_dialog_event);                               //  run dialog - parallel
   zdialog_send_event(zd,"init");
   return;
}


//  dialog event and completion function

int fix_motionblur_RL_dialog_event(zdialog *zd, const char *event)
{
   using namespace fix_motionblur_RL_names;
   
   void  fix_motionblur_RL_mousefunc();
   
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  from f_open()
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  [reset]
         zd->zstat = 0;                                                          //  keep dialog active
         edit_reset();
         zdialog_stuff(zd,"span",0);
         zdialog_stuff(zd,"angle",0);
         zdialog_stuff(zd,"iter",0);
         zdialog_stuff(zd,"supring",0);
         span = angle = iter = supring = 0;
      }

      else if (zd->zstat == 2) {                                                 //  [apply]
         zd->zstat = 0;                                                          //  keep dialog active
         zdialog_fetch(zd,"angle",angle);                                        //  get all inputs
         zdialog_fetch(zd,"span",span);
         zdialog_fetch(zd,"iter",iter);
         zdialog_fetch(zd,"supring",supring);

         if (angle < 0) {
            angle = 180 + angle;                                                 //  -1 --> 179
            zdialog_stuff(zd,"angle",angle);
         }
         if (angle >= 180) {                                                     //  180 --> 0
            angle = angle - 180;
            zdialog_stuff(zd,"angle",angle);
         }
         
         signal_thread();                                                        //  process
      }

      else if (zd->zstat == 3) {                                                 //  [ OK ] commit edit 
         snprintf(editparms,40,"%d %d %d %d",span,angle,iter,supring);           //  log exif hist data                    21.0
         EFfixmotionblur.editparms = editparms;
         edit_done(0);
      }

      else edit_cancel(0);                                                       //  [cancel] or [x] discard edit
      return 1;
   }
   
   if (strmatch(event,"focus")) {
      takeMouse(fix_motionblur_RL_mousefunc,0);                                  //  connect mouse
      return 1;
   }

   return 1;
}


//  mouse function - capture mouse drag direction and set rotate angle

void fix_motionblur_RL_mousefunc()
{
   using namespace fix_motionblur_RL_names;

   int         dx, dy;
   float       R;
   
   if (! KBshiftkey) {
      takeMouse(fix_motionblur_RL_mousefunc,0);
      return;
   }

   takeMouse(fix_motionblur_RL_mousefunc,dragcursor);

   if (! Mxdrag && ! Mydrag) return;
   
   dx = Mxdrag - Mxdown;                                                         //  drag vector
   dy = Mydrag - Mydown;
   Mxdrag = Mydrag = 0;

   R = sqrtf(dx*dx + dy*dy);                                                     //  get angle of drag
   angle = RAD * acosf(dx/R);
   if (dy > 0) angle = 180 - angle;                                              //  top quadrant only, 0-180 deg.
   if (angle == 180) angle = 0;
   if (CEF) zdialog_stuff(CEF->zd,"angle",angle);

   return;
}


//  thread function

void * fix_motionblur_RL_thread(void *)
{
   using namespace fix_motionblur_RL_names;

   void  * fix_motionblur_RL_wthread0(void *arg);                                //  worker threads
   void  * fix_motionblur_RL_wthread1(void *arg);
   void  * fix_motionblur_RL_wthread2(void *arg);
      
   float    rotate;
   int      mww, mhh;
   
   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request
      
      paintlock(1);                                                              //  prevent window updates

      if (angle <= 90) rotate = angle;                                           //  avoid upside-down result
      else rotate = angle - 180;

      PXM_free(E3pxm);
      E3pxm = PXM_rotate(E1pxm,rotate);                                          //  rotate to make blur angle 0
      E9pxm = PXM_copy(E3pxm);                                                   //  E3 = E9 = rotated E1
      
      do_wthreads(fix_motionblur_RL_wthread0,NWT);                               //  fill black margins from edge pixels

      progressmon_reset(E3pxm->hh);                                              //  progress monitor = rows

      do_wthreads(fix_motionblur_RL_wthread1,NWT);                               //  do worker thread

      progressmon_reset(0);

      PXM_free(E9pxm);
      E9pxm = 0;

      E2pxm = PXM_rotate(E3pxm,-rotate);                                         //  un-rotate
      mww = (E2pxm->ww - Eww) / 2;
      mhh = (E2pxm->hh - Ehh) / 2;                                               //  cut-off margins from rotate
      PXM_free(E3pxm);
      E3pxm = PXM_copy_area(E2pxm,mww,mhh,Eww,Ehh);
      PXM_free(E2pxm);
      E2pxm = 0;

      if (sa_stat == 3)                                                          //  process select area
         do_wthreads(fix_motionblur_RL_wthread2,NWT);

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;                                                           //  not saved

      paintlock(0);
      Fpaint2();
   }

   return 0;                                                                     //  not executed, stop warning
}


//  propagate image edge pixels into the black margins created from rotation

void * fix_motionblur_RL_wthread0(void *arg)
{
   using namespace fix_motionblur_RL_names;
   
   int      index = *((int *) arg);
   int      Eww = E3pxm->ww;                                                     //  rotated and larger image
   int      Ehh = E3pxm->hh;
   int      px, py, pxL;
   float    *pix3, R, G, B;

   if (angle == 0) return 0;
   
   for (py = index; py < Ehh; py += NWT)                                         //  loop image rows
   {
      for (px = 0; px < Eww; px++)                                               //  loop columns
      {
         pix3 = PXMpix(E3pxm,px,py);
         if (pix3[0] > 0 || pix3[1] > 0 || pix3[2] > 0) break;                   //  find first non-black pixel in row
      }
      
      pxL = px + 2;                                                              //  use next pixel
      if (pxL < Eww) {
         pix3 = PXMpix(E3pxm,pxL,py);                                            //  get RGB values
         R = pix3[0];
         G = pix3[1];
         B = pix3[2];

         for (px = 0; px < pxL; px++) {                                          //  fill black pixels with RGB
            pix3 = PXMpix(E3pxm,px,py);
            pix3[0] = R;
            pix3[1] = G;
            pix3[2] = B;
         }
      }

      for (px = Eww-1; px > 0; px--)                                             //  find last non-black pixel in row
      {
         pix3 = PXMpix(E3pxm,px,py);
         if (pix3[0] > 0 || pix3[1] > 0 || pix3[2] > 0) break;
      }
      
      pxL = px - 2;                                                              //  use previous pixel
      if (pxL > 0) {
         pix3 = PXMpix(E3pxm,pxL,py);                                            //  get RGB values
         R = pix3[0];
         G = pix3[1];
         B = pix3[2];

         for (px = pxL+1; px < Eww; px++) {                                      //  fill black pixels with RGB
            pix3 = PXMpix(E3pxm,px,py);
            pix3[0] = R;
            pix3[1] = G;
            pix3[2] = B;
         }
      }
   }
   
   return 0;
}


//  fix motion blur worker thread function
//  E3 = rotated blurred image so that blur angle = 0
//  span = N: each blurred pixel is average of N input pixels

void * fix_motionblur_RL_wthread1(void *arg)
{
   using namespace fix_motionblur_RL_names;

   void RLdecon(float *pixels, int Np, int Nd, int Nt);

   int      index = *((int *) arg);
   int      px, py, rgb;
   int      Eww = E3pxm->ww;                                                     //  rotated and larger image
   int      Ehh = E3pxm->hh;
   int      nc = E3pxm->nc;
   int      pcc = nc * sizeof(float);
   float    *pix2, *pix3, *RLval;
   float    *pix9, *pix9a, *pix9b;
   float    con, F1, F2;
   
   if (span < 1) return 0;
   
   RLval = (float *) zmalloc(Eww * sizeof(float));

   for (py = index; py < Ehh; py += NWT)                                         //  loop image rows
   {
      busy_add(index,1);                                                         //  count progress
      
      for (rgb = 0; rgb < 2; rgb++)                                              //  loop RGB
      {
         for (px = 0; px < Eww; px++)                                            //  loop pixels in row
         {
            pix3 = PXMpix(E3pxm,px,py);                                          //  one RGB value per pixel
            RLval[px] = pix3[rgb];
         }

         RLdecon(RLval,Eww,span,iter);                                           //  do R-L algorithm on pixel array

         for (px = 0; px < Eww; px++)
         {
            pix3 = PXMpix(E3pxm,px,py);                                          //  save revised RGB values
            pix3[rgb] = RLval[px];
         }
      }

      for (px = Eww-1; px > span; px--)                                          //  fix position shift
      {
         pix3 = PXMpix(E3pxm,px,py);
         pix2 = PXMpix(E3pxm,px-span/2,py);
         memcpy(pix3,pix2,pcc);
      }
      
      for (px = 1; px < Eww-1; px++)                                             //  suppress ringing
      for (rgb = 0; rgb < 3; rgb++)
      {
         pix9 = PXMpix(E9pxm,px,py);                                             //  blurred input image
         pix9a = pix9 - nc;
         pix9b = pix9 + nc;
         con = span * 0.0039 * fabsf(pix9a[rgb] - pix9b[rgb]);                   //  0 - 1 = max. contrast
         if (con > 1.0) con = 1.0;
         F1 = 0.1 * supring;                                                     //  0 - 1 = max. suppression
         F2 = F1 * (1.0 - con);                                                  //  min. contrast --> max. suppression
         pix3 = PXMpix(E3pxm,px,py);                                             //  sharpened output image
         pix3[rgb] = F2 * pix9[rgb] + (1.0 - F2) * pix3[rgb];
      }         
   }

   zfree(RLval);
   return 0;                                                                     //  exit thread
}


//  fix motion blur worker thread function
//  replace output with input image in areas outside the select area

void * fix_motionblur_RL_wthread2(void *arg)
{
   using namespace fix_motionblur_RL_names;

   int      index = *((int *) arg);
   int      px, py, ii, dist;
   float    *pix1, *pix3, f1, f2;
   int      pcc = E1pxm->nc * sizeof(float);

   for (py = index; py < Ehh; py += NWT)
   for (px = 0; px < Eww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      ii = py * Eww + px;
      dist = sa_pixmap[ii];
      if (dist) {
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix1[0];
         pix3[1] = f1 * pix3[1] + f2 * pix1[1];
         pix3[2] = f1 * pix3[2] + f2 * pix1[2];
      }

      else memcpy(pix3,pix1,pcc);
   }
   
   return 0;
}


/********************************************************************************

   Richardson-Lucy deconvolution for special case: linear uniform blur.
      pixels[]       row of motion blurred pixels in blur direction (one RGB channel)
      Np             pixel row length
      Nd             blur span: Nd pixels contribute equally to each blurred pixel
                                (inclusive: Nd = 1 means no blur)
      Nt             algorithm iterations 
   Variable names follow Wikipedia article on Richardson-Lucy deconvolution.

***/

void RLdecon(float *pixels, int Np, int Nd, int Nt)
{
   int      ii, jj, tt;
   float    Pij = 1.0 / Nd;                                                      //  pixel contribution factor
   float    *Ci = (float *) malloc(Np * sizeof(float));                          //  Ci factor per pixel
   float    *Uj = (float *) malloc(Np * sizeof(float));                          //  old/new estimated value per pixel

   for (jj = 0; jj < Np; jj++)                                                   //  initial Uj per pixel jj
      Uj[jj] = pixels[jj];

   for (tt = 0; tt < Nt; tt++)                                                   //  algorithm iterations
   {
      for (ii = Nd; ii < Np-Nd; ii++)                                            //  compute Ci per pixel ii
      {
         Ci[ii] = 0;
         for (jj = 0; jj < Nd; jj++)                                             //  Nd pixels contributing to ii
            Ci[ii] += Pij * Uj[ii-jj];
         if (Ci[ii] <= 0) Ci[ii] = 1;
      }
      
      for (jj = Nd; jj < Np-Nd-Nd; jj++)                                         //  compute new Uj per pixel jj
      {
         float S = 0;
         for (ii = 0; ii < Nd; ii++)                                             //  Nd pixels contributing to pixel jj
            S += pixels[jj+ii] / Ci[jj+ii] * Pij;
         Uj[jj] = Uj[jj] * S;                                                    //  new Uj replaces old Uj
      }
   }

   for (jj = 0; jj < Np; jj++) 
   {
      pixels[jj] = Uj[jj];
      if (pixels[jj] < 0) pixels[jj] = 0;
      if (pixels[jj] > 255) pixels[jj] = 255;
   }
   
   free(Ci);
   free(Uj);
   return;
}


/********************************************************************************/

//  image blur function

namespace blur_names
{
   int         Fnormblur;                                                        //  normal blur
   float       Nblur_radius;                                                     //  blur radius
   float       blur_weight[1415];                                                //  blur radius limit 999 

   int         Fradblur;                                                         //  radial blur
   int         RBrad, RBlen;                                                     //  radial blur radius, length
   int         Cx, Cy;                                                           //  image center of radial blur
   
   int         Fdirblur;                                                         //  directed blur
   float       Dmdx, Dmdy, Dmdw, Dmdh;
   float       DD, Dspan, Dintens;
   
   int         Fgradblur;                                                        //  graduated blur
   float       gblur_radius;                                                     //  blur radius
   int         con_limit;                                                        //  contrast limit
   uint8       *pixcon;
   int         pixseq_done[122][122];                                            //  up to gblur_radius = 60
   int         pixseq_angle[1000];
   int         pixseq_dx[13000], pixseq_dy[13000];
   int         pixseq_rad[13000];
   int         max1 = 999, max2 = 12999;                                         //  for later overflow check
   
   int         Fpaintblur;                                                       //  paint blur
   int         pmode = 1;                                                        //  1/2 = blend/restore
   int         pblur_radius = 20;                                                //  mouse radius
   float       powcent, powedge;                                                 //  power at center and edge
   float       kernel[402][402];                                                 //  radius limit 200
   int         mousex, mousey;                                                   //  mouse click/drag position
   
   int         Fblurbackground;                                                  //  blur background via select area
   int         Faddmotionblur;                                                   //  add motion blur to image              21.0

   editfunc    EFblur;
   VOL int     Fcancel;
   PXM         *E2pxm;
   int         E3ww, E3hh;                                                       //  image dimensions
}


//  menu function

void m_blur(GtkWidget *, cchar *menu)                                            //  consolidate all blur functions
{
   using namespace blur_names;

   int    blur_dialog_event(zdialog *zd, cchar *event);
   void   blur_mousefunc();
   void * blur_thread(void *);

   cchar  *radblur_tip = "Click to set center";
   cchar  *dirblur_tip = "Pull image using the mouse";
   cchar  *paintblur_tip = "left drag: blend image \n"
                           "right drag: restore image";
   cchar  *blurbackground_tip = "blur image outside selected area";

   F1_help_topic = "blur";

   EFblur.menuname = "Blur";
   EFblur.menufunc = m_blur;
   EFblur.Farea = 2;                                                             //  select area usable
   EFblur.threadfunc = blur_thread;                                              //  thread function
   EFblur.mousefunc = blur_mousefunc;
   EFblur.Frestart = 1;                                                          //  allow restart
   if (! edit_setup(EFblur)) return;                                             //  setup edit

/***
          _________________________________________
         |             Blur Image                  |
         |                                         |
         |  [x] Normal Blur   Radius [____]        |
         |  - - - - - - - - - - - - - - - - - - -  |
         |  [x] Radial Blur                        |
         |  Radius [___] Length [___]              |                             //  central area, no blur                 21.0
         |  Center X [___] Y [___]                 |
         |  - - - - - - - - - - - - - - - - - - -  |
         |  [X] Directed Blur                      |
         |  Blur Span  [___]  Intensity   [___]    |
         |  - - - - - - - - - - - - - - - - - - -  |
         |  [X] Graduated Blur                     |
         |  Radius [___]  Contrast Limit [___]     |
         |  - - - - - - - - - - - - - - - - - - -  |
         |  [X] Paint Blur                         |
         |  Radius [___]  Power [___]  Edge [___]  |
         |  - - - - - - - - - - - - - - - - - - -  |
         |  [X] Blur Background                    |
         |  [X] Add Motion Blur                    |                             //  21.0
         |                                         |
         |         [Reset] [Apply] [ OK ] [Cancel] |
         |_________________________________________|

***/

   zdialog *zd = zdialog_new("Blur Radius",Mwin,Breset,Bapply,BOK,Bcancel,null);
   EFblur.zd = zd;

   zdialog_add_widget(zd,"hbox","hbnb","dialog");
   zdialog_add_widget(zd,"check","Fnormblur","hbnb","Normal Blur","space=2");
   zdialog_add_widget(zd,"label","space","hbnb",0,"space=5");
   zdialog_add_widget(zd,"label","labrad","hbnb",Bradius,"space=5");
   zdialog_add_widget(zd,"zspin","Nblur_radius","hbnb","1|999|1|10","space=5|size=3");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbrb1","dialog");
   zdialog_add_widget(zd,"check","Fradblur","hbrb1","Radial Blur","space=2");
   zdialog_add_widget(zd,"hbox","hbrb2","dialog");
   zdialog_add_widget(zd,"label","labrbr","hbrb2",Bradius,"space=5");
   zdialog_add_widget(zd,"zspin","RBrad","hbrb2","1|999|1|100","space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbrb2",0,"space=5");
   zdialog_add_widget(zd,"label","labrbl","hbrb2",Blength,"space=5");
   zdialog_add_widget(zd,"zspin","RBlen","hbrb2","1|999|1|100","space=3|size=3");
   zdialog_add_widget(zd,"hbox","hbrb3","dialog");
   zdialog_add_widget(zd,"label","labc","hbrb3",Bcenter,"space=5");
   zdialog_add_widget(zd,"label","labcx","hbrb3","X","space=3");
   zdialog_add_widget(zd,"zentry","Cx","hbrb3",0,"space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbrb3",0,"space=5");
   zdialog_add_widget(zd,"label","labcy","hbrb3","Y","space=3");
   zdialog_add_widget(zd,"zentry","Cy","hbrb3",0,"space=3|size=3");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbdb1","dialog");
   zdialog_add_widget(zd,"check","Fdirblur","hbdb1","Directed Blur","space=2");
   zdialog_add_widget(zd,"hbox","hbdb2","dialog");
   zdialog_add_widget(zd,"label","labspan","hbdb2","Blur Span","space=5");
   zdialog_add_widget(zd,"zspin","span","hbdb2","0.00|1.0|0.01|0.1","space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbdb2",0,"space=5");
   zdialog_add_widget(zd,"label","labint","hbdb2","Intensity");
   zdialog_add_widget(zd,"zspin","intens","hbdb2","0.00|1.0|0.01|0.2","space=3|size=3");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbgb1","dialog");
   zdialog_add_widget(zd,"check","Fgradblur","hbgb1","Graduated Blur","space=2");
   zdialog_add_widget(zd,"hbox","hbgb2","dialog");
   zdialog_add_widget(zd,"label","labgrad","hbgb2",Bradius,"space=5");
   zdialog_add_widget(zd,"zspin","gblur_radius","hbgb2","1|50|1|10","space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbgb2",0,"space=5");
   zdialog_add_widget(zd,"label","lablim","hbgb2","Contrast Limit");
   zdialog_add_widget(zd,"zspin","con_limit","hbgb2","1|255|1|50","space=3|size=3");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbpb1","dialog");
   zdialog_add_widget(zd,"check","Fpaintblur","hbpb1","Paint Blur","space=2");
   zdialog_add_widget(zd,"hbox","hbpb2","dialog");
   zdialog_add_widget(zd,"label","labpaint","hbpb2",Bradius,"space=5");
   zdialog_add_widget(zd,"zspin","pblur_radius","hbpb2","2|200|1|20","space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbpb2",0,"space=5");
   zdialog_add_widget(zd,"label","labpow","hbpb2","Power");
   zdialog_add_widget(zd,"zspin","powcent","hbpb2","0|100|1|30","space=3|size=3");
   zdialog_add_widget(zd,"label","space","hbpb2",0,"space=5");
   zdialog_add_widget(zd,"label","labedge","hbpb2",Bedge);
   zdialog_add_widget(zd,"zspin","powedge","hbpb2","0|100|1|10","space=3|size=3");
   
   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbbg","dialog");
   zdialog_add_widget(zd,"check","Fblurbackground","hbbg","Blur Background","space=2");
   zdialog_add_widget(zd,"hbox","hbamb","dialog");                                                                     //  21.0
   zdialog_add_widget(zd,"check","Faddmotionblur","hbamb","Add Motion Blur","space=2");

   zdialog_add_ttip(zd,"Fradblur",radblur_tip);
   zdialog_add_ttip(zd,"Fdirblur",dirblur_tip);
   zdialog_add_ttip(zd,"Fpaintblur",paintblur_tip);
   zdialog_add_ttip(zd,"Fblurbackground",blurbackground_tip);
   
   E3ww = E3pxm->ww;                                                             //  image dimensions
   E3hh = E3pxm->hh;

   Fcancel = 0;                                                                  //  initial status
   E2pxm = 0;

   Fnormblur = 0;                                                                //  default settings
   Nblur_radius = 10;

   Fradblur = 0;
   RBrad = 100;
   RBlen = 100;
   Cx = E3ww / 2;
   Cy = E3hh / 2;
   zdialog_stuff(zd,"Cx",Cx);
   zdialog_stuff(zd,"Cy",Cy);

   Fdirblur = 0;
   Dspan = 0.1;
   Dintens = 0.2;

   Fgradblur = 0;
   con_limit = 1;
   gblur_radius = 10;
   
   Fpaintblur = 0;
   pmode = 1;
   pblur_radius = 20;
   powcent = 30;
   powedge = 10;
   
   zdialog_restore_inputs(zd);

   zdialog_stuff(zd,"Fblurbackground",0);
   Fblurbackground = 0;

   zdialog_stuff(zd,"Faddmotionblur",0);                                         //  21.0
   Faddmotionblur = 0;

   zdialog_run(zd,blur_dialog_event,"save");                                     //  run dialog - parallel
   zdialog_send_event(zd,"pblur_radius");                                        //  get kernel initialized
   return;
}


//  dialog event and completion callback function

int blur_dialog_event(zdialog * zd, cchar *event)
{
   using namespace blur_names;

   void   blur_mousefunc();

   float    frad, kern;
   int      rad, dx, dy;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  from f_open()
   
   if (zstrstr("Fnormblur Fradblur Fdirblur Fgradblur Fpaintblur"                //  checkboxes work like radio buttons
               "Fblurbackground Faddmotionblur Ffixmotionblur",event)) {
      zdialog_stuff(zd,"Fnormblur",0);
      zdialog_stuff(zd,"Fradblur",0);
      zdialog_stuff(zd,"Fdirblur",0);
      zdialog_stuff(zd,"Fgradblur",0);
      zdialog_stuff(zd,"Fpaintblur",0);
      zdialog_stuff(zd,"Fblurbackground",0);
      zdialog_stuff(zd,"Faddmotionblur",0);
      zdialog_stuff(zd,event,1);
      zdialog_fetch(zd,"Fnormblur",Fnormblur);
      zdialog_fetch(zd,"Fradblur",Fradblur);
      zdialog_fetch(zd,"Fdirblur",Fdirblur);
      zdialog_fetch(zd,"Fgradblur",Fgradblur);
      zdialog_fetch(zd,"Fpaintblur",Fpaintblur);
      zdialog_fetch(zd,"Fblurbackground",Fblurbackground);
      zdialog_fetch(zd,"Faddmotionblur",Faddmotionblur);                         //  21.0
   }
   
   if (Fradblur || Fdirblur)                                                     //  connect mouse
      takeMouse(blur_mousefunc,dragcursor);
   else if (Fpaintblur)
      takeMouse(blur_mousefunc,0);
   else freeMouse();

   zdialog_fetch(zd,"Fnormblur",Fnormblur);                                      //  get all dialog inputs
   zdialog_fetch(zd,"Nblur_radius",Nblur_radius);

   zdialog_fetch(zd,"Fradblur",Fradblur);
   zdialog_fetch(zd,"RBrad",RBrad);
   zdialog_fetch(zd,"RBlen",RBlen);
   zdialog_fetch(zd,"Cx",Cx);
   zdialog_fetch(zd,"Cy",Cy);

   zdialog_fetch(zd,"Fdirblur",Fdirblur);
   zdialog_fetch(zd,"span",Dspan);
   zdialog_fetch(zd,"intens",Dintens);

   zdialog_fetch(zd,"Fgradblur",Fgradblur);
   zdialog_fetch(zd,"gblur_radius",gblur_radius);
   zdialog_fetch(zd,"con_limit",con_limit);

   zdialog_fetch(zd,"Fpaintblur",Fpaintblur);
   zdialog_fetch(zd,"pblur_radius",pblur_radius);
   zdialog_fetch(zd,"powcent",powcent);
   zdialog_fetch(zd,"powedge",powedge);
   
   if (zstrstr("pblur_radius powcent powedge",event))                            //  paint blur parameters
   {
      zdialog_fetch(zd,"pblur_radius",pblur_radius);                             //  mouse radius
      zdialog_fetch(zd,"powcent",powcent);                                       //  center transparency
      zdialog_fetch(zd,"powedge",powedge);                                       //  powedge transparency

      powcent = 0.01 * powcent;                                                  //  scale 0 ... 1
      powedge = 0.01 * powedge;
      rad = pblur_radius;

      for (dy = -rad; dy <= rad; dy++)                                           //  build kernel
      for (dx = -rad; dx <= rad; dx++)
      {
         frad = sqrt(dx*dx + dy*dy);
         kern = (rad - frad) / rad;                                              //  center ... powedge  >>  1 ... 0
         kern = kern * (powcent - powedge) + powedge;                            //  strength  center ... powedge
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         if (frad > rad) kern = 2;                                               //  beyond radius, within square
         kernel[dx+rad][dy+rad] = kern;
      }
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1)                                                        //  [reset]
      {
         zd->zstat = 0;                                                          //  keep dialog active
         edit_reset();
      }

      else if (zd->zstat == 2)                                                   //  [apply]
      {
         if (Fblurbackground) {                                                  //  cancel and start new function
            Fcancel = 1;
            edit_cancel(0);
            if (E2pxm) PXM_free(E2pxm);
            E2pxm = 0;
            m_blur_background(0,0);
            return 1;
         }

         if (Faddmotionblur) {                                                   //  cancel and start new function         21.0
            Fcancel = 1;
            edit_cancel(0);
            if (E2pxm) PXM_free(E2pxm);
            E2pxm = 0;
            m_add_motionblur(0,0);
            return 1;
         }
      
         zd->zstat = 0;                                                          //  keep dialog active
         signal_thread();                                                        //  trigger thread
         return 1;                                                               //  do not free E2 
      }

      else if (zd->zstat == 3)                                                   //  [ OK ]
         edit_done(0);

      else {                                                                     //  [cancel]
         Fcancel = 1;
         edit_cancel(0);                                                         //  discard edit
      }

      if (E2pxm) PXM_free(E2pxm);                                                //  free memory
      E2pxm = 0;
   }

   return 1;
}


//  blur mouse function

void blur_mousefunc()                                                            //  mouse function
{
   using namespace blur_names;
   
   if (! CEF) return;
   
   if (Fnormblur) 
   {
      freeMouse();
      return;
   }

   if (Fradblur && LMclick)                                                      //  radial blur, new center
   {
      zdialog *zd = CEF->zd;
      Cx = Mxposn;
      Cy = Myposn;
      zdialog_stuff(zd,"Cx",Cx);
      zdialog_stuff(zd,"Cy",Cy);
      LMclick = 0;
      signal_thread();
   }

   if (Fdirblur && (Mxdrag || Mydrag))                                           //  directed blur, mouse drag
   {
      Dmdx = Mxdown;                                                             //  drag origin
      Dmdy = Mydown;
      Dmdw = Mxdrag - Mxdown;                                                    //  drag increment
      Dmdh = Mydrag - Mydown;
      Mxdrag = Mydrag = 0;
      signal_thread();
   }
   
   if (Fpaintblur)                                                               //  paint blur
   {
      int      px, py, rr;

      if (LMclick || RMclick)                                                    //  mouse click
      {
         if (LMclick) pmode = 1;                                                 //  left click, paint
         if (RMclick) pmode = 2;                                                 //  right click, erase
         mousex = Mxclick;
         mousey = Myclick;
         signal_thread();
      }

      else if (Mxdrag || Mydrag)                                                 //  mouse drag in progress
      {
         if (Mbutton == 1) pmode = 1;                                            //  left drag, paint
         if (Mbutton == 3) pmode = 2;                                            //  right drag, erase
         mousex = Mxdrag;
         mousey = Mydrag;
         signal_thread();
      }

      cairo_t *cr = draw_context_create(gdkwin,draw_context);

      px = mousex - pblur_radius - 1;                                            //  repaint modified area
      py = mousey - pblur_radius - 1;
      rr = 2 * pblur_radius + 3;
      Fpaint3(px,py,rr,rr,cr);

      draw_mousecircle(Mxposn,Myposn,pblur_radius,0,cr);                         //  redraw mouse circle
      draw_context_destroy(draw_context);

      LMclick = RMclick = Mxdrag = Mydrag = 0;                                   //  reset mouse
   }

   return;
}


//  image blur thread function

void * blur_thread(void *)
{
   using namespace blur_names;

   void * normblur_wthread(void *);
   void * radblur_wthread(void *);
   void * dirblur_wthread(void *);
   void * gradblur_wthread(void *);
   void * paintblur_wthread(void *);

   int         ii, jj;
   float       rad, w, wsum;
   float       dd, d1, d2, d3, d4;
   int         px, py, dx, dy, adx, ady;
   float       *pix1, *pix2;
   float       contrast, maxcon;
   float       rad1, rad2, angle, astep;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request
      
      paintlock(1);                                                              //  block window paint

      if (Fnormblur)                                                             //  normal blur
      {
         if (E2pxm) PXM_free(E2pxm);
         E2pxm = PXM_copy(E1pxm);                                                //  intermediate image

         rad = Nblur_radius;
         wsum = 0;
         
         for (ii = 0; ii < rad; ii++)                                            //  set pixel weight per distance
         {                                                                       //      example, rad = 10
            w = 1.0 - ii / rad;                                                  //  dist:   0   1   2   3   5   7   9
            w = w * w;                                                           //  weight: 1  .81 .64 .49 .25 .09 .01
            blur_weight[ii] = w;
            wsum += w;
         }
         
         for (ii = 0; ii < rad; ii++)                                            //  make weights sum to 1.0
            blur_weight[ii] = blur_weight[ii] / wsum;

         if (sa_stat == 3) progressmon_reset(sa_Npixel * 2);                     //  initz. progress counter
         else  progressmon_reset(E3ww * E3hh * 2);

         do_wthreads(normblur_wthread,NWT);                                      //  worker threads
      }

      if (Fradblur)                                                              //  radial blur
      {
         if (E2pxm) PXM_free(E2pxm);

         if (Fnormblur)                                                          //  if normal blur done before,
            E2pxm = PXM_copy(E3pxm);                                             //    use the blur output image
         else E2pxm = PXM_copy(E1pxm);                                           //  else use the original image

         if (sa_stat == 3) progressmon_reset(sa_Npixel);                         //  initz. progress counter
         else  progressmon_reset(E3ww * E3hh);

         do_wthreads(radblur_wthread,NWT);                                       //  worker threads
      }
      
      if (Fdirblur)                                                              //  directed blur
      {
         d1 = (Dmdx-0) * (Dmdx-0) + (Dmdy-0) * (Dmdy-0);                         //  distance, mouse to 4 corners
         d2 = (E3ww-Dmdx) * (E3ww-Dmdx) + (Dmdy-0) * (Dmdy-0);
         d3 = (E3ww-Dmdx) * (E3ww-Dmdx) + (E3hh-Dmdy) * (E3hh-Dmdy);
         d4 = (Dmdx-0) * (Dmdx-0) + (E3hh-Dmdy) * (E3hh-Dmdy);

         dd = d1;
         if (d2 > dd) dd = d2;                                                   //  find greatest corner distance
         if (d3 > dd) dd = d3;
         if (d4 > dd) dd = d4;

         DD = dd * 0.5 * Dspan;

         do_wthreads(dirblur_wthread,NWT);                                       //  worker threads
      }
      
      if (Fgradblur)                                                             //  graduated blur
      {
         pixcon = (uint8 *) zmalloc(E3ww * E3hh);                                //  pixel contrast map

         for (py = 1; py < E3hh-1; py++)                                         //  loop interior pixels
         for (px = 1; px < E3ww-1; px++)
         {
            pix1 = PXMpix(E1pxm,px,py);                                          //  this pixel in base image E1
            contrast = maxcon = 0.0;

            for (dx = px-1; dx <= px+1; dx++)                                    //  loop neighbor pixels
            for (dy = py-1; dy <= py+1; dy++)
            {
               pix2 = PXMpix(E1pxm,dx,dy);
               contrast = 1.0 - PIXMATCH(pix1,pix2);                             //  contrast, 0-1
               if (contrast > maxcon) maxcon = contrast;
            }

            ii = py * E3ww + px;                                                 //  ii maps to (px,py)
            pixcon[ii] = 255 * maxcon;                                           //  pixel contrast, 0 to 255
         }

         rad1 = gblur_radius;

         for (dy = 0; dy <= 2*rad1; dy++)                                        //  no pixels mapped yet
         for (dx = 0; dx <= 2*rad1; dx++)
            pixseq_done[dx][dy] = 0;

         ii = jj = 0;

         astep = 0.5 / rad1;                                                     //  0.5 pixel steps at rad1 from center

         for (angle = 0; angle < 2*PI; angle += astep)                           //  loop full circle
         {
            pixseq_angle[ii] = jj;                                               //  start pixel sequence for this angle
            ii++;

            for (rad2 = 1; rad2 <= rad1; rad2++)                                 //  loop rad2 from center to edge
            {
               dx = lround(rad2 * cos(angle));                                   //  pixel at angle and rad2
               dy = lround(rad2 * sin(angle));
               adx = rad1 + dx;
               ady = rad1 + dy;
               if (pixseq_done[adx][ady]) continue;                              //  pixel already mapped
               pixseq_done[adx][ady] = 1;                                        //  map pixel
               pixseq_dx[jj] = dx;                                               //  save pixel sequence for angle
               pixseq_dy[jj] = dy;
               pixseq_rad[jj] = rad2;                                            //  pixel radius
               jj++;
            }
            pixseq_rad[jj] = 9999;                                               //  mark end of pixels for angle
            jj++;
         }

         pixseq_angle[ii] = 9999;                                                //  mark end of angle steps

         if (ii > max1 || jj > max2)                                             //  should not happen
            zappcrash("gradblur array overflow");

         if (sa_stat == 3) progressmon_reset(sa_Npixel);                         //  initz. progress counter
         else  progressmon_reset(E3ww * E3hh);

         do_wthreads(gradblur_wthread,NWT);                                      //  worker threads

         zfree(pixcon);
      }
      
      if (Fpaintblur)
         do_wthreads(paintblur_wthread,NWT);                                     //  worker threads
      
      progressmon_reset(0); 
      CEF->Fmods++;
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();
   }
   
   return 0;
}


//  normal blur worker thread

void * normblur_wthread(void *arg)
{
   using namespace blur_names;

   int      index = *((int *) arg);
   int      rad = Nblur_radius;
   int      ii, dist = 0;
   int      px, py, qx, qy;
   int      ylo, yhi, xlo, xhi;
   float    R, G, B, w1, w2, f1, f2;
   float    *pix1, *pix3, *pix2;

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (Fcancel) return 0;                                                     //  user cancel

      ii = py * E3ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  source pixel
      pix2 = PXMpix(E2pxm,px,py);                                                //  target pixel - intermediate image
      
      ylo = py - rad;
      if (ylo < 0) ylo = 0;
      yhi = py + rad;
      if (yhi > E3hh-1) yhi = E3hh - 1;
      
      R = G = B = 0;
      w2 = 0;

      for (qy = ylo; qy <= yhi; qy++)                                            //  loop pixels in same column
      {
         if (sa_stat == 3) {  
            ii = qy * E3ww + px;                                                 //  don't use pixels outside area   
            dist = sa_pixmap[ii];
            if (! dist) continue;
         }
         
         w1 = blur_weight[abs(qy-py)];                                           //  weight based on radius
         pix1 = PXMpix(E1pxm,px,qy);
         R += w1 * pix1[0];                                                      //  accumulate RGB * weight
         G += w1 * pix1[1];
         B += w1 * pix1[2];
         w2 += w1;                                                               //  accumulate weights
      }
      
      R = R / w2;                                                                //  normalize
      G = G / w2;
      B = B / w2;

      pix2[0] = R;                                                               //  weighted average
      pix2[1] = G;
      pix2[2] = B;

      busy_add(index,1);                                                         //  track progress
   }

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (Fcancel) return 0;                                                     //  user cancel

      ii = py * E3ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix2 = PXMpix(E2pxm,px,py);                                                //  source pixel - intermediate image
      pix3 = PXMpix(E3pxm,px,py);                                                //  target pixel - final image
      
      xlo = px - rad;
      if (xlo < 0) xlo = 0;
      xhi = px + rad;
      if (xhi > E3ww-1) xhi = E3ww - 1;
      
      R = G = B = 0;
      w2 = 0;

      for (qx = xlo; qx <= xhi; qx++)                                            //  loop pixels in same row
      {
         if (sa_stat == 3) {  
            ii = py * E3ww + qx;                                                 //  don't use pixels outside area   
            dist = sa_pixmap[ii];
            if (! dist) continue;
         }
         
         w1 = blur_weight[abs(qx-px)];                                           //  weight based on radius
         pix2 = PXMpix(E2pxm,qx,py);
         R += w1 * pix2[0];                                                      //  accumulate RGB * weight
         G += w1 * pix2[1];
         B += w1 * pix2[2];
         w2 += w1;                                                               //  accumulate weights
      }
      
      R = R / w2;                                                                //  normalize
      G = G / w2;
      B = B / w2;

      pix3[0] = R;                                                               //  weighted average
      pix3[1] = G;
      pix3[2] = B;

      busy_add(index,1);                                                         //  track progress
   }

   if (sa_stat == 3 && sa_blendwidth > 0)                                        //  select area has edge blend
   {
      for (py = index; py < E3hh; py += NWT)                                     //  loop all image pixels
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  omit pixels outside area
         if (dist >= sa_blendwidth) continue;                                    //  omit if > blendwidth from edge

         pix1 = PXMpix(E1pxm,px,py);                                             //  source pixel
         pix3 = PXMpix(E3pxm,px,py);                                             //  target pixel
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix1[0];
         pix3[1] = f1 * pix3[1] + f2 * pix1[1];
         pix3[2] = f1 * pix3[2] + f2 * pix1[2];
      }
   }

   return 0;
}


//  radial blur worker thread

void * radblur_wthread(void *arg)
{
   using namespace blur_names;

   int      index = *((int *) arg);
   int      ii, dist = 0;
   int      px, py, qx, qy, qz;
   float    RBlen2;
   float    *pix2, *pix3;
   float    R, Rx, Ry, Rz;
   float    f1, f2;
   int      Rsum, Gsum, Bsum, Npix;

   for (py = index+1; py < E3hh-1; py += NWT)                                    //  loop all image pixels
   for (px = 1; px < E3ww-1; px++)
   {
      if (Fcancel) return 0;                                                     //  user cancel

      ii = py * E3ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }
      
      Rsum = Gsum = Bsum = Npix = 0;                                             //  reset RGB sums
      
      R = sqrtf((px-Cx)*(px-Cx) + (py-Cy)*(py-Cy));                              //  distance (Cx,Cy) to (px,py)
      if (R == 0) continue;
      Rx = (px-Cx)/R;                                                            //  unit vector along (Cx,Cy) to (px,py)
      Ry = (py-Cy)/R;
      if (fabsf(Rx) > fabsf(Ry)) Rz = 1.0 / fabsf(Rx);                           //  Rz is 1.0 .. 1.414
      else Rz = 1.0 / fabsf(Ry);
      Rx = Rx * Rz;                                                              //  vector with max x/y component = 1
      Ry = Ry * Rz;
      
      RBlen2 = E3ww;                                                             //  lesser image dimension                21.0
      if (E3hh < E3ww) RBlen2 = E3hh;
      RBlen2 = RBlen * (R - RBrad) / RBlen2;                                     //  R = 0-max >> RBlen2 = 0..RBlen        21.0
      if (RBlen2 < 1) RBlen2 = 1;                                                //  central area, no blur
      
      for (qz = 0; qz < RBlen2; qz++)                                            //  loop (qx,qy) from (px,py) 
      {                                                                          //    in direction to (Cx,Cy)
         qx = px - qz * Rx;                                                      //      for distance RBlen
         qy = py - qz * Ry;
         if (qx < 0 || qx > E3ww-1) break;
         if (qy < 0 || qy > E3hh-1) break;

         if (sa_stat == 3) {  
            ii = qy * E3ww + qx;                                                 //  don't use pixels outside area   
            dist = sa_pixmap[ii];
            if (! dist) continue;
         }

         pix2 = PXMpix(E2pxm,qx,qy);                                             //  sum RGB values
         Rsum += pix2[0];
         Gsum += pix2[1];
         Bsum += pix2[2];
         Npix++;                                                                 //  count pixels in sum
      }
      
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel is average of 
      pix3[0] = Rsum / Npix;                                                     //   pixels in line of length RBlen
      pix3[1] = Gsum / Npix;                                                     //    from (px,py) to (Cx,Cy)
      pix3[2] = Bsum / Npix;

      busy_add(index,1);                                                         //  track progress
   }

   if (sa_stat == 3 && sa_blendwidth > 0)                                        //  select area has edge blend
   {
      for (py = index; py < E3hh; py += NWT)                                     //  loop all image pixels
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  omit pixels outside area
         if (dist >= sa_blendwidth) continue;                                    //  omit if > blendwidth from edge

         pix2 = PXMpix(E2pxm,px,py);                                             //  source pixel
         pix3 = PXMpix(E3pxm,px,py);                                             //  target pixel
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix2[0];
         pix3[1] = f1 * pix3[1] + f2 * pix2[1];
         pix3[2] = f1 * pix3[2] + f2 * pix2[2];
      }
   }

   return 0;
}


//  directed blur worker thread

void * dirblur_wthread(void *arg)
{
   using namespace blur_names;

   int      index = *((int *) arg);
   int      ii, px, py, dist = 0, vstat;
   float    d, mag, dispx, dispy;
   float    F1, F2, f1, f2;
   float    vpix[4], *pix1, *pix3;

   F1 = Dintens * Dintens;
   F2 = 1.0 - F1;

   for (py = index; py < E3hh; py += NWT)                                        //  process all pixels
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel outside area
      }

      d = (px-Dmdx)*(px-Dmdx) + (py-Dmdy)*(py-Dmdy);
      mag = (1.0 - d / DD);
      if (mag < 0) continue;

      mag = mag * mag;                                                           //  faster than pow(mag,4);
      mag = mag * mag;

      dispx = -Dmdw * mag;                                                       //  displacement = drag * mag
      dispy = -Dmdh * mag;

      vstat = vpixel(E3pxm,px+dispx,py+dispy,vpix);                              //  input virtual pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      if (vstat) {
         pix3[0] = F2 * pix3[0] + F1 * vpix[0];                                  //  output = input pixel blend
         pix3[1] = F2 * pix3[1] + F1 * vpix[1];
         pix3[2] = F2 * pix3[2] + F1 * vpix[2];
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  within edge blend area
         pix1 = PXMpix(E1pxm,px,py);                                             //  input pixel
         f1 = sa_blendfunc(dist);
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix1[0];
         pix3[1] = f1 * pix3[1] + f2 * pix1[1];
         pix3[2] = f1 * pix3[2] + f2 * pix1[2];
      }
   }

   return 0;                                                                     //  exit thread
}


//  graduated blur worker thread

void * gradblur_wthread(void *arg)
{
   using namespace blur_names;

   int      index = *((int *) arg);
   int      ii, jj, npix, dist = 0;
   int      px, py, dx, dy;
   float    red, green, blue, f1, f2;
   float    *pix1, *pix3, *pixN;
   int      rad, blurrad, con;
   
   for (py = index+1; py < E3hh-1; py += NWT)                                    //  loop interior pixels
   for (px = 1; px < E3ww-1; px++)
   {
      if (Fcancel) return 0;

      ii = py * E3ww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel outside area
      }

      if (pixcon[ii] > con_limit) continue;                                      //  high contrast pixel

      pix1 = PXMpix(E1pxm,px,py);                                                //  source pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  target pixel

      red = pix1[0];                                                             //  blur center pixel
      green = pix1[1];
      blue = pix1[2];
      npix = 1;

      blurrad = 1.0 + gblur_radius * (con_limit - pixcon[ii]) / con_limit;       //  blur radius for pixel, 1 - gblur_radius

      for (ii = 0; ii < 2000; ii++)                                              //  loop angle around center pixel
      {
         jj = pixseq_angle[ii];                                                  //  1st pixel for angle step ii
         if (jj == 9999) break;                                                  //  none, end of angle loop

         while (true)                                                            //  loop pixels from center to radius
         {
            rad = pixseq_rad[jj];                                                //  next pixel step radius
            if (rad > blurrad) break;                                            //  stop here if beyond blur radius
            dx = pixseq_dx[jj];                                                  //  next pixel step px/py
            dy = pixseq_dy[jj];
            if (px+dx < 0 || px+dx > E3ww-1) break;                              //  stop here if off the edge
            if (py+dy < 0 || py+dy > E3hh-1) break;

            pixN = PXMpix(E1pxm,px+dx,py+dy);
            con = 255 * (1.0 - PIXMATCH(pix1,pixN));
            if (con > con_limit) break;

            red += pixN[0];                                                      //  add pixel RGB values
            green += pixN[1];
            blue += pixN[2];
            npix++;
            jj++;
         }
      }

      pix3[0] = red / npix;                                                      //  average pixel values
      pix3[1] = green / npix;
      pix3[2] = blue / npix;
      
      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  within edge blend area
         f1 = sa_blendfunc(dist);
         f2 = 1.0 - f1;
         pix3[0] = f1 * pix3[0] + f2 * pix1[0];
         pix3[1] = f1 * pix3[1] + f2 * pix1[1];
         pix3[2] = f1 * pix3[2] + f2 * pix1[2];
      }

      busy_add(index,1);                                                         //  track progress
   }

   return 0;
}


//  paint blur worker thread

void * paintblur_wthread(void *arg)
{
   using namespace blur_names;
   
   int         index = *((int *) arg);
   float       *pix1, *pix3, *pixm;
   int         radius, radius2, npix;
   int         px, py, dx, dy, qx, qy, rx, ry, sx, sy;
   int         ii, dist = 0;
   float       kern, kern2, meanR, meanG, meanB;

   px = mousex;
   py = mousey;
   radius = pblur_radius;

   for (dy = -radius+index; dy <= radius; dy += NWT)                             //  loop within mouse radius
   for (dx = -radius; dx <= radius; dx++)
   {
      qx = px + dx;
      qy = py + dy;

      if (qx < 0 || qx > E3ww-1) continue;                                       //  off image
      if (qy < 0 || qy > E3hh-1) continue;

      if (sa_stat == 3) {                                                        //  select area active
         ii = qy * E3ww + qx;
         dist = sa_pixmap[ii];
         if (! dist) continue;                                                   //  pixel is outside area
      }

      kern = kernel[dx+radius][dy+radius];                                       //  mouse transparencies
      if (kern > 1) continue;                                                    //  outside mouse radius

      if (sa_stat == 3 && dist < sa_blendwidth)                                  //  within blend distance
         kern = kern * sa_blendfunc(dist);
         
      pix1 = PXMpix(E1pxm,qx,qy);                                                //  original pixel
      pix3 = PXMpix(E3pxm,qx,qy);                                                //  edited pixel
      
      meanR = meanG = meanB = npix = 0;
      radius2 = sqrtf(radius);                                                   //  radius = 2..99  >>  radius2 = 1..9
      
      for (ry = -radius2; ry <= radius2; ry++)
      for (rx = -radius2; rx <= radius2; rx++)
      {
         sx = qx + rx;
         sy = qy + ry;
         
         if (px - sx < -radius || px - sx > radius) continue;                    //  outside mouse radius
         if (py - sy < -radius || py - sy > radius) continue;

         if (sx < 0 || sx > E3ww-1) continue;                                    //  off image
         if (sy < 0 || sy > E3hh-1) continue;
         
         pixm = PXMpix(E3pxm,sx,sy);
         meanR += pixm[0];         
         meanG += pixm[1];         
         meanB += pixm[2];         
         npix++;
      }
      
      if (npix == 0) continue;
      
      meanR = meanR / npix;
      meanG = meanG / npix;
      meanB = meanB / npix;
      
      if (pmode == 1) {                                                          //  blend
         kern2 = 0.5 * kern;
         pix3[0] = kern2 * meanR + (1.0 - kern2) * pix3[0];                      //  pix3 tends to regional mean
         pix3[1] = kern2 * meanG + (1.0 - kern2) * pix3[1];
         pix3[2] = kern2 * meanB + (1.0 - kern2) * pix3[2];
      }

      if (pmode == 2) {                                                          //  restore
         kern2 = 0.1 * kern;
         pix3[0] = kern2 * pix1[0] + (1.0 - kern2) * pix3[0];                    //  pix3 tends to pix1
         pix3[1] = kern2 * pix1[1] + (1.0 - kern2) * pix3[1];
         pix3[2] = kern2 * pix1[2] + (1.0 - kern2) * pix3[2];
      }
   }

   return 0;
}


/********************************************************************************/

//  Blur Background
//  Blur the image outside of a selected area or areas. 
//  Blur increases with distance from selected area edges.

namespace blur_BG_names 
{
   int         conrad, incrad;               //  constant or increasing blur
   int         conbrad;                      //  constant blur radius
   int         minbrad;                      //  min. blur radius
   int         maxbrad;                      //  max. blur radius
   VOL int     Fcancel;                      //  GCC inconsistent
   int         E3ww, E3hh;                   //  image dimensions
   int         maxdist;                      //  max. area edge distance
   editfunc    EFblurBG;
}


//  called from main blur function, no separate menu

void m_blur_background(GtkWidget *, const char *menu)
{
   using namespace blur_BG_names;

   int    blur_BG_dialog_event(zdialog* zd, const char *event);
   void * blur_BG_thread(void *);

   EFblurBG.menufunc = m_blur_background;
   EFblurBG.menuname = "Blur Background";                                        //  function name
   EFblurBG.Farea = 2;                                                           //  select area usable (required)
   EFblurBG.threadfunc = blur_BG_thread;                                         //  thread function
      
   if (! edit_setup(EFblurBG)) return;                                           //  setup edit

   minbrad = 10;                                                                 //  defaults
   maxbrad = 20;
   conbrad = 10;
   conrad = 1;
   incrad = 0;
   Fcancel = 0;

   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;
   
/***
       ____________________________________
      |          Blur Background           |
      |                                    |
      |  [x] constant blur [ 20 ]          |
      |                                    |
      |  [x] increase blur with distance   |
      |    min. blur radius [ 20 ]         |
      |    max. blur radius [ 90 ]         |
      |                                    |
      |            [Apply] [ OK ] [Cancel] |
      |____________________________________|

***/

   zdialog *zd = zdialog_new("Blur Background",Mwin,Bapply,BOK,Bcancel,null);
   CEF->zd = zd;
   
   zdialog_add_widget(zd,"hbox","hbcon","dialog",0,"space=5");
   zdialog_add_widget(zd,"check","conrad","hbcon","constant blur","space=3");
   zdialog_add_widget(zd,"zspin","conbrad","hbcon","1|100|1|10","space=8");
   zdialog_add_widget(zd,"hbox","hbinc","dialog");
   zdialog_add_widget(zd,"check","incrad","hbinc","increase blur with distance","space=3");
   zdialog_add_widget(zd,"hbox","hbmin","dialog");
   zdialog_add_widget(zd,"label","labmin","hbmin","min. blur radius","space=8");
   zdialog_add_widget(zd,"zspin","minbrad","hbmin","0|100|1|10","space=3");
   zdialog_add_widget(zd,"hbox","hbmax","dialog");
   zdialog_add_widget(zd,"label","labmax","hbmax","max. blur radius","space=8");
   zdialog_add_widget(zd,"zspin","maxbrad","hbmax","1|100|1|20","space=3");
   
   zdialog_stuff(zd,"conrad",conrad);
   zdialog_stuff(zd,"incrad",incrad);

   zdialog_resize(zd,300,0);
   zdialog_restore_inputs(zd);                                                   //  restore previous inputs
   zdialog_run(zd,blur_BG_dialog_event,"save");                                  //  run dialog - parallel
   
   if (sa_stat != 3) m_select_area(0,0);                                         //  start select area dialog
   return;
}


//  blur_BG dialog event and completion function

int blur_BG_dialog_event(zdialog *zd, const char *event)                         //  blur_BG dialog event function
{
   using namespace blur_BG_names;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1)                                                        //  [apply]
      {
         zd->zstat = 0;                                                          //  keep dialog active

         if (sa_stat != 3) {
            zmessageACK(Mwin,"no active Select Area");
            m_select_area(0,0);
            return 1;
         }
         
         if (incrad && ! sa_edgecalc_done)                                       //  if increasing blur radius,
            sa_edgecalc();                                                       //    calc. area edge distances

         sa_show(0,0);
         edit_reset();
         signal_thread();
         return 1;
      }
      
      else if (zd->zstat == 2) {
         edit_done(0);                                                           //  [ OK ]
         if (zd_sela) zdialog_send_event(zd_sela,"done");                        //  kill select area dialog
      }

      else {
         Fcancel = 1;                                                            //  kill threads
         edit_cancel(0);                                                         //  [cancel] or [x], discard edit
         if (zd_sela) zdialog_send_event(zd_sela,"done");
      }

      return 1;
   }
   
   if (zstrstr("conrad incrad",event)) {
      zdialog_stuff(zd,"conrad",0);
      zdialog_stuff(zd,"incrad",0);
      zdialog_stuff(zd,event,1);
   }
   
   zdialog_fetch(zd,"conrad",conrad);
   zdialog_fetch(zd,"incrad",incrad);
   zdialog_fetch(zd,"conbrad",conbrad);
   zdialog_fetch(zd,"minbrad",minbrad);
   zdialog_fetch(zd,"maxbrad",maxbrad);
   
   return 1;
}


//  thread function - multiple working threads to update image

void * blur_BG_thread(void *)
{
   using namespace blur_BG_names;

   void  * blur_BG_wthread(void *arg);                                           //  worker thread
   
   int      ii, dist;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request
      
      paintlock(1);                                                              //  block window paint

      if (incrad && sa_edgecalc_done) {                                          //  if increasing blur radius,
         maxdist = 0;                                                            //    get max. area edge distance
         for (ii = 0; ii < E3ww * E3hh; ii++) {
            dist = sa_pixmap[ii];
            if (dist > maxdist) maxdist = dist;
         }
      }

      progressmon_reset(sa_Npixel);                                              //  initz. progress counter

      do_wthreads(blur_BG_wthread,NWT);                                          //  worker threads
      
      progressmon_reset(0);

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;                                                           //  not saved

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


void * blur_BG_wthread(void *arg)                                                //  worker thread function
{
   using namespace blur_BG_names;

   int         index = *((int *) (arg));
   int         ii, rad = 0, dist, npix;
   int         px, py, qx, qy;
   float       *pix1, *pix3;
   float       red, green, blue, F;
   
   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (Fcancel) break;                                                        //  cancel edit

      ii = py * E3ww + px;
      dist = sa_pixmap[ii];                                                      //  area edge distance
      if (! dist) continue;                                                      //  pixel outside the area
      
      if (conrad) rad = conbrad;                                                 //  use constant blur radius

      if (incrad) {                                                              //  use increasing blur radius
         if (! sa_edgecalc_done) return 0;                                       //    depending on edge distance
         rad = minbrad + (maxbrad - minbrad) * dist / maxdist;
      }

      npix = 0;
      red = green = blue = 0;      

      for (qy = py-rad; qy <= py+rad; qy++)                                      //  average surrounding pixels
      for (qx = px-rad; qx <= px+rad; qx++)
      {
         if (qy < 0 || qy > E3hh-1) continue;
         if (qx < 0 || qx > E3ww-1) continue;
         ii = qy * E3ww + qx;
         if (! sa_pixmap[ii]) continue;
         pix1 = PXMpix(E1pxm,qx,qy);
         red += pix1[0];
         green += pix1[1];
         blue += pix1[2];
         npix++;
      }
      
      F = 0.999 / npix;
      red = F * red;                                                             //  blurred pixel RGB
      green = F * green;
      blue = F * blue;
      pix3 = PXMpix(E3pxm,px,py);
      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;

      busy_add(index,1);                                                         //  count pixels done
   }
   
   return 0;                                                                     //  exit thread
}


/********************************************************************************/

//  add motion blur to an image

namespace  add_motionblur_names
{
   editfunc    EFaddmotionblur;
   int         Eww, Ehh;                                                         //  image dimensions
   int         span;                                                             //  blur span, pixels
   int         angle;                                                            //  blur angle, 0-180 deg.
}


//  menu function

void m_add_motionblur(GtkWidget *, const char *menu)                             //  21.0
{
   using namespace add_motionblur_names;

   void   add_motionblur_mousefunc();
   int    add_motionblur_dialog_event(zdialog* zd, const char *event);
   void * add_motionblur_thread(void *);

   cchar *hintmess = "Drag mouse across image \n"
                     " to indicate blur direction";

   EFaddmotionblur.menufunc = m_add_motionblur;
   EFaddmotionblur.menuname = "Add Motion Blur";
   EFaddmotionblur.Farea = 2;                                                    //  select area usable
   EFaddmotionblur.threadfunc = add_motionblur_thread;                           //  thread function
   EFaddmotionblur.mousefunc = add_motionblur_mousefunc;                         //  mouse function

   if (! edit_setup(EFaddmotionblur)) return;                                    //  setup edit
   
   Eww = E3pxm->ww;                                                              //  image dimensions
   Ehh = E3pxm->hh;

/***
          ___________________________________
         |         Add Motion Blur           |
         |                                   |
         | Drag mouse across image           |
         |  to indicate blur direction       |
         |                                   |
         | Blur Span (pixels) [___]          |
         | Blur Angle (degrees) [___]        |
         |                                   |
         | [Reset] [Apply] [ OK ] [Cancel]   |
         |___________________________________|
         
***/

   zdialog *zd = zdialog_new("Add Motion Blur",Mwin,Breset,Bapply,BOK,Bcancel,null);
   CEF->zd = zd;
   
   zdialog_add_widget(zd,"label","labhint","dialog",hintmess,"space=5");
   zdialog_add_widget(zd,"hbox","hbspan","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labspan","hbspan","Blur Span (pixels)","space=5");
   zdialog_add_widget(zd,"zspin","span","hbspan","0|50|1|0");
   zdialog_add_widget(zd,"label","space","hbspan",0,"space=20");
   zdialog_add_widget(zd,"hbox","hbangle","dialog");
   zdialog_add_widget(zd,"label","labangle","hbangle","Blur Angle (degrees)","space=5");
   zdialog_add_widget(zd,"zspin","angle","hbangle","0|180|1|0");
   zdialog_add_widget(zd,"label","space","hbangle",0,"space=20");

   angle = 0;
   span = 0;
   
   takeMouse(add_motionblur_mousefunc,dragcursor);                               //  connect mouse

   zdialog_run(zd,add_motionblur_dialog_event);                                  //  run dialog - parallel
   return;
}


//  dialog event and completion function

int add_motionblur_dialog_event(zdialog *zd, const char *event)
{
   using namespace add_motionblur_names;
   
   void  add_motionblur_mousefunc();
   void  add_motionblur_update();
   
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  from f_open()
   
   if (strmatch(event,"focus")) 
      takeMouse(add_motionblur_mousefunc,dragcursor);                            //  connect mouse

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  [reset]
         zd->zstat = 0;                                                          //  keep dialog active
         edit_reset();
      }

      else if (zd->zstat == 2) {                                                 //  [apply]
         zd->zstat = 0;                                                          //  keep dialog active
         add_motionblur_update();
      }

      else if (zd->zstat == 3) edit_done(0);                                     //  [ OK ] commit edit 
      else edit_cancel(0);                                                       //  [cancel] or [x] discard edit

      Fpaint2();
      return 1;
   }
   
   if (strmatch(event,"span"))                                                   //  span input
      zdialog_fetch(zd,"span",span);

   if (strmatch(event,"angle")) {                                                //  angle input
      zdialog_fetch(zd,"angle",angle);
      if (angle < 0) {
         angle = 180 + angle;                                                    //  -1 --> 179
         zdialog_stuff(zd,"angle",angle);
      }
      if (angle >= 180) {                                                        //  180 --> 0
         angle = angle - 180;
         zdialog_stuff(zd,"angle",angle);
      }
   }

   return 1;
}


//  finish addition of motion blur

void add_motionblur_update()
{
   using namespace add_motionblur_names;

   int      rotate, mww, mhh;
   int      ii, px, py, dist = 0;
   float    *pix1, *pix3, f1, f2;
   int      pcc = E1pxm->nc * sizeof(float);

   if (angle <= 90) rotate = angle;                                              //  avoid upside-down result
   else rotate = angle - 180;

   PXM_free(E3pxm);                                                              //  rotate image so blur is horizontal
   E3pxm = PXM_rotate(E1pxm,rotate);

   E9pxm = PXM_copy(E3pxm);                                                      //  create input image for E3 output
   signal_thread();                                                              //  add motion blur
   wait_thread_idle();
   PXM_free(E9pxm);

   E9pxm = PXM_rotate(E3pxm,-rotate);                                            //  un-rotate
   mww = (E9pxm->ww - Eww) / 2;
   mhh = (E9pxm->hh - Ehh) / 2;                                                  //  cut-off margins from rotate
   PXM_free(E3pxm);
   E3pxm = PXM_copy_area(E9pxm,mww,mhh,Eww,Ehh);
   PXM_free(E9pxm);
   E9pxm = 0;
   
   if (sa_stat == 3)
   {
      for (py = 0; py < Ehh; py++)                                               //  replace output with input image
      for (px = 0; px < Ehh; px++)                                               //    in areas outside the select area
      {
         pix1 = PXMpix(E1pxm,px,py);
         pix3 = PXMpix(E3pxm,px,py);

         ii = py * Eww + px;
         dist = sa_pixmap[ii];
         if (dist) {
            f1 = sa_blendfunc(dist);                                             //  blend changes over sa_blendwidth
            f2 = 1.0 - f1;
            pix3[0] = f1 * pix3[0] + f2 * pix1[0];
            pix3[1] = f1 * pix3[1] + f2 * pix1[1];
            pix3[2] = f1 * pix3[2] + f2 * pix1[2];
         }

         else memcpy(pix3,pix1,pcc);
      }
      
      return;
   }
   
   Eww = E3pxm->ww;                                                              //  cut-off fuzzy margins
   Ehh = E3pxm->hh;
   E9pxm = PXM_copy_area(E3pxm,span,span,Eww-span,Ehh-span);
   PXM_free(E3pxm);
   E3pxm = E9pxm;
   E9pxm = 0;
   
   sa_clear();                                                                   //  select area no longer valid
   return;
}


//  mouse function - capture mouse drag direction and set rotate angle

void add_motionblur_mousefunc()
{
   using namespace add_motionblur_names;

   int         dx, dy;
   float       R;
   
   if (! Mxdrag && ! Mydrag) return;
   
   dx = Mxdrag - Mxdown;                                                         //  drag vector
   dy = Mydrag - Mydown;
   Mxdrag = Mydrag = 0;

   R = sqrtf(dx*dx + dy*dy);                                                     //  get angle of drag
   angle = RAD * acosf(dx/R);
   if (dy > 0) angle = 180 - angle;                                              //  top quadrant only, 0-180 deg.
   if (angle == 180) angle = 0;
   if (CEF) zdialog_stuff(CEF->zd,"angle",angle);

   return;
}


//  thread function - multiple working threads to update image

void * add_motionblur_thread(void *)
{
   using namespace add_motionblur_names;

   void  * add_motionblur_wthread(void *arg);                                    //  worker thread
   
   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);

      progressmon_reset(E3pxm->hh);                                              //  progress monitor
      
      do_wthreads(add_motionblur_wthread,NWT);                                   //  do worker threads
      
      progressmon_reset(0);

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;                                                           //  not saved

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;                                                                     //  not executed, stop warning
}


//  add motion blur worker thread function
//  span = N: each blurred pixel is average of N original pixels

void * add_motionblur_wthread(void *arg)
{
   using namespace add_motionblur_names;

   int      index = *((int *) arg);
   int      px, py, ii;
   float    *pix1, *pix3;
   float    R, G, B;
   int      Eww = E3pxm->ww;                                                     //  rotated and larger image
   int      Ehh = E3pxm->hh;
   
   for (py = index; py < Ehh; py += NWT)                                         //  loop all image pixels
   {
      if (CEF->thread_command == 9) break;                                       //  cancel thread
      busy_add(index,1);                                                         //  count progress

      for (px = span+1; px < Eww; px++)
      {
         pix3 = PXMpix(E3pxm,px,py);                                             //  output pixel (blurred)

         R = G = B = 0;
         
         for (ii = 0; ii <= span; ii++)                                          //  sum input pixels 
         {
            pix1 = PXMpix(E9pxm,px-ii,py);
            R += pix1[0];
            G += pix1[1];
            B += pix1[2];
         }
         
         R = R / (span + 1);                                                     //  average input pixels
         G = G / (span + 1);
         B = B / (span + 1);
          
         pix3[0] = R;                                                            //  output RGB
         pix3[1] = G;
         pix3[2] = B;
      }
   }

   return 0;                                                                     //  exit thread
}


/********************************************************************************/

//  image noise reduction

namespace denoise_names
{
   enum     denoise_method { voodoo, chroma, anneal, flatten, median, tophat, wavelets }
            denoise_method;

   int      noise_histogram[3][256];
   int      denoise_radius, denoise_thresh;
   float    wavelets_thresh;
   float    denoise_darkareas;

   zdialog  *zd_denoise_measure;
   cchar    *mformat = "  mean RGB:  %5.0f %5.0f %5.0f ";
   cchar    *nformat = " mean noise: %5.2f %5.2f %5.2f ";
   
   int      E3ww, E3hh;                                                          //  image dimensions
   int8     *Fpix, *Gpix;

   editfunc    EFdenoise;
   char        editparms[40];
   GtkWidget   *denoise_measure_drawwin;
}


//  menu function

void m_denoise(GtkWidget *, cchar *menu)
{
   using namespace denoise_names;

   void   denoise_characterize();
   int    denoise_dialog_event(zdialog *zd, cchar *event);
   void * denoise_thread(void *);
   
   cchar  *tip = "Apply repeatedly while watching the image.";

   F1_help_topic = "denoise";

   EFdenoise.menuname = "Denoise";
   EFdenoise.menufunc = m_denoise;
   EFdenoise.Farea = 2;                                                          //  select area usable
   EFdenoise.threadfunc = denoise_thread;                                        //  thread function
   EFdenoise.Frestart = 1;                                                       //  allow restart
   EFdenoise.Fpaint = 1;                                                         //  use with paint edits OK
   EFdenoise.Fscript = 1;                                                        //  scripting supported 
   if (! edit_setup(EFdenoise)) return;                                          //  setup edit

   E3ww = E3pxm->ww;                                                             //  image dimensions
   E3hh = E3pxm->hh;

/***
          _____________________________________________
         |           Noise Reduction                   |
         |  Apply repeatedly while watching the image. |
         | - - - - - - - - - - - - - - - - - - - - - - |          sep0
         |  Method                                     |          hbm1
         |    (o) Voodoo   (o) Chroma   (o) Anneal     |          hbm2  vbm1  vbm2  vbm3
         |    (o) Flatten  (o) Median   (o) Top Hat    |          
         |  Radius [___]    Threshold [___]            |          hbrt
         | - - - - - - - - - - - - - - - - - - - - - - |          sep1
         |    (o) Wavelets Threshold [___]             |          hbwav
         | - - - - - - - - - - - - - - - - - - - - - - |          sep2
         | dark areas =========[]=========== all areas |          hbar
         |                                             |
         |   [Measure] [Apply] [Reset] [ OK ] [Cancel] |
         |_____________________________________________|

***/

   zdialog *zd = zdialog_new("Noise Reduction",Mwin,Bmeasure,Bapply,Breset,BOK,Bcancel,null);
   EFdenoise.zd = zd;
   
   zdialog_add_widget(zd,"label","labtip","dialog",tip,"space=3");
   
   zdialog_add_widget(zd,"hsep","sep0","dialog",0,"space=3");

   zdialog_add_widget(zd,"hbox","hbm1","dialog",0);
   zdialog_add_widget(zd,"label","labm","hbm1","Method","space=3");

   zdialog_add_widget(zd,"hbox","hbm2","dialog",0);
   zdialog_add_widget(zd,"label","space","hbm2",0,"space=8"); 
   zdialog_add_widget(zd,"vbox","vbm1","hbm2",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbm2","hbm2",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbm3","hbm2",0,"space=3");
   zdialog_add_widget(zd,"check","voodoo","vbm1","Voodoo");
   zdialog_add_widget(zd,"check","chroma","vbm2","Chroma");
   zdialog_add_widget(zd,"check","anneal","vbm3","Anneal");
   zdialog_add_widget(zd,"check","flatten","vbm1","Flatten");
   zdialog_add_widget(zd,"check","median","vbm2","Median");
   zdialog_add_widget(zd,"check","tophat","vbm3","Top Hat");
   
   zdialog_add_widget(zd,"hbox","hbrt","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labrad","hbrt",Bradius,"space=3");
   zdialog_add_widget(zd,"zspin","denoise_radius","hbrt","1|9|1|3","size=4");
   zdialog_add_widget(zd,"label","space","hbrt",0,"space=8");
   zdialog_add_widget(zd,"label","labthresh","hbrt",Bthresh,"space=3");
   zdialog_add_widget(zd,"zspin","denoise_thresh","hbrt","1|50|1|10","size=4");

   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=3");

   zdialog_add_widget(zd,"hbox","hbwav","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","space","hbwav",0,"space=8"); 
   zdialog_add_widget(zd,"check","wavelets","hbwav","Wavelets");
   zdialog_add_widget(zd,"label","labthresh","hbwav",Bthresh);
   zdialog_add_widget(zd,"zspin","wavelets_thresh","hbwav","0.1|8.0|0.1|1","size=4|space=12");

   zdialog_add_widget(zd,"hsep","sep2","dialog",0,"space=3");

   zdialog_add_widget(zd,"hbox","hbar","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labdark","hbar","dark areas","space=8");
   zdialog_add_widget(zd,"hscale","darkareas","hbar","0|256|1|256","expand");
   zdialog_add_widget(zd,"label","laball","hbar","all areas","space=8");

   denoise_characterize();                                                       //  characterize image noise
   zdialog_stuff(zd,"denoise_thresh",denoise_thresh);                            //    = default threshold

   wavelets_thresh = 1.0;                                                        //  default wavelets threshold
   
   zdialog_stuff(zd,"voodoo",1);                                                 //  initial selected method

   zdialog_run(zd,denoise_dialog_event,"save");                                  //  run dialog - parallel
   return;
}


//  characterize noise levels

void denoise_characterize()
{
   using namespace denoise_names;

   void * denoise_characterize_wthread(void *arg);

   int      ii, rgb, Npix, Tpix;
   double   val, sum, sum2, mean, sigma, varnc, thresh, limit;

   for (rgb = 0; rgb < 3; rgb++)                                                 //  clear histogram
   for (ii = 0; ii < 256; ii++)
      noise_histogram[rgb][ii] = 0;

   do_wthreads(denoise_characterize_wthread,NWT);                                //  make histogram of pixel-pixel diffs

   thresh = 0;
   limit = 100;

   for (rgb = 0; rgb < 3; rgb++)                                                 //  get mean and sigma
   {
      sum = sum2 = Tpix = 0;
      
      for (ii = 0; ii < limit; ii++) {                                           //  omit diffs > limit
         Npix = noise_histogram[rgb][ii];
         Tpix += Npix;
         val = ii * Npix;
         sum += val;
         sum2 += val * val;
      }
      
      mean = sum / limit;
      varnc = (sum2 - 2.0 * mean * mean) / limit + mean * mean;
      sigma = sqrtf(varnc);
      
      mean = mean / Tpix * limit;                                                //  mean and sigma
      sigma = sigma / Tpix * limit;
      if (sigma > thresh) thresh = sigma;
   }
   
   sigma = thresh;                                                               //  greatest RGB sigma
   
   thresh = 0;
   limit = 3 * sigma;
   
   for (rgb = 0; rgb < 3; rgb++)                                                 //  make another histogram
   {                                                                             //    ignoring values > 3 * sigma
      sum = sum2 = Tpix = 0;
      
      for (ii = 0; ii < limit; ii++) {
         Npix = noise_histogram[rgb][ii];
         Tpix += Npix;
         val = ii * Npix;
         sum += val;
         sum2 += val * val;
      }
      
      mean = sum / limit;
      varnc = (sum2 - 2.0 * mean * mean) / limit + mean * mean;
      sigma = sqrtf(varnc);
      
      mean = mean / Tpix * limit;
      sigma = sigma / Tpix * limit;
      if (sigma > thresh) thresh = sigma;
   }
   
   denoise_thresh = thresh + 0.5;                                                //  greatest RGB sigma
   return;
}


void * denoise_characterize_wthread(void *arg)
{
   using namespace denoise_names;

   int      index = *((int *) arg);
   int      ii, px, py, diff, rgb;
   float    *pix1, *pix2;
   
   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww-1; px++)                                               //  omit last column
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         if (sa_pixmap[ii] == 0) continue;                                       //  pixel outside area, ignore
      }

      pix1 = PXMpix(E1pxm,px,py);
      pix2 = PXMpix(E1pxm,px+1,py);
      
      for (rgb = 0; rgb < 3; rgb++)
      {
         diff = pix1[rgb] - pix2[rgb];                                           //  pixel-pixel RGB difference
         diff = abs(diff);
         noise_histogram[rgb][diff] += 1;
      }
   }
   
   return 0;
   return 0;
}


//  dialog event and completion callback function

int denoise_dialog_event(zdialog * zd, cchar *event)                             //  reworked for script files
{
   using namespace denoise_names;

   void   denoise_measure();

   int    ii;
   
   wait_thread_idle();
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"apply")) zd->zstat = 2;                                   //  from script file
   if (strmatch(event,"done")) zd->zstat = 4;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 5;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  [measure] 
         zd->zstat = 0;                                                          //  keep dialog active
         denoise_measure();
         return 1;
      }
      
      if (zd->zstat == 2) {                                                      //  [apply]
         zd->zstat = 0;                                                          //  keep dialog active
         signal_thread();
         return 1;
      }

      if (zd->zstat == 3) {                                                      //  [reset]
         edit_undo();                                                            //  undo edits
         zd->zstat = 0;                                                          //  keep dialog active
         return 1;
      }

      if (zd->zstat == 4) {                                                      //  [ OK ]  commit edit
         EFdenoise.editparms = editparms;                                        //  save exif edit hist                   21.0
         edit_done(0);
      }

      else edit_cancel(0);                                                       //  [cancel] or [x]  discard edit

      if (zd_denoise_measure) {                                                  //  kill measure dialog 
         freeMouse();
         zdialog_free(zd_denoise_measure);
         zd_denoise_measure = 0;
      }

      return 1;
   }
   
   if (strmatch(event,"blendwidth")) signal_thread();

   zdialog_fetch(zd,"denoise_radius",denoise_radius);
   zdialog_fetch(zd,"denoise_thresh",denoise_thresh);
   zdialog_fetch(zd,"wavelets_thresh",wavelets_thresh);
   zdialog_fetch(zd,"darkareas",denoise_darkareas);
   
   if (zstrstr("voodoo chroma anneal flatten median tophat wavelets",event)) {   //  capture choice
      zdialog_stuff(zd,"voodoo",0);
      zdialog_stuff(zd,"chroma",0);
      zdialog_stuff(zd,"anneal",0);
      zdialog_stuff(zd,"flatten",0);
      zdialog_stuff(zd,"median",0);
      zdialog_stuff(zd,"tophat",0);
      zdialog_stuff(zd,"wavelets",0);
      zdialog_stuff(zd,event,1);
   }
   
   zdialog_fetch(zd,"voodoo",ii);
   if (ii) denoise_method = voodoo;

   zdialog_fetch(zd,"chroma",ii);
   if (ii) denoise_method = chroma;

   zdialog_fetch(zd,"anneal",ii);
   if (ii) denoise_method = anneal;

   zdialog_fetch(zd,"flatten",ii);
   if (ii) denoise_method = flatten;

   zdialog_fetch(zd,"median",ii);
   if (ii) denoise_method = median;
   
   zdialog_fetch(zd,"tophat",ii);
   if (ii) denoise_method = tophat;
   
   zdialog_fetch(zd,"wavelets",ii);
   if (ii) denoise_method = wavelets;

   return 1;
}


//  image noise reduction thread

void * denoise_thread(void *)
{
   using namespace denoise_names;

   void * denoise_chroma_wthread1(void *arg);
   void * denoise_chroma_wthread2(void *arg);
   void * denoise_anneal_wthread(void *arg);
   void * denoise_flatten_wthread(void *arg);
   void * denoise_median_wthread(void *arg);
   void * denoise_tophat_wthread(void *arg);
   void * denoise_wavelets_wthread(void *arg);

   int      ii, px, py, dist = 0;
   float    *pix1, *pix9;
   int      save_thresh;
   int      nc = E3pxm->nc, pcc = nc * sizeof(float);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

      E9pxm = PXM_copy(E3pxm);                                                   //  E3 is source, E9 is modified
      
      if (denoise_method == wavelets) {                                          //  wavelets method
         do_wthreads(denoise_wavelets_wthread,3);                                //  worker threads, 1 per RGB color
         snprintf(editparms,40,"wavelets %.1f",wavelets_thresh);                 //  record edit hist                      21.0
      }

      else                                                                       //  other method
      {
         if (sa_stat == 3) progressmon_reset(sa_Npixel);                         //  initz. progress counter
         else  progressmon_reset(E3ww * E3hh);

         if (denoise_method == voodoo)
         {
            if (sa_stat == 3) progressmon_reset(sa_Npixel * 4);                  //  progress counter
            else  progressmon_reset(E3ww * E3hh * 4);

            save_thresh = denoise_thresh;
            denoise_thresh = 255;                                                //  salt and pepper elimination
            denoise_radius = 1;
            do_wthreads(denoise_median_wthread,NWT);
 
            denoise_thresh = save_thresh;

            for (ii = 0; ii < 3; ii++)
            {
               PXM_free(E3pxm);                                                  //  accumulate changes
               E3pxm = E9pxm;
               E9pxm = PXM_copy(E3pxm);
               denoise_radius = 2 + 2 * ii;                                      //  2  4  6
               if (ii > 0) denoise_thresh *= 0.8;                                //  1.0  0.8  0.64
               do_wthreads(denoise_anneal_wthread,NWT);                          //  general noise reduction
            }

            denoise_thresh = save_thresh;
            
            snprintf(editparms,40,"voodoo");                                     //  exif edit hist                        21.0
         }
     
         if (denoise_method == chroma) {
            E8pxm = PXM_make(E3ww,E3hh,3);                                       //  allocate HSL image
            do_wthreads(denoise_chroma_wthread1,NWT);
            do_wthreads(denoise_chroma_wthread2,NWT);
            PXM_free(E8pxm);
            snprintf(editparms,40,"chroma %d %d",                                //  exif edit hist                        21.0
                                       denoise_radius,denoise_thresh);
         }

         if (denoise_method == anneal) {
            do_wthreads(denoise_anneal_wthread,NWT);
            snprintf(editparms,40,"anneal %d %d",                                //  exif edit hist                        21.0
                                       denoise_radius,denoise_thresh);
         }

         if (denoise_method == flatten) {
            do_wthreads(denoise_flatten_wthread,NWT);
            snprintf(editparms,40,"flatten %d %d",                               //  exif edit hist                        21.0
                                       denoise_radius,denoise_thresh);
         }

         if (denoise_method == median) {
            do_wthreads(denoise_median_wthread,NWT);
            snprintf(editparms,40,"median %d %d",                                //  exif edit hist                        21.0
                                       denoise_radius,denoise_thresh);
         }

         if (denoise_method == tophat) {
            do_wthreads(denoise_tophat_wthread,NWT);
            snprintf(editparms,40,"top hat %d %d",                               //  exif edit hist                        21.0
                                       denoise_radius,denoise_thresh);
         }

         progressmon_reset(0); 
      }

      if (denoise_darkareas < 256)                                               //  if brightness threshold set,
      {                                                                          //    revert brighter areas
         for (py = 0; py < E3hh; py++)
         for (px = 0; px < E3ww; px++)
         {
            if (sa_stat == 3) {                                                  //  select area active
               ii = py * E3ww + px;
               dist = sa_pixmap[ii];                                             //  distance from edge
               if (! dist) continue;                                             //  outside pixel
            }

            pix1 = PXMpix(E1pxm,px,py);                                          //  source pixel
            pix9 = PXMpix(E9pxm,px,py);                                          //  target pixel

            if (pix1[0] + pix1[1] + pix1[2] > 3 * denoise_darkareas)             //  revert brighter pixels
               memcpy(pix9,pix1,pcc);
         }
      }

      PXM_free(E3pxm);                                                           //  image9 >> image3
      E3pxm = E9pxm;
      E9pxm = 0;

      CEF->Fmods++;
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


//  Chroma:
//  Exchange color data between neighbor pixels that match closely enough.

void RGB_YCbCr(float R, float G, float B, float &Y, float &Cb, float &Cr)
{
   Y  =  0.299 * R + 0.587 * G + 0.114 * B;
   Cb =  128 - 0.168736 * R - 0.331264 * G + 0.5 * B;
   Cr =  128 + 0.5 * R - 0.418688 * G - 0.081312 * B;
   return;
}

void YCbCr_RGB(float Y, float Cb, float Cr, float &R, float &G, float &B)
{
   R = Y + 1.402 * (Cr - 128);
   G = Y - 0.344136 * (Cb - 128) - 0.714136 * (Cr - 128);
   B = Y + 1.772 * (Cb - 128);
   return;
}


void * denoise_chroma_wthread1(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         px, py;
   float       *pix3, *pix8;

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);                                                //  E3 RGB image >> E8 YCbCr image
      pix8 = PXMpix(E8pxm,px,py);
      RGB_YCbCr(pix3[0],pix3[1],pix3[2],pix8[0],pix8[1],pix8[2]);
   }

   return 0;
}


void * denoise_chroma_wthread2(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         rad, thresh;
   int         ii, dist = 0;
   int         px, py, dx, dy, rgb;
   float       *pix1, *pix9;
   float       *pix8, *pix8N, pixM[3];
   float       Frad, f1, f2;
   float       match, reqmatch;
   
   rad = denoise_radius;
   thresh = denoise_thresh;
   reqmatch = 1.0 - 0.01 * thresh;                                               //  20 >> 0.8
   Frad = rad + rad + 1;
   Frad = 1.0 / (Frad * Frad);

   for (py = index+rad; py < E3hh-rad; py += NWT)                                //  loop all image pixels
   for (px = rad; px < E3ww-rad; px++)                                           //  skip outermost rows and cols
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      busy_add(index,1);                                                         //  track progress

      pix8 = PXMpix(E8pxm,px,py);                                                //  source YCbCr pixel

      memset(pixM,0,3*sizeof(float));                                            //  clear totals

      for (dy = -rad; dy <= +rad; dy++)                                          //  loop neighbor pixels
      for (dx = -rad; dx <= +rad; dx++)
      {
         pix8N = pix8 + (dy * E3ww + dx) * 3;                                    //  neighbor pixel
         
         match = PIXMATCH(pix8,pix8N);                                           //  match level, 0..1 = perfect match
         if (match > reqmatch) f1 = 0.5;                                         //  color exchange factor
         else f1 = 0;
         f2 = 1.0 - f1;

         for (rgb = 0; rgb < 3; rgb++)
            pixM[rgb] += f2 * pix8[rgb] + f1 * pix8N[rgb];                       //  accumulate exchange colors
      }

      for (rgb = 0; rgb < 3; rgb++)                                              //  normalize to 0-255
         pixM[rgb] = pixM[rgb] * Frad;      
      
      match = PIXMATCH(pix8,pixM);                                               //  final old:new comparison
      if (match < reqmatch) continue;                                            //  reject larger changes
      
      pix9 = PXMpix(E9pxm,px,py);                                                //  target RGB pixel
      YCbCr_RGB(pixM[0],pixM[1],pixM[2],pix9[0],pix9[1],pix9[2]);                //  YCbCr pix8 >> RGB pix9

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm,px,py);                                             //  source image pixel
         pix9[0] = f1 * pix9[0] + f2 * pix1[0];
         pix9[1] = f1 * pix9[1] + f2 * pix1[1];
         pix9[2] = f1 * pix9[2] + f2 * pix1[2];
      }
   }

   return 0;
}


//  Anneal:
//  Exchange RGB data between neighbor pixels that match closely enough.

void * denoise_anneal_wthread(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         rad, thresh;
   int         ii, rgb, dist = 0;
   int         px, py, dx, dy;
   int         nc = E3pxm->nc;
   float       *pix1, *pix3, *pix9, *pix3N;
   float       f1, f2, Frad, pixM[3];
   float       match, reqmatch;

   rad = denoise_radius;
   thresh = denoise_thresh;
   reqmatch = 1.0 - 0.01 * thresh;                                               //  20 >> 0.8
   Frad = rad + rad + 1;
   Frad = 1.0 / (Frad * Frad);
   
   for (py = index+rad; py < E3hh-rad; py += NWT)                                //  loop all image pixels
   for (px = rad; px < E3ww-rad; px++)                                           //  skip outermost rows and cols
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      for (rgb = 0; rgb < 3; rgb++)
         pixM[rgb] = 0;

      pix1 = PXMpix(E1pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);                                                //  source image pixel
      pix9 = PXMpix(E9pxm,px,py);                                                //  target image pixel

      for (dy = -rad; dy <= +rad; dy++)                                          //  loop neighbor pixels
      for (dx = -rad; dx <= +rad; dx++)
      {
         pix3N = pix3 + (dy * E3ww + dx) * nc;                                   //  neighbor pixel
         match = PIXMATCH(pix3,pix3N);                                           //  match level, 0..1 = perfect match
         if (match > reqmatch) f1 = 0.5;                                         //  RGB exchange factors
         else f1 = 0;
         f2 = 1.0 - f1;

         for (rgb = 0; rgb < 3; rgb++)                                           //  loop all RGB colors
            pixM[rgb] += f2 * pix3[rgb] + f1 * pix3N[rgb];                       //  accumulate exchange colors
      }
      
      for (rgb = 0; rgb < 3; rgb++)                                              //  normalize to 0-255
         pixM[rgb] = Frad * pixM[rgb];
      
      match = PIXMATCH(pix1,pixM);                                               //  final old:new comparison
      if (match < reqmatch) continue;                                            //  reject larger changes
      
      for (rgb = 0; rgb < 3; rgb++)
         pix9[rgb] = pixM[rgb];

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix9[0] = f1 * pix9[0] + f2 * pix1[0];
         pix9[1] = f1 * pix9[1] + f2 * pix1[1];
         pix9[2] = f1 * pix9[2] + f2 * pix1[2];
      }

      busy_add(index,1);                                                         //  track progress
   }

   return 0;
}


//  Flatten: 
//  Flatten outlyer pixels within neighborhood group.
//  An outlier pixel has an RGB value outside mean +- sigma.
//  Process groups increasing from radius = 1 to denoise_radius.

void * denoise_flatten_wthread(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         ii, rgb, dist = 0;
   int         px, py, dx, dy, rad1, rad2, thresh;
   int         nc = E3pxm->nc;
   float       *pix1, *pix3, *pix9, *pixN;
   float       nn, val, sum, sum2, mean, variance, sigma, thresh2;
   float       f1, f2;

   rad1 = denoise_radius;
   thresh = denoise_thresh;

   for (py = index+rad1; py < E3hh-rad1; py += NWT)                              //  loop all image pixels
   for (px = rad1; px < E3ww-rad1; px++)                                         //  skip outermost rows and cols
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix3 = PXMpix(E3pxm,px,py);                                                //  source image pixel
      pix9 = PXMpix(E9pxm,px,py);                                                //  target image pixel

      for (rad2 = 1; rad2 <= rad1; rad2++)                                       //  radius from 1 to max
      {
         nn = (rad2 * 2 + 1);                                                    //  pixel group NxN size
         nn = nn * nn;                                                           //  pixels in group

         for (rgb = 0; rgb < 3; rgb++)                                           //  loop RGB color
         {
            sum = sum2 = 0;

            for (dy = -rad2; dy <= rad2; dy++)                                   //  loop surrounding pixels
            for (dx = -rad2; dx <= rad2; dx++)
            {
               pixN = pix3 + (dy * E3ww + dx) * nc;
               val = pixN[rgb];
               sum += val;
               sum2 += val * val;
            }

            mean = sum / nn;                                                     //  compute mean and sigma
            variance = (sum2 - 2.0 * mean * sum) / nn + mean * mean;
            sigma = sqrtf(variance);
            thresh2 = 0.5 * (thresh + sigma);

            if (pix3[rgb] > mean + sigma && pix3[rgb] - mean < thresh2)          //  if | pixel - mean | > sigma
               pix9[rgb] = mean;                                                 //    flatten pixel
            else if (pix3[rgb] < mean - sigma && mean - pix3[rgb] < thresh2)
               pix9[rgb] = mean;
         }
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm,px,py);                                             //  source image pixel
         pix9[0] = f1 * pix9[0] + f2 * pix1[0];
         pix9[1] = f1 * pix9[1] + f2 * pix1[1];
         pix9[2] = f1 * pix9[2] + f2 * pix1[2];
      }

      busy_add(index,1);                                                         //  track progress
   }

   return 0;
}


//  Median:
//  Use median RGB brightness for pixels within radius

void * denoise_median_wthread(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         ii, rgb, dist = 0;
   int         px, py, dx, dy, rad, Frad, thresh;
   int         nc = E3pxm->nc;
   float       *pix1, *pix3, *pix9, *pixN;
   float       f1, f2, median;
   int16       rgbdist[256];
   int         rgbsum;

   rad = denoise_radius;
   thresh = denoise_thresh;

   Frad = 2 * rad + 1;
   Frad = Frad * Frad;

   for (py = index+rad; py < E3hh-rad; py += NWT)                                //  loop all image pixels
   for (px = rad; px < E3ww-rad; px++)                                           //  skip outermost rows and cols
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix3 = PXMpix(E3pxm,px,py);                                                //  source image pixel
      pix9 = PXMpix(E9pxm,px,py);                                                //  target image pixel

      for (rgb = 0; rgb < 3; rgb++)                                              //  loop all RGB colors
      {
         memset(rgbdist,0,256*sizeof(int16));                                    //  clear rgb distribution
         rgbsum = 0;

         for (dy = -rad; dy <= rad; dy++)                                        //  loop surrounding pixels
         for (dx = -rad; dx <= rad; dx++)                                        //  get RGB values
         {
            pixN = pix3 + (dy * E3ww + dx) * nc;                                 //  make distribution of RGB values
            ii = pixN[rgb];
            rgbdist[ii]++;
         }
         
         for (ii = 0; ii < 256; ii++)                                            //  sum distribution from 0 to 255
         {
            rgbsum += rgbdist[ii];                                               //  break when half of RGB values
            if (rgbsum > Frad / 2) break;                                        //    have been counted
         }
         
         median = ii;                                                            //  >> median RGB value

         if (pix3[rgb] > median && pix3[rgb] - median < thresh)                  //  if | rgb - median | < threshold
            pix9[rgb] = median;                                                  //    moderate rgb

         else if (pix3[rgb] < median && median - pix3[rgb] < thresh)
            pix9[rgb] = median;
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm,px,py);                                             //  source image pixel
         pix9[0] = f1 * pix9[0] + f2 * pix1[0];
         pix9[1] = f1 * pix9[1] + f2 * pix1[1];
         pix9[2] = f1 * pix9[2] + f2 * pix1[2];
      }

      busy_add(index,1);                                                         //  track progress
   }

   return 0;
}


//  Top Hat:
//  Execute with increasing radius from 1 to limit.
//  Detect outlier by comparing with pixels along outer radius only.

void * denoise_tophat_wthread(void *arg)
{
   using namespace denoise_names;

   int         index = *((int *) arg);
   int         ii, dist = 0;
   int         px, py, dx, dy, rad1, rad2;
   int         nc = E3pxm->nc;
   float       *pix1, *pix3, *pix9, *pixN;
   float       minR, minG, minB, maxR, maxG, maxB;
   float       f1, f2;

   rad1 = denoise_radius;

   for (py = index+rad1; py < E3hh-rad1; py += NWT)                              //  loop all image pixels
   for (px = rad1; px < E3ww-rad1; px++)                                         //  skip outermost rows and cols
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside pixel
      }

      pix3 = PXMpix(E3pxm,px,py);                                                //  source image pixel
      pix9 = PXMpix(E9pxm,px,py);                                                //  target image pixel

      for (rad2 = 1; rad2 <= rad1; rad2++)                                       //  loop rad2 from 1 to max.
      for (int loops = 0; loops < 2; loops++)
      {
         minR = minG = minB = 255;
         maxR = maxG = maxB = 0;

         for (dy = -rad2; dy <= rad2; dy++)                                      //  loop all pixels within rad2
         for (dx = -rad2; dx <= rad2; dx++)
         {
            if (dx > -rad2 && dx < rad2) continue;                               //  skip inner pixels
            if (dy > -rad2 && dy < rad2) continue;

            pixN = pix3 + (dy * E3ww + dx) * nc;
            if (pixN[0] < minR) minR = pixN[0];                                  //  find min and max per color
            if (pixN[0] > maxR) maxR = pixN[0];                                  //    among outermost pixels
            if (pixN[1] < minG) minG = pixN[1];
            if (pixN[1] > maxG) maxG = pixN[1];
            if (pixN[2] < minB) minB = pixN[2];
            if (pixN[2] > maxB) maxB = pixN[2];
         }

         if (pix3[0] < minR && pix9[0] < 254) pix9[0] += 2;                      //  if central pixel is outlier,
         if (pix3[0] > maxR && pix9[0] > 1)   pix9[0] -= 2;                      //    moderate its values
         if (pix3[1] < minG && pix9[1] < 254) pix9[1] += 2;
         if (pix3[1] > maxG && pix9[1] > 1)   pix9[1] -= 2;
         if (pix3[2] < minB && pix9[2] < 254) pix9[2] += 2;
         if (pix3[2] > maxB && pix9[2] > 1)   pix9[2] -= 2;
      }

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //  blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         pix1 = PXMpix(E1pxm,px,py);                                             //  source image pixel
         pix9[0] = f1 * pix9[0] + f2 * pix1[0];
         pix9[1] = f1 * pix9[1] + f2 * pix1[1];
         pix9[2] = f1 * pix9[2] + f2 * pix1[2];
      }

      busy_add(index,1);                                                         //  track progress
   }

   return 0;
}


//  Wavelets:
//  worker thread for wavelets method
//  do wavelets denoise for one color in each of 3 threads

void * denoise_wavelets_wthread(void *arg)
{
   using namespace denoise_names;

   void denoise_wavelets(float *fimg[3], uint ww2, uint hh2, float);

   int      rgb = *((int *) arg);                                                //  rgb color 0/1/2
   int      ii, jj;
   float    *fimg[3];
   float    f256 = 1.0 / 256.0;
   int      nc = E3pxm->nc;

   if (sa_stat == 3) goto denoise_area;                                          //  select area is active

   fimg[0] = (float *) zmalloc(E3ww * E3hh * sizeof(float));
   fimg[1] = (float *) zmalloc(E3ww * E3hh * sizeof(float));
   fimg[2] = (float *) zmalloc(E3ww * E3hh * sizeof(float));

   for (ii = 0; ii < E3ww * E3hh; ii++)                                          //  extract one noisy color from E3
      fimg[0][ii] = E3pxm->pixels[nc*ii+rgb] * f256;

   denoise_wavelets(fimg,E3ww,E3hh,wavelets_thresh);

   for (ii = 0; ii < E3ww * E3hh; ii++)                                          //  save one denoised color to E9
      E9pxm->pixels[nc*ii+rgb] = 256.0 * fimg[0][ii];

   zfree(fimg[0]);
   zfree(fimg[1]);
   zfree(fimg[2]);

   return 0;

denoise_area:

   int      px, py, pxl, pxh, pyl, pyh, ww2, hh2, dist;
   float    f1, f2;

   pxl = sa_minx - 16;
   if (pxl < 0) pxl = 0;
   pxh = sa_maxx + 16;
   if (pxh > E3ww) pxh = E3ww;

   pyl = sa_miny - 16;
   if (pyl < 0) pyl = 0;
   pyh = sa_maxy + 16;
   if (pyh > E3hh) pyh = E3hh;

   ww2 = pxh - pxl;
   hh2 = pyh - pyl;

   fimg[0] = (float *) zmalloc(ww2 * hh2 * sizeof(float));
   fimg[1] = (float *) zmalloc(ww2 * hh2 * sizeof(float));
   fimg[2] = (float *) zmalloc(ww2 * hh2 * sizeof(float));

   for (py = 0; py < hh2; py++)
   for (px = 0; px < ww2; px++)
   {
      ii = py * ww2 + px;
      jj = (py + pyl) * E3ww + (px + pxl);
      fimg[0][ii] = E3pxm->pixels[nc*jj+rgb] * f256;
   }

   denoise_wavelets(fimg,ww2,hh2,wavelets_thresh);

   for (py = 0; py < hh2; py++)
   for (px = 0; px < ww2; px++)
   {
      ii = py * ww2 + px;
      jj = (py + pyl) * E3ww + (px + pxl);

      dist = sa_pixmap[jj];
      if (! dist) continue;

      if (dist < sa_blendwidth) {
         f1 = sa_blendfunc(dist);
         f2 = 1.0 - f1;
         E9pxm->pixels[nc*jj+rgb] = f1 * 256.0 * fimg[0][ii] + f2 * E3pxm->pixels[nc*jj+rgb];
      }
      else E9pxm->pixels[nc*jj+rgb] = 256.0 * fimg[0][ii];
   }

   zfree(fimg[0]);
   zfree(fimg[1]);
   zfree(fimg[2]);

   return 0;
}


//  wavelets denoise algorithm
//  Adapted from Gimp wavelets plugin (and ultimately DCraw by Dave Coffin).
//  fimg[0][rows][cols] = one color of image to denoise
//  fimg[1] and [2] = working space
//  thresh (0-10) is the adjustable parameter

void denoise_wavelets(float *fimg[3], uint ww2, uint hh2, float thresh)
{
   void denoise_wavelets_avgpix(float *temp, float *fimg, int st, int size, int sc);

   float    *temp, thold, stdev[5];
   uint     ii, lev, lpass, hpass, size, col, row;
   uint     samples[5];

   size = ww2 * hh2;
   temp = (float *) zmalloc ((ww2 + hh2) * sizeof(float));
   hpass = 0;

   for (lev = 0; lev < 5; lev++)
   {
      lpass = ((lev & 1) + 1);                                                   //  1, 2, 1, 2, 1

      for (row = 0; row < hh2; row++)                                            //  average row pixels
      {
         denoise_wavelets_avgpix(temp, fimg[hpass] + row * ww2, 1, ww2, 1 << lev);

         for (col = 0; col < ww2; col++)
            fimg[lpass][row * ww2 + col] = temp[col];
      }

      for (col = 0; col < ww2; col++)                                            //  average column pixels
      {
         denoise_wavelets_avgpix(temp, fimg[lpass] + col, ww2, hh2, 1 << lev);

         for (row = 0; row < hh2; row++)
            fimg[lpass][row * ww2 + col] = temp[row];
      }

      thold = 5.0 / (1 << 6) * exp (-2.6 * sqrt (lev + 1)) * 0.8002 / exp (-2.6);

      stdev[0] = stdev[1] = stdev[2] = stdev[3] = stdev[4] = 0.0;
      samples[0] = samples[1] = samples[2] = samples[3] = samples[4] = 0;

      for (ii = 0; ii < size; ii++)
      {
         fimg[hpass][ii] -= fimg[lpass][ii];

         if (fimg[hpass][ii] < thold && fimg[hpass][ii] > -thold)
         {
            if (fimg[lpass][ii] > 0.8) {
               stdev[4] += fimg[hpass][ii] * fimg[hpass][ii];
               samples[4]++;
            }
            else if (fimg[lpass][ii] > 0.6) {
               stdev[3] += fimg[hpass][ii] * fimg[hpass][ii];
               samples[3]++;
            }
            else if (fimg[lpass][ii] > 0.4) {
               stdev[2] += fimg[hpass][ii] * fimg[hpass][ii];
               samples[2]++;
            }
            else if (fimg[lpass][ii] > 0.2) {
               stdev[1] += fimg[hpass][ii] * fimg[hpass][ii];
               samples[1]++;
            }
            else {
               stdev[0] += fimg[hpass][ii] * fimg[hpass][ii];
               samples[0]++;
            }
         }
      }

      stdev[0] = sqrt (stdev[0] / (samples[0] + 1));
      stdev[1] = sqrt (stdev[1] / (samples[1] + 1));
      stdev[2] = sqrt (stdev[2] / (samples[2] + 1));
      stdev[3] = sqrt (stdev[3] / (samples[3] + 1));
      stdev[4] = sqrt (stdev[4] / (samples[4] + 1));

      for (ii = 0; ii < size; ii++)                                              //  do thresholding
      {
         if (fimg[lpass][ii] > 0.8)
            thold = thresh * stdev[4];
         else if (fimg[lpass][ii] > 0.6)
            thold = thresh * stdev[3];
         else if (fimg[lpass][ii] > 0.4)
            thold = thresh * stdev[2];
         else if (fimg[lpass][ii] > 0.2)
            thold = thresh * stdev[1];
         else
            thold = thresh * stdev[0];

         if (fimg[hpass][ii] < -thold)
            fimg[hpass][ii] += thold;
         else if (fimg[hpass][ii] > thold)
            fimg[hpass][ii] -= thold;
         else
            fimg[hpass][ii] = 0;

         if (hpass) fimg[0][ii] += fimg[hpass][ii];
      }

      hpass = lpass;
   }

   for (ii = 0; ii < size; ii++)
      fimg[0][ii] = fimg[0][ii] + fimg[lpass][ii];

   zfree(temp);
   return;
}


//  average pixels in one column or row
//  st = row stride (row length) or column stride (1)
//  sc = 1, 2, 4, 8, 16 = pixels +/- from target pixel to average

void denoise_wavelets_avgpix(float *temp, float *fimg, int st, int size, int sc)
{
  int ii;

  for (ii = 0; ii < sc; ii++)
    temp[ii] = 0.25*(2*fimg[st*ii] + fimg[st*(sc-ii)] + fimg[st*(ii+sc)]);

  for (NOP; ii < size - sc; ii++)
    temp[ii] = 0.25*(2*fimg[st*ii] + fimg[st*(ii-sc)] + fimg[st*(ii+sc)]);

  for (NOP; ii < size; ii++)
    temp[ii] = 0.25*(2*fimg[st*ii] + fimg[st*(ii-sc)] + fimg[st*(2*size-2-(ii+sc))]);

  return;
}


//  dialog to measure noise at mouse position

void denoise_measure()
{
   using namespace denoise_names;

   int denoise_measure_dialog_event(zdialog *zd, cchar *event);

   GtkWidget   *frdraw, *drawwin;
   char        text[100];
   cchar       *title = "Measure Noise";
   cchar       *mousemess = "Move mouse in a monotone image area.";

/***
          _______________________________________
         |           Measure Noise               |
         |                                       |
         |  Move mouse in a monotone image area. |
         | _____________________________________ |
         ||                                     ||
         ||                                     ||
         ||.....................................||
         ||                                     ||
         ||                                     ||
         ||_____________________________________||       drawing area
         ||                                     ||
         ||                                     ||
         ||.....................................||
         ||                                     ||
         ||_____________________________________||
         | center                           edge |
         |                                       |
         |   mean RGB:   100   150   200         |
         |  mean noise:  1.51  1.23  0.76        |
         |                                       |
         |                              [cancel] |
         |_______________________________________|

***/

   if (zd_denoise_measure) return;

   zdialog *zd = zdialog_new(title,Mwin,Bcancel,null);                           //  measure noise dialog
   zd_denoise_measure = zd;

   zdialog_add_widget(zd,"label","clab","dialog",mousemess,"space=5");

   zdialog_add_widget(zd,"frame","frdraw","dialog",0,"expand");                  //  frame for drawing areas
   frdraw = zdialog_gtkwidget(zd,"frdraw");
   drawwin = gtk_drawing_area_new();                                             //  drawing area
   gtk_container_add(GTK_CONTAINER(frdraw),drawwin);
   denoise_measure_drawwin = drawwin;
   
   zdialog_add_widget(zd,"hbox","hbce","dialog");
   zdialog_add_widget(zd,"label","labcen","hbce",Bcenter,"space=3");
   zdialog_add_widget(zd,"label","space","hbce",0,"expand");
   zdialog_add_widget(zd,"label","labend","hbce",Bedge,"space=5");

   snprintf(text,100,mformat,0.0,0.0,0.0);                                       //  mean RGB:     0     0     0
   zdialog_add_widget(zd,"label","mlab","dialog",text);
   snprintf(text,100,nformat,0.0,0.0,0.0);                                       //  mean noise:  0.00  0.00  0.00
   zdialog_add_widget(zd,"label","nlab","dialog",text);

   zdialog_resize(zd,300,300);
   zdialog_run(zd,denoise_measure_dialog_event,"save");                          //  run dialog
   return;
}


//  dialog event and completion function

int denoise_measure_dialog_event(zdialog *zd, cchar *event)
{
   using namespace denoise_names;

   void denoise_measure_mousefunc();
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0

   if (strmatch(event,"focus"))
      takeMouse(denoise_measure_mousefunc,dragcursor);                           //  connect mouse function
   
   if (zd->zstat) {
      freeMouse();                                                               //  free mouse
      zdialog_free(zd);
      zd_denoise_measure = 0;
   }
   
   return 1;
}


//  mouse function
//  sample noise where the mouse is clicked
//  assumed: mouse is on a monotone image area

void denoise_measure_mousefunc()
{
   using namespace denoise_names;

   GtkWidget   *drawwin;
   GdkWindow   *gdkwin;
   cairo_t     *cr;
   zdialog     *zd = zd_denoise_measure;

   char        text[100];
   int         mx, my, px, py, qx, qy, Npix;
   float       *pix3, R;
   float       Rm, Gm, Bm, Rn, Gn, Bn, Ro, Go, Bo;
   int         dww, dhh;
   float       max, xscale, yscale;
   float       rx, ry;
   float       Noise[400][3];
   double      dashes[2] = { 1, 3 };
   
   if (! E3pxm) return;
   if (! zd) return;
   if (Fthreadbusy) return;

   mx = Mxposn;                                                                  //  mouse position
   my = Myposn;

   if (mx < 13 || mx >= E3ww-13) return;                                         //  must be 12+ pixels from image edge
   if (my < 13 || my >= E3hh-13) return;
   
   draw_mousecircle(Mxposn,Myposn,10,0,0);                                       //  draw mouse circle, radius 10

   Npix = 0;
   Rm = Gm = Bm = 0;
   
   for (py = my-10; py <= my+10; py++)                                           //  loop pixels within mouise circle
   for (px = mx-10; px <= mx+10; px++)                                           //  (approx. 314 pixels)
   {
      R = sqrtf((px-mx)*(px-mx) + (py-my)*(py-my));                              //  distance from center
      if (R > 10) continue;

      pix3 = PXMpix(E3pxm,px,py);                                                //  get pixel RGB values
      Rm += pix3[0];                                                             //  accumulate
      Gm += pix3[1];
      Bm += pix3[2];

      Npix++;
   }

   Rm = Rm / Npix;                                                               //  mean RGB values
   Gm = Gm / Npix;                                                               //    for pixels within mouse
   Bm = Bm / Npix;

   Npix = 0;
   Rn = Gn = Bn = 0;

   for (py = my-10; py <= my+10; py++)                                           //  loop pixels within mouise circle
   for (px = mx-10; px <= mx+10; px++)                                           //  (approx. 314 pixels)
   {
      R = sqrtf((px-mx)*(px-mx) + (py-my)*(py-my));                              //  distance from center
      if (R > 10) continue;

      Ro = Go = Bo = 0;

      for (qy = py-2; qy <= py+2; qy++)                                          //  for each pixel, get mean RGB
      for (qx = px-2; qx <= px+2; qx++)                                          //    for 5x5 surrounding pixels
      {
         pix3 = PXMpix(E3pxm,qx,qy);
         Ro += pix3[0];
         Go += pix3[1];
         Bo += pix3[2];
      }
      
      Ro = Ro / 25;                                                              //  mean RGB for surrounding pixels
      Go = Go / 25;
      Bo = Bo / 25;

      pix3 = PXMpix(E3pxm,px,py);                                                //  get pixel RGB noise levels

      Noise[Npix][0] = pix3[0] - Ro;                                             //  noise = pixel value - mean
      Noise[Npix][1] = pix3[1] - Go;
      Noise[Npix][2] = pix3[2] - Bo;
      
      Rn += fabsf(Noise[Npix][0]);                                               //  accumulate absolute values
      Gn += fabsf(Noise[Npix][1]);
      Bn += fabsf(Noise[Npix][2]);

      Npix++;
   }

   Rn = Rn / Npix;                                                               //  mean RGB noise levels 
   Gn = Gn / Npix;                                                               //    for pixels within mouse
   Bn = Bn / Npix;

   snprintf(text,100,mformat,0.0,0.0,0.0);                                       //  clear dialog data
   zdialog_stuff(zd,"mlab",text);
   snprintf(text,100,nformat,0.0,0.0,0.0);
   zdialog_stuff(zd,"nlab",text);

   snprintf(text,100,mformat,Rm,Gm,Bm);                                          //  mean RGB:   NNN   NNN   NNN
   zdialog_stuff(zd,"mlab",text);

   snprintf(text,100,nformat,Rn,Gn,Bn);                                          //  mean noise:  N.NN  N.NN  N.NN
   zdialog_stuff(zd,"nlab",text);

   max = Rn;
   if (Gn > max) max = Gn;
   if (Bn > max) max = Bn;

   drawwin = denoise_measure_drawwin;
   gdkwin = gtk_widget_get_window(drawwin);                                      //  GDK drawing window

   dww = gtk_widget_get_allocated_width(drawwin);                                //  drawing window size
   dhh = gtk_widget_get_allocated_height(drawwin);

   xscale = dww / 10.0;                                                          //  x scale:  0 to max radius
   yscale = dhh / 20.0;                                                          //  y scale: -10 to +10 noise level
   
   cr = draw_context_create(gdkwin,draw_context);
   
   cairo_set_source_rgb(cr,1,1,1);                                               //  white background
   cairo_paint(cr);
   
   cairo_set_source_rgb(cr,0,0,0);                                               //  paint black

   cairo_set_line_width(cr,2);                                                   //  center line
   cairo_set_dash(cr,dashes,0,0);
   cairo_move_to(cr,0,0.5*dhh);
   cairo_line_to(cr,dww,0.5*dhh);
   cairo_stroke(cr);
   
   cairo_set_dash(cr,dashes,2,0);                                                //  dash lines at -5 and +5
   cairo_move_to(cr,0,0.25*dhh);
   cairo_line_to(cr,dww,0.25*dhh);
   cairo_move_to(cr,0,0.75*dhh);
   cairo_line_to(cr,dww,0.75*dhh);
   cairo_stroke(cr);

   cairo_set_source_rgb(cr,1,0,0);

   Npix = 0;

   for (py = my-10; py <= my+10; py++)                                           //  loop pixels within mouise circle
   for (px = mx-10; px <= mx+10; px++)                                           //  (approx. 314 pixels)
   {
      R = sqrtf((px-mx)*(px-mx) + (py-my)*(py-my));                              //  distance from center
      if (R > 10) continue;
      Rn = Noise[Npix][0];                                                       //  RED noise
      rx = R * xscale;                                                           //  px, 0 to dww
      ry = 0.5 * dhh - Rn * yscale;                                              //  red py, 0 to dhh
      cairo_move_to(cr,rx,ry);
      cairo_arc(cr,rx,ry,1,0,2*PI);
      Npix++;
   }      

   cairo_stroke(cr);

   cairo_set_source_rgb(cr,0,1,0);

   Npix = 0;

   for (py = my-10; py <= my+10; py++)                                           //  same for GREEN noise
   for (px = mx-10; px <= mx+10; px++)
   {
      R = sqrtf((px-mx)*(px-mx) + (py-my)*(py-my));
      if (R > 10) continue;
      Gn = Noise[Npix][1];
      rx = R * xscale;
      ry = 0.5 * dhh - Gn * yscale;
      cairo_move_to(cr,rx,ry);
      cairo_arc(cr,rx,ry,1,0,2*PI);
      Npix++;
   }      

   cairo_stroke(cr);

   cairo_set_source_rgb(cr,0,0,1);

   Npix = 0;

   for (py = my-10; py <= my+10; py++)                                           //  same for BLUE noise
   for (px = mx-10; px <= mx+10; px++)
   {
      R = (px-mx)*(px-mx) + (py-my)*(py-my);
      if (R > 100) continue;
      R = 0.1 * R;
      Bn = Noise[Npix][2];
      rx = R * xscale;
      ry = 0.5 * dhh - Bn * yscale;
      cairo_move_to(cr,rx,ry);
      cairo_arc(cr,rx,ry,1,0,2*PI);
      Npix++;
   }      

   cairo_stroke(cr);

   draw_context_destroy(draw_context); 
   return;
}


/********************************************************************************/

//  red eye removal function

namespace redeye_names
{
   struct sredmem {                                                              //  red-eye struct in memory
      char        type, space[3];
      int         cx, cy, ww, hh, rad, clicks;
      float       thresh, tstep;
   };
   sredmem  redmem[100];                                                         //  store up to 100 red-eyes

   int      E3ww, E3hh;
   int      Nredmem = 0, maxredmem = 100;

   editfunc    EFredeye;

   #define PIXRED(pix) (25*pix[0]/(PIXBRIGHT(pix)+1))                            //  red brightness 0-100%
}


//  menu function

void m_redeyes(GtkWidget *, cchar *menu)
{
   using namespace redeye_names;

   int      redeye_dialog_event(zdialog *zd, cchar *event);
   void     redeye_mousefunc();

   cchar    *redeye_message = 
               "Method 1:\n"
               "  Left-click on red-eye to darken.\n"
               "Method 2:\n"
               "  Drag down and right to enclose red-eye.\n"
               "  Left-click on red-eye to darken.\n"
               "Undo red-eye:\n"
               "  Right-click on red-eye.";

   F1_help_topic = "red eyes";

   EFredeye.menufunc = m_redeyes;
   EFredeye.menuname = "Red Eyes";
   EFredeye.Farea = 1;                                                           //  select area ignored
   EFredeye.mousefunc = redeye_mousefunc;
   if (! edit_setup(EFredeye)) return;                                           //  setup edit
   
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

   zdialog *zd = zdialog_new("Red Eye Reduction",Mwin,BOK,Bcancel,null);
   EFredeye.zd = zd;

   zdialog_add_widget(zd,"label","lab1","dialog",redeye_message);
   zdialog_run(zd,redeye_dialog_event,"save");                                   //  run dialog - parallel

   Nredmem = 0;
   takeMouse(redeye_mousefunc,dragcursor);                                       //  connect mouse function
   return;
}


//  dialog event and completion callback function

int redeye_dialog_event(zdialog *zd, cchar *event)
{
   using namespace redeye_names;

   void     redeye_mousefunc();

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()
   
   if (zd->zstat)
   {
      if (Nredmem > 0) {
         CEF->Fmods++;
         CEF->Fsaved = 0;
      }
      if (zd->zstat == 1) edit_done(0);                                          //  commit edit
      else edit_cancel(0);                                                       //  discard edit
      return 1;
   }

   if (strmatch(event,"focus"))                                                  //  toggle mouse capture
      takeMouse(redeye_mousefunc,dragcursor);                                    //  connect mouse function

   return 1;
}


int      redeye_createF(int px, int py);                                         //  create 1-click red-eye (type F)
int      redeye_createR(int px, int py, int ww, int hh);                         //  create robust red-eye (type R)
void     redeye_darken(int ii);                                                  //  darken red-eye
void     redeye_distr(int ii);                                                   //  build pixel redness distribution
int      redeye_find(int px, int py);                                            //  find red-eye at mouse position
void     redeye_remove(int ii);                                                  //  remove red-eye at mouse position
int      redeye_radlim(int cx, int cy);                                          //  compute red-eye radius limit

void redeye_mousefunc()
{
   using namespace redeye_names;

   int         ii, px, py, ww, hh;

   if (Nredmem == maxredmem) {
      zmessageACK(Mwin,"%d red-eye limit reached",maxredmem);                    //  too many red-eyes
      return;
   }

   if (LMclick)                                                                  //  left mouse click
   {
      px = Mxclick;                                                              //  click position
      py = Myclick;
      if (px < 0 || px > E3ww-1 || py < 0 || py > E3hh-1)                        //  outside image area
         return;

      ii = redeye_find(px,py);                                                   //  find existing red-eye
      if (ii < 0) ii = redeye_createF(px,py);                                    //  or create new type F
      redeye_darken(ii);                                                         //  darken red-eye
      Fpaint2();
   }

   if (RMclick)                                                                  //  right mouse click
   {
      px = Mxclick;                                                              //  click position
      py = Myclick;
      ii = redeye_find(px,py);                                                   //  find red-eye
      if (ii >= 0) redeye_remove(ii);                                            //  if found, remove
      Fpaint2();
   }

   LMclick = RMclick = 0;

   if (Mxdrag || Mydrag)                                                         //  mouse drag underway
   {
      px = Mxdown;                                                               //  initial position
      py = Mydown;
      ww = Mxdrag - Mxdown;                                                      //  increment
      hh = Mydrag - Mydown;
      Mxdrag = Mydrag = 0;
      if (ww < 2 && hh < 2) return;
      if (ww < 2) ww = 2;
      if (hh < 2) hh = 2;
      if (px < 1) px = 1;                                                        //  keep within image area
      if (py < 1) py = 1;
      if (px + ww > E3ww-1) ww = E3ww-1 - px;
      if (py + hh > E3hh-1) hh = E3hh-1 - py;
      ii = redeye_find(px,py);                                                   //  find existing red-eye
      if (ii >= 0) redeye_remove(ii);                                            //  remove it
      ii = redeye_createR(px,py,ww,hh);                                          //  create new red-eye type R
   }

   return;
}


//  create type F redeye (1-click automatic)

int redeye_createF(int cx, int cy)
{
   using namespace redeye_names;

   int         cx0, cy0, cx1, cy1, px, py, rad, radlim;
   int         loops, ii;
   int         Tnpix, Rnpix, R2npix;
   float       rd, rcx, rcy, redpart;
   float       Tsum, Rsum, R2sum, Tavg, Ravg, R2avg;
   float       sumx, sumy, sumr;
   float       *ppix;

   cx0 = cx;
   cy0 = cy;

   for (loops = 0; loops < 8; loops++)
   {
      cx1 = cx;
      cy1 = cy;

      radlim = redeye_radlim(cx,cy);                                             //  radius limit (image edge)
      Tsum = Tavg = Ravg = Tnpix = 0;

      for (rad = 0; rad < radlim-2; rad++)                                       //  find red-eye radius from (cx,cy)
      {
         Rsum = Rnpix = 0;
         R2sum = R2npix = 0;

         for (py = cy-rad-2; py <= cy+rad+2; py++)
         for (px = cx-rad-2; px <= cx+rad+2; px++)
         {
            rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
            ppix = PXMpix(E3pxm,px,py);
            redpart = PIXRED(ppix);

            if (rd <= rad + 0.5 && rd > rad - 0.5) {                             //  accum. redness at rad
               Rsum += redpart;
               Rnpix++;
            }
            else if (rd <= rad + 2.5 && rd > rad + 1.5) {                        //  accum. redness at rad+2
               R2sum += redpart;
               R2npix++;
            }
         }

         Tsum += Rsum;
         Tnpix += Rnpix;
         Tavg = Tsum / Tnpix;                                                    //  avg. redness over 0-rad
         Ravg = Rsum / Rnpix;                                                    //  avg. redness at rad
         R2avg = R2sum / R2npix;                                                 //  avg. redness at rad+2
         if (R2avg > Ravg || Ravg > Tavg) continue;
         if ((Ravg - R2avg) < 0.2 * (Tavg - Ravg)) break;                        //  0.1 --> 0.2
      }

      sumx = sumy = sumr = 0;
      rad = int(1.2 * rad + 1);
      if (rad > radlim) rad = radlim;

      for (py = cy-rad; py <= cy+rad; py++)                                      //  compute center of gravity for
      for (px = cx-rad; px <= cx+rad; px++)                                      //   pixels within rad of (cx,cy)
      {
         rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (rd > rad + 0.5) continue;
         ppix = PXMpix(E3pxm,px,py);
         redpart = PIXRED(ppix);                                                 //  weight by redness
         sumx += redpart * (px - cx);
         sumy += redpart * (py - cy);
         sumr += redpart;
      }

      rcx = cx + 1.0 * sumx / sumr;                                              //  new center of red-eye
      rcy = cy + 1.0 * sumy / sumr;
      if (fabsf(cx0 - rcx) > 0.6 * rad) break;                                   //  give up if big movement
      if (fabsf(cy0 - rcy) > 0.6 * rad) break;
      cx = int(rcx + 0.5);
      cy = int(rcy + 0.5);
      if (cx == cx1 && cy == cy1) break;                                         //  done if no change
   }

   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   ii = Nredmem++;                                                               //  add red-eye to memory
   redmem[ii].type = 'F';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  create type R red-eye (drag an ellipse over red-eye area)

int redeye_createR(int cx, int cy, int ww, int hh)
{
   using namespace redeye_names;

   int      rad, radlim;

   draw_mousearc(cx,cy,2*ww,2*hh,0,0);                                           //  draw ellipse around mouse pointer

   if (ww > hh) rad = ww;
   else rad = hh;
   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   int ii = Nredmem++;                                                           //  add red-eye to memory
   redmem[ii].type = 'R';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].ww = 2 * ww;
   redmem[ii].hh = 2 * hh;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  darken a red-eye and increase click count

void redeye_darken(int ii)
{
   using namespace redeye_names;

   int         cx, cy, ww, hh, px, py, rad, clicks;
   float       rd, thresh, tstep;
   char        type;
   float       *ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   thresh = redmem[ii].thresh;
   tstep = redmem[ii].tstep;
   clicks = redmem[ii].clicks++;

   if (thresh == 0)                                                              //  1st click
   {
      redeye_distr(ii);                                                          //  get pixel redness distribution
      thresh = redmem[ii].thresh;                                                //  initial redness threshold
      tstep = redmem[ii].tstep;                                                  //  redness step size
      draw_mousearc(0,0,0,0,1,0);                                                //  erase mouse ellipse
   }

   tstep = (thresh - tstep) / thresh;                                            //  convert to reduction factor
   thresh = thresh * pow(tstep,clicks);                                          //  reduce threshold by total clicks

   for (py = cy-rad; py <= cy+rad; py++)                                         //  darken pixels over threshold
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm,px,py);                                                //  set redness = threshold
      if (PIXRED(ppix) > thresh)
         ppix[0] = int(thresh * (0.65 * ppix[1] + 0.10 * ppix[2] + 1) / (25 - 0.25 * thresh));
   }

   return;
}


//  Build a distribution of redness for a red-eye. Use this information
//  to set initial threshold and step size for stepwise darkening.

void redeye_distr(int ii)
{
   using namespace redeye_names;

   int         cx, cy, ww, hh, rad, px, py;
   int         bin, npix, dbins[20], bsum, blim;
   float       rd, maxred, minred, redpart, dbase, dstep;
   char        type;
   float       *ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;

   maxred = 0;
   minred = 100;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm,px,py);
      redpart = PIXRED(ppix);
      if (redpart > maxred) maxred = redpart;
      if (redpart < minred) minred = redpart;
   }

   dbase = minred;
   dstep = (maxred - minred) / 19.99;

   for (bin = 0; bin < 20; bin++) dbins[bin] = 0;
   npix = 0;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = PXMpix(E3pxm,px,py);
      redpart = PIXRED(ppix);
      bin = int((redpart - dbase) / dstep);
      ++dbins[bin];
      ++npix;
   }

   bsum = 0;
   blim = int(0.5 * npix);

   for (bin = 0; bin < 20; bin++)                                                //  find redness level for 50% of
   {                                                                             //    pixels within red-eye radius
      bsum += dbins[bin];
      if (bsum > blim) break;
   }

   redmem[ii].thresh = dbase + dstep * bin;                                      //  initial redness threshold
   redmem[ii].tstep = dstep;                                                     //  redness step (5% of range)

   return;
}


//  find a red-eye (nearly) overlapping the mouse click position

int redeye_find(int cx, int cy)
{
   using namespace redeye_names;

   for (int ii = 0; ii < Nredmem; ii++)
   {
      if (cx > redmem[ii].cx - 2 * redmem[ii].rad &&
          cx < redmem[ii].cx + 2 * redmem[ii].rad &&
          cy > redmem[ii].cy - 2 * redmem[ii].rad &&
          cy < redmem[ii].cy + 2 * redmem[ii].rad)
            return ii;                                                           //  found
   }
   return -1;                                                                    //  not found
}


//  remove a red-eye from memory

void redeye_remove(int ii)
{
   using namespace redeye_names;

   int      cx, cy, rad, px, py;
   float    *pix1, *pix3;
   int      nc = E1pxm->nc, pcc = nc * sizeof(float);

   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   rad = redmem[ii].rad;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);
      memcpy(pix3,pix1,pcc);
   }

   for (ii++; ii < Nredmem; ii++)
      redmem[ii-1] = redmem[ii];
   Nredmem--;

   draw_mousearc(0,0,0,0,1,0);                                                   //  erase mouse ellipse
   return;
}


//  compute red-eye radius limit: smaller of 100 and nearest image edge

int redeye_radlim(int cx, int cy)
{
   using namespace redeye_names;

   int radlim = 100;
   if (cx < 100) radlim = cx;
   if (E3ww-1 - cx < 100) radlim = E3ww-1 - cx;
   if (cy < 100) radlim = cy;
   if (E3hh-1 - cy < 100) radlim = E3hh-1 - cy;
   return radlim;
}


/********************************************************************************/

//  Smart Erase menu function - Replace pixels inside a select area
//    with a reflection of pixels outside the area.

namespace smarterase_names
{
   editfunc    EFsmarterase;
   int         E3ww, E3hh;
}


//  menu function

void m_smart_erase(GtkWidget *, const char *menu)
{
   using namespace smarterase_names;

   int smart_erase_dialog_event(zdialog* zd, const char *event);

   cchar    *erase_message = "Drag mouse to select. Erase. Repeat. \n"
                             "Click: extend selection to mouse.";
   F1_help_topic = "smart erase";

   EFsmarterase.menufunc = m_smart_erase;
   EFsmarterase.menuname = "Smart Erase";
   EFsmarterase.Farea = 0;                                                       //  select area deleted
   EFsmarterase.mousefunc = sa_mouse_select;                                     //  mouse function (use select area)
   if (! edit_setup(EFsmarterase)) return;                                       //  setup edit
   
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

/***
       _________________________________________
      |                                         |
      | Drag mouse to select. Erase. Repeat.    |
      | Click: extend selection to mouse.       |
      |                                         |
      | Radius [ 10 ]    Blur [ 1.5 ]           |
      | [New Area] [Show] [Hide] [Erase] [Undo] |
      |                                         |
      |                                 [ OK ]  |
      |_________________________________________|

***/

   zdialog *zd = zdialog_new("Smart Erase",Mwin,BOK,null);
   EFsmarterase.zd = zd;

   zdialog_add_widget(zd,"label","lab1","dialog",erase_message,"space=3");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labr","hb2","Radius","space=5");
   zdialog_add_widget(zd,"zspin","radius","hb2","1|30|1|10");
   zdialog_add_widget(zd,"label","labb","hb2","Blur","space=10");
   zdialog_add_widget(zd,"zspin","blur","hb2","0|9|0.5|1");
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","newarea","hb3","New Area","space=3");
   zdialog_add_widget(zd,"button","show","hb3",Bshow,"space=3");
   zdialog_add_widget(zd,"button","hide","hb3",Bhide,"space=3");
   zdialog_add_widget(zd,"button","erase","hb3",Berase,"space=3");
   zdialog_add_widget(zd,"button","undo1","hb3",Bundo,"space=3");

   sa_clear();                                                                   //  clear area if any
   sa_pixmap_create();                                                           //  allocate select area pixel maps
   sa_mode = mode_mouse;                                                         //  mode = select by mouse
   sa_stat = 1;                                                                  //  status = active edit
   sa_fww = E1pxm->ww;
   sa_fhh = E1pxm->hh;
   sa_searchrange = 1;                                                           //  search within mouse radius
   sa_mouseradius = 10;                                                          //  initial mouse select radius
   sa_lastx = sa_lasty = 0;                                                      //  initz. for sa_mouse_select 
   takeMouse(sa_mouse_select,0);                                                 //  use select area mouse function
   sa_show(1,0);

   zdialog_run(zd,smart_erase_dialog_event,"save");                              //  run dialog - parallel
   return;
}


//  dialog event and completion function

int smart_erase_dialog_event(zdialog *zd, const char *event)
{
   using namespace smarterase_names;

   void smart_erase_func(int mode);
   int  smart_erase_blur(float radius);

   float       radius;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (zd->zstat)
   {
      sa_clear();                                                                //  clear select area
      freeMouse();                                                               //  disconnect mouse
      if (zd->zstat == 1) edit_done(0);                                          //  commit edit
      else edit_cancel(0);                                                       //  discard edit
      return 1;
   }

   if (strmatch(event,"radius"))
      zdialog_fetch(zd,"radius",sa_mouseradius);

   if (strmatch(event,"newarea")) {
      sa_clear();
      sa_lastx = sa_lasty = 0;                                                   //  forget last click
      sa_pixmap_create();                                                        //  allocate select area pixel maps
      sa_mode = mode_mouse;                                                      //  mode = select by mouse
      sa_stat = 1;                                                               //  status = active edit
      sa_fww = E1pxm->ww;
      sa_fhh = E1pxm->hh;
      sa_show(1,0);
      takeMouse(sa_mouse_select,0);
   }

   if (strmatch(event,"show")) {
      sa_show(1,0);
      takeMouse(sa_mouse_select,0);
   }

   if (strmatch(event,"hide")) {
      sa_show(0,0);
      freeMouse();
   }

   if (strmatch(event,"erase")) {                                                //  do smart erase
      sa_finish_auto();                                                          //  finish the area
      smart_erase_func(1);
      zdialog_fetch(zd,"blur",radius);                                           //  add optional blur
      if (radius > 0) smart_erase_blur(radius);
      sa_show(0,0);
   }

   if (strmatch(event,"undo1"))                                                  //  dialog undo, undo last erase
      smart_erase_func(2);

   return 1;
}


//  erase the area or restore last erased area
//  mode = 1 = erase, mode = 2 = restore

void smart_erase_func(int mode)
{
   using namespace smarterase_names;

   int         px, py, npx, npy;
   int         qx, qy, sx, sy, tx, ty;
   int         ww, hh, ii, rad, inc, cc;
   int         dist2, mindist2;
   float       slope;
   char        *pmap;
   float       *pix1, *pix3;
   int         nc = E1pxm->nc, pcc = nc * sizeof(float);

   if (sa_stat != 3) return;                                                     //  nothing selected
   if (! sa_validate()) return;                                                  //  area invalid for curr. image file

   ww = E1pxm->ww;
   hh = E1pxm->hh;

   for (py = sa_miny; py < sa_maxy; py++)                                        //  undo all pixels in area
   for (px = sa_minx; px < sa_maxx; px++)
   {
      ii = py * ww + px;
      if (! sa_pixmap[ii]) continue;                                             //  pixel not selected

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      memcpy(pix3,pix1,pcc);
   }

   Fpaint2();                                                                    //  update window

   if (mode == 2) return;                                                        //  mode = undo, done

   cc = ww * hh;                                                                 //  allocate pixel done map
   pmap = (char *) zmalloc(cc);
   memset(pmap,0,cc);

   for (py = sa_miny; py < sa_maxy; py++)                                        //  loop all pixels in area
   for (px = sa_minx; px < sa_maxx; px++)
   {
      ii = py * ww + px;
      if (! sa_pixmap[ii]) continue;                                             //  pixel not selected
      if (pmap[ii]) continue;                                                    //  pixel already done

      mindist2 = 999999;                                                         //  find nearest edge
      npx = npy = 0;

      for (rad = 1; rad < 50; rad++)                                             //  50 pixel limit
      {
         for (qx = px-rad; qx <= px+rad; qx++)                                   //  search within rad
         for (qy = py-rad; qy <= py+rad; qy++)
         {
            if (qx < 0 || qx >= ww) continue;                                    //  off image edge
            if (qy < 0 || qy >= hh) continue;
            ii = qy * ww + qx;
            if (sa_pixmap[ii]) continue;                                         //  within selected area

            dist2 = (px-qx) * (px-qx) + (py-qy) * (py-qy);                       //  distance**2 to edge pixel
            if (dist2 < mindist2) {
               mindist2 = dist2;
               npx = qx;                                                         //  save nearest edge pixel found
               npy = qy;
            }
         }

         if (rad * rad >= mindist2) break;                                       //  found edge, can quit now
      }

      if (! npx && ! npy) continue;                                              //  edge not found, should not happen

      qx = npx;                                                                  //  nearest edge pixel from px/py
      qy = npy;

      if (abs(qy - py) > abs(qx - px))                                           //  line px/py to qx/qy is more
      {                                                                          //    vertical than horizontal
         slope = 1.0 * (qx - px) / (qy - py);
         if (qy > py) inc = 1;
         else inc = -1;

         for (sy = py; sy != qy; sy += inc) {                                    //  sx/sy = line from px/py to qx/qy
            sx = px + slope * (sy - py);

            ii = sy * ww + sx;
            if (pmap[ii]) continue;                                              //  skip done pixels
            pmap[ii] = 1;

            tx = qx + (qx - sx);                                                 //  tx/ty = extended line from qx/qy
            ty = qy + (qy - sy);

            if (tx < 0) tx = 0;                                                  //  don't go off edge
            if (tx > ww-1) tx = ww-1;
            if (ty < 0) ty = 0;
            if (ty > hh-1) ty = hh-1;

            pix1 = PXMpix(E1pxm,tx,ty);                                          //  copy pixel from tx/ty to sx/sy
            pix3 = PXMpix(E3pxm,sx,sy);                                          //  simplified
            memcpy(pix3,pix1,pcc);
         }
      }

      else                                                                       //  more horizontal than vertical
      {
         slope = 1.0 * (qy - py) / (qx - px);
         if (qx > px) inc = 1;
         else inc = -1;

         for (sx = px; sx != qx; sx += inc) {
            sy = py + slope * (sx - px);

            ii = sy * ww + sx;
            if (pmap[ii]) continue;
            pmap[ii] = 1;

            tx = qx + (qx - sx);
            ty = qy + (qy - sy);

            if (tx < 0) tx = 0;
            if (tx > ww-1) tx = ww-1;
            if (ty < 0) ty = 0;
            if (ty > hh-1) ty = hh-1;

            pix1 = PXMpix(E1pxm,tx,ty);
            pix3 = PXMpix(E3pxm,sx,sy);
            memcpy(pix3,pix1,pcc);
         }
      }
   }

   zfree(pmap);                                                                  //  free memory
   CEF->Fmods++;
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return;
}


//  add blur to the erased area to help mask the side-effects

int smart_erase_blur(float radius)
{
   using namespace smarterase_names;

   int         ii, px, py, dx, dy, adx, ady;
   float       blur_weight[12][12];                                              //  up to blur radius = 10
   float       rad, radflat2;
   float       m, d, w, sum, weight;
   float       red, green, blue;
   float       *pix9, *pix3, *pixN;
   int         nc = E3pxm->nc;

   if (sa_stat != 3) return 0;

   rad = radius - 0.2;
   radflat2 = rad * rad;

   for (dx = 0; dx < 12; dx++)                                                   //  clear weights array
   for (dy = 0; dy < 12; dy++)
      blur_weight[dx][dy] = 0;

   for (dx = -rad-1; dx <= rad+1; dx++)                                          //  blur_weight[dx][dy] = no. of pixels
   for (dy = -rad-1; dy <= rad+1; dy++)                                          //    at distance (dx,dy) from center
      ++blur_weight[abs(dx)][abs(dy)];

   m = sqrt(radflat2 + radflat2);                                                //  corner pixel distance from center
   sum = 0;

   for (dx = 0; dx <= rad+1; dx++)                                               //  compute weight of pixel
   for (dy = 0; dy <= rad+1; dy++)                                               //    at distance dx, dy
   {
      d = sqrt(dx*dx + dy*dy);
      w = (m + 1.2 - d) / m;
      w = w * w;
      sum += blur_weight[dx][dy] * w;
      blur_weight[dx][dy] = w;
   }

   for (dx = 0; dx <= rad+1; dx++)                                               //  make weights add up to 1.0
   for (dy = 0; dy <= rad+1; dy++)
      blur_weight[dx][dy] = blur_weight[dx][dy] / sum;

   E9pxm = PXM_copy(E3pxm);                                                      //  copy edited image

   for (py = sa_miny; py < sa_maxy; py++)                                        //  loop all pixels in area
   for (px = sa_minx; px < sa_maxx; px++)
   {
      ii = py * E1pxm->ww + px;
      if (! sa_pixmap[ii]) continue;                                             //  pixel not in area

      pix9 = PXMpix(E9pxm,px,py);                                                //  source pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  target pixel

      rad = radius;
      red = green = blue = 0;

      for (dy = -rad-1; dy <= rad+1; dy++)                                       //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx >= E3ww) continue;                               //  omit pixels off edge
         if (py+dy < 0 || py+dy >= E3hh) continue;
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix9 + (dy * E3ww + dx) * nc;
         weight = blur_weight[adx][ady];                                         //  weight at distance (dx,dy)
         red += pixN[0] * weight;                                                //  accumulate contributions
         green += pixN[1] * weight;
         blue += pixN[2] * weight;
      }

      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }

   PXM_free(E9pxm);

   CEF->Fmods++;
   CEF->Fsaved = 0;

   Fpaint2();                                                                    //  update window
   return 0;
}


/********************************************************************************/

//  Adjust RGB menu function
//  Adjust Brightness, contrast, and color levels using RGB or CMY colors

namespace adjust_RGB_names
{
   editfunc    EF_RGB;                                                           //  edit function data
   float       RGB_inputs[8];
   int         E3ww, E3hh;
}


//  menu function

void m_adjust_RGB(GtkWidget *, cchar *menu)
{
   using namespace adjust_RGB_names;

   int    RGB_dialog_event(zdialog *zd, cchar *event);
   void * RGB_thread(void *);

   F1_help_topic = "adjust RGB";

   EF_RGB.menuname = "Adjust RGB";
   EF_RGB.menufunc = m_adjust_RGB;
   EF_RGB.FprevReq = 1;                                                          //  use preview
   EF_RGB.Farea = 2;                                                             //  select area usable
   EF_RGB.Frestart = 1;                                                          //  allow restart
   EF_RGB.Fscript = 1;                                                           //  scripting supported
   EF_RGB.Fpaint = 1;                                                            //  paint edits supported
   EF_RGB.threadfunc = RGB_thread;                                               //  thread function
   if (! edit_setup(EF_RGB)) return;                                             //  setup edit
   
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

/***
    ________________________________
   |                                |
   |   +Brightness    =====[]=====  |
   |    +Red -Cyan    =====[]=====  |
   | +Green -Magenta  =====[]=====  |
   |   +Blue -Yellow  =====[]=====  |
   |                                |
   |     Contrast     =====[]=====  |
   |       Red        =====[]=====  |
   |      Green       =====[]=====  |
   |       Blue       =====[]=====  |
   |                                |
   |        [reset] [ OK ] [cancel] |
   |________________________________|

***/

   zdialog *zd = zdialog_new("Adjust RGB",Mwin,Breset,BOK,Bcancel,null); 
   EF_RGB.zd = zd;

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb2",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"homog|expand");
   zdialog_add_widget(zd,"label","labBriteDens","vb1","+Brightness");
   zdialog_add_widget(zd,"label","labRedDens","vb1","+Red -Cyan");
   zdialog_add_widget(zd,"label","labGreenDens","vb1","+Green -Magenta");
   zdialog_add_widget(zd,"label","labBlueDens","vb1","+Blue -Yellow");
   zdialog_add_widget(zd,"hsep","sep1","vb1");
   zdialog_add_widget(zd,"label","labContrast","vb1","Contrast All");
   zdialog_add_widget(zd,"label","labRedCon","vb1","Contrast Red");
   zdialog_add_widget(zd,"label","labGreenCon","vb1","Contrast Green");
   zdialog_add_widget(zd,"label","labBlueCon","vb1","Contrast Blue");
   zdialog_add_widget(zd,"hscale","BriteDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","RedDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","GreenDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","BlueDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hsep","sep2","vb2");
   zdialog_add_widget(zd,"hscale","Contrast","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","RedCon","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","GreenCon","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","BlueCon","vb2","-1|+1|0.001|0","expand");

   zdialog_rescale(zd,"BriteDens",-1,0,+1);
   zdialog_rescale(zd,"RedDens",-1,0,+1);
   zdialog_rescale(zd,"GreenDens",-1,0,+1);
   zdialog_rescale(zd,"BlueDens",-1,0,+1);
   zdialog_rescale(zd,"Contrast",-1,0,+1);
   zdialog_rescale(zd,"RedCon",-1,0,+1);
   zdialog_rescale(zd,"GreenCon",-1,0,+1);
   zdialog_rescale(zd,"BlueCon",-1,0,+1);

   zdialog_resize(zd,300,0);
   zdialog_restore_inputs(zd);                                                   //  restore prior inputs
   zdialog_run(zd,RGB_dialog_event,"save");                                      //  run dialog - parallel
   
   zdialog_send_event(zd,"apply");
   return;
}


//  RGB dialog event and completion function

int RGB_dialog_event(zdialog *zd, cchar *event)                                  //  RGB dialog event function
{
   using namespace adjust_RGB_names;

   int      mod = 0;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   
   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      E3ww = E3pxm->ww;
      E3hh = E3pxm->hh;
      signal_thread();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         RGB_inputs[0] = 0;
         RGB_inputs[1] = 0;
         RGB_inputs[2] = 0;
         RGB_inputs[3] = 0;
         RGB_inputs[4] = 0;
         RGB_inputs[5] = 0;
         RGB_inputs[6] = 0;
         RGB_inputs[7] = 0;
         zdialog_stuff(zd,"BriteDens",0);
         zdialog_stuff(zd,"RedDens",0);
         zdialog_stuff(zd,"GreenDens",0);
         zdialog_stuff(zd,"BlueDens",0);
         zdialog_stuff(zd,"Contrast",0);
         zdialog_stuff(zd,"RedCon",0);
         zdialog_stuff(zd,"GreenCon",0);
         zdialog_stuff(zd,"BlueCon",0);
         edit_reset();
         return 1;
      }
      else if (zd->zstat == 2) {                                                 //  done
         edit_fullsize();                                                        //  get full size image
         E3ww = E3pxm->ww;
         E3hh = E3pxm->hh;
         signal_thread();
         edit_done(0);                                                           //  commit edit
         return 1;
      }
      else {
         edit_cancel(0);                                                         //  discard edit
         return 1;
      }
   }

   if (strmatch("focus",event)) return 1;
   
   zdialog_fetch(zd,"BriteDens",RGB_inputs[0]);                                  //  get all inputs
   zdialog_fetch(zd,"RedDens",RGB_inputs[1]);
   zdialog_fetch(zd,"GreenDens",RGB_inputs[2]);
   zdialog_fetch(zd,"BlueDens",RGB_inputs[3]);
   zdialog_fetch(zd,"Contrast",RGB_inputs[4]);
   zdialog_fetch(zd,"RedCon",RGB_inputs[5]);
   zdialog_fetch(zd,"GreenCon",RGB_inputs[6]);
   zdialog_fetch(zd,"BlueCon",RGB_inputs[7]);

   if (RGB_inputs[0]) mod++;
   if (RGB_inputs[1]) mod++;
   if (RGB_inputs[2]) mod++;
   if (RGB_inputs[3]) mod++;
   if (RGB_inputs[4]) mod++;
   if (RGB_inputs[5]) mod++;
   if (RGB_inputs[6]) mod++;
   if (RGB_inputs[7]) mod++;
   
   if (mod) signal_thread();                                                     //  trigger update thread
   return 1;
}


//  thread function - multiple working threads to update image

void * RGB_thread(void *)
{
   using namespace adjust_RGB_names;

   void  * RGB_wthread(void *arg);                                               //  worker thread

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

       if (sa_stat == 3) progressmon_reset(sa_Npixel);                           //  initz. progress counter
       else  progressmon_reset(E3ww * E3hh);

      do_wthreads(RGB_wthread,NWT);                                              //  worker threads

      progressmon_reset(0);
      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


//  worker thread function

void * RGB_wthread(void *arg)
{
   using namespace adjust_RGB_names;

   float    R1, G1, B1, R3, G3, B3;
   float    briA, briR, briG, briB;
   float    conA, conR, conG, conB;
   float    R, G, B;

   int      index = *((int *) (arg));
   int      px, py, ii, dist = 0;
   float    *pix1, *pix3;
   float    f1, f2;
   float    max$;

   briA = RGB_inputs[0];                                                         //  color brightness inputs, -1 to +1
   briR = RGB_inputs[1];
   briG = RGB_inputs[2];
   briB = RGB_inputs[3];

   R = briR - 0.5 * briG - 0.5 * briB + briA;                                    //  red = red - green - blue + all
   G = briG - 0.5 * briR - 0.5 * briB + briA;                                    //  etc.
   B = briB - 0.5 * briR - 0.5 * briG + briA;

   R += 1;                                                                       //  -1 ... 0 ... +1  >>  0 ... 1 ... 2
   G += 1;                                                                       //  increase the range
   B += 1;

   briR = R;                                                                     //  final color brightness factors
   briG = G;
   briB = B;

   conA = RGB_inputs[4];                                                         //  contrast inputs, -1 to +1
   conR = RGB_inputs[5];
   conG = RGB_inputs[6];
   conB = RGB_inputs[7];

   if (conA < 0) conA = 0.5 * conA + 1;                                          //  -1 ... 0  >>  0.5 ... 1.0
   else conA = conA + 1;                                                         //   0 ... 1  >>  1.0 ... 2.0
   if (conR < 0) conR = 0.5 * conR + 1;
   else conR = conR + 1;
   if (conG < 0) conG = 0.5 * conG + 1;
   else conG = conG + 1;
   if (conB < 0) conB = 0.5 * conB + 1;
   else conB = conB + 1;

   conR = conR * conA;                                                           //  apply overall contrast
   conG = conG * conA;
   conB = conB * conA;

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      R1 = pix1[0];                                                              //  input RGB values, 0-255
      G1 = pix1[1];
      B1 = pix1[2];

      R3 = R1 * briR;                                                            //  apply color brightness factors
      G3 = G1 * briG;
      B3 = B1 * briB;

      R3 = conR * (R3 - 128) + 128;                                              //  apply contrast factors
      G3 = conG * (G3 - 128) + 128;
      B3 = conB * (B3 - 128) + 128;

      RGBFIX(R3,G3,B3)                                                           //  21.0

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }

      pix3[0] = R3;                                                              //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   return 0;
}


/********************************************************************************/

//  HSL color menu function
//  Adjust colors using the HSL (hue/saturation/lightness) color model

namespace adjust_HSL_names
{
   GtkWidget   *RGBframe, *RGBcolor;
   GtkWidget   *Hframe, *Hscale;
   editfunc    EFHSL;                                                            //  edit function data
   int         E3ww, E3hh;
   int         Huse, Suse, Luse;                                                 //  "match using" flags, 0 or 1
   int         Hout, Sout, Lout;                                                 //  "output color" flags, 0 or 1
   float       Rm, Gm, Bm;                                                       //  RGB image color to match
   float       Hm, Sm, Lm;                                                       //  corresp. HSL color
   float       Mlev;                                                             //  match level 0..1 = 100%
   float       Hc, Sc, Lc;                                                       //  new color to add
   float       Rc, Gc, Bc;                                                       //  corresp. RGB color
   float       Adj;                                                              //  color adjustment, 0..1 = 100%
}


//  menu function

void m_adjust_HSL(GtkWidget *, cchar *menu)
{
   using namespace adjust_HSL_names;

   void   HSL_RGBcolor(GtkWidget *drawarea, cairo_t *cr, int *);
   void   HSL_Hscale(GtkWidget *drawarea, cairo_t *cr, int *);
   int    HSL_dialog_event(zdialog *zd, cchar *event);
   void   HSL_mousefunc();
   void * HSL_thread(void *);
   
   F1_help_topic = "adjust HSL";

   EFHSL.menuname = "Adjust HSL";
   EFHSL.menufunc = m_adjust_HSL;
   EFHSL.FprevReq = 1;                                                           //  use preview
   EFHSL.Farea = 2;                                                              //  select area usable
   EFHSL.Frestart = 1;                                                           //  allow restart
   EFHSL.Fpaint = 1;                                                             //  use with paint edits OK
   EFHSL.mousefunc = HSL_mousefunc;                                              //  mouse function
   EFHSL.threadfunc = HSL_thread;                                                //  thread function
   if (! edit_setup(EFHSL)) return;                                              //  setup edit
   
   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

/***
       ___________________________________________________
      |                                                   |
      |  Input color to match and adjust: [#####]         |
      |  Match using: [] Hue  [] Saturation  [] Lightness |
      |  Match Level: ==================[]========== 100% |
      |  - - - - - - - - - - - - - - - - - - - - - - - -  |
      |  Output Color                                     |
      |  [########]  [#################################]  |                      //  new color and hue spectrum
      |  [] Color Hue   ================[]==============  |
      |  [] Saturation  =====================[]=========  |
      |  [] Lightness   ===========[]===================  |
      |  Adjustment  ===================[]========== 100% |
      |                                                   |
      |                          [reset] [ OK ] [cancel]  |
      |___________________________________________________|

***/

   zdialog *zd = zdialog_new("Adjust HSL",Mwin,Breset,BOK,Bcancel,null);
   EFHSL.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","labmatch","hb1","Input color to match and adjust:","space=5");
   zdialog_add_widget(zd,"colorbutt","matchRGB","hb1","0|0|0");
   zdialog_add_ttip(zd,"matchRGB","shift+click on image to select color");
   
   zdialog_add_widget(zd,"hbox","hbmu","dialog");
   zdialog_add_widget(zd,"label","labmu","hbmu","Match using:","space=5");
   zdialog_add_widget(zd,"check","Huse","hbmu","Hue","space=3");
   zdialog_add_widget(zd,"check","Suse","hbmu","Saturation","space=3");
   zdialog_add_widget(zd,"check","Luse","hbmu","Lightness","space=3");
   
   zdialog_add_widget(zd,"hbox","hbmatch","dialog");
   zdialog_add_widget(zd,"label","labmatch","hbmatch",Bmatchlevel,"space=5");
   zdialog_add_widget(zd,"hscale","Mlev","hbmatch","0|1|0.001|1.0","expand");
   zdialog_add_widget(zd,"label","lab100%","hbmatch","100%","space=4");

   zdialog_add_widget(zd,"hsep","sep","dialog",0,"space=5");
   
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","laboutput","hb1","Output Color");

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb2",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"homog|expand");

   zdialog_add_widget(zd,"frame","RGBframe","vb1",0,"space=1");                  //  drawing area for RGB color
   RGBframe = zdialog_gtkwidget(zd,"RGBframe");
   RGBcolor = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(RGBframe),RGBcolor);
   gtk_widget_set_size_request(RGBcolor,0,16);
   G_SIGNAL(RGBcolor,"draw",HSL_RGBcolor,0);

   zdialog_add_widget(zd,"frame","Hframe","vb2",0,"space=1");                    //  drawing area for hue scale
   Hframe = zdialog_gtkwidget(zd,"Hframe");
   Hscale = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(Hframe),Hscale);
   gtk_widget_set_size_request(Hscale,200,16);
   G_SIGNAL(Hscale,"draw",HSL_Hscale,0);

   zdialog_add_widget(zd,"check","Hout","vb1","Color Hue");
   zdialog_add_widget(zd,"check","Sout","vb1","Saturation");
   zdialog_add_widget(zd,"check","Lout","vb1","Lightness");
   zdialog_add_widget(zd,"label","labadjust","vb1","Adjustment");

   zdialog_add_widget(zd,"hscale","Hc","vb2","0|359.9|0.1|180","expand");
   zdialog_add_widget(zd,"hscale","Sc","vb2","0|1|0.001|0.5","expand");
   zdialog_add_widget(zd,"hscale","Lc","vb2","0|1|0.001|0.5","expand");
   zdialog_add_widget(zd,"hbox","vb2hb","vb2");
   zdialog_add_widget(zd,"hscale","Adj","vb2hb","0|1|0.001|0.0","expand");
   zdialog_add_widget(zd,"label","lab100%","vb2hb","100%","space=4");

   zdialog_stuff(zd,"Huse",1);                                                   //  default: match on hue and saturation
   zdialog_stuff(zd,"Suse",1);
   zdialog_stuff(zd,"Luse",0);
   zdialog_stuff(zd,"Hout",1);                                                   //  default: replace only hue
   zdialog_stuff(zd,"Sout",0);
   zdialog_stuff(zd,"Lout",0);

   Rm = Gm = Bm = 0;                                                             //  color to match = black = not set
   Hm = Sm = Lm = 0;
   Huse = Suse = 1;
   Luse = 0;
   Hout = 1;
   Sout = Lout = 0;
   Mlev = 1.0;
   Hc = 180;                                                                     //  new HSL color to set / mix
   Sc = 0.5;
   Lc = 0.5;
   Adj = 0.0;
   
   zdialog_run(zd,HSL_dialog_event,"save");                                      //  run dialog - parallel
   takeMouse(HSL_mousefunc,arrowcursor);                                         //  connect mouse function
   return;
}


//  Paint RGBcolor drawing area with RGB color from new HSL color

void HSL_RGBcolor(GtkWidget *drawarea, cairo_t *cr, int *)
{
   using namespace adjust_HSL_names;

   int      ww, hh;

   ww = gtk_widget_get_allocated_width(drawarea);                                //  drawing area size
   hh = gtk_widget_get_allocated_height(drawarea);
   
   HSLtoRGB(Hc,Sc,Lc,Rc,Gc,Bc);                                                  //  new RGB color

   cairo_set_source_rgb(cr,Rc,Gc,Bc);
   cairo_rectangle(cr,0,0,ww-1,hh-1);
   cairo_fill(cr);

   return;
}


//  Paint Hscale drawing area with all hue values in a horizontal scale

void HSL_Hscale(GtkWidget *drawarea, cairo_t *cr, int *)
{
   using namespace adjust_HSL_names;

   int      px, ww, hh;
   float    H, S, L, R, G, B;

   ww = gtk_widget_get_allocated_width(drawarea);                                //  drawing area size
   hh = gtk_widget_get_allocated_height(drawarea);
   
   S = L = 0.5;

   for (px = 0; px < ww; px++)                                                   //  paint hue color scale
   {
      H = 360 * px / ww;
      HSLtoRGB(H,S,L,R,G,B);
      cairo_set_source_rgb(cr,R,G,B);
      cairo_move_to(cr,px,0);
      cairo_line_to(cr,px,hh-1);
      cairo_stroke(cr);
   }

   return;
}


//  HSL dialog event and completion function

int HSL_dialog_event(zdialog *zd, cchar *event)                                  //  HSL dialog event function
{
   using namespace adjust_HSL_names;

   void   HSL_mousefunc();

   int      mod = 0;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  from f_open()
   
   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      E3ww = E3pxm->ww;
      E3hh = E3pxm->hh;
      signal_thread();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  reset
         zd->zstat = 0;                                                          //  keep dialog active
         Mlev = 1.0;
         Hc = 180;                                                               //  set defaults
         Sc = 0.5;
         Lc = 0.5;
         Adj = 0.0;
         zdialog_stuff(zd,"Mlev",Mlev);
         zdialog_stuff(zd,"Hc",Hc);
         zdialog_stuff(zd,"Sc",Sc);
         zdialog_stuff(zd,"Lc",Lc);
         zdialog_stuff(zd,"Adj",Adj);
         edit_reset();
         return 1;
      }
      else if (zd->zstat == 2) {                                                 //  done
         edit_fullsize();                                                        //  get full size image
         E3ww = E3pxm->ww;
         E3hh = E3pxm->hh;
         signal_thread();
         edit_done(0);                                                           //  commit edit
         return 1;
      }
      else {
         edit_cancel(0);                                                         //  discard edit
         return 1;
      }
   }

   if (strmatch("focus",event)) {
      takeMouse(HSL_mousefunc,arrowcursor);
      return 1;
   }
   
   if (strmatch(event,"Huse")) {                                                 //  match on Hue, 0/1
      zdialog_fetch(zd,"Huse",Huse);
      mod = 1;
   }
   
   if (strmatch(event,"Suse")) {                                                 //  match on Saturation, 0/1
      zdialog_fetch(zd,"Suse",Suse);
      mod = 1;
   }
   
   if (strmatch(event,"Luse")) {                                                 //  match on Lightness, 0/1
      zdialog_fetch(zd,"Luse",Luse);
      mod = 1;
   }
   
   if (strmatch(event,"Hout")) {                                                 //  replace Hue, 0/1
      zdialog_fetch(zd,"Hout",Hout);
      mod = 1;
   }
   
   if (strmatch(event,"Sout")) {                                                 //  replace Saturation, 0/1
      zdialog_fetch(zd,"Sout",Sout);
      mod = 1;
   }
   
   if (strmatch(event,"Lout")) {                                                 //  replace Lightness, 0/1
      zdialog_fetch(zd,"Lout",Lout);
      mod = 1;
   }
   
   if (strmatch("Mlev",event)) {                                                 //  color match 0..1 = 100%
      zdialog_fetch(zd,"Mlev",Mlev);
      mod = 1;
   }

   if (strmatch("Hc",event)) {                                                   //  new color hue 0-360
      zdialog_fetch(zd,"Hc",Hc);
      mod = 1;
   }
      
   if (strmatch("Sc",event)) {                                                   //  saturation 0-1
      zdialog_fetch(zd,"Sc",Sc);
      mod = 1;
   }

   if (strmatch("Lc",event)) {                                                   //  lightness 0-1
      zdialog_fetch(zd,"Lc",Lc);
      mod = 1;
   }

   if (strmatch("Adj",event)) {                                                  //  adjustment 0..1 = 100%
      zdialog_fetch(zd,"Adj",Adj);
      mod = 1;
   }
   
   if (strmatch("blendwidth",event)) mod = 1;                                    //  area blend width changed
   
   if (mod) {
      gtk_widget_queue_draw(RGBcolor);                                           //  draw current RGB color
      signal_thread();                                                           //  trigger update thread
   }

   return 1;
}


//  mouse function
//  click on image to set the color to match and change

void HSL_mousefunc()
{
   using namespace adjust_HSL_names;

   int         mx, my, px, py;
   char        color[20];
   float       *pix1, R, G, B;
   float       f256 = 1.0 / 256.0;
   zdialog     *zd = EFHSL.zd;

   if (! KBshiftkey) return;                                                     //  check shift + left or right click
   if (! LMclick && ! RMclick) return;

   mx = Mxclick;                                                                 //  clicked pixel on image
   my = Myclick;

   if (mx < 1) mx = 1;                                                           //  pull back from image edge
   if (mx > E3ww - 2) mx = E3ww - 2;
   if (my < 1) my = 1;
   if (my > E3hh - 2) my = E3hh - 2;
   
   R = G = B = 0;

   for (py = my-1; py <= my+1; py++)                                             //  compute average RGB for 3x3
   for (px = mx-1; px <= mx+1; px++)                                             //    block of pixels
   {
      pix1 = PXMpix(E1pxm,px,py);
      R += pix1[0];
      G += pix1[1];
      B += pix1[2];
   }
   
   R = R / 9;
   G = G / 9;
   B = B / 9;

   if (LMclick)                                                                  //  left mouse click
   {                                                                             //  pick MATCH color from image
      LMclick = 0;
      snprintf(color,19,"%.0f|%.0f|%.0f",R,G,B);                                 //  draw new match color button
      if (zd) zdialog_stuff(zd,"matchRGB",color);
      Rm = R * f256;
      Gm = G * f256;
      Bm = B * f256;
      RGBtoHSL(Rm,Gm,Bm,Hm,Sm,Lm);                                               //  set HSL color to match
      signal_thread();                                                           //  trigger update thread
   }
   
   if (RMclick)                                                                  //  right mouse click
   {                                                                             //  pick OUTPUT color from image
      RMclick = 0;
      R = R * f256;
      G = G * f256;
      B = B * f256;
      RGBtoHSL(R,G,B,Hc,Sc,Lc);                                                  //  output HSL
      zdialog_stuff(zd,"Hc",Hc);
      zdialog_stuff(zd,"Sc",Sc);
      zdialog_stuff(zd,"Lc",Lc);
      gtk_widget_queue_draw(RGBcolor);                                           //  draw current RGB color
      signal_thread();                                                           //  trigger update thread
   }

   return;
}


//  thread function - multiple working threads to update image

void * HSL_thread(void *)
{
   using namespace adjust_HSL_names;

   void  * HSL_wthread(void *arg);                                               //  worker thread

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

      do_wthreads(HSL_wthread,NWT);                                              //  worker threads

      CEF->Fmods++;                                                              //  image modified
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


//  worker thread function

void * HSL_wthread(void *arg)
{
   using namespace adjust_HSL_names;
   
   int      index = *((int *) (arg));
   int      px, py, ii, dist = 0;
   float    *pix1, *pix3;
   float    R1, G1, B1, R3, G3, B3;
   float    H1, S1, L1, H3, S3, L3;
   float    dH, dS, dL, match;
   float    a1, a2, f1, f2;
   float    f256 = 1.0 / 256.0;

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py * E3ww + px;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                                //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel
      
      R1 = f256 * pix1[0];                                                       //  input pixel RGB
      G1 = f256 * pix1[1];
      B1 = f256 * pix1[2];
      
      RGBtoHSL(R1,G1,B1,H1,S1,L1);                                               //  convert to HSL

      match = 1.0;                                                               //  compare image pixel to match HSL

      if (Huse) {
         dH = fabsf(Hm - H1);
         if (360 - dH < dH) dH = 360 - dH;
         dH *= 0.002778;                                                         //  H difference, normalized 0..1
         match = 1.0 - dH;
      }

      if (Suse) {
         dS = fabsf(Sm - S1);                                                    //  S difference, 0..1
         match *= (1.0 - dS);
      }

      if (Luse) {
         dL = fabsf(Lm - L1);                                                    //  L difference, 0..1
         match *= (1.0 - dL);
      }

      a1 = pow(match, 10.0 * Mlev);                                              //  color selectivity, 0..1 = max
      a1 = Adj * a1;
      a2 = 1.0 - a1;
      
      if (Hout) H3 = a1 * Hc + a2 * H1;                                          //  output HSL = a1 * new HSL
      else H3 = H1;                                                              //             + a2 * old HSL
      if (Sout) S3 = a1 * Sc + a2 * S1;
      else S3 = S1;
      if (Lout) L3 = a1 * Lc + a2 * L1;
      else L3 = L1;
      
      HSLtoRGB(H3,S3,L3,R3,G3,B3);
      
      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  select area is active,
         f1 = sa_blendfunc(dist);                                                //    blend changes over sa_blendwidth
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }
      
      pix3[0] = 255.0 * R3;
      pix3[1] = 255.0 * G3;
      pix3[2] = 255.0 * B3;
   }

   return 0;
}


//  HSL to RGB converter (Wikipedia)
//  H = 0-360 deg.  S = 0-1   L = 0-1
//  output RGB values = 0-1

void HSLtoRGB(float H, float S, float L, float &R, float &G, float &B)
{
   float    C, X, M;
   float    h1, h2;
   
   h1 = H / 60;
   h2 = h1 - 2 * int(h1/2);
   
   C = (1 - fabsf(2*L-1)) * S;
   X = C * (1 - fabsf(h2-1));
   M = L - C/2;
   
   if (H < 60) {
      R = C;
      G = X;
      B = 0;
   }
   
   else if (H < 120) {
      R = X;
      G = C;
      B = 0;
   }
   
   else if (H < 180) {
      R = 0;
      G = C;
      B = X;
   }
   
   else if (H < 240) {
      R = 0;
      G = X;
      B = C;
   }
   
   else if (H < 300) {
      R = X;
      G = 0;
      B = C;
   }
   
   else {
      R = C;
      G = 0;
      B = X;
   }
   
   R = R + M;
   G = G + M;
   B = B + M;
   
   if (R < 0) R = 0;                                                             //  21.32
   if (G < 0) G = 0;
   if (B < 0) B = 0;
   
   if (R > 255.0) R = 255.0;                                                     //  21.32
   if (G > 255.0) G = 255.0;
   if (B > 255.0) B = 255.0;
   
   return;
}


//  RGB to HSL converter 
//  input RGB values 0-1
//  outputs: H = 0-360 deg.  S = 0-1   L = 0-1

void RGBtoHSL(float R, float G, float B, float &H, float &S, float &L)
{
   float    max, min, D;

   max = R;
   if (G > max) max = G;
   if (B > max) max = B;
   
   min = R;
   if (G < min) min = G;
   if (B < min) min = B;
   
   D = max - min;
   
   L = 0.5 * (max + min);
   
   if (D < 0.004) {
      H = S = 0;
      return;
   }
   
   if (L > 0.5)
      S = D / (2 - max - min);
   else
      S = D / (max + min);
   
   if (max == R) 
      H = (G - B) / D;
   else if (max == G)
      H = 2 + (B - R) / D;
   else
      H = 4 + (R - G) / D;

   H = H * 60;
   if (H < 0) H += 360;

   return;
}


/********************************************************************************/

//  match_colors edit function
//  Adjust colors of image 2 to match the colors of image 1
//  using small selected areas in each image as the match standard.

namespace match_colors_names
{
   float    match_colors_RGB1[3];                                                //  image 1 base colors to match
   float    match_colors_RGB2[3];                                                //  image 2 target colors to match
   int      match_colors_radius = 10;                                            //  mouse radius
   int      match_colors_mode = 0;
   int      E3ww, E3hh;

   editfunc    EFmatchcolors;
}


//  menu function

void m_match_colors(GtkWidget *, const char *menu)
{
   using namespace match_colors_names;

   int    match_colors_dialog_event(zdialog* zd, const char *event);
   void * match_colors_thread(void *);
   void   match_colors_mousefunc();

   cchar    *title = "Color Match Images";

   F1_help_topic = "match colors";

   if (Fblock("blocked edits")) return;                                          //  check, no block
                                                                                 //  (edit_setup() follows)
/***
          ____________________________________________
         |       Color Match Images                   |
         |                                            |
         | 1  [ 10 ]   mouse radius for color sample  |
         | 2  [Open]   image for source color         |
         | 3  click on image to get source color      |
         | 4  [Open]   image for target color         |
         | 5  click on image to set target color      |
         |                                            |
         |                            [ OK ] [cancel] |
         |____________________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,BOK,Bcancel,null);                       //  match_colors dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"label","labn1","vb1","1");
   zdialog_add_widget(zd,"label","labn2","vb1","2");
   zdialog_add_widget(zd,"label","labn3","vb1","3");
   zdialog_add_widget(zd,"label","labn4","vb1","4");
   zdialog_add_widget(zd,"label","labn5","vb1","5");
   zdialog_add_widget(zd,"hbox","hbrad","vb2");
   zdialog_add_widget(zd,"zspin","radius","hbrad","1|20|1|10","space=5");
   zdialog_add_widget(zd,"label","labrad","hbrad","mouse radius for color sample");
   zdialog_add_widget(zd,"hbox","hbop1","vb2");
   zdialog_add_widget(zd,"button","open1","hbop1","Open","space=5");
   zdialog_add_widget(zd,"label","labop1","hbop1","image for source color");
   zdialog_add_widget(zd,"hbox","hbclik1","vb2");
   zdialog_add_widget(zd,"label","labclik1","hbclik1","click on image to get source color");
   zdialog_add_widget(zd,"hbox","hbop2","vb2");
   zdialog_add_widget(zd,"button","open2","hbop2","Open","space=5");
   zdialog_add_widget(zd,"label","labop2","hbop2","image to set matching color");
   zdialog_add_widget(zd,"hbox","hbclik2","vb2");
   zdialog_add_widget(zd,"label","labclik2","hbclik2","click on image to set matching color");

   zdialog_stuff(zd,"radius",match_colors_radius);                               //  remember last radius

   EFmatchcolors.menuname = "Match Colors";
   EFmatchcolors.Farea = 1;                                                      //  select area ignored
   EFmatchcolors.zd = zd;
   EFmatchcolors.threadfunc = match_colors_thread;
   EFmatchcolors.mousefunc = match_colors_mousefunc;

   match_colors_mode = 0;
   if (curr_file) {
      match_colors_mode = 1;                                                     //  image 1 ready to click
      takeMouse(match_colors_mousefunc,0);                                       //  connect mouse function
   }

   zdialog_run(zd,match_colors_dialog_event,"parent");                           //  run dialog - parallel
   return;
}


//  match_colors dialog event and completion function

int match_colors_dialog_event(zdialog *zd, const char *event)
{
   using namespace match_colors_names;

   void   match_colors_mousefunc();

   int      err;
   char     *file;

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()
   
   if (zd->zstat)
   {
      if (match_colors_mode == 4) {                                              //  edit was started
         if (zd->zstat == 1) edit_done(0);                                       //  commit edit
         else edit_cancel(0);                                                    //  discard edit
         match_colors_mode = 0;
         return 1;
      }

      freeMouse();                                                               //  abandoned
      zdialog_free(zd);
      match_colors_mode = 0;
      return 1;
   }

   if (strmatch(event,"radius"))                                                 //  set new mouse radius
      zdialog_fetch(zd,"radius",match_colors_radius);

   if (strmatch(event,"open1"))                                                  //  get image 1 for color source
   {
      if (match_colors_mode == 4) edit_cancel(1);                                //  cancel edit, keep dialog
      match_colors_mode = 0;
      file = gallery_select1(null);                                              //  open image 1
      if (file) {
         err = f_open(file);
         if (! err) match_colors_mode = 1;                                       //  image 1 ready to click
      }
   }

   if (strmatch(event,"open2"))                                                  //  get image 2 to set matching color
   {
      if (match_colors_mode < 2) {
         zmessageACK(Mwin,"select source image color first");                    //  check that RGB1 has been set
         return 1;
      }
      match_colors_mode = 2;
      file = gallery_select1(null);                                              //  open image 2
      if (! file) return 1;
      err = f_open(file);
      if (err) return 1;
      match_colors_mode = 3;                                                     //  image 2 ready to click
   }

   takeMouse(match_colors_mousefunc,0);                                          //  reconnect mouse function
   return 1;
}


//  mouse function - click on image and get colors to match

void match_colors_mousefunc()
{
   using namespace match_colors_names;

   void  match_colors_getRGB(int px, int py, float rgb[3]);

   int      px, py;

   if (match_colors_mode < 1) return;                                            //  no image available yet

   draw_mousecircle(Mxposn,Myposn,match_colors_radius,0,0);                      //  draw circle around pointer

   if (LMclick)
   {
      LMclick = 0;
      px = Mxclick;
      py = Myclick;

      if (match_colors_mode == 1 || match_colors_mode == 2)                      //  image 1 ready to click
      {
         match_colors_getRGB(px,py,match_colors_RGB1);                           //  get RGB1 color
         match_colors_mode = 2;
         return;
      }

      if (match_colors_mode == 3 || match_colors_mode == 4)                      //  image 2 ready to click
      {
         if (match_colors_mode == 4) edit_reset();
         else {
            if (! edit_setup(EFmatchcolors)) return;                             //  setup edit - thread will launch
            E3ww = E3pxm->ww;
            E3hh = E3pxm->hh;
            match_colors_mode = 4;                                               //  edit waiting for cancel or done
         }

         match_colors_getRGB(px,py,match_colors_RGB2);                           //  get RGB2 color
         signal_thread();                                                        //  update the target image
         return;
      }
   }

   return;
}


//  get the RGB averages for pixels within mouse radius

void match_colors_getRGB(int px, int py, float rgb[3])
{
   using namespace match_colors_names;

   int      radflat1 = match_colors_radius;
   int      radflat2 = radflat1 * radflat1;
   int      rad, npix, qx, qy;
   float    red, green, blue;
   float    *pix1;
   PXM      *pxm;

   pxm = PXM_load(curr_file,1);                                                  //  popup ACK if error
   if (! pxm) return;

   npix = 0;
   red = green = blue = 0;

   for (qy = py-radflat1; qy <= py+radflat1; qy++)
   for (qx = px-radflat1; qx <= px+radflat1; qx++)
   {
      if (qx < 0 || qx > pxm->ww-1) continue;
      if (qy < 0 || qy > pxm->hh-1) continue;
      rad = (qx-px) * (qx-px) + (qy-py) * (qy-py);
      if (rad > radflat2) continue;
      pix1 = PXMpix(pxm,qx,qy);
      red += pix1[0];
      green += pix1[1];
      blue += pix1[2];
      npix++;
   }

   rgb[0] = red / npix;
   rgb[1] = green / npix;
   rgb[2] = blue / npix;

   PXM_free(pxm);
   return;
}


//  thread function - start multiple working threads

void * match_colors_thread(void *)
{
   using namespace match_colors_names;

   void * match_colors_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

      do_wthreads(match_colors_wthread,NWT);                                     //  worker threads

      CEF->Fmods++;
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


void * match_colors_wthread(void *arg)                                           //  worker thread function
{
   using namespace match_colors_names;

   int         index = *((int *) (arg));
   int         px, py;
   float       *pix3;
   float       Rred, Rgreen, Rblue;
   float       red, green, blue;
   float       max$;

   Rred = match_colors_RGB1[0] / match_colors_RGB2[0];                           //  color adjustment ratios
   Rgreen = match_colors_RGB1[1] / match_colors_RGB2[1];
   Rblue = match_colors_RGB1[2] / match_colors_RGB2[2];

   for (py = index; py < E3hh; py += NWT)                                        //  loop all image pixels
   for (px = 0; px < E3ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);

      red = pix3[0] * Rred;                                                      //  adjust colors
      green = pix3[1] * Rgreen;
      blue = pix3[2] * Rblue;

      RGBFIX(red,green,blue)                                                     //  21.0

      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }

   return 0;
}


/********************************************************************************/

//  add a brightness/color curved ramp to an image, in a chosen direction

namespace brite_ramp_names 
{
   editfunc    EFbrite_ramp;
   int         Fline, linex1, liney1, linex2, liney2;
   float       A, B, C;
   float       ex1, ey1, ex2, ey2;
   int         E3ww, E3hh;
}


//  menu function

void m_brite_ramp(GtkWidget *, cchar *menu) 
{
   using namespace brite_ramp_names;

   void   brite_ramp_curvedit(int spc);
   int    brite_ramp_dialog_event(zdialog* zd, cchar *event);
   void * brite_ramp_thread(void *);
   void   brite_ramp_mousefunc();

   cchar    *mess = "Draw a line across the image in \n"
                    "direction of brightness change.";

   F1_help_topic = "brite ramp";

   EFbrite_ramp.menuname = "Brite Ramp";
   EFbrite_ramp.menufunc = m_brite_ramp;
   EFbrite_ramp.FprevReq = 1;                                                    //  use preview
   EFbrite_ramp.Fscript = 1;                                                     //  scripting supported
   EFbrite_ramp.Farea = 2;                                                       //  select area usable
   EFbrite_ramp.threadfunc = brite_ramp_thread;
   EFbrite_ramp.mousefunc = brite_ramp_mousefunc;

   if (! edit_setup(EFbrite_ramp)) return;                                       //  setup edit

   E3ww = E3pxm->ww;
   E3hh = E3pxm->hh;

   Fline = 0;                                                                    //  no drawn line initially

/***
          _______________________________________________
         |              Brightness Ramp                  |
         |                                               |
         |    Draw a line across the image in            |
         |    direction of brightness change.            |
         |  ___________________________________________  |
         | |                                           | |                       //  5 curves are maintained:
         | |                                           | |                       //  curve 0: current display curve
         | |                                           | |                       //        1: curve for all colors
         | |         curve edit area                   | |                       //        2,3,4: red, green, blue
         | |                                           | |
         | |                                           | |
         | |                                           | |
         | |___________________________________________| |
         |   (o) all  (o) red  (o) green  (o) blue       |                       //  select curve to display
         |                                               |
         |                      [reset] [ OK ] [Cancel]  |
         |_______________________________________________|

***/

   zdialog *zd = zdialog_new("Brightness Ramp",Mwin,Breset,BOK,Bcancel,null);
   EFbrite_ramp.zd = zd;
   zdialog_add_widget(zd,"label","labmess","dialog",mess);
   zdialog_add_widget(zd,"frame","frameH","dialog",0,"expand");                  //  edit-curves
   zdialog_add_widget(zd,"hbox","hbrgb","dialog");                               //  radio buttons all/red/green/blue
   zdialog_add_widget(zd,"radio","all","hbrgb",Ball,"space=5");
   zdialog_add_widget(zd,"radio","red","hbrgb",Bred,"space=3");
   zdialog_add_widget(zd,"radio","green","hbrgb",Bgreen,"space=3");
   zdialog_add_widget(zd,"radio","blue","hbrgb",Bblue,"space=3");

   GtkWidget *frameH = zdialog_gtkwidget(zd,"frameH");                           //  setup edit curves
   spldat *sd = splcurve_init(frameH,brite_ramp_curvedit);
   EFbrite_ramp.sd = sd;

   sd->Nscale = 1;                                                               //  horizontal fixed line, neutral curve
   sd->xscale[0][0] = 0.01;
   sd->yscale[0][0] = 0.50;
   sd->xscale[1][0] = 0.99;
   sd->yscale[1][0] = 0.50;

   for (int ii = 0; ii < 4; ii++)                                                //  loop curves 0-3
   {
      sd->nap[ii] = 3;                                                           //  initial curves are neutral
      sd->vert[ii] = 0;
      sd->fact[ii] = 0;
      sd->apx[ii][0] = 0.01;
      sd->apx[ii][1] = 0.50;                                                     //  curve 0 = overall brightness
      sd->apx[ii][2] = 0.99;                                                     //  curve 1/2/3 = R/G/B adjustment
      sd->apy[ii][0] = 0.5;
      sd->apy[ii][1] = 0.5;
      sd->apy[ii][2] = 0.5;
      splcurve_generate(sd,ii);
   }

   sd->Nspc = 4;                                                                 //  4 curves
   sd->fact[0] = 1;                                                              //  curve 0 active
   zdialog_stuff(zd,"all",1);                                                    //  stuff default selection, all

   zdialog_resize(zd,200,200);
   zdialog_run(zd,brite_ramp_dialog_event,"save");                               //  run dialog - parallel

   takeMouse(brite_ramp_mousefunc,dragcursor);                                   //  connect mouse
   return;
}


//  dialog event and completion callback function

int brite_ramp_dialog_event(zdialog *zd, cchar *event)
{
   using namespace brite_ramp_names;

   void   brite_ramp_mousefunc();

   int      ii, Fupdate = 0;

   spldat *sd = EFbrite_ramp.sd;    

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 2;                                    //  apply and quit
   if (strmatch(event,"cancel")) zd->zstat = 3;                                  //  cancel
   if (strmatch(event,"apply")) Fupdate++;                                       //  from script
   
   if (strmatch(event,"focus")) 
      takeMouse(brite_ramp_mousefunc,dragcursor);                                //  connect mouse
   
   if (strmatch(event,"fullsize")) {                                             //  from select area
      edit_fullsize();
      E3ww = E3pxm->ww;
      E3hh = E3pxm->hh;
      signal_thread();
      return 1;
   }

   if (zd->zstat)
   {
      if (zd->zstat == 1)                                                        //  reset
      {
         for (int ii = 0; ii < 4; ii++) {                                        //  loop curves 0-3
            sd->nap[ii] = 3;                                                     //  all curves are neutral
            sd->vert[ii] = 0;
            sd->fact[ii] = 0;
            sd->apx[ii][0] = 0.01;
            sd->apx[ii][1] = 0.50;
            sd->apx[ii][2] = 0.99;
            sd->apy[ii][0] = 0.5;
            sd->apy[ii][1] = 0.5;
            sd->apy[ii][2] = 0.5;
            splcurve_generate(sd,ii);
            sd->fact[ii] = 0;
         }

         sd->fact[0] = 1;
         gtk_widget_queue_draw(sd->drawarea);                                    //  draw curve 0
         zdialog_stuff(zd,"all",1);
         zdialog_stuff(zd,"red",0); 
         zdialog_stuff(zd,"green",0);
         zdialog_stuff(zd,"blue",0);
         edit_reset();                                                           //  restore initial image
         zd->zstat = 0;
         return 1;
      }
         
      if (zd->zstat == 2) {                                                      //  done
         edit_fullsize();                                                        //  get full size image
         E3ww = E3pxm->ww;
         E3hh = E3pxm->hh;
         signal_thread();
         edit_done(0);                                                           //  commit edit
      }
      else edit_cancel(0);                                                       //  discard edit

      Ntoplines = 0;
      Fpaint2();
      return 1;
   }

   if (zstrstr("all red green blue",event))                                      //  new choice of curve
   {
      zdialog_fetch(zd,event,ii);
      if (! ii) return 0;                                                        //  button OFF event, wait for ON event

      for (ii = 0; ii < 4; ii++)
         sd->fact[ii] = 0;
      ii = strmatchV(event,"all","red","green","blue",null);
      ii = ii-1;                                                                 //  new active curve: 0, 1, 2, 3
      sd->fact[ii] = 1;

      splcurve_generate(sd,ii);                                                  //  regenerate curve
      gtk_widget_queue_draw(sd->drawarea);                                       //  draw curve

      Fupdate = 1;
   }

   if (strmatch(event,"blendwidth")) Fupdate = 1;

   if (Fupdate) signal_thread();                                                 //  trigger image update

   return 1;
}


//  brite_ramp mouse function

void brite_ramp_mousefunc()
{
   using namespace brite_ramp_names;
   
   int      mx = 0, my = 0;
   float    d1, d2;
   
   if (! (LMclick || RMclick || Mxdrag || Mydrag)) return;                       //  ignore mouse movement
   
   if (LMclick || RMclick) {                                                     //  left or right mouse click
      mx = Mxclick;
      my = Myclick;
      LMclick = RMclick = 0;
   }

   if (Mxdrag || Mydrag) {                                                       //  mouse drag
      mx = Mxdrag;
      my = Mydrag;
      Mxdrag = Mydrag = 0;
   }
   
   if (! Fline && (mx || my))
   {
      Fline = 1;
      linex1 = mx;                                                               //  draw arbitrary line to start with
      liney1 = my;
      linex2 = mx + 100;
      liney2 = my + 100;
   }
   
   else                                                                          //  move nearest line end point to mouse
   {
      d1 = (linex1 - mx) * (linex1 - mx) + (liney1 - my) * (liney1 - my);
      d2 = (linex2 - mx) * (linex2 - mx) + (liney2 - my) * (liney2 - my);

      if (d1 < d2) {
         linex1 = mx;
         liney1 = my;
      }
      else {
         linex2 = mx;
         liney2 = my;
      }
   }
   
   Ntoplines = 1;                                                                //  update line data
   toplines[0].x1 = linex1;
   toplines[0].y1 = liney1;
   toplines[0].x2 = linex2;
   toplines[0].y2 = liney2;
   toplines[0].type = 3;                                                         //  black/white dual line                 21.0

   signal_thread();                                                              //  update image
   return;
}


//  this function is called when a curve is edited

void brite_ramp_curvedit(int spc)
{
   signal_thread();
   return;
}


//  brite_ramp thread function

void * brite_ramp_thread(void *arg)
{
   using namespace brite_ramp_names;

   void brite_ramp_equation();
   void brite_ramp_rampline();
   void * brite_ramp_wthread(void *);

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window paint

      ex1 = linex1;                                                              //  ramp line end points
      ey1 = liney1;
      ex2 = linex2;
      ey2 = liney2;

      brite_ramp_equation();                                                     //  compute line equation
      brite_ramp_rampline();                                                     //  compute new end points

      do_wthreads(brite_ramp_wthread,NWT);                                       //  worker threads

      CEF->Fmods++;                                                              //  image3 modified
      CEF->Fsaved = 0;

      paintlock(0);                                                              //  unblock window paint
      Fpaint2();                                                                 //  update window
   }

   return 0;
}


void * brite_ramp_wthread(void *arg)                                             //  worker thread function
{
   using namespace brite_ramp_names;

   void brite_ramp_posn(int px, int py, float &rx, float &ry);

   int         index = *((int *) arg);
   int         ii, dist = 0, px3, py3;
   float       x3, y3;
   float       d1, d2, rampval;
   float       *pix1, *pix3;
   float       red1, green1, blue1;
   float       red3, green3, blue3;
   float       Fall, Fred, Fgreen, Fblue;
   float       dold, dnew;
   float       max$;

   spldat *sd = EFbrite_ramp.sd;    

   for (py3 = index; py3 < E3hh; py3 += NWT)                                     //  loop output pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      if (sa_stat == 3) {                                                        //  select area active
         ii = py3 * E3ww + px3;
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel outside area
      }

      brite_ramp_posn(px3,py3,x3,y3);                                            //  nearest point on ramp line

      d1 = sqrtf((x3-ex1) * (x3-ex1) + (y3-ey1) * (y3-ey1));                     //  compute ramp value
      d2 = sqrtf((x3-ex2) * (x3-ex2) + (y3-ey2) * (y3-ey2));
      rampval = d1 / (d1 + d2);                                                  //  0.0 ... 1.0

      ii = 999.0 * rampval;                                                      //  corresp. curve index 0-999

      Fall = sd->yval[0][ii] * 2.0;                                              //  curve values 0.0 - 1.0
      Fred = sd->yval[1][ii] * 2.0;                                              //  (0.5 is neutral value)
      Fgreen = sd->yval[2][ii] * 2.0;
      Fblue = sd->yval[3][ii] * 2.0;
      
      pix1 = PXMpix(E1pxm,px3,py3);                                              //  input pixel
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = red1 * Fall;                                                        //  curve "all" adjustment
      green3 = green1 * Fall;                                                    //    projected on each RGB color
      blue3 = blue1 * Fall;

      red3 = red3 * Fred;                                                        //  add additional RGB adjustments
      green3 = green3 * Fgreen;
      blue3 = blue3 * Fblue;

      RGBFIX(red3,green3,blue3)                                                  //  21.0

      if (sa_stat == 3 && dist < sa_blendwidth) {                                //  blend changes over blendwidth
         dnew = sa_blendfunc(dist);
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3 = PXMpix(E3pxm,px3,py3);                                              //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   return 0;
}


//  get equation of ramp line in the form  Ax + By + C = 0
//  end points are (ex1,ey1) and (ex2,ey2)

void brite_ramp_equation()
{
   using namespace brite_ramp_names;

   if (ex1 != ex2)
   {
      A = (ey2 - ey1) / (ex2 - ex1);
      B = -1;
      C = ey1 - A * ex1;
   }
   else {
      A = 1;
      B = 0;
      C = -ex1;
   }
   return;
}


//  compute nearest point on ramp line for given image pixel position

void brite_ramp_posn(int px, int py, float &rx, float &ry)
{
   using namespace brite_ramp_names;
   
   float    F1, F2;

   F1 = B * px - A * py;
   F2 = A * A + B * B;
   rx = (B * F1 - A * C) / F2;
   ry = (-A * F1 - B * C) / F2;

   return;
}


//  extend ramp line end points long enough for entire image

void brite_ramp_rampline()
{
   using namespace brite_ramp_names;

   void brite_ramp_posn(int px, int py, float &rx, float &ry);

   float    rx, ry, d1, d2;
   
   if (B == 0) {                                                                 //  vertical line
      ey1 = 0;
      ey2 = E3hh - 1;
      return;
   }

   if (A == 0) {                                                                 //  horizontal line
      ex1 = 0;
      ex2 = E3ww - 1;
      return;
   }

   brite_ramp_posn(0,0,rx,ry);
   if (rx < 0 || ry < 0) {
      d1 = (rx - ex1) * (rx - ex1) + (ry - ey1) * (ry - ey1);
      d2 = (rx - ex2) * (rx - ex2) + (ry - ey2) * (ry - ey2);
      if (d1 < d2) {
         ex1 = rx;
         ey1 = ry;
      }
      else {
         ex2 = rx;
         ey2 = ry;
      }
   }      

   brite_ramp_posn(E3ww,0,rx,ry);
   if (rx > E3ww || ry < 0) {
      d1 = (rx - ex1) * (rx - ex1) + (ry - ey1) * (ry - ey1);
      d2 = (rx - ex2) * (rx - ex2) + (ry - ey2) * (ry - ey2);
      if (d1 < d2) {
         ex1 = rx;
         ey1 = ry;
      }
      else {
         ex2 = rx;
         ey2 = ry;
      }
   }      

   brite_ramp_posn(E3ww,E3hh,rx,ry);
   if (rx > E3ww || ry > E3hh) {
      d1 = (rx - ex1) * (rx - ex1) + (ry - ey1) * (ry - ey1);
      d2 = (rx - ex2) * (rx - ex2) + (ry - ey2) * (ry - ey2);
      if (d1 < d2) {
         ex1 = rx;
         ey1 = ry;
      }
      else {
         ex2 = rx;
         ey2 = ry;
      }
   }      

   brite_ramp_posn(0,E3hh,rx,ry);
   if (rx < 0 || ry > E3hh) {
      d1 = (rx - ex1) * (rx - ex1) + (ry - ey1) * (ry - ey1);
      d2 = (rx - ex2) * (rx - ex2) + (ry - ey2) * (ry - ey2);
      if (d1 < d2) {
         ex1 = rx;
         ey1 = ry;
      }
      else {
         ex2 = rx;
         ey2 = ry;
      }
   }      

   return;
}


/********************************************************************************/

//  find and remove "dust" from an image (e.g. from a scanned dusty slide)
//  dust is defined as small dark areas surrounded by brighter areas
//  image 1   original with prior edits
//  image 3   accumulated dust removals that have been committed
//  image 9   committed dust removals + pending removal (work in process)

namespace dust_names
{
   editfunc    EFdust;

   int         spotspan;                                                         //  max. dustspot span, pixels
   int         spotspan2;                                                        //  spotspan **2
   float       brightness;                                                       //  brightness limit, 0 to 1 = white
   float       contrast;                                                         //  min. contrast, 0 to 1 = black/white
   int         *pixgroup;                                                        //  maps (px,py) to pixel group no.
   int         Fred;                                                             //  red pixels are on

   int         Nstack;

   struct spixstack {
      uint16      px, py;                                                        //  pixel group search stack
      uint16      direc;
   }  *pixstack;

   #define maxgroups 1000000
   int         Ngroups;
   int         groupcount[maxgroups];                                            //  count of pixels in each group
   float       groupbright[maxgroups];
   int         edgecount[maxgroups];                                             //  group edge pixel count
   float       edgebright[maxgroups];                                            //  group edge pixel brightness sum

   typedef struct {
      uint16      px1, py1, px2, py2;                                            //  pixel group extreme pixels
      int         span2;                                                         //  span from px1/py1 to px2/py2
   }  sgroupspan;

   sgroupspan    groupspan[maxgroups];
}


//  menu function

void m_remove_dust(GtkWidget *, const char *menu)
{
   using namespace dust_names;

   int    dust_dialog_event(zdialog *zd, cchar *event);
   void * dust_thread(void *);

   F1_help_topic = "remove dust";
   m_viewmode(0,"F");                                                            //  file view mode 

   EFdust.menufunc = m_remove_dust;
   EFdust.menuname = "Remove Dust";
   EFdust.Farea = 2;                                                             //  select area usable
   EFdust.Frestart = 1;                                                          //  restart allowed
   EFdust.threadfunc = dust_thread;                                              //  thread function
   if (! edit_setup(EFdust)) return;                                             //  setup edit

   E9pxm = PXM_copy(E3pxm);                                                      //  image 9 = copy of image3
   Fred = 0;

   int cc = E1pxm->ww * E1pxm->hh * sizeof(int);
   pixgroup = (int *) zmalloc(cc);                                               //  maps pixels to assigned groups

   cc = E1pxm->ww * E1pxm->hh * sizeof(spixstack);
   pixstack = (spixstack *) zmalloc(cc);                                         //  pixel group search stack

/***
       ____________________________________________
      |                Remove Dust                 |
      |                                            |
      | spot size limit    =========[]===========  |
      | max. brightness    =============[]=======  |
      | min. contrast      ========[]============  |
      | [erase] [red] [undo last] [apply]          |
      |                                            |
      |                            [ OK ] [Cancel] |
      |____________________________________________|

***/

   zdialog *zd = zdialog_new("Remove Dust",Mwin,BOK,Bcancel,null);
   EFdust.zd = zd;

   zdialog_add_widget(zd,"hbox","hbssl","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labssl","hbssl","spot size limit","space=5");
   zdialog_add_widget(zd,"hscale","spotspan","hbssl","1|50|1|20","space=5|expand");
   zdialog_add_widget(zd,"hbox","hbmb","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labmb","hbmb","max. brightness","space=5");
   zdialog_add_widget(zd,"hscale","brightness","hbmb","1|999|1|700","space=5|expand");
   zdialog_add_widget(zd,"hbox","hbmc","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labmb","hbmc","min. contrast","space=5");
   zdialog_add_widget(zd,"hscale","contrast","hbmc","1|500|1|40","space=5|expand");
   zdialog_add_widget(zd,"hbox","hbbutts","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","erase","hbbutts",Berase,"space=5");
   zdialog_add_widget(zd,"button","red","hbbutts",Bred,"space=5");
   zdialog_add_widget(zd,"button","undo1","hbbutts",Bundolast,"space=5");
   zdialog_add_widget(zd,"button","apply","hbbutts",Bapply,"space=5");

   zdialog_resize(zd,300,0);
   zdialog_restore_inputs(zd);                                                   //  preload prior user inputs

   zdialog_fetch(zd,"spotspan",spotspan);                                        //  max. dustspot span (pixels)
   spotspan2 = spotspan * spotspan;

   zdialog_fetch(zd,"brightness",brightness);                                    //  max. dustspot brightness
   brightness = 0.001 * brightness;                                              //  scale 0 to 1 = white

   zdialog_fetch(zd,"contrast",contrast);                                        //  min. dustspot contrast
   contrast = 0.001 * contrast;                                                  //  scale 0 to 1 = black/white

   zdialog_run(zd,dust_dialog_event,"save");                                     //  run dialog - parallel

   signal_thread();
   return;
}


//  dialog event and completion callback function

int dust_dialog_event(zdialog *zd, cchar *event)
{
   using namespace dust_names;

   void dust_erase();

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  done
         wrapup_thread(8);                                                       //  thread finish, exit
         PXM_free(E3pxm);
         E3pxm = E9pxm;                                                          //  image 3 = image 9
         E9pxm = 0;
         edit_done(0);                                                           //  commit edit
      }
      else {                                                                     //  cancel
         wrapup_thread(8);                                                       //  thread finish, exit
         PXM_free(E9pxm);
         edit_cancel(0);                                                         //  discard edit
      }
      zfree(pixgroup);                                                           //  free memory
      zfree(pixstack);
      return 1;
   }

   if (zstrstr("spotspan brightness contrast red",event))
   {
      zdialog_fetch(zd,"spotspan",spotspan);                                     //  max. dustspot span (pixels)
      spotspan2 = spotspan * spotspan;

      zdialog_fetch(zd,"brightness",brightness);                                 //  max. dustspot brightness
      brightness = 0.001 * brightness;                                           //  scale 0 to 1 = white

      zdialog_fetch(zd,"contrast",contrast);                                     //  min. dustspot contrast
      contrast = 0.001 * contrast;                                               //  scale 0 to 1 = black/white

      signal_thread();                                                           //  do the work
   }

   if (strmatch(event,"erase")) dust_erase();
   if (strmatch(event,"blendwidth")) dust_erase();

   if (strmatch(event,"undo1")) {
      PXM_free(E3pxm);
      E3pxm = PXM_copy(E9pxm);
      Fred = 0;
      Fpaint2();
   }

   if (strmatch(event,"apply")) {                                                //  button
      if (Fred) dust_erase();
      PXM_free(E9pxm);                                                           //  image 9 = copy of image 3
      E9pxm = PXM_copy(E3pxm);
      CEF->Fmods++;
      CEF->Fsaved = 0;
   }

   return 1;
}


//  dust find thread function - find the dust particles and mark them

void * dust_thread(void *)
{
   using namespace dust_names;

   int         xspan, yspan, span2;
   int         group, cc, ii, kk, Nremoved;
   int         px, py, dx, dy, ppx, ppy, npx, npy;
   float       gbright, pbright, pcontrast;
   float       ff = 1.0 / 256.0;
   uint16      direc;
   float       *pix3;

   while (true)
   {
      thread_idle_loop();                                                        //  wait for work or exit request

      paintlock(1);                                                              //  block window updates

      PXM_free(E3pxm);
      E3pxm = PXM_copy(E9pxm);

      paintlock(0);                                                              //  unblock window updates

      cc = E1pxm->ww * E1pxm->hh * sizeof(int);                                  //  clear group arrays
      memset(pixgroup,0,cc);
      cc = maxgroups * sizeof(int);
      memset(groupcount,0,cc);
      memset(edgecount,0,cc);
      cc = maxgroups * sizeof(float );
      memset(groupbright,0,cc);
      memset(edgebright,0,cc);
      cc = maxgroups * sizeof(sgroupspan);
      memset(groupspan,0,cc);

      group = 0;

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;
         if (sa_stat == 3 && ! sa_pixmap[ii]) continue;                          //  not in active area
         if (pixgroup[ii]) continue;                                             //  already assigned to a group

         pix3 = PXMpix(E3pxm,px,py);                                             //  get pixel brightness
         gbright = ff * PIXBRIGHT(pix3);                                         //  0 to 1.0 = white
         if (gbright > brightness) continue;                                     //  ignore bright pixel

         if (group == maxgroups-1) break;                                        //  too many groups, make no more

         pixgroup[ii] = ++group;                                                 //  assign next group
         groupcount[group] = 1;
         groupbright[group] = gbright;

         pixstack[0].px = px;                                                    //  put pixel into stack with
         pixstack[0].py = py;                                                    //    direction = ahead
         pixstack[0].direc = 0;
         Nstack = 1;

         while (Nstack)
         {
            kk = Nstack - 1;                                                     //  get last pixel in stack
            px = pixstack[kk].px;
            py = pixstack[kk].py;
            direc = pixstack[kk].direc;                                          //  next search direction

            if (direc == 'x') {
               Nstack--;                                                         //  none left
               continue;
            }

            if (Nstack > 1) {
               ii = Nstack - 2;                                                  //  get prior pixel in stack
               ppx = pixstack[ii].px;
               ppy = pixstack[ii].py;
            }
            else {
               ppx = px - 1;                                                     //  if only one, assume prior = left
               ppy = py;
            }

            dx = px - ppx;                                                       //  vector from prior to this pixel
            dy = py - ppy;

            switch (direc)
            {
               case 0:
                  npx = px + dx;
                  npy = py + dy;
                  pixstack[kk].direc = 1;
                  break;

               case 1:
                  npx = px + dy;
                  npy = py + dx;
                  pixstack[kk].direc = 3;
                  break;

               case 2:
                  npx = px - dx;                                                 //  back to prior pixel
                  npy = py - dy;                                                 //  (this path never taken)
                  zappcrash("stack search bug");
                  break;

               case 3:
                  npx = px - dy;
                  npy = py - dx;
                  pixstack[kk].direc = 4;
                  break;

               case 4:
                  npx = px - dx;
                  npy = py + dy;
                  pixstack[kk].direc = 5;
                  break;

               case 5:
                  npx = px - dy;
                  npy = py + dx;
                  pixstack[kk].direc = 6;
                  break;

               case 6:
                  npx = px + dx;
                  npy = py - dy;
                  pixstack[kk].direc = 7;
                  break;

               case 7:
                  npx = px + dy;
                  npy = py - dx;
                  pixstack[kk].direc = 'x';
                  break;

               default:
                  npx = npy = 0;
                  zappcrash("stack search bug");
            }

            if (npx < 0 || npx > E1pxm->ww-1) continue;                          //  pixel off the edge
            if (npy < 0 || npy > E1pxm->hh-1) continue;

            ii = npy * E1pxm->ww + npx;
            if (pixgroup[ii]) continue;                                          //  pixel already assigned
            if (sa_stat == 3 && ! sa_pixmap[ii]) continue;                       //  pixel outside area

            pix3 = PXMpix(E3pxm,npx,npy);                                        //  pixel brightness
            pbright = ff * PIXBRIGHT(pix3);
            if (pbright > brightness) continue;                                  //  brighter than limit

            pixgroup[ii] = group;                                                //  assign pixel to group
            ++groupcount[group];                                                 //  count pixels in group
            groupbright[group] += pbright;                                       //  sum brightness for group

            kk = Nstack++;                                                       //  put pixel into stack
            pixstack[kk].px = npx;
            pixstack[kk].py = npy;
            pixstack[kk].direc = 0;                                              //  search direction
         }
      }

      Ngroups = group;                                                           //  group numbers are 1-Ngroups
      Nremoved = 0;

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;
         group = pixgroup[ii];
         if (! group) continue;
         if (groupspan[group].px1 == 0) {                                        //  first pixel found in this group
            groupspan[group].px1 = px;                                           //  group px1/py1 = this pixel
            groupspan[group].py1 = py;
            continue;
         }
         xspan = groupspan[group].px1 - px;                                      //  span from group px1/py1 to this pixel
         yspan = groupspan[group].py1 - py;
         span2 = xspan * xspan + yspan * yspan;
         if (span2 > groupspan[group].span2) {
            groupspan[group].span2 = span2;                                      //  if greater, group px2/py2 = this pixel
            groupspan[group].px2 = px;
            groupspan[group].py2 = py;
         }
      }

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;
         group = pixgroup[ii];
         if (! group) continue;
         if (groupspan[group].span2 > spotspan2) continue;
         xspan = groupspan[group].px2 - px;                                      //  span from this pixel to group px2/py2
         yspan = groupspan[group].py2 - py;
         span2 = xspan * xspan + yspan * yspan;
         if (span2 > groupspan[group].span2) {
            groupspan[group].span2 = span2;                                      //  if greater, group px1/py1 = this pixel
            groupspan[group].px1 = px;
            groupspan[group].py1 = py;
         }
      }

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;                                               //  eliminate group if span > limit
         group = pixgroup[ii];
         if (! group) continue;
         if (! groupcount[group]) pixgroup[ii] = 0;
         else if (groupspan[group].span2 > spotspan2) {
            pixgroup[ii] = 0;
            groupcount[group] = 0;
            Nremoved++;
         }
      }

      for (py = 1; py < E1pxm->hh-1; py++)                                       //  loop all pixels except image edges
      for (px = 1; px < E1pxm->ww-1; px++)
      {
         ii = py * E1pxm->ww + px;
         group = pixgroup[ii];
         if (group) continue;                                                    //  find pixels bordering group pixels
         pix3 = PXMpix(E3pxm,px,py);
         pbright = ff * PIXBRIGHT(pix3);

         group = pixgroup[ii-E1pxm->ww-1];
         if (group) {
            ++edgecount[group];                                                  //  accumulate pixel count and
            edgebright[group] += pbright;                                        //      bordering the groups
         }

         group = pixgroup[ii-E1pxm->ww];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii-E1pxm->ww+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii-1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+E1pxm->ww-1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+E1pxm->ww];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }

         group = pixgroup[ii+E1pxm->ww+1];
         if (group) {
            ++edgecount[group];
            edgebright[group] += pbright;
         }
      }

      for (group = 1; group <= Ngroups; group++)                                 //  compute group pixel and edge pixel
      {                                                                          //    mean brightness
         if (groupcount[group] && edgecount[group]) {
            edgebright[group] = edgebright[group] / edgecount[group];
            groupbright[group] = groupbright[group] / groupcount[group];
            pcontrast = edgebright[group] - groupbright[group];                  //  edge - group contrast
            if (pcontrast < contrast) {
               groupcount[group] = 0;
               Nremoved++;
            }
         }
      }

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;                                               //  eliminate group if low contrast
         group = pixgroup[ii];
         if (! group) continue;
         if (! groupcount[group]) pixgroup[ii] = 0;
      }

      for (py = 0; py < E1pxm->hh; py++)                                         //  loop all pixels
      for (px = 0; px < E1pxm->ww; px++)
      {
         ii = py * E1pxm->ww + px;
         if (! pixgroup[ii]) continue;                                           //  not a dust pixel
         pix3 = PXMpix(E3pxm,px,py);                                             //  paint it red
         pix3[0] = 255;
         pix3[1] = pix3[2] = 0;
      }

      Fred = 1;

      Fpaint2();                                                                 //  update window
   }

   return 0;
}


//  erase the selected dust areas

void dust_erase()
{
   using namespace dust_names;

   int         cc, ii, px, py, inc;
   int         qx, qy, npx, npy;
   int         sx, sy, tx, ty;
   int         rad, dist, dist2, mindist2;
   float       slope, f1, f2;
   float       *pix1, *pix3;
   char        *pmap;
   int         nc = E1pxm->nc, pcc = nc * sizeof(float);

   Ffuncbusy = 1;
   PXM_free(E3pxm);                                                              //  not a thread
   E3pxm = PXM_copy(E9pxm);

   cc = E1pxm->ww * E1pxm->hh;                                                   //  allocate pixel done map
   pmap = (char *) zmalloc(cc);
   memset(pmap,0,cc);

   for (py = 0; py < E1pxm->hh; py++)                                            //  loop all pixels
   for (px = 0; px < E1pxm->ww; px++)
   {
      ii = py * E1pxm->ww + px;
      if (! pixgroup[ii]) continue;                                              //  not a dust pixel
      if (pmap[ii]) continue;                                                    //  skip pixels already done

      mindist2 = 999999;
      npx = npy = 0;

      for (rad = 1; rad < 10; rad++)                                             //  find nearest edge (10 pixel limit)
      {
         for (qx = px-rad; qx <= px+rad; qx++)                                   //  search within rad
         for (qy = py-rad; qy <= py+rad; qy++)
         {
            if (qx < 0 || qx >= E1pxm->ww) continue;                             //  off image edge
            if (qy < 0 || qy >= E1pxm->hh) continue;
            ii = qy * E1pxm->ww + qx;
            if (pixgroup[ii]) continue;                                          //  within dust area

            dist2 = (px-qx) * (px-qx) + (py-qy) * (py-qy);                       //  distance**2 to edge pixel
            if (dist2 < mindist2) {
               mindist2 = dist2;
               npx = qx;                                                         //  save nearest pixel found
               npy = qy;
            }
         }

         if (rad * rad >= mindist2) break;                                       //  can quit now
      }

      if (! npx && ! npy) continue;                                              //  should not happen

      qx = npx;                                                                  //  nearest edge pixel
      qy = npy;

      if (abs(qy - py) > abs(qx - px)) {                                         //  qx/qy = near edge from px/py
         slope = 1.0 * (qx - px) / (qy - py);
         if (qy > py) inc = 1;
         else inc = -1;
         for (sy = py; sy != qy+inc; sy += inc)                                  //  line from px/py to qx/qy
         {
            sx = px + slope * (sy - py);
            ii = sy * E1pxm->ww + sx;
            if (pmap[ii]) continue;
            pmap[ii] = 1;
            tx = qx + (qx - sx);                                                 //  tx/ty = parallel line from qx/qy
            ty = qy + (qy - sy);
            if (tx < 0) tx = 0;
            if (tx > E1pxm->ww-1) tx = E1pxm->ww-1;
            if (ty < 0) ty = 0;
            if (ty > E1pxm->hh-1) ty = E1pxm->hh-1;
            pix1 = PXMpix(E3pxm,tx,ty);                                          //  copy pixel from tx/ty to sx/sy
            pix3 = PXMpix(E3pxm,sx,sy);
            memcpy(pix3,pix1,pcc);
         }
      }

      else {
         slope = 1.0 * (qy - py) / (qx - px);
         if (qx > px) inc = 1;
         else inc = -1;
         for (sx = px; sx != qx+inc; sx += inc)
         {
            sy = py + slope * (sx - px);
            ii = sy * E1pxm->ww + sx;
            if (pmap[ii]) continue;
            pmap[ii] = 1;
            tx = qx + (qx - sx);
            ty = qy + (qy - sy);
            if (tx < 0) tx = 0;
            if (tx > E1pxm->ww-1) tx = E1pxm->ww-1;
            if (ty < 0) ty = 0;
            if (ty > E1pxm->hh-1) ty = E1pxm->hh-1;
            pix1 = PXMpix(E3pxm,tx,ty);
            pix3 = PXMpix(E3pxm,sx,sy);
            memcpy(pix3,pix1,pcc);
         }
      }
   }

   zfree(pmap);

   if (sa_stat == 3)                                                             //  area edge blending
   {
      for (ii = 0; ii < E1pxm->ww * E1pxm->hh; ii++)                             //  find pixels in select area
      {
         dist = sa_pixmap[ii];
         if (! dist || dist >= sa_blendwidth) continue;

         py = ii / E1pxm->ww;
         px = ii - py * E1pxm->ww;
         pix1 = PXMpix(E1pxm,px,py);                                             //  input pixel, unchanged image
         pix3 = PXMpix(E3pxm,px,py);                                             //  output pixel, changed image

         f2 = sa_blendfunc(dist);
         f1 = 1.0 - f2;

         pix3[0] = f1 * pix1[0] + f2 * pix3[0];                                  //  blend the pixels
         pix3[1] = f1 * pix1[1] + f2 * pix3[1];
         pix3[2] = f1 * pix1[2] + f2 * pix3[2];
      }
   }

   Fred = 0;
   Ffuncbusy = 0;
   Fpaint2();                                                                    //  update window
   return;
}


/********************************************************************************/

//  Shift R/B color planes to maximize overlap with G plane.
//  Radius of a pixel in R-plane or B-plane is shifted using the formula:
//    R2 = F1 * R1 + F2 * R1 * R1 + F3 * sqrt(R1) 
//    R1: pixel old radius   R2: pixel new radius
//    F1 F2 F3: computed values for maximum overlap
//    F1 is near 1.0
//    F2 and F3 are near 0.0

namespace chromatic1_names
{
   editfunc    EFchromatic1;
   double      Rf1, Rf2, Rf3, Bf1, Bf2, Bf3;
   char        editparms[40]; 
   int         Eww, Ehh;
   int         cx, cy;
   uint8       *pixcon;
   int         threshcon;
   int         evrgb;
   float       evF1, evF2, evF3;
   double      evRsum1[max_threads];
   double      evRsum2;
}


//  menu function

void m_chromatic1(GtkWidget *, cchar *menu)
{
   using namespace chromatic1_names;
   
   void  chromatic1_threshcon();
   int   chromatic1_dialog_event(zdialog* zd, cchar *event);
   
   cchar    *title = "Chromatic Aberration";
   F1_help_topic = "chromatic 1";

   EFchromatic1.menuname = "Chromatic 1";                                        //  setup edit
   EFchromatic1.menufunc = m_chromatic1;
   EFchromatic1.Farea = 1;                                                       //  select area ignored
   if (! edit_setup(EFchromatic1)) return;

   Eww = E3pxm->ww;                                                              //  image dimensions
   Ehh = E3pxm->hh;

   cx = Eww / 2;                                                                 //  image center
   cy = Ehh / 2;
   
   chromatic1_threshcon();                                                       //  get image threshold contrast
   
/***
       ____________________________________________
      |          Chromatic Aberration              |
      |                                            |
      |  Red Factors:   [ 1.0 ]  [ 0.0 ]  [ 0.0 ]  |
      |  Blue Factors:  [ 1.0 ]  [ 0.0 ]  [ 0.0 ]  |
      |                                            |
      |  Find optimum factors: [ Search ]          |
      |                                            |
      |                            [ OK ] [Cancel] |
      |____________________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,BOK,Bcancel,null);
   CEF->zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=3|homog");
   zdialog_add_widget(zd,"label","labR","vb1","Red Factors");
   zdialog_add_widget(zd,"zspin","Rf1","vb2","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"zspin","Rf2","vb3","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"zspin","Rf3","vb4","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"label","labB","vb1","Blue Factors");
   zdialog_add_widget(zd,"zspin","Bf1","vb2","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"zspin","Bf2","vb3","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"zspin","Bf3","vb4","-3|+3|0.2|0.0","size=6");
   zdialog_add_widget(zd,"hbox","hbopt","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labopt","hbopt","Find optimum factors:");
   zdialog_add_widget(zd,"button","search","hbopt",Bsearch,"space=5");

   zdialog_run(zd,chromatic1_dialog_event,"save");                               //  run dialog - parallel
   return;
}


//  dialog event and completion function

int chromatic1_dialog_event(zdialog *zd, cchar *event)
{
   using namespace chromatic1_names;

   void  chromatic1_RBshift();
   void  chromatic1_RBsearch();

   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 1;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 2;                                  //  from f_open()

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                      //  done
         snprintf(editparms,40,"%.1f %.1f %.1f %.1f %.1f %.1f",
                                 Rf1, Rf2, Rf3, Bf1, Bf2, Bf3);                  //  log exif edit hist                    21.0
         EFchromatic1.editparms = editparms;
         edit_done(0);
      }
      else edit_cancel(0);                                                       //  discard edit
      zfree(pixcon);                                                             //  free memory
      return 1;
   }

   if (zstrstr("Rf1 Rf2 Rf3 Bf1 Bf2 Bf3",event))
   {
      zdialog_fetch(zd,"Rf1",Rf1);                                               //  get manually adjusted factors
      zdialog_fetch(zd,"Rf2",Rf2);
      zdialog_fetch(zd,"Rf3",Rf3);
      zdialog_fetch(zd,"Bf1",Bf1);
      zdialog_fetch(zd,"Bf2",Bf2);
      zdialog_fetch(zd,"Bf3",Bf3);
      zmainloop();
      
      chromatic1_RBshift();                                                      //  shift R/B color planes using factors
   }
   
   if (strmatch(event,"search"))
   {
      chromatic1_RBsearch();                                                     //  search for optimum factors
      
      zdialog_stuff(zd,"Rf1",Rf1);                                               //  stuff dialog with found factors
      zdialog_stuff(zd,"Rf2",Rf2);
      zdialog_stuff(zd,"Rf3",Rf3);
      zdialog_stuff(zd,"Bf1",Bf1);
      zdialog_stuff(zd,"Bf2",Bf2);
      zdialog_stuff(zd,"Bf3",Bf3);

      chromatic1_RBshift();                                                      //  shift R/B color planes using factors
   }

   return 1;
}


//  Find contrast threshold for highest-contrast pixels in color green.

void chromatic1_threshcon()
{
   using namespace chromatic1_names;

   int      ii, jj;
   int      px, py, qx, qy;
   int      Gcon, Gmax;
   int      condist[256];
   float    *pix1, *pix2;

   pixcon = (uint8 *) zmalloc(Eww * Ehh);                                        //  map of pixel green contrast
   
   for (py = 9; py < Ehh-9; py++)                                                //  loop all pixels
   for (px = 9; px < Eww-9; px++)
   {
      Gmax = 0;
      pix1 = PXMpix(E1pxm,px,py);

      for (qy = py - 5; qy <= py + 5; qy += 10)                                  //  loop 4 neighbors offset by 5
      for (qx = px - 5; qx <= px + 5; qx += 10)
      {
         pix2 = PXMpix(E1pxm,qx,qy);                                             //  find max. green contrast
         Gcon = pix1[1] - pix2[1];
         Gcon = abs(Gcon);
         if (Gcon > Gmax) Gmax = Gcon;
      }
      
      ii = py * Eww + px;                                                        //  save pixel green contrast
      pixcon[ii] = Gmax;
   }
   
   memset(condist, 0, 256 * sizeof(int));

   for (ii = 0; ii < Eww * Ehh; ii++)                                            //  make histogram of contrast values
   {
      jj = pixcon[ii];                                                           //  condist[jj] = pixels with
      ++condist[jj];                                                             //    contrast value jj 
   }
   
   for (jj = 0, ii = 255; ii > 0; ii--)                                          //  find minimum contrast for 200K
   {                                                                             //    highest green-contrast pixels
      jj += condist[ii];
      if (ii > 100 && jj > 100000) break;
      if (jj > 200000) break;
   }

   threshcon = ii;                                                               //  threshold for chromatic1_evaluate()
   return;
}


//  Compute factors Rf1 Rf2 Bf1 Bf2 that minimize
//    Rplane - Gplane  and  Bplane - Gplane

void chromatic1_RBsearch()
{
   using namespace chromatic1_names;

   double chromatic1_evaluate(int rgb, float F1, float F2, float F3);

   int      rgb;                                                                 //  0/1/2 = red/green/blue
   float    F1, F2, F3;
   float    R1, R2, R3;
   float    S1, S2, S3;
   double   Rsum, Rmin;
   
   Ffuncbusy = 1;
   progressmon_reset(1372);                                                      //  initz. progress counter

   for (rgb = 0; rgb <= 2; rgb += 2)                                             //  loop rgb = 0, 2  (red, blue) 
   {
      Rmin = 1.0 * Eww * Ehh * 256;                                              //  max. possible image difference
      R1 = R2 = R3 = 0;

      for (F1 = -3; F1 <= +3; F1 += 1.0)                                         //  loop all combinations F1, F2, F3
      for (F2 = -3; F2 <= +3; F2 += 1.0)                                         //    in 1-pixel steps
      for (F3 = -3; F3 <= +3; F3 += 1.0)
      {
         Rsum = chromatic1_evaluate(rgb, F1, F2, F3);                            //  evaluate each combination
         if (Rsum < Rmin) {
            Rmin = Rsum;                                                         //  remember best combination
            R1 = F1;
            R2 = F2;
            R3 = F3;
         }
         
         busy_add(0,1);
         zmainloop();
      }
      
      S1 = R1; S2 = R2; S3 = R3;                                                 //  loop around best combination
      
      for (F1 = R1-1; F1 <= R1+1; F1 += 0.333)                                   //  loop all combinations F1, F2, F3
      for (F2 = R2-1; F2 <= R2+1; F2 += 0.333)                                   //    in 0.333 pixel steps
      for (F3 = R3-1; F3 <= R3+1; F3 += 0.333)
      {
         Rsum = chromatic1_evaluate(rgb, F1, F2, F3);                            //  evaluate each combination
         if (Rsum < Rmin) {
            Rmin = Rsum;                                                         //  remember best combination
            S1 = F1;
            S2 = F2;
            S3 = F3;
         }

         busy_add(0,1);
         zmainloop();
      }
      
      if (rgb == 0) {
         Rf1 = S1;                                                               //  red plane factors
         Rf2 = S2;
         Rf3 = S3;
      }
      else {
         Bf1 = S1;                                                               //  blue plane factors
         Bf2 = S2;
         Bf3 = S3;
      }
   }
   
   Ffuncbusy = 0;
   progressmon_reset(0);
   return;
}


//  evaluate the alignment of a shifted R/B color plane with the G plane
//  R/B color plane is shifted using factors F1 F2 F3
//  rgb is 0/2 for R/B

double chromatic1_evaluate(int rgb, float F1, float F2, float F3)
{
   using namespace chromatic1_names;

   void * chromatic1_evaluate_wthread(void *);

   evrgb = rgb;                                                                  //  make args avail. for thread
   evF1 = F1;
   evF2 = F2;
   evF3 = F3;

   do_wthreads(chromatic1_evaluate_wthread,NWT);                                 //  do worker threads
   
   evRsum2 = 0;
   for (int ii = 0; ii < NWT; ii++) evRsum2 += evRsum1[ii];
   return evRsum2;
}


void * chromatic1_evaluate_wthread(void *arg)                                    //  worker thread
{
   using namespace chromatic1_names;

   int      index = *((int *) arg);
   int      ii, px1, py1;
   float    fx1, fy1, fx2, fy2, fx3, fy3;
   float    *pix1, vpix[4];
   double   Rsum = 0;
   
   for (py1 = index + 9; py1 < Ehh-9; py1 += NWT)                                //  loop all image pixels
   for (px1 = 9; px1 < Eww-9; px1++)
   {
      ii = py1 * Eww + px1;                                                      //  skip low contrast pixel
      if (pixcon[ii] < threshcon) continue;
      fx1 = 1.0 * (px1 - cx) / cx;                                               //  -1 to +1 at edges, 0 at center
      fy1 = 1.0 * (py1 - cy) / cy;
      fx2 = fx1 * fabsf(fx1);                                                    //  square, keep sign
      fy2 = fy1 * fabsf(fy1);
      fx3 = fx1 * fx2;                                                           //  cube
      fy3 = fy1 * fy2;
      fx1 = fx1 * evF1;                                                          //  * F1
      fy1 = fy1 * evF1;
      fx2 = fx2 * evF2;                                                          //  * F2
      fy2 = fy2 * evF2;
      fx3 = fx3 * evF3;                                                          //  * F3
      fy3 = fy3 * evF3;
      pix1 = PXMpix(E1pxm,px1,py1);                                              //  image unshifted pixel (G)
      vpixel(E1pxm, px1+fx1+fx2+fx3, py1+fy1+fy2+fy3, vpix);                     //  virtual pixel, shifted (R/B) 
      Rsum += fabs(vpix[evrgb] - pix1[1]);                                       //  sum shifted R/B - unshifted G
   }

   evRsum1[index] = Rsum;

   return 0;
}

   
//  shift R/B color planes using factors Rf1 Rf2 Bf1 Bf2

void chromatic1_RBshift()
{
   void * chromatic1_RBshift_wthread(void *);

   do_wthreads(chromatic1_RBshift_wthread,NWT);                                  //  do worker threads

   CEF->Fmods++;                                                                 //  image is modified
   CEF->Fsaved = 0;                                                              //  and not saved

   Fpaint2();                                                                    //  update window
   return;
}


void * chromatic1_RBshift_wthread(void *arg)                                     //  worker thread
{
   using namespace chromatic1_names;

   int      index = *((int *) arg);
   int      px3, py3, ok;
   float    px1, py1;
   float    fx1, fy1, fx2, fy2, fx3, fy3;
   float    *pix3, vpix[4];
   
   for (py3 = index; py3 < Ehh; py3 += NWT)                                      //  loop all image pixels
   for (px3 = 0; px3 < Eww; px3++)
   {
      pix3 = PXMpix(E3pxm,px3,py3);                                              //  output pixel

      fx1 = 1.0 * (px3 - cx) / cx;                                               //  -1 to +1 at edges, 0 at center
      fy1 = 1.0 * (py3 - cy) / cy;
      fx2 = fx1 * fabsf(fx1);                                                    //  square, keep sign
      fy2 = fy1 * fabsf(fy1);
      fx3 = fx1 * fx2;                                                           //  cube
      fy3 = fy1 * fy2;

      px1 = px3 + Rf1 * fx1 + Rf2 * fx2 + Rf3 * fx3;                             //  red shift
      py1 = py3 + Rf1 * fy1 + Rf2 * fy2 + Rf3 * fy3;
      ok = vpixel(E1pxm,px1,py1,vpix);                                           //  red input pixel
      if (ok) pix3[0] = vpix[0]; 

      px1 = px3 + Bf1 * fx1 + Bf2 * fx2 + Bf3 * fx3;                             //  blue shift
      py1 = py3 + Bf1 * fy1 + Bf2 * fy2 + Bf3 * fy3;
      ok = vpixel(E1pxm,px1,py1,vpix);                                           //  blue input pixel
      if (ok) pix3[2] = vpix[2];
   }
   
   return 0;
}


/********************************************************************************/

//  Remove axial chromatic aberration, color bands often seen on
//    dark image features against a bright background.

namespace chromatic2_names
{
   editfunc    EFchromatic2;
   zdialog     *zd;
   int         Eww, Ehh;                                                         //  image dimensions
   float       Crgb[3];                                                          //  chromatic color
   float       Rrgb[3];                                                          //  replacement color
   float       Brgb[3];                                                          //  background color
   float       Rhsl[3];                                                          //  replacement color, HSL space
   float       Cmatch;                                                           //  chromatic color match level
   int         pcc = 3 * sizeof(float);                                          //  RGB pixel size
   uint8       *Pmark;                                                           //  pixel undo markers
   int         Vmark;                                                            //  current Pmark value 0-255
   int         Wcolor;                                                           //  which color check button
   int         Bprox;                                                            //  background proximity range
   int         Fusehue;                                                          //  flag, use hue
}


//  menu function

void m_chromatic2(GtkWidget *, cchar *menu)
{
   using namespace chromatic2_names;
   
   int   chromatic2_dialog_event(zdialog* zd, cchar *event);
   void  chromatic2_mousefunc();
   
   cchar    *title = "Chromatic Aberration";
   int      cc;

   F1_help_topic = "chromatic 2";

   EFchromatic2.menuname = "Chromatic 2";                                        //  setup edit
   EFchromatic2.menufunc = m_chromatic2;
   EFchromatic2.mousefunc = chromatic2_mousefunc;                                //  mouse function
   EFchromatic2.Farea = 2;                                                       //  select area OK
   if (! edit_setup(EFchromatic2)) return;

   Eww = E3pxm->ww;                                                              //  image dimensions
   Ehh = E3pxm->hh;
   
   cc = Eww * Ehh;                                                               //  allocate pixel markers
   Pmark = (uint8 *) zmalloc(cc);
   memset(Pmark,0,cc);

/***
       ______________________________________
      |       Chromatic Aberration           |
      |                                      |
      |  [x] Chromatic Color [ ### ]         |
      |  [_] Replacement Color [ ### ]       |
      |  [_] Background Color [ ### ]        |      
      |  Color match level [ 70 ]            |
      |  Background proximity [ 10 ]         |
      |                                      |
      |       [apply] [undo] [ OK ] [cancel] |
      |______________________________________|

***/

   zd = zdialog_new(title,Mwin,Bapply,Bundo,BOK,Bcancel,null);
   CEF->zd = zd;
   
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog");

   zdialog_add_widget(zd,"check","Ccheck","vb1","Chromatic Color","space=3");
   zdialog_add_widget(zd,"colorbutt","Crgb","vb2","0|0|0");

   zdialog_add_widget(zd,"check","Rcheck","vb1","Replacement Color","space=3");
   zdialog_add_widget(zd,"colorbutt","Rrgb","vb2","0|0|0");

   zdialog_add_widget(zd,"check","Bcheck","vb1","Background Color","space=3");
   zdialog_add_widget(zd,"colorbutt","Brgb","vb2","0|0|0");

   zdialog_add_widget(zd,"hbox","hb4","dialog");
   zdialog_add_widget(zd,"label","labmatch","hb4","Color match level","space=3");
   zdialog_add_widget(zd,"zspin","Cmatch","hb4","50|100|1|70","space=3");

   zdialog_add_widget(zd,"hbox","hb5","dialog");
   zdialog_add_widget(zd,"label","labp","hb5","Background Proximity","space=3");
   zdialog_add_widget(zd,"zspin","Bprox","hb5","1|100|1|10","space=3");
   
   zdialog_run(zd,chromatic2_dialog_event,"save");                               //  run dialog - parallel

   zdialog_stuff(zd,"Ccheck",1);
   zdialog_stuff(zd,"Rcheck",0);
   zdialog_stuff(zd,"Bcheck",0);

   Wcolor = 1;                                                                   //  next color = chromatic
   Cmatch = 70;                                                                  //  match level %
   Bprox = 10;                                                                   //  background proximity
   Vmark = 0;                                                                    //  no marks yet

   takeMouse(chromatic2_mousefunc,dragcursor);      
   return;
}


//  dialog event and completion function

int chromatic2_dialog_event(zdialog *zd, cchar *event)
{
   using namespace chromatic2_names;
   
   void chromatic2_mousefunc();
   void chromatic2_repair();
   
   int      ii;
   int      px, py;
   float    *pix1, *pix3;
   
   if (strmatch(event,"escape")) zd->zstat = -2;                                 //  escape key                            21.0
   if (strmatch(event,"done")) zd->zstat = 3;                                    //  from edit_setup() or f_save()
   if (strmatch(event,"cancel")) zd->zstat = 4;                                  //  from f_open()

   if (zd->zstat)
   {  
      if (zd->zstat == 1)                                                        //  [apply]
      {
         zd->zstat = 0;                                                          //  keep dialog active

         if (Vmark == 255) {                                                     //  check for max. marker value
            zmessageACK(Mwin,"255 iterations, cannot continue");
            return 1;
         }

         chromatic2_repair();                                                    //  do repair function
         return 1;
      }

      if (zd->zstat == 2)                                                        //  [undo]
      {
         zd->zstat = 0;                                                          //  keep dialog active

         if (Vmark < 1) return 1;                                                //  nothing to undo
         
         for (py = 0; py < Ehh; py++)
         for (px = 0; px < Eww; px++)
         {
            ii = py * Eww + px;
            if (Pmark[ii] == Vmark) {
               pix1 = PXMpix(E1pxm,px,py);
               pix3 = PXMpix(E3pxm,px,py);
               memcpy(pix3,pix1,pcc);
            }
         }
         
         Vmark -= 1;
         Fpaint2();
         return 1;
      }
      
      if (zd->zstat == 3) edit_done(0);                                          //  [ OK ]
      else edit_cancel(0);                                                       //  [cancel]

      freeMouse();
      zfree(Pmark);
      return 1;
   }
   
   if (strmatch(event,"focus"))
      takeMouse(chromatic2_mousefunc,dragcursor);
   
   if (zstrstr("Ccheck Rcheck Bcheck",event)) {                                  //  mark which color to be selected
      zdialog_stuff(zd,"Ccheck",0);
      zdialog_stuff(zd,"Rcheck",0);
      zdialog_stuff(zd,"Bcheck",0);
      zdialog_stuff(zd,event,1);
      if (strmatch(event,"Ccheck")) Wcolor = 1;
      if (strmatch(event,"Rcheck")) Wcolor = 2;
      if (strmatch(event,"Bcheck")) Wcolor = 3;
   }
   
   if (strmatch(event,"Bprox"))                                                  //  background proximity limit
      zdialog_fetch(zd,"Bprox",Bprox);
   
   if (strmatch(event,"Cmatch"))
      zdialog_fetch(zd,"Cmatch",Cmatch);

   return 1;
}


//  mouse function

void chromatic2_mousefunc()
{
   using namespace chromatic2_names;
   
   float    *pix;
   char     text[20];

   if (! LMclick) return;
   LMclick = 0;
   if (! zd) return;
   
   pix = PXBpix(E1pxm,Mxclick,Myclick);                                          //  pick new color from image
   snprintf(text,20,"%.0f|%.0f|%.0f",pix[0],pix[1],pix[2]);
   
   if (Wcolor == 1) {                                                            //  chromatic color to fix
      Crgb[0] = pix[0];
      Crgb[1] = pix[1];
      Crgb[2] = pix[2];
      zdialog_stuff(zd,"Crgb",text);                                             //  update dialog color button
   }   

   if (Wcolor == 2) {                                                            //  replacement color
      Rrgb[0] = pix[0];
      Rrgb[1] = pix[1];
      Rrgb[2] = pix[2];
      zdialog_stuff(zd,"Rrgb",text);
   }   

   if (Wcolor == 3) {                                                            //  background color
      Brgb[0] = pix[0];
      Brgb[1] = pix[1];
      Brgb[2] = pix[2];
      zdialog_stuff(zd,"Brgb",text);
   }
   
   return;
}


//  repair the chromatic aberraton

void chromatic2_repair()
{
   using namespace chromatic2_names;

   void * chromatic2_repair_wthread(void *arg);
   
   float    rmin, rmax;
   char     text[20];
   cchar    *pp;
   
   zdialog_fetch(zd,"Crgb",text,20);                                             //  get chromatic color to fix
   pp = substring(text,'|',1);
   if (pp) Crgb[0] = atoi(pp);
   pp = substring(text,'|',2);
   if (pp) Crgb[1] = atoi(pp);
   pp = substring(text,'|',3);
   if (pp) Crgb[2] = atoi(pp);

   zdialog_fetch(zd,"Rrgb",text,20);                                             //  get replacement color
   pp = substring(text,'|',1);
   if (pp) Rrgb[0] = atoi(pp);
   pp = substring(text,'|',2);
   if (pp) Rrgb[1] = atoi(pp);
   pp = substring(text,'|',3);
   if (pp) Rrgb[2] = atoi(pp);

   zdialog_fetch(zd,"Brgb",text,20);                                             //  get background color
   pp = substring(text,'|',1);
   if (pp) Brgb[0] = atoi(pp);
   pp = substring(text,'|',2);
   if (pp) Brgb[1] = atoi(pp);
   pp = substring(text,'|',3);
   if (pp) Brgb[2] = atoi(pp);
   
   zdialog_fetch(zd,"Bprox",Bprox);                                              //  background proximity range
   zdialog_fetch(zd,"Cmatch",Cmatch);                                            //  color match level 
   
   RGBtoHSL(Rrgb[0],Rrgb[1],Rrgb[2],Rhsl[0],Rhsl[1],Rhsl[2]);                    //  replacement color in HSL units

   rmin = rmax = Rrgb[0];
   if (Rrgb[1] < rmin) rmin = Rrgb[1];                                           //  get replacement color RGB range
   if (Rrgb[2] < rmin) rmin = Rrgb[2]; 
   if (Rrgb[1] > rmax) rmax = Rrgb[1];
   if (Rrgb[2] > rmax) rmax = Rrgb[2];

   Fusehue = 0;                                                                  //  use replacement color hue
   if (rmax > 0 && (rmax-rmin) / rmax > 0.1) Fusehue = 1;                        //    only if distinct hue

   Vmark += 1;                                                                   //  next pixel marker value

   do_wthreads(chromatic2_repair_wthread,NWT);                                   //  do worker threads

   CEF->Fmods++;                                                                 //  image is modified
   CEF->Fsaved = 0;                                                              //  and not saved
   Fpaint2();                                                                    //  update window
   return;
}


void * chromatic2_repair_wthread(void *arg)                                      //  worker threads
{
   using namespace chromatic2_names;

   int      index = *((int *) arg);
   int      ii, dist;
   int      px, py, qx, qy;
   float    f1, f2;
   float    *pix1, *pix3;
   float    Prgb[3], Phsl[3];
   int      dx, dy, R, D;
   
   R = Bprox;
   
   for (py = R+index; py < Ehh-R; py += NWT)                                     //  loop all image pixels
   for (px = R; px < Eww-R; px++)                                                //  find pix1 matching chromatic color
   {
      ii = py * Eww + px;

      if (sa_stat == 3) {                                                        //  select area active
         dist = sa_pixmap[ii];                                                   //  distance from edge
         if (! dist) continue;                                                   //  pixel outside area
      }

      pix3 = PXMpix(E1pxm,px,py);                                                //  input pixel
      f1 = PIXMATCH(pix3,Crgb);                                                  //  compare to chromatic color
      if (f1 < 0.01 * Cmatch) continue;                                          //  match level, 0.7 ... 1.0

      f1 = 3.333 * f1 - 2.333;                                                   //  part replacement color, 0.0 ... 1.0
      f1 = sqrtf(f1);                                                            //  use curve 
      f2 = 1.0 - f1;                                                             //  part original color, 1.0 ... 0.0
      
      D = 999;

      for (qy = py-R; qy <= py+R; qy++)                                          //  loop pixels within Bprox of pix3
      for (qx = px-R; qx <= px+R; qx++)
      {
         pix1 = PXMpix(E1pxm,qx,qy);
         if (PIXMATCH(pix1,Brgb) < 0.01 * Cmatch) continue;                      //  not a background pixel
         dx = qx - px;
         dy = qy - py;
         D = sqrtf(dx*dx + dy*dy);                                               //  image pixel distance to background
         if (D <= Bprox) goto break2;                                            //  exit both loops
      }  break2:

      if (D > Bprox) continue;                                                   //  pix3 too far from background
      
      pix3 = PXMpix(E3pxm,px,py);                                                //  output pixel

      if (Fusehue) {                                                             //  use replacement color hue only
         RGBtoHSL(pix3[0],pix3[1],pix3[2],Phsl[0],Phsl[1],Phsl[2]);              //  convert to HSL
         Phsl[0] = Rhsl[0];                                                      //  replace hue only
         HSLtoRGB(Phsl[0],Phsl[1],Phsl[2],Prgb[0],Prgb[1],Prgb[2]);              //  back to RGB
         pix3[0] = f1 * Prgb[0] + f2 * pix3[0];                                  //  new = mix of new + old
         pix3[1] = f1 * Prgb[1] + f2 * pix3[1];
         pix3[2] = f1 * Prgb[2] + f2 * pix3[2];
      }
      
      else {
         pix3[0] = f1 * Rrgb[0] + f2 * pix3[0];                                  //  use replacement color RGB
         pix3[1] = f1 * Rrgb[1] + f2 * pix3[1];
         pix3[2] = f1 * Rrgb[2] + f2 * pix3[2];
      }

      Pmark[ii] = Vmark;                                                         //  mark undo level
   }
   
   return 0;
}


