/* Copyright (C) 2008 and 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/emitter.h>
#include <c++-gtk-utils/thread.h>

extern "C" {
  static gboolean cgu_callback_wrapper(void*);
  static gboolean cgu_tracked_callback_wrapper(void*);
}

gboolean cgu_callback_wrapper(void* data) {
  const Cgu::Callback::Callback* cb = static_cast<Cgu::Callback::Callback*>(data);

  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    cb->dispatch();
  }
  catch (...) {
    g_critical("Exception thrown in callback_wrapper() for Callback::post() function\n");
  }
  delete cb;
  return false;
}

gboolean cgu_tracked_callback_wrapper(void* data) {
  const Cgu::SafeEmitter* e = static_cast<Cgu::SafeEmitter*>(data);

  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    e->emit();
  }
  catch (...) {
    g_critical("Exception thrown in tracked_callback_wrapper() for Callback::post() function\n");
  }
  delete e;
  return false;
}

namespace Cgu {

namespace Callback {

void post(const Callback* cb, gint priority, GMainContext* context) {
  // context has a default value of NULL which will attach the idle source
  // to the default program main context

  GSource* source_p = g_idle_source_new();
  if (priority != G_PRIORITY_DEFAULT_IDLE)
    g_source_set_priority(source_p, priority);
  g_source_set_callback(source_p, cgu_callback_wrapper, const_cast<Callback*>(cb), 0);

  g_source_attach(source_p, context);
  g_source_unref(source_p);
}

void post(const Callback* cb, Releaser& r, gint priority, GMainContext* context) {
  // context has a default value of NULL which will attach the idle source
  // to the default program main context

  SafeFunctor f(cb);   // take ownership
  SafeEmitter* e = new SafeEmitter;
  try {
    e->connect(f, r);
  }
  catch (...) {
    delete e;
    throw;
  }
  
  GSource* source_p = g_idle_source_new();
  if (priority != G_PRIORITY_DEFAULT_IDLE)
    g_source_set_priority(source_p, priority);
  g_source_set_callback(source_p, cgu_tracked_callback_wrapper, e, 0);

  g_source_attach(source_p, context);
  g_source_unref(source_p);
}

} // namespace Callback

} // namespace Cgu
