/************************************************************************
 *
 *  ConverterPalette.java
 *
 *  Copyright: 2002-2023 by Henrik Just
 *
 *  This file is part of Writer2LaTeX.
 *  
 *  Writer2LaTeX is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  Writer2LaTeX is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with Writer2LaTeX.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *  Version 2.0 (2023-06-29)
 *
 */

package writer2latex.latex;

import org.w3c.dom.Element;

import java.io.IOException;
import java.util.List;

import writer2latex.api.Config;
import writer2latex.api.ConverterFactory;
import writer2latex.base.ConverterBase;
import writer2latex.latex.i18n.ClassicI18n;
import writer2latex.latex.i18n.I18n;
import writer2latex.latex.i18n.XeTeXI18n;
import writer2latex.latex.tikz.TikZConverter;
import writer2latex.latex.util.Context;
import writer2latex.util.CSVList;
import writer2latex.util.Calc;
import writer2latex.util.ExportNameCollection;
import writer2latex.util.Misc;
import writer2latex.office.MIMETypes;

/** This class converts a Writer XML file to a LaTeX file
 */
public final class ConverterPalette extends ConverterBase {

    // Configuration
    private LaTeXConfig config;
	
    public Config getConfig() { return config; }

    // The main outfile
    private LaTeXDocument texDoc;
    
    // Various data used in conversion
    private Context mainContext; // main context
    private CSVList globalOptions; // global options

    // The helpers (the "colors" of the palette)
    private I18n i18n;
    private ColorConverter colorCv;
    private MicrotypeConverter microtypeCv;
    private FrameStyleConverter frameSc;
    private CharStyleConverter charSc;
    private HeadingStyleConverter headingSc;
    private SectionStyleConverter sectionSc;
    private PageStyleConverter pageSc;
    private BlockConverter blockCv;
    private ParConverter parCv;
    private HeadingConverter headingCv;
    private IndexConverter indexCv;
    private BibConverter bibCv;
    private SectionConverter sectionCv;
    private TableConverter tableCv;
    private ListConverter listCv;
    private NoteConverter noteCv;
    private CaptionConverter captionCv;
    private InlineConverter inlineCv;
    private FieldConverter fieldCv;
    private DrawConverter drawCv;
    private TikZConverter tikZCv;
    private MathConverter mathCv;
    private Info info;
	
    // Constructor
    public ConverterPalette() {
        super();
        config = new LaTeXConfig();
    }
	
    // Accessor methods for data
	
    public String getOutFileName() { return sTargetFileName; }

    public Context getMainContext() { return mainContext; }
	
    public void addGlobalOption(String sOption) {
	    globalOptions.addValue(sOption);
    }
    
    // Accessor methods for helpers
    public I18n getI18n() { return i18n; }
    public ColorConverter getColorCv() { return colorCv; }
    public MicrotypeConverter getMicrotypeCv() { return microtypeCv; }
    public FrameStyleConverter getFrameStyleSc() { return frameSc; }
    public CharStyleConverter getCharSc() { return charSc; }
    //public HeadingStyleConverter getHeadingSc() { return headingSc; }
    public SectionStyleConverter getSectionSc() { return sectionSc; }
    public PageStyleConverter getPageSc() { return pageSc; }
    public BlockConverter getBlockCv() { return blockCv; }
    public ParConverter getParCv() { return parCv; }
    public HeadingConverter getHeadingCv() { return headingCv; }
    public IndexConverter getIndexCv() { return indexCv; }
    public BibConverter getBibCv() { return bibCv; }
    public SectionConverter getSectionCv() { return sectionCv; }
    public TableConverter getTableCv() { return tableCv; }
    public ListConverter getListCv() { return listCv; }
    public NoteConverter getNoteCv() { return noteCv; }
    public CaptionConverter getCaptionCv() { return captionCv; }
    public InlineConverter getInlineCv() { return inlineCv; }
    public FieldConverter getFieldCv() { return fieldCv; }
    public DrawConverter getDrawCv() { return drawCv; }
    public TikZConverter getTikZCv() { return tikZCv; }
    public MathConverter getMathCv() { return mathCv; }
    public Info getInfo() { return info; }
	
	
    // fill out inner converter method
    public void convertInner() throws IOException {
        sTargetFileName = Misc.trimDocumentName(sTargetFileName,".tex");
        String sSafeTargetFileName = new ExportNameCollection(true).getExportName(sTargetFileName);
        imageConverter.setBaseFileName(sSafeTargetFileName+"-img");
        if (config.saveImagesInSubdir()) {
        	imageConverter.setUseSubdir(sSafeTargetFileName+"-img");
        }
		
        // Set graphics formats depending on backend
        if (config.backend()==LaTeXConfig.PDFTEX || config.backend()==LaTeXConfig.XETEX) {
            imageConverter.setDefaultFormat(MIMETypes.PNG);
            imageConverter.setDefaultVectorFormat(MIMETypes.PDF);
            imageConverter.addAcceptedFormat(MIMETypes.JPEG);
        }
        else if (config.backend()==LaTeXConfig.DVIPS) {
            imageConverter.setDefaultFormat(MIMETypes.EPS);
        }
        // Other values: keep original format
		
        // Inject user sequence names for tables and figures into OfficeReader
        if (config.tableSequenceName().length()>0) {
            ofr.addTableSequenceName(config.tableSequenceName());
        }
        if (config.figureSequenceName().length()>0) {
            ofr.addFigureSequenceName(config.figureSequenceName());
        }
		
        // Create helpers
        if (config.backend()!=LaTeXConfig.XETEX) {
            i18n = new ClassicI18n(ofr,config,this);        	
        }
        else {
            i18n = new XeTeXI18n(ofr,config,this);        	        	
        }
        colorCv = new ColorConverter(ofr,config,this);
        microtypeCv = new MicrotypeConverter(ofr,config,this);
        frameSc = new FrameStyleConverter(ofr,config,this);
        charSc = new CharStyleConverter(ofr,config,this);
        headingSc = new HeadingStyleConverter(ofr,config,this);
        sectionSc = new SectionStyleConverter(ofr,config,this);
        pageSc = new PageStyleConverter(ofr,config,this);
        blockCv = new BlockConverter(ofr,config,this);
        parCv = new ParConverter(ofr,config,this);
        headingCv = new HeadingConverter(ofr,config,this);
        indexCv = new IndexConverter(ofr,config,this);
        bibCv = new BibConverter(ofr,config,this);
        sectionCv = new SectionConverter(ofr,config,this);
        tableCv = new TableConverter(ofr,config,this);
        listCv = new ListConverter(ofr,config,this);
        noteCv = new NoteConverter(ofr,config,this);
        captionCv = new CaptionConverter(ofr,config,this);
        inlineCv = new InlineConverter(ofr,config,this);
        fieldCv = new FieldConverter(ofr,config,this);
        drawCv = new DrawConverter(ofr,config,this);
        tikZCv = new TikZConverter(ofr,config,this);
        mathCv = new MathConverter(ofr,config,this);
        info = new Info(ofr,config,this);

        // Create master document and add this
        this.texDoc = new LaTeXDocument(sTargetFileName,config.wrapLinesAfter(),true);
        if (config.backend()!=LaTeXConfig.XETEX) {
            texDoc.setEncoding(ClassicI18n.writeJavaEncoding(config.inputencoding()));        	
        }
        else {
            texDoc.setEncoding("UTF-8");        	
        	
        }
        converterResult.addDocument(texDoc);

        // Setup context.
        // The default language is specified in the default paragraph style:
        mainContext = new Context();
        mainContext.resetFormattingFromStyle(ofr.getDefaultParStyle());
        pageSc.updateContext(mainContext);
		
        // Create main LaTeXDocumentPortions
        LaTeXPacman packages = new LaTeXPacman(false);
        LaTeXDocumentPortion declarations = new LaTeXDocumentPortion(false);
        LaTeXDocumentPortion body = new LaTeXDocumentPortion(true);
        
        // Create additional data for the preamble
        globalOptions = new CSVList(',');
        globalOptions.parse(config.globalOptions());

        // Traverse the content
        Element content = ofr.getContent();
        blockCv.traverseBlockText(content,body,mainContext);
        noteCv.insertEndnotes(body);
        
        // Always use calc.sty
        packages.usepackage("calc");
        // Add declarations from our helpers
        i18n.appendDeclarations(packages,declarations);
        	// common: usepackage tipa, tipx, bbding, ifsym, pifont, eurosym, amsmath, wasysym, amssymb, amsfonts
        	// classic: usepackage inputenc, babel, textcomp, fontenc, cmbright, ccfonts, eulervm, iwona, kurier, anttor,
        	// kmath, kerkis, fouriernc, pxfonts, mathpazo, mathpple, txfonts, mathptmx, arev, mathdesign, fourier
        	// xetex: usepackage fontspec, mathspec, xepersian
        colorCv.appendDeclarations(packages,declarations); // usepackage xolor
        microtypeCv.appendDeclarations(packages,declarations); // usepackage microtype, letterspace
        frameSc.appendDeclarations(packages,declarations); // usepackage longfbox
        noteCv.appendDeclarations(packages,declarations); // usepackage endnotes, perpage
        headingSc.appendDeclarations(packages,declarations); // usepackage titlesec
        sectionSc.appendDeclarations(packages,declarations); // usepackage multicol
        charSc.appendDeclarations(packages,declarations); // usepackage ulem
        headingCv.appendDeclarations(packages,declarations); // no packages
        parCv.appendDeclarations(packages,declarations); // no packages
        pageSc.appendDeclarations(packages,declarations); // usepackage geometry, fancyhdr
        blockCv.appendDeclarations(packages,declarations); // no packages
        indexCv.appendDeclarations(packages,declarations); // usepackage makeidx
        bibCv.appendDeclarations(packages,declarations); // no packages
        sectionCv.appendDeclarations(packages,declarations); // no packages
        tableCv.appendDeclarations(packages,declarations); // usepackage array, longtable, supertabular, tabulary, hhline, colortbl
        listCv.appendDeclarations(packages,declarations); // usepackage enumitem
        captionCv.appendDeclarations(packages,declarations); // usepackage caption
        inlineCv.appendDeclarations(packages,declarations); // no packages
        fieldCv.appendDeclarations(packages,declarations); // lastpage, titleref, hyperref 
        drawCv.appendDeclarations(packages,declarations); // usepackage graphicx 
        tikZCv.appendDeclarations(packages,declarations); // usepakcage tikz
        mathCv.appendDeclarations(packages,declarations); // 

        // Add custom preamble
        String sCustomPreamble = config.getCustomPreamble();
        if (sCustomPreamble.length()>0) {
        	declarations.append(sCustomPreamble).nl();
        }

        // Set \title, \author and \date (for \maketitle)
        createMeta("title",metaData.getTitle(),declarations);
        if (config.metadata()) {
            createMeta("author",metaData.getCreator(),declarations);
            // According to the spec, the date has the format YYYY-MM-DDThh:mm:ss
            String sDate = metaData.getDate();
            if (sDate!=null) {
            	createMeta("date",Misc.dateOnly(sDate),declarations);
            }
        }
		
        // Add font size documentclass option if so configured
        List<String> fontSizes = config.fontSizes();
        if (!fontSizes.isEmpty()) {
        	globalOptions.addValue(chooseFontSize(ofr.getFontSize(),fontSizes));
        }
        
        // Assemble the document
        LaTeXDocumentPortion result = texDoc.getContents();

        if (!config.noPreamble()) {
            // Create document class declaration
	        result.append("% This file was converted to LaTeX by Writer2LaTeX ver. "+ConverterFactory.getVersion()).nl()
                  .append("% see http://writer2latex.sourceforge.net for more info").nl();
            result.append("\\documentclass");
            if (!globalOptions.isEmpty()) {
                result.append("[").append(globalOptions.toString()).append("]");
            }
            result.append("{").append(config.documentclass()).append("}").nl();

            result.append(packages)
                  .append(declarations)
                  .append("\\begin{document}").nl();
        }

        result.append(body);

        if (!config.noPreamble()) {
            result.append("\\end{document}").nl();
        }
        else {
            result.append("\\endinput").nl();
        }
		
        // Add BibTeX document if there's any bibliographic data
        if (bibCv.getBibTeXDocument()!=null) {
            converterResult.addDocument(bibCv.getBibTeXDocument());
        }
    }
    
	private String chooseFontSize(String sFontSize, List<String> available) {
		String sPrevSize = null;
		for (String sSize : available) {
			if (sPrevSize!=null) {
				String sMidSize = Calc.multiply("50%",Calc.add(sPrevSize,sSize));
				if (Calc.isLessThan(sFontSize,sMidSize)) {
				    return sPrevSize;
				}
			}
			sPrevSize = sSize;
		}
		return sPrevSize;
	}
	
    private void createMeta(String sName, String sValue,LaTeXDocumentPortion ldp) {
        if (sValue!=null && !sValue.isEmpty()) {
	        // Meta data is assumed to be in the default language:
	        ldp.append("\\"+sName+"{"+i18n.convert(sValue,false,mainContext.getLang())+"}").nl();
        }
    }

}