/**
 * Video test suite
 */

#include <stdio.h>
#include <string.h>

/* Visual Studio 2008 doesn't have stdint.h */
#if defined(_MSC_VER) && _MSC_VER <= 1500
#define UINT8_MAX   ~(Uint8)0
#define UINT16_MAX  ~(Uint16)0
#define UINT32_MAX  ~(Uint32)0
#define UINT64_MAX  ~(Uint64)0
#else
#include <stdint.h>
#endif

#include "SDL.h"
#include "SDL_test.h"

/* Private helpers */

/*
 * Create a test window
 */
SDL_Window *_createVideoSuiteTestWindow(const char *title)
{
    SDL_Window *window;
    int x, y, w, h;
    SDL_WindowFlags flags;
    SDL_bool needs_renderer = SDL_FALSE;

    /* Standard window */
    x = SDLTest_RandomIntegerInRange(1, 100);
    y = SDLTest_RandomIntegerInRange(1, 100);
    w = SDLTest_RandomIntegerInRange(320, 1024);
    h = SDLTest_RandomIntegerInRange(320, 768);
    flags = SDL_WINDOW_SHOWN | SDL_WINDOW_RESIZABLE | SDL_WINDOW_BORDERLESS;

    window = SDL_CreateWindow(title, x, y, w, h, flags);
    SDLTest_AssertPass("Call to SDL_CreateWindow('Title',%d,%d,%d,%d,%d)", x, y, w, h, flags);
    SDLTest_AssertCheck(window != NULL, "Validate that returned window struct is not NULL");

    /* Wayland and XWayland windows require that a frame be presented before they are fully mapped and visible onscreen.
     * This is required for the mouse/keyboard grab tests to pass.
     */
    if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "wayland") == 0) {
        needs_renderer = SDL_TRUE;
    } else if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "x11") == 0) {
        /* Try to detect if the x11 driver is running under XWayland */
        const char *session_type = SDL_getenv("XDG_SESSION_TYPE");
        if (session_type && SDL_strcasecmp(session_type, "wayland") == 0) {
            needs_renderer = SDL_TRUE;
        }
    }

    if (needs_renderer) {
        SDL_Renderer *renderer = SDL_CreateRenderer(window, -1, 0);
        if (renderer) {
            SDL_SetRenderDrawColor(renderer, 0x00, 0x00, 0x00, 0xFF);
            SDL_RenderClear(renderer);
            SDL_RenderPresent(renderer);

            /* Some desktops don't display the window immediately after presentation,
             * so delay to give the window time to actually appear on the desktop.
             */
            SDL_Delay(100);
        } else {
            SDLTest_Log("Unable to create a renderer, some tests may fail on Wayland/XWayland");
        }
    }

    return window;
}

/*
 * Destroy test window
 */
void _destroyVideoSuiteTestWindow(SDL_Window *window)
{
    if (window) {
        SDL_DestroyWindow(window);
        window = NULL;
        SDLTest_AssertPass("Call to SDL_DestroyWindow()");
    }
}

/* Test case functions */

/**
 * @brief Enable and disable screensaver while checking state
 */
int video_enableDisableScreensaver(void *arg)
{
    SDL_bool initialResult;
    SDL_bool result;

    /* Get current state and proceed according to current state */
    initialResult = SDL_IsScreenSaverEnabled();
    SDLTest_AssertPass("Call to SDL_IsScreenSaverEnabled()");
    if (initialResult == SDL_TRUE) {

        /* Currently enabled: disable first, then enable again */

        /* Disable screensaver and check */
        SDL_DisableScreenSaver();
        SDLTest_AssertPass("Call to SDL_DisableScreenSaver()");
        result = SDL_IsScreenSaverEnabled();
        SDLTest_AssertPass("Call to SDL_IsScreenSaverEnabled()");
        SDLTest_AssertCheck(result == SDL_FALSE, "Verify result from SDL_IsScreenSaverEnabled, expected: %i, got: %i", SDL_FALSE, result);

        /* Enable screensaver and check */
        SDL_EnableScreenSaver();
        SDLTest_AssertPass("Call to SDL_EnableScreenSaver()");
        result = SDL_IsScreenSaverEnabled();
        SDLTest_AssertPass("Call to SDL_IsScreenSaverEnabled()");
        SDLTest_AssertCheck(result == SDL_TRUE, "Verify result from SDL_IsScreenSaverEnabled, expected: %i, got: %i", SDL_TRUE, result);

    } else {

        /* Currently disabled: enable first, then disable again */

        /* Enable screensaver and check */
        SDL_EnableScreenSaver();
        SDLTest_AssertPass("Call to SDL_EnableScreenSaver()");
        result = SDL_IsScreenSaverEnabled();
        SDLTest_AssertPass("Call to SDL_IsScreenSaverEnabled()");
        SDLTest_AssertCheck(result == SDL_TRUE, "Verify result from SDL_IsScreenSaverEnabled, expected: %i, got: %i", SDL_TRUE, result);

        /* Disable screensaver and check */
        SDL_DisableScreenSaver();
        SDLTest_AssertPass("Call to SDL_DisableScreenSaver()");
        result = SDL_IsScreenSaverEnabled();
        SDLTest_AssertPass("Call to SDL_IsScreenSaverEnabled()");
        SDLTest_AssertCheck(result == SDL_FALSE, "Verify result from SDL_IsScreenSaverEnabled, expected: %i, got: %i", SDL_FALSE, result);
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_CreateWindow function using different positions
 */
int video_createWindowVariousPositions(void *arg)
{
    SDL_Window *window;
    const char *title = "video_createWindowVariousPositions Test Window";
    int x, y, w, h;
    int xVariation, yVariation;

    for (xVariation = 0; xVariation < 6; xVariation++) {
        for (yVariation = 0; yVariation < 6; yVariation++) {
            switch (xVariation) {
            default:
            case 0:
                /* Zero X Position */
                x = 0;
                break;
            case 1:
                /* Random X position inside screen */
                x = SDLTest_RandomIntegerInRange(1, 100);
                break;
            case 2:
                /* Random X position outside screen (positive) */
                x = SDLTest_RandomIntegerInRange(10000, 11000);
                break;
            case 3:
                /* Random X position outside screen (negative) */
                x = SDLTest_RandomIntegerInRange(-1000, -100);
                break;
            case 4:
                /* Centered X position */
                x = SDL_WINDOWPOS_CENTERED;
                break;
            case 5:
                /* Undefined X position */
                x = SDL_WINDOWPOS_UNDEFINED;
                break;
            }

            switch (yVariation) {
            default:
            case 0:
                /* Zero X Position */
                y = 0;
                break;
            case 1:
                /* Random X position inside screen */
                y = SDLTest_RandomIntegerInRange(1, 100);
                break;
            case 2:
                /* Random X position outside screen (positive) */
                y = SDLTest_RandomIntegerInRange(10000, 11000);
                break;
            case 3:
                /* Random Y position outside screen (negative) */
                y = SDLTest_RandomIntegerInRange(-1000, -100);
                break;
            case 4:
                /* Centered Y position */
                y = SDL_WINDOWPOS_CENTERED;
                break;
            case 5:
                /* Undefined Y position */
                y = SDL_WINDOWPOS_UNDEFINED;
                break;
            }

            w = SDLTest_RandomIntegerInRange(32, 96);
            h = SDLTest_RandomIntegerInRange(32, 96);
            window = SDL_CreateWindow(title, x, y, w, h, SDL_WINDOW_SHOWN);
            SDLTest_AssertPass("Call to SDL_CreateWindow('Title',%d,%d,%d,%d,SHOWN)", x, y, w, h);
            SDLTest_AssertCheck(window != NULL, "Validate that returned window struct is not NULL");

            /* Clean up */
            _destroyVideoSuiteTestWindow(window);
        }
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_CreateWindow function using different sizes
 */
int video_createWindowVariousSizes(void *arg)
{
    SDL_Window *window;
    const char *title = "video_createWindowVariousSizes Test Window";
    int x, y, w, h;
    int wVariation, hVariation;

    x = SDLTest_RandomIntegerInRange(1, 100);
    y = SDLTest_RandomIntegerInRange(1, 100);
    for (wVariation = 0; wVariation < 3; wVariation++) {
        for (hVariation = 0; hVariation < 3; hVariation++) {
            switch (wVariation) {
            case 0:
                /* Width of 1 */
                w = 1;
                break;
            case 1:
                /* Random "normal" width */
                w = SDLTest_RandomIntegerInRange(320, 1920);
                break;
            case 2:
                /* Random "large" width */
                w = SDLTest_RandomIntegerInRange(2048, 4095);
                break;
            }

            switch (hVariation) {
            case 0:
                /* Height of 1 */
                h = 1;
                break;
            case 1:
                /* Random "normal" height */
                h = SDLTest_RandomIntegerInRange(320, 1080);
                break;
            case 2:
                /* Random "large" height */
                h = SDLTest_RandomIntegerInRange(2048, 4095);
                break;
            }

            window = SDL_CreateWindow(title, x, y, w, h, SDL_WINDOW_SHOWN);
            SDLTest_AssertPass("Call to SDL_CreateWindow('Title',%d,%d,%d,%d,SHOWN)", x, y, w, h);
            SDLTest_AssertCheck(window != NULL, "Validate that returned window struct is not NULL");

            /* Clean up */
            _destroyVideoSuiteTestWindow(window);
        }
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_CreateWindow function using different flags
 */
int video_createWindowVariousFlags(void *arg)
{
    SDL_Window *window;
    const char *title = "video_createWindowVariousFlags Test Window";
    int x, y, w, h;
    int fVariation;
    SDL_WindowFlags flags;

    /* Standard window */
    x = SDLTest_RandomIntegerInRange(1, 100);
    y = SDLTest_RandomIntegerInRange(1, 100);
    w = SDLTest_RandomIntegerInRange(320, 1024);
    h = SDLTest_RandomIntegerInRange(320, 768);

    for (fVariation = 0; fVariation < 14; fVariation++) {
        switch (fVariation) {
        default:
        case 0:
            flags = SDL_WINDOW_FULLSCREEN;
            /* Skip - blanks screen; comment out next line to run test */
            continue;
            break;
        case 1:
            flags = SDL_WINDOW_FULLSCREEN_DESKTOP;
            /* Skip - blanks screen; comment out next line to run test */
            continue;
            break;
        case 2:
            flags = SDL_WINDOW_OPENGL;
            /* Skip - not every video driver supports OpenGL; comment out next line to run test */
            continue;
            break;
        case 3:
            flags = SDL_WINDOW_SHOWN;
            break;
        case 4:
            flags = SDL_WINDOW_HIDDEN;
            break;
        case 5:
            flags = SDL_WINDOW_BORDERLESS;
            break;
        case 6:
            flags = SDL_WINDOW_RESIZABLE;
            break;
        case 7:
            flags = SDL_WINDOW_MINIMIZED;
            break;
        case 8:
            flags = SDL_WINDOW_MAXIMIZED;
            break;
        case 9:
            flags = SDL_WINDOW_MOUSE_GRABBED;
            break;
        case 10:
            flags = SDL_WINDOW_INPUT_FOCUS;
            break;
        case 11:
            flags = SDL_WINDOW_MOUSE_FOCUS;
            break;
        case 12:
            flags = SDL_WINDOW_FOREIGN;
            break;
        case 13:
            flags = SDL_WINDOW_KEYBOARD_GRABBED;
            break;
        }

        window = SDL_CreateWindow(title, x, y, w, h, flags);
        SDLTest_AssertPass("Call to SDL_CreateWindow('Title',%d,%d,%d,%d,%d)", x, y, w, h, flags);
        SDLTest_AssertCheck(window != NULL, "Validate that returned window struct is not NULL");

        /* Clean up */
        _destroyVideoSuiteTestWindow(window);
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_GetWindowFlags function
 */
int video_getWindowFlags(void *arg)
{
    SDL_Window *window;
    const char *title = "video_getWindowFlags Test Window";
    SDL_WindowFlags flags;
    Uint32 actualFlags;

    /* Reliable flag set always set in test window */
    flags = SDL_WINDOW_SHOWN;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (window != NULL) {
        actualFlags = SDL_GetWindowFlags(window);
        SDLTest_AssertPass("Call to SDL_GetWindowFlags()");
        SDLTest_AssertCheck((flags & actualFlags) == flags, "Verify returned value has flags %d set, got: %" SDL_PRIu32, flags, actualFlags);
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_GetNumDisplayModes function
 */
int video_getNumDisplayModes(void *arg)
{
    int result;
    int displayNum;
    int i;

    /* Get number of displays */
    displayNum = SDL_GetNumVideoDisplays();
    SDLTest_AssertPass("Call to SDL_GetNumVideoDisplays()");

    /* Make call for each display */
    for (i = 0; i < displayNum; i++) {
        result = SDL_GetNumDisplayModes(i);
        SDLTest_AssertPass("Call to SDL_GetNumDisplayModes(%d)", i);
        SDLTest_AssertCheck(result >= 1, "Validate returned value from function; expected: >=1; got: %d", result);
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests negative call to SDL_GetNumDisplayModes function
 */
int video_getNumDisplayModesNegative(void *arg)
{
    int result;
    int displayNum;
    int displayIndex;

    /* Get number of displays */
    displayNum = SDL_GetNumVideoDisplays();
    SDLTest_AssertPass("Call to SDL_GetNumVideoDisplays()");

    /* Invalid boundary values */
    displayIndex = SDLTest_RandomSint32BoundaryValue(0, displayNum, SDL_FALSE);
    result = SDL_GetNumDisplayModes(displayIndex);
    SDLTest_AssertPass("Call to SDL_GetNumDisplayModes(%d=out-of-bounds/boundary)", displayIndex);
    SDLTest_AssertCheck(result < 0, "Validate returned value from function; expected: <0; got: %d", result);

    /* Large (out-of-bounds) display index */
    displayIndex = SDLTest_RandomIntegerInRange(-2000, -1000);
    result = SDL_GetNumDisplayModes(displayIndex);
    SDLTest_AssertPass("Call to SDL_GetNumDisplayModes(%d=out-of-bounds/large negative)", displayIndex);
    SDLTest_AssertCheck(result < 0, "Validate returned value from function; expected: <0; got: %d", result);

    displayIndex = SDLTest_RandomIntegerInRange(1000, 2000);
    result = SDL_GetNumDisplayModes(displayIndex);
    SDLTest_AssertPass("Call to SDL_GetNumDisplayModes(%d=out-of-bounds/large positive)", displayIndex);
    SDLTest_AssertCheck(result < 0, "Validate returned value from function; expected: <0; got: %d", result);

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_GetClosestDisplayMode function against current resolution
 */
int video_getClosestDisplayModeCurrentResolution(void *arg)
{
    int result;
    SDL_DisplayMode current;
    SDL_DisplayMode target;
    SDL_DisplayMode closest;
    SDL_DisplayMode *dResult;
    int displayNum;
    int i;
    int variation;

    /* Get number of displays */
    displayNum = SDL_GetNumVideoDisplays();
    SDLTest_AssertPass("Call to SDL_GetNumVideoDisplays()");

    /* Make calls for each display */
    for (i = 0; i < displayNum; i++) {
        SDLTest_Log("Testing against display: %d", i);

        /* Get first display mode to get a sane resolution; this should always work */
        result = SDL_GetDisplayMode(i, 0, &current);
        SDLTest_AssertPass("Call to SDL_GetDisplayMode()");
        SDLTest_AssertCheck(result == 0, "Verify return value, expected: 0, got: %d", result);
        if (result != 0) {
            return TEST_ABORTED;
        }

        /* Set the desired resolution equals to current resolution */
        target.w = current.w;
        target.h = current.h;
        for (variation = 0; variation < 8; variation++) {
            /* Vary constraints on other query parameters */
            target.format = (variation & 1) ? current.format : 0;
            target.refresh_rate = (variation & 2) ? current.refresh_rate : 0;
            target.driverdata = (variation & 4) ? current.driverdata : 0;

            /* Make call */
            dResult = SDL_GetClosestDisplayMode(i, &target, &closest);
            SDLTest_AssertPass("Call to SDL_GetClosestDisplayMode(target=current/variation%d)", variation);
            SDLTest_Assert(dResult != NULL, "Verify returned value is not NULL");

            /* Check that one gets the current resolution back again */
            SDLTest_AssertCheck(closest.w == current.w, "Verify returned width matches current width; expected: %d, got: %d", current.w, closest.w);
            SDLTest_AssertCheck(closest.h == current.h, "Verify returned height matches current height; expected: %d, got: %d", current.h, closest.h);
            /* NOLINTBEGIN(clang-analyzer-core.NullDereference): Checked earlier for NULL */
            SDLTest_AssertCheck(closest.w == dResult->w, "Verify return value matches assigned value; expected: %d, got: %d", closest.w, dResult->w);
            SDLTest_AssertCheck(closest.h == dResult->h, "Verify return value matches assigned value; expected: %d, got: %d", closest.h, dResult->h);
            /* NOLINTEND(clang-analyzer-core.NullDereference) */
        }
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests the functionality of the SDL_GetClosestDisplayMode function against random resolution
 */
int video_getClosestDisplayModeRandomResolution(void *arg)
{
    SDL_DisplayMode target;
    SDL_DisplayMode closest;
    int displayNum;
    int i;
    int variation;

    /* Get number of displays */
    displayNum = SDL_GetNumVideoDisplays();
    SDLTest_AssertPass("Call to SDL_GetNumVideoDisplays()");

    /* Make calls for each display */
    for (i = 0; i < displayNum; i++) {
        SDLTest_Log("Testing against display: %d", i);

        for (variation = 0; variation < 16; variation++) {

            /* Set random constraints */
            target.w = (variation & 1) ? SDLTest_RandomIntegerInRange(1, 4096) : 0;
            target.h = (variation & 2) ? SDLTest_RandomIntegerInRange(1, 4096) : 0;
            target.format = (variation & 4) ? SDLTest_RandomIntegerInRange(1, 10) : 0;
            target.refresh_rate = (variation & 8) ? SDLTest_RandomIntegerInRange(25, 120) : 0;
            target.driverdata = 0;

            /* Make call; may or may not find anything, so don't validate any further */
            SDL_GetClosestDisplayMode(i, &target, &closest);
            SDLTest_AssertPass("Call to SDL_GetClosestDisplayMode(target=random/variation%d)", variation);
        }
    }

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowBrightness
 *
* @sa http://wiki.libsdl.org/SDL_GetWindowBrightness
 */
int
video_getWindowBrightness(void *arg)
{
  SDL_Window* window;
  const char* title = "video_getWindowBrightness Test Window";
  float result;

  /* Call against new test window */
  window = _createVideoSuiteTestWindow(title);
  if (window != NULL) {
      result = SDL_GetWindowBrightness(window);
      SDLTest_AssertPass("Call to SDL_GetWindowBrightness()");
      SDLTest_AssertCheck(result >= 0.0 && result <= 1.0, "Validate range of result value; expected: [0.0, 1.0], got: %f", result);
  }

  /* Clean up */
  _destroyVideoSuiteTestWindow(window);

  return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowBrightness with invalid input
 *
* @sa http://wiki.libsdl.org/SDL_GetWindowBrightness
 */
int
video_getWindowBrightnessNegative(void *arg)
{
  const char *invalidWindowError = "Invalid window";
  char *lastError;
  float result;

  /* Call against invalid window */
  result = SDL_GetWindowBrightness(NULL);
  SDLTest_AssertPass("Call to SDL_GetWindowBrightness(window=NULL)");
  SDLTest_AssertCheck(result == 1.0, "Validate result value; expected: 1.0, got: %f", result);
  lastError = (char *)SDL_GetError();
  SDLTest_AssertPass("SDL_GetError()");
  SDLTest_AssertCheck(lastError != NULL, "Verify error message is not NULL");
  if (lastError != NULL) {
      SDLTest_AssertCheck(SDL_strcmp(lastError, invalidWindowError) == 0,
         "SDL_GetError(): expected message '%s', was message: '%s'",
         invalidWindowError,
         lastError);
  }

  return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowDisplayMode
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowDisplayMode
 */
int video_getWindowDisplayMode(void *arg)
{
    SDL_Window *window;
    const char *title = "video_getWindowDisplayMode Test Window";
    SDL_DisplayMode mode;
    int result;

    /* Invalidate part of the mode content so we can check values later */
    mode.w = -1;
    mode.h = -1;
    mode.refresh_rate = -1;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (window) {
        result = SDL_GetWindowDisplayMode(window, &mode);
        SDLTest_AssertPass("Call to SDL_GetWindowDisplayMode()");
        SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);
        SDLTest_AssertCheck(mode.w > 0, "Validate mode.w content; expected: >0, got: %d", mode.w);
        SDLTest_AssertCheck(mode.h > 0, "Validate mode.h content; expected: >0, got: %d", mode.h);
        SDLTest_AssertCheck(mode.refresh_rate > 0, "Validate mode.refresh_rate content; expected: >0, got: %d", mode.refresh_rate);
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    return TEST_COMPLETED;
}

/* Helper function that checks for an 'Invalid window' error */
void _checkInvalidWindowError(void)
{
    const char *invalidWindowError = "Invalid window";
    char *lastError;

    lastError = (char *)SDL_GetError();
    SDLTest_AssertPass("SDL_GetError()");
    SDLTest_AssertCheck(lastError != NULL, "Verify error message is not NULL");
    if (lastError != NULL) {
        SDLTest_AssertCheck(SDL_strcmp(lastError, invalidWindowError) == 0,
                            "SDL_GetError(): expected message '%s', was message: '%s'",
                            invalidWindowError,
                            lastError);
        SDL_ClearError();
        SDLTest_AssertPass("Call to SDL_ClearError()");
    }
}

/**
 * @brief Tests call to SDL_GetWindowDisplayMode with invalid input
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowDisplayMode
 */
int video_getWindowDisplayModeNegative(void *arg)
{
    const char *expectedError = "Parameter 'mode' is invalid";
    char *lastError;
    SDL_Window *window;
    const char *title = "video_getWindowDisplayModeNegative Test Window";
    SDL_DisplayMode mode;
    int result;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (window != NULL) {
        result = SDL_GetWindowDisplayMode(window, NULL);
        SDLTest_AssertPass("Call to SDL_GetWindowDisplayMode(...,mode=NULL)");
        SDLTest_AssertCheck(result == -1, "Validate result value; expected: -1, got: %d", result);
        lastError = (char *)SDL_GetError();
        SDLTest_AssertPass("SDL_GetError()");
        SDLTest_AssertCheck(lastError != NULL, "Verify error message is not NULL");
        if (lastError != NULL) {
            SDLTest_AssertCheck(SDL_strcmp(lastError, expectedError) == 0,
                                "SDL_GetError(): expected message '%s', was message: '%s'",
                                expectedError,
                                lastError);
        }
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Call against invalid window */
    result = SDL_GetWindowDisplayMode(NULL, &mode);
    SDLTest_AssertPass("Call to SDL_GetWindowDisplayMode(window=NULL,...)");
    SDLTest_AssertCheck(result == -1, "Validate result value; expected: -1, got: %d", result);
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowGammaRamp
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowGammaRamp
 */
int
video_getWindowGammaRamp(void *arg)
{
  SDL_Window* window;
  const char* title = "video_getWindowGammaRamp Test Window";
  Uint16 red[256];
  Uint16 green[256];
  Uint16 blue[256];
  int result;

  /* Call against new test window */
  window = _createVideoSuiteTestWindow(title);
  if (!window) return TEST_ABORTED;

  /* Retrieve no channel */
  result = SDL_GetWindowGammaRamp(window, NULL, NULL, NULL);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(all NULL)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  /* Retrieve single channel */
  result = SDL_GetWindowGammaRamp(window, red, NULL, NULL);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(r)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  result = SDL_GetWindowGammaRamp(window, NULL, green, NULL);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(g)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  result = SDL_GetWindowGammaRamp(window, NULL, NULL, blue);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(b)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  /* Retrieve two channels */
  result = SDL_GetWindowGammaRamp(window, red, green, NULL);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(r, g)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  result = SDL_GetWindowGammaRamp(window, NULL, green, blue);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(g,b)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  result = SDL_GetWindowGammaRamp(window, red, NULL, blue);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(r,b)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  /* Retrieve all channels */
  result = SDL_GetWindowGammaRamp(window, red, green, blue);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(r,g,b)");
  SDLTest_AssertCheck(result == 0, "Validate result value; expected: 0, got: %d", result);

  /* Clean up */
  _destroyVideoSuiteTestWindow(window);

  return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowGammaRamp with invalid input
 *
* @sa http://wiki.libsdl.org/SDL_GetWindowGammaRamp
 */
int
video_getWindowGammaRampNegative(void *arg)
{
  Uint16 red[256];
  Uint16 green[256];
  Uint16 blue[256];
  int result;

  SDL_ClearError();
  SDLTest_AssertPass("Call to SDL_ClearError()");

  /* Call against invalid window */
  result = SDL_GetWindowGammaRamp(NULL, red, green, blue);
  SDLTest_AssertPass("Call to SDL_GetWindowGammaRamp(window=NULL,r,g,b)");
  SDLTest_AssertCheck(result == -1, "Validate result value; expected: -1, got: %i", result);
  _checkInvalidWindowError();

  return TEST_COMPLETED;
}

/* Helper for setting and checking the window mouse grab state */
void _setAndCheckWindowMouseGrabState(SDL_Window *window, SDL_bool desiredState)
{
    SDL_bool currentState;

    /* Set state */
    SDL_SetWindowMouseGrab(window, desiredState);
    SDLTest_AssertPass("Call to SDL_SetWindowMouseGrab(%s)", (desiredState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE");

    /* Get and check state */
    currentState = SDL_GetWindowMouseGrab(window);
    SDLTest_AssertPass("Call to SDL_GetWindowMouseGrab()");
    SDLTest_AssertCheck(
        currentState == desiredState,
        "Validate returned state; expected: %s, got: %s",
        (desiredState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE",
        (currentState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE");

    if (desiredState) {
        SDLTest_AssertCheck(
            SDL_GetGrabbedWindow() == window,
            "Grabbed window should be to our window");
        SDLTest_AssertCheck(
            SDL_GetWindowGrab(window),
            "SDL_GetWindowGrab() should return SDL_TRUE");
        SDLTest_AssertCheck(
            SDL_GetWindowFlags(window) & SDL_WINDOW_MOUSE_GRABBED,
            "SDL_WINDOW_MOUSE_GRABBED should be set");
    } else {
        SDLTest_AssertCheck(
            !(SDL_GetWindowFlags(window) & SDL_WINDOW_MOUSE_GRABBED),
            "SDL_WINDOW_MOUSE_GRABBED should be unset");
    }
}

/* Helper for setting and checking the window keyboard grab state */
void _setAndCheckWindowKeyboardGrabState(SDL_Window *window, SDL_bool desiredState)
{
    SDL_bool currentState;

    /* Set state */
    SDL_SetWindowKeyboardGrab(window, desiredState);
    SDLTest_AssertPass("Call to SDL_SetWindowKeyboardGrab(%s)", (desiredState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE");

    /* Get and check state */
    currentState = SDL_GetWindowKeyboardGrab(window);
    SDLTest_AssertPass("Call to SDL_GetWindowKeyboardGrab()");
    SDLTest_AssertCheck(
        currentState == desiredState,
        "Validate returned state; expected: %s, got: %s",
        (desiredState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE",
        (currentState == SDL_FALSE) ? "SDL_FALSE" : "SDL_TRUE");

    if (desiredState) {
        SDLTest_AssertCheck(
            SDL_GetGrabbedWindow() == window,
            "Grabbed window should be set to our window");
        SDLTest_AssertCheck(
            SDL_GetWindowGrab(window),
            "SDL_GetWindowGrab() should return SDL_TRUE");
        SDLTest_AssertCheck(
            SDL_GetWindowFlags(window) & SDL_WINDOW_KEYBOARD_GRABBED,
            "SDL_WINDOW_KEYBOARD_GRABBED should be set");
    } else {
        SDLTest_AssertCheck(
            !(SDL_GetWindowFlags(window) & SDL_WINDOW_KEYBOARD_GRABBED),
            "SDL_WINDOW_KEYBOARD_GRABBED should be unset");
    }
}

/**
 * @brief Tests keyboard and mouse grab support
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowGrab
 * @sa http://wiki.libsdl.org/SDL_SetWindowGrab
 */
int video_getSetWindowGrab(void *arg)
{
    const char *title = "video_getSetWindowGrab Test Window";
    SDL_Window *window;
    SDL_bool originalMouseState, originalKeyboardState;
    SDL_bool hasFocusGained = SDL_FALSE;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    /* Need to raise the window to have and SDL_EVENT_WINDOW_FOCUS_GAINED,
     * so that the window gets the flags SDL_WINDOW_INPUT_FOCUS,
     * so that it can be "grabbed" */
    SDL_RaiseWindow(window);

    {
        SDL_Event evt;
        SDL_zero(evt);
        while (SDL_PollEvent(&evt)) {
            if (evt.type == SDL_WINDOWEVENT) {
                if (evt.window.event == SDL_WINDOWEVENT_FOCUS_GAINED) {
                    hasFocusGained = SDL_TRUE;
                }
            }
        }
    }

    SDLTest_AssertCheck(hasFocusGained == SDL_TRUE, "Expected window with focus");

    /* Get state */
    originalMouseState = SDL_GetWindowMouseGrab(window);
    SDLTest_AssertPass("Call to SDL_GetWindowMouseGrab()");
    originalKeyboardState = SDL_GetWindowKeyboardGrab(window);
    SDLTest_AssertPass("Call to SDL_GetWindowKeyboardGrab()");

    /* F */
    _setAndCheckWindowKeyboardGrabState(window, SDL_FALSE);
    _setAndCheckWindowMouseGrabState(window, SDL_FALSE);
    SDLTest_AssertCheck(!SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab should return SDL_FALSE");
    SDLTest_AssertCheck(SDL_GetGrabbedWindow() == NULL,
                        "Expected NULL grabbed window");

    /* F --> F */
    _setAndCheckWindowMouseGrabState(window, SDL_FALSE);
    _setAndCheckWindowKeyboardGrabState(window, SDL_FALSE);
    SDLTest_AssertCheck(SDL_GetGrabbedWindow() == NULL,
                        "Expected NULL grabbed window");

    /* F --> T */
    _setAndCheckWindowMouseGrabState(window, SDL_TRUE);
    _setAndCheckWindowKeyboardGrabState(window, SDL_TRUE);
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");

    /* T --> T */
    _setAndCheckWindowKeyboardGrabState(window, SDL_TRUE);
    _setAndCheckWindowMouseGrabState(window, SDL_TRUE);
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");

    /* M: T --> F */
    /* K: T --> T */
    _setAndCheckWindowKeyboardGrabState(window, SDL_TRUE);
    _setAndCheckWindowMouseGrabState(window, SDL_FALSE);
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");

    /* M: F --> T */
    /* K: T --> F */
    _setAndCheckWindowMouseGrabState(window, SDL_TRUE);
    _setAndCheckWindowKeyboardGrabState(window, SDL_FALSE);
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");

    /* M: T --> F */
    /* K: F --> F */
    _setAndCheckWindowMouseGrabState(window, SDL_FALSE);
    _setAndCheckWindowKeyboardGrabState(window, SDL_FALSE);
    SDLTest_AssertCheck(!SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_FALSE");
    SDLTest_AssertCheck(SDL_GetGrabbedWindow() == NULL,
                        "Expected NULL grabbed window");

    /* Using the older SDL_SetWindowGrab API should only grab mouse by default */
    SDL_SetWindowGrab(window, SDL_TRUE);
    SDLTest_AssertPass("Call to SDL_SetWindowGrab(SDL_TRUE)");
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");
    SDLTest_AssertCheck(SDL_GetWindowMouseGrab(window),
                        "SDL_GetWindowMouseGrab() should return SDL_TRUE");
    SDLTest_AssertCheck(!SDL_GetWindowKeyboardGrab(window),
                        "SDL_GetWindowKeyboardGrab() should return SDL_FALSE");
    SDL_SetWindowGrab(window, SDL_FALSE);
    SDLTest_AssertCheck(!SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_FALSE");
    SDLTest_AssertCheck(!SDL_GetWindowMouseGrab(window),
                        "SDL_GetWindowMouseGrab() should return SDL_FALSE");
    SDLTest_AssertCheck(!SDL_GetWindowKeyboardGrab(window),
                        "SDL_GetWindowKeyboardGrab() should return SDL_FALSE");

    /* Now test with SDL_HINT_GRAB_KEYBOARD set. We should get keyboard grab now. */
    SDL_SetHint(SDL_HINT_GRAB_KEYBOARD, "1");
    SDL_SetWindowGrab(window, SDL_TRUE);
    SDLTest_AssertPass("Call to SDL_SetWindowGrab(SDL_TRUE)");
    SDLTest_AssertCheck(SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_TRUE");
    SDLTest_AssertCheck(SDL_GetWindowMouseGrab(window),
                        "SDL_GetWindowMouseGrab() should return SDL_TRUE");
    SDLTest_AssertCheck(SDL_GetWindowKeyboardGrab(window),
                        "SDL_GetWindowKeyboardGrab() should return SDL_TRUE");
    SDL_SetWindowGrab(window, SDL_FALSE);
    SDLTest_AssertCheck(!SDL_GetWindowGrab(window),
                        "SDL_GetWindowGrab() should return SDL_FALSE");
    SDLTest_AssertCheck(!SDL_GetWindowMouseGrab(window),
                        "SDL_GetWindowMouseGrab() should return SDL_FALSE");
    SDLTest_AssertCheck(!SDL_GetWindowKeyboardGrab(window),
                        "SDL_GetWindowKeyboardGrab() should return SDL_FALSE");

    /* Negative tests */
    SDL_GetWindowGrab(NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowGrab(window=NULL)");
    _checkInvalidWindowError();

    SDL_GetWindowKeyboardGrab(NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowKeyboardGrab(window=NULL)");
    _checkInvalidWindowError();

    SDL_SetWindowGrab(NULL, SDL_FALSE);
    SDLTest_AssertPass("Call to SDL_SetWindowGrab(window=NULL,SDL_FALSE)");
    _checkInvalidWindowError();

    SDL_SetWindowKeyboardGrab(NULL, SDL_FALSE);
    SDLTest_AssertPass("Call to SDL_SetWindowKeyboardGrab(window=NULL,SDL_FALSE)");
    _checkInvalidWindowError();

    SDL_SetWindowGrab(NULL, SDL_TRUE);
    SDLTest_AssertPass("Call to SDL_SetWindowGrab(window=NULL,SDL_TRUE)");
    _checkInvalidWindowError();

    SDL_SetWindowKeyboardGrab(NULL, SDL_TRUE);
    SDLTest_AssertPass("Call to SDL_SetWindowKeyboardGrab(window=NULL,SDL_TRUE)");
    _checkInvalidWindowError();

    /* Restore state */
    _setAndCheckWindowMouseGrabState(window, originalMouseState);
    _setAndCheckWindowKeyboardGrabState(window, originalKeyboardState);

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowID and SDL_GetWindowFromID
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowID
 * @sa http://wiki.libsdl.org/SDL_SetWindowFromID
 */
int video_getWindowId(void *arg)
{
    const char *title = "video_getWindowId Test Window";
    SDL_Window *window;
    SDL_Window *result;
    Uint32 id, randomId;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    /* Get ID */
    id = SDL_GetWindowID(window);
    SDLTest_AssertPass("Call to SDL_GetWindowID()");

    /* Get window from ID */
    result = SDL_GetWindowFromID(id);
    SDLTest_AssertPass("Call to SDL_GetWindowID(%" SDL_PRIu32 ")", id);
    SDLTest_AssertCheck(result == window, "Verify result matches window pointer");

    /* Get window from random large ID, no result check */
    randomId = SDLTest_RandomIntegerInRange(UINT8_MAX, UINT16_MAX);
    result = SDL_GetWindowFromID(randomId);
    SDLTest_AssertPass("Call to SDL_GetWindowID(%" SDL_PRIu32 "/random_large)", randomId);

    /* Get window from 0 and Uint32 max ID, no result check */
    result = SDL_GetWindowFromID(0);
    SDLTest_AssertPass("Call to SDL_GetWindowID(0)");
    result = SDL_GetWindowFromID(UINT32_MAX);
    SDLTest_AssertPass("Call to SDL_GetWindowID(UINT32_MAX)");

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Get window from ID for closed window */
    result = SDL_GetWindowFromID(id);
    SDLTest_AssertPass("Call to SDL_GetWindowID(%" SDL_PRIu32 "/closed_window)", id);
    SDLTest_AssertCheck(result == NULL, "Verify result is NULL");

    /* Negative test */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    id = SDL_GetWindowID(NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowID(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowPixelFormat
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowPixelFormat
 */
int video_getWindowPixelFormat(void *arg)
{
    const char *title = "video_getWindowPixelFormat Test Window";
    SDL_Window *window;
    Uint32 format;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    /* Get format */
    format = SDL_GetWindowPixelFormat(window);
    SDLTest_AssertPass("Call to SDL_GetWindowPixelFormat()");
    SDLTest_AssertCheck(format != SDL_PIXELFORMAT_UNKNOWN, "Verify that returned format is valid; expected: != %d, got: %" SDL_PRIu32, SDL_PIXELFORMAT_UNKNOWN, format);

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Negative test */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    format = SDL_GetWindowPixelFormat(NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowPixelFormat(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}


static SDL_bool getPositionFromEvent(int *x, int *y)
{
    SDL_bool ret = SDL_FALSE;
    SDL_Event evt;
    SDL_zero(evt);
    while (SDL_PollEvent(&evt)) {
        if (evt.type == SDL_WINDOWEVENT && evt.window.event == SDL_WINDOWEVENT_MOVED) {
            *x = evt.window.data1;
            *y = evt.window.data2;
            ret = SDL_TRUE;
        }
    }
    return ret;
}

static SDL_bool getSizeFromEvent(int *w, int *h)
{
    SDL_bool ret = SDL_FALSE;
    SDL_Event evt;
    SDL_zero(evt);
    while (SDL_PollEvent(&evt)) {
        if (evt.type == SDL_WINDOWEVENT && evt.window.event == SDL_WINDOWEVENT_RESIZED) {
            *w = evt.window.data1;
            *h = evt.window.data2;
            ret = SDL_TRUE;
        }
    }
    return ret;
}

/**
 * @brief Tests call to SDL_GetWindowPosition and SDL_SetWindowPosition
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowPosition
 * @sa http://wiki.libsdl.org/SDL_SetWindowPosition
 */
int video_getSetWindowPosition(void *arg)
{
    const char *title = "video_getSetWindowPosition Test Window";
    SDL_Window *window;
    int maxxVariation, maxyVariation;
    int xVariation, yVariation;
    int referenceX, referenceY;
    int currentX, currentY;
    int desiredX, desiredY;
    SDL_Rect display_bounds;

    if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "wayland") == 0) {
        SDLTest_Log("Skipping test: wayland does not support window positioning");
        return TEST_SKIPPED;
    }

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "x11") == 0) {
        /* The X11 server allows arbitrary window placement, but compositing
         * window managers such as GNOME and KDE force windows to be within
         * desktop bounds.
         */
        maxxVariation = 2;
        maxyVariation = 2;

        SDL_GetDisplayUsableBounds(SDL_GetWindowDisplayIndex(window), &display_bounds);
    } else if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "cocoa") == 0) {
        /* Platform doesn't allow windows with negative Y desktop bounds */
        maxxVariation = 4;
        maxyVariation = 3;

        SDL_GetDisplayUsableBounds(SDL_GetWindowDisplayIndex(window), &display_bounds);
    } else {
        /* Platform allows windows to be placed out of bounds */
        maxxVariation = 4;
        maxyVariation = 4;

        SDL_GetDisplayBounds(SDL_GetWindowDisplayIndex(window), &display_bounds);
    }

    for (xVariation = 0; xVariation < maxxVariation; xVariation++) {
        for (yVariation = 0; yVariation < maxyVariation; yVariation++) {
            switch (xVariation) {
            default:
            case 0:
                /* Zero X Position */
                desiredX = display_bounds.x > 0 ? display_bounds.x : 0;
                break;
            case 1:
                /* Random X position inside screen */
                desiredX = SDLTest_RandomIntegerInRange(display_bounds.x + 1, display_bounds.x + 100);
                break;
            case 2:
                /* Random X position outside screen (positive) */
                desiredX = SDLTest_RandomIntegerInRange(10000, 11000);
                break;
            case 3:
                /* Random X position outside screen (negative) */
                desiredX = SDLTest_RandomIntegerInRange(-1000, -100);
                break;
            }

            switch (yVariation) {
            default:
            case 0:
                /* Zero Y Position */
                desiredY = display_bounds.y > 0 ? display_bounds.y : 0;
                break;
            case 1:
                /* Random Y position inside screen */
                desiredY = SDLTest_RandomIntegerInRange(display_bounds.y + 1, display_bounds.y + 100);
                break;
            case 2:
                /* Random Y position outside screen (positive) */
                desiredY = SDLTest_RandomIntegerInRange(10000, 11000);
                break;
            case 3:
                /* Random Y position outside screen (negative) */
                desiredY = SDLTest_RandomIntegerInRange(-1000, -100);
                break;
            }

            /* Set position */
            SDL_SetWindowPosition(window, desiredX, desiredY);
            SDLTest_AssertPass("Call to SDL_SetWindowPosition(...,%d,%d)", desiredX, desiredY);

            /* Get position */
            currentX = desiredX + 1;
            currentY = desiredY + 1;
            SDL_GetWindowPosition(window, &currentX, &currentY);
            SDLTest_AssertPass("Call to SDL_GetWindowPosition()");

            if (desiredX == currentX && desiredY == currentY) {
                SDLTest_AssertCheck(desiredX == currentX, "Verify returned X position; expected: %d, got: %d", desiredX, currentX);
                SDLTest_AssertCheck(desiredY == currentY, "Verify returned Y position; expected: %d, got: %d", desiredY, currentY);
            } else {
                SDL_bool hasEvent;
                /* SDL_SetWindowPosition() and SDL_SetWindowSize() will make requests of the window manager and set the internal position and size,
                 * and then we get events signaling what actually happened, and they get passed on to the application if they're not what we expect. */
                desiredX = currentX + 1;
                desiredY = currentY + 1;
                hasEvent = getPositionFromEvent(&desiredX, &desiredY);
                SDLTest_AssertCheck(hasEvent == SDL_TRUE, "Changing position was not honored by WM, checking present of SDL_WINDOWEVENT_MOVED");
                if (hasEvent) {
                    SDLTest_AssertCheck(desiredX == currentX, "Verify returned X position is the position from SDL event; expected: %d, got: %d", desiredX, currentX);
                    SDLTest_AssertCheck(desiredY == currentY, "Verify returned Y position is the position from SDL event; expected: %d, got: %d", desiredY, currentY);
                }
            }

            /* Get position X */
            currentX = desiredX + 1;
            SDL_GetWindowPosition(window, &currentX, NULL);
            SDLTest_AssertPass("Call to SDL_GetWindowPosition(&y=NULL)");
            SDLTest_AssertCheck(desiredX == currentX, "Verify returned X position; expected: %d, got: %d", desiredX, currentX);

            /* Get position Y */
            currentY = desiredY + 1;
            SDL_GetWindowPosition(window, NULL, &currentY);
            SDLTest_AssertPass("Call to SDL_GetWindowPosition(&x=NULL)");
            SDLTest_AssertCheck(desiredY == currentY, "Verify returned Y position; expected: %d, got: %d", desiredY, currentY);
        }
    }

    /* Dummy call with both pointers NULL */
    SDL_GetWindowPosition(window, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowPosition(&x=NULL,&y=NULL)");

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Set some 'magic' value for later check that nothing was changed */
    referenceX = SDLTest_RandomSint32();
    referenceY = SDLTest_RandomSint32();
    currentX = referenceX;
    currentY = referenceY;
    desiredX = SDLTest_RandomSint32();
    desiredY = SDLTest_RandomSint32();

    /* Negative tests */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    SDL_GetWindowPosition(NULL, &currentX, &currentY);
    SDLTest_AssertPass("Call to SDL_GetWindowPosition(window=NULL)");
    SDLTest_AssertCheck(
        currentX == referenceX && currentY == referenceY,
        "Verify that content of X and Y pointers has not been modified; expected: %d,%d; got: %d,%d",
        referenceX, referenceY,
        currentX, currentY);
    _checkInvalidWindowError();

    SDL_GetWindowPosition(NULL, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowPosition(NULL, NULL, NULL)");
    _checkInvalidWindowError();

    SDL_SetWindowPosition(NULL, desiredX, desiredY);
    SDLTest_AssertPass("Call to SDL_SetWindowPosition(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/* Helper function that checks for an 'Invalid parameter' error */
void _checkInvalidParameterError(void)
{
    const char *invalidParameterError = "Parameter";
    char *lastError;

    lastError = (char *)SDL_GetError();
    SDLTest_AssertPass("SDL_GetError()");
    SDLTest_AssertCheck(lastError != NULL, "Verify error message is not NULL");
    if (lastError != NULL) {
        SDLTest_AssertCheck(SDL_strncmp(lastError, invalidParameterError, SDL_strlen(invalidParameterError)) == 0,
                            "SDL_GetError(): expected message starts with '%s', was message: '%s'",
                            invalidParameterError,
                            lastError);
        SDL_ClearError();
        SDLTest_AssertPass("Call to SDL_ClearError()");
    }
}

/**
 * @brief Tests call to SDL_GetWindowSize and SDL_SetWindowSize
 *
 * @sa http://wiki.libsdl.org/SDL_GetWindowSize
 * @sa http://wiki.libsdl.org/SDL_SetWindowSize
 */
int video_getSetWindowSize(void *arg)
{
    const char *title = "video_getSetWindowSize Test Window";
    SDL_Window *window;
    int result;
    SDL_Rect display;
    int maxwVariation, maxhVariation;
    int wVariation, hVariation;
    int referenceW, referenceH;
    int currentW, currentH;
    int desiredW, desiredH;

    /* Get display bounds for size range */
    result = SDL_GetDisplayUsableBounds(0, &display);
    SDLTest_AssertPass("SDL_GetDisplayBounds()");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    if (result != 0) {
        return TEST_ABORTED;
    }

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "windows") == 0 ||
        SDL_strcmp(SDL_GetCurrentVideoDriver(), "x11") == 0) {
        /* Platform clips window size to screen size */
        maxwVariation = 4;
        maxhVariation = 4;
    } else {
        /* Platform allows window size >= screen size */
        maxwVariation = 5;
        maxhVariation = 5;
    }

    for (wVariation = 0; wVariation < maxwVariation; wVariation++) {
        for (hVariation = 0; hVariation < maxhVariation; hVariation++) {
            switch (wVariation) {
            default:
            case 0:
                /* 1 Pixel Wide */
                desiredW = 1;
                break;
            case 1:
                /* Random width inside screen */
                desiredW = SDLTest_RandomIntegerInRange(1, 100);
                break;
            case 2:
                /* Width 1 pixel smaller than screen */
                desiredW = display.w - 1;
                break;
            case 3:
                /* Width at screen size */
                desiredW = display.w;
                break;
            case 4:
                /* Width 1 pixel larger than screen */
                desiredW = display.w + 1;
                break;
            }

            switch (hVariation) {
            default:
            case 0:
                /* 1 Pixel High */
                desiredH = 1;
                break;
            case 1:
                /* Random height inside screen */
                desiredH = SDLTest_RandomIntegerInRange(1, 100);
                break;
            case 2:
                /* Height 1 pixel smaller than screen */
                desiredH = display.h - 1;
                break;
            case 3:
                /* Height at screen size */
                desiredH = display.h;
                break;
            case 4:
                /* Height 1 pixel larger than screen */
                desiredH = display.h + 1;
                break;
            }

            /* Set size */
            SDL_SetWindowSize(window, desiredW, desiredH);
            SDLTest_AssertPass("Call to SDL_SetWindowSize(...,%d,%d)", desiredW, desiredH);

            /* Get size */
            currentW = desiredW + 1;
            currentH = desiredH + 1;
            SDL_GetWindowSize(window, &currentW, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowSize()");

            if (desiredW == currentW && desiredH == currentH) {
                SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentW);
                SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredH, currentH);
            } else {
                SDL_bool hasEvent;
                /* SDL_SetWindowPosition() and SDL_SetWindowSize() will make requests of the window manager and set the internal position and size,
                 * and then we get events signaling what actually happened, and they get passed on to the application if they're not what we expect. */
                desiredW = currentW + 1;
                desiredH = currentH + 1;
                hasEvent = getSizeFromEvent(&desiredW, &desiredH);
                SDLTest_AssertCheck(hasEvent == SDL_TRUE, "Changing size was not honored by WM, checking presence of SDL_WINDOWEVENT_RESIZED");
                if (hasEvent) {
                    SDLTest_AssertCheck(desiredW == currentW, "Verify returned width is the one from SDL event; expected: %d, got: %d", desiredW, currentW);
                    SDLTest_AssertCheck(desiredH == currentH, "Verify returned height is the one from SDL event; expected: %d, got: %d", desiredH, currentH);
                }
            }

            /* Get just width */
            currentW = desiredW + 1;
            SDL_GetWindowSize(window, &currentW, NULL);
            SDLTest_AssertPass("Call to SDL_GetWindowSize(&h=NULL)");
            SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentW);

            /* Get just height */
            currentH = desiredH + 1;
            SDL_GetWindowSize(window, NULL, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowSize(&w=NULL)");
            SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredH, currentH);
        }
    }

    /* Dummy call with both pointers NULL */
    SDL_GetWindowSize(window, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowSize(&w=NULL,&h=NULL)");

    /* Negative tests for parameter input */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    for (desiredH = -2; desiredH < 2; desiredH++) {
        for (desiredW = -2; desiredW < 2; desiredW++) {
            if (desiredW <= 0 || desiredH <= 0) {
                SDL_SetWindowSize(window, desiredW, desiredH);
                SDLTest_AssertPass("Call to SDL_SetWindowSize(...,%d,%d)", desiredW, desiredH);
                _checkInvalidParameterError();
            }
        }
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Set some 'magic' value for later check that nothing was changed */
    referenceW = SDLTest_RandomSint32();
    referenceH = SDLTest_RandomSint32();
    currentW = referenceW;
    currentH = referenceH;
    desiredW = SDLTest_RandomSint32();
    desiredH = SDLTest_RandomSint32();

    /* Negative tests for window input */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    SDL_GetWindowSize(NULL, &currentW, &currentH);
    SDLTest_AssertPass("Call to SDL_GetWindowSize(window=NULL)");
    SDLTest_AssertCheck(
        currentW == referenceW && currentH == referenceH,
        "Verify that content of W and H pointers has not been modified; expected: %d,%d; got: %d,%d",
        referenceW, referenceH,
        currentW, currentH);
    _checkInvalidWindowError();

    SDL_GetWindowSize(NULL, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowSize(NULL, NULL, NULL)");
    _checkInvalidWindowError();

    SDL_SetWindowSize(NULL, desiredW, desiredH);
    SDLTest_AssertPass("Call to SDL_SetWindowSize(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowMinimumSize and SDL_SetWindowMinimumSize
 *
 */
int video_getSetWindowMinimumSize(void *arg)
{
    const char *title = "video_getSetWindowMinimumSize Test Window";
    SDL_Window *window;
    int result;
    SDL_Rect display;
    int wVariation, hVariation;
    int referenceW, referenceH;
    int currentW, currentH;
    int desiredW = 1;
    int desiredH = 1;

    /* Get display bounds for size range */
    result = SDL_GetDisplayBounds(0, &display);
    SDLTest_AssertPass("SDL_GetDisplayBounds()");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    if (result != 0) {
        return TEST_ABORTED;
    }

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    for (wVariation = 0; wVariation < 5; wVariation++) {
        for (hVariation = 0; hVariation < 5; hVariation++) {
            switch (wVariation) {
            case 0:
                /* 1 Pixel Wide */
                desiredW = 1;
                break;
            case 1:
                /* Random width inside screen */
                desiredW = SDLTest_RandomIntegerInRange(2, display.w - 1);
                break;
            case 2:
                /* Width at screen size */
                desiredW = display.w;
                break;
            }

            switch (hVariation) {
            case 0:
                /* 1 Pixel High */
                desiredH = 1;
                break;
            case 1:
                /* Random height inside screen */
                desiredH = SDLTest_RandomIntegerInRange(2, display.h - 1);
                break;
            case 2:
                /* Height at screen size */
                desiredH = display.h;
                break;
            case 4:
                /* Height 1 pixel larger than screen */
                desiredH = display.h + 1;
                break;
            }

            /* Set size */
            SDL_SetWindowMinimumSize(window, desiredW, desiredH);
            SDLTest_AssertPass("Call to SDL_SetWindowMinimumSize(...,%d,%d)", desiredW, desiredH);

            /* Get size */
            currentW = desiredW + 1;
            currentH = desiredH + 1;
            SDL_GetWindowMinimumSize(window, &currentW, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize()");
            SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentW);
            SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredH, currentH);

            /* Get just width */
            currentW = desiredW + 1;
            SDL_GetWindowMinimumSize(window, &currentW, NULL);
            SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize(&h=NULL)");
            SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentH);

            /* Get just height */
            currentH = desiredH + 1;
            SDL_GetWindowMinimumSize(window, NULL, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize(&w=NULL)");
            SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredW, currentH);
        }
    }

    /* Dummy call with both pointers NULL */
    SDL_GetWindowMinimumSize(window, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize(&w=NULL,&h=NULL)");

    /* Negative tests for parameter input */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    for (desiredH = -2; desiredH < 2; desiredH++) {
        for (desiredW = -2; desiredW < 2; desiredW++) {
            if (desiredW <= 0 || desiredH <= 0) {
                SDL_SetWindowMinimumSize(window, desiredW, desiredH);
                SDLTest_AssertPass("Call to SDL_SetWindowMinimumSize(...,%d,%d)", desiredW, desiredH);
                _checkInvalidParameterError();
            }
        }
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Set some 'magic' value for later check that nothing was changed */
    referenceW = SDLTest_RandomSint32();
    referenceH = SDLTest_RandomSint32();
    currentW = referenceW;
    currentH = referenceH;
    desiredW = SDLTest_RandomSint32();
    desiredH = SDLTest_RandomSint32();

    /* Negative tests for window input */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    SDL_GetWindowMinimumSize(NULL, &currentW, &currentH);
    SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize(window=NULL)");
    SDLTest_AssertCheck(
        currentW == referenceW && currentH == referenceH,
        "Verify that content of W and H pointers has not been modified; expected: %d,%d; got: %d,%d",
        referenceW, referenceH,
        currentW, currentH);
    _checkInvalidWindowError();

    SDL_GetWindowMinimumSize(NULL, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowMinimumSize(NULL, NULL, NULL)");
    _checkInvalidWindowError();

    SDL_SetWindowMinimumSize(NULL, desiredW, desiredH);
    SDLTest_AssertPass("Call to SDL_SetWindowMinimumSize(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_GetWindowMaximumSize and SDL_SetWindowMaximumSize
 *
 */
int video_getSetWindowMaximumSize(void *arg)
{
    const char *title = "video_getSetWindowMaximumSize Test Window";
    SDL_Window *window;
    int result;
    SDL_Rect display;
    int wVariation, hVariation;
    int referenceW, referenceH;
    int currentW, currentH;
    int desiredW, desiredH;

    /* Get display bounds for size range */
    result = SDL_GetDisplayBounds(0, &display);
    SDLTest_AssertPass("SDL_GetDisplayBounds()");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    if (result != 0) {
        return TEST_ABORTED;
    }

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    for (wVariation = 0; wVariation < 3; wVariation++) {
        for (hVariation = 0; hVariation < 3; hVariation++) {
            switch (wVariation) {
            case 0:
                /* 1 Pixel Wide */
                desiredW = 1;
                break;
            case 1:
                /* Random width inside screen */
                desiredW = SDLTest_RandomIntegerInRange(2, display.w - 1);
                break;
            case 2:
                /* Width at screen size */
                desiredW = display.w;
                break;
            }

            switch (hVariation) {
            case 0:
                /* 1 Pixel High */
                desiredH = 1;
                break;
            case 1:
                /* Random height inside screen */
                desiredH = SDLTest_RandomIntegerInRange(2, display.h - 1);
                break;
            case 2:
                /* Height at screen size */
                desiredH = display.h;
                break;
            }

            /* Set size */
            SDL_SetWindowMaximumSize(window, desiredW, desiredH);
            SDLTest_AssertPass("Call to SDL_SetWindowMaximumSize(...,%d,%d)", desiredW, desiredH);

            /* Get size */
            currentW = desiredW + 1;
            currentH = desiredH + 1;
            SDL_GetWindowMaximumSize(window, &currentW, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize()");
            SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentW);
            SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredH, currentH);

            /* Get just width */
            currentW = desiredW + 1;
            SDL_GetWindowMaximumSize(window, &currentW, NULL);
            SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize(&h=NULL)");
            SDLTest_AssertCheck(desiredW == currentW, "Verify returned width; expected: %d, got: %d", desiredW, currentH);

            /* Get just height */
            currentH = desiredH + 1;
            SDL_GetWindowMaximumSize(window, NULL, &currentH);
            SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize(&w=NULL)");
            SDLTest_AssertCheck(desiredH == currentH, "Verify returned height; expected: %d, got: %d", desiredW, currentH);
        }
    }

    /* Dummy call with both pointers NULL */
    SDL_GetWindowMaximumSize(window, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize(&w=NULL,&h=NULL)");

    /* Negative tests for parameter input */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    for (desiredH = -2; desiredH < 2; desiredH++) {
        for (desiredW = -2; desiredW < 2; desiredW++) {
            if (desiredW <= 0 || desiredH <= 0) {
                SDL_SetWindowMaximumSize(window, desiredW, desiredH);
                SDLTest_AssertPass("Call to SDL_SetWindowMaximumSize(...,%d,%d)", desiredW, desiredH);
                _checkInvalidParameterError();
            }
        }
    }

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

    /* Set some 'magic' value for later check that nothing was changed */
    referenceW = SDLTest_RandomSint32();
    referenceH = SDLTest_RandomSint32();
    currentW = referenceW;
    currentH = referenceH;
    desiredW = SDLTest_RandomSint32();
    desiredH = SDLTest_RandomSint32();

    /* Negative tests */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");
    SDL_GetWindowMaximumSize(NULL, &currentW, &currentH);
    SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize(window=NULL)");
    SDLTest_AssertCheck(
        currentW == referenceW && currentH == referenceH,
        "Verify that content of W and H pointers has not been modified; expected: %d,%d; got: %d,%d",
        referenceW, referenceH,
        currentW, currentH);
    _checkInvalidWindowError();

    SDL_GetWindowMaximumSize(NULL, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowMaximumSize(NULL, NULL, NULL)");
    _checkInvalidWindowError();

    SDL_SetWindowMaximumSize(NULL, desiredW, desiredH);
    SDLTest_AssertPass("Call to SDL_SetWindowMaximumSize(window=NULL)");
    _checkInvalidWindowError();

    return TEST_COMPLETED;
}

/**
 * @brief Tests call to SDL_SetWindowData and SDL_GetWindowData
 *
 * @sa http://wiki.libsdl.org/SDL_SetWindowData
 * @sa http://wiki.libsdl.org/SDL_GetWindowData
 */
int video_getSetWindowData(void *arg)
{
    int returnValue = TEST_COMPLETED;
    const char *title = "video_setGetWindowData Test Window";
    SDL_Window *window;
    const char *referenceName = "TestName";
    const char *name = "TestName";
    const char *referenceName2 = "TestName2";
    const char *name2 = "TestName2";
    int datasize;
    char *referenceUserdata = NULL;
    char *userdata = NULL;
    char *referenceUserdata2 = NULL;
    char *userdata2 = NULL;
    char *result;
    int iteration;

    /* Call against new test window */
    window = _createVideoSuiteTestWindow(title);
    if (!window) {
        return TEST_ABORTED;
    }

    /* Create testdata */
    datasize = SDLTest_RandomIntegerInRange(1, 32);
    referenceUserdata = SDLTest_RandomAsciiStringOfSize(datasize);
    if (!referenceUserdata) {
        returnValue = TEST_ABORTED;
        goto cleanup;
    }
    userdata = SDL_strdup(referenceUserdata);
    if (!userdata) {
        returnValue = TEST_ABORTED;
        goto cleanup;
    }
    datasize = SDLTest_RandomIntegerInRange(1, 32);
    referenceUserdata2 = SDLTest_RandomAsciiStringOfSize(datasize);
    if (!referenceUserdata2) {
        returnValue = TEST_ABORTED;
        goto cleanup;
    }
    userdata2 = SDL_strdup(referenceUserdata2);
    if (!userdata2) {
        returnValue = TEST_ABORTED;
        goto cleanup;
    }

    /* Get non-existent data */
    result = (char *)SDL_GetWindowData(window, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s)", name);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);

    /* Set data */
    result = (char *)SDL_SetWindowData(window, name, userdata);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,%s)", name, userdata);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);

    /* Get data (twice) */
    for (iteration = 1; iteration <= 2; iteration++) {
        result = (char *)SDL_GetWindowData(window, name);
        SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s) [iteration %d]", name, iteration);
        SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata, result);
        SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    }

    /* Set data again twice */
    for (iteration = 1; iteration <= 2; iteration++) {
        result = (char *)SDL_SetWindowData(window, name, userdata);
        SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,%s) [iteration %d]", name, userdata, iteration);
        SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata, result);
        SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
        SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);
    }

    /* Get data again */
    result = (char *)SDL_GetWindowData(window, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s) [again]", name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);

    /* Set data with new data */
    result = (char *)SDL_SetWindowData(window, name, userdata2);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,%s) [new userdata]", name, userdata2);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, userdata2) == 0, "Validate that userdata2 was not changed, expected: %s, got: %s", referenceUserdata2, userdata2);

    /* Set data with new data again */
    result = (char *)SDL_SetWindowData(window, name, userdata2);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,%s) [new userdata again]", name, userdata2);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata2, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, userdata2) == 0, "Validate that userdata2 was not changed, expected: %s, got: %s", referenceUserdata2, userdata2);

    /* Get new data */
    result = (char *)SDL_GetWindowData(window, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s)", name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata2, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);

    /* Set data with NULL to clear */
    result = (char *)SDL_SetWindowData(window, name, NULL);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,NULL)", name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata2, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, userdata2) == 0, "Validate that userdata2 was not changed, expected: %s, got: %s", referenceUserdata2, userdata2);

    /* Set data with NULL to clear again */
    result = (char *)SDL_SetWindowData(window, name, NULL);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,NULL) [again]", name);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata2, userdata2) == 0, "Validate that userdata2 was not changed, expected: %s, got: %s", referenceUserdata2, userdata2);

    /* Get non-existent data */
    result = (char *)SDL_GetWindowData(window, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s)", name);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);

    /* Get non-existent data new name */
    result = (char *)SDL_GetWindowData(window, name2);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s)", name2);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName2, name2) == 0, "Validate that name2 was not changed, expected: %s, got: %s", referenceName2, name2);

    /* Set data (again) */
    result = (char *)SDL_SetWindowData(window, name, userdata);
    SDLTest_AssertPass("Call to SDL_SetWindowData(...%s,%s) [again, after clear]", name, userdata);
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, userdata) == 0, "Validate that userdata was not changed, expected: %s, got: %s", referenceUserdata, userdata);

    /* Get data (again) */
    result = (char *)SDL_GetWindowData(window, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(..,%s) [again, after clear]", name);
    SDLTest_AssertCheck(SDL_strcmp(referenceUserdata, result) == 0, "Validate that correct result was returned; expected: %s, got: %s", referenceUserdata, result);
    SDLTest_AssertCheck(SDL_strcmp(referenceName, name) == 0, "Validate that name was not changed, expected: %s, got: %s", referenceName, name);

    /* Negative test */
    SDL_ClearError();
    SDLTest_AssertPass("Call to SDL_ClearError()");

    /* Set with invalid window */
    result = (char *)SDL_SetWindowData(NULL, name, userdata);
    SDLTest_AssertPass("Call to SDL_SetWindowData(window=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidWindowError();

    /* Set data with NULL name, valid userdata */
    result = (char *)SDL_SetWindowData(window, NULL, userdata);
    SDLTest_AssertPass("Call to SDL_SetWindowData(name=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Set data with empty name, valid userdata */
    result = (char *)SDL_SetWindowData(window, "", userdata);
    SDLTest_AssertPass("Call to SDL_SetWindowData(name='')");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Set data with NULL name, NULL userdata */
    result = (char *)SDL_SetWindowData(window, NULL, NULL);
    SDLTest_AssertPass("Call to SDL_SetWindowData(name=NULL,userdata=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Set data with empty name, NULL userdata */
    result = (char *)SDL_SetWindowData(window, "", NULL);
    SDLTest_AssertPass("Call to SDL_SetWindowData(name='',userdata=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Get with invalid window */
    result = (char *)SDL_GetWindowData(NULL, name);
    SDLTest_AssertPass("Call to SDL_GetWindowData(window=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidWindowError();

    /* Get data with NULL name */
    result = (char *)SDL_GetWindowData(window, NULL);
    SDLTest_AssertPass("Call to SDL_GetWindowData(name=NULL)");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Get data with empty name */
    result = (char *)SDL_GetWindowData(window, "");
    SDLTest_AssertPass("Call to SDL_GetWindowData(name='')");
    SDLTest_AssertCheck(result == NULL, "Validate that result is NULL");
    _checkInvalidParameterError();

    /* Clean up */
    _destroyVideoSuiteTestWindow(window);

cleanup:
    SDL_free(referenceUserdata);
    SDL_free(referenceUserdata2);
    SDL_free(userdata);
    SDL_free(userdata2);

    return returnValue;
}

/**
 * @brief Tests the functionality of the SDL_WINDOWPOS_CENTERED_DISPLAY along with SDL_WINDOW_FULLSCREEN_DESKTOP.
 *
 * Espeically useful when run on a multi-monitor system with different DPI scales per monitor,
 * to test that the window size is maintained when moving between monitors.
 */
int video_setWindowCenteredOnDisplay(void *arg)
{
    SDL_Window *window;
    const char *title = "video_setWindowCenteredOnDisplay Test Window";
    int x, y, w, h;
    int xVariation, yVariation;
    int displayNum;
    int result;
    SDL_Rect display0, display1;

    displayNum = SDL_GetNumVideoDisplays();
    SDLTest_AssertPass("SDL_GetNumVideoDisplays()");
    SDLTest_AssertCheck(displayNum >= 1, "Validate result (current: %d, expected >= 1)", displayNum);
    if (displayNum <= 0) {
        return TEST_ABORTED;
    }

    /* Get display bounds */
    result = SDL_GetDisplayBounds(0 % displayNum, &display0);
    SDLTest_AssertPass("SDL_GetDisplayBounds()");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    if (result != 0) {
        return TEST_ABORTED;
    }

    result = SDL_GetDisplayBounds(1 % displayNum, &display1);
    SDLTest_AssertPass("SDL_GetDisplayBounds()");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    if (result != 0) {
        return TEST_ABORTED;
    }

    for (xVariation = 0; xVariation < 2; xVariation++) {
        for (yVariation = 0; yVariation < 2; yVariation++) {
            int currentX = 0, currentY = 0;
            int currentW = 0, currentH = 0;
            int expectedX = 0, expectedY = 0;
            int currentDisplay;
            int expectedDisplay;
            SDL_Rect expectedDisplayRect;
            SDL_bool video_driver_is_wayland = SDL_strcmp(SDL_GetCurrentVideoDriver(), "wayland") == 0;

            /* xVariation is the display we start on */
            expectedDisplay = xVariation % displayNum;
            x = SDL_WINDOWPOS_CENTERED_DISPLAY(expectedDisplay);
            y = SDL_WINDOWPOS_CENTERED_DISPLAY(expectedDisplay);
            w = SDLTest_RandomIntegerInRange(640, 800);
            h = SDLTest_RandomIntegerInRange(400, 600);
            expectedDisplayRect = (xVariation == 0) ? display0 : display1;
            expectedX = (expectedDisplayRect.x + ((expectedDisplayRect.w - w) / 2));
            expectedY = (expectedDisplayRect.y + ((expectedDisplayRect.h - h) / 2));

            window = SDL_CreateWindow(title, x, y, w, h, SDL_WINDOW_SHOWN | SDL_WINDOW_ALLOW_HIGHDPI | SDL_WINDOW_BORDERLESS);
            SDLTest_AssertPass("Call to SDL_CreateWindow('Title',%d,%d,%d,%d,SHOWN)", x, y, w, h);
            SDLTest_AssertCheck(window != NULL, "Validate that returned window struct is not NULL");

            /* Wayland windows require that a frame be presented before they are fully mapped and visible onscreen. */
            if (video_driver_is_wayland) {
                SDL_Renderer *renderer = SDL_CreateRenderer(window, -1, 0);

                if (renderer) {
                    SDL_SetRenderDrawColor(renderer, 0x00, 0x00, 0x00, 0xFF);
                    SDL_RenderClear(renderer);
                    SDL_RenderPresent(renderer);

                    /* Some desktops don't display the window immediately after presentation,
                     * so delay to give the window time to actually appear on the desktop.
                     */
                    SDL_Delay(100);
                } else {
                    SDLTest_Log("Unable to create a renderer, tests may fail under Wayland");
                }
            }

            /* Check the window is centered on the requested display */
            currentDisplay = SDL_GetWindowDisplayIndex(window);
            SDL_GetWindowSize(window, &currentW, &currentH);
            SDL_GetWindowPosition(window, &currentX, &currentY);

            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentDisplay == expectedDisplay, "Validate display index (current: %d, expected: %d)", currentDisplay, expectedDisplay);
            } else {
                SDLTest_Log("Skipping display index validation: Wayland driver does not support window positioning");
            }
            SDLTest_AssertCheck(currentW == w, "Validate width (current: %d, expected: %d)", currentW, w);
            SDLTest_AssertCheck(currentH == h, "Validate height (current: %d, expected: %d)", currentH, h);
            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentX == expectedX, "Validate x (current: %d, expected: %d)", currentX, expectedX);
                SDLTest_AssertCheck(currentY == expectedY, "Validate y (current: %d, expected: %d)", currentY, expectedY);
            } else {
                SDLTest_Log("Skipping window position validation: Wayland driver does not support window positioning");
            }

            /* Enter fullscreen desktop */
            result = SDL_SetWindowFullscreen(window, SDL_WINDOW_FULLSCREEN_DESKTOP);
            SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);

            /* Check we are filling the full display */
            currentDisplay = SDL_GetWindowDisplayIndex(window);
            SDL_GetWindowSize(window, &currentW, &currentH);
            SDL_GetWindowPosition(window, &currentX, &currentY);

            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentDisplay == expectedDisplay, "Validate display index (current: %d, expected: %d)", currentDisplay, expectedDisplay);
            } else {
                SDLTest_Log("Skipping display index validation: Wayland driver does not support window positioning");
            }
            SDLTest_AssertCheck(currentW == expectedDisplayRect.w, "Validate width (current: %d, expected: %d)", currentW, expectedDisplayRect.w);
            SDLTest_AssertCheck(currentH == expectedDisplayRect.h, "Validate height (current: %d, expected: %d)", currentH, expectedDisplayRect.h);
            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentX == expectedDisplayRect.x, "Validate x (current: %d, expected: %d)", currentX, expectedDisplayRect.x);
                SDLTest_AssertCheck(currentY == expectedDisplayRect.y, "Validate y (current: %d, expected: %d)", currentY, expectedDisplayRect.y);
            } else {
                SDLTest_Log("Skipping window position validation: Wayland driver does not support window positioning");
            }

            /* Leave fullscreen desktop */
            result = SDL_SetWindowFullscreen(window, 0);
            SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);

            /* Check window was restored correctly */
            currentDisplay = SDL_GetWindowDisplayIndex(window);
            SDL_GetWindowSize(window, &currentW, &currentH);
            SDL_GetWindowPosition(window, &currentX, &currentY);

            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentDisplay == expectedDisplay, "Validate display index (current: %d, expected: %d)", currentDisplay, expectedDisplay);
            } else {
                SDLTest_Log("Skipping display index validation: Wayland driver does not support window positioning");
            }
            SDLTest_AssertCheck(currentW == w, "Validate width (current: %d, expected: %d)", currentW, w);
            SDLTest_AssertCheck(currentH == h, "Validate height (current: %d, expected: %d)", currentH, h);
            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentX == expectedX, "Validate x (current: %d, expected: %d)", currentX, expectedX);
                SDLTest_AssertCheck(currentY == expectedY, "Validate y (current: %d, expected: %d)", currentY, expectedY);
            } else {
                SDLTest_Log("Skipping window position validation: Wayland driver does not support window positioning");
            }

            /* Center on display yVariation, and check window properties */

            expectedDisplay = yVariation % displayNum;
            x = SDL_WINDOWPOS_CENTERED_DISPLAY(expectedDisplay);
            y = SDL_WINDOWPOS_CENTERED_DISPLAY(expectedDisplay);
            expectedDisplayRect = (expectedDisplay == 0) ? display0 : display1;
            expectedX = (expectedDisplayRect.x + ((expectedDisplayRect.w - w) / 2));
            expectedY = (expectedDisplayRect.y + ((expectedDisplayRect.h - h) / 2));
            SDL_SetWindowPosition(window, x, y);

            currentDisplay = SDL_GetWindowDisplayIndex(window);
            SDL_GetWindowSize(window, &currentW, &currentH);
            SDL_GetWindowPosition(window, &currentX, &currentY);

            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentDisplay == expectedDisplay, "Validate display index (current: %d, expected: %d)", currentDisplay, expectedDisplay);
            } else {
                SDLTest_Log("Skipping display index validation: Wayland driver does not support window positioning");
            }
            SDLTest_AssertCheck(currentW == w, "Validate width (current: %d, expected: %d)", currentW, w);
            SDLTest_AssertCheck(currentH == h, "Validate height (current: %d, expected: %d)", currentH, h);
            if (!video_driver_is_wayland) {
                SDLTest_AssertCheck(currentX == expectedX, "Validate x (current: %d, expected: %d)", currentX, expectedX);
                SDLTest_AssertCheck(currentY == expectedY, "Validate y (current: %d, expected: %d)", currentY, expectedY);
            } else {
                SDLTest_Log("Skipping window position validation: Wayland driver does not support window positioning");
            }

            /* Clean up */
            _destroyVideoSuiteTestWindow(window);
        }
    }

    return TEST_COMPLETED;
}

/**
 * Tests window surface functionality
 */
static int video_getWindowSurface(void *arg)
{
    const char *title = "video_getWindowSurface Test Window";
    SDL_Window *window;
    SDL_Surface *surface;
    SDL_Renderer *renderer;
    Uint32 renderer_flags = SDL_RENDERER_ACCELERATED;
    int result;

    if (SDL_strcmp(SDL_GetCurrentVideoDriver(), "dummy") == 0) {
        renderer_flags = SDL_RENDERER_SOFTWARE;
    }

    /* Make sure we're testing interaction with an accelerated renderer */
    SDL_SetHint(SDL_HINT_FRAMEBUFFER_ACCELERATION, "1");

    window = SDL_CreateWindow(title, SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED, 320, 320, 0);
    SDLTest_AssertPass("Call to SDL_CreateWindow('%s', SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED, 320, 320, 0)", title);
    SDLTest_AssertCheck(window != NULL, "Validate that returned window is not NULL");

    surface = SDL_GetWindowSurface(window);
    SDLTest_AssertPass("Call to SDL_GetWindowSurface(window)");
    SDLTest_AssertCheck(surface != NULL, "Validate that returned surface is not NULL");
    SDLTest_AssertCheck(SDL_HasWindowSurface(window), "Validate that window has a surface");

    result = SDL_UpdateWindowSurface(window);
    SDLTest_AssertPass("Call to SDL_UpdateWindowSurface(window)");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);

    /* We shouldn't be able to create a renderer on a window with a surface */
    renderer = SDL_CreateRenderer(window, -1, renderer_flags);
    SDLTest_AssertPass("Call to SDL_CreateRenderer(window)");
    SDLTest_AssertCheck(renderer == NULL, "Validate that returned renderer is NULL");

    result = SDL_DestroyWindowSurface(window);
    SDLTest_AssertPass("Call to SDL_DestroyWindowSurface(window)");
    SDLTest_AssertCheck(result == 0, "Verify return value; expected: 0, got: %d", result);
    SDLTest_AssertCheck(!SDL_HasWindowSurface(window), "Validate that window does not have a surface");

    /* We should be able to create a renderer on the window now */
    renderer = SDL_CreateRenderer(window, -1, renderer_flags);
    SDLTest_AssertPass("Call to SDL_CreateRenderer(window)");
    SDLTest_AssertCheck(renderer != NULL, "Validate that returned renderer is not NULL");

    /* We should not be able to create a window surface now, unless it was created by the renderer */
    if (!SDL_HasWindowSurface(window)) {
        surface = SDL_GetWindowSurface(window);
        SDLTest_AssertPass("Call to SDL_GetWindowSurface(window)");
        SDLTest_AssertCheck(surface == NULL, "Validate that returned surface is NULL");
    }

    SDL_DestroyRenderer(renderer);
    SDLTest_AssertPass("Call to SDL_DestroyRenderer(renderer)");
    SDLTest_AssertCheck(!SDL_HasWindowSurface(window), "Validate that window does not have a surface");

    /* We should be able to create a window surface again */
    surface = SDL_GetWindowSurface(window);
    SDLTest_AssertPass("Call to SDL_GetWindowSurface(window)");
    SDLTest_AssertCheck(surface != NULL, "Validate that returned surface is not NULL");
    SDLTest_AssertCheck(SDL_HasWindowSurface(window), "Validate that window has a surface");

    /* Clean up */
    SDL_DestroyWindow(window);

    return TEST_COMPLETED;
}

/* ================= Test References ================== */

/* Video test cases */
static const SDLTest_TestCaseReference videoTest1 = {
    (SDLTest_TestCaseFp)video_enableDisableScreensaver, "video_enableDisableScreensaver", "Enable and disable screenaver while checking state", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest2 = {
    (SDLTest_TestCaseFp)video_createWindowVariousPositions, "video_createWindowVariousPositions", "Create windows at various locations", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest3 = {
    (SDLTest_TestCaseFp)video_createWindowVariousSizes, "video_createWindowVariousSizes", "Create windows with various sizes", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest4 = {
    (SDLTest_TestCaseFp)video_createWindowVariousFlags, "video_createWindowVariousFlags", "Create windows using various flags", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest5 = {
    (SDLTest_TestCaseFp)video_getWindowFlags, "video_getWindowFlags", "Get window flags set during SDL_CreateWindow", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest6 = {
    (SDLTest_TestCaseFp)video_getNumDisplayModes, "video_getNumDisplayModes", "Use SDL_GetNumDisplayModes function to get number of display modes", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest7 = {
    (SDLTest_TestCaseFp)video_getNumDisplayModesNegative, "video_getNumDisplayModesNegative", "Negative tests for SDL_GetNumDisplayModes", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest8 = {
    (SDLTest_TestCaseFp)video_getClosestDisplayModeCurrentResolution, "video_getClosestDisplayModeCurrentResolution", "Use function to get closes match to requested display mode for current resolution", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest9 = {
    (SDLTest_TestCaseFp)video_getClosestDisplayModeRandomResolution, "video_getClosestDisplayModeRandomResolution", "Use function to get closes match to requested display mode for random resolution", TEST_ENABLED
};

static const SDLTest_TestCaseReference videoTest10 =
        { (SDLTest_TestCaseFp)video_getWindowBrightness, "video_getWindowBrightness",  "Get window brightness", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest11 =
        { (SDLTest_TestCaseFp)video_getWindowBrightnessNegative, "video_getWindowBrightnessNegative",  "Get window brightness with invalid input", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest12 =
        { (SDLTest_TestCaseFp)video_getWindowDisplayMode, "video_getWindowDisplayMode",  "Get window display mode", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest13 =
        { (SDLTest_TestCaseFp)video_getWindowDisplayModeNegative, "video_getWindowDisplayModeNegative",  "Get window display mode with invalid input", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest14 =
        { (SDLTest_TestCaseFp)video_getWindowGammaRamp, "video_getWindowGammaRamp",  "Get window gamma ramp", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest15 =
        { (SDLTest_TestCaseFp)video_getWindowGammaRampNegative, "video_getWindowGammaRampNegative",  "Get window gamma ramp against invalid input", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest16 =
        { (SDLTest_TestCaseFp)video_getSetWindowGrab, "video_getSetWindowGrab",  "Checks SDL_GetWindowGrab and SDL_SetWindowGrab positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest17 =
        { (SDLTest_TestCaseFp)video_getWindowId, "video_getWindowId",  "Checks SDL_GetWindowID and SDL_GetWindowFromID", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest18 =
        { (SDLTest_TestCaseFp)video_getWindowPixelFormat, "video_getWindowPixelFormat",  "Checks SDL_GetWindowPixelFormat", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest19 =
        { (SDLTest_TestCaseFp)video_getSetWindowPosition, "video_getSetWindowPosition",  "Checks SDL_GetWindowPosition and SDL_SetWindowPosition positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest20 =
        { (SDLTest_TestCaseFp)video_getSetWindowSize, "video_getSetWindowSize",  "Checks SDL_GetWindowSize and SDL_SetWindowSize positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest21 =
        { (SDLTest_TestCaseFp)video_getSetWindowMinimumSize, "video_getSetWindowMinimumSize",  "Checks SDL_GetWindowMinimumSize and SDL_SetWindowMinimumSize positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest22 =
        { (SDLTest_TestCaseFp)video_getSetWindowMaximumSize, "video_getSetWindowMaximumSize",  "Checks SDL_GetWindowMaximumSize and SDL_SetWindowMaximumSize positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest23 =
        { (SDLTest_TestCaseFp)video_getSetWindowData, "video_getSetWindowData",  "Checks SDL_SetWindowData and SDL_GetWindowData positive and negative cases", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest24 =
        { (SDLTest_TestCaseFp) video_setWindowCenteredOnDisplay, "video_setWindowCenteredOnDisplay", "Checks using SDL_WINDOWPOS_CENTERED_DISPLAY centers the window on a display", TEST_ENABLED };

static const SDLTest_TestCaseReference videoTest25 = {
    (SDLTest_TestCaseFp)video_getWindowSurface, "video_getWindowSurface", "Checks window surface functionality", TEST_ENABLED
};

/* Sequence of Video test cases */
static const SDLTest_TestCaseReference *videoTests[] = {
    &videoTest1, &videoTest2, &videoTest3, &videoTest4, &videoTest5, &videoTest6,
    &videoTest7, &videoTest8, &videoTest9, &videoTest10, &videoTest11, &videoTest12,
    &videoTest13, &videoTest14, &videoTest15, &videoTest16, &videoTest17,
    &videoTest18, &videoTest19, &videoTest20, &videoTest21, &videoTest22,
    &videoTest23, &videoTest24, &videoTest25, NULL
};

/* Video test suite (global) */
SDLTest_TestSuiteReference videoTestSuite = {
    "Video",
    NULL,
    videoTests,
    NULL
};
