//===-- BFloat16 log(x) function ------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#include "src/math/log_bf16.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/bfloat16.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/common.h"
#include "src/__support/macros/config.h"
#include "src/__support/macros/optimization.h"
#include "src/__support/macros/properties/cpu_features.h"

namespace LIBC_NAMESPACE_DECL {

// Generated by Sollya with the following commands:
//   > display = hexadecimal;
//   > round(log(2), SG, RN);
static constexpr float LOGF_2 = 0x1.62e43p-1f;

// Generated by Sollya with the following commands:
//   > display = hexadecimal;
//   > for i from 0 to 127 do print(round(log(1 + i * 2^-7), SG, RN));
static constexpr float LOG_1_PLUS_M[128] = {
    0x0.0p0f,       0x1.fe02a6p-8f, 0x1.fc0a8cp-7f, 0x1.7b91bp-6f,
    0x1.f829bp-6f,  0x1.39e87cp-5f, 0x1.77459p-5f,  0x1.b42dd8p-5f,
    0x1.f0a30cp-5f, 0x1.16536ep-4f, 0x1.341d7ap-4f, 0x1.51b074p-4f,
    0x1.6f0d28p-4f, 0x1.8c345ep-4f, 0x1.a926d4p-4f, 0x1.c5e548p-4f,
    0x1.e27076p-4f, 0x1.fec914p-4f, 0x1.0d77e8p-3f, 0x1.1b72aep-3f,
    0x1.29553p-3f,  0x1.371fc2p-3f, 0x1.44d2b6p-3f, 0x1.526e5ep-3f,
    0x1.5ff308p-3f, 0x1.6d60fep-3f, 0x1.7ab89p-3f,  0x1.87fa06p-3f,
    0x1.9525aap-3f, 0x1.a23bc2p-3f, 0x1.af3c94p-3f, 0x1.bc2868p-3f,
    0x1.c8ff7cp-3f, 0x1.d5c216p-3f, 0x1.e27076p-3f, 0x1.ef0adcp-3f,
    0x1.fb9186p-3f, 0x1.04025ap-2f, 0x1.0a324ep-2f, 0x1.1058cp-2f,
    0x1.1675cap-2f, 0x1.1c898cp-2f, 0x1.22942p-2f,  0x1.2895a2p-2f,
    0x1.2e8e2cp-2f, 0x1.347ddap-2f, 0x1.3a64c6p-2f, 0x1.404308p-2f,
    0x1.4618bcp-2f, 0x1.4be5fap-2f, 0x1.51aad8p-2f, 0x1.576772p-2f,
    0x1.5d1bdcp-2f, 0x1.62c83p-2f,  0x1.686c82p-2f, 0x1.6e08eap-2f,
    0x1.739d8p-2f,  0x1.792a56p-2f, 0x1.7eaf84p-2f, 0x1.842d1ep-2f,
    0x1.89a338p-2f, 0x1.8f11e8p-2f, 0x1.947942p-2f, 0x1.99d958p-2f,
    0x1.9f323ep-2f, 0x1.a4840ap-2f, 0x1.a9cecap-2f, 0x1.af1294p-2f,
    0x1.b44f78p-2f, 0x1.b9858ap-2f, 0x1.beb4dap-2f, 0x1.c3dd7ap-2f,
    0x1.c8ff7cp-2f, 0x1.ce1afp-2f,  0x1.d32fe8p-2f, 0x1.d83e72p-2f,
    0x1.dd46ap-2f,  0x1.e24882p-2f, 0x1.e74426p-2f, 0x1.ec399ep-2f,
    0x1.f128f6p-2f, 0x1.f6124p-2f,  0x1.faf588p-2f, 0x1.ffd2ep-2f,
    0x1.02552ap-1f, 0x1.04bdfap-1f, 0x1.0723e6p-1f, 0x1.0986f4p-1f,
    0x1.0be72ep-1f, 0x1.0e4498p-1f, 0x1.109f3ap-1f, 0x1.12f71ap-1f,
    0x1.154c3ep-1f, 0x1.179eacp-1f, 0x1.19ee6cp-1f, 0x1.1c3b82p-1f,
    0x1.1e85f6p-1f, 0x1.20cdcep-1f, 0x1.23130ep-1f, 0x1.2555bcp-1f,
    0x1.2795e2p-1f, 0x1.29d38p-1f,  0x1.2c0e9ep-1f, 0x1.2e4744p-1f,
    0x1.307d74p-1f, 0x1.32b134p-1f, 0x1.34e28ap-1f, 0x1.37117cp-1f,
    0x1.393e0ep-1f, 0x1.3b6844p-1f, 0x1.3d9026p-1f, 0x1.3fb5b8p-1f,
    0x1.41d8fep-1f, 0x1.43f9fep-1f, 0x1.4618bcp-1f, 0x1.48353ep-1f,
    0x1.4a4f86p-1f, 0x1.4c679ap-1f, 0x1.4e7d82p-1f, 0x1.50913cp-1f,
    0x1.52a2d2p-1f, 0x1.54b246p-1f, 0x1.56bf9ep-1f, 0x1.58cadcp-1f,
    0x1.5ad404p-1f, 0x1.5cdb1ep-1f, 0x1.5ee02ap-1f, 0x1.60e33p-1f,
};

LLVM_LIBC_FUNCTION(bfloat16, log_bf16, (bfloat16 x)) {
  using FPBits = fputil::FPBits<bfloat16>;
  FPBits x_bits(x);

  uint16_t x_u = x_bits.uintval();

  // If x <= 0, or x is 1, or x is +inf, or x is NaN.
  if (LIBC_UNLIKELY(x_u == 0U || x_u == 0x3f80U || x_u >= 0x7f80U)) {
    // log(NaN) = NaN
    if (x_bits.is_nan()) {
      if (x_bits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }

      return x;
    }

    // log(+/-0) = −inf
    if ((x_u & 0x7fffU) == 0U) {
      fputil::raise_except_if_required(FE_DIVBYZERO);
      return FPBits::inf(Sign::NEG).get_val();
    }

    // log(1) = 0
    if (x_u == 0x3f80U)
      return FPBits::zero().get_val();

    // x < 0
    if (x_u > 0x8000U) {
      fputil::set_errno_if_required(EDOM);
      fputil::raise_except_if_required(FE_INVALID);
      return FPBits::quiet_nan().get_val();
    }

    // log(+inf) = +inf
    return FPBits::inf().get_val();
  }

#ifndef LIBC_TARGET_CPU_HAS_FMA
  // log(0.00000000000000171390679426508540927898138761520386)
  //     ~= -34.00000095
  if (LIBC_UNLIKELY(x_u == 0x26F7U))
    return bfloat16(-34.0000009);
#endif // LIBC_TARGET_CPU_HAS_FMA

  int e = -FPBits::EXP_BIAS;

  // When x is subnormal, normalize it.
  if ((x_u & FPBits::EXP_MASK) == 0U) {
    // Can't pass an integer to fputil::cast directly.
    constexpr float NORMALIZE_EXP = 1U << FPBits::FRACTION_LEN;
    x_bits = FPBits(x_bits.get_val() * fputil::cast<bfloat16>(NORMALIZE_EXP));
    x_u = x_bits.uintval();
    e -= FPBits::FRACTION_LEN;
  }

  // To compute log(x), we perform the following range reduction:
  //   x = 2^e * (1 + m),
  //   log(x) = e * log(2) + log(1 + m).
  // for BFloat16, mantissa is at most 7 explicit bits, so we lookup
  // log(1 + m) in LOG_1_PLUS_M table using `m` as key.

  // Get the 7-bit mantissa directly as the table index
  uint16_t m = x_bits.get_mantissa();

  // Get unbiased exponent
  e += x_u >> FPBits::FRACTION_LEN;

  return fputil::cast<bfloat16>(
      fputil::multiply_add(static_cast<float>(e), LOGF_2, LOG_1_PLUS_M[m]));
}

} // namespace LIBC_NAMESPACE_DECL
