﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Tag.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/CreateDBParameterGroupMessage">AWS
 * API Reference</a></p>
 */
class CreateDBParameterGroupRequest : public RDSRequest {
 public:
  AWS_RDS_API CreateDBParameterGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDBParameterGroup"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the DB parameter group.</p> <p>Constraints:</p> <ul> <li> <p>Must
   * be 1 to 255 letters, numbers, or hyphens.</p> </li> <li> <p>First character must
   * be a letter</p> </li> <li> <p>Can't end with a hyphen or contain two consecutive
   * hyphens</p> </li> </ul>  <p>This value is stored as a lowercase
   * string.</p>
   */
  inline const Aws::String& GetDBParameterGroupName() const { return m_dBParameterGroupName; }
  inline bool DBParameterGroupNameHasBeenSet() const { return m_dBParameterGroupNameHasBeenSet; }
  template <typename DBParameterGroupNameT = Aws::String>
  void SetDBParameterGroupName(DBParameterGroupNameT&& value) {
    m_dBParameterGroupNameHasBeenSet = true;
    m_dBParameterGroupName = std::forward<DBParameterGroupNameT>(value);
  }
  template <typename DBParameterGroupNameT = Aws::String>
  CreateDBParameterGroupRequest& WithDBParameterGroupName(DBParameterGroupNameT&& value) {
    SetDBParameterGroupName(std::forward<DBParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The DB parameter group family name. A DB parameter group can be associated
   * with one and only one DB parameter group family, and can be applied only to a DB
   * instance running a database engine and engine version compatible with that DB
   * parameter group family.</p> <p>To list all of the available parameter group
   * families for a DB engine, use the following command:</p> <p> <code>aws rds
   * describe-db-engine-versions --query "DBEngineVersions[].DBParameterGroupFamily"
   * --engine &lt;engine&gt;</code> </p> <p>For example, to list all of the available
   * parameter group families for the MySQL DB engine, use the following command:</p>
   * <p> <code>aws rds describe-db-engine-versions --query
   * "DBEngineVersions[].DBParameterGroupFamily" --engine mysql</code> </p>
   * <p>The output contains duplicates.</p>  <p>The following are the valid DB
   * engine values:</p> <ul> <li> <p> <code>aurora-mysql</code> </p> </li> <li> <p>
   * <code>aurora-postgresql</code> </p> </li> <li> <p> <code>db2-ae</code> </p>
   * </li> <li> <p> <code>db2-se</code> </p> </li> <li> <p> <code>mysql</code> </p>
   * </li> <li> <p> <code>oracle-ee</code> </p> </li> <li> <p>
   * <code>oracle-ee-cdb</code> </p> </li> <li> <p> <code>oracle-se2</code> </p>
   * </li> <li> <p> <code>oracle-se2-cdb</code> </p> </li> <li> <p>
   * <code>postgres</code> </p> </li> <li> <p> <code>sqlserver-ee</code> </p> </li>
   * <li> <p> <code>sqlserver-se</code> </p> </li> <li> <p> <code>sqlserver-ex</code>
   * </p> </li> <li> <p> <code>sqlserver-web</code> </p> </li> </ul>
   */
  inline const Aws::String& GetDBParameterGroupFamily() const { return m_dBParameterGroupFamily; }
  inline bool DBParameterGroupFamilyHasBeenSet() const { return m_dBParameterGroupFamilyHasBeenSet; }
  template <typename DBParameterGroupFamilyT = Aws::String>
  void SetDBParameterGroupFamily(DBParameterGroupFamilyT&& value) {
    m_dBParameterGroupFamilyHasBeenSet = true;
    m_dBParameterGroupFamily = std::forward<DBParameterGroupFamilyT>(value);
  }
  template <typename DBParameterGroupFamilyT = Aws::String>
  CreateDBParameterGroupRequest& WithDBParameterGroupFamily(DBParameterGroupFamilyT&& value) {
    SetDBParameterGroupFamily(std::forward<DBParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for the DB parameter group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDBParameterGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags to assign to the DB parameter group.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDBParameterGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDBParameterGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBParameterGroupName;

  Aws::String m_dBParameterGroupFamily;

  Aws::String m_description;

  Aws::Vector<Tag> m_tags;
  bool m_dBParameterGroupNameHasBeenSet = false;
  bool m_dBParameterGroupFamilyHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
