﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/elasticmapreduce/EMR_EXPORTS.h>
#include <aws/elasticmapreduce/model/SpotProvisioningAllocationStrategy.h>
#include <aws/elasticmapreduce/model/SpotProvisioningTimeoutAction.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace EMR {
namespace Model {

/**
 * <p>The launch specification for Spot Instances in the instance fleet, which
 * determines the defined duration, provisioning timeout behavior, and allocation
 * strategy.</p>  <p>The instance fleet configuration is available only in
 * Amazon EMR releases 4.8.0 and later, excluding 5.0.x versions. Spot Instance
 * allocation strategy is available in Amazon EMR releases 5.12.1 and later.</p>
 *   <p>Spot Instances with a defined duration (also known as Spot
 * blocks) are no longer available to new customers from July 1, 2021. For
 * customers who have previously used the feature, we will continue to support Spot
 * Instances with a defined duration until December 31, 2022. </p>
 * <p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SpotProvisioningSpecification">AWS
 * API Reference</a></p>
 */
class SpotProvisioningSpecification {
 public:
  AWS_EMR_API SpotProvisioningSpecification() = default;
  AWS_EMR_API SpotProvisioningSpecification(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMR_API SpotProvisioningSpecification& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_EMR_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Spot provisioning timeout period in minutes. If Spot Instances are not
   * provisioned within this time period, the <code>TimeOutAction</code> is taken.
   * Minimum value is 5 and maximum value is 1440. The timeout applies only during
   * initial provisioning, when the cluster is first created.</p>
   */
  inline int GetTimeoutDurationMinutes() const { return m_timeoutDurationMinutes; }
  inline bool TimeoutDurationMinutesHasBeenSet() const { return m_timeoutDurationMinutesHasBeenSet; }
  inline void SetTimeoutDurationMinutes(int value) {
    m_timeoutDurationMinutesHasBeenSet = true;
    m_timeoutDurationMinutes = value;
  }
  inline SpotProvisioningSpecification& WithTimeoutDurationMinutes(int value) {
    SetTimeoutDurationMinutes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action to take when <code>TargetSpotCapacity</code> has not been
   * fulfilled when the <code>TimeoutDurationMinutes</code> has expired; that is,
   * when all Spot Instances could not be provisioned within the Spot provisioning
   * timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
   * <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot
   * Instances are available, On-Demand Instances should be provisioned to fulfill
   * any remaining Spot capacity.</p>
   */
  inline SpotProvisioningTimeoutAction GetTimeoutAction() const { return m_timeoutAction; }
  inline bool TimeoutActionHasBeenSet() const { return m_timeoutActionHasBeenSet; }
  inline void SetTimeoutAction(SpotProvisioningTimeoutAction value) {
    m_timeoutActionHasBeenSet = true;
    m_timeoutAction = value;
  }
  inline SpotProvisioningSpecification& WithTimeoutAction(SpotProvisioningTimeoutAction value) {
    SetTimeoutAction(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The defined duration for Spot Instances (also known as Spot blocks) in
   * minutes. When specified, the Spot Instance does not terminate before the defined
   * duration expires, and defined duration pricing for Spot Instances applies. Valid
   * values are 60, 120, 180, 240, 300, or 360. The duration period starts as soon as
   * a Spot Instance receives its instance ID. At the end of the duration, Amazon EC2
   * marks the Spot Instance for termination and provides a Spot Instance termination
   * notice, which gives the instance a two-minute warning before it terminates. </p>
   *  <p>Spot Instances with a defined duration (also known as Spot blocks) are
   * no longer available to new customers from July 1, 2021. For customers who have
   * previously used the feature, we will continue to support Spot Instances with a
   * defined duration until December 31, 2022. </p>
   */
  inline int GetBlockDurationMinutes() const { return m_blockDurationMinutes; }
  inline bool BlockDurationMinutesHasBeenSet() const { return m_blockDurationMinutesHasBeenSet; }
  inline void SetBlockDurationMinutes(int value) {
    m_blockDurationMinutesHasBeenSet = true;
    m_blockDurationMinutes = value;
  }
  inline SpotProvisioningSpecification& WithBlockDurationMinutes(int value) {
    SetBlockDurationMinutes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies one of the following strategies to launch Spot Instance fleets:
   * <code>capacity-optimized</code>, <code>price-capacity-optimized</code>,
   * <code>lowest-price</code>, or <code>diversified</code>, and
   * <code>capacity-optimized-prioritized</code>. For more information on the
   * provisioning strategies, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
   * strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide for Linux
   * Instances</i>.</p>  <p>When you launch a Spot Instance fleet with the old
   * console, it automatically launches with the <code>capacity-optimized</code>
   * strategy. You can't change the allocation strategy from the old console.</p>
   *
   */
  inline SpotProvisioningAllocationStrategy GetAllocationStrategy() const { return m_allocationStrategy; }
  inline bool AllocationStrategyHasBeenSet() const { return m_allocationStrategyHasBeenSet; }
  inline void SetAllocationStrategy(SpotProvisioningAllocationStrategy value) {
    m_allocationStrategyHasBeenSet = true;
    m_allocationStrategy = value;
  }
  inline SpotProvisioningSpecification& WithAllocationStrategy(SpotProvisioningAllocationStrategy value) {
    SetAllocationStrategy(value);
    return *this;
  }
  ///@}
 private:
  int m_timeoutDurationMinutes{0};

  SpotProvisioningTimeoutAction m_timeoutAction{SpotProvisioningTimeoutAction::NOT_SET};

  int m_blockDurationMinutes{0};

  SpotProvisioningAllocationStrategy m_allocationStrategy{SpotProvisioningAllocationStrategy::NOT_SET};
  bool m_timeoutDurationMinutesHasBeenSet = false;
  bool m_timeoutActionHasBeenSet = false;
  bool m_blockDurationMinutesHasBeenSet = false;
  bool m_allocationStrategyHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMR
}  // namespace Aws
