/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define XITK_WINDOW_C

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "image.h"
#include "default_font.h"
#include "window.h"
#include "_backend.h"

#include "xitk_x11.h"

#define TITLE_BAR_HEIGHT 20

void xitk_window_set_input_focus (xitk_window_t *w) {
  xitk_window_flags (w, XITK_WINF_FOCUS, XITK_WINF_FOCUS);
}

/*
 *
 */

void xitk_window_define_window_cursor (xitk_window_t *xwin, xitk_cursors_t cursor) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_CURSOR, (uintptr_t)cursor},
    {XITK_TAG_END, 0}
  };

  if (!xwin || (cursor >= xitk_cursor_num_glyphs))
    return;

  xwin->last_cursor = cursor;
  xwin->bewin->set_props (xwin->bewin, tags);
}

/*
 * Is window is size match with given args
 */

const char *xitk_window_set_window_title (xitk_window_t *xwin, const char *title) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_TITLE, (uintptr_t)NULL},
    {XITK_TAG_END, 0}
  };
  const char *t;
 
  if (!xwin)
    return NULL;
  if (!xwin->bewin)
    return NULL;
 
  xwin->bewin->get_props (xwin->bewin, tags);
  t = (const char *)tags[0].value;
  if (!title)
    return t;
  if (t && !strcmp (t, title))
    return t;
 
  tags[0].value = (uintptr_t)title;
  xwin->bewin->set_props (xwin->bewin, tags);
  xwin->bewin->get_props (xwin->bewin, tags);
  t = (const char *)tags[0].value;
  return t;
}
 
/*
 *
 */

void xitk_window_set_window_icon (xitk_window_t *w, xitk_image_t *icon) {

  xitk_tagitem_t tags[] = {
    {XITK_TAG_ICON, (uintptr_t)(icon ? icon->beimg : NULL)},
    {XITK_TAG_END, 0}
  };

  if (w == NULL)
    return;

  w->bewin->set_props (w->bewin, tags);
}

void xitk_window_set_window_layer(xitk_window_t *w, int layer) {

  xitk_tagitem_t tags[] = {
    {XITK_TAG_LAYER, (uintptr_t)layer},
    {XITK_TAG_END, 0}
  };

  if (w == NULL)
    return;

  w->bewin->set_props (w->bewin, tags);
}

void xitk_window_set_layer_above(xitk_window_t *w) {

  xitk_tagitem_t tags[] = {
    {XITK_TAG_LAYER_ABOVE, (uintptr_t)1},
    {XITK_TAG_END, 0}
  };

  if (w == NULL)
    return;

  w->bewin->set_props (w->bewin, tags);
}

/*
 *
 */

void xitk_window_set_window_class(xitk_window_t *w, const char *res_name, const char *res_class) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_RES_NAME, (uintptr_t)res_name},
    {XITK_TAG_RES_CLASS, (uintptr_t)res_class},
    {XITK_TAG_END, 0}
  };

  if (w == NULL)
    return;

  w->bewin->set_props (w->bewin, tags);
}

void xitk_window_set_wm_window_type (xitk_window_t *w, xitk_wm_window_type_t type) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_WINDOW_TYPE, (uintptr_t)type},
    {XITK_TAG_END, 0}
  };

  if (!w)
    return;

  w->bewin->set_props (w->bewin, tags);
  w->type = type;
}

void xitk_window_raise_window (xitk_window_t *xwin) {
  if (xwin && xwin->bewin)
    xwin->bewin->raise (xwin->bewin);
}

/*
void xitk_window_reparent_window (xitk_window_t *xwin, xitk_window_t *parent, int x, int y) {
  if (xwin && xwin->bewin) {
    xitk_tagitem_t tags[] = {
      {XITK_TAG_PARENT, (uintptr_t)(parent ? parent->bewin : NULL)},
      {XITK_TAG_X, x},
      {XITK_TAG_Y, y},
      {XITK_TAG_END, 0}
    };
    xwin->bewin->set_props (xwin->bewin, tags);
  }
}
*/

/*
 * Get (safely) window pos.
 */
void xitk_window_get_window_position (xitk_window_t *xwin, xitk_rect_t *r) {
  if (xwin && xwin->bewin && r) {
    xitk_tagitem_t tags[] = {
      {XITK_TAG_X, 0},
      {XITK_TAG_Y, 0},
      {XITK_TAG_WIDTH, 0},
      {XITK_TAG_HEIGHT, 0},
      {XITK_TAG_END, 0}
    };
    xwin->bewin->get_props (xwin->bewin, tags);
    r->x = tags[0].value;
    r->y = tags[1].value;
    r->width = xwin->width = tags[2].value;
    r->height = xwin->height = tags[3].value;
  }
}

void xitk_window_move_resize (xitk_window_t *xwin, const xitk_rect_t *r) {
  if (xwin && xwin->bewin && r) {
    xitk_tagitem_t tags[5], *p = tags;
    if (r->x != XITK_INT_KEEP) {
      p->type = XITK_TAG_X; p->value = r->x; p++;
    }
    if (r->y != XITK_INT_KEEP) {
      p->type = XITK_TAG_Y; p->value = r->y; p++;
    }
    if (r->width != XITK_INT_KEEP) {
      p->type = XITK_TAG_WIDTH; p->value = r->width; p++;
    }
    if (r->height != XITK_INT_KEEP) {
      p->type = XITK_TAG_HEIGHT; p->value = r->height; p++;
    }
    p->type = XITK_TAG_END; p->value = 0;
    xwin->bewin->set_props (xwin->bewin, tags);
  }
}

static void _xitk_window_draw_frame (xitk_window_t *xwin, const char *title, xitk_msg_type_t type) {
  xitk_image_t *bg, *tbar;
  unsigned int   colorblack, colorwhite, colorgray;
  xitk_font_t   *fs = NULL;
  int            lbear, rbear, wid, asc, des;
  int            bar_style;
  size_t         tlen;
  static const unsigned int typecolors[XITK_MSG_TYPE_LAST][4] = {
    [XITK_MSG_TYPE_INFO]  = { 0x0d800d, 0x082c08, 0x64c864, 0x000500 }, /* green */
    [XITK_MSG_TYPE_WARN]  = { 0x1919f9, 0x0a0a47, 0x7d7dfa, 0x000008 }, /* blue */
    [XITK_MSG_TYPE_ERROR] = { 0xa21010, 0x4c0b0b, 0xd76d6d, 0x070000 }  /* red */
  };

  tbar = xitk_image_new (xwin->xitk, NULL, 0, xwin->width, 2 * TITLE_BAR_HEIGHT);
  if (!tbar)
    return;

  bar_style  = xitk_get_cfg_num (xwin->xitk, XITK_BAR_STYLE);
  colorblack = xitk_get_cfg_num (xwin->xitk, XITK_BLACK_COLOR);
  colorwhite = xitk_get_cfg_num (xwin->xitk, XITK_WHITE_COLOR);
  colorgray  = xitk_get_cfg_num (xwin->xitk, XITK_BG_COLOR);

  /* The window is not yet visible on screen. Rendering into the bg directly
   * should not harm there. Even worse: xitk_image_copy () seems not to wait
   * for pending renders of the source image on some servers. */
  bg = xwin->bg_image;

  tlen = xitk_find_byte (title, 0);
  if (!tlen)
    title = "Window", tlen = 6;
  fs = xitk_font_load_font (xwin->xitk, DEFAULT_BOLD_FONT_12);
  xitk_font_string_extent (fs, title, &lbear, &rbear, &wid, &asc, &des);

 /* Draw window title bar background */
  if (bar_style) {
    int s, bl;
    unsigned int colorblue;

    bl = 0x505050;
    colorblue = xitk_color_db_get (xwin->xitk, bl);
    for (s = 0; s < TITLE_BAR_HEIGHT; s++, bl -= 0x080808) {
      xitk_image_draw_line (tbar, 0, s, xwin->width, s, colorblue);
      colorblue = xitk_color_db_get (xwin->xitk, bl);
    }
    bl = typecolors[type][0];
    colorblue = xitk_color_db_get (xwin->xitk, bl);
    for (s = 0; s < TITLE_BAR_HEIGHT; s++, bl -= typecolors[type][3]) {
      xitk_image_draw_line (tbar, 0, TITLE_BAR_HEIGHT + s, xwin->width, s, colorblue);
      colorblue = xitk_color_db_get (xwin->xitk, bl);
    }
    xitk_image_draw_string (tbar, fs, (xwin->width - wid) - TITLE_BAR_HEIGHT,
      ((TITLE_BAR_HEIGHT + asc + des) >> 1) - des, title, tlen, colorwhite);
    xitk_image_draw_string (tbar, fs, (xwin->width - wid) - TITLE_BAR_HEIGHT,
      TITLE_BAR_HEIGHT + ((TITLE_BAR_HEIGHT + asc + des) >> 1) - des, title, tlen, colorwhite);
  } else {
    int s;
    unsigned int bar1, bar2, bar0 = xitk_color_db_get (xwin->xitk, typecolors[type][2]);

    xitk_image_fill_rectangle (tbar, 0, 0,                xwin->width, TITLE_BAR_HEIGHT, colorgray);
    xitk_image_fill_rectangle (tbar, 0, TITLE_BAR_HEIGHT, xwin->width, TITLE_BAR_HEIGHT, bar0);
    xitk_image_draw_rectangular_box (tbar, 0, 0, xwin->width,     TITLE_BAR_HEIGHT,     XITK_DRAW_OUTTER);
    xitk_image_draw_rectangular_box (tbar, 3, 2, xwin->width - 6, TITLE_BAR_HEIGHT - 4, XITK_DRAW_INNER);
    xitk_image_draw_rectangular_box (tbar, 0, TITLE_BAR_HEIGHT,     xwin->width,     TITLE_BAR_HEIGHT,     XITK_DRAW_OUTTER);
    xitk_image_draw_rectangular_box (tbar, 3, TITLE_BAR_HEIGHT + 2, xwin->width - 6, TITLE_BAR_HEIGHT - 4, XITK_DRAW_INNER);

    bar1 = xitk_color_db_get (xwin->xitk, 0x909090); /* gray */
    bar2 = xitk_color_db_get (xwin->xitk, 0x5f5f5f);
    for (s = 6; s <= (TITLE_BAR_HEIGHT - 6); s += 3) {
      xitk_image_draw_line (tbar, 6, s,     xwin->width - 7, s,     bar2);
      xitk_image_draw_line (tbar, 6, s + 1, xwin->width - 7, s + 1, bar1);
    }
    bar1 = xitk_color_db_get (xwin->xitk, typecolors[type][0]);
    bar2 = xitk_color_db_get (xwin->xitk, typecolors[type][1]);
    for (s = 6; s <= (TITLE_BAR_HEIGHT - 6); s += 3) {
      xitk_image_draw_line (tbar, 6, TITLE_BAR_HEIGHT + s,     xwin->width - 7, TITLE_BAR_HEIGHT + s,     bar2);
      xitk_image_draw_line (tbar, 6, TITLE_BAR_HEIGHT + s + 1, xwin->width - 7, TITLE_BAR_HEIGHT + s + 1, bar1);
    }

    xitk_image_fill_rectangle (tbar, ((xwin->width - wid) - TITLE_BAR_HEIGHT) - 7,
      6, wid + 14, TITLE_BAR_HEIGHT - 11, colorgray);
    xitk_image_fill_rectangle (tbar, ((xwin->width - wid) - TITLE_BAR_HEIGHT) - 7,
      TITLE_BAR_HEIGHT + 6, wid + 14, TITLE_BAR_HEIGHT - 11, bar0);
    xitk_image_draw_string (tbar, fs, (xwin->width - wid) - TITLE_BAR_HEIGHT,
      ((TITLE_BAR_HEIGHT + asc + des) >> 1) - des, title, tlen, xitk_color_db_get (xwin->xitk, 0x404040));
    xitk_image_draw_string (tbar, fs, (xwin->width - wid) - TITLE_BAR_HEIGHT,
      TITLE_BAR_HEIGHT + ((TITLE_BAR_HEIGHT + asc + des) >> 1) - des, title, tlen, colorblack);
  }
  xitk_font_unload_font (fs);
  tbar->last_state = XITK_IMG_STATE_FOCUS;
  {
    xitk_widget_list_t *wl = xitk_window_widget_list (xwin);
    xitk_image_widget_t iw = {.nw = {.wl = wl, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE}};
    xitk_part_image_t image = {
      .image = tbar,
      .x = 0,
      .y = 0,
      .width = xwin->width,
      .height = 2 * TITLE_BAR_HEIGHT,
      .num_states = -2
    };
    unsigned int bar00 = colorwhite, bar01 = colorgray, bar02 = colorblack;
    unsigned int bar10 = xitk_color_db_get (xwin->xitk, typecolors[type][2]);
    unsigned int bar11 = xitk_color_db_get (xwin->xitk, typecolors[type][0]);
    unsigned int bar12 = xitk_color_db_get (xwin->xitk, typecolors[type][1]);
    xitk_color_rect_widget_t rw = {
      .nw = {.wl = wl, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE},
      .width = 1,
      .height = xwin->height - TITLE_BAR_HEIGHT - 1,
      .color[0] = bar00,
      .color[1] = bar10,
      .color[2] = bar00,
      .color[3] = bar00,
      .color[4] = bar00,
      .color[5] = bar00
    };
    xitk_widget_t *wbar;

    /* top */
    wbar = xitk_noskin_part_image_create (&iw, &image, 0, 0);
    xitk_widget_set_window_focusable (wbar);
    xitk_image_free_image (&tbar);

    /* left, right */
    wbar = xitk_noskin_color_rect_create (&rw, 0, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);
    rw.height = xwin->height - TITLE_BAR_HEIGHT - 4;
    wbar = xitk_noskin_color_rect_create (&rw, xwin->width - 4, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);

    rw.width = 2;
    rw.height = xwin->height - TITLE_BAR_HEIGHT - 3;
    rw.color[0] = bar01;
    rw.color[1] = bar11;
    wbar = xitk_noskin_color_rect_create (&rw, 1, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);
    wbar = xitk_noskin_color_rect_create (&rw, xwin->width - 3, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);

    rw.width = 1;
    rw.height = xwin->height - TITLE_BAR_HEIGHT - 4;
    rw.color[0] = bar02;
    rw.color[1] = bar12;
    wbar = xitk_noskin_color_rect_create (&rw, 3, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);
    rw.height = xwin->height - TITLE_BAR_HEIGHT - 1;
    wbar = xitk_noskin_color_rect_create (&rw, xwin->width - 1, TITLE_BAR_HEIGHT);
    xitk_widget_set_window_focusable (wbar);

    /* bottom */
    rw.width = xwin->width - 8;
    rw.height = 1;
    rw.color[0] = bar00;
    rw.color[1] = bar10;
    wbar = xitk_noskin_color_rect_create (&rw, 4, xwin->height - 4);
    xitk_widget_set_window_focusable (wbar);
    rw.width = xwin->width - 2;
    rw.height = 2;
    rw.color[0] = bar01;
    rw.color[1] = bar11;
    wbar = xitk_noskin_color_rect_create (&rw, 1, xwin->height - 3);
    xitk_widget_set_window_focusable (wbar);
    rw.height = 1;
    rw.color[0] = bar02;
    rw.color[1] = bar12;
    wbar = xitk_noskin_color_rect_create (&rw, 1, xwin->height - 1);
    xitk_widget_set_window_focusable (wbar);
  }

  /* xitk_image_draw_line (bg, 0, 0, xwin->width, 0, colorwhite); */
  /* xitk_image_draw_line (bg, 0, 0, 0, TITLE_BAR_HEIGHT - 1, colorwhite); */

  /* xitk_image_draw_rectangular_box (bg,
    4, TITLE_BAR_HEIGHT - 1, xwin->width - 8, xwin->height - (TITLE_BAR_HEIGHT - 1) - 4, XITK_DRAW_INNER); */
  xitk_window_set_background_image (xwin, bg);
}

/*
 * Create a simple (empty) window.
 */
xitk_window_t *xitk_window_create_window_ext (xitk_t *xitk, int x, int y, int width, int height,
    const char *title, const char *res_name, const char *res_class,
    int override_redirect, int layer_above, xitk_image_t *icon, xitk_image_t *bg_image) {
  static const char typenames[XITK_MSG_TYPE_LAST][12] = {
    [XITK_MSG_TYPE_INFO]  = N_("Information"),
    [XITK_MSG_TYPE_WARN]  = N_("Warning"),
    [XITK_MSG_TYPE_ERROR] = N_("Error")
  };
  xitk_window_t *xwin;
  xitk_msg_type_t type;
  int mode = (bg_image == XITK_WINDOW_BG_SIMPLE) ? 0
           : (bg_image == XITK_WINDOW_BG_FRAME) ? 1
           : 2;

  if (xitk == NULL)
    return NULL;
  if ((mode < 2) && ((width <= 0 || height <= 0)))
    return NULL;

  xwin = (xitk_window_t *) xitk_xmalloc (sizeof (*xwin));
  if (!xwin)
    return NULL;

  {
    uintptr_t t = (uintptr_t)title;
    if (t >= XITK_MSG_TYPE_LAST) {
      type = XITK_MSG_TYPE_INFO;
    } else {
      type = (xitk_msg_type_t)t;
      title = dgettext ("xitk", typenames[type]);
    }
  }

  if (x < 0)
    x = XITK_XY_CENTER;
  if (y < 0)
    y = XITK_XY_CENTER;

  if (!title)
    title = "xiTK Window";

  xwin->xitk        = xitk;
  xwin->bewin       = NULL;
  /* will be set by xitk_window_update_tree (). */
  xwin->type        = WINDOW_TYPE_END;
  xwin->role        = XITK_WR_HELPER;
  xwin->key         = 0;
  xwin->last_cursor = xitk_cursor_default;

  if (mode < 2) {
    unsigned int colordgray = xitk_get_cfg_num (xitk, XITK_SELECT_COLOR);

    bg_image = xitk_image_new (xitk, NULL, 0, width, height);
    if (!bg_image) {
      XITK_FREE (xwin);
      return NULL;
    }
    xitk_image_draw_relief (bg_image, width, height, XITK_DRAW_OUTTER);
    xitk_image_draw_line (bg_image, width - 2, 2, width - 2, height - 2, colordgray);
    xitk_image_draw_line (bg_image, 2, height - 2, width - 2, height - 2, colordgray);
  } else {
    xitk_image_ref (bg_image);
  }
  xwin->bg_image = bg_image;

  {
    xitk_tagitem_t tags[] = {
      {XITK_TAG_X, x},
      {XITK_TAG_Y, y},
      {XITK_TAG_IMAGE, (uintptr_t)xwin->bg_image->beimg},
      {XITK_TAG_TITLE, (uintptr_t)title},
      {XITK_TAG_ICON, (uintptr_t)(icon ? icon->beimg : NULL)},
      {XITK_TAG_LAYER_ABOVE, layer_above},
      {XITK_TAG_RES_NAME, (uintptr_t)res_name},
      {XITK_TAG_RES_CLASS, (uintptr_t)res_class},
      {XITK_TAG_WIDTH, width},
      {XITK_TAG_HEIGHT, height},
      {XITK_TAG_WIN_FLAGS, (XITK_WINF_OVERRIDE_REDIRECT << 16) | (override_redirect ? XITK_WINF_OVERRIDE_REDIRECT : 0)},
      {XITK_TAG_END, 0}
    };
    xwin->bewin = xitk->d->window_new (xitk->d, tags);
    if (!xwin->bewin) {
      xitk_image_free_image (&xwin->bg_image);
      XITK_FREE (xwin);
      return NULL;
    }
    xwin->bewin->get_props (xwin->bewin, tags + 8);
    xwin->width = tags[8].value;
    xwin->height = tags[9].value;
    xwin->flags = tags[10].value;
    xwin->bewin->data = xwin;
  }

  if (mode == 1)
    _xitk_window_draw_frame (xwin, title, type);

  return xwin;
}

xitk_widget_list_t *xitk_window_widget_list (xitk_window_t *xwin) {
  if (xwin->widget_list)
    return xwin->widget_list;

  if (!xwin->xitk) {
    XITK_WARNING("xitk_window_widget_list() failed for non-xitk window %p\n", xwin);
    return NULL;
  }

  xwin->widget_list = xitk_widget_list_get (xwin->xitk, xwin);

  return xwin->widget_list;
}

/*
 * Apply (draw) window background.
 */
void xitk_window_apply_background (xitk_window_t *xwin) {
  if (!xwin)
    return;
  if (!xwin->bewin || !xwin->bg_image)
    return;
  {
    xitk_tagitem_t tags[2] = {
      {XITK_TAG_IMAGE, (uintptr_t)xwin->bg_image->beimg},
      {XITK_TAG_END, 0}
    };
    xwin->bewin->set_props (xwin->bewin, tags);
  }
}

xitk_image_t *xitk_window_get_background_image (xitk_window_t *w) {
  return w ? w->bg_image : NULL;
}

int xitk_window_set_background_image (xitk_window_t *xwin, xitk_image_t *bg) {
  if (xwin && xwin->bewin) {
    xitk_image_t *old_bg = xwin->bg_image;
    xitk_tagitem_t tags[] = {
      {XITK_TAG_IMAGE, (uintptr_t)bg->beimg},
      {XITK_TAG_END, 0}
    };
    xwin->bg_image = bg;
    xitk_image_ref (bg);
    xitk_image_free_image (&old_bg);
    return xwin->bewin->set_props (xwin->bewin, tags);
  } else {
    return 0;
  }
}

void xitk_window_destroy_window (xitk_window_t *xwin) {
  if (!xwin)
    return;
  if (xwin->widget_list) {
    xwin->widget_list->xwin = NULL;
    xitk_widget_list_defferred_destroy (xwin->widget_list);
    xwin->widget_list = NULL;
  }
  if (xwin->bewin)
    xwin->bewin->_delete (&xwin->bewin);
  xitk_image_free_image (&xwin->bg_image);
  XITK_FREE (xwin);
}

int xitk_window_get_backend_type(xitk_window_t *xwin) {
  return xwin->bewin->type;
}

uintptr_t xitk_window_get_native_display_id(xitk_window_t *xwin) {
  return xwin->bewin->display->id;
}

uintptr_t xitk_window_get_native_id(xitk_window_t *xwin) {
  return xwin->bewin->id;
}

xitk_window_t *xitk_window_wrap_native_window (xitk_t *xitk, xitk_be_display_t *be_display, uintptr_t window) {
  xitk_window_t *xwin;

  if (be_display) {
    /* display must match witk xitk display */
    if (xitk && xitk->d != be_display) {
      XITK_WARNING("Tried to wrap native window from display %p to xitk display %p\n", be_display, xitk->d);
      return NULL;
    }
  } else if (xitk) {
    be_display = xitk->d;
  }
  if (!be_display)
    return NULL;

  xwin = xitk_xmalloc (sizeof (*xwin));
  if (!xwin)
    return NULL;

  {
    xitk_tagitem_t tags[2] = {
      {XITK_TAG_WRAP, window},
      {XITK_TAG_END, 0}
    };
    xwin->bewin = be_display->window_new (be_display, tags);
  }
  if (!xwin->bewin) {
    XITK_FREE (xwin);
    return NULL;
  }

  xwin->xitk = xitk;
  xwin->bewin->data = xwin;
  xwin->bg_image = NULL;
  xwin->last_cursor = xitk_cursor_default;

  /* will be set by xitk_window_update_tree (). */
  xwin->type = WINDOW_TYPE_END;
  xwin->role = XITK_WR_HELPER;

  if (xitk)
    xwin->widget_list = xitk_widget_list_get (xitk, xwin);

  {
    xitk_tagitem_t tags[] = {
      {XITK_TAG_WIDTH, 0},
      {XITK_TAG_HEIGHT, 0},
      {XITK_TAG_WIN_FLAGS, 0},
      {XITK_TAG_END, 0}
    };
    xwin->bewin->get_props (xwin->bewin, tags);
    xwin->width = tags[0].value;
    xwin->height = tags[1].value;
    xwin->flags = tags[2].value;
  }
  return xwin;
}

uint32_t xitk_window_flags (xitk_window_t *xwin, uint32_t mask, uint32_t value) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_WIN_FLAGS, 0},
    {XITK_TAG_END, 0}
  };

  if (!xwin)
    return 0;
  if (!xwin->bewin)
    return 0;

  if ((mask & value & XITK_WINF_FOCUS) && xwin->xitk)
    xitk_set_focus_key (xwin->xitk, xwin->key, 0);
  tags[0].value = (mask << 16) | (value & 0xffff);
  if ((mask & (XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED)) && (xwin->role != XITK_WR_SUBMENU)) {
    xitk_window_update_tree (xwin, tags[0].value);
  } else {
    xwin->bewin->set_props (xwin->bewin, tags);
  }
  xwin->bewin->get_props (xwin->bewin, tags);
  xwin->flags = tags[0].value;


  return tags[0].value & 0xffff;
}

void xitk_window_set_role (xitk_window_t *xwin, xitk_window_role_t role) {
  if (!xwin)
    return;
  if (xwin->role == role)
    return;
  xwin->role = role;
  xitk_window_update_tree (xwin, 0);
}

int xitk_window_change_skin (xitk_window_t *xwin, xitk_skin_config_t *skin, const char *background_skin_element_name) {
  const xitk_skin_element_info_t *info;
  xitk_image_t *bg_image;
  xitk_widget_list_t *wl;
  xitk_tagitem_t tags[] = {
    {XITK_TAG_WIDTH, 0},
    {XITK_TAG_HEIGHT, 0},
    {XITK_TAG_END, 0}
  };

  if (!xwin)
    return 1;
  if (!skin || !background_skin_element_name)
    return 2;
  wl = xitk_window_widget_list (xwin);
  if (!wl)
    return 1;

  info = xitk_skin_get_info (skin, background_skin_element_name);
  bg_image = info ? info->pixmap_img.image : NULL;
  if (!bg_image)
    return 2;

  /* disable widget painting temporarily. */
  wl->flags &= ~XITK_WL_EXPOSED;
  xitk_widget_list_change_skin (wl, skin);
  /* does trigger EXPOSE events that redraw widgets when background really has changed. */
  xitk_window_set_background_image (xwin, bg_image);

  if (xwin->bewin) {
    xwin->bewin->get_props (xwin->bewin, tags);
    xwin->width = tags[0].value;
    xwin->height = tags[1].value;
  }
  return 0;
}
