/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifndef __MEDIAMARK_H__
#define __MEDIAMARK_H__

#include "common.h"

typedef struct {
  unsigned int max, used, gen;
  int          cur;
  char       **mrl, **ident, **len, *buf;
} mmk_string_list_t;

static inline void gui_playlist_init_strings (mmk_string_list_t *list) {
  list->max = list->used = 0;
  list->gen = ~0;
  list->cur = -1;
  list->mrl = list->ident = list->len = NULL;
  list->buf = NULL;
}
/** set or update string list. if list->used > 1, this will add a fake
 *  "total playtime" entry @ [list->used] that may show on any skin :-) */
int gui_playlist_get_strings (gGui_t *gui, mmk_string_list_t *list);
/** free string list contents. */
void gui_playlist_free_strings (gGui_t *gui, mmk_string_list_t *list);

typedef struct alt_mrl_s alt_mrl_t;

typedef struct {
  char      *ident, *mrl, *sub;
  alt_mrl_t *cur_alt, *alternates;
  int        start; /** << seconds */
  int        end;   /** << seconds, -1 == till the end of stream */
  int        av_offset, spu_offset; /** << milliseconds */
  uint32_t   played; /** << | 1 (played, used with shuffle loop mode), | 2 (played at least 1 time or probed) */
  int        length; /** << milliseconds, or -1 if not known */
  enum mediamark_type_e {
    MMK_TYPE_FILE = 0,
    MMK_TYPE_NET
  }          type;
  enum mediamark_from_e {
    MMK_FROM_USER = 0,
    MMK_FROM_PLAYLIST,
    MMK_FROM_DIR
  }          from;
} mediamark_t;

typedef enum {
  MMK_VAL_IDENT = 0,
  MMK_VAL_MRL,
  MMK_VAL_SUB,
  MMK_VAL_ADD_ALTER
} mmk_val_t;

/** you can read mediamark_t yourself. to modify, use these.
 *  next 3 return these flags. */
#define MMK_CHANGED_IDENT      1
#define MMK_CHANGED_MRL        2
#define MMK_CHANGED_SUB        4
#define MMK_CHANGED_START      8
#define MMK_CHANGED_END       16
#define MMK_CHANGED_LENGTH    32
#define MMK_CHANGED_PLAYED    64
#define MMK_CHANGED_AV_OFFS  128
#define MMK_CHANGED_SPU_OFFS 256
#define MMK_CHANGED_INDEX    512
#define MMK_CHANGED_FAIL    4096
int mediamark_copy (mediamark_t **to, const mediamark_t *from);
int mediamark_set_str_val (mediamark_t **mmk, const char *value, mmk_val_t what);
int mediamark_clear (mediamark_t *mmk);

#define mediamark_have_alternates(_mmk) ((_mmk)->alternates != NULL)
const char *mediamark_get_first_alternate_mrl(mediamark_t *mmk);
const char *mediamark_get_next_alternate_mrl(mediamark_t *mmk);
const char *mediamark_get_current_alternate_mrl(mediamark_t *mmk);


/** gui currently played item. */
#define GUI_MMK_NONE -1
#define GUI_MMK_CURRENT -2
#define GUI_MMK_LAST -3 /** << GUI_MMK_LAST - 1 for second last etc. */
/** returns MMK_CHANGED_*. */
int gui_current_set_index (gGui_t *gui, int idx);
void gui_current_free (gGui_t *gui);


/** gui playlist stuff. */
#define gui_playlist_load(_gui,_filename) gui_playlist_add_file (_gui, _filename, 1)
#define GUI_MAX_DIR_LEVELS 8
typedef enum {
  GUI_ITEM_TYPE_AUTO = 0,
  GUI_ITEM_TYPE_AV,
  GUI_ITEM_TYPE_SPU,
  GUI_ITEM_TYPE_PLAYLIST
} gui_item_type_t;

/** have gui->mmk_mutex locked for these 2. */
void gui_playlist_now_changed (gGui_t *gui);
#define gui_playlist_now_changed(_gui) \
  (_gui)->playlist.gen = ((_gui)->playlist.gen + 1) & 0x7fffffff
int gui_playlist_has_changed (gGui_t *gui, uint32_t *known_gen);
#define gui_playlist_has_changed(_gui, _known_gen) ({ \
  int res = (_gui)->playlist.gen != *(_known_gen); \
  *(_known_gen) = (_gui)->playlist.gen; \
  res; \
})

/** recursively scan this dir for playable files or add this file. return found count.
 *  a negative max_levels will add _all_ files, even hidden and unknown ext ones.
 *  a zero max_levels forbids scanning dirs and playlist files.
 *  if replace is set, previous playlist items will be dropped first.
 *  a NULL or empty path will be ignored. if you mean "current dir", use ".". */
int gui_playlist_add_item (gGui_t *gui, const char *path, int max_levels, gui_item_type_t type, int replace);
/** add 1 entry manually. */
#define gui_playlist_append(_gui,_mrl,_ident,_sub,_start,_end,_av_offset,_spu_offset) \
  gui_playlist_insert (_gui, GUI_MMK_NONE, _mrl, _ident, _sub, _start, _end, _av_offset, _spu_offset)
int gui_playlist_insert (gGui_t *gui, int index, const char *mrl, const char *ident,
  const char *sub, int start, int end, int av_offset, int spu_offset);
/** move n entries starting at index by diff. return new start index. */
int gui_playlist_move (gGui_t *gui, int index, int n, int diff);
/** remove entry/entries manually. return new entries count. */
typedef enum {
  GUI_PLAYLIST_REMOVE_ALL = 0,
  GUI_PLAYLIST_REMOVE_1,
  GUI_PLAYLIST_REMOVE_SAME_DEVICE,
  GUI_PLAYLIST_REMOVE_LAST
} gui_playlist_remove_t;
int gui_playlist_remove (gGui_t *gui, int index, gui_playlist_remove_t mode);
/** returns the real index used. */
int gui_playlist_set_str_val (gGui_t *gui, const char *value, mmk_val_t what, int idx);
/** return 0 (no change), 1 (current item changed), 2 (current item and playlist changed). */
int gui_playlist_set_length (gGui_t *gui, int length);
#define gui_playlist_set_av_offs(_gui,_offs) gui_playlist_set_offs (_gui, 0, _offs)
#define gui_playlist_set_spu_offs(_gui,_offs) gui_playlist_set_offs (_gui, 1, _offs)
int gui_playlist_set_offs (gGui_t *gui, int what, int offs);
mediamark_t *mediamark_get_current_mmk (gGui_t *gui);
const char *mediamark_get_current_mrl (gGui_t *gui);
void gui_playlist_save (gGui_t *gui, const char *filename);
void gui_playlist_free (gGui_t *gui);

void mediamark_reset_played_state (gGui_t *gui);
int gui_playlist_all_played (gGui_t *gui);
#define gui_playlist_all_played(_gui) (((_gui)->playlist.played >= (_gui)->playlist.num))
void gui_playlist_mark_played (gGui_t *gui);
#define gui_playlist_mark_played(_gui) do { \
  (_gui)->playlist.played += 1 - ((_gui)->playlist.mmk[(_gui)->playlist.cur]->played & 1); \
  (_gui)->playlist.probed += 1 - (((_gui)->playlist.mmk[(_gui)->playlist.cur]->played >> 1) & 1); \
  (_gui)->playlist.mmk[(_gui)->playlist.cur]->played |= 3; \
} while (0)
int mediamark_get_shuffle_next (gGui_t *gui);

/** example: https://vids.anywhere.net/ready/to/rumble/trailer.mp4?again=1#start=0:02:55
 *  buf[0]:
 *    pad[8]
 *  start:
 *    https
 *  protend:
 *    ://
 *  host:
 *    vids.anywhere.net
 *  root:
 *    /ready/to/rumble/
 *  lastpart:
 *    trailer.
 *  ext:
 *    mp4
 *  args:
 *    ?again=1
 *  info:
 *    #start=0:02:55
 *  end:
 *    \0, free[n]
 *  max:
 *    pad[8]
 *  buf[sizeof (buf)]
 */
typedef struct {
  char *start, *protend, *host, *root, *lastpart, *ext, *args, *info, *end, *max;
  char buf[2048];
} mrl_buf_t;
void mrl_buf_init (mrl_buf_t *mrlb);
/** base here is just a hint for interpreting ? and #. can be NULL. */
int mrl_buf_set (mrl_buf_t *mrlb, const mrl_buf_t *base, const char *name);
/** for security, this will drop info for network type base. */
void mrl_buf_merge (mrl_buf_t *to, const mrl_buf_t *base, const mrl_buf_t *name);
int mrl_buf_is_file (const mrl_buf_t *mrlb);

size_t mrl_get_lowercase_prot (char *buf, size_t bsize, const char *mrl);
int mrl_look_like_playlist (gGui_t *gui, const char *mrl);
int mrl_look_like_file (const char *mrl);

/** mediamark editor window. */
/** if callback == NULL, this will only update an existing window.
 *  if playlist_index == GUI_MMK_NONE, this will only affect a window open on current entry. */
void mmk_edit_mediamark (gGui_t *gui, void (*apply_callback) (void *userdata, int index), void *userdata, int playlist_index);
void mmk_editor_raise_window (gGui_t *gui);
void mmk_editor_end (gGui_t *gui);

#endif

