Flows
=====
We maintain a
[webpage of streams generated by Liquidsoap](http://flows.liquidsoap.info/).
In order to register your radio on this page, a simple
operator called `register_flow` is provided. If your stream is called `stream`,
just wrap it as follows before outputting it:

```liquidsoap
stream = register_flow(
  radio="My radio",
  website="http://my.web.site/",
  description="The sound of my computer",
  genre="reggae",
  user="foo",
  password="bar",
  streams=[("mp3/128k","http://my.web.site/stream")],
  stream)
```

Most parameters are pretty explicit: `radio` is the name of the radio,
`website` is the url of the website of the radio, etc.

The parameter `streams` is a list of pairs of strings describing the format of the
stream and the url of the stream. We use a list here because one radio can be
broadcasted under multiple stream formats. The format should always be of the
form `codec/bitrate` (e.g. `mp3/128k`) or `codec` if it is encoded in variable
bitrate.

You can register a stream with any `user` and `password` parameters. Those
parameters are only useful if you use additional services such as the
command-line client, or if you want to claim that you ``own'' a registered radio.
All this is relatively informal for now, but may be tightened in future
developments of Flows.

Getting information from flows
------------------------------
If you develop a website and registered your radio as explained above, flows can
be used as a convenient way to retrieve metadata in javascript, get a list of
friendly radios, etc. The flows service currently consists of a list of radios
broadcasting using Liquidsoap, with their corresponding streams and
metadata.

List of radios
--------------
**Note: When the list gets big enough, this API should be changed to return paginated results.**

A list of radios, encoded in JSON format, can be obtained by querying the
following url:

```
http://flows.liquidsoap.info/radios
```

Output is a JSON string like this:

```javascript
[{ "token"       : "a60f5cadf2645321d4d061896318a2d99f2ff6a6",
   "name"        : "RadioPi - Canal Jazz",
   "website"     : "http://www.radiopi.org/",
   "description" : "Cool vibes from Chatenay!",
   "genre"       : "jazz",
   "longitude"   : 2.26670002937317,
   "latitude"    : 48.7667007446289,
   "title"       : "Bud Powell - Un Poco Loco",
   "artist"      : "Va-1939-1999 - 60 Ans De Jazz",
   "streams"     : [{ "format" : "mp3/128k",
                      "url"    : "http://radiopi.org:8080/jazz" }]}
 ...
```

Single radio
------------
A single radio, encoded in JSON format, can be obtained by querying the
following url:

```
http://flows.liquidsoap.info/radio?name=foo&website=bar
```

All argument are optional and should be in UTF8 and properly encoded for a HTTP
GET request.

A direct request using a radio's token can also be performed at this URL:

```
http://flows.liquidsoap.info/radio/:token
```

Output is a JSON string like this:

```javascript
{ "token"       : "a60f5cadf2645321d4d061896318a2d99f2ff6a6",
  "name"        : "RadioPi - Canal Jazz",
  "website"     : "http://www.radiopi.org/",
  "description" : "Cool vibes from Chatenay!",
  "genre"       : "jazz",
  "longitude"   : 2.26670002937317,
  "latitude"    : 48.7667007446289,
  "title"       : "Bud Powell - Un Poco Loco",
  "artist"      : "Va-1939-1999 - 60 Ans De Jazz",
  "streams"     : [{ "format" : "mp3/128k",
                     "url"    : "http://radiopi.org:8080/jazz" }]}
```

Stream redirection
------------------
Flows can also be used to provide a single url for all your listeners, allowing
you to change the actual listening url of your radio while preserving your
playlists.

If you radio's token is `:token` and has a stream of format `:format`, then the
following url will redirect any request to your stream's URL.

```
http://flows.liquidsoap.info/radio/:token/:format
```

Playlist
--------
As for streams, if you radio's token is `:token` then the following link will
return a [PLS](http://en.wikipedia.org/wiki/PLS_(file_format)) playlist:

```
http://flows.liquidsoap.info/radio/:token.pls
```

Real-time notifications
-----------------------
It is oftentimes difficult to properly implement a regular update of your
radio's currently playing information on a website or an online player. Using
Flows, you can very easily implement real-time notifications of the song
currently being played by your radio.

First, you need to know your radio's token. You can do so by querying a single
radio, as described above, or querying all radios and finding yours in the list.

Then, in your webpage's head, you need to add javascript code adapted from this
example:

```html
<script src="http://flows.liquidsoap.info/socket.io/socket.io.js"></script>
<script type="text/javascript">
  var socket = io.connect("http://flows.liquidsoap.info");

  socket.emit('join', radio_token);

  socket.on('joined', function (radio) {
    console.log("Successfully joined " + radio.name + " notification channel. Current title is: " + radio.title + ".");
  });

  socket.on('error', function (data) {
    console.log("Error: " + data + ".");
  });

  socket.on(radio_token, function (data) {
    console.log("Got new notification " + data.cmd + " for radio " + data.radio.name + ": " + JSON.stringify(data));
  });
</script>
```

Received messages are objects of the form:

```javascript
{ cmd   : "metadata",
  radio : { token       : "a60f5cadf2645321d4d061896318a2d99f2ff6a6",
            name        : "RadioPi - Canal Jazz",
            website     : "http://www.radiopi.org/",
            description : "Cool vibes from Chatenay!",
            genre       : "jazz",
            longitude   : 2.26670002937317,
            latitude    : 48.7667007446289,
            title       : "Bud Powell - Un Poco Loco",
            artist      : "Va-1939-1999 - 60 Ans De Jazz",
            streams     : [{ format : "mp3/128k",
                             url    : "http://radiopi.org:8080/jazz" }]}}
```

Messages with command `"metadata"` are the one you want to use to update
information displayed about your radio.

Twitter notifications
---------------------
**Note: The twitter API is deprecated for now.**

You can register twitter notifications on any twitter account with radio that
you own. The API is designed to allow implementing this feature on your
website. It works as follows:

First you issue a `HTTP` `GET` request, authenticated with your radio
credentials at this address:

```
http://flows.liquidsoap.info/radio/:token/twitter/auth?redirect_to=:link
```

Then you should receive a response of the form:

```
{"url":"https://api.twitter.com/oauth/authenticate?oauth_token=..."}
```

You should then visit the returned url with a browser or redirect your website
visitor to this address. There, twitter should ask you to authorize the ```
Savonet
Flows```
 application for the twitter user that you or your visitor are using.

Once done, you should be automatically redirected to `:link` where `:link` is
the parameter passed in the initial `HTTP` `GET` request. From this point, all
metadata updates for the that radio should be forwarded to the twitter account
that was just authorized.

Command-line client
-------------------
A command-line client is also available. It can be used to:

* Change your radio's name
* Change your radio's location
* Register and unregister twitter notifications

The code is still being developped. It can be accessed at this url:
[https://github.com/savonet/flows-client](https://github.com/savonet/flows-client)


