package printer

import (
	"bytes"
	"strings"
	"testing"

	"github.com/dty1er/kubecolor/testutil"
)

func Test_VersionPrinter_Print(t *testing.T) {
	tests := []struct {
		name           string
		darkBackground bool
		recursive      bool
		input          string
		expected       string
	}{
		{
			name:           "go struct dump can be colorized",
			darkBackground: true,
			input: testutil.NewHereDoc(`
				Client Version: version.Info{Major:"1", Minor:"19", GitVersion:"v1.19.3", GitCommit:"1e11e4a2108024935ecfcb2912226cedeafd99df", GitTreeState:"clean", BuildDate:"2020-10-14T18:49:28Z", GoVersion:"go1.15.2", Compiler:"gc", Platform:"darwin/amd64"}
				Server Version: version.Info{Major:"1", Minor:"19", GitVersion:"v1.19.2", GitCommit:"f5743093fd1c663cb0cbc89748f730662345d44d", GitTreeState:"clean", BuildDate:"2020-09-16T13:32:58Z", GoVersion:"go1.15", Compiler:"gc", Platform:"linux/amd64"}`),
			expected: testutil.NewHereDoc(`
				[33mClient Version[0m: [37mversion.Info[0m{[33mMajor[0m:"[36m1[0m", [33mMinor[0m:"[36m19[0m", [33mGitVersion[0m:"[36mv1.19.3[0m", [33mGitCommit[0m:"[36m1e11e4a2108024935ecfcb2912226cedeafd99df[0m", [33mGitTreeState[0m:"[36mclean[0m", [33mBuildDate[0m:"[36m2020-10-14T18:49:28Z[0m", [33mGoVersion[0m:"[36mgo1.15.2[0m", [33mCompiler[0m:"[36mgc[0m", [33mPlatform[0m:"[36mdarwin/amd64[0m"}
				[33mServer Version[0m: [37mversion.Info[0m{[33mMajor[0m:"[36m1[0m", [33mMinor[0m:"[36m19[0m", [33mGitVersion[0m:"[36mv1.19.2[0m", [33mGitCommit[0m:"[36mf5743093fd1c663cb0cbc89748f730662345d44d[0m", [33mGitTreeState[0m:"[36mclean[0m", [33mBuildDate[0m:"[36m2020-09-16T13:32:58Z[0m", [33mGoVersion[0m:"[36mgo1.15[0m", [33mCompiler[0m:"[36mgc[0m", [33mPlatform[0m:"[36mlinux/amd64[0m"}
			`),
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			r := strings.NewReader(tt.input)
			var w bytes.Buffer
			printer := VersionPrinter{DarkBackground: tt.darkBackground}
			printer.Print(r, &w)
			testutil.MustEqual(t, tt.expected, w.String())
		})
	}
}

func Test_VersionShortPrinter_Print(t *testing.T) {
	tests := []struct {
		name           string
		darkBackground bool
		input          string
		expected       string
	}{
		{
			name:           "--short can be colorized",
			darkBackground: true,
			input: testutil.NewHereDoc(`
				Client Version: v1.19.3
				Server Version: v1.19.2`),
			expected: testutil.NewHereDoc(`
				[33mClient Version[0m: [36mv1.19.3[0m
				[33mServer Version[0m: [36mv1.19.2[0m
			`),
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			r := strings.NewReader(tt.input)
			var w bytes.Buffer
			printer := VersionShortPrinter{DarkBackground: tt.darkBackground}
			printer.Print(r, &w)
			testutil.MustEqual(t, tt.expected, w.String())
		})
	}
}
