/* $Id$ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __ENTITY_HPP_INCLUDED
#define __ENTITY_HPP_INCLUDED

#include <string>
#include <list>
#include <set>
#include "frontend/ast/AttributableDeclaration.hpp"
#include "frontend/ast/LibUnit.hpp"
#include "frontend/ast/SignalDeclaration.hpp"
#include "frontend/ast/ConstantDeclaration.hpp"

namespace ast {

//! One VHDL entity definition.
/** This node represents one Entity definition in VHDL.
 */
class Entity : public LibUnit {
public:
	//! c'tor
	/** initialize members
	 *  @param ename name of entity.
	 *  @param plist list of associated Ports. 
	 *  @param glist list of associated Generics.
	 *  @param libs library clauses
         *  @param useCs use clauses. 
         *  @param decls list of declared items.
         *  @param loc location of the Architecture declaration. */
	Entity(
		std::string* ename, 
		std::list<SignalDeclaration*> *plist,
		std::list<ConstantDeclaration*> *glist,
		std::list<SimpleName*> *libs,
		std::list<Name*> *useCs,
		std::list<SymbolDeclaration*> *decls,
		Location loc
		) : 	LibUnit(ename, libs, useCs, decls, loc),
			ports(plist),
			generics(glist) {}
	
	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) {
		visitor.visit(*this);
	}

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const {
		assert(this->name);
		stream << "ENTITY " << *this->name;

		if ((this->generics == NULL) && (this->ports == NULL)) {
			return;
		}

		stream << " IS";
		if (this->generics != NULL) {
			stream << " GENERIC (";
			util::MiscUtil::listPut(this->generics, stream, ", ");
			stream << ");";
		}

		if (this->ports != NULL) {
			stream << " PORT (";
			util::MiscUtil::listPut(this->ports, stream, ", ");
			stream << ");";
		}
	}


	/** List of associated Ports. */
	std::list<SignalDeclaration*> *ports;
	/** List of associated Generics */
	std::list<ConstantDeclaration*> *generics;

protected:
	/** Destructor */
	virtual ~Entity() {
		util::MiscUtil::lterminate(ports);
		util::MiscUtil::lterminate(generics);
	}
};


}; /* namespace ast */

#endif /* __ENTITY_HPP_INCLUDED */
