#!/usr/bin/env perl
#
# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: stats-monitor.pl,v 1.9 2003/02/13 18:41:02 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Data::Dumper;
use Event qw(loop unloop);
use Getopt::Long;
use Time::HiRes qw(gettimeofday);

use ISC::Net::CRLF;
use ISC::Net::Listen;
use ISC::CC::Group::Connect;

use constant STATS_AGING => 120;

my @groups = ();
my $data = {};

my $port = 5545;
my $addr = "0.0.0.0";
my $result = GetOptions("group=s" => \@groups,
			"port=s" => \$port,
			"addr=s" => \$addr);

my $sock = new ISC::CC::Group::Connect(connect_cb => \&_connect_cb,
				       connect_error_cb => \&_connect_err,
				       error_cb => \&_msg_err);

my $listener = new ISC::Net::Listen(cb => \&_listen_newsock,
				    error_cb => \&_listen_err,
				    LocalPort => $port,
				    LocalAddr => $addr,
				    ReuseAddr => 1,
				    Proto => "tcp");

if (scalar(@groups) == 0) {
    push @groups, "stats.epp-frontend";
    push @groups, "stats.whois";
    push @groups, "stats.dns";
}

sub _listen_err {
    warn "listen error";

    unloop();
}

sub _listen_newsock {
    my ($l, $newsock) = @_;

    my $data = {};

    my $crlf = new ISC::Net::CRLF(cb => \&_crlf_cb,
				  error_cb => \&_crlf_err,
				  data => $data,
				  oneshot => 1,
				  flush => 1,
				  socket => $newsock);
    print "New connection.\n";

    send_stats($crlf);
    $crlf->shutdown(5);
}

sub _crlf_err {
    my ($crlf, $msg) = @_;

    warn "Shutting down crlf reader " . $crlf->name;

    $crlf->cancel;
    $crlf = undef;
}

sub _connect_cb {
    my ($gc, $sock) = @_;

    warn "connected, local name is " . $gc->myname;

    foreach my $g (@groups) {
	my ($group, $instance) = split(/\./, $g);
	my %args = (group => $group, cb => \&_group_cb);
	$args{instance} = $instance if ($instance);

	print "subscribing to group $group";
	print " instance $instance" if ($instance);
	print "\n";

	my $c = $gc->join(%args);
    }
}

sub _connect_err {
    my ($foo, $msg) = @_;

    warn "ERROR: $msg";
    unloop();
}

sub _msg_err {
    my ($lp) = @_;

    print "Error on socket: " . $lp->name . "\n";
    unloop();
}

sub _group_cb {
    my ($c, $msg, $wmsg) = @_;

#    print Data::Dumper->Dump([$msg, $wmsg], ["msg", "wmsg"]);

    my $stats = $msg->{stats};
    return unless ($stats);

    my $type = $wmsg->{instance};
    my $id = $stats->{ident};

    print "Got stats on $type from $id\n";

    $data->{$type}->{$id} = $msg;
}

sub send_stats {
    my ($crlf) = @_;

    my $now = gettimeofday;

    my @lines;

    foreach my $type (keys %$data) {
	my $ids = $data->{$type};
	foreach my $id (keys %$ids) {
	    my $msg = $ids->{$id};
	    my $stats = $msg->{stats};
	    if ($msg->{now} + STATS_AGING < $now) {
		delete $data->{$type}->{$id};
		next;
	    }

	    my $counters = $stats->{counters};
	    my $gauges = $stats->{gauges};
	    my $start = $stats->{start};
	    my $their_now = $msg->{now};
	    my $interval = $msg->{interval};

	    push @lines, "stats $type $id";
	    push @lines, "start $start";
	    push @lines, "last $their_now";
	    push @lines, "interval $interval";
	    if (defined($counters) && ref($counters) eq "HASH") {
		foreach my $k (keys %$counters) {
		    push @lines, "counter $k $counters->{$k}";
		}
	    }
	    if (defined($gauges) && ref($counters) eq "HASH") {
		foreach my $k (keys %$gauges) {
		    push @lines, "gauge $k $gauges->{$k}";
		}
	    }
	}
    }
    $crlf->send(\@lines);
}

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

Event::loop();
