#!/usr/bin/env perl
#
# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: replication-master.pl,v 1.12 2002/12/06 21:38:44 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Data::Dumper;
use Event qw(loop unloop);
use Getopt::Long;
use MIME::Base64;
use POSIX qw(gmtime strftime);
use Time::HiRes qw(gettimeofday);

use ISC::CC::Group::Connect;

use ISC::SRS::DB;
use ISC::SRS::EPPResultCode;

my $channel = "srs_rep";
my $result = GetOptions("channel=s" => \$channel);

my $db = new ISC::SRS::DB(replication => "slave",
			  replication_name => "srs",
			  dbname => "srsclone");

my $srs = new ISC::CC::Group::Connect(timeout => 10,
				      connect_error_cb => \&_connect_err,
				      error_cb => \&_err,
				      connect_cb => \&_connect);

Event->timer(cb => \&_tick, interval => 1);

my $versions = {};

my $transactions = 0;
my $start = gettimeofday;

sub _tick {
    my ($ev) = @_;

    my $now = gettimeofday;
    my $interval = $now - $start;
    $start = $now;

    if ($transactions) {
	printf("%04d done in %8.3f seconds (%8.4f/sec)\n",
	       $transactions, $interval, $transactions / $interval);
	$transactions = 0;
    }

    foreach my $v (sort { $a <=> $b } keys %$versions) {
	my $log = $versions->{$v};
	if (!ref($log)) {
	    if ($log == -1) {
		print "\t$now $v ROLLBACK\n";
		delete $versions->{$v};
	    }
	} else {
	    print "\t$now $v COMMITTED";
	    $db->begin;
	    foreach my $item (@$log) {
		my $delta = ISC::CC::Message::fromwire(decode_base64($item->{delta}));
		print " seq ", $item->{seq}, " type ", $delta->{type};
#		print Data::Dumper->Dump([$delta], ["delta"]);
		if ($delta->{type} eq "domain_add") {
		    my ($ret, $msg, $list) = $db->domain_add(%{$delta->{args}});
		    if ($ret != EPP_RESULT_SUCCESS) {
			print "$ret $msg\n";
		    }
		}
	    }
	    $db->commit;
	    print "\n";
	    delete $versions->{$v};
	}
    }
}

sub _connect {
    my ($gc) = @_;

    warn "connected to msgbus, local name is " . $gc->myname;

    $gc->join(group => $channel,
	      instance => "rep_seq",
	      cb => \&_msg_cb);
}

sub _msg_cb {
    my ($c, $msg, $wmsg) = @_;

    my $type = $msg->{type};
    my $version = $msg->{version};

    my ($log, $xlog);
    if ($msg->{log} && $type eq "commit") {
	$log = $msg->{log};
#	print Data::Dumper->Dump([$log], ["log"]);
    }

    if ($type eq "rollback") {
	if ($versions->{$version}) {
	    warn "VERSION $version ALREADY EXISTS";
	} else {
	    $versions->{$version} = -1;
	}
    } elsif ($type eq "commit") {
	if ($versions->{$version}) {
	    warn "VERSION $version ALREADY EXISTS";
	} else {
	    $versions->{$version} = $log;
	}
    } else {
	warn "Unknown type $type";
    }
}

sub _err {
    my ($srs, $msg) = @_;

    warn "SRS error: $msg";
    unloop();
}

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

Event::loop();
