# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: EPPHandler.pm,v 1.18 2003/02/18 22:50:56 explorer Exp $

package ISC::SRS::SAX::EPPHandler;
use base qw(XML::SAX::Base);

use strict;
use warnings;

use Carp;

use Data::Dumper;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.18 $ =~ /\d+/g); sprintf "%d."."%02d"
 x $#r, @r};
    @ISA = qw(Exporter XML::SAX::Base);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

# The following is a list of functions which are called in response
# to a particular element start event, a character event within an
# element, or an element end event. If no processing is required, the
# function reference is undef.

my $action = {
    "clID" => [ undef, \&_set_parms, undef ],
    "clTRID" => [ undef, \&_set_parms, undef ],
    "hello" => [ \&_set_command, undef, undef ],
    "login" => [ \&_set_command, undef, undef ],
    "logout" => [ \&_set_command, undef, undef ],
    "new_pw" => [ undef, \&_set_parms, undef ],
    "pw" => [ undef, \&_set_parms, undef ],
    "transfer" => [ \&_s_transfer, undef, undef ],

    "contact:authInfo" => [ &_s_authinfo, undef, undef ],
    "contact:cc" => [ undef, \&_c_contact_cc, undef ],
    "contact:check" => [ \&_set_command, undef, undef ],
    "contact:city" => [ undef, \&_c_contact_city, undef ],
    "contact:create" => [ \&_set_command, undef, undef ],
    "contact:delete" => [ \&_set_command, undef, undef ],
    "contact:email" => [ undef, \&_set_parms, undef ],
    "contact:fax" => [ undef, \&_set_parms, undef ],
    "contact:id" => [ undef, \&_c_contact_id, undef ],
    "contact:info" => [ \&_set_command, undef, undef ],
    "contact:name" => [ undef, \&_c_contact_name, undef ],
    "contact:org" => [ undef, \&_c_contact_org, undef ],
    "contact:postalInfo" => [ \&_s_postal_info, undef, undef ],
    "contact:sp" => [ undef, \&_c_contact_sp, undef ],
    "contact:status" => [ \&_s_status, \&_c_status, \&_e_status ],
    "contact:street" => [ undef, \&_c_contact_street, undef ],
    "contact:transfer" => [ \&_set_command, undef, undef ],
    "contact:update" => [ \&_set_command, undef, undef ],
    "contact:voice" => [ \&_s_contact_voice, \&_set_parms, undef ],

    "domain:authInfo" => [ &_s_authinfo, undef, undef ],
    "domain:check" => [ \&_set_command, undef, undef ],
    "domain:contact" => [ \&_s_domain_contact, \&_c_domain_contact, undef ],
    "domain:create" => [ \&_set_command, undef, undef ],
    "domain:curExpDate" => [ undef, \&_set_parms, undef ],
    "domain:delete" => [ \&_set_command, undef, undef ],
    "domain:info" => [ \&_set_command, undef, undef ],
    "domain:name" => [ undef, \&_c_domain_name, undef ],
    "domain:ns" => [ undef, \&_c_domain_ns, undef ],
    "domain:period" => [ \&_s_domain_period, \&_set_parms, undef ],
    "domain:pw" => [ undef, \&_c_authinfo_pw, undef ],
    "domain:registrant" => [ undef, \&_set_parms, undef ],
    "domain:renew" => [ \&_set_command, undef, undef ],
    "domain:status" => [ \&_s_status, \&_c_status, \&_e_status ],
    "domain:transfer" => [ \&_set_command, undef, undef ],
    "domain:update" => [ \&_set_command, undef, undef ],

    "host:authInfo" => [ \&_s_authinfo, undef, undef ],
    "host:addr" => [ \&_s_host_addr, \&_c_host_addr, undef ],
    "host:check" => [ \&_set_command, undef, undef ],
    "host:create" => [ \&_set_command, undef, undef ],
    "host:delete" => [ \&_set_command, undef, undef ],
    "host:info" => [ \&_set_command, undef, undef ],
    "host:name" => [ undef, \&_c_host_name, undef ],
    "host:pw" => [ undef, \&_c_authinfo_pw, undef, undef ],
    "host:update" => [ \&_set_command, undef, undef ]
};


sub start_document {
    my ($self, $data) = @_;

    $self->{heritage} = [];
    $self->{req} = {};
    $self->{work} = {};

    $self->SUPER::start_document($data);
}


sub start_element {
    my ($self, $data) = @_;

    my $name = $data->{Name};
    push @{$self->{heritage}}, $name;
    my $handler = @{$action->{$name}}[0];

    if ($handler) {
        &{$handler}($self, $data->{Attributes});
    }
}


sub characters {
    my ($self, $data) = @_;

    my $name = @{$self->{heritage}}[-1];
    my $handler = @{$action->{$name}}[1];

    if ($handler) {
        &{$handler}($self, $data->{Data});
    }
}


sub end_element {
    my ($self, $data) = @_;

    my $name = $data->{Name};
    my $handler = @{$action->{$name}}[2];

    if ($handler) {
        &{$handler}($self);
    }

    pop @{$self->{heritage}};
}


sub end_document {
    my ($self, $data) = @_;

    return ($self->{req});
}


# EPP Parsing Happens Here (tm)


# character section handlers

my $data_map = {
    "clID" => "id",
    "clTRID" => "clTRID",
    "new_pw" => "new_password",
    "pw" => "password",

    "contact:email" => "email",
    "contact:fax" => "fax",
    "contact:voice" => "voice",

    "domain:curExpDate" => "current_date",   
    "domain:period" => "period",
    "domain:registrant" => "registrant"
};

sub _set_parms {
    my ($self, $text) = @_;

    my $name = @{$self->{heritage}}[-1];
    my $parm = $data_map->{$name};

    $self->{req}->{$parm} = $text;
}

sub _c_contact_id {
    my ($self, $text) = @_;

    if (@{$self->{heritage}}[-2] =~ /:check$/) {
        push @{$self->{req}->{ids}}, $text;
    } else {
        $self->{req}->{id} = $text;
    }
}

sub _c_domain_name {
    my ($self, $text) = @_;

    if (@{$self->{heritage}}[-2] =~ /:check$/) {
        push @{$self->{req}->{names}}, $text;
    } else {
        $self->{req}->{name} = $text;
    }
}

sub _c_host_name {
    my ($self, $text) = @_;

    if (@{$self->{heritage}}[-2] =~ /:check$/) {
        push @{$self->{req}->{names}}, $text;       
    } elsif (@{$self->{heritage}}[-2] =~ /:chg$/) {
        $self->{req}->{rename} = $text;
    } else {
        $self->{req}->{name} = $text;
    }
}


sub _c_domain_ns {
    my ($self, $text) = @_;

    if (@{$self->{heritage}}[-2] =~ /:add$/) {
        push @{$self->{req}->{nameservers_add}}, $text;
    } elsif (@{$self->{heritage}}[-2] =~ /:rem$/) {
        push @{$self->{req}->{nameservers_del}}, $text;
    } else {
        push @{$self->{req}->{ns}}, $text;
    }
}

sub _c_status {
    my ($self, $text) = @_;

    $self->{work}->{status_descr} = $text;
}

sub _c_domain_contact {
    my ($self, $text) = @_;

    my $con = [ $self->{work}->{contact_type}, $text ];

    if (@{$self->{heritage}}[-2] =~ /:add$/) {
        push @{$self->{req}->{contact_add}}, $con;
    } elsif (@{$self->{heritage}}[-2] =~ /:rem$/) {
        push @{$self->{req}->{contact_del}}, $con;
    } else {
        push @{$self->{req}->{contact}}, $con;
    }
}

sub _c_host_addr {
    my ($self, $text) = @_;

    my %proto_map = ( "v4" => 4, "v6" => 6 );

    my $proto = $proto_map{$self->{work}->{proto}};

    if (@{$self->{heritage}}[-2] =~ /:add$/) {
        push @{$self->{req}->{addr_add}}, [ $proto, $text ];
    } elsif (@{$self->{heritage}}[-2] =~ /:rem$/) {
        push @{$self->{req}->{addr_del}}, [ $proto, $text ];
    } else {
        push @{$self->{req}->{addrs}}, [ $proto, $text ];
    }
}

sub _c_authinfo_pw {
    my ($self, $text) = @_;

    $self->{req}->{authinfo}->{type} = "pw";
    $self->{req}->{authinfo}->{roid} = $self->{work}->{roid};
    $self->{req}->{authinfo}->{_val} = $text;
}

sub _c_postal_scalar {
  my ($self, $text, $attr) = @_;

    my $type = $self->{work}->{postal_type};
    my $ancestor = @{$self->{heritage}}[-4];

    if ($ancestor =~ /:add$/) {
        $self->{req}->{postal_add}->{$type}->{$attr} = $text;
    } elsif ($ancestor =~ /:del$/) {
        $self->{req}->{postal_del}->{$type}->{$attr} = $text;
    } elsif ($ancestor =~ /:chg$/) {
        $self->{req}->{postal_chg}->{$type}->{$attr} = $text;
    } else {
        $self->{req}->{postal}->{$type}->{$attr} = $text;
    }
}



sub _c_contact_name {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "name");
}

sub _c_contact_org {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "org");
}

sub _c_contact_street {
    my ($self, $text) = @_;

    my $type = $self->{work}->{postal_type};
    my $ancestor = @{$self->{heritage}}[-4];

    if ($ancestor =~ /:add$/) {
        push @{$self->{req}->{postal_add}->{$type}->{street}}, $text;
    } elsif ($ancestor =~ /:del$/) {
        push @{$self->{req}->{postal_del}->{$type}->{street}}, $text;
    } elsif ($ancestor =~ /:chg$/) {
        push @{$self->{req}->{postal_chg}->{$type}->{street}}, $text;
    } else {
        push @{$self->{req}->{postal}->{$type}->{street}}, $text;
    }
}

sub _c_contact_city {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "city");
}

sub _c_contact_sp {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "sp");
}

sub _c_contact_pc {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "pc");
}

sub _c_contact_cc {
    my ($self, $text) = @_;

    _c_postal_scalar($self, $text, "cc");
}

# element start handlers

my $command_map = {
    "hello" => "hello",
    "login" => "login",
    "logout" => "logout",

    "contact:check" => "contact_check",
    "contact:create" => "contact_add",
    "contact:delete" => "contact_del",
    "contact:info" => "contact_info",
    "contact:transfer" => "contact_transfer",
    "contact:update" => "contact_mod",

    "domain:check" => "domain_check",    
    "domain:create" => "domain_add",      
    "domain:delete" => "domain_del",    
    "domain:renew" => "domain_renew",
    "domain:transfer" => "domain_transfer",
    "domain:update" => "domain_mod",
    "domain:info" => "domain_info",

    "host:check" => "host_check",
    "host:create" => "host_add",
    "host:delete" => "host_del",
    "host:info" => "host_info",
    "host:update" => "host_mod"
};


sub _set_command {
    my ($self, $attr) = @_;

    my $name = @{$self->{heritage}}[-1];
    my $command = $command_map->{$name};
    $self->{req}->{command} = $command;
}

sub _s_status {
    my ($self, $attr) = @_;

    $self->{work}->{status_name} = $attr->{'{}s'}->{Value};
    $self->{work}->{status_lang} = $attr->{'{}lang'}->{Value};
}

sub _s_domain_period {
    my ($self, $attr) = @_;

    if ($attr->{'{}unit'}) {
        $self->{req}->{'period_unit'} = $attr->{'{}unit'}->{Value};
    }
}

sub _s_domain_contact {
    my ($self, $attr) = @_;

    $self->{work}->{contact_type} = $attr->{'{}type'}->{Value};
}

sub _s_host_addr {
    my ($self, $attr) = @_;

    $self->{work}->{proto} = $attr->{'{}ip'}->{Value};
}

sub _s_authinfo {
    my ($self, $attr) = @_;

    $self->{work}->{roid} = $attr->{'{}roid'}->{Value};
}

sub _s_transfer {
    my ($self, $attr) = @_;

    $self->{req}->{transfer_op} = $attr->{'{}op'}->{Value};
}

sub _s_postal_info {
    my ($self, $attr) = @_;

    $self->{work}->{postal_type} = $attr->{'{}type'}->{Value};
}

sub _s_contact_voice {
    my ($self, $attr) = @_;

    my $x = $attr->{'{}x'}->{Value};

    if ($x) {
      $self->{req}->{voice_ext} = $attr->{'{}x'}->{Value};
    }
}

sub _e_status {
    my ($self) = @_;

    my $stat = $self->{work}->{status_name};
    my $lang = $self->{work}->{status_lang};
    my $descr = $self->{work}->{status_descr};
    $lang = "en" unless defined($lang);
    $descr = "" unless defined($descr);

    if (@{$self->{heritage}}[-2] =~ /:add$/) {
        push @{$self->{req}->{status_add}}, [ $stat, $lang, $descr ];
    } elsif (@{$self->{heritage}}[-2] =~ /:rem$/) {    
        push @{$self->{req}->{status_del}}, [ $stat, $descr ];
    }
}


1;
