# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: EPPWriter.pm,v 1.19 2003/02/13 20:46:59 lidl Exp $

package ISC::SRS::EPPWriter;

use XML::Writer;
use XML::Writer::String;
use ISC::SRS::Version;
use ISC::SRS::EPPResultCode;
use ISC::Date(date_epp);

use strict;
use warnings;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.19 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw(epp_writer);
    @EXPORT_OK = qw();
    %EXPORT_TAGS = (all => \@EXPORT_OK);
}

my $command_handler_map = {
    "greeting"         => \&_greeting,
    "login"            => \&_common,
    "logout"           => \&_common,
    "unknown"          => \&_common,
    "contact_check"    => \&_contact_check,
    "contact_info"     => \&_contact_info,
    "contact_transfer" => \&_contact_transfer,
    "contact_add"      => \&_contact_add,
    "contact_del"      => \&_common,
    "contact_mod"      => \&_common,
    "domain_check"     => \&_domain_check,
    "domain_add"       => \&_domain_add,
    "domain_del"       => \&_common,
    "domain_info"      => \&_domain_info,
    "domain_renew"     => \&_domain_renew,
    "domain_transfer"  => \&_domain_transfer,
    "domain_mod"       => \&_common,
    "host_check"       => \&_host_check,
    "host_add"         => \&_host_add,
    "host_del"         => \&_common,
    "host_info"        => \&_host_info,
    "host_mod"         => \&_common };

sub epp_writer {
    my ($resp) = @_;

    my $s = new XML::Writer::String();
    my $writer = new XML::Writer(OUTPUT => $s, 
            DATA_MODE => 'true', DATA_INDENT => 4);

    $writer->xmlDecl("UTF-8", "no");
    $writer->startTag("epp",
        "xmlns" => "urn:ietf:params:xml:ns:epp-1.0",
        "xmlns:xsi" => "http://www.w3.org/2001/XMLSchema-instance",
        "xsi:schemaLocation" => "urn:ietf:params:xml:ns:epp-1.0 epp-1.0.xsd");
    &{$command_handler_map->{$resp->{command}}}($writer, $resp);
    $writer->endTag(); # </epp>
    $writer->end();

    return ($s->value());
}

sub _greeting {
    my ($writer, $resp) = @_;

    $writer->startTag("greeting");

        $writer->dataElement("svID", ISC_SRS_SERVERNAME);
        $writer->dataElement("svDate", date_epp());

        $writer->startTag("svcMenu");

            foreach my $ver (ISC_EPP_VERSIONS) {
                $writer->dataElement("version", $ver);
            }

            foreach my $lang (epp_languages()) {
                $writer->dataElement("lang", $lang);
            }

            foreach my $objuri (ISC_EPP_OBJURIS) {
                $writer->dataElement("objURI", $objuri);
            }

        $writer->endTag(); # </svcMenu>

        $writer->startTag("dcp");
            $writer->startTag("statement");
                $writer->startTag("purpose");
                    $writer->emptyTag("admin");
                    $writer->emptyTag("prov");
                $writer->endTag(); # </purpose>
                $writer->startTag("recipient");
                    $writer->emptyTag("ours");
                    $writer->emptyTag("public");
                $writer->endTag(); # </recipient>
                $writer->startTag("retention");
                    $writer->emptyTag("stated");
                $writer->endTag(); # </retention>
            $writer->endTag(); # </statement>
        $writer->endTag(); # </dcp>
    $writer->endTag(); # </greeting>
}

sub _preamble {
    my ($writer, $resp) = @_;

    my $lang = "en";
    if (defined($resp->{lang})) {
        $lang = $resp->{lang};
    }

    my $code = $resp->{resultCode};

    $writer->startTag("response");
        $writer->startTag("result", code => $code);
            $writer->dataElement("msg", epp_result_totext($code, $lang));
        $writer->endTag(); # </result>
}

sub _postamble {
    my ($writer, $resp) = @_;

    if (defined($resp->{clTRID}) || defined($resp->{svTRID})) {
        $writer->startTag("trID");
            $writer->dataElement("clTRID", $resp->{clTRID});  
            $writer->dataElement("svTRID", $resp->{svTRID});
        $writer->endTag(); # </trID>
    }
    $writer->endTag(); # </response>
}


sub _common {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);
    _postamble($writer, $resp);
}

sub _contact_check {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("contact:chkData",
                    "xmlns:contact" => "urn:ietf:params:xml:ns:contact-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:contact-1.0 contact-1.0.xsd");
                foreach my $ids (@{$resp->{ids}}) {
                    $writer->startTag("contact:cd");
                        $writer->dataElement("contact:id", 
                            @{$ids}[0], avail => @{$ids}[1]);
                        if (defined(@{$ids}[2])) {
                            $writer->dataElement("contact:reason", @{$ids}[2]);
                        }
                    $writer->endTag(); # </contact:cd>
                }
            $writer->endTag(); # </contact:chkData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _contact_info {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("contact:infData",
                    "xmlns:contact" => "urn:ietf:params:xml:ns:contact-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:contact-1.0 contact-1.0.xsd");
                $writer->dataElement("contact:id", $resp->{id});
                $writer->dataElement("contact:roid", $resp->{roid});

                if (scalar(@{$resp->{status}})) {
                    foreach my $s (@{$resp->{status}}) {
                        if (defined(@{$s}[1]) && defined(@{$s}[2])) {
                            $writer->dataElement("contact:status", @{$s}[2],
                                s => ${$s}[0], lang => ${$s}[1]);
                        } else {
                            $writer->emptyTag("contact:status", s=> ${$s}[0]);
                        }
                    }
                }

                foreach my $t (keys %{$resp->{postal}}) {
                    $writer->startTag("contact:postalInfo", type => $t);
                        $writer->dataElement("contact:name",
                            $resp->{postal}->{$t}->{name});
                        $writer->dataElement("contact:org",
                            $resp->{postal}->{$t}->{org});
                        $writer->startTag("contact:addr");
                            foreach my $s (@{$resp->{postal}->{$t}->{street}}) {
                                $writer->dataElement("contact:street", $s);
                            }
                            $writer->dataElement("contact:city",
                                $resp->{postal}->{$t}->{city});
                            if (defined($resp->{postal}->{$t}->{sp})) {
				$writer->dataElement("contact:sp",
                                    $resp->{postal}->{$t}->{sp});
			    }
                            if (defined($resp->{postal}->{$t}->{pc})) {
                                $writer->dataElement("contact:pc",
                                    $resp->{postal}->{$t}->{pc});
			    }
                            $writer->dataElement("contact:cc",
                                $resp->{postal}->{$t}->{cc});
                        $writer->endTag(); # </contact:addr>
                    $writer->endTag(); # </contact:postalInfo>
                }
                if (defined($resp->{voice_ext})) {
                    $writer->dataElement("contact:voice", $resp->{voice},
                        x => $resp->{voice_ext});
                } else {
                    $writer->dataElement("contact:voice", $resp->{voice});
                }
                $writer->dataElement("contact:fax", $resp->{fax});
                $writer->dataElement("contact:email", $resp->{email});
                $writer->dataElement("contact:clID", $resp->{clID});
                $writer->dataElement("contact:crID", $resp->{crID});
                $writer->dataElement("contact:crDate", $resp->{crDate});
                if (defined($resp->{upID})) {
                    $writer->dataElement("contact:upID", $resp->{upID});
                }
                if (defined($resp->{upDate})) {
                    $writer->dataElement("contact:upDate", $resp->{upDate});
                }
                if (defined($resp->{trDate})) {
                    $writer->dataElement("contact:trDate", $resp->{trDate});
                }
                # XXXJNA spit out authinfo structure here
            $writer->endTag(); # </contact:infData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _contact_transfer {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("contact:trnData",
                    "xmlns:contact" => "urn:ietf:params:xml:ns:contact-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:contact-1.0 contact-1.0.xsd");
                $writer->dataElement("contact:id", $resp->{id});
                $writer->dataElement("contact:trStatus", $resp->{trStatus});
                $writer->dataElement("contact:reID", $resp->{reID});
                $writer->dataElement("contact:reDate", $resp->{reDate});
                $writer->dataElement("contact:acID", $resp->{acID});
                $writer->dataElement("contact:acDate", $resp->{acDate});
            $writer->endTag(); # </contact:trnData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _contact_add {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("contact:creData",
                    "xmlns:contact" => "urn:ietf:params:xml:ns:contact-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:contact-1.0 contact-1.0.xsd");
                $writer->dataElement("contact:id", $resp->{id});
                $writer->dataElement("contact:crDate", $resp->{crDate});
            $writer->endTag(); # </contact:creData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _domain_check {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("domain:chkData", (
                    "xmlns:contact" => "urn:ietf:params:xml:ns:domain-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:domain-1.0 domain-1.0.xsd"));
                foreach my $names (@{$resp->{names}}) {       
                    $writer->startTag("domain:cd");
                        $writer->dataElement("domain:name",
                            @{$names}[0], avail => @{$names}[1]);
                        if (defined(@{$names}[2])) {
                            $writer->dataElement("domain:reason", @{$names}[2]);
                        }
                    $writer->endTag(); # </domain:cd>
                }
            $writer->endTag(); # </domain:chkData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _domain_add {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("domain:creData",
                    "xmlns:contact" => "urn:ietf:params:xml:ns:domain-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:domain-1.0 domain-1.0.xsd");
                $writer->dataElement("domain:name", $resp->{name});
                $writer->dataElement("domain:crDate", $resp->{crDate});
                $writer->dataElement("domain:exDate", $resp->{exDate});
            $writer->endTag(); # </domain:creData>
        $writer->endTag(); # </resData>   
    }

    _postamble($writer, $resp);   
}

sub _domain_info {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {

        $writer->startTag("resData");
            $writer->startTag("domain:infData",
                    "xmlns:domain" => "urn:ietf:params:xml:ns:domain-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:domain-1.0 domain-1.0.xsd");
                $writer->dataElement("domain:name", $resp->{name});
                $writer->dataElement("domain:roid", $resp->{roid});

                if (defined($resp->{status}) && scalar(@{$resp->{status}})) {
                    foreach my $s (@{$resp->{status}}) {
                        if (defined(@{$s}[1]) && defined(@{$s}[2])) {
                            $writer->dataElement("domain:status", @{$s}[2],
                                s => ${$s}[0], lang => ${$s}[1]);
                        } else {
                            $writer->emptyTag("domain:status", s=> ${$s}[0]);
                        }
                    }
                }

                if (defined($resp->{registrant})) {
		    $writer->dataElement("domain:registrant", $resp->{registrant});
		}
                foreach my $t (@{$resp->{contact}}) {
                    $writer->dataElement("domain:contact", @{$t}[1],
                        type => ${$t}[0]);
                }
                foreach my $n (@{$resp->{ns}}) {
                    $writer->dataElement("domain:ns", $n);
                }
                foreach my $h (@{$resp->{host}}) {
                    $writer->dataElement("domain:host", $h);
                }
                $writer->dataElement("domain:clID", $resp->{clID});
		if (defined($resp->{crID})) {
                    $writer->dataElement("domain:crID", $resp->{crID});
		}
		if (defined($resp->{crDate})) {
                    $writer->dataElement("domain:crDate", $resp->{crDate});
		}
		if (defined($resp->{exDate})) {
                    $writer->dataElement("domain:exDate", $resp->{exDate});
		}
                if (defined($resp->{upID})) {
                    $writer->dataElement("domain:upID", $resp->{upID});
                }
                if (defined($resp->{upDate})) {
                    $writer->dataElement("domain:upDate", $resp->{upDate});
                }
                if (defined($resp->{trDate})) {
                    $writer->dataElement("domain:trDate", $resp->{trDate});
                }
                ## XXXJNA handle authinfo
            $writer->endTag(); # </domain:infData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _domain_renew {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {
        $writer->startTag("resData");
            $writer->startTag("domain:renData",
                    "xmlns:domain" => "urn:ietf:params:xml:ns:domain-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:domain-1.0 domain-1.0.xsd");
                $writer->dataElement("domain:name", $resp->{name});
                $writer->dataElement("domain:exDate", $resp->{exDate});
            $writer->endTag(); # </domain:renData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _domain_transfer {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {
        $writer->startTag("resData");
            $writer->startTag("domain:trnData",
                    "xmlns:domain" => "urn:ietf:params:xml:ns:domain-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:domain-1.0 domain-1.0.xsd");

                $writer->dataElement("domain:name", $resp->{name});
                $writer->dataElement("domain:trStatus", $resp->{trStatus});
                $writer->dataElement("domain:reID", $resp->{reID});
                $writer->dataElement("domain:reDate", $resp->{reDate});
                $writer->dataElement("domain:acID", $resp->{acID});
                $writer->dataElement("domain:acDate", $resp->{acDate});
                $writer->dataElement("domain:expDate", $resp->{expDate});

            $writer->endTag(); # </domain:trnData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _host_check {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {
        $writer->startTag("resData");
            $writer->startTag("host:chkData",
                    "xmlns:host" => "urn:ietf:params:xml:ns:host-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:host-1.0 host-1.0.xsd");

                foreach my $n (@{$resp->{names}}) {
                    $writer->startTag("host:cd");
                        $writer->dataElement("host:name", @{$n}[0],
                            avail => @{$n}[1]);
                        if (defined(@{$n}[2])) {
                            $writer->dataElement("host:reason", @{$n}[2]);
                        }
                    $writer->endTag(); # </host:cd>
                }
            $writer->endTag(); # </host:chkData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _host_add {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {
        $writer->startTag("resData");
            $writer->startTag("host:creData",
                    "xmlns:host" => "urn:ietf:params:xml:ns:host-1.0",      
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:host-1.0 host-1.0.xsd");
                $writer->dataElement("host:name", $resp->{name});
                $writer->dataElement("host:crDate", $resp->{crDate});
            $writer->endTag(); # </host:addData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}

sub _host_info {
    my ($writer, $resp) = @_;

    _preamble($writer, $resp);

    if (epp_success($resp->{resultCode})) {
        $writer->startTag("resData");
            $writer->startTag("host:infData",
                    "xmlns:host" => "urn:ietf:params:xml:ns:host-1.0",
                    "xsi:schemaLocation" => "urn:ietf:params:xml:ns:host-1.0 host-1.0.xsd"); 
               $writer->dataElement("host:name", $resp->{name});
               $writer->dataElement("host:roid", $resp->{roid});

                if (scalar(@{$resp->{status}})) {
                    foreach my $s (@{$resp->{status}}) {
                        if (defined(@{$s}[1]) && defined(@{$s}[2])) {
                            $writer->dataElement("host:status", @{$s}[2],
                                s => ${$s}[0], lang => ${$s}[1]);
                        } else {
                            $writer->emptyTag("host:status", s=> ${$s}[0]);
                        }
                    }
                }

                foreach my $a (@{$resp->{addr}}) {
                    $writer->dataElement("host:addr", @{$a}[1],
                        ip => "v" . ${$a}[0]); 
                }
                $writer->dataElement("host:clID", $resp->{clID});
                $writer->dataElement("host:crID", $resp->{crID});
                $writer->dataElement("host:crDate", $resp->{crDate});
                if (defined($resp->{upID})) {
                    $writer->dataElement("host:upID", $resp->{upID});    
		}
                if (defined($resp->{upDate})) {
                    $writer->dataElement("host:upDate", $resp->{upDate});
		}
                if (defined($resp->{trDate})) {
                    $writer->dataElement("host:trDate", $resp->{trDate});
		}
           $writer->endTag(); # </host:infData>
        $writer->endTag(); # </resData>
    }

    _postamble($writer, $resp);
}


1;

