# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Connect.pm,v 1.7 2002/12/06 02:21:12 lidl Exp $

package ISC::Net::Connect;

use strict;
use warnings;

use Carp;
use Event;
use IO::Socket::INET;

eval {
    require IO::Socket::SSL;
};
our $ssl_available;
if ($@) {
    $ssl_available = 0;
} else {
    $ssl_available = 1;
    $IO::Socket::SSL::DEBUG = 1;
}

our @EXPORT_OK;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.7 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

END {
}

use constant TIMEOUT => 30;

my $sock_count = 0;

sub _open_socket {
    my ($self) = @_;

    my $poke_writer;  # set to trigger immediate writable event

    my $peer;

    if ($self->{_proto} eq "tcp") {
	my %largs;

	$largs{LocalPort} = $self->{_tcp_args}->{LocalPort};
	$largs{LocalAddr} = $self->{_tcp_args}->{LocalAddr};

	$largs{Proto} = "tcp";
	$largs{Type} = SOCK_STREAM;

	$self->{sock} = new IO::Socket::INET(%largs);
	$self->{sock}->sockopt(SO_KEEPALIVE, 1);

	my ($addr) = inet_aton($self->{_tcp_args}->{PeerAddr});
	$peer = sockaddr_in($self->{_tcp_args}->{PeerPort}, $addr);
    } elsif ($self->{_proto} eq "ssl") {
	my %largs;

	$largs{LocalPort} = $self->{_tcp_args}->{LocalPort};
	$largs{LocalAddr} = $self->{_tcp_args}->{LocalAddr};

	$largs{Proto} = "tcp";
	$largs{Type} = SOCK_STREAM;

	$self->{sock} = new IO::Socket::INET(%largs);
	$self->{sock}->sockopt(SO_KEEPALIVE, 1);

	$self->{sock}->autoflush(1);

	my ($addr) = inet_aton($self->{_tcp_args}->{PeerAddr});
	$peer = sockaddr_in($self->{_tcp_args}->{PeerPort}, $addr);
    } elsif ($self->{_proto} eq "local") {
	my %largs;

	$largs{Type} = SOCK_STREAM;

	$self->{sock} = new IO::Socket::UNIX(%largs);

	$peer = sockaddr_un($self->{_local_args}->{Peer});
    }

    $self->{sock}->blocking(0);

    $self->{sock}->connect($peer);
    $poke_writer = 1 if ($! && $! != &Errno::EINPROGRESS);

    $self->{writer} = Event->io(desc => $self->name . " connect",
				fd => $self->{sock},
				cb => \&_sock_connect,
				poll => "wt",
				timeout => $self->{timeout},
				timeout_cb => \&_connect_timeout);

    $self->{writer}->now if ($poke_writer);
}

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $name;
    if ($args{name}) {
	$name = $args{name};
    } else {
	$name = "connect" . $sock_count++;
    }

    my $ret = bless({}, $class);

    croak "Missing argument:  cb" if (!$args{cb});
    croak "Missing argument:  error_cb" if (!$args{error_cb});
    croak "Missing argument:  Proto" if (!$args{Proto});

    $ret->{cb} = $args{cb};
    $ret->{error_cb} = $args{error_cb};
    $ret->{data} = $args{data};
    $ret->{name} = $name;

    $ret->{timeout} = $args{timeout};
    $ret->{timeout} = TIMEOUT if (!$ret->{timeout});

    my $proto = $args{Proto} || "tcp";

    if ($proto eq "tcp") {
	$ret->{_proto} = "tcp";
	$ret->{_tcp_args}->{LocalPort} = $args{LocalPort};
	$ret->{_tcp_args}->{LocalAddr} = $args{LocalAddr};
	$ret->{_tcp_args}->{PeerPort} = $args{PeerPort};
	$ret->{_tcp_args}->{PeerAddr} = $args{PeerAddr};
    } elsif ($proto eq "ssl") {
	croak "SSL support is not available" if (!$ssl_available);
	$ret->{_proto} = "ssl";
	$ret->{_tcp_args}->{LocalPort} = $args{LocalPort};
	$ret->{_tcp_args}->{LocalAddr} = $args{LocalAddr};
	$ret->{_tcp_args}->{PeerPort} = $args{PeerPort};
	$ret->{_tcp_args}->{PeerAddr} = $args{PeerAddr};
    } elsif ($proto eq "local" || $proto eq "unix") {
	$ret->{_proto} = "local";
	$ret->{_local_args}->{Peer} = $args{Peer};
	$ret->{_local_args}->{Local} = $args{Local};
    } else {
	croak "Invalid or missing argument:  Proto";
    }

    _open_socket($ret);

    $ret->{writer}->data($ret);

    return $ret;
}

sub data {
    my ($self, $data) = @_;

    if (defined($data)) {
	$self->{data} = $data;
    }

    return $self->{data};
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub socket {
    my ($self) = @_;

    return $self->{sock};
}

sub cancel {
    my ($self) = @_;

    $self->{writer}->cancel if defined($self->{writer});

    #
    # destroy this socket fully
    #
    $self->{writer} = undef;

    #
    # close the socket down
    #
    $self->{sock} = undef;
}

#
# new connection on our local socket
#
sub _sock_connect {
    my ($ev) = @_;

    my $self = $ev->w->data;
    my $sock = $self->{sock};

    if (!$sock || !$sock->connected) {
	_connect_error(@_);
	return;
    }

    $self->{writer}->stop;

    $self->{writer} = undef;

    if ($self->{_proto} eq "ssl") {
	IO::Socket::SSL::context_init({ SSL_verify_mode => 0 });
	$self->{sock} = IO::Socket::SSL::socketToSSL($sock);
    }

    $self->{cb}->($self);
}

sub _connect_error {
    my ($ev) = @_;

    my $self = $ev->w->data;

    $self->{writer}->cancel;

    $self->{sock} = undef;
    $self->{error_cb}->($self, "CONNECT-FAILED: $!");
}

sub _connect_timeout {
    my ($ev) = @_;

    my $self = $ev->w->data;

    $self->{writer}->cancel;

    $self->{sock} = undef;
    $self->{error_cb}->($self, "TIMEOUT");
}

sub reconnect {
    my ($self) = @_;

    $self->{sock} = _open_socket($self);
}

1;
