# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Date.pm,v 1.12 2002/12/06 02:21:09 lidl Exp $

package ISC::Date;

use strict;
use warnings;

use Time::HiRes qw (gettimeofday);

use POSIX;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.12 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw(date_add
		    date_ansi from_date_ansi
		    date_rrp from_date_rrp
		    date_epp from_date_epp);
    %EXPORT_TAGS = ("all" => \@EXPORT_OK);
}

our @EXPORT_OK;

sub date_ansi {
    my ($ts) = @_;

    $ts =gettimeofday if (!$ts);
    my ($sec, $usec) = split(/\./, $ts);

    my $s = POSIX::strftime("%Y-%m-%d %H:%M:%S", gmtime($ts));
    $s .= "." . $usec if ($usec);

    return $s;
}

sub from_date_ansi {
    my ($s) = @_;

    my ($year, $month, $day,
	$hour, $min, $sec, $usec) = split(/[\ \.\:\-]+/, $s);

    $hour = 0 unless defined($hour);
    $min = 0 unless defined($min);
    $sec = 0 unless defined($sec);
    $usec = 0 unless defined($usec);

    $month--;
    $year -= 1900;

    my $tz = $ENV{TZ};
    $ENV{TZ} = "UTC";
    my $ts = POSIX::mktime($sec, $min, $hour, $day, $month, $year);
    if (defined($tz)) {
	$ENV{TZ} = $tz;
    } else {
	delete($ENV{TZ});
    }

    $ts .= ".$usec" if ($usec);

    return $ts;
}

sub date_rrp {
    my ($ts) = @_;

    $ts =gettimeofday if (!$ts);
    my ($sec, $usec) = split(/\./, $ts);

    my $s = POSIX::strftime("%Y-%m-%d %H:%M:%S.0", gmtime($ts));

    return $s;
}

sub date_epp {
    my ($ts) = @_;

    $ts =gettimeofday if (!$ts);
    my ($sec, $usec) = split(/\./, $ts);

    my $s = POSIX::strftime("%Y-%m-%dT%H:%M:%S", gmtime($ts));
    $usec = 0 unless($usec);
    $usec = int($usec / 100000);
    $s .= '.' . $usec . 'Z';

    return $s;
}

sub from_date_rrp {
    my ($s) = @_;

    my ($year, $month, $day,
	$hour, $min, $sec, $usec) = split(/[\ \.\:\-]+/, $s);

    $hour = 0 unless defined($hour);
    $min = 0 unless defined($min);
    $sec = 0 unless defined($sec);
    $usec = 0 unless defined($usec);

    $month--;
    $year -= 1900;

    my $tz = $ENV{TZ};
    $ENV{TZ} = "UTC";
    my $ts = POSIX::mktime($sec, $min, $hour, $day, $month, $year);
    if (defined($tz)) {
	$ENV{TZ} = $tz;
    } else {
	delete($ENV{TZ});
    }

    if ($usec) {
	$ts += ($usec / 10);
    }

    return $ts;
}

sub from_date_epp {
    my ($s) = @_;

    my ($year, $month, $day,
	$hour, $min, $sec, $usec) = split(/[\ \.\:\-TZ]+/, $s);

    $hour = 0 unless defined($hour);
    $min = 0 unless defined($min);
    $sec = 0 unless defined($sec);
    $usec = 0 unless defined($usec);

    $month--;
    $year -= 1900;

    my $tz = $ENV{TZ};
    $ENV{TZ} = "UTC";
    my $ts = POSIX::mktime($sec, $min, $hour, $day, $month, $year);
    if (defined($tz)) {
	$ENV{TZ} = $tz;
    } else {
	delete($ENV{TZ});
    }

    if ($usec =~ /^[0-9]+$/) {
	$ts += ($usec / (10 ** length($usec)));
    }

    return $ts;
}

sub date_add {
    my ($now, $add) = @_;

    $now = gettimeofday unless ($now);

    my $tz = $ENV{TZ};
    $ENV{TZ} = "UTC";

    my ($sec, $min, $hour, $day, $month, $year) = POSIX::gmtime($now);

    while ($add--) {
	$month++;
	if ($month == 12) {
	    $month = 0;
	    $year++;
	}
    }

    my $then;
    while (!defined($then)) {
	$then = POSIX::mktime($sec, $min, $hour, $day, $month, $year);

	if (!defined($then)) {
	    $day = 1;
	    $month++;
	    if ($month == 12) {
		$month = 0;
		$year++;
	    }
	}
    }

    if (defined($tz)) {
	$ENV{TZ} = $tz;
    } else {
	delete($ENV{TZ});
    }

    return $then;
}

1;

__END__

=head1 NAME

ISC::Date - Date and Time functions

=head1 SYNOPSIS

use ISC::Date qw(:all);


=head1 DESCRIPTION

This package handles date and time conversion to and from ANSI dates,
RRP protocol dates, and other pretty-printing of dates used in the ISC
SRS system.  Many of these are of general use.

UTC is assumed in all conversions.

=over

=item date_ansi TS

Return the date in ANSI format, suitable for passing into PostgreSQL or
other database engines.  The format is:

    YYYY-MM-DD HH:MM:SS[.ssssss]

TS is the Unix timestamp value.  If it is not provided, or is 0, this
function will use C<gettimeofday> from the L<Time::HiRes> package.

UTC is assumed in all conversions.

=item from_date_ansi STRING

Converts C<STRING> from an ANSI date value to Unix timestamp.

If the time part is omitted, it will use "00:00:00".

For example:

    2002-08-15

would be processed as:

    2002-08-15 00:00:00

If subsecond resolution is provided, it will be parsed and used.

UTC is assumed in all conversions.

=item date_rrp TS

Return the date in RRP protocol format, suitable for passing to a RRP
client or server.  The format is:

    YYYY-MM-DD HH:MM:SS.0

The trailing sub-second is currently hard-coded to 0, since it is unclear
if RRP clients will actually understand better resolution, since no
RRP servers known to exist will return anything but 0.

TS is the Unix timestamp value.  If it is not provided, or is 0, this
function will use C<gettimeofday> from the L<Time::HiRes> package.

UTC is assumed in all conversions.

=item date_epp TS

Return the date in EPP protocol format, suitable for passing to a EPP
client or server.  The format is:

    YYYY-MM-DDTHH:MM:SS.0Z (for example, 2002-05-19T12:34:56.7Z)

TS is the Unix timestamp value.  If it is not provided, or is 0, this
function will use C<gettimeofday> from the L<Time::HiRes> package.

UTC is assumed in all conversions.

=item from_date_rrp

Same as C<from_date_ansi> but for RRP format data.

If sub-seconds are provided (that is, the last digit isn't .0) it will
be returned.

UTC is assumed in all conversions.

=item from_date_epp

Same as C<from_date_ansi> but for EPP format data.

Sub-seconds are provided.

UTC is assumed in all conversions.

=item date_add TS MONTHS

Add C<MONTHS> months to the timestamp C<TS>.  Will properly handle
leap-years by moving to the first of the following month.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
