# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: LengthPrefix.pm,v 1.18.4.1 2003/04/08 16:35:35 lidl Exp $

package ISC::Net::LengthPrefix;

use strict;
use warnings;

#
# Perl, by default, returns the number of CHARACTERS when length($foo) is
# called, which is usually the same as the number of bytes used to store
# that string.  However, if Perl thinks that string is a utf8 encoded
# character string, length() != sizeof(), and substr() changes to work
# on characters, etc.
#
# Do NOT remove this "use bytes" unless you REALLY know what you're doing.
#
use bytes;

use Carp;
use Event;
use Errno;
use IO::Socket;
use Time::HiRes qw(gettimeofday);

our @EXPORT_OK;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.18.4.1 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

use constant MAX_IO_SIZE => 256;
use constant MAX_MSG_SIZE => (1024 * 128);
use constant MAX_QUEUE_LENGTH => (MAX_MSG_SIZE * 4);

use constant STATE_READ_LENGTH => 1;
use constant STATE_READ_DATA   => 2;
use constant STATE_DEAD_JIM    => 3;
use constant STATE_READ_ONESHOT => 4;
use constant STATE_READ_ONESHOTRESET => 5;
use constant STATE_SHUTTING_DOWN => 6;

my $sock_count = 0;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $name;
    if ($args{name}) {
	$name = $args{name};
    } else {
	$name = "ListenPrefix" . $sock_count++;
    }

    my $self = bless({}, $class);

    croak "Missing argument:  cb" if (!$args{cb});
    croak "Missing argument:  error_cb" if (!$args{error_cb});
    croak "Missing argument:  socket" if (!$args{socket});

    if ($args{lenbytes}) {
	croak "Invalid argument: lenbytes must be 1, 2 or 4"
	    if ($args{lenbytes} != 1
		&& $args{lenbytes} != 2
		&& $args{lenbytes} != 4);
    }

    $self->{cb} = $args{cb};
    $self->{error_cb} = $args{error_cb};
    $self->{data} = $args{data};
    $self->{name} = $name;

    $self->{lenbytes} = $args{lenbytes} || 4;

    $self->{sock} = $args{socket};

    $self->{sock}->blocking(0);
    $self->{sock}->sockopt(SO_KEEPALIVE, 1);
#    $self->{sock}->sockopt(SO_SNDBUF, 128 * 1024);
#    $self->{sock}->sockopt(SO_RCVLOWAT, 4);
#    $self->{sock}->sockopt(SO_SNDTIMEO, 0);

    $self->{write_queue} = "";
    $self->{input_queue} = "";
    $self->{input_len} = 0;
    $self->{state} = STATE_READ_LENGTH;

    $self->{oneshot} = $args{oneshot} || 0;
    $self->{flush} = $args{flush} || 0;
    $self->{headercounted} = $args{headercounted} || 0;

    $self->{max_read_size} = $args{max_read_size} || MAX_MSG_SIZE;
    $self->{max_queue_length} = $args{max_queue_length} || MAX_QUEUE_LENGTH;

    if ($self->{max_read_size} > $self->{max_queue_length}) {
	croak "Invalid arguments:  max_read_size must be <= max_queue_length";
    }

    $self->{reader} = Event->io(desc => "$name reader",
			       fd => $self->{sock},
			       cb => \&_sock_read,
			       poll => "r");
    $self->{writer} = Event->io(desc => "$name writer",
			       fd => $self->{sock},
			       cb => \&_sock_write,
			       poll => "w",
			       parked => 1);
    $self->{timer} = Event->timer(cb => \&_timer_cb,
				  parked => 1);


    $self->{reader}->data($self);
    $self->{writer}->data($self);
    $self->{timer}->data($self);

    return $self;
}

sub data {
    my ($self, $data) = @_;

    if (defined($data)) {
	$self->{data} = $data;
    }

    return $self->{data};
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub socket {
    my ($self) = @_;

    return $self->{sock};
}

sub stop {
    my ($self) = @_;

    $self->{reader}->stop if defined($self->{reader});
    $self->{writer}->stop if defined($self->{writer});
    $self->{timer}->stop if defined($self->{timer});
}

sub start {
    my ($self) = @_;

    $self->{reader}->start if defined($self->{reader});
    $self->{writer}->start if defined($self->{writer});
    $self->{timer}->start if defined($self->{timer});
}

sub cancel {
    my ($self) = @_;

    $self->{reader}->cancel;
    $self->{writer}->cancel;
    $self->{timer}->cancel;

    #
    # destroy this socket fully
    #
    $self->{reader} = undef;
    $self->{writer} = undef;
    $self->{timer} = undef;

    #
    # close the socket down
    #
    $self->{sock} = undef;

    $self->{state} = STATE_DEAD_JIM;
    $self->{data} = undef;
}

#
# queue a write on a socket.  This code will append the new data to the
# queue, then set the look-for-writable flag on the socket.  The event
# system should handle the rest.
#
sub send {
    my ($self, $data) = @_;

    croak "It's dead, jim"
	if ($self->{state} == STATE_DEAD_JIM);
    croak "Cannot send when shutting down"
	if ($self->{state} == STATE_SHUTTING_DOWN);

    my $sock = $self->{sock};
    my $was_empty = (length($self->{write_queue}) == 0);

    my $datalen = length($data);

    if ($self->{max_queue_length}) {
	my $len = $datalen + length($self->{write_queue}) + 4;
	if ($len > $self->{max_queue_length}) {
	    croak "Send queue length exceeded";
	}
    }

    if ($self->{lenbytes} == 4) {
	# don't even bother checking length...
	$self->{write_queue} .= pack("N", $datalen + ($self->{headercounted} ? 4 : 0)) . $data;
    } elsif ($self->{lenbytes} == 2) {
	if ($datalen > 65535) {
	    croak "Error:  lenbytes == 2, but length is $datalen";
	}
	$self->{write_queue} .= pack("n", $datalen + ($self->{headercounted} ? 2 : 0)) . $data;
    } elsif ($self->{lenbytes} == 1) {
	if ($datalen > 255) {
	    croak "Error:  lenbytes == 1, but length is $datalen";
	}
	$self->{write_queue} .= pack("C", $datalen + ($self->{headercounted} ? 1 : 0)) . $data;
    }

    if ($was_empty) {
	$self->{writer}->start;
    }
}

sub _timer_cb {
    my ($event) = @_;

    my $self = $event->w->data;

    if ($self->{state} == STATE_SHUTTING_DOWN) {
	$self->{error_cb}->($self, "EOF/TIMEOUT");
	$self->stop;
    }
}

#
# read data from a socket.  Reads the length and data, and will call upper
# layers when there is a complete read.
#
sub _sock_read {
    my ($event) = @_;

    my $self = $event->w->data;

    if ($self->{state} == STATE_DEAD_JIM) {
	$self->stop;
	return;
    }

    if ($self->{state} == STATE_READ_ONESHOT) {
	$self->{reader}->stop;
	return;
    }

    my $sock = $self->{sock};

    if ($self->{state} == STATE_READ_ONESHOTRESET) {
	$self->{state} = STATE_READ_LENGTH;
    } else {
	my $buf;
	my $cc;
	eval {
	    $cc = $sock->sysread($buf, MAX_IO_SIZE);
	};
	if ($@ || !defined($cc)) {
	    if ($!{EWOULDBLOCK} || $!{EINPROGRESS}) {
		return;
	    }
	    $self->{error_cb}->($self, "SOCKET READ: $!");
	    $self->{state} = STATE_DEAD_JIM;
	    $self->stop;
	    return;
	}
	if ($cc == 0) {
	    $self->{error_cb}->($self, "EOF");
	    $self->{state} = STATE_DEAD_JIM;
	    $self->stop;
	    return;
	}
	if ($self->{state} == STATE_SHUTTING_DOWN) {
	    return;
	}
	$self->{input} .= $buf;
    }

  again:
    #
    # First read the 4 byte prefix, then the data.  Do not combine these
    # with an else-if, since it's possible that both will be hit if
    # a single read returns both length and msg.
    #
    if ($self->{state} == STATE_READ_LENGTH
	&& length($self->{input}) > $self->{lenbytes}) {
	my ($len, $remaining);
	if ($self->{lenbytes} == 4) {
	    ($len, $remaining) = unpack("N a*", $self->{input});
	} elsif ($self->{lenbytes} == 2) {
	    ($len, $remaining) = unpack("n a*", $self->{input});
	} else {
	    ($len, $remaining) = unpack("C a*", $self->{input});
	}

	$self->{input} = $remaining;
	$self->{len} = $len - ($self->{headercounted} ? $self->{lenbytes} : 0);

	if ($self->{max_read_size} < $len) {
	    $self->{error_cb}->($self, "SOCKET READ: $len exceeds io size");
	    $self->{state} = STATE_DEAD_JIM;
	    $self->stop;
	    return;
	}
	$self->{state} = STATE_READ_DATA;
    }
    if (($self->{state} == STATE_READ_DATA) &&
	(length($self->{input}) >= $self->{len})) {
	my $msg = substr($self->{input}, 0, $self->{len});
	$self->{input} = substr($self->{input}, $self->{len});
	$self->{len} = 0;
	$self->{state} = STATE_READ_LENGTH;

	if ($self->{oneshot}) {
	    $self->{reader}->stop;
	    $self->{state} = STATE_READ_ONESHOT;
	}
	$self->{cb}->($self, $msg);

	if (length($self->{input}) > $self->{lenbytes}
	    && $self->{state} != STATE_READ_ONESHOT) {
	  goto again;
	}
    }
}

sub oneshot_reset {
    my ($self) = @_;

    return if ($self->{state} != STATE_READ_ONESHOT);

    $self->{state} = STATE_READ_ONESHOTRESET;
    if ($self->{flush} && $self->{write_queue}) {
	return;
    }
    $self->{reader}->start;
    $self->{reader}->now;
}

#
# write available on a socket
#
sub _sock_write {
    my ($event) = @_;

    my $self = $event->w->data;

    if (length($self->{write_queue}) == 0) {
	$self->{writer}->stop;
	return;
    }

    $self->_write_data;
}

sub _write_data {
    my ($self) = @_;

    my $sock = $self->{sock};

    my $cc;
    eval {
	$cc = $sock->syswrite($self->{write_queue});
    };
    if ($@ || !defined($cc)) {
	if ($!{EWOULDBLOCK} || $!{EINPROGRESS}) {
	    return;
	}
	$self->{error_cb}->($self, "SOCKET WRITE: $!");
	$self->{state} = STATE_DEAD_JIM;
	$self->stop;
	return;
    }
    if ($cc) {
	$self->{write_queue} = substr($self->{write_queue}, $cc);
	if (length($self->{write_queue}) == 0) {
	    $self->{writer}->stop;
	}
    }

    if ($self->{state} == STATE_READ_ONESHOTRESET
	&& $self->{flush}
	&& !$self->{write_queue}) {
	$self->{reader}->start;
	$self->{reader}->now;
    }

    if ($self->{state} == STATE_SHUTTING_DOWN
	&& !$self->{write_queue}) {
	$self->{sock}->shutdown(1);  # shut down writer
	$self->{input_queue} = "";
	$self->{reader}->start; # start reader.  It will toss all input.
	$self->{reader}->now;
    }
}

sub shutdown {
    my ($self, $timeout) = @_;

    $self->{state} = STATE_SHUTTING_DOWN;
    $self->{shutdown_timeout} = $timeout || 120;

    if (!$self->{write_queue}) {
	$self->{sock}->shutdown(1);
	$self->{input_queue} = "";
	$self->{reader}->start; # start reader.  It will toss all input.
	$self->{reader}->now;
    }

    #
    # Set the timer of death.  We may want to touch the timer if
    # writes are happening, to avoid it going off when it really isn't
    # needed yet.  But, on the other hand, we may not.  Setting it here
    # gives us a definite time to live before we will drop the other
    # end forcefully.  XXXMLG
    #
    if ($self->{shutdown_timeout}) {
	$self->{timer}->at(time + $self->{shutdown_timeout});
	$self->{timer}->start;
    }
}

1;

__END__

=head1 NAME

ISC::Net::LengthPrefix - Read a 1, 2, or 4 byte length, then data

=head1 SYNOPSIS

use ISC::Net::LengthPrefix;

my $lp = new ISC::Net::LengthPrefix(
    cb => \&callback,
    error_cb => \&error_callback,
    socket => $sock,
    ...);

callback($lp, $lines);

error_callback($lp, $errmsg);

=head1 DESCRIPTION

This package uses the C<Event> to listen on a socket, and will read a 1,
2, or 4-byte length prefix (in network byte order) and then that many
data bytes.  On each successful read of a data block, the callback is
called.

=over

=item new ARGS

Create a new LengthPrefix object.

C<cb> is the application callback, and is required.  When a block of
input is read, this function is called for every block.  See C<oneshot>,
below, for a method to keep the application in control of when these
messages are sent.

C<error_cb> is called when there is a socket error.  Errors are fatal, and
the socket should be closed down.

The C<socket> argument should be an C<IO::Socket> or a subclass, such as
C<IO::Socket::INET>.

C<data> is application specific state.  It can be set during the C<new>
call, or using the data() method.

C<name> is the application's name for this object.  It can be retrieved
using the name() method.  If it is not set, a default name will be
generated.

C<oneshot> can be set, which gives the application control over the reading
process.  If it is set, after a callback, the application must call
oneshot_reset() to receive any more events on this object.

C<flush> is used in conjunction with <oneshot> and will only cause another
read to happen after the write queue is fully written.

C<lenbytes> specifies the size of the prefix (both for input and output)
and can be 1, 2, or 4.  If not specified, it defaults the commonly used
4.

C<max_read_size> controls the maximum size for a block of data for input.
As soon as enough bytes are read to know the length, this value is checked,
and if exceeded, the error callback will be performed and the reader shut
down.  The default is 128K (131,072 bytes.)

C<max_queue_length> controls the maximum output queue size.  If it is
exceeded, the send() method will fail.  Callers may wish to call send()
in an eval block so this can be caught.

=item data DATA

Returns the current application-defined data.  If C<DATA> is provided, it
will set a new value (and return it).

=item name

Returns the object's name.

=item socket

Returns the socket associated with this object.

=item stop

Stop all events on this object.  They may be restarted with C<start>.

=item start

Start all events on this object, after being stopped with C<stop>.

=item cancel

Fully shut down this object.  All pending events are cancelled, and
the socket is released.

=item send DATA

Queue C<DATA> for output to the socket.

If this will exceed a quota, it will raise an error using C<croak>.

This will implicitly start the writer process if it is stopped.

=item oneshot_reset

If the LengthPrefix object is created with C<oneshot> set, this function
must be called when the application wishes to continue receiving callbacks
for new data.

=item shutdown TIME

Shut down the write side of the socket.  If TIME is specified, it is a
timeout on how long we will wait for output to flush and the other end
to close the socket.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
