# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Message.pm,v 1.12 2002/12/06 02:21:10 lidl Exp $

package ISC::CC::Message;

use strict;
use warnings;

use Carp;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.12 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw(towire fromwire);
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

my $PROTOCOL_VERSION = 0x536b616e;

my $ITEM_DATA = 0x01;
my $ITEM_HASH = 0x02;
my $ITEM_LIST = 0x03;
my $ITEM_NULL = 0x04;
my $ITEM_TYPE_MASK = 0x0f;

my $ITEM_LENGTH_32 = 0x00;
my $ITEM_LENGTH_16 = 0x10;
my $ITEM_LENGTH_8  = 0x20;
my $ITEM_LENGTH_MASK = 0x30;

my $_typeenc = {
    "SCALAR" => { code => $ITEM_DATA,
		  pack => \&_pack_scalarref },
    "HASH" => { code => $ITEM_HASH,
		pack => \&_pack_hashref,
		unpack => \&_unpack_hash },
    "ARRAY" => { code => $ITEM_LIST,
		 pack => \&_pack_arrayref,
		 unpack => \&_unpack_list }
};
my @_typedec;
$_typedec[$ITEM_HASH] = $_typeenc->{"HASH"};
$_typedec[$ITEM_LIST] = $_typeenc->{"ARRAY"};

sub towire {
    my ($msg) = @_;

    my $buf = pack "N", $PROTOCOL_VERSION;

    $buf .= _encode_hash($msg);

    return $buf;
}

sub _pack_scalarref {
    my ($code, $valp) = @_;

    my $val = $valp;

    if (!defined($val)) {
	return pack("C", $ITEM_NULL);
    }

    my $len = length($val);
    if ($len < 0xff) {
	return pack("C C/a*", $code | $ITEM_LENGTH_8, $val);
    } elsif ($len < 0xffff) {
	return pack("C n/a*", $code | $ITEM_LENGTH_16, $val);
    }

    return pack("C N/a*", $code, $val);
}

sub _pack_hashref {
    my ($code, $val) = @_;

    my $x = _encode_hash($val);
    my $len = length($x);
    if ($len < 0xff) {
	return pack("C C/a*", $code | $ITEM_LENGTH_8, $x);
    } elsif ($len < 0xffff) {
	return pack("C n/a*", $code | $ITEM_LENGTH_16, $x);
    }

    return pack("C N/a*", $code, $x);
}

sub _pack_arrayref {
    my ($code, $val) = @_;

    my $x = _encode_array($val);
    my $len = length($x);
    if ($len < 0xff) {
	return pack("C C/a*", $code | $ITEM_LENGTH_8, $x);
    } elsif ($len < 0xffff) {
	return pack("C n/a*", $code | $ITEM_LENGTH_16, $x);
    }

    return pack("C N/a*", $code, $x);
}

sub _encode_val {
    my ($val) = @_;

    my $type = ref($val);
    if (!$type) {
	if (!defined($val)) {
	    return pack("C", $ITEM_NULL);
	}
	my $len = length($val);
	if ($len < 0xff) {
	    return pack("C C/a*", $ITEM_DATA | $ITEM_LENGTH_8, $val);
	} elsif ($len < 0xffff) {
	    return pack("C n/a*", $ITEM_DATA | $ITEM_LENGTH_16, $val);
	}
	return pack("C N/a*", $ITEM_DATA, $val);
    } else {
	my $func = $_typeenc->{$type};
	if (!$func) {
	    die "Unknown reference type in _encode_val ($type)";
	}

	return $func->{pack}->($func->{code}, $val);
    }
}

sub _encode_array {
    my ($arr) = @_;

    my $ret = "";

    foreach my $val (@$arr) {
	if (!defined($val)) {
	    $ret .= pack("C", $ITEM_NULL);
	} else {
	    my $len = length($val);
	    if (!ref($val)) {
		my $len = length($val);
		if ($len < 0xff) {
		    $ret .= pack("C C/a*", $ITEM_DATA | $ITEM_LENGTH_8, $val);
		} elsif ($len < 0xffff) {
		    $ret .= pack("C n/a*", $ITEM_DATA | $ITEM_LENGTH_16, $val);
		} else {
		    $ret .= pack("C N/a*", $ITEM_DATA, $val);
		}
	    } else {
		$ret .= _encode_val($val);
	    }
	}
    }

    return $ret;
}

sub _encode_hash {
    my ($hash) = @_;

    my $ret = "";

    foreach my $tag (keys %$hash) {
	if (!ref($hash->{$tag})) {
	    my $val = $hash->{$tag};
	    if (!defined($val)) {
		$ret .= pack("C/A* C", $tag, $ITEM_NULL);
	    } else {
		my $len = length($val);
		if ($len < 0xff) {
		    $ret .= pack("C/A* C C/a*", $tag,
				 $ITEM_DATA | $ITEM_LENGTH_8, $val);
		} elsif ($len < 0xffff) {
		    $ret .= pack("C/A* C n/a*", $tag,
				 $ITEM_DATA | $ITEM_LENGTH_16, $val);
		} else {
		    $ret .= pack("C/A* C N/a*", $tag, $ITEM_DATA, $val);
		}
	    }
	} else {
	    $ret .= pack("C/A* a*", $tag, _encode_val($hash->{$tag}));
	}
    }

    return $ret;
}

sub fromwire {
    my ($buf) = @_;

    croak "undef value" if (!defined($buf));

    return _unpack($buf);
}

sub _unpack {
    my ($buf) = @_;

    my $msg;

    my $proto_version = unpack("N", $buf);
    die "Malformed packet" if (!defined($proto_version));
    die "Protocol version mismatch" if ($proto_version != $PROTOCOL_VERSION);

    $buf = substr($buf, 4);

    return _unpack_hash($buf);
}

sub _unpack_val {
    my ($type, $buf) = @_;

    if ($type == $ITEM_DATA) {
	return $buf;
    }

    if ($type == $ITEM_NULL) {
	return undef;
    }

    $type &= $ITEM_TYPE_MASK;
    foreach my $t (keys %$_typeenc) {
	my $tt = $_typeenc->{$t};
	if ($tt->{code} == $type) {
	    return $tt->{unpack}->($buf);
	}
    }
#    my $func = $_typedec[ord($type) - 0x30];
#    if (!$func) {
#	print Data::Dumper->Dump([\@_typedec], ['t']);
	die "Unknown value type $type";
#    }

#    return $func->{unpack}->($func->{code}, $buf);
}

sub _unpack_hash {
    my ($buf) = @_;

    my $ret = {};

    my $cur = $buf;
    while ($cur) {
	my ($tag, $typelen, $val);
	($tag, $typelen, $cur) = unpack("C/A* C a*", $cur);
	my $type = $typelen & $ITEM_TYPE_MASK;
	my $len = $typelen & $ITEM_LENGTH_MASK;
	if ($type == $ITEM_NULL) {
	    $ret->{$tag} = undef;
	} else {
	    if ($len == $ITEM_LENGTH_8) {
		($val, $cur) = unpack("C/a* a*", $cur);
	    } elsif ($len == $ITEM_LENGTH_16) {
		($val, $cur) = unpack("n/a* a*", $cur);
	    } else {
		($val, $cur) = unpack("N/a* a*", $cur);
	    }
	    if ($type == $ITEM_DATA) {
		$ret->{$tag} = $val;
	    } else {
		$ret->{$tag} = _unpack_val($type, $val);
	    }
	}
    }

    return $ret;
}

sub _unpack_list {
    my ($cur) = @_;

    my $arr = [];

    while ($cur) {
	my ($typelen, $val);
	($typelen, $cur) = unpack("C a*", $cur);
	my $type = $typelen & $ITEM_TYPE_MASK;
	my $len = $typelen & $ITEM_LENGTH_MASK;
	if ($type == $ITEM_NULL) {
	    push @$arr, undef;
	} else {
	    if ($len == $ITEM_LENGTH_8) {
		($val, $cur) = unpack("C/a* a*", $cur);
	    } elsif ($len == $ITEM_LENGTH_16) {
		($val, $cur) = unpack("n/a* a*", $cur);
	    } else {
		($val, $cur) = unpack("N/a* a*", $cur);
	    }
	    if ($type == $ITEM_DATA) {
		push @$arr, $val;
	    } else {
		push @$arr, _unpack_val($type, $val);
	    }
	}
    }

    return $arr;
}

1;
