#!/usr/bin/env perl
#
# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: xaction.pl,v 1.99.2.1 2003/02/21 23:10:52 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Carp;
use Data::Dumper;
use Event qw(loop unloop);
use Time::HiRes qw(gettimeofday);
use POSIX qw(gmtime strftime);
use Getopt::Long;

use ISC::CC::Group::Connect;

use ISC::SRS::DB;
use ISC::SRS::EPPResultCode;
use ISC::SRS::Version;

use ISC::Date ":all";
use ISC::Log;

my $srsdb_channel = "srsdb";
my $result = GetOptions("srschannel=s" => \$srsdb_channel);

my $msgbus = new ISC::CC::Group::Connect(timeout => 10,
					 connect_error_cb => \&_connect_err,
					 error_cb => \&_err,
					 connect_cb => \&_connect);

my $log;
my $db;
my $apex;

Event->timer(cb => \&_tick, interval => 1);

my $transactions = 0;
my $start = gettimeofday;

# a counter for the active session number
my $sessionid = 100;
my $state = {};
my ($sdate, $usec) = split(/\./, from_date_ansi(date_ansi()));
$state->{starttime} = $sdate;
#print "setting starttime to $state->{starttime}\n";

sub _tick {
    my ($ev) = @_;

    my $now = gettimeofday;
    my $interval = $now - $start;
    $start = $now;

    if ($transactions) {
	printf("%04d done in %8.3f seconds (%8.4f/sec)\n",
	       $transactions, $interval, $transactions / $interval);
	$transactions = 0;
    }
}

sub _connect {
    my ($gc) = @_;

    $log = new ISC::Log(facility => "xaction($srsdb_channel)",

			maxlevel_stderr => ISC_LOG_DEBUG,

			msgbus => $gc,
			group => "log",
			instance => "xaction",
			maxlevel_msgbus => ISC_LOG_INFO,
			);

    $log->log(ISC_LOG_INFO,
	       "Connected to msgbus, local name is " . $gc->myname);

    $gc->join(group => $srsdb_channel,
	      instance => "epp_req",
	      cb => \&_msg_cb);

    $db = new ISC::SRS::DB(replication => "master",
			   replication_name => "srs",
			   group_handle => $gc);

    $apex = $db->apex;

    $log->log(ISC_LOG_INFO, "Zone apex: $apex");
}

# function name, need_args

my $typemap = {
    fe_start => [ \&_msg_fe_start, 0 ],
    login => [ \&_msg_login, 1 ],
    logout => [ \&_msg_logout, 1 ],
    hello => [ \&_msg_hello, 0 ],

    contact_add => [ \&_msg_contact_add, 1 ],
    contact_check => [ \&_msg_contact_check, 1 ],
    contact_del => [ \&_msg_contact_del, 1 ],
    contact_info => [ \&_msg_contact_info, 1 ],
    contact_mod => [ \&_msg_contact_mod, 1 ],

    domain_add => [ \&_msg_domain_add, 1 ],
    domain_check => [ \&_msg_domain_check, 1 ],
    domain_del => [ \&_msg_domain_del, 1 ],
    domain_mod => [ \&_msg_domain_mod, 1 ],
    domain_renew => [ \&_msg_domain_renew, 1 ],
    domain_info => [ \&_msg_domain_info, 1 ],
    domain_transfer => [ \&_msg_domain_transfer, 1 ],

    host_add => [ \&_msg_host_add, 1 ],
    host_check => [ \&_msg_host_check, 1 ],
    host_del => [ \&_msg_host_del, 1 ],
    host_mod => [ \&_msg_host_mod, 1 ],
    host_info => [ \&_msg_host_info, 1 ],

    iscadmin => [ \&_msg_iscadmin, 1 ],
};

sub _msg_cb {
    my ($c, $msg, $wmsg) = @_;

    #
    # quick stab at transaction recording
    #
    $transactions++;

    my $type = $msg->{type};
    my $args = $msg->{args};
    my $addl = $msg->{addl} || {};
    my $repl;
   
#    print Data::Dumper->Dump([$wmsg, $msg], ['wmsg', 'msg']);
 
    if (!$type) {
	$repl= _make_result(EPP_RF_FAIL,
			       'Missing argument "type"');
	$log->log(ISC_LOG_WARNING, "replying with missing \"type\"");
	$c->reply(msg => $repl);
	return;
    }

    my ($func, $needargs) = @{$typemap->{$type}};
    if (!$func) {
	$repl = _make_result(EPP_RF_FAIL,
				"Unknown operation type $type");
	$log->log(ISC_LOG_WARNING, "replying with unknown type");
	$c->reply(msg => $repl);
	return;
    }

    if ($needargs && !$args) {
	$repl= _make_result(EPP_RF_FAIL,
			       'Missing argument "arg"');
	$log->log(ISC_LOG_WARNING, "replying with missing \"arg\"");
	$c->reply(msg => $repl);
	return;
    }

    # this is where we enforce the shared state before the frontend
    # and the backend processes.
    #
    # we punt any command (except fe_start, login, and logout)
    # that does not already have (feid/sid) state stored in the backend
    #
    # if the command is login, and we haven't heard from the
    # frontend previously, create some state for the frontend
    # and continue processing the command
    #
    # for any other command, if we find a previously unknown frontend,
    # we create new state for that frontend and reject that command in
    # a manner such that the frontend will terminate the session.

    if ($type =~ m/^(fe_start)$/) {
	$repl = $func->($args, $addl);
    } elsif ($type =~ m/^(login|logout)$/) {
	my $feid = $addl->{feid};

	if (!defined($state->{$feid}->{name})) {
	    _fe_create($feid, "Implicit");
	}

	$repl = $func->($args, $addl);

    } else {
	my $feid = $addl->{feid};
	my $sid = $addl->{sid};

	if (!defined($state->{$feid}->{name})) {
	    _fe_create($feid, "Implicit");
	}

	if (!defined($state->{$feid}->{$sid})) {
	    $repl= _make_result(EPP_RF_CLOSING,'backend restart found: no sid in state');
	} else {
	    my ($stime,$trash) = split(/-/, $sid);
	    if ($state->{starttime} > $stime) {
		# we should *never* get here
		print "****\n**** $state->{starttime}, $stime\n****\n";
		$repl= _make_result(EPP_RF_CLOSING,'backend restart detected - old sid found in request!');
	    } else {
		$repl = $func->($args, $addl);
	    }
	}
    }

    if (defined($ENV{ISC_SRS_DEBUG})) {
	print Data::Dumper->Dump([$repl], ['repl']);
    }
    $c->reply(msg => $repl);
}

# create the state needed for a frontend and log it
sub _fe_create {
    my ($feid, $type) = @_;

    $state->{$feid}->{name} = $feid;
    my $date = from_date_ansi(date_ansi());
    $state->{$feid}->{starttime} = $date;
#    print "created backend state for feid $feid (at $date)!\n";

    $log->log(ISC_LOG_INFO, "$type front end start: $feid");

    return;
}

# --- FE_START ----------------------------------------------------- #

sub _msg_fe_start {
    my ($args, $addl) = @_;

    my $repl = {};
    my $feid = $addl->{feid};

    if (defined($state->{$feid})) {
#	print "deleting existing backend state for feid $feid.\n";
	# dump state here
#	print Data::Dumper->Dump([$state->{$feid}], ['$state->{$feid}']);
	delete $state->{$feid};
    }
    _fe_create($feid, "Explicit");
    return $repl;
}


# --- LOGIN -------------------------------------------------------- #

sub _msg_login {
    my ($args, $addl) = @_;

    my ($code, $xmsg, $lines, $reg) = $db->login(%$args);

    my $repl = _make_result($code, $xmsg);

    if ($code == EPP_RS_SUCCESS) {
	my $feid = $addl->{feid};
	my $id = $reg->{registrar_id};
	my $date = from_date_ansi(date_ansi());
	my $sid = $state->{starttime} . "-" . $sessionid++;

	if (!defined($state->{$feid}->{name})) {
	    # something bad has happened -- the backend has been
	    # restarted, with one or more existing frontend connections.
	    # reject the login with some error message such that the
	    # frontend will kill all client sessions
	    print "**** backend received a message from an unknown frontend!\n";
	    print "**** this should never happen!\n";
	}
	$repl->{sid} = $sid;
	$repl->{id} = $id;
	$state->{$feid}->{$sid}->{sid} = $sid;
	$state->{$feid}->{$sid}->{id} = $id;
	$state->{$feid}->{$sid}->{starttime} = $date;

#	print "created session state for feid $feid -> $sid (at $date)\n";
	$log->log(ISC_LOG_INFO, "LOGIN registrar_id $id feid $feid sid $sid");
    }

    return $repl;
}


# --- LOGOUT ------------------------------------------------------- #

sub _msg_logout {
    my ($args, $addl) = @_;

    my $repl = _make_result(EPP_RS_END, undef);

    my $id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my $feid = $state->{$addl->{feid}}->{name};
    my $sid = $state->{$addl->{feid}}->{$addl->{sid}}->{sid};

    if (defined($sid)) {
	$log->log(ISC_LOG_INFO, "LOGOUT registrar_id $id feid $feid sid $sid");

	# remove the state for this session
#	print "deleting state for session $feid -> $sid\n";

	delete $state->{$feid}->{$sid};
    } else {
	$log->log(ISC_LOG_INFO, "LOGOUT from feid $feid unknown client!");
    }

    return $repl;
}


# --- HELLO ----------------------------------------------------- #

sub _msg_hello {
    my ($args, $addl) = @_;

#    print Data::Dumper->Dump([$state->{$addl->{feid}}], ['$state->{$feid}']);
#    print Data::Dumper->Dump([$addl], ['$addl']);

    my $repl = _make_result(EPP_RS_SUCCESS, undef);

    $repl->{now} = date_epp();
     
    return $repl;
}


# --- CONTACT_ADD --------------------------------------------------- #

sub _msg_contact_add {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "id"')
      unless ($args->{id});

#    print Data::Dumper->Dump([$args], ["args"]);

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my %largs = (created_id => $registrar_id,
		 registrar_id => $registrar_id,
		 authinfo => $args->{authinfo},
		 postal => $args->{postal},
		 id => $args->{id},
		 fax => $args->{fax},
		 voice => $args->{voice},
		 voice_ext => $args->{voice_ext},
		 email => $args->{email},
		 owner_id => $registrar_id);
    $largs{status} = $args->{status} if ($args->{status});

    my ($code, $xmsg, $list) = $db->contact_add(%largs);

    my $repl = _make_result($code, $xmsg);
    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- CONTACT_CHECK ------------------------------------------------- #

sub _msg_contact_check {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "ids"')
        unless ($args->{ids});

    my %largs = ( ids => $args->{ids} );

    my ($code, $xmsg, $list) = $db->contact_check(%largs);

    my $repl = _make_result($code, $xmsg);

    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- CONTACT_DEL --------------------------------------------------- #

sub _msg_contact_del {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "id"')
        unless ($args->{id});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my %largs = ( id => $args->{id},
                  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->contact_del(%largs);

    my $repl = _make_result($code, $xmsg);
    return $repl;
}


# --- CONTACT_INFO ------------------------------------------------- #

sub _msg_contact_info {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "id"')
        unless ($args->{id});

    my %largs = ( id => $args->{id} );

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my ($code, $xmsg, $list) = $db->contact_info(%largs);

    my $repl = _make_result($code, $xmsg);

    # owner identity checking -- remove items unless the requester
    # is the sponsering registrar

    if ($list->{registrar_id} == $registrar_id) {
	delete $list->{registrar_id};
    } else {
	delete $list->{registrar_id};

	delete $list->{authinfo};
    }

    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- CONTACT_MOD ------------------------------------------------- #

sub _msg_contact_mod {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "id"')
        unless ($args->{id});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my %largs = ( id => $args->{id},
		  authinfo => $args->{authinfo},
		  voice => $args->{voice},
		  voice_ext => $args->{voice_ext},
		  fax => $args->{fax},
		  email => $args->{email},
                  postal_add => $args->{postal_add},
                  postal_del => $args->{postal_del},
                  postal_chg => $args->{postal_chg},
                  status_add => $args->{status_add},
                  status_del => $args->{status_del},
		  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->contact_mod(%largs);

    my $repl = _make_result($code, $xmsg);
    return $repl;
}

# --- DOMAIN_ADD --------------------------------------------------- #

sub _msg_domain_add {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
      unless ($args->{name});

    my $period = $args->{period} || $addl->{period} || 2;
    my $units = $args->{units} || $addl->{units} || 'y';

    if (defined($ENV{ISC_SRS_DEBUG})) {
	print Data::Dumper->Dump([$args,$addl], ["args","addl"]);
    }

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = (period => $period,
		 units => $units,
		 fqdn => $args->{name},
		 created_id => $registrar_id,
		 status => $args->{status},
		 registrar_id => $registrar_id,
		 contact => $args->{contact},
		 nameservers => $args->{nameservers},
		 authinfo => $args->{authinfo},
		 owner_id => $args->{owner_id});
    $largs{nameservers} = $args->{nameservers} if ($args->{nameservers});

    my ($code, $xmsg, $list) = $db->domain_add(%largs);

    my $repl = _make_result($code, $xmsg);
    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- DOMAIN_CHECK ------------------------------------------------- #

sub _msg_domain_check {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "names"')
        unless ($args->{names});

    my %largs = ( names => $args->{names} );

    my ($code, $xmsg, $list) = $db->domain_check(%largs);

    my $repl = _make_result($code, $xmsg);

    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- DOMAIN_DEL --------------------------------------------------- #

sub _msg_domain_del {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = ( fqdn => $args->{name},
                  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->domain_del(%largs);

    my $repl = _make_result($code, $xmsg);
    return $repl;
}


# --- DOMAIN_MOD --------------------------------------------------- #

sub _msg_domain_mod {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});

    if (defined($ENV{ISC_SRS_DEBUG})) {
	print Data::Dumper->Dump([$args, $addl], ['args', 'addl']);
    }

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = ( fqdn => $args->{name},
		  authinfo => $args->{authinfo},
		  owner_id => $args->{owner_id},
                  nameservers_add => $args->{nameservers_add},
                  nameservers_del => $args->{nameservers_del},
                  contact_add => $args->{contact_add},
                  contact_del => $args->{contact_del},
                  status_add => $args->{status_add},
                  status_del => $args->{status_del},
		  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->domain_mod(%largs);

    my $repl = _make_result($code, $xmsg);
    return $repl;
}


# --- DOMAIN_RENEW ------------------------------------------------- #

sub _msg_domain_renew {
    my ($args, $addl) = @_;

    if (defined($ENV{ISC_SRS_DEBUG})) {
        print Data::Dumper->Dump([$args], ["argsXX"]);
    }

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});

    return _make_result(EPP_RF_PARAM, 'missing "current_date"')
        if (!defined($args->{current_date}));

    return _make_result(EPP_RF_PARAM, 'missing "period"')
       if (!defined($args->{period}));

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = ( fqdn => $args->{name},
		  registrar_id => $registrar_id,
		  current_date => $args->{current_date},
		  period_unit => $args->{period_unit},
		  period => $args->{period} );

    my ($code, $xmsg, $list) = $db->domain_renew(%largs);

    my $repl = _make_result($code, $xmsg);

    $repl->{expiry} = $list->{expiry} || undef;

    return $repl;
}


# --- DOMAIN_INFO ------------------------------------------------ #

sub _msg_domain_info {
    my ($args, $addl) = @_;

#    print Data::Dumper->Dump([$args, $addl], ['args', 'addl']);

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless($args->{name});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = (fqdn => $args->{name});

    my ($code, $xmsg, $list) = $db->domain_info(%largs);

    # owner identity checking -- remove items unless the requester
    # is the owner -- revisit when we have authinfo support - XXX
    if ($list->{registrar_id} == $registrar_id) {
	delete $list->{registrar_id};
    } else {
	delete $list->{registrar_id};

	# this is pretty much the entire list of optional attributes
	# listed on page 13 of the EPP Domain Name Mapping (draft-06)
	delete $list->{status};
	delete $list->{registrant};
	delete $list->{contact};
	delete $list->{ns};
	delete $list->{hosts};
	delete $list->{crID};
	delete $list->{crDate};
	delete $list->{exDate};
	delete $list->{upID} if (defined($list->{upID}));
	delete $list->{upDate} if (defined($list->{upDate}));
	delete $list->{trDate} if (defined($list->{trDate}));
	delete $list->{authinfo} if (defined($list->{authinfo}));
    }

    my $repl = _make_result($code, $xmsg, $list);

    return $repl;
}


# --- DOMAIN_TRANSFER ---------------------------------------------- #

sub _msg_domain_transfer {
    my ($args, $addl) = @_;

    my $repl = _make_result(EPP_RF_UNIMPLSERVICE, "notyet");

    return $repl;
}


# --- HOST_ADD ------------------------------------------------------- #

sub _msg_host_add {
    my ($args, $addl) = @_;

    my $fqdn = uc $args->{name};

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($fqdn);

    my $in_zone = ($fqdn =~ m/$apex$/i);
    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

#    print Data::Dumper->Dump([$args], ["args"]);

    return _make_result(EPP_RF_PARAM, 'missing ip address')
        if ($in_zone && !$args->{addrs});

    return _make_result(EPP_RF_PARAM, 'addresses specified for out-of-zone host')
        if (!$in_zone && $args->{addrs});

    my %largs = (fqdn => $fqdn,
		 created_id => $registrar_id,
		 registrar_id => $registrar_id,
		 status => $args->{status},
		 addrs => $args->{addrs});
    my ($code, $xmsg, $list) = $db->host_add(%largs);

    my $repl = _make_result($code, $xmsg);
    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}


# --- HOST_CHECK ----------------------------------------------------- #

sub _msg_host_check {
    my ($args, $addl) = @_;
 
    return _make_result(EPP_RF_PARAM, 'missing "names"')
        unless ($args->{names});

    my %largs = ( names => $args->{names} );

    my ($code, $xmsg, $list) = $db->host_check(%largs);

    my $repl = _make_result($code, $xmsg);

    foreach my $tag (keys %$list) {
	$repl->{$tag} = $list->{$tag};
    }

    return $repl;
}

# --- HOST_DEL ------------------------------------------------------- #

sub _msg_host_del {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my %largs = ( fqdn => $args->{name},
		  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->host_del(%largs);
    my $repl = _make_result($code, $xmsg);

    return $repl;
}


# --- HOST_MOD ----------------------------------------------------- #

sub _msg_host_mod {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});

    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};
    my %largs = ( fqdn => $args->{name},
                  rename => $args->{rename},
                  addr_add => $args->{addr_add},
		  addr_del => $args->{addr_del},
                  status_add => $args->{status_add},
		  status_del => $args->{status_del},
                  registrar_id => $registrar_id );

    my ($code, $xmsg, $list) = $db->host_mod(%largs);

    my $repl = _make_result($code, $xmsg);
    return $repl;
}


# --- HOST_INFO -------------------------------------------------- #

sub _msg_host_info {
    my ($args, $addl) = @_;

    return _make_result(EPP_RF_PARAM, 'missing "name"')
        unless ($args->{name});
    
    my $registrar_id = $state->{$addl->{feid}}->{$addl->{sid}}->{id};

    my %largs = ( fqdn => $args->{name} );
     
    my ($code, $xmsg, $list) = $db->host_info(%largs);

    # owner identity checking -- remove items unless the requester
    # is the owner -- currently this is an empty operation, but
    # this code is where the modification would be done, if it was
    # needed
    if ($list->{registrar_id} == $registrar_id) {
	delete $list->{registrar_id};
    } else {
	delete $list->{registrar_id};
    }

    my $repl = _make_result($code, $xmsg, $list);

    return $repl;
}


# --- ISCADMIN -------------------------------------------------- #

sub _msg_iscadmin {
    my ($args, $addl) = @_;

    my $repl = _make_result(EPP_RF_UNIMPCMD);

    return $repl;
}

# ---------------------------------------------------------------- #

sub _make_result {
    my ($code, $msg, $extra) = @_;

    my $ret = {
	result => {
	    resultcode => $code,
	},
    };

    if ($msg) {
	$ret->{result}->{xresult} = $msg;
#	print "result: $msg\n";
    }

    if ($extra) {
	foreach my $k (keys %$extra) {
	    $ret->{$k} = $extra->{$k};
	}
    }

    return $ret;
}

sub _err {
    my ($msgbus, $msg) = @_;

    $log->log(ISC_LOG_ERR, "MSGBUS error: $msg");
    unloop();
}

$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

$SIG{PIPE} = "IGNORE";

Event::loop();
