# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: DB.pm,v 1.160.2.1 2003/02/21 23:10:51 lidl Exp $

package ISC::SRS::DB;

use strict;
use warnings;

use Carp;

use Data::Dumper;
use MIME::Base64;

use ISC::CC::Message;
use ISC::DB::ReplicationMaster;
use ISC::DB::ReplicationSlave;
use ISC::DBI;
use ISC::Date qw(:all);
use ISC::SRS::EPPResultCode;
use ISC::SRS::Version;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.160.2.1 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

sub _check_required_args {
    my ($func, $args, $reqd) = @_;

    foreach my $i (@$reqd) {
	if (!exists($args->{$i})) {
	    croak "$func: Missing argument: $i";
	}
    }
}

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    my %largs;
    $largs{dbname} = $args{dbname} || $ENV{ISC_SRS_DBNAME} || "srs";
    $largs{dbuser} = $args{dbuser} || $ENV{ISC_SRS_DBUSER} || "srs";
    $largs{dbpass} = $args{dbpass} || $ENV{ISC_SRS_DBPASS} || undef;
    $largs{serializable} = $args{serializable};

    $self->{db} = new ISC::DBI(%largs);
    die "cannot connect to database" if (!$self->{db});
    $self->_init_sql;

    my $reptype = $args{replication} || "none";
    if ($reptype eq "none") {
    } elsif ($reptype eq "master") {
	_check_required_args("new", \%args, [ "replication_name" ]);
	my $repname = $args{replication_name};
	my $gc = $args{group_handle};
	$self->{rep} = new ISC::DB::ReplicationMaster(db => $self->{db},
						      name => $repname,
						      group_handle => $gc);
    } elsif ($reptype eq "slave") {
	_check_required_args("new", \%args, [ "replication_name" ]);
	my $repname = $args{replication_name};
	$self->{rep} = new ISC::DB::ReplicationSlave(db => $self->{db},
						     name => $repname);
    } else {
	die "Unknown replication type $reptype";
    }

    return $self;
}

sub DESTROY {
    my ($self) = @_;

    if ($self->{db}) {
	if ($self->{db}->in_transaction) {
	    warn "Destroying ISC::SRS::DB with transaction in progress";
	}
	if ($self->{db}->db) {
	    $self->{db}->db->disconnect;
	}
	$self->{db} = undef;
    }
    $self->{rep} = undef;
}

sub _init_sql {
    my ($self) = @_;

    my $db = $self->db->db;

    $self->{sql_newdid} = $db->prepare("
	SELECT nextval('domainid_seq') AS did");
    $self->{sql_newhid} = $db->prepare("
	SELECT nextval('hostid_seq') AS hid");
    $self->{sql_newcid} = $db->prepare("
	SELECT nextval('contactid_seq') AS cid");
    $self->{sql_newrid} = $db->prepare("
	SELECT nextval('registrarid_seq') AS rid");

    $self->{sql_contact_add} = $db->prepare("INSERT INTO contact
	     (st, contact_id, handle, guid, voice, voice_ext,
              fax, email, update, upid, crdate, crid, clid)
	      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $self->{sql_contact_setvoice} = $db->prepare("UPDATE contact
         SET st=?, voice=? WHERE contact_id=?");

    $self->{sql_contact_setvoice_ext} = $db->prepare("UPDATE contact
         SET st=?, voice_ext=? WHERE contact_id=?");

    $self->{sql_contact_setfax} = $db->prepare("UPDATE contact
         SET st=?, fax=? WHERE contact_id=?");

    $self->{sql_contact_setemail} = $db->prepare("UPDATE contact
         SET st=?, email=? WHERE contact_id=?");

    $self->{sql_contact_setupdate} = $db->prepare("UPDATE contact
         SET st=?, update=?, upid=? WHERE contact_id=?");

    $self->{sql_ca_add} = $db->prepare(
	 "INSERT INTO contact_authinfo
             (st, contact_id, authtype, authinfo)
             VALUES (?, ?, ?, ?)");

    $self->{sql_ca_del} = $db->prepare("
         DELETE FROM contact_authinfo WHERE contact_id=?");

    $self->{sql_cp_add} = $db->prepare("INSERT INTO contact_postal
         (st, contact_id, type, name, org, street1, street2, street3,
          city, sp, pc, cc)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $self->{sql_cp_del} = $db->prepare("DELETE FROM contact_postal
         WHERE contact_postal.contact_id=?
             AND contact_postal.type=?");

    $self->{sql_cp_update} = $db->prepare("UPDATE contact_postal
	 SET st=?, name=?, org=?, street1=?, street2=?,
	     street3=?, city=?, sp=?, pc=?, cc=?
	 WHERE type=? AND contact_id=?");

    $self->{sql_csm_add} = $db->prepare("INSERT INTO contact_status_map
         (st, contact_id, lang, msg, status)
         SELECT ? AS st, ? AS contact_id, ? as lang, ? as msg,
             cs.code FROM contact_status cs WHERE cs.status=?");

    $self->{sql_csm_del} = $db->prepare("DELETE FROM contact_status_map
         WHERE contact_status_map.contact_id=?
             AND contact_status.status=?
             AND contact_status.code=contact_status_map.status");

    $self->{sql_domain_add} = $db->prepare("INSERT INTO domain
         (st, domain_id, expire_date, fqdn, owner_id, updated_date,
          updated_id, created_date, created_id, registrar_id)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $self->{sql_domain_setowner} = $db->prepare("UPDATE domain
         SET st=?, owner_id=? WHERE domain_id=?");

    $self->{sql_domain_setupdate} = $db->prepare("UPDATE domain
         SET st=?, updated_date=?, updated_id=? WHERE domain_id=?");

    $self->{sql_dhm_add} = $db->prepare("INSERT INTO domain_host_map
         (st, domain_id, host_id)
         SELECT ? AS st, ? AS domain_id, host.host_id FROM host WHERE fqdn=?");

    $self->{sql_dhm_del} = $db->prepare("DELETE FROM domain_host_map
         WHERE domain_id=?
             AND host.host_id=domain_host_map.host_id
             AND host.fqdn=?");

    $self->{sql_dsm_add} = $db->prepare("INSERT INTO domain_status_map
         (st, domain_id, lang, msg, status)
         SELECT ? AS st, ? AS domain_id, ? as lang, ? as msg,
             ds.code FROM domain_status ds WHERE ds.status=?");

    $self->{sql_dsm_del} = $db->prepare("DELETE FROM domain_status_map
         WHERE domain_status_map.domain_id=?
             AND domain_status.status=?
             AND domain_status.code=domain_status_map.status");

    $self->{sql_dcm_add} = $db->prepare(
	 "INSERT INTO domain_contact_map
             (st, domain_id, type, contact_id)
             SELECT ? AS st, ? AS domain_id, ? AS type,
                 c.contact_id FROM contact c WHERE c.handle=?");

    $self->{sql_dcm_del} = $db->prepare(
	 "DELETE FROM domain_contact_map
          WHERE domain_contact_map.domain_id=?
              AND domain_contact_map.type=?
              AND contact.handle=?
              AND contact.contact_id=domain_contact_map.contact_id");

    $self->{sql_da_add} = $db->prepare(
	 "INSERT INTO domain_authinfo
             (st, domain_id, authtype, authinfo)
             VALUES (?, ?, ?, ?)");

    $self->{sql_da_del} = $db->prepare("
         DELETE FROM domain_authinfo WHERE domain_id=?");

    $self->{sql_host_add} = $db->prepare("INSERT INTO host
	     (st, host_id, fqdn, domain_id, updated_date, updated_id,
	      created_date, created_id, registrar_id)
	      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");

    $self->{sql_him_add} = $db->prepare("INSERT INTO host_ip_map
         (st, host_id, type, addr)
         VALUES (?, ?, ?, ?)");

    $self->{sql_him_del} = $db->prepare("DELETE FROM host_ip_map
         WHERE host_id=?
             AND type=?
             AND addr=?");

    $self->{sql_hsm_add} = $db->prepare("INSERT INTO host_status_map
         (st, host_id, lang, msg, status)
         SELECT ? AS st, ? AS host_id, ? as lang, ? as msg,
             hs.code FROM host_status hs WHERE hs.status=?");

    $self->{sql_hsm_del} = $db->prepare("DELETE FROM host_status_map
         WHERE host_status_map.host_id=?
             AND host_status.status=?
             AND host_status.code=host_status_map.status");
}

sub db {
    my ($self) = @_;

    return $self->{db};
}

sub rep {
    my ($self) = @_;

    return $self->{rep};
}

sub rep_newst {
    my ($self, @args) = @_;

    my $st = 0;
    if ($self->rep) {
	$st = $self->rep->newst(@args);
    }

    return $st;
}

sub _new_did {
    my ($self) = @_;

    my $db = $self->db->db;

    my $sth = $self->{sql_newdid};
    $sth->execute;
    my $arr = $sth->fetchrow_hashref;
    my $id = $arr->{did};
    $sth->finish;

    return $id;
}

sub _new_hid {
    my ($self) = @_;

    my $db = $self->db->db;

    my $sth = $self->{sql_newhid};
    $sth->execute;
    my $arr = $sth->fetchrow_hashref;
    my $id = $arr->{hid};
    $sth->finish;

    return $id;
}

sub _new_cid {
    my ($self) = @_;

    my $db = $self->db->db;

    my $sth = $self->{sql_newcid};
    $sth->execute;
    my $arr = $sth->fetchrow_hashref;
    my $id = $arr->{cid};
    $sth->finish;

    return $id;
}

sub _new_rid {
    my ($self) = @_;

    my $db = $self->db->db;

    my $sth = $self->{sql_newrid};
    $sth->execute;
    my $arr = $sth->fetchrow_hashref;
    my $id = $arr->{rid};
    $sth->finish;

    return $id;
}

sub rep_addlog {
    my ($self, $type, $args) = @_;

    if ($self->rep) {
	my $msg = { type => $type,
		    args => $args };
#	print Data::Dumper->Dump([$msg], ["log_msg"]);
	my $buf = ISC::CC::Message::towire($msg);
	$self->rep->addlog(encode_base64($buf, ''));
    }
}

sub rep_finish {
    my ($self, @args) = @_;

    if ($self->rep) {
	$self->rep->finish(@args);
    }
}

sub begin {
    my $self = shift;

    return $self->db->begin(@_);
}

sub commit {
    my $self = shift;

    return $self->db->commit(@_);
}

sub rollback {
    my $self = shift;

    return $self->db->rollback(@_);
}

sub bump_soa_serial {
    my ($self, %args) = @_;

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $apex = $args{apex};
    carp "Missing argument: apex" unless ($apex);

    my $sth = $db->prepare("SELECT * FROM soa WHERE apex=? FOR UPDATE");
    $sth->execute($apex);
    my %soa = %{ $sth->fetchrow_hashref };
    $sth->finish;

    my %nsoa = %soa;
    $nsoa{serial}++;
    $nsoa{st} = $st;

    $sth = $db->prepare("UPDATE soa SET st=?, serial=? WHERE apex=?");
    $sth->execute($nsoa{st}, $nsoa{serial}, $apex);
    $sth->finish;

    # XXXMLG need replication goo

    $self->commit;
    $self->rep_finish;

    return unless(defined(wantarray));
    return wantarray ? (\%soa, \%nsoa) : \%nsoa;
}

sub registrar_get {
    my ($self, %args) = @_;

    my $db = $self->{db}->{db};

    my $sth;
    if ($args{userid}) {
	$sth = $db->prepare_cached("SELECT * FROM registrar WHERE userid=?");
	$sth->execute($args{userid});
    } elsif ($args{registrar_id}) {
	$sth = $db->prepare_cached("SELECT * FROM registrar WHERE registrar_id=?");
	$sth->execute($args{registrar_id});
    } else {
	croak "Requires either 'userid' or 'registrar_id'";
    }

    my ($info) = $sth->fetchrow_hashref;
    if (!$info) {
	$sth->finish;
	return undef;
    }

    my %deepcopy = %{ $info };
    $sth->finish;

#    print Data::Dumper->Dump([\%ret], ["ret"]);

    return \%deepcopy;
}

sub registrar_setpw {
    my ($self, %args) = @_;

    my @reqd = qw(st userid newpassword);
    _check_required_args("registrar_setpw", \%args, \@reqd);

    my $st = $args{st};
    my $userid = $args{userid};
    my $newpassword = $args{newpassword};

    my $db = $self->db->db;

    my $sth = $db->prepare_cached("
	UPDATE registrar SET st=?, password=? WHERE userid=?");
    $sth->execute($st, $newpassword, $userid);
    my $rows = $sth->rows;
    $sth->finish;

    if (!$rows || $rows != 1) {
	return "registrar_setpw: " . $sth->errstr;
    }

    return 0;
}

sub _registrar_name {
    my ($self, $id) = @_;

    my $db = $self->db->db;

    my $sth = $db->prepare_cached("
	SELECT name FROM registrar WHERE registrar_id=?");
    $sth->execute($id);
    my (@info) = $sth->fetchrow_array;
    if (!scalar(@info)) {
	$sth->finish;
	return undef;
    }
    $sth->finish;

    return $info[0];
}

sub _contact_name {
    my ($self, $id) = @_;

    my $db = $self->db->db;

    my $sth = $db->prepare_cached("
	SELECT handle FROM contact WHERE contact_id=?");
    $sth->execute($id);
    my (@info) = $sth->fetchrow_array;
    if (!scalar(@info)) {
	$sth->finish;
	return undef;
    }
    $sth->finish;

    return $info[0];
}

sub _contact_id_byname {
    my ($self, $id) = @_;

    my $db = $self->db->db;

    my $sth = $db->prepare_cached("
	SELECT contact_id FROM contact WHERE handle=?");
    $sth->execute($id);
    my (@info) = $sth->fetchrow_array;
    if (!scalar(@info)) {
	$sth->finish;
	return undef;
    }
    $sth->finish;

    return $info[0];
}

#
# manipulate contact_authinfo for a contact
#
sub _ca {
    my ($self, %args) = @_;

    my $contact_id = $args{contact_id};
    my $st = $args{st};
    my $ai = $args{authinfo};

    return (EPP_RS_SUCCESS) if (!$ai);

    my $aitype = $args{authinfo}->{type};
    my $ai_val = $args{authinfo}->{_val};

    croak "missing {authinfo}->{type}" if (!$aitype && $ai_val);
    croak "missing {authinfo}->{_val}" if ($aitype && !$ai_val);

    return (EPP_RS_SUCCESS) if (!$aitype && !$ai_val);

    $self->begin;

    #
    # remove any previous authinfo.  Ignore the deleted item count.
    #
    my $sth = $self->{sql_ca_del};
    $sth->execute($contact_id);
    if ($sth->err) {
	my $msg = "Cannot delete authinfo from contact: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	return (EPP_RF_FAIL, $msg);
    }
    $sth->finish;

    $sth = $self->{sql_ca_add};
    $sth->execute($st, $contact_id, $aitype, $ai_val);
    if ($sth->err) {
	my $msg = "Cannot add authinfo to contact: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	return (EPP_RF_FAIL, $msg);
    }
    $sth->finish;

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of postal addresses to add/delete/change for a contact,
# do the delete operations, then the add operations and finally any
# change operations.
#
# Will return an error if any operation fails, or EPP_RS_SUCCESS on success.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _cp {
    my ($self, %args) = @_;

    my $contact_id = $args{contact_id};
    my $st = $args{st};
    my $add = $args{add} || {};
    my $del = $args{del} || {};
    my $chg = $args{chg} || {};

#    print Data::Dumper->Dump([\%args], ['args']);

    croak "missing contact_id" unless ($contact_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS)
	if (scalar($add) == 0 && scalar($del) == 0 && scalar($chg) == 0) ;

    my $db = $self->begin;

    if ($del && scalar($del)) {
	my $sth = $self->{sql_cp_del};
	foreach my $type (keys %$del) {
	    eval {
		$sth->execute($contact_id, $type);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap contact_postal: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap contact_postal: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar($add)) {
	my $sth = $self->{sql_cp_add};
	foreach my $type (keys %$add) {
	    my $name = $add->{$type}->{name};
	    my $org = $add->{$type}->{org};
	    my $streets = $add->{$type}->{street};
	    my $street1 = shift(@$streets) if (scalar(@$streets) >= 1);
	    my $street2 = shift(@$streets) if (scalar(@$streets) >= 1);
	    my $street3 = shift(@$streets) if (scalar(@$streets) >= 1);
	    my $city = $add->{$type}->{city};
	    my $sp = $add->{$type}->{sp};
	    my $pc = $add->{$type}->{pc};
	    my $cc = $add->{$type}->{cc};
	    
	    eval {
		$sth->execute($st, $contact_id, $type, $name, $org,
		      $street1, $street2, $street3, $city, $sp, $pc, $cc);
	    };
	    if ($sth->err) {
		my $msg = "Cannot add contact_postal: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot add contact_postal: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($chg && scalar($chg)) {
	# lookup all the current values for the contact, so we can do
	# a high-speed merge of the data with the stuff passed in

        my $cp = $self->_c_postal_get_byid(id => $contact_id);

	my $sth = $self->{sql_cp_update};

	# carefully override the existing values with the new values
	foreach my $type (keys %$chg) {
	    if (!defined($cp->{$type})) {
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, "postal address of type $type does not exist");
	    }
	    my $name = $cp->{$type}->{name};
	    $name = $chg->{$type}->{name} if (defined($chg->{$type}->{name}));

	    my $org = $cp->{$type}->{org};
	    $org = $chg->{$type}->{org} if (defined($chg->{$type}->{org}));

	    # if any streets were passed in, replace them all
            my ($street1, $street2, $street3);
	    if (defined($chg->{$type}->{street})) {
		my $streets = $chg->{$type}->{street};
		$street1 = shift(@$streets) if (scalar(@$streets) >= 1);
		$street2 = shift(@$streets) if (scalar(@$streets) >= 1);
		$street3 = shift(@$streets) if (scalar(@$streets) >= 1);
            } else {
		$street1 = $cp->{$type}->{street1};
		$street2 = $cp->{$type}->{street2};
		$street3 = $cp->{$type}->{street3};
	    }
	
	    my $city = $cp->{$type}->{city};
	    $city = $chg->{$type}->{city} if (defined($chg->{$type}->{city}));

	    my $sp = $cp->{$type}->{sp};
	    $sp = $chg->{$type}->{sp} if (defined($chg->{$type}->{sp}));

	    my $pc = $cp->{$type}->{pc};
	    $pc = $chg->{$type}->{pc} if (defined($chg->{$type}->{pc}));

	    my $cc = $cp->{$type}->{cc};
	    $cc = $chg->{$type}->{cc} if (defined($chg->{$type}->{cc}));
	    
	    eval {
		$sth->execute($st, $name, $org,
		      $street1, $street2, $street3, $city, $sp, $pc, $cc,
		      $type, $contact_id);
	    };
	    if ($sth->err) {
		my $msg = "Cannot update contact_postal: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot update contact_postal: UPDATE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of statuses to delete from a contact, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails, or EPP_RS_SUCCESS on success.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _csm {
    my ($self, %args) = @_;

    my $contact_id = $args{contact_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

#    print Data::Dumper->Dump([\%args], ['args']);

    croak "missing contact_id" unless ($contact_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS) if (scalar(@$add) == 0 && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_csm_del};
	foreach my $i (@$del) {
	    eval {
		$sth->execute($contact_id, $i);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap status: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_csm_add};
	foreach my $i (@$add) {
	    my ($code, $lang, $msg) = @$i;

# print "csm adding: $code, $lang, $msg\n";

	    eval {
		$sth->execute($st, $contact_id, $lang, $msg, $code);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map status: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of nameservers to delete from a domain, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails due to the host not
# already being mapped onto the domain, or the host not existing.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _dhm {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

    croak "missing domain_id" unless ($domain_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS) if (scalar(@$add) == 0 && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_dhm_del};
	foreach my $ns (@$del) {
	    $ns = uc $ns;
	    eval {
		$sth->execute($domain_id, $ns);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap host: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap host: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_dhm_add};
	foreach my $ns (@$add) {
	    $ns = uc $ns;
	    eval {
		$sth->execute($st, $domain_id, $ns);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map host: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map host: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of contacts to delete from a domain, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails due to the contact not
# already being mapped onto the domain, or the contact not existing.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _dcm {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

    croak "missing domain_id" unless ($domain_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS) if (scalar(@$add) == 0 && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_dcm_del};
	foreach my $i (@$del) {
	    my ($type, $handle) = @$i;
	    $handle = uc $handle;
	    eval {
		$sth->execute($domain_id, $type, $handle);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap contact: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap contact: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_dcm_add};
	foreach my $i (@$add) {
	    my ($type, $handle) = @$i;
	    $handle = uc $handle;
	    eval {
		$sth->execute($st, $domain_id, $type, $handle);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map contact: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map contact: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of statuses to delete from a domain, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails, or EPP_RS_SUCCESS on success.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _dsm {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

    croak "missing domain_id" unless ($domain_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS) if (scalar(@$add) == 0 && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_dsm_del};
	foreach my $i (@$del) {
	    eval {
		$sth->execute($domain_id, $i);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap status: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_dsm_add};
	foreach my $i (@$add) {
	    my ($code, $lang, $msg) = @$i;
	    eval {
		$sth->execute($st, $domain_id, $lang, $msg, $code);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map status: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

sub _set_d_owner {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $st = $args{st};
    my $owner_id = $args{owner_id};

    $self->begin;

    my $owner_cid = $self->_contact_id_byname($args{owner_id});
    if (!$owner_cid) {
	$self->rollback;
	return (EPP_RF_FAIL, "contact $args{owner_id} not known");
    }

    my $sth = $self->{sql_domain_setowner};
    $sth->execute($st, $owner_cid, $domain_id);
    if ($sth->err) {
	my $msg = "Cannot set owner: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	return (EPP_RF_FAIL, $msg);
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

sub _da {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $st = $args{st};
    my $ai = $args{authinfo};

    return (EPP_RS_SUCCESS) if (!$ai);

    my $aitype = $args{authinfo}->{type};
    my $ai_val = $args{authinfo}->{_val};

    croak "missing {authinfo}->{type}" if (!$aitype && $ai_val);
    croak "missing {authinfo}->{_val}" if ($aitype && !$ai_val);

    return (EPP_RS_SUCCESS) if (!$aitype && !$ai_val);

    $self->begin;

    #
    # remove any previous authinfo.  Ignore the deleted item count.
    #
    my $sth = $self->{sql_da_del};
    $sth->execute($domain_id);
    if ($sth->err) {
	my $msg = "Cannot delete authinfo from domain: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	return (EPP_RF_FAIL, $msg);
    }
    $sth->finish;

    $sth = $self->{sql_da_add};
    $sth->execute($st, $domain_id, $aitype, $ai_val);
    if ($sth->err) {
	my $msg = "Cannot add authinfo to domain: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	return (EPP_RF_FAIL, $msg);
    }
    $sth->finish;

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# return a list of host_id and fqdn's for the given domain_id
#
# WARNING: no replication or transaction locking is performed here.
#          This is a helper function only.
#
sub _domain_get_hosts {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my $hosts = [];

    my $sth = $db->prepare_cached("
        SELECT dhm.host_id, h.fqdn
          FROM domain_host_map dhm, host h
          WHERE dhm.domain_id=? AND h.host_id=dhm.host_id");

    $sth->execute($domain_id);
    while (my $arr = $sth->fetchrow_hashref) {
	my %deep_copy = %{ $arr };
	push(@$hosts, \%deep_copy);
    }
    $sth->finish;

    return $hosts;
}

sub _domain_get_subhosts {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my $hosts = [];

    my $sth = $db->prepare_cached("
        SELECT h.host_id, h.fqdn
          FROM host h
          WHERE h.domain_id=?");

    $sth->execute($domain_id);
    while (my $arr = $sth->fetchrow_hashref) {
	my %deep_copy = %{ $arr };
	push(@$hosts, \%deep_copy);
    }
    $sth->finish;

    return $hosts;
}

sub _domain_status {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my $s = [];

    my $sth = $db->prepare_cached("
        SELECT ds.status, dsm.lang, dsm.msg
          FROM domain_status ds, domain_status_map dsm
          WHERE dsm.domain_id=? AND ds.code=dsm.status");

    $sth->execute($domain_id);
    while (my $arr = $sth->fetchrow_arrayref) {
	push @$s, [ @{ $arr } ];
    }
    $sth->finish;

    if (scalar(@$s) == 0) {
	$s = [ [ 'ok' ] ];
    }

    return $s;
}

#
# Given an array of ip address to delete from a domain, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails due to the host not existing.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _him {
    my ($self, %args) = @_;

    my $host_id = $args{host_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

#    print Data::Dumper->Dump([$host_id, $add, $del], ['host_id', 'add', 'del']);

    croak "missing host_id" unless ($host_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS)
	if (scalar(@$add) == 0
	    && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_him_del};
	foreach my $addr (@$del) {
	    my ($type, $address) = @$addr;
	    eval {
		$sth->execute($host_id, $type, $address);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap ip address: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap ip address: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_him_add};
	foreach my $addr (@$add) {
	    my ($type, $address) = @$addr;
	    eval {
		$sth->execute($st, $host_id, $type, $address);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map ip address: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map ip address: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

#
# Given an array of statuses to delete from a host, delete them,
# and then add the ones requested.
#
# Will return an error if any operation fails, or EPP_RS_SUCCESS on success.
#
# This function does no permissions checking.
#
# This function does begin and end its own transaction, although it is
# quite likely the caller will also already be in one.
#
sub _hsm {
    my ($self, %args) = @_;

    my $host_id = $args{host_id};
    my $st = $args{st};
    my $add = $args{add} || [];
    my $del = $args{del} || [];

#    print Data::Dumper->Dump([\%args], ['args']);

    croak "missing host_id" unless ($host_id);
    croak "missing st" unless (defined($st));

    return (EPP_RS_SUCCESS) if (scalar(@$add) == 0 && scalar(@$del) == 0);

    my $db = $self->begin;

    if ($del && scalar(@$del)) {
	my $sth = $self->{sql_hsm_del};
	foreach my $i (@$del) {
	    eval {
		$sth->execute($host_id, $i);
	    };
	    if ($sth->err) {
		my $msg = "Cannot unmap status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot unmap status: DELETE returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    if ($add && scalar(@$add)) {
	my $sth = $self->{sql_hsm_add};
	foreach my $i (@$add) {
	    my ($code, $lang, $msg) = @$i;
	    eval {
		$sth->execute($st, $host_id, $lang, $msg, $code);
	    };
	    if ($sth->err) {
		my $msg = "Cannot map status: " . $sth->errstr;
		$sth->finish;
		$self->rollback;
		return (EPP_RF_FAIL, $msg);
	    }
	    if ($sth->rows != 1) {
		my $msg = "Cannot map status: INSERT returned "
		    . $sth->rows . " rows";
		$sth->finish;
		$self->rollback;
		return (EPP_RF_NOTEXISTS, $msg);
	    }
	}
	$sth->finish;
    }

    $self->commit;
    return (EPP_RS_SUCCESS);
}

sub _host_status {
    my ($self, %args) = @_;

    my $host_id = $args{host_id};
    my $db = $self->{db}->{db};

    my $s = [];

    my $sth = $db->prepare_cached("
        SELECT hs.status, hsm.msg
          FROM host_status hs, host_status_map hsm
          WHERE hsm.host_id=? AND hs.code=hsm.status");
    $sth->execute($host_id);
    while (my $arr = $sth->fetchrow_arrayref) {
	push @$s, [ @{ $arr } ];
    }
    $sth->finish;

    if (scalar(@$s) == 0) {
	$s = [ [ 'ok' ] ];
    }

    return $s;
}

sub _domain_get_contact_handles {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my $s = [];

    my $sth = $db->prepare_cached("
        SELECT dcm.type as type, c.handle as handle
          FROM domain_contact_map dcm, contact c
          WHERE dcm.domain_id=? AND c.contact_id=dcm.contact_id");

    $sth->execute($domain_id);
    while (my $arr = $sth->fetchrow_hashref) {
	my %c = %{ $arr };
	push @$s, \%c;
    }
    $sth->finish;

    return $s;
}

# looks in
#	domain_contact_map
#	registrar_contact_map

sub _contact_inuse {
    my ($self, %args) = @_;

    my $contact_id = $args{contact_id};
    my $db = $self->{db}->{db};

    my (@dids, @rids);
    my $sth;

    $sth = $db->prepare("SELECT domain_id FROM domain_contact_map dcm
            WHERE contact_id=?");
    $sth->execute($contact_id);
    while (my @row = $sth->fetchrow_array) {
#	print "contact_del: Found domain_id $row[0]\n";
	push @dids, $row[0];
    }
    $sth->finish;

    $sth = $db->prepare("SELECT registrar_id FROM registrar_contact_map dcm
            WHERE contact_id=?");
    $sth->execute($contact_id);
    while (my @row = $sth->fetchrow_array) {
#	print "contact_del: Found registrar_id $row[0]\n";
	push @rids, $row[0];
    }
    $sth->finish;

    my $count = @dids + @rids;
    return $count;
}

sub _domain_inuse {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my @ids;
    my %dids;

    my $sth = $db->prepare("SELECT host_id FROM host h WHERE h.domain_id=?");
    $sth->execute($domain_id);
    while (my @row = $sth->fetchrow_array) {
#	print "Found host_id $row[0]\n";
	push @ids, $row[0];
    }
    $sth->finish;

    return 0 if (scalar(@ids) == 0);

    $sth = $db->prepare("SELECT domain_id FROM domain_host_map dhm
            WHERE host_id=?");
    foreach my $id (@ids) {
	$sth->execute($id);
	while (my @row = $sth->fetchrow_array) {
#	    print "Found domain_id $row[0]\n";
	    $dids{$row[0]} = 1 unless ($row[0] == $domain_id);
	}
	$sth->finish;
    }

    my $count = scalar(keys %dids);
    return $count;
}

sub _nameserver_inuse {
    my ($self, %args) = @_;

    my $host_id = $args{host_id};
    my $db = $self->{db}->{db};

    my $hosts = [];

    my $sth = $db->prepare("
        SELECT count(*) as count FROM domain_host_map dhm
           WHERE host_id=?");

    $sth->execute($host_id);
    my @row = $sth->fetchrow_array;
    $sth->finish;

    return $row[0];
}

sub _domain_get {
    my ($self, %args) = @_;

    my $fqdn = $args{fqdn};
    my $domain_id = $args{domain_id};

    if (!$fqdn && !$domain_id) {
	croak "Error, must specify one of fqdn or domain_id";
    }
    my $db = $self->{db}->{db};

    my $sth;
    if ($domain_id) {
	$sth = $db->prepare("SELECT * FROM domain WHERE domain_id=?");
	$sth->execute($domain_id);
    } elsif ($fqdn) {
	$sth = $db->prepare("SELECT * FROM domain WHERE fqdn=?");
	$sth->execute($fqdn);
    } else {
	croak "Requires 'domain_id' or 'fqdn' or 'handle'";
    }

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
	$sth->finish;
	return undef;
    }

    my %deep_copy = %{ $arr };
    $sth->finish;

    return \%deep_copy;
}

sub apex {
    my ($self) = @_;

    return $self->{apex} if ($self->{apex});

    my $d = $self->_domain_get(domain_id => 1);

    $self->{apex} = $d->{fqdn};

    return $d->{fqdn};
}

sub _host_get_addresses {
    my ($self, %args) = @_;

    my $id = $args{id};
    my $db = $self->{db}->{db};

    my $addrs = [];

    my $sth = $db->prepare_cached("
        SELECT type, addr FROM host_ip_map WHERE host_id=?");
    $sth->execute($id);
    # XXXMLG need to check for error here

    while (my $arr = $sth->fetchrow_arrayref) {
	my @dc = @{ $arr };
	push(@$addrs, \@dc);
    }
    $sth->finish;

    return $addrs;
}

sub _domain_get_contacts {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $type = $args{type};
    my $db = $self->{db}->{db};

    my $contacts = [];

    my $sth = $db->prepare_cached("
        SELECT contact_id from domain_contact_map WHERE domain_id=? and type=?");
    $sth->execute($domain_id, $type);
    while (my $arr = $sth->fetchrow_hashref) {
	my %deep_copy = %{ $arr };
	push(@$contacts, \%deep_copy);
    }
    $sth->finish;

    return $contacts;
}

sub _domain_get_authinfo {
    my ($self, %args) = @_;

    my $domain_id = $args{domain_id};
    my $db = $self->{db}->{db};

    my $sth = $db->prepare_cached("SELECT authinfo, authtype FROM domain_authinfo WHERE domain_id=?");
    $sth->execute($domain_id);

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
        $sth->finish;
        return undef;
    }

    my %deepcopy = %{ $arr };
    $sth->finish;

    return \%deepcopy;
}

sub _host_get {
    my ($self, %args) = @_;

    my $fqdn = $args{fqdn};
    my $id = $args{id};

    if (!$id && !$fqdn) {
	croak "Error, must specify one of id or fqdn";
    }
    my $db = $self->{db}->{db};
    my $sth;

    if ($id) {
	$sth = $db->prepare_cached("SELECT * FROM host WHERE domain_id=?");
	$sth->execute($id);
    } elsif ($fqdn) {
	$sth = $db->prepare_cached("SELECT * FROM host WHERE fqdn=?");
	$sth->execute($fqdn);
    } else {
	croak "Requires 'id' or 'fqdn'";
    }

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
	$sth->finish;
	return undef;
    }

    my %deep_copy = %{ $arr };
    $sth->finish;

    return \%deep_copy;
}

sub _contact_get {
    my ($self, %args) = @_;

    my $id = $args{id};
    my $handle = $args{handle};

    if (!$id && !$handle) {
	croak "Error, must specify one of id or handle";
    }
    my $db = $self->{db}->{db};

    my $sth;
    if ($id) {
	$sth = $db->prepare("SELECT * FROM contact WHERE contact_id=?");
	$sth->execute($id);
    } elsif ($handle) {
	$sth = $db->prepare("SELECT * FROM contact WHERE handle=?");
	$sth->execute($handle);
    } else {
	croak "Requires 'id' or 'handle'";
    }

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
	$sth->finish;
	return undef;
    }

    my %deep_copy = %{ $arr };
    $sth->finish;

    return \%deep_copy;
}

sub _contact_status {
    my ($self, %args) = @_;
    my $contact_id = $args{contact_id};
    my $db = $self->{db}->{db};

    my $s = [];

    my $sth = $db->prepare_cached("
        SELECT cs.status, csm.lang, csm.msg
          FROM contact_status cs, contact_status_map csm
          WHERE csm.contact_id=? AND cs.code=csm.status");

    $sth->execute($contact_id);
    while (my $arr = $sth->fetchrow_arrayref) {
	push @$s, [ @{ $arr } ];
    }
    $sth->finish;

    if (scalar(@$s) == 0) {
	$s = [ [ 'ok' ] ];
    }

    return $s;
}


sub _c_authinfo_get_byid {
    my ($self, %args) = @_;

    my $id = $args{id};
    my $db = $self->{db}->{db};

    my $sth = $db->prepare_cached("SELECT authinfo, authtype FROM contact_authinfo WHERE contact_id=?");
    $sth->execute($id);

    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
        $sth->finish;
        return undef;
    }

    my %deepcopy = %{ $arr };
    $sth->finish;

    return \%deepcopy;
}

sub _c_postal_get_byid {
    my ($self, %args) = @_;

    my $id = $args{id};
    my $db = $self->{db}->{db};

    my $p = {};

    my $sth = $db->prepare_cached("SELECT * FROM contact_postal WHERE contact_id=?");
    $sth->execute($id);
    while (my $arr = $sth->fetchrow_hashref) {
	$p->{ $arr->{type}}->{name} = $arr->{name};
	$p->{ $arr->{type}}->{org} = $arr->{org};
	$p->{ $arr->{type}}->{city} = $arr->{city};
	$p->{ $arr->{type}}->{pc} = $arr->{pc};
	$p->{ $arr->{type}}->{sp} = $arr->{sp};
	$p->{ $arr->{type}}->{cc} = $arr->{cc};
	my $street = [];
	push (@$street, $arr->{street1}) if ($arr->{street1});
	push (@$street, $arr->{street2}) if ($arr->{street2});
	push (@$street, $arr->{street3}) if ($arr->{street3});
	$p->{ $arr->{type}}->{street} = $street;
	
    }
    $sth->finish;

#    print Data::Dumper->Dump([$p], ["p"]);

    return $p;
}

sub _get_hostid {
    my ($self, %args) = @_;

    my $fqdn = $args{fqdn};
    my $db = $self->{db}->{db};

    my $sth = $db->prepare_cached("
               SELECT host_id FROM host WHERE fqdn=?");
    $sth->execute($fqdn);
    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
	$sth->finish;
	return undef;
    }
    my $host_id = $arr->{host_id};
    $sth->finish;

    return $host_id;
}

sub _get_hostid_registrarid {
    my ($self, %args) = @_;
    
    my $fqdn = $args{fqdn};
    my $db = $self->{db}->{db};
    
    my $sth = $db->prepare_cached("
               SELECT host_id, registrar_id FROM host WHERE fqdn=?");
    $sth->execute($fqdn);
    my ($host_id, $registrar_id) = $sth->fetchrow_array;
    $sth->finish;
    return ($host_id, $registrar_id);
}

sub _get_domainid {
    my ($self, %args) = @_;

    my $fqdn = $args{fqdn};   
    my $db = $self->{db}->{db};

    my $sth = $db->prepare_cached("
               SELECT domain_id from domain WHERE fqdn=?");
    $sth->execute($fqdn);
    my $arr = $sth->fetchrow_hashref;
    if (!$arr) {
        $sth->finish;
        return undef;
    }
    my $domain_id = $arr->{domain_id};
    $sth->finish;

    return $domain_id;
}

# --- LOGIN -------------------------------------------------------- #

sub login {
    my ($self, %args) = @_;

    my $reg = $self->registrar_get(userid => $args{id});

    if (!$reg) {
	return (EPP_RF_AUTHENTICATION, "no such registrar");
    }

    if ($reg->{password} ne $args{password}) {
	return (EPP_RF_AUTHENTICATION, "password mismatch");
    }

    if ($args{newpassword}) {
	my $st = $self->rep_newst;
	my $db = $self->begin;

	my $ret = $self->registrar_setpw(st => $st,
					 userid => $args{id},
					 newpassword => $args{newpassword});
	if ($ret) {
	    $self->rollback;
	    $self->rep_finish;
	    return (EPP_RF_FAIL, "could not change password: $ret");
	}
	$self->commit;
	$self->rep_finish;
    }

    return (EPP_RS_SUCCESS, undef, undef,
	    { registrar_id => $reg->{registrar_id}});
}


# --- CONTACT_ADD ------------------------------------------------------- #

sub contact_add {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ["args"]);

    # ignore what the client requested and assign them a contact number
    my $forceassign = 0;

    my @reqd;
    if ($forceassign) {
	@reqd = qw(postal email authinfo registrar_id);
    } else {
	@reqd = qw(id postal email authinfo registrar_id);
    }
    _check_required_args("contact_add", \%args, \@reqd);

    my $st = $self->rep_newst;
    my $db = $self->begin;
    my $ret = {};

    my $now_ts;
    if ($args{created_date}) {
	$now_ts = from_date_ansi($args{created_date});
    } else {
	$now_ts = time;
    }
    my $now_ansi = date_ansi($now_ts);

    my ($contact_id, $handle);
    if ($forceassign) {
	$contact_id = $self->_new_cid;
        $handle = 'C' . $contact_id;
    } else {
        $handle = uc $args{id};
	$contact_id = $self->_new_cid;
    }
    # kludge -- hardcoded XXXKJL
    my $guid = $handle . '-ISC';

    my $sth = $self->{sql_contact_add};
    eval {
	$sth->execute($st,  $contact_id, $handle, $guid,
		      $args{voice}, $args{voice_ext}, $args{fax},
		      $args{email}, $args{updated_date}, $args{updated_id},
		      $now_ansi, $args{registrar_id}, $args{registrar_id});
    };
    if ($sth->err) {
	my $msg = "Cannot insert contact into database: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_EXISTS, $msg);
    }
    $sth->finish;

    my ($errcode, @errx);

    #
    # For each postal address provided, add it.
    #
    my $address = $args{postal};
    ($errcode, @errx) = $self->_cp(st => $st,
				    contact_id => $contact_id,
				    add => $address);
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    # process statuses.
    ($errcode, @errx) = $self->_csm(st => $st,
				    contact_id => $contact_id,
				    add => $args{status});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    # process authinfo for contact
    ($errcode, @errx) = $self->_ca(st => $st,
				   contact_id => $contact_id,
				   authinfo => $args{authinfo});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    my $epp_create_date = date_epp($now_ts);
    my $log = {
	contact_id => $contact_id,
	handle => $handle,
	guid => $guid,
	updated_date => $args{updated_date},
	updated_id => $args{updated_id},
	created_date => $epp_create_date,
	created_id => $args{created_id},
	registrar_id => $args{registrar_id},
    };
    $self->rep_addlog("contact_add", $log);

    $self->commit;
    $self->rep_finish;

    $ret->{id} = $handle;
    $ret->{crDate} = $epp_create_date;

    return (EPP_RS_SUCCESS, undef, $ret);
}


# --- CONTACT_CHECK ----------------------------------------------------- #

sub contact_check {
    my ($self, %args) = @_;

    my @reqd = qw(ids);
    _check_required_args("contact_check", \%args, \@reqd);

    my $ret = [];
    my $ids = $args{ids};

    if (scalar(@$ids) == 0) {
	return (EPP_RS_SUCCESS);
    }

    my $db = $self->begin;

    while (my $id = shift @$ids) {
	my $item = [ $id ];

	my $uid = uc $id;
	my $c = $self->_contact_get(handle => $uid);
	if ($c) {
	    push @$item, 0, 'Contact already exists';
	} else {
	    # not really, we're going to ignore it later
	    push @$item, 1;
	}

	push @$ret, $item;
    }

    $self->commit;

    return (EPP_RS_SUCCESS, undef, { ids => $ret });
}


# --- CONTACT_DEL ------------------------------------------------------- #

sub contact_del {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ["args"]);

    my @reqd = qw(id);
    _check_required_args("contact_del", \%args, \@reqd);

    my $handle = uc $args{id};

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $contact = $self->_contact_get(handle => $handle);

    if (!$contact) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_NOTEXISTS, "contact_del: no such contact");
    }

    if ($contact->{clid} != $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for contact");
    }

    my $domains = $self->_contact_inuse(contact_id => $contact->{contact_id});
    if ($domains) {
	$self->rollback;
        $self->rep_finish;
	return (EPP_RF_INUSE, "contact in use by domains");
    }

    my $statae = $self->_contact_status(contact_id => $contact->{contact_id});

#    print Data::Dumper->Dump([$statae], ["statae"]);

    foreach my $s (@$statae) {
	my ($status, $lang, $msg) = @$s;
	if ($status =~ m/.*(UpdateProhibited|DeleteProhibited)$/) {
	    $self->rollback;
            $self->rep_finish;
	    return (EPP_RF_DATAMGT, "contact status prevents deletion");
	}
    }

    my $now_ts = time;
    my $now_ansi = date_ansi($now_ts);

    my $sth = $db->prepare_cached("DELETE FROM contact WHERE contact_id = ?");
    $sth->execute($contact->{contact_id});

    if ($sth->err) {
        my $msg = "DELETE failed: " . $sth->errstr;
        $sth->finish;
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_FAIL, $msg);
    }

    $sth->finish;

    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}


# --- CONTACT_INFO ----------------------------------------------------- #

sub contact_info {
    my ($self, %args) = @_;

    my @reqd = qw(id);
    _check_required_args("contact_info", \%args, \@reqd);

    my $id = uc $args{id};

    if (!$id) {
	return (EPP_RS_SUCCESS);
    }

    my $db = $self->begin;

    my $c = $self->_contact_get(handle => $id);
    if (!$c) {
	# contact does not exist
	$self->rollback;
	return (EPP_RF_NOTEXISTS, 'Contact does not exist');
    }

    my $ret = {};

    my $postals = $self->_c_postal_get_byid(id => $c->{contact_id});
    my $auths = $self->_c_authinfo_get_byid(id => $c->{contact_id});

    $ret->{id} = $c->{handle};
    $ret->{roid} = $c->{guid};
    $ret->{status} = $self->_contact_status(contact_id => $c->{contact_id});

    $ret->{authinfo} = $auths if ($auths);
    $ret->{postal} = $postals if ($postals);

    $ret->{voice} = $c->{voice} if ($c->{voice});
    $ret->{voice_ext} = $c->{voice_ext} if ($c->{voice_ext});
    $ret->{fax} = $c->{fax} if ($c->{fax});
    $ret->{email} = $c->{email} if ($c->{email});
    $ret->{clID} = "R" . $c->{clid};
    $ret->{crID} = "R" . $c->{crid};
    $ret->{crDate} = date_epp(from_date_ansi($c->{crdate}));
    $ret->{upID} = "R" . $c->{upid} if ($c->{upid});
    $ret->{upDate} = date_epp(from_date_ansi($c->{update}))
	if ($c->{upid} && $c->{update});
    $ret->{trDate} = date_epp(from_date_ansi($c->{trdate}))
	if ($c->{trdate});

    # this last item is for easy identity checking
    $ret->{registrar_id} = $c->{clid};
        
    $self->commit;

    return (EPP_RS_SUCCESS, undef, $ret);
}


# --- CONTACT_MOD ------------------------------------------------------- #

sub contact_mod {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ["args"]);

    my @reqd = qw(id registrar_id);
    _check_required_args("contact_mod", \%args, \@reqd);

    my $handle = uc $args{id};

    my $st = $self->rep_newst;
    my $db =  $self->begin;

    my $contact_data = $self->_contact_get(handle => $handle);

    if (!$contact_data) {
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_NOTEXISTS, "contact $handle not found");
    }

    my $contact_id = $contact_data->{contact_id};
    my $registrar_id = $contact_data->{clid};

    if ($registrar_id != $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for contact");
    }

    my $now_ts;
    if ($args{updated_date}) {
	$now_ts = from_date_ansi($args{updated_date});
    } else {
	$now_ts = time;
    }
    my $now_ansi = date_ansi($now_ts);

    # need to check/update statuses here -- XXX

    my $pc = 0;
    $pc += scalar($args{postal_add}) if ($args{postal_add});
    $pc += scalar($args{postal_del}) if ($args{postal_del});
    $pc += scalar($args{postal_chg}) if ($args{postal_chg});

    my ($errcode, @errx);
    if ($pc > 0) {
	($errcode, @errx) = $self->_cp(st => $st,
				       contact_id => $contact_id,
				       add => $args{postal_add},
				       del => $args{postal_del},
				       chg => $args{postal_chg});
	if ($errcode != EPP_RS_SUCCESS) {
	    $self->rollback;
	    $self->rep_finish;
	    return ($errcode, @errx);
	}
    }

    # update voice element, if needed
    # must use defined to catch setting to the empty string
    if (defined($args{voice})) {
	my $sth = $self->{sql_contact_setvoice};
	$sth->execute($st, $args{voice}, $contact_id);
	if ($sth->err) {
	    my $msg = "Cannot set voice: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }

    # update voice_ext element, if needed
    # must use defined to catch setting to the empty string
    if (defined($args{voice_ext})) {
	my $sth = $self->{sql_contact_setvoice_ext};
	$sth->execute($st, $args{voice_ext}, $contact_id);
	if ($sth->err) {
	    my $msg = "Cannot set voice_ext: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }

    # update fax element, if needed
    # must use defined to catch setting to the empty string
    if (defined($args{fax})) {
	my $sth = $self->{sql_contact_setfax};
	$sth->execute($st, $args{fax}, $contact_id);
	if ($sth->err) {
	    my $msg = "Cannot set fax: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }

    # update email element, if needed
    if ($args{email}) {
	my $sth = $self->{sql_contact_setemail};
	$sth->execute($st, $args{email}, $contact_id);
	if ($sth->err) {
	    my $msg = "Cannot set email: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }

    ($errcode, @errx) = $self->_csm(st => $st,
				    contact_id => $contact_id,
				    add => $args{status_add},
				    del => $args{status_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    ($errcode, @errx) = $self->_ca(st => $st,
				   contact_id => $contact_id,
				   authinfo => $args{authinfo});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    # last but not least, set the last update time field
    if (1){
	my $sth = $self->{sql_contact_setupdate};
	$sth->execute($st, $now_ansi, $registrar_id, $contact_id);
	if ($sth->err) {
	    my $msg = "Cannot set update times: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }

    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}


# --- DOMAIN_ADD --------------------------------------------------- #

sub domain_add {
    my ($self, %args) = @_;

    my @reqd = qw(period units fqdn created_id registrar_id);
    _check_required_args("domain_add", \%args, \@reqd);

    my $apex = $self->apex;

    my $fqdn = uc $args{fqdn};

    return (EPP_RF_VALRANGE, "$fqdn is not named under $apex", undef)
        if ($fqdn !~ m/$apex$/);

    my $units = 1;
    $units = 12 if ($args{units} eq 'y');

    my $now_ts;
    if ($args{created_date}) {
	$now_ts = from_date_ansi($args{created_date});
    } else {
	$now_ts = time;
    }
    my $now_ansi = date_ansi($now_ts);

    my $then;
    if ($args{units} eq 'y') {
	$then = date_add($now_ts, $args{period} * 12);
    } else {
	$then = date_add($now_ts, $args{period});
    }
    my $expiration_date = date_epp($then);
    my $expiration_ansi = date_ansi($then);

    my $st = $self->rep_newst;
    my $db = $self->begin;

#    print Data::Dumper->Dump([\%args], ["args"]);

    my $domain_id = $args{domain_id} || $st || $self->_new_did;

    my $owner_cid = undef;
    if ($args{owner_id}) {
	$owner_cid = $self->_contact_id_byname($args{owner_id});
	if (!$owner_cid) {
	    $self->rollback;
	    $self->rep_finish;
	    return (EPP_RF_FAIL, "contact $args{owner_id} not known");
	}
    }

    my $sth = $self->{sql_domain_add};
    eval {
	$sth->execute($st, $domain_id, $expiration_ansi,
		      $fqdn, $owner_cid,
		      $args{updated_date}, $args{updated_id},
		      $now_ansi, $args{created_id},
		      $args{registrar_id});
    };
    if ($sth->err) {
	my $msg = "domain_add: Cannot insert into database: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_EXISTS, $msg);
    }
    $sth->finish;

    my ($errcode, @errx);

    #
    # Now, for each name server found, run through and attach this
    # domain to it.  Failure to find one results in an error, and
    # will abort the transaction.
    #
    ($errcode, @errx) = $self->_dhm(st => $st,
				    domain_id => $domain_id,
				    add => $args{nameservers});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    #
    # And the same for contacts.
    #
    ($errcode, @errx) = $self->_dcm(st => $st,
				    domain_id => $domain_id,
				    add => $args{contact});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    #
    # And the same for statuses.
    #
    ($errcode, @errx) = $self->_dsm(st => $st,
				    domain_id => $domain_id,
				    add => $args{status});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    #
    # And authinfo.
    #
    ($errcode, @errx) = $self->_da(st => $st,
				   domain_id => $domain_id,
				   authinfo => $args{authinfo});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    # XXXMLG need replication here

    $self->commit;
    $self->rep_finish;

    my $lines = {};
    $lines->{exDate} = $expiration_date;
    $lines->{crDate} = date_epp($now_ts);
    $lines->{name} = $fqdn;

    return (EPP_RS_SUCCESS, undef, $lines);
}


# --- DOMAIN_CHECK ------------------------------------------------- #

sub domain_check {
    my ($self, %args) = @_;

    my @reqd = qw(names);
    _check_required_args("domain_check", \%args, \@reqd);

    my $apex = $self->apex;
    my $ret = [];
    my $names = $args{names};

    if (scalar(@$names) == 0) {
	return (EPP_RS_SUCCESS);
    }

    my $db = $self->begin;

    while (my $name = shift @$names) {
	my $item = [ $name ];

	if ($name !~ m/($apex)$/) {
	    push @$item, 0, 'Bad name for registry';
	} else {
	    my $d = $self->_get_domainid(fqdn => $name);
	    if ($d) {
		push @$item, 0, 'Domain exists';
	    } else {
		push @$item, 1;
	    }
	}

	push @$ret, $item;
    }

    $self->commit;

    return (EPP_RS_SUCCESS, undef, { names => $ret });
}


# --- DOMAIN_MOD --------------------------------------------------- #

sub domain_mod {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ["args"]);

    my @reqd = qw(fqdn registrar_id);
    _check_required_args("domain_mod", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $st = $self->rep_newst;
    my $db =  $self->begin;

    my $domain_data = $self->_domain_get(fqdn => $fqdn);

    if (!$domain_data) {
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_NOTEXISTS, "domain $fqdn not found");
    }

    my $domain_id = $domain_data->{domain_id};
    my $registrar_id = $domain_data->{registrar_id};

    if ($registrar_id != $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for domain");
    }

    my $now_ts;
    if ($args{updated_date}) {
	$now_ts = from_date_ansi($args{updated_date});
    } else {
	$now_ts = time;
    }
    my $now_ansi = date_ansi($now_ts);

    # need to check/update status here XXXMLG

    my ($errcode, @errx);
    ($errcode, @errx) = $self->_dhm(st => $st,
				    domain_id => $domain_id,
				    add => $args{nameservers_add},
				    del => $args{nameservers_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    ($errcode, @errx) = $self->_dcm(st => $st,
				    domain_id => $domain_id,
				    add => $args{contact_add},
				    del => $args{contact_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }


    ($errcode, @errx) = $self->_dsm(st => $st,
				    domain_id => $domain_id,
				    add => $args{status_add},
				    del => $args{status_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    ($errcode, @errx) = $self->_da(st => $st,
				   domain_id => $domain_id,
				   authinfo => $args{authinfo});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    if ($args{owner_id}) {
	($errcode, @errx) = $self->_set_d_owner(st => $st,
						domain_id => $domain_id,
						owner_id => $args{owner_id});
	if ($errcode != EPP_RS_SUCCESS) {
	    $self->rollback;
	    $self->rep_finish;
	    return ($errcode, @errx);
	}
    }

    # last but not least, set the last update time field
    if (1) {
	my $sth = $self->{sql_domain_setupdate};
	$sth->execute($st, $now_ansi, $registrar_id, $domain_id);
	if ($sth->err) {
	    my $msg = "Cannot set update times: " . $sth->errstr;
	    $sth->finish;
	    $self->rollback;
	    return (EPP_RF_FAIL, $msg);
	}
    }


    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}


# --- DOMAIN_RENEW ------------------------------------------------- #

sub domain_renew {
    my ($self, %args) = @_;

    my @reqd = qw(fqdn registrar_id current_date);
    _check_required_args("domain_renew", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $st = $self->rep_newst;
    my $db = $self->begin;

# find the current expiration_date date

    my $sth = $db->prepare_cached("SELECT registrar_id, expire_date
        FROM domain WHERE fqdn = ?");
    eval {
        $sth->execute($fqdn);
    };

    if ($sth->err) {
        my $msg = "Select failed(domain_renew 1): " . $sth->errstr;
        $sth->finish;
	$self->rollback;
        $self->rep_finish;
        return (EPP_RF_FAIL, $msg); ## wrong
    }

    my ($registrar_id, $expiration_date) = $sth->fetchrow_array;

    if (!$expiration_date) {
	$sth->finish;
	$self->rollback;
        $self->rep_finish;
	return (EPP_RF_FAIL, "select failed(domain_renew 2): "
		. $sth->errstr);
    }

    if ($registrar_id != $args{registrar_id}) {
	$sth->finish;
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, 
            "can't renew someone else's domain");
    }

    my ($expiry, undef) = split(/ /, $expiration_date);

    if ($expiry ne $args{current_date}) {
	$sth->finish;
	$self->rollback;
        $self->rep_finish;
	return (EPP_RF_VALRANGE,
		"we said current_date=$expiry, they say current_date=$args{current_date}");
    }

    $args{period} *= 12 if ($args{period_unit} eq 'y');
    my $new_expiry = date_add(from_date_ansi($expiration_date),
			      $args{period});

    $sth->finish;
    $sth = $db->prepare_cached("UPDATE domain SET st=?, expire_date=? WHERE
        fqdn=?");
    eval {
	$sth->execute($st, date_ansi($new_expiry), $fqdn);
    };
    if ($sth->err) {
        my $msg = "UPDATE failed: " . $sth->errstr;
        $sth->finish;
	$self->rollback;
        $self->rep_finish;
        return (EPP_RF_FAIL, $msg); ## wrong
    }
    $sth->finish;
    $self->commit;
    $self->rep_finish;

    my $repl = { "expiration_date" => $new_expiry };

    return (EPP_RS_SUCCESS, undef, $repl);
}


# --- DOMAIN_DEL --------------------------------------------------- #

sub domain_del {
    my ($self, %args) = @_;

    my @reqd = qw(fqdn registrar_id);
    _check_required_args("domain_del", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $domain = $self->_domain_get(fqdn => $fqdn);

    if (!$domain) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_NOTEXISTS, "no such domain");
    }

    if ($domain->{registrar_id} != $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for domain");
    }

    my $hosts = $self->_domain_inuse(domain_id => $domain->{domain_id});
    if ($hosts) {
	$self->rollback;
        $self->rep_finish;
	return (EPP_RF_INUSE, "domain in use by $hosts nameservers");
    }

    my $now_ts = time;
    my $now_ansi = date_ansi($now_ts);

    my $sth = $db->prepare_cached("DELETE FROM domain WHERE domain_id = ?");
    $sth->execute($domain->{domain_id});

    if ($sth->err) {
        my $msg = "DELETE failed: " . $sth->errstr;
        $sth->finish;
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_FAIL, $msg);
    }

    $sth->finish;

    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}


# --- DOMAIN_INFO ------------------------------------------------ #

sub domain_info {
    my ($self, %args) = @_;

    my @reqd = qw(fqdn);
    _check_required_args("domain_info", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};
    my $hostinfo = $args{hostinfo} || 'all';

    my $db = $self->begin;

    my $d = $self->_domain_get(fqdn => $fqdn);
    if (!$d) {
	$self->rollback;
	return(EPP_RF_NOTEXISTS, 'Domain does not exist');
    }

    my $lines = {};
    $lines->{name} = $d->{fqdn};
    $lines->{roid} = "D" . $d->{domain_id} . "-ISC" ; # XXXKJL

    # get list of status attributes

    $lines->{status} = $self->_domain_status(domain_id => $d->{domain_id});

    # get list of nameservers
    if ($hostinfo eq 'all' || $hostinfo eq 'del') {
	$lines->{ns} = [];
	my $hosts = $self->_domain_get_hosts(domain_id => $d->{domain_id});
	
	foreach my $h (@$hosts) {
	    push(@{$lines->{ns}}, $h->{fqdn});
	}
    }

    if ($hostinfo eq 'all' || $hostinfo eq 'sub') {
	$lines->{hosts} = [];
	my $hosts = $self->_domain_get_subhosts(domain_id => $d->{domain_id});
	
	foreach my $h (@$hosts) {
	    push(@{$lines->{hosts}}, $h->{fqdn});
	}
    }

    $lines->{contact} = [];
    my $contacts = $self->_domain_get_contact_handles(domain_id => $d->{domain_id});
	
    foreach my $h (@$contacts) {
	push(@{$lines->{contact}}, [ $h->{type}, $h->{handle} ]);
    }

    my $auths = $self->_domain_get_authinfo(domain_id => $d->{domain_id});
    $lines->{authinfo} = $auths if ($auths);

    $lines->{exDate} = date_epp(from_date_ansi($d->{expire_date}))
	if ($d->{expire_date});
    $lines->{crDate} = date_epp(from_date_ansi($d->{created_date}));
    $lines->{crID} = $self->_registrar_name($d->{created_id});
    $lines->{registrant} = $self->_registrar_name($d->{registrar_id});
    $lines->{clID} = $self->_contact_name($d->{owner_id});
    $lines->{upDate} = date_epp(from_date_ansi($d->{updated_date}))
	if ($d->{updated_date});
    $lines->{upID} = $self->_registrar_name($d->{updated_id})
	if ($d->{updated_id});
    $lines->{trDate} = date_epp(from_date_ansi($d->{last_transfer_date}))
        if ($d->{last_transfer_date});

    # this last item is for easy identity checking
    $lines->{registrar_id} = $d->{registrar_id};

    $self->commit;

#    print Data::Dumper->Dump([$lines], ['lines']);

    return (EPP_RS_SUCCESS, undef, $lines);
}


# --- HOST_ADD ------------------------------------------------------- #

sub host_add {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ["args"]);

    my @reqd = qw(fqdn registrar_id);
    _check_required_args("host_add", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $apex = $self->apex;

    my $parent_id;

    # if the nameserver is in-zone, check the parent domain exists
    if ($fqdn =~ m/($apex)$/) {
        my $uqdn = $fqdn;
        $uqdn =~ s/($apex)//;
        $uqdn =~ s/\.$//;
        my @labels = split(/\./, $uqdn);
	
        my $parent = $labels[scalar(@labels) - 1] . "." . $apex;

        # check the domain exists and is not pending deletion
	my $domain = $self->_domain_get(fqdn => $parent);
        if (!$domain) {
            my $msg = "host_add: Cannot find parent domain $parent for $fqdn";
            $self->rollback;
            $self->rep_finish;
            return (EPP_RF_DATAMGT, $msg);
        }

#        if ($domain->{pending_delete_date}) {
#            $self->rollback;
#            $self->rep_finish;
#            return (EPP_RF_DATAMGT,
#		    "no undeleted parent domain $parent");
#        }

        if ($domain->{registrar_id} != $args{registrar_id}) {
            $self->rollback;
            $self->rep_finish;
            return (EPP_RF_AUTHORIZATION,
		    "domain owned by different registrar");
        }

	$parent_id = $domain->{domain_id};
    }

    my $now_ts;
    if ($args{created_date}) {
	$now_ts = from_date_ansi($args{created_date});
    } else {
	$now_ts = time;
    }
    my $now_ansi = date_ansi($now_ts);

    my $host_id = $args{host_id} || $st || $self->_new_hid;

    my $sth = $self->{sql_host_add};
    eval {
	$sth->execute($st, $host_id, $fqdn, $parent_id,
		      $args{updated_date}, $args{updated_id},
		      $now_ansi, $args{created_id},
		      $args{registrar_id});
    };
    if ($sth->err) {
	my $msg = "host_add: Cannot insert into database: " . $sth->errstr;
	$sth->finish;
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_EXISTS, $msg);
    }
    $sth->finish;

    my ($errcode, @errx);

    #
    # For each IP address provided, add it.
    #
    my $addrs = $args{addrs};
    ($errcode, @errx) = $self->_him(st => $st,
				    host_id => $host_id,
				    add => $addrs);
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    #
    # Process statuses.
    #
    ($errcode, @errx) = $self->_hsm(st => $st,
				    host_id => $host_id,
				    add => $args{status});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    my $log = {
	host_id => $host_id,
	fqdn => $fqdn,
	updated_date => $args{updated_date},
	updated_id => $args{updated_id},
	created_date => $args{created_date},
	created_id => $args{created_id},
	registrar_id => $args{registrar_id},
    };
    $log->{addrs} = $addrs if ($addrs);
    $self->rep_addlog("host_add", $log);

    $self->commit;
    $self->rep_finish;

    my $lines = {};
    $lines->{crDate} = date_epp($now_ts);
    $lines->{name} = $fqdn;

    return (EPP_RS_SUCCESS, undef, $lines);
}


# --- HOST_CHECK ----------------------------------------------------- #

sub host_check {
    my ($self, %args) = @_;

    my @reqd = qw(names);
    _check_required_args("host_check", \%args, \@reqd);

    my $ret = [];
    my $names = $args{names};

    if (scalar(@$names) == 0) {
	return (EPP_RS_SUCCESS);
    }

    my $db = $self->begin;

    while (my $name = shift @$names) {
	my $item = [ $name ];

	my $n = $self->_get_hostid(fqdn => $name);
	if ($n) {
	    push @$item, 0, 'Host exists';
	} else {
	    push @$item, 1;
	}

	push @$ret, $item;
    }

    $self->commit;

    return (EPP_RS_SUCCESS, undef, { names => $ret });
}


# --- HOST_DEL ------------------------------------------------------- #

sub host_del {
    my ($self, %args) = @_;

    my @reqd = qw(fqdn registrar_id);
    _check_required_args("host_del", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my ($hostid, $registrar_id) = $self->_get_hostid_registrarid(fqdn => $fqdn);

    if (!$hostid) {
        $self->rollback;
	$self->rep_finish;
        return (EPP_RF_NOTEXISTS, "no such host: $fqdn");
    }

    if ($registrar_id != $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for nameserver");
    }

    my $inuse = $self->_nameserver_inuse(host_id => $hostid);
    if ($inuse) {
	$self->rollback;
	$self->rep_finish;
	return (EPP_RF_INUSE, "host in use by $inuse domains");
    }

    my $sth = $db->prepare("DELETE FROM host WHERE host_id=?");
    $sth->execute($hostid);
    if ($sth->err) {
        my $msg = "DELETE failed: " . $sth->errstr;
        $sth->finish;
        $self->rollback;
	$self->rep_finish;
        return (EPP_RF_FAIL, $msg);
    }

    # XXXMLG need to check numrows here!

    $self->commit;
    $self->rep_finish;

    # need to send replication log XXXMLG

    return (EPP_RS_SUCCESS);
}

# --- HOST_MOD ------------------------------------------------------- #

sub host_mod {
    my ($self, %args) = @_;

#    print Data::Dumper->Dump([\%args], ['args']);

    my @reqd = qw(fqdn registrar_id);
    _check_required_args("host_mod", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $apex = $self->apex;

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $h = $self->_host_get(fqdn => $fqdn);

    if ($h->{registrar_id} !=  $args{registrar_id}) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_AUTHORIZATION, "not registrar for nameserver");
    }

    # address modifications are only appropriate to in-zone nameservers
    if ($fqdn !~ m/($apex)$/ && ($args{addr_del} || $args{addr_add})) {
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_DATAMGT, "address modification attempted on out-of-zone nameserver");
    }

    my ($errcode, @errx);
    ($errcode, @errx) = $self->_him(st => $st,
				    host_id => $h->{host_id},
				    add => $args{addr_add},
				    del => $args{addr_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    #
    # Process statuses.
    #
    ($errcode, @errx) = $self->_hsm(st => $st,
				    host_id => $h->{host_id},
				    add => $args{status_add},
				    del => $args{status_del});
    if ($errcode != EPP_RS_SUCCESS) {
	$self->rollback;
	$self->rep_finish;
	return ($errcode, @errx);
    }

    # update modify date for host, renaming if necessary
    my $new_fqdn = uc $args{rename} || $fqdn;

    my $now_ts = time;
    my $updated_date = date_ansi($now_ts);

    my $sth2 = $db->prepare_cached("UPDATE host SET st=?, fqdn=?, updated_date=? WHERE host_id=?");
    eval {
        $sth2->execute($st, $new_fqdn, $updated_date, $h->{host_id});
    };
    if ($sth2->err) {
        my $msg = "attempt to update host record failed: " . $sth2->errstr;
        $sth2->finish;
        $self->rollback;
        $self->rep_finish;
        return (EPP_RF_FAIL, $msg); ## wrong code
    }

    $sth2->finish;
    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}


# --- HOST_INFO ---------------------------------------------------- #

sub host_info {
    my ($self, %args) = @_;

    my @reqd = qw(fqdn);
    _check_required_args("host_info", \%args, \@reqd);

    my $fqdn = uc $args{fqdn};

    my $db = $self->begin;

    my $lines = {};

    my $h = $self->_host_get(fqdn => $fqdn);

    if (!$h) {
	$self->rollback;
	return (EPP_RF_NOTEXISTS, 'host does not exist');
    }

    $lines->{name} = $fqdn;
    $lines->{roid} = $fqdn . "-ISC"; # XXXKJL
    $lines->{crDate} = date_epp(from_date_ansi($h->{created_date}));
    $lines->{crID} = $self->_registrar_name($h->{created_id});
    $lines->{clID} = $self->_registrar_name($h->{registrar_id});
    $lines->{owner_id} = $self->_contact_name($h->{owner_id});
    $lines->{upDate} = date_epp(from_date_ansi($h->{updated_date}))
	if ($h->{updated_date});
    $lines->{upID} = $self->_registrar_name($h->{updated_id})
	if ($h->{updated_id});

    $lines->{addrs} = $self->_host_get_addresses(id => $h->{host_id});
    $lines->{status} = $self->_host_status(host_id => $h->{host_id});

    # this last item is for easy identity checking
    $lines->{registrar_id} = $h->{registrar_id};

    $self->commit;
        
    return (EPP_RS_SUCCESS, undef, $lines);
}

# XXXMLG needs changes for EPP-style commands
sub iscadmin {
    my ($self, %args) = @_;

    my @reqd = qw(name userid password whois url);
    _check_required_args("iscadmin", \%args, \@reqd);

    my $st = $self->rep_newst;
    my $db = $self->begin;

    my $rid = $args{registrar_id} || $st || $self->_new_rid;

    my $sth = $db->prepare("INSERT INTO registrar (st, registrar_id, name, userid, password, whois, url) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $sth->execute($st, $rid, $args{name}, $args{userid}, $args{password},
		  $args{whois}, $args{url});

    $self->commit;
    $self->rep_finish;

    return (EPP_RS_SUCCESS);
}

1;

__END__

=head1 NAME

ISC::SRS::DB - interface to the database layer for the ISC SRS

=head1 SYNOPSIS

use ISC::SRS::DB;

my $db = new ISC::SRS::DB(
    dbname => "srs",
    dbuser => "srs",
    dbpass => undef,
    serializable => undef,

    replication => "none",
);

The default values are shown.  The first set (C<dbname>, C<dbuser>, C<dbpass>,
and serializable) are passed used to create a L<ISC::DBI> object.  See
that module for the meaning of these variables.

If dbname, dbuser, or dbpass are not passed in, the environment variables
ISC_SRS_DBNAME, ISC_SRS_DBUSER, and ISC_SRS_DBPASS are used.  If those
do not exist, the default values are used.

Using a replication other than "none" is not supposed at this time.

=head1 DESCRIPTION

This module provides a low-level interface to the registry database to
manipulate the various objects stored in it.  Currently, only PostgreSQL
is supported as the storage database backend.

Transaction support is provided with the $db->begin(), $db->commit(), and
$db->rollback() calls, although an application should rarely need to call
these directly.  Replication, if configured, is also handled internally.

Each function is broken down into a group, such as registrar, contact,
domain, host, dns, or miscellaneous.  Other functions are available
internally, but typically those not listed here are internal only, and
any internal function beginning with an underscore (_) are purely internal.

=head1 REGISTRAR FUNCTIONS

=over

=item registrar_get(userid => USERID)

Get the registrar information associated with registrar USERID.  This is
a by-name lookup.

=back

=head1 CONTACT FUNCTIONS

=over

=item foo

=item bar

=back

=head1 DOMAIN FUNCTIONS

=over

=item foo

=item bar

=back

=head1 HOST FUNCTIONS

=over

=item foo

=item bar

=back

=head1 DNS FUNCTIONS

=over

=item bump_soa_serial(apex => APEX)

$db->bump_soa_serial(apex => "isc.org");

Bump the serial number for the registry DNS zone file for domain APEX.

=back

=head1 MISCELLANEOUS FUNCTIONS

=over

=item foo

=item bar

=back

=head1 AUTHOR

Written by Michael Graff, Joe Abley, and Kurt Lidl for the Internet
Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
