# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: NameVerify.pm,v 1.4 2002/12/06 02:21:09 lidl Exp $

package ISC::NameVerify;

use strict;
use warnings;

BEGIN {
  use Exporter();
  our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

  $VERSION = do { my @r = (q$Revision: 1.4 $ =~ /\d+/g); \
    sprintf "%d."."%02d" x $#r, @r};
  @ISA = qw(Exporter);
  @EXPORT = qw();
  @EXPORT_OK = qw();
  %EXPORT_TAGS = ();
}

our @EXPORT_OK;


# HostNameValid
#
# Check that a supplied character string is valid for use as a hostname.
#
# The following checks are convenient placeholders that should be
# replaced at some point with something more authoritative. For now:
#
# + /[a-z0-9]([a-z0-9-]{0,61}[a-z0-9]){0,1}/i matches a valid label;
#
# + there can be a maximum of 125 labels in a hostname;
#
# + the total length of the hostname will not exceed 253 characters;
#
# + there will be no trailing dot.

sub HostNameValid {
  my ($name) = @_;

  return ($name =~ m/^([a-z0-9]([a-z0-9-]{0,61}[a-z0-9]){0,1}\.){0,125}[a-z0-9]([a-z0-9-]{0,61}[a-z0-9])$/i && length($name) < 254)
}


# DomainNameValid
#
# Check that a supplied character string is valid for use as a domain.
#
# Domain names must:
#
# + contains exactly one dot;
#
# + be valid hostnames (since they are frequently used as hostnames).

sub DomainNameValid {
  my ($name) = @_;

  return ($name =~ m/^[^\.]+\.[^\.]+/ && HostNameValid($name));
}

1;

__END__

=head1 NAME

ISC::NameVerify - Verify that domain and host names are textually valid

=head1 SYNOPSIS

use ISC::NameVerify;

ISC::NameVerify::HostNameValid($text) or die "$text not valid";

ISC::NameVerify::DomainNameValid($text) or die "$text not valid";

=head1 DESCRIPTION

This package provides two functions which validate a host name and
a domain name.

=over

=item HostNameValid NAME

Returns a value which will test true if NAME is a valid host name.

=item DomainNameValid NAME

Returns a value which will test true if NAME is a valid domain name.

=back

=head1 AUTHOR

Written by Joe Abley for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.

