# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Listen.pm,v 1.4 2002/12/06 02:21:10 lidl Exp $

package ISC::CC::Direct::Listen;

use strict;
use warnings;

use Carp;

use ISC::CC::Direct;
use ISC::Net::Listen;
use ISC::Net::LengthPrefix;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.4 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter ISC::CC::Direct);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

END {
}

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $self = bless({}, $class);
    $self = $self->SUPER::new(%args);

    $self = bless($self, $class);

    $self->{connect_cb} = $args{connect_cb} if ($args{connect_cb});

    my $proto = $args{Proto} || "tcp";

    my %largs;

    $largs{cb} = \&_listen_cb;
    $largs{error_cb} = \&_listen_err;
    $largs{data} = $self;

    if ($proto eq "tcp") {
	$largs{LocalPort} = $args{LocalPort} if ($args{LocalPort});
	$largs{LocalAddr} = $args{LocalAddr} if ($args{LocalAddr});
	$largs{ReuseAddr} = $args{ReuseAddr} if ($args{ReuseAddr});
	$largs{Proto} = "tcp";
	$largs{Listen} = $args{Listen} if ($args{Listen});
    } elsif ($proto eq "local" || $proto eq "unix") {
	$largs{Proto} = "local";
	$largs{Peer} = $args{Peer} if ($args{Peer});
	$largs{Local} = $args{Local} if ($args{Local});
	$largs{Listen} = $args{Listen} if ($args{Listen});
    } else {
	croak "Unknown protocol type: $proto";
    }

    $self->{sock} = new ISC::Net::Listen(%largs);
    croak "Cannot open socket" unless ($self->{sock});

    return $self;
}

sub DESTROY {
    my ($self) = @_;

    if ($self->{sock}) {
	$self->{sock}->close();
    }

    $self->SUPER::DESTROY;
}

sub _listen_cb {
    my ($mp, $newsock) = @_;

    my $self = $mp->data;

    $self->{lp} = new ISC::Net::LengthPrefix(cb => \&ISC::CC::Direct::_msg_cb,
					     error_cb => \&ISC::CC::Direct::_msg_err,
					     socket => $newsock,
					     data => $self);

    bless($self->{lp}, "ISC::CC::DirectSocket");

    if ($self->{connect_cb}) {
	$self->{connect_cb}->($self->{lp}, $newsock);
    }
}

sub _listen_err {
    my ($con, $msg) = @_;

    my $self = $con->{data};

    $self->{error_cb}->($self, $msg);
    $con->socket->close;
    $con->cancel;
}

sub cancel {
    my ($self) = @_;

    $self->SUPER::cancel();

    $self->{sock}->close;
    $self->{sock} = undef;
}

1;
