# Copyright (C) 2010  Internet Systems Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SYSTEMS CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SYSTEMS CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#
# Tests for the configuration manager run script
#

import unittest
import os
import sys
from isc.testutils.parse_args import OptsError, TestOptParser

class MyConfigManager:
    def __init__(self, path, filename):
        self._path = path
        self.read_config_called = False
        self.notify_boss_called = False
        self.run_called = False
        self.write_config_called = False
        self.running = True

    def read_config(self):
        self.read_config_called = True

    def notify_boss(self):
        self.notify_boss_called = True

    def run(self):
        self.run_called = True

    def write_config(self):
        self.write_config_called = True

class TestConfigManagerStartup(unittest.TestCase):
    def test_cfgmgr(self):
        # some creative module use;
        # b10-cfgmgr has a hypen, so we use __import__
        # this also gives us the chance to override the imported
        # module ConfigManager in it.
        b = __import__("b10-cfgmgr")
        b.ConfigManager = MyConfigManager

        b.main()

        self.assertTrue(b.cm.read_config_called)
        self.assertTrue(b.cm.notify_boss_called)
        self.assertTrue(b.cm.run_called)
        # if there are no changes, config is not written
        self.assertFalse(b.cm.write_config_called)

        self.assertTrue(b.cm.running)
        b.signal_handler(None, None)
        self.assertFalse(b.cm.running)

        # TODO: take value from the 'global config module'
        # (and rename the .in away from this file again)
        data_path = "${prefix}/var/bind10-devel".replace("${prefix}", "/home/reed/opt/bind10")
        self.assertEqual(data_path, b.DATA_PATH)

        # remove the 'module' again, or later tests may fail
        # (if it is already present it won't be loaded again)
        sys.modules.pop("b10-cfgmgr")

    def test_cfgmgr_from_source(self):
        tmp_env_var = "/just/some/dir"
        env_var = None
        if "B10_FROM_SOURCE" in os.environ:
            env_var = os.environ["B10_FROM_SOURCE"]

        os.environ["B10_FROM_SOURCE"] = tmp_env_var
        b = __import__("b10-cfgmgr", globals(), locals())
        b.ConfigManager = MyConfigManager
        self.assertEqual(tmp_env_var, b.DATA_PATH)

        if env_var != None:
            os.environ["B10_FROM_SOURCE"] = env_var

        sys.modules.pop("b10-cfgmgr")

class TestParseArgs(unittest.TestCase):
    """
    Test for the parsing of command line arguments. We provide a different
    array to parse instead.
    """

    def test_defaults(self):
        """
        Test the default values when no options are provided.
        """
        # Pass it empty array, not our arguments
        b = __import__("b10-cfgmgr")
        parsed = b.parse_options([], TestOptParser)
        self.assertEqual(b.DATA_PATH, parsed.data_path)
        self.assertEqual(b.DEFAULT_CONFIG_FILE, parsed.config_file)

    def test_wrong_args(self):
        """
        Test it fails when we pass invalid option.
        """
        b = __import__("b10-cfgmgr")
        self.assertRaises(OptsError, b.parse_options, ['--wrong-option'],
                          TestOptParser)

    def test_not_arg(self):
        """
        Test it fails when there's an argument that's not option
        (eg. without -- at the beginning).
        """
        b = __import__("b10-cfgmgr")
        self.assertRaises(OptsError, b.parse_options, ['not-option'],
                          TestOptParser)

    def test_datapath(self):
        """
        Test overwriting the data path.
        """
        b = __import__("b10-cfgmgr")
        parsed = b.parse_options(['--data-path=/path'], TestOptParser)
        self.assertEqual('/path', parsed.data_path)
        self.assertEqual(b.DEFAULT_CONFIG_FILE, parsed.config_file)
        parsed = b.parse_options(['-p', '/path'], TestOptParser)
        self.assertEqual('/path', parsed.data_path)
        self.assertEqual(b.DEFAULT_CONFIG_FILE, parsed.config_file)
        self.assertRaises(OptsError, b.parse_options, ['-p'], TestOptParser)
        self.assertRaises(OptsError, b.parse_options, ['--data-path'],
                          TestOptParser)

    def test_db_filename(self):
        """
        Test setting the configuration database file.
        """
        b = __import__("b10-cfgmgr")
        parsed = b.parse_options(['--config-filename=filename'],
                                 TestOptParser)
        self.assertEqual(b.DATA_PATH, parsed.data_path)
        self.assertEqual("filename", parsed.config_file)
        parsed = b.parse_options(['-c', 'filename'], TestOptParser)
        self.assertEqual(b.DATA_PATH, parsed.data_path)
        self.assertEqual("filename", parsed.config_file)
        self.assertRaises(OptsError, b.parse_options, ['-c'], TestOptParser)
        self.assertRaises(OptsError, b.parse_options, ['--config-filename'],
                          TestOptParser)

if __name__ == '__main__':
    unittest.main()

