
/*
 * driver/usb/usb.c
 *
 * (C) Copyright Linus Torvalds 1999
 *
 * NOTE! This is not actually a driver at all, rather this is
 * just a collection of helper routines that implement the
 * generic USB things that the real drivers can use..
 *
 * Think of this as a "USB library" rather than anything else.
 * It should be considered a slave, with no callbacks. Callbacks
 * are evil.
 */

#include <linux/string.h>
#include <linux/bitops.h>
#include <linux/malloc.h>

#include "usb.h"

/*
 * We have a per-interface "registered driver" list.
 */


/*
 * This entrypoint gets called for each new device.
 *
 * We now walk the list of registered USB drivers,
 * looking for one that will accept this device as
 * his..
 *
 * Or rather, we _would_ do that, if we had implemented
 * that part.. Just show the information.
 */
void usb_device_descriptor(struct usb_device *dev)
{
	printk("New device:\n");
	usb_show_device(dev);
}

/*
 * Parse the fairly incomprehensible output of
 * the USB configuration data, and build up the
 * USB device database.
 */
static int usb_parse_endpoint(struct usb_endpoint_descriptor *endpoint, unsigned char *ptr, int len)
{
	if (len < 7)
		return -1;

	if (*(__u16 *) ptr != (USB_ENDPOINT_DESC | 7))
		return -1;

	memcpy(endpoint, ptr, 7);
	return 7;
}

static int usb_parse_interface(struct usb_interface_descriptor *interface, unsigned char *ptr, int len)
{
	int i;
	int parsed = 0;

	if (len < 9)
		return -1;

	if (*(__u16 *) ptr != (USB_INTERFACE_DESC | 9))
		return -1;

	memcpy(interface, ptr, 9);
	len -= 9;
	parsed += 9;

	if (interface->bNumEndpoints > USB_MAXENDPOINTS)
		return -1;

	for (i = 0; i < interface->bNumEndpoints; i++) {
		int retval = usb_parse_endpoint(interface->endpoint + i, ptr + parsed, len);
		if (retval < 0)
			return retval;
		parsed += retval;
		len -= retval;
	}
	return parsed;
}

static int usb_parse_config(struct usb_config_descriptor *config, unsigned char *ptr, int len)
{
	int i;
	int parsed = 0;

	if (len < 9)
		return -1;

	if (*(__u16 *) ptr != (USB_CONFIG_DESC | 9))
		return -1;

	memcpy(config, ptr, 9);
	len -= 9;
	parsed += 9;

	if (config->bNumInterfaces > USB_MAXINTERFACES)
		return -1;

	for (i = 0; i < config->bNumInterfaces; i++) {
		int retval = usb_parse_interface(config->interface + i, ptr + parsed, len);
		if (retval < 0)
			return retval;
		parsed += retval;
		len -= retval;
	}
	return parsed;
}

int usb_parse_configuration(struct usb_device *dev, void *__buf, int bytes)
{
	int i;
	unsigned char *ptr = __buf;

	if (dev->descriptor.bNumConfigurations > USB_MAXCONFIG)
		return -1;

	for (i = 0; i < dev->descriptor.bNumConfigurations; i++) {
		int retval = usb_parse_config(dev->config + i, ptr, bytes);
		if (retval < 0)
			return retval;
		ptr += retval;
		bytes += retval;
	}
	return 0;
}

void usb_init_root_hub(struct usb_device *dev)
{
	dev->devnum = -1;
	dev->slow = 0;
}

void usb_disconnect(struct usb_device *dev, struct usb_devmap * devmap)
{
	if (dev->devnum > 0)
		clear_bit(dev->devnum, devmap->devicemap);
}

/*
 * Connect a new USB device. This basically just initializes
 * the USB device information and sets up the topology - it's
 * up to the low-level driver to reset the port and actually
 * do the setup (the upper levels don't know how to do that).
 */
void usb_connect(struct usb_device *dev, struct usb_devmap *devmap)
{
	int devnum;

	memset(dev, 0, sizeof(*dev));
	dev->descriptor.bMaxPacketSize0 = 8;

	devnum = find_next_zero_bit(devmap->devicemap, 128, 1);
	if (devnum < 128) {
		set_bit(devnum, devmap->devicemap);
		dev->devnum = devnum;
	}
}
