#ifndef _SUPERMOUNT_I_H
#define _SUPERMOUNT_I_H

/*
 *  $Id: supermount.h,v 1.55 2004/01/14 20:24:56 bor Exp $
 */

#include <linux/module.h>

#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/list.h>
#include <linux/locks.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <linux/poll.h>
#include <linux/smp_lock.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/dnotify.h>

/*
 * The supermount superblock magic number
 */

#define SUPERMOUNT_SUPER_MAGIC	0x9fa1
#define SUPERMOUNT_VERSION		"1.2.11a"

#define S_DBG_DEBUG			0x001
#define S_DBG_TRACE_DENTRY		0x002
#define S_DBG_TRACE_FILE		0x004
#define S_DBG_TRACE_FILEMAP		0x008
#define S_DBG_TRACE_MEDIACTL		0x010
#define S_DBG_TRACE_NAMEI		0x020
#define S_DBG_TRACE_SUBFS		0x040
#define S_DBG_TRACE_SUPER		0x080

/*
 * The subfs umount reason
 */
#define SUBFS_UMNT_NORMAL		0 /* normal umount */
#define SUBFS_UMNT_MEDIA		1 /* media change detected */
#define SUBFS_UMNT_USER			2 /* user request */

/*
 * When to lock media
 */
#define TRAY_LOCK_NEVER			0
#define TRAY_LOCK_ONWRITE		1
#define TRAY_LOCK_ALWAYS		2

extern struct file_system_type supermount_fs_type;

#ifdef SUPERMOUNT_DEBUG

#define SUPERMOUNT_BUG_ON(x) BUG_ON(x)
#define SUPERMOUNT_BUG_LOCKED_ON(sb, x) \
do { \
	if (x) { \
		subfs_unlock(sb); \
		BUG(); \
	} \
} while (0)

#define supermount_debug(sb, args...) \
do { \
	struct supermount_sb_info *sbi = supermount_sbi(sb); \
	char *dev = sbi->devname ? sbi->devname : "unknown"; \
\
	if (supermount_dbg(sb, S_DBG_DEBUG)) { \
		printk("%sSUPERMOUNT DEBUG [dev=%s] <%s:%d> ", \
			KERN_DEBUG, dev, __FUNCTION__, __LINE__); \
		printk(args); \
		printk("\n"); \
	} \
} while(0)

#define ENTER(sb, args...) \
do { \
	struct supermount_sb_info *sbi = supermount_sbi(sb); \
	char *dev = sbi->devname ? sbi->devname : "unknown"; \
\
	if (supermount_dbg(sb, S_DBG_TRACE_CURRENT)) { \
		printk("%sSUPERMOUNT TRACE [dev=%s] PID=%lu ENTER %s", \
			KERN_DEBUG, dev, (unsigned long)current->pid, \
			__FUNCTION__); \
		printk(" " args); \
		printk("\n"); \
	} \
} while (0)

#define LEAVE(sb, args...) \
do { \
	struct supermount_sb_info *sbi = supermount_sbi(sb); \
	char *dev = sbi->devname ? sbi->devname : "unknown"; \
\
	if (supermount_dbg(sb, S_DBG_TRACE_CURRENT)) { \
		printk("%sSUPERMOUNT TRACE [dev=%s] PID=%lu LEAVE %s", \
			KERN_DEBUG, dev, (unsigned long)current->pid, \
			__FUNCTION__); \
		printk(" " args); \
		printk("\n"); \
	} \
} while (0)

#else /* SUPERMOUNT_DEBUG */
#define supermount_debug(f, a...) /**/
#define SUPERMOUNT_BUG_ON(sb, x) do { } while (0)
#define SUPERMOUNT_BUG_LOCKED_ON(x) do { } while (0)

#define ENTER(sb, args...) do { (void)sb; } while (0)
#define LEAVE(sb, args...) do { (void)sb; } while (0)
#endif				/* SUPERMOUNT_DEBUG */

#define supermount_warning(sb, ...) \
do { \
	struct supermount_sb_info *sbi = supermount_sbi(sb); \
	char *dev = sbi->devname ? sbi->devname : "unknown"; \
\
	printk("%sSUPERMOUNT WARNING [dev=%s] ", KERN_WARNING, dev); \
	printk(__VA_ARGS__); \
	printk("\n"); \
} while(0)

#define supermount_error(sb, ...) \
do { \
	struct supermount_sb_info *sbi = supermount_sbi(sb); \
	char *dev = sbi->devname ? sbi->devname : "unknown"; \
\
	printk("%sSUPERMOUNT ERROR [dev=%s] ", KERN_ERR, dev); \
	printk(__VA_ARGS__); \
	printk("\n"); \
} while(0)

/*
 * The following is drived from fs/inode.c:update_atime()
 */
#define NEED_WRITE_ATIME(inode) \
	(!(IS_NOATIME(inode) || \
	 (IS_NODIRATIME(inode) && S_ISDIR(inode->i_mode)) || \
	 IS_RDONLY(inode)))


/*
 * supermount super-block data in memory
 */
struct supermount_sb_info;
struct supermount_sb_info {
	/* == options == */
	unsigned long s_debug;	/* debug flags S_DBG_* */
	char *s_type;		/* Type of fs to be sub-mounted */
	char *devname;		/* Where to mount the subfs from */
	char *s_data;		/* Data to pass when mounting subfs */
	/* == end of options == */
	struct vfsmount *s_undermount;
				/* Mount point for subfs */
	int readcount;		/* Refcount of read access
				   on the filesystem */
	int writecount;		/* Refcount of write access
				   on the filesystem */
	int lockcount;		/* Refcount of requests to lock tray */
	struct list_head s_inodes;
				/* list of active inodes */
	struct list_head s_files;
				/* list of active files */
	struct list_head s_dentries;
				/* list of active dentries */
	struct semaphore sem;
	struct super_block *host;
				/* needed for procfs support */
	struct supermount_sb_info *next;
				/* list of all supermount fs */
	unsigned int tray_lock:2;
	unsigned int disabled:1;
	unsigned int rw:1;
};

static inline struct supermount_sb_info *
supermount_sbi(struct super_block *sb)
{
	struct supermount_sb_info *sbi;
	SUPERMOUNT_BUG_ON(!sb);
	SUPERMOUNT_BUG_ON(sb->s_type != &supermount_fs_type);

	sbi = (struct supermount_sb_info *) (sb->u.generic_sbp);
	SUPERMOUNT_BUG_ON(!sbi);

	return sbi;
}

static inline struct vfsmount *
subfs_mnt(struct super_block *sb)
{
	return supermount_sbi(sb)->s_undermount;
}

static inline struct super_block *
subfs_sb(struct super_block *sb)
{
	struct vfsmount *mnt = subfs_mnt(sb);

	if (mnt) return mnt->mnt_sb;

	return 0;
}

/* this is expected to run under sb->sem */
static inline int
subfs_is_mounted(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	return sbi->s_undermount != 0;
}

/* this is expected to run under sb->sem */
static inline int
subfs_is_busy(struct super_block *sb)
{
	struct vfsmount *mnt = subfs_mnt(sb);

	/*
	 * In "normal" case mnt_count is 2. But we currently do
	 * not insert subfs into task namespace so count is 1
	 * FIXME
	 * This also means we do not do_umount i.e. do not run
	 * either umount_begin or DQUOT_OFF for subfs.
	 */
	if (mnt) return atomic_read(&mnt->mnt_count) > 1;

	return 0;
}

/* this is expected to run under sb->sem */
static inline int
subfs_is_rw(struct super_block *sb)
{
	struct super_block *subsb;

	if (!subfs_is_mounted(sb))
		return 0;
	
	subsb = subfs_sb(sb);
	return !(subsb->s_flags & MS_RDONLY);
}

static inline void
subfs_lock(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	down(&sbi->sem);
}

static inline void
subfs_unlock(struct super_block *sb)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	up(&sbi->sem);
}

/*
 * query debug flags set
 */
static inline int supermount_dbg(struct super_block *sb, unsigned long flags)
{
	struct supermount_sb_info *sbi = supermount_sbi(sb);

	return sbi->s_debug & flags;
}

/*
 * supermount inode info
 */

struct supermount_inode_info {
	struct list_head list;
  	struct inode *inode;
	struct inode *host;
	int readcount;
	int writecount;
};

static inline int
is_inode_supermounted(struct inode *inode)
{
	return inode && inode->i_sb && inode->i_sb->s_type == &supermount_fs_type;
}

static inline struct supermount_inode_info *
supermount_i(struct inode *inode)
{
	SUPERMOUNT_BUG_ON(!is_inode_supermounted(inode));
	SUPERMOUNT_BUG_ON(is_bad_inode(inode) ^ (inode->u.generic_ip == 0));

	return inode->u.generic_ip;
}

static inline int
is_inode_obsolete(struct inode *inode)
{
	struct supermount_inode_info *sii = supermount_i(inode);

	return sii->inode == 0;
}

static inline void
supermount_list_add_inode(struct inode *inode)
{
	struct supermount_inode_info *sii = supermount_i(inode);

	list_add(&(sii->list), &(supermount_sbi(inode->i_sb)->s_inodes));
}

/*
 * FIXME
 * Should we propagate all flags? *_QUOTA looks very possible candidate
 * We can't just assign them because other flags may be set by VFS
 */
#define SMNT_INODE_FLAGS (S_IMMUTABLE|S_NOATIME|S_APPEND|S_SYNC)
static inline void
set_inode_flags(struct inode *inode, struct inode *subi)
{
	inode->i_flags &= ~SMNT_INODE_FLAGS;
	if (subi)
		inode->i_flags |= (subi->i_flags & SMNT_INODE_FLAGS);
}

/*
 * supermount dentry info
 */

struct supermount_dentry_info {
	struct list_head list;
	struct dentry *dentry;
	struct dentry *host;
};

static inline int
is_dentry_supermounted(struct dentry *dentry)
{
	return (dentry && dentry->d_sb && dentry->d_sb->s_type == &supermount_fs_type);
}

static inline struct supermount_dentry_info *
supermount_d(struct dentry *dentry)
{
	SUPERMOUNT_BUG_ON(!is_dentry_supermounted(dentry));
	SUPERMOUNT_BUG_ON(!dentry->d_fsdata);

	return  (struct supermount_dentry_info *)dentry->d_fsdata;
}

static inline int
is_dentry_obsolete(struct dentry *dentry)
{
	struct supermount_dentry_info *sdi = supermount_d(dentry);

	return sdi->dentry == 0;
}

/*
 * Supermount file info 
 */

struct supermount_file_info {
	struct list_head	list;
	struct file *		host;
	struct file		*file;
	pid_t			owner;
	struct vm_operations_struct *vm_ops;
	unsigned int		fake:1;
};

static inline int
is_file_supermounted(struct file *file)
{
	return file && file->f_dentry && is_dentry_supermounted(file->f_dentry);
}

static inline struct supermount_file_info *
supermount_f(struct file *file)
{
	SUPERMOUNT_BUG_ON(!is_file_supermounted(file));
	SUPERMOUNT_BUG_ON(!file->f_supermount);

	return  file->f_supermount;
}

static inline int
is_file_obsolete(struct file *file)
{
	struct supermount_file_info *sfi = supermount_f(file);

	return sfi->file == NULL;
}

static inline int
is_file_fake(struct file *file)
{
	struct supermount_file_info *sfi = supermount_f(file);

	return sfi->fake;
}

/* dentry.c */
extern struct dentry_operations supermount_dops;
extern int init_dentry_info(struct dentry *);
extern void attach_subfs_dentry(struct dentry *, struct dentry *);
extern struct dentry *get_subfs_dentry(struct dentry *dentry);
 
/* file.c */
extern struct file_operations supermount_dir_operations;
extern struct file_operations supermount_file_operations;

/* filemap.c */
extern struct vm_operations_struct supermount_vm_ops;
extern int supermount_file_mmap(struct file *, struct vm_area_struct *);
extern struct file *get_subfs_file(struct file*);

/* mediactl.c */
extern void supermount_mediactl(struct super_block *, int, int);
static inline void
supermount_lock_door(struct super_block *sb)
{
	supermount_mediactl(sb, MEDIA_LOCK, 1);
}
static inline void
supermount_unlock_door(struct super_block *sb)
{
	supermount_mediactl(sb, MEDIA_UNLOCK, 1);
}

static inline void
mark_media_supermounted(struct super_block *sb)
{
	supermount_mediactl(sb, SUPERMOUNT_INC_COUNT, 0);
	supermount_mediactl(sb, MEDIA_UNLOCK, 0);
}

static inline void
unmark_media_supermounted(struct super_block *sb)
{
	supermount_mediactl(sb, SUPERMOUNT_DEC_COUNT, 0);
	supermount_mediactl(sb, MEDIA_UNLOCK, 0);
}

/* namei.c */
extern struct inode_operations supermount_dir_iops;
extern struct inode_operations supermount_file_iops;
extern struct inode_operations supermount_symlink_iops;
extern struct inode *create_supermount_inode(struct super_block *);
extern struct inode *get_subfs_inode(struct inode *inode);

/* proc.c */
#ifdef CONFIG_PROC_FS
extern void supermount_proc_register(void);
extern void supermount_proc_unregister(void);
extern void supermount_proc_insert(struct supermount_sb_info *);
extern void supermount_proc_remove(struct supermount_sb_info *);
#else
#define supermount_proc_register() do { \
	printk(KERN_INFO "Supermount version %s for kernel 2.4\n", \
	       SUPERMOUNT_VERSION); \
} while(0)
#define supermount_proc_unregister() do { } while(0)
#define supermount_proc_insert(sbi) do { } while(0)
#define supermount_proc_remove(sbi) do { } while(0)
#endif

/* subfs.c */
extern void subfs_umount(struct super_block *sb, int);
extern struct vfsmount *subfs_go_online(struct super_block *);
extern void subfs_go_offline(struct super_block *, struct vfsmount *);
extern int subfs_get_access(struct inode *, int);
extern void subfs_put_access(struct inode *, int);
extern int subfs_check_disk_change(struct super_block *);
extern struct vfsmount *subfs_prevent_umount(struct super_block *);
extern void subfs_allow_umount(struct super_block *, struct vfsmount *);
extern struct vfsmount *subfs_get_mnt(struct super_block *sb);
extern struct super_block *subfs_get_sb(struct super_block *sb);
extern void subfs_clear_inode(struct inode *);

/* super.c */
extern struct super_block *supermount_read_super(struct super_block *,
						 void *, int);
extern void attach_subfs_inode(struct inode *, struct inode *);
extern void supermount_init_root_inode(struct inode *);
extern char *strdup(const char *);

#endif				/* _SUPERMOUNT_I_H */
