/*
 * $Id: lrcmd.c,v 1.8 2004/03/02 11:59:43 yumo Exp $
 * lrcfg -- lr device driver configurator program
 *
 * Author: Yumo (Katsuyuki Yumoto) 2001-2004
 *         yumo@st.rim.or.jp
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netinet/in.h>

typedef unsigned short u16;
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,5)
#include "miicompat.h"
#else
#include <linux/mii.h>
#endif

#ifdef ETHMAP
#include <ethmap.h>
#endif

#include <linux/sockios.h>
#include "lr.h"
#include "lrcfg.h"

/* internal error codes */
#define ERR_FAILCNF -1
#define ERR_FEWARGS -2
#define ERR_INVALPIF -3
#define ERR_INVALPNAM -4
#define ERR_INVALPTR -5
#define ERR_INVALPVAL -6
#define ERR_INVALSD -7
#define ERR_INVALVIF -8
#define ERR_NODRV -9
#define ERR_NOMEM -10
#define ERR_UNKNCOMM -11

/* maximum name length */
#define MAX_NAME 7


char *get_field_n(char *d, char *s, int n)
{
	char *p, *q, c;
	int i;

	if (!d || !s)
		goto exit;

	p = strrchr (s, '\n');
	if (p){
		*p = '\0';
	}
	for (p = q = s, i = 1; p && *p;){
		for (; (*p == ' ' || *p == '\t') && *p; p++)
			;
		q = strchr(p, ' ');
		if (!q){
			q = strchr(p, '\t');
		}
		if (!q){
			q = strchr(p, '\0');
		}
		if (i == n){
			c = *q;
			*q = '\0';
			strcpy(d, p);
			*q = c;
			return d;
		}
		i++;
		p = q;
	}

exit:
	return NULL;
}



char *get_module_name(char *d, char *ifname)
{
	FILE *fp;
	char astr[MAXLINE][MAXCOL+1];
	char s1[MAXCOL+1];
	char s2[MAXCOL+1];
	char s3[MAXCOL+1];
	char s4[MAXCOL+1];
	char s5[MAXCOL+1];
	char cmd[128];
	char *p, *q;
	int i = 0, j = 0, n;

	if (!d || !ifname || strlen(ifname) > MAX_NAME){
		return NULL;
	}

	/* first stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 up", ifname);

	if (system(cmd)){
		return NULL;
	}

	fp = popen("/bin/cat /proc/modules", "r");
	if (!fp){
		return NULL;
	}
	for (i = 0; fgets(astr[i]+1, MAXCOL, fp) && i < MAXLINE; i++){
		if (get_field_n(s1, astr[i]+1, 3)){
			if (!atoi(s1)){
				astr[i][0] = 1; /* exclusive mark */
			} else {
				astr[i][0] = 0;
			}
		}
	}
	pclose (fp);

	n = i;

	/* second stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 down", ifname);

	if (system(cmd)){
		return NULL;
	}
	
	if ((fp = popen("/bin/cat /proc/modules", "r")) == NULL){
		return NULL;
	}
	for (; fgets(s1, MAXCOL, fp);){
		if (get_field_n(s2, s1, 1)){
			for (i = 0; i < n; i++){
				if (!astr[i][0] && get_field_n(s3, astr[i]+1, 1)){
					if (strcmp(s2, s3) == 0){
						p = get_field_n(s4, astr[i]+1, 3);
						q = get_field_n(s5, s1, 3);
						if (p && q && atoi(p) != atoi(q)+1){
							astr[i][0] = 1; /* exclusive mark */
						}
					}
				}
			}
		} else {
			break;
		}
	}
	pclose (fp);


	/* third stage */
	sprintf(cmd, "/sbin/ifconfig %s 0 up", ifname);

	if (system(cmd)){
		return NULL;
	}
	
	if ((fp = popen("/bin/cat /proc/modules", "r")) == NULL){
		return NULL;
	}
	for (j = 0; fgets(s1, MAXCOL, fp);){
		if (get_field_n(s2, s1, 1)){
			for (i = 0; i < n; i++){
				if (!astr[i][0] && get_field_n(s3, astr[i]+1, 1)){
					if (strcmp(s2, s3) == 0){
						p = get_field_n(s4, astr[i]+1, 3);
						q = get_field_n(s5, s1, 3);
						if (p && q && atoi(p) != atoi(q)){
							astr[i][0] = 1; /* exclusive mark */
						} else {
						    j++;
							strcpy(d, s2);
						}
					}
				}
			}
		} else {
			break;
		}
	}
	pclose (fp);


	/* check if single line is remaining */
	if (j != 1){
		return NULL;
	}
	return d;
}







int lr_config(int sock, struct ifreq *req)
{
	int ret;

	ret = ioctl(sock, SIOCDEVPRIVATE, req);
	if (ret < 0){
		perror(PROG_NAME);
	}
	return ret;
}


void print_version(struct u_drv_info di)
{
	printf ("%s %d.%d.%d\n", di.name, di.major, di.minor, di.patch);
	printf ("Copyright 2001-2004 Yumo (Katsuyuki Yumoto)\n");
	printf ("\
This is free software; see the source for copying conditions. There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n");
	printf ("\n");
	return;
}


int version_match(struct u_drv_info di)
{
	return((di.major == LR_VER_MAJOR) && (di.minor == LR_VER_MINOR))?1:0;
}


void print_general_info(char *s)
{
	char sbuf[1024];
	int len;

	strcpy(sbuf, s);
	len = strlen(sbuf);
	sprintf(sbuf+len, "lrcfg version            : %d.%d.%d\n",
		   LR_VER_MAJOR, LR_VER_MINOR, LR_VER_PATCH);
	len = strlen(sbuf);
	sprintf(sbuf+len,"Please type \"lrcfg -h\" to show usage.\n\n");
	printf("%s", sbuf);
	return;
}


int does_driver_exists(int s, struct u_drv_info *di)
{
	struct ifreq req;
	struct lrconf vc;
	int ret = 1;

	if (s < 0 || !di){
		ret = 0;
		goto exit;
	}
	(struct lrconf *)req.ifr_data = &vc;
	vc.buf = (unsigned char *)di;
	vc.size = sizeof(struct u_drv_info);

	/* check if lr device driver exists... */
	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	vc.action = LRCTL_VERSION;
	vc.magic = LR_MAGIC;
	if (lr_config(s, &req))
		ret = 0;

	if (ret && strcmp(di->name, "LR") != 0)
		ret = 0;

exit:
	return ret;
}

int add_phyif(int s, char *pif, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	char str[MAXCOL];
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!pif ||	strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	(struct lrconf *)req.ifr_data = &vc;
	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	strcpy(vc.vif_name, vif);
	strcpy(vc.pif_name, pif);
	if (get_module_name(str, pif)){
		strcpy(vc.pif_module, str);
	} else {
		vc.pif_module[0] = '\0';
	}
	vc.action = LRCTL_ADD_PHYIF;
	vc.magic = LR_MAGIC;
	vc.size = 0;
	vc.buf = NULL;
	if (lr_config(s, &req))
		goto exit;
	ret = 0;

exit:
	return ret;
}

int del_phyif(int s, char *pif)
{
	struct ifreq req;
	struct lrconf vc;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!pif ||	strlen(pif) > MAX_NAME){
		ret = ERR_INVALPIF;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.pif_name, pif);
	vc.action = LRCTL_DEL_PHYIF;
	vc.magic = LR_MAGIC;
	vc.size = 0;
	vc.buf = NULL;

	if (lr_config(s, &req)){
		goto exit;
	}
	ret = 0;

exit:
	return ret;
}

int print_dist_info(int s, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	unsigned char buf[1024*1024];
	struct u_dist_info *di = (struct u_dist_info *)buf;
	int i, j;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.vif_name, vif);
	vc.action = LRCTL_GET_DISTINFO;
	vc.magic = LR_MAGIC;
	vc.size = 1024*1024;
	vc.buf = buf;

	if (lr_config(s, &req))
		goto exit;

	printf("\nlr distributing information\n");
	printf("\nlr device: %s\n\n", vif);
	printf("  neighbors       sub-group      age\n");
	for (i = 0; i < vc.size; i++){
		printf("%c ", di[i].is_bfu?'*':' ');
		for (j = 0; j < 3; j++)
			printf("%02x", di[i].m_addr[j]);
		printf("-");
		for (j = 3; j < 6; j++)
			printf("%02x", di[i].m_addr[j]);
		printf("      %2d        %5d\n", di[i].sgid, di[i].age);
	}
	printf("\n");

	ret = 0;

exit:
	return ret;
}

int print_bfu_info(int s, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	unsigned char *buf = NULL;
	struct u_bfu_info *di = NULL;
	int i, j;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	buf = (unsigned char *)malloc(sizeof(unsigned char)*1024*1024);
	di = (struct u_bfu_info *)buf;
	if (!buf){
		ret = ERR_NOMEM;
		goto exit;
	}
	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.vif_name, vif);
	vc.action = LRCTL_GET_BFUINFO;
	vc.magic = LR_MAGIC;
	vc.size = 1024*1024;
	vc.buf = buf;

	if (lr_config(s, &req))
		goto exit;

	printf("\nlr broadcast filtering unit information\n");
	printf("\nlr device: %s\n\n", vif);
	printf("neighbors       rcv sub-group      age\n");
	for (i = 0; i < vc.size; i++){
		for (j = 0; j < 3; j++)
			printf("%02x", di[i].gaddr[j]);
		printf("-");
		for (j = 3; j < 6; j++)
			printf("%02x", di[i].gaddr[j]);
		printf("      %2d        %5d\n", di[i].min_sgid, di[i].age);
	}
	printf("\n");

	ret = 0;
exit:
	if (buf)
		free(buf);

	return ret;
}


int print_group_info(struct u_group_info u, char *vif)
{
	int i;
	int ret = 0;

	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	printf("\nlr group information\n");
	printf("\nlr device: %s\n\n", vif);
	printf("group id:           %2d\n", u.gid);
	printf("group address:     ");
	for (i = 0; i < 3; i++)
		printf("%02x", u.gaddr[i]);
	printf("-");
	for (i = 3; i < 6; i++)
		printf("%02x", u.gaddr[i]);
	printf("\n");
	printf("periodic time:   %5d\n", u.perio_time);
	printf("age time:        %5d\n", u.age_time);
	printf("topology change detection/notification: %s\n",
		   u.tchg_opt?"yes":"no");
	printf("LRCP Phase II compatibility: %s\n",
		   u.pii_opt?"yes":"no");
	printf("group link status:   %s\n", u.lstat?"up":"down");
	printf("\n\n");

exit:
	return ret;
}

int get_group_info(struct u_group_info *u, int s, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	int i;
	int ret = ERR_FAILCNF;

	if (!u){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.vif_name, vif);
	vc.action = LRCTL_GET_GRPINFO;
	vc.magic = LR_MAGIC;
	vc.size = sizeof(struct u_group_info);
	(struct u_group_info *)vc.buf = u;

	if (lr_config(s, &req))
		goto exit;
	ret = 0;

exit:
	return ret;
}

int set_group_info(struct u_group_info *u, int s, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	int ret = ERR_FAILCNF;

	if (!u){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	(struct vethconf *)req.ifr_data = &vc;
	strcpy(vc.vif_name, vif);
	vc.action = LRCTL_SET_GRPINFO;
	vc.magic = LR_MAGIC;
	vc.size = sizeof(struct u_group_info);
	(struct u_group_info *)vc.buf = u;
	
	if (lr_config(s, &req))
		goto exit;
	ret = 0;

exit:
	return ret;
}

int group_param_set(struct u_group_info *u, char *pnam, char *pval)
{
	int ret = 0;

	if (!u){
		ret = ERR_INVALPTR;
		goto exit;
	}
	if (!pnam || strlen(pnam) > MAX_NAME){
		ret = ERR_INVALPNAM;
		goto exit;
	}
	if (!pval || strlen(pval) > MAX_NAME){
		ret = ERR_INVALPVAL;
		goto exit;
	}

	if (strcmp(pnam, "perio") == 0){
		u->perio_time = atoi(pval)&0x7fff;
	} else if (strcmp(pnam, "age") == 0){
		u->age_time = atoi(pval)&0x7fff;
	} else if (strcmp(pnam, "tchg") == 0){
		u->tchg_opt = (strcmp(pval, "yes") == 0)?1:0;
	} else if (strcmp(pnam, "pii") == 0){
		u->pii_opt = (strcmp(pval, "yes") == 0)?1:0;
	} else {
		ret = ERR_INVALPNAM;
		goto exit;
	}

	if (u->perio_time == 0)
		ret = ERR_INVALPVAL;

exit:
	return ret;
}

int print_port_list(int s, char *vif)
{
	struct ifreq req;
	struct lrconf vc;
	unsigned char buf[1024*1024];
	int i;
	struct u_port_info *p = (struct u_port_info *)buf;
	int ret = ERR_FAILCNF;
	char ifname[MAX_NAME*4];
#ifdef ETHMAP
	char *sticky_intf = NULL;
#endif

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}
	if (!vif ||	strlen(vif) > MAX_NAME){
		ret = ERR_INVALVIF;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.vif_name, vif);
	vc.action = LRCTL_GET_PORTLIST;
	vc.magic = LR_MAGIC;
	vc.size = 1024*1024;
	vc.buf = buf;

	if (lr_config(s, &req)){
		goto exit;
	}

	printf("\nlr port information\n");
	printf("\nlr device: %s\n\n", vif);
	printf("active      port       name        sub-group  group  link\n");
	for (i = 0; i < vc.size; i++){
#ifdef ETHMAP
		sticky_intf = ethmap_reverse_query(p[i].dev_name);
		if (sticky_intf)
			sprintf(ifname, "[%s]", sticky_intf);
		else
#endif
			sprintf(ifname, "%s", p[i].dev_name);
		printf("   %c         %2d   %10s          %2d       %2d     %-4s\n",
			   (p[i].pid==p[i].aid)?'*':' ',
			   p[i].pid,
			   ifname,
			   p[i].sgid, p[i].gid,
			   p[i].lstat?"up":"down");
	}
	printf("\n");
	ret = 0;

exit:
	return ret;
}


int print_sys_info(int s)
{
	struct ifreq req;
	struct lrconf vc;
	struct u_sys_info si;
	int i;
	int ret = ERR_FAILCNF;

	if (s < 0){
		ret = ERR_INVALSD;
		goto exit;
	}

	strcpy(req.ifr_name, DEFAULT_LR_NAME);
	req.ifr_data = (unsigned char *)&vc;
	strcpy(vc.vif_name, DEFAULT_LR_NAME);
	vc.action = LRCTL_GET_SYSINFO;
	vc.magic = LR_MAGIC;
	vc.size = sizeof(struct u_sys_info);
	vc.buf = (unsigned char *)&si;

	if (lr_config(s, &req)){
		goto exit;
	}

	printf("\nlr system global information\n\n");
	printf("system id:  ");
	for (i = 0; i < 3; i++)
		printf("%02x", si.sys_id[i]);
	printf("-");
	for (i = 3; i < 6; i++)
		printf("%02x", si.sys_id[i]);
	printf("\n\n");
	
	ret = 0;

exit:
	return ret;
}

void print_reason(int code)
{
	switch(code){
	case ERR_FAILCNF:
		fprintf(stderr, "%s: Fail to ioctl.\n", PROG_NAME);
		break;
	case ERR_FEWARGS:
		fprintf(stderr, "%s: Too few arguments.\n", PROG_NAME);
		break;
	case ERR_INVALPIF:
		fprintf(stderr, "%s: Invalid physical interface name\n", PROG_NAME);
		break;
	case ERR_INVALPNAM:
		fprintf(stderr, "%s: Invalid parameter name\n", PROG_NAME);
		break;
	case ERR_INVALPTR:
		fprintf(stderr, "%s: Invalid pointer\n", PROG_NAME);
		break;
	case ERR_INVALPVAL:
		fprintf(stderr, "%s: Invalid parameter value\n", PROG_NAME);
		break;
	case ERR_INVALSD:
		fprintf(stderr, "%s: Invalid socket id\n", PROG_NAME);
		break;
	case ERR_INVALVIF:
		fprintf(stderr, "%s: Invalid virtual interface name\n", PROG_NAME);
		break;
	case ERR_NODRV:
		fprintf(stderr, "%s: lr driver does not exists.\n", PROG_NAME);
		break;
	case ERR_NOMEM:
		fprintf(stderr, "%s: Cannot allocate memory.\n", PROG_NAME);
		break;
	case ERR_UNKNCOMM:
		fprintf(stderr, "%s: Unknown command\n", PROG_NAME);
		break;
	}
}

int lr_command(int argc, char **argv)
{
	int sock;
	char sbuf[1024];
	char str[MAXCOL];
	struct u_drv_info di;
	int ret = 0;
#ifdef ETHMAP
	char *sticky_intf = NULL;
#endif

	if (argc < 2) {
		usage();
		ret = 1;
		goto close_and_ret;
	}
	if (strcmp(argv[1], "-h") == 0 ||
		strcmp(argv[1], "--help") == 0){
		usage();
		goto close_and_ret;
	}
	
	/* a silly raw socket just for ioctl()ling it */
	sock = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
	if (sock < 0) {
		perror(PROG_NAME);
		print_general_info("");
		ret = 1;
		goto close_and_ret;
	}
	
	if (!does_driver_exists(sock, &di)){
		print_general_info("lr device driver does not exists.\n");
		ret = 1;
		goto close_and_ret;
	}
	
	if (!version_match(di)){
		print_general_info("Version mismatch between lrcfg and lr driver.\nUpdate lrcfg please.\n\n");
		printf("lr device driver version : %d.%d.%d\n", di.major, di.minor, di.patch);
		printf("lrcfg version            : %d.%d.%d\n",
			   LR_VER_MAJOR, LR_VER_MINOR, LR_VER_PATCH);
		ret = 1;
		goto close_and_ret;
	}
	
	
	if (strcmp(argv[1], "add") == 0){
		if (argc < 4){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		ret = add_phyif(sock, sticky_intf, argv[3]);
#else
		ret = add_phyif(sock, argv[2], argv[3]);
#endif
	} else if (strcmp(argv[1], "del") == 0){
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
#ifdef ETHMAP
		sticky_intf = ethmap_query(argv[2]);
		ret =del_phyif(sock, sticky_intf);
#else
		ret =del_phyif(sock, argv[2]);
#endif
	} else if (strcmp(argv[1], "list") == 0){
		FILE *fp = fopen("/proc/net/lrcp", "r");
		if (!fp){
			ret = ERR_NODRV;
			goto close_and_ret;
		}
		for (; fgets(str, MAXCOL, fp);){
			printf("%s", str);
		}
	} else if (strcmp(argv[1], "showdist") == 0){
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
		ret = print_dist_info(sock, argv[2]);
	} else if (strcmp(argv[1], "showbfu") == 0){
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
		ret = print_bfu_info(sock, argv[2]);
	} else if (strcmp(argv[1], "showport") == 0){
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
		ret = print_port_list(sock, argv[2]);
	} else if (strcmp(argv[1], "showsys") == 0){
		ret = print_sys_info(sock);
	} else if (strcmp(argv[1], "showgrp") == 0){
		struct u_group_info u;
		
		if (argc < 3){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
		ret = get_group_info(&u, sock, argv[2]);
		if (ret)
			goto close_and_ret;

		ret = print_group_info(u, argv[2]);
	
	} else if (strcmp(argv[1], "setgrp") == 0){
	
		struct u_group_info u;
		
		if (argc < 5){
			ret = ERR_FEWARGS;
			goto close_and_ret;
		}
		
		if (strncmp(argv[2], "lr", 2)){
			ret = ERR_INVALVIF;
			goto close_and_ret;
		}
		
		ret =get_group_info(&u, sock, argv[2]);
		if (ret){
			goto close_and_ret;
		}
		
		ret = group_param_set(&u, argv[3], argv[4]);
		if (ret){
			goto close_and_ret;
		}
		
		ret = set_group_info(&u, sock, argv[2]);
		
	} else if (strcmp(argv[1], "ver") == 0){
		print_version(di);
		ret = 0;
		goto close_and_ret;
	} else {
		ret = ERR_UNKNCOMM;
		goto close_and_ret;
	}
	
close_and_ret:
	if (ret < 0)
		print_reason(ret);
	
	if (ret){
		ret = 1;
	}
	close(sock);
	return ret;
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  indent-tabs-mode: t
 * End:
 */
