// **********************************************************************
// * IBMASM HEADER FILE                                                 *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// -------------------------------------------------------------------------------------------------------
// SLIM.H
//
// CHANGE HISTORY:
// Date      Author       Description
//
//  ??         ??         Creation
//
// 08/22/97  A. Parsons   Formatted, added comments.  Looks OK for Kiowa.
//
// 09/17/97  A. Parsons   Modified the trailer/trailer_buf union so that trailer_buf[3]
//                        is now trailer_buf[4], which fixes a dynamic memory leak in
//                        the Netware device driver.
//
// 04/23/99  M. DeLoera   Added a _FAR directive in order to make this a common file
//                        for all platforms. OS/2 requires all 32-bit pointers to use
//                        the far keyword, so the directive will insert that keyword in
//                        all appropriate places when building in an OS/2 environment.
//
// -------------------------------------------------------------------------------------------------------

/**************************************************************************
*
*  Description:
*       This file contains an implementation of an enhanced SLIP protocol
*               developed by Synoptics and utilized license free. The protocol is
*               described in the following sections.
*
*               The SlipClass is intended to be used as a base class for several
*               serial classes. SlipClass itself inherits form MsTransClass and
*               implements SendSlimMessage and ReadMessage. SlipClass uses SendRaw and
*               ReadRaw to actually perform the IO operations. ReadRaw may either
*               suspend waiting for characters or return FALSE if no characters exist.
*               However, in this case ReadMessage will return FALSE only if there has not
*               been a start of a message. This avoids an extra level of buffering that
*               would otherwise be required.
*

OUT-OF-BAND SNMP COMMUNICATIONS

The protocol described in this document is used for carrying SNMP packets over
asynchronous serial lines.  SynOptics agents and network management station
software applications use this protocol for out-of-band management access to
remote sites and for situations when the usual inband communications channels
are temporarily dysfunctional.

In addition to the description of the packet framing, we have enclosed
fragments of the code used to implement this protocol.  It is designed to
interface to the Epilogue SNMP agent libraries.

This information is made available without copyright, although we would
appreciate your giving credit to SynOptics Communications if you distribute
the information further.  SynOptics does not provide support for your use of
this protocol, and is not responsible for problems that you may have with it.

DESCRIPTION

Out-of-band SNMP traffic is carried in packets, the format of which is shown
below:

1.  The packet begins with a distinctive character pattern: SYN followed by
    STX (both in ASCII). By discarding all bytes that precede this pattern,
    the receiver may eliminate inter-packet line noise.

2.  Following the synchronization pattern is a two-byte type field.  The
    purpose of the type is to allow for future extension beyond SNMP.  The
    two-byte length field is a big-endian number (i.e., the first byte is the
    high-order byte, the second byte is the low-order byte).  It indicates the
    number of bytes following in the length field.

3.  The data follows the length field.  If the data length is odd, a pad byte
    is appended.  (The value of the pad is not significant.)

4.  After the data (and pad) comes a two-byte checksum, computed using the
    standard IP algorithm.

5.  Finally, the end of the packet is marked by an END character (as defined
    by SLIP, RFC1055).

The following figure shows the ASYNC Packet framing format (before ESC
processing):

     1 Byte         1 Byte           2 Bytes               2 Bytes
  _____/\_____   _____/\_____   ________/\_________   ________/\_________
 /            \ /            \ /                   \ /                   \
+--------------+--------------+---------------------+---------------------+
|  SYN = 0x16  |  STX = 0x02  |    Type = 0x0001    |    Length = DATA    |
|              |              |        (SNMP)       |    bytes, no PAD    |
+--------------+--------------+---------------------+---------------------+

                                           2 Bytes             1 Byte
                                      ________/\_________   _____/\_____
                                     /                   \ /            \
+---------------------+  +----------+---------------------+--------------+
|    SNMP PDU         \  \          |     Checksum        |  END = 0xC0  |
|    Data and PAD     /  /          |                     |              |
+---------------------+  +----------+---------------------+--------------+


In the figure, the following definitions apply:

TYPE = 0x00, 0x01 for SNMP, 0x02 for NETFIN 2 bytes

LENGTH = is the number of bytes in the DATA field excluding the PAD, 2 bytes

PAD = byte will be added only if the DATA field has an odd length, 1 byte

CHECKSUM = includes TYPE, LENGTH, DATA, and PAD fields, 2 bytes

To enable the network management module hardware to find the END character on
incoming packets, it is necessary to prevent the occurrence of END within the
packet body.  This is done by applying the SLIP escape algorithm to the packet
before transmission.  The entire packet, from synchronization pattern to
checksum (but not the true END), is passed throughout the algorithm.  Note
that the checksum is computed on outgoing packets before the escape algorithm
is applied.

In the SLIP algorithm, any END characters (0xC0) are translated to the two-
character pair (0xDB, 0xDC).  Any ESC characters (0xDB, not the same as the
ASCII ESC character) is translated to the two-character pair (0xDB, 0xDE).

On reception, the escape algorithm must be reversed before computing the
checksum on the packet or using the type and length fields.

The out-of-band facility assumes that there is an eight-bit path.  In other
words, there should be nothing trying to generate parity or forcing the high-
order bit to some standard value.

***************************************************************************/

// ***********************************************************************
// * MMD - Setting a conditional far directive for the OS/2 environment. *
// ***********************************************************************

#ifdef __OS2__
   #define _FAR far
#else
   #define _FAR
#endif

// ***********************************************************************
// * MMD - End of change                                                 *
// ***********************************************************************

#ifndef NULL
#define   NULL  0
#endif

#ifndef TRUE
#define   TRUE  1
#endif

#ifndef FALSE
#define   FALSE 0
#endif

#ifndef  SERIAL_HPP
#define  SERIAL_HPP

#pragma pack(1)

#ifdef DEBUG
   #ifdef __NT__
      #define KernalConsolePrint(string) kdprint((string))
   #elif defined(SCO_UNIXWARE7)
      #define KernalConsolePrint(string)
   #else
      #define KernalConsolePrint(string) puts(string)
   #endif
#else
   #define KernalConsolePrint(string)
#endif

// receive state machine
typedef enum _MsRcvState
{
   MsRcvStateIdle,
   MsRcvStateNeedSTX,
   MsRcvStatePktType1,
   MsRcvStatePktType2,
   MsRcvStateESC,
   MsDataStateLgth1,
   MsDataStateLgth2,
   MsDataStateType,
   MsDataStateData,
   MsDataStatePad,
   MsDataStateCksum1,
   MsDataStateCksum2,
   MsDataStateEnd
} MsRcvState;

// send state machine
typedef enum _MsSndState
{
   SendComplete = 0,
   SendHeader,
   SendData,
   SendPad,
   SendTrailer
} MsSndState;

static const unsigned char SYN           = 0x16;
static const unsigned char STX           = 0x02;

static const unsigned short TYPE_SNMP   = 0x0001;
static const unsigned short TYPE_NETFIN = 0x0002;
static const unsigned short TYPE_I2C    = 0x0003;

// The following codes are taken from RFC1055
#define END           0xC0
#define ESC           0xDB
#define ACK           0xFD
#define ESC_END       0xDC
#define ESC_ESC       0xDE
#define ESC_ACK       0xDF    // this code was added for I2C interrupt scheme

typedef struct  _HEADER
{
   unsigned char   syn;
   unsigned char   stx;
   unsigned short  type;      // Checksum starts here and includes this
   unsigned short  length;
} HEADER;

typedef struct _TRAILER
{
   unsigned short checksum;
   unsigned short end;
} TRAILER;

typedef struct slim_s
{
   union
   {
      HEADER      header;
      unsigned char       header_buf[6];
   }u1;
   union
   {
      TRAILER     trailer;
      unsigned char       trailer_buf[4];
   }u2;
   unsigned char       escape;
   unsigned short      charCount;
   unsigned short      bufferSize;
   unsigned short      state;
   unsigned char _FAR  *buffer;
} SLIM_STATUS;


unsigned short rfc_cksum(unsigned short     type,
                         unsigned short     len,
                         unsigned char _FAR *buf,
                         unsigned short     size);

void InitializeSlimReceive (SLIM_STATUS        *slim,
                            unsigned char _FAR *buffer,
                            unsigned short     bufferSize);

void InitializeSlimSend (SLIM_STATUS        *slim,
                         unsigned char _FAR *buffer,
                         unsigned short     bufferSize);

unsigned short SetNextSlimByte (SLIM_STATUS        *slim,
                                unsigned char _FAR *ch);

unsigned short GetNextSlimByte (SLIM_STATUS        *slim,
                                unsigned char _FAR *ch);

unsigned int CheckESC(SLIM_STATUS        *slim,
                      unsigned char _FAR *ch);

void ResetSlimSend (SLIM_STATUS *slim);

extern short GetSLIMBuffer(SLIM_STATUS   *slim,
                           unsigned char ch);

#pragma pack()
#endif // SERIAL_HPP
