/*
 * Copyright (C) 2013 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#pragma once

#include <unity/util/NonCopyable.h>
#include <unity/util/DefinesPtrs.h>
#include <unity/scopes/Variant.h>
#include <unity/scopes/CannedQuery.h>
#include <string>
#include <memory>

namespace unity
{

namespace scopes
{
    class CategoryRenderer;

namespace internal
{
    class CategoryImpl;
    class CategoryRegistry;
    class SearchReplyImpl;
}

/**
   \brief A set of related results returned by a scope
   and displayed within a single pane in the Unity dash.

   To create a Category, use SearchReplyProxy::register_category.
   \see ResultItem
*/
class Category
{
public:
    /// @cond
    NONCOPYABLE(Category);
    UNITY_DEFINES_PTRS(Category);

    virtual ~Category();
    /// @endcond

    /**
     \brief Get identifier of this Category
     \return The category identifier.
    */
    std::string id() const;

    /**
     \brief Get title of this Category
     \return The category title.
    */
    std::string title() const;

    /**
     \brief Get icon of this Category
     \return The category icon.
    */
    std::string icon() const;

    /**
     \brief Query to perform when this category is expanded
     \return The expansion query or nullptr.
    */
    CannedQuery::SCPtr query() const;

    /**
     \brief Get renderer template of this Category
     \return The category renderer template.
     */
    CategoryRenderer const& renderer_template() const;

    // @cond
    VariantMap serialize() const;
    // @endcond

protected:
    /// @cond
    Category(std::string const& id, std::string const& title,
             std::string const &icon, CategoryRenderer const& renderer_template);
    Category(std::string const& id, std::string const& title,
             std::string const &icon, CannedQuery::SCPtr const& query,
             CategoryRenderer const& renderer_template);
    Category(VariantMap const& variant_map);
    /// @endcond

    friend class internal::CategoryRegistry;
    friend class internal::SearchReplyImpl;

private:
    std::unique_ptr<internal::CategoryImpl> p;
};

} // namespace scopes

} // namespace unity
