-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Characters.Latin_1;
with GNAT.Command_Line;

package body CommandLine is
   --# hide CommandLine; -- remainder is not SPARK.

   -- Switch_Delimiter is '-' on all platforms. This package is now *only* used
   -- by SPARKMake so no other tools are affected by this.
   Switch_Delimiter     : constant Character := '-';
   Switch_Arg_Separator : constant String    := "=";

   --------------------------------------------------------------------------

   procedure Setup is
   begin
      GNAT.Command_Line.Initialize_Option_Scan (Switch_Char => Switch_Delimiter);
   end Setup;

   --------------------------------------------------------------------------

   procedure Read (Switch   : out E_Strings.T;
                   Argument : out E_Strings.T;
                   Success  : out Boolean) is
      The_Switch_String : E_Strings.T;
      LC_Switch_String  : E_Strings.T;
      Arg_Found         : Boolean;
      Arg_Pos           : E_Strings.Positions;
      Char              : Character;
   begin
      -- set up default out parameters
      Switch   := E_Strings.Empty_String;
      Argument := E_Strings.Empty_String;
      Success  := False;

      -- Read the command line
      Char := GNAT.Command_Line.Getopt ("* *" & Switch_Arg_Separator (1));

      if Char /= Ada.Characters.Latin_1.NUL then

         -- Get the current switch
         The_Switch_String := E_Strings.Copy_String (Str => GNAT.Command_Line.Full_Switch);

         -- Convert to lower case
         LC_Switch_String := E_Strings.Lower_Case (E_Str => The_Switch_String);

         -- Is this a switch?
         if E_Strings.Get_Element (E_Str => The_Switch_String,
                                   Pos   => E_Strings.Positions'First) = Switch_Delimiter then

            -- Yes, does it have an argument?
            E_Strings.Find_Sub_String
              (E_Str         => LC_Switch_String,
               Search_String => Switch_Arg_Separator,
               String_Found  => Arg_Found,
               String_Start  => Arg_Pos);

            if Arg_Found then
               -- Yes it has an argument
               Success := True;
               Switch  := E_Strings.Section (E_Str     => LC_Switch_String,
                                             Start_Pos => 2,
                                             Length    => Arg_Pos - 2);

               Argument :=
                 E_Strings.Section
                 (E_Str     => The_Switch_String,
                  Start_Pos => Arg_Pos + 1,
                  Length    => E_Strings.Get_Length (E_Str => The_Switch_String) - Arg_Pos);
            else
               -- No argument
               Success := True;
               Switch  :=
                 E_Strings.Section
                 (E_Str     => LC_Switch_String,
                  Start_Pos => 2,
                  Length    => E_Strings.Get_Length (E_Str => LC_Switch_String) - 1);
            end if;

         else
            -- must be an argument
            Argument := E_Strings.Copy_String (Str => GNAT.Command_Line.Full_Switch);
            Success  := not E_Strings.Is_Empty (E_Str => Argument);
         end if;
      end if;
   end Read;

end CommandLine;
