if (typeof T === 'undefined') require('../setup');

T('sqrt', function () {

  function t(expected, n, sd, rm) {
    Decimal.precision = sd;
    Decimal.rounding = rm;
    T.assertEqual(expected, new Decimal(n).sqrt().valueOf());
  }

  Decimal.config({
    precision: 20,
    rounding: 4,
    toExpNeg: -9e15,
    toExpPos: 9e15,
    minE: -9e15,
    maxE: 9e15
  });

  t('NaN', NaN, 20, 4);
  t('2', 4, 20, 4);
  t('0.1', 0.01, 20, 4);
  t('0', 0, 20, 4);
  t('-0', -0, 20, 4);
  t('0', '0', 20, 4);
  t('-0', '-0', 20, 4);

  t('Infinity', Infinity, 20, 4);
  t('NaN', -Infinity, 20, 4);

  t('NaN', -1, 20, 4);
  t('NaN', -35.999, 20, 4);
  t('NaN', '-0.00000000000001', 20, 4);

  // Test against Math.sqrt of squared integers.
  for (var i = 0; i < 500; i++) {
    var j = Math.floor(Math.random() * Math.pow(2, Math.floor(Math.random() * 26) + 1));
    j *= j;
    t(Math.sqrt(j).toString(), j.toString(), 20, 4);
  }

  // Initial rounding tests.
  t('11', '101', 2, 0);           // 10.04...
  t('11', '111', 2, 0);           // 10.53...
  t('1000', 999000.25, 3, 0);     // 999.5

  t('10', '101', 2, 1);
  t('10', '111', 2, 1);
  t('999', 999000.25, 3, 1);

  t('11', '101', 2, 2);
  t('11', '111', 2, 2);
  t('1000', 999000.25, 3, 2);

  t('10', '101', 2, 3);
  t('10', '111', 2, 3);
  t('999', 999000.25, 3, 3);

  t('10', '101', 2, 4);
  t('11', '111', 2, 4);
  t('1000', 999000.25, 3, 4);

  t('10', '101', 2, 5);
  t('11', '111', 2, 5);
  t('999', 999000.25, 3, 5);

  t('10', '101', 2, 6);
  t('11', '111', 2, 6);
  t('1000', 999000.25, 3, 6);

  t('10', '101', 2, 7);
  t('11', '111', 2, 7);
  t('1000', 999000.25, 2, 7);

  t('10', '101', 2, 8);
  t('11', '111', 2, 8);
  t('999', 999000.25, 3, 8);

  t('3.7', '13.69', 2, 6);
  t('14.8', '219.04', 3, 6);
  t('3.578708384438441', '12.80715370085', 16, 6);
  t('49.56954417823', '2457.139710037', 13, 6);
  t('4.9808503290100979', '24.80887', 17, 6);
  t('54108953.7650348268963935650889893674872', '2927778877546676.5690784769302', 39, 6);
  t('1.0000000000001', '1.000000000000000001402', 14, 2);
  t('18.000001', '324.0000000000000000000008467', 8, 2);
  t('11.00000000000000001', '121.000000000000000000000000001944', 19, 0);
  t('2.00000000000001', '4.0000000000000000000000000000000005485', 15, 0);
  t('7.1', '49.00000000000000000000007338', 2, 0);
  t('19.00000000000001', '361.0000000000000000000000005645', 16, 2);
  t('1.0000001', '1.00000000000000000000000000000006986', 8, 0);
  t('30771277.0274560007853901874282412', '946871489900441.411712749999999999999999999999999999999999999999', 33, 6);
  t('668.093253221434649746', '446348.594999999999999999999999999999', 21, 5);
  t('0.9999999999999999999999999999999999999999', '0.9999999999999999999999999999999999999998', 40, 5);
  t('0.9999999999999999999997187097975391240474677563824', '0.9999999999999999999994374195950782480949355918890297218761543222489261', 49, 3);
  t('0.0000000000000298840312673111495324166498096924391126062701', '8.9305532478563043E-28', 45, 5);
  t('0.0000000000000000000000000000000000000000025385115498645', '6.4440408887956349899441566682289191111111111111111111E-84', 14, 5);
  t('0.0000000000000000000000000000000000000000000000000000140471526414353053458592377645343274655348630930173397', '1.97322497331782876158645511100784258994087355048761111111111111E-106', 54, 5);
  t('0.00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002598902022257132166703490111187879281161770014282895', '6.7542917212922111E-232', 52, 5);
  t('0.00999999999999999999999999999949', '0.00009999999999999999999999999999', 30, 1);

  Decimal.toExpNeg = Decimal.toExpPos = 0;

  t('2.381905579721376621997401692376763128098896247600842643458999047531881813e+2763945', '5.673474190707827242511954659916889896808177096292E+5527890', 73, 6);
  t('1.55701854242008e+46782286', '2.4243067414399499211289451653514095936017185263165775841595802781681024396317601E+93564572', 16, 4);
  t('9.9154007211093362846141108573489066595648018e+35533561334', '9.831517146017554599160038642659577233621137326296138739740339E+71067122669', 44, 0);
  t('2.233685316997e+25', '498935009536979677258304315852850344980736298107223.17', 13, 4);
  t('5.6790521336e+45232', '3.2251633136582109206181686755297199980106981532535904612E+90465', 11, 1);
  t('9.6241485526139652228166208472049625290167818011e+1987400693503696', '9.262423536278148172467483107114956962585715251496115897712401E+3974801387007393', 47, 1);
  t('4.86348548114065869236934236337303e+2212121902', '2.365349102526598437724926901530502896022219085536E+4424243805', 33, 3);
  t('8.94274851486936491099499873074488855754871805417147437494613942878964986078571e+3335853085908', '7.9972751000198231728604329670070818587227114744864529357097524020E+6671706171817', 78, 6);
  t('2.0885911135683898927279398736031752344015545504649127170644896370353379e-4', '4.36221283967684692666426416376714916311505871923000872592271965E-8', 71, 6);
  t('8.026722039667633182260408511913520451120702453123749778e+1023704129', '6.442826670208612947748038374549405718510075898758412768822080597099708051296751984704859E+2047408259', 55, 6);
  t('2.460787425752332228460163766079909e+3858273752019344', '6.05547475474079E+7716547504038688', 34, 4);
  t('4.4571175021534285e+429190', '1.986589642800241768950253744134264443767020159900398126805300834E+858381', 17, 5);
  t('9.4527e+117835031340212', '8.935356909E+235670062680425', 6, 1);
  t('3e+4904361806', '9.24E+9808723612', 1, 5);
  t('1.957204568476697755184197782894827661216187868664840919284816726085640525e+438', '3.83064972286605667223095591399211128840797098184470613547390920507003215236913837823828877362E+876', 73, 0);
  t('8.3032395380167e+34528872', '6.89437868256825998885730788226521728356623423193748251E+69057745', 14, 0);
  t('8.39e+1711992', '7.041883770880118425878239862595091826896816E+3423985', 3, 6);
  t('9.790699916656927140739512410042075323596621289368295024430694756362358e+1202', '9.585780485802596005974448193E+2405', 70, 2);
  t('9.847056990214885399626178374025660624889212963684855438227064306989910207e+269432', '9.696453136853983765292198202484493417225363497463637965E+538865', 73, 2);
  t('1.7359099425e+2747', '3.013383328415989492093650804382794630962008467500528389245351791E+5494', 11, 5);
  t('6.834069539249451103168749378990917542555757999511237679195e-204174793001024', '4.670450646729720489233321886824609831695025907588563949E-408349586002047', 58, 4);
  t('3.138284384168760310983231869196862541278664719980474337283125219910900471e-34848021177070', '9.848828875917495153223800564466445801240606621764417514110475E-69696042354140', 73, 0);
  t('4.00929156084328544059400729461e+7547814714', '1.6074418819849188E+15095629429', 30, 5);
  t('9.385699983e+163', '8.80913641710599005691978950691155189458746110141164551755555797229303947815057725607373E+327', 10, 5);
  t('2.064258493001581567732632230025244966622767522065920193424751662709921e+4395', '4.2611631259291605782486841695067759569661374268622981237845E+8790', 70, 0);
  t('2.2215318528416409230049e+218258', '4.93520377319001414151248980920968837885642082455528005489000815E+436516', 25, 3);
  t('5.1330527926591638410946628719813202245977800293e+46304917529689', '2.6348230972226040854267304152051578667034961411058694242668334095640189241199450266226838E+92609835059379', 48, 0);
  t('2.3253579055706672e-196', '5.407289389E-392', 18, 6);
  t('2.503e+176755402701774', '6.2667203408185268363604194770009720671739738664656179598783769617970284596514793072E+353510805403548', 4, 5);
  t('9.6369204975569156895116858673802898447044593766985526107265769189051484968e+360924246678463', '9.287023667623263145602226199798895429249583886386354524632320596574924425720535087689428938434E+721848493356927', 74, 3);
  t('4.65424913912969424744230257268e+92793524825107', '2.16620350490895E+185587049650215', 30, 3);
  t('6.01309285636782899752364197007684920017398969437949e-1790', '3.61572856993018165706134046141609828036662065978484170009821375392882703132369124228069E-3579', 51, 1);
  t('9.509773153923080543680696577222501710491282865e+412437011050', '9.04357854390761345545547419189409265295580849E+824874022101', 46, 0);
  t('1.257819302229712520962174337277372176639810260661e-44311330', '1.582109397061640889607372828919871087547023353618955613E-88622660', 49, 1);
  t('5.60801706098533725826615935476591733e+4703349458', '3.144985535630261990939138487693918541845307202896600571E+9406698917', 36, 6);
  t('4.520612769066843919448985334359993005870337091660370281988247839555503e+33279', '2.043593980785019831258664920277034009002215803890866800625E+66559', 70, 5);
  t('3.9031901238112199422244215131891752614639270293050848390039e+24964497874732', '1.5234893142617446461800063194E+49928995749465', 59, 1);
  t('1.67399696850587052605991771251848e+43519462757', '2.802265850566844477905639463307E+87038925514', 33, 2);
  t('9.5408249368370749687627590642478e-80750', '9.1027340475372175567247E-161499', 32, 4);
  t('2.4283599683e+171849', '5.896932135288822595493869E+343698', 11, 0);
  t('2.1e+498604095779994', '4.2961949068353E+997208191559988', 2, 0);
  t('2.20453501e-357801904289', '4.859974646744411255356204868440980E-715603808578', 9, 1);
  t('5.8147365156793540374533779611537425212e+496502126799', '3.3811160746774874681847190789838042E+993004253599', 38, 4);
  t('9.6614424956e+33826', '9.3343471095787978641040077990615E+67653', 12, 1);
  t('6.840414600748481921789126213883996657219910998072878638534975744655859056193334e+365397755685166', '4.6791271910133013331845019336634770531377970763582548051313437429709245928538732275037320149E+730795511370333', 79, 1);
  t('2.4239992207971581775833393874986290518e+2157096700', '5.8757722224252300019927334806719E+4314193400', 39, 2);
  t('8.2220388454970543626652913918343189618675218734756075784049153881385219e+3', '67601922.776862534581068550153256371638256888653847213492979247915305924432354685660', 71, 6);
  t('1.64244486917406282584413656933972205192398555455095912262410121870819135e+9340508786387', '2.6976251482762043514129832714354246217401306620466914174989739297066373978821E+18681017572774', 72, 5);
  t('9.79090692715782911695847092370781210596001172136e+3115677438', '9.586185845626716371804688230667770283863635486094832181975090257E+6231354877', 48, 5);
  t('1.944278705796582640753477900250165813717781441628885212874406345152815184e+2116000927238534', '3.78021968581403435601384837583880755915455510843E+4232001854477068', 73, 5);
  t('4.8812606303344129781885860648384540427484823894462977105e+44750108717', '2.3826705341252710709297819787306105797004429874259653221553023657650287E+89500217435', 56, 3);
  t('1.93086613186193490703915304629355648384e+4', '372824401.9171471', 39, 5);
  t('3.1595298775034276812747891834935798726890112101053025250708e+161894746', '9.982629046836824729046499174522512414789196378588531892708638648812537073974580E+323789492', 59, 1);
  t('1.390343871134044306144904825487640471083e+45251442', '1.93305608E+90502884', 40, 0);
  t('2.96906709876922593571912431731393589808133769511469663747755797342953511989756e-481589284892170', '8.8153594369939084379355385E-963178569784340', 78, 4);
  t('5.06340826804083097846774064356525633462189965902266516580957088231376938e+37447664404211', '2.563810328886424765192984306061855294460082173029294221865432386446793287430868E+74895328808423', 72, 6);
  t('5.972915579625786738e+332753', '3.567572052133644795106468790474408870787279617581832818761015389739299185393099191398142701264032348E+665507', 19, 0);
  t('3.13014348207873691395112227e+295812860174', '9.7977982184E+591625720348', 27, 5);
  t('1.612e+29382080784098', '2.59549542566482274489688514562643282E+58764161568196', 4, 0);
  t('2.21e+957', '4.88485916656072684684336288223350318510382769049147063964695662933003E+1914', 3, 1);
  t('8.62822128158912829157811086596256560701532576e-8740918', '7.44462024840675394866139306388888378538969579877716771591568819584308250214770680342900129457E-17481835', 45, 0);
  t('1.608973391338221287e+497446804197772', '2.588795374034416984035765330740978157904525653087970199E+994893608395544', 19, 4);
  t('8e+3075307718', '6.892258379923950714603548436010577093809697037965095264743890886686628674142064667547853365035865E+6150615437', 1, 3);
  t('1.7653932336326458078973623890876147360894729071996321e+26135', '3.116613269355929545695883758738082553843696E+52270', 53, 6);
  t('9.256926799025012329e+345215572125607', '8.5690693762507461E+690431144251215', 20, 3);
  t('5.03544006612282891778e+46195820165472', '2.53556566595150796637413433477866041357009510845074403852348490716635E+92391640330945', 21, 1);
  t('2e-48295616097338', '4.896457515808248945640496172912496043632630141762245831365224969393680585716718487583992616E-96591232194676', 1, 5);
  t('2.48878965963768168184605548022681e-363573591781789', '6.19407396991944743243101974561954532539988692605208840104874752842254662E-727147183563578', 33, 6);
  t('5.3851648071345040312507104915e+740361058', '2.9E+1480722117', 29, 3);
  t('8.3413437402428516353e+40', '6.95780153928886055351192302500612019875561321E+81', 20, 5);
  t('2.79630657985938724890492968896027809909551e+913991292804', '7.81933048856490367778167731312448221121779492491922341813551588784321465352788E+1827982585608', 42, 6);
  t('3.068981347405821994329370885223153147810167995e-134', '9.4186465107248546707630495330709378212295744168927053819144383155698062121265279905897752271E-268', 46, 5);
  t('6.0271249500033e-10781347401', '3.6326235162953181809577645353279210E-21562694801', 14, 1);
  t('3.199125028856e+25795', '1.023440095025536568635708921517360961793E+51591', 13, 6);
  t('1.8666114157357738546399e+3', '3484238.1773551099774', 23, 0);
  t('5.0973657635624059747024576803245965274249298527568449922667e+1692063437920', '2.598313772753815009002620426366701559965597345321262947368693172122554312019668885696226081E+3384126875841', 59, 1);
  t('2.013328509206219057784323487222458501070404001508e-3238', '4.05349168598253649731512E-6476', 49, 2);
  t('2.69875493659779856084765987819363578742030895189293632989705922592411229915e+4020061289691140', '7.28327820781098773E+8040122579382280', 75, 3);
  t('1.6e+10', '2.56E+20', 13, 5);
  t('2.3483542723023e+2733546344672534', '5.51476778824035712778945842083575307125895176285682237794763477393558870924644509568829084003207E+5467092689345068', 14, 6);
  t('1.69575e+13', '287553939655483170302812649.82525337373758443', 6, 0);
  t('1.32333278406215884733018366812891959496232303384e+8285944289080', '1.75120965737370433697922956385416755835455646442365E+16571888578160', 48, 3);
  t('8.619857317385143165541404858165098391611310078189659021366304034052302190198e+423674679393446', '7.4301940172078196756317577124190795116444648589485897991299472748929474290863699960E+847349358786893', 76, 6);
  t('1.5321135075443986e+3740', '2.3473718E+7480', 17, 3);
  t('1.72340673782204014190446948179592390656068185428e+1299869167652256', '2.9701307839704062069609474255783085619215767002151887992880369960243164553E+2599738335304512', 48, 6);
  t('2.6489093408250026638898665191805392071832816e+37475657', '7.01672069590995012448612570344661497015681231635520477773E+74951314', 44, 0);
  t('7.552132321848183098292052759964393e+3', '57034702.6067040290231940776601570114357688435285519762590782006101866084636618667', 34, 1);
  t('2.501915642047260445562789772566841521129383e+346367499153', '6.2595818799207554600051E+692734998306', 43, 2);
  t('3.46410161513775458705489268301e+23505129984', '1.20E+47010259969', 30, 5);
  t('5.5156817867298553573934377553326519175269141671462400489745247227004977385606e+2535798792', '3.0422745572463449598911698968741967861068325178395628338533947208549426458197501E+5071597585', 77, 0);
  t('7e+413483559541878', '4.9446671019E+826967119083757', 2, 1);
  t('2.3540150674351154530407163580243074454938968196e+2160300', '5.54138693771155115387408E+4320600', 47, 1);
  t('2.737454074105672e+2878546', '7.49365480783773944781718916811056453166482639687697E+5757092', 16, 5);
  t('7.454070919313777677e+2341606971625', '5.556317327015934668013876961309779446079811E+4683213943251', 19, 5);
  t('1.1573376026021382616589933753247694e+358389221', '1.33943032639686490800430635561822376604163268393006414770600700814939E+716778442', 35, 1);
  t('1.869736722457198060474037674032156648711972906521e+32904235983', '3.49591541130498529E+65808471966', 49, 3);
  t('7.47395e+36118', '5.5859818323020679E+72237', 6, 0);
  t('2.8316440369487011e+3', '8018207.9519871366038039483866918309675753055797380955099583126482331855373', 17, 0);
  t('9.7472668692913041841429e+1309536008', '9.50092114211839024068782916975E+2619072017', 23, 4);
  t('8.114992298209531e-2596906057289', '6.58531E-5193812114577', 16, 4);
  t('6.703595e+401', '4.49381922866E+803', 7, 6);
  t('9.7608119539308818711822517856627039556134970352903e+1249', '9.527345E+2499', 50, 0);
  t('3.0066592756745816541757861126198711210412117e+139049925673', '9.04E+278099851346', 44, 4);
  t('1.896015165815e+347585', '3.594873509E+695170', 13, 5);
  t('2e+1005', '7.1236555794326143979309006340E+2010', 1, 3);
  t('4.4721359549995793928183473374625524708812367e-2934', '2E-5867', 44, 6);
  t('9.81380264259950340885274731932875646303790396039793934033661431416500117441683e+4610414097817', '9.63107223078929964397336E+9220828195635', 78, 0);
  t('8.9376309368079848924360696e+36390757', '7.98812467625871776375673355260343E+72781515', 26, 2);
  t('5.026652112874930197997606836236916458650894248379748587283939631792571414e+2570662179', '2.526723146387E+5141324359', 73, 3);
  t('8.49293409e+2918903274734', '7.212992952916182126166161894531743826E+5837806549469', 9, 6);
  t('3.7717367e+4831878', '1.4225997558149815433987209975829468787464264438709401098671706038917792294221101238217259236638882E+9663757', 8, 6);
  t('5.65256722208745359096703556418756741838091330344004243880813554127204475354963e-196', '3.19515162002174718875005691558428708625371358895103E-391', 78, 3);
  t('6.63236207127e+23317', '4.39882266444868312494846698153144297198308093870920987520682921E+46635', 12, 1);
  t('1.0299091215176223197e+255', '1.06071279858520053783126987482987178921249572000906994047388785433323881546006E+510', 20, 2);
  t('5.040737517733564457496704009949621179767208845752392579636770019086715865e+1', '2540.90347226867370536208924268972271855242897', 73, 3);
  t('7.2998671170787362309999885642096413366997803441454046e-3', '0.000053288059927007419736192864488572091336455240976257828090775274140470718273556545610819', 53, 1);
  t('5.11117005049595857693e+9', '26124059285086859749.111136068760', 21, 5);
  t('1.89935540125106114644973371135677739395218e+49685', '3.607550940261579490044139098611195000453932161335896883380180003006836843018711910962E+99370', 42, 0);
  t('6.18443052847326250790598455586e-2615403185', '3.82471809615120769875272018837130277169788007341654336023385958906492891E-5230806369', 31, 3);
  t('3.426242491487763270227e+18', '1.1739137610476275565237282E+37', 22, 4);
  t('2.96275501530970122781213e-3678', '8.7779172807427879568E-7356', 24, 5);
  t('2.4428719e+4859', '5.9676230838955962140555831660429425584144796031304001629260382152102420202408466740096475145421788E+9718', 9, 0);
  t('1.02832795192e+442784', '1.05745837668529806545913238826109255689541898043645621468495695830204838516639758771694008345342674E+885568', 12, 2);
  t('6.514808353931537716266282122142219803851312224850046724512e-2045', '4.2442727888456152E-4089', 58, 0);
  t('2.550511488760636812199575483724868e-1447172750541', '6.5051088543E-2894345501082', 34, 5);
  t('2.4730609367420230671388e+12', '6116030396839332618454241.9526022655923587418704940810349550246004', 23, 5);
  t('3.036323149912413372105696029126241615660685857583888271128260449310702453141608e-1349985', '9.21925827069403988819584961758405864669074310023419843237E-2699970', 79, 2);
  t('2.6874987079843743011487811643134819286318198642391692062301e-91', '7.222649305417681173051748752199025762E-182', 59, 5);
  t('8.8814616618312878350673062471290132392165769044e+163440250', '7.888036125057898099450298304157513376364713379776394983E+326880501', 48, 6);
  t('8.003905394943370898574902868328421705723747639106911357526019275287081172297e+30146897', '6.4062501571203598084182532613957872182544E+60293795', 76, 0);
  t('6.6869e+3882', '4.47141713850001280335127436138703209048665710727894216146638123448970973413333877E+7765', 5, 0);
  t('9.46715778767100012677112937995161815357113020444e-327073528', '8.962707657665966551992881052251375867047729017714166968495362566E-654147055', 48, 6);
  t('6.303431321113571e+275056324656282', '3.97332464199955790588897588438551310212027677784886403329179755857444550904597263134605E+550112649312565', 18, 4);
  t('6.576174355328564536662125352346e+21602351976988', '4.32460691516810613850278067385309182411399820938966096E+43204703953977', 31, 2);
  t('2.07630346262337137491871083e-3090618529044467', '4.311036068901801732099430365860481210539754465419778810616524065681770638162008279484055169E-6181237058088934', 27, 2);
  t('4.2148890965989444057133600997628746172165814692968051092e-8712289445', '1.7765290096628665705861617573159354925958E-17424578889', 56, 6);
  t('2.6474912249241999225341670266712257663525163332933359876990081330726543e+162573', '7.0092097860506405451155195833366486325362949236190817489245792816903602999857931117317190E+325146', 71, 0);
  t('9.12874162316595732639551387628400033604695187619e-3', '0.00008333392362252263723524459', 48, 4);
  t('2.7235179981923310319738235894879139608807e+1033760346', '7.4175502864775620583462352475547105037730945778584608279E+2067520692', 41, 5);
  t('7.099057482524069554788222715721382261331093541531504796459e-5', '5.0396617140180980112290131689304401510152E-9', 58, 5);
  t('9.10809918720977210707253937500924e+27380', '8.295747080405131128480935E+54761', 33, 6);
  t('1.285681486112942019135229306830687883569187630472e+3834782', '1.652976883733583122E+7669564', 49, 4);
  t('9.568088e+2018', '9.15482993151972E+4037', 7, 0);
  t('1.987355257754938992377160858509897774952049248e+15', '3.9495809205262E+30', 46, 2);
  t('1.9768903173746098651102397889231284119168749816365290144722209101587590107964e+46924932068', '3.908095326929485719118550903578424879636511190787119806290E+93849864136', 77, 4);
  t('6.85223487409204247121078477321652642e+13', '4695312276972318913824783267.23915150099092256', 36, 2);
  t('1.89753938328297503876364212659e+4', '360065571.110993325', 30, 2);
  t('1.73793786659448789397819131927279578513780780522093623215e-26159', '3.020428028143E-52318', 57, 6);
  t('8.6448435822785381294005301901316419084244682800468991364630579854249684e+199772394975', '7.473332056206242784465850133264347507502598072E+399544789951', 71, 3);
  t('5.152189634588319893862857733009123865557155838508261990197856939965755e-21896842502066', '2.6545058030759325273618511956767499403152E-43793685004131', 70, 4);
  t('6.363328661925045799106506614064522175879e+45790995', '4.04919516596767938142398731881702747229058975396741269844545638111172628708619095858703698307014719E+91581991', 41, 5);
  t('2.809603385469e+3452', '7.8938711836403788689485387722739902228108362611929174E+6904', 13, 6);
  t('8.387292474579902077507212611207840585e-13228993663', '7.03466750541446573370027044489E-26457987325', 38, 4);
  t('1.8280039989118977381747692279217097e+2570948', '3.341598620037889427132828227624777069946213265099768868E+5141896', 35, 5);
  t('4.242077070790137478716517628874194271194060827692514184105e+389854', '1.7995217874523433061846222162668474214071163108613415E+779709', 58, 2);
  t('9.29961911573539491036147371e-27015695', '8.64829156977511683564833506322917189811629558525289718715023664251559870029893040656186008376212E-54031389', 27, 5);
  t('2.9190392183495259081103419481759400578499514075079378347e+31', '8.52078995826261119108447319667254438902793796629435580E+62', 57, 3);
  t('8.937227578505540103845413636584692573241532357433996111863e+1790938', '7.987403679E+3581877', 58, 1);
  t('3.070283187058663486241563846238e+2571', '9.426638848735104E+5142', 31, 6);
  t('2.8538810158098743e+3473261643654553', '8.1446368524E+6946523287309106', 18, 2);
  t('2.008039e+36729025', '4.032219417259540272204478866331908221389931400048743477485727321671916860580259161402773798107E+73458050', 7, 6);
  t('7.51692279236539994967372050974313334001215192497833104202e-2440875511683227', '5.6504128266382441683929602449473591619319549900002408802881702717481E-4881751023366453', 57, 2);
  t('3.0435123169140879772323396228229690669146412039176588151e+8', '92629672232077598.90063863301233712326343840597041607655456994461', 56, 6);
  t('2.236067977499789696409173668731276235440618359611525724e+702888559122', '5E+1405777118244', 55, 6);
  t('5.97990172243658198399814740059437396277732066046e+370', '3.575922461E+741', 49, 4);
  t('2.82561653740855466419452851014456248628635025976235076e-4280625271453', '7.984108816476710E-8561250542906', 54, 0);
  t('9.48683298050513799599668063329815560115867e-434737', '9E-869473', 42, 5);
  t('8.882e-4097945006776965', '7.88869662270039764413484643614524697066192137552471897225988700736686713038E-8195890013553929', 4, 5);
  t('3.67889e+14891', '1.35342176496813904540880394246590582429142627731764911712414478149020053276032E+29783', 6, 0);
  t('8.473e+456', '7.179707176136375341401913486978796960524639825994932608331410951477613527254629E+913', 4, 6);
  t('6.061122e+156', '3.67371987435196618057562397883355060133829216965154000472540684500E+313', 8, 0);
  t('1.2162592664785351757282744e-47741523717344', '1.4792866032949044394019E-95483047434688', 27, 5);
  t('1.0226100780319713142530417271134325435661232545102289895560582934036408017e-3070736', '1.0457313716925544603251531895365106855371052455015295127376768840262251474879407584E-6141472', 74, 5);
  t('6.40378773473547483807376041857055923862407e-289422539068310', '4.100849735154850424998568954883333023944579871020634818881424885849663612599045128003798737145657E-578845078136619', 42, 0);
  t('5.1725935520949720972771088800926908173244154e+2824962274', '2.675572405517448082E+5649924549', 44, 2);
  t('6.864734640020072049718705304013039739459327696999280834990366263684919488693e+28', '4.712458167789150819E+57', 76, 3);
  t('9.4403813337678765859e+2275528598282', '8.9120799726952952463715474436836438839285693903355297048145204533165099122605783431631909389375E+4551057196565', 20, 2);
  t('1.401091686602968166559525283171170204216e+4918493', '1.9630579142679499665419988066E+9836986', 40, 1);
  t('5.76860337023413964786931e+2461399906291', '3.327678484307667442355379899894139310335808838737692146122103503686495579534E+4922799812583', 24, 4);
  t('4.930871521981228238e+3437032097726610', '2.43134939662854741919117399694122150170471210277624540E+6874064195453221', 19, 5);
  t('1.0245391862903256753964503966393139032236946080277416684518e-38', '1.0496805442444426583754480311188446871883996972187399715782318606844535332564E-76', 59, 1);
  t('1.7128627267646388333852869565663674027391841238236919212997718917108311794e-4002', '2.9338987207395937897002373949538359655732613253567763605075447290049969107E-8004', 74, 1);
  t('2.8264310169258223109845135255342978316223511737459003e-16767859699', '7.988712293440338047E-33535719398', 53, 1);
  t('6.903495420085316148538509004607763967209023254e+44768102', '4.76582490151389356813825520590783598202999337271191949072678494086332063317360130E+89536205', 47, 6);
  t('2.95564580436384800253055711281369395501536452408051186314e-203298238353891', '8.73584212085361806007883039348577463893244695975888298999502229344131381589611093355072253081E-406596476707782', 57, 5);
  t('6.649283812592150138142528590709054870664998836855393073088808237856384012e+16918', '4.42129752204E+33837', 73, 1);
  t('3.1418363802846828888954674287282088166631670326029874978921587241662126193072e-10', '9.8711358404803585142715995165965562371006E-20', 77, 0);
  t('9.0168891708498077340870422538771763991860996900736805340459998564777522117427e+1646179540063', '8.1304290319388533208865555457549157885068286554112709492694227672649E+3292359080127', 77, 0);
  t('8.977858829152028877e+40', '8.0601949156183038822620916355105579058051056436917994500749067325934920E+81', 19, 2);
  t('4.2017305790590619e+646808528', '1.7654539859E+1293617057', 17, 3);
  t('5.5524814508191389246383339131600308902340023710576826212458e+2497373294220', '3.08300502616906098687255826013116108848672731202638827475459671548604743541183982705838E+4994746588441', 59, 4);
  t('1.67271971876579712062058760927597244127329496726295947385574838534557761275087e+4084885214', '2.79799125754792741188580781151558042045245397938161747962330378605256347960E+8169770428', 78, 3);
  t('3.085527497069488331267423581241150396840258e+42246909', '9.520479935171901322697389268713897072458655100066709408989906359E+84493818', 43, 3);
  t('3e+158107000193', '9.6977619149278219422781E+316214000386', 1, 3);
  t('2.90723345116412399903677294614733205e+380', '8.4520063395676629612501328633225132880746874537228627270266E+760', 36, 1);
  t('9.775891681481346189754834e-710', '9.5568058168056182185442353049791687180E-1419', 26, 4);
  t('4.6646957424189037236e-233070856320311', '2.17593863693410473953572E-466141712640621', 20, 5);
  t('8.53425633807185478645377489395742278102449403e+2105124743', '7.28335312439196245774226929758991441342086268907461367722377958885353595551884935516785591E+4210249487', 46, 4);
  t('2.8161913e+30949787971256', '7.9309336869113277937283E+61899575942512', 8, 5);
  t('6.6443937050711499376864490195641899272283176042527e+261607', '4.4147967707989123421155030572424236370263103192892052949087277363710E+523215', 50, 6);
  t('3.887130387e+4462928527512', '1.5109782641945858394E+8925857055025', 10, 4);
  t('6.9295847389150837205027607357674e-337585815724', '4.8019144653804829012013755178407300496433787057977746765392593580534900280159767925073E-675171631447', 32, 0);
  t('1.67762269562204e+2724901494014124', '2.81441790886617380509520155082877145938117037134E+5449802988028248', 15, 3);
  t('1.378806846317145211527751556938e+3615', '1.9011083194510316937617450165824777298618984876923310846235512081728267868E+7230', 31, 4);
  t('7.2282380669809344745123929711647933739752741869853493e+129', '5.2247425552952276174806603769135304185149515310446734318136860424E+259', 53, 3);
  t('5.112514728527103756119083043541744424478486522645230981812126003154941892e+920836812802990', '2.6137806849406565416963702733639313080984002961475E+1841673625605981', 74, 5);
  t('3.15240118526970354837192462164291327592722e+12762', '9.93763323288983179604546E+25524', 42, 4);
  t('3.49559682514708909727859773952360298460003998789958980573672291458393004e+132', '1.22191971639784089879E+265', 72, 1);
  t('7.4813100457072356252346588698556564868296818103801416175589911024114791569995e+284130830840784', '5.597E+568261661681569', 77, 0);
  t('2e-3768574628', '7.2713214641933444306829997615377506867292707248931E-7537149256', 1, 3);
  t('2.902077e+1588760762', '8.422051691577715414082824473132488156131471218330734604226191089990683764449E+3177521524', 7, 1);
  t('2.18935e-192883592280', '4.79327125880129020E-385767184560', 6, 1);
  t('4.48463019222796485951602654336860567938893190030116242172e+473496', '2.011190796104263304745319026570265454712539877236299939E+946993', 57, 4);
  t('2.546525295e+29021565225819', '6.4847910760149283558212070695669048254074329962785677108576986411961086444961000E+58043130451638', 10, 6);
  t('1.51185657328663290102911156905e+18', '2.285710298190E+36', 30, 5);
  t('2.6598141458335522206769057345931e-1022221889', '7.074611290376269E-2044443778', 32, 6);
  t('2.1400246921846e+473622', '4.5797056831596035941207627491245833008597417378183068825736455247163834770170E+947244', 14, 0);
  t('2.3774545577213805010498159234873880660721178e+422716', '5.65229017403016496862404612990739165210984813062645469906426138589591E+845432', 44, 1);
  t('2.203089e+335', '4.85360341581551E+670', 7, 3);
  t('4.910625878493595047710096578187419155296332340361665271615e+158478680481', '2.411424651853099211331932586857671419333629451402446313055708226355472325441681706664245120160E+316957360963', 58, 0);
  t('2.14658304513912472592069256949618e+0', '4.607818769678757580622617058038482756337638794621523984131268073532516530956591238950305713306', 33, 4);
  t('2.755609393656315990095837385538e-32204', '7.59338313040692946359911961843531565497832081882557075265876961832563840683131278114866271E-64408', 31, 3);
  t('6.0796177733655821078861178138872182814062298888138e+13', '3696175227022267849032414092.016773404702311234937146358672775619', 51, 0);
  t('9.322011536195057384357328326782263568235613559128e-24931530574', '8.68998990809537336703600486E-49863061147', 49, 0);
  t('2.71986591529447641250156426872327014101855e+430301407', '7.39767059718065993935568543488443748452082512031990E+860602814', 43, 4);
  t('9.748823047418070634441154844604945179095866321330356660832934187282699223e-3702188646680283', '9.5039550809869757481802467684142816071629587849671095519782938514286705E-7404377293360565', 73, 0);
  t('1.878382903912616955109258360633381976169159358491282086149580414136009e+120243', '3.52832233371119558076316851E+240486', 70, 0);
  t('4.621003503308429392531555236351223553159256034625873954713601722906751e+22257918209422', '2.135367337758877761572808626061137813597705703296648884770002410767354094917580791680252657855994460E+44515836418845', 71, 1);
  t('8.09675556793333478408703745283365e+2697370827636895', '6.55574507268594587077340408367234333953545565465682867336894127238643198993462187195128E+5394741655273791', 33, 2);
  t('2.02207593759537562422226807353364843462858518254559656e-28012230172050', '4.08879109740221741578687735602952948096051E-56024460344100', 54, 1);
  t('5.3731000956898735111861968797250897647473152496311640907459600751218763861e+454513139585766', '2.88702046383025278824610014957155384890357084971884613239765424667693727485508799293067464689E+909026279171533', 74, 1);
  t('7.6531546e-3210839', '5.85707760360226795537203615034919E-6421677', 8, 3);
  t('5.1545158401e-849458529874', '2.6569033546401399505222840937237E-1698917059747', 11, 3);
  t('1.362796e+820366697', '1.8572130578086568903670432137573555816173399866229650898073903609534634685E+1640733394', 7, 3);
  t('1.803028018886897386311411e+392', '3.2509100368912099980074744E+784', 27, 5);
  t('3.194063e+59854163', '1.020203979681842516145028680059578232774062730558684033505119017106807172711E+119708327', 7, 4);
  t('4.261774950512222945050604e+766620941569952', '1.816272572881346033232615148233362658915452774043972856579675E+1533241883139905', 25, 0);
  t('6.4982834054101674517e-6690629769341', '4.222768721702916271457625051478794776571420E-13381259538681', 21, 3);
  t('4.63890650784339381668800704976722120910871991068098e+3684414202057', '2.1519453588511791178106435314081634243235207843258897805541E+7368828404115', 51, 2);
  t('2.9151839322904732383444730694225e+30759', '8.498297359084546458280003048168914699436364769023695844300652045190667728E+61518', 32, 3);
  t('2.766532719204307910633754595392000551442511483883250429903676087804405565590643e+455779478', '7.65370328642798200007936195028468256694001674934111932023715848409184276391931286002453021414093E+911558956', 79, 2);
  t('7.15809196006477e+33292581643', '5.1238280508743809055025574209743282729559196766140070189951812797362009162E+66585163287', 15, 0);
  t('3.86118619398318356259490388719113598185201634813542416e+406721231', '1.490875882460634284411863820052569490029137331918395153873776527023749083262931386830E+813442463', 54, 4);
  t('9.689731193707917826178936968367758565636403e+40068362191', '9.389089060631627013431474273256976055318787324073601735849070341827498807062287216165E+80136724383', 44, 3);
  t('2.380529012851904667362202911642711024107714e+198681597152426', '5.666918381029663696953608649066197933231308460645707103242820405560172691798118435135524461103430183E+397363194304852', 45, 5);
  t('2.54746401587652656029608220458044130513888776543718542834122918527624080395488e+32532', '6.489572912185759966860293134205566227925787911969794771866E+65064', 78, 2);
  t('6.92698673696026591387308445e+107400231637266', '4.79831452540234321937856594164763711E+214800463274533', 27, 4);
  t('2e-3687675859352948', '2.87850545774362267882888183908800107382947760934094690201972229957797333658228601684365936E-7375351718705896', 1, 6);
  t('1.618027041879108501297810734567701e+50804486666', '2.6180115082520583359185144823153085187283124373602323397858607127478347167082796349429269911729E+101608973332', 34, 2);
  t('1.468859842402168586360692550046800918056478722e-36', '2.15754923662172353660E-72', 46, 6);
  t('8.49215738290834886674655744364e+3929100150329', '7.21167370160847769931713219E+7858200300659', 30, 3);
  t('3.080752255068078689981071323748588537107e-36384305560', '9.49103445710705218035789748393813719855362668161846E-72768611120', 40, 0);
  t('1.173890248587e+32564710208321', '1.3780183157276814234744165885721310974270604540920153529405097095176115665588872141E+65129420416642', 13, 4);
  t('3.3574798416124266047747707108073142426282137e-218933383480676', '1.1272670886833805240621800360101550017149869005946946591265232560766193937035542597E-437866766961351', 44, 0);
  t('2.23944945243244996980913928295535257356722976e+3460351267', '5.01513385E+6920702534', 45, 0);
  t('9.0144388862969379134e-45686027', '8.1260108434782378341695197333453540906E-91372053', 20, 0);
  t('3.0154321001794005197296683806041624213563546270536367710423e+100086943546', '9.092830750792350171931235260936892651832323052358883721226795146110156733755114783658E+200173887092', 59, 6);
  t('2.44369944036062486685492557249769572445990273975135928852505135432044491079041e+3973481', '5.9716669548188311704969626601027296830809246015644883490860409903708929680E+7946962', 78, 3);
  t('9.9766657347831312895713804962380972879872133625410810249905126198373397e+4902005181527', '9.9533859183595836960393341304494815647224E+9804010363055', 71, 2);
  t('1.55e+26246', '2.401659339220326238148181906366427704224780832955898700E+52492', 4, 6);
  t('2.34305570615805937922628235012415265e-7', '5.4899100421598422967909002186123690839377971716E-14', 37, 0);
  t('9.486832980505137995996680633298155601159e+996590685', '9E+1993181371', 40, 0);
  t('2.2472831e+47363932428322', '5.05028103859772651487706403994190016339078351595454246848636742496802495048742202447079630739E+94727864856644', 8, 0);
  t('3.09835648876358033e+47479603614', '9.59981293146338228219992128891802378333E+94959207228', 18, 5);
  t('8.55135371363729866747197e+16028', '7.3125650335738419022159017437377735662828E+32057', 24, 3);
  t('7.541423725038898641083750859464767137139684e+12', '56873071800579577894480029.9913916338494896187006254131382792065617290802105695797397', 43, 5);
  t('6.8002746166354257e+11', '462437348616560861444716.469443110926995223677083942242611169677743650813236751337419207369559128520', 17, 4);
  t('2.5652177e-244584256384357', '6.5803421671668747304683336293E-489168512768714', 8, 3);
  t('2.23793617809742898841363911292888853966977964817766258759532718005259729e+41087371', '5.00835833723732739992314400670207943302403206440405252937804763314822035527687053E+82174742', 72, 3);
  t('5.23407164867045527149076121386206081046e-3542587', '2.739550602341585775997338219789670446820E-7085173', 39, 0);
  t('3.0742131314253906754227821858e+96706326503', '9.450786377428306361560309171177304320240134953027243301587893245054872736802088043071493639E+193412653006', 29, 1);
  t('9.6526054e+6', '93172790036170.613444694347666418780549753491881062334620953897827553473780799108203', 8, 0);
  t('2.764550263766756063892234262466594517907118311533593420860995828119465404401809e-431912159', '7.6427381608928405257681569294347818369102356176996211973E-863824318', 79, 3);
  t('4.726185048245726940640138085401458332003103e-2387', '2.2336825110261464289548790065115800838432265E-4773', 44, 0);
  t('2.06700039e+27664', '4.2724906128719667659691356003183719988200264E+55328', 10, 4);
  t('3.0394538316874649464598276081160724e+106145874', '9.23827959495961249186367387478693953484941E+212291748', 37, 4);
  t('3.006875499403192963673188892347469682563777463469587792654e+20786', '9.041300268911201088838782013152679784074371858101273E+41572', 58, 1);
  t('2.85909217357538160219868784409618103905259254542127e-30883978641744', '8.174408057E-61767957283488', 51, 1);
  t('3.3260210094135369128622169846679339403575787017793503761274203309290260436e+3815171969334', '1.10624157550602430015244848049439711918773950918589248425841376113701453944821290918195E+7630343938669', 74, 3);
  t('2.2219659043755149394630379e+39005143', '4.9371324802073E+78010286', 26, 5);
  t('9.5438933355313647910084093295703513e+363658650', '9.10859E+727317301', 35, 3);
  t('3.00666e+472622870', '9.04E+945245740', 7, 2);
  t('2.837780823108084927845098354231270362354547981102e+3799336425332871', '8.053E+7598672850665742', 49, 6);
  t('2.63098328861837901379246055794913493258563428302007e+18474', '6.922073064989180646258163061425595248339645318548644890836321470068254933488292945284735322631E+36948', 51, 6);
  t('2.764594001353016120387309e+1656763472', '7.64297999231708049848271603772247952906531812E+3313526944', 25, 0);
  t('4.99358959259322115994443893492663456542067e+21163752', '2.49359370192553324844573464720642409E+42327505', 42, 3);
  t('2.027895093136415935e-33', '4.11235850876675305992625132144861230486080916955691E-66', 19, 3);
  t('2.9545024474453888695924718967207697287659510106764672439630841479565241968e+3025138', '8.729084711960792819353415468437350816065576927664405E+6050276', 74, 5);
  t('5.274723829579e+147', '2.782271147832686219E+295', 13, 4);
  t('2.8438354740227241040111232893777368e+5046190127387', '8.08740020331005190220391258674305873512363906032235E+10092380254774', 35, 4);
  t('4.7022502775686244544842e+785', '2.21111576728942057265374867434598092629448251165002373458069847941389903130518761599813923261736272E+1571', 23, 0);
  t('1.9179842370578362891185422e+3826111', '3.6786635336023303507152224116804583704219412283495446611E+7652222', 26, 3);
  t('1.294105809149064963260379265e+33012', '1.6747098452733561507694771361540079238390082400869150000987128223006851969007181E+66024', 28, 6);
  t('2.6259981818539021981659e+339340', '6.8958664511E+678680', 23, 2);
  t('1.650095715573914339995571593605530228843869509809576660267100705710199e+460348481858', '2.72281587055538841173686928082096620997022877836901707243689996E+920696963716', 71, 1);
  t('2.5961509971494339104070358328276508155305955410565531e+27506184764', '6.7400E+55012369528', 53, 0);
  t('6.98318893856197088096157423785895630275546e+1', '4876.49277516542655231337713128136167574490306845440883621473529483335813066', 42, 6);
  t('9.0231652121710693691311381001357983609378069176681239052663552490593273e+472', '8.1417510446134179304795008671281142077622655031292884E+945', 71, 0);
  t('3.614556567833e+905', '1.30650191820664E+1811', 13, 3);
  t('9.1398161452037739597682402466114e+2', '835362.39168127574079882209155115811445457403061495355495204551760538374', 32, 6);
  t('1.48195910345e+265', '2.1962027843207439856796146949959914521607703076789660614150E+530', 12, 3);
  t('1.5556e+14554', '2.419900615649606588649129133146016064773740775801480E+29108', 6, 6);
  t('9.577563741597269956621661494375676941761541216043791043763111011243284670677e+148495594025', '9.1729727224358697241613267368499559222661984280657781520944350143634252209378072E+296991188051', 76, 0);
  t('1.8506236700427283766082924116700167702e+869890601624', '3.424807968122417190265786224803414598744313823843255233059744E+1739781203248', 38, 3);
  t('9.826305933387008239476331241821733277265221856e+322094481799', '9.655628829651672320852209236659488021285268281974300650419875326E+644188963599', 46, 6);
  t('2.624529228150198019388839742009995469e+207078651009687', '6.88815366941467416777000064485475542793749913477244591699232199964730E+414157302019374', 37, 0);
  t('1.6072e+444880748', '2.5829424561385859658292788297683281827868597224516601440089262848450E+889761496', 5, 5);
  t('4.14372056656893949780344576110128553784727806963084424e+35972549165834', '1.71704201338064129520393922040426764484532952194442522349724910206873998394075971283951735644E+71945098331669', 54, 2);
  t('5.0531262501544526519574040306536300193230181695512694054594791474960429e+43540097', '2.553408490E+87080195', 71, 1);
  t('7.82604e+4', '6124701461.30253396696741160592218892208', 6, 3);
  t('2.2398583586237716716486413363400318012837209715720882626185e+47', '5.0169654666967765488407729714423591421E+94', 59, 6);
  t('9.880231068423594927351688244474459194937005e+644353', '9.7618965965442852147114138973091944182243234357713569889829E+1288707', 43, 0);
  t('4.962894523835293602541700098665678245210425304527057712473799410572054423102e+99', '2.463032205471434562E+199', 76, 1);
  t('3.079618529786431417491547e+33473302', '9.48405028900394137180787430531224638858239194922517032274376483841159408163465435135E+66946604', 25, 2);
  t('2.55106412528206453900085028710092456169084330858943744594958242411029891729e-36', '6.507928171301145077839025428E-72', 75, 3);
  t('1.599636494099434519120495776736793381468862256137471e+4144316766399', '2.558836913254730207106858879274826091287738890342230033749159837590085448680594119987E+8288633532798', 52, 0);
  t('2.07272418e-12884693243', '4.2961855225494E-25769386486', 9, 6);
  t('8.69228216461714213735790110145447192421003957520348071e-1861021855949449', '7.5555769229321270082798708075E-3722043711898897', 54, 2);
  t('8.24962547017544740347522820382134272105255583449e+2865', '6.805632039816747163674E+5731', 48, 2);
  t('7.27483292499446870635760964657613202175753147622053e-148221990584', '5.292319408658357715078502155762867585E-296443981167', 51, 5);
  t('4.90795797150709105376962677572069305380908614111297365553e+4086', '2.4088051450080E+8173', 57, 1);
  t('7.34786550466656744803006128046e+2795776346528318', '5.399112747466886993138311223824932339056153376414279E+5591552693056637', 30, 3);
  t('9.70637064951188141912617586468613169489040932e+9263851', '9.4213631185705702766011583868449956283599883063266590478542230153211906005516607505881153E+18527703', 45, 4);
  t('3.14106344730876479475193343625504e-910', '9.8662795800192214301268467968737763973E-1820', 33, 2);
  t('6.8863641511930920159e+1434', '4.742201122283735467286752580639151338402147978758E+2869', 20, 5);
  t('9.1536625949357836061871605822069283444244503528403697e+93673802', '8.3789538901926503620943368526446290753065226079800268864837819950868226858046E+187347605', 53, 5);
  t('2.050219946066689694568885168e+41880885', '4.2034018272497E+83761770', 28, 5);
  t('2.58797367988247572680336566731031833e+3461935', '6.69760776776444294842577371961987758394584550633726992061849437263890087941E+6923870', 36, 0);
  t('6.0585255787e-340010557', '3.670573218788407507689514263121673716122995116133303311009468985401267408779E-680021113', 11, 6);
  t('2.033581e+3480', '4.135448217987841390586067157756058403489565598382242753255526546540540095751E+6960', 7, 2);
  t('2.4573907806034206508145897417338517767341675713702823338890812489722423677833e+38617943', '6.038769448594689087910861123468200183610626181299843431865263126830407743069E+77235886', 77, 5);
  t('1.8941557084570725928674400770347757019608538011645519465e+304618795294', '3.587825847880514585709358773677324854320097535772497696497628452E+609237590588', 56, 3);
  t('2.32812250747228299412e+313837', '5.4201544097990303858141E+627674', 21, 5);
  t('8.60645693084276008079123831702972740385e+154', '7.40711009024513815760164703475393233970877193508242149988958928091491914E+309', 39, 4);
  t('2.2706488927283296329062209956647601583635908763632678803367026671603381e+37', '515584639404838941246901894159595954745567752911318738407100564309750626516.1791617', 71, 1);
  t('3.39893741999727121585845223399480273146968e-1897475', '1.1552775585057706466940388889077953341599272409994467228416821444233007E-3794949', 43, 4);
  t('5.5136e+166880406', '3.04E+333760813', 5, 5);
  t('3.030059156073167836053285715808361515527841237119577e-4844973118878', '9.181258489302838079121284106371E-9689946237756', 52, 3);
  t('3.01621123e+412557134', '9.097530205431698341141736543467287960383290070775938518339807290620844422120352E+825114268', 9, 1);
  t('2.7692987e+14174', '7.66901510445223617671270401903922314736658973717857677577742198978502434481269112716631157122727993E+28348', 8, 2);
  t('8.1163781879836183909409375363404068136137574908934e+204971512154', '6.58755948903762446750976327895628343674985502155875440613151906714494399548343042750830709062E+409943024309', 50, 2);
  t('2.2572378462909762847780411032884150727038656700793e+46981', '5.0951226947283250806636033744756427856016086366224E+93962', 50, 2);
  t('9.0281062255204740105304768996726311145672013275416789795275497995048942097e+226702402736093', '8.150670201928153993411272129594404183375704672E+453404805472187', 74, 3);
  t('2.64772769892363307093059365640630051052232478304226906666646516317583776e+295003889110756', '7.0104619676474369342365609690167930592871571199219650021680E+590007778221512', 72, 2);
  t('3.13423173774023388222090399070014419184070085892e+2', '98234.08585858166222466900865352520714306', 48, 4);
  t('7.5875e-145', '5.7570159774745196978712550411544E-289', 6, 4);
  t('9.019814533002418483218041677971520150196e+3587590862646067', '8.13570542097616366291558237185955227709177843801016027464366840371504045382881466535960323E+7175181725292135', 40, 5);
  t('1.35383017303503354367849439544524371635138052709984972e-317514', '1.8328561374200688659993455967130023996868394E-635028', 54, 6);
  t('9.37621997569568303472309542853412757889934365090285260995e+2197', '8.791350103263475495836258678650727954866E+4395', 57, 3);
  t('3.0504548177905637298e-12020179286', '9.30527459538166136476938267777527342524470683073510498487415309409144843033563741336552377142559E-24040358572', 20, 5);
  t('3.086354295687660799750652512900547113767349372521616998883899383188989461278e-12', '9.5255828385096767511E-24', 76, 4);
  t('6.01199e+2879412', '3.61440703564099768E+5758825', 6, 1);
  t('6.4466262346647814710667613602892424745155045175684165271079e+3092341', '4.1558989809468218098954125905590E+6184683', 59, 3);
  t('2.33792157009946435002902338510510938904819610116259987000091846057423155518e+1006816', '5.46587726793634459876766095307E+2013632', 75, 6);
  t('7.52637894342292600459229452291474918313e+41', '5.66463800E+83', 39, 5);
  t('2.5626769096419963476340903043996166294265167882558946536e-526', '6.56731294321225271290059931898686901695490329384659866588358271112311008476E-1052', 56, 5);
  t('1.97484176581314990174384610437230990660852618680820080988126186147286133697613e+39070', '3.90E+78140', 78, 5);
  t('1.45440475274653711374621434742103752e+477036835198', '2.115293184811715756111035E+954073670396', 36, 5);
  t('6.1992027227922108538280901695224715931051043157545802926356061644672662141e-4', '3.843011439827436064752567943485332632909203518838336622636143406313606434503297E-7', 74, 0);
  t('1.861442505620019066170426433408338273749989194997433001e+25044421131632', '3.46496820172893471274450231665851340050925461712723830282019910966406769659794792E+50088842263264', 55, 2);
  t('2.96906281918975236955954438611698479173299363074681248825253049431984948e+2137858258433490', '8.815334024295000171588787248553700566011567E+4275716516866980', 72, 3);
  t('5.175035338263985141e-19506965', '2.678099075228103912053736336553650966855863923E-39013929', 19, 1);
  t('5.6856749819172745252775650565521e+444143632171956', '3.23269E+888287264343913', 32, 4);
  t('7.8957265213531807075664078327696484000996048294363154908871e+11785538569', '6.234249730E+23571077139', 59, 0);
  t('1.8283610369653862758941525973088006356e-393', '3.3429040814931426E-786', 38, 2);
  t('6.324555320336758664e+2759', '4E+5519', 21, 5);
  t('2.622555500020207157785460692451466974946442382680558e-42303129343', '6.8777973506862387855796638472705729011046488677364623807540946604353320E-84606258686', 52, 3);
  t('1.9200171460327483539026314778967671885846184667896514124254004538864392e+46889086360', '3.6864658410597401179937293588455580384542595E+93778172720', 71, 5);
  t('1.32991783013086940115047303330105611393593893453e+2137996', '1.7686814349E+4275992', 48, 5);
  t('8.504876072821706451854804787270754918424940970464942356230569216208409282293e+1954155074', '7.23329170140551722658511E+3908310149', 76, 0);
  t('8.989429645304548256737997439398050197279956353931606298464860617251463e-2145138768798', '8.080984534788025628E-4290277537595', 70, 5);
  t('2e+87988', '4.168731982E+175976', 2, 5);
  t('5.2064477870639497e-28517', '2.71070985594230989175938821603202E-57033', 17, 1);
  t('7.2266787484135478395239424677393393e+34292647220436', '5.222488573277200227050900075314803902522307294599971E+68585294440873', 35, 5);
  t('2.65893085873818746784799462956040824680806779629363141352213215464903828502e+41', '7.06991331155019503904475228560073103E+82', 75, 2);
  t('8.881482619e+2095399787486186', '7.888073352767506531631057273649734668539553289948510498240822582019489548040318306E+4190799574972373', 10, 1);
  t('9.80332759539657e+1', '9610.523194266382981096232960460004238837580104560636452314738102557965139756002214949', 15, 4);
  t('9.082984330509808229878255253074802885120895971e+4349613472438', '8.2500604348286709226838364974088837798915132923152211474564761463852274150514234984081825E+8699226944877', 46, 0);
  t('2.3051841717555917375661e-3805332', '5.3138740657125134679225950980E-7610664', 23, 5);
  t('1.58649846866898977e+17', '25169773910890495218058815869093616.163248025716892103133483661936354157144385347', 18, 6);
  t('1.832487619892217619778605787594363341371e+489189', '3.35801087705824464519194153474023E+978378', 41, 6);
  t('8.46288196383477866594835930945604975e+68975897', '7.16203711338E+137951795', 37, 0);
  t('2.15903826476511767107e-426122', '4.661446228719970353170383319438925741090187385001608608366288E-852244', 21, 5);
  t('3.25286507801e+13148279697', '1.05811312157440264046383518534621E+26296559395', 12, 3);
  t('3.926939618857446e+2701650423077205', '1.542085477015226E+5403300846154411', 16, 4);
  t('2.300649433664730442433670197221423917e+33169', '5.292987816621844920407591519790997148291859608855681470609202096937600310235150613372875022280048179E+66338', 37, 5);
  t('8.587979082919625568762e+0', '73.753384728665013020438300', 22, 4);
  t('2.068686747170362320265134050006653074465e+34', '4.2794648579182945572703279910205170577550420354118714938358645443205E+68', 40, 4);
  t('7.69e-46140374415', '5.9182163506875549546680576005992720902216355454971484890245944725555E-92280748829', 3, 5);
  t('1.8175779968958926637429567383064725023e+489880727845850', '3.303589774800085601596674E+979761455691700', 38, 2);
  t('8.919264136401625271884219060858252e+9', '79553272734900230263.41763128677730026531493235388640071381802652591', 34, 1);
  t('2.684345427e+44565', '7.205710371469E+89130', 12, 4);
  t('2.9534086394e+225110078', '8.722622591246883914133438987850898330671972588389730184309208739334634887878346367329202E+450220156', 11, 4);
  t('2.2293525e-29', '4.970012461272812033092318803761409304947044073671573877964943E-58', 8, 0);
  t('2.12326974896414051580077536570530341003518e+129130', '4.5082744268662442849713715921E+258260', 42, 3);
  t('2.5833750802158470838638220584e+18913396', '6.6738268050802343551355260020766554923882450343012342734378848E+37826792', 30, 1);
  t('1.0482628915520200483e+1', '109.8855089805002144728403440947030093340246871753530', 20, 5);
  t('1.49777148251106029879789692329117828615e-494834655520847', '2.24331941382337940650496041556953252175484515884385876892851088684239094331708000189663420216840E-989669311041694', 39, 5);
  t('6.95456711664340560996070853420178762114841e-38627491422785', '4.8366003779897772450979302026969538397871395603455798715295598491356099722574E-77254982845569', 42, 3);
  t('3.10149148573625247e-4', '9.6192494360944667705321067387464882437575358E-8', 18, 5);
  t('1.9575110120741046328968e+257921497', '3.83184936239138541387679462617387283711241414931326900113204911095477961E+515842994', 23, 0);
  t('7.7671e+1441473951', '6.0327727774912176230101E+2882947903', 5, 6);
  t('1.55009951392721507762173021041426457646284757348955667596e+4760', '2.40280850307738845039513019493014927568794808938729593338641386181207198737391954656000091310896E+9520', 57, 3);
  t('6.97734691743388618569498e-29', '4.8683370006224153769363821987754694567429898400322317885097538299174433336292355572118157532513893E-57', 24, 4);
  t('1.4993331851193049612203995625656077760057082447306644612348e+2339610533576283', '2.2480E+4679221067152566', 59, 3);
  t('6.04693544698469e-220', '3.65654283E-439', 15, 1);
  t('4.92955e+230982431475', '2.4300466966592452130220374934414E+461964862951', 7, 6);
  t('3.0764244910060828198383307889390376085e+4181965985826', '9.46438764886203575285E+8363931971652', 38, 0);
  t('7.4513572904573959e+293', '5.5522725470052584411450816E+587', 17, 0);
  t('1.245345961467927340281476373719978090064081519e+42', '1.550886563744476367740974406041099100674095756961953547612335899E+84', 46, 1);
  t('1.5499077392523e-4', '2.40221400019421603693694579636622069903401E-8', 14, 5);
  t('3.07185786e+12919', '9.43631065159385809410362801786965384978640704874013E+25838', 9, 0);
  t('2.361286043755e+281976835', '5.5756717804303509418805557E+563953670', 13, 5);
  t('5.47722557505166113456969782800802e+405615457129', '3E+811230914259', 33, 6);
  t('1.7178698335153e+144561611584745', '2.9510767649015787272590443004662758146251322168953340785510127474489250982992775086288323854164153E+289123223169490', 14, 2);
  t('2.2011e+12378959348', '4.84515086803287598716648233379771635161708841282201798840424143306973602425697887937893645861532E+24757918696', 5, 3);
  t('3.071695622941830478526650605376658041302645824732e+28109511538117', '9.435314E+56219023076234', 49, 2);
  t('2.307714697590689989764e+284', '5.325547125476089750844425629104815411440491176993481E+568', 22, 0);
  t('2.5862546754279728496914821072086e-33773833', '6.6887132461730491917584965749675495749719281864675587283858319408183402627833958E-67547666', 32, 3);
  t('5.74069223118780469804395537303763292412320069e-28', '3.295554729322001530304774161762307365140729343455866587659018918667945699638475469150683153123653323E-55', 45, 5);
  t('2.53435553329728248339409747785119424231260889160560714e+25943171346', '6.422957969154553102396E+51886342692', 54, 2);
  t('1.4069894208184455429860946489038258613329117231491e+114452', '1.979619230295024840325033863086398472578753367030579325792446632992E+228904', 50, 0);
  t('1.118294004710828525178841841394721727540545598338176097e+1', '125.058148097218257166460091', 55, 4);
  t('5.888425440747775688e+2065204358486413', '3.467355417124563636358862403139990401147625419807197821070468E+4130408716972827', 19, 0);
  t('3.0558360616281147776649165585589148e+202', '9.3381340355468272974654269623217477382915449047483811417E+404', 35, 4);
  t('8.49192304e+1387453895', '7.2112757E+2774907791', 9, 4);
  t('1.6985163959801521e+3936837', '2.88495794741340509204059372399499124467343954532861012543791870383E+7873674', 17, 3);
  t('4.991092064869170494e-124451984', '2.4911E-248903967', 19, 0);
  t('5.9823664195735900053440901869404771842353600892916293363151e+2227900957', '3.57887079780417347338180368431509195268242653234667944437089019357950841991042084010E+4455801915', 59, 0);
  t('1.90934970073455410278410575623977888122382805484e+422554664715', '3.6456162796951313121087727108063524737724086412612828706606881122179782708607529986314383405E+845109329430', 48, 1);
  t('2.599310207388995318083e+398467', '6.75641355423662185068716628560209294936028E+796934', 22, 6);
  t('3.7633102634136297417364966e+778600439757179', '1.416250413871436327348908304665878915237348394E+1557200879514359', 26, 0);
  t('1.2047068181e+31361471610', '1.45131851760962261131924534974E+62722943220', 11, 4);
  t('4.35951628054340993989341222113828625466454197009e-21152924900022', '1.90053822003230473596535984483816E-42305849800043', 48, 6);
  t('1.576844611205999099662237540201844126209316481620105534e+335357340260426', '2.486438927889398461388938E+670714680520852', 55, 1);
  t('3e-414034', '6.47961009652494321346331325444614339994562352820030434758628977937286889619097368135E-828068', 1, 2);
  t('2.7450159803706115172884307982554068123557452994357916574e+29493402848956', '7.535112732490029474794930517417328483099871453263111571224994801873955541077813E+58986805697912', 56, 1);
  t('8.450708163926728117884231904577733365414280601e-3356220959', '7.1414468471857852311230634694477427663911004991250862311231174531532437446075636178902521E-6712441917', 46, 2);
  t('2.21717604626595076e+2233028', '4.91586962013551338731303957E+4466056', 18, 2);
});
