////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB Device Driver
//
///	\file     PicoUsbDevice_MacOSX.cpp
///	\brief    Generic Pico USB device class
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: PicoUsbDevice_MacOSX.cpp,v 1.5 2007/08/01 17:34:21 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////

#include "PicoPortability.h"

#ifdef PICO_OS_MACOSX

#include <assert.h>
#include <pthread.h>
#include <stdio.h>
#include <mach/mach.h>

#include <CoreFoundation/CFNumber.h>

#include <IOKit/IOKitLib.h>
#include <IOKit/IOCFPlugIn.h>
#include <IOKit/usb/IOUSBLib.h>

#include "PicoUsbDevice_MacOSX.h"
#include "PicoUsbID.h"

//#define DEBUG 1

pthread_mutex_t PicoUsbDevice_MacOSX::handlesMutex = PTHREAD_MUTEX_INITIALIZER;
bool PicoUsbDevice_MacOSX::handlesInitialized = 0;
unsigned char * PicoUsbDevice_MacOSX::deviceHandles = NULL;

mach_port_t PicoUsbDevice_MacOSX::masterPort;
//////////////////////////////////////////////////////////////////////////
// Obtains an IO Kit master port
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice::Init() {
	IOReturn err;
	err = IOMasterPort(MACH_PORT_NULL, &(PicoUsbDevice_MacOSX::masterPort));
    if(err) {
#if DEBUG
        printf("PicoUsbDevice::Init: could not create master port, err = %08x\n", err);
#endif
        return -1;
    }
	return 0;
}

//////////////////////////////////////////////////////////////////////////
// Count the number of devices with the Pico VID and a matching PID
// Does not check DID so does not differentiate between variants
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice::Count(unsigned short product) {
	IOReturn				err;
    CFMutableDictionaryRef 	matchingDictionary = 0;		// requires <IOKit/IOKitLib.h>
    SInt32					idVendor=VENDOR_ID_PICO_TECHNOLOGY;
    SInt32					idProduct=product;
    CFNumberRef				numberRef;
    io_service_t			usbDevice=-1;
	//SInt32					score; 
    //IOCFPlugInInterface 	**plugInInterface=NULL;
	//HRESULT					res;
    //IOUSBDeviceInterface	**dev=NULL;
    io_iterator_t			iterator = 0;
	
    
#if DEBUG
	printf("PicoUsbDevice::Count()\n");
#endif
	
    // Create a matching dictionary
    matchingDictionary = IOServiceMatching(kIOUSBDeviceClassName);	// requires <IOKit/usb/IOUSBLib.h>
    if (!matchingDictionary) {
#if DEBUG
        printf("PicoUsbDevice::Count: could not create matching dictionary\n");
#endif
        return -1;
    }
	
    // Add VID reference
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &idVendor);
    if (!numberRef) {
#if DEBUG
        printf("PicoUsbDevice::Count: could not create CFNumberRef for vendor\n");
#endif
        return -1;
    }
    CFDictionaryAddValue(matchingDictionary, CFSTR(kUSBVendorID), numberRef);
    CFRelease(numberRef);
    numberRef = 0;
	
    // Add PID reference
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &idProduct);
    if(!numberRef) {
#if DEBUG
        printf("PicoUsbDevice::Count: could not create CFNumberRef for product\n");
#endif
        return -1;
    }
    CFDictionaryAddValue(matchingDictionary, CFSTR(kUSBProductID), numberRef);
    CFRelease(numberRef);
    numberRef = 0;
    
    // Get all matching devices
    err = IOServiceGetMatchingServices(PicoUsbDevice_MacOSX::masterPort, matchingDictionary, &iterator);
    matchingDictionary = 0;			// this was consumed by the above call
    
    // Iterate through devices counting them
	int deviceCount=0;
	while(usbDevice=IOIteratorNext(iterator)) {
#if DEBUG
		printf("PicoUsbDevice::Count: Found device %p\n", (void*)usbDevice);
#endif
		deviceCount++;
		IOObjectRelease(usbDevice);			// no longer need this reference
	}
	
    // Clean up
    IOObjectRelease(iterator);
    iterator = 0;
    
    mach_port_deallocate(mach_task_self(),PicoUsbDevice_MacOSX::masterPort);
	
    return deviceCount;
}

//////////////////////////////////////////////////////////////////////////
// Get a list of connected devices by PID
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice::Enumerate(PicoUsbDevice **list,unsigned int length,unsigned short product) {
	IOReturn				err;
    CFMutableDictionaryRef 	matchingDictionary = 0;		// requires <IOKit/IOKitLib.h>
    SInt32					idVendor=VENDOR_ID_PICO_TECHNOLOGY;
    SInt32					idProduct=product;
    CFNumberRef				numberRef;
    io_service_t			usbDevice=-1;
	SInt32					score; 
    IOCFPlugInInterface 	**plugInInterface=NULL;
	HRESULT					res;
    IOUSBDeviceInterface	**dev=NULL;
    io_iterator_t			iterator = 0;
	CFMutableDictionaryRef usbDeviceProperties=NULL;
    
#if DEBUG
	printf("PicoUsbDevice::Enumerate()\n");
#endif
	
    // Create a matching dictionary
    matchingDictionary = IOServiceMatching(kIOUSBDeviceClassName);	// requires <IOKit/usb/IOUSBLib.h>
    if (!matchingDictionary) {
#if DEBUG
        printf("PicoUsbDevice::Enumerate: could not create matching dictionary\n");
#endif
        return -1;
    }
	
    // Create VID reference
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &idVendor);
    if (!numberRef) {
#if DEBUG
        printf("PicoUsbDevice::Enumerate: could not create CFNumberRef for vendor\n");
#endif
        return -1;
    }
    CFDictionaryAddValue(matchingDictionary, CFSTR(kUSBVendorID), numberRef);
    CFRelease(numberRef);
    numberRef = 0;
	
    // Create PID reference
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &idProduct);
    if(!numberRef) {
#if DEBUG
        printf("PicoUsbDevice::Enumerate: could not create CFNumberRef for product\n");
#endif
        return -1;
    }
    CFDictionaryAddValue(matchingDictionary, CFSTR(kUSBProductID), numberRef);
    CFRelease(numberRef);
    numberRef = 0;
    
    // Get all matching devices
    err = IOServiceGetMatchingServices(PicoUsbDevice_MacOSX::masterPort, matchingDictionary, &iterator);
    matchingDictionary = 0;			// this was consumed by the above call
    
	int curDevice=0;
	// Iterate through connected devices
	for(;(curDevice<length)&&(usbDevice!=0);curDevice++) {
		while(usbDevice=IOIteratorNext(iterator)) {
#if DEBUG
			printf("PicoUsbDevice::Enumerate: Found device %p\n", (void*)usbDevice);
#endif
			
			// Create plugin for device
			err = IOCreatePlugInInterfaceForService(usbDevice, kIOUSBDeviceUserClientTypeID, kIOCFPlugInInterfaceID, &plugInInterface, &score);
			
			if ((kIOReturnSuccess != err) || !plugInInterface) {
#if DEBUG
				printf("PicoUsbDevice::Enumerate: unable to create a plugin (%08x)\n", err);
#endif
				IOObjectRelease(usbDevice);
				continue;
			}
			
			// Got device interface, get properties of the device
#if DEBUG
			printf("PicoUsbDevice::Enumerate: Creating device properties\n");
#endif			
			usbDeviceProperties = NULL;
			res = IORegistryEntryCreateCFProperties(usbDevice, &usbDeviceProperties,kCFAllocatorDefault, kNilOptions);
			if(res) {
				usbDeviceProperties = NULL;
			}
			
			// Done with usbDevice
			IOObjectRelease(usbDevice);
			
			// I have the device plugin, I need the device interface
#if DEBUG
			printf("PicoUsbDevice::Enumerate: Creating device interface\n");
#endif
			res = (*plugInInterface)->QueryInterface(plugInInterface,CFUUIDGetUUIDBytes(kIOUSBDeviceInterfaceID),(LPVOID*)&dev);
			IODestroyPlugInInterface(plugInInterface);			// done with this
			
			if (res || !dev) {
#if DEBUG
				printf("PicoUsbDevice::Enumerate: couldn't create a device interface (%08x)\n", (int) res);
#endif
				continue;
			}
			
			// Create a PicoUsbDevice object for our device
			list[curDevice]=(PicoUsbDevice *)new PicoUsbDevice_MacOSX(dev,usbDeviceProperties);
			
			IOObjectRelease(usbDevice);			// no longer need this reference
			
			break;
		}
	}
	
	
	
    IOObjectRelease(iterator);
    iterator = 0;
    
    mach_port_deallocate(mach_task_self(),PicoUsbDevice_MacOSX::masterPort);
	
    return curDevice-1;
}


//////////////////////////////////////////////////////////////////////////
// Constructor. Only called by Enumerate()
//////////////////////////////////////////////////////////////////////////
PicoUsbDevice_MacOSX::PicoUsbDevice_MacOSX(IOUSBDeviceInterface **newDevice) {
#if DEBUG
	printf("PicoUsbDevice_MacOSX::PicoUsbDevice_MacOSX()\n");
#endif
	
	// Just set up some member variables
	assert(newDevice);
	
	device=newDevice;
	interface=NULL;
	properties=NULL;
	serialString=NULL;
	state = PICODEVICE_STATE_CLOSED;	
	pthread_mutex_init(&mutex,NULL);
}

//////////////////////////////////////////////////////////////////////////
// Constructor. Only called by Enumerate()
//////////////////////////////////////////////////////////////////////////
PicoUsbDevice_MacOSX::PicoUsbDevice_MacOSX(IOUSBDeviceInterface **newDevice,CFMutableDictionaryRef deviceProperties) {
#if DEBUG
	printf("PicoUsbDevice_MacOSX::PicoUsbDevice_MacOSX()\n");
#endif
	
	// Just set up some member variables
	assert(newDevice);
	
	device=newDevice;
	interface=NULL;
	properties=deviceProperties;
	serialString=NULL;
	state = PICODEVICE_STATE_CLOSED;
	pthread_mutex_init(&mutex,NULL);
}

//////////////////////////////////////////////////////////////////////////
// Destructor
//////////////////////////////////////////////////////////////////////////
PicoUsbDevice_MacOSX::~PicoUsbDevice_MacOSX(void) {
	IOReturn err;
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::~PicoUsbDevice_MacOSX()\n");
#endif
	
	// Clean up
	
	if(device) {
		err=(*device)->Release(device);
	}
	if(serialString) {
		free(serialString);
	}
	
	pthread_mutex_destroy(&mutex);
}

//////////////////////////////////////////////////////////////////////////
// Get current state
//////////////////////////////////////////////////////////////////////////
PICODEVICE_STATES PicoUsbDevice_MacOSX::GetDeviceState() {
	// Return state variable
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetDeviceState()");
#endif
	
	PICODEVICE_STATES tmpState=state;
	
	pthread_mutex_unlock(&mutex);
	
#if DEBUG
	printf("=%i\n",tmpState);
#endif
	
	return tmpState;
}

//////////////////////////////////////////////////////////////////////////
// Open a device
//////////////////////////////////////////////////////////////////////////
PICODEVICE_STATES PicoUsbDevice_MacOSX::Open(void) {
	PICODEVICE_STATES tmpState;
	IOReturn err;
	int i, j;
	
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::Open()\n");
#endif
	assert(device);

	handle = 0;
	
	// Try to open the IOKit device
	err = (*device)->USBDeviceOpen(device);
	if(kIOReturnExclusiveAccess==err) {
		// "Exclusive Access Error" means the device is already open
#if DEBUG
		printf("PicoUsbDevice_MacOSX::Open: Exclusive access err\n");
#endif
		state=tmpState=PICODEVICE_STATE_LOCKED;
	} else {
		// Success
		// Look for an available handle
		pthread_mutex_lock(&handlesMutex);
		
		if (!handlesInitialized) {
			handlesInitialized = true;
			deviceHandles = new unsigned char(4096);
			assert(deviceHandles);
			deviceHandles[0]=0x01; // Don't use 0 as a handle
			for (int i = 1; i < sizeof(deviceHandles); i++) {
				deviceHandles[i] = 0;
			}
		}
		
		for (i = 0; i < sizeof(deviceHandles); i++) {
			// Look for a handle location which isn't in use
			if (deviceHandles[i] != 0xFF) 
				break;
		}
		if (i == sizeof(deviceHandles)) { 
			// We've run out of handles.
			state=tmpState=PICODEVICE_STATE_CLOSED;
		} else {
			for (j = 0; j < 8; j++) {  
				if (!(deviceHandles[i] & (1<<j))) 
					break;
			}
	
			handle = (i * 8) + j;
			deviceHandles[i] |= (1<<j);
			
			state=tmpState=PICODEVICE_STATE_OPEN;
		
			setInterface(0);
		}

		pthread_mutex_unlock(&handlesMutex);
	}
	
	pthread_mutex_unlock(&mutex);
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::Open: Exit\n");
#endif
	return tmpState;
}


//////////////////////////////////////////////////////////////////////////
/// Get handle for use with C API
/// <returns>The handle of the device</returns>
//////////////////////////////////////////////////////////////////////////
short PicoUsbDevice_MacOSX::GetHandle(void) {

#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetHandle()\n");
#endif

	return handle;
}


//////////////////////////////////////////////////////////////////////////
// Get Product ID
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice_MacOSX::GetPID(void) {
	unsigned short product=0;
	IOReturn err;
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetPID()\n");
#endif
	
	assert(device);
	
	// Ask IOKit for PID
	err = (*device)->GetDeviceProduct(device, &product);
	if (err!=kIOReturnSuccess) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::GetPID: Error getting product\n");
#endif
		return -1;
	}
	return product;
}

//////////////////////////////////////////////////////////////////////////
// Get Device ID
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice_MacOSX::GetDID(void) {
	unsigned short release=0;
	IOReturn err;
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetDID()\n");
#endif
	
	assert(device);
	
	// Ask IOKit for DID ("Release Number")
	err = (*device)->GetDeviceReleaseNumber(device, &release);
	if (err!=kIOReturnSuccess) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::GetDID: Error getting release\n");
#endif
		return -1;
	}
	return release;
}

//////////////////////////////////////////////////////////////////////////
// Get serial number. Requires Mac FW
//////////////////////////////////////////////////////////////////////////
const char *PicoUsbDevice_MacOSX::GetSerialString(void) {
	CFStringRef serialRef;
	const char *str=NULL;
	
	// Look up the "USB Serial Number"
	if((serialString==NULL)&&(properties)) {
		serialRef = (CFStringRef)CFDictionaryGetValue(properties, CFSTR("USB Serial Number"));
		if(serialRef) {
			str=CFStringGetCStringPtr(serialRef,CFStringGetFastestEncoding(serialRef));
			serialString=(char*)malloc((strlen(str)+1)*sizeof(char));
			strcpy(serialString,str);
			CFRelease(serialRef);
		}
	}
	return serialString;
}

//////////////////////////////////////////////////////////////////////////
// Thread-safe wrapper for setInterface
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice_MacOSX::SetInterface(int interfaceNum) {
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::SetInterface(%i)\n", interfaceNum);
#endif
	
	int ret=setInterface(interfaceNum);
	
	pthread_mutex_unlock(&mutex);
	
	return ret;
}

//////////////////////////////////////////////////////////////////////////
// Private fn to select a particular interface
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice_MacOSX::setInterface(int interfaceNum) {
    IOUSBFindInterfaceRequest		interfaceRequest;
    io_iterator_t					iterator;
    io_service_t					usbInterfaceRef;
	IOCFPlugInInterface				**iodev;
	SInt32							score;
	IOUSBInterfaceInterface			**newInterface;
	IOReturn err;
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::setInterface(%i)\n", interfaceNum);
#endif
	
	assert(device);
	
	// Get list of interfaces
	interfaceRequest.bInterfaceClass = kIOUSBFindInterfaceDontCare;		// requested class
    interfaceRequest.bInterfaceSubClass = kIOUSBFindInterfaceDontCare;		// requested subclass
    interfaceRequest.bInterfaceProtocol = kIOUSBFindInterfaceDontCare;		// requested protocol
    interfaceRequest.bAlternateSetting = kIOUSBFindInterfaceDontCare;		// requested alt setting
    
    err = (*device)->CreateInterfaceIterator(device, &interfaceRequest, &iterator);
    if(err) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::SetInterface: unable to create interface iterator\n");
#endif
		return -1;
    }
    
    // Iterate through avaialble interfaces
    while (usbInterfaceRef=IOIteratorNext(iterator)) {
		//printf("PicoUsbDevice_MacOSX::SetInterface: found interface: %p\n", (void*)usbInterfaceRef);
		
		err = IOCreatePlugInInterfaceForService(usbInterfaceRef, kIOUSBInterfaceUserClientTypeID, kIOCFPlugInInterfaceID, &iodev, &score);
		if (err || !iodev) {
#if DEBUG
			printf("PicoUsbDevice_MacOSX::SetInterface: unable to create plugin. ret = %08x, iodev = %p\n", err, iodev);
#endif
			return -1;
		}
		err = (*iodev)->QueryInterface(iodev, CFUUIDGetUUIDBytes(kIOUSBInterfaceInterfaceID), (LPVOID*)&newInterface);
		IODestroyPlugInInterface(iodev);				// done with this
		if (err || !newInterface) {
#if DEBUG
			printf("PicoUsbDevice_MacOSX::SetInterface: unable to create a device interface. ret = %08x, intf = %p\n", err, newInterface);
#endif
			return -1;
		}
		
		UInt8 intfNum;
		err=(*newInterface)->GetInterfaceNumber(newInterface,&intfNum);
		if(err) {
#if DEBUG
			printf("PicoUsbDevice_MacOSX::SetInterface: unable to get number of interface.");
#endif
			return -1;
		}
		
		// Have we found the interface we are looking for?
		if(intfNum==interfaceNum) {
#if DEBUG
			printf("PicoUsbDevice_MacOSX::SetInterface: Found interface: %i\n",interfaceNum);
#endif
			
			err = (*newInterface)->USBInterfaceOpen(newInterface);
			if (err){
#if DEBUG
				printf("PicoUsbDevice_MacOSX::SetInterface: unable to open interface. ret = %08x\n", err);
#endif
				return -1;
			}
			
			interface=newInterface;
			break;
		} else {
			interface=NULL;
		}
		
		IOObjectRelease(usbInterfaceRef);				// no longer need this reference
    }
    
    IOObjectRelease(iterator);
    iterator = 0;
	
	return 0;
}
//////////////////////////////////////////////////////////////////////////
// Get pipes (number of endpoints)
//////////////////////////////////////////////////////////////////////////
int PicoUsbDevice_MacOSX::GetPipes() {
	IOReturn err;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetPipes()\n");
#endif
	
	assert(interface);
	
	UInt8 numEndpoints=0;
	err=(*interface)->GetNumEndpoints(interface,&numEndpoints);
	if(err==kIOReturnSuccess) {
		
		pthread_mutex_unlock(&mutex);
		
		return numEndpoints;
	} else if (err==kIOReturnNoDevice) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::GetPipes: Error (kIOReturnNoDevice) getting number of endpoints from interface.\n");
#endif		
		state=PICODEVICE_STATE_DISCONNECTED;
		
		pthread_mutex_unlock(&mutex);
		
		return -1;
	} else {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::GetPipes: Error (unknown)  getting number of endpoints from interface.\n");
#endif		
		
		pthread_mutex_unlock(&mutex);
		
		return -1;
	}
}
//////////////////////////////////////////////////////////////////////////
// Get status of a given pipe
//////////////////////////////////////////////////////////////////////////
PicoUsbDevice::PipeInformation PicoUsbDevice_MacOSX::GetPipeInformation(int pipeNumber) {
	IOReturn err;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::GetPipeInformation(%i)\n",pipeNumber);
#endif
	
	assert(interface);
	
	PicoUsbDevice::PipeInformation info;
	err=(*interface)->GetPipeProperties(interface,pipeNumber,&info.direction,&info.number,&info.transferType,&info.maxPacketSize,&info.interval);
	if(err) {
		info.status=PicoUsbDevice::ErrorGettingStatus;
	} else {
		err=(*interface)->GetPipeStatus(interface,pipeNumber);  
		if((err==kIOReturnNoDevice)||(err==kIOReturnNotOpen)) {
			info.status=PicoUsbDevice::ErrorGettingStatus;
		} else if(err==kIOUSBPipeStalled) {
			info.status=PicoUsbDevice::ErrorGettingInfo;
		} else {
			info.status=PicoUsbDevice::StatusOK;
		}
	}
	
	pthread_mutex_unlock(&mutex);
	
	return info;
}

//////////////////////////////////////////////////////////////////////////
// Read bytes from given pipe
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoUsbDevice_MacOSX::ReadPipe(int pipeNumber,void *buf,unsigned int *size) {
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::ReadPipe(%i)\n",pipeNumber);
#endif
	IOReturn err;
	
	assert(interface);
	
	int ret=-1;
	err=(*interface)->ReadPipe(interface,pipeNumber,buf,(UInt32 *)size);
	if(err==kIOReturnSuccess) {	
		ret=0;
	} else if (err==kIOReturnNoDevice) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ReadPipe: Error (kIOReturnNoDevice) reading from pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_DISCONNECTED;
	} else if (err==kIOReturnNotOpen) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ReadPipe: Error (kIOReturnNotOpen) reading from pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_CLOSED;
	} else {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ReadPipe: Error (unknown %i) reading from pipe %i\n",err,pipeNumber);
#endif		
	}
	
	pthread_mutex_unlock(&mutex);
	
	return (PICO_RETURNS)ret;
}

//////////////////////////////////////////////////////////////////////////
// Write bytes to given pipe
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoUsbDevice_MacOSX::WritePipe(int pipeNumber,void *buf,unsigned int size) {
	IOReturn err;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::WritePipe(%i)\n",pipeNumber);
#endif
	
	assert(interface);
	
	int ret=-1;
	err=(*interface)->WritePipe(interface,pipeNumber,buf,size);  
	if(err==kIOReturnSuccess) {
		ret=0;
	} else if (err==kIOReturnNoDevice) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::WritePipe: Error (kIOReturnNoDevice) writing to pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_DISCONNECTED;
	} else if (err==kIOReturnNotOpen) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::WritePipe: Error (kIOReturnNotOpen) writing to pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_CLOSED;
	} else {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::WritePipe: Error (unknown %i) writing to pipe %i\n",err,pipeNumber);
#endif
	}
	
	pthread_mutex_unlock(&mutex);
	
	return (PICO_RETURNS)ret;
}

//////////////////////////////////////////////////////////////////////////
// Flush pipe
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS PicoUsbDevice_MacOSX::ResetPipe(int pipeNumber) {
	IOReturn err;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::ResetPipe(%i)\n",pipeNumber);
#endif
	
	assert(interface);
	
	int ret=1;
	err=(*interface)->ResetPipe(interface,pipeNumber);  
	if(err==kIOReturnSuccess) {
		ret=0;
	} else if (err==kIOReturnNoDevice) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ResetPipe: Error (kIOReturnNoDevice) resetting pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_DISCONNECTED;
	} else if (err==kIOReturnNotOpen) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ResetPipe: Error (kIOReturnNotOpen) resetting pipe %i\n",pipeNumber);
#endif		
		state=PICODEVICE_STATE_CLOSED;
	} else {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::ResetPipe: Error (unknown) resetting pipe %i\n",pipeNumber);
#endif		
	}
	
	pthread_mutex_unlock(&mutex);
	
	return (PICO_RETURNS)ret;
}

//////////////////////////////////////////////////////////////////////////
// Close USB device
//////////////////////////////////////////////////////////////////////////
void PicoUsbDevice_MacOSX::Close() {
	IOReturn err;
	pthread_mutex_lock(&mutex);
#if DEBUG
	printf("PicoUsbDevice_MacOSX::Close()\n");
#endif
	
	assert(device);
	
	// Close interface
#if DEBUG
	printf("PicoUsbDevice_MacOSX::Close: Closing interface\n");
#endif
	err = (*interface)->USBInterfaceClose(interface);
	if(kIOReturnExclusiveAccess==err) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::Close: Exclusive access err\n");
#endif
	}
	// Close device
	
#if DEBUG
	printf("PicoUsbDevice_MacOSX::Close: Closing device\n");
#endif
	err = (*device)->USBDeviceClose(device);
	if(kIOReturnExclusiveAccess==err) {
#if DEBUG
		printf("PicoUsbDevice_MacOSX::Close: Exclusive access err\n");
#endif
	}
	state=PICODEVICE_STATE_CLOSED;

	// Relinquish our handle
	pthread_mutex_lock(&handlesMutex);
	deviceHandles[handle / 8] |= (1<<(handle / 8));
	handle = 0;
	pthread_mutex_unlock(&handlesMutex);
	
	
	pthread_mutex_unlock(&mutex);
}
#endif
