/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      PicoPortability.h 
 * \brief     Definitions and routines to aid porting of Pico drivers
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: PicoPortability.h,v 1.13 2008/02/18 16:58:30 douglas Exp $
 *
 *************************************************************************** */

/* Avoid including this header more than once */
#ifndef PICOPORTABILITY_H
#define PICOPORTABILITY_H

/* ***************************************************************************
 * Determines the OS we are compiling for.
 *************************************************************************** */

/* Sets a Pico-specific OS definition. 
 * Not all of the following OSs are supported (see below) */
#if defined(WIN32) || defined(_WIN32) || defined(__WIN32__)
	#define PICO_OS_WIN32
#elif defined(__APPLE__)
	#define PICO_OS_MACOSX
#elif (defined(sun) || defined(__sun) || defined(__sun__)) && defined(__SVR4)
	#define PICO_OS_SOLARIS
#elif defined(linux) || defined(__linux) || defined(__linux__)
	#define PICO_OS_LINUX
#elif defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
	#define PICO_OS_BSD
#else
	#error "Sorry, the OS you are compiling for does not appear to be supported."
#endif

/* We support Windows, Linux and Mac OSX.
 * We don't yet have any BSD or Solaris support.
 * (But why not implement some if you're interested!) */
#if defined(PICO_OS_SOLARIS) || defined(PICO_OS_BSD)
	#error "Sorry, there's not yet support for Solaris or BSD."
#endif

/* ***************************************************************************
 * Platform-specific include files needed to compile this header only
 *************************************************************************** */
#ifdef PICO_OS_LINUX
	#include <stdlib.h> 	
	#include <stdio.h> 	
	#include <unistd.h>
	#include <stdint.h>
	#include <errno.h>
	#include <string.h>
	#include <assert.h>
	#include <sys/times.h>
#endif

/* ***************************************************************************
// Platform-specific data type definitions
 *************************************************************************** */

// If we're not compiling on Windows, map Win32-specific
// types onto standard C types
#ifndef PICO_OS_WIN32
	typedef int8_t 		__int8;
	typedef int16_t 	__int16;
	typedef int32_t 	__int32;
	typedef int64_t 	__int64;
	
	typedef uint8_t 	UCHAR;
	typedef uint8_t 	BYTE;
	typedef uint16_t 	WORD;
	typedef uint32_t 	DWORD;
	typedef uint32_t 	UINT;
	
	typedef int16_t 	SHORT;
	typedef uint16_t 	USHORT;
	
	typedef int32_t 	LONG;
	
	#ifndef PICO_OS_MACOSX // Apple have already defined this in CFPlugInCOM.h
		typedef uint32_t 	ULONG;
	#endif
	
	typedef int64_t 	LONGLONG;
	
	typedef short		BOOL;
	typedef uint32_t 	SIZE_T;
	typedef struct {int64_t QuadPart;} LARGE_INTEGER;
	typedef char * LPCTSTR, * LPSTR;
	typedef void * LPVOID, * PVOID;

#endif

// If we're not compiling on OSX, map Mac-specific
// types onto standard C types
#ifndef PICO_OS_MACOSX
	typedef uint8_t UInt8;
	typedef uint16_t UInt16;
	typedef uint32_t UInt32;
	typedef uint64_t UInt64;
	
	typedef int8_t Int8;
	typedef int16_t Int16;
	typedef int32_t Int32;
	
	typedef int8_t SInt8;
	typedef int16_t SInt16;
	typedef int32_t SInt32;
#endif

// Make sure we have TRUE and FALSE on all OSs
#ifndef TRUE
	#define TRUE  1
#endif
#ifndef FALSE
	#define FALSE 0
#endif

/* ***************************************************************************
// Map the new Win32 "secure" stdlib calls onto ANSI ones
 *************************************************************************** */

#ifndef PICO_OS_WIN32
	#define printf_s printf
	#define sprintf_s snprintf
	#define wsprintf sprintf
	#define wsprintf_s snprintf
	#define fprintf_s fprintf
	#define scanf_s scanf
	#define sscanf_s sscanf
	#define fscanf_s fscanf
	#define strtok_s(a,b,c) strtok(a,b)
	#define strncpy_s(a,b,c,d) strncpy(a,c,d)
	#define memcpy_s(a,b,c,d) memcpy(a,c,d)
	int fopen_s(FILE ** a, const char * b, const char * c);
#endif


/* ***************************************************************************
// Map some general Win32 calls onto equivalent POSIX/ANSI ones
 *************************************************************************** */

#ifndef PICO_OS_WIN32
	#define Sleep(x) usleep(1000*(x))
	#define ZeroMemory(a,b) memset(a,0,b)
	#define CopyMemory(a,b,c) memcpy(a,b,c)
	#define __min(a,b) (((a) < (b)) ? (a) : (b))
	#define __max(a,b) (((a) > (b)) ? (a) : (b))
	#define __noop {}
	#define _msize(a) malloc_usable_size(a)
#endif
	
/* ***************************************************************************
// Simulate Win32 GetLastError()
 *************************************************************************** */

#ifndef PICO_OS_WIN32
	long GetLastError(void);
	void OutputDebugStr(char * str);
#define OutputDebugString OutputDebugStr
#endif

/* ***************************************************************************
// We don't need to worry about windows calling conventions on other OSs
 *************************************************************************** */

#ifndef PICO_OS_WIN32
	#define __cdecl
	#define __stdcall
	#define __thiscall
	#define WINAPI
	#define CALLBACK
#endif


/* ***************************************************************************
// Timing routines.
 *************************************************************************** */

#ifndef PICO_OS_WIN32

	// Get value of a millisecond counter (arbitrary starttime)
	DWORD GetTickCount(void);
	
	// Get frequency in counts per second of the "Performance Counter"
	BOOL  QueryPerformanceFrequency(LARGE_INTEGER * lpFrequency);
	
	// Get value of the "Performance Counter"
	BOOL  QueryPerformanceCounter(LARGE_INTEGER * lpPerformanceCount);

#endif


/* ***************************************************************************
// Threading and synchronisation routines.
 *************************************************************************** */

#ifndef PICO_OS_WIN32
	// We map the behaviour of the other OSs onto the Windows function calls
	// so for windows we just need this:
	// By using a struct these are as general as possible for
	// optimal portability. On some OSs the struct amy only have one
	// member: on others several objects are required to achieve the 
	// function (e.g. we use a mutex and condition variable to 
	// implement a Win32-style semaphore on Linux)
	// The structs are defined in the platform-specific cpp files.
	
	struct 	PICO_HANDLE;
	typedef struct PICO_HANDLE *HANDLE, *HWND, *HINSTANCE, **PHANDLE;
	
	struct PICO_CRITICAL_SECTION;
	typedef struct PICO_CRITICAL_SECTION *CRITICAL_SECTION;
	
	typedef DWORD (* LPTHREAD_START_ROUTINE) (void * param);
	typedef void (* LPTHREAD_VOID_START_ROUTINE) (void * param);
	
	// Argument / return values for synchronisation functions
	#define INVALID_HANDLE_VALUE  (NULL)
	#define INFINITE              ((DWORD)0xFFFFFFFF)
	#define WAIT_ABANDONED        ((DWORD)0x00000080)
	#define WAIT_OBJECT_0         ((DWORD)0x00000000)
	#define WAIT_TIMEOUT          ((DWORD)0x00000102)
	#define WAIT_FAILED           ((DWORD)0xFFFFFFFF)
	#define STILL_ACTIVE          ((DWORD)0x00000103)

	#define THREAD_PRIORITY_ABOVE_NORMAL   1
	#define THREAD_PRIORITY_BELOW_NORMAL  -1
	#define THREAD_PRIORITY_HIGHEST        2
	#define THREAD_PRIORITY_IDLE         -15
	#define THREAD_PRIORITY_LOWEST        -2
	#define THREAD_PRIORITY_NORMAL         0
	#define THREAD_PRIORITY_TIME_CRITICAL 15
	
	// Prototypes of synchronisation functions. Note that these may not implement 
	// all the functionality of the corresponding Win32 API.
	
	// Semaphores
	HANDLE  CreateSemaphore(void * lpSemaphoreAttributes, LONG lInitialCount, LONG lMaximumCount, void * lpName);
	BOOL    ReleaseSemaphore(HANDLE hSemaphore, LONG lReleaseCount, long * lpPreviousCount);
	
	// Mutexes
	HANDLE  CreateMutex(void * lpMutexAttributes, BOOL bInitialOwner, void * lpName);
	BOOL    ReleaseMutex(HANDLE hMutex);
	
	// Events
	HANDLE  CreateEvent(void * lpEventAttributes, BOOL bManualReset, BOOL bInitialState, void * lpName);
	BOOL    SetEvent(PICO_HANDLE * hEvent);
	BOOL    ResetEvent(PICO_HANDLE * hEvent);
	
	// Threads
	HANDLE  _beginthread(void ( *start_address )( void * ), unsigned int stack_size, void *arglist);	
	HANDLE  _beginthreadex(void *security, unsigned int stack_size, unsigned int ( *start_address )( void * ), void *arglist, unsigned int initflag, unsigned int *thrdaddr );
	HANDLE  CreateThread(void * lpThreadAttributes, SIZE_T dwStackSize, LPTHREAD_START_ROUTINE lpStartAddress, void * lpParameter, DWORD dwCreationFlags, DWORD * lpThreadId);
	BOOL    GetExitCodeThread(HANDLE hThread, DWORD * lpExitCode);
	BOOL 	SetThreadPriority(HANDLE hThread, int nPriority);
	BOOL 	SuspendThread(HANDLE hThread);
	BOOL 	ResumeThread(HANDLE hThread);
	// Common to semaphores and threads
	DWORD   WaitForSingleObject(HANDLE hHandle, DWORD dwMilliseconds);
	BOOL    CloseHandle (HANDLE hObject); 
	
	// Critical sections
	BOOL InitializeCriticalSectionAndSpinCount(CRITICAL_SECTION * lpCriticalSection, DWORD dwSpinCount);
	BOOL InitializeCriticalSection(CRITICAL_SECTION *  lpCriticalSection);
	void DeleteCriticalSection(CRITICAL_SECTION *  lpCriticalSection);
	void EnterCriticalSection(CRITICAL_SECTION *  lpCriticalSection);
	void LeaveCriticalSection(CRITICAL_SECTION *  lpCriticalSection);

#endif // not defined PICO_OS_WIN32


#endif // not defined PICO_PORTABILITY_H
