/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      PicoDownloader_Linux.cpp 
 * \brief     Background firmware downloader for Pico products: Linux-specific implementation
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: PicoDownloader_Linux.cpp,v 1.16 2008/05/13 12:53:27 pei Exp $
 *
 *************************************************************************** */

#include <assert.h>
#include <stdio.h>

#include <errno.h>
#include <string.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <dirent.h>

#include "PicoUsbID.h"
#include "PicoStatus.h"
#include "PicoDownloader_Linux.h"
#include "PicoUsbDevice_Linux.h"
#include "PicoPortability.h"
#include "LinuxUsbFS.h"

#include <sys/wait.h>

#include "IntelHexRecord.h"

#define USB_TYPE_VENDOR			(0x02 << 5)

//#define DEBUG 1
//#define NARRATEDOWNLOAD 1

// USB chip FW download reg addresses
#define	kFX_FX2_USBCS			0x7f92
#define	kFX1_FX2LP_USBCS		0xe600


//////////////////////////////////////////////////////////////////////////
// Write the contents of writeBuffer to the device
//////////////////////////////////////////////////////////////////////////

PICO_RETURNS AnchorWrite(int dev, UInt16 anchorAddress, UInt16 count, UInt8 writeBuffer[])
{
	struct linux_usbfs_ctrltransfer ctrl;
	int ret;

	ctrl.bRequestType = USB_TYPE_VENDOR;
	ctrl.bRequest = 0xa0;
	ctrl.wValue = anchorAddress;
	ctrl.wIndex = 0;
	ctrl.wLength = count;
	ctrl.data = (uint8_t *)writeBuffer;
	ctrl.timeout = 1000;

	ret = ioctl(dev, IOCTL_USBFS_CONTROL, &ctrl);
	return (ret == count)?PICO_SUCCESS:PICO_FAILURE;
}

//////////////////////////////////////////////////////////////////////////
// Download firmware to the specified processor type
//////////////////////////////////////////////////////////////////////////

PICO_RETURNS DownloadToAnchorDevice(int dev, PicoDownloader::UsbChip chip, INTEL_HEX_RECORD *firmware) {
    UInt8 	writeVal;
    PICO_RETURNS	kr;
	
	// Select download method based on processor type
	UInt16 usbReg=0;
	switch(chip) {
		case PicoDownloader::FX:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX\n");
#endif
			usbReg=kFX_FX2_USBCS;
			break;
		case PicoDownloader::FX2:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX2\n");
#endif
			usbReg=kFX_FX2_USBCS;
			break;
		case PicoDownloader::FX1:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX1\n");
#endif
			usbReg=kFX1_FX2LP_USBCS;
			break;
		case PicoDownloader::FX2LP:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX2LP\n");
#endif
			usbReg=kFX1_FX2LP_USBCS;
			break;
		default:
#if DEBUG
			printf("DownloadToAnchorDevice: Unknown device\n");
#endif
			return PICO_FAILURE;
	}
	
    // Assert reset
    writeVal = 1;
    kr = AnchorWrite(dev, usbReg, 1, &writeVal);
    if (PICO_SUCCESS != kr) {
#if DEBUG
        printf("AnchorWrite reset returned err 0x%x!\n", kr);
#endif
      return kr;
    }
    
    
    
    // Download code using AnchorWrite
    do {
		kr=AnchorWrite(dev,firmware->Address,firmware->Length,firmware->Data);
        if (PICO_SUCCESS != kr)  {
#if DEBUG
            printf("AnchorWrite download %lx returned err 0x%x!\n", firmware->Address, kr);
#endif
            return kr;
        }
#if NARRATEDOWNLOAD
        printf("%04lx ",firmware->Address);
#endif
		// The last record has a length of 0
    } while((++firmware)->Type==0);
#if NARRATEDOWNLOAD
    printf("\n");
#endif
	
    // De-assert reset
    writeVal = 0;
    kr = AnchorWrite(dev, usbReg, 1, &writeVal);
    if (PICO_SUCCESS != kr) {
#if DEBUG
        printf("AnchorWrite run returned err 0x%x!\n", kr);
#endif
    }
    
    return kr;
}


//////////////////////////////////////////////////////////////////////////
/// threadProc is started in a new thread by the PicoDownloader constructor
/// and just calls the DownloadThread method 
/// <param name="args">Pointer to the PicoLinuxDownloader class whose
/// DownloadThread method is to be run</param>
/// <returns>Always NULL. When this function returns, its thread terminates. </returns>
//////////////////////////////////////////////////////////////////////////
void *PicoDownloader_Linux::threadProc(void *args) {
#if DEBUG
	printf("PicoDownloader::threadProc: Started\n");
#endif
	
	int err;
	
	// Make the thread cancellable so we can stop it if needed
	err = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::threadProc: Failed to set thread cancel state, error %d\n", err);
#endif
	}
	
	err = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::threadProc: Failed to set thread cancel type, error %d\n", err);
#endif
	}
	
	// Run the polling function
	if(args!=NULL) {
		((PicoDownloader_Linux *)args)->DownloadThread();
	}
	return NULL;
}


//////////////////////////////////////////////////////////////////////////
/// Start downloader.
/// For the moment, this works by polling for new USB devices every 500ms.
/// This seems to use very little CPU time but a callback-based solution
/// would be nicer. Perhaps when libusb-1.0 or openusb are released
/// we can look at this again?
//////////////////////////////////////////////////////////////////////////
void PicoDownloader_Linux::DownloadThread (void) {

	DIR *busdir, *dir;
	struct dirent *entry;
	char buspath[PATH_MAX + 1];
	char dirpath[PATH_MAX + 1];
	char filepath[PATH_MAX + 1];


	int  err;     // Return value from system calls
	long sleepTime;
	bool threadStopSet = FALSE;

	#if DEBUG
		printf("PicoDownloader::DownloadThread()\n");
	#endif	
	
		
	// Poll for new devices and download FW
	do {
		snprintf(buspath, PATH_MAX, "%s", "/dev/bus/usb");
		busdir = opendir(buspath);
		if (!busdir)
		{

			snprintf(buspath, PATH_MAX, "%s", "/proc/bus/usb");
			busdir = opendir(buspath);
			if (!busdir)
			{
				continue;
			}
		}
		while (entry = readdir(busdir)) {
			
			if (entry->d_name[0] == '.' || !strchr("0123456789", entry->d_name[strlen(entry->d_name) - 1])) 
				continue;
			
			snprintf(dirpath, PATH_MAX, "%s/%s", buspath, entry->d_name);

			dir = opendir(dirpath);
			if (!dir)
				continue;

			while (entry = readdir(dir)) {
				DeviceDescriptor device_desc;
				
				int fd, ret;

				/* Skip anything starting with a . */
				if (entry->d_name[0] == '.')
					continue;
				
				snprintf(filepath, PATH_MAX, "%s/%s", dirpath, entry->d_name);

#if DEBUG
	printf("PicoDownloader::DownloadThread::opening %s\n\n", filepath);
#endif	
				fd = open(filepath, O_RDWR);
				if (fd < 0) {
					// Don't try read-only - this will be no use
#if DEBUG
	printf("PicoDownloader::DownloadThread::trying read-only on %s\n\n", filepath);
#endif	
					// fd = open(entry->d_name, O_RDONLY);
					if (fd < 0) {
#if DEBUG
	printf("PicoDownloader::DownloadThread::failed %s\n\n", filepath);
#endif	
						continue;
					}
				}

				ret = read(fd, (void *)&device_desc, 18);
				if (ret < 0) {
					close(fd);
					continue;
				}

#if DEBUG
	printf("PicoDownloader::DownloadThread::Device %x::%x\n\n", device_desc.idVendor, device_desc.idProduct);
#endif	

				std::vector<DeviceSpec *>::const_iterator it;
				for (it = devicelist.begin(); it != devicelist.end(); it++)
				{
#if DEBUG
	printf("PicoDownloader::DownloadThread::Looking for %x::%x\n\n", (*it)->vendor, (*it)->product);
#endif	
					if (((*it)->vendor != ANY_USB_ID && (*it)->vendor != device_desc.idVendor) ||
					((*it)->product != ANY_USB_ID && (*it)->product != device_desc.idProduct) ||
					((*it)->release != ANY_USB_ID && (*it)->release != device_desc.bcdDevice) ||
					((*it)->deviceclass != ANY_USB_ID && (*it)->deviceclass != device_desc.bDeviceClass))
					{
						continue;
					}
					// We have a device, do stuff!
					DownloadToAnchorDevice(fd, (*it)->chip, (*it)->firmware);
					// We don't need to protect either initialized or
					// initialDevices by a mutex in the following  
					// as initialized can only be changed by this thread.
					if (!initialized)
					{
						initialDevices++;
					}
							
					
				}

				close(fd);

				// Check threadStop
				err = pthread_mutex_lock(threadStopMutex);
				if (err != 0)
				{
	#if DEBUG
					printf("PicoDownloader::DownloadThread(): Failed to lock mutex, error %d\n", err);
	#endif
				} 
				threadStopSet = threadStop;
				
				err = pthread_mutex_unlock(threadStopMutex);			
				if (err != 0)
				{
	#if DEBUG
					printf("PicoDownloader::DownloadThread(): Failed to unlock mutex, error %d\n", err);
	#endif
				}
				if (threadStopSet)
					break;
				
			}

			closedir(dir);
			
		}
		
		closedir(busdir);

		// We only need to protect initialized when writing, not reading, in
		// this thread.
		if (!initialized && !threadStopSet)
		{			
			err = pthread_mutex_lock(initialDeviceMutex);
			if (err != 0)
			{
		#if DEBUG
				printf("PicoDownloader::DownloadThread(): Failed to lock mutex, error %d\n", err);
		#endif
			} else	{
				initialized = TRUE;
			}
			
			err = pthread_mutex_unlock(initialDeviceMutex);
			if (err != 0)
			{
		#if DEBUG
				printf("PicoDownloader::DownloadThread(): Failed to unlock mutex, error %d\n", err);
		#endif
			}
		}
		
		sleepTime = 0;
		const unsigned long sleepStep = 100000; // Wake up every 100ms to check for threadStop
		while (!threadStopSet && (sleepTime < pollInterval))
		{
			usleep(sleepStep);
			sleepTime += sleepStep;
			err = pthread_mutex_lock(threadStopMutex);
			if (err != 0)
			{
		#if DEBUG
				printf("PicoDownloader::DownloadThread(): Failed to lock mutex, error %d\n",  err);
		#endif
			} 
			threadStopSet = threadStop;
			
			err = pthread_mutex_unlock(threadStopMutex);			
			if (err != 0)
			{
		#if DEBUG
				printf("PicoDownloader::DownloadThread(): Failed to unlock mutex, error %d\n",  err);
		#endif
			}
		}
	} while (!threadStopSet);
	#if DEBUG
	printf("PicoDownloader::DownloadThread(): Exiting\n");
	#endif

}


//////////////////////////////////////////////////////////////////////////
/// Get a count of already-connected devices to which we downloaded firmware
/// at class instantiation. Intended usage is (pseudocode):
/// <code>
/// <para>Count number of devices available to your driver (match PID,VID & DID, firmware loaded)
/// <para>Create a PicoDownloader for each PID/VID/DID combination required
/// <para>Call InitialDeviceCount until it returns >=0 (or a timeout occurs)
/// <para>Count number of available devices until it has increased by InitialDeviceCount (or a timeout occurs)
/// <para></code>
/// The timeout is required because not all devices to which we download firmware may
/// successfully reappear on the bus (for example, the user may disconnect one).
/// <returns>If we have finished the first full scan through the USB bus looking
/// for matching devices, the number of devices to which firmware was downloaded
/// on that first scan. If we have not yet finished one full scan, returns -1.</returns>
//////////////////////////////////////////////////////////////////////////
int PicoDownloader_Linux::InitialDeviceCount(void)
{
#if DEBUG
	printf("PicoDownloader::InitialDeviceCount()\n");
#endif

	int retVal;  // The value this function will return
	int err;     // Return value from system calls
	
	err = pthread_mutex_lock(initialDeviceMutex);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::InitialDeviceCount(): Failed to lock mutex, error %d\n", err);
#endif
		return -1;
	}
	retVal =  initialized ? initialDevices : -1;
	err = pthread_mutex_unlock(initialDeviceMutex);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::InitialDeviceCount(): Failed to unlock mutex, error %d\n", err);
#endif
	}
	
	return retVal;
}


//////////////////////////////////////////////////////////////////////////
/// Downloader class constructor. This should only be called by 
/// PicoDownloader::Create() which will instantiate an appropriate downloader
/// for the current platform. See PicoDownloader::Create() for parameter documentation.
//////////////////////////////////////////////////////////////////////////
PicoDownloader_Linux::PicoDownloader_Linux(std::vector<DeviceSpec *> devices) {
	int err;
	
#if DEBUG
	printf("PicoDownloader::PicoDownloader()\n");
#endif

	initialDeviceMutex = NULL;
	initialDevices = 0;
	initialized = false;
	threadStop = false;
	threadStopping = false;
	threadStopMutex = NULL;
	thread = NULL;
	
	// Validate parameters
	if (!devices.size())
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Empty device list was supplied, exiting.\n");
#endif
		return;
	}	
	
	// Save our parameters
	std::vector<DeviceSpec *>::const_iterator it;
	for (it = devices.begin(); it != devices.end(); it++)
	{
		if (!(*it)->firmware)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): No firmware was supplied, skipping.\n");
	#endif
			continue;  // Can't do anything with no firmware
		}	
		if ((*it)->chip > FX2LP)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): Invalid chip was specified, skipping.\n");
	#endif
			continue;  // Can't do anything with no chip
		}	
		if ((*it)->vendor > 0xffff && (*it)->vendor != ANY_USB_ID)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): Invalid vendor, skipping.\n");
	#endif
			continue;
		}	
		if ((*it)->product > 0xffff && (*it)->product != ANY_USB_ID)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): Invalid product, skipping.\n");
	#endif
			continue;
		}	
		if ((*it)->release > 0xffff && (*it)->release != ANY_USB_ID)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): Invalid release, skipping.\n");
	#endif
			continue;
		}	
		if ((*it)->deviceclass > 0xffff && (*it)->deviceclass != ANY_USB_ID)
		{
	#if DEBUG
			printf("PicoDownloader::PicoDownloader(): Invalid vendor, skipping.\n");
	#endif
			continue;
		}
	
		// Copy the device spec to a new struct and put it in our vector
		DeviceSpec * newDev = new DeviceSpec();
		*newDev = **it;
		devicelist.push_back(newDev);
	
	}

	
	// Set up the initial device counting
	initialDeviceMutex = new pthread_mutex_t();
	if (!initialDeviceMutex)
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Failed to create device mutex\n");
#endif
		return; // Couldn't create mutex (out of memory?)		
	}
	err = pthread_mutex_init(initialDeviceMutex, NULL);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Failed to init device mutex, error %d\n", err);
#endif
		return; // Couldn't init mutex		
	}

	
	// Setup threading
	threadStopMutex = new pthread_mutex_t();
	if (!threadStopMutex)
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Failed to create thread mutex\n");
#endif
		return; // Couldn't create mutex (out of memory?)		
	}
	err = pthread_mutex_init(threadStopMutex, NULL);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Failed to init thread mutex, error %d\n", err);
#endif
		return; // Couldn't init mutex		
	}
	
	thread = new pthread_t();
	if (!thread)
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Failed to create thread\n");
#endif
		return; // Couldn't create thread (out of memory?)		
	}

	
	// Start the polling thread
#if DEBUG
	printf("PicoDownloader::PicoDownloader: Starting thread...\n");
#endif
	err = pthread_create(thread,NULL,threadProc,this);
#if DEBUG
	if(err != 0) {
		printf("PicoDownloader::PicoDownloader: Thread failed to start, error %d\n", err);
	} else {
		printf("PicoDownloader::PicoDownloader: Thread started OK\n");
	}
#endif
}

//////////////////////////////////////////////////////////////////////////
/// Downloader class destructor. Terminates the downloading thread and
/// frees resources.
//////////////////////////////////////////////////////////////////////////
PicoDownloader_Linux::~PicoDownloader_Linux() {
	int err;
	
#if DEBUG
	printf("PicoDownloader::~PicoDownloader()\n");
#endif
	
	if (!(thread && threadStopMutex))
	{
#if DEBUG
		printf("PicoDownloader::PicoDownloader(): Can't find thread or mutex\n");
#endif
		return; 
	}
	
	// Signal the thread to stop
	err = pthread_mutex_lock (threadStopMutex);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::~PicoDownloader(): Failed to lock mutex, error %d\n", err);
#endif
	}
	// We continue into the mutex-protected code even if we couldn't lock the mutex. 
	// We'll probably do less harm trying to stop the thread nicely (even without
	// mutex lock) than cancelling it.
	threadStop = true;
	err = pthread_mutex_unlock (threadStopMutex);
	if (err != 0)
	{
#if DEBUG
		printf("PicoDownloader::~PicoDownloader(): Failed to unlock mutex, error %d\n", err);
#endif
	}
	
	// Wait for the thread to terminate
	const unsigned long waitIncrement = 1000; // Sleep in 1ms steps
	bool acknowledged = false; // Has the thread indicated that it is stopping?
	unsigned long timeout = 0;
	do {
		usleep(waitIncrement); // TODO: usleep is deprecated. Change to nanosleep.
		timeout += waitIncrement;
		err = pthread_mutex_lock (threadStopMutex);
		if (err != 0)
		{
	#if DEBUG
			printf("PicoDownloader::~PicoDownloader(): Failed to lock mutex, error %d\n", err);
	#endif
			break; // If we couldn't lock the mutex, give up on stopping the thread nicely
		}
		
		acknowledged = threadStopping;
		err = pthread_mutex_unlock (threadStopMutex);
		if (err != 0)
		{
	#if DEBUG
			printf("PicoDownloader::~PicoDownloader(): Failed to unlock mutex, error %d\n", err);
	#endif
			// Don't handle this in any way, just note it for debugging.
		}
	
		
	} while (!acknowledged && timeout < (2 * pollInterval)); // Wait up to twice the polling interval
	
	if (acknowledged) { 
	// The thread is terminating
#if DEBUG
		printf("PicoDownloader::~PicoDownloader: Joining thread...\n");
#endif
		// Ensure that the thread exits safely
		err = pthread_join(*(thread),NULL);
#if DEBUG
		if(err) {
			printf("PicoDownloader::~PicoDownloader: Joining thread...error %d\n", err);
			if(err==EINVAL) {
				printf("PicoDownloader::~PicoDownloader: Thread does not refer to a joinable thread.\n" );
			} else if(err==ESRCH) {
				printf("PicoDownloader::~PicoDownloader: Thread could not be found.\n");
			} else if(err==EDEADLK) {
				printf("PicoDownloader::~PicoDownloader: A deadlock was detected.\n");
			}
		} else {
			printf("PicoDownloader::~PicoDownloader: Joining thread...ok!\n");
		}
#endif
	
#if DEBUG
		printf("PicoDownloader::~PicoDownloader: Detaching thread...");
#endif
		err=pthread_detach(*(thread));
#if DEBUG
		if(err) {
			printf("PicoDownloader::~PicoDownloader: Detaching thread...error %d\n", err);
			if(err==EINVAL) {
				printf("PicoDownloader::~PicoDownloader: Thread does not refer to a joinable thread.\n");
			} else if(err==ESRCH) {
				printf("PicoDownloader::~PicoDownloader: Thread could not be found.\n");
			}
		} else {
			printf("ok!\n");
		}
#endif
	} else {
	// The thread didn't acknowledge that it is terminating
		// Cancel thread.
	
		
	}
	
	// Free resources
	if (thread)
		delete thread;
	if (threadStopMutex)
	{
		pthread_mutex_destroy(threadStopMutex);
		delete (threadStopMutex);
	}
	if (initialDeviceMutex){
		pthread_mutex_destroy(initialDeviceMutex);
		delete initialDeviceMutex;
	}
	
}


