/* Go IPP - IPP core protocol implementation in pure Go
 *
 * Copyright (C) 2020 and up by Alexander Pevzner (pzz@apevzner.com)
 * See LICENSE for license terms and conditions
 */

package goipp

import (
	"bytes"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"reflect"
	"strings"
	"testing"
	"time"
)

// errWriter implements io.Writer interface
//
// it accepts some first bytes and after that always returns an error
type errWriter struct{ skip int }

var _ = io.Writer(&errWriter{})

func (ewr *errWriter) Write(data []byte) (int, error) {
	if len(data) <= ewr.skip {
		ewr.skip -= len(data)
		return len(data), nil
	}

	n := ewr.skip
	ewr.skip = 0

	return n, errors.New("I/O error")
}

// errValue implements Value interface and returns
// error from its encode() and decode() methods
type errValue struct{}

var _ = Value(errValue{})

func (errValue) String() string { return "" }
func (errValue) Type() Type     { return TypeInteger }

func (errValue) encode() ([]byte, error) {
	return nil, errors.New("encode error")
}

func (errValue) decode([]byte) (Value, error) {
	return nil, errors.New("decode error")
}

// Check that err == nil
func assertNoError(t *testing.T, err error) {
	if err != nil {
		t.Errorf("%s", err)
	}
}

// Check that err != nil
func assertWithError(t *testing.T, err error) {
	if err == nil {
		t.Errorf("Error expected")
	}
}

// Check that err != nil and contains expected test
func assertErrorIs(t *testing.T, err error, s string) {
	if err == nil {
		if s != "" {
			t.Errorf("Error expected")
		}
		return
	}

	if !strings.HasPrefix(err.Error(), s) {
		t.Errorf("Error is %q, expected %q", err, s)
	}
}

// Check that value type is as specified
func assertValueType(t *testing.T, val Value, typ Type) {
	if val.Type() != typ {
		t.Errorf("%s: type is %s, must be %s", reflect.TypeOf(val).Name(), val.Type(), typ)
	}
}

func assertDataSize(t *testing.T, data []byte, size int) {
	if len(data) != size {
		t.Errorf("data size must be %d, present %d", size, len(data))
	}
}

// Check that encode() works without error and returns expected size
func assertEncodeSize(t *testing.T, encode func() ([]byte, error), size int) {
	data, err := encode()
	assertNoError(t, err)
	assertDataSize(t, data, size)
}

// Check that decode() works without error and returns expected value
func assertDecode(t *testing.T, data []byte, expected Value) {
	val, err := expected.decode(data)
	assertNoError(t, err)

	if !ValueEqual(val, expected) {
		t.Errorf("decode: expected %s, present %s", val, expected)
	}
}

// Check that decode returns error
func assertDecodeErr(t *testing.T, data []byte, val Value) {
	_, err := val.decode(data)
	if err == nil {
		t.Errorf("decode: expected error")
	}
}

// parseTime parses time given in time.Layout format.
// this function panics if time cannot be parsed
func parseTime(s string) Time {
	t, err := time.Parse(time.Layout, s)
	if err != nil {
		panic(fmt.Sprintf("parseTime(%q): %s", s, err))
	}
	return Time{t}
}

// Test String() methods for various types
func TestString(t *testing.T) {
	// Here we test that T.String() doesn't crash for
	// values out of range
	for i := -1000; i <= 10000; i++ {
		_ = Op(i).String()
		_ = Status(i).String()
		_ = Type(i).String()
		_ = Tag(i).String()
		_ = Units(i).String()
	}
}

// Test Void Value
func TestVoidValue(t *testing.T) {
	var v Void

	assertValueType(t, v, TypeVoid)
	assertEncodeSize(t, v.encode, 0)

	assertDecode(t, []byte{}, Void{})
	assertDecode(t, []byte{1, 2, 3, 4}, Void{})
}

// Test Integer Value
func TestIntegerValue(t *testing.T) {
	var v Integer

	assertValueType(t, v, TypeInteger)
	assertEncodeSize(t, v.encode, 4)

	assertDecode(t, []byte{1, 2, 3, 4}, Integer(0x01020304))
	assertDecodeErr(t, []byte{1, 2, 3}, Integer(0))
}

// Test Boolean Value
func TestBooleanValue(t *testing.T) {
	var v Boolean

	assertValueType(t, v, TypeBoolean)
	assertEncodeSize(t, v.encode, 1)

	assertDecode(t, []byte{0}, Boolean(false))
	assertDecode(t, []byte{1}, Boolean(true))
	assertDecodeErr(t, []byte{1, 2, 3}, Integer(0))
}

// Test String Value
func TestStringValue(t *testing.T) {
	var v String

	assertValueType(t, v, TypeString)
	assertEncodeSize(t, v.encode, 0)

	v = "12345"
	assertEncodeSize(t, v.encode, 5)

	assertDecode(t, []byte{}, String(""))
	assertDecode(t, []byte("hello"), String("hello"))
}

// Test Time Value
func TestDateTimeValue(t *testing.T) {
	var v Time

	assertValueType(t, v, TypeDateTime)
	assertEncodeSize(t, v.encode, 11)

	tm := time.Date(2020, 1, 13, 15, 35, 12, 300000000, time.UTC)

	v = Time{tm}
	data, _ := v.encode()

	assertDecode(t, data, Time{tm})
	assertDecodeErr(t, []byte{1, 2, 3}, Time{})
}

// Test Resolution value
func TestResolutionValue(t *testing.T) {
	v := Resolution{100, 100, UnitsDpi}

	assertValueType(t, v, TypeResolution)
	assertEncodeSize(t, v.encode, 9)

	data, _ := v.encode()
	assertDecode(t, data, v)
	assertDecodeErr(t, []byte{1, 2, 3}, Resolution{})
}

// Test Range value
func TestRangeValue(t *testing.T) {
	v := Range{100, 200}

	assertValueType(t, v, TypeRange)
	assertEncodeSize(t, v.encode, 8)

	data, _ := v.encode()
	assertDecode(t, data, v)
	assertDecodeErr(t, []byte{1, 2, 3}, Range{})
}

// Test TextWithLang value
func TestTextWithLang(t *testing.T) {
	v := TextWithLang{"ru_RU", "строка на росском языке"}

	data, err := v.encode()
	if err != nil {
		t.Errorf("(TestTextWithLang) encode(): %s", err)
	}

	v2, err := v.decode(data)
	if err != nil {
		t.Errorf("(TestTextWithLang) decode(): %s", err)
	}

	//if v != v2.(TextWithLang) {
	if !ValueEqual(v, v2) {
		t.Errorf("TestTextWithLang not the same after encode and decode")
	}
}

// Test Binary value
func TestBinaryValue(t *testing.T) {
	v := Binary([]byte("12345"))

	assertValueType(t, v, TypeBinary)
	assertEncodeSize(t, v.encode, 5)

	data, _ := v.encode()
	assertDecode(t, data, v)
}

// Test (Attributes) Equal()
func TestAttributesEqual(t *testing.T) {
	attr1 := MakeAttribute("attr1", TagInteger, Integer(1))
	attr2 := MakeAttribute("attr2", TagInteger, Integer(2))
	attr3 := MakeAttribute("attr3", TagInteger, Integer(3))

	var attrs1, attrs2 Attributes

	attrs1.Add(attr1)
	attrs1.Add(attr2)

	attrs2.Add(attr1)
	attrs2.Add(attr2)

	if !attrs1.Equal(attrs2) {
		t.Errorf("(Attributes) Equal(): failed for equal attributes")
	}

	attrs2.Add(attr3)
	if attrs1.Equal(attrs2) {
		t.Errorf("(Attributes) Equal(): failed attributes of different length")
	}

	attrs2 = attrs2[:2]

	if !attrs1.Equal(attrs2) {
		t.Errorf("(Attributes) Equal(): failed for equal attributes")
	}

	attrs2[1] = attr3
	if attrs1.Equal(attrs2) {
		t.Errorf("(Attributes) Equal(): failed attributes of different value")
	}
}

// Test (Message) Equal()
func TestMessageEqual(t *testing.T) {
	var m1, m2 Message
	// Test: Version, Code, RequestID mismatch makes messages non-equal
	m1 = Message{Version: 0, Code: 1, RequestID: 2}

	m2 = m1
	m2.Version++
	if m1.Equal(m2) {
		t.Errorf("(Message) Equal(): ignores difference in Version")
	}

	m2 = m1
	m2.Code++
	if m1.Equal(m2) {
		t.Errorf("(Message) Equal(): ignores difference in Code")
	}

	m2 = m1
	m2.RequestID++
	if m1.Equal(m2) {
		t.Errorf("(Message) Equal(): ignores difference in RequestID")
	}
}

// Test Version
func TestVersion(t *testing.T) {
	v := MakeVersion(1, 2)
	if v.Major() != 1 || v.Minor() != 2 {
		t.Errorf("Version test failed")
	}
	if v.String() != "1.2" {
		t.Errorf("(Version)String() test failed")
	}
}

// testEncodeDecodeMessage creates a quite complex message
// for Encode/Decode test
func testEncodeDecodeMessage() *Message {
	m := &Message{
		Version:   DefaultVersion,
		Code:      0x1234,
		RequestID: 0x87654321,
	}

	// Populate all groups
	m.Operation.Add(MakeAttribute("grp_operation", TagInteger,
		Integer(1)))
	m.Job.Add(MakeAttribute("grp_job", TagInteger,
		Integer(2)))
	m.Printer.Add(MakeAttribute("grp_printer", TagInteger,
		Integer(3)))
	m.Unsupported.Add(MakeAttribute("grp_unsupported", TagInteger,
		Integer(4)))
	m.Subscription.Add(MakeAttribute("grp_subscription", TagInteger,
		Integer(5)))
	m.EventNotification.Add(MakeAttribute("grp_eventnotification", TagInteger,
		Integer(6)))
	m.Resource.Add(MakeAttribute("grp_resource", TagInteger,
		Integer(7)))
	m.Document.Add(MakeAttribute("grp_document", TagInteger,
		Integer(8)))
	m.System.Add(MakeAttribute("grp_system", TagInteger,
		Integer(9)))
	m.Future11.Add(MakeAttribute("grp_future11", TagInteger,
		Integer(10)))
	m.Future12.Add(MakeAttribute("grp_future12", TagInteger,
		Integer(11)))
	m.Future13.Add(MakeAttribute("grp_future13", TagInteger,
		Integer(12)))
	m.Future14.Add(MakeAttribute("grp_future14", TagInteger,
		Integer(13)))
	m.Future15.Add(MakeAttribute("grp_future15", TagInteger,
		Integer(14)))

	// Use all possible attribute types
	m.Operation.Add(MakeAttribute("type_integer", TagInteger, Integer(123)))

	m.Operation.Add(MakeAttribute("type_boolean_t", TagBoolean, Boolean(true)))
	m.Operation.Add(MakeAttribute("type_boolean_f", TagBoolean, Boolean(false)))

	m.Operation.Add(MakeAttribute("type_void", TagUnsupportedValue, Void{}))

	m.Operation.Add(MakeAttribute("type_string_1", TagText, String("hello")))
	m.Operation.Add(MakeAttribute("type_string_2", TagText, String("")))

	m.Operation.Add(MakeAttribute("type_time_1", TagDateTime,
		parseTime("01/02 03:04:05PM '06 -0700")))
	m.Operation.Add(MakeAttribute("type_time_2", TagDateTime,
		parseTime("01/02 03:04:05PM '06 +0700")))
	m.Operation.Add(MakeAttribute("type_time_3", TagDateTime,
		parseTime("01/02 03:04:05PM '06 -0730")))

	m.Operation.Add(MakeAttribute("type_resolution_1", TagResolution,
		Resolution{123, 456, UnitsDpi}))
	m.Operation.Add(MakeAttribute("type_resolution_2", TagResolution,
		Resolution{78, 90, UnitsDpcm}))

	m.Operation.Add(MakeAttribute("type_range", TagRange,
		Range{100, 1000}))

	m.Operation.Add(MakeAttribute("type_textlang_1", TagTextLang,
		TextWithLang{"hello", "en"}))
	m.Operation.Add(MakeAttribute("type_textlang_1", TagTextLang,
		TextWithLang{"привет", "ru"}))

	return m
}

// Test Message Encode, then Decode
func TestEncodeDecode(t *testing.T) {
	m1 := testEncodeDecodeMessage()

	data, err := m1.EncodeBytes()
	assertNoError(t, err)

	m2 := &Message{}
	err = m2.DecodeBytes(data)
	assertNoError(t, err)
}

// Test encode errors
func TestEncodeErrors(t *testing.T) {
	// Attribute without name
	m := NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a := MakeAttribute("attr", TagInteger, Integer(123))
	a.Name = ""
	m.Operation.Add(a)
	err := m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Attribute without name")

	// Attribute without value
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagInteger, Integer(123))
	a.Values = nil
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Attribute without value")

	// Attribute name exceeds...
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagInteger, Integer(123))
	a.Name = strings.Repeat("x", 32767)
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertNoError(t, err)

	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagInteger, Integer(123))
	a.Name = strings.Repeat("x", 32767+1)
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Attribute name exceeds 32767 bytes")

	// Attribute value exceeds...
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagText, String(strings.Repeat("x", 32767)))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertNoError(t, err)

	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagText, String(strings.Repeat("x", 32767+1)))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Attribute value exceeds 32767 bytes")

	// Tag XXX cannot be used with value
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagJobGroup, Integer(123))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Tag job-attributes-tag cannot be used with value")

	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagMemberName, Integer(123))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Tag memberAttrName cannot be used with value")

	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagEndCollection, Integer(123))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Tag endCollection cannot be used with value")

	// Collection member without name
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagBeginCollection, Collection{
		MakeAttribute("", TagInteger, Integer(123)),
	})
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Collection member without name")

	// Tag XXX: YYY value required, ZZZ present
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagText, Integer(123))
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "Tag textWithoutLanguage: String value required, Integer present")

	// I/O error
	m = testEncodeDecodeMessage()

	data, err := m.EncodeBytes()
	assertNoError(t, err)

	for skip := 0; skip < len(data); skip++ {
		err = m.Encode(&errWriter{skip})
		assertErrorIs(t, err, "I/O error")
	}

	// encode error
	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagInteger, errValue{})
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "encode error")

	m = NewRequest(DefaultVersion, OpGetPrinterAttributes, 0x12345678)
	a = MakeAttribute("attr", TagBeginCollection, Collection{
		MakeAttribute("attr", TagInteger, errValue{}),
	})
	m.Operation.Add(a)
	err = m.Encode(ioutil.Discard)
	assertErrorIs(t, err, "encode error")
}

// Test decode errors
func TestDecodeErrors(t *testing.T) {
	var d []byte
	var err error
	var m = &Message{}

	hdr := []byte{
		0x01, 0x01, // IPP version
		0x00, 0x02, // Print-Job operation
		0x01, 0x02, 0x03, 0x04, // Request ID
	}

	body := []byte{}

	// Message truncated
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagInteger),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Message truncated at")

	d, err = testEncodeDecodeMessage().EncodeBytes()
	assertNoError(t, err)

	for i := 0; i < len(d); i++ {
		err = m.DecodeBytes(d[:i])
		assertErrorIs(t, err, "Message truncated at")
	}

	d = goodMessage1
	for i := 0; i < len(d); i++ {
		err = m.DecodeBytes(d[:i])
		assertErrorIs(t, err, "Message truncated at")
	}

	// Invalid tag 0
	d = append(hdr, 0)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Invalid tag 0 at 0x8")

	// Attribute without a group
	body = []byte{
		uint8(TagInteger),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, uint8(TagJobGroup))
	d = append(d, body...)
	err = m.DecodeBytes(d)
	assertNoError(t, err)

	err = m.DecodeBytes(append(hdr, body...))
	assertErrorIs(t, err, "Attribute without a group at")

	// Additional value without preceding attribute
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagInteger),
		0x00, 0x00, // No name
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	err = m.DecodeBytes(append(hdr, body...))
	assertErrorIs(t, err, "Additional value without preceding attribute")

	// "Unexpected tag XXX"
	for _, tag := range []Tag{TagMemberName, TagEndCollection} {
		body = []byte{
			uint8(TagJobGroup),
			uint8(TagInteger),
			0x00, 0x04, // Name length + name
			'a', 't', 't', 'r',
			0x00, 0x04, // Value length + value
			0x00, 0x00, 0x54, 0x56,
		}

		d = append(hdr, body...)
		d = append(d, uint8(tag))
		d = append(d, uint8(TagEnd))
		err = m.DecodeBytes(d)
		assertErrorIs(t, err, "Unexpected tag")
	}

	// Collection: unexpected tag XXX
	for tag := TagZero; tag.IsDelimiter(); tag++ {
		body = []byte{
			uint8(TagJobGroup),

			uint8(TagBeginCollection),
			0x00, 0x0a, // Name length + name
			'c', 'o', 'l', 'l', 'e', 'c', 't', 'i', 'o', 'n',
			0x00, 0x00, // No value

			uint8(TagMemberName),
			0x00, 0x00, // No name
			0x00, 0x06, // Value length + value
			'm', 'e', 'm', 'b', 'e', 'r',

			uint8(TagInteger),
			0x00, 0x00, // No name
			0x00, 0x04, // Value length + value
			0x00, 0x00, 0x54, 0x56,

			uint8(tag),

			uint8(TagEndCollection),
			0x00, 0x00, // No name
			0x00, 0x00, // No value
		}

		d = append(hdr, body...)
		d = append(d, uint8(TagEnd))
		err = m.DecodeBytes(d)
		assertErrorIs(t, err, "Collection: unexpected tag")
	}

	// Collection: unexpected endCollection, expected value tag
	body = []byte{
		uint8(TagJobGroup),

		uint8(TagBeginCollection),
		0x00, 0x0a, // Name length + name
		'c', 'o', 'l', 'l', 'e', 'c', 't', 'i', 'o', 'n',
		0x00, 0x00, // No value

		uint8(TagMemberName),
		0x00, 0x00, // No name
		0x00, 0x06, // Value length + value
		'm', 'e', 'm', 'b', 'e', 'r',

		uint8(TagEndCollection),
		0x00, 0x00, // No name
		0x00, 0x00, // No value

		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Collection: unexpected endCollection, expected value tag")

	// Collection: unexpected memberAttrName, expected value tag
	body = []byte{
		uint8(TagJobGroup),

		uint8(TagBeginCollection),
		0x00, 0x0a, // Name length + name
		'c', 'o', 'l', 'l', 'e', 'c', 't', 'i', 'o', 'n',
		0x00, 0x00, // No value

		uint8(TagMemberName),
		0x00, 0x00, // No name
		0x00, 0x07, // Value length + value
		'm', 'e', 'm', 'b', 'e', 'r', '1',

		uint8(TagMemberName),
		0x00, 0x00, // No name
		0x00, 0x07, // Value length + value
		'm', 'e', 'm', 'b', 'e', 'r', '2',

		uint8(TagInteger),
		0x00, 0x00, // No name
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,

		uint8(TagEndCollection),
		0x00, 0x00, // No name
		0x00, 0x00, // No value

		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Collection: unexpected memberAttrName, expected value tag")

	// Collection: memberAttrName value is empty
	body = []byte{
		uint8(TagJobGroup),

		uint8(TagBeginCollection),
		0x00, 0x0a, // Name length + name
		'c', 'o', 'l', 'l', 'e', 'c', 't', 'i', 'o', 'n',
		0x00, 0x00, // No value

		uint8(TagMemberName),
		0x00, 0x00, // No name
		0x00, 0x00, // No value

		uint8(TagInteger),
		0x00, 0x00, // No name
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,

		uint8(TagEndCollection),
		0x00, 0x00, // No name
		0x00, 0x00, // No value

		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Collection: memberAttrName value is empty")

	// Collection: unexpected integer, expected memberAttrName
	body = []byte{
		uint8(TagJobGroup),

		uint8(TagBeginCollection),
		0x00, 0x0a, // Name length + name
		'c', 'o', 'l', 'l', 'e', 'c', 't', 'i', 'o', 'n',
		0x00, 0x00, // No value

		//uint8(TagMemberName),
		//0x00, 0x00, // No name
		//0x00, 0x06, // Value length + value
		//'m', 'e', 'm', 'b', 'e', 'r',

		uint8(TagInteger),
		0x00, 0x00, // No name
		0x00, 0x04, // Value length + value
		0x00, 0x00, 0x54, 0x56,

		uint8(TagEndCollection),
		0x00, 0x00, // No name
		0x00, 0x00, // No value

		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Collection: unexpected integer, expected memberAttrName")
}

// Test errors in decoding values
func TestDecodeValueErrors(t *testing.T) {
	var d []byte
	var err error
	var m = &Message{}

	hdr := []byte{
		0x01, 0x01, // IPP version
		0x00, 0x02, // Print-Job operation
		0x01, 0x02, 0x03, 0x04, // Request ID
	}

	body := []byte{}

	// integer: value must be 4 bytes
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagInteger),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "integer: value must be 4 bytes")

	// boolean: value must be 1 byte
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagBoolean),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "boolean: value must be 1 byte")

	// dateTime: value must be 11 bytes
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagDateTime),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "dateTime: value must be 11 bytes")

	// dateTime: bad XXX
	var datetest = []struct {
		in  []byte
		err string
	}{
		//      year        month day   hour  min   sec   s/10
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0x32, 0x05, '+', 0x04, 0x00},
			""},
		{[]byte{0x07, 0xe7, 0xff, 0x15, 0x11, 0x23, 0x32, 0x05, '+', 0x04, 0x00},
			"dateTime: bad month 255"},
		{[]byte{0x07, 0xe7, 0x02, 0xff, 0x11, 0x23, 0x32, 0x05, '+', 0x04, 0x00},
			"dateTime: bad day 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0xff, 0x23, 0x32, 0x05, '+', 0x04, 0x00},
			"dateTime: bad hours 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0xff, 0x32, 0x05, '+', 0x04, 0x00},
			"dateTime: bad minutes 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0xff, 0x05, '+', 0x04, 0x00},
			"dateTime: bad seconds 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0x32, 0xff, '+', 0x04, 0x00},
			"dateTime: bad deciseconds 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0x32, 0x05, '?', 0x04, 0x00},
			"dateTime: bad UTC sign"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0x32, 0x05, '-', 0xff, 0x00},
			"dateTime: bad UTC hours 255"},
		{[]byte{0x07, 0xe7, 0x02, 0x15, 0x11, 0x23, 0x32, 0x05, '-', 0x04, 0xff},
			"dateTime: bad UTC minutes 255"},
	}

	for _, test := range datetest {
		body = []byte{
			uint8(TagJobGroup),
			uint8(TagDateTime),
			0x00, 0x04, // Name length + name
			'a', 't', 't', 'r',
			0x00, 0x0b, // Value length + value
		}

		d = append(hdr, body...)
		d = append(d, test.in...)
		d = append(d, uint8(TagEnd))

		err = m.DecodeBytes(d)
		assertErrorIs(t, err, test.err)
	}

	// resolution: value must be 9 bytes
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagResolution),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "resolution: value must be 9 bytes ")

	// rangeOfInteger: value must be 8 bytes
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagRange),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "rangeOfInteger: value must be 8 bytes")

	// textWithLanguage: truncated language length
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagTextLang),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x01, // Value length
		0x00,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "textWithLanguage: truncated language length")

	// textWithLanguage: truncated language name
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagTextLang),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length
		0x00, 0x02, // Language length
		'e',
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "textWithLanguage: truncated language name")

	// textWithLanguage: truncated text length
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagTextLang),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x05, // Value length
		0x00, 0x02, // Language length
		'e', 'n', // Language name
		0x00, // Text length
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "textWithLanguage: truncated text length")

	// textWithLanguage: truncated text string
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagTextLang),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x09, // Value length
		0x00, 0x02, // Language length
		'e', 'n', // Language name
		0x00, 0x05, // Text length
		'h', 'e', 'l',
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "textWithLanguage: truncated text string")

	// textWithLanguage: extra 2 bytes at the end of value
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagTextLang),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x0d, // Value length
		0x00, 0x02, // Language length
		'e', 'n', // Language name
		0x00, 0x05, // Text length
		'h', 'e', 'l', 'l', 'o', // Test string
		'?', '?', // Extra 2 bytes
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "textWithLanguage: extra 2 bytes at the end of value")
}

// Test TagExtension
func TestTagExtension(t *testing.T) {
	// Ensure extension tag encodes and decodes well
	m1 := NewResponse(DefaultVersion, StatusOk, 0x12345678)
	m1.Operation.Add(MakeAttribute("attr", 0x12345678,
		Binary{1, 2, 3, 4, 5}))

	data, err := m1.EncodeBytes()
	assertNoError(t, err)

	m2 := Message{}
	err = m2.DecodeBytes(data)
	assertNoError(t, err)

	if !m1.Equal(m2) {
		t.Errorf("Message is not the same after encoding and decoding")
	}

	// Tag can't exceed 0x7fffffff, check that encoder validates it
	m1 = NewResponse(DefaultVersion, StatusOk, 0x12345678)
	tmp := uint32(0x81234567)
	m1.Operation.Add(MakeAttribute("attr", Tag(tmp),
		Binary{1, 2, 3, 4, 5}))

	_, err = m1.EncodeBytes()
	assertErrorIs(t, err, "Tag 0x81234567 exceeds extension tag range")

	// Now prepare to decoder tests
	var d []byte
	var m = &Message{}

	hdr := []byte{
		0x01, 0x01, // IPP version
		0x00, 0x02, // Print-Job operation
		0x01, 0x02, 0x03, 0x04, // Request ID
	}

	body := []byte{}

	// Extension tag truncated
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagExtension),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x03, // Value length + value
		0x00, 0x54, 0x56,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Extension tag truncated")

	// Extension tag out of range
	body = []byte{
		uint8(TagJobGroup),
		uint8(TagExtension),
		0x00, 0x04, // Name length + name
		'a', 't', 't', 'r',
		0x00, 0x08, // Value length + value
		0xff, 0xff, 0xff, 0xff, 0, 0, 0, 0,
		uint8(TagEnd),
	}

	d = append(hdr, body...)
	err = m.DecodeBytes(d)
	assertErrorIs(t, err, "Extension tag out of range")
}

// Test message decoding
func testDecode(t *testing.T, data []byte, opt DecoderOptions,
	mustFail, mustEncode bool) {

	var m Message
	err := m.DecodeEx(bytes.NewBuffer(data), opt)

	if mustFail {
		assertWithError(t, err)
	} else {
		assertNoError(t, err)
	}

	if err != nil {
		return
	}

	//m.Print(os.Stdout, true)

	if !m.Equal(m) {
		t.Errorf("Message is not equal to itself")
	}

	if mustEncode {
		buf, err := m.EncodeBytes()
		assertNoError(t, err)

		if !bytes.Equal(buf, data) {
			t.Errorf("Message is not the same after decoding and encoding")
		}
	}

	// We can't test a lot of (*Message) Print(), so lets test
	// at least that it doesn't hand
	m.Print(ioutil.Discard, true)
}

func TestDecodeGoodMessage1(t *testing.T) {
	testDecode(t, goodMessage1, DecoderOptions{}, false, true)
}

func TestDecodeGoodMessage2(t *testing.T) {
	testDecode(t, goodMessage2, DecoderOptions{}, false, true)
}

func TestDecodeBadMessage1(t *testing.T) {
	testDecode(t, badMessage1, DecoderOptions{}, true, false)
	testDecode(t, badMessage1,
		DecoderOptions{EnableWorkarounds: true}, false, false)
}

func TestDecodeHPOfficeJetPro8730(t *testing.T) {
	testDecode(t, attrsHPOfficeJetPro8730, DecoderOptions{}, false, true)
}

func TestDecodePantumM7300FDW(t *testing.T) {
	testDecode(t, attrsPantumM7300FDW,
		DecoderOptions{EnableWorkarounds: false}, true, false)

	testDecode(t, attrsPantumM7300FDW,
		DecoderOptions{EnableWorkarounds: true}, false, false)
}

// ------------------------ Test Data ------------------------
// The good message - 1
var goodMessage1 = []byte{
	0x01, 0x01, // IPP version
	0x00, 0x02, // Print-Job operation
	0x00, 0x00, 0x00, 0x01, // Request ID

	uint8(TagOperationGroup),

	uint8(TagCharset),
	0x00, 0x12, // Name length + name
	'a', 't', 't', 'r', 'i', 'b', 'u', 't', 'e', 's', '-',
	'c', 'h', 'a', 'r', 's', 'e', 't',
	0x00, 0x05, // Value length + value
	'u', 't', 'f', '-', '8',

	uint8(TagLanguage),
	0x00, 0x1b, // Name length + name
	'a', 't', 't', 'r', 'i', 'b', 'u', 't', 'e', 's', '-',
	'n', 'a', 't', 'u', 'r', 'a', 'l', '-', 'l', 'a', 'n',
	'g', 'u', 'a', 'g', 'e',
	0x00, 0x02, // Value length + value
	'e', 'n',

	uint8(TagURI),
	0x00, 0x0b, // Name length + name
	'p', 'r', 'i', 'n', 't', 'e', 'r', '-', 'u', 'r', 'i',
	0x00, 0x1c, // Value length + value
	'i', 'p', 'p', ':', '/', '/', 'l', 'o', 'c', 'a', 'l',
	'h', 'o', 's', 't', '/', 'p', 'r', 'i', 'n', 't', 'e',
	'r', 's', '/', 'f', 'o', 'o',

	uint8(TagJobGroup),

	uint8(TagBeginCollection),
	0x00, 0x09, // Name length + name
	'm', 'e', 'd', 'i', 'a', '-', 'c', 'o', 'l',
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0a, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 's', 'i', 'z', 'e',

	uint8(TagBeginCollection),
	0x00, 0x00, // Name length + name
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'x', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',

	uint8(TagInteger),
	0x00, 0x00, // No name
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x54, 0x56,

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'y', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',

	uint8(TagInteger),
	0x00, 0x00, // No name
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x6d, 0x24,

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 'c', 'o', 'l', 'o', 'r',

	uint8(TagKeyword),
	0x00, 0x00, // No name
	0x00, 0x04, // Value length + value
	'b', 'l', 'u', 'e',

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0a, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 't', 'y', 'p', 'e',

	uint8(TagKeyword),
	0x00, 0x00, // No name
	0x00, 0x05, // Value length + value
	'p', 'l', 'a', 'i', 'n',

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagBeginCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0a, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 's', 'i', 'z', 'e',

	uint8(TagBeginCollection),
	0x00, 0x00, // Name length + name
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'x', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',

	uint8(TagInteger),
	0x00, 0x00, // No name
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x52, 0x08,

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'y', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',

	uint8(TagInteger),
	0x00, 0x00, // No name
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x74, 0x04,

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0b, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 'c', 'o', 'l', 'o', 'r',

	uint8(TagKeyword),
	0x00, 0x00, // No name
	0x00, 0x05, // Value length + value
	'p', 'l', 'a', 'i', 'd',

	uint8(TagMemberName),
	0x00, 0x00, // No name
	0x00, 0x0a, // Value length + value
	'm', 'e', 'd', 'i', 'a', '-', 't', 'y', 'p', 'e',

	uint8(TagKeyword),
	0x00, 0x00, // No name
	0x00, 0x06, // Value length + value
	'g', 'l', 'o', 's', 's', 'y',

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagEnd),
}

// The good message - 2
var goodMessage2 = []byte{
	0x01, 0x01, // IPP version
	0x00, 0x02, // Print-Job operation
	0x00, 0x00, 0x00, 0x01, // Request ID

	uint8(TagOperationGroup),

	uint8(TagInteger),
	0x00, 0x1f, // Name length + name
	'n', 'o', 't', 'i', 'f', 'y', '-', 'l', 'e', 'a', 's', 'e',
	'-', 'd', 'u', 'r', 'a', 't', 'i', 'o', 'n', '-', 's', 'u',
	'p', 'p', 'o', 'r', 't', 'e', 'd',
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x00, 0x01,

	uint8(TagRange),
	0x00, 0x00, // No name
	0x00, 0x08, // Value length + value
	0x00, 0x00, 0x00, 0x10,
	0x00, 0x00, 0x00, 0x20,

	uint8(TagEnd),
}

// The bad message - 1
//
// This message violates IPP encoding rules: instead of
// using TagMemberName it uses named attributes
//
// It must not decode normally, but must decode with workarounds
var badMessage1 = []byte{
	0x01, 0x01, // IPP version */
	0x00, 0x02, // Print-Job operation */
	0x00, 0x00, 0x00, 0x01, // Request ID */

	uint8(TagOperationGroup),

	uint8(TagCharset),

	0x00, 0x12, // Name length + name
	'a', 't', 't', 'r', 'i', 'b', 'u', 't', 'e', 's', '-',
	'c', 'h', 'a', 'r', 's', 'e', 't',
	0x00, 0x05, // Value length + value
	'u', 't', 'f', '-', '8',

	uint8(TagLanguage),
	0x00, 0x1b, // Name length + name
	'a', 't', 't', 'r', 'i', 'b', 'u', 't', 'e', 's', '-',
	'n', 'a', 't', 'u', 'r', 'a', 'l', '-', 'l', 'a', 'n',
	'g', 'u', 'a', 'g', 'e',
	0x00, 0x02, // Value length + value
	'e', 'n',

	uint8(TagURI),
	0x00, 0x0b, // Name length + name
	'p', 'r', 'i', 'n', 't', 'e', 'r', '-', 'u', 'r', 'i',
	0x00, 0x1c, // Value length + value
	'i', 'p', 'p', ':', '/', '/', 'l', 'o', 'c', 'a', 'l',
	'h', 'o', 's', 't', '/', 'p', 'r', 'i', 'n', 't', 'e',
	'r', 's', '/', 'f', 'o', 'o',

	uint8(TagJobGroup),

	uint8(TagBeginCollection),
	0x00, 0x09, // Name length + name
	'm', 'e', 'd', 'i', 'a', '-', 'c', 'o', 'l',
	0x00, 0x00, // No value

	uint8(TagBeginCollection),
	0x00, 0x0a, // Name length + name
	'm', 'e', 'd', 'i', 'a', '-', 's', 'i', 'z', 'e',
	0x00, 0x00, // No value

	uint8(TagInteger),
	0x00, 0x0b, // Name length + name
	'x', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x54, 0x56,

	uint8(TagInteger),
	0x00, 0x0b, // Name length + name
	'y', '-', 'd', 'i', 'm', 'e', 'n', 's', 'i', 'o', 'n',
	0x00, 0x04, // Value length + value
	0x00, 0x00, 0x6d, 0x24,

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagEndCollection),
	0x00, 0x00, // No name
	0x00, 0x00, // No value

	uint8(TagEnd),
}

// The big real example, Get-Printer-Attributes output
// from the HP OfficeJet Pro 8730
var attrsHPOfficeJetPro8730 = []byte{
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x47, 0x00, 0x12, 0x61, 0x74, 0x74, 0x72,
	0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x2d, 0x63, 0x68, 0x61, 0x72, 0x73, 0x65, 0x74, 0x00, 0x05,
	0x75, 0x74, 0x66, 0x2d, 0x38, 0x48, 0x00, 0x1b, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
	0x65, 0x73, 0x2d, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x61, 0x6c, 0x2d, 0x6c, 0x61, 0x6e, 0x67, 0x75,
	0x61, 0x67, 0x65, 0x00, 0x02, 0x65, 0x6e, 0x04, 0x45, 0x00, 0x15, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x75, 0x72, 0x69, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x19, 0x69, 0x70, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73,
	0x74, 0x2f, 0x69, 0x70, 0x70, 0x2f, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x45, 0x00, 0x00, 0x00, 0x1e,
	0x69, 0x70, 0x70, 0x73, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74,
	0x3a, 0x34, 0x34, 0x33, 0x2f, 0x69, 0x70, 0x70, 0x2f, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x44, 0x00,
	0x16, 0x75, 0x72, 0x69, 0x2d, 0x73, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x2d, 0x73, 0x75,
	0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x6e, 0x6f, 0x6e, 0x65, 0x44, 0x00, 0x00,
	0x00, 0x03, 0x74, 0x6c, 0x73, 0x44, 0x00, 0x1c, 0x75, 0x72, 0x69, 0x2d, 0x61, 0x75, 0x74, 0x68,
	0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f,
	0x72, 0x74, 0x65, 0x64, 0x00, 0x14, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67,
	0x2d, 0x75, 0x73, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x14, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x2d, 0x75, 0x73, 0x65, 0x72, 0x2d, 0x6e,
	0x61, 0x6d, 0x65, 0x44, 0x00, 0x25, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x65,
	0x74, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x2d, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65,
	0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x11, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x2d, 0x77, 0x69, 0x66, 0x69, 0x2d, 0x73, 0x73, 0x69, 0x64, 0x44, 0x00,
	0x00, 0x00, 0x15, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x77, 0x69, 0x66, 0x69, 0x2d,
	0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x42, 0x00, 0x11, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x77, 0x69, 0x66, 0x69, 0x2d, 0x73, 0x73, 0x69, 0x64, 0x00, 0x09, 0x41, 0x31,
	0x2d, 0x35, 0x45, 0x46, 0x33, 0x37, 0x31, 0x23, 0x00, 0x12, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x2d, 0x77, 0x69, 0x66, 0x69, 0x2d, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x04, 0x42, 0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d,
	0x65, 0x00, 0x08, 0x48, 0x50, 0x30, 0x38, 0x43, 0x32, 0x32, 0x39, 0x41, 0x00, 0x10, 0x70, 0x72,
	0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6c, 0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x00,
	0x45, 0x00, 0x11, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6d, 0x6f, 0x72, 0x65, 0x2d,
	0x69, 0x6e, 0x66, 0x6f, 0x00, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63,
	0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x2f, 0x23, 0x68, 0x49, 0x64, 0x2d, 0x70, 0x67, 0x41, 0x69,
	0x72, 0x50, 0x72, 0x69, 0x6e, 0x74, 0x41, 0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x2d, 0x69, 0x6e, 0x66, 0x6f, 0x00, 0x1e, 0x48, 0x50, 0x20, 0x4f, 0x66, 0x66, 0x69, 0x63, 0x65,
	0x4a, 0x65, 0x74, 0x20, 0x50, 0x72, 0x6f, 0x20, 0x38, 0x37, 0x33, 0x30, 0x20, 0x5b, 0x30, 0x38,
	0x43, 0x32, 0x32, 0x39, 0x5d, 0x42, 0x00, 0x13, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d,
	0x64, 0x6e, 0x73, 0x2d, 0x73, 0x64, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x1e, 0x48, 0x50, 0x20,
	0x4f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4a, 0x65, 0x74, 0x20, 0x50, 0x72, 0x6f, 0x20, 0x38, 0x37,
	0x33, 0x30, 0x20, 0x5b, 0x30, 0x38, 0x43, 0x32, 0x32, 0x39, 0x5d, 0x41, 0x00, 0x16, 0x70, 0x72,
	0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x61, 0x6e, 0x64, 0x2d, 0x6d,
	0x6f, 0x64, 0x65, 0x6c, 0x00, 0x15, 0x48, 0x50, 0x20, 0x4f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4a,
	0x65, 0x74, 0x20, 0x50, 0x72, 0x6f, 0x20, 0x38, 0x37, 0x33, 0x30, 0x23, 0x00, 0x0d, 0x70, 0x72,
	0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x03, 0x44, 0x00, 0x15, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x74, 0x61, 0x74,
	0x65, 0x2d, 0x72, 0x65, 0x61, 0x73, 0x6f, 0x6e, 0x73, 0x00, 0x1a, 0x77, 0x69, 0x66, 0x69, 0x2d,
	0x6e, 0x6f, 0x74, 0x2d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x65, 0x64, 0x2d, 0x72,
	0x65, 0x70, 0x6f, 0x72, 0x74, 0x41, 0x00, 0x15, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d,
	0x73, 0x74, 0x61, 0x74, 0x65, 0x2d, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x00, 0x00, 0x44,
	0x00, 0x16, 0x69, 0x70, 0x70, 0x2d, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x73,
	0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x03, 0x31, 0x2e, 0x30, 0x44, 0x00, 0x00,
	0x00, 0x03, 0x31, 0x2e, 0x31, 0x44, 0x00, 0x00, 0x00, 0x03, 0x32, 0x2e, 0x30, 0x23, 0x00, 0x14,
	0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f,
	0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x04, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x23, 0x00, 0x00, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x39, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x09, 0x23, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0a, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0b,
	0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x06, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x23, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x40, 0x29, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x40, 0x2a, 0x23, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x07, 0x23,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x3b, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x3c, 0x47, 0x00, 0x12, 0x63, 0x68, 0x61, 0x72, 0x73, 0x65, 0x74, 0x2d, 0x63, 0x6f, 0x6e, 0x66,
	0x69, 0x67, 0x75, 0x72, 0x65, 0x64, 0x00, 0x05, 0x75, 0x74, 0x66, 0x2d, 0x38, 0x47, 0x00, 0x11,
	0x63, 0x68, 0x61, 0x72, 0x73, 0x65, 0x74, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x05, 0x75, 0x74, 0x66, 0x2d, 0x38, 0x48, 0x00, 0x1b, 0x6e, 0x61, 0x74, 0x75, 0x72,
	0x61, 0x6c, 0x2d, 0x6c, 0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x2d, 0x63, 0x6f, 0x6e, 0x66,
	0x69, 0x67, 0x75, 0x72, 0x65, 0x64, 0x00, 0x02, 0x65, 0x6e, 0x48, 0x00, 0x24, 0x67, 0x65, 0x6e,
	0x65, 0x72, 0x61, 0x74, 0x65, 0x64, 0x2d, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x61, 0x6c, 0x2d, 0x6c,
	0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x02, 0x65, 0x6e, 0x48, 0x00, 0x00, 0x00, 0x02, 0x66, 0x72, 0x48, 0x00, 0x00, 0x00,
	0x02, 0x64, 0x65, 0x48, 0x00, 0x00, 0x00, 0x02, 0x65, 0x73, 0x48, 0x00, 0x00, 0x00, 0x02, 0x69,
	0x74, 0x48, 0x00, 0x00, 0x00, 0x02, 0x73, 0x76, 0x48, 0x00, 0x00, 0x00, 0x02, 0x64, 0x61, 0x48,
	0x00, 0x00, 0x00, 0x02, 0x6e, 0x6f, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6e, 0x6c, 0x48, 0x00, 0x00,
	0x00, 0x02, 0x66, 0x69, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6a, 0x61, 0x48, 0x00, 0x00, 0x00, 0x02,
	0x70, 0x74, 0x48, 0x00, 0x00, 0x00, 0x02, 0x70, 0x6c, 0x48, 0x00, 0x00, 0x00, 0x02, 0x74, 0x72,
	0x48, 0x00, 0x00, 0x00, 0x05, 0x7a, 0x68, 0x2d, 0x74, 0x77, 0x48, 0x00, 0x00, 0x00, 0x05, 0x7a,
	0x68, 0x2d, 0x63, 0x6e, 0x48, 0x00, 0x00, 0x00, 0x02, 0x72, 0x75, 0x48, 0x00, 0x00, 0x00, 0x02,
	0x63, 0x73, 0x48, 0x00, 0x00, 0x00, 0x02, 0x68, 0x75, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6b, 0x6f,
	0x48, 0x00, 0x00, 0x00, 0x02, 0x68, 0x65, 0x48, 0x00, 0x00, 0x00, 0x02, 0x65, 0x6c, 0x48, 0x00,
	0x00, 0x00, 0x02, 0x61, 0x72, 0x48, 0x00, 0x00, 0x00, 0x02, 0x62, 0x67, 0x48, 0x00, 0x00, 0x00,
	0x02, 0x68, 0x72, 0x48, 0x00, 0x00, 0x00, 0x02, 0x72, 0x6f, 0x48, 0x00, 0x00, 0x00, 0x02, 0x73,
	0x6b, 0x48, 0x00, 0x00, 0x00, 0x02, 0x73, 0x6c, 0x48, 0x00, 0x23, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x6c, 0x61, 0x6e, 0x67, 0x75,
	0x61, 0x67, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x02,
	0x65, 0x6e, 0x48, 0x00, 0x00, 0x00, 0x02, 0x66, 0x72, 0x48, 0x00, 0x00, 0x00, 0x02, 0x64, 0x65,
	0x48, 0x00, 0x00, 0x00, 0x02, 0x65, 0x73, 0x48, 0x00, 0x00, 0x00, 0x02, 0x69, 0x74, 0x48, 0x00,
	0x00, 0x00, 0x02, 0x73, 0x76, 0x48, 0x00, 0x00, 0x00, 0x02, 0x64, 0x61, 0x48, 0x00, 0x00, 0x00,
	0x02, 0x6e, 0x6f, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6e, 0x6c, 0x48, 0x00, 0x00, 0x00, 0x02, 0x66,
	0x69, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6a, 0x61, 0x48, 0x00, 0x00, 0x00, 0x02, 0x70, 0x74, 0x48,
	0x00, 0x00, 0x00, 0x02, 0x70, 0x6c, 0x48, 0x00, 0x00, 0x00, 0x02, 0x74, 0x72, 0x48, 0x00, 0x00,
	0x00, 0x05, 0x7a, 0x68, 0x2d, 0x74, 0x77, 0x48, 0x00, 0x00, 0x00, 0x05, 0x7a, 0x68, 0x2d, 0x63,
	0x6e, 0x48, 0x00, 0x00, 0x00, 0x02, 0x72, 0x75, 0x48, 0x00, 0x00, 0x00, 0x02, 0x63, 0x73, 0x48,
	0x00, 0x00, 0x00, 0x02, 0x68, 0x75, 0x48, 0x00, 0x00, 0x00, 0x02, 0x6b, 0x6f, 0x48, 0x00, 0x00,
	0x00, 0x02, 0x68, 0x65, 0x48, 0x00, 0x00, 0x00, 0x02, 0x65, 0x6c, 0x48, 0x00, 0x00, 0x00, 0x02,
	0x61, 0x72, 0x48, 0x00, 0x00, 0x00, 0x02, 0x62, 0x67, 0x48, 0x00, 0x00, 0x00, 0x02, 0x68, 0x72,
	0x48, 0x00, 0x00, 0x00, 0x02, 0x72, 0x6f, 0x48, 0x00, 0x00, 0x00, 0x02, 0x73, 0x6b, 0x48, 0x00,
	0x00, 0x00, 0x02, 0x73, 0x6c, 0x45, 0x00, 0x13, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d,
	0x73, 0x74, 0x72, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x75, 0x72, 0x69, 0x00, 0x25, 0x68, 0x74, 0x74,
	0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x2f, 0x69, 0x70,
	0x70, 0x2f, 0x66, 0x69, 0x6c, 0x65, 0x73, 0x2f, 0x65, 0x6e, 0x2e, 0x73, 0x74, 0x72, 0x69, 0x6e,
	0x67, 0x73, 0x49, 0x00, 0x17, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x66, 0x6f,
	0x72, 0x6d, 0x61, 0x74, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x18, 0x61, 0x70,
	0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x6f, 0x63, 0x74, 0x65, 0x74, 0x2d,
	0x73, 0x74, 0x72, 0x65, 0x61, 0x6d, 0x49, 0x00, 0x19, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
	0x74, 0x2d, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
	0x65, 0x64, 0x00, 0x16, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f,
	0x76, 0x6e, 0x64, 0x2e, 0x68, 0x70, 0x2d, 0x50, 0x43, 0x4c, 0x49, 0x00, 0x00, 0x00, 0x18, 0x61,
	0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x76, 0x6e, 0x64, 0x2e, 0x68,
	0x70, 0x2d, 0x50, 0x43, 0x4c, 0x58, 0x4c, 0x49, 0x00, 0x00, 0x00, 0x16, 0x61, 0x70, 0x70, 0x6c,
	0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x70, 0x6f, 0x73, 0x74, 0x73, 0x63, 0x72, 0x69,
	0x70, 0x74, 0x49, 0x00, 0x00, 0x00, 0x0f, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69,
	0x6f, 0x6e, 0x2f, 0x70, 0x64, 0x66, 0x49, 0x00, 0x00, 0x00, 0x0a, 0x69, 0x6d, 0x61, 0x67, 0x65,
	0x2f, 0x6a, 0x70, 0x65, 0x67, 0x49, 0x00, 0x00, 0x00, 0x10, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
	0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x50, 0x43, 0x4c, 0x6d, 0x49, 0x00, 0x00, 0x00, 0x09, 0x69,
	0x6d, 0x61, 0x67, 0x65, 0x2f, 0x75, 0x72, 0x66, 0x49, 0x00, 0x00, 0x00, 0x10, 0x69, 0x6d, 0x61,
	0x67, 0x65, 0x2f, 0x70, 0x77, 0x67, 0x2d, 0x72, 0x61, 0x73, 0x74, 0x65, 0x72, 0x49, 0x00, 0x00,
	0x00, 0x18, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x6f, 0x63,
	0x74, 0x65, 0x74, 0x2d, 0x73, 0x74, 0x72, 0x65, 0x61, 0x6d, 0x41, 0x00, 0x21, 0x64, 0x6f, 0x63,
	0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2d, 0x76, 0x65, 0x72,
	0x73, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x05,
	0x50, 0x43, 0x4c, 0x35, 0x63, 0x41, 0x00, 0x00, 0x00, 0x05, 0x50, 0x43, 0x4c, 0x58, 0x4c, 0x41,
	0x00, 0x00, 0x00, 0x04, 0x50, 0x53, 0x2f, 0x33, 0x41, 0x00, 0x00, 0x00, 0x0c, 0x4e, 0x41, 0x54,
	0x49, 0x56, 0x45, 0x4f, 0x46, 0x46, 0x49, 0x43, 0x45, 0x41, 0x00, 0x00, 0x00, 0x07, 0x50, 0x44,
	0x46, 0x2f, 0x31, 0x2e, 0x37, 0x41, 0x00, 0x00, 0x00, 0x07, 0x50, 0x43, 0x4c, 0x33, 0x47, 0x55,
	0x49, 0x41, 0x00, 0x00, 0x00, 0x04, 0x50, 0x43, 0x4c, 0x33, 0x41, 0x00, 0x00, 0x00, 0x03, 0x50,
	0x4a, 0x4c, 0x41, 0x00, 0x00, 0x00, 0x09, 0x41, 0x75, 0x74, 0x6f, 0x6d, 0x61, 0x74, 0x69, 0x63,
	0x41, 0x00, 0x00, 0x00, 0x04, 0x4a, 0x50, 0x45, 0x47, 0x41, 0x00, 0x00, 0x00, 0x04, 0x50, 0x43,
	0x4c, 0x4d, 0x41, 0x00, 0x00, 0x00, 0x0b, 0x41, 0x70, 0x70, 0x6c, 0x65, 0x52, 0x61, 0x73, 0x74,
	0x65, 0x72, 0x41, 0x00, 0x00, 0x00, 0x09, 0x50, 0x57, 0x47, 0x52, 0x61, 0x73, 0x74, 0x65, 0x72,
	0x22, 0x00, 0x19, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x69, 0x73, 0x2d, 0x61, 0x63,
	0x63, 0x65, 0x70, 0x74, 0x69, 0x6e, 0x67, 0x2d, 0x6a, 0x6f, 0x62, 0x73, 0x00, 0x01, 0x01, 0x21,
	0x00, 0x10, 0x71, 0x75, 0x65, 0x75, 0x65, 0x64, 0x2d, 0x6a, 0x6f, 0x62, 0x2d, 0x63, 0x6f, 0x75,
	0x6e, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x16, 0x70, 0x64, 0x6c, 0x2d, 0x6f,
	0x76, 0x65, 0x72, 0x72, 0x69, 0x64, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x09, 0x61, 0x74, 0x74, 0x65, 0x6d, 0x70, 0x74, 0x65, 0x64, 0x21, 0x00, 0x0f, 0x70,
	0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x75, 0x70, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x00, 0x04,
	0x00, 0x00, 0x07, 0xdf, 0x31, 0x00, 0x14, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x63,
	0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x00, 0x0b, 0x07, 0xe4, 0x01,
	0x13, 0x0b, 0x17, 0x17, 0x00, 0x2b, 0x00, 0x00, 0x44, 0x00, 0x15, 0x63, 0x6f, 0x6d, 0x70, 0x72,
	0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x04, 0x6e, 0x6f, 0x6e, 0x65, 0x44, 0x00, 0x00, 0x00, 0x07, 0x64, 0x65, 0x66, 0x6c, 0x61,
	0x74, 0x65, 0x44, 0x00, 0x00, 0x00, 0x04, 0x67, 0x7a, 0x69, 0x70, 0x22, 0x00, 0x0f, 0x63, 0x6f,
	0x6c, 0x6f, 0x72, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x01, 0x01,
	0x44, 0x00, 0x21, 0x6a, 0x6f, 0x62, 0x2d, 0x63, 0x72, 0x65, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d,
	0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f,
	0x72, 0x74, 0x65, 0x64, 0x00, 0x06, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00,
	0x0a, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x44, 0x00, 0x00, 0x00, 0x0e,
	0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x44, 0x00,
	0x00, 0x00, 0x11, 0x6a, 0x6f, 0x62, 0x2d, 0x70, 0x61, 0x67, 0x65, 0x73, 0x2d, 0x70, 0x65, 0x72,
	0x2d, 0x73, 0x65, 0x74, 0x44, 0x00, 0x00, 0x00, 0x05, 0x73, 0x69, 0x64, 0x65, 0x73, 0x44, 0x00,
	0x00, 0x00, 0x15, 0x6f, 0x72, 0x69, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x65, 0x64, 0x44, 0x00, 0x00, 0x00, 0x05, 0x6d, 0x65, 0x64,
	0x69, 0x61, 0x44, 0x00, 0x00, 0x00, 0x0d, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x71, 0x75, 0x61,
	0x6c, 0x69, 0x74, 0x79, 0x44, 0x00, 0x00, 0x00, 0x12, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x2d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x0a,
	0x6f, 0x75, 0x74, 0x70, 0x75, 0x74, 0x2d, 0x62, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x09, 0x6d,
	0x65, 0x64, 0x69, 0x61, 0x2d, 0x63, 0x6f, 0x6c, 0x44, 0x00, 0x00, 0x00, 0x0b, 0x6f, 0x75, 0x74,
	0x70, 0x75, 0x74, 0x2d, 0x6d, 0x6f, 0x64, 0x65, 0x44, 0x00, 0x00, 0x00, 0x16, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x6f, 0x70, 0x74, 0x69, 0x6d,
	0x69, 0x7a, 0x65, 0x44, 0x00, 0x00, 0x00, 0x16, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x73, 0x6f, 0x75,
	0x72, 0x63, 0x65, 0x2d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x44, 0x00,
	0x00, 0x00, 0x10, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x2d, 0x6d,
	0x6f, 0x64, 0x65, 0x44, 0x00, 0x00, 0x00, 0x16, 0x69, 0x70, 0x70, 0x2d, 0x61, 0x74, 0x74, 0x72,
	0x69, 0x62, 0x75, 0x74, 0x65, 0x2d, 0x66, 0x69, 0x64, 0x65, 0x6c, 0x69, 0x74, 0x79, 0x44, 0x00,
	0x00, 0x00, 0x08, 0x6a, 0x6f, 0x62, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0b,
	0x70, 0x61, 0x67, 0x65, 0x2d, 0x72, 0x61, 0x6e, 0x67, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x1a,
	0x6d, 0x75, 0x6c, 0x74, 0x69, 0x70, 0x6c, 0x65, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
	0x74, 0x2d, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x69, 0x6e, 0x67, 0x44, 0x00, 0x00, 0x00, 0x18, 0x6a,
	0x6f, 0x62, 0x2d, 0x6d, 0x61, 0x6e, 0x64, 0x61, 0x74, 0x6f, 0x72, 0x79, 0x2d, 0x61, 0x74, 0x74,
	0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x09, 0x6f, 0x76, 0x65, 0x72,
	0x72, 0x69, 0x64, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x16, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d,
	0x72, 0x65, 0x6e, 0x64, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x2d, 0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74,
	0x44, 0x00, 0x00, 0x00, 0x0d, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x73, 0x63, 0x61, 0x6c, 0x69,
	0x6e, 0x67, 0x46, 0x00, 0x1f, 0x72, 0x65, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x63, 0x65, 0x2d, 0x75,
	0x72, 0x69, 0x2d, 0x73, 0x63, 0x68, 0x65, 0x6d, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f,
	0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x68, 0x74, 0x74, 0x70, 0x46, 0x00, 0x00, 0x00, 0x05, 0x68,
	0x74, 0x74, 0x70, 0x73, 0x41, 0x00, 0x11, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x64,
	0x65, 0x76, 0x69, 0x63, 0x65, 0x2d, 0x69, 0x64, 0x01, 0x99, 0x4d, 0x46, 0x47, 0x3a, 0x48, 0x50,
	0x3b, 0x4d, 0x44, 0x4c, 0x3a, 0x48, 0x50, 0x20, 0x4f, 0x66, 0x66, 0x69, 0x63, 0x65, 0x4a, 0x65,
	0x74, 0x20, 0x50, 0x72, 0x6f, 0x20, 0x38, 0x37, 0x33, 0x30, 0x3b, 0x43, 0x4d, 0x44, 0x3a, 0x50,
	0x43, 0x4c, 0x35, 0x63, 0x2c, 0x50, 0x43, 0x4c, 0x58, 0x4c, 0x2c, 0x50, 0x4f, 0x53, 0x54, 0x53,
	0x43, 0x52, 0x49, 0x50, 0x54, 0x2c, 0x4e, 0x41, 0x54, 0x49, 0x56, 0x45, 0x4f, 0x46, 0x46, 0x49,
	0x43, 0x45, 0x2c, 0x50, 0x44, 0x46, 0x2c, 0x50, 0x43, 0x4c, 0x33, 0x47, 0x55, 0x49, 0x2c, 0x50,
	0x43, 0x4c, 0x33, 0x2c, 0x50, 0x4a, 0x4c, 0x2c, 0x41, 0x75, 0x74, 0x6f, 0x6d, 0x61, 0x74, 0x69,
	0x63, 0x2c, 0x4a, 0x50, 0x45, 0x47, 0x2c, 0x50, 0x43, 0x4c, 0x4d, 0x2c, 0x41, 0x70, 0x70, 0x6c,
	0x65, 0x52, 0x61, 0x73, 0x74, 0x65, 0x72, 0x2c, 0x50, 0x57, 0x47, 0x52, 0x61, 0x73, 0x74, 0x65,
	0x72, 0x2c, 0x38, 0x30, 0x32, 0x2e, 0x31, 0x31, 0x2c, 0x38, 0x30, 0x32, 0x2e, 0x33, 0x2c, 0x44,
	0x45, 0x53, 0x4b, 0x4a, 0x45, 0x54, 0x2c, 0x44, 0x59, 0x4e, 0x3b, 0x43, 0x4c, 0x53, 0x3a, 0x50,
	0x52, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x3b, 0x44, 0x45, 0x53, 0x3a, 0x44, 0x39, 0x4c, 0x32, 0x30,
	0x41, 0x3b, 0x43, 0x49, 0x44, 0x3a, 0x48, 0x50, 0x49, 0x4a, 0x56, 0x49, 0x50, 0x41, 0x56, 0x39,
	0x3b, 0x4c, 0x45, 0x44, 0x4d, 0x44, 0x49, 0x53, 0x3a, 0x55, 0x53, 0x42, 0x23, 0x46, 0x46, 0x23,
	0x43, 0x43, 0x23, 0x30, 0x30, 0x2c, 0x55, 0x53, 0x42, 0x23, 0x46, 0x46, 0x23, 0x30, 0x34, 0x23,
	0x30, 0x31, 0x3b, 0x4d, 0x43, 0x54, 0x3a, 0x4d, 0x46, 0x3b, 0x4d, 0x43, 0x4c, 0x3a, 0x44, 0x49,
	0x3b, 0x4d, 0x43, 0x56, 0x3a, 0x33, 0x2e, 0x30, 0x3b, 0x53, 0x4e, 0x3a, 0x43, 0x4e, 0x37, 0x38,
	0x33, 0x46, 0x36, 0x30, 0x57, 0x31, 0x3b, 0x53, 0x3a, 0x30, 0x33, 0x38, 0x30, 0x38, 0x30, 0x43,
	0x34, 0x38, 0x34, 0x31, 0x30, 0x30, 0x30, 0x30, 0x31, 0x30, 0x30, 0x35, 0x38, 0x30, 0x30, 0x38,
	0x30, 0x30, 0x30, 0x30, 0x34, 0x34, 0x31, 0x38, 0x30, 0x30, 0x33, 0x63, 0x34, 0x35, 0x31, 0x38,
	0x30, 0x30, 0x34, 0x36, 0x34, 0x36, 0x31, 0x38, 0x30, 0x30, 0x33, 0x63, 0x34, 0x31, 0x31, 0x38,
	0x30, 0x30, 0x34, 0x36, 0x3b, 0x5a, 0x3a, 0x30, 0x35, 0x30, 0x30, 0x30, 0x30, 0x30, 0x39, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x39, 0x30, 0x30, 0x30, 0x30, 0x30, 0x39, 0x30, 0x30, 0x30, 0x30, 0x30,
	0x39, 0x30, 0x30, 0x30, 0x30, 0x30, 0x39, 0x2c, 0x31, 0x32, 0x30, 0x30, 0x30, 0x2c, 0x31, 0x37,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x2c, 0x31,
	0x38, 0x31, 0x3b, 0x30, 0x00, 0x0d, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x61, 0x6c,
	0x65, 0x72, 0x74, 0x00, 0x27, 0x63, 0x6f, 0x64, 0x65, 0x3d, 0x75, 0x6e, 0x6b, 0x6e, 0x6f, 0x77,
	0x6e, 0x3b, 0x73, 0x65, 0x76, 0x65, 0x72, 0x69, 0x74, 0x79, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72,
	0x3b, 0x67, 0x72, 0x6f, 0x75, 0x70, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x30, 0x00, 0x00, 0x00,
	0x27, 0x63, 0x6f, 0x64, 0x65, 0x3d, 0x75, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x3b, 0x73, 0x65,
	0x76, 0x65, 0x72, 0x69, 0x74, 0x79, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x3b, 0x67, 0x72, 0x6f,
	0x75, 0x70, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x30, 0x00, 0x00, 0x00, 0x27, 0x63, 0x6f, 0x64,
	0x65, 0x3d, 0x75, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x3b, 0x73, 0x65, 0x76, 0x65, 0x72, 0x69,
	0x74, 0x79, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x3b, 0x67, 0x72, 0x6f, 0x75, 0x70, 0x3d, 0x6f,
	0x74, 0x68, 0x65, 0x72, 0x30, 0x00, 0x00, 0x00, 0x27, 0x63, 0x6f, 0x64, 0x65, 0x3d, 0x75, 0x6e,
	0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x3b, 0x73, 0x65, 0x76, 0x65, 0x72, 0x69, 0x74, 0x79, 0x3d, 0x6f,
	0x74, 0x68, 0x65, 0x72, 0x3b, 0x67, 0x72, 0x6f, 0x75, 0x70, 0x3d, 0x6f, 0x74, 0x68, 0x65, 0x72,
	0x41, 0x00, 0x19, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x61, 0x6c, 0x65, 0x72, 0x74,
	0x2d, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x09, 0x67, 0x65,
	0x6e, 0x75, 0x69, 0x6e, 0x65, 0x48, 0x50, 0x41, 0x00, 0x00, 0x00, 0x09, 0x67, 0x65, 0x6e, 0x75,
	0x69, 0x6e, 0x65, 0x48, 0x50, 0x41, 0x00, 0x00, 0x00, 0x09, 0x67, 0x65, 0x6e, 0x75, 0x69, 0x6e,
	0x65, 0x48, 0x50, 0x41, 0x00, 0x00, 0x00, 0x09, 0x67, 0x65, 0x6e, 0x75, 0x69, 0x6e, 0x65, 0x48,
	0x50, 0x45, 0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x75, 0x75, 0x69, 0x64,
	0x00, 0x2d, 0x75, 0x72, 0x6e, 0x3a, 0x75, 0x75, 0x69, 0x64, 0x3a, 0x64, 0x62, 0x63, 0x63, 0x34,
	0x62, 0x35, 0x38, 0x2d, 0x66, 0x63, 0x34, 0x63, 0x2d, 0x66, 0x36, 0x66, 0x64, 0x2d, 0x62, 0x34,
	0x64, 0x36, 0x2d, 0x32, 0x62, 0x30, 0x30, 0x64, 0x35, 0x35, 0x34, 0x61, 0x64, 0x34, 0x37, 0x23,
	0x00, 0x29, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x63, 0x61, 0x70, 0x65, 0x2d, 0x6f, 0x72, 0x69, 0x65,
	0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x65,
	0x64, 0x2d, 0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x05, 0x34, 0x00, 0x19, 0x6a, 0x6f, 0x62, 0x2d, 0x63, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69,
	0x6e, 0x74, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x00, 0x4a,
	0x00, 0x00, 0x00, 0x0d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d,
	0x65, 0x42, 0x00, 0x00, 0x00, 0x0f, 0x66, 0x75, 0x6c, 0x6c, 0x62, 0x6c, 0x65, 0x65, 0x64, 0x2d,
	0x73, 0x69, 0x7a, 0x65, 0x73, 0x4a, 0x00, 0x00, 0x00, 0x10, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d,
	0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x13, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f,
	0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c,
	0x65, 0x66, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x72, 0x69,
	0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x44, 0x00, 0x00,
	0x00, 0x14, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x76, 0x6f, 0x69, 0x63, 0x65, 0x5f, 0x35, 0x2e, 0x35,
	0x78, 0x38, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69, 0x6e,
	0x64, 0x65, 0x78, 0x2d, 0x35, 0x78, 0x38, 0x5f, 0x35, 0x78, 0x38, 0x69, 0x6e, 0x44, 0x00, 0x00,
	0x00, 0x14, 0x6e, 0x61, 0x5f, 0x6f, 0x66, 0x69, 0x63, 0x69, 0x6f, 0x5f, 0x38, 0x2e, 0x35, 0x78,
	0x31, 0x33, 0x2e, 0x34, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x6f, 0x6d, 0x5f, 0x31, 0x36,
	0x6b, 0x5f, 0x31, 0x39, 0x35, 0x78, 0x32, 0x37, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10,
	0x6f, 0x6d, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x31, 0x38, 0x34, 0x78, 0x32, 0x36, 0x30, 0x6d, 0x6d,
	0x44, 0x00, 0x00, 0x00, 0x14, 0x72, 0x6f, 0x63, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x37, 0x2e, 0x37,
	0x35, 0x78, 0x31, 0x30, 0x2e, 0x37, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6a, 0x70,
	0x6e, 0x5f, 0x6f, 0x75, 0x66, 0x75, 0x6b, 0x75, 0x5f, 0x31, 0x34, 0x38, 0x78, 0x32, 0x30, 0x30,
	0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x18, 0x6e, 0x61, 0x5f, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72,
	0x2d, 0x31, 0x30, 0x5f, 0x34, 0x2e, 0x31, 0x32, 0x35, 0x78, 0x39, 0x2e, 0x35, 0x69, 0x6e, 0x44,
	0x00, 0x00, 0x00, 0x16, 0x6e, 0x61, 0x5f, 0x6d, 0x6f, 0x6e, 0x61, 0x72, 0x63, 0x68, 0x5f, 0x33,
	0x2e, 0x38, 0x37, 0x35, 0x78, 0x37, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69,
	0x73, 0x6f, 0x5f, 0x62, 0x35, 0x5f, 0x31, 0x37, 0x36, 0x78, 0x32, 0x35, 0x30, 0x6d, 0x6d, 0x44,
	0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x35, 0x5f, 0x31, 0x36, 0x32, 0x78, 0x32,
	0x32, 0x39, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x36, 0x5f,
	0x31, 0x31, 0x34, 0x78, 0x31, 0x36, 0x32, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73,
	0x6f, 0x5f, 0x64, 0x6c, 0x5f, 0x31, 0x31, 0x30, 0x78, 0x32, 0x32, 0x30, 0x6d, 0x6d, 0x44, 0x00,
	0x00, 0x00, 0x13, 0x6a, 0x70, 0x6e, 0x5f, 0x63, 0x68, 0x6f, 0x75, 0x33, 0x5f, 0x31, 0x32, 0x30,
	0x78, 0x32, 0x33, 0x35, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6a, 0x70, 0x6e, 0x5f, 0x63,
	0x68, 0x6f, 0x75, 0x34, 0x5f, 0x39, 0x30, 0x78, 0x32, 0x30, 0x35, 0x6d, 0x6d, 0x37, 0x00, 0x00,
	0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0d, 0x72, 0x65, 0x73, 0x6f,
	0x6c, 0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x42, 0x00, 0x00, 0x00, 0x0c, 0x64, 0x75,
	0x70, 0x6c, 0x65, 0x78, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x73, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x73,
	0x69, 0x64, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x14, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64,
	0x65, 0x64, 0x2d, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x44, 0x00, 0x00,
	0x00, 0x13, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x2d, 0x6c, 0x6f, 0x6e, 0x67,
	0x2d, 0x65, 0x64, 0x67, 0x65, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x44,
	0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x6c, 0x65, 0x74, 0x74, 0x65, 0x72, 0x5f, 0x38, 0x2e,
	0x35, 0x78, 0x31, 0x31, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x11, 0x6e, 0x61, 0x5f, 0x6c, 0x65,
	0x67, 0x61, 0x6c, 0x5f, 0x38, 0x2e, 0x35, 0x78, 0x31, 0x34, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00,
	0x18, 0x6e, 0x61, 0x5f, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x69, 0x76, 0x65, 0x5f, 0x37, 0x2e,
	0x32, 0x35, 0x78, 0x31, 0x30, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6e, 0x61,
	0x5f, 0x66, 0x6f, 0x6f, 0x6c, 0x73, 0x63, 0x61, 0x70, 0x5f, 0x38, 0x2e, 0x35, 0x78, 0x31, 0x33,
	0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x34, 0x5f, 0x32, 0x31,
	0x30, 0x78, 0x32, 0x39, 0x37, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x6a, 0x69, 0x73, 0x5f,
	0x62, 0x35, 0x5f, 0x31, 0x38, 0x32, 0x78, 0x32, 0x35, 0x37, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00,
	0x14, 0x6e, 0x61, 0x5f, 0x6f, 0x66, 0x69, 0x63, 0x69, 0x6f, 0x5f, 0x38, 0x2e, 0x35, 0x78, 0x31,
	0x33, 0x2e, 0x34, 0x69, 0x6e, 0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d,
	0x74, 0x79, 0x70, 0x65, 0x44, 0x00, 0x00, 0x00, 0x15, 0x63, 0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e,
	0x61, 0x64, 0x76, 0x61, 0x6e, 0x63, 0x65, 0x64, 0x2d, 0x70, 0x68, 0x6f, 0x74, 0x6f, 0x44, 0x00,
	0x00, 0x00, 0x09, 0x63, 0x61, 0x72, 0x64, 0x73, 0x74, 0x6f, 0x63, 0x6b, 0x37, 0x00, 0x00, 0x00,
	0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0d, 0x72, 0x65, 0x73, 0x6f, 0x6c,
	0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x42, 0x00, 0x00, 0x00, 0x0c, 0x64, 0x75, 0x70,
	0x6c, 0x65, 0x78, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x73, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x73, 0x69,
	0x64, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x14, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65,
	0x64, 0x2d, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x44, 0x00, 0x00, 0x00,
	0x13, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x2d, 0x6c, 0x6f, 0x6e, 0x67, 0x2d,
	0x65, 0x64, 0x67, 0x65, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x44, 0x00,
	0x00, 0x00, 0x14, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x76, 0x6f, 0x69, 0x63, 0x65, 0x5f, 0x35, 0x2e,
	0x35, 0x78, 0x38, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69,
	0x6e, 0x64, 0x65, 0x78, 0x2d, 0x33, 0x78, 0x35, 0x5f, 0x33, 0x78, 0x35, 0x69, 0x6e, 0x44, 0x00,
	0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x64, 0x65, 0x78, 0x2d, 0x34, 0x78, 0x36, 0x5f,
	0x34, 0x78, 0x36, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x0c, 0x6e, 0x61, 0x5f, 0x35, 0x78, 0x37,
	0x5f, 0x35, 0x78, 0x37, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69, 0x6e,
	0x64, 0x65, 0x78, 0x2d, 0x35, 0x78, 0x38, 0x5f, 0x35, 0x78, 0x38, 0x69, 0x6e, 0x44, 0x00, 0x00,
	0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x35, 0x5f, 0x31, 0x34, 0x38, 0x78, 0x32, 0x31, 0x30,
	0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x36, 0x5f, 0x31, 0x30,
	0x35, 0x78, 0x31, 0x34, 0x38, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6f, 0x65, 0x5f, 0x70,
	0x68, 0x6f, 0x74, 0x6f, 0x2d, 0x6c, 0x5f, 0x33, 0x2e, 0x35, 0x78, 0x35, 0x69, 0x6e, 0x44, 0x00,
	0x00, 0x00, 0x18, 0x6f, 0x6d, 0x5f, 0x73, 0x6d, 0x61, 0x6c, 0x6c, 0x2d, 0x70, 0x68, 0x6f, 0x74,
	0x6f, 0x5f, 0x31, 0x30, 0x30, 0x78, 0x31, 0x35, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10,
	0x6f, 0x6d, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x31, 0x39, 0x35, 0x78, 0x32, 0x37, 0x30, 0x6d, 0x6d,
	0x44, 0x00, 0x00, 0x00, 0x10, 0x6f, 0x6d, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x31, 0x38, 0x34, 0x78,
	0x32, 0x36, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x14, 0x72, 0x6f, 0x63, 0x5f, 0x31, 0x36,
	0x6b, 0x5f, 0x37, 0x2e, 0x37, 0x35, 0x78, 0x31, 0x30, 0x2e, 0x37, 0x35, 0x69, 0x6e, 0x44, 0x00,
	0x00, 0x00, 0x14, 0x6a, 0x70, 0x6e, 0x5f, 0x68, 0x61, 0x67, 0x61, 0x6b, 0x69, 0x5f, 0x31, 0x30,
	0x30, 0x78, 0x31, 0x34, 0x38, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6a, 0x70, 0x6e, 0x5f,
	0x6f, 0x75, 0x66, 0x75, 0x6b, 0x75, 0x5f, 0x31, 0x34, 0x38, 0x78, 0x32, 0x30, 0x30, 0x6d, 0x6d,
	0x44, 0x00, 0x00, 0x00, 0x18, 0x6e, 0x61, 0x5f, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x2d, 0x31,
	0x30, 0x5f, 0x34, 0x2e, 0x31, 0x32, 0x35, 0x78, 0x39, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00,
	0x00, 0x16, 0x6e, 0x61, 0x5f, 0x6d, 0x6f, 0x6e, 0x61, 0x72, 0x63, 0x68, 0x5f, 0x33, 0x2e, 0x38,
	0x37, 0x35, 0x78, 0x37, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f,
	0x5f, 0x62, 0x35, 0x5f, 0x31, 0x37, 0x36, 0x78, 0x32, 0x35, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00,
	0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x35, 0x5f, 0x31, 0x36, 0x32, 0x78, 0x32, 0x32, 0x39,
	0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x36, 0x5f, 0x31, 0x31,
	0x34, 0x78, 0x31, 0x36, 0x32, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f,
	0x64, 0x6c, 0x5f, 0x31, 0x31, 0x30, 0x78, 0x32, 0x32, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00,
	0x13, 0x6a, 0x70, 0x6e, 0x5f, 0x63, 0x68, 0x6f, 0x75, 0x33, 0x5f, 0x31, 0x32, 0x30, 0x78, 0x32,
	0x33, 0x35, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6a, 0x70, 0x6e, 0x5f, 0x63, 0x68, 0x6f,
	0x75, 0x34, 0x5f, 0x39, 0x30, 0x78, 0x32, 0x30, 0x35, 0x6d, 0x6d, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x17, 0x6a, 0x6f, 0x62, 0x2d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x76, 0x65, 0x72, 0x73,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00,
	0x0d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x42, 0x00,
	0x00, 0x00, 0x0f, 0x66, 0x75, 0x6c, 0x6c, 0x62, 0x6c, 0x65, 0x65, 0x64, 0x2d, 0x73, 0x69, 0x7a,
	0x65, 0x73, 0x4a, 0x00, 0x00, 0x00, 0x10, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70,
	0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28,
	0x4a, 0x00, 0x00, 0x00, 0x13, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f,
	0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01,
	0x28, 0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74,
	0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28,
	0x4a, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x72, 0x69, 0x67, 0x68, 0x74,
	0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28,
	0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0d, 0x72,
	0x65, 0x73, 0x6f, 0x6c, 0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x42, 0x00, 0x00, 0x00,
	0x0c, 0x64, 0x75, 0x70, 0x6c, 0x65, 0x78, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x73, 0x4a, 0x00, 0x00,
	0x00, 0x05, 0x73, 0x69, 0x64, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x09, 0x6f, 0x6e, 0x65, 0x2d,
	0x73, 0x69, 0x64, 0x65, 0x64, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a,
	0x00, 0x00, 0x00, 0x0d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x76, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d,
	0x65, 0x42, 0x00, 0x00, 0x00, 0x0c, 0x64, 0x75, 0x70, 0x6c, 0x65, 0x78, 0x2d, 0x73, 0x69, 0x7a,
	0x65, 0x73, 0x4a, 0x00, 0x00, 0x00, 0x05, 0x73, 0x69, 0x64, 0x65, 0x73, 0x44, 0x00, 0x00, 0x00,
	0x09, 0x6f, 0x6e, 0x65, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x37, 0x00, 0x00, 0x00, 0x00, 0x44,
	0x00, 0x16, 0x69, 0x70, 0x70, 0x2d, 0x66, 0x65, 0x61, 0x74, 0x75, 0x72, 0x65, 0x73, 0x2d, 0x73,
	0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x0c, 0x61, 0x69, 0x72, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x2d, 0x31, 0x2e, 0x35, 0x44, 0x00, 0x00, 0x00, 0x0e, 0x69, 0x70, 0x70, 0x2d, 0x65,
	0x76, 0x65, 0x72, 0x79, 0x77, 0x68, 0x65, 0x72, 0x65, 0x44, 0x00, 0x14, 0x77, 0x68, 0x69, 0x63,
	0x68, 0x2d, 0x6a, 0x6f, 0x62, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x09, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74, 0x65, 0x64, 0x44, 0x00, 0x00, 0x00, 0x0d,
	0x6e, 0x6f, 0x74, 0x2d, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74, 0x65, 0x64, 0x44, 0x00, 0x00,
	0x00, 0x03, 0x61, 0x6c, 0x6c, 0x22, 0x00, 0x11, 0x6a, 0x6f, 0x62, 0x2d, 0x69, 0x64, 0x73, 0x2d,
	0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x01, 0x01, 0x22, 0x00, 0x1d, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x2d, 0x75, 0x73, 0x65, 0x72, 0x2d, 0x75,
	0x72, 0x69, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x01, 0x01, 0x44,
	0x00, 0x22, 0x6d, 0x75, 0x6c, 0x74, 0x69, 0x70, 0x6c, 0x65, 0x2d, 0x6f, 0x70, 0x65, 0x72, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x2d, 0x6f, 0x75, 0x74, 0x2d, 0x61, 0x63,
	0x74, 0x69, 0x6f, 0x6e, 0x00, 0x0b, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x73, 0x73, 0x2d, 0x6a, 0x6f,
	0x62, 0x12, 0x00, 0x14, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x67, 0x65, 0x6f, 0x2d,
	0x6c, 0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x00, 0x41, 0x00, 0x10, 0x6d, 0x6f, 0x70,
	0x72, 0x69, 0x61, 0x2d, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x65, 0x64, 0x00, 0x03, 0x31,
	0x2e, 0x33, 0x22, 0x00, 0x1e, 0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x2d, 0x61,
	0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72,
	0x74, 0x65, 0x64, 0x00, 0x01, 0x00, 0x31, 0x00, 0x1f, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x2d, 0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x2d, 0x64,
	0x61, 0x74, 0x65, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x00, 0x0b, 0x07, 0xe4, 0x01, 0x13, 0x0a, 0x31,
	0x10, 0x00, 0x2b, 0x00, 0x00, 0x21, 0x00, 0x1a, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d,
	0x63, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x2d, 0x74, 0x69,
	0x6d, 0x65, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x31, 0x00, 0x1e, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x73, 0x74, 0x61, 0x74, 0x65, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x2d,
	0x64, 0x61, 0x74, 0x65, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x00, 0x0b, 0x07, 0xe4, 0x01, 0x13, 0x0a,
	0x31, 0x10, 0x00, 0x2b, 0x00, 0x00, 0x21, 0x00, 0x19, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x2d, 0x73, 0x74, 0x61, 0x74, 0x65, 0x2d, 0x63, 0x68, 0x61, 0x6e, 0x67, 0x65, 0x2d, 0x74, 0x69,
	0x6d, 0x65, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x1b, 0x64, 0x6f, 0x63, 0x75, 0x6d,
	0x65, 0x6e, 0x74, 0x2d, 0x70, 0x61, 0x73, 0x73, 0x77, 0x6f, 0x72, 0x64, 0x2d, 0x73, 0x75, 0x70,
	0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x0c, 0x70,
	0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6b, 0x69, 0x6e, 0x64, 0x00, 0x08, 0x64, 0x6f, 0x63,
	0x75, 0x6d, 0x65, 0x6e, 0x74, 0x44, 0x00, 0x00, 0x00, 0x08, 0x65, 0x6e, 0x76, 0x65, 0x6c, 0x6f,
	0x70, 0x65, 0x44, 0x00, 0x00, 0x00, 0x05, 0x70, 0x68, 0x6f, 0x74, 0x6f, 0x44, 0x00, 0x00, 0x00,
	0x08, 0x70, 0x6f, 0x73, 0x74, 0x63, 0x61, 0x72, 0x64, 0x44, 0x00, 0x0f, 0x6d, 0x65, 0x64, 0x69,
	0x61, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x12, 0x6e, 0x61, 0x5f,
	0x6c, 0x65, 0x74, 0x74, 0x65, 0x72, 0x5f, 0x38, 0x2e, 0x35, 0x78, 0x31, 0x31, 0x69, 0x6e, 0x44,
	0x00, 0x00, 0x00, 0x11, 0x6e, 0x61, 0x5f, 0x6c, 0x65, 0x67, 0x61, 0x6c, 0x5f, 0x38, 0x2e, 0x35,
	0x78, 0x31, 0x34, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x18, 0x6e, 0x61, 0x5f, 0x65, 0x78, 0x65,
	0x63, 0x75, 0x74, 0x69, 0x76, 0x65, 0x5f, 0x37, 0x2e, 0x32, 0x35, 0x78, 0x31, 0x30, 0x2e, 0x35,
	0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x76, 0x6f, 0x69, 0x63,
	0x65, 0x5f, 0x35, 0x2e, 0x35, 0x78, 0x38, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x14,
	0x6e, 0x61, 0x5f, 0x66, 0x6f, 0x6f, 0x6c, 0x73, 0x63, 0x61, 0x70, 0x5f, 0x38, 0x2e, 0x35, 0x78,
	0x31, 0x33, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x64, 0x65,
	0x78, 0x2d, 0x33, 0x78, 0x35, 0x5f, 0x33, 0x78, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12,
	0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x64, 0x65, 0x78, 0x2d, 0x34, 0x78, 0x36, 0x5f, 0x34, 0x78, 0x36,
	0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x0c, 0x6e, 0x61, 0x5f, 0x35, 0x78, 0x37, 0x5f, 0x35, 0x78,
	0x37, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6e, 0x61, 0x5f, 0x69, 0x6e, 0x64, 0x65, 0x78,
	0x2d, 0x35, 0x78, 0x38, 0x5f, 0x35, 0x78, 0x38, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69,
	0x73, 0x6f, 0x5f, 0x61, 0x34, 0x5f, 0x32, 0x31, 0x30, 0x78, 0x32, 0x39, 0x37, 0x6d, 0x6d, 0x44,
	0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x35, 0x5f, 0x31, 0x34, 0x38, 0x78, 0x32,
	0x31, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x36, 0x5f,
	0x31, 0x30, 0x35, 0x78, 0x31, 0x34, 0x38, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x6a, 0x69,
	0x73, 0x5f, 0x62, 0x35, 0x5f, 0x31, 0x38, 0x32, 0x78, 0x32, 0x35, 0x37, 0x6d, 0x6d, 0x44, 0x00,
	0x00, 0x00, 0x12, 0x6f, 0x65, 0x5f, 0x70, 0x68, 0x6f, 0x74, 0x6f, 0x2d, 0x6c, 0x5f, 0x33, 0x2e,
	0x35, 0x78, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x18, 0x6f, 0x6d, 0x5f, 0x73, 0x6d, 0x61,
	0x6c, 0x6c, 0x2d, 0x70, 0x68, 0x6f, 0x74, 0x6f, 0x5f, 0x31, 0x30, 0x30, 0x78, 0x31, 0x35, 0x30,
	0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6e, 0x61, 0x5f, 0x6f, 0x66, 0x69, 0x63, 0x69, 0x6f,
	0x5f, 0x38, 0x2e, 0x35, 0x78, 0x31, 0x33, 0x2e, 0x34, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10,
	0x6f, 0x6d, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x31, 0x39, 0x35, 0x78, 0x32, 0x37, 0x30, 0x6d, 0x6d,
	0x44, 0x00, 0x00, 0x00, 0x10, 0x6f, 0x6d, 0x5f, 0x31, 0x36, 0x6b, 0x5f, 0x31, 0x38, 0x34, 0x78,
	0x32, 0x36, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x14, 0x72, 0x6f, 0x63, 0x5f, 0x31, 0x36,
	0x6b, 0x5f, 0x37, 0x2e, 0x37, 0x35, 0x78, 0x31, 0x30, 0x2e, 0x37, 0x35, 0x69, 0x6e, 0x44, 0x00,
	0x00, 0x00, 0x14, 0x6a, 0x70, 0x6e, 0x5f, 0x68, 0x61, 0x67, 0x61, 0x6b, 0x69, 0x5f, 0x31, 0x30,
	0x30, 0x78, 0x31, 0x34, 0x38, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x14, 0x6a, 0x70, 0x6e, 0x5f,
	0x6f, 0x75, 0x66, 0x75, 0x6b, 0x75, 0x5f, 0x31, 0x34, 0x38, 0x78, 0x32, 0x30, 0x30, 0x6d, 0x6d,
	0x44, 0x00, 0x00, 0x00, 0x18, 0x6e, 0x61, 0x5f, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x2d, 0x31,
	0x30, 0x5f, 0x34, 0x2e, 0x31, 0x32, 0x35, 0x78, 0x39, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00,
	0x00, 0x16, 0x6e, 0x61, 0x5f, 0x6d, 0x6f, 0x6e, 0x61, 0x72, 0x63, 0x68, 0x5f, 0x33, 0x2e, 0x38,
	0x37, 0x35, 0x78, 0x37, 0x2e, 0x35, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f,
	0x5f, 0x62, 0x35, 0x5f, 0x31, 0x37, 0x36, 0x78, 0x32, 0x35, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00,
	0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x35, 0x5f, 0x31, 0x36, 0x32, 0x78, 0x32, 0x32, 0x39,
	0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x63, 0x36, 0x5f, 0x31, 0x31,
	0x34, 0x78, 0x31, 0x36, 0x32, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f,
	0x64, 0x6c, 0x5f, 0x31, 0x31, 0x30, 0x78, 0x32, 0x32, 0x30, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00,
	0x13, 0x6a, 0x70, 0x6e, 0x5f, 0x63, 0x68, 0x6f, 0x75, 0x33, 0x5f, 0x31, 0x32, 0x30, 0x78, 0x32,
	0x33, 0x35, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6a, 0x70, 0x6e, 0x5f, 0x63, 0x68, 0x6f,
	0x75, 0x34, 0x5f, 0x39, 0x30, 0x78, 0x32, 0x30, 0x35, 0x6d, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x10,
	0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x5f, 0x6d, 0x69, 0x6e, 0x5f, 0x33, 0x78, 0x35, 0x69, 0x6e,
	0x44, 0x00, 0x00, 0x00, 0x13, 0x63, 0x75, 0x73, 0x74, 0x6f, 0x6d, 0x5f, 0x6d, 0x61, 0x78, 0x5f,
	0x38, 0x2e, 0x35, 0x78, 0x31, 0x34, 0x69, 0x6e, 0x44, 0x00, 0x0d, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f, 0x61, 0x34,
	0x5f, 0x32, 0x31, 0x30, 0x78, 0x32, 0x39, 0x37, 0x6d, 0x6d, 0x44, 0x00, 0x13, 0x6d, 0x65, 0x64,
	0x69, 0x61, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x44, 0x00, 0x00, 0x00,
	0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x44, 0x00, 0x00, 0x00, 0x10,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x44, 0x00, 0x00, 0x00, 0x11, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74, 0x2d,
	0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x44, 0x00, 0x00,
	0x00, 0x13, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d,
	0x61, 0x72, 0x67, 0x69, 0x6e, 0x44, 0x00, 0x00, 0x00, 0x0c, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d,
	0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0f, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x73, 0x69, 0x7a, 0x65, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x34, 0x00, 0x11, 0x6d, 0x65, 0x64,
	0x69, 0x61, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x00,
	0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x34,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e,
	0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x74, 0x04, 0x37, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x10,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x13, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d,
	0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x0c, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x44, 0x00, 0x00, 0x00, 0x04, 0x6d,
	0x61, 0x69, 0x6e, 0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x79,
	0x70, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72,
	0x79, 0x37, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x17, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69,
	0x6e, 0x67, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x64, 0x61, 0x74, 0x61, 0x62, 0x61, 0x73, 0x65,
	0x00, 0x00, 0x21, 0x00, 0x1b, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74, 0x2d,
	0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x21,
	0x00, 0x1c, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61,
	0x72, 0x67, 0x69, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04,
	0x00, 0x00, 0x01, 0x28, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x1a,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x1d, 0x6d, 0x65, 0x64, 0x69,
	0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x2d,
	0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x21,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x16, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x04, 0x6d, 0x61, 0x69, 0x6e, 0x44, 0x00, 0x14, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d,
	0x74, 0x79, 0x70, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x0a,
	0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72, 0x79, 0x44, 0x00, 0x00, 0x00, 0x14, 0x63,
	0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x65, 0x63, 0x6f, 0x73, 0x6d, 0x61, 0x72, 0x74, 0x2d, 0x6c,
	0x69, 0x74, 0x65, 0x44, 0x00, 0x00, 0x00, 0x21, 0x63, 0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x70,
	0x72, 0x65, 0x6d, 0x69, 0x75, 0x6d, 0x2d, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74,
	0x69, 0x6f, 0x6e, 0x2d, 0x6d, 0x61, 0x74, 0x74, 0x65, 0x44, 0x00, 0x00, 0x00, 0x25, 0x63, 0x6f,
	0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x74, 0x72, 0x69, 0x66, 0x6f, 0x6c, 0x64, 0x2d, 0x62, 0x72, 0x6f,
	0x63, 0x68, 0x75, 0x72, 0x65, 0x2d, 0x67, 0x6c, 0x6f, 0x73, 0x73, 0x79, 0x2d, 0x31, 0x38, 0x30,
	0x67, 0x73, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x15, 0x63, 0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x62,
	0x72, 0x6f, 0x63, 0x68, 0x75, 0x72, 0x65, 0x2d, 0x6d, 0x61, 0x74, 0x74, 0x65, 0x44, 0x00, 0x00,
	0x00, 0x16, 0x63, 0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x62, 0x72, 0x6f, 0x63, 0x68, 0x75, 0x72,
	0x65, 0x2d, 0x67, 0x6c, 0x6f, 0x73, 0x73, 0x79, 0x44, 0x00, 0x00, 0x00, 0x15, 0x63, 0x6f, 0x6d,
	0x2e, 0x68, 0x70, 0x2e, 0x61, 0x64, 0x76, 0x61, 0x6e, 0x63, 0x65, 0x64, 0x2d, 0x70, 0x68, 0x6f,
	0x74, 0x6f, 0x44, 0x00, 0x00, 0x00, 0x16, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72,
	0x79, 0x2d, 0x6c, 0x69, 0x67, 0x68, 0x74, 0x77, 0x65, 0x69, 0x67, 0x68, 0x74, 0x44, 0x00, 0x00,
	0x00, 0x13, 0x63, 0x6f, 0x6d, 0x2e, 0x68, 0x70, 0x2e, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x74, 0x65, 0x44, 0x00, 0x00, 0x00, 0x09, 0x63, 0x61, 0x72, 0x64, 0x73, 0x74,
	0x6f, 0x63, 0x6b, 0x44, 0x00, 0x00, 0x00, 0x16, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65,
	0x72, 0x79, 0x2d, 0x68, 0x65, 0x61, 0x76, 0x79, 0x77, 0x65, 0x69, 0x67, 0x68, 0x74, 0x44, 0x00,
	0x00, 0x00, 0x15, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72, 0x79, 0x2d, 0x6c, 0x65,
	0x74, 0x74, 0x65, 0x72, 0x68, 0x65, 0x61, 0x64, 0x34, 0x00, 0x14, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x73, 0x69, 0x7a, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x6d, 0x24, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x8a, 0xe8, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x47, 0xef, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x68, 0x2e, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x36, 0x92, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x80, 0xfc, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x1d, 0xc4, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x31, 0x9c, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x27, 0xb0, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x3b, 0x88, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x31, 0x9c, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x45, 0x74, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x31, 0x9c, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x4f, 0x60, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x74, 0x04, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x39, 0xd0, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x29, 0x04, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x39, 0xd0, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x47, 0x18, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x64, 0x64, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x22, 0xba, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x31, 0x9c, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x27, 0x10, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x3a, 0x98, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x84, 0xf4, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x4c, 0x2c, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x69, 0x78, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x47, 0xe0, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x65, 0x90, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x4c, 0xe5, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x6a, 0xa9, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x27, 0x10, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x39, 0xd0, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x39, 0xd0, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x4e, 0x20, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x28, 0xed, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x5e, 0x42, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x26, 0x72, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x4a, 0x6a, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x44, 0xc0, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x61, 0xa8, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x3f, 0x48, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x59, 0x74, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x2c, 0x88, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d,
	0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x3f, 0x48, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x2a, 0xf8, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69,
	0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x55, 0xf0, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x2e, 0xe0, 0x4a, 0x00,
	0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x5b, 0xcc, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x23, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x50, 0x14, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x33, 0x00, 0x00,
	0x00, 0x08, 0x00, 0x00, 0x1d, 0xc4, 0x00, 0x00, 0x54, 0x56, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x33, 0x00, 0x00, 0x00, 0x08, 0x00,
	0x00, 0x31, 0x9c, 0x00, 0x00, 0x8a, 0xe8, 0x37, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x0b, 0x6d,
	0x65, 0x64, 0x69, 0x61, 0x2d, 0x72, 0x65, 0x61, 0x64, 0x79, 0x00, 0x10, 0x69, 0x73, 0x6f, 0x5f,
	0x61, 0x34, 0x5f, 0x32, 0x31, 0x30, 0x78, 0x32, 0x39, 0x37, 0x6d, 0x6d, 0x34, 0x00, 0x0f, 0x6d,
	0x65, 0x64, 0x69, 0x61, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x72, 0x65, 0x61, 0x64, 0x79, 0x00, 0x00,
	0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x34,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e,
	0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x4a, 0x00, 0x00,
	0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00,
	0x00, 0x04, 0x00, 0x00, 0x74, 0x04, 0x37, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x10,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x13, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69,
	0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d,
	0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a, 0x00, 0x00, 0x00, 0x0c, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x44, 0x00, 0x00, 0x00, 0x04, 0x6d,
	0x61, 0x69, 0x6e, 0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x79,
	0x70, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72,
	0x79, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0a,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a,
	0x00, 0x00, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64,
	0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x74,
	0x04, 0x37, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x10, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x74, 0x6f, 0x70, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x13, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x62,
	0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d,
	0x6c, 0x65, 0x66, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x12, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x72,
	0x69, 0x67, 0x68, 0x74, 0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04,
	0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0c, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73,
	0x6f, 0x75, 0x72, 0x63, 0x65, 0x44, 0x00, 0x00, 0x00, 0x04, 0x6d, 0x61, 0x69, 0x6e, 0x4a, 0x00,
	0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x44, 0x00, 0x00,
	0x00, 0x0a, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72, 0x79, 0x37, 0x00, 0x00, 0x00,
	0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65, 0x64, 0x69, 0x61,
	0x2d, 0x73, 0x69, 0x7a, 0x65, 0x34, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x78,
	0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
	0x00, 0x52, 0x08, 0x4a, 0x00, 0x00, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73,
	0x69, 0x6f, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x74, 0x04, 0x37, 0x00, 0x00, 0x00,
	0x00, 0x4a, 0x00, 0x00, 0x00, 0x10, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x74, 0x6f, 0x70, 0x2d,
	0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a,
	0x00, 0x00, 0x00, 0x13, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d,
	0x2d, 0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28,
	0x4a, 0x00, 0x00, 0x00, 0x11, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x6c, 0x65, 0x66, 0x74, 0x2d,
	0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a,
	0x00, 0x00, 0x00, 0x12, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d,
	0x6d, 0x61, 0x72, 0x67, 0x69, 0x6e, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x01, 0x28, 0x4a,
	0x00, 0x00, 0x00, 0x0c, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65,
	0x44, 0x00, 0x00, 0x00, 0x04, 0x6d, 0x61, 0x69, 0x6e, 0x4a, 0x00, 0x00, 0x00, 0x0a, 0x6d, 0x65,
	0x64, 0x69, 0x61, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x73, 0x74, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x65, 0x72, 0x79, 0x37, 0x00, 0x00, 0x00, 0x00, 0x13, 0x00, 0x14, 0x66,
	0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x72, 0x65,
	0x61, 0x64, 0x79, 0x00, 0x00, 0x44, 0x00, 0x18, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e,
	0x67, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x12, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x2d, 0x74, 0x65, 0x6d, 0x70,
	0x6c, 0x61, 0x74, 0x65, 0x30, 0x00, 0x10, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x66,
	0x69, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x72, 0x00, 0x00, 0x30, 0x00, 0x1c, 0x70, 0x72, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x2d, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x65, 0x72, 0x2d, 0x64, 0x65, 0x73,
	0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x00, 0x21, 0x00, 0x10, 0x70, 0x61, 0x67,
	0x65, 0x73, 0x2d, 0x70, 0x65, 0x72, 0x2d, 0x6d, 0x69, 0x6e, 0x75, 0x74, 0x65, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x18, 0x21, 0x00, 0x16, 0x70, 0x61, 0x67, 0x65, 0x73, 0x2d, 0x70, 0x65, 0x72, 0x2d,
	0x6d, 0x69, 0x6e, 0x75, 0x74, 0x65, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x14, 0x33, 0x00, 0x17, 0x6a, 0x70, 0x65, 0x67, 0x2d, 0x6b, 0x2d, 0x6f, 0x63, 0x74, 0x65,
	0x74, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x08, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x33, 0x00, 0x1a, 0x6a, 0x70, 0x65, 0x67, 0x2d, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72,
	0x74, 0x65, 0x64, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x33, 0x00, 0x1a,
	0x6a, 0x70, 0x65, 0x67, 0x2d, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x40, 0x00, 0x33, 0x00, 0x16, 0x70, 0x64, 0x66, 0x2d, 0x6b, 0x2d, 0x6f, 0x63, 0x74,
	0x65, 0x74, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x00, 0x33, 0x00, 0x14, 0x70, 0x64, 0x66, 0x2d, 0x73, 0x69,
	0x7a, 0x65, 0x2d, 0x63, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69, 0x6e, 0x74, 0x73, 0x00, 0x08,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x00, 0x22, 0x00, 0x17, 0x70, 0x64, 0x66, 0x2d, 0x66,
	0x69, 0x74, 0x2d, 0x74, 0x6f, 0x2d, 0x70, 0x61, 0x67, 0x65, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75,
	0x6c, 0x74, 0x00, 0x01, 0x00, 0x22, 0x00, 0x19, 0x70, 0x64, 0x66, 0x2d, 0x66, 0x69, 0x74, 0x2d,
	0x74, 0x6f, 0x2d, 0x70, 0x61, 0x67, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x01, 0x00, 0x44, 0x00, 0x16, 0x70, 0x64, 0x66, 0x2d, 0x76, 0x65, 0x72, 0x73, 0x69,
	0x6f, 0x6e, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x10, 0x69,
	0x73, 0x6f, 0x2d, 0x33, 0x32, 0x30, 0x30, 0x30, 0x2d, 0x31, 0x5f, 0x32, 0x30, 0x30, 0x38, 0x44,
	0x00, 0x00, 0x00, 0x09, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x37, 0x44, 0x00, 0x00,
	0x00, 0x09, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x36, 0x44, 0x00, 0x00, 0x00, 0x09,
	0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x35, 0x44, 0x00, 0x00, 0x00, 0x10, 0x69, 0x73,
	0x6f, 0x2d, 0x31, 0x39, 0x30, 0x30, 0x35, 0x2d, 0x31, 0x5f, 0x32, 0x30, 0x30, 0x35, 0x44, 0x00,
	0x00, 0x00, 0x09, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x34, 0x44, 0x00, 0x00, 0x00,
	0x09, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x33, 0x44, 0x00, 0x00, 0x00, 0x09, 0x61,
	0x64, 0x6f, 0x62, 0x65, 0x2d, 0x31, 0x2e, 0x32, 0x44, 0x00, 0x0d, 0x75, 0x72, 0x66, 0x2d, 0x73,
	0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x03, 0x43, 0x50, 0x31, 0x44, 0x00, 0x00,
	0x00, 0x0f, 0x4d, 0x54, 0x31, 0x2d, 0x32, 0x2d, 0x38, 0x2d, 0x39, 0x2d, 0x31, 0x30, 0x2d, 0x31,
	0x31, 0x44, 0x00, 0x00, 0x00, 0x07, 0x50, 0x51, 0x33, 0x2d, 0x34, 0x2d, 0x35, 0x44, 0x00, 0x00,
	0x00, 0x09, 0x52, 0x53, 0x33, 0x30, 0x30, 0x2d, 0x36, 0x30, 0x30, 0x44, 0x00, 0x00, 0x00, 0x06,
	0x53, 0x52, 0x47, 0x42, 0x32, 0x34, 0x44, 0x00, 0x00, 0x00, 0x04, 0x4f, 0x42, 0x31, 0x30, 0x44,
	0x00, 0x00, 0x00, 0x02, 0x57, 0x38, 0x44, 0x00, 0x00, 0x00, 0x05, 0x44, 0x45, 0x56, 0x57, 0x38,
	0x44, 0x00, 0x00, 0x00, 0x08, 0x44, 0x45, 0x56, 0x52, 0x47, 0x42, 0x32, 0x34, 0x44, 0x00, 0x00,
	0x00, 0x0a, 0x41, 0x44, 0x4f, 0x42, 0x45, 0x52, 0x47, 0x42, 0x32, 0x34, 0x44, 0x00, 0x00, 0x00,
	0x03, 0x44, 0x4d, 0x33, 0x44, 0x00, 0x00, 0x00, 0x03, 0x46, 0x4e, 0x33, 0x44, 0x00, 0x00, 0x00,
	0x05, 0x49, 0x53, 0x31, 0x2d, 0x32, 0x44, 0x00, 0x00, 0x00, 0x04, 0x56, 0x31, 0x2e, 0x34, 0x42,
	0x00, 0x0c, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x72, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x73, 0x00, 0x08,
	0x63, 0x79, 0x61, 0x6e, 0x20, 0x69, 0x6e, 0x6b, 0x42, 0x00, 0x00, 0x00, 0x0b, 0x6d, 0x61, 0x67,
	0x65, 0x6e, 0x74, 0x61, 0x20, 0x69, 0x6e, 0x6b, 0x42, 0x00, 0x00, 0x00, 0x0a, 0x79, 0x65, 0x6c,
	0x6c, 0x6f, 0x77, 0x20, 0x69, 0x6e, 0x6b, 0x42, 0x00, 0x00, 0x00, 0x09, 0x62, 0x6c, 0x61, 0x63,
	0x6b, 0x20, 0x69, 0x6e, 0x6b, 0x42, 0x00, 0x0d, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x72, 0x2d, 0x63,
	0x6f, 0x6c, 0x6f, 0x72, 0x73, 0x00, 0x07, 0x23, 0x30, 0x30, 0x46, 0x46, 0x46, 0x46, 0x42, 0x00,
	0x00, 0x00, 0x07, 0x23, 0x46, 0x46, 0x30, 0x30, 0x46, 0x46, 0x42, 0x00, 0x00, 0x00, 0x07, 0x23,
	0x46, 0x46, 0x46, 0x46, 0x30, 0x30, 0x42, 0x00, 0x00, 0x00, 0x07, 0x23, 0x30, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x44, 0x00, 0x0c, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x72, 0x2d, 0x74, 0x79, 0x70, 0x65,
	0x73, 0x00, 0x0d, 0x69, 0x6e, 0x6b, 0x2d, 0x63, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65,
	0x44, 0x00, 0x00, 0x00, 0x0d, 0x69, 0x6e, 0x6b, 0x2d, 0x63, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64,
	0x67, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0d, 0x69, 0x6e, 0x6b, 0x2d, 0x63, 0x61, 0x72, 0x74, 0x72,
	0x69, 0x64, 0x67, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0d, 0x69, 0x6e, 0x6b, 0x2d, 0x63, 0x61, 0x72,
	0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x21, 0x00, 0x11, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x72, 0x2d,
	0x6c, 0x6f, 0x77, 0x2d, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x73, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x01, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x21, 0x00, 0x12, 0x6d, 0x61,
	0x72, 0x6b, 0x65, 0x72, 0x2d, 0x68, 0x69, 0x67, 0x68, 0x2d, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x73,
	0x00, 0x04, 0x00, 0x00, 0x00, 0x64, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x64, 0x21,
	0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x64, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x64, 0x21, 0x00, 0x0d, 0x6d, 0x61, 0x72, 0x6b, 0x65, 0x72, 0x2d, 0x6c, 0x65, 0x76, 0x65, 0x6c,
	0x73, 0x00, 0x04, 0x00, 0x00, 0x00, 0x3c, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x46,
	0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x3c, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x46, 0x30, 0x00, 0x0e, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x75, 0x70,
	0x70, 0x6c, 0x79, 0x00, 0x65, 0x74, 0x79, 0x70, 0x65, 0x3d, 0x69, 0x6e, 0x6b, 0x43, 0x61, 0x72,
	0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x3b, 0x6d, 0x61, 0x78, 0x63, 0x61, 0x70, 0x61, 0x63, 0x69,
	0x74, 0x79, 0x3d, 0x31, 0x30, 0x30, 0x3b, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x3d, 0x36, 0x30, 0x3b,
	0x63, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x73, 0x75, 0x70, 0x70, 0x6c, 0x79, 0x54, 0x68, 0x61, 0x74,
	0x49, 0x73, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x64, 0x3b, 0x75, 0x6e, 0x69, 0x74, 0x3d,
	0x70, 0x65, 0x72, 0x63, 0x65, 0x6e, 0x74, 0x3b, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x61, 0x6e, 0x74,
	0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x63, 0x79, 0x61, 0x6e, 0x3b, 0x30, 0x00, 0x00, 0x00, 0x68, 0x74,
	0x79, 0x70, 0x65, 0x3d, 0x69, 0x6e, 0x6b, 0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65,
	0x3b, 0x6d, 0x61, 0x78, 0x63, 0x61, 0x70, 0x61, 0x63, 0x69, 0x74, 0x79, 0x3d, 0x31, 0x30, 0x30,
	0x3b, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x3d, 0x37, 0x30, 0x3b, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x3d,
	0x73, 0x75, 0x70, 0x70, 0x6c, 0x79, 0x54, 0x68, 0x61, 0x74, 0x49, 0x73, 0x43, 0x6f, 0x6e, 0x73,
	0x75, 0x6d, 0x65, 0x64, 0x3b, 0x75, 0x6e, 0x69, 0x74, 0x3d, 0x70, 0x65, 0x72, 0x63, 0x65, 0x6e,
	0x74, 0x3b, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x61, 0x6e, 0x74, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x6d,
	0x61, 0x67, 0x65, 0x6e, 0x74, 0x61, 0x3b, 0x30, 0x00, 0x00, 0x00, 0x67, 0x74, 0x79, 0x70, 0x65,
	0x3d, 0x69, 0x6e, 0x6b, 0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x3b, 0x6d, 0x61,
	0x78, 0x63, 0x61, 0x70, 0x61, 0x63, 0x69, 0x74, 0x79, 0x3d, 0x31, 0x30, 0x30, 0x3b, 0x6c, 0x65,
	0x76, 0x65, 0x6c, 0x3d, 0x36, 0x30, 0x3b, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x73, 0x75, 0x70,
	0x70, 0x6c, 0x79, 0x54, 0x68, 0x61, 0x74, 0x49, 0x73, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65,
	0x64, 0x3b, 0x75, 0x6e, 0x69, 0x74, 0x3d, 0x70, 0x65, 0x72, 0x63, 0x65, 0x6e, 0x74, 0x3b, 0x63,
	0x6f, 0x6c, 0x6f, 0x72, 0x61, 0x6e, 0x74, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x79, 0x65, 0x6c, 0x6c,
	0x6f, 0x77, 0x3b, 0x30, 0x00, 0x00, 0x00, 0x66, 0x74, 0x79, 0x70, 0x65, 0x3d, 0x69, 0x6e, 0x6b,
	0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x3b, 0x6d, 0x61, 0x78, 0x63, 0x61, 0x70,
	0x61, 0x63, 0x69, 0x74, 0x79, 0x3d, 0x31, 0x30, 0x30, 0x3b, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x3d,
	0x37, 0x30, 0x3b, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x73, 0x75, 0x70, 0x70, 0x6c, 0x79, 0x54,
	0x68, 0x61, 0x74, 0x49, 0x73, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x65, 0x64, 0x3b, 0x75, 0x6e,
	0x69, 0x74, 0x3d, 0x70, 0x65, 0x72, 0x63, 0x65, 0x6e, 0x74, 0x3b, 0x63, 0x6f, 0x6c, 0x6f, 0x72,
	0x61, 0x6e, 0x74, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x62, 0x6c, 0x61, 0x63, 0x6b, 0x3b, 0x41, 0x00,
	0x1a, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6c, 0x79, 0x2d,
	0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x00, 0x0e, 0x43, 0x79, 0x61,
	0x6e, 0x20, 0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x41, 0x00, 0x00, 0x00, 0x11,
	0x4d, 0x61, 0x67, 0x65, 0x6e, 0x74, 0x61, 0x20, 0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67,
	0x65, 0x41, 0x00, 0x00, 0x00, 0x10, 0x59, 0x65, 0x6c, 0x6c, 0x6f, 0x77, 0x20, 0x43, 0x61, 0x72,
	0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x41, 0x00, 0x00, 0x00, 0x0f, 0x42, 0x6c, 0x61, 0x63, 0x6b,
	0x20, 0x43, 0x61, 0x72, 0x74, 0x72, 0x69, 0x64, 0x67, 0x65, 0x42, 0x00, 0x15, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x2d, 0x66, 0x69, 0x72, 0x6d, 0x77, 0x61, 0x72, 0x65, 0x2d, 0x6e, 0x61,
	0x6d, 0x65, 0x00, 0x16, 0x57, 0x45, 0x42, 0x50, 0x44, 0x4c, 0x50, 0x50, 0x31, 0x4e, 0x30, 0x30,
	0x31, 0x2e, 0x31, 0x39, 0x31, 0x39, 0x41, 0x2e, 0x30, 0x30, 0x41, 0x00, 0x1f, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x2d, 0x66, 0x69, 0x72, 0x6d, 0x77, 0x61, 0x72, 0x65, 0x2d, 0x73, 0x74,
	0x72, 0x69, 0x6e, 0x67, 0x2d, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x16, 0x57, 0x45,
	0x42, 0x50, 0x44, 0x4c, 0x50, 0x50, 0x31, 0x4e, 0x30, 0x30, 0x31, 0x2e, 0x31, 0x39, 0x31, 0x39,
	0x41, 0x2e, 0x30, 0x30, 0x30, 0x00, 0x18, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x66,
	0x69, 0x72, 0x6d, 0x77, 0x61, 0x72, 0x65, 0x2d, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x00,
	0x16, 0x57, 0x45, 0x42, 0x50, 0x44, 0x4c, 0x50, 0x50, 0x31, 0x4e, 0x30, 0x30, 0x31, 0x2e, 0x31,
	0x39, 0x31, 0x39, 0x41, 0x2e, 0x30, 0x30, 0x30, 0x00, 0x12, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x2d, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x2d, 0x74, 0x72, 0x61, 0x79, 0x00, 0x86, 0x74, 0x79,
	0x70, 0x65, 0x3d, 0x73, 0x68, 0x65, 0x65, 0x74, 0x46, 0x65, 0x65, 0x64, 0x41, 0x75, 0x74, 0x6f,
	0x4e, 0x6f, 0x6e, 0x52, 0x65, 0x6d, 0x6f, 0x76, 0x61, 0x62, 0x6c, 0x65, 0x3b, 0x64, 0x69, 0x6d,
	0x75, 0x6e, 0x69, 0x74, 0x3d, 0x6d, 0x69, 0x63, 0x72, 0x6f, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x73,
	0x3b, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x66, 0x65, 0x65, 0x64, 0x3d, 0x32, 0x39, 0x37, 0x30, 0x30,
	0x30, 0x3b, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x78, 0x66, 0x65, 0x65, 0x64, 0x3d, 0x32, 0x31, 0x30,
	0x30, 0x30, 0x30, 0x3b, 0x6d, 0x61, 0x78, 0x63, 0x61, 0x70, 0x61, 0x63, 0x69, 0x74, 0x79, 0x3d,
	0x2d, 0x32, 0x3b, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x3d, 0x2d, 0x32, 0x3b, 0x73, 0x74, 0x61, 0x74,
	0x75, 0x73, 0x3d, 0x30, 0x3b, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x49, 0x6e, 0x70, 0x75, 0x74, 0x54,
	0x72, 0x61, 0x79, 0x31, 0x30, 0x00, 0x13, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6f,
	0x75, 0x74, 0x70, 0x75, 0x74, 0x2d, 0x74, 0x72, 0x61, 0x79, 0x00, 0x6e, 0x74, 0x79, 0x70, 0x65,
	0x3d, 0x75, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x3b, 0x6d, 0x61, 0x78, 0x63, 0x61, 0x70, 0x61,
	0x63, 0x69, 0x74, 0x79, 0x3d, 0x2d, 0x32, 0x3b, 0x72, 0x65, 0x6d, 0x61, 0x69, 0x6e, 0x69, 0x6e,
	0x67, 0x3d, 0x2d, 0x32, 0x3b, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x3d, 0x35, 0x3b, 0x73, 0x74,
	0x61, 0x63, 0x6b, 0x69, 0x6e, 0x67, 0x6f, 0x72, 0x64, 0x65, 0x72, 0x3d, 0x75, 0x6e, 0x6b, 0x6e,
	0x6f, 0x77, 0x6e, 0x3b, 0x70, 0x61, 0x67, 0x65, 0x64, 0x65, 0x6c, 0x69, 0x76, 0x65, 0x72, 0x79,
	0x3d, 0x66, 0x61, 0x63, 0x65, 0x44, 0x6f, 0x77, 0x6e, 0x3b, 0x6e, 0x61, 0x6d, 0x65, 0x3d, 0x4f,
	0x75, 0x74, 0x70, 0x75, 0x74, 0x54, 0x72, 0x61, 0x79, 0x31, 0x21, 0x00, 0x0e, 0x63, 0x6f, 0x70,
	0x69, 0x65, 0x73, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x01, 0x23, 0x00, 0x12, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x64,
	0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x13, 0x00, 0x16, 0x66,
	0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x2d, 0x64, 0x65,
	0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x00, 0x23, 0x00, 0x1d, 0x6f, 0x72, 0x69, 0x65, 0x6e, 0x74,
	0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x65, 0x64, 0x2d,
	0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x23, 0x00, 0x15,
	0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x71, 0x75, 0x61, 0x6c, 0x69, 0x74, 0x79, 0x2d, 0x64, 0x65,
	0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x32, 0x00, 0x1a, 0x70, 0x72,
	0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e,
	0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x09, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00,
	0x02, 0x58, 0x03, 0x44, 0x00, 0x0d, 0x73, 0x69, 0x64, 0x65, 0x73, 0x2d, 0x64, 0x65, 0x66, 0x61,
	0x75, 0x6c, 0x74, 0x00, 0x09, 0x6f, 0x6e, 0x65, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x44, 0x00,
	0x12, 0x6f, 0x75, 0x74, 0x70, 0x75, 0x74, 0x2d, 0x62, 0x69, 0x6e, 0x2d, 0x64, 0x65, 0x66, 0x61,
	0x75, 0x6c, 0x74, 0x00, 0x09, 0x66, 0x61, 0x63, 0x65, 0x2d, 0x64, 0x6f, 0x77, 0x6e, 0x44, 0x00,
	0x13, 0x6f, 0x75, 0x74, 0x70, 0x75, 0x74, 0x2d, 0x6d, 0x6f, 0x64, 0x65, 0x2d, 0x64, 0x65, 0x66,
	0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x18, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x2d, 0x6d, 0x6f, 0x64, 0x65, 0x2d, 0x64, 0x65,
	0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x22, 0x6d, 0x75,
	0x6c, 0x74, 0x69, 0x70, 0x6c, 0x65, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d,
	0x68, 0x61, 0x6e, 0x64, 0x6c, 0x69, 0x6e, 0x67, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74,
	0x00, 0x24, 0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d,
	0x65, 0x6e, 0x74, 0x73, 0x2d, 0x75, 0x6e, 0x63, 0x6f, 0x6c, 0x6c, 0x61, 0x74, 0x65, 0x64, 0x2d,
	0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x21, 0x00, 0x11, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x2d,
	0x75, 0x70, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01,
	0x44, 0x00, 0x28, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d,
	0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x6e, 0x75, 0x6d, 0x62, 0x65, 0x72,
	0x2d, 0x75, 0x70, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x10, 0x74, 0x6f, 0x72,
	0x69, 0x67, 0x68, 0x74, 0x2d, 0x74, 0x6f, 0x62, 0x6f, 0x74, 0x74, 0x6f, 0x6d, 0x44, 0x00, 0x1e,
	0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x72, 0x65, 0x6e, 0x64, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x2d,
	0x69, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04,
	0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x15, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x73, 0x63, 0x61,
	0x6c, 0x69, 0x6e, 0x67, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x61, 0x75,
	0x74, 0x6f, 0x33, 0x00, 0x10, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70,
	0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x63, 0x23,
	0x00, 0x14, 0x66, 0x69, 0x6e, 0x69, 0x73, 0x68, 0x69, 0x6e, 0x67, 0x73, 0x2d, 0x73, 0x75, 0x70,
	0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x33, 0x00, 0x1b, 0x6a,
	0x6f, 0x62, 0x2d, 0x70, 0x61, 0x67, 0x65, 0x73, 0x2d, 0x70, 0x65, 0x72, 0x2d, 0x73, 0x65, 0x74,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01,
	0x7f, 0xff, 0xff, 0xff, 0x23, 0x00, 0x1f, 0x6f, 0x72, 0x69, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69,
	0x6f, 0x6e, 0x2d, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x65, 0x64, 0x2d, 0x73, 0x75, 0x70,
	0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x23, 0x00, 0x00, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x04, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x05, 0x23, 0x00,
	0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x06, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x07,
	0x23, 0x00, 0x17, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x71, 0x75, 0x61, 0x6c, 0x69, 0x74, 0x79,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03,
	0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x23, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
	0x00, 0x05, 0x32, 0x00, 0x1c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x72, 0x65, 0x73,
	0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x09, 0x00, 0x00, 0x01, 0x2c, 0x00, 0x00, 0x01, 0x2c, 0x03, 0x32, 0x00, 0x00, 0x00,
	0x09, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x02, 0x58, 0x03, 0x32, 0x00, 0x00, 0x00, 0x09, 0x00,
	0x00, 0x04, 0xb0, 0x00, 0x00, 0x04, 0xb0, 0x03, 0x44, 0x00, 0x0f, 0x73, 0x69, 0x64, 0x65, 0x73,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x09, 0x6f, 0x6e, 0x65, 0x2d,
	0x73, 0x69, 0x64, 0x65, 0x64, 0x44, 0x00, 0x00, 0x00, 0x14, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69,
	0x64, 0x65, 0x64, 0x2d, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x44, 0x00,
	0x00, 0x00, 0x13, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x2d, 0x6c, 0x6f, 0x6e,
	0x67, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x44, 0x00, 0x14, 0x6f, 0x75, 0x74, 0x70, 0x75, 0x74, 0x2d,
	0x62, 0x69, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x09, 0x66,
	0x61, 0x63, 0x65, 0x2d, 0x64, 0x6f, 0x77, 0x6e, 0x44, 0x00, 0x15, 0x6f, 0x75, 0x74, 0x70, 0x75,
	0x74, 0x2d, 0x6d, 0x6f, 0x64, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x04, 0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x00, 0x00, 0x0f, 0x61, 0x75, 0x74, 0x6f, 0x2d,
	0x6d, 0x6f, 0x6e, 0x6f, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x6d,
	0x6f, 0x6e, 0x6f, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x05, 0x63, 0x6f,
	0x6c, 0x6f, 0x72, 0x44, 0x00, 0x1a, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6c, 0x6f,
	0x72, 0x2d, 0x6d, 0x6f, 0x64, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64,
	0x00, 0x04, 0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x00, 0x00, 0x0f, 0x61, 0x75, 0x74, 0x6f, 0x2d,
	0x6d, 0x6f, 0x6e, 0x6f, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x6d,
	0x6f, 0x6e, 0x6f, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x65, 0x44, 0x00, 0x00, 0x00, 0x05, 0x63, 0x6f,
	0x6c, 0x6f, 0x72, 0x44, 0x00, 0x00, 0x00, 0x12, 0x70, 0x72, 0x6f, 0x63, 0x65, 0x73, 0x73, 0x2d,
	0x6d, 0x6f, 0x6e, 0x6f, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x65, 0x22, 0x00, 0x15, 0x70, 0x61, 0x67,
	0x65, 0x2d, 0x72, 0x61, 0x6e, 0x67, 0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
	0x65, 0x64, 0x00, 0x01, 0x01, 0x44, 0x00, 0x24, 0x6d, 0x75, 0x6c, 0x74, 0x69, 0x70, 0x6c, 0x65,
	0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x69,
	0x6e, 0x67, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x24, 0x73, 0x65,
	0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x73,
	0x2d, 0x75, 0x6e, 0x63, 0x6f, 0x6c, 0x6c, 0x61, 0x74, 0x65, 0x64, 0x2d, 0x63, 0x6f, 0x70, 0x69,
	0x65, 0x73, 0x44, 0x00, 0x00, 0x00, 0x22, 0x73, 0x65, 0x70, 0x61, 0x72, 0x61, 0x74, 0x65, 0x2d,
	0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0x2d, 0x63, 0x6f, 0x6c, 0x6c, 0x61, 0x74,
	0x65, 0x64, 0x2d, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x21, 0x00, 0x13, 0x6e, 0x75, 0x6d, 0x62,
	0x65, 0x72, 0x2d, 0x75, 0x70, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x01, 0x44, 0x00, 0x2a, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x6e,
	0x75, 0x6d, 0x62, 0x65, 0x72, 0x2d, 0x75, 0x70, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
	0x65, 0x64, 0x00, 0x10, 0x74, 0x6f, 0x72, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x74, 0x6f, 0x62, 0x6f,
	0x74, 0x74, 0x6f, 0x6d, 0x44, 0x00, 0x13, 0x6f, 0x76, 0x65, 0x72, 0x72, 0x69, 0x64, 0x65, 0x73,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x05, 0x70, 0x61, 0x67, 0x65,
	0x73, 0x44, 0x00, 0x00, 0x00, 0x05, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x44, 0x00, 0x00, 0x00, 0x09,
	0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x63, 0x6f, 0x6c, 0x44, 0x00, 0x20, 0x70, 0x72, 0x69, 0x6e,
	0x74, 0x2d, 0x72, 0x65, 0x6e, 0x64, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x2d, 0x69, 0x6e, 0x74, 0x65,
	0x6e, 0x74, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x61, 0x75,
	0x74, 0x6f, 0x44, 0x00, 0x00, 0x00, 0x0a, 0x70, 0x65, 0x72, 0x63, 0x65, 0x70, 0x74, 0x75, 0x61,
	0x6c, 0x44, 0x00, 0x17, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x73, 0x63, 0x61, 0x6c, 0x69, 0x6e,
	0x67, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x61, 0x75, 0x74,
	0x6f, 0x44, 0x00, 0x00, 0x00, 0x08, 0x61, 0x75, 0x74, 0x6f, 0x2d, 0x66, 0x69, 0x74, 0x44, 0x00,
	0x00, 0x00, 0x04, 0x66, 0x69, 0x6c, 0x6c, 0x44, 0x00, 0x00, 0x00, 0x03, 0x66, 0x69, 0x74, 0x44,
	0x00, 0x00, 0x00, 0x04, 0x6e, 0x6f, 0x6e, 0x65, 0x45, 0x00, 0x0d, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x00, 0x31, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x2f, 0x77, 0x65, 0x62, 0x41, 0x70,
	0x70, 0x73, 0x2f, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x2f, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x2d, 0x73, 0x6d, 0x61, 0x6c, 0x6c, 0x2e, 0x70, 0x6e, 0x67, 0x45, 0x00, 0x00, 0x00, 0x2b,
	0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74,
	0x2f, 0x77, 0x65, 0x62, 0x41, 0x70, 0x70, 0x73, 0x2f, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x2f,
	0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2e, 0x70, 0x6e, 0x67, 0x45, 0x00, 0x00, 0x00, 0x31,
	0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74,
	0x2f, 0x77, 0x65, 0x62, 0x41, 0x70, 0x70, 0x73, 0x2f, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x73, 0x2f,
	0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6c, 0x61, 0x72, 0x67, 0x65, 0x2e, 0x70, 0x6e,
	0x67, 0x45, 0x00, 0x17, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x73, 0x75, 0x70, 0x70,
	0x6c, 0x79, 0x2d, 0x69, 0x6e, 0x66, 0x6f, 0x2d, 0x75, 0x72, 0x69, 0x00, 0x26, 0x68, 0x74, 0x74,
	0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x2f, 0x23, 0x68,
	0x49, 0x64, 0x2d, 0x70, 0x67, 0x49, 0x6e, 0x6b, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6d, 0x61, 0x62,
	0x6c, 0x65, 0x73, 0x44, 0x00, 0x1e, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6e, 0x74,
	0x65, 0x6e, 0x74, 0x2d, 0x6f, 0x70, 0x74, 0x69, 0x6d, 0x69, 0x7a, 0x65, 0x2d, 0x64, 0x65, 0x66,
	0x61, 0x75, 0x6c, 0x74, 0x00, 0x04, 0x61, 0x75, 0x74, 0x6f, 0x44, 0x00, 0x20, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x2d, 0x63, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x6f, 0x70, 0x74, 0x69, 0x6d,
	0x69, 0x7a, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x61,
	0x75, 0x74, 0x6f, 0x44, 0x00, 0x00, 0x00, 0x05, 0x70, 0x68, 0x6f, 0x74, 0x6f, 0x44, 0x00, 0x00,
	0x00, 0x08, 0x67, 0x72, 0x61, 0x70, 0x68, 0x69, 0x63, 0x73, 0x44, 0x00, 0x00, 0x00, 0x04, 0x74,
	0x65, 0x78, 0x74, 0x44, 0x00, 0x00, 0x00, 0x11, 0x74, 0x65, 0x78, 0x74, 0x2d, 0x61, 0x6e, 0x64,
	0x2d, 0x67, 0x72, 0x61, 0x70, 0x68, 0x69, 0x63, 0x73, 0x44, 0x00, 0x1e, 0x70, 0x77, 0x67, 0x2d,
	0x72, 0x61, 0x73, 0x74, 0x65, 0x72, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d,
	0x73, 0x68, 0x65, 0x65, 0x74, 0x2d, 0x62, 0x61, 0x63, 0x6b, 0x00, 0x07, 0x72, 0x6f, 0x74, 0x61,
	0x74, 0x65, 0x64, 0x44, 0x00, 0x22, 0x70, 0x77, 0x67, 0x2d, 0x72, 0x61, 0x73, 0x74, 0x65, 0x72,
	0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x74, 0x79, 0x70, 0x65, 0x2d, 0x73,
	0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x07, 0x73, 0x67, 0x72, 0x61, 0x79, 0x5f,
	0x38, 0x44, 0x00, 0x00, 0x00, 0x06, 0x73, 0x72, 0x67, 0x62, 0x5f, 0x38, 0x44, 0x00, 0x00, 0x00,
	0x0b, 0x61, 0x64, 0x6f, 0x62, 0x65, 0x2d, 0x72, 0x67, 0x62, 0x5f, 0x38, 0x44, 0x00, 0x00, 0x00,
	0x05, 0x72, 0x67, 0x62, 0x5f, 0x38, 0x32, 0x00, 0x28, 0x70, 0x77, 0x67, 0x2d, 0x72, 0x61, 0x73,
	0x74, 0x65, 0x72, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x72, 0x65, 0x73,
	0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x09, 0x00, 0x00, 0x01, 0x2c, 0x00, 0x00, 0x01, 0x2c, 0x03, 0x32, 0x00, 0x00, 0x00,
	0x09, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x02, 0x58, 0x03, 0x41, 0x00, 0x16, 0x65, 0x70, 0x63,
	0x6c, 0x2d, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72,
	0x74, 0x65, 0x64, 0x00, 0x03, 0x31, 0x2e, 0x30, 0x22, 0x00, 0x17, 0x6d, 0x61, 0x6e, 0x75, 0x61,
	0x6c, 0x2d, 0x64, 0x75, 0x70, 0x6c, 0x65, 0x78, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74,
	0x65, 0x64, 0x00, 0x01, 0x00, 0x32, 0x00, 0x20, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x73, 0x6f, 0x75,
	0x72, 0x63, 0x65, 0x2d, 0x72, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x73,
	0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x09, 0x00, 0x00, 0x01, 0x2c, 0x00, 0x00,
	0x01, 0x2c, 0x03, 0x32, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x02, 0x58,
	0x03, 0x32, 0x00, 0x1e, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x2d,
	0x72, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75,
	0x6c, 0x74, 0x00, 0x09, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x02, 0x58, 0x03, 0x21, 0x00, 0x1b,
	0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x73, 0x74, 0x72, 0x69, 0x70, 0x2d, 0x68, 0x65, 0x69, 0x67, 0x68,
	0x74, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x20, 0x21, 0x00, 0x1b, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x73, 0x74, 0x72, 0x69, 0x70, 0x2d, 0x68,
	0x65, 0x69, 0x67, 0x68, 0x74, 0x2d, 0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x20, 0x44, 0x00, 0x15, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x72, 0x61, 0x73,
	0x74, 0x65, 0x72, 0x2d, 0x62, 0x61, 0x63, 0x6b, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x00, 0x07, 0x72,
	0x6f, 0x74, 0x61, 0x74, 0x65, 0x64, 0x44, 0x00, 0x21, 0x70, 0x63, 0x6c, 0x6d, 0x2d, 0x63, 0x6f,
	0x6d, 0x70, 0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x2d, 0x6d, 0x65, 0x74, 0x68, 0x6f, 0x64,
	0x2d, 0x70, 0x72, 0x65, 0x66, 0x65, 0x72, 0x72, 0x65, 0x64, 0x00, 0x04, 0x6a, 0x70, 0x65, 0x67,
	0x44, 0x00, 0x00, 0x00, 0x05, 0x66, 0x6c, 0x61, 0x74, 0x65, 0x44, 0x00, 0x00, 0x00, 0x03, 0x72,
	0x6c, 0x65, 0x44, 0x00, 0x22, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x66, 0x6f,
	0x72, 0x6d, 0x61, 0x74, 0x2d, 0x76, 0x61, 0x72, 0x79, 0x69, 0x6e, 0x67, 0x2d, 0x61, 0x74, 0x74,
	0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x00, 0x06, 0x63, 0x6f, 0x70, 0x69, 0x65, 0x73, 0x44,
	0x00, 0x00, 0x00, 0x15, 0x6f, 0x72, 0x69, 0x65, 0x6e, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d,
	0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x65, 0x64, 0x44, 0x00, 0x20, 0x70, 0x72, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x2d, 0x67, 0x65, 0x74, 0x2d, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
	0x65, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x0f, 0x64, 0x6f,
	0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x41, 0x00, 0x14,
	0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x00, 0x00, 0x41, 0x00, 0x1b, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x2d, 0x6f, 0x72, 0x67, 0x61, 0x6e, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x61, 0x6c, 0x2d,
	0x75, 0x6e, 0x69, 0x74, 0x00, 0x00, 0x44, 0x00, 0x18, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66,
	0x79, 0x2d, 0x61, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c,
	0x74, 0x00, 0x07, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79, 0x44, 0x00, 0x1a, 0x69, 0x64, 0x65,
	0x6e, 0x74, 0x69, 0x66, 0x79, 0x2d, 0x61, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x73, 0x75,
	0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x07, 0x64, 0x69, 0x73, 0x70, 0x6c, 0x61, 0x79,
	0x44, 0x00, 0x00, 0x00, 0x05, 0x73, 0x6f, 0x75, 0x6e, 0x64, 0x44, 0x00, 0x00, 0x00, 0x03, 0x70,
	0x69, 0x6e, 0x23, 0x00, 0x1a, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x2d, 0x6f, 0x70, 0x65, 0x72, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x0a, 0x21, 0x00, 0x1b, 0x6d, 0x75, 0x6c, 0x74, 0x69, 0x70, 0x6c, 0x65,
	0x2d, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2d, 0x74, 0x69, 0x6d, 0x65, 0x2d,
	0x6f, 0x75, 0x74, 0x00, 0x04, 0x00, 0x00, 0x00, 0x78, 0x22, 0x00, 0x20, 0x6d, 0x75, 0x6c, 0x74,
	0x69, 0x70, 0x6c, 0x65, 0x2d, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d, 0x6a, 0x6f,
	0x62, 0x73, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x01, 0x00, 0x03,
}

// Get-Printer-Attributes output from Pantum M7300FDW
var attrsPantumM7300FDW = []byte{
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x47, 0x00, 0x12,
	0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x73, 0x2d, 0x63,
	0x68, 0x61, 0x72, 0x73, 0x65, 0x74, 0x00, 0x05, 0x75, 0x74, 0x66, 0x2d,
	0x38, 0x48, 0x00, 0x1b, 0x61, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
	0x65, 0x73, 0x2d, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x61, 0x6c, 0x2d, 0x6c,
	0x61, 0x6e, 0x67, 0x75, 0x61, 0x67, 0x65, 0x00, 0x05, 0x65, 0x6e, 0x2d,
	0x55, 0x53, 0x04, 0x22, 0x00, 0x0f, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x2d,
	0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x01, 0x00,
	0x49, 0x00, 0x19, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x2d,
	0x66, 0x6f, 0x72, 0x6d, 0x61, 0x74, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f,
	0x72, 0x74, 0x65, 0x64, 0x00, 0x09, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x2f,
	0x75, 0x72, 0x66, 0x49, 0x00, 0x00, 0x00, 0x10, 0x69, 0x6d, 0x61, 0x67,
	0x65, 0x2f, 0x70, 0x77, 0x67, 0x2d, 0x72, 0x61, 0x73, 0x74, 0x65, 0x72,
	0x49, 0x00, 0x00, 0x00, 0x18, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63, 0x61,
	0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x6f, 0x63, 0x74, 0x65, 0x74, 0x2d, 0x73,
	0x74, 0x72, 0x65, 0x61, 0x6d, 0x41, 0x00, 0x10, 0x6d, 0x6f, 0x70, 0x72,
	0x69, 0x61, 0x2d, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x65, 0x64,
	0x00, 0x03, 0x31, 0x2e, 0x33, 0x41, 0x00, 0x10, 0x70, 0x72, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x2d, 0x6c, 0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e,
	0x00, 0x0e, 0x70, 0x61, 0x6e, 0x74, 0x75, 0x6d, 0x20, 0x70, 0x72, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x41, 0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x69, 0x6e, 0x66, 0x6f, 0x00, 0x12, 0x50, 0x61, 0x6e,
	0x74, 0x75, 0x6d, 0x20, 0x49, 0x50, 0x50, 0x20, 0x70, 0x72, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x45, 0x00, 0x11, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x2d, 0x6d, 0x6f, 0x72, 0x65, 0x2d, 0x69, 0x6e, 0x66, 0x6f, 0x00,
	0x26, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61,
	0x6c, 0x68, 0x6f, 0x73, 0x74, 0x3a, 0x36, 0x30, 0x30, 0x30, 0x30, 0x2f,
	0x69, 0x6e, 0x64, 0x65, 0x78, 0x2d, 0x6a, 0x75, 0x6d, 0x70, 0x2e, 0x68,
	0x74, 0x6d, 0x6c, 0x41, 0x00, 0x16, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x2d, 0x6d, 0x61, 0x6b, 0x65, 0x2d, 0x61, 0x6e, 0x64, 0x2d, 0x6d,
	0x6f, 0x64, 0x65, 0x6c, 0x00, 0x16, 0x50, 0x61, 0x6e, 0x74, 0x75, 0x6d,
	0x20, 0x4d, 0x37, 0x33, 0x30, 0x30, 0x46, 0x44, 0x57, 0x20, 0x53, 0x65,
	0x72, 0x69, 0x65, 0x73, 0x41, 0x00, 0x11, 0x70, 0x72, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x2d, 0x64, 0x65, 0x76, 0x69, 0x63, 0x65, 0x2d, 0x69, 0x64,
	0x00, 0x45, 0x4d, 0x46, 0x47, 0x3a, 0x50, 0x61, 0x6e, 0x74, 0x75, 0x6d,
	0x3b, 0x43, 0x4d, 0x44, 0x3a, 0x3a, 0x44, 0x57, 0x2d, 0x50, 0x53, 0x2c,
	0x44, 0x57, 0x2d, 0x50, 0x43, 0x4c, 0x2c, 0x55, 0x52, 0x46, 0x3b, 0x4d,
	0x44, 0x4c, 0x3a, 0x2d, 0x50, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x3b,
	0x43, 0x4c, 0x53, 0x3a, 0x50, 0x52, 0x49, 0x4e, 0x54, 0x45, 0x52, 0x3b,
	0x44, 0x45, 0x53, 0x3a, 0x50, 0x61, 0x6e, 0x74, 0x75, 0x6d, 0x3b, 0x45,
	0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x75, 0x75,
	0x69, 0x64, 0x00, 0x2d, 0x75, 0x72, 0x6e, 0x3a, 0x75, 0x75, 0x69, 0x64,
	0x3a, 0x30, 0x37, 0x62, 0x63, 0x35, 0x61, 0x34, 0x37, 0x2d, 0x65, 0x64,
	0x66, 0x32, 0x2d, 0x34, 0x36, 0x36, 0x31, 0x2d, 0x38, 0x37, 0x34, 0x38,
	0x2d, 0x38, 0x36, 0x32, 0x30, 0x66, 0x37, 0x39, 0x64, 0x34, 0x61, 0x63,
	0x34, 0x45, 0x00, 0x0d, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d,
	0x69, 0x63, 0x6f, 0x6e, 0x73, 0x00, 0x25, 0x68, 0x74, 0x74, 0x70, 0x3a,
	0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x3a,
	0x36, 0x30, 0x30, 0x30, 0x30, 0x2f, 0x66, 0x6c, 0x61, 0x67, 0x73, 0x68,
	0x69, 0x70, 0x34, 0x38, 0x2e, 0x70, 0x6e, 0x67, 0x45, 0x00, 0x00, 0x00,
	0x26, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61,
	0x6c, 0x68, 0x6f, 0x73, 0x74, 0x3a, 0x36, 0x30, 0x30, 0x30, 0x30, 0x2f,
	0x66, 0x6c, 0x61, 0x67, 0x73, 0x68, 0x69, 0x70, 0x31, 0x32, 0x38, 0x2e,
	0x70, 0x6e, 0x67, 0x45, 0x00, 0x00, 0x00, 0x26, 0x68, 0x74, 0x74, 0x70,
	0x3a, 0x2f, 0x2f, 0x6c, 0x6f, 0x63, 0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74,
	0x3a, 0x36, 0x30, 0x30, 0x30, 0x30, 0x2f, 0x66, 0x6c, 0x61, 0x67, 0x73,
	0x68, 0x69, 0x70, 0x35, 0x31, 0x32, 0x2e, 0x70, 0x6e, 0x67, 0x42, 0x00,
	0x13, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x64, 0x6e, 0x73,
	0x2d, 0x73, 0x64, 0x2d, 0x6e, 0x61, 0x6d, 0x65, 0x00, 0x1d, 0x50, 0x61,
	0x6e, 0x74, 0x75, 0x6d, 0x20, 0x4d, 0x37, 0x33, 0x30, 0x30, 0x46, 0x44,
	0x57, 0x20, 0x53, 0x65, 0x72, 0x69, 0x65, 0x73, 0x20, 0x31, 0x43, 0x36,
	0x46, 0x43, 0x44, 0x44, 0x00, 0x0d, 0x75, 0x72, 0x66, 0x2d, 0x73, 0x75,
	0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x04, 0x56, 0x31, 0x2e,
	0x34, 0x44, 0x00, 0x00, 0x00, 0x02, 0x57, 0x38, 0x44, 0x00, 0x00, 0x00,
	0x03, 0x49, 0x53, 0x31, 0x44, 0x00, 0x00, 0x00, 0x04, 0x43, 0x50, 0x39,
	0x39, 0x44, 0x00, 0x00, 0x00, 0x03, 0x50, 0x51, 0x34, 0x44, 0x00, 0x00,
	0x00, 0x04, 0x4f, 0x42, 0x31, 0x30, 0x44, 0x00, 0x00, 0x00, 0x05, 0x52,
	0x53, 0x36, 0x30, 0x30, 0x44, 0x00, 0x00, 0x00, 0x03, 0x44, 0x4d, 0x31,
	0x44, 0x00, 0x0c, 0x70, 0x72, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x2d, 0x6b,
	0x69, 0x6e, 0x64, 0x00, 0x08, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e,
	0x74, 0x44, 0x00, 0x00, 0x00, 0x08, 0x65, 0x6e, 0x76, 0x65, 0x6c, 0x6f,
	0x70, 0x65, 0x34, 0x00, 0x14, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x2d, 0x73,
	0x69, 0x7a, 0x65, 0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65,
	0x64, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x54, 0x56, 0x21,
	0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x00, 0x04, 0x00, 0x00, 0x6d, 0x24, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69,
	0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x52,
	0x08, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73,
	0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x74, 0x04, 0x37, 0x00, 0x00,
	0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00,
	0x00, 0x39, 0xd0, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x52, 0x08, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b,
	0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00,
	0x04, 0x00, 0x00, 0x29, 0x04, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69,
	0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x39,
	0xd0, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x21,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x00, 0x04, 0x00, 0x00, 0x2c, 0x88, 0x21, 0x00, 0x0b, 0x79, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00,
	0x00, 0x3f, 0x48, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
	0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73,
	0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x2a, 0xf8, 0x21, 0x00, 0x0b,
	0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00,
	0x04, 0x00, 0x00, 0x55, 0xf0, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00,
	0x00, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x27, 0x10, 0x21,
	0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x00, 0x04, 0x00, 0x00, 0x39, 0xd0, 0x37, 0x00, 0x00, 0x00, 0x00,
	0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69,
	0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x2e,
	0xe0, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73,
	0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x5b, 0xcc, 0x37, 0x00, 0x00,
	0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b, 0x78, 0x2d,
	0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00,
	0x00, 0x28, 0xed, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65,
	0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x5e, 0x42, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x0b,
	0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00,
	0x04, 0x00, 0x00, 0x54, 0x56, 0x21, 0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69,
	0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x04, 0x00, 0x00, 0x8a,
	0xe8, 0x37, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x00, 0x33,
	0x00, 0x0b, 0x78, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x00, 0x08, 0x00, 0x00, 0x27, 0x10, 0x00, 0x00, 0x39, 0xd0, 0x33,
	0x00, 0x0b, 0x79, 0x2d, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x73, 0x69, 0x6f,
	0x6e, 0x00, 0x08, 0x00, 0x00, 0x54, 0x60, 0x00, 0x00, 0x8b, 0x10, 0x37,
	0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x0f, 0x73, 0x69, 0x64, 0x65, 0x73,
	0x2d, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x65, 0x64, 0x00, 0x09,
	0x6f, 0x6e, 0x65, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x44, 0x00, 0x00,
	0x00, 0x14, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64, 0x2d,
	0x73, 0x68, 0x6f, 0x72, 0x74, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x44, 0x00,
	0x00, 0x00, 0x13, 0x74, 0x77, 0x6f, 0x2d, 0x73, 0x69, 0x64, 0x65, 0x64,
	0x2d, 0x6c, 0x6f, 0x6e, 0x67, 0x2d, 0x65, 0x64, 0x67, 0x65, 0x03,
}
