/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "globus_rls_client.h"
#include "version.h"
#include <stdio.h>
#include <string.h>
#include <sys/time.h>

static globus_module_descriptor_t	*modules[] = {
  GLOBUS_COMMON_MODULE,
  GLOBUS_IO_MODULE,
  GLOBUS_RLS_CLIENT_MODULE,
};					      
#define NMODS	(sizeof(modules) / sizeof(globus_module_descriptor_t *))

#define SHOW		0
#define LRCADD		1
#define LRCDELETE	2
#define PING		3
#define QUIT		4
#define SSU		5
#define LRCADDBF	6
#define STATS		7
#define GETCONF		8
#define CLEAR		9
#define SETCONF		10
#define RLIADD		11
#define RLIDELETE	12

static void	usage(char *prog);

int
main(int argc, char **argv)

{
  extern char			*optarg;
  extern int			optind;
  int				cmd;
  int				i;
  globus_rls_handle_t		*h;
  char				*rli;
  char				*pattern;
  char				*lrc;
  globus_result_t		r;
  int				rc;
  char				errmsg[MAXERRMSG];
  time_t			now;
  globus_list_t			*list;
  globus_list_t			*p;
  globus_rls_string2_t		*str2;
  globus_rls_stats_t		rlsstats;
  char				buf[100];
  globus_rls_rli_info_t		*rliinfo;
  globus_rls_sender_info_t	*senderinfo;
  char				*option;
  char				*value;
  int				timeout = 0;

  cmd = SHOW;
  while ((i = getopt(argc, argv, "AaC:c:DdepqrSst:uv")) != -1)
    switch (i) {
      case 'A': cmd = LRCADDBF;
		break;
      case 'a':	cmd = LRCADD;
		break;
      case 'C': cmd = SETCONF;
		option = optarg;
		if (optind >= argc - 1)
		  usage(*argv);
		value = argv[optind++];
		break;
      case 'c': cmd = GETCONF;
		if (strcmp(optarg, "all") == 0)
		  option = NULL;
		else
		  option = optarg;
		break;
      case 'D':	cmd = RLIDELETE;
		break;
      case 'd':	cmd = LRCDELETE;
		break;
      case 'e': cmd = CLEAR;
		break;
      case 'p': cmd = PING;
		break;
      case 'q': cmd = QUIT;
		break;
      case 'r':	cmd = RLIADD;
		break;
      case 'S':	cmd = STATS;
		break;
      case 's':	cmd = SHOW;
		break;
      case 't': timeout = atoi(optarg);
		break;
      case 'u': cmd = SSU;
		break;
      case 'v':	printf("Version: %d.%d\n", MAJOR, MINOR);
		exit(0);
      case '?':	usage(*argv);
		break;
    }
  if ((cmd == LRCADD || cmd == LRCDELETE || cmd == LRCADDBF ||
       cmd == RLIADD || cmd == RLIDELETE) && optind > argc - 2)
    usage(*argv);
  else if(optind > argc - 1)
    usage(*argv);
  rli = optind < argc - 1 ? argv[optind++] : NULL;
  pattern = optind < argc - 1 ? argv[optind++] : NULL;
  lrc = argv[optind];
  for (i = 0; i < NMODS; i++)
    if ((rc = globus_module_activate(modules[i])) != GLOBUS_SUCCESS) {
      fprintf(stderr, "globus_module_activate(%d): %d\n", rc, i);
      exit(1);
    }
  if (timeout)
    globus_rls_client_set_timeout(timeout);
  if ((r = globus_rls_client_connect(lrc, &h)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "connect(%s): %s\n", lrc, errmsg);
    exit(1);
  }

  switch (cmd) {
    case LRCADD:
      r = globus_rls_client_lrc_rli_add(h, rli, 0, pattern);
      break;

    case LRCADDBF:
      r = globus_rls_client_lrc_rli_add(h, rli, FRLI_BLOOMFILTER, NULL);
      break;

    case CLEAR:
      r = globus_rls_client_lrc_clear(h);
      break;

    case LRCDELETE:
      r = globus_rls_client_lrc_rli_delete(h, rli, pattern);
      break;

    case RLIDELETE:
      r = globus_rls_client_rli_rli_delete(h, rli, pattern);
      break;

    case GETCONF:
      r = globus_rls_client_get_configuration(h, option, &list);
      if (r == GLOBUS_SUCCESS) {
	for (p = list; p; p = globus_list_rest(p)) {
	  str2 = (globus_rls_string2_t *) globus_list_first(p);
	  printf("  %-24s %s\n", str2->s1, str2->s2);
	}
	globus_rls_client_free_list(list);
      }
      break;

    case RLIADD:
      r = globus_rls_client_rli_rli_add(h, rli, pattern);
      break;

    case SHOW:
      r = globus_rls_client_lrc_rli_get_part(h, rli, pattern, &list);
      if (r == GLOBUS_SUCCESS) {
	printf("LRC updates:\n");
	for (p = list; p; p = globus_list_rest(p)) {
	  str2 = (globus_rls_string2_t *) globus_list_first(p);
	  printf("  %-32s %s\n", str2->s1, *str2->s2 ? str2->s2 : "all LFNs");
	}
	globus_rls_client_free_list(list);
      }

      r = globus_rls_client_rli_rli_get_part(h, rli, pattern, &list);
      if (r == GLOBUS_SUCCESS) {
	printf("RLI updates:\n");
	for (p = list; p; p = globus_list_rest(p)) {
	  str2 = (globus_rls_string2_t *) globus_list_first(p);
	  printf("  %-32s %s\n", str2->s1, *str2->s2 ? str2->s2 : "all LFNs");
	}
	globus_rls_client_free_list(list);
      }
      break;

    case PING:
      now = time(0);
      r = globus_rls_client_admin(h, globus_rls_admin_cmd_ping);
      if (r == GLOBUS_SUCCESS)
	printf("ping %s: %d seconds\n", lrc, (int) (time(0) - now));
      break;

    case QUIT:
      r = globus_rls_client_admin(h, globus_rls_admin_cmd_quit);
      break;

    case SETCONF:
      r = globus_rls_client_set_configuration(h, option, value);
      break;

    case SSU:
      r = globus_rls_client_admin(h, globus_rls_admin_cmd_ssu);
      break;

    case STATS:
      r = globus_rls_client_stats(h, &rlsstats);
      if (r == GLOBUS_SUCCESS) {
	printf("Version:    %s\n", rlsstats.version);
	printf("Uptime:     %02d:%02d:%02d\n", (int) rlsstats.uptime / 3600,
	       (int) (rlsstats.uptime / 60) % 60, (int) rlsstats.uptime % 60);
	if (rlsstats.flags & RLS_LRCSERVER) {
	  printf("LRC stats\n");
	  printf("  initialized: %s\n",
          (rlsstats.flags & RLS_INITIALIZED) ? "yes" : "no");
	  if (rlsstats.flags & RLS_SNDLFNLIST)
	    printf("  update method: lfnlist\n");
	  if (rlsstats.flags & RLS_SNDBLOOMFILTER)
	    printf("  update method: bloomfilter\n");
	  if (globus_rls_client_lrc_rli_list(h, &list) == GLOBUS_RLS_SUCCESS) {
	    for (p = list; p; p = globus_list_rest(p)) {
	      rliinfo = (globus_rls_rli_info_t *) globus_list_first(p);
	      mycftime(buf, 100, "%D %T", rliinfo->lastupdate);
	      if (rliinfo->flags & FRLI_BLOOMFILTER)
		printf("  updates bloomfilter: %s last %s\n",rliinfo->url,buf);
	      else
		printf("  updates lfnlist:     %s last %s\n",rliinfo->url,buf);
	    }
	    globus_rls_client_free_list(list);
	  }
	  printf("  lfnlist update interval: %d\n", rlsstats.lrc_lfnlistui);
	  printf("  bloomfilter update interval: %d\n",
		 rlsstats.lrc_bloomfilterui);
	  printf("  numlfn: %d\n", rlsstats.lrc_numlfn);
	  printf("  numpfn: %d\n", rlsstats.lrc_numpfn);
	  printf("  nummap: %d\n", rlsstats.lrc_nummap);
	}

	if (rlsstats.flags & RLS_RLISERVER) {
	  printf("RLI stats\n");
	  if (globus_rls_client_rli_sender_list(h, &list) == GLOBUS_RLS_SUCCESS) {
	    for (p = list; p; p = globus_list_rest(p)) {
	      senderinfo = (globus_rls_sender_info_t *) globus_list_first(p);
	      mycftime(buf, 100, "%D %T", senderinfo->lastupdate);
	      printf("  updated by: %s last %s\n", senderinfo->url, buf);
	    }
	    globus_rls_client_free_list(list);
	  }
	  if (rlsstats.flags & RLS_RCVBLOOMFILTER)
	    printf("  updated via bloomfilters\n");
	  else
	    printf("  updated via lfnlists\n");
	  if (globus_rls_client_rli_rli_list(h, &list) == GLOBUS_RLS_SUCCESS) {
	    for (p = list; p; p = globus_list_rest(p)) {
	      rliinfo = (globus_rls_rli_info_t *) globus_list_first(p);
	      mycftime(buf, 100, "%D %T", rliinfo->lastupdate);
	      if (rliinfo->flags & FRLI_BLOOMFILTER)
		printf("  updates bloomfilter: %s last %s\n",rliinfo->url,buf);
	      else
		printf("  updates lfnlist:     %s last %s\n",rliinfo->url,buf);
	    }
	    globus_rls_client_free_list(list);
	  }
	  if (rlsstats.flags & RLS_RCVLFNLIST) {
	    printf("  numlfn: %d\n", rlsstats.rli_numlfn);
	    printf("  numlrc: %d\n", rlsstats.rli_numlrc);
	    printf("  numsender: %d\n", rlsstats.rli_numsender);
	    printf("  nummap: %d\n", rlsstats.rli_nummap);
	  }
	}
      }
      break;
  }
  if (r != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "%s\n", errmsg);
  }
  if ((r = globus_rls_client_close(h)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "close: %s\n", errmsg);
  }
  for (i = NMODS - 1; i >= 0; i--)
    globus_module_deactivate(modules[i]);
  exit(0);
}

static void
usage(char *prog)

{
  fprintf(stderr,"Usage: %s -A|-a|-C option value|-c option|-D|-d|-e|-p|-r|-S|-s|-t timeout|-u|-v [rli [pattern]] rls-server\n", prog);
  exit(1);
}
