/*
 * Copyright (c) 2016, Alliance for Open Media. All rights reserved
 *
 * This source code is subject to the terms of the BSD 2 Clause License and
 * the Alliance for Open Media Patent License 1.0. If the BSD 2 Clause License
 * was not distributed with this source code in the LICENSE file, you can
 * obtain it at www.aomedia.org/license/software. If the Alliance for Open
 * Media Patent License 1.0 was not distributed with this source code in the
 * PATENTS file, you can obtain it at www.aomedia.org/license/patent.
 */

#ifndef AV1_ENCODER_TX_PRUNE_MODEL_WEIGHTS_H_
#define AV1_ENCODER_TX_PRUNE_MODEL_WEIGHTS_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "av1/encoder/ml.h"

// Tx type model for 4x4 block.
static const float av1_tx_type_nn_weights_4x4_layer0[32] = {
  0.72406f,  -0.40019f, 0.51795f,  -0.43881f, -0.49746f, -0.41780f, -0.39409f,
  -0.16183f, -1.00135f, -0.41733f, -0.96534f, 0.93272f,  1.06229f,  0.04188f,
  0.60919f,  0.92405f,  -0.39359f, 0.70570f,  0.75375f,  1.11966f,  -1.86360f,
  -0.35421f, 0.18743f,  0.13346f,  -0.21262f, 0.07050f,  0.10533f,  -0.47402f,
  1.33417f,  1.72899f,  1.17983f,  0.10552f,
};

static const float av1_tx_type_nn_bias_4x4_layer0[8] = {
  1.96273f, -0.69845f, -0.10999f, -1.11311f,
  1.35101f, 0.43842f,  -0.29264f, -1.15376f,
};

static const float av1_tx_type_nn_weights_4x4_layer1[32] = {
  0.79770f,  0.08520f,  0.23298f,  0.05285f,  0.87506f,  -0.90784f, -0.06197f,
  -1.00580f, 0.68639f,  -0.34881f, 0.15366f,  -1.64658f, 0.80755f,  -0.26293f,
  0.10253f,  -0.23915f, 1.14696f,  -0.10928f, -1.61377f, 0.00863f,  0.98599f,
  -0.43872f, 0.61196f,  -0.03787f, 1.01060f,  0.17643f,  -0.00208f, -0.15738f,
  0.06517f,  0.72885f,  0.24387f,  1.28535f,
};

static const float av1_tx_type_nn_bias_4x4_layer1[4] = {
  1.23769f,
  1.40308f,
  0.09871f,
  1.82070f,
};

static const NN_CONFIG av1_tx_type_nnconfig_4x4 = {
  4,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_4x4_layer0,
      av1_tx_type_nn_weights_4x4_layer1,
  },
  {
      av1_tx_type_nn_bias_4x4_layer0,
      av1_tx_type_nn_bias_4x4_layer1,
  },
};
/******************************************************************************/

// Tx type model for 4x8 block.
static const float av1_tx_type_nn_weights_4x8_hor_layer0[32] = {
  0.68355f,  -0.06887f, 0.68525f,  -0.86048f, -0.35906f, -0.28597f, -0.21108f,
  0.12591f,  -1.13025f, -0.65695f, -0.25658f, 0.39155f,  0.89011f,  0.19258f,
  0.28316f,  0.61172f,  0.52587f,  0.99182f,  0.75704f,  0.66788f,  -1.61814f,
  -1.23483f, -0.62868f, -0.11902f, 0.33295f,  0.64796f,  0.92345f,  -0.71821f,
  0.07575f,  0.34687f,  0.20518f,  -0.87850f,
};

static const float av1_tx_type_nn_bias_4x8_hor_layer0[8] = {
  1.14049f, -0.18583f, 1.92114f, -0.72057f,
  1.32715f, 0.96713f,  1.09877f, -0.64345f,
};

static const float av1_tx_type_nn_weights_4x8_hor_layer1[32] = {
  0.71978f,  0.06896f,  1.48617f,  0.97124f,  -0.02487f, -0.95359f, 0.68983f,
  -0.16313f, 0.51324f,  -0.33770f, 0.45938f,  -1.08238f, 0.72938f,  0.42300f,
  0.85691f,  -0.03783f, 1.12617f,  -0.04034f, 0.36923f,  0.25638f,  1.10167f,
  0.41633f,  0.72602f,  -0.14797f, 0.66888f,  0.11437f,  -0.99797f, -0.20725f,
  1.01163f,  2.06308f,  1.23331f,  -0.15481f,
};

static const float av1_tx_type_nn_bias_4x8_hor_layer1[4] = {
  2.14443f,
  1.98356f,
  0.74616f,
  2.58795f,
};

static const NN_CONFIG av1_tx_type_nnconfig_4x8_hor = {
  4,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_4x8_hor_layer0,
      av1_tx_type_nn_weights_4x8_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_4x8_hor_layer0,
      av1_tx_type_nn_bias_4x8_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_4x8_ver_layer0[128] = {
  0.88859f,  1.02796f,  1.15509f,  0.61719f,  0.85804f,  1.17581f,  0.93524f,
  0.06546f,  0.08018f,  -0.78562f, -0.36614f, 0.14149f,  -0.30069f, -0.52647f,
  -0.82789f, 0.60527f,  -1.74026f, -0.20271f, 0.09875f,  0.03708f,  0.09430f,
  -0.24043f, -0.38433f, 1.21014f,  1.42443f,  0.69586f,  1.07812f,  1.21748f,
  1.10989f,  0.93122f,  1.04127f,  0.39424f,  0.95592f,  0.12904f,  0.46330f,
  0.49722f,  0.46303f,  0.36979f,  0.60227f,  0.39345f,  -2.01632f, -0.05706f,
  0.07766f,  -0.01271f, -0.16577f, -0.21957f, -0.14800f, 0.24898f,  0.27662f,
  0.42028f,  0.44748f,  1.14585f,  1.38805f,  0.46182f,  -0.22982f, -0.07324f,
  0.29886f,  -0.46959f, -0.04228f, -0.01064f, 0.24260f,  -0.32282f, -0.23804f,
  1.44466f,  -0.42190f, -0.36385f, 0.39746f,  0.38557f,  -0.09624f, -0.21540f,
  0.57385f,  -0.72878f, -0.39677f, -0.00717f, 0.60499f,  1.33849f,  1.05337f,
  1.11947f,  0.38487f,  0.86534f,  -0.33970f, 0.71140f,  0.20772f,  0.61132f,
  0.06181f,  -0.20027f, 0.13736f,  -0.72321f, 0.64586f,  -0.56740f, -0.90912f,
  -0.20452f, 0.15381f,  -0.84346f, 0.19550f,  0.63164f,  1.35441f,  0.63218f,
  0.82883f,  0.38803f,  -0.23874f, -0.02962f, 0.23846f,  -0.06822f, -0.40159f,
  -0.17850f, -0.69524f, 1.12299f,  -0.08286f, -0.14150f, -0.28456f, -0.41519f,
  -0.12792f, -0.55286f, 0.51655f,  0.06636f,  0.73759f,  0.70072f,  0.12616f,
  0.31282f,  0.17130f,  -1.34233f, 0.37221f,  0.95838f,  0.16286f,  1.04301f,
  0.73600f,  -0.11233f,
};

static const float av1_tx_type_nn_bias_4x8_ver_layer0[16] = {
  -0.89131f, 0.09124f,  -0.71678f, -1.19929f, 0.98963f,  0.16896f,
  -0.44943f, -0.97532f, -0.13997f, 1.07136f,  -0.46362f, -0.45253f,
  -0.63015f, -0.20008f, 1.24048f,  -0.21265f,
};

static const float av1_tx_type_nn_weights_4x8_ver_layer1[64] = {
  -0.79795f, 0.45973f,  -0.54188f, -1.05095f, 0.64404f,  -0.56470f, -0.57018f,
  0.61644f,  0.50229f,  1.14006f,  0.13805f,  -0.42058f, -0.07468f, 0.66203f,
  0.93180f,  -0.59662f, -0.25152f, 0.00336f,  1.09769f,  -1.11921f, 0.15151f,
  0.58750f,  -0.42480f, -0.95908f, -0.10980f, 1.31715f,  0.06665f,  -0.52371f,
  0.37228f,  -0.12364f, 0.54876f,  -0.32698f, 0.39863f,  -0.97669f, -1.06351f,
  1.82755f,  1.02851f,  0.10322f,  -0.08322f, 0.08891f,  -0.05715f, 0.93503f,
  0.02096f,  -0.39506f, -0.99330f, -0.09407f, 0.75108f,  -0.30104f, 1.78314f,
  -0.01786f, -0.17392f, 0.00461f,  0.41394f,  0.92566f,  1.11251f,  -0.71380f,
  -0.04907f, 0.12736f,  0.00208f,  0.94451f,  -0.31783f, -0.19655f, 0.64619f,
  0.50359f,
};

static const float av1_tx_type_nn_bias_4x8_ver_layer1[4] = {
  0.39274f,
  1.27276f,
  0.30322f,
  2.55238f,
};

static const NN_CONFIG av1_tx_type_nnconfig_4x8_ver = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_4x8_ver_layer0,
      av1_tx_type_nn_weights_4x8_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_4x8_ver_layer0,
      av1_tx_type_nn_bias_4x8_ver_layer1,
  },
};
/******************************************************************************/

// Tx type model for 8x4 block.
static const float av1_tx_type_nn_weights_8x4_hor_layer0[128] = {
  0.64828f,  0.61618f,  0.98975f,  -0.14562f, 0.26957f,  1.80872f,  0.58299f,
  -0.06917f, 0.00937f,  -0.74073f, -0.66045f, -0.04576f, -0.39802f, -0.76960f,
  -0.85166f, 0.88799f,  -0.70694f, -0.34366f, -0.54906f, -0.39502f, -0.29465f,
  -0.49650f, -0.32171f, 1.37181f,  1.30432f,  0.71843f,  1.01916f,  1.01582f,
  0.90999f,  0.86334f,  1.04603f,  0.40734f,  0.96187f,  0.53742f,  0.07510f,
  0.44167f,  0.02049f,  -0.02874f, 0.97191f,  1.03647f,  -2.62751f, -0.01390f,
  -0.09282f, -0.02522f, -0.30849f, -0.19386f, -0.51356f, 0.52778f,  0.77191f,
  0.75416f,  0.69067f,  0.93561f,  1.35982f,  0.76193f,  0.57869f,  0.00251f,
  -0.87244f, -0.26922f, -0.06682f, 0.07176f,  0.51142f,  0.58948f,  0.13914f,
  0.71165f,  -0.40329f, -0.33201f, 0.35293f,  0.33437f,  -0.01812f, -0.24765f,
  0.26810f,  -0.77088f, 1.35707f,  0.22243f,  0.78402f,  0.66191f,  0.79890f,
  1.90669f,  0.73189f,  0.24222f,  -0.34682f, 0.66990f,  0.19554f,  0.58414f,
  0.05060f,  -0.21271f, 0.11656f,  -0.74907f, 0.68837f,  -0.39147f, -1.78263f,
  -0.69918f, -0.06838f, -0.26927f, 0.38502f,  0.08305f,  1.29848f,  0.67328f,
  0.67269f,  0.65805f,  -0.47778f, -1.02617f, 0.16523f,  0.12223f,  -0.35294f,
  -0.15866f, -0.56224f, 1.25895f,  -0.21422f, -0.33518f, -0.33519f, -0.37414f,
  0.55122f,  0.14806f,  0.44312f,  -0.07865f, 0.75295f,  0.10766f,  0.59922f,
  0.48837f,  -0.19099f, -2.07991f, 0.35755f,  0.87813f,  0.07559f,  1.00724f,
  0.25223f,  -0.06761f,
};

static const float av1_tx_type_nn_bias_8x4_hor_layer0[16] = {
  -0.54227f, 0.08599f,  -0.77447f, -1.10920f, 0.89298f,  0.05454f,
  -0.73681f, 0.21048f,  -0.41041f, 1.25690f,  -0.60918f, 0.14661f,
  -0.65392f, -0.25881f, 1.67995f,  -0.03550f,
};

static const float av1_tx_type_nn_weights_8x4_hor_layer1[64] = {
  -0.22312f, 0.73552f,  0.48399f,  -0.66996f, 0.36527f,  -0.42228f, -1.10793f,
  0.31167f,  0.16177f,  1.69315f,  -0.06287f, -0.35804f, -0.24889f, 0.80824f,
  1.08952f,  -0.62838f, 0.30066f,  -0.19043f, -0.00518f, -1.31005f, 0.65797f,
  1.07714f,  -0.24253f, 0.49779f,  0.05848f,  1.08914f,  0.08015f,  -0.38853f,
  0.35108f,  -0.11026f, 0.64528f,  -0.37615f, 0.39995f,  -0.58117f, -1.29627f,
  1.74169f,  0.75558f,  -0.04910f, 0.35020f,  0.04556f,  0.12634f,  1.27223f,
  0.02608f,  -0.19687f, -0.78649f, -0.22746f, 1.02589f,  -0.28411f, 1.42443f,
  -0.42115f, -0.21153f, -0.01733f, 0.62001f,  0.87167f,  1.66008f,  -0.39179f,
  -0.06293f, 0.27012f,  0.16871f,  0.64597f,  0.67358f,  -0.20053f, 0.95830f,
  0.44232f,
};

static const float av1_tx_type_nn_bias_8x4_hor_layer1[4] = {
  0.14889f,
  1.74197f,
  0.53696f,
  2.87574f,
};

static const NN_CONFIG av1_tx_type_nnconfig_8x4_hor = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_8x4_hor_layer0,
      av1_tx_type_nn_weights_8x4_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_8x4_hor_layer0,
      av1_tx_type_nn_bias_8x4_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_8x4_ver_layer0[32] = {
  0.81919f,  0.15527f,  0.60055f,  -0.54617f, -0.35510f, -0.28223f, -0.20478f,
  0.15001f,  -1.84806f, -0.30274f, -0.00865f, 0.33939f,  1.11970f,  0.44630f,
  0.32074f,  0.39637f,  0.08149f,  1.28070f,  0.86703f,  0.76503f,  -1.83991f,
  -1.13575f, -0.68605f, -0.23690f, 0.07099f,  0.64960f,  0.82543f,  -0.72028f,
  0.08220f,  0.34338f,  0.20245f,  -0.88920f,
};

static const float av1_tx_type_nn_bias_8x4_ver_layer0[8] = {
  1.14995f, -0.16021f, 2.38325f, -0.65179f,
  1.09624f, 1.07662f,  0.63837f, -0.64847f,
};

static const float av1_tx_type_nn_weights_8x4_ver_layer1[32] = {
  0.10278f,  0.06819f,  1.73885f,  1.29889f,  -0.18482f, -1.06132f, 0.67003f,
  -0.23280f, 0.50181f,  -0.33890f, 0.43524f,  -1.03147f, 1.09640f,  0.66332f,
  0.47652f,  -0.02251f, 0.94245f,  -0.03861f, 0.84776f,  0.28377f,  0.92044f,
  0.23572f,  0.52082f,  -0.16266f, 0.45290f,  0.11342f,  -0.50310f, -0.92633f,
  1.46345f,  1.84714f,  1.06804f,  -0.13610f,
};

static const float av1_tx_type_nn_bias_8x4_ver_layer1[4] = {
  2.41028f,
  1.95675f,
  0.82387f,
  2.41923f,
};

static const NN_CONFIG av1_tx_type_nnconfig_8x4_ver = {
  4,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      8,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_8x4_ver_layer0,
      av1_tx_type_nn_weights_8x4_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_8x4_ver_layer0,
      av1_tx_type_nn_bias_8x4_ver_layer1,
  },
};
/******************************************************************************/

// Tx type model for 8x8 block.
static const float av1_tx_type_nn_weights_8x8_layer0[128] = {
  0.98214f,  1.05643f,  0.91173f,  0.24165f,  0.39961f,  0.25736f,  0.68593f,
  0.10553f,  0.13353f,  -0.49687f, -1.66413f, 1.16584f,  2.25147f,  -0.72247f,
  -2.65486f, -0.03628f, -1.47746f, -1.07644f, -1.25551f, -0.91260f, -1.26199f,
  -1.06022f, -1.42138f, 1.10500f,  2.96552f,  -0.40638f, 0.02258f,  -0.23137f,
  0.34922f,  -0.01454f, 0.41251f,  0.35944f,  -1.56742f, 0.01406f,  0.88114f,
  1.42462f,  0.87243f,  0.02439f,  0.07035f,  0.34303f,  -3.16843f, 0.25798f,
  0.07494f,  0.38926f,  -0.12267f, 0.09049f,  -0.36711f, 0.01551f,  1.41269f,
  1.33505f,  1.43627f,  1.41909f,  1.44605f,  1.43008f,  1.36721f,  0.19443f,
  -0.08606f, 0.17285f,  0.63692f,  0.92092f,  0.61007f,  0.87100f,  -0.33631f,
  1.98025f,  -0.40686f, -0.33808f, 0.34919f,  0.33817f,  -0.01807f, -0.25259f,
  0.26442f,  -0.76979f, 1.07788f,  -1.38747f, 1.34315f,  2.79947f,  2.02838f,
  -0.25062f, 0.00174f,  1.25888f,  0.17344f,  0.20897f,  1.28765f,  1.95749f,
  1.62351f,  1.04556f,  0.43858f,  0.12463f,  1.66399f,  0.03971f,  0.36614f,
  0.56932f,  0.15982f,  0.11587f,  0.21402f,  1.89386f,  -0.91267f, -0.79781f,
  1.79155f,  0.60147f,  -0.90118f, -4.32718f, -0.58154f, -0.02181f, -0.40734f,
  -0.11409f, -0.79470f, 0.69697f,  -0.16588f, -0.16090f, -0.21236f, -0.52776f,
  -0.64455f, 0.09173f,  0.80766f,  0.76097f,  0.20295f,  -0.93467f, -0.43509f,
  0.59659f,  0.07788f,  -3.79459f, 0.16268f,  0.47343f,  0.05106f,  -0.24880f,
  1.18941f,  0.10346f,
};

static const float av1_tx_type_nn_bias_8x8_layer0[16] = {
  0.75780f,  0.25628f,  0.19911f,  -0.41384f, 1.33909f,  0.31498f,
  -1.37171f, -1.09561f, -0.44056f, 0.49001f,  -0.65804f, -1.96031f,
  0.64806f,  -0.52520f, 1.38838f,  0.15519f,
};

static const float av1_tx_type_nn_weights_8x8_layer1[64] = {
  -0.63856f, -2.02670f, -0.92947f, 0.00216f,  1.47710f,  -2.01099f, -2.11289f,
  -0.92288f, 0.19296f,  1.37866f,  -0.85975f, -0.78624f, -2.10392f, 0.13976f,
  1.06968f,  -2.04120f, 0.57991f,  -1.84941f, -0.81512f, -2.08254f, -0.47334f,
  0.12256f,  -1.39594f, -1.02829f, 0.06134f,  2.25646f,  -1.25196f, -2.65317f,
  -1.94473f, 0.10989f,  0.55446f,  -1.76557f, 0.33455f,  -1.85556f, -3.01878f,
  -0.25100f, 1.65520f,  -1.61409f, 1.16336f,  -1.15560f, 0.13631f,  1.50733f,
  -1.07538f, -0.91200f, -1.93132f, 0.09271f,  0.24425f,  -1.80655f, -0.01138f,
  -1.36421f, -0.62970f, -0.84766f, -0.34714f, -0.50531f, 1.91005f,  -1.60316f,
  -0.02495f, 1.04938f,  0.28411f,  -0.79809f, -1.48232f, 0.00766f,  0.94016f,
  -1.11974f,
};

static const float av1_tx_type_nn_bias_8x8_layer1[4] = {
  0.53574f,
  1.57736f,
  -0.13698f,
  2.64613f,
};

static const NN_CONFIG av1_tx_type_nnconfig_8x8 = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_8x8_layer0,
      av1_tx_type_nn_weights_8x8_layer1,
  },
  {
      av1_tx_type_nn_bias_8x8_layer0,
      av1_tx_type_nn_bias_8x8_layer1,
  },
};
/******************************************************************************/

// Tx type model for 8x16 block.
static const float av1_tx_type_nn_weights_8x16_hor_layer0[128] = {
  1.36274f,  1.37313f,  1.26859f,  1.26459f,  1.37979f,  1.47217f,  1.29710f,
  0.15765f,  0.31552f,  -0.05727f, 0.25562f,  0.47925f,  -0.32913f, -0.55757f,
  -0.98010f, 0.08568f,  -0.62754f, 0.12834f,  -0.03717f, 0.06286f,  0.26159f,
  0.26023f,  -0.62605f, 1.34500f,  1.47720f,  0.47937f,  0.84793f,  0.87866f,
  0.81260f,  0.74761f,  0.84217f,  0.53321f,  -0.78232f, 0.35321f,  0.41240f,
  0.45002f,  0.88973f,  0.51055f,  0.91115f,  -0.45512f, -2.37418f, -0.25205f,
  0.05893f,  -0.15685f, -0.25156f, -0.17104f, -0.12230f, 0.17802f,  0.18796f,
  -0.05797f, 0.26484f,  1.23515f,  1.70393f,  0.46022f,  -0.14354f, 0.08501f,
  -0.84625f, -0.42578f, -0.29345f, -0.51797f, -0.56515f, -0.47465f, 0.23970f,
  1.59912f,  -0.40332f, -0.33209f, 0.37274f,  0.36831f,  -0.00248f, -0.24295f,
  0.29539f,  -0.76136f, -0.22531f, 0.12371f,  0.37889f,  1.02639f,  1.73330f,
  1.09686f,  1.04111f,  0.69006f,  -1.27157f, 0.94013f,  0.61621f,  0.62274f,
  0.48759f,  0.55672f,  0.62597f,  -0.38846f, 1.72124f,  0.08214f,  -0.06650f,
  0.32617f,  0.10958f,  0.24650f,  0.10740f,  1.16861f,  0.50701f,  0.45383f,
  0.90016f,  -0.00695f, -0.11986f, -0.07834f, 0.20346f,  0.25863f,  -0.40889f,
  -0.11344f, -0.79108f, 0.76259f,  -0.14562f, -0.15459f, -0.20954f, -0.51306f,
  0.02743f,  -0.82456f, -0.00861f, -0.27274f, 0.28762f,  0.07282f,  0.26410f,
  0.53413f,  -0.22208f, -0.85031f, -1.39129f, -0.74519f, 0.09771f,  0.80313f,
  1.07698f,  0.02531f,
};

static const float av1_tx_type_nn_bias_8x16_hor_layer0[16] = {
  -1.30434f, -1.19259f, -0.43467f, -0.85386f, 0.96584f,  0.29276f,
  -0.41990f, -0.96924f, -0.30933f, 0.95264f,  -0.25330f, -1.19584f,
  1.46564f,  -0.42959f, 1.55720f,  0.18479f,
};

static const float av1_tx_type_nn_weights_8x16_hor_layer1[64] = {
  -1.72959f, -0.21670f, 0.10616f,  -0.02006f, 0.15084f,  -0.85303f, -0.27535f,
  0.58704f,  0.23683f,  1.19743f,  0.77971f,  0.49874f,  0.19508f,  0.19641f,
  1.47895f,  -0.52173f, -0.56746f, -0.50761f, 0.15864f,  -0.95168f, 0.48103f,
  0.91904f,  -0.11700f, 0.62863f,  0.06526f,  1.63803f,  -0.72325f, -1.80449f,
  0.66373f,  0.12831f,  0.27139f,  -0.26346f, 1.50852f,  0.25079f,  -0.54255f,
  1.78815f,  1.39691f,  -0.44989f, -0.18511f, -1.52903f, 0.13983f,  1.06906f,
  -0.30184f, 0.37566f,  0.46209f,  0.10440f,  0.64695f,  -0.34002f, 1.96990f,
  0.21189f,  -0.91248f, -0.11263f, 0.26708f,  1.27405f,  1.89776f,  0.02081f,
  -0.06977f, -0.02584f, 0.47733f,  0.27117f,  1.33315f,  -0.09175f, 0.48747f,
  1.16772f,
};

static const float av1_tx_type_nn_bias_8x16_hor_layer1[4] = {
  1.25783f,
  1.19452f,
  0.69964f,
  2.41982f,
};

static const NN_CONFIG av1_tx_type_nnconfig_8x16_hor = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_8x16_hor_layer0,
      av1_tx_type_nn_weights_8x16_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_8x16_hor_layer0,
      av1_tx_type_nn_bias_8x16_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_8x16_ver_layer0[128] = {
  0.90888f,  0.86305f,  0.81674f,  0.75352f,  1.07834f,  0.99048f,  0.96355f,
  0.13836f,  -0.51334f, 0.19906f,  1.84608f,  0.67828f,  0.45876f,  0.08325f,
  0.28190f,  -0.01958f, -1.96553f, 0.27837f,  -0.05929f, 0.13491f,  0.21036f,
  0.05797f,  -0.01373f, 0.73765f,  1.39603f,  -0.53767f, 0.10362f,  0.03420f,
  0.41909f,  0.09510f,  0.32284f,  0.83860f,  0.13954f,  0.48434f,  1.47762f,
  0.45891f,  0.23613f,  0.13013f,  0.82097f,  -0.03251f, -1.89757f, 0.21589f,
  -0.10370f, 0.02530f,  -0.25659f, 0.01466f,  -0.23661f, 0.22783f,  0.92100f,
  1.02915f,  1.20358f,  1.17251f,  0.97749f,  1.04696f,  0.91333f,  0.54576f,
  -0.52792f, 0.02217f,  0.25652f,  0.31405f,  -0.18398f, 0.04572f,  -0.81359f,
  1.82883f,  -0.40047f, -0.33056f, 0.35255f,  0.34448f,  -0.00339f, -0.23857f,
  0.28925f,  -0.77175f, -0.24325f, -0.21420f, 1.11451f,  1.39553f,  0.51573f,
  0.05476f,  1.13791f,  0.94959f,  -0.35710f, 0.67467f,  0.16722f,  0.61213f,
  0.07683f,  -0.20613f, 0.13440f,  -0.72131f, -0.15418f, -0.17688f, -0.16510f,
  -0.19226f, 0.09270f,  -2.43559f, -0.12669f, 0.05074f,  0.30414f,  0.00927f,
  0.60630f,  0.00801f,  -1.07310f, -0.06227f, 2.10607f,  0.02382f,  -0.39891f,
  -0.09149f, -0.78596f, 0.83966f,  -0.14802f, -0.14083f, -0.20831f, -0.55136f,
  0.08566f,  -0.00647f, 0.07044f,  0.53408f,  0.85720f,  -0.07393f, 0.24476f,
  0.43767f,  0.30519f,  -1.89430f, 0.23252f,  1.63790f,  0.17316f,  -0.03903f,
  0.25269f,  0.01562f,
};

static const float av1_tx_type_nn_bias_8x16_ver_layer0[16] = {
  -0.83370f, -0.20704f, -0.60437f, -0.81664f, 1.16998f,  0.16745f,
  -1.34680f, -1.07083f, -0.34649f, 0.65598f,  -0.56278f, 0.22660f,
  -0.25956f, -0.29608f, 1.24359f,  -0.09167f,
};

static const float av1_tx_type_nn_weights_8x16_ver_layer1[64] = {
  -0.71147f, -0.63964f, -0.69220f, 0.22326f,  0.67191f,  -0.58894f, -0.98464f,
  0.23583f,  0.22824f,  1.39838f,  0.09920f,  -0.59411f, -0.67101f, 0.19088f,
  0.83025f,  -0.66991f, -0.42889f, -0.49969f, 1.39532f,  -1.02000f, 0.62101f,
  0.57175f,  -0.83226f, 0.01551f,  0.05604f,  1.23028f,  0.02030f,  -0.55995f,
  -0.42349f, 0.15375f,  0.52132f,  -0.52421f, 0.89586f,  -0.73778f, -0.10911f,
  0.22447f,  1.16858f,  -0.48169f, 1.73890f,  -0.69860f, 0.12504f,  1.10492f,
  0.04391f,  -0.85670f, -0.49257f, 0.09616f,  0.76518f,  -0.44854f, 1.50938f,
  0.62246f,  -0.40366f, -0.11182f, -0.01680f, 0.59724f,  1.32170f,  -1.09061f,
  -0.04278f, -0.02449f, 0.25024f,  1.26239f,  0.42345f,  -0.10031f, 0.80871f,
  0.44198f,
};

static const float av1_tx_type_nn_bias_8x16_ver_layer1[4] = {
  0.68329f,
  1.33555f,
  0.25943f,
  3.23439f,
};

static const NN_CONFIG av1_tx_type_nnconfig_8x16_ver = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_8x16_ver_layer0,
      av1_tx_type_nn_weights_8x16_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_8x16_ver_layer0,
      av1_tx_type_nn_bias_8x16_ver_layer1,
  },
};
/******************************************************************************/

// Tx type model for 16x8 block.
static const float av1_tx_type_nn_weights_16x8_hor_layer0[128] = {
  0.89821f,  0.90804f,  1.13052f,  0.74855f,  1.02053f,  0.91260f,  0.97102f,
  0.16808f,  -0.19982f, -0.33296f, -0.22490f, -0.22481f, -0.09332f, -2.44338f,
  -0.12236f, -0.03158f, -1.43561f, 0.07794f,  0.16586f,  0.09731f,  0.12967f,
  0.09725f,  -0.16826f, 1.26640f,  0.88004f,  0.27312f,  -0.07993f, 0.33640f,
  0.11732f,  0.33384f,  0.97066f,  -0.61744f, -0.48545f, 0.44622f,  0.73744f,
  0.32262f,  -0.05713f, 0.42280f,  1.10378f,  0.18540f,  -2.07906f, 0.11443f,
  0.37877f,  0.24136f,  -0.12524f, -0.12434f, 0.02116f,  0.11716f,  1.28267f,
  1.01508f,  1.26184f,  1.22545f,  1.29582f,  1.18855f,  1.27564f,  0.42001f,
  -0.41481f, 0.06725f,  -0.13133f, -0.24801f, 0.16515f,  0.16228f,  0.35197f,
  0.53610f,  -0.39805f, -0.32584f, 0.40096f,  0.38621f,  -0.00030f, -0.23434f,
  0.29149f,  -0.76542f, 0.04996f,  -0.30036f, 1.48687f,  0.90852f,  -0.03083f,
  -0.15953f, 1.19259f,  0.87690f,  -1.08977f, 0.78757f,  0.81149f,  0.54089f,
  0.35400f,  0.37919f,  0.84997f,  -0.20449f, 0.39601f,  -0.37596f, 0.64748f,
  0.26021f,  0.37354f,  0.23593f,  0.16335f,  1.70681f,  0.31800f,  -0.00964f,
  0.82687f,  -0.78372f, -1.47438f, 0.32410f,  1.37436f,  0.07476f,  -0.40574f,
  -0.10353f, -0.79300f, 0.74381f,  -0.15601f, -0.14380f, -0.20961f, -0.52697f,
  0.04669f,  -0.00870f, 0.05624f,  -0.09036f, 0.25701f,  0.30336f,  0.24199f,
  0.45579f,  0.66330f,  -1.81834f, 0.74965f,  1.22747f,  0.25072f,  0.25100f,
  0.43289f,  -0.00362f,
};

static const float av1_tx_type_nn_bias_16x8_hor_layer0[16] = {
  -0.87643f, 0.36754f,  -0.86409f, 1.37761f,  1.22688f,  0.09074f,
  -1.47139f, -1.06100f, -0.24087f, 1.10382f,  -0.32837f, -1.39592f,
  -0.14741f, -0.43954f, 1.72137f,  -0.21704f,
};

static const float av1_tx_type_nn_weights_16x8_hor_layer1[64] = {
  -0.81860f, -0.80745f, -0.43612f, 0.58656f,  0.37455f, -0.56519f, -1.71536f,
  0.23278f,  0.23951f,  1.09610f,  0.49986f,  0.43375f, -0.53182f, 0.17376f,
  1.05626f,  -0.61743f, -0.71777f, -0.66943f, 1.40091f, 0.34426f,  1.14295f,
  0.45571f,  -0.52504f, -0.00303f, 0.06044f,  0.66119f, -0.60340f, -1.14344f,
  -0.28045f, 0.12742f,  0.61484f,  -0.41016f, 1.36102f, -0.86969f, -0.52728f,
  1.01725f,  0.67083f,  -0.10138f, 1.36406f,  0.34066f, 0.12498f,  0.86595f,
  -0.39636f, -0.27888f, -0.40244f, 0.09847f,  0.81178f, -0.45313f, 1.39127f,
  0.99865f,  -0.57908f, 0.55072f,  0.49638f,  1.11524f, 1.85504f,  -0.28316f,
  -0.05195f, -0.23284f, 0.26461f,  -1.28120f, 0.60707f, -0.06110f, 0.74085f,
  0.63304f,
};

static const float av1_tx_type_nn_bias_16x8_hor_layer1[4] = {
  0.71765f,
  1.40400f,
  0.32221f,
  3.07234f,
};

static const NN_CONFIG av1_tx_type_nnconfig_16x8_hor = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_16x8_hor_layer0,
      av1_tx_type_nn_weights_16x8_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_16x8_hor_layer0,
      av1_tx_type_nn_bias_16x8_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_16x8_ver_layer0[128] = {
  1.20497f,  1.23691f,  1.23738f,  1.07773f,  1.15264f,  1.31959f,  1.15365f,
  0.17179f,  0.68612f,  0.55636f,  0.57145f,  0.67022f,  0.19636f,  -1.27420f,
  -1.36428f, -0.16706f, -1.20934f, -0.87794f, -0.97146f, -0.74722f, -1.14493f,
  -1.02689f, -0.88153f, 0.83857f,  1.53355f,  0.13601f,  0.35451f,  0.53750f,
  0.62381f,  0.32438f,  0.59405f,  0.33090f,  -1.52948f, -0.46094f, 0.42634f,
  0.48763f,  0.30707f,  0.52553f,  0.71427f,  -0.31287f, -2.37106f, -0.18756f,
  0.16561f,  -0.00431f, -0.13747f, -0.09336f, -0.16511f, 0.13454f,  0.45010f,
  -0.00317f, -0.06403f, 0.95442f,  1.59636f,  0.30602f,  -0.05515f, 0.05467f,
  -0.21758f, -0.19192f, -0.17935f, -0.00545f, 0.35409f,  0.26141f,  -0.32174f,
  1.78129f,  -0.40161f, -0.33158f, 0.38084f,  0.38081f,  0.01053f,  -0.23567f,
  0.29239f,  -0.76159f, -0.19373f, 0.13649f,  0.66949f,  1.19733f,  1.92557f,
  1.16691f,  0.94955f,  0.62324f,  -0.85434f, -0.07699f, 0.87683f,  0.95911f,
  0.86106f,  0.57959f,  0.40146f,  -0.35851f, 1.55427f,  0.15349f,  -0.01582f,
  0.32517f,  0.03784f,  0.15916f,  0.09024f,  1.43187f,  0.56160f,  0.11521f,
  0.52476f,  -0.26107f, -0.38167f, -0.31596f, 0.31304f,  -0.65366f, -0.40680f,
  -0.11082f, -0.78585f, 0.77906f,  -0.13322f, -0.13747f, -0.21001f, -0.53204f,
  -0.06752f, -0.84741f, -0.53442f, -0.16284f, 0.54027f,  0.13586f,  -0.42001f,
  0.85388f,  0.08300f,  -0.89325f, -1.73681f, -0.70473f, 0.23151f,  0.69549f,
  0.72124f,  0.12769f,
};

static const float av1_tx_type_nn_bias_16x8_ver_layer0[16] = {
  -1.15644f, -0.31062f, 0.20697f,  -0.60304f, -1.19498f, 0.21451f,
  -0.42825f, -0.71800f, -0.25816f, 1.47408f,  -0.24423f, -1.45773f,
  -0.55834f, -0.36938f, 1.56759f,  0.07238f,
};

static const float av1_tx_type_nn_weights_16x8_ver_layer1[64] = {
  -1.45227f, -0.67141f, 0.75237f,  0.32681f,  -0.70528f, -0.76730f, -0.49777f,
  0.02418f,  0.25096f,  1.14840f,  0.23548f,  0.48755f,  0.33164f,  0.21050f,
  1.41651f,  -0.28888f, -0.76668f, 0.04439f,  0.67538f,  -1.06438f, 0.68128f,
  0.95824f,  0.08530f,  -0.03635f, 0.06820f,  1.38621f,  -0.50424f, -1.72992f,
  -0.20949f, 0.13400f,  0.93366f,  -0.05324f, 1.41593f,  -0.75119f, -1.80912f,
  1.05440f,  0.62580f,  -0.30867f, -0.07025f, -0.34654f, 0.13621f,  1.74426f,
  -0.22417f, 0.47031f,  -0.08142f, 0.10151f,  0.42498f,  0.06635f,  1.50623f,
  1.04130f,  0.85107f,  0.23382f,  0.69800f,  1.10856f,  1.18767f,  -0.69395f,
  -0.07985f, 0.50412f,  0.46019f,  0.49214f,  0.44219f,  -0.09502f, 0.75745f,
  0.99208f,
};

static const float av1_tx_type_nn_bias_16x8_ver_layer1[4] = {
  0.68774f,
  0.88572f,
  0.77462f,
  3.05667f,
};

static const NN_CONFIG av1_tx_type_nnconfig_16x8_ver = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_16x8_ver_layer0,
      av1_tx_type_nn_weights_16x8_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_16x8_ver_layer0,
      av1_tx_type_nn_bias_16x8_ver_layer1,
  },
};
/******************************************************************************/

// Tx type model for 16x16 block.
static const float av1_tx_type_nn_weights_16x16_layer0[128] = {
  1.26592f,  1.36313f,  1.30956f,  1.29926f,  1.48816f,  1.68851f,  1.32000f,
  0.13321f,  -0.22477f, -0.88906f, -0.19622f, 1.69605f,  1.22180f,  -1.57771f,
  -1.15765f, 0.05710f,  -1.13355f, -0.85486f, -0.99971f, -0.91571f, -1.06031f,
  -0.77952f, -1.15723f, 1.17809f,  1.35602f,  -0.05243f, -0.37596f, 0.26108f,
  0.17611f,  -0.10323f, 0.77279f,  -0.48911f, -0.79308f, 0.55112f,  0.43918f,
  0.27872f,  0.28714f,  0.45830f,  1.05689f,  0.03705f,  -2.49975f, -0.01940f,
  0.05709f,  0.07942f,  -0.13290f, -0.10359f, 0.00143f,  0.37303f,  0.96470f,
  0.53293f,  1.14459f,  0.89185f,  0.43378f,  0.47764f,  0.90924f,  0.15279f,
  -0.15361f, 0.02949f,  0.42240f,  0.68143f,  0.89588f,  0.73754f,  0.10974f,
  1.57755f,  -0.39870f, -0.32914f, 0.35638f,  0.34991f,  -0.00003f, -0.23373f,
  0.29630f,  -0.76699f, -0.01356f, 0.04234f,  0.84253f,  1.92078f,  0.93160f,
  0.71993f,  0.71604f,  0.76455f,  -1.59782f, 0.32332f,  1.11628f,  0.33062f,
  -0.03728f, -0.05710f, 0.80447f,  -0.14719f, 1.34658f,  -0.05718f, 0.64015f,
  0.21926f,  0.41653f,  0.12720f,  0.54092f,  1.39411f,  1.81819f,  -0.24513f,
  0.00955f,  0.38011f,  -0.57787f, -0.41759f, 0.68834f,  -0.31783f, -0.40607f,
  -0.10107f, -0.79374f, 0.75599f,  -0.16282f, -0.14490f, -0.20783f, -0.55019f,
  -0.13793f, -0.22293f, 0.18305f,  0.12445f,  0.56830f,  0.24567f,  0.09278f,
  0.70803f,  0.35803f,  -1.52676f, -0.89624f, 0.77665f,  0.19877f,  0.77175f,
  0.50355f,  0.08592f,
};

static const float av1_tx_type_nn_bias_16x16_layer0[16] = {
  -1.31834f, 0.14346f,  -0.10062f, 0.84489f,  0.95617f,  -0.06720f,
  -0.68502f, -0.91442f, -0.31932f, 0.25276f,  -0.15138f, -1.57661f,
  -0.14062f, -0.42120f, 0.94573f,  -0.09287f,
};

static const float av1_tx_type_nn_weights_16x16_layer1[64] = {
  -1.80333f, -1.06353f, 0.55139f,  0.74644f,  0.13747f, -0.93018f, -0.10286f,
  0.67133f,  0.24460f,  1.44583f,  0.02173f,  0.26037f, -0.73687f, 0.19566f,
  0.61846f,  -0.58601f, -1.03196f, -0.74415f, 0.30041f, -0.41967f, 1.08740f,
  0.96224f,  -0.59139f, 0.03813f,  0.05403f,  1.33427f, -0.54375f, -1.92181f,
  0.54704f,  0.13608f,  0.22151f,  -0.38076f, 1.18390f, -0.77508f, -1.84283f,
  1.00894f,  0.62318f,  -0.15296f, 1.27600f,  0.22822f, 0.12751f,  0.93910f,
  -0.28502f, 0.53912f,  -0.96889f, 0.10182f,  0.81508f, -0.43028f, 2.67386f,
  0.52204f,  0.49820f,  -0.41711f, 1.05038f,  1.12192f, 0.74349f,  -0.75417f,
  -0.03718f, -0.35769f, 0.89651f,  0.63236f,  0.54215f, -0.07894f, 0.48274f,
  1.08829f,
};

static const float av1_tx_type_nn_bias_16x16_layer1[4] = {
  0.81986f,
  1.26865f,
  0.11118f,
  2.48404f,
};

static const NN_CONFIG av1_tx_type_nnconfig_16x16 = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_16x16_layer0,
      av1_tx_type_nn_weights_16x16_layer1,
  },
  {
      av1_tx_type_nn_bias_16x16_layer0,
      av1_tx_type_nn_bias_16x16_layer1,
  },
};
/******************************************************************************/

// Tx type model for 16x32 block.
static const float av1_tx_type_nn_weights_16x32_hor_layer0[128] = {
  0.89821f,  0.90804f,  1.13052f,  0.74855f,  1.02053f,  0.91260f,  0.97102f,
  0.16808f,  -0.19982f, -0.33296f, -0.22490f, -0.22481f, -0.09332f, -2.44338f,
  -0.12236f, -0.03158f, -1.43561f, 0.07794f,  0.16586f,  0.09731f,  0.12967f,
  0.09725f,  -0.16826f, 1.26640f,  0.88004f,  0.27312f,  -0.07993f, 0.33640f,
  0.11732f,  0.33384f,  0.97066f,  -0.61744f, -0.48545f, 0.44622f,  0.73744f,
  0.32262f,  -0.05713f, 0.42280f,  1.10378f,  0.18540f,  -2.07906f, 0.11443f,
  0.37877f,  0.24136f,  -0.12524f, -0.12434f, 0.02116f,  0.11716f,  1.28267f,
  1.01508f,  1.26184f,  1.22545f,  1.29582f,  1.18855f,  1.27564f,  0.42001f,
  -0.41481f, 0.06725f,  -0.13133f, -0.24801f, 0.16515f,  0.16228f,  0.35197f,
  0.53610f,  -0.39805f, -0.32584f, 0.40096f,  0.38621f,  -0.00030f, -0.23434f,
  0.29149f,  -0.76542f, 0.04996f,  -0.30036f, 1.48687f,  0.90852f,  -0.03083f,
  -0.15953f, 1.19259f,  0.87690f,  -1.08977f, 0.78757f,  0.81149f,  0.54089f,
  0.35400f,  0.37919f,  0.84997f,  -0.20449f, 0.39601f,  -0.37596f, 0.64748f,
  0.26021f,  0.37354f,  0.23593f,  0.16335f,  1.70681f,  0.31800f,  -0.00964f,
  0.82687f,  -0.78372f, -1.47438f, 0.32410f,  1.37436f,  0.07476f,  -0.40574f,
  -0.10353f, -0.79300f, 0.74381f,  -0.15601f, -0.14380f, -0.20961f, -0.52697f,
  0.04669f,  -0.00870f, 0.05624f,  -0.09036f, 0.25701f,  0.30336f,  0.24199f,
  0.45579f,  0.66330f,  -1.81834f, 0.74965f,  1.22747f,  0.25072f,  0.25100f,
  0.43289f,  -0.00362f,
};

static const float av1_tx_type_nn_bias_16x32_hor_layer0[16] = {
  -0.87643f, 0.36754f,  -0.86409f, 1.37761f,  1.22688f,  0.09074f,
  -1.47139f, -1.06100f, -0.24087f, 1.10382f,  -0.32837f, -1.39592f,
  -0.14741f, -0.43954f, 1.72137f,  -0.21704f,
};

static const float av1_tx_type_nn_weights_16x32_hor_layer1[64] = {
  -0.81860f, -0.80745f, -0.43612f, 0.58656f,  0.37455f, -0.56519f, -1.71536f,
  0.23278f,  0.23951f,  1.09610f,  0.49986f,  0.43375f, -0.53182f, 0.17376f,
  1.05626f,  -0.61743f, -0.71777f, -0.66943f, 1.40091f, 0.34426f,  1.14295f,
  0.45571f,  -0.52504f, -0.00303f, 0.06044f,  0.66119f, -0.60340f, -1.14344f,
  -0.28045f, 0.12742f,  0.61484f,  -0.41016f, 1.36102f, -0.86969f, -0.52728f,
  1.01725f,  0.67083f,  -0.10138f, 1.36406f,  0.34066f, 0.12498f,  0.86595f,
  -0.39636f, -0.27888f, -0.40244f, 0.09847f,  0.81178f, -0.45313f, 1.39127f,
  0.99865f,  -0.57908f, 0.55072f,  0.49638f,  1.11524f, 1.85504f,  -0.28316f,
  -0.05195f, -0.23284f, 0.26461f,  -1.28120f, 0.60707f, -0.06110f, 0.74085f,
  0.63304f,
};

static const float av1_tx_type_nn_bias_16x32_hor_layer1[4] = {
  0.71765f,
  1.40400f,
  0.32221f,
  3.07234f,
};

static const NN_CONFIG av1_tx_type_nnconfig_16x32_hor = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_16x32_hor_layer0,
      av1_tx_type_nn_weights_16x32_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_16x32_hor_layer0,
      av1_tx_type_nn_bias_16x32_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_16x32_ver_layer0[512] = {
  -0.01219f, 0.51494f,  0.25450f,  0.45788f,  -0.87277f, 0.32954f,  -0.04851f,
  -0.24321f, -0.40000f, 0.21915f,  0.14108f,  0.98268f,  0.18989f,  0.54298f,
  0.36349f,  0.38931f,  1.08124f,  0.87199f,  1.03553f,  1.14777f,  1.04254f,
  1.11336f,  0.92198f,  0.84715f,  1.89363f,  1.21587f,  0.72377f,  1.25097f,
  0.84231f,  0.95529f,  1.12346f,  0.19113f,  -0.04559f, 0.56859f,  0.59747f,
  0.60176f,  0.82465f,  0.59009f,  0.67240f,  1.58674f,  -0.92951f, -0.23449f,
  0.11923f,  -0.19151f, -0.15914f, 0.03146f,  -0.16541f, 0.17181f,  -0.21834f,
  0.21906f,  0.96708f,  0.36085f,  -0.42380f, -2.25681f, -0.48812f, 0.72875f,
  0.06585f,  0.18818f,  -0.02109f, -0.10996f, 0.00187f,  -0.02078f, 0.04484f,
  -0.07171f, 0.94773f,  -0.33466f, 0.28484f,  0.14791f,  0.30274f,  0.13377f,
  0.40970f,  0.45133f,  1.69265f,  -0.36422f, -0.15889f, 0.07670f,  0.44675f,
  -0.28665f, -0.07097f, 1.03803f,  -0.83274f, -0.24571f, 0.08039f,  -0.23790f,
  -0.23276f, -0.28031f, 0.26451f,  -0.18513f, -2.23336f, -0.62073f, 0.32495f,
  -0.67644f, -0.08559f, -0.36382f, -0.24515f, -0.01899f, 0.09012f,  0.19723f,
  0.04017f,  0.31624f,  0.58369f,  0.30411f,  -0.81165f, -2.58541f, -0.20491f,
  0.68089f,  -0.14799f, 0.13925f,  0.12867f,  0.15229f,  0.06887f,  -0.03784f,
  0.02288f,  -0.28712f, 0.14107f,  0.29485f,  -0.11662f, 0.25239f,  0.30311f,
  -0.07377f, -0.10962f, 0.59856f,  0.47967f,  0.01847f,  -0.27889f, 0.46786f,
  0.18118f,  0.09355f,  -2.10076f, 0.38823f,  0.28202f,  0.29104f,  0.86977f,
  0.52377f,  0.21161f,  0.72888f,  -0.00952f, 0.15982f,  -0.14651f, 0.28763f,
  -0.14155f, 0.00093f,  0.08351f,  0.34685f,  -0.22066f, 0.20378f,  0.25416f,
  0.03423f,  -0.11068f, -0.41612f, 0.56913f,  -0.06697f, -0.12585f, -0.21033f,
  -0.14513f, -0.04477f, -0.35778f, 0.03437f,  0.06956f,  -0.25356f, -1.46010f,
  -0.08142f, 0.11926f,  -0.63551f, -0.13882f, 0.34164f,  0.10821f,  1.07323f,
  -0.62435f, -0.27116f, 0.25971f,  0.11952f,  -0.39480f, -0.05474f, -0.12582f,
  0.28289f,  0.13723f,  0.58369f,  0.41865f,  0.28574f,  1.01357f,  0.46661f,
  0.61717f,  0.85708f,  -0.03930f, -0.38013f, -0.33888f, -0.20561f, -0.19087f,
  -0.01041f, 0.12119f,  -0.20786f, 0.55915f,  0.67511f,  0.55554f,  0.56540f,
  0.76647f,  0.54766f,  0.45166f,  0.61384f,  0.95407f,  -0.06811f, -0.62132f,
  0.12713f,  0.63713f,  2.04090f,  1.17054f,  0.00469f,  -0.93692f, -0.24136f,
  -0.04281f, -0.15787f, 0.37956f,  -0.09174f, -0.72494f, 0.55285f,  -1.40996f,
  -0.54077f, 0.38445f,  -0.08258f, 0.64259f,  -0.54058f, -0.49865f, 1.41371f,
  0.89014f,  0.78788f,  0.37919f,  0.87447f,  -0.00760f, -0.00947f, 0.16323f,
  -0.36632f, -1.38115f, -0.24619f, 0.40490f,  -0.08871f, -0.25365f, -0.60842f,
  0.11128f,  0.18658f,  -0.86001f, -0.28271f, 0.39572f,  -0.29930f, -0.10110f,
  0.33706f,  0.21731f,  0.15383f,  -0.01707f, 0.02812f,  0.31192f,  0.39742f,
  0.38260f,  -0.48263f, 0.57385f,  0.53239f,  -0.60013f, -0.63211f, -0.45140f,
  -0.73520f, -0.95260f, -0.70633f, -0.96190f, 0.01747f,  -0.05195f, -0.07138f,
  -1.09535f, -0.63548f, -1.55700f, -0.35721f, -0.18923f, 0.77568f,  0.09419f,
  0.36919f,  -0.32761f, -0.06597f, -0.38988f, -0.43674f, -0.24284f, 0.36906f,
  0.28414f,  0.19273f,  -0.68516f, 0.09514f,  -0.45381f, 0.19917f,  -0.32377f,
  1.32549f,  0.08244f,  -0.64405f, 0.13195f,  2.85307f,  0.47631f,  -0.33408f,
  0.04168f,  0.18585f,  -0.18029f, 0.07986f,  -0.08816f, -0.00703f, -0.01515f,
  -0.13164f, 0.00571f,  0.05676f,  1.51425f,  0.73360f,  0.43486f,  -0.08223f,
  -0.06183f, -0.57098f, -0.29948f, 0.05945f,  0.19238f,  -0.47980f, -0.35902f,
  -0.19931f, 0.43443f,  0.67436f,  0.78573f,  0.25703f,  1.01863f,  0.99047f,
  0.95228f,  1.02429f,  1.19264f,  0.29935f,  -0.26583f, -0.98749f, -0.46167f,
  -0.29727f, -0.10515f, -0.39790f, -0.59321f, -0.61925f, -0.95452f, 0.04292f,
  -0.48273f, -0.91195f, -0.45971f, -0.46355f, -0.88319f, -0.51712f, -0.47682f,
  -0.86110f, -0.59178f, -0.57163f, -0.94946f, 0.19627f,  -0.18699f, 0.11037f,
  1.39110f,  0.05715f,  3.00762f,  1.52243f,  0.25028f,  0.12779f,  -0.12871f,
  0.04764f,  0.08288f,  -0.16572f, -0.06580f, 0.05845f,  -0.01474f, 0.04886f,
  -0.10000f, 0.12911f,  -0.01416f, -0.12472f, 0.14358f,  0.16554f,  0.08853f,
  0.13418f,  -0.05408f, -0.13871f, -0.00049f, 0.20725f,  -0.05603f, 0.27885f,
  -0.14277f, 0.29653f,  -0.24739f, 0.10101f,  -0.17068f, -2.43802f, 0.41834f,
  0.49784f,  0.34949f,  0.98487f,  0.16792f,  1.07355f,  0.32546f,  1.32377f,
  -0.08584f, 0.85214f,  -0.05721f, 0.90307f,  0.20167f,  0.52664f,  -0.14478f,
  0.64997f,  0.06846f,  0.32475f,  0.64453f,  0.70143f,  -0.03091f, -0.24958f,
  -0.39021f, -0.57693f, -0.18319f, 0.11793f,  -0.05948f, 0.36670f,  -0.27932f,
  0.14800f,  -0.55459f, -0.89673f, 0.65922f,  0.54308f,  -0.16731f, -0.59731f,
  -0.20705f, -0.18183f, -0.05645f, -0.06829f, -0.40210f, -0.27955f, 0.28054f,
  0.57665f,  0.14171f,  0.54693f,  -0.22144f, -0.59664f, 0.13295f,  0.07057f,
  -0.19698f, 0.03328f,  -0.09687f, -0.32390f, -0.11506f, -0.40406f, -0.11473f,
  0.10399f,  -0.29824f, 0.16028f,  0.00053f,  0.22699f,  0.04203f,  -0.43880f,
  -0.12654f, 0.12172f,  0.21087f,  -0.46350f, -0.22081f, -0.06173f, -0.23287f,
  0.90314f,  0.04466f,  -0.06149f, 0.32682f,  0.16609f,  -0.58991f, -0.03786f,
  -0.41329f, 0.02632f,  0.23411f,  0.25344f,  0.16468f,  0.31007f,  0.21845f,
  0.32462f,  0.33945f,  0.11527f,  -0.35926f, -0.18584f, 0.29340f,  0.78199f,
  2.39287f,  0.53838f,  -1.55085f, 0.02238f,  -0.26153f, -0.42498f, -0.02460f,
  0.19261f,  -0.10870f, -0.08453f, -0.39561f, 0.08600f,  0.36310f,  0.58439f,
  -0.59526f, 0.13104f,  -0.06703f, -0.17529f, -0.41431f, -0.23121f, -0.32394f,
  -0.33324f, -0.21405f, -0.41702f, -0.29236f, -0.31766f, -0.33512f, -0.22679f,
  -0.13680f, -0.00118f, -1.81744f, -2.34798f, -1.08048f, -0.29883f, -0.29123f,
  -0.01752f,
};

static const float av1_tx_type_nn_bias_16x32_ver_layer0[32] = {
  1.02458f,  -1.02185f, -0.18978f, 0.05981f,  -0.94931f, 0.34544f,  0.04415f,
  -0.60036f, -0.11368f, -0.14154f, 1.23438f,  0.51640f,  -0.57587f, -0.91380f,
  0.95720f,  0.68298f,  -0.06353f, -2.14960f, -0.11080f, 0.79380f,  -0.94199f,
  0.43040f,  0.01358f,  0.07201f,  -0.49689f, -0.14839f, -0.80132f, -0.13925f,
  -0.11834f, -0.24998f, -0.33976f, 0.35497f,
};

static const float av1_tx_type_nn_weights_16x32_ver_layer1[128] = {
  0.87367f,  -1.06469f, -0.50829f, -0.70540f, 1.14596f,  -1.12346f, -0.94467f,
  0.01380f,  -0.18911f, 0.07961f,  -0.18626f, 0.61902f,  -0.64423f, 1.21545f,
  1.01149f,  0.26309f,  1.50380f,  1.93940f,  -0.64064f, 1.03987f,  -1.88000f,
  -0.44574f, -1.53303f, 1.36307f,  1.00292f,  0.37031f,  0.21594f,  0.16758f,
  0.02592f,  -0.77431f, -0.31797f, -1.53826f, 1.14013f,  -1.21957f, 0.04571f,
  -0.22168f, 0.32299f,  0.25949f,  -0.13306f, 0.17850f,  0.92494f,  0.19999f,
  0.07494f,  -0.03362f, -0.53453f, 1.02970f,  -0.22947f, 0.73964f,  1.08445f,
  0.16855f,  -0.02686f, 0.25254f,  0.05952f,  0.02194f,  0.05649f,  0.39195f,
  0.14139f,  0.53843f,  -0.06959f, -0.06993f, -0.14151f, -0.53147f, 0.17481f,
  -1.21977f, 0.62932f,  1.07173f,  0.24049f,  -0.51574f, 0.97492f,  -0.28169f,
  -0.15406f, -0.05441f, -0.25415f, 0.16583f,  0.43674f,  -0.00593f, -0.09277f,
  0.61402f,  1.35562f,  -0.03926f, 0.18967f,  -0.29548f, -0.55509f, 0.23661f,
  0.05023f,  0.36226f,  -0.83314f, 0.39357f,  0.19943f,  -0.63431f, -0.03847f,
  0.12213f,  0.62024f,  -0.11704f, -0.22483f, 0.96624f,  0.18518f,  0.09181f,
  -0.63068f, 0.66797f,  0.74107f,  0.40624f,  0.70636f,  -0.06921f, 0.34175f,
  -0.15513f, 2.07844f,  0.22126f,  0.52919f,  0.26793f,  -0.50018f, 1.10549f,
  0.10970f,  0.05831f,  0.82842f,  -1.22975f, 1.78377f,  0.92679f,  2.01480f,
  -1.19011f, -0.53381f, 0.38533f,  0.45579f,  -0.10683f, -0.40828f, 0.31398f,
  0.14978f,  0.91325f,
};

static const float av1_tx_type_nn_bias_16x32_ver_layer1[4] = {
  1.03659f,
  1.80249f,
  1.25710f,
  1.32000f,
};

static const NN_CONFIG av1_tx_type_nnconfig_16x32_ver = {
  16,  // num_inputs
  4,   // num_outputs
  1,   // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_16x32_ver_layer0,
      av1_tx_type_nn_weights_16x32_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_16x32_ver_layer0,
      av1_tx_type_nn_bias_16x32_ver_layer1,
  },
};
/******************************************************************************/

// Tx type model for 32x16 block.
static const float av1_tx_type_nn_weights_32x16_hor_layer0[512] = {
  -0.07289f, 0.30798f,  0.41881f,  0.33434f,  -0.01599f, 0.85307f,  -0.16060f,
  -0.07922f, -0.04693f, 0.29186f,  0.44117f,  1.02417f,  0.12447f,  0.46321f,
  0.40060f,  0.50140f,  0.48338f,  0.47298f,  0.36585f,  0.42821f,  0.41289f,
  0.47534f,  0.42900f,  0.26061f,  0.45887f,  0.38163f,  0.17302f,  1.00888f,
  1.79910f,  1.36140f,  0.24471f,  0.04557f,  1.10823f,  0.74325f,  0.91210f,
  0.81387f,  0.98865f,  -0.09874f, 0.55146f,  0.19385f,  -0.50752f, -0.17249f,
  0.27261f,  -0.02763f, -0.03286f, 0.09122f,  0.07015f,  0.20012f,  0.68983f,
  -1.25345f, -0.00145f, 0.71567f,  0.54948f,  -0.56154f, -0.28918f, 0.11997f,
  -0.09907f, 0.09195f,  0.05768f,  0.15558f,  0.11284f,  -0.35195f, -0.08723f,
  -0.03571f, 0.94031f,  0.63737f,  0.98202f,  0.93826f,  0.87126f,  0.88530f,
  0.97697f,  0.55283f,  0.58670f,  0.86502f,  0.97008f,  0.99709f,  0.66214f,
  0.96660f,  0.99890f,  0.31945f,  -1.00301f, 0.13215f,  -0.03950f, 0.21148f,
  0.05128f,  0.10955f,  0.44839f,  -0.33438f, -2.09773f, 0.13908f,  0.58669f,
  0.25268f,  -0.24006f, 0.01286f,  -0.05732f, 0.03401f,  -0.06896f, 0.35397f,
  0.05133f,  -0.21449f, -0.38437f, -0.32326f, -0.38731f, -0.44419f, 0.25968f,
  -0.29422f, -0.12553f, -0.08896f, -0.16400f, -0.22309f, 0.21380f,  -0.26912f,
  0.06866f,  -0.25694f, 0.17632f,  0.32032f,  -0.10666f, 0.26278f,  0.31877f,
  -0.09338f, -0.14289f, 0.54232f,  0.46070f,  0.00059f,  -0.27914f, 0.45177f,
  0.16274f,  -0.08811f, -0.45791f, 0.53946f,  -0.16794f, 0.16229f,  0.11840f,
  -0.24435f, 0.26894f,  -0.33180f, -0.47314f, 0.34061f,  -0.13939f, 0.13321f,
  -0.05208f, -0.18139f, -0.35234f, 1.37298f,  -0.19360f, 0.21728f,  0.26088f,
  0.04045f,  -0.10763f, -0.40470f, 0.50026f,  -0.06726f, -0.12871f, -0.20963f,
  -0.14583f, -0.04711f, -0.35988f, 0.03091f,  0.06491f,  -0.31668f, -0.52190f,
  0.23397f,  -0.13984f, -0.15207f, -0.49977f, 0.51205f,  0.12559f,  -0.03631f,
  0.33447f,  -0.36684f, 0.17533f,  0.15671f,  -0.00096f, 0.06817f,  0.20922f,
  0.34006f,  0.71260f,  0.45024f,  0.53033f,  0.15645f,  0.76019f,  0.56870f,
  0.83066f,  0.63022f,  1.74436f,  -0.24798f, 0.06795f,  -0.00749f, 0.17795f,
  0.10371f,  0.06527f,  0.41054f,  0.49003f,  0.34630f,  0.02615f,  0.30320f,
  -0.47133f, -0.49584f, 0.21775f,  0.27530f,  -0.29977f, -0.64269f, 0.52627f,
  -0.02492f, 0.08077f,  0.40786f,  -0.36015f, -0.70714f, -1.98185f, -0.28187f,
  0.35018f,  -0.06105f, -0.12710f, 0.06606f,  -0.27805f, 0.44630f,  -0.84731f,
  -0.26699f, 0.25856f,  0.06194f,  -0.18674f, -0.11560f, -0.43277f, 1.10579f,
  0.95876f,  0.17415f,  0.56386f,  0.68426f,  0.50180f,  0.24844f,  0.12347f,
  0.15281f,  -0.19089f, 0.52279f,  0.41860f,  -0.05270f, -0.17029f, -0.03542f,
  0.10621f,  -0.25088f, 0.24070f,  -0.08951f, 0.29950f,  -0.36720f, 0.02151f,
  0.20129f,  -0.70066f, -0.23144f, -0.20070f, -0.39262f, -0.01597f, -0.05591f,
  0.23814f,  -0.25991f, 0.05812f,  0.60554f,  -0.06106f, -0.58326f, 0.28762f,
  -0.18747f, 0.08232f,  -0.04243f, -0.03293f, 0.14722f,  -0.13017f, -0.67263f,
  0.38698f,  -0.18207f, -0.11496f, -0.27976f, -0.55345f, 1.42872f,  0.04684f,
  0.04214f,  0.00030f,  0.02410f,  0.19966f,  -0.04246f, 0.00442f,  0.23121f,
  0.13364f,  0.21548f,  -0.12748f, -0.14066f, -0.28354f, 0.59937f,  -0.27553f,
  1.57503f,  -0.01050f, -0.17724f, 0.44110f,  -0.80334f, 0.72064f,  1.00501f,
  -0.72638f, 0.02774f,  0.48540f,  -0.72016f, -0.27721f, 0.31559f,  0.07322f,
  0.20279f,  -0.19647f, 0.02352f,  0.12662f,  0.19743f,  0.30543f,  0.25712f,
  0.44702f,  0.16417f,  0.17888f,  -2.58469f, 0.20555f,  0.57782f,  -0.10892f,
  0.14527f,  0.82251f,  0.04200f,  0.44626f,  0.10818f,  0.71204f,  0.62903f,
  0.69178f,  0.73603f,  0.52717f,  0.83020f,  0.48824f,  1.03270f,  -0.00152f,
  0.07958f,  0.24181f,  -0.78839f, -0.74214f, -0.72998f, -1.58694f, 0.17735f,
  0.56318f,  0.32580f,  -0.58503f, -0.33673f, -0.00838f, 0.48924f,  0.43362f,
  0.12750f,  0.00295f,  0.38624f,  0.17037f,  0.00729f,  -0.26256f, -0.41669f,
  0.36847f,  0.22424f,  1.33334f,  0.18112f,  0.37682f,  0.49173f,  -0.45240f,
  -0.04857f, -0.35038f, -0.83099f, -0.01988f, 0.03497f,  0.38033f,  0.13685f,
  0.17597f,  0.28668f,  0.31193f,  -0.43281f, 0.43267f,  -0.50495f, 0.01969f,
  0.14131f,  -0.09326f, -0.39425f, -0.62048f, -0.09119f, -0.28306f, -0.52671f,
  -0.38584f, -0.10953f, 0.19669f,  0.34540f,  -0.49941f, 0.04605f,  -0.43535f,
  0.27519f,  0.03659f,  -0.31961f, 0.13330f,  0.87009f,  0.20101f,  -0.70392f,
  -0.27883f, 0.33874f,  -0.34308f, 0.67760f,  0.88195f,  0.55752f,  -0.26563f,
  0.17875f,  0.06964f,  0.87607f,  1.47616f,  0.46747f,  -0.56408f, -0.39352f,
  -0.16427f, -0.41185f, 0.14187f,  0.19265f,  -0.58613f, 0.56345f,  -0.17729f,
  -0.11320f, 0.08752f,  -0.01329f, 1.20981f,  0.45170f,  -0.20571f, -0.01150f,
  0.26476f,  0.13508f,  0.22020f,  -0.42684f, -0.22499f, -1.51212f, 0.86648f,
  0.21776f,  0.24666f,  0.71339f,  0.42742f,  -0.00952f, 0.14762f,  0.07693f,
  -0.19599f, 0.03075f,  -0.09703f, -0.32483f, -0.11616f, -0.40461f, -0.11693f,
  0.10038f,  -0.30038f, 0.14686f,  0.00548f,  0.20350f,  0.00763f,  -0.43756f,
  -0.01997f, 0.00902f,  0.07470f,  -0.41441f, -0.20605f, 0.07626f,  -0.34973f,
  0.47455f,  -0.15251f, -0.05325f, 0.04964f,  0.32477f,  -0.54604f, 0.25273f,
  -0.18461f, -0.30841f, 0.64908f,  0.60752f,  0.64148f,  0.72788f,  0.71232f,
  0.58597f,  0.73017f,  0.58857f,  0.71908f,  0.59860f,  0.61849f,  0.99398f,
  0.39572f,  -0.36165f, -1.88646f, 0.14384f,  -0.60541f, -0.21380f, -0.55498f,
  -0.50960f, -0.08801f, 0.51892f,  0.19126f,  0.57879f,  1.19447f,  0.25673f,
  -0.21631f, -0.43562f, -0.27497f, -0.02206f, -0.56169f, 0.58952f,  -0.60983f,
  -0.64088f, -0.69087f, -0.56261f, -0.74089f, -0.65063f, -0.66978f, -0.60836f,
  -0.92770f, -0.77182f, -1.61031f, -0.70007f, -0.68402f, -0.42242f, -0.66722f,
  -0.14533f,
};

static const float av1_tx_type_nn_bias_32x16_hor_layer0[32] = {
  1.53781f,  -0.49320f, -0.31646f, 0.02826f,  -1.05554f, 0.06559f,  -0.12399f,
  -0.61671f, -0.28956f, -0.15419f, 0.87189f,  -0.43375f, -1.08477f, -0.66006f,
  0.36233f,  0.82678f,  -0.92342f, -1.47101f, -0.02937f, -0.16497f, -0.75457f,
  0.50173f,  -0.07560f, 0.71598f,  1.50795f,  -0.04745f, -0.14008f, -0.18510f,
  -0.14988f, -0.67044f, 0.79659f,  0.70610f,
};

static const float av1_tx_type_nn_weights_32x16_hor_layer1[128] = {
  0.84983f,  -0.62530f, -0.82600f, -0.52563f, -0.11942f, -0.50279f, -0.13425f,
  -0.02850f, 0.50767f,  0.10252f,  0.24540f,  0.67748f,  -0.43483f, -0.22242f,
  0.23431f,  0.57287f,  0.69560f,  1.13814f,  -0.47427f, -0.55858f, -1.47072f,
  0.26587f,  -0.36335f, 0.83060f,  1.01645f,  -0.52895f, -0.11614f, 0.17390f,
  -0.13664f, -0.83098f, -0.07985f, -1.36820f, 0.47759f,  -0.55612f, 0.46852f,
  0.07406f,  -0.80467f, 0.23059f,  0.09992f,  -0.06164f, 0.13541f,  0.06135f,
  0.83605f,  -0.53224f, -0.13867f, 0.93838f,  -0.61290f, 0.27732f,  -0.46688f,
  -0.41810f, 0.12885f,  0.13619f,  -0.24612f, 0.07215f,  0.98866f,  0.10993f,
  1.05799f,  -0.27146f, -0.00079f, -0.08585f, 0.08322f,  -0.33809f, 0.67598f,
  -1.06515f, 1.28866f,  0.61028f,  -0.31704f, -0.59905f, 1.62151f,  0.10969f,
  0.20671f,  -0.17818f, 0.14170f,  0.19322f,  0.30602f,  0.93111f,  0.19011f,
  -0.45609f, 0.82506f,  0.32936f,  -0.07858f, -0.27106f, -0.31638f, 0.23299f,
  0.81491f,  0.32584f,  -0.52093f, -0.32472f, 0.53643f,  -0.42605f, 0.01641f,
  0.09002f,  0.15832f,  -0.08790f, 0.05511f,  1.00730f,  0.46309f,  0.68166f,
  -0.18835f, 0.64512f,  -1.00540f, 0.86802f,  0.18981f,  -0.06982f, -0.24514f,
  -0.08027f, 0.61199f,  -0.20830f, 0.72001f,  0.17477f,  0.06511f,  0.00801f,
  -0.43590f, 0.37257f,  0.70323f,  0.60233f,  1.62541f,  0.74383f,  -0.22254f,
  -0.33892f, 0.22881f,  0.62817f,  0.68915f,  -0.06417f, 0.00969f,  1.65869f,
  0.89060f,  0.75948f,
};

static const float av1_tx_type_nn_bias_32x16_hor_layer1[4] = {
  0.95359f,
  1.56043f,
  1.06017f,
  2.54520f,
};

static const NN_CONFIG av1_tx_type_nnconfig_32x16_hor = {
  16,  // num_inputs
  4,   // num_outputs
  1,   // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_32x16_hor_layer0,
      av1_tx_type_nn_weights_32x16_hor_layer1,
  },
  {
      av1_tx_type_nn_bias_32x16_hor_layer0,
      av1_tx_type_nn_bias_32x16_hor_layer1,
  },
};

static const float av1_tx_type_nn_weights_32x16_ver_layer0[128] = {
  1.30219f,  1.30548f,  1.33334f,  1.20560f,  1.01572f,  1.38100f,  1.37504f,
  0.12599f,  -0.96957f, 0.19400f,  0.75734f,  0.11295f,  -0.40447f, -1.53062f,
  -0.82980f, 0.02168f,  -1.11289f, -0.66861f, -0.83663f, -0.91455f, -0.78618f,
  -0.87176f, -1.10711f, 0.71207f,  1.49689f,  -0.12715f, 0.29357f,  0.35234f,
  0.61016f,  0.80708f,  0.83564f,  1.05961f,  -0.99842f, 0.82004f,  0.02638f,
  0.44606f,  0.32298f,  0.21321f,  0.47290f,  -0.71442f, -2.81050f, -0.02520f,
  -0.08919f, 0.00369f,  -0.05257f, -0.07011f, -0.16394f, 0.06290f,  0.80086f,
  0.32349f,  0.47411f,  1.36126f,  1.68162f,  0.91325f,  -0.27495f, 0.00262f,
  0.06025f,  0.42832f,  0.36965f,  0.38063f,  0.32772f,  0.40914f,  0.44510f,
  3.02239f,  -1.84077f, 0.49536f,  -0.27340f, -0.10437f, -0.34293f, -0.08047f,
  -0.29651f, -0.97111f, -0.34187f, 0.52869f,  1.27240f,  1.20306f,  1.19121f,
  1.28742f,  0.26393f,  -0.62319f, 0.92285f,  -0.08303f, -0.33118f, -0.13053f,
  0.24875f,  -0.52089f, 0.44691f,  -1.08908f, 1.20921f,  0.36538f,  -0.46792f,
  -0.18855f, -0.13443f, -0.28472f, -0.10353f, 0.06911f,  0.68519f,  0.08228f,
  -0.49027f, -0.34381f, 0.04719f,  -0.33298f, 0.72525f,  0.09538f,  -0.29216f,
  -0.07260f, -0.55827f, 0.54542f,  -0.10144f, -0.09292f, -0.14427f, -0.38361f,
  -0.41559f, 0.75338f,  -0.04530f, 0.27944f,  0.06932f,  -0.11537f, 0.29568f,
  1.92155f,  -0.98996f, -0.08841f, 0.49386f,  0.15947f,  0.53290f,  1.46747f,
  0.59360f,  0.25468f,
};

static const float av1_tx_type_nn_bias_32x16_ver_layer0[16] = {
  -1.19673f, 0.33043f,  0.24408f, 0.46221f,  2.00646f, 0.19031f,
  -0.64944f, -0.43452f, 1.04400f, 1.47371f,  0.52460f, -1.39577f,
  0.83852f,  -0.25536f, 1.33200f, -0.24444f,
};

static const float av1_tx_type_nn_weights_32x16_ver_layer1[64] = {
  -1.31447f, -0.86455f, 0.85217f,  1.00048f,  0.37395f, -1.35713f, -0.54032f,
  0.82803f,  0.89606f,  1.57696f,  0.68067f,  0.42512f, -0.26250f, 0.14621f,
  0.93249f,  -0.77690f, -0.93652f, -0.44488f, 0.68360f, -0.88178f, 1.89111f,
  0.67700f,  -0.29310f, 0.91604f,  -1.21881f, 1.11188f, 0.45045f,  -0.86119f,
  -0.09294f, 0.09360f,  0.80794f,  0.41027f,  1.80399f, -0.50362f, -1.44689f,
  0.85148f,  0.90707f,  -0.18458f, 0.14165f,  1.17367f, 0.70869f,  1.57147f,
  0.24692f,  0.16626f,  0.56794f,  0.07313f,  0.14728f, -0.74296f, 1.74127f,
  1.26560f,  0.17753f,  1.10194f,  0.56435f,  1.73779f, 1.42841f,  -1.16773f,
  0.24584f,  0.10813f,  -0.60187f, 0.79802f,  0.75229f, -0.06112f, 1.77282f,
  1.01058f,
};

static const float av1_tx_type_nn_bias_32x16_ver_layer1[4] = {
  0.83082f,
  2.03845f,
  0.59627f,
  2.31341f,
};

static const NN_CONFIG av1_tx_type_nnconfig_32x16_ver = {
  8,  // num_inputs
  4,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_type_nn_weights_32x16_ver_layer0,
      av1_tx_type_nn_weights_32x16_ver_layer1,
  },
  {
      av1_tx_type_nn_bias_32x16_ver_layer0,
      av1_tx_type_nn_bias_32x16_ver_layer1,
  },
};
/******************************************************************************/

// Map tx_size to its corresponding neural net model for tx type prediction.
static const NN_CONFIG *av1_tx_type_nnconfig_map_hor[] = {
  &av1_tx_type_nnconfig_4x4,        // 4x4
  &av1_tx_type_nnconfig_8x8,        // 8x8
  &av1_tx_type_nnconfig_16x16,      // 16x16
  NULL,                             // 32x32
  NULL,                             // 64x64
  &av1_tx_type_nnconfig_4x8_hor,    // 4x8
  &av1_tx_type_nnconfig_8x4_hor,    // 8x4
  &av1_tx_type_nnconfig_8x16_hor,   // 8x16
  &av1_tx_type_nnconfig_16x8_hor,   // 16x8
  &av1_tx_type_nnconfig_16x32_hor,  // 16x32
  &av1_tx_type_nnconfig_32x16_hor,  // 32x16
  NULL,                             // 32x64
  NULL,                             // 64x32
  NULL,                             // 4x16
  NULL,                             // 16x4
  NULL,                             // 8x32
  NULL,                             // 32x8
  NULL,                             // 16x64
  NULL,                             // 64x16
};

static const NN_CONFIG *av1_tx_type_nnconfig_map_ver[] = {
  &av1_tx_type_nnconfig_4x4,        // 4x4 transform
  &av1_tx_type_nnconfig_8x8,        // 8x8 transform
  &av1_tx_type_nnconfig_16x16,      // 16x16 transform
  NULL,                             // 32x32 transform
  NULL,                             // 64x64 transform
  &av1_tx_type_nnconfig_4x8_ver,    // 4x8 transform
  &av1_tx_type_nnconfig_8x4_ver,    // 8x4 transform
  &av1_tx_type_nnconfig_8x16_ver,   // 8x16 transform
  &av1_tx_type_nnconfig_16x8_ver,   // 16x8 transform
  &av1_tx_type_nnconfig_16x32_ver,  // 16x32 transform
  &av1_tx_type_nnconfig_32x16_ver,  // 32x16 transform
  NULL,                             // 32x64 transform
  NULL,                             // 64x32 transform
  NULL,                             // 4x16 transform
  NULL,                             // 16x4 transform
  NULL,                             // 8x32 transform
  NULL,                             // 32x8 transform
  NULL,                             // 16x64 transform
  NULL,                             // 64x16 transform
};

// Tx split model for 4x8 block.
static const float av1_tx_split_nn_weights_4x8_layer0[8 * 16] = {
  0.068650f,  -0.732073f, -0.040361f, 0.322550f,  -0.021123f, 0.212518f,
  -0.350546f, 0.435987f,  -0.111756f, -0.401568f, 0.069548f,  -0.313000f,
  0.073918f,  -0.373805f, -0.775810f, -0.124753f, 0.181094f,  -0.602641f,
  -0.026219f, -0.350112f, 0.020599f,  -0.311752f, -0.476482f, -0.669465f,
  -0.310921f, 0.348869f,  -0.115984f, 0.154250f,  0.200485f,  -0.016689f,
  0.020392f,  0.413810f,  0.634064f,  -0.627530f, 0.399178f,  -0.012284f,
  0.472030f,  0.091087f,  -0.706100f, -0.447944f, -0.274226f, 0.445656f,
  0.309339f,  0.505522f,  0.038496f,  -0.152809f, 0.408684f,  -0.068151f,
  0.271612f,  0.353233f,  -0.150365f, 0.075212f,  -0.035096f, 0.346615f,
  0.124382f,  0.477072f,  0.216288f,  0.070548f,  -0.106362f, 0.681613f,
  -0.145502f, -0.218631f, -0.099248f, -0.001983f, -0.196819f, -0.969045f,
  0.063009f,  -0.123053f, 0.104875f,  -0.137581f, -0.282933f, -0.003624f,
  -0.315659f, -0.333523f, -0.503000f, -0.100063f, -0.536711f, -0.059978f,
  -0.670248f, -0.353762f, 0.181109f,  0.289715f,  -0.071206f, 0.261141f,
  0.052796f,  -0.114554f, -0.139214f, -0.261380f, 0.075984f,  -0.647925f,
  -0.099528f, -0.677814f, 0.015712f,  -0.389385f, -0.095622f, -0.165117f,
  -0.109454f, -0.175240f, -0.393914f, 0.212330f,  0.037822f,  0.248280f,
  0.180197f,  0.110493f,  -0.525727f, -0.092329f, -0.524029f, -0.407364f,
  -0.542373f, -0.435626f, -0.912194f, 0.062794f,  0.160433f,  0.741485f,
  -0.103659f, -0.119327f, -0.055275f, 0.334358f,  0.014713f,  0.046327f,
  0.831114f,  -0.576682f, 0.354369f,  -0.082088f, 0.452331f,  0.039730f,
  -0.792429f, -0.385862f,
};

static const float av1_tx_split_nn_bias_4x8_layer0[16] = {
  0.238621f,  2.186830f,  1.383035f,  -0.867139f, 1.257119f, -0.351571f,
  -0.240650f, -0.971692f, 2.744843f,  1.116991f,  0.139062f, -0.165332f,
  0.262171f,  -1.598153f, -1.427340f, -1.602306f,
};

static const float av1_tx_split_nn_weights_4x8_layer1[16] = {
  -0.367134f, 1.373058f, -0.897039f, -0.326819f, -0.734030f, -0.290413f,
  -0.501249f, 0.505321f, -0.537692f, -0.767893f, 0.268697f,  0.278987f,
  0.085082f,  0.614986f, 0.847904f,  0.637578f,
};

static const float av1_tx_split_nn_bias_4x8_layer1[1] = {
  0.20586078f,
};

static const NN_CONFIG av1_tx_split_nnconfig_4x8 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_4x8_layer0,
      av1_tx_split_nn_weights_4x8_layer1,
  },
  {
      av1_tx_split_nn_bias_4x8_layer0,
      av1_tx_split_nn_bias_4x8_layer1,
  },
};
/******************************************************************************/

// Tx split model for 8x8 block.
static const float av1_tx_split_nn_weights_8x8_layer0[144] = {
  0.177983f,  -0.938386f, -0.074460f, -0.221843f, -0.073182f, -0.295155f,
  -0.098202f, -0.279510f, 0.001054f,  -0.119319f, -1.835282f, -0.581507f,
  -1.222222f, -1.049006f, -0.807508f, -0.454252f, -0.774879f, -0.180607f,
  -0.886976f, -0.231971f, -0.824677f, -0.351872f, -1.323819f, 0.235378f,
  0.015331f,  -0.341818f, 0.145549f,  -0.348362f, 0.147647f,  -0.323400f,
  0.047558f,  -0.553025f, -0.295485f, -0.330368f, -0.530605f, -0.407516f,
  0.447740f,  0.782381f,  -0.179164f, -0.584675f, -0.052645f, 0.038656f,
  -0.096783f, 0.038342f,  -0.170762f, -0.405844f, -0.552665f, -0.509866f,
  0.757204f,  -1.296465f, 0.631015f,  0.009265f,  0.646192f,  0.044523f,
  0.653161f,  0.033820f,  0.849639f,  -0.068555f, -1.036085f, -0.511652f,
  0.104693f,  -1.458690f, 0.286051f,  -0.089800f, 0.381564f,  -0.302640f,
  0.304465f,  -0.268706f, 0.432603f,  -0.117914f, -2.070031f, -0.565696f,
  -0.073027f, -1.783570f, -0.318144f, -0.320990f, -0.343966f, -0.140996f,
  -0.322977f, -0.232147f, -0.373210f, -0.158266f, -1.922305f, -0.634373f,
  0.101894f,  -0.221847f, 0.018412f,  -0.423887f, -0.266684f, -0.444930f,
  -0.196237f, 0.106638f,  -0.065834f, -0.538401f, -0.280772f, -0.620348f,
  1.089957f,  -0.799928f, 0.504112f,  -0.165763f, 0.578741f,  -0.172653f,
  0.547316f,  -0.143484f, 0.717220f,  -0.297190f, -1.237854f, -0.074819f,
  -0.977304f, -0.484092f, -0.646427f, -0.451443f, -0.612126f, -0.224475f,
  -0.731608f, -0.257077f, -0.665857f, -0.346742f, -1.216372f, 0.227267f,
  0.231249f,  -1.693073f, -0.035899f, 0.380845f,  -0.058476f, 0.409405f,
  -0.066679f, 0.406731f,  -0.068501f, 0.396748f,  0.639462f,  0.150834f,
  -0.418659f, -1.421931f, 0.101889f,  0.083573f,  0.129746f,  0.134460f,
  0.081185f,  0.127420f,  0.083664f,  0.051096f,  1.361688f,  0.386093f,
};

static const float av1_tx_split_nn_bias_8x8_layer0[12] = {
  4.280443f, 2.218902f, -0.256953f, 3.161431f,  2.082548f, 2.506052f,
  2.563224f, 1.421976f, -1.627813f, -1.436085f, 2.297265f, 1.500469f,
};

static const float av1_tx_split_nn_weights_8x8_layer1[12] = {
  1.178833f,  -0.428527f, -0.078737f, 0.381434f, -0.466895f, -0.901745f,
  -0.766968f, -0.356663f, 0.450146f,  0.509370f, -0.356604f, -0.443506f,
};

static const float av1_tx_split_nn_bias_8x8_layer1[1] = {
  -0.156294f,
};

static const NN_CONFIG av1_tx_split_nnconfig_8x8 = {
  12,  // num_inputs
  1,   // num_outputs
  1,   // num_hidden_layers
  {
      12,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_8x8_layer0,
      av1_tx_split_nn_weights_8x8_layer1,
  },
  {
      av1_tx_split_nn_bias_8x8_layer0,
      av1_tx_split_nn_bias_8x8_layer1,
  },
};
/******************************************************************************/

// Tx split model for 8x16 block.
static const float av1_tx_split_nn_weights_8x16_layer0[8 * 64] = {
  0.374660f,  0.218905f,  -0.139779f, 0.212141f,  0.056517f,  0.051114f,
  0.042860f,  -0.273258f, -0.340809f, 0.138983f,  -0.216996f, -0.241519f,
  -0.123244f, 0.078577f,  -0.472273f, -0.194201f, 0.125056f,  0.239761f,
  -0.332782f, 0.174782f,  -0.211400f, -0.129795f, 0.062195f,  0.113176f,
  -0.008869f, 0.140764f,  0.059833f,  0.163826f,  0.359293f,  -0.109797f,
  -0.022091f, -0.059536f, -0.188226f, 0.179709f,  0.031386f,  0.164790f,
  0.214364f,  0.198555f,  0.152262f,  -0.242980f, 0.319367f,  -0.136902f,
  0.046524f,  -0.043591f, 0.342178f,  -0.011757f, -0.014286f, 0.072871f,
  -0.278314f, -0.345303f, -0.252103f, -0.107154f, -0.235101f, -0.106739f,
  -0.120865f, -0.160042f, 0.240028f,  0.112902f,  -0.141587f, -0.703012f,
  -0.136591f, 0.318993f,  -0.154417f, -0.054668f, 0.192870f,  0.176166f,
  -0.029965f, 0.266942f,  -0.178384f, 0.038680f,  0.134403f,  -0.002426f,
  0.534825f,  -0.070923f, 0.413281f,  0.418148f,  0.093729f,  0.016454f,
  0.305358f,  -0.040512f, 0.069904f,  -0.227588f, -0.362220f, -0.031604f,
  -0.394901f, 0.071506f,  -0.342833f, -0.142550f, -0.164005f, 0.182600f,
  0.213062f,  0.076805f,  0.278758f,  0.125613f,  -0.035552f, 0.040971f,
  0.182785f,  -0.227961f, -0.105413f, -0.074949f, -0.084629f, -0.254767f,
  0.114657f,  0.047121f,  0.195902f,  0.264759f,  0.017799f,  0.210230f,
  0.150749f,  -0.142142f, 0.182494f,  -0.142415f, -0.259782f, -0.114830f,
  -0.198826f, 0.000061f,  -0.375668f, -0.276656f, -0.373202f, 0.210298f,
  0.422680f,  0.066960f,  0.351106f,  -0.209034f, 0.367195f,  -0.110274f,
  0.115573f,  -0.066642f, -0.389673f, -0.260447f, 0.056949f,  -0.180425f,
  0.069922f,  -0.153506f, -0.097053f, -0.111757f, 0.094069f,  0.144837f,
  -0.052984f, -0.506681f, -0.034474f, 0.279057f,  -0.105025f, 0.006656f,
  -0.125017f, -0.114096f, 0.103153f,  -0.117402f, -0.359472f, 0.072534f,
  0.110291f,  0.003088f,  -0.456897f, 0.038331f,  -0.322298f, 0.113942f,
  -0.119916f, -0.194392f, 0.093167f,  0.193459f,  0.074671f,  0.033602f,
  0.004440f,  -0.179578f, -0.036637f, -0.216172f, -0.296530f, -0.318992f,
  0.319160f,  -0.066218f, 0.291246f,  0.181292f,  0.089914f,  0.025273f,
  0.303128f,  0.019063f,  0.078545f,  -0.396919f, 0.014065f,  -0.122121f,
  0.037107f,  -0.151886f, -0.299392f, -0.172207f, -0.124571f, -0.232553f,
  0.102970f,  -0.225040f, 0.061059f,  -0.258188f, -0.469871f, -0.099607f,
  -0.061524f, -0.213700f, 0.070237f,  -0.289134f, -0.238225f, 0.256403f,
  -0.119344f, 0.067782f,  -0.398983f, -0.123975f, -0.200205f, -0.047038f,
  0.026569f,  0.031037f,  0.094302f,  -0.101239f, 0.433307f,  -0.303612f,
  0.088537f,  -0.164436f, 0.202471f,  -0.048592f, -0.251904f, 0.122577f,
  -0.309874f, -0.263405f, -0.292503f, 0.216589f,  0.035378f,  0.136599f,
  -0.145844f, -0.018211f, 0.174084f,  -0.449941f, -0.001428f, 0.064134f,
  0.039652f,  0.111083f,  -0.246076f, -0.204733f, 0.056559f,  -0.000123f,
  0.104049f,  0.138512f,  -0.128309f, 0.087855f,  0.232784f,  0.247138f,
  0.162766f,  0.154829f,  0.313605f,  -0.164115f, -0.050844f, 0.156549f,
  0.185279f,  -0.238962f, -0.308281f, -0.179592f, -0.193262f, 0.201670f,
  -0.203399f, -0.096831f, -0.127867f, 0.310674f,  -0.008181f, 0.004078f,
  -0.211038f, -0.193480f, -0.185639f, -0.150202f, -0.204858f, -0.240758f,
  0.114268f,  -0.032535f, -0.052403f, -0.234333f, -0.064072f, -0.208444f,
  -0.352853f, -0.224001f, -0.156330f, 0.215436f,  0.171846f,  0.291849f,
  0.108832f,  0.046991f,  -0.127801f, 0.032485f,  0.141493f,  0.123319f,
  -0.057250f, 0.315346f,  -0.061317f, -0.465086f, -0.130179f, -0.217841f,
  -0.239089f, -0.073251f, -0.327718f, 0.054905f,  -0.283169f, -0.028900f,
  0.071450f,  0.270072f,  0.248891f,  0.088052f,  0.253319f,  0.122808f,
  0.175490f,  -0.147805f, 0.089169f,  -0.045457f, -0.330788f, 0.099791f,
  -0.137376f, -0.195977f, -0.350942f, -0.284930f, -0.559037f, 0.030504f,
  0.162554f,  -0.199100f, -0.050453f, -0.131320f, -0.077863f, -0.066253f,
  -0.379723f, -0.424047f, -0.081182f, -0.252261f, -0.102815f, 0.058240f,
  -0.182036f, 0.176772f,  -0.070823f, 0.216054f,  -0.211533f, -0.232992f,
  0.279346f,  0.117984f,  0.236674f,  0.126625f,  -0.046220f, 0.044919f,
  0.278492f,  0.083944f,  0.180512f,  0.217994f,  0.401170f,  -0.064417f,
  0.011636f,  -0.139597f, -0.050020f, -0.268438f, -0.032803f, 0.024908f,
  -0.085713f, -0.012984f, -0.055192f, -0.338657f, 0.045826f,  -0.312849f,
  -0.023393f, -0.168800f, -0.030886f, -0.131816f, -0.253542f, -0.104812f,
  -0.354389f, 0.169464f,  0.094151f,  -0.217122f, -0.456397f, 0.211478f,
  0.219232f,  -0.155519f, -0.353700f, -0.264759f, -0.034709f, 0.034409f,
  -0.148639f, -0.132850f, -0.216791f, -0.118492f, 0.173721f,  -0.144181f,
  0.335028f,  0.176439f,  0.105980f,  0.169390f,  0.155615f,  -0.040618f,
  -0.176029f, 0.155569f,  -0.184833f, -0.171099f, -0.178663f, -0.032051f,
  -0.434334f, 0.092238f,  -0.263103f, 0.061804f,  -0.172957f, 0.005962f,
  -0.100176f, 0.125898f,  0.048092f,  -0.088141f, 0.247196f,  -0.221601f,
  -0.114474f, -0.124410f, -0.156393f, -0.181782f, -0.083562f, 0.034937f,
  0.403401f,  -0.046200f, 0.322259f,  0.219678f,  0.109850f,  0.051837f,
  0.196861f,  -0.019118f, 0.248818f,  -0.137567f, 0.127862f,  0.052293f,
  0.298726f,  0.275788f,  0.015344f,  0.058714f,  0.283691f,  -0.053794f,
  -0.123270f, -0.227761f, -0.141744f, -0.268515f, -0.007189f, -0.242117f,
  -0.252396f, -0.069017f, 0.034803f,  -0.003388f, -0.262577f, 0.062115f,
  -0.298393f, 0.215415f,  -0.153615f, 0.289902f,  0.085886f,  -0.504290f,
  0.077178f,  0.150861f,  -0.228848f, -0.261020f, 0.198204f,  0.162113f,
  0.346418f,  -0.286950f, 0.354756f,  -0.226419f, 0.024720f,  0.208037f,
  0.107286f,  -0.110849f, 0.104415f,  -0.207725f, 0.063932f,  -0.037748f,
  -0.167037f, -0.068282f, 0.320815f,  -0.051884f, 0.099989f,  -0.078388f,
  0.127071f,  0.046675f,  -0.336571f, -0.273080f, 0.264694f,  -0.007352f,
  -0.093828f, 0.094773f,  -0.144434f, 0.091795f,  -0.031615f, 0.056914f,
  0.064673f,  -0.136669f, 0.344734f,  0.225926f,  0.283451f,  -0.068354f,
  0.030572f,  0.180784f,  -0.378047f, -0.092962f, -0.083291f, 0.038970f,
  0.052094f,  -0.017932f, 0.216302f,  -0.184396f, 0.079888f,  0.210406f,
  -0.020627f, 0.244744f,  0.336972f,  -0.182914f, -0.220976f, -0.304225f,
  -0.330974f, -0.370868f, -0.084935f, -0.136489f, -0.210082f, -0.188088f,
  -0.408768f, 0.184693f,
};

static const float av1_tx_split_nn_bias_8x16_layer0[64] = {
  -0.274107f, 0.445751f,  0.234359f,  0.291593f,  0.163298f,  0.183707f,
  -0.548839f, -0.190779f, -0.163346f, -0.669028f, 0.399209f,  -0.354974f,
  0.000000f,  -0.254630f, 0.220149f,  0.371104f,  0.789759f,  0.270300f,
  0.195126f,  -0.206958f, 0.917708f,  -0.256232f, 1.131933f,  1.178944f,
  0.461270f,  0.246169f,  -0.818614f, -0.111986f, 0.759355f,  0.154889f,
  0.470299f,  -1.025250f, 0.678678f,  0.959346f,  -0.164105f, 0.544079f,
  -0.448733f, 0.649221f,  -0.536672f, 0.962758f,  -0.256427f, 0.808664f,
  -0.118694f, 0.684873f,  -0.015635f, -0.046469f, 0.075481f,  0.412647f,
  0.454456f,  -0.107169f, 0.775235f,  -0.261629f, -1.194849f, 0.010093f,
  -0.231289f, 0.658286f,  -0.769320f, 0.564545f,  0.482962f,  -0.131378f,
  -0.255844f, -0.078400f, 0.476752f,  0.643001f,
};

static const float av1_tx_split_nn_weights_8x16_layer1[64] = {
  -0.145065f, -0.145101f, 0.174786f,  0.196692f,  0.102025f,  -0.087735f,
  0.386353f,  -0.660539f, -0.183940f, 0.490045f,  -0.276404f, -0.145669f,
  0.209846f,  -0.085574f, -0.156821f, -0.377450f, -0.950010f, 0.450709f,
  -0.108545f, -0.261181f, 1.435606f,  -0.176621f, -1.158548f, 2.035680f,
  0.218069f,  -0.138629f, 0.305958f,  -0.277194f, -0.602468f, 0.203873f,
  0.120720f,  0.216095f,  -0.434502f, -0.579746f, -0.239450f, 0.755529f,
  0.545643f,  0.232091f,  0.330169f,  0.988136f,  -0.070465f, -0.345584f,
  -0.162455f, -0.617064f, 0.123881f,  -0.201098f, 0.222756f,  0.112932f,
  0.048647f,  -0.147890f, 0.394584f,  -0.262148f, 0.280564f,  -0.195432f,
  -0.047515f, 1.133410f,  0.255415f,  -0.299032f, -0.397807f, -0.153246f,
  -0.256734f, 0.177370f,  0.213522f,  -0.530158f,
};

static const float av1_tx_split_nn_bias_8x16_layer1[1] = {
  0.14910713f,
};

static const NN_CONFIG av1_tx_split_nnconfig_8x16 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      64,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_8x16_layer0,
      av1_tx_split_nn_weights_8x16_layer1,
  },
  {
      av1_tx_split_nn_bias_8x16_layer0,
      av1_tx_split_nn_bias_8x16_layer1,
  },
};
/******************************************************************************/

// Tx split model for 16x16 block.
static const float av1_tx_split_nn_weights_16x16_layer0[12 * 24] = {
  -0.177215f, -0.297166f, 0.299924f,  0.207878f,  0.216871f,  0.173264f,
  0.295464f,  0.048395f,  0.154731f,  0.305880f,  0.056787f,  -0.166617f,
  0.115653f,  -0.529477f, -0.073995f, -0.211746f, -0.018169f, 0.000788f,
  -0.024940f, -0.007055f, 0.001392f,  0.021678f,  -1.594600f, -0.099593f,
  0.332930f,  0.103574f,  0.158249f,  0.182601f,  0.332665f,  0.226207f,
  -0.139566f, 0.185531f,  0.099074f,  -0.185654f, -0.203121f, -0.285678f,
  -0.313453f, -0.294452f, -0.143707f, -0.031265f, -0.453030f, -0.061874f,
  -0.066150f, -0.099058f, -0.458879f, 0.127544f,  0.338314f,  -0.161350f,
  0.030091f,  -0.075528f, 0.004320f,  0.353690f,  -0.013480f, -0.420402f,
  -0.004659f, -0.329401f, -0.001745f, 0.227384f,  -0.055183f, 0.121405f,
  0.160340f,  0.143603f,  -0.221813f, 0.079107f,  -0.657639f, -0.084348f,
  -0.303414f, 0.046774f,  -0.367679f, 0.060005f,  0.168645f,  0.084421f,
  -0.133625f, 0.301375f,  0.079412f,  -0.419303f, 0.017235f,  0.068637f,
  0.018384f,  -0.428325f, -0.019753f, 0.149444f,  -0.474836f, -0.287162f,
  0.198083f,  0.028292f,  -0.299092f, -0.005849f, -0.256245f, 0.233277f,
  -0.217561f, -0.264003f, 0.269411f,  0.207032f,  -0.339411f, -0.198431f,
  -0.028521f, 0.158076f,  0.177116f,  0.345702f,  -0.145132f, 0.064623f,
  -0.090867f, 0.288816f,  -0.263198f, -0.071028f, -0.044546f, 0.380017f,
  -0.014100f, -0.271192f, -0.318559f, 0.129015f,  -0.050314f, -0.093355f,
  -0.578498f, 0.099090f,  -0.133080f, -0.029975f, -0.059828f, -0.157765f,
  -0.321153f, -0.343671f, -0.242959f, 0.128304f,  0.017170f,  0.072787f,
  -0.475838f, -0.003806f, -0.068615f, 0.150556f,  -0.159903f, -0.416513f,
  0.218794f,  -0.290456f, -0.084569f, -0.170014f, -0.044414f, -0.153069f,
  -0.077329f, -0.089747f, -0.096526f, 0.537952f,  0.134725f,  -0.006469f,
  -0.323335f, -0.168183f, -0.107163f, -0.139954f, 0.011286f,  -0.021712f,
  -0.513992f, 0.259135f,  -0.319808f, 0.077811f,  0.104613f,  0.370571f,
  0.185244f,  0.065530f,  -0.091098f, -0.573741f, 0.111934f,  0.437417f,
  -0.123691f, 0.220641f,  -0.024783f, -0.149460f, -0.354185f, -0.134127f,
  0.038015f,  -0.380596f, 0.250980f,  0.142208f,  0.135170f,  -0.131129f,
  -0.357556f, -0.530945f, 0.159672f,  -0.147025f, -0.377829f, -0.504508f,
  -0.492870f, 0.020753f,  0.142818f,  0.025172f,  0.086140f,  0.091283f,
  0.087491f,  -0.186415f, 0.177785f,  -0.195121f, -1.191148f, -0.477102f,
  0.023371f,  0.227004f,  -0.023502f, -0.242913f, -0.074398f, -0.153480f,
  0.162900f,  0.415509f,  -0.162565f, -0.131709f, -0.258852f, -0.252027f,
  -0.080845f, -0.330274f, 0.021874f,  0.232398f,  0.069277f,  0.220567f,
  -0.024237f, -0.366771f, 0.081673f,  -0.429906f, -0.302170f, 0.061045f,
  0.352777f,  -0.230376f, 0.408153f,  0.064758f,  0.142051f,  0.007219f,
  0.622878f,  0.212577f,  0.036489f,  0.081150f,  -0.284767f, 0.107763f,
  -0.529786f, -0.072190f, -0.300421f, -0.287959f, -0.568900f, 0.011547f,
  -0.131696f, -0.356854f, -0.587962f, -0.026598f, 0.405829f,  0.057565f,
  0.414265f,  -0.159155f, 0.221456f,  0.146314f,  0.265776f,  -0.006516f,
  0.473978f,  -0.186431f, 0.288672f,  -0.060437f, 0.083380f,  -0.205641f,
  0.360016f,  0.222041f,  0.420011f,  0.024579f,  0.377546f,  0.250380f,
  -0.069900f, 0.296743f,  0.073532f,  -0.243225f, -0.374987f, -0.387288f,
  -0.237255f, -0.287013f, 0.417831f,  -0.252988f, -0.257652f, -0.066775f,
  -0.253926f, 0.057841f,  0.346133f,  -0.157797f, -0.406028f, -0.286893f,
  0.274507f,  -0.452561f, 0.143381f,  -0.097755f, 0.021242f,  0.034561f,
  0.044115f,  0.004065f,  0.066729f,  0.043558f,  0.102991f,  -0.477574f,
};

static const float av1_tx_split_nn_bias_16x16_layer0[24] = {
  -0.479033f, 1.467402f,  -0.366291f, 0.372511f,  0.715322f,  -0.605500f,
  0.176848f,  0.032318f,  0.237429f,  -0.046047f, 0.452082f,  0.451805f,
  -0.822845f, 0.636762f,  -0.057350f, 1.163978f,  0.728287f,  0.603654f,
  -0.245519f, -0.893569f, -1.428185f, 0.808870f,  -0.076159f, 1.231976f,
};

static const float av1_tx_split_nn_weights_16x16_layer1[24] = {
  -0.176161f, 1.670188f, -0.180755f, -0.321326f, 0.249728f,  -0.170504f,
  -0.538432f, 0.033893f, 0.149842f,  0.404140f,  -0.377812f, 0.338838f,
  -0.176091f, 0.249844f, -0.362533f, 1.412460f,  0.196862f,  0.278194f,
  -0.140444f, 0.297746f, 0.172533f,  0.116470f,  -0.151656f, -0.603250f,
};

static const float av1_tx_split_nn_bias_16x16_layer1[1] = {
  0.184803f,
};

static const NN_CONFIG av1_tx_split_nnconfig_16x16 = {
  12,  // num_inputs
  1,   // num_outputs
  1,   // num_hidden_layers
  {
      24,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_16x16_layer0,
      av1_tx_split_nn_weights_16x16_layer1,
  },
  {
      av1_tx_split_nn_bias_16x16_layer0,
      av1_tx_split_nn_bias_16x16_layer1,
  },
};
/******************************************************************************/

// Tx split model for 32x32 block.
static const float av1_tx_split_nn_weights_32x32_layer0[12 * 32] = {
  -0.439303f, 0.004813f,  -0.365052f, -0.116868f, -0.356716f, -0.196537f,
  -0.196770f, -0.076096f, 0.357004f,  -0.044909f, -0.112910f, -0.129081f,
  0.156725f,  -0.386346f, 0.038971f,  0.160696f,  0.204923f,  -0.384333f,
  -0.319546f, 0.028179f,  -0.250524f, -0.289669f, -0.284138f, -0.258963f,
  -0.180854f, -0.000807f, -0.029620f, -0.353134f, 0.212408f,  0.141414f,
  0.303016f,  0.098066f,  0.482455f,  0.036069f,  -0.166279f, 0.210119f,
  -0.086337f, -0.023550f, -0.250796f, -0.183945f, -0.393856f, 0.170608f,
  -0.306403f, 0.026318f,  -0.277296f, 0.092684f,  -0.033584f, -0.018371f,
  -0.025043f, -0.257659f, -0.139163f, -0.206949f, -0.190105f, 0.028053f,
  0.361851f,  -0.364726f, -0.096771f, -0.184166f, -0.433228f, -0.182191f,
  -0.097051f, 0.259172f,  0.016432f,  0.259358f,  0.145059f,  0.037196f,
  0.091581f,  -0.219644f, 0.140384f,  -0.446837f, -0.234531f, 0.149508f,
  -0.083429f, 0.186189f,  -0.099890f, -0.111277f, 0.495214f,  0.085053f,
  -0.266613f, -0.051366f, 0.148593f,  0.111875f,  0.077787f,  -0.371653f,
  -0.146157f, -0.229235f, 0.076203f,  0.488975f,  0.096771f,  -0.009483f,
  0.192985f,  0.246273f,  -0.192671f, -0.557890f, -0.292650f, -0.088907f,
  -0.106892f, -0.329659f, 0.012105f,  -0.359326f, 0.170723f,  -0.004357f,
  0.171593f,  -0.478768f, -0.236016f, -0.035077f, 0.133731f,  0.137962f,
  -0.397926f, -0.155164f, -0.276709f, -0.186602f, -0.258301f, 0.036965f,
  -0.649359f, 0.127605f,  0.097930f,  0.182775f,  -0.313324f, 0.053349f,
  0.204203f,  -0.222948f, -0.059008f, -0.049759f, -0.056848f, 0.087497f,
  -0.039987f, -0.055042f, -0.041623f, -0.078424f, -0.317291f, -0.191398f,
  0.632147f,  0.221825f,  0.268394f,  -0.096357f, 0.442545f,  -0.007117f,
  -0.036125f, 0.000525f,  0.088092f,  -0.203653f, 0.086925f,  0.439141f,
  0.329889f,  -0.370050f, -0.194306f, -0.207430f, 0.132779f,  -0.217614f,
  -0.039444f, -0.053019f, -0.260725f, -0.116563f, -0.271048f, 0.283737f,
  -0.007300f, 0.062257f,  -0.347865f, -0.296767f, -0.359123f, 0.230459f,
  -0.189117f, -0.087622f, -0.561091f, 0.184182f,  -0.044980f, 0.012643f,
  0.241672f,  0.050272f,  -0.204851f, -0.159285f, -0.064081f, -0.118666f,
  -0.269471f, 0.231668f,  0.135749f,  -0.131162f, 0.062760f,  0.100949f,
  0.074967f,  -0.056918f, 0.251707f,  0.034098f,  0.341290f,  -0.105027f,
  0.313246f,  -0.092679f, -0.014632f, -0.390967f, 0.136881f,  -0.241554f,
  0.097674f,  0.110832f,  -0.390245f, 0.017654f,  -0.506222f, 0.065252f,
  0.244834f,  -0.171352f, -0.331702f, 0.111043f,  0.125217f,  -0.058116f,
  -0.382595f, -0.052545f, 0.114261f,  -0.493617f, 0.243984f,  -0.171053f,
  0.165009f,  -0.063020f, 0.096502f,  0.341339f,  -0.013443f, 0.056372f,
  0.339284f,  0.398376f,  0.389409f,  0.257252f,  0.517368f,  0.078856f,
  0.087716f,  -0.171092f, 0.227461f,  0.125307f,  -0.054423f, -0.143161f,
  0.224041f,  -0.086477f, -0.092548f, 0.072392f,  -0.061608f, 0.258347f,
  0.147033f,  -0.478244f, -0.204869f, 0.038552f,  -0.144563f, 0.224087f,
  -0.296705f, 0.153889f,  -0.064624f, 0.085265f,  -0.103826f, 0.127971f,
  0.019965f,  0.111937f,  -0.074187f, -0.029518f, -0.127305f, -0.012210f,
  0.042714f,  0.070052f,  -0.202360f, 0.348144f,  -0.132097f, -0.209585f,
  -0.248286f, -0.065774f, -0.089482f, -0.133226f, 0.325430f,  -0.013468f,
  -0.406090f, -0.144936f, 0.208620f,  0.343445f,  -0.059639f, 0.114857f,
  -0.069431f, -0.218725f, 0.190575f,  -0.368101f, 0.030030f,  0.062815f,
  -0.239369f, -0.537852f, 0.022487f,  0.023038f,  0.190788f,  0.040123f,
  -0.004304f, 0.060749f,  -0.108929f, 0.136796f,  -0.542875f, -0.227074f,
  -0.182244f, 0.082559f,  0.019149f,  0.178854f,  0.120284f,  0.009070f,
  0.068268f,  -0.544822f, 0.120536f,  0.354028f,  -0.119890f, -0.122055f,
  -0.405335f, 0.122341f,  -0.304412f, 0.062405f,  -0.302568f, -0.276505f,
  -0.120915f, -0.221841f, 0.282007f,  -0.253971f, 0.059517f,  -0.144976f,
  0.149391f,  -0.047355f, -0.167742f, -0.392333f, -0.041132f, 0.342135f,
  0.017485f,  0.021038f,  -0.023728f, -0.192181f, -0.103996f, 0.092873f,
  -0.114365f, -0.397732f, -0.065421f, 0.053084f,  0.035201f,  0.053019f,
  -0.105377f, -0.039500f, 0.131904f,  -0.123911f, -0.390328f, -0.125198f,
  -0.000126f, 0.014864f,  -0.220187f, 0.084056f,  -0.492155f, -0.164979f,
  0.133592f,  0.121519f,  -0.240813f, 0.186680f,  0.118673f,  0.235006f,
  -0.239894f, -0.185759f, -0.336992f, 0.209620f,  -0.298845f, 0.127803f,
  -0.083992f, 0.194340f,  -0.245378f, 0.212308f,  0.142512f,  -0.163324f,
  0.383495f,  0.291065f,  0.286620f,  -0.239957f, 0.225127f,  -0.174424f,
  0.297231f,  -0.045434f, 0.156444f,  -0.184273f, -0.204567f, 0.202551f,
  0.370019f,  -0.073910f, 0.344897f,  0.063100f,  0.338547f,  -0.099145f,
  0.391863f,  -0.214244f, -0.241734f, -0.281851f, -0.035133f, -0.153157f,
};

static const float av1_tx_split_nn_bias_32x32_layer0[32] = {
  0.143343f,  -0.021982f, -0.314939f, 0.170867f,  -0.081248f, 0.125758f,
  -0.355762f, 0.279798f,  1.027712f,  -0.434660f, 1.072005f,  0.668893f,
  -0.031216f, -0.528650f, 0.328349f,  0.543645f,  -0.188810f, 0.221110f,
  -1.638637f, 0.058045f,  -1.731105f, -0.444284f, 0.513693f,  0.890025f,
  0.160288f,  0.393312f,  0.332856f,  -0.080767f, 0.299822f,  0.235876f,
  0.254942f,  -0.017796f,
};

static const float av1_tx_split_nn_weights_32x32_layer1[32] = {
  -0.090326f, -0.267553f, -0.026071f, 0.100912f,  0.279137f,  0.079064f,
  -0.074885f, 0.053804f,  0.736810f,  -0.031693f, -0.970514f, 0.174069f,
  0.095940f,  -0.065047f, 0.052911f,  0.176728f,  -0.058274f, 0.148364f,
  -0.162210f, 0.093875f,  -0.367663f, 0.020876f,  0.137280f,  -1.099116f,
  0.146854f,  0.075590f,  0.228534f,  0.141993f,  0.072143f,  0.101421f,
  -0.068547f, -0.154148f,
};

static const float av1_tx_split_nn_bias_32x32_layer1[1] = {
  0.316622f,
};

static const NN_CONFIG av1_tx_split_nnconfig_32x32 = {
  12,  // num_inputs
  1,   // num_outputs
  1,   // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_32x32_layer0,
      av1_tx_split_nn_weights_32x32_layer1,
  },
  {
      av1_tx_split_nn_bias_32x32_layer0,
      av1_tx_split_nn_bias_32x32_layer1,
  },
};
/******************************************************************************/

// Tx split model for 64x64 block.
static const float av1_tx_split_nn_weights_64x64_layer0[12 * 32] = {
  -0.006828f, 0.149944f,  -0.017614f, -0.044599f, -0.024517f, 0.507698f,
  0.001039f,  0.037164f,  0.015091f,  -0.306620f, -0.162047f, -0.369440f,
  0.396310f,  0.087121f,  0.208609f,  -0.083068f, 0.493774f,  0.217682f,
  0.377393f,  0.172879f,  0.397422f,  0.078919f,  0.741350f,  0.064169f,
  -0.099989f, -0.192983f, -0.278230f, -0.310048f, -0.439965f, -0.226698f,
  -0.436596f, -0.007551f, -0.396721f, 0.153570f,  -0.190838f, -0.071869f,
  0.048799f,  -0.301301f, -0.005015f, 0.500480f,  -0.030622f, -0.559095f,
  -0.032634f, -0.054160f, -0.056979f, -0.456545f, 0.306536f,  -0.411323f,
  -0.005366f, -0.069496f, 0.019990f,  0.327931f,  -0.002516f, 0.393190f,
  0.001759f,  0.035093f,  -0.030302f, -0.528984f, 0.174781f,  0.241462f,
  -0.415427f, -0.164502f, 0.143065f,  -0.122595f, 0.082049f,  -0.143346f,
  0.055642f,  -0.124701f, 0.004050f,  -0.216235f, -2.681730f, 0.101658f,
  0.381239f,  0.465936f,  0.331154f,  0.301708f,  -0.360171f, 0.054886f,
  -0.118658f, 0.287921f,  0.277859f,  0.203784f,  0.247809f,  0.656924f,
  -0.354628f, 0.315081f,  0.105108f,  -0.510179f, 0.059267f,  0.061386f,
  0.076423f,  0.347119f,  0.100134f,  0.028402f,  -0.118621f, -0.238689f,
  0.080141f,  -0.138863f, 0.009009f,  -0.100526f, -0.138875f, 0.066992f,
  0.005949f,  0.564336f,  0.046994f,  0.004655f,  0.366047f,  0.014695f,
  -0.146928f, -0.024665f, -0.440357f, -0.109395f, 0.527231f,  -0.020925f,
  -0.227236f, -0.068141f, 0.282009f,  0.040192f,  -0.267100f, 0.229228f,
  0.133861f,  0.338706f,  -0.030178f, -0.040919f, -0.026343f, -0.330338f,
  -0.066931f, -0.110580f, -0.072056f, 0.599457f,  -0.020738f, 0.169200f,
  0.836240f,  -0.157548f, 0.386273f,  0.002404f,  0.329410f,  -0.007020f,
  0.351705f,  -0.041259f, 0.388861f,  0.003899f,  0.582627f,  0.023572f,
  0.409912f,  -0.158472f, 0.536383f,  0.525093f,  0.604247f,  0.439159f,
  0.692832f,  0.046272f,  0.590367f,  -0.082166f, 0.262357f,  0.478671f,
  0.031935f,  0.042675f,  0.120002f,  0.398616f,  -0.078967f, 0.227986f,
  -0.044679f, 0.151061f,  -0.085564f, 0.220205f,  -0.265606f, -0.203623f,
  0.204719f,  -0.125922f, 0.038544f,  -0.269379f, 0.025866f,  0.109967f,
  0.019064f,  -0.237297f, -0.309746f, -0.329118f, -0.278368f, -0.063859f,
  0.278496f,  0.018620f,  0.209971f,  0.296250f,  0.142850f,  0.288689f,
  0.137084f,  0.130517f,  0.128171f,  -0.155396f, -0.008449f, -0.099845f,
  0.173455f,  -0.059909f, -0.147318f, 0.102851f,  -0.251389f, -0.001448f,
  0.103907f,  0.297273f,  -0.027846f, 0.028260f,  -0.382601f, 0.346695f,
  -0.601641f, 0.162366f,  -0.477495f, -0.042731f, -0.387871f, -0.051791f,
  -0.401498f, -0.048446f, -0.456270f, -0.062287f, 0.493919f,  0.003008f,
  0.099917f,  -0.358525f, -0.094903f, -0.022811f, -0.062259f, 0.019455f,
  -0.050644f, 0.020041f,  -0.132912f, -0.061578f, -3.083691f, -0.014961f,
  -0.129115f, -0.710559f, 0.157213f,  -0.844037f, -0.121991f, -0.943386f,
  -0.231269f, -0.003462f, 0.331478f,  -0.132703f, -1.285993f, -0.120957f,
  -0.373755f, -0.322609f, 0.309059f,  -0.131523f, -0.118334f, -0.063805f,
  -0.104251f, 0.012166f,  -0.094699f, -0.283753f, 0.128168f,  -0.526929f,
  -0.050331f, 0.186153f,  0.005913f,  -0.221236f, 0.036363f,  0.160909f,
  -0.001342f, -0.382749f, 0.037820f,  0.281689f,  -0.024275f, 0.028854f,
  0.318291f,  0.318526f,  0.035778f,  0.034031f,  0.189663f,  -0.293367f,
  0.082022f,  0.127923f,  0.078866f,  -0.081361f, -0.268117f, 0.246675f,
  0.248605f,  -0.215479f, -0.073084f, 0.496140f,  -0.067327f, 0.396237f,
  -0.120739f, 0.033752f,  -0.044120f, -0.218941f, -0.028078f, 0.195132f,
  -0.040400f, 0.281604f,  -0.100471f, 0.415207f,  -0.258503f, -0.429749f,
  0.150569f,  -0.010859f, 0.136448f,  0.026589f,  0.148466f,  0.110764f,
  0.380967f,  0.009177f,  0.103075f,  0.116417f,  0.226273f,  -0.327746f,
  0.169346f,  0.284553f,  -0.094986f, 0.312745f,  -0.147840f, 0.025062f,
  -0.494482f, 0.112388f,  -0.213962f, 0.107050f,  -0.433371f, -0.096276f,
  -0.244835f, -0.003518f, -0.459148f, -0.145080f, 0.017150f,  0.042846f,
  -0.237479f, 0.104746f,  0.158677f,  0.358937f,  0.099921f,  0.277109f,
  0.012410f,  -0.062897f, 0.116130f,  0.255309f,  0.341628f,  0.145002f,
  -0.429344f, -0.016433f, -0.068985f, 0.285194f,  -0.286719f, -0.018298f,
  -0.179369f, -0.194655f, -0.165380f, 0.026071f,  -0.428268f, -0.379929f,
  -0.727543f, 0.179610f,  -0.963979f, -0.042026f, -0.616202f, 0.133401f,
  -0.784966f, 0.061205f,  -0.713357f, 0.129795f,  0.120512f,  -0.339545f,
  0.353557f,  0.114906f,  -0.329813f, -0.209987f, 0.085410f,  0.214313f,
  -0.122082f, 0.335770f,  -0.020937f, 0.202456f,  0.289023f,  -0.421186f,
  0.337905f,  0.407663f,  0.132771f,  0.071734f,  0.213914f,  0.128595f,
  0.302659f,  -0.209501f, 0.217756f,  0.253079f,  -0.089505f, -0.205614f,
};

static const float av1_tx_split_nn_bias_64x64_layer0[32] = {
  0.296914f,  -1.826816f, 0.346130f,  0.969520f,  -0.528154f, 1.175862f,
  -0.075985f, -0.097323f, -0.233059f, 0.004846f,  0.401279f,  -2.272435f,
  0.086257f,  0.414162f,  -0.194786f, -0.233887f, -0.113215f, -2.453546f,
  0.861214f,  0.298361f,  0.267397f,  -0.158557f, -0.119911f, -0.098134f,
  -0.339263f, 0.385871f,  -0.678123f, 0.263218f,  0.251611f,  -1.155773f,
  -0.365437f, 0.229255f,
};

static const float av1_tx_split_nn_weights_64x64_layer1[32] = {
  0.502104f,  -0.708023f, 0.419648f,  1.583418f,  0.419355f,  -1.462981f,
  -0.439623f, 0.405691f,  0.823257f,  0.061654f,  0.750875f,  0.775031f,
  -0.387909f, 0.447385f,  0.284690f,  0.353262f,  -0.224347f, 0.832864f,
  -1.708491f, -1.042447f, -0.272829f, 0.540640f,  0.310509f,  0.723745f,
  0.245592f,  -0.218417f, -0.597987f, -0.362301f, 0.702217f,  -0.692614f,
  0.207812f,  0.513560f,
};

static const float av1_tx_split_nn_bias_64x64_layer1[1] = { -0.2307045f };

static const NN_CONFIG av1_tx_split_nnconfig_64x64 = {
  12,  // num_inputs
  1,   // num_outputs
  1,   // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_64x64_layer0,
      av1_tx_split_nn_weights_64x64_layer1,
  },
  {
      av1_tx_split_nn_bias_64x64_layer0,
      av1_tx_split_nn_bias_64x64_layer1,
  },
};
/******************************************************************************/

// Tx split model for 4x16 block.
static const float av1_tx_split_nn_weights_4x16_layer0[8 * 16] = {
  -1.344184f, -1.454625f, -0.703110f, -0.140570f, -0.841536f, -0.068131f,
  -2.128968f, -0.655518f, 0.432180f,  0.879752f,  -0.222211f, 0.061615f,
  -0.230969f, 0.569496f,  1.424188f,  0.598063f,  -0.436005f, -0.737606f,
  -0.137875f, -0.085730f, -0.076512f, -0.583101f, -0.937377f, -0.203556f,
  -0.215797f, -0.015361f, -0.124098f, -0.411917f, 0.340441f,  -0.331752f,
  -0.472607f, -0.097714f, -0.930572f, -1.354713f, -0.550724f, 0.176212f,
  -0.636060f, 0.183271f,  -0.610212f, 0.345895f,  -1.100906f, -1.605713f,
  0.111888f,  -0.140937f, 0.063013f,  -0.013315f, -0.273472f, -0.255870f,
  1.200328f,  0.274002f,  1.005776f,  0.322392f,  1.222373f,  0.158227f,
  0.408810f,  0.145022f,  0.139842f,  -1.249412f, 0.286672f,  -0.635699f,
  0.312562f,  -0.495606f, -1.117034f, -0.085107f, -0.097484f, -0.341521f,
  -0.132199f, -0.863055f, 0.217579f,  -1.161425f, -0.302087f, -1.357271f,
  -0.520724f, -1.211069f, -1.048729f, -0.333087f, -1.171527f, -0.280824f,
  -2.057684f, -0.228755f, 0.606278f,  0.101198f,  -0.314847f, -1.303255f,
  -0.294964f, 1.301923f,  0.041712f,  0.077593f,  -1.152746f, 0.495315f,
  -0.751566f, 0.230249f,  -0.840661f, 0.100731f,  1.346269f,  0.649898f,
  -1.432258f, -0.456710f, -1.018123f, -0.348559f, -1.225226f, -0.170717f,
  -0.354072f, 0.068292f,  -0.234168f, 0.277503f,  0.179134f,  0.907420f,
  0.354626f,  -0.627210f, 0.905779f,  0.512612f,  0.161190f,  -0.843177f,
  0.014953f,  -0.354983f, 0.011116f,  -0.429598f, -1.017138f, -0.211432f,
  0.941840f,  -0.281747f, 0.957776f,  -0.541914f, 1.041880f,  -0.433580f,
  -1.416451f, -0.166467f,
};

static const float av1_tx_split_nn_bias_4x16_layer0[16] = {
  3.086118f,  -3.235095f, 4.830956f,  -0.165706f, 0.955031f,  4.055783f,
  -0.311489f, 4.660205f,  -0.576277f, -0.248111f, -0.790519f, -1.686412f,
  -1.191704f, -3.800073f, 4.121552f,  -1.399397f,
};

static const float av1_tx_split_nn_weights_4x16_layer1[16] = {
  -0.758677f, 0.388776f,  0.439906f,  0.011390f, -0.084319f, -0.667969f,
  -0.467316f, -0.875491f, -0.160668f, 0.805292f, 0.114393f,  -0.549682f,
  0.462109f,  0.343315f,  1.092593f,  0.483152f,
};

static const float av1_tx_split_nn_bias_4x16_layer1[1] = {
  0.8205083f,
};

static const NN_CONFIG av1_tx_split_nnconfig_4x16 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_4x16_layer0,
      av1_tx_split_nn_weights_4x16_layer1,
  },
  {
      av1_tx_split_nn_bias_4x16_layer0,
      av1_tx_split_nn_bias_4x16_layer1,
  },
};
/******************************************************************************/

// Tx split model for 16x32 block.
static const float av1_tx_split_nn_weights_16x32_layer0[8 * 32] = {
  0.180713f,  0.033211f,  0.607561f,  0.138642f,  0.637204f,  -0.000940f,
  0.012630f,  0.358109f,  0.022238f,  0.190418f,  0.079088f,  0.065925f,
  0.038242f,  0.162380f,  -0.122728f, 0.379382f,  -0.303283f, -0.327550f,
  0.029120f,  -0.284553f, 0.269588f,  -0.309805f, -0.241036f, -0.161103f,
  -0.304887f, 0.239843f,  -0.149146f, 0.311234f,  -0.073640f, -0.132718f,
  0.178901f,  0.474712f,  0.020280f,  0.063685f,  -0.609170f, -0.013658f,
  -0.338074f, 0.250429f,  0.082978f,  -0.186315f, -0.788959f, 0.039859f,
  -0.426461f, -0.001524f, -0.447211f, 0.378102f,  0.315617f,  0.017428f,
  0.745494f,  -0.219024f, 0.512836f,  0.200522f,  0.680449f,  0.313686f,
  -0.412569f, -0.132927f, 0.631120f,  0.042735f,  0.336153f,  0.044772f,
  0.432606f,  0.175681f,  -0.634411f, -0.073509f, -0.040643f, -0.559260f,
  -0.104034f, -0.570495f, -0.247365f, 0.063256f,  -0.582021f, -0.492585f,
  -0.194955f, -0.207934f, -0.506627f, 0.021743f,  -0.416518f, 0.320876f,
  0.115889f,  0.149399f,  -0.229376f, 0.095505f,  0.115191f,  -0.471921f,
  0.113068f,  0.343684f,  -0.036831f, 0.021240f,  0.295112f,  0.031166f,
  0.448201f,  -0.132241f, 0.164032f,  0.355572f,  0.072154f,  0.017335f,
  -0.046113f, 0.178719f,  -0.026881f, -0.242590f, 0.055073f,  -0.012958f,
  0.077904f,  0.351356f,  0.107655f,  0.260568f,  -0.080052f, -0.197553f,
  0.085763f,  0.263416f,  -0.327741f, 0.158855f,  0.056899f,  -0.162121f,
  0.339518f,  -0.571204f, 0.264966f,  -0.252214f, -0.202560f, -0.134213f,
  -0.330188f, 0.009470f,  -0.468376f, -0.065240f, -0.307957f, 0.116479f,
  -0.222238f, -0.458716f, 0.186493f,  -0.391415f, 0.118649f,  -0.104653f,
  -0.259958f, -0.332081f, -0.403785f, -0.050147f, -0.573511f, 0.177117f,
  -0.598358f, 0.164947f,  -0.119694f, -0.058520f, 0.203829f,  -0.267404f,
  -0.048202f, -0.600006f, 0.181594f,  -0.731805f, 0.146417f,  -0.687148f,
  -1.210525f, -0.450101f, -0.620635f, 0.208825f,  -0.611357f, 0.112202f,
  -0.309468f, -0.323545f, 0.357770f,  0.308061f,  0.553199f,  0.049012f,
  0.530093f,  -0.208597f, 0.607882f,  -0.058120f, -0.527634f, 0.018136f,
  0.060753f,  0.118894f,  0.175649f,  0.014731f,  0.428318f,  -0.106465f,
  -0.119077f, 0.080179f,  0.524997f,  0.368286f,  0.528286f,  0.213659f,
  0.639286f,  0.195079f,  -0.049815f, -0.092008f, -0.302958f, 0.298149f,
  -0.173870f, -0.145205f, -0.233589f, -0.303368f, 0.141275f,  0.325622f,
  -0.115293f, 0.155188f,  0.047225f,  0.231050f,  -0.167447f, 0.349754f,
  0.295544f,  -0.319466f, 0.095144f,  0.174612f,  -0.194652f, 0.305915f,
  -0.239008f, -0.037453f, 0.280696f,  0.125850f,  0.749196f,  -0.101919f,
  0.791808f,  -0.236811f, 0.064157f,  0.032865f,  -0.225911f, 0.350384f,
  0.723183f,  -0.103992f, 0.483085f,  -0.123992f, 0.602138f,  0.023895f,
  -0.692601f, -0.118387f, 0.162527f,  0.145178f,  -0.184702f, -0.017753f,
  -0.159436f, 0.124105f,  -0.131067f, 0.310275f,  0.151499f,  0.138924f,
  0.537459f,  0.263212f,  0.615896f,  0.281255f,  0.021293f,  -0.473459f,
  0.210145f,  -0.056682f, 0.063658f,  0.377254f,  -0.314410f, -0.183487f,
  0.300384f,  0.328471f,  0.164694f,  -0.159272f, -0.160942f, -0.502861f,
  -0.129147f, 0.045916f,  -0.606865f, -0.101378f,
};

static const float av1_tx_split_nn_bias_16x32_layer0[32] = {
  0.051664f,  -0.212487f, -0.077596f, -0.818467f, 0.638475f,  -0.759937f,
  0.157198f,  0.989640f,  1.586035f,  0.431144f,  0.041605f,  0.543085f,
  0.498379f,  0.320504f,  0.134233f,  0.670979f,  -0.105562f, -1.574879f,
  1.261812f,  -0.287530f, -1.610592f, 0.730899f,  -0.894240f, -0.657790f,
  0.270806f,  -0.181708f, 0.298578f,  0.817240f,  -0.221508f, -0.201771f,
  -0.294389f, 1.456413f,
};

static const float av1_tx_split_nn_weights_16x32_layer1[32] = {
  1.208914f,  0.324728f,  0.383352f,  -0.874321f, 0.172565f,  -0.580927f,
  -0.432927f, 0.433698f,  -0.801935f, 0.672028f,  0.563493f,  0.260077f,
  -0.200557f, -0.121638f, 0.530735f,  -0.525196f, 0.281799f,  0.624204f,
  -0.662775f, -0.230887f, 0.980989f,  0.223437f,  -0.790591f, 0.600724f,
  -0.273445f, 0.427635f,  -0.501641f, -0.878390f, 0.234731f,  -0.172550f,
  0.418904f,  1.792187f,
};

static const float av1_tx_split_nn_bias_16x32_layer1[1] = {
  -0.29233751f,
};

static const NN_CONFIG av1_tx_split_nnconfig_16x32 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_16x32_layer0,
      av1_tx_split_nn_weights_16x32_layer1,
  },
  {
      av1_tx_split_nn_bias_16x32_layer0,
      av1_tx_split_nn_bias_16x32_layer1,
  },
};
/******************************************************************************/

// Tx split model for 32x64 block.
static const float av1_tx_split_nn_weights_32x64_layer0[8 * 32] = {
  0.031614f,  -0.110926f, 0.052418f,  -0.702506f, 0.045708f,  0.238329f,
  -0.021806f, -0.208128f, 0.509745f,  -0.293891f, 0.277788f,  0.113937f,
  0.741576f,  0.062848f,  0.351878f,  0.212532f,  0.385842f,  0.081517f,
  0.398502f,  -0.015156f, 0.242616f,  0.214619f,  -0.182678f, -0.170546f,
  0.110605f,  -0.236749f, -0.023831f, -0.285243f, 0.147156f,  -0.257639f,
  0.341355f,  -0.571641f, -0.721797f, 0.139588f,  -0.518494f, -0.206526f,
  -0.570560f, -0.184295f, 0.110271f,  0.210292f,  -0.109132f, -0.001080f,
  0.129251f,  -0.204230f, -0.396312f, -0.183024f, 0.421243f,  -0.013154f,
  0.222627f,  0.169826f,  0.226037f,  0.218153f,  -0.343528f, 0.274906f,
  -0.156632f, 0.250261f,  -0.484020f, 0.019909f,  -0.349575f, -0.286643f,
  -0.507396f, 0.202446f,  -0.154110f, -0.292644f, 0.122666f,  0.306963f,
  0.424895f,  0.005579f,  0.494094f,  -0.079551f, 0.473740f,  0.352414f,
  -0.356917f, 0.264331f,  -0.554487f, 0.119978f,  0.012291f,  -0.141641f,
  -0.254714f, -0.213723f, -0.116701f, -0.011267f, 0.190025f,  -0.118501f,
  0.305151f,  -0.316782f, -0.220801f, -0.308420f, -0.324285f, 0.421329f,
  -0.177066f, -0.055114f, 0.229698f,  -0.199523f, 0.054278f,  0.365020f,
  -0.060586f, -0.300618f, 0.157563f,  -0.064338f, -0.005711f, -0.176991f,
  -0.424502f, -0.111914f, 0.092608f,  0.126621f,  0.078547f,  0.148008f,
  0.024221f,  0.124599f,  0.001343f,  0.059402f,  0.453753f,  0.047102f,
  0.242544f,  0.055735f,  -0.067451f, -0.170061f, -0.170469f, -0.232173f,
  0.214908f,  0.248889f,  0.544348f,  -0.084566f, 0.402478f,  0.298031f,
  0.099038f,  -0.238019f, -0.475085f, -0.070042f, -0.754955f, -0.049095f,
  -0.783801f, -0.099857f, -0.582008f, -0.055194f, -0.103655f, 0.143689f,
  0.100219f,  0.293934f,  0.099271f,  -0.036320f, 0.356626f,  -0.261445f,
  0.879544f,  0.000878f,  0.532920f,  -0.093918f, 0.508867f,  -0.040215f,
  -0.789042f, -0.145380f, -0.090040f, -0.066636f, 0.015212f,  0.352989f,
  -0.058831f, -0.164588f, 0.039890f,  0.122861f,  0.222508f,  0.061217f,
  0.466487f,  0.022666f,  0.423777f,  -0.002200f, -0.656835f, -0.099760f,
  -0.520606f, 0.303204f,  -0.563620f, -0.160922f, -0.243203f, 0.313354f,
  -0.336516f, -0.206764f, -0.236040f, 0.325899f,  -0.418748f, 0.163205f,
  -0.476242f, -0.121928f, 0.139178f,  -0.157193f, -0.531766f, -0.180202f,
  -0.485254f, 0.187703f,  -0.440072f, 0.137854f,  0.029139f,  0.109530f,
  -0.078475f, -0.360618f, -0.334672f, -0.350890f, -0.403976f, 0.180336f,
  -0.304542f, 0.005123f,  0.413995f,  0.314639f,  0.342648f,  -0.293264f,
  0.358135f,  -0.180425f, -0.369530f, -0.048413f, 0.498366f,  0.121875f,
  0.270948f,  -0.187966f, 0.342503f,  0.174420f,  -0.352105f, 0.088080f,
  0.008277f,  0.020275f,  -0.002381f, 0.504389f,  -0.018832f, -0.366047f,
  -0.090947f, -0.168150f, 0.016184f,  -0.328914f, 0.089579f,  -0.017349f,
  0.005844f,  -0.005010f, -1.857514f, -0.282426f, 0.010177f,  -0.214727f,
  -0.182529f, 0.156943f,  -0.162032f, -0.472654f, 0.069432f,  0.016901f,
  -0.767905f, 0.137129f,  -0.411463f, 0.049056f,  -0.431657f, -0.037641f,
  0.785500f,  0.046225f,  0.195831f,  0.245204f,  0.368614f,  0.212261f,
  0.440626f,  -0.158048f, -0.461031f, -0.146280f,
};

static const float av1_tx_split_nn_bias_32x64_layer0[32] = {
  0.490777f,  -1.894238f, 0.621333f,  -0.076756f, 0.286298f, 0.286375f,
  -0.126431f, -0.350034f, -1.017572f, 0.620125f,  0.408128f, 0.238756f,
  -0.060728f, 0.210912f,  0.043124f,  0.445649f,  0.907025f, 0.360272f,
  1.083101f,  -0.068952f, 1.062348f,  0.396354f,  0.280075f, 0.501732f,
  0.328422f,  0.066241f,  0.474697f,  0.126313f,  0.741206f, 0.314796f,
  0.552712f,  0.299410f,
};

static const float av1_tx_split_nn_weights_32x64_layer1[32] = {
  1.033823f,  0.603439f,  0.304591f,  -0.279940f, -0.780909f, -0.132801f,
  0.154059f,  0.662014f,  -0.718368f, 0.198733f,  0.039766f,  -0.208516f,
  -0.104909f, -0.394209f, 0.081617f,  0.365041f,  -0.874960f, -0.063315f,
  -1.189897f, 0.337225f,  0.410893f,  0.307519f,  0.221323f,  0.233895f,
  0.469536f,  0.438557f,  0.280144f,  0.422423f,  -1.394513f, 0.781900f,
  0.352981f,  0.111265f,
};

static const float av1_tx_split_nn_bias_32x64_layer1[1] = {
  -0.18160765f,
};

static const NN_CONFIG av1_tx_split_nnconfig_32x64 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      32,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_32x64_layer0,
      av1_tx_split_nn_weights_32x64_layer1,
  },
  {
      av1_tx_split_nn_bias_32x64_layer0,
      av1_tx_split_nn_bias_32x64_layer1,
  },
};
/******************************************************************************/

// Tx split model for 8x32 block.
static const float av1_tx_split_nn_weights_8x32_layer0[8 * 24] = {
  -0.687846f, 0.121404f,  -0.372905f, 0.126770f,  -0.103298f, -0.101650f,
  -0.148490f, -0.271740f, 0.682915f,  -0.079765f, 0.634347f,  -0.151503f,
  0.287692f,  -0.079072f, -0.236948f, 0.065064f,  0.713383f,  0.397123f,
  0.553621f,  0.368529f,  0.767663f,  -0.046601f, -0.392402f, -0.294822f,
  -0.292325f, -0.010573f, -0.837945f, 0.050113f,  -0.811360f, 0.199162f,
  0.150832f,  0.011602f,  0.369694f,  -0.225876f, 0.234113f,  -0.269808f,
  0.303805f,  -0.190281f, -0.451136f, 0.209755f,  -0.308894f, 0.326956f,
  0.313591f,  0.089923f,  -0.095754f, 0.390981f,  0.467366f,  0.169670f,
  0.853322f,  0.054055f,  0.830319f,  -0.121918f, 0.262019f,  -0.093526f,
  0.385558f,  0.419174f,  0.040198f,  -0.347030f, -0.450492f, -0.106764f,
  0.487502f,  -0.204188f, 0.430374f,  -0.116388f, 0.236407f,  -0.157376f,
  0.732294f,  -0.651387f, 0.347446f,  0.342575f,  0.048406f,  0.187657f,
  0.434899f,  -0.447782f, 0.032728f,  -0.071168f, -0.255327f, 0.104174f,
  0.095689f,  -0.431743f, 0.725694f,  0.031797f,  0.523171f,  0.061801f,
  0.469804f,  -0.071068f, -0.059024f, -0.211937f, 0.392134f,  -0.321490f,
  0.366060f,  -0.427798f, 0.166771f,  0.299652f,  0.044660f,  0.205142f,
  0.039133f,  -0.051835f, -0.465475f, 0.216976f,  -0.341156f, 0.095358f,
  0.230807f,  0.201674f,  0.279266f,  -0.713534f, -0.091690f, -0.569708f,
  -0.119001f, 0.252160f,  -1.544578f, -0.284477f, 0.555348f,  0.226471f,
  0.347690f,  0.034365f,  0.770835f,  -0.241859f, -0.130241f, 0.292936f,
  0.396622f,  -0.417916f, 0.492224f,  0.125517f,  0.344824f,  0.232172f,
  -0.432106f, -0.278745f, 0.035069f,  -0.307247f, -0.120760f, 0.170950f,
  0.433601f,  0.044286f,  0.141463f,  -0.041382f, 0.529346f,  0.010868f,
  -0.323674f, 0.185205f,  0.623459f,  0.232842f,  -0.406693f, -0.142944f,
  0.222988f,  0.343634f,  0.065401f,  0.002621f,  0.805335f,  -0.426926f,
  0.279181f,  0.131364f,  0.192339f,  -0.402391f, 0.544120f,  -0.060618f,
  0.467780f,  0.165224f,  -0.373131f, 0.002427f,  0.688064f,  0.322317f,
  0.259713f,  0.130583f,  0.185032f,  -0.189111f, -0.067821f, 0.010875f,
  0.644724f,  -0.179291f, 0.463222f,  0.155230f,  0.721384f,  -0.046019f,
  0.438501f,  0.440027f,  -0.462090f, -0.002039f, -0.468026f, -0.008890f,
  -0.328530f, 0.370102f,  0.482531f,  0.043471f,  -0.469732f, -0.532663f,
  0.122081f,  -0.379659f, 0.037219f,  -0.519913f, -0.128975f, -0.404365f,
};

static const float av1_tx_split_nn_bias_8x32_layer0[24] = {
  -1.198965f, 0.395204f,  -0.408627f, -0.021654f, -0.658355f, 0.154525f,
  -0.288354f, 1.207574f,  0.411608f,  0.964678f,  -1.176893f, 1.059006f,
  -0.472969f, 2.087975f,  1.065536f,  0.595569f,  0.197907f,  -0.349938f,
  1.013651f,  -0.931093f, -0.973595f, -0.459094f, -1.253062f, 1.624782f,
};

static const float av1_tx_split_nn_weights_8x32_layer1[24] = {
  0.815787f,  -0.393465f, -0.483427f, -0.565592f, 0.493494f,  0.430229f,
  -0.507073f, -0.251379f, -0.353418f, -0.495445f, 0.820029f,  0.649146f,
  -0.487383f, 1.844503f,  0.480324f,  -0.982705f, -0.501446f, -0.220584f,
  0.334299f,  0.802238f,  0.805838f,  -0.487848f, 0.300772f,  -1.232857f,
};

static const float av1_tx_split_nn_bias_8x32_layer1[1] = {
  0.13435879f,
};

static const NN_CONFIG av1_tx_split_nnconfig_8x32 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      24,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_8x32_layer0,
      av1_tx_split_nn_weights_8x32_layer1,
  },
  {
      av1_tx_split_nn_bias_8x32_layer0,
      av1_tx_split_nn_bias_8x32_layer1,
  },
};
/******************************************************************************/

// Tx split model for 16x32 block.
static const float av1_tx_split_nn_weights_16x64_layer0[8 * 16] = {
  -0.378223f, -0.124216f, -0.514089f, -0.110117f, -0.585801f, -0.094838f,
  -0.455385f, -0.220254f, -0.504568f, -0.082351f, -0.476420f, -0.253993f,
  -0.454709f, -0.059461f, 0.210313f,  -0.155683f, 0.192968f,  -0.127804f,
  0.471996f,  0.253377f,  0.472625f,  0.485322f,  0.150560f,  0.164868f,
  -0.475587f, 0.447559f,  -0.455759f, -0.306665f, -0.194866f, -0.283716f,
  -0.243897f, 0.293020f,  -0.308298f, -0.191904f, -0.468568f, 0.014053f,
  -0.618848f, 0.096273f,  -0.444586f, 0.347750f,  -0.280643f, -0.062872f,
  0.118661f,  0.540099f,  0.104141f,  -0.279300f, -0.098721f, -0.173427f,
  -0.984558f, -0.424559f, -0.411928f, -0.120875f, -0.488999f, -0.050716f,
  -0.523103f, 0.093620f,  -0.930396f, -0.431997f, -1.163297f, 0.190384f,
  -0.422581f, -0.005354f, 0.450552f,  0.369210f,  0.562484f,  0.679922f,
  0.282099f,  -0.039075f, 0.404196f,  0.006371f,  0.069679f,  -0.196160f,
  -0.213675f, 0.275187f,  -0.104235f, -0.193090f, 0.003116f,  -0.252454f,
  -0.094591f, 0.210439f,  -0.137070f, 0.145043f,  0.024558f,  0.121718f,
  0.010138f,  0.301651f,  -0.377990f, 0.444414f,  0.001845f,  -0.095334f,
  0.550259f,  0.087603f,  0.792492f,  -0.044584f, 0.641706f,  -0.328458f,
  -0.447791f, 0.135376f,  0.356385f,  0.135748f,  0.310370f,  0.293757f,
  -0.062000f, -0.056368f, 0.343930f,  0.312039f,  0.370763f,  0.452381f,
  -0.023630f, -0.185909f, 0.422277f,  -0.006306f, 0.045166f,  0.423359f,
  -0.157735f, -0.084901f, 0.219527f,  -0.209510f, 0.575057f,  0.249276f,
  0.069267f,  0.233898f,  -0.229392f, 0.117197f,  -0.038551f, 0.293976f,
  0.101996f,  0.120878f,
};

static const float av1_tx_split_nn_bias_16x64_layer0[16] = {
  1.036995f,  0.160249f,  0.100264f,  0.694881f,  0.694677f,  0.128379f,
  -0.843405f, -0.405515f, 0.104139f,  0.182980f,  -0.025472f, 0.901067f,
  -0.299866f, -0.103079f, -0.190352f, -0.048121f,
};

static const float av1_tx_split_nn_weights_16x64_layer1[16] = {
  -1.778868f, 0.174690f,  0.211991f, 0.712138f,  0.589352f,  0.466652f,
  1.029146f,  -0.490044f, 0.483015f, 0.600215f,  -0.577776f, -0.755546f,
  0.348337f,  -0.205082f, 0.347129f, -0.322277f,
};

static const float av1_tx_split_nn_bias_16x64_layer1[1] = {
  0.04230947f,
};

static const NN_CONFIG av1_tx_split_nnconfig_16x64 = {
  8,  // num_inputs
  1,  // num_outputs
  1,  // num_hidden_layers
  {
      16,
  },  // num_hidden_nodes
  {
      av1_tx_split_nn_weights_16x64_layer0,
      av1_tx_split_nn_weights_16x64_layer1,
  },
  {
      av1_tx_split_nn_bias_16x64_layer0,
      av1_tx_split_nn_bias_16x64_layer1,
  },
};
/******************************************************************************/

// Map block size to its corresponding neural net model for tx split prediction.
static const NN_CONFIG *av1_tx_split_nnconfig_map[TX_SIZES_ALL] = {
  NULL,                          // TX_4X4,
  &av1_tx_split_nnconfig_8x8,    // TX_8X8,
  &av1_tx_split_nnconfig_16x16,  // TX_16X16,
  &av1_tx_split_nnconfig_32x32,  // TX_32X32,
  &av1_tx_split_nnconfig_64x64,  // TX_64X64,
  &av1_tx_split_nnconfig_4x8,    // TX_4X8,
  &av1_tx_split_nnconfig_4x8,    // TX_8X4,
  &av1_tx_split_nnconfig_8x16,   // TX_8X16,
  &av1_tx_split_nnconfig_8x16,   // TX_16X8,
  &av1_tx_split_nnconfig_16x32,  // TX_16X32,
  &av1_tx_split_nnconfig_16x32,  // TX_32X16,
  &av1_tx_split_nnconfig_32x64,  // TX_32X64,
  &av1_tx_split_nnconfig_32x64,  // TX_64X32,
  &av1_tx_split_nnconfig_4x16,   // TX_4X16,
  &av1_tx_split_nnconfig_4x16,   // TX_16X4,
  &av1_tx_split_nnconfig_8x32,   // TX_8X32,
  &av1_tx_split_nnconfig_8x32,   // TX_32X8,
  &av1_tx_split_nnconfig_16x64,  // TX_16X64,
  &av1_tx_split_nnconfig_16x64,  // TX_64X16,
};

#ifdef __cplusplus
}  // extern "C"
#endif

#endif  // AV1_ENCODER_TX_PRUNE_MODEL_WEIGHTS_H_
