/*
 *  Copyright (C) 2021-2022  The Exult Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "verify.h"

#include "exult_constants.h"
#include "fnames.h"
#include "databuf.h"
#include "utils.h"
#include "sha1/sha1.h"
#include "modmgr.h"
#include "hash_utils.h"

#include <cstring>
#include <iostream>
#include <map>
#include <vector>

using File2HashMap = std::map<const char *, const char *>;
extern const File2HashMap blackgate_files;
extern const File2HashMap portenoire_files;
extern const File2HashMap puertanegra_files;
extern const File2HashMap schwarzepforte_files;
extern const File2HashMap forgeofvirtue_files;
extern const File2HashMap serpentbeta_files;
extern const File2HashMap serpentisle_files;
extern const File2HashMap islaserpiente_files;
extern const File2HashMap silverseed_files;

int verify_files(BaseGameInfo *game) {
	const File2HashMap empty_map;
	const auto get_filelist = [&empty_map, game]() -> const File2HashMap& {
		switch (game->get_game_type()) {
		case BLACK_GATE:
			if (game->have_expansion()) {
				return forgeofvirtue_files;
			}
			switch (game->get_game_language()) {
				case ENGLISH:
					return blackgate_files;
				case FRENCH:
					return portenoire_files;
				case GERMAN:
					return schwarzepforte_files;
				case SPANISH:
					return puertanegra_files;
			};
			break;
		case SERPENT_ISLE:
			if (game->have_expansion()) {
				return silverseed_files;
			}
			if (game->is_si_beta()) {
				return serpentbeta_files;
			}
			switch (game->get_game_language()) {
				case ENGLISH:
					return blackgate_files;
				case FRENCH:
				case GERMAN:
					// These do not exist
					break;
				case SPANISH:
					return islaserpiente_files;
			};
			break;
		case EXULT_DEVEL_GAME:
		case NONE:
		case EXULT_MENU_GAME:
			// Nothing to do
			break;
		};
		return empty_map;
	};
	const auto& files = get_filelist();
	if (files.empty()) {
		return 1;
	}
	std::cerr << std::endl
	          << std::endl
	          << "========================================" << std::endl
	          << "Starting file verification" << std::endl;
	int errors = 0;
	for (const auto& file_hash : files) {
		const auto& file = file_hash.first;
		const auto& hash = file_hash.second;
		IFileDataSource ds(file);
		if (!ds.good()) {
			errors++;
			continue;
		}
		const auto length = ds.getSize();
		const auto data = ds.readN(length);
		const auto hash_str = sha1::toHexString(sha1::calc(data.get(), length));
		if (strcmp(hash, hash_str.data()) != 0) {
			errors++;
			std::cerr << "Hash mismatch for file '" << file << "':" << std::endl
			          << "\tExpected hash: " << hash << std::endl
			          << "\tActual hash:   " << hash_str.data() << std::endl;
		}
	}
	std::cerr << "========================================" << std::endl
	          << "File verification finshed" << std::endl;
	if (errors == 0) {
		std::cerr << "All files match, your data files are good" << std::endl;
	} else {
		const std::string files{errors > 1 ? "files" : "file"};
		const std::string hashes{errors > 1 ? "hashes" : "hash"};
		const std::string these{errors > 1 ? "these" : "this"};
		const std::string they_are{errors > 1 ? "they are" : "it is"};
		std::cerr << errors << ' ' << files << " did not match the expected "
		          << hashes << "; see the affected " << files << " above." << std::endl
		          << "If you did not modify " << these << ' ' << files
		          << ", " << they_are << " probably corrupt." << std::endl;
	}
	return errors;
}


const File2HashMap blackgate_files{
		{MAINMUS_AD, "0ec151bd93e07ea36715d3bea56ea996bf9f7720"},
		{AMMO, "1c9313c1cf105a43933a29c6b7388872a7d8b21d"},
		{ARMOR, "32b00f5c8da9ccbeb503e9db08b6d06423282f07"},
		{ENDGAME, "5fde1edc8eb90b436146bbdad5c96d046e371f31"},
		{ENDSCORE_XMI, "5b364cd61b2c431b4f8ddd835652fb7447d7e5db"},
		{ENDSHAPE_FLX, "831899ce7b270ac27d297147181e98c5a2643bcb"},
		{EQUIP, "006c7bad2899fde897d0a09488574b624a28b90b"},
		{FACES_VGA, "6752ca0d454adf45aa3dab9d061e538afec99e48"},
		{FONTS_VGA, "848bd7f677c4e4ac496482e4c1c24273d9d607e6"},
		{GUMPS_VGA, "372ebae26a85aefca04352cec51b492b0f361968"},
		{INITGAME, "45f2deef795f2c9785f1ad509f9842a806ef93c7"},
		{INTROMUS_AD, "7966b060c395b32aea00bc421167e27f74cf5db8"},
		{INTROPAL_DAT, "e4234ba7b53d83c882f6e1662a1a654d9da90547"},
		{INTROMUS, "9c52d65e43fd96db49ce20986833f161f59b945a"},
		{INTROSND, "2191c1ddb93a6a78f983fc77a2cf87cc368647e4"},
		{MAINSHP_FLX, "4d7178806a2bd817467cb27a416304439930c302"},
		{MONSTERS, "d57a80ed8112ab55f3a5f5c2797bdf08d7f7f56b"},
		{MAINMUS, "2a48db4b099c950a6dd68a247785320c9df4d1c5"},
		{OCCLUDE, "506a97cdfcef2ee062b366275651d14c869cdfbe"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{POINTERS, "29d19ef760a3832721c41432cdf364d457c6df22"},
		{READY, "2b5b95c0ebe0ddf58a6196abfc89fc1ee39ed50c"},
		{SCHEDULE_DAT, "abf7b1d577039556bb0b5b94a8f26d88bc860485"},
		{SHAPES_VGA, "7eeffc7f3e3bd1d1d59f7c3e3b36774b3de1aab6"},
		{SHPDIMS, "00fd33bd081c6fe6b28be4e44ff3cc636bc50099"},
		{SPRITES_VGA, "716d601f8d26dedbb601f55f119f2f3f4702a91c"},
		{TEXT_FLX, "d8c8d8f5a5812413b9a4520a4d3d2ca81e4253ff"},
		{TFA, "8519d9744a75219691b2eb5e7aed14f15618d27d"},
		{U7CHUNKS, "3327be23e8b241a0cc113d68b3989537ec63889d"},
		{INTRO_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{U7MAP, "960b7834f455e918c4088076c2629eb50cfb827b"},
		{U7SPEECH, "7af1fbda6e0fbdbc07ed8a851de16aa99fa4bb4e"},
		{U7VOICE_FLX, "cc276cc23dd3c900ef3e8af26d6384df3dbee14c"},
		{USECODE, "570aa419ee8ccd76343162b59ec71fcbea72e1a7"},
		{WEAPONS, "cea42a72a7cf46703f1168f9d49c912aa8d157e6"},
		{WGTVOL, "554d1788918468507b0bb4e4fa19087cd99d3610"},
		{WIHH, "2546cfe8e7d227d170f82b82db51c422ec04f88f"},
		{XFORMTBL, "c345d416ac6164c84e3276b85d6ee5a548c6125b"},
		{XMIDI_AD, "a1d5af1540a1ac23f259899bafd2b9013e9f449b"},
		{XMIDI_MT, "f4b0e6a1b6a5af50f98baa130dae36f448df3acd"},
		{U7IFIX "00", "e6e87969aaeeee0a5266ae4bd6d2d54b09563f05"},
		{U7IFIX "01", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "02", "3ea16eff9480c4084ad85b22ff0daf987c40cff3"},
		{U7IFIX "03", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "04", "97aa3e9c3e46e67c1c0796633c95dd5f3a0d1e04"},
		{U7IFIX "05", "81da20dedc86f3a7a9cc99805d2c44eff391b444"},
		{U7IFIX "06", "34b68f210182d48934677b9c2cead690d31a1f08"},
		{U7IFIX "07", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "08", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "09", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "0a", "c40332bf473ab6cf367c560952656fcc4666f296"},
		{U7IFIX "0b", "e7a231eecd9eaee03b687379294a0524b952e67c"},
		{U7IFIX "0c", "a202efdeacb9f126fecc3d53c8097c1353fe8c22"},
		{U7IFIX "0d", "e9ded9c7e354992b9a3236f5fe8bc022ac9a9eef"},
		{U7IFIX "0e", "8ea1fd177a5a27ec66273c7a7316866f01b45052"},
		{U7IFIX "0f", "3b9013c7e83e0706ec3586093994dc2c799591db"},
		{U7IFIX "10", "55a3ca7638f5f4b10cf221ae7fc8724bbf46f881"},
		{U7IFIX "11", "b9e39ea98e3143f3155460cb12ebb983c38c9641"},
		{U7IFIX "12", "895c662b900a7300f1572b2d478300ed57269ac6"},
		{U7IFIX "13", "52990250a518369d50a4276865cb6ed3359795b1"},
		{U7IFIX "14", "497f5e37bc07bfe8b840232642ecec4885be8b38"},
		{U7IFIX "15", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "16", "f1379823d36da37913b04a05a777bc9e78b9671b"},
		{U7IFIX "17", "c8ca5993a3d699fd188140ee903c7baad8a4d652"},
		{U7IFIX "18", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "19", "688deb0662e9cd0cbb1429fb1cd0dcaa0f920b87"},
		{U7IFIX "1a", "b7113f16ad52db755677e61276d5b0dda79bb0e6"},
		{U7IFIX "1b", "5638cd5c36411b27611cd17d1bc92ede39ec38cd"},
		{U7IFIX "1c", "9cda878d0232cac9fe95e4859ceb051fcde5d4b4"},
		{U7IFIX "1d", "4ac5d40e515e241ddecaa8b28d0c7d5a07592dff"},
		{U7IFIX "1e", "837c725afc74eb0c9205931bae292f0eca38b39a"},
		{U7IFIX "1f", "abaf27da575c16b8ef988da62d31790f7f122a37"},
		{U7IFIX "20", "9819584c33e1142b24759cf0d3e660602b9390a4"},
		{U7IFIX "21", "d593ccbd3bd1aede43788e3fa05f9006196cab12"},
		{U7IFIX "22", "13b6797f433422f889528d71d09d442fb42f7e1e"},
		{U7IFIX "23", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "24", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "25", "92a66084450e66865e8b281523033912ecb6dbc2"},
		{U7IFIX "26", "082130deec589470762dfaf49e90635ce4014160"},
		{U7IFIX "27", "c0fe5e4d37e2232866bb7b59f941819bfdba4b5a"},
		{U7IFIX "28", "c83c5c4a3e7041af86ea1c278c5f080b6fc3ba30"},
		{U7IFIX "29", "15f10541c17de6bbae4dcb994c5273e1c3216d19"},
		{U7IFIX "2a", "c07e6802eac421c7dcf1a5bce105b1425bfe168c"},
		{U7IFIX "2b", "f064291feddf9022b4d423dcd25d9f9feee408ba"},
		{U7IFIX "2c", "cae675cf5fe77dafeb9261c3f447a2320a16e98f"},
		{U7IFIX "2d", "f24a42b459ad4a44eff39c22e5699d6dbbb153e7"},
		{U7IFIX "2e", "8dc8a8efbbbada1bb9952ddd23e6865d11b396d2"},
		{U7IFIX "2f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "30", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "31", "ddb020787ba59f6784ad0a081d7c6269b2f4011e"},
		{U7IFIX "32", "b22bff67502aa1bb5ada25bd6b6f7ba0393bfdec"},
		{U7IFIX "33", "79a3feba90f83ee49fdb6ca3a4927bd4351fa918"},
		{U7IFIX "34", "47e17d5ed8d6a823b8c373de9b83870c60884e5f"},
		{U7IFIX "35", "b664c3a4823915fca4b59c7731706ae145faed14"},
		{U7IFIX "36", "9c1682adda84946ad338df5980a0467479c3b65e"},
		{U7IFIX "37", "1bbf093d27780a34f0e765fb76d298970785581c"},
		{U7IFIX "38", "5521a5f0b469f070da0ee8f7067d76e839be47ca"},
		{U7IFIX "39", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "3a", "86424e2a75cd6b1a6a7a4d08cc0ce5c795d29531"},
		{U7IFIX "3b", "3322eeca60fffc9b5e7979f6dbd63e29785b4a13"},
		{U7IFIX "3c", "fa9a73e9c5cf6c3a1acc020002df03beb4721c49"},
		{U7IFIX "3d", "40ad586ac63607f4a887903c71a131b54a5b864b"},
		{U7IFIX "3e", "2b041288e41ed9f21054988f3e28e0770d9876c6"},
		{U7IFIX "3f", "027f1f75cd74a3487d3b530f3e6a0d927af00f7e"},
		{U7IFIX "40", "dd1c5c36b18ec462e4433f703b99d74a5d3f6af5"},
		{U7IFIX "41", "cb813d85e2eb989343592c9a031ec259485ddcd8"},
		{U7IFIX "42", "a3b1b09942f82648175fd5623af025569b233200"},
		{U7IFIX "43", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "44", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "45", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "46", "ab649f1a8d24ee8ea8dc5f1a929751166480b7b8"},
		{U7IFIX "47", "cbc8aa320e6bf910e6b678d00b7dcb14177bb634"},
		{U7IFIX "48", "eec082df80200f47f9d6f4adaceb556798901d21"},
		{U7IFIX "49", "4f368037159fce26eadbab1cf1d5b1160dd16d0b"},
		{U7IFIX "4a", "db5d6c7ad8fec3c3faf11c4021f249e1d01d6385"},
		{U7IFIX "4b", "bcd9dfad8b46e8abcb69e27abf9b64e8aac220c6"},
		{U7IFIX "4c", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "4d", "51df9c9941cb2630adab2b96e84be01094302775"},
		{U7IFIX "4e", "c19fed72d6c1760d45c92144118160d89de97bc5"},
		{U7IFIX "4f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "50", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "51", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "52", "8b49773bc0016d6d16e2c6bceefea35e91d7746f"},
		{U7IFIX "53", "a3616d5d6f4d40fff5b9046630fbc69f3249f3c1"},
		{U7IFIX "54", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "55", "9fa54ae682b2034809f0b1c0a2f458ba581c0b63"},
		{U7IFIX "56", "402455cc4756b0b6f0b4897901c2473ffb9e3692"},
		{U7IFIX "57", "9893e8b0a3e27ae398413e02e71dc278ed765a54"},
		{U7IFIX "58", "262d9a7508e56bf3c19c14ea9177c372da43638f"},
		{U7IFIX "59", "032afdce757b1ca66bd7107f058d01cc49832030"},
		{U7IFIX "5a", "0f252204f41861b08b6fb201e8bd750270e2b70d"},
		{U7IFIX "5b", "59387297607fddef7ee5605b885f87dc52eb011b"},
		{U7IFIX "5c", "d54f40caefaca9d0997a64f2277f8266f0673bae"},
		{U7IFIX "5d", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "5e", "7c9bc69a731df11a891d1764fc5244a6288e51a2"},
		{U7IFIX "5f", "e17ea9496ce1944f337db4563134f6b09170622f"},
		{U7IFIX "60", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "61", "b741f0befb470ccf1260e28e334447404ac825a8"},
		{U7IFIX "62", "c8a9d3908b0e5d18e62a9085a9498fdc44af9824"},
		{U7IFIX "63", "b98c2b741ff569df79f1575f79241f470f28ae76"},
		{U7IFIX "64", "dce1fa95e393a292c35d684d3b4ab13b570f843b"},
		{U7IFIX "65", "1c5bc75352c5ec3a68cef7d4c13593d98479710e"},
		{U7IFIX "66", "2880531459cd9e78fa440536ef1c9dd83121fd9e"},
		{U7IFIX "67", "2b22bd245302e6c18352f040194005445e2171be"},
		{U7IFIX "68", "8c4e1b7639e9123544d4316081ae399d7b29a1cc"},
		{U7IFIX "69", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "6a", "34b56453285014bc9d3357dd90b98d7860619667"},
		{U7IFIX "6b", "5c499e6df55ca88c56189e98a393c0b0df3ae77a"},
		{U7IFIX "6c", "1ac2d3e25be041b6e34e667546c86f9a3823032f"},
		{U7IFIX "6d", "39b3d1cd3f68cc79cd38f7379bcde7c80e1840d4"},
		{U7IFIX "6e", "c0387a805b24ffe67cc5048b3dd87022d7a90fa7"},
		{U7IFIX "6f", "8c5eae177455438234b7f21d27b215d48d52e67d"},
		{U7IFIX "70", "60cb9c43a18ab5df73a1f8b4313e5ee4ab0e0c10"},
		{U7IFIX "71", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "72", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "73", "54a5f96eb652fdce6124fd1047ca94191ec38cea"},
		{U7IFIX "74", "8136fabe27cad5717496eb5c958307920ebb5525"},
		{U7IFIX "75", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "76", "6dde9535938a5b4157d894915b5b22b7bdfb227c"},
		{U7IFIX "77", "fd50c150f08dce06549965fc38469927ca4c629e"},
		{U7IFIX "78", "04ac24bee791f5dfc01171a568d469fb5115bfaf"},
		{U7IFIX "79", "80de8eb6f1330b3c54b700a891923c50351f2e6b"},
		{U7IFIX "7a", "5fb6ce39a6ecbd59f88d490217c28ec546ab2260"},
		{U7IFIX "7b", "df30a2e94ccb6e68382b3d8f395f9097ea8fb0fb"},
		{U7IFIX "7c", "a2d4f122b2f71f8fc3de06a06859fc8862998cee"},
		{U7IFIX "7d", "b9458ba9e345ae9e2441d65bf09e750bfa054888"},
		{U7IFIX "7e", "d0286271c657f06cb36c202480991b6b8927e0f2"},
		{U7IFIX "7f", "5349a97972897092fad9ffce065bfdb0e9569e73"},
		{U7IFIX "80", "db8142e4eccdab10e2b1ea2adbc9e9ac83f416f8"},
		{U7IFIX "81", "e408895ac373d0627b76cb1eab0c8eb270bfecd6"},
		{U7IFIX "82", "2ec59c9639e79948b4830fd6a953e8811fdc7991"},
		{U7IFIX "83", "b7e399bd9419c36e854797cffcbea65246da54eb"},
		{U7IFIX "84", "d11d4da0e85d9463deed854ee2c3cf793be9a570"},
		{U7IFIX "85", "6964877d706be9017a232d5253c581dd9ed5191a"},
		{U7IFIX "86", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "87", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "88", "d97a018065022a1dbbbf620a7d2b7b232ea21a49"},
		{U7IFIX "89", "c943e73826fc325cea2c75713852af5a84b8f0ee"},
		{U7IFIX "8a", "f0fb50192d12c402237d9744a1f90db637962599"},
		{U7IFIX "8b", "e0d6e7f6d2467b7db5cab9fa3b315f1a160c354d"},
		{U7IFIX "8c", "fcaac53e029ad8350fb574924284ca7eb34fd8a4"},
		{U7IFIX "8d", "b8c9ffc019bb0729a0ec08be312d3a06f7eb6251"},
		{U7IFIX "8e", "629de2763fc14f8a93b5a207d099ee410d9172fa"},
		{U7IFIX "8f", "a60aaa1878c4f0128c9a609d5875c5838da3fb37"}};

const File2HashMap portenoire_files{
		{MAINMUS_AD, "0ec151bd93e07ea36715d3bea56ea996bf9f7720"},
		{AMMO, "1c9313c1cf105a43933a29c6b7388872a7d8b21d"},
		{ARMOR, "32b00f5c8da9ccbeb503e9db08b6d06423282f07"},
		{ENDGAME, "17a420a406b5ee67f4e2813b1ac699ffb6d990b3"},
		{ENDSCORE_XMI, "5b364cd61b2c431b4f8ddd835652fb7447d7e5db"},
		{ENDSHAPE_FLX, "376e32e677712f7ee03e12134f224e1919b4b41a"},
		{EQUIP, "006c7bad2899fde897d0a09488574b624a28b90b"},
		{FACES_VGA, "6752ca0d454adf45aa3dab9d061e538afec99e48"},
		{FONTS_VGA, "d94e6613255d7fc4104ab0ae81a17ccd97fbb1a1"},
		{GUMPS_VGA, "385b479402f97c7d41c51986b660881fa9c2a4bc"},
		{INITGAME, "45f2deef795f2c9785f1ad509f9842a806ef93c7"},
		{INTROMUS_AD, "7966b060c395b32aea00bc421167e27f74cf5db8"},
		{INTROPAL_DAT, "e4234ba7b53d83c882f6e1662a1a654d9da90547"},
		{INTROMUS, "9c52d65e43fd96db49ce20986833f161f59b945a"},
		{INTROSND, "7fbae1a1f3d0f7195454d3f347709e7608cc8595"},
		{MAINSHP_FLX, "3aef855ad56d5adce22f3a7bf27318980fa58c3c"},
		{MONSTERS, "d57a80ed8112ab55f3a5f5c2797bdf08d7f7f56b"},
		{MAINMUS, "2a48db4b099c950a6dd68a247785320c9df4d1c5"},
		{OCCLUDE, "506a97cdfcef2ee062b366275651d14c869cdfbe"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{POINTERS, "6bad04c09c1d021db01110b9758ae15f6a3202bd"},
		{READY, "2b5b95c0ebe0ddf58a6196abfc89fc1ee39ed50c"},
		{SCHEDULE_DAT, "abf7b1d577039556bb0b5b94a8f26d88bc860485"},
		{SHAPES_VGA, "7eeffc7f3e3bd1d1d59f7c3e3b36774b3de1aab6"},
		{SHPDIMS, "00fd33bd081c6fe6b28be4e44ff3cc636bc50099"},
		{SPRITES_VGA, "716d601f8d26dedbb601f55f119f2f3f4702a91c"},
		{TEXT_FLX, "51b33f2d0cde06cd2a4bb30822e48c2912feccb4"},
		{TFA, "8519d9744a75219691b2eb5e7aed14f15618d27d"},
		{U7CHUNKS, "3327be23e8b241a0cc113d68b3989537ec63889d"},
		{INTRO_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{U7MAP, "960b7834f455e918c4088076c2629eb50cfb827b"},
		{U7SPEECH, "83224fd6da0d4fb4419f9749c24bf7b6d82f6c0e"},
		{U7VOICE_FLX, "cc276cc23dd3c900ef3e8af26d6384df3dbee14c"},
		{USECODE, "08b0056520b5f130c6b4e1fcf1ab90ebf8e6cedb"},
		{WEAPONS, "cea42a72a7cf46703f1168f9d49c912aa8d157e6"},
		{WGTVOL, "554d1788918468507b0bb4e4fa19087cd99d3610"},
		{WIHH, "2546cfe8e7d227d170f82b82db51c422ec04f88f"},
		{XFORMTBL, "c345d416ac6164c84e3276b85d6ee5a548c6125b"},
		{XMIDI_AD, "a1d5af1540a1ac23f259899bafd2b9013e9f449b"},
		{XMIDI_MT, "f4b0e6a1b6a5af50f98baa130dae36f448df3acd"},
		{U7IFIX "00", "e6e87969aaeeee0a5266ae4bd6d2d54b09563f05"},
		{U7IFIX "01", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "02", "3ea16eff9480c4084ad85b22ff0daf987c40cff3"},
		{U7IFIX "03", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "04", "97aa3e9c3e46e67c1c0796633c95dd5f3a0d1e04"},
		{U7IFIX "05", "81da20dedc86f3a7a9cc99805d2c44eff391b444"},
		{U7IFIX "06", "34b68f210182d48934677b9c2cead690d31a1f08"},
		{U7IFIX "07", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "08", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "09", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "0a", "c40332bf473ab6cf367c560952656fcc4666f296"},
		{U7IFIX "0b", "e7a231eecd9eaee03b687379294a0524b952e67c"},
		{U7IFIX "0c", "a202efdeacb9f126fecc3d53c8097c1353fe8c22"},
		{U7IFIX "0d", "e9ded9c7e354992b9a3236f5fe8bc022ac9a9eef"},
		{U7IFIX "0e", "8ea1fd177a5a27ec66273c7a7316866f01b45052"},
		{U7IFIX "0f", "3b9013c7e83e0706ec3586093994dc2c799591db"},
		{U7IFIX "10", "55a3ca7638f5f4b10cf221ae7fc8724bbf46f881"},
		{U7IFIX "11", "b9e39ea98e3143f3155460cb12ebb983c38c9641"},
		{U7IFIX "12", "895c662b900a7300f1572b2d478300ed57269ac6"},
		{U7IFIX "13", "52990250a518369d50a4276865cb6ed3359795b1"},
		{U7IFIX "14", "497f5e37bc07bfe8b840232642ecec4885be8b38"},
		{U7IFIX "15", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "16", "f1379823d36da37913b04a05a777bc9e78b9671b"},
		{U7IFIX "17", "c8ca5993a3d699fd188140ee903c7baad8a4d652"},
		{U7IFIX "18", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "19", "688deb0662e9cd0cbb1429fb1cd0dcaa0f920b87"},
		{U7IFIX "1a", "b7113f16ad52db755677e61276d5b0dda79bb0e6"},
		{U7IFIX "1b", "5638cd5c36411b27611cd17d1bc92ede39ec38cd"},
		{U7IFIX "1c", "9cda878d0232cac9fe95e4859ceb051fcde5d4b4"},
		{U7IFIX "1d", "4ac5d40e515e241ddecaa8b28d0c7d5a07592dff"},
		{U7IFIX "1e", "837c725afc74eb0c9205931bae292f0eca38b39a"},
		{U7IFIX "1f", "abaf27da575c16b8ef988da62d31790f7f122a37"},
		{U7IFIX "20", "9819584c33e1142b24759cf0d3e660602b9390a4"},
		{U7IFIX "21", "d593ccbd3bd1aede43788e3fa05f9006196cab12"},
		{U7IFIX "22", "13b6797f433422f889528d71d09d442fb42f7e1e"},
		{U7IFIX "23", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "24", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "25", "92a66084450e66865e8b281523033912ecb6dbc2"},
		{U7IFIX "26", "082130deec589470762dfaf49e90635ce4014160"},
		{U7IFIX "27", "c0fe5e4d37e2232866bb7b59f941819bfdba4b5a"},
		{U7IFIX "28", "c83c5c4a3e7041af86ea1c278c5f080b6fc3ba30"},
		{U7IFIX "29", "15f10541c17de6bbae4dcb994c5273e1c3216d19"},
		{U7IFIX "2a", "c07e6802eac421c7dcf1a5bce105b1425bfe168c"},
		{U7IFIX "2b", "f064291feddf9022b4d423dcd25d9f9feee408ba"},
		{U7IFIX "2c", "cae675cf5fe77dafeb9261c3f447a2320a16e98f"},
		{U7IFIX "2d", "f24a42b459ad4a44eff39c22e5699d6dbbb153e7"},
		{U7IFIX "2e", "8dc8a8efbbbada1bb9952ddd23e6865d11b396d2"},
		{U7IFIX "2f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "30", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "31", "ddb020787ba59f6784ad0a081d7c6269b2f4011e"},
		{U7IFIX "32", "b22bff67502aa1bb5ada25bd6b6f7ba0393bfdec"},
		{U7IFIX "33", "79a3feba90f83ee49fdb6ca3a4927bd4351fa918"},
		{U7IFIX "34", "47e17d5ed8d6a823b8c373de9b83870c60884e5f"},
		{U7IFIX "35", "b664c3a4823915fca4b59c7731706ae145faed14"},
		{U7IFIX "36", "9c1682adda84946ad338df5980a0467479c3b65e"},
		{U7IFIX "37", "1bbf093d27780a34f0e765fb76d298970785581c"},
		{U7IFIX "38", "5521a5f0b469f070da0ee8f7067d76e839be47ca"},
		{U7IFIX "39", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "3a", "86424e2a75cd6b1a6a7a4d08cc0ce5c795d29531"},
		{U7IFIX "3b", "3322eeca60fffc9b5e7979f6dbd63e29785b4a13"},
		{U7IFIX "3c", "fa9a73e9c5cf6c3a1acc020002df03beb4721c49"},
		{U7IFIX "3d", "40ad586ac63607f4a887903c71a131b54a5b864b"},
		{U7IFIX "3e", "2b041288e41ed9f21054988f3e28e0770d9876c6"},
		{U7IFIX "3f", "027f1f75cd74a3487d3b530f3e6a0d927af00f7e"},
		{U7IFIX "40", "dd1c5c36b18ec462e4433f703b99d74a5d3f6af5"},
		{U7IFIX "41", "cb813d85e2eb989343592c9a031ec259485ddcd8"},
		{U7IFIX "42", "a3b1b09942f82648175fd5623af025569b233200"},
		{U7IFIX "43", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "44", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "45", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "46", "ab649f1a8d24ee8ea8dc5f1a929751166480b7b8"},
		{U7IFIX "47", "cbc8aa320e6bf910e6b678d00b7dcb14177bb634"},
		{U7IFIX "48", "eec082df80200f47f9d6f4adaceb556798901d21"},
		{U7IFIX "49", "4f368037159fce26eadbab1cf1d5b1160dd16d0b"},
		{U7IFIX "4a", "db5d6c7ad8fec3c3faf11c4021f249e1d01d6385"},
		{U7IFIX "4b", "bcd9dfad8b46e8abcb69e27abf9b64e8aac220c6"},
		{U7IFIX "4c", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "4d", "51df9c9941cb2630adab2b96e84be01094302775"},
		{U7IFIX "4e", "c19fed72d6c1760d45c92144118160d89de97bc5"},
		{U7IFIX "4f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "50", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "51", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "52", "8b49773bc0016d6d16e2c6bceefea35e91d7746f"},
		{U7IFIX "53", "a3616d5d6f4d40fff5b9046630fbc69f3249f3c1"},
		{U7IFIX "54", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "55", "9fa54ae682b2034809f0b1c0a2f458ba581c0b63"},
		{U7IFIX "56", "402455cc4756b0b6f0b4897901c2473ffb9e3692"},
		{U7IFIX "57", "9893e8b0a3e27ae398413e02e71dc278ed765a54"},
		{U7IFIX "58", "262d9a7508e56bf3c19c14ea9177c372da43638f"},
		{U7IFIX "59", "032afdce757b1ca66bd7107f058d01cc49832030"},
		{U7IFIX "5a", "0f252204f41861b08b6fb201e8bd750270e2b70d"},
		{U7IFIX "5b", "59387297607fddef7ee5605b885f87dc52eb011b"},
		{U7IFIX "5c", "d54f40caefaca9d0997a64f2277f8266f0673bae"},
		{U7IFIX "5d", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "5e", "7c9bc69a731df11a891d1764fc5244a6288e51a2"},
		{U7IFIX "5f", "e17ea9496ce1944f337db4563134f6b09170622f"},
		{U7IFIX "60", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "61", "b741f0befb470ccf1260e28e334447404ac825a8"},
		{U7IFIX "62", "c8a9d3908b0e5d18e62a9085a9498fdc44af9824"},
		{U7IFIX "63", "b98c2b741ff569df79f1575f79241f470f28ae76"},
		{U7IFIX "64", "dce1fa95e393a292c35d684d3b4ab13b570f843b"},
		{U7IFIX "65", "1c5bc75352c5ec3a68cef7d4c13593d98479710e"},
		{U7IFIX "66", "2880531459cd9e78fa440536ef1c9dd83121fd9e"},
		{U7IFIX "67", "2b22bd245302e6c18352f040194005445e2171be"},
		{U7IFIX "68", "8c4e1b7639e9123544d4316081ae399d7b29a1cc"},
		{U7IFIX "69", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "6a", "34b56453285014bc9d3357dd90b98d7860619667"},
		{U7IFIX "6b", "5c499e6df55ca88c56189e98a393c0b0df3ae77a"},
		{U7IFIX "6c", "1ac2d3e25be041b6e34e667546c86f9a3823032f"},
		{U7IFIX "6d", "39b3d1cd3f68cc79cd38f7379bcde7c80e1840d4"},
		{U7IFIX "6e", "c0387a805b24ffe67cc5048b3dd87022d7a90fa7"},
		{U7IFIX "6f", "8c5eae177455438234b7f21d27b215d48d52e67d"},
		{U7IFIX "70", "60cb9c43a18ab5df73a1f8b4313e5ee4ab0e0c10"},
		{U7IFIX "71", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "72", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "73", "54a5f96eb652fdce6124fd1047ca94191ec38cea"},
		{U7IFIX "74", "8136fabe27cad5717496eb5c958307920ebb5525"},
		{U7IFIX "75", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "76", "6dde9535938a5b4157d894915b5b22b7bdfb227c"},
		{U7IFIX "77", "fd50c150f08dce06549965fc38469927ca4c629e"},
		{U7IFIX "78", "04ac24bee791f5dfc01171a568d469fb5115bfaf"},
		{U7IFIX "79", "80de8eb6f1330b3c54b700a891923c50351f2e6b"},
		{U7IFIX "7a", "5fb6ce39a6ecbd59f88d490217c28ec546ab2260"},
		{U7IFIX "7b", "df30a2e94ccb6e68382b3d8f395f9097ea8fb0fb"},
		{U7IFIX "7c", "a2d4f122b2f71f8fc3de06a06859fc8862998cee"},
		{U7IFIX "7d", "b9458ba9e345ae9e2441d65bf09e750bfa054888"},
		{U7IFIX "7e", "d0286271c657f06cb36c202480991b6b8927e0f2"},
		{U7IFIX "7f", "5349a97972897092fad9ffce065bfdb0e9569e73"},
		{U7IFIX "80", "db8142e4eccdab10e2b1ea2adbc9e9ac83f416f8"},
		{U7IFIX "81", "e408895ac373d0627b76cb1eab0c8eb270bfecd6"},
		{U7IFIX "82", "2ec59c9639e79948b4830fd6a953e8811fdc7991"},
		{U7IFIX "83", "b7e399bd9419c36e854797cffcbea65246da54eb"},
		{U7IFIX "84", "d11d4da0e85d9463deed854ee2c3cf793be9a570"},
		{U7IFIX "85", "6964877d706be9017a232d5253c581dd9ed5191a"},
		{U7IFIX "86", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "87", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "88", "d97a018065022a1dbbbf620a7d2b7b232ea21a49"},
		{U7IFIX "89", "c943e73826fc325cea2c75713852af5a84b8f0ee"},
		{U7IFIX "8a", "f0fb50192d12c402237d9744a1f90db637962599"},
		{U7IFIX "8b", "e0d6e7f6d2467b7db5cab9fa3b315f1a160c354d"},
		{U7IFIX "8c", "fcaac53e029ad8350fb574924284ca7eb34fd8a4"},
		{U7IFIX "8d", "b8c9ffc019bb0729a0ec08be312d3a06f7eb6251"},
		{U7IFIX "8e", "629de2763fc14f8a93b5a207d099ee410d9172fa"},
		{U7IFIX "8f", "a60aaa1878c4f0128c9a609d5875c5838da3fb37"}};

const File2HashMap puertanegra_files{
		{MAINMUS_AD, "0ec151bd93e07ea36715d3bea56ea996bf9f7720"},
		{AMMO, "1c9313c1cf105a43933a29c6b7388872a7d8b21d"},
		{ARMOR, "32b00f5c8da9ccbeb503e9db08b6d06423282f07"},
		{ENDGAME, "c5003203b913ab9dcf16dd7d78e70b0a3b66482b"},
		{ENDSCORE_XMI, "5b364cd61b2c431b4f8ddd835652fb7447d7e5db"},
		{ENDSHAPE_FLX, "a0208c2e25896f85f69c8f858de72f60c5ee639d"},
		{EQUIP, "006c7bad2899fde897d0a09488574b624a28b90b"},
		{FACES_VGA, "6752ca0d454adf45aa3dab9d061e538afec99e48"},
		{FONTS_VGA, "848bd7f677c4e4ac496482e4c1c24273d9d607e6"},
		{GUMPS_VGA, "e08df5b6d594a3dcd98864f49f327b03a5863a2a"},
		{INITGAME, "45f2deef795f2c9785f1ad509f9842a806ef93c7"},
		{INTROMUS_AD, "7966b060c395b32aea00bc421167e27f74cf5db8"},
		{INTROPAL_DAT, "e4234ba7b53d83c882f6e1662a1a654d9da90547"},
		{INTROMUS, "9c52d65e43fd96db49ce20986833f161f59b945a"},
		{INTROSND, "12d90cb4e8c84421a9b668d470582d9e322851f4"},
		{MAINSHP_FLX, "959c48e582d8f03d2683378c5d0674f6c0b0635c"},
		{MONSTERS, "d57a80ed8112ab55f3a5f5c2797bdf08d7f7f56b"},
		{MAINMUS, "2a48db4b099c950a6dd68a247785320c9df4d1c5"},
		{OCCLUDE, "506a97cdfcef2ee062b366275651d14c869cdfbe"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{POINTERS, "ff8c6aa032973866a6775895b68473550e877c6a"},
		{READY, "2b5b95c0ebe0ddf58a6196abfc89fc1ee39ed50c"},
		{SCHEDULE_DAT, "abf7b1d577039556bb0b5b94a8f26d88bc860485"},
		{SHAPES_VGA, "7eeffc7f3e3bd1d1d59f7c3e3b36774b3de1aab6"},
		{SHPDIMS, "00fd33bd081c6fe6b28be4e44ff3cc636bc50099"},
		{SPRITES_VGA, "716d601f8d26dedbb601f55f119f2f3f4702a91c"},
		{TEXT_FLX, "91076b0da851d84e61b2ef3dac8a5317133f8394"},
		{TFA, "8519d9744a75219691b2eb5e7aed14f15618d27d"},
		{U7CHUNKS, "3327be23e8b241a0cc113d68b3989537ec63889d"},
		{INTRO_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{U7MAP, "960b7834f455e918c4088076c2629eb50cfb827b"},
		{U7SPEECH, "770d68ffb44206cef9e9a5d27a4ddb5bcb7c2aa0"},
		{U7VOICE_FLX, "cc276cc23dd3c900ef3e8af26d6384df3dbee14c"},
		{USECODE, "818a0705fd624ee2957c54bc83ff9c61e8ad74ca"},
		{WEAPONS, "cea42a72a7cf46703f1168f9d49c912aa8d157e6"},
		{WGTVOL, "554d1788918468507b0bb4e4fa19087cd99d3610"},
		{WIHH, "2546cfe8e7d227d170f82b82db51c422ec04f88f"},
		{XFORMTBL, "c345d416ac6164c84e3276b85d6ee5a548c6125b"},
		{XMIDI_AD, "a1d5af1540a1ac23f259899bafd2b9013e9f449b"},
		{XMIDI_MT, "f4b0e6a1b6a5af50f98baa130dae36f448df3acd"},
		{U7IFIX "00", "e6e87969aaeeee0a5266ae4bd6d2d54b09563f05"},
		{U7IFIX "01", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "02", "3ea16eff9480c4084ad85b22ff0daf987c40cff3"},
		{U7IFIX "03", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "04", "97aa3e9c3e46e67c1c0796633c95dd5f3a0d1e04"},
		{U7IFIX "05", "81da20dedc86f3a7a9cc99805d2c44eff391b444"},
		{U7IFIX "06", "34b68f210182d48934677b9c2cead690d31a1f08"},
		{U7IFIX "07", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "08", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "09", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "0a", "c40332bf473ab6cf367c560952656fcc4666f296"},
		{U7IFIX "0b", "e7a231eecd9eaee03b687379294a0524b952e67c"},
		{U7IFIX "0c", "a202efdeacb9f126fecc3d53c8097c1353fe8c22"},
		{U7IFIX "0d", "e9ded9c7e354992b9a3236f5fe8bc022ac9a9eef"},
		{U7IFIX "0e", "8ea1fd177a5a27ec66273c7a7316866f01b45052"},
		{U7IFIX "0f", "3b9013c7e83e0706ec3586093994dc2c799591db"},
		{U7IFIX "10", "55a3ca7638f5f4b10cf221ae7fc8724bbf46f881"},
		{U7IFIX "11", "b9e39ea98e3143f3155460cb12ebb983c38c9641"},
		{U7IFIX "12", "895c662b900a7300f1572b2d478300ed57269ac6"},
		{U7IFIX "13", "52990250a518369d50a4276865cb6ed3359795b1"},
		{U7IFIX "14", "497f5e37bc07bfe8b840232642ecec4885be8b38"},
		{U7IFIX "15", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "16", "f1379823d36da37913b04a05a777bc9e78b9671b"},
		{U7IFIX "17", "c8ca5993a3d699fd188140ee903c7baad8a4d652"},
		{U7IFIX "18", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "19", "688deb0662e9cd0cbb1429fb1cd0dcaa0f920b87"},
		{U7IFIX "1a", "b7113f16ad52db755677e61276d5b0dda79bb0e6"},
		{U7IFIX "1b", "5638cd5c36411b27611cd17d1bc92ede39ec38cd"},
		{U7IFIX "1c", "9cda878d0232cac9fe95e4859ceb051fcde5d4b4"},
		{U7IFIX "1d", "4ac5d40e515e241ddecaa8b28d0c7d5a07592dff"},
		{U7IFIX "1e", "837c725afc74eb0c9205931bae292f0eca38b39a"},
		{U7IFIX "1f", "abaf27da575c16b8ef988da62d31790f7f122a37"},
		{U7IFIX "20", "9819584c33e1142b24759cf0d3e660602b9390a4"},
		{U7IFIX "21", "d593ccbd3bd1aede43788e3fa05f9006196cab12"},
		{U7IFIX "22", "13b6797f433422f889528d71d09d442fb42f7e1e"},
		{U7IFIX "23", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "24", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "25", "92a66084450e66865e8b281523033912ecb6dbc2"},
		{U7IFIX "26", "082130deec589470762dfaf49e90635ce4014160"},
		{U7IFIX "27", "c0fe5e4d37e2232866bb7b59f941819bfdba4b5a"},
		{U7IFIX "28", "c83c5c4a3e7041af86ea1c278c5f080b6fc3ba30"},
		{U7IFIX "29", "15f10541c17de6bbae4dcb994c5273e1c3216d19"},
		{U7IFIX "2a", "c07e6802eac421c7dcf1a5bce105b1425bfe168c"},
		{U7IFIX "2b", "f064291feddf9022b4d423dcd25d9f9feee408ba"},
		{U7IFIX "2c", "cae675cf5fe77dafeb9261c3f447a2320a16e98f"},
		{U7IFIX "2d", "f24a42b459ad4a44eff39c22e5699d6dbbb153e7"},
		{U7IFIX "2e", "8dc8a8efbbbada1bb9952ddd23e6865d11b396d2"},
		{U7IFIX "2f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "30", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "31", "ddb020787ba59f6784ad0a081d7c6269b2f4011e"},
		{U7IFIX "32", "b22bff67502aa1bb5ada25bd6b6f7ba0393bfdec"},
		{U7IFIX "33", "79a3feba90f83ee49fdb6ca3a4927bd4351fa918"},
		{U7IFIX "34", "47e17d5ed8d6a823b8c373de9b83870c60884e5f"},
		{U7IFIX "35", "b664c3a4823915fca4b59c7731706ae145faed14"},
		{U7IFIX "36", "9c1682adda84946ad338df5980a0467479c3b65e"},
		{U7IFIX "37", "1bbf093d27780a34f0e765fb76d298970785581c"},
		{U7IFIX "38", "5521a5f0b469f070da0ee8f7067d76e839be47ca"},
		{U7IFIX "39", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "3a", "86424e2a75cd6b1a6a7a4d08cc0ce5c795d29531"},
		{U7IFIX "3b", "3322eeca60fffc9b5e7979f6dbd63e29785b4a13"},
		{U7IFIX "3c", "fa9a73e9c5cf6c3a1acc020002df03beb4721c49"},
		{U7IFIX "3d", "40ad586ac63607f4a887903c71a131b54a5b864b"},
		{U7IFIX "3e", "2b041288e41ed9f21054988f3e28e0770d9876c6"},
		{U7IFIX "3f", "027f1f75cd74a3487d3b530f3e6a0d927af00f7e"},
		{U7IFIX "40", "dd1c5c36b18ec462e4433f703b99d74a5d3f6af5"},
		{U7IFIX "41", "cb813d85e2eb989343592c9a031ec259485ddcd8"},
		{U7IFIX "42", "a3b1b09942f82648175fd5623af025569b233200"},
		{U7IFIX "43", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "44", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "45", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "46", "ab649f1a8d24ee8ea8dc5f1a929751166480b7b8"},
		{U7IFIX "47", "cbc8aa320e6bf910e6b678d00b7dcb14177bb634"},
		{U7IFIX "48", "eec082df80200f47f9d6f4adaceb556798901d21"},
		{U7IFIX "49", "4f368037159fce26eadbab1cf1d5b1160dd16d0b"},
		{U7IFIX "4a", "db5d6c7ad8fec3c3faf11c4021f249e1d01d6385"},
		{U7IFIX "4b", "bcd9dfad8b46e8abcb69e27abf9b64e8aac220c6"},
		{U7IFIX "4c", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "4d", "51df9c9941cb2630adab2b96e84be01094302775"},
		{U7IFIX "4e", "c19fed72d6c1760d45c92144118160d89de97bc5"},
		{U7IFIX "4f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "50", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "51", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "52", "8b49773bc0016d6d16e2c6bceefea35e91d7746f"},
		{U7IFIX "53", "a3616d5d6f4d40fff5b9046630fbc69f3249f3c1"},
		{U7IFIX "54", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "55", "9fa54ae682b2034809f0b1c0a2f458ba581c0b63"},
		{U7IFIX "56", "402455cc4756b0b6f0b4897901c2473ffb9e3692"},
		{U7IFIX "57", "9893e8b0a3e27ae398413e02e71dc278ed765a54"},
		{U7IFIX "58", "262d9a7508e56bf3c19c14ea9177c372da43638f"},
		{U7IFIX "59", "032afdce757b1ca66bd7107f058d01cc49832030"},
		{U7IFIX "5a", "0f252204f41861b08b6fb201e8bd750270e2b70d"},
		{U7IFIX "5b", "59387297607fddef7ee5605b885f87dc52eb011b"},
		{U7IFIX "5c", "d54f40caefaca9d0997a64f2277f8266f0673bae"},
		{U7IFIX "5d", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "5e", "7c9bc69a731df11a891d1764fc5244a6288e51a2"},
		{U7IFIX "5f", "e17ea9496ce1944f337db4563134f6b09170622f"},
		{U7IFIX "60", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "61", "b741f0befb470ccf1260e28e334447404ac825a8"},
		{U7IFIX "62", "c8a9d3908b0e5d18e62a9085a9498fdc44af9824"},
		{U7IFIX "63", "b98c2b741ff569df79f1575f79241f470f28ae76"},
		{U7IFIX "64", "dce1fa95e393a292c35d684d3b4ab13b570f843b"},
		{U7IFIX "65", "1c5bc75352c5ec3a68cef7d4c13593d98479710e"},
		{U7IFIX "66", "2880531459cd9e78fa440536ef1c9dd83121fd9e"},
		{U7IFIX "67", "2b22bd245302e6c18352f040194005445e2171be"},
		{U7IFIX "68", "8c4e1b7639e9123544d4316081ae399d7b29a1cc"},
		{U7IFIX "69", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "6a", "34b56453285014bc9d3357dd90b98d7860619667"},
		{U7IFIX "6b", "5c499e6df55ca88c56189e98a393c0b0df3ae77a"},
		{U7IFIX "6c", "1ac2d3e25be041b6e34e667546c86f9a3823032f"},
		{U7IFIX "6d", "39b3d1cd3f68cc79cd38f7379bcde7c80e1840d4"},
		{U7IFIX "6e", "c0387a805b24ffe67cc5048b3dd87022d7a90fa7"},
		{U7IFIX "6f", "8c5eae177455438234b7f21d27b215d48d52e67d"},
		{U7IFIX "70", "60cb9c43a18ab5df73a1f8b4313e5ee4ab0e0c10"},
		{U7IFIX "71", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "72", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "73", "54a5f96eb652fdce6124fd1047ca94191ec38cea"},
		{U7IFIX "74", "8136fabe27cad5717496eb5c958307920ebb5525"},
		{U7IFIX "75", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "76", "6dde9535938a5b4157d894915b5b22b7bdfb227c"},
		{U7IFIX "77", "fd50c150f08dce06549965fc38469927ca4c629e"},
		{U7IFIX "78", "04ac24bee791f5dfc01171a568d469fb5115bfaf"},
		{U7IFIX "79", "80de8eb6f1330b3c54b700a891923c50351f2e6b"},
		{U7IFIX "7a", "5fb6ce39a6ecbd59f88d490217c28ec546ab2260"},
		{U7IFIX "7b", "df30a2e94ccb6e68382b3d8f395f9097ea8fb0fb"},
		{U7IFIX "7c", "a2d4f122b2f71f8fc3de06a06859fc8862998cee"},
		{U7IFIX "7d", "b9458ba9e345ae9e2441d65bf09e750bfa054888"},
		{U7IFIX "7e", "d0286271c657f06cb36c202480991b6b8927e0f2"},
		{U7IFIX "7f", "5349a97972897092fad9ffce065bfdb0e9569e73"},
		{U7IFIX "80", "db8142e4eccdab10e2b1ea2adbc9e9ac83f416f8"},
		{U7IFIX "81", "e408895ac373d0627b76cb1eab0c8eb270bfecd6"},
		{U7IFIX "82", "2ec59c9639e79948b4830fd6a953e8811fdc7991"},
		{U7IFIX "83", "b7e399bd9419c36e854797cffcbea65246da54eb"},
		{U7IFIX "84", "d11d4da0e85d9463deed854ee2c3cf793be9a570"},
		{U7IFIX "85", "6964877d706be9017a232d5253c581dd9ed5191a"},
		{U7IFIX "86", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "87", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "88", "d97a018065022a1dbbbf620a7d2b7b232ea21a49"},
		{U7IFIX "89", "c943e73826fc325cea2c75713852af5a84b8f0ee"},
		{U7IFIX "8a", "f0fb50192d12c402237d9744a1f90db637962599"},
		{U7IFIX "8b", "e0d6e7f6d2467b7db5cab9fa3b315f1a160c354d"},
		{U7IFIX "8c", "fcaac53e029ad8350fb574924284ca7eb34fd8a4"},
		{U7IFIX "8d", "b8c9ffc019bb0729a0ec08be312d3a06f7eb6251"},
		{U7IFIX "8e", "629de2763fc14f8a93b5a207d099ee410d9172fa"},
		{U7IFIX "8f", "a60aaa1878c4f0128c9a609d5875c5838da3fb37"}};

const File2HashMap schwarzepforte_files{
		{MAINMUS_AD, "0ec151bd93e07ea36715d3bea56ea996bf9f7720"},
		{AMMO, "1c9313c1cf105a43933a29c6b7388872a7d8b21d"},
		{ARMOR, "32b00f5c8da9ccbeb503e9db08b6d06423282f07"},
		{ENDGAME, "5347705b2f7df66baeb6788f5b8974b3d967ef3d"},
		{ENDSCORE_XMI, "5b364cd61b2c431b4f8ddd835652fb7447d7e5db"},
		{ENDSHAPE_FLX, "4ea63cc5f98ea8b51b4167c10c13d3cea9e8c5db"},
		{EQUIP, "006c7bad2899fde897d0a09488574b624a28b90b"},
		{FACES_VGA, "6752ca0d454adf45aa3dab9d061e538afec99e48"},
		{FONTS_VGA, "514698f78df730845c176b518f80e685dac5ebd1"},
		{GUMPS_VGA, "bbf3576c1c1e256318f5d8c41db6ca386d280376"},
		{INITGAME, "45f2deef795f2c9785f1ad509f9842a806ef93c7"},
		{INTROMUS_AD, "7966b060c395b32aea00bc421167e27f74cf5db8"},
		{INTROPAL_DAT, "e4234ba7b53d83c882f6e1662a1a654d9da90547"},
		{INTROMUS, "9c52d65e43fd96db49ce20986833f161f59b945a"},
		{INTROSND, "764244bac7b99e5dbfe5233ec3336322c7bed3d3"},
		{MAINSHP_FLX, "4c530310070df641897580c747e49c184374d46f"},
		{MONSTERS, "d57a80ed8112ab55f3a5f5c2797bdf08d7f7f56b"},
		{MAINMUS, "2a48db4b099c950a6dd68a247785320c9df4d1c5"},
		{OCCLUDE, "506a97cdfcef2ee062b366275651d14c869cdfbe"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{POINTERS, "17ba747558f7e288ed8e12b227d4f3d598fda893"},
		{READY, "2b5b95c0ebe0ddf58a6196abfc89fc1ee39ed50c"},
		{SCHEDULE_DAT, "abf7b1d577039556bb0b5b94a8f26d88bc860485"},
		{SHAPES_VGA, "099740e6b4508b00eb23e0185f1973d53cc9791d"},
		{SHPDIMS, "00fd33bd081c6fe6b28be4e44ff3cc636bc50099"},
		{SPRITES_VGA, "716d601f8d26dedbb601f55f119f2f3f4702a91c"},
		{TEXT_FLX, "e424bd9a527e1f2b9ebb523cb9af20045272218e"},
		{TFA, "8519d9744a75219691b2eb5e7aed14f15618d27d"},
		{U7CHUNKS, "3327be23e8b241a0cc113d68b3989537ec63889d"},
		{INTRO_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{U7MAP, "960b7834f455e918c4088076c2629eb50cfb827b"},
		{U7SPEECH, "120112e5e49a3e94ce98b7e5be71f08dcc36ebca"},
		{U7VOICE_FLX, "cc276cc23dd3c900ef3e8af26d6384df3dbee14c"},
		{USECODE, "01955304f9abbca2b36f7f92b8aed138e8ffb22e"},
		{WEAPONS, "cea42a72a7cf46703f1168f9d49c912aa8d157e6"},
		{WGTVOL, "554d1788918468507b0bb4e4fa19087cd99d3610"},
		{WIHH, "2546cfe8e7d227d170f82b82db51c422ec04f88f"},
		{XFORMTBL, "c345d416ac6164c84e3276b85d6ee5a548c6125b"},
		{XMIDI_AD, "a1d5af1540a1ac23f259899bafd2b9013e9f449b"},
		{XMIDI_MT, "f4b0e6a1b6a5af50f98baa130dae36f448df3acd"},
		{U7IFIX "00", "e6e87969aaeeee0a5266ae4bd6d2d54b09563f05"},
		{U7IFIX "01", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "02", "3ea16eff9480c4084ad85b22ff0daf987c40cff3"},
		{U7IFIX "03", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "04", "97aa3e9c3e46e67c1c0796633c95dd5f3a0d1e04"},
		{U7IFIX "05", "81da20dedc86f3a7a9cc99805d2c44eff391b444"},
		{U7IFIX "06", "34b68f210182d48934677b9c2cead690d31a1f08"},
		{U7IFIX "07", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "08", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "09", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "0a", "c40332bf473ab6cf367c560952656fcc4666f296"},
		{U7IFIX "0b", "e7a231eecd9eaee03b687379294a0524b952e67c"},
		{U7IFIX "0c", "a202efdeacb9f126fecc3d53c8097c1353fe8c22"},
		{U7IFIX "0d", "e9ded9c7e354992b9a3236f5fe8bc022ac9a9eef"},
		{U7IFIX "0e", "8ea1fd177a5a27ec66273c7a7316866f01b45052"},
		{U7IFIX "0f", "3b9013c7e83e0706ec3586093994dc2c799591db"},
		{U7IFIX "10", "55a3ca7638f5f4b10cf221ae7fc8724bbf46f881"},
		{U7IFIX "11", "b9e39ea98e3143f3155460cb12ebb983c38c9641"},
		{U7IFIX "12", "895c662b900a7300f1572b2d478300ed57269ac6"},
		{U7IFIX "13", "52990250a518369d50a4276865cb6ed3359795b1"},
		{U7IFIX "14", "497f5e37bc07bfe8b840232642ecec4885be8b38"},
		{U7IFIX "15", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "16", "f1379823d36da37913b04a05a777bc9e78b9671b"},
		{U7IFIX "17", "c8ca5993a3d699fd188140ee903c7baad8a4d652"},
		{U7IFIX "18", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "19", "688deb0662e9cd0cbb1429fb1cd0dcaa0f920b87"},
		{U7IFIX "1a", "b7113f16ad52db755677e61276d5b0dda79bb0e6"},
		{U7IFIX "1b", "5638cd5c36411b27611cd17d1bc92ede39ec38cd"},
		{U7IFIX "1c", "9cda878d0232cac9fe95e4859ceb051fcde5d4b4"},
		{U7IFIX "1d", "4ac5d40e515e241ddecaa8b28d0c7d5a07592dff"},
		{U7IFIX "1e", "837c725afc74eb0c9205931bae292f0eca38b39a"},
		{U7IFIX "1f", "abaf27da575c16b8ef988da62d31790f7f122a37"},
		{U7IFIX "20", "9819584c33e1142b24759cf0d3e660602b9390a4"},
		{U7IFIX "21", "d593ccbd3bd1aede43788e3fa05f9006196cab12"},
		{U7IFIX "22", "13b6797f433422f889528d71d09d442fb42f7e1e"},
		{U7IFIX "23", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "24", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "25", "92a66084450e66865e8b281523033912ecb6dbc2"},
		{U7IFIX "26", "082130deec589470762dfaf49e90635ce4014160"},
		{U7IFIX "27", "c0fe5e4d37e2232866bb7b59f941819bfdba4b5a"},
		{U7IFIX "28", "c83c5c4a3e7041af86ea1c278c5f080b6fc3ba30"},
		{U7IFIX "29", "15f10541c17de6bbae4dcb994c5273e1c3216d19"},
		{U7IFIX "2a", "c07e6802eac421c7dcf1a5bce105b1425bfe168c"},
		{U7IFIX "2b", "f064291feddf9022b4d423dcd25d9f9feee408ba"},
		{U7IFIX "2c", "cae675cf5fe77dafeb9261c3f447a2320a16e98f"},
		{U7IFIX "2d", "f24a42b459ad4a44eff39c22e5699d6dbbb153e7"},
		{U7IFIX "2e", "8dc8a8efbbbada1bb9952ddd23e6865d11b396d2"},
		{U7IFIX "2f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "30", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "31", "ddb020787ba59f6784ad0a081d7c6269b2f4011e"},
		{U7IFIX "32", "b22bff67502aa1bb5ada25bd6b6f7ba0393bfdec"},
		{U7IFIX "33", "79a3feba90f83ee49fdb6ca3a4927bd4351fa918"},
		{U7IFIX "34", "47e17d5ed8d6a823b8c373de9b83870c60884e5f"},
		{U7IFIX "35", "b664c3a4823915fca4b59c7731706ae145faed14"},
		{U7IFIX "36", "9c1682adda84946ad338df5980a0467479c3b65e"},
		{U7IFIX "37", "1bbf093d27780a34f0e765fb76d298970785581c"},
		{U7IFIX "38", "5521a5f0b469f070da0ee8f7067d76e839be47ca"},
		{U7IFIX "39", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "3a", "86424e2a75cd6b1a6a7a4d08cc0ce5c795d29531"},
		{U7IFIX "3b", "3322eeca60fffc9b5e7979f6dbd63e29785b4a13"},
		{U7IFIX "3c", "fa9a73e9c5cf6c3a1acc020002df03beb4721c49"},
		{U7IFIX "3d", "40ad586ac63607f4a887903c71a131b54a5b864b"},
		{U7IFIX "3e", "2b041288e41ed9f21054988f3e28e0770d9876c6"},
		{U7IFIX "3f", "027f1f75cd74a3487d3b530f3e6a0d927af00f7e"},
		{U7IFIX "40", "dd1c5c36b18ec462e4433f703b99d74a5d3f6af5"},
		{U7IFIX "41", "cb813d85e2eb989343592c9a031ec259485ddcd8"},
		{U7IFIX "42", "a3b1b09942f82648175fd5623af025569b233200"},
		{U7IFIX "43", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "44", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "45", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "46", "ab649f1a8d24ee8ea8dc5f1a929751166480b7b8"},
		{U7IFIX "47", "cbc8aa320e6bf910e6b678d00b7dcb14177bb634"},
		{U7IFIX "48", "eec082df80200f47f9d6f4adaceb556798901d21"},
		{U7IFIX "49", "4f368037159fce26eadbab1cf1d5b1160dd16d0b"},
		{U7IFIX "4a", "db5d6c7ad8fec3c3faf11c4021f249e1d01d6385"},
		{U7IFIX "4b", "bcd9dfad8b46e8abcb69e27abf9b64e8aac220c6"},
		{U7IFIX "4c", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "4d", "51df9c9941cb2630adab2b96e84be01094302775"},
		{U7IFIX "4e", "c19fed72d6c1760d45c92144118160d89de97bc5"},
		{U7IFIX "4f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "50", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "51", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "52", "8b49773bc0016d6d16e2c6bceefea35e91d7746f"},
		{U7IFIX "53", "a3616d5d6f4d40fff5b9046630fbc69f3249f3c1"},
		{U7IFIX "54", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "55", "9fa54ae682b2034809f0b1c0a2f458ba581c0b63"},
		{U7IFIX "56", "402455cc4756b0b6f0b4897901c2473ffb9e3692"},
		{U7IFIX "57", "9893e8b0a3e27ae398413e02e71dc278ed765a54"},
		{U7IFIX "58", "262d9a7508e56bf3c19c14ea9177c372da43638f"},
		{U7IFIX "59", "032afdce757b1ca66bd7107f058d01cc49832030"},
		{U7IFIX "5a", "0f252204f41861b08b6fb201e8bd750270e2b70d"},
		{U7IFIX "5b", "59387297607fddef7ee5605b885f87dc52eb011b"},
		{U7IFIX "5c", "d54f40caefaca9d0997a64f2277f8266f0673bae"},
		{U7IFIX "5d", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "5e", "7c9bc69a731df11a891d1764fc5244a6288e51a2"},
		{U7IFIX "5f", "e17ea9496ce1944f337db4563134f6b09170622f"},
		{U7IFIX "60", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "61", "b741f0befb470ccf1260e28e334447404ac825a8"},
		{U7IFIX "62", "c8a9d3908b0e5d18e62a9085a9498fdc44af9824"},
		{U7IFIX "63", "b98c2b741ff569df79f1575f79241f470f28ae76"},
		{U7IFIX "64", "dce1fa95e393a292c35d684d3b4ab13b570f843b"},
		{U7IFIX "65", "1c5bc75352c5ec3a68cef7d4c13593d98479710e"},
		{U7IFIX "66", "2880531459cd9e78fa440536ef1c9dd83121fd9e"},
		{U7IFIX "67", "2b22bd245302e6c18352f040194005445e2171be"},
		{U7IFIX "68", "8c4e1b7639e9123544d4316081ae399d7b29a1cc"},
		{U7IFIX "69", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "6a", "34b56453285014bc9d3357dd90b98d7860619667"},
		{U7IFIX "6b", "5c499e6df55ca88c56189e98a393c0b0df3ae77a"},
		{U7IFIX "6c", "1ac2d3e25be041b6e34e667546c86f9a3823032f"},
		{U7IFIX "6d", "39b3d1cd3f68cc79cd38f7379bcde7c80e1840d4"},
		{U7IFIX "6e", "c0387a805b24ffe67cc5048b3dd87022d7a90fa7"},
		{U7IFIX "6f", "8c5eae177455438234b7f21d27b215d48d52e67d"},
		{U7IFIX "70", "60cb9c43a18ab5df73a1f8b4313e5ee4ab0e0c10"},
		{U7IFIX "71", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "72", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "73", "54a5f96eb652fdce6124fd1047ca94191ec38cea"},
		{U7IFIX "74", "8136fabe27cad5717496eb5c958307920ebb5525"},
		{U7IFIX "75", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "76", "6dde9535938a5b4157d894915b5b22b7bdfb227c"},
		{U7IFIX "77", "fd50c150f08dce06549965fc38469927ca4c629e"},
		{U7IFIX "78", "04ac24bee791f5dfc01171a568d469fb5115bfaf"},
		{U7IFIX "79", "80de8eb6f1330b3c54b700a891923c50351f2e6b"},
		{U7IFIX "7a", "5fb6ce39a6ecbd59f88d490217c28ec546ab2260"},
		{U7IFIX "7b", "df30a2e94ccb6e68382b3d8f395f9097ea8fb0fb"},
		{U7IFIX "7c", "a2d4f122b2f71f8fc3de06a06859fc8862998cee"},
		{U7IFIX "7d", "b9458ba9e345ae9e2441d65bf09e750bfa054888"},
		{U7IFIX "7e", "d0286271c657f06cb36c202480991b6b8927e0f2"},
		{U7IFIX "7f", "5349a97972897092fad9ffce065bfdb0e9569e73"},
		{U7IFIX "80", "db8142e4eccdab10e2b1ea2adbc9e9ac83f416f8"},
		{U7IFIX "81", "e408895ac373d0627b76cb1eab0c8eb270bfecd6"},
		{U7IFIX "82", "2ec59c9639e79948b4830fd6a953e8811fdc7991"},
		{U7IFIX "83", "b7e399bd9419c36e854797cffcbea65246da54eb"},
		{U7IFIX "84", "d11d4da0e85d9463deed854ee2c3cf793be9a570"},
		{U7IFIX "85", "6964877d706be9017a232d5253c581dd9ed5191a"},
		{U7IFIX "86", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "87", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "88", "d97a018065022a1dbbbf620a7d2b7b232ea21a49"},
		{U7IFIX "89", "c943e73826fc325cea2c75713852af5a84b8f0ee"},
		{U7IFIX "8a", "f0fb50192d12c402237d9744a1f90db637962599"},
		{U7IFIX "8b", "e0d6e7f6d2467b7db5cab9fa3b315f1a160c354d"},
		{U7IFIX "8c", "fcaac53e029ad8350fb574924284ca7eb34fd8a4"},
		{U7IFIX "8d", "b8c9ffc019bb0729a0ec08be312d3a06f7eb6251"},
		{U7IFIX "8e", "629de2763fc14f8a93b5a207d099ee410d9172fa"},
		{U7IFIX "8f", "a60aaa1878c4f0128c9a609d5875c5838da3fb37"}};

const File2HashMap forgeofvirtue_files{
		{MAINMUS_AD, "0ec151bd93e07ea36715d3bea56ea996bf9f7720"},
		{AMMO, "1c9313c1cf105a43933a29c6b7388872a7d8b21d"},
		{ARMOR, "32b00f5c8da9ccbeb503e9db08b6d06423282f07"},
		{ENDGAME, "5fde1edc8eb90b436146bbdad5c96d046e371f31"},
		{ENDSCORE_XMI, "5b364cd61b2c431b4f8ddd835652fb7447d7e5db"},
		{ENDSHAPE_FLX, "831899ce7b270ac27d297147181e98c5a2643bcb"},
		{EQUIP, "006c7bad2899fde897d0a09488574b624a28b90b"},
		{FACES_VGA, "5d3be8996f1301b08eadff933869d718b0be097a"},
		{FONTS_VGA, "848bd7f677c4e4ac496482e4c1c24273d9d607e6"},
		{GUMPS_VGA, "372ebae26a85aefca04352cec51b492b0f361968"},
		{INITGAME, "c13aaa7b5be6f40b95ab8399198e0c339342d5eb"},
		{INTROMUS_AD, "7966b060c395b32aea00bc421167e27f74cf5db8"},
		{INTROPAL_DAT, "e4234ba7b53d83c882f6e1662a1a654d9da90547"},
		{INTROMUS, "9c52d65e43fd96db49ce20986833f161f59b945a"},
		{INTROSND, "2191c1ddb93a6a78f983fc77a2cf87cc368647e4"},
		{MAINSHP_FLX, "a8cdc79a2bcd77309e36d9b43030175d3d6ea409"},
		{MONSTERS, "7c06cf913d634d6d5b8a73f5af63afe9047d3415"},
		{MAINMUS, "2a48db4b099c950a6dd68a247785320c9df4d1c5"},
		{OCCLUDE, "18e0de95abd033085ad313df525638bd8564d51b"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{POINTERS, "29d19ef760a3832721c41432cdf364d457c6df22"},
		{READY, "083ac53121bbaa2b659d23074b778136579177c8"},
		{SCHEDULE_DAT, "abf7b1d577039556bb0b5b94a8f26d88bc860485"},
		{SHAPES_VGA, "099740e6b4508b00eb23e0185f1973d53cc9791d"},
		{SHPDIMS, "f35d81e9dd1e5fef02f4193316e4a04ee090b5ee"},
		{SPRITES_VGA, "716d601f8d26dedbb601f55f119f2f3f4702a91c"},
		{TEXT_FLX, "40bb782cbeceac96f379a6491c97e6bc2a458efd"},
		{TFA, "60927a82a08be74053e590f308a00e480a913612"},
		{U7CHUNKS, "4f9310fc15c5ad611db1424296ff041a38316087"},
		{INTRO_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{U7MAP, "64dab40006c528d6964ab13f9e36fd487f4b204e"},
		{U7SPEECH, "7af1fbda6e0fbdbc07ed8a851de16aa99fa4bb4e"},
		{U7VOICE_FLX, "cc276cc23dd3c900ef3e8af26d6384df3dbee14c"},
		{USECODE, "2b8f2e630375e785435809ae8d3bafc919152359"},
		{WEAPONS, "1852b353e6f142e2535e873b8503cc5cf0c8a2b3"},
		{WGTVOL, "7e5ade73bea610bd4a73bda4ca3ceee242a035d6"},
		{WIHH, "4866e3526b49cdfbd7ccd1943a1fc9abb979f894"},
		{XFORMTBL, "c345d416ac6164c84e3276b85d6ee5a548c6125b"},
		{XMIDI_AD, "a1d5af1540a1ac23f259899bafd2b9013e9f449b"},
		{XMIDI_MT, "f4b0e6a1b6a5af50f98baa130dae36f448df3acd"},
		{U7IFIX "00", "e6e87969aaeeee0a5266ae4bd6d2d54b09563f05"},
		{U7IFIX "01", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "02", "3ea16eff9480c4084ad85b22ff0daf987c40cff3"},
		{U7IFIX "03", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "04", "97aa3e9c3e46e67c1c0796633c95dd5f3a0d1e04"},
		{U7IFIX "05", "81da20dedc86f3a7a9cc99805d2c44eff391b444"},
		{U7IFIX "06", "34b68f210182d48934677b9c2cead690d31a1f08"},
		{U7IFIX "07", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "08", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "09", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "0a", "c40332bf473ab6cf367c560952656fcc4666f296"},
		{U7IFIX "0b", "e7a231eecd9eaee03b687379294a0524b952e67c"},
		{U7IFIX "0c", "a202efdeacb9f126fecc3d53c8097c1353fe8c22"},
		{U7IFIX "0d", "e9ded9c7e354992b9a3236f5fe8bc022ac9a9eef"},
		{U7IFIX "0e", "8ea1fd177a5a27ec66273c7a7316866f01b45052"},
		{U7IFIX "0f", "3b9013c7e83e0706ec3586093994dc2c799591db"},
		{U7IFIX "10", "55a3ca7638f5f4b10cf221ae7fc8724bbf46f881"},
		{U7IFIX "11", "b9e39ea98e3143f3155460cb12ebb983c38c9641"},
		{U7IFIX "12", "895c662b900a7300f1572b2d478300ed57269ac6"},
		{U7IFIX "13", "52990250a518369d50a4276865cb6ed3359795b1"},
		{U7IFIX "14", "497f5e37bc07bfe8b840232642ecec4885be8b38"},
		{U7IFIX "15", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "16", "f1379823d36da37913b04a05a777bc9e78b9671b"},
		{U7IFIX "17", "c8ca5993a3d699fd188140ee903c7baad8a4d652"},
		{U7IFIX "18", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "19", "688deb0662e9cd0cbb1429fb1cd0dcaa0f920b87"},
		{U7IFIX "1a", "b7113f16ad52db755677e61276d5b0dda79bb0e6"},
		{U7IFIX "1b", "5638cd5c36411b27611cd17d1bc92ede39ec38cd"},
		{U7IFIX "1c", "9cda878d0232cac9fe95e4859ceb051fcde5d4b4"},
		{U7IFIX "1d", "4ac5d40e515e241ddecaa8b28d0c7d5a07592dff"},
		{U7IFIX "1e", "837c725afc74eb0c9205931bae292f0eca38b39a"},
		{U7IFIX "1f", "abaf27da575c16b8ef988da62d31790f7f122a37"},
		{U7IFIX "20", "9819584c33e1142b24759cf0d3e660602b9390a4"},
		{U7IFIX "21", "d593ccbd3bd1aede43788e3fa05f9006196cab12"},
		{U7IFIX "22", "13b6797f433422f889528d71d09d442fb42f7e1e"},
		{U7IFIX "23", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "24", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "25", "92a66084450e66865e8b281523033912ecb6dbc2"},
		{U7IFIX "26", "082130deec589470762dfaf49e90635ce4014160"},
		{U7IFIX "27", "c0fe5e4d37e2232866bb7b59f941819bfdba4b5a"},
		{U7IFIX "28", "c83c5c4a3e7041af86ea1c278c5f080b6fc3ba30"},
		{U7IFIX "29", "15f10541c17de6bbae4dcb994c5273e1c3216d19"},
		{U7IFIX "2a", "c07e6802eac421c7dcf1a5bce105b1425bfe168c"},
		{U7IFIX "2b", "f064291feddf9022b4d423dcd25d9f9feee408ba"},
		{U7IFIX "2c", "cae675cf5fe77dafeb9261c3f447a2320a16e98f"},
		{U7IFIX "2d", "f24a42b459ad4a44eff39c22e5699d6dbbb153e7"},
		{U7IFIX "2e", "8dc8a8efbbbada1bb9952ddd23e6865d11b396d2"},
		{U7IFIX "2f", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "30", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "31", "ddb020787ba59f6784ad0a081d7c6269b2f4011e"},
		{U7IFIX "32", "b22bff67502aa1bb5ada25bd6b6f7ba0393bfdec"},
		{U7IFIX "33", "79a3feba90f83ee49fdb6ca3a4927bd4351fa918"},
		{U7IFIX "34", "47e17d5ed8d6a823b8c373de9b83870c60884e5f"},
		{U7IFIX "35", "b664c3a4823915fca4b59c7731706ae145faed14"},
		{U7IFIX "36", "9c1682adda84946ad338df5980a0467479c3b65e"},
		{U7IFIX "37", "1bbf093d27780a34f0e765fb76d298970785581c"},
		{U7IFIX "38", "5521a5f0b469f070da0ee8f7067d76e839be47ca"},
		{U7IFIX "39", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "3a", "86424e2a75cd6b1a6a7a4d08cc0ce5c795d29531"},
		{U7IFIX "3b", "3322eeca60fffc9b5e7979f6dbd63e29785b4a13"},
		{U7IFIX "3c", "fa9a73e9c5cf6c3a1acc020002df03beb4721c49"},
		{U7IFIX "3d", "40ad586ac63607f4a887903c71a131b54a5b864b"},
		{U7IFIX "3e", "2b041288e41ed9f21054988f3e28e0770d9876c6"},
		{U7IFIX "3f", "027f1f75cd74a3487d3b530f3e6a0d927af00f7e"},
		{U7IFIX "40", "dd1c5c36b18ec462e4433f703b99d74a5d3f6af5"},
		{U7IFIX "41", "cb813d85e2eb989343592c9a031ec259485ddcd8"},
		{U7IFIX "42", "a3b1b09942f82648175fd5623af025569b233200"},
		{U7IFIX "43", "785bc792c0712fd666429161959cb761ccde7639"},
		{U7IFIX "44", "4f0b354de9513a524498a4934f4f2b9667ff2bf9"},
		{U7IFIX "45", "51b7eb703ae3a89b8606d47d1ba9d0fead18c67a"},
		{U7IFIX "46", "ab649f1a8d24ee8ea8dc5f1a929751166480b7b8"},
		{U7IFIX "47", "cbc8aa320e6bf910e6b678d00b7dcb14177bb634"},
		{U7IFIX "48", "eec082df80200f47f9d6f4adaceb556798901d21"},
		{U7IFIX "49", "4f368037159fce26eadbab1cf1d5b1160dd16d0b"},
		{U7IFIX "4a", "db5d6c7ad8fec3c3faf11c4021f249e1d01d6385"},
		{U7IFIX "4b", "bcd9dfad8b46e8abcb69e27abf9b64e8aac220c6"},
		{U7IFIX "4c", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "4d", "51df9c9941cb2630adab2b96e84be01094302775"},
		{U7IFIX "4e", "c19fed72d6c1760d45c92144118160d89de97bc5"},
		{U7IFIX "4f", "04db0271b3a0b3623773ce000f11f671b304ed0b"},
		{U7IFIX "50", "34ca081f4982610d770c6a2bea4ed844fce90956"},
		{U7IFIX "51", "10f44a19181bf3d6d433ce82c0acc70979d9c718"},
		{U7IFIX "52", "8b49773bc0016d6d16e2c6bceefea35e91d7746f"},
		{U7IFIX "53", "a3616d5d6f4d40fff5b9046630fbc69f3249f3c1"},
		{U7IFIX "54", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "55", "9fa54ae682b2034809f0b1c0a2f458ba581c0b63"},
		{U7IFIX "56", "402455cc4756b0b6f0b4897901c2473ffb9e3692"},
		{U7IFIX "57", "9893e8b0a3e27ae398413e02e71dc278ed765a54"},
		{U7IFIX "58", "262d9a7508e56bf3c19c14ea9177c372da43638f"},
		{U7IFIX "59", "032afdce757b1ca66bd7107f058d01cc49832030"},
		{U7IFIX "5a", "0f252204f41861b08b6fb201e8bd750270e2b70d"},
		{U7IFIX "5b", "59387297607fddef7ee5605b885f87dc52eb011b"},
		{U7IFIX "5c", "d54f40caefaca9d0997a64f2277f8266f0673bae"},
		{U7IFIX "5d", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "5e", "7c9bc69a731df11a891d1764fc5244a6288e51a2"},
		{U7IFIX "5f", "e17ea9496ce1944f337db4563134f6b09170622f"},
		{U7IFIX "60", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "61", "b741f0befb470ccf1260e28e334447404ac825a8"},
		{U7IFIX "62", "c8a9d3908b0e5d18e62a9085a9498fdc44af9824"},
		{U7IFIX "63", "b98c2b741ff569df79f1575f79241f470f28ae76"},
		{U7IFIX "64", "dce1fa95e393a292c35d684d3b4ab13b570f843b"},
		{U7IFIX "65", "1c5bc75352c5ec3a68cef7d4c13593d98479710e"},
		{U7IFIX "66", "2880531459cd9e78fa440536ef1c9dd83121fd9e"},
		{U7IFIX "67", "2b22bd245302e6c18352f040194005445e2171be"},
		{U7IFIX "68", "8c4e1b7639e9123544d4316081ae399d7b29a1cc"},
		{U7IFIX "69", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "6a", "34b56453285014bc9d3357dd90b98d7860619667"},
		{U7IFIX "6b", "5c499e6df55ca88c56189e98a393c0b0df3ae77a"},
		{U7IFIX "6c", "1ac2d3e25be041b6e34e667546c86f9a3823032f"},
		{U7IFIX "6d", "39b3d1cd3f68cc79cd38f7379bcde7c80e1840d4"},
		{U7IFIX "6e", "c0387a805b24ffe67cc5048b3dd87022d7a90fa7"},
		{U7IFIX "6f", "8c5eae177455438234b7f21d27b215d48d52e67d"},
		{U7IFIX "70", "60cb9c43a18ab5df73a1f8b4313e5ee4ab0e0c10"},
		{U7IFIX "71", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "72", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "73", "54a5f96eb652fdce6124fd1047ca94191ec38cea"},
		{U7IFIX "74", "8136fabe27cad5717496eb5c958307920ebb5525"},
		{U7IFIX "75", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "76", "6dde9535938a5b4157d894915b5b22b7bdfb227c"},
		{U7IFIX "77", "fd50c150f08dce06549965fc38469927ca4c629e"},
		{U7IFIX "78", "04ac24bee791f5dfc01171a568d469fb5115bfaf"},
		{U7IFIX "79", "80de8eb6f1330b3c54b700a891923c50351f2e6b"},
		{U7IFIX "7a", "5fb6ce39a6ecbd59f88d490217c28ec546ab2260"},
		{U7IFIX "7b", "df30a2e94ccb6e68382b3d8f395f9097ea8fb0fb"},
		{U7IFIX "7c", "a2d4f122b2f71f8fc3de06a06859fc8862998cee"},
		{U7IFIX "7d", "b9458ba9e345ae9e2441d65bf09e750bfa054888"},
		{U7IFIX "7e", "d0286271c657f06cb36c202480991b6b8927e0f2"},
		{U7IFIX "7f", "5349a97972897092fad9ffce065bfdb0e9569e73"},
		{U7IFIX "80", "db8142e4eccdab10e2b1ea2adbc9e9ac83f416f8"},
		{U7IFIX "81", "e408895ac373d0627b76cb1eab0c8eb270bfecd6"},
		{U7IFIX "82", "2ec59c9639e79948b4830fd6a953e8811fdc7991"},
		{U7IFIX "83", "b7e399bd9419c36e854797cffcbea65246da54eb"},
		{U7IFIX "84", "d11d4da0e85d9463deed854ee2c3cf793be9a570"},
		{U7IFIX "85", "6964877d706be9017a232d5253c581dd9ed5191a"},
		{U7IFIX "86", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "87", "bf111dbdbf8a322dcdf7ba862cc32e6f00790915"},
		{U7IFIX "88", "d97a018065022a1dbbbf620a7d2b7b232ea21a49"},
		{U7IFIX "89", "c943e73826fc325cea2c75713852af5a84b8f0ee"},
		{U7IFIX "8a", "f0fb50192d12c402237d9744a1f90db637962599"},
		{U7IFIX "8b", "e0d6e7f6d2467b7db5cab9fa3b315f1a160c354d"},
		{U7IFIX "8c", "fcaac53e029ad8350fb574924284ca7eb34fd8a4"},
		{U7IFIX "8d", "b8c9ffc019bb0729a0ec08be312d3a06f7eb6251"},
		{U7IFIX "8e", "629de2763fc14f8a93b5a207d099ee410d9172fa"},
		{U7IFIX "8f", "a60aaa1878c4f0128c9a609d5875c5838da3fb37"}};

const File2HashMap serpentbeta_files{
		{A_SINTRO, "fadc12356803b0c2c06375c813247d684da505b8"},
		{MAINMUS_AD, "ac89dea0a024ff79315df1ed8c8648c137f28508"},
		{AMMO, "93641547da762df5e29b73a1560869b69b6ee625"},
		{ARMOR, "7154839d0c102c5244d7ebfc2c5e4c36cd185dcd"},
		{EQUIP, "f93ee9e7c21f91dc7567d6e014f9d14913bf9c1c"},
		{FACES_VGA, "a2639f6e45abf8d097ede61ef6b9630d7422cb20"},
		{FONTS_VGA, "d852da3709f42ca12a50110ad4fef7f5241fdb69"},
		{GUMPS_VGA, "9c3e236603b942822557bdd8793145eb84b2f275"},
		{INITGAME, "c55261c5f65aa140f40ceeefde0736eebfd5d6ed"},
		{INTRO_DAT, "eae097b213ca9e17dafd9755ec0db9e378dd9d2f"},
		{MAINMENU_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{MAINSHP_FLX, "fbc785da3a7565fcb4cc90ece1db51e7938cff83"},
		{MONSTERS, "cf3f064c08cb2a338e423de39d52805390fd8f4b"},
		{MAINMUS, "199ac092eb11a7bfcaaaa0bc5a6786f2f952d6f2"},
		{OCCLUDE, "9778e4a5f359bee323d663a03fd6784c15bbf28a"},
		{PALETTES_FLX, "a2306ce38a9478c5ac5c1f3f7ca721440e9eec1e"},
		{PAPERDOL, "931b8c7a2fc650d492b4ea8682761ec7a5bf4fb3"},
		{POINTERS, "501e371f934c92d62a9b05304478b017467499e4"},
		{R_SINTRO, "1606f5f00dec70cf22a62c3e9cb3d4e92d0d00ed"},
		{READY, "a415542b7eae62533606d51fb34bef04bcda9a1d"},
		{SCHEDULE_DAT, "41eee6c80c8c9fe2db9e2fbcf666146da5baf262"},
		{SHAPES_VGA, "d0d7725b1c146434bf96e51738376bc5862548d5"},
		{SHPDIMS, "b5d0cca3587d7d7cf47a950d4eaa7a0f39f2bfe0"},
		{SISPEECH, "b3824016ada236fafc6ba5051eec06b86bc1be9e"},
		{SPRITES_VGA, "a15cbc690c1e41fa8d93827459a45c89a52123dd"},
		{TEXT_FLX, "961ba69f11308e838b297d8492e6193a081d7ab5"},
		{TFA, "6f4fc557eeaf3dbd667180fada657d22afb41f6e"},
		{U7CHUNKS, "94082444533320bb2099841543496f68e2c7dbc4"},
		{U7MAP, "66040c53cf5d6cdd4491238dafc86d196474ae1b"},
		{USECODE, "517c5b088f3ef8188f2d7f4cf078cf7fff921826"},
		{WEAPONS, "6ab5c199b2e62bc928ba376f8611717466ea5215"},
		{WGTVOL, "1ea22c6dc78eaa35c0aa80af0f6ddf52168482df"},
		{WIHH, "efe9d817e5eeca2bb95ef859a5a8241116f116f0"},
		{XFORMTBL, "a1f25f60dd75737e26c0a7116d6d64f668fba429"},
		{XMIDI_AD, "48261bafb7932680577f324a0f85bad95a7a77cd"},
		{XMIDI_MT, "1181e55b2ad06368ae262b9bbd7fcc90ad0f86e0"},
		{U7IFIX "00", "ec11ebcd215725607aba701c3e1290876abe4a23"},
		{U7IFIX "01", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "02", "0facb469073f223ecca792dea7fd5e433b043146"},
		{U7IFIX "03", "88d9a221bb6cc06543f91786492843219c436c51"},
		{U7IFIX "04", "f2c3de3b3e05e62ee7752f5c982fd5147fc78a85"},
		{U7IFIX "05", "426ddf3a5230386d0e8ae35b05a993d33b4e3bec"},
		{U7IFIX "06", "a8ec8b03849433f848d9999d0e1ae49ce0b0d87e"},
		{U7IFIX "07", "e1e4777e66668988048400a03439602f5e77b842"},
		{U7IFIX "08", "a0a6d417e1774a628a0623ec57685eb2e86df842"},
		{U7IFIX "09", "72c9bdf03487643c8de9e8ad6c7ee9b46679673f"},
		{U7IFIX "0a", "f7d03cdd4ed26fe7ca52a64a791eb384d70d8585"},
		{U7IFIX "0b", "74b94c048c0c9328f899cd6ff3caafb5ba6d8c3c"},
		{U7IFIX "0c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0e", "0058030bfc7ed68b2ac4eaf1ebe8720db963d865"},
		{U7IFIX "0f", "df2991f26caf0a75c9586005a9cdcebdeb95285e"},
		{U7IFIX "10", "1b3935224ae29ff56b72bf56f8a7a1252691399d"},
		{U7IFIX "11", "29af79d0bc5fc852ccc866d22bbca9a990c8f5b2"},
		{U7IFIX "12", "fcd7c893260b51c8d40bb26eb32ae1fb0fd7cd6c"},
		{U7IFIX "13", "15b4e8cb27ae8dd126284e72c63ac06f47c03a68"},
		{U7IFIX "14", "caa9845510f8504156cf859452b6dfb1a8349aa3"},
		{U7IFIX "15", "4a9d5ed39f72bb94b00abfdbe3092684aceaba2c"},
		{U7IFIX "16", "2b0ab00e89bee950afcbcdafa7c5c21cb174401d"},
		{U7IFIX "17", "665e4b9c23cd6ae51658f0fdce53862ff66e4b60"},
		{U7IFIX "18", "66836c7611589686e2b13381b22da6ca3072bbe2"},
		{U7IFIX "19", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1b", "266e15c1353cbb9d1a39eff67c36d8864bf78d7b"},
		{U7IFIX "1c", "8b28eaa587623aaccac1d0d6e994a047838026dc"},
		{U7IFIX "1d", "caf7a52dd4bb5cb6642e222f784b46297ba5d30b"},
		{U7IFIX "1e", "72172a4f493a7d6715b64c0803bb29afd3229413"},
		{U7IFIX "1f", "59301594b49811fdbd98ec0593020c9471629032"},
		{U7IFIX "20", "4323cc1d3bad9a5a05a5d7c9f9e020df490c6868"},
		{U7IFIX "21", "ae7d40937516d01b3ef3189177ac504c0ecbca68"},
		{U7IFIX "22", "0de0683ed71c54f1629d7231a1ad49cf66297796"},
		{U7IFIX "23", "4fde987f7e611f5689f15f557073b4b52e1a8253"},
		{U7IFIX "24", "99cae41f15e9b16b9c6af504149ffa9467e2a075"},
		{U7IFIX "25", "adab01ceab0f7d0f18edaa62e4f7d636a9be7158"},
		{U7IFIX "26", "cb96656113b268e7efb1a5ec154fcfdbca7ecacb"},
		{U7IFIX "27", "a3bc42f7eb30521bf362482ceb1c214faedbebca"},
		{U7IFIX "28", "5eacbbbf1bc6cc4fab51bfe305e8c8f972a34718"},
		{U7IFIX "29", "4ea17e6f550da99a7c453a2449165cde141fb8df"},
		{U7IFIX "2a", "414b1c55b47825ae5e4c4923bb80be648dfa2cb3"},
		{U7IFIX "2b", "e7006b1bfe1243deac74ce2d99f321f19d1eaf24"},
		{U7IFIX "2c", "0ddefad211ab9088072d40ac187f8d4e312e06a0"},
		{U7IFIX "2d", "1b65380a41dc593b1449f76d0e5ba9b5107301bb"},
		{U7IFIX "2e", "77c2591abed867f7533f4a5ea22da51ef37e917a"},
		{U7IFIX "2f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "30", "30f06b306527340d95504673db2abe44fec514de"},
		{U7IFIX "31", "5e85416ebbe29ca50b0cdf3efa18459c3986662a"},
		{U7IFIX "32", "a01631dae9e9714ca3cf646d3073909dd57daa4e"},
		{U7IFIX "33", "731cc2dd7f958a07ffdd3becf4e4e8e73835e153"},
		{U7IFIX "34", "679b7a208017259360bb40e4873264cdf6bbc7ae"},
		{U7IFIX "35", "fbf1554679f18ddaba316dcdc338a88b3c6b7d68"},
		{U7IFIX "36", "3b790fc8bafa275ba9c29df18de253de9d1f69ab"},
		{U7IFIX "37", "3d9ce92e646ede25cb57719237c6b421b27e4330"},
		{U7IFIX "38", "70419a84495af5f6d3171b512449a3079b80d35c"},
		{U7IFIX "39", "c161a992aa8b9e1564fbfe8f7f4dd233e69d7e8b"},
		{U7IFIX "3a", "7cefc88cc95591bc338eda8e984b92919d177f2f"},
		{U7IFIX "3b", "252a6829d721624f2cbda9b13dba34f45a6ed58c"},
		{U7IFIX "3c", "a921d1b5263069926ccebd648415f93c94f4936c"},
		{U7IFIX "3d", "d83b43c39a92a1a64a30286854e807748fc3ae53"},
		{U7IFIX "3e", "c5f645927635ca479b41bb8b675451d51307eb7b"},
		{U7IFIX "3f", "4e5767a73229cda6e4ce69de1b42361ef92709ba"},
		{U7IFIX "40", "2e888891124143e5586900909744b484b80a3d20"},
		{U7IFIX "41", "31fbba9e39daafcfbb0c1788ff056c87b9121b25"},
		{U7IFIX "42", "6163a061b174b25b53d3be6934caac57556c948e"},
		{U7IFIX "43", "4826e4cb128cade01f24431391aa9c6a2d045cf1"},
		{U7IFIX "44", "9e739c8eb88efd27a962f17c082dc5e5a463fbb2"},
		{U7IFIX "45", "767f50296ede7f23d2fe1eaaa72754dc6d8d05ad"},
		{U7IFIX "46", "bba226a7bfe42f525d8cb93082f6b206c2ea1475"},
		{U7IFIX "47", "affa556aa668e2697f8b1367e1a8106a7b025b88"},
		{U7IFIX "48", "0594fc04fbb2e82cc32296dc5e9a47f612d659a4"},
		{U7IFIX "49", "9d1018a90ebb3667c12692d46b6f517abb9eed0f"},
		{U7IFIX "4a", "e0da20b60a7555f929379701dc17ad0bfc6fe20e"},
		{U7IFIX "4b", "063b9289cb8d54b24e98cb0e4844854c1acefc96"},
		{U7IFIX "4c", "801ba5e817a66de9e4d0cb947513881989e27d0c"},
		{U7IFIX "4d", "ecd0105fe2cd80bc590d13ab51f3e788b88b13fc"},
		{U7IFIX "4e", "6561428ec7790cdbce4db80847c7c9816470c167"},
		{U7IFIX "4f", "eb8430bf275430729251ef2424197a2f9d7e1120"},
		{U7IFIX "50", "222e54c1141603614ad8d4bf612e382578dba649"},
		{U7IFIX "51", "b30005490c30ad8731591ba4daf995cd9614c4f3"},
		{U7IFIX "52", "5d00aef748bc16ab852e971e100bd9ba548af2bd"},
		{U7IFIX "53", "81380061482a01880aabec1754a0998a5139b7e6"},
		{U7IFIX "54", "a9e5001e258f32f1d8380dbacd072275f500ce9e"},
		{U7IFIX "55", "a6846d13e637f51f8c4679cdc511e1098e9a3a54"},
		{U7IFIX "56", "b9b55e6087b6354f6c7ccef9d86ea045f3e7d7fe"},
		{U7IFIX "57", "570bf17f58bb5dbb9d3e0fb1063176ae142be09a"},
		{U7IFIX "58", "08c3a3c7b5d5d69f9d1404ea61d38a4fe1bb4f77"},
		{U7IFIX "59", "b4268812d030d2835f60808193a3a51addcd9591"},
		{U7IFIX "5a", "27204d3218084fa7658363cc7c5cc6176562f5e5"},
		{U7IFIX "5b", "a016b669b1483acc3938bad9bd11c748d850c3e6"},
		{U7IFIX "5c", "aadf99e5e970747184dd3b96795ad0b3f85a7848"},
		{U7IFIX "5d", "d4fcb39534fd8d5d6dce2efb5d57216318d33095"},
		{U7IFIX "5e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "5f", "f4b5256e54973f287cb32cb7ab63493ed73f0596"},
		{U7IFIX "60", "11651c7ab552384e9d022fd4c657cd261783216f"},
		{U7IFIX "61", "75e62157adb26ac349b3f80773eb975c8dc01270"},
		{U7IFIX "62", "dd2e5a69eb988925706c5ab1be4fde3570f9cd43"},
		{U7IFIX "63", "4888d5354d1c2cfa3a13b85225ae36e8a143fbcb"},
		{U7IFIX "64", "ec605673c2e9b95cac1496498ce240b0ec7560a7"},
		{U7IFIX "65", "b5303de8983d74114ea0aaa770f7ea47586fa76f"},
		{U7IFIX "66", "da3a28aad5a15e4dd5937eefb4244ca2a220c1c6"},
		{U7IFIX "67", "0299c8854112cbfac4ea22eb79b180451460a82a"},
		{U7IFIX "68", "212bb7ce61e62d3ab1b71ec16620830c53c2c532"},
		{U7IFIX "69", "c9d36899d6b22ae94c51fef148e346b4eb67c464"},
		{U7IFIX "6a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6b", "61f5f3937173e669edf5534595765841314c2834"},
		{U7IFIX "6c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6d", "a49f33b618a9afd097d271d998ad144f903b79f6"},
		{U7IFIX "6e", "d7d10c949f7f4b59fce0bb65f8ddcccffb7f43df"},
		{U7IFIX "6f", "3c0a35619bc79b5d814136d8cad2bc5e5eb0cab9"},
		{U7IFIX "70", "03460e2bab8683358c6feb1fe82ae4144eab4863"},
		{U7IFIX "71", "cfdd4527e0137467a4c6ddc632c81d538cd3ccab"},
		{U7IFIX "72", "1db9d5a95c477171d807a746e75a572856bdbae1"},
		{U7IFIX "73", "f3941f1f4daedb06769529f89252c4239a63eae0"},
		{U7IFIX "74", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "75", "fe98379209871394d78d9bed388cf5d2f910572b"},
		{U7IFIX "76", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "77", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "78", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "79", "2b5ad3ad7ed6b873b77596df0ce788ab6b4e7b49"},
		{U7IFIX "7a", "708f0f7e556218406dd5ad04c992b16b1f70f2fa"},
		{U7IFIX "7b", "cbe09d7b598e61c32ea5bcc8facb692cfa04fc60"},
		{U7IFIX "7c", "927448f4e5366c30256a1fecaa75ed2b6b7bbeaf"},
		{U7IFIX "7d", "bfba14f9e884fbba246e77828b5477ae6a716101"},
		{U7IFIX "7e", "a48f094837a5b7ff80fdea3c938199f6ffa396e6"},
		{U7IFIX "7f", "91cbad22a24d155443befca76e2d32088047fe92"},
		{U7IFIX "80", "e690f9e499c7667bb5ec27e3d749c97ec8bfcb63"},
		{U7IFIX "81", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "82", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "83", "48460a43e3339267e7c666652e0c46ddff170c38"},
		{U7IFIX "84", "51b7e91624a560104749e88f5bc0af580a7a3845"},
		{U7IFIX "85", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "86", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "87", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "88", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "89", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8b", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"}};

const File2HashMap serpentisle_files{
		{A_SEND, "a784e3fb68822084b17523af4f3ede6d32a4807b"},
		{A_SINTRO, "3c911191ef83502fe6a304db963504a54b0c6bdb"},
		{MAINMUS_AD, "d739fd2bf63484f2e6845e4b850bde6342f539a8"},
		{AMMO, "12d690572b28f2901f7e076a70196ca9392110de"},
		{ARMOR, "6cd6dfda39da05ee802e460d6313f09fb60136d8"},
		{EQUIP, "ed19102d4539da752851abd61d5bc6330b59a051"},
		{FACES_VGA, "fce7beb4914cd62dff4033b0f0d914db106b57ea"},
		{FONTS_VGA, "d852da3709f42ca12a50110ad4fef7f5241fdb69"},
		{GUMPS_VGA, "78db01dd77d637b7ddbcb0e4fbb88941c544323f"},
		{INITGAME, "6851438b9f106713c0e65f6280036e76c89e3176"},
		{INTRO_DAT, "eae097b213ca9e17dafd9755ec0db9e378dd9d2f"},
		{MAINMENU_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{MAINSHP_FLX, "ba11603f3284d27f6ffa3668551064412c9a0af8"},
		{MONSTERS, "8ee769056af1ce90cc632a80c44a1fabe9337eac"},
		{MAINMUS, "a2557010abe3b0f0734cbe14eb1f5d06125f79a6"},
		{OCCLUDE, "9778e4a5f359bee323d663a03fd6784c15bbf28a"},
		{PALETTES_FLX, "7b427540ddbd585fe2f41364b98c4472193b1d72"},
		{PAPERDOL, "f0f36c70660ccf98ef8af3ae7bdeee587b7c0d37"},
		{POINTERS, "501e371f934c92d62a9b05304478b017467499e4"},
		{R_SEND, "0a64e70f833ec93c9a83000cc1f877d44f62e384"},
		{R_SINTRO, "3df365884c93ff408ed5612c1be48064b19fdbe4"},
		{READY, "3b0a1b5280b1957b85e484f204df9906afeac315"},
		{SCHEDULE_DAT, "882783ddc396a6268e191ee44a211ec2da392a48"},
		{SHAPES_VGA, "90362b4c801b2e80d97ce69261fa6cc91496c409"},
		{SHPDIMS, "0f4cdd1682718d61caea197299d9ef33a52e3f06"},
		{SISPEECH, "b3824016ada236fafc6ba5051eec06b86bc1be9e"},
		{SPRITES_VGA, "e2e2a8cfa55506535db1f16e14234aed2fe374cc"},
		{TEXT_FLX, "e3d4a801b6ebb6e9b22905581e42939a5dfbbbbb"},
		{TFA, "8d295622330a3cbefe5531910639a32555f9a85a"},
		{U7CHUNKS, "fc55b795d6bb43342fc49a72cba8e1850e7e2de9"},
		{U7MAP, "4311aa37af9bc53d17e6822e929abd291aadcb9a"},
		{USECODE, "979c47e7005454b3cefb203e9413985fd03e00f9"},
		{WEAPONS, "00e808039535fc36f92255440e3e9bd43791c265"},
		{WGTVOL, "c1d22db3e6ecdb7745716c7679c82cf9ee46b39a"},
		{WIHH, "0bbe120a00c962d826da1b9c4387c0a6b4ae3ed3"},
		{XFORMTBL, "a1f25f60dd75737e26c0a7116d6d64f668fba429"},
		{XMIDI_AD, "23e5be524b21c1b1976341c4adca4ceb18fc1681"},
		{XMIDI_MT, "611cdc42586a65f99e3ec265197764fc4dc523ac"},
		{U7IFIX "00", "ec11ebcd215725607aba701c3e1290876abe4a23"},
		{U7IFIX "01", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "02", "0facb469073f223ecca792dea7fd5e433b043146"},
		{U7IFIX "03", "88a6fb6cb4afd657ec1e8437befef07974ba85ed"},
		{U7IFIX "04", "c71affd8deaabfa1d9cf48de25156f85e169f978"},
		{U7IFIX "05", "9940fc09ee49072393062924557b9000eef23fd5"},
		{U7IFIX "06", "7ee8313e6999aa26b8a20b851b8d8764f42a037e"},
		{U7IFIX "07", "4b96aa546d528e8a929bf07a4f5ebbded1d8bef7"},
		{U7IFIX "08", "40c5d8d38aa982e6000d7e4ff4f1ff679cbdef98"},
		{U7IFIX "09", "7f1f1a7756534fd84f2f939c9e4d80023b9cb1b7"},
		{U7IFIX "0a", "f0bbb381d2fec18f44885d5a1e5fbeb4598a567d"},
		{U7IFIX "0b", "9fb5052fb5303df773c9200baf451a9ff2490f71"},
		{U7IFIX "0c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0e", "0058030bfc7ed68b2ac4eaf1ebe8720db963d865"},
		{U7IFIX "0f", "df2991f26caf0a75c9586005a9cdcebdeb95285e"},
		{U7IFIX "10", "b0e0e65d994ecc28ec3633b40d2cb5f939e29d63"},
		{U7IFIX "11", "cc4531c12a38a9878a40be981f42c63c50e063ea"},
		{U7IFIX "12", "54277ee12781917fea7718dca1af3058065a9026"},
		{U7IFIX "13", "6e3a0cba96dd251f9af12fd17597132702bc5f31"},
		{U7IFIX "14", "0dfd52a9133053f11bb5e3223879e4d50430fcd1"},
		{U7IFIX "15", "1dd41d6e0a896ffd6d2e9bb9661fc3f374f14379"},
		{U7IFIX "16", "725f338ce6510d07b50949b49b0054468c96c13d"},
		{U7IFIX "17", "665e4b9c23cd6ae51658f0fdce53862ff66e4b60"},
		{U7IFIX "18", "a24639d69c3d938a1489e178f2ba814b333be28c"},
		{U7IFIX "19", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1b", "266e15c1353cbb9d1a39eff67c36d8864bf78d7b"},
		{U7IFIX "1c", "a3ce4c89bf92de55dddb0863263b2d0a3c744040"},
		{U7IFIX "1d", "0b17e043ba324f695f981e1862c2217ba467bf77"},
		{U7IFIX "1e", "5de0b3a0ff517f825b0a8aa89be5c3f8884c14c0"},
		{U7IFIX "1f", "1adeb58fc43609d3b7aa17cd524837258f2252be"},
		{U7IFIX "20", "49f6f1342bcbf1272e6915548640b60c39445158"},
		{U7IFIX "21", "f7a5ee5b4b8a7d4475a6976284e7b56fcb5ab6bd"},
		{U7IFIX "22", "f7b27e24f24c1f1e85fd0af2c845d8b06a03ab4a"},
		{U7IFIX "23", "5ccf68aa57a6b019c34270240bfa31c4907011bb"},
		{U7IFIX "24", "eb8fc4d6e69889399fc5bab167dd2783d3572f2f"},
		{U7IFIX "25", "5946aa65996719701714c69924c669f195490fa4"},
		{U7IFIX "26", "bd657cf05d5ca5bc02df35c675b075616afbbe20"},
		{U7IFIX "27", "9208c23679a180d2765b7c37d7c28840e39b8b40"},
		{U7IFIX "28", "2374b44dda8215881d8c8a8afa7e08925a49cb96"},
		{U7IFIX "29", "270bac66d2053360e00e9509e4afa56fb08006b3"},
		{U7IFIX "2a", "1908df70343d7820c97b189e879b208ad326c948"},
		{U7IFIX "2b", "73cf390120811ad7fca273dd702daa25b0566e90"},
		{U7IFIX "2c", "69205d7deb29cf4f15b47bb534f5c281aa586b0e"},
		{U7IFIX "2d", "e039038d03e1aff5539d3913c6babc754d74741b"},
		{U7IFIX "2e", "3c3622c5451f7b0a7f07a009b7aacc332a8c1635"},
		{U7IFIX "2f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "30", "2a35a26cffa65ffe22196f5ebc77a023b677b9a6"},
		{U7IFIX "31", "8006b9212a75aed4a626813fc4ac480b57d8cd4b"},
		{U7IFIX "32", "0c6fac168e1bd72891701078976ade0a98ef3c24"},
		{U7IFIX "33", "963fdda025e05135df180e5baaeba6b083e1c1b9"},
		{U7IFIX "34", "4f56eb8383fcb1e109cdd19b6c8d20f40cd3c2d1"},
		{U7IFIX "35", "64622ccb625e411186bf280b8bef411137133463"},
		{U7IFIX "36", "067586a3cb84d79d885769eb3c9ff15ab2665dee"},
		{U7IFIX "37", "46f1228b46d8e04aaf33836be591cea63c9728c3"},
		{U7IFIX "38", "c9fc5438425403fbd793360ec44568b6e6cc1a42"},
		{U7IFIX "39", "6f3938cc2e3b368e6e619b831be92c87ae87c178"},
		{U7IFIX "3a", "a24be07f574e040b6b2df0482ae4f4c2a61b1cb2"},
		{U7IFIX "3b", "091fd08b275fe1f40ee87ca22b9a82b8ff4dd864"},
		{U7IFIX "3c", "4ea65dba8704252a27f4ff7f7ab27ea4a9890549"},
		{U7IFIX "3d", "d3a1a1ff8d5c7d7ac53bec230a354ebb0280f9e4"},
		{U7IFIX "3e", "c120a30cb4511165a901dedcbaea523b385809c8"},
		{U7IFIX "3f", "4cc31c9134b0ec3edf681f0a369c09fb5456ba9a"},
		{U7IFIX "40", "2e888891124143e5586900909744b484b80a3d20"},
		{U7IFIX "41", "31fbba9e39daafcfbb0c1788ff056c87b9121b25"},
		{U7IFIX "42", "75ffd804aa0ab8bde82c0d13cb6500b13592712c"},
		{U7IFIX "43", "30fe548b12ce0c95fef90391ceea53bc150ab398"},
		{U7IFIX "44", "dc4efa63bcad631ab604ab282102134d8dd8e277"},
		{U7IFIX "45", "d0959558614391cc203fc187d01042467def4acf"},
		{U7IFIX "46", "7d03b1189b0d47fe7b6c0c75f86c52c81e1c0e03"},
		{U7IFIX "47", "8c52aa2fc6f2f5af8a3214f713441de34f602802"},
		{U7IFIX "48", "e1ae2ccc290df882bf1ec84760fc95926a1e5481"},
		{U7IFIX "49", "b8e503849fbc57b5087bffc35e541039e4c27e28"},
		{U7IFIX "4a", "e4132d9d4d1f429cde7d8c75c27166061ccc5fab"},
		{U7IFIX "4b", "7ead3f73bed572e80af4bc94462473dbc0956cb9"},
		{U7IFIX "4c", "a782b13660f5a098480b0df3341b6976eef16bcb"},
		{U7IFIX "4d", "78cd0b09b66592405d897318ae04135ee584a389"},
		{U7IFIX "4e", "6561428ec7790cdbce4db80847c7c9816470c167"},
		{U7IFIX "4f", "eb8430bf275430729251ef2424197a2f9d7e1120"},
		{U7IFIX "50", "d32984f4d3eedcc8ee63d8c943df58c4556eae20"},
		{U7IFIX "51", "8509b8c02330cf698e680ed84bdc495494b2791d"},
		{U7IFIX "52", "0f4e099f2d8387e4a0dd182542a1656b969ff977"},
		{U7IFIX "53", "c9098d7d822c94ad6e9fdc7ad3c95e95b2878dd7"},
		{U7IFIX "54", "dedee0edb215fa553fa7073ed14bbd77a48f6007"},
		{U7IFIX "55", "c44d13299ecb2025b54995e7eb933ab71df88427"},
		{U7IFIX "56", "d878caa9a6f707b6cc2d4259fae6e154e11a5acf"},
		{U7IFIX "57", "ee7ad613da58b49af5d67306af18a07e287911b2"},
		{U7IFIX "58", "80e438ead863cee3ffb306eaf01709fe73feaa44"},
		{U7IFIX "59", "255d54fd722ad364885ec67c812a7f3a87f4aac1"},
		{U7IFIX "5a", "a6e19cccf3019a2913b918161fe2a7f95e62e088"},
		{U7IFIX "5b", "ebd149eccda59593284cd30132862bd954096f64"},
		{U7IFIX "5c", "6936c0cf636593c20e2006daf39692e73daa8050"},
		{U7IFIX "5d", "00635726145e279a9e91d632d114e026e136382a"},
		{U7IFIX "5e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "5f", "c220486e664c42c16daf60367854957c5b18bf68"},
		{U7IFIX "60", "ea916c48ccb43cf0d692622217386c8d23b5dfdd"},
		{U7IFIX "61", "1bbf8def7e4256fa3038b71e2b2e4c568111533d"},
		{U7IFIX "62", "dd2e5a69eb988925706c5ab1be4fde3570f9cd43"},
		{U7IFIX "63", "95e6c1da90503d93bce1b9421b4d4ca927ac966a"},
		{U7IFIX "64", "d29c51e6ad8f73a1409c8993178beffd602bcd02"},
		{U7IFIX "65", "e2d3983066f71fe94cbb274d1d559d1e6e363b7c"},
		{U7IFIX "66", "35613383f33c1c4d5bb71189069b36478c4b83a5"},
		{U7IFIX "67", "53e6d103f36d254e88127ce937530c991d3cd649"},
		{U7IFIX "68", "561472d337975cd8aa7b09277da3fe574f34ae34"},
		{U7IFIX "69", "8136799008e29b583b6e84b8a5de9476a31431d3"},
		{U7IFIX "6a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6b", "7bc9a2231b26b0cca09ef3408d4211b42f816912"},
		{U7IFIX "6c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6d", "a49f33b618a9afd097d271d998ad144f903b79f6"},
		{U7IFIX "6e", "d7d10c949f7f4b59fce0bb65f8ddcccffb7f43df"},
		{U7IFIX "6f", "a65eeaf0aeab5aef7fd046aad17e7e82d048359d"},
		{U7IFIX "70", "64105bd8a45ae284ccd70b07510d2524ac9b4207"},
		{U7IFIX "71", "e6d1c0bceb48ae381c0ee565e712e73d0617bfd3"},
		{U7IFIX "72", "12a060447180c1bba4634da64518d752ae5c9d90"},
		{U7IFIX "73", "31658b70a370e7758ec2ca39b4074033454473b6"},
		{U7IFIX "74", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "75", "fe98379209871394d78d9bed388cf5d2f910572b"},
		{U7IFIX "76", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "77", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "78", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "79", "2b5ad3ad7ed6b873b77596df0ce788ab6b4e7b49"},
		{U7IFIX "7a", "d27549e9ed45c56bb39afaedf2d3d51e45934d7f"},
		{U7IFIX "7b", "a66acd3ed45be14a60a5ae2939428209e28a64c7"},
		{U7IFIX "7c", "a76ccdb32aa62de2d40f8f75d2bdefcb1107bf10"},
		{U7IFIX "7d", "076eef4e7b6ab0d9b533eedb4276078b1c51d551"},
		{U7IFIX "7e", "96d479b62275cd43847a98be7f0584fdb462bb12"},
		{U7IFIX "7f", "1879e8482d88aec76fe6cfb1cb9cb5d29e6a5370"},
		{U7IFIX "80", "e690f9e499c7667bb5ec27e3d749c97ec8bfcb63"},
		{U7IFIX "81", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "82", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "83", "48460a43e3339267e7c666652e0c46ddff170c38"},
		{U7IFIX "84", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "85", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "86", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "87", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "88", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "89", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8b", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"}};

const File2HashMap islaserpiente_files{
		{A_SEND, "a784e3fb68822084b17523af4f3ede6d32a4807b"},
		{A_SINTRO, "3c911191ef83502fe6a304db963504a54b0c6bdb"},
		{MAINMUS_AD, "d739fd2bf63484f2e6845e4b850bde6342f539a8"},
		{AMMO, "12d690572b28f2901f7e076a70196ca9392110de"},
		{ARMOR, "6cd6dfda39da05ee802e460d6313f09fb60136d8"},
		{EQUIP, "ed19102d4539da752851abd61d5bc6330b59a051"},
		{FACES_VGA, "b03042237f090c2fe7661242023480efb601779d"},
		{FONTS_VGA, "d852da3709f42ca12a50110ad4fef7f5241fdb69"},
		{GUMPS_VGA, "cf114059f5a768d4417e2e9abbfa1486ad70f9c3"},
		{INITGAME, "6851438b9f106713c0e65f6280036e76c89e3176"},
		{INTRO_DAT, "7964b030b417b4c80bfee702ef4b467c9d040687"},
		{MAINMENU_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{MAINSHP_FLX, "f426745fd5177895b21c641ff19c42869c79a80c"},
		{MONSTERS, "8ee769056af1ce90cc632a80c44a1fabe9337eac"},
		{MAINMUS, "a2557010abe3b0f0734cbe14eb1f5d06125f79a6"},
		{OCCLUDE, "9778e4a5f359bee323d663a03fd6784c15bbf28a"},
		{PALETTES_FLX, "7b427540ddbd585fe2f41364b98c4472193b1d72"},
		{PAPERDOL, "f0f36c70660ccf98ef8af3ae7bdeee587b7c0d37"},
		{POINTERS, "3a204b59df0006fb536e4f5fcac5ccd217e756b1"},
		{R_SEND, "0a64e70f833ec93c9a83000cc1f877d44f62e384"},
		{R_SINTRO, "3df365884c93ff408ed5612c1be48064b19fdbe4"},
		{READY, "3b0a1b5280b1957b85e484f204df9906afeac315"},
		{SCHEDULE_DAT, "882783ddc396a6268e191ee44a211ec2da392a48"},
		{SHAPES_VGA, "90362b4c801b2e80d97ce69261fa6cc91496c409"},
		{SHPDIMS, "0f4cdd1682718d61caea197299d9ef33a52e3f06"},
		{SISPEECH, "c14df0419c5ce0b8e4fe57e81c9a9732cf5a4de1"},
		{SPRITES_VGA, "1dc0a58517ddc98a73629dfe1948c5ebe0c42ce1"},
		{TEXT_FLX, "f4791de842e8731b0b288670cf7eb78bb93a96aa"},
		{TFA, "8d295622330a3cbefe5531910639a32555f9a85a"},
		{U7CHUNKS, "fc55b795d6bb43342fc49a72cba8e1850e7e2de9"},
		{U7MAP, "4311aa37af9bc53d17e6822e929abd291aadcb9a"},
		{USECODE, "6e3c6a6985efe8f9dffe88980eb3c465ed2313a0"},
		{WEAPONS, "00e808039535fc36f92255440e3e9bd43791c265"},
		{WGTVOL, "c1d22db3e6ecdb7745716c7679c82cf9ee46b39a"},
		{WIHH, "0bbe120a00c962d826da1b9c4387c0a6b4ae3ed3"},
		{XFORMTBL, "a1f25f60dd75737e26c0a7116d6d64f668fba429"},
		{XMIDI_AD, "23e5be524b21c1b1976341c4adca4ceb18fc1681"},
		{XMIDI_MT, "611cdc42586a65f99e3ec265197764fc4dc523ac"},
		{U7IFIX "00", "ec11ebcd215725607aba701c3e1290876abe4a23"},
		{U7IFIX "01", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "02", "0facb469073f223ecca792dea7fd5e433b043146"},
		{U7IFIX "03", "88a6fb6cb4afd657ec1e8437befef07974ba85ed"},
		{U7IFIX "04", "c71affd8deaabfa1d9cf48de25156f85e169f978"},
		{U7IFIX "05", "9940fc09ee49072393062924557b9000eef23fd5"},
		{U7IFIX "06", "7ee8313e6999aa26b8a20b851b8d8764f42a037e"},
		{U7IFIX "07", "4b96aa546d528e8a929bf07a4f5ebbded1d8bef7"},
		{U7IFIX "08", "40c5d8d38aa982e6000d7e4ff4f1ff679cbdef98"},
		{U7IFIX "09", "7f1f1a7756534fd84f2f939c9e4d80023b9cb1b7"},
		{U7IFIX "0a", "f0bbb381d2fec18f44885d5a1e5fbeb4598a567d"},
		{U7IFIX "0b", "9fb5052fb5303df773c9200baf451a9ff2490f71"},
		{U7IFIX "0c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0e", "0058030bfc7ed68b2ac4eaf1ebe8720db963d865"},
		{U7IFIX "0f", "df2991f26caf0a75c9586005a9cdcebdeb95285e"},
		{U7IFIX "10", "b0e0e65d994ecc28ec3633b40d2cb5f939e29d63"},
		{U7IFIX "11", "cc4531c12a38a9878a40be981f42c63c50e063ea"},
		{U7IFIX "12", "54277ee12781917fea7718dca1af3058065a9026"},
		{U7IFIX "13", "6e3a0cba96dd251f9af12fd17597132702bc5f31"},
		{U7IFIX "14", "0dfd52a9133053f11bb5e3223879e4d50430fcd1"},
		{U7IFIX "15", "1dd41d6e0a896ffd6d2e9bb9661fc3f374f14379"},
		{U7IFIX "16", "725f338ce6510d07b50949b49b0054468c96c13d"},
		{U7IFIX "17", "665e4b9c23cd6ae51658f0fdce53862ff66e4b60"},
		{U7IFIX "18", "a24639d69c3d938a1489e178f2ba814b333be28c"},
		{U7IFIX "19", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1b", "266e15c1353cbb9d1a39eff67c36d8864bf78d7b"},
		{U7IFIX "1c", "a3ce4c89bf92de55dddb0863263b2d0a3c744040"},
		{U7IFIX "1d", "0b17e043ba324f695f981e1862c2217ba467bf77"},
		{U7IFIX "1e", "5de0b3a0ff517f825b0a8aa89be5c3f8884c14c0"},
		{U7IFIX "1f", "1adeb58fc43609d3b7aa17cd524837258f2252be"},
		{U7IFIX "20", "49f6f1342bcbf1272e6915548640b60c39445158"},
		{U7IFIX "21", "f7a5ee5b4b8a7d4475a6976284e7b56fcb5ab6bd"},
		{U7IFIX "22", "f7b27e24f24c1f1e85fd0af2c845d8b06a03ab4a"},
		{U7IFIX "23", "5ccf68aa57a6b019c34270240bfa31c4907011bb"},
		{U7IFIX "24", "eb8fc4d6e69889399fc5bab167dd2783d3572f2f"},
		{U7IFIX "25", "5946aa65996719701714c69924c669f195490fa4"},
		{U7IFIX "26", "bd657cf05d5ca5bc02df35c675b075616afbbe20"},
		{U7IFIX "27", "9208c23679a180d2765b7c37d7c28840e39b8b40"},
		{U7IFIX "28", "2374b44dda8215881d8c8a8afa7e08925a49cb96"},
		{U7IFIX "29", "270bac66d2053360e00e9509e4afa56fb08006b3"},
		{U7IFIX "2a", "1908df70343d7820c97b189e879b208ad326c948"},
		{U7IFIX "2b", "73cf390120811ad7fca273dd702daa25b0566e90"},
		{U7IFIX "2c", "69205d7deb29cf4f15b47bb534f5c281aa586b0e"},
		{U7IFIX "2d", "e039038d03e1aff5539d3913c6babc754d74741b"},
		{U7IFIX "2e", "3c3622c5451f7b0a7f07a009b7aacc332a8c1635"},
		{U7IFIX "2f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "30", "2a35a26cffa65ffe22196f5ebc77a023b677b9a6"},
		{U7IFIX "31", "8006b9212a75aed4a626813fc4ac480b57d8cd4b"},
		{U7IFIX "32", "0c6fac168e1bd72891701078976ade0a98ef3c24"},
		{U7IFIX "33", "963fdda025e05135df180e5baaeba6b083e1c1b9"},
		{U7IFIX "34", "4f56eb8383fcb1e109cdd19b6c8d20f40cd3c2d1"},
		{U7IFIX "35", "64622ccb625e411186bf280b8bef411137133463"},
		{U7IFIX "36", "067586a3cb84d79d885769eb3c9ff15ab2665dee"},
		{U7IFIX "37", "46f1228b46d8e04aaf33836be591cea63c9728c3"},
		{U7IFIX "38", "c9fc5438425403fbd793360ec44568b6e6cc1a42"},
		{U7IFIX "39", "6f3938cc2e3b368e6e619b831be92c87ae87c178"},
		{U7IFIX "3a", "a24be07f574e040b6b2df0482ae4f4c2a61b1cb2"},
		{U7IFIX "3b", "091fd08b275fe1f40ee87ca22b9a82b8ff4dd864"},
		{U7IFIX "3c", "4ea65dba8704252a27f4ff7f7ab27ea4a9890549"},
		{U7IFIX "3d", "d3a1a1ff8d5c7d7ac53bec230a354ebb0280f9e4"},
		{U7IFIX "3e", "c120a30cb4511165a901dedcbaea523b385809c8"},
		{U7IFIX "3f", "4cc31c9134b0ec3edf681f0a369c09fb5456ba9a"},
		{U7IFIX "40", "2e888891124143e5586900909744b484b80a3d20"},
		{U7IFIX "41", "31fbba9e39daafcfbb0c1788ff056c87b9121b25"},
		{U7IFIX "42", "75ffd804aa0ab8bde82c0d13cb6500b13592712c"},
		{U7IFIX "43", "30fe548b12ce0c95fef90391ceea53bc150ab398"},
		{U7IFIX "44", "dc4efa63bcad631ab604ab282102134d8dd8e277"},
		{U7IFIX "45", "d0959558614391cc203fc187d01042467def4acf"},
		{U7IFIX "46", "7d03b1189b0d47fe7b6c0c75f86c52c81e1c0e03"},
		{U7IFIX "47", "8c52aa2fc6f2f5af8a3214f713441de34f602802"},
		{U7IFIX "48", "e1ae2ccc290df882bf1ec84760fc95926a1e5481"},
		{U7IFIX "49", "b8e503849fbc57b5087bffc35e541039e4c27e28"},
		{U7IFIX "4a", "e4132d9d4d1f429cde7d8c75c27166061ccc5fab"},
		{U7IFIX "4b", "7ead3f73bed572e80af4bc94462473dbc0956cb9"},
		{U7IFIX "4c", "a782b13660f5a098480b0df3341b6976eef16bcb"},
		{U7IFIX "4d", "78cd0b09b66592405d897318ae04135ee584a389"},
		{U7IFIX "4e", "6561428ec7790cdbce4db80847c7c9816470c167"},
		{U7IFIX "4f", "eb8430bf275430729251ef2424197a2f9d7e1120"},
		{U7IFIX "50", "d32984f4d3eedcc8ee63d8c943df58c4556eae20"},
		{U7IFIX "51", "8509b8c02330cf698e680ed84bdc495494b2791d"},
		{U7IFIX "52", "0f4e099f2d8387e4a0dd182542a1656b969ff977"},
		{U7IFIX "53", "c9098d7d822c94ad6e9fdc7ad3c95e95b2878dd7"},
		{U7IFIX "54", "dedee0edb215fa553fa7073ed14bbd77a48f6007"},
		{U7IFIX "55", "c44d13299ecb2025b54995e7eb933ab71df88427"},
		{U7IFIX "56", "d878caa9a6f707b6cc2d4259fae6e154e11a5acf"},
		{U7IFIX "57", "ee7ad613da58b49af5d67306af18a07e287911b2"},
		{U7IFIX "58", "80e438ead863cee3ffb306eaf01709fe73feaa44"},
		{U7IFIX "59", "255d54fd722ad364885ec67c812a7f3a87f4aac1"},
		{U7IFIX "5a", "a6e19cccf3019a2913b918161fe2a7f95e62e088"},
		{U7IFIX "5b", "ebd149eccda59593284cd30132862bd954096f64"},
		{U7IFIX "5c", "6936c0cf636593c20e2006daf39692e73daa8050"},
		{U7IFIX "5d", "00635726145e279a9e91d632d114e026e136382a"},
		{U7IFIX "5e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "5f", "c220486e664c42c16daf60367854957c5b18bf68"},
		{U7IFIX "60", "ea916c48ccb43cf0d692622217386c8d23b5dfdd"},
		{U7IFIX "61", "1bbf8def7e4256fa3038b71e2b2e4c568111533d"},
		{U7IFIX "62", "dd2e5a69eb988925706c5ab1be4fde3570f9cd43"},
		{U7IFIX "63", "95e6c1da90503d93bce1b9421b4d4ca927ac966a"},
		{U7IFIX "64", "d29c51e6ad8f73a1409c8993178beffd602bcd02"},
		{U7IFIX "65", "e2d3983066f71fe94cbb274d1d559d1e6e363b7c"},
		{U7IFIX "66", "35613383f33c1c4d5bb71189069b36478c4b83a5"},
		{U7IFIX "67", "53e6d103f36d254e88127ce937530c991d3cd649"},
		{U7IFIX "68", "561472d337975cd8aa7b09277da3fe574f34ae34"},
		{U7IFIX "69", "8136799008e29b583b6e84b8a5de9476a31431d3"},
		{U7IFIX "6a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6b", "7bc9a2231b26b0cca09ef3408d4211b42f816912"},
		{U7IFIX "6c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6d", "a49f33b618a9afd097d271d998ad144f903b79f6"},
		{U7IFIX "6e", "d7d10c949f7f4b59fce0bb65f8ddcccffb7f43df"},
		{U7IFIX "6f", "a65eeaf0aeab5aef7fd046aad17e7e82d048359d"},
		{U7IFIX "70", "64105bd8a45ae284ccd70b07510d2524ac9b4207"},
		{U7IFIX "71", "e6d1c0bceb48ae381c0ee565e712e73d0617bfd3"},
		{U7IFIX "72", "12a060447180c1bba4634da64518d752ae5c9d90"},
		{U7IFIX "73", "31658b70a370e7758ec2ca39b4074033454473b6"},
		{U7IFIX "74", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "75", "fe98379209871394d78d9bed388cf5d2f910572b"},
		{U7IFIX "76", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "77", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "78", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "79", "2b5ad3ad7ed6b873b77596df0ce788ab6b4e7b49"},
		{U7IFIX "7a", "d27549e9ed45c56bb39afaedf2d3d51e45934d7f"},
		{U7IFIX "7b", "a66acd3ed45be14a60a5ae2939428209e28a64c7"},
		{U7IFIX "7c", "a76ccdb32aa62de2d40f8f75d2bdefcb1107bf10"},
		{U7IFIX "7d", "076eef4e7b6ab0d9b533eedb4276078b1c51d551"},
		{U7IFIX "7e", "96d479b62275cd43847a98be7f0584fdb462bb12"},
		{U7IFIX "7f", "1879e8482d88aec76fe6cfb1cb9cb5d29e6a5370"},
		{U7IFIX "80", "e690f9e499c7667bb5ec27e3d749c97ec8bfcb63"},
		{U7IFIX "81", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "82", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "83", "48460a43e3339267e7c666652e0c46ddff170c38"},
		{U7IFIX "84", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "85", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "86", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "87", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "88", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "89", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8b", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"}};

const File2HashMap silverseed_files{
		{A_SEND, "a784e3fb68822084b17523af4f3ede6d32a4807b"},
		{A_SINTRO, "3c911191ef83502fe6a304db963504a54b0c6bdb"},
		{MAINMUS_AD, "d739fd2bf63484f2e6845e4b850bde6342f539a8"},
		{AMMO, "e1e55087ccf603ac45b5bca07f30a95feb801242"},
		{ARMOR, "f04e16f33b8e2f014a5cdad3d90454ba1472d50a"},
		{EQUIP, "6e1900e5631f0de34929090dcb09daf923fd87b1"},
		{FACES_VGA, "ebf92c21069d76ad319b7106a786e4f36ca256e3"},
		{FONTS_VGA, "b1720e459ea0dd1dc374e3665b8e8dcecf473bef"},
		{GUMPS_VGA, "78db01dd77d637b7ddbcb0e4fbb88941c544323f"},
		{INITGAME, "94216d0ea59a2e94b7980cda7502246d3038b2e3"},
		{INTRO_DAT, "eae097b213ca9e17dafd9755ec0db9e378dd9d2f"},
		{MAINMENU_TIM, "8f9cc17fdb730044648a0611cfe00b6fc1a46e2b"},
		{MAINSHP_FLX, "1aa3067825879dacf76f36ea205d3cca8b75a122"},
		{MONSTERS, "e6ced2be45695c0259e81d5db4012c5ed445c967"},
		{MAINMUS, "a2557010abe3b0f0734cbe14eb1f5d06125f79a6"},
		{OCCLUDE, "2aec0367547d7faf2710e9e39c0513931e55a6f3"},
		{PALETTES_FLX, "7b427540ddbd585fe2f41364b98c4472193b1d72"},
		{PAPERDOL, "b75f29cd0f2607b40b98e4427f3672bbb6f902bb"},
		{POINTERS, "501e371f934c92d62a9b05304478b017467499e4"},
		{R_SEND, "0a64e70f833ec93c9a83000cc1f877d44f62e384"},
		{R_SINTRO, "3df365884c93ff408ed5612c1be48064b19fdbe4"},
		{READY, "4bb1f6b37e7bb148dd219da12dc628b58f5da385"},
		{SCHEDULE_DAT, "882783ddc396a6268e191ee44a211ec2da392a48"},
		{SHAPES_VGA, "fa58477485aada497e7192667c676f206d33f440"},
		{SHPDIMS, "d3fe3d37ef342927dc87428d456486128c8c7f3b"},
		{SISPEECH, "b3824016ada236fafc6ba5051eec06b86bc1be9e"},
		{SPRITES_VGA, "059918cad8f10e0cfceb8880ec958d4521d9f478"},
		{TEXT_FLX, "93e6ea02a229ee49d6e83d4a62ca6cdb8b97fe52"},
		{TFA, "88659ddb9c246f283035e39da05bdc4e3b23c68a"},
		{U7CHUNKS, "98c0d200867138491f47ba143542540f99cc681f"},
		{U7MAP, "046457b59bb630c59c7ee62312af18c8c0fa77e8"},
		{USECODE, "fd36db018bf6f4326d1c3e9f85555f12f5f14346"},
		{WEAPONS, "ae80301e1c3a418d4672f5c1b92248f97eba4482"},
		{WGTVOL, "c17e5c2b577c8bf0966c0963f8f5a84c7a1800d8"},
		{WIHH, "60492d0b83f4bbe244bf2380fdfa5a5e2fe68b76"},
		{XFORMTBL, "a1f25f60dd75737e26c0a7116d6d64f668fba429"},
		{XMIDI_AD, "23e5be524b21c1b1976341c4adca4ceb18fc1681"},
		{XMIDI_MT, "611cdc42586a65f99e3ec265197764fc4dc523ac"},
		{U7IFIX "00", "ec11ebcd215725607aba701c3e1290876abe4a23"},
		{U7IFIX "01", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "02", "0facb469073f223ecca792dea7fd5e433b043146"},
		{U7IFIX "03", "88a6fb6cb4afd657ec1e8437befef07974ba85ed"},
		{U7IFIX "04", "c71affd8deaabfa1d9cf48de25156f85e169f978"},
		{U7IFIX "05", "9940fc09ee49072393062924557b9000eef23fd5"},
		{U7IFIX "06", "7ee8313e6999aa26b8a20b851b8d8764f42a037e"},
		{U7IFIX "07", "4b96aa546d528e8a929bf07a4f5ebbded1d8bef7"},
		{U7IFIX "08", "40c5d8d38aa982e6000d7e4ff4f1ff679cbdef98"},
		{U7IFIX "09", "7f1f1a7756534fd84f2f939c9e4d80023b9cb1b7"},
		{U7IFIX "0a", "f0bbb381d2fec18f44885d5a1e5fbeb4598a567d"},
		{U7IFIX "0b", "9fb5052fb5303df773c9200baf451a9ff2490f71"},
		{U7IFIX "0c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0d", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "0e", "0058030bfc7ed68b2ac4eaf1ebe8720db963d865"},
		{U7IFIX "0f", "df2991f26caf0a75c9586005a9cdcebdeb95285e"},
		{U7IFIX "10", "b0e0e65d994ecc28ec3633b40d2cb5f939e29d63"},
		{U7IFIX "11", "cc4531c12a38a9878a40be981f42c63c50e063ea"},
		{U7IFIX "12", "54277ee12781917fea7718dca1af3058065a9026"},
		{U7IFIX "13", "6e3a0cba96dd251f9af12fd17597132702bc5f31"},
		{U7IFIX "14", "0dfd52a9133053f11bb5e3223879e4d50430fcd1"},
		{U7IFIX "15", "1dd41d6e0a896ffd6d2e9bb9661fc3f374f14379"},
		{U7IFIX "16", "725f338ce6510d07b50949b49b0054468c96c13d"},
		{U7IFIX "17", "665e4b9c23cd6ae51658f0fdce53862ff66e4b60"},
		{U7IFIX "18", "a24639d69c3d938a1489e178f2ba814b333be28c"},
		{U7IFIX "19", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "1b", "266e15c1353cbb9d1a39eff67c36d8864bf78d7b"},
		{U7IFIX "1c", "a3ce4c89bf92de55dddb0863263b2d0a3c744040"},
		{U7IFIX "1d", "0b17e043ba324f695f981e1862c2217ba467bf77"},
		{U7IFIX "1e", "5de0b3a0ff517f825b0a8aa89be5c3f8884c14c0"},
		{U7IFIX "1f", "1adeb58fc43609d3b7aa17cd524837258f2252be"},
		{U7IFIX "20", "49f6f1342bcbf1272e6915548640b60c39445158"},
		{U7IFIX "21", "f7a5ee5b4b8a7d4475a6976284e7b56fcb5ab6bd"},
		{U7IFIX "22", "f7b27e24f24c1f1e85fd0af2c845d8b06a03ab4a"},
		{U7IFIX "23", "5ccf68aa57a6b019c34270240bfa31c4907011bb"},
		{U7IFIX "24", "eb8fc4d6e69889399fc5bab167dd2783d3572f2f"},
		{U7IFIX "25", "5946aa65996719701714c69924c669f195490fa4"},
		{U7IFIX "26", "bd657cf05d5ca5bc02df35c675b075616afbbe20"},
		{U7IFIX "27", "9208c23679a180d2765b7c37d7c28840e39b8b40"},
		{U7IFIX "28", "2374b44dda8215881d8c8a8afa7e08925a49cb96"},
		{U7IFIX "29", "270bac66d2053360e00e9509e4afa56fb08006b3"},
		{U7IFIX "2a", "1908df70343d7820c97b189e879b208ad326c948"},
		{U7IFIX "2b", "73cf390120811ad7fca273dd702daa25b0566e90"},
		{U7IFIX "2c", "69205d7deb29cf4f15b47bb534f5c281aa586b0e"},
		{U7IFIX "2d", "e039038d03e1aff5539d3913c6babc754d74741b"},
		{U7IFIX "2e", "3c3622c5451f7b0a7f07a009b7aacc332a8c1635"},
		{U7IFIX "2f", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "30", "2a35a26cffa65ffe22196f5ebc77a023b677b9a6"},
		{U7IFIX "31", "8006b9212a75aed4a626813fc4ac480b57d8cd4b"},
		{U7IFIX "32", "0c6fac168e1bd72891701078976ade0a98ef3c24"},
		{U7IFIX "33", "963fdda025e05135df180e5baaeba6b083e1c1b9"},
		{U7IFIX "34", "4f56eb8383fcb1e109cdd19b6c8d20f40cd3c2d1"},
		{U7IFIX "35", "64622ccb625e411186bf280b8bef411137133463"},
		{U7IFIX "36", "067586a3cb84d79d885769eb3c9ff15ab2665dee"},
		{U7IFIX "37", "46f1228b46d8e04aaf33836be591cea63c9728c3"},
		{U7IFIX "38", "c9fc5438425403fbd793360ec44568b6e6cc1a42"},
		{U7IFIX "39", "6f3938cc2e3b368e6e619b831be92c87ae87c178"},
		{U7IFIX "3a", "a24be07f574e040b6b2df0482ae4f4c2a61b1cb2"},
		{U7IFIX "3b", "091fd08b275fe1f40ee87ca22b9a82b8ff4dd864"},
		{U7IFIX "3c", "4ea65dba8704252a27f4ff7f7ab27ea4a9890549"},
		{U7IFIX "3d", "d3a1a1ff8d5c7d7ac53bec230a354ebb0280f9e4"},
		{U7IFIX "3e", "c120a30cb4511165a901dedcbaea523b385809c8"},
		{U7IFIX "3f", "4cc31c9134b0ec3edf681f0a369c09fb5456ba9a"},
		{U7IFIX "40", "2e888891124143e5586900909744b484b80a3d20"},
		{U7IFIX "41", "31fbba9e39daafcfbb0c1788ff056c87b9121b25"},
		{U7IFIX "42", "75ffd804aa0ab8bde82c0d13cb6500b13592712c"},
		{U7IFIX "43", "30fe548b12ce0c95fef90391ceea53bc150ab398"},
		{U7IFIX "44", "dc4efa63bcad631ab604ab282102134d8dd8e277"},
		{U7IFIX "45", "d0959558614391cc203fc187d01042467def4acf"},
		{U7IFIX "46", "7d03b1189b0d47fe7b6c0c75f86c52c81e1c0e03"},
		{U7IFIX "47", "8c52aa2fc6f2f5af8a3214f713441de34f602802"},
		{U7IFIX "48", "e1ae2ccc290df882bf1ec84760fc95926a1e5481"},
		{U7IFIX "49", "b8e503849fbc57b5087bffc35e541039e4c27e28"},
		{U7IFIX "4a", "e4132d9d4d1f429cde7d8c75c27166061ccc5fab"},
		{U7IFIX "4b", "7ead3f73bed572e80af4bc94462473dbc0956cb9"},
		{U7IFIX "4c", "a782b13660f5a098480b0df3341b6976eef16bcb"},
		{U7IFIX "4d", "78cd0b09b66592405d897318ae04135ee584a389"},
		{U7IFIX "4e", "6561428ec7790cdbce4db80847c7c9816470c167"},
		{U7IFIX "4f", "eb8430bf275430729251ef2424197a2f9d7e1120"},
		{U7IFIX "50", "d32984f4d3eedcc8ee63d8c943df58c4556eae20"},
		{U7IFIX "51", "8509b8c02330cf698e680ed84bdc495494b2791d"},
		{U7IFIX "52", "0f4e099f2d8387e4a0dd182542a1656b969ff977"},
		{U7IFIX "53", "c9098d7d822c94ad6e9fdc7ad3c95e95b2878dd7"},
		{U7IFIX "54", "dedee0edb215fa553fa7073ed14bbd77a48f6007"},
		{U7IFIX "55", "c44d13299ecb2025b54995e7eb933ab71df88427"},
		{U7IFIX "56", "d878caa9a6f707b6cc2d4259fae6e154e11a5acf"},
		{U7IFIX "57", "ee7ad613da58b49af5d67306af18a07e287911b2"},
		{U7IFIX "58", "80e438ead863cee3ffb306eaf01709fe73feaa44"},
		{U7IFIX "59", "255d54fd722ad364885ec67c812a7f3a87f4aac1"},
		{U7IFIX "5a", "a6e19cccf3019a2913b918161fe2a7f95e62e088"},
		{U7IFIX "5b", "ebd149eccda59593284cd30132862bd954096f64"},
		{U7IFIX "5c", "6936c0cf636593c20e2006daf39692e73daa8050"},
		{U7IFIX "5d", "00635726145e279a9e91d632d114e026e136382a"},
		{U7IFIX "5e", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "5f", "c220486e664c42c16daf60367854957c5b18bf68"},
		{U7IFIX "60", "ea916c48ccb43cf0d692622217386c8d23b5dfdd"},
		{U7IFIX "61", "1bbf8def7e4256fa3038b71e2b2e4c568111533d"},
		{U7IFIX "62", "dd2e5a69eb988925706c5ab1be4fde3570f9cd43"},
		{U7IFIX "63", "95e6c1da90503d93bce1b9421b4d4ca927ac966a"},
		{U7IFIX "64", "d29c51e6ad8f73a1409c8993178beffd602bcd02"},
		{U7IFIX "65", "e2d3983066f71fe94cbb274d1d559d1e6e363b7c"},
		{U7IFIX "66", "35613383f33c1c4d5bb71189069b36478c4b83a5"},
		{U7IFIX "67", "53e6d103f36d254e88127ce937530c991d3cd649"},
		{U7IFIX "68", "561472d337975cd8aa7b09277da3fe574f34ae34"},
		{U7IFIX "69", "8136799008e29b583b6e84b8a5de9476a31431d3"},
		{U7IFIX "6a", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6b", "7bc9a2231b26b0cca09ef3408d4211b42f816912"},
		{U7IFIX "6c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "6d", "a49f33b618a9afd097d271d998ad144f903b79f6"},
		{U7IFIX "6e", "d7d10c949f7f4b59fce0bb65f8ddcccffb7f43df"},
		{U7IFIX "6f", "a65eeaf0aeab5aef7fd046aad17e7e82d048359d"},
		{U7IFIX "70", "64105bd8a45ae284ccd70b07510d2524ac9b4207"},
		{U7IFIX "71", "e6d1c0bceb48ae381c0ee565e712e73d0617bfd3"},
		{U7IFIX "72", "12a060447180c1bba4634da64518d752ae5c9d90"},
		{U7IFIX "73", "31658b70a370e7758ec2ca39b4074033454473b6"},
		{U7IFIX "74", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "75", "fe98379209871394d78d9bed388cf5d2f910572b"},
		{U7IFIX "76", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "77", "3031444a6fd0305aba756a49ee0b3ef9b21f73f9"},
		{U7IFIX "78", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "79", "2b5ad3ad7ed6b873b77596df0ce788ab6b4e7b49"},
		{U7IFIX "7a", "d27549e9ed45c56bb39afaedf2d3d51e45934d7f"},
		{U7IFIX "7b", "a66acd3ed45be14a60a5ae2939428209e28a64c7"},
		{U7IFIX "7c", "a76ccdb32aa62de2d40f8f75d2bdefcb1107bf10"},
		{U7IFIX "7d", "076eef4e7b6ab0d9b533eedb4276078b1c51d551"},
		{U7IFIX "7e", "96d479b62275cd43847a98be7f0584fdb462bb12"},
		{U7IFIX "7f", "1879e8482d88aec76fe6cfb1cb9cb5d29e6a5370"},
		{U7IFIX "80", "e690f9e499c7667bb5ec27e3d749c97ec8bfcb63"},
		{U7IFIX "81", "e78bb95c9a063917026ff5b63f090e830515e64e"},
		{U7IFIX "82", "a8f4741bfa13c359c9017a1a994c0116af573357"},
		{U7IFIX "83", "e37127889fb9883bdefab5d1384b50d3ef392c7e"},
		{U7IFIX "84", "02271357d495cb777c6533675e48e710b24775b0"},
		{U7IFIX "85", "38e28064a2ca2dcd8e5ee03b3d54ca3c8f4c5035"},
		{U7IFIX "86", "945aaa09786ba844bfc43cee9f2ab9900a95662a"},
		{U7IFIX "87", "688f28b6eb521bfb99ad1e6a3c75e42a0891eb85"},
		{U7IFIX "88", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "89", "0c8aa95b296c044c994528a3646b6d2350c05b7a"},
		{U7IFIX "8a", "bd4efc8e929ae198c9ab235ebc76754d5e783bb9"},
		{U7IFIX "8b", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8c", "f86d8c1e20b53088e447dba4ef9c71cae1d23c86"},
		{U7IFIX "8d", "dff19d2dbd87cc97703bcef3d7b26f4b4230d0de"},
		{U7IFIX "8e", "283f5bd025894742f55dc49d4cb104e08a7f015c"},
		{U7IFIX "8f", "79b77743e96eb7409c5c3e81de3ebf2f0a4eada0"}};
