/*
 * Copyright (c) 2016 Meltytech, LLC
 * Author: Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.1
import QtQuick.Dialogs 1.1
import QtQuick.Controls 1.1
import QtQuick.Layouts 1.0
import QtQuick.Window 2.1
import Shotcut.Controls 1.0 as Shotcut
import org.shotcut.qml 1.0 as Shotcut

Item {
    id: lut3dRoot
    width: 350
    height: 100
    property string settingsOpenPath: settings.openPath

    SystemPalette { id: activePalette; colorGroup: SystemPalette.Active }
    Shotcut.File { id: lutFile }

    // This signal is used to workaround context properties not available in
    // the FileDialog onAccepted signal handler on Qt 5.5.
    signal fileOpened(string path)
    onFileOpened: settings.openPath = path

    Component.onCompleted: {
        var resource = filter.get('av.file')
        lutFile.url = resource

        if (filter.isNew) {
            interpolationCombo.currentIndex = 1
        } else {
            interpolationCombo.currentIndex = interpolationCombo.valueToIndex()
        }

        if (lutFile.exists()) {
            fileLabel.text = lutFile.fileName
            fileLabelTip.text = lutFile.url
        } else {
            console.log('lutFile.url = ' + lutFile.url)
            fileLabel.text = qsTr("No File Loaded")
            fileLabel.color = 'red'
            fileLabelTip.text = qsTr('No 3D LUT file loaded.\nClick "Open" to load a file.')
        }
    }

    FileDialog {
        id: fileDialog
        modality: Qt.WindowModal
        selectMultiple: false
        selectFolder: false
        folder: settingsOpenPath
        nameFilters: ['3D-LUT Files (*.3dl *.cube *.dat *.m3d)', 'AfterEffects (*.3dl)', 'Iridas (*.cube)', 'DaVinci (*.dat)', 'Pandora (*.m3d)', 'All Files (*)']
        selectedNameFilter: nameFilters[0]
        onAccepted: {
            lutFile.url = fileDialog.fileUrl
            lut3dRoot.fileOpened(lutFile.path)
            fileLabel.text = lutFile.fileName
            fileLabel.color = activePalette.text
            fileLabelTip.text = lutFile.url
            filter.set('av.file', lutFile.url)
        }
    }

    GridLayout {
        columns: 3
        anchors.fill: parent
        anchors.margins: 8

        Button {
            id: openButton
            text: qsTr('Open...')
            Layout.alignment: Qt.AlignRight
            onClicked: {
                fileDialog.selectExisting = true
                fileDialog.title = qsTr( "Open 3D LUT File" )
                fileDialog.open()
            }
        }
        Label {
            id: fileLabel
            Layout.columnSpan: 2
            Layout.fillWidth: true
            Shotcut.ToolTip { id: fileLabelTip }
        }

        Label {
            text: qsTr('Interpolation')
            Layout.alignment: Qt.AlignRight
        }
        ComboBox {
            id: interpolationCombo
            implicitWidth: 180
            model: [qsTr('Nearest'), qsTr('Trilinear'), qsTr('Tetrahedral')]
            property var values: ['nearest', 'trilinear', 'tetrahedral']
            onCurrentIndexChanged: filter.set('av.interp', values[currentIndex])

            function valueToIndex() {
                var w = filter.get('av.interp')
                for (var i = 0; i < values.length; ++i)
                    if (values[i] === w) break;
                if (i === values.length) i = 1;
                return i;
            }
        }
        Shotcut.UndoButton {
            onClicked: interpolationCombo.currentIndex = 1
        }

        Item {
            Layout.fillHeight: true
            Layout.columnSpan: 3
        }
    }
}
