/**
*** \file informationitemoptions.h
*** \brief Information item options classes.
*** \details Classes for setting information item options.
**/

/*
** This file is part of moonphase.
** Copyright (C) 2014-2018 by Alan Wise <awisesoftware@gmail.com>.
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef   INFORMATIONITEMOPTIONS_H
/**
*** \brief informationitemoptions.h identifier.
*** \details Identifier for informationitemoptions.h.
**/
#define   INFORMATIONITEMOPTIONS_H


/****
*****
***** INCLUDES
*****
****/

#include  "datetime.h"

#include  <QColor>
#include  <QFont>


/****
*****
***** DEFINES
*****
****/


/****
*****
***** DATA TYPES
*****
****/

/**
*** \brief Set item options base class.
*** \details Base class for setting item options.
**/
class INFORMATIONITEMOPTIONS_C
{
  public:
    /**
    *** \brief Constructor.
    *** \details Constructor.
    **/
    INFORMATIONITEMOPTIONS_C(void);
    /**
    *** \brief Copy constructor.
    *** \details Copy constructor.
    *** \param Options Option data.
    **/
    INFORMATIONITEMOPTIONS_C(INFORMATIONITEMOPTIONS_C const &Options);

    /**
    *** \brief Destructor.
    *** \details Destructor.
    **/
    ~INFORMATIONITEMOPTIONS_C(void);

    /**
    *** \brief Convert to DATETIMEOPTIONS_T format.
    *** \details Converts from the INFORMATIONITEMOPTIONS_C format to the 
    ****  DATETIMEOPTIONS_T format.
    *** \param DateTimeOptions DATETIMEOPTIONS_T data.
    **/
    void Convert(DATETIMEOPTIONS_T &DateTimeOptions);

    /**
    *** \brief Return 24 hour format flag.
    *** \details Returns the 24 hour format flag.
    *** \retval true Use 24 hour format.
    *** \retval false Use 12 hour (am/pm) format.
    **/
    bool Get24HourFormatFlag(void) const;

    /**
    *** \brief Return 4 digit year format flag.
    *** \details Returns the 4 digit year format flag.
    *** \retval true Use 4 digit year format.
    *** \retval false Use 2 digit year format.
    **/
    bool Get4DigitYearFlag(void) const;

    /**
    *** \brief Return long day of the week format flag.
    *** \details Returns the long day of the week format flag.
    *** \retval true Use long day of the week format (Sunday).
    *** \retval false Use short day of the week format (Sun.).
    **/
    bool GetLongDayOfWeekFormatFlag(void) const;

    /**
    *** \brief Return month format flag.
    *** \details Returns the month format flag.
    *** \retval true Use long month format (January).
    *** \retval false Use short month format (Jan.).
    **/
    bool GetLongMonthFormatFlag(void) const;

    /**
    *** \brief Return show the day of the week flag.
    *** \details Returns the show the day of the week flag.
    *** \retval true Show the day of the week.
    *** \retval false Don't show the day of the week.
    **/
    bool GetShowDayOfWeekFlag(void) const;

    /**
    *** \brief Return show the seconds flag.
    *** \details Returns the show the seconds flag.
    *** \retval true Show the seconds.
    *** \retval false Don't show the seconds.
    **/
    bool GetShowSecondsFlag(void) const;

    /**
    *** \brief Return unit or format index.
    *** \details Returns the index of the display unit or the date/time format
    ***   index.
    *** \returns Unit or format index.
    **/
    int GetUnitFormatIndex(void) const;

    /**
    *** \brief Set 24 hour format flag.
    *** \details Sets the 24 hour format flag.
    *** \param Flag true - Use 24 hour format,\n
    ***   false - Use 12 hour (am/pm) format.
    **/
    void Set24HourFormatFlag(bool Flag);

    /**
    *** \brief Set 4 digit year format flag.
    *** \details Sets the 4 digit year format flag.
    *** \param Flag true - Use 4 digit year format,\n
    ***   false - Use 2 digit year format.
    **/
    void Set4DigitYearFlag(bool Flag);

    /**
    *** \brief Set long day of the week format flag.
    *** \details Sets the long day of the week format flag.
    *** \param Flag true - Use long day of the week format (Sunday),\n
    ***   false - Use short day of the week format (Sun.).
    **/
    void SetLongDayOfWeekFormatFlag(bool Flag);

    /**
    *** \brief Set long month format flag.
    *** \details Sets the long month format flag.
    *** \param Flag true - Use long month format (January),\n
    ***   false - Use short month format (Jan.).
    **/
    void SetLongMonthFormatFlag(bool Flag);

    /**
    *** \brief Set show the day of the week flag.
    *** \details Sets the show the day of the week flag.
    *** \param Flag true - Show the day of the week,\n
    ***   false - Don't show the day of the week.
    **/
    void SetShowDayOfWeekFlag(bool Flag);

    /**
    *** \brief Set show the seconds flag.
    *** \details Sets the show the seconds flag.
    *** \param Flag true - Show the seconds,\n
    ***   false - Don't show the seconds.
    **/
    void SetShowSecondsFlag(bool Flag);

    /**
    *** \brief Set unit or format index.
    *** \details Sets the index of the display unit or the index of the
    ***   date/time format index.
    *** \param Index Unit or format index.
    **/
    void SetUnitFormatIndex(int Index);

    /**
    *** \brief Assignment operator (const) (=)
    *** \details Assignment operator (const) (=)
    *** \param RHS Object on right hand side of operator.
    *** \return Copy of data.
    **/
    INFORMATIONITEMOPTIONS_C & operator=(INFORMATIONITEMOPTIONS_C const &RHS);

    /**
    *** \brief == operator.
    *** \details Equality operator.
    *** \param RHS INFORMATIONITEMOPTIONS_C on right hand side.
    *** \retval true INFORMATIONITEMOPTIONS_C are equivalent.
    *** \retval false INFORMATIONITEMOPTIONS_C are not equivalent.
    **/
    bool operator==(INFORMATIONITEMOPTIONS_C const &RHS) const;

    /**
    *** \brief != operator.
    *** \details Inequality operator.
    *** \param RHS INFORMATIONITEMOPTIONS_C on right hand side.
    *** \retval true INFORMATIONITEMOPTIONS_C are not equivalent.
    *** \retval false INFORMATIONITEMOPTIONS_C are equivalent.
    **/
    bool operator!=(INFORMATIONITEMOPTIONS_C const &RHS) const;

  private:
    /**
    *** \brief Date/time options.
    *** \details Options for the date/time print functions.
    **/
    DATETIMEOPTIONS_T m_DateTimeOptions;

    /**
    *** \brief Unit or format index.
    *** \details Index of the display unit or index of the date/time format.
    **/
    int m_UnitFormatIndex;
};

/**
*** \brief Set information panel item options class.
*** \details Class for setting information panel item options.
**/
class INFORMATIONPANELITEMOPTIONS_C : public INFORMATIONITEMOPTIONS_C
{
  public:
    /**
    *** \brief Constructor.
    *** \details Constructor.
    **/
    INFORMATIONPANELITEMOPTIONS_C(void);

    /**
    *** \brief Destructor.
    *** \details Destructor.
    **/
    ~INFORMATIONPANELITEMOPTIONS_C(void);

    /**
    *** \brief == operator.
    *** \details Equality operator.
    *** \param RHS INFORMATIONPANELITEMOPTIONS_C on right hand side.
    *** \retval true INFORMATIONPANELITEMOPTIONS_C are equivalent.
    *** \retval false INFORMATIONPANELITEMOPTIONS_C are not equivalent.
    **/
    bool operator==(INFORMATIONPANELITEMOPTIONS_C const &RHS) const;

    /**
    *** \brief != operator.
    *** \details Inequality operator.
    *** \param RHS INFORMATIONPANELITEMOPTIONS_C on right hand side.
    *** \retval true INFORMATIONPANELITEMOPTIONS_C are not equivalent.
    *** \retval false INFORMATIONPANELITEMOPTIONS_C are equivalent.
    **/
    bool operator!=(INFORMATIONPANELITEMOPTIONS_C const &RHS) const;

    /**
    *** \brief Build a style sheet string.
    *** \details Build a style sheet string using text and background colors.
    *** \returns Style sheet string.
    **/
    QString BuildStyleSheetString(void) const;

    /**
    *** \brief Return background color.
    *** \details Returns the color of the background.
    *** \returns Background color.
    **/
    QColor GetBackgroundColor(void) const;

    /**
    *** \brief Return font.
    *** \details Returns the font.
    *** \returns Font.
    **/
    QFont GetFont(void) const;

    /**
    *** \brief Return margin.
    *** \details Returns the margin.
    *** \returns Margin.
    **/
    int GetMargin(void) const;

    /**
    *** \brief Return text color.
    *** \details Returns the color of the text.
    *** \returns Text color.
    **/
    QColor GetTextColor(void) const;

    /**
    *** \brief Return use system background color flag.
    *** \details Returns the use system background color flag.
    *** \retval true Use system background color.
    *** \retval false Use user defined color.
    **/
    bool GetUseSystemBackgroundColorFlag(void) const;

    /**
    *** \brief Return use system font flag
    *** \details Returns the use system font flag.
    *** \retval true Use system font.
    *** \retval false Use user defined font.
    **/
    bool GetUseSystemFontFlag(void) const;

    /**
    *** \brief Return use system text color flag.
    *** \details Returns the use system text color flag.
    *** \retval true Use system text color.
    *** \retval false Use user defined color.
    **/
    bool GetUseSystemTextColorFlag(void) const;

    /**
    *** \brief Set background color.
    *** \details Sets the color of the background.
    *** \param Color Background color.
    **/
    void SetBackgroundColor(QColor Color);

    /**
    *** \brief Set font.
    *** \details Sets the font.
    *** \param Font Font.
    **/
    void SetFont(QFont Font);

    /**
    *** \brief Set margin.
    *** \details Sets the margin.
    *** \param Margin Margin.
    **/
    void SetMargin(int Margin);

    /**
    *** \brief Set text color.
    *** \details Sets the color of the text.
    *** \param Color Text color.
    **/
    void SetTextColor(QColor Color);

    /**
    *** \brief Set use system background color flag.
    *** \details Sets the use system background color flag.
    *** \param Flag true - Use system background color,\n
    ***   false - Use user defined color.
    **/
    void SetUseSystemBackgroundColorFlag(bool Flag);

    /**
    *** \brief Sets the use system font flag.
    *** \details Sets the use system font flag.
    *** \param Flag true - Use system background font,\n
    ***   false - Use user defined font.
    **/
    void SetUseSystemFontFlag(bool Flag);

    /**
    *** \brief Sets the use system text color flag.
    *** \details Sets the use system text color flag.
    *** \param Flag true - Use system text color,\n
    ***   false - Use user defined color.
    **/
    void SetUseSystemTextColorFlag(bool Flag);

  private:
    /**
    *** \brief Background color.
    *** \details The color of the background.
    **/
    QColor m_BackgroundColor;

    /**
    *** \brief Font.
    *** \details Text font.
    **/
    QFont m_Font;

    /**
    *** \brief Margin.
    *** \details The pad area around the text.
    **/
    int m_Margin;

    /**
    *** \brief Text color.
    *** \details The color of the text.
    **/
    QColor m_TextColor;

    /**
    *** \brief Use system background color flag.
    *** \details Use the system background color flag.
    **/
    bool m_UseSystemBackgroundColorFlag;

    /**
    *** \brief Use system font flag.
    *** \details Use the system font flag.
    **/
    bool m_UseSystemFontFlag;

    /**
    *** \brief Use system text color flag.
    *** \details Use the system text color flag.
    **/
    bool m_UseSystemTextColorFlag;
};

/**
*** \brief Set tool tip item options class.
*** \details Class for setting tool tip item options.
**/
class TOOLTIPITEMOPTIONS_C : public INFORMATIONITEMOPTIONS_C
{
  public:
    /**
    *** \brief Constructor.
    *** \details Constructor.
    **/
    TOOLTIPITEMOPTIONS_C(void);

    /**
    *** \brief Destructor.
    *** \details Destructor.
    **/
    ~TOOLTIPITEMOPTIONS_C(void);
};


/****
*****
***** DATA
*****
****/


/****
*****
***** VARIABLES
*****
****/


/****
*****
***** PROTOTYPES
*****
****/


#endif    /* INFORMATIONITEMOPTIONS_H */


/**
*** informationitemoptions.h
**/
