/*
 * $RCSfile: PropertyChangeEventJAI.java,v $
 *
 * Copyright (c) 2005 Sun Microsystems, Inc. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * $Revision: 1.1 $
 * $Date: 2005/02/11 04:57:16 $
 * $State: Exp $
 */
package com.lightcrafts.mediax.jai;

import java.beans.PropertyChangeEvent;

/**
 * A class instances of which represent Java Bean-style events emitted by
 * JAI objects.  This class definition adds no functionality to that provided
 * by the superclass.  The significance of the derivation is that bean events
 * will be easily identifiable as having been generated by JAI classes by
 * virtue of their being instances of this event class.  Note that this does
 * not prevent JAI properties from colliding with other Java Bean properties
 * in the Bean property name space.
 *
 * @since JAI 1.1
 */
public class PropertyChangeEventJAI extends PropertyChangeEvent {
    /**
     * The case-retained property name as supplied to the constructor.
     */
    private String originalPropertyName;

    /**
     * Constructs a <code>PropertyChangeEventJAI</code>.
     * <code>propertyName</code> is forced to lower case; all other
     * parameters are passed unmodified to the superclass constructor.
     * The original property name may be obtained by invoking
     * <code>getOriginalPropertyName()</code>.
     *
     * @exception NullPointerException if <code>propertyName</code> is
     *            <code>null</code>.
     * @exception IllegalArgumentException if <code>source</code> is
     *            <code>null</code> or if <code>oldValue</code> and
     *            <code>newValue</code> are both <code>null</code>.
     */
    public PropertyChangeEventJAI(Object source,
				  String propertyName,
				  Object oldValue,
				  Object newValue) {
        super(source, propertyName.toLowerCase(), oldValue, newValue);

        if(source == null) {
            throw new IllegalArgumentException(JaiI18N.getString("PropertyChangeEventJAI0"));
        } else if(oldValue == null && newValue == null) {
            throw new IllegalArgumentException(JaiI18N.getString("PropertyChangeEventJAI1"));
        }

        originalPropertyName = propertyName.equals(getPropertyName()) ?
            getPropertyName() : propertyName;
    }

    /**
     * Returns the value of <code>propertyName</code> originally passed to
     * the class constructor.  This name has its case retained.
     */
    public String getOriginalPropertyName() {
        return originalPropertyName;
    }
}
