
/*

  KLayout Layout Viewer
  Copyright (C) 2006-2016 Matthias Koefferlein

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


#include "layLayoutViewConfigPages.h"
#include "layConfig.h"
#include "layConverters.h"

#include "LayoutViewConfigPage.h"
#include "LayoutViewConfigPage1.h"
#include "LayoutViewConfigPage2a.h"
#include "LayoutViewConfigPage2b.h"
#include "LayoutViewConfigPage2c.h"
#include "LayoutViewConfigPage3a.h"
#include "LayoutViewConfigPage3b.h"
#include "LayoutViewConfigPage3c.h"
#include "LayoutViewConfigPage3f.h"
#include "LayoutViewConfigPage4.h"
#include "LayoutViewConfigPage5.h"
#include "LayoutViewConfigPage6.h"
#include "LayoutViewConfigPage7.h"

#include "laySelectStippleForm.h"

#include "layWidgets.h"
#include "layFileDialog.h"
#include "dbHershey.h"

#include <QtGui/QColorDialog>
#include <QtGui/QPixmap>
#include <QtGui/QPainter>

namespace lay
{

// ------------------------------------------------------------
//  LayoutConfigPage implementation
//  The configuration pages are declared via a "dummy" plugin

LayoutViewConfigPage::LayoutViewConfigPage (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage::~LayoutViewConfigPage ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage::setup (lay::Plugin *root)
{
  lay::ColorConverter cc;
  QColor color;

  root->config_get (cfg_background_color, color, cc);
  mp_ui->bkgnd_color_pb->set_color (color);
}

void 
LayoutViewConfigPage::commit (lay::Plugin *root)
{
  lay::ColorConverter cc;
  root->config_set (cfg_background_color, mp_ui->bkgnd_color_pb->get_color (), cc);
}

// ------------------------------------------------------------
//  LayoutConfigPage1 implementation

LayoutViewConfigPage1::LayoutViewConfigPage1 (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage1 ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage1::~LayoutViewConfigPage1 ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage1::setup (lay::Plugin *root)
{
  lay::ColorConverter cc;

  QColor color;
  int ctx_dimming = 0;
  bool ctx_hollow = false;

  root->config_get (cfg_ctx_color, color, cc);
  mp_ui->ctx_color_pb->set_color (color);

  root->config_get (cfg_ctx_dimming, ctx_dimming);
  mp_ui->ctx_dimming_spinbx->setValue (ctx_dimming);

  root->config_get (cfg_ctx_hollow, ctx_hollow);
  mp_ui->ctx_hollow_cbx->setChecked (ctx_hollow);

  QColor ccolor;
  int cctx_dimming = 0;
  bool cctx_hollow = false;
  bool cctx_enabled = false;

  root->config_get (cfg_child_ctx_color, ccolor, cc);
  mp_ui->cctx_color_pb->set_color (ccolor);

  root->config_get (cfg_child_ctx_dimming, cctx_dimming);
  mp_ui->cctx_dimming_spinbx->setValue (cctx_dimming);

  root->config_get (cfg_child_ctx_hollow, cctx_hollow);
  mp_ui->cctx_hollow_cbx->setChecked (cctx_hollow);

  root->config_get (cfg_child_ctx_enabled, cctx_enabled);
  mp_ui->cctx_grp->setChecked (cctx_enabled);

  double aw = 10.0;
  bool am = false;

  root->config_get (cfg_abstract_mode_width, aw);
  mp_ui->abstract_mode_width_le->setText (tl::to_qstring (tl::to_string (aw)));

  root->config_get (cfg_abstract_mode_enabled, am);
  mp_ui->abstract_mode_grp->setChecked (am);
}

void 
LayoutViewConfigPage1::commit (lay::Plugin *root)
{
  lay::ColorConverter cc;

  root->config_set (cfg_ctx_color, mp_ui->ctx_color_pb->get_color (), cc);
  root->config_set (cfg_ctx_dimming, mp_ui->ctx_dimming_spinbx->value ());
  root->config_set (cfg_ctx_hollow, mp_ui->ctx_hollow_cbx->isChecked ());

  root->config_set (cfg_child_ctx_color, mp_ui->cctx_color_pb->get_color (), cc);
  root->config_set (cfg_child_ctx_dimming, mp_ui->cctx_dimming_spinbx->value ());
  root->config_set (cfg_child_ctx_hollow, mp_ui->cctx_hollow_cbx->isChecked ());
  root->config_set (cfg_child_ctx_enabled, mp_ui->cctx_grp->isChecked ());

  root->config_set (cfg_abstract_mode_enabled, mp_ui->abstract_mode_grp->isChecked ());

  double w = 10.0;
  tl::from_string (tl::to_string (mp_ui->abstract_mode_width_le->text ()), w);
  root->config_set (cfg_abstract_mode_width, w);
  if (w <= 0.0) {
    throw tl::Exception (tl::to_string (QObject::tr ("Invalid abstract mode border with - must be larger than 0")));
  }
}

// ------------------------------------------------------------
//  LayoutConfigPage2a implementation

LayoutViewConfigPage2a::LayoutViewConfigPage2a (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage2a ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage2a::~LayoutViewConfigPage2a ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage2a::setup (lay::Plugin *root)
{
  QColor color;
  root->config_get (cfg_cell_box_color, color, ColorConverter ());
  mp_ui->cell_box_color_pb->set_color (color);

  bool flag = false;
  root->config_get (cfg_cell_box_text_transform, flag);
  mp_ui->cell_xform_text_cbx->setChecked (flag);

  root->config_get (cfg_cell_box_visible, flag);
  mp_ui->cell_group->setChecked (flag);

  int font = 0;
  root->config_get (cfg_cell_box_text_font, font);
  mp_ui->cell_font_cb->clear ();
  if (mp_ui->cell_font_cb->count () == 0) {
    std::vector<std::string> ff = db::Hershey::font_names ();
    for (std::vector<std::string>::const_iterator f = ff.begin (); f != ff.end (); ++f) {
      mp_ui->cell_font_cb->addItem (tl::to_qstring (*f));
    }
  }
  mp_ui->cell_font_cb->setCurrentIndex (font);

  int n = 0;
  root->config_get (cfg_min_inst_label_size, n);
  mp_ui->cell_min_size_for_label_edit->setText (tl::to_qstring (tl::to_string (n)));

  bool gs_visible = true;
  root->config_get (cfg_guiding_shape_visible, gs_visible);
  mp_ui->pcell_gs_group->setChecked (gs_visible);

  int gs_lw = 1;
  root->config_get (cfg_guiding_shape_line_width, gs_lw);
  mp_ui->pcell_gs_lw->setValue (gs_lw);

  QColor gs_color;
  root->config_get (cfg_guiding_shape_color, gs_color, ColorConverter ());
  mp_ui->pcell_gs_color_pb->set_color (gs_color);

  int gs_vs = 6;
  root->config_get (cfg_guiding_shape_vertex_size, gs_vs);
  mp_ui->pcell_gs_vs->setValue (gs_vs);
}

void 
LayoutViewConfigPage2a::commit (lay::Plugin *root)
{
  root->config_set (cfg_cell_box_text_transform, mp_ui->cell_xform_text_cbx->isChecked ());
  root->config_set (cfg_cell_box_text_font, mp_ui->cell_font_cb->currentIndex ());
  root->config_set (cfg_cell_box_color, mp_ui->cell_box_color_pb->get_color (), ColorConverter ());
  root->config_set (cfg_cell_box_visible, mp_ui->cell_group->isChecked ());

  root->config_set (cfg_guiding_shape_visible, mp_ui->pcell_gs_group->isChecked ());
  root->config_set (cfg_guiding_shape_line_width, mp_ui->pcell_gs_lw->value ());
  root->config_set (cfg_guiding_shape_color, mp_ui->pcell_gs_color_pb->get_color (), ColorConverter ());
  root->config_set (cfg_guiding_shape_vertex_size, mp_ui->pcell_gs_vs->value ());

  try {
    int n;
    tl::from_string (tl::to_string (mp_ui->cell_min_size_for_label_edit->text ()), n);
    root->config_set (cfg_min_inst_label_size, n);
  } catch (...) { }
}

// ------------------------------------------------------------
//  LayoutConfigPage2b implementation

LayoutViewConfigPage2b::LayoutViewConfigPage2b (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage2b ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage2b::~LayoutViewConfigPage2b ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage2b::setup (lay::Plugin *root)
{
  QColor color;
  root->config_get (cfg_text_color, color, ColorConverter ());
  mp_ui->text_color_pb->set_color (color);

  bool flag = false;
  root->config_get (cfg_apply_text_trans, flag);
  mp_ui->text_apply_trans_cbx->setChecked (flag);

  root->config_get (cfg_text_visible, flag);
  mp_ui->text_group->setChecked (flag);

  root->config_get (cfg_show_properties, flag);
  mp_ui->show_properties_cbx->setChecked (flag);

  int font = 0;
  root->config_get (cfg_text_font, font);
  mp_ui->text_font_cb->clear ();
  if (mp_ui->text_font_cb->count () == 0) {
    std::vector<std::string> ff = db::Hershey::font_names ();
    for (std::vector<std::string>::const_iterator f = ff.begin (); f != ff.end (); ++f) {
      mp_ui->text_font_cb->addItem (tl::to_qstring (*f));
    }
  }
  mp_ui->text_font_cb->setCurrentIndex (font);

  double s = 0.0;
  root->config_get (cfg_default_text_size, s);
  mp_ui->text_def_size_edit->setText (tl::to_qstring (tl::to_string (s)));
}

void 
LayoutViewConfigPage2b::commit (lay::Plugin *root)
{
  root->config_set (cfg_text_color, mp_ui->text_color_pb->get_color (), ColorConverter ());
  root->config_set (cfg_apply_text_trans, mp_ui->text_apply_trans_cbx->isChecked ());
  root->config_set (cfg_text_visible, mp_ui->text_group->isChecked ());
  root->config_set (cfg_show_properties, mp_ui->show_properties_cbx->isChecked ());
  root->config_set (cfg_text_font, mp_ui->text_font_cb->currentIndex ());

  try {
    double s;
    tl::from_string (tl::to_string (mp_ui->text_def_size_edit->text ()), s);
    root->config_set (cfg_default_text_size, s);
  } catch (...) { }
}

// ------------------------------------------------------------
//  LayoutConfigPage2c implementation

LayoutViewConfigPage2c::LayoutViewConfigPage2c (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage2c ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage2c::~LayoutViewConfigPage2c ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage2c::setup (lay::Plugin *root)
{
  QColor color;
  root->config_get (cfg_sel_color, color, lay::ColorConverter ());
  mp_ui->color_pb->set_color (color);

  int lw = 0;
  root->config_get (cfg_sel_line_width, lw);
  mp_ui->lw_spinbx->setValue (lw);

  int vs = 0;
  root->config_get (cfg_sel_vertex_size, vs);
  mp_ui->vs_spinbx->setValue (vs);

  int dp = 0;
  root->config_get (cfg_sel_dither_pattern, dp);
  mp_ui->stipple_pb->set_dither_pattern (dp);

  bool halo = 0;
  root->config_get (cfg_sel_halo, halo);
  mp_ui->halo_cb->setChecked (halo);

  bool tm = 0;
  root->config_get (cfg_sel_transient_mode, tm);
  mp_ui->transient_mode_cb->setChecked (tm);

  bool ipm = 0;
  root->config_get (cfg_sel_inside_pcells_mode, ipm);
  mp_ui->sel_inside_pcells_cb->setChecked (ipm);
}

void 
LayoutViewConfigPage2c::commit (lay::Plugin *root)
{
  lay::ColorConverter cc;
  root->config_set (cfg_sel_color, mp_ui->color_pb->get_color (), cc);
  root->config_set (cfg_sel_line_width, mp_ui->lw_spinbx->value ());
  root->config_set (cfg_sel_vertex_size, mp_ui->vs_spinbx->value ());
  root->config_set (cfg_sel_dither_pattern, mp_ui->stipple_pb->dither_pattern ());
  root->config_set (cfg_sel_halo, mp_ui->halo_cb->isChecked ());
  root->config_set (cfg_sel_transient_mode, mp_ui->transient_mode_cb->isChecked ());
  root->config_set (cfg_sel_inside_pcells_mode, mp_ui->sel_inside_pcells_cb->isChecked ());
}

// ------------------------------------------------------------
//  LayoutConfigPage3a implementation

LayoutViewConfigPage3a::LayoutViewConfigPage3a (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage3a ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage3a::~LayoutViewConfigPage3a ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage3a::setup (lay::Plugin *root)
{
  bool flag = true;

  root->config_get (cfg_fit_new_cell, flag);
  mp_ui->fit_new_cell_cbx->setChecked (flag);
  root->config_get (cfg_full_hier_new_cell, flag);
  mp_ui->full_hier_new_cell_cbx->setChecked (flag);
  root->config_get (cfg_clear_ruler_new_cell, flag);
  mp_ui->clear_ruler_new_cell_cbx->setChecked (flag);
}

void 
LayoutViewConfigPage3a::commit (lay::Plugin *root)
{
  root->config_set (cfg_fit_new_cell, mp_ui->fit_new_cell_cbx->isChecked ());
  root->config_set (cfg_full_hier_new_cell, mp_ui->full_hier_new_cell_cbx->isChecked ());
  root->config_set (cfg_clear_ruler_new_cell, mp_ui->clear_ruler_new_cell_cbx->isChecked ());
}

// ------------------------------------------------------------
//  LayoutConfigPage3b implementation

LayoutViewConfigPage3b::LayoutViewConfigPage3b (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage3b ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage3b::~LayoutViewConfigPage3b ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage3b::setup (lay::Plugin *root)
{
  bool flag = true;
  double pd = 0.0;
  int dm = 0;

  root->config_get (cfg_paste_display_mode, dm);
  mp_ui->paste_dont_change_rb->setChecked (dm == 0);
  mp_ui->paste_pan_rb->setChecked (dm == 1);
  mp_ui->paste_zoom_rb->setChecked (dm == 2);

  root->config_get (cfg_pan_distance, pd);
  pd *= 100.0;
  mp_ui->pan_distance_le->setText (tl::to_qstring (tl::to_string (pd)));

  root->config_get (cfg_mouse_wheel_mode, flag);
  mp_ui->alt_mouse_wheel_mode_cbx->setChecked (flag);
}

void 
LayoutViewConfigPage3b::commit (lay::Plugin *root)
{
  root->config_set (cfg_mouse_wheel_mode, mp_ui->alt_mouse_wheel_mode_cbx->isChecked () ? 1 : 0);

  double pd = 0.0;
  try {
    tl::from_string (tl::to_string (mp_ui->pan_distance_le->text ()), pd);
  } catch (...) { }
  if (pd <= 0.0) {
    throw tl::Exception (tl::to_string (QObject::tr ("Invalid pan distance: must be larger than 0.0")));
  }
  root->config_set (cfg_pan_distance, pd * 0.01);

  if (mp_ui->paste_dont_change_rb->isChecked ()) {
    root->config_set (cfg_paste_display_mode, 0);
  } else if (mp_ui->paste_pan_rb->isChecked ()) {
    root->config_set (cfg_paste_display_mode, 1);
  } else if (mp_ui->paste_zoom_rb->isChecked ()) {
    root->config_set (cfg_paste_display_mode, 2);
  }
}

// ------------------------------------------------------------
//  LayoutConfigPage3c implementation

LayoutViewConfigPage3c::LayoutViewConfigPage3c (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage3c ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage3c::~LayoutViewConfigPage3c ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage3c::setup (lay::Plugin *root)
{
  bool flag = true;

  root->config_get (cfg_dbu_units, flag);
  mp_ui->dbu_units_cbx->setChecked (flag);
  root->config_get (cfg_abs_units, flag);
  mp_ui->abs_units_cbx->setChecked (flag);
}

void 
LayoutViewConfigPage3c::commit (lay::Plugin *root)
{
  root->config_set (cfg_dbu_units, mp_ui->dbu_units_cbx->isChecked ());
  root->config_set (cfg_abs_units, mp_ui->abs_units_cbx->isChecked ());
}

// ------------------------------------------------------------
//  LayoutConfigPage3f implementation

LayoutViewConfigPage3f::LayoutViewConfigPage3f (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage3f ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage3f::~LayoutViewConfigPage3f ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage3f::setup (lay::Plugin *root)
{
  int workers = 1;
  bool flag = true;

  root->config_get (cfg_drawing_workers, workers);
  mp_ui->drawing_workers_spbx->setValue (workers);

  root->config_get (cfg_drop_small_cells, flag);
  mp_ui->drop_small_cells_cbx->setChecked (flag);

  unsigned int n = 0;
  root->config_get (cfg_drop_small_cells_cond, n);
  mp_ui->drop_small_cells_cond_cb->setCurrentIndex (n);
  root->config_get (cfg_drop_small_cells_value, n);
  mp_ui->drop_small_cells_value_le->setText (tl::to_qstring (tl::to_string (n)));

  root->config_get (cfg_array_border_instances, flag);
  mp_ui->array_border_insts_cbx->setChecked (flag);

  root->config_get (cfg_text_lazy_rendering, flag);
  mp_ui->text_lazy_rendering_cbx->setChecked (flag);

  root->config_get (cfg_bitmap_caching, flag);
  mp_ui->bitmap_caching_cbx->setChecked (flag);

  n = 0;
  root->config_get (cfg_image_cache_size, n);
  mp_ui->image_cache_size_spbx->setValue (int (n));
}

void 
LayoutViewConfigPage3f::commit (lay::Plugin *root)
{
  root->config_set (cfg_drawing_workers, mp_ui->drawing_workers_spbx->value ());

  root->config_set (cfg_drop_small_cells, mp_ui->drop_small_cells_cbx->isChecked ());
  root->config_set (cfg_drop_small_cells_cond, mp_ui->drop_small_cells_cond_cb->currentIndex ());

  try {
    unsigned int s;
    tl::from_string (tl::to_string (mp_ui->drop_small_cells_value_le->text ()), s);
    root->config_set (cfg_drop_small_cells_value, s);
  } catch (...) { }

  root->config_set (cfg_array_border_instances, mp_ui->array_border_insts_cbx->isChecked ());

  root->config_set (cfg_text_lazy_rendering, mp_ui->text_lazy_rendering_cbx->isChecked ());
  root->config_set (cfg_bitmap_caching, mp_ui->bitmap_caching_cbx->isChecked ());

  root->config_set (cfg_image_cache_size, mp_ui->image_cache_size_spbx->value ());
}


// ------------------------------------------------------------
//  LayoutConfigPage4 implementation

class PaletteOp 
  : public db::Op
{
public:
  PaletteOp (const lay::ColorPalette &p, bool e, bool b)
    : palette (p), edit_order (e), before (b)
  { }

  lay::ColorPalette palette;
  bool edit_order, before;
};

static QToolButton * (Ui::LayoutViewConfigPage4::*cfg4_buttons []) = {
  &Ui::LayoutViewConfigPage4::cb_1,  &Ui::LayoutViewConfigPage4::cb_2,  &Ui::LayoutViewConfigPage4::cb_3,  &Ui::LayoutViewConfigPage4::cb_4,  &Ui::LayoutViewConfigPage4::cb_5,
  &Ui::LayoutViewConfigPage4::cb_6,  &Ui::LayoutViewConfigPage4::cb_7,  &Ui::LayoutViewConfigPage4::cb_8,  &Ui::LayoutViewConfigPage4::cb_9,  &Ui::LayoutViewConfigPage4::cb_10,
  &Ui::LayoutViewConfigPage4::cb_11, &Ui::LayoutViewConfigPage4::cb_12, &Ui::LayoutViewConfigPage4::cb_13, &Ui::LayoutViewConfigPage4::cb_14, &Ui::LayoutViewConfigPage4::cb_15,
  &Ui::LayoutViewConfigPage4::cb_16, &Ui::LayoutViewConfigPage4::cb_17, &Ui::LayoutViewConfigPage4::cb_18, &Ui::LayoutViewConfigPage4::cb_19, &Ui::LayoutViewConfigPage4::cb_20,
  &Ui::LayoutViewConfigPage4::cb_21, &Ui::LayoutViewConfigPage4::cb_22, &Ui::LayoutViewConfigPage4::cb_23, &Ui::LayoutViewConfigPage4::cb_24, &Ui::LayoutViewConfigPage4::cb_25,
  &Ui::LayoutViewConfigPage4::cb_26, &Ui::LayoutViewConfigPage4::cb_27, &Ui::LayoutViewConfigPage4::cb_28, &Ui::LayoutViewConfigPage4::cb_29, &Ui::LayoutViewConfigPage4::cb_30,
  &Ui::LayoutViewConfigPage4::cb_31, &Ui::LayoutViewConfigPage4::cb_32, &Ui::LayoutViewConfigPage4::cb_33, &Ui::LayoutViewConfigPage4::cb_34, &Ui::LayoutViewConfigPage4::cb_35,
  &Ui::LayoutViewConfigPage4::cb_36, &Ui::LayoutViewConfigPage4::cb_37, &Ui::LayoutViewConfigPage4::cb_38, &Ui::LayoutViewConfigPage4::cb_39, &Ui::LayoutViewConfigPage4::cb_40,
  &Ui::LayoutViewConfigPage4::cb_41, &Ui::LayoutViewConfigPage4::cb_42
};

LayoutViewConfigPage4::LayoutViewConfigPage4 (QWidget *parent)
  : lay::ConfigPage (parent),
    m_edit_order_changed_disabled (false)
{
  //  install the manager at db::Object
  manager (&m_manager);

  mp_ui = new Ui::LayoutViewConfigPage4 ();
  mp_ui->setupUi (this);

  for (unsigned int i = 0; i < sizeof (cfg4_buttons) / sizeof (cfg4_buttons [0]); ++i) {
    connect (mp_ui->*(cfg4_buttons [i]), SIGNAL (clicked ()), this, SLOT (color_button_clicked ()));
  }

  connect (mp_ui->undo_pb, SIGNAL (clicked ()), this, SLOT (undo_button_clicked ()));
  connect (mp_ui->redo_pb, SIGNAL (clicked ()), this, SLOT (redo_button_clicked ()));
  connect (mp_ui->reset_pb, SIGNAL (clicked ()), this, SLOT (reset_button_clicked ()));
  connect (mp_ui->edit_order_cbx, SIGNAL (stateChanged (int)), this, SLOT (edit_order_changed (int)));
}

LayoutViewConfigPage4::~LayoutViewConfigPage4 ()
{
  //  deinstall the manager
  manager (0);

  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage4::set_edit_order (bool edit_order)
{
  m_edit_order_changed_disabled = true;
  mp_ui->edit_order_cbx->setChecked (edit_order);
  m_edit_order_changed_disabled = false;
}

void 
LayoutViewConfigPage4::setup (lay::Plugin *root)
{
  m_manager.clear ();

  std::string s;
  root->config_get (cfg_color_palette, s);

  lay::ColorPalette palette = lay::ColorPalette::default_palette ();
  try {
    if (! s.empty ()) {
      palette.from_string (s);
    }
  } catch (...) {
    //  ignore errors: just reset the palette 
    palette = lay::ColorPalette::default_palette ();
  }

  m_palette = palette;

  set_edit_order (false);
  update ();
}

void 
LayoutViewConfigPage4::commit (lay::Plugin *root)
{
  root->config_set (cfg_color_palette, m_palette.to_string ());
}

void 
LayoutViewConfigPage4::undo (db::Op *op)
{
  PaletteOp *pop = dynamic_cast<PaletteOp *> (op);
  if (pop && pop->before) {
    m_palette = pop->palette;
    set_edit_order (pop->edit_order);
  }
}

void 
LayoutViewConfigPage4::redo (db::Op *op)
{
  PaletteOp *pop = dynamic_cast<PaletteOp *> (op);
  if (pop && ! pop->before) {
    m_palette = pop->palette;
    set_edit_order (pop->edit_order);
  }
}

void 
LayoutViewConfigPage4::undo_button_clicked ()
{
  m_manager.undo ();
  update ();
}

void 
LayoutViewConfigPage4::redo_button_clicked ()
{
  m_manager.redo ();
  update ();
}

void 
LayoutViewConfigPage4::reset_button_clicked ()
{
  m_manager.transaction (tl::to_string (QObject::tr ("Reset palette")));
  m_manager.queue (this, new PaletteOp (m_palette, mp_ui->edit_order_cbx->isChecked (), true /*before*/));
  m_palette = lay::ColorPalette::default_palette ();
  m_manager.queue (this, new PaletteOp (m_palette, false, false /*after*/));
  m_manager.commit ();
  set_edit_order (false);
  update ();
}

void 
LayoutViewConfigPage4::edit_order_changed (int s)
{
  if (m_edit_order_changed_disabled) {
    return;
  }

  if (s) {
    m_manager.transaction (tl::to_string (QObject::tr ("Clear assignment order")));
    m_manager.queue (this, new PaletteOp (m_palette, false, true /*before*/));
    m_palette.clear_luminous_colors ();
    m_manager.queue (this, new PaletteOp (m_palette, true, false /*after*/));
    m_manager.commit ();
    update ();
  } else {
    m_manager.transaction (tl::to_string (QObject::tr ("Stop changing assignment order")));
    m_manager.queue (this, new PaletteOp (m_palette, true, true /*before*/));
    m_manager.queue (this, new PaletteOp (m_palette, false, false /*after*/));
    m_manager.commit ();
  }
}

void 
LayoutViewConfigPage4::color_button_clicked ()
{
  if (mp_ui->edit_order_cbx->isChecked ()) {

    for (unsigned int i = 0; i < sizeof (cfg4_buttons) / sizeof (cfg4_buttons [0]); ++i) {
      if (sender () == mp_ui->*(cfg4_buttons [i])) {

        bool found = false;
        for (unsigned int j = 0; j < m_palette.luminous_colors () && !found; ++j) {
          if (m_palette.luminous_color_index_by_index (j) == i) {
            found = true;
          }
        }
        if (! found) {
          m_manager.transaction (tl::to_string (QObject::tr ("Set assignment order")));
          m_manager.queue (this, new PaletteOp (m_palette, true, true /*before*/));
          m_palette.set_luminous_color_index (m_palette.luminous_colors (), i);
          m_manager.queue (this, new PaletteOp (m_palette, true, false /*after*/));
          m_manager.commit ();
          update ();
        }
        break;
      }
    }

  } else {

    for (unsigned int i = 0; i < sizeof (cfg4_buttons) / sizeof (cfg4_buttons [0]); ++i) {

      if (sender () == mp_ui->*(cfg4_buttons [i])) {

        QColor c;
        if (m_palette.colors () > i) {
          c = QColorDialog::getColor (m_palette.color_by_index (i));
        } else {
          c = QColorDialog::getColor ();
        }
        if (c.isValid ()) {
          m_manager.transaction (tl::to_string (QObject::tr ("Set color")));
          m_manager.queue (this, new PaletteOp (m_palette, false, true /*before*/));
          m_palette.set_color (i, c.rgb ());
          m_manager.queue (this, new PaletteOp (m_palette, false, false /*after*/));
          m_manager.commit ();
          update ();
        }

        break;

      }
    }

  }
}

void 
LayoutViewConfigPage4::update ()
{
  for (unsigned int i = 0; i < sizeof (cfg4_buttons) / sizeof (cfg4_buttons [0]); ++i) {

    QColor color;
    if (i < m_palette.colors ()) {
      color = QColor (m_palette.color_by_index (i));
    }

    QColor text_color = color.green () > 128 ? QColor (0, 0, 0) : QColor (255, 255, 255);

    QString text = QString::fromAscii ("  "); 
    for (unsigned int j = 0; j < m_palette.luminous_colors (); ++j) {
      if (i == m_palette.luminous_color_index_by_index (j)) {
        text = tl::to_qstring (tl::sprintf ("%d", j));
        break;
      }
    }

    QFontMetrics fm (font (), this);
    QRect rt (fm.boundingRect (QString::fromAscii ("AA")));
    QPixmap pxmp (rt.width () + 10, rt.height () + 10);

    QPainter pxpainter (&pxmp);
    pxpainter.setPen (QPen (palette ().color (QPalette::Active, QPalette::Text)));
    pxpainter.setBrush (QBrush (color));
    QRect r (0, 0, pxmp.width () - 1, pxmp.height () - 1);
    pxpainter.drawRect (r);
    pxpainter.setFont (font ());
    pxpainter.setPen (QPen (text_color));
    pxpainter.drawText (r, Qt::AlignHCenter | Qt::AlignVCenter | Qt::TextSingleLine, text);

    (mp_ui->*(cfg4_buttons [i]))->setIconSize (pxmp.size ());
    (mp_ui->*(cfg4_buttons [i]))->setIcon (QIcon (pxmp));

  }
}

// ------------------------------------------------------------
//  LayoutConfigPage5 implementation

LayoutViewConfigPage5::LayoutViewConfigPage5 (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage5 ();
  mp_ui->setupUi (this);
  connect (mp_ui->browse_pb, SIGNAL (clicked ()), this, SLOT (browse_clicked ()));
}

LayoutViewConfigPage5::~LayoutViewConfigPage5 ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage5::setup (lay::Plugin *root)
{
  std::string lyp_file;
  root->config_get (cfg_default_lyp_file, lyp_file);
  mp_ui->lyp_file_gbx->setChecked (! lyp_file.empty ());
  mp_ui->lyp_file_le->setText (tl::to_qstring (lyp_file));

  bool add_other_layers = false;
  root->config_get (cfg_default_add_other_layers, add_other_layers);
  mp_ui->add_other_layers_cb->setChecked (add_other_layers);

  bool always_show_source = false;
  root->config_get (cfg_layers_always_show_source, always_show_source);
  mp_ui->source_display_cb->setChecked (always_show_source);

  bool always_show_ld = false;
  root->config_get (cfg_layers_always_show_ld, always_show_ld);
  mp_ui->ld_display_cb->setChecked (always_show_ld);

  bool always_show_li = false;
  root->config_get (cfg_layers_always_show_layout_index, always_show_li);
  mp_ui->ly_index_cb->setChecked (always_show_li);
}

void 
LayoutViewConfigPage5::commit (lay::Plugin *root)
{
  if (mp_ui->lyp_file_gbx->isChecked ()) {
    root->config_set (cfg_default_lyp_file, tl::to_string (mp_ui->lyp_file_le->text ()));
  } else {
    root->config_set (cfg_default_lyp_file, std::string ());
  }
  root->config_set (cfg_default_add_other_layers, mp_ui->add_other_layers_cb->isChecked ());
  root->config_set (cfg_layers_always_show_source, mp_ui->source_display_cb->isChecked ());
  root->config_set (cfg_layers_always_show_ld, mp_ui->ld_display_cb->isChecked ());
  root->config_set (cfg_layers_always_show_layout_index, mp_ui->ly_index_cb->isChecked ());
}

void 
LayoutViewConfigPage5::browse_clicked ()
{
  std::string fn = tl::to_string (mp_ui->lyp_file_le->text ());
  lay::FileDialog file_dialog (this,
    tl::to_string (QObject::tr ("Layer Properties File")), 
    tl::to_string (QObject::tr ("Layer properties files (*.lyp);;All files (*)")), 
    "lyp");

  if (file_dialog.get_open (fn)) {
    mp_ui->lyp_file_le->setText (tl::to_qstring (fn));
  }
}

// ------------------------------------------------------------
//  LayoutConfigPage6 implementation

class StipplePaletteOp 
  : public db::Op
{
public:
  StipplePaletteOp (const lay::StipplePalette &p, bool e, bool b)
    : palette (p), edit_order (e), before (b)
  { }

  lay::StipplePalette palette;
  bool edit_order, before;
};

static QToolButton * (Ui::LayoutViewConfigPage6::*cfg6_buttons []) = {
  &Ui::LayoutViewConfigPage6::cb_1,  &Ui::LayoutViewConfigPage6::cb_2,  &Ui::LayoutViewConfigPage6::cb_3,  &Ui::LayoutViewConfigPage6::cb_4,  
  &Ui::LayoutViewConfigPage6::cb_5,  &Ui::LayoutViewConfigPage6::cb_6,  &Ui::LayoutViewConfigPage6::cb_7,  &Ui::LayoutViewConfigPage6::cb_8,  
  &Ui::LayoutViewConfigPage6::cb_9,  &Ui::LayoutViewConfigPage6::cb_10, &Ui::LayoutViewConfigPage6::cb_11, &Ui::LayoutViewConfigPage6::cb_12, 
  &Ui::LayoutViewConfigPage6::cb_13, &Ui::LayoutViewConfigPage6::cb_14, &Ui::LayoutViewConfigPage6::cb_15, &Ui::LayoutViewConfigPage6::cb_16, 
};

LayoutViewConfigPage6::LayoutViewConfigPage6 (QWidget *parent)
  : lay::ConfigPage (parent),
    m_edit_order_changed_disabled (false)
{
  //  install the manager at db::Object
  manager (&m_manager);

  mp_ui = new Ui::LayoutViewConfigPage6 ();
  mp_ui->setupUi (this);

  for (unsigned int i = 0; i < sizeof (cfg6_buttons) / sizeof (cfg6_buttons [0]); ++i) {
    connect (mp_ui->*(cfg6_buttons [i]), SIGNAL (clicked ()), this, SLOT (stipple_button_clicked ()));
  }

  connect (mp_ui->undo_pb, SIGNAL (clicked ()), this, SLOT (undo_button_clicked ()));
  connect (mp_ui->redo_pb, SIGNAL (clicked ()), this, SLOT (redo_button_clicked ()));
  connect (mp_ui->reset_pb, SIGNAL (clicked ()), this, SLOT (reset_button_clicked ()));
  connect (mp_ui->edit_order_cbx, SIGNAL (stateChanged (int)), this, SLOT (edit_order_changed (int)));
}

LayoutViewConfigPage6::~LayoutViewConfigPage6 ()
{
  //  deinstall the manager
  manager (0);

  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage6::set_edit_order (bool edit_order)
{
  m_edit_order_changed_disabled = true;
  mp_ui->edit_order_cbx->setChecked (edit_order);
  m_edit_order_changed_disabled = false;
}

void 
LayoutViewConfigPage6::setup (lay::Plugin *root)
{
  m_manager.clear ();

  std::string s;
  root->config_get (cfg_stipple_palette, s);

  lay::StipplePalette palette = lay::StipplePalette::default_palette ();
  try {
    if (! s.empty ()) {
      palette.from_string (s);
    }
  } catch (...) {
    //  ignore errors: just reset the palette 
    palette = lay::StipplePalette::default_palette ();
  }

  m_palette = palette;

  bool f = true;
  root->config_get (cfg_stipple_offset, f);
  mp_ui->stipple_offset_cbx->setChecked (f);

  set_edit_order (false);
  update ();
}

void 
LayoutViewConfigPage6::commit (lay::Plugin *root)
{
  if (m_palette.stipples () == 0) {
    throw tl::Exception (tl::to_string (QObject::tr ("No stipples set")));
  }
  if (m_palette.standard_stipples () == 0) {
    throw tl::Exception (tl::to_string (QObject::tr ("No stipples selected for automatic assignment - use 'Set assignment order' to do so")));
  }
  root->config_set (cfg_stipple_palette, m_palette.to_string ());
  root->config_set (cfg_stipple_offset, mp_ui->stipple_offset_cbx->isChecked ());
}

void 
LayoutViewConfigPage6::undo (db::Op *op)
{
  StipplePaletteOp *pop = dynamic_cast<StipplePaletteOp *> (op);
  if (pop && pop->before) {
    m_palette = pop->palette;
    set_edit_order (pop->edit_order);
  }
}

void 
LayoutViewConfigPage6::redo (db::Op *op)
{
  StipplePaletteOp *pop = dynamic_cast<StipplePaletteOp *> (op);
  if (pop && ! pop->before) {
    m_palette = pop->palette;
    set_edit_order (pop->edit_order);
  }
}

void 
LayoutViewConfigPage6::undo_button_clicked ()
{
  m_manager.undo ();
  update ();
}

void 
LayoutViewConfigPage6::redo_button_clicked ()
{
  m_manager.redo ();
  update ();
}

void 
LayoutViewConfigPage6::reset_button_clicked ()
{
  m_manager.transaction (tl::to_string (QObject::tr ("Reset palette")));
  m_manager.queue (this, new StipplePaletteOp (m_palette, mp_ui->edit_order_cbx->isChecked (), true /*before*/));
  m_palette = lay::StipplePalette::default_palette ();
  m_manager.queue (this, new StipplePaletteOp (m_palette, false, false /*after*/));
  m_manager.commit ();
  set_edit_order (false);
  update ();
}

void 
LayoutViewConfigPage6::edit_order_changed (int s)
{
  if (m_edit_order_changed_disabled) {
    return;
  }

  if (s) {
    m_manager.transaction (tl::to_string (QObject::tr ("Clear assignment order")));
    m_manager.queue (this, new StipplePaletteOp (m_palette, false, true /*before*/));
    m_palette.clear_standard_stipples ();
    m_manager.queue (this, new StipplePaletteOp (m_palette, true, false /*after*/));
    m_manager.commit ();
    update ();
  } else {
    m_manager.transaction (tl::to_string (QObject::tr ("Stop changing assignment order")));
    m_manager.queue (this, new StipplePaletteOp (m_palette, true, true /*before*/));
    m_manager.queue (this, new StipplePaletteOp (m_palette, false, false /*after*/));
    m_manager.commit ();
  }
}

void 
LayoutViewConfigPage6::stipple_button_clicked ()
{
  if (mp_ui->edit_order_cbx->isChecked ()) {

    for (unsigned int i = 0; i < sizeof (cfg6_buttons) / sizeof (cfg6_buttons [0]); ++i) {
      if (sender () == mp_ui->*(cfg6_buttons [i])) {

        bool found = false;
        for (unsigned int j = 0; j < m_palette.standard_stipples () && !found; ++j) {
          if (m_palette.standard_stipple_index_by_index (j) == i) {
            found = true;
          }
        }
        if (! found) {
          m_manager.transaction (tl::to_string (QObject::tr ("Set assignment order")));
          m_manager.queue (this, new StipplePaletteOp (m_palette, true, true /*before*/));
          m_palette.set_standard_stipple_index (m_palette.standard_stipples (), i);
          m_manager.queue (this, new StipplePaletteOp (m_palette, true, false /*after*/));
          m_manager.commit ();
          update ();
        }
        break;
      }
    }

  } else {

    for (unsigned int i = 0; i < sizeof (cfg6_buttons) / sizeof (cfg6_buttons [0]); ++i) {

      if (sender () == mp_ui->*(cfg6_buttons [i])) {

        if (m_palette.stipples () > i) {

          unsigned int s = m_palette.stipple_by_index (i);
          SelectStippleForm stipples_form (0, m_pattern);
          if (stipples_form.exec () && stipples_form.selected () >= 0) {

            s = stipples_form.selected ();

            m_manager.transaction (tl::to_string (QObject::tr ("Set stipple")));
            m_manager.queue (this, new StipplePaletteOp (m_palette, false, true /*before*/));
            m_palette.set_stipple (i, s);
            m_manager.queue (this, new StipplePaletteOp (m_palette, false, false /*after*/));
            m_manager.commit ();
            update ();

          }

        }

        break;

      }
    }

  }
}

void 
LayoutViewConfigPage6::update ()
{
  for (unsigned int i = 0; i < sizeof (cfg6_buttons) / sizeof (cfg6_buttons [0]); ++i) {

    int s = -1;
    if (i < m_palette.stipples ()) {
      s = (int)m_palette.stipple_by_index (i);
    }

    QString text = QString::fromAscii ("  "); 
    for (unsigned int j = 0; j < m_palette.standard_stipples (); ++j) {
      if (i == m_palette.standard_stipple_index_by_index (j)) {
        text = tl::to_qstring (tl::sprintf ("%d", j));
        break;
      }
    }

    QFontMetrics fm (font (), this);
    QRect rt (fm.boundingRect (QString::fromAscii ("AA")));

    const unsigned int h = rt.height () + 10;
    const unsigned int w = rt.width () + 10;

    unsigned int color0 = palette ().color (QPalette::Active, QPalette::Button).rgb();
    unsigned int color1 = palette ().color (QPalette::Active, QPalette::Dark).rgb();

    QImage image (w, h, QImage::Format_RGB32);
    if (s >= 0) {
      const uint32_t *dp = m_pattern.pattern ((unsigned int) s);
      for (unsigned int l = 0; l < h; ++l, ++dp) {
        uint32_t m = *dp;
        if (l == 0 || l == h - 1) { 
          m |= ((1 << w) - 1);
        } else {
          m |= ((1 << (w - 1)) | 1);
        }
        color_t *pt = (color_t *) image.scanLine (h - 1 - l);
        for (unsigned int b = 0; b < w; ++b) {
          *pt++ = (m & 1) ? color1 : color0;
          m >>= 1;
        }
      }
    }

    QPixmap pxmp = QPixmap::fromImage (image); // Qt 4.6.0 workaround

    QPainter pxpainter (&pxmp);
    pxpainter.setPen (QPen (palette ().color (QPalette::Active, QPalette::Text)));
    QRect r (0, 0, pxmp.width () - 1, pxmp.height () - 1);
    pxpainter.drawRect (r);
    pxpainter.setFont (font ());
    pxpainter.drawText (r, Qt::AlignHCenter | Qt::AlignVCenter | Qt::TextSingleLine, text);

    (mp_ui->*(cfg6_buttons [i]))->setIconSize (pxmp.size ());
    (mp_ui->*(cfg6_buttons [i]))->setIcon (QIcon (pxmp));

  }
}

// ------------------------------------------------------------
//  LayoutConfigPage7 implementation

LayoutViewConfigPage7::LayoutViewConfigPage7 (QWidget *parent)
  : lay::ConfigPage (parent)
{
  mp_ui = new Ui::LayoutViewConfigPage7 ();
  mp_ui->setupUi (this);
}

LayoutViewConfigPage7::~LayoutViewConfigPage7 ()
{
  delete mp_ui;
  mp_ui = 0;
}

void 
LayoutViewConfigPage7::setup (lay::Plugin *root)
{
  int oversampling = 1;
  root->config_get (cfg_bitmap_oversampling, oversampling);
  mp_ui->oversampling->setCurrentIndex (oversampling - 1);

  int default_font_size = 0;
  root->config_get (cfg_default_font_size, default_font_size);
  mp_ui->default_font_size->setCurrentIndex (default_font_size);

  std::string s;
  root->config_get (cfg_global_trans, s);
  tl::Extractor ex (s.c_str ());

  try {
    db::DCplxTrans t;
    ex.read (t);
    mp_ui->global_trans->setCurrentIndex (t.rot ());
  } catch (...) { }

  int def_depth = 0;
  root->config_get (cfg_initial_hier_depth, def_depth);
  mp_ui->def_depth->setValue(def_depth);
}

void 
LayoutViewConfigPage7::commit (lay::Plugin *root)
{
  root->config_set (cfg_bitmap_oversampling, mp_ui->oversampling->currentIndex () + 1);
  root->config_set (cfg_default_font_size, mp_ui->default_font_size->currentIndex ());
  root->config_set (cfg_global_trans, db::DCplxTrans (db::FTrans (mp_ui->global_trans->currentIndex ())).to_string ());
  root->config_set (cfg_initial_hier_depth, mp_ui->def_depth->value ());
}

// ------------------------------------------------------------
//  The dummy plugin declaration to register the configuration options

class LayoutViewConfigDeclaration
  : public lay::PluginDeclaration
{
public:
  virtual void get_options (std::vector < std::pair<std::string, std::string> > &options) const
  {
    lay::ColorConverter cc;
    options.push_back (std::pair<std::string, std::string> (cfg_default_lyp_file, ""));
    options.push_back (std::pair<std::string, std::string> (cfg_default_add_other_layers, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_layers_always_show_source, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_layers_always_show_ld, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_layers_always_show_layout_index, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_test_shapes_in_view, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_flat_cell_list, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_split_cell_list, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_cell_list_sorting, "by-name"));
    options.push_back (std::pair<std::string, std::string> (cfg_hide_empty_layers, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_min_inst_label_size, "16"));
    options.push_back (std::pair<std::string, std::string> (cfg_cell_box_text_font, "0"));
    options.push_back (std::pair<std::string, std::string> (cfg_cell_box_text_transform, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_cell_box_color, "auto"));
    options.push_back (std::pair<std::string, std::string> (cfg_cell_box_visible, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_text_color, "auto"));
    options.push_back (std::pair<std::string, std::string> (cfg_text_visible, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_text_lazy_rendering, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_bitmap_caching, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_show_properties, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_apply_text_trans, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_global_trans, "r0"));
    options.push_back (std::pair<std::string, std::string> (cfg_default_text_size, "0.1"));
    options.push_back (std::pair<std::string, std::string> (cfg_text_font, "0"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_color, cc.to_string (QColor ())));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_line_width, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_vertex_size, "3"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_dither_pattern, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_halo, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_transient_mode, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_sel_inside_pcells_mode, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_background_color, cc.to_string (QColor ())));
    options.push_back (std::pair<std::string, std::string> (cfg_ctx_color, cc.to_string (QColor ())));
    options.push_back (std::pair<std::string, std::string> (cfg_ctx_dimming, "50"));
    options.push_back (std::pair<std::string, std::string> (cfg_ctx_hollow, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_child_ctx_color, cc.to_string (QColor ())));
    options.push_back (std::pair<std::string, std::string> (cfg_child_ctx_dimming, "50"));
    options.push_back (std::pair<std::string, std::string> (cfg_child_ctx_hollow, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_child_ctx_enabled, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_abstract_mode_width, "10.0"));
    options.push_back (std::pair<std::string, std::string> (cfg_abstract_mode_enabled, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_fit_new_cell, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_full_hier_new_cell, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_initial_hier_depth, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_clear_ruler_new_cell, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_mouse_wheel_mode, "0"));
    options.push_back (std::pair<std::string, std::string> (cfg_pan_distance, "0.15"));
    options.push_back (std::pair<std::string, std::string> (cfg_paste_display_mode, "2"));
    options.push_back (std::pair<std::string, std::string> (cfg_guiding_shape_visible, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_guiding_shape_line_width, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_guiding_shape_color, cc.to_string (QColor ())));
    options.push_back (std::pair<std::string, std::string> (cfg_guiding_shape_vertex_size, "5"));
    options.push_back (std::pair<std::string, std::string> (cfg_abs_units, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_dbu_units, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_drawing_workers, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_drop_small_cells, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_drop_small_cells_cond, "0"));
    options.push_back (std::pair<std::string, std::string> (cfg_drop_small_cells_value, "10"));
    options.push_back (std::pair<std::string, std::string> (cfg_array_border_instances, "false"));
    options.push_back (std::pair<std::string, std::string> (cfg_bitmap_oversampling, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_image_cache_size, "1"));
    options.push_back (std::pair<std::string, std::string> (cfg_default_font_size, "0"));
    options.push_back (std::pair<std::string, std::string> (cfg_color_palette, lay::ColorPalette ().to_string ()));
    options.push_back (std::pair<std::string, std::string> (cfg_stipple_palette, lay::StipplePalette ().to_string ()));
    options.push_back (std::pair<std::string, std::string> (cfg_stipple_offset, "true"));
    options.push_back (std::pair<std::string, std::string> (cfg_no_stipple, "false"));
  }

  virtual std::vector<std::pair <std::string, ConfigPage *> > config_pages (QWidget *parent) const 
  {
    std::vector<std::pair <std::string, ConfigPage *> > pages;

    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|General")),         new LayoutViewConfigPage7 (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Cells")),           new LayoutViewConfigPage2a (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Texts")),           new LayoutViewConfigPage2b (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Color Palette")),   new LayoutViewConfigPage4 (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Stipple Palette")), new LayoutViewConfigPage6 (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Background")),      new LayoutViewConfigPage (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Context")),         new LayoutViewConfigPage1 (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Display|Optimization")),    new LayoutViewConfigPage3f (parent)));

    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Application|Selection")),   new LayoutViewConfigPage2c (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Application|Layer Properties")),  new LayoutViewConfigPage5 (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Application|Units")),       new LayoutViewConfigPage3c (parent)));

    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Navigation|New Cell")),     new LayoutViewConfigPage3a (parent)));
    pages.push_back (std::make_pair (tl::to_string (QObject::tr ("Navigation|Zoom And Pan")), new LayoutViewConfigPage3b (parent)));

    return pages;
  }
};

static tl::RegisteredClass<lay::PluginDeclaration> config_decl (new LayoutViewConfigDeclaration (), 2000, "LayoutViewConfig");

}

