/*
   Copyright (C) 2005 korewaisai
   korewaisai@yahoo.co.jp

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
*/
#include <QtCore>

#include "Crontab.h"
#include "Clib.h"


Crontab::Crontab(const QString &user)
    : CronType(CronType::CRON), cronOwner(user), changed(false)
{
    QString str = getCrontab(user);
    if ( !str.isEmpty() ) setup(str);
}

Crontab::~Crontab()
{
    for (auto& c : tCommands) delete c;
    for (auto& v : variables) delete v;
}

QString Crontab::getCrontab(const QString &user)
{
    QString ret;
    estr = "";
    if (user == "/etc/crontab") {
        QFile f(user);
        if (!f.open(QIODevice::ReadOnly | QIODevice::Text)) {
            estr = "can't open /etc/crontab\n\n" + f.errorString();
            return ret;
        }
        ret = QString::fromUtf8(f.readAll());

    } else {
        QProcess p;
        if (user == Clib::uName())
            p.start("crontab", QStringList() << "-l");
        else
            p.start("crontab", QStringList() << "-u" << user << "-l");

        if (!p.waitForStarted()) {
            estr = "can't get crontab\n\nQProcess::waitForStarted():"
                    + QString::number(p.error());
            return ret;
        }

        if (!p.waitForFinished()) {
            estr = "can't read crontab\n\nQProcess::waitForFinished():"
                    + QString::number(p.error());
            return ret;
        }

        QString err = QString::fromUtf8(p.readAllStandardError());
        if (p.exitCode() || !err.isEmpty()) {
            estr = "crontab update error\n\n" + err;
            return ret;
        }

        ret =  QString::fromUtf8(p.readAllStandardOutput());

    }
    return ret;
}

QString Crontab::writeTempFile(const QString &text, const QString &tmp)
{
    QString fdir = QDir::tempPath() + "/job-scheduler-" + Clib::uName();
    if (!QFileInfo(fdir).exists()) {
        if (!QDir(fdir).mkdir(fdir)) {
            estr = "can't create directory " + fdir;
            return QString();
        }
    }
    QTemporaryFile f(fdir+"/"+tmp);
    f.setAutoRemove(false);
    if (!f.open()) {
        estr = "can't open temporary file\n\n" + f.errorString();
        return QString();
    }
    QTextStream t(&f);
    t << text;
    qDebug() << "File Saved :" << f.fileName();
    return f.fileName();
}

bool Crontab::putCrontab(const QString &text)
{
    estr = "";
    if (cronOwner == "/etc/crontab") {
        QString saveFile = writeTempFile(text, "etccron");
        QFile f(cronOwner);
        if (!f.open(QIODevice::WriteOnly)) {
            estr = "can't open /etc/crontab for write\n\n" + f.errorString();
            return false;
        }
        QTextStream t(&f);
        t << text;
    } else {
        QString fname = writeTempFile(text, cronOwner);
        if (fname.isEmpty())
            return false;

        QProcess p;
        if (Clib::uId() == 0)
            p.start("crontab", QStringList() << "-u" << cronOwner << fname );
        else
            p.start("crontab", QStringList() << fname );

        if (!p.waitForStarted()) {
            estr = "can't update crontab\n\nQProcess::waitForStarted():"
                    + QString::number(p.error());
            return false;
        }

        if (!p.waitForFinished()) {
            estr = "can't update crontab\n\nQProcess::waitForFinished():"
                    + QString::number(p.error());
            return false;
        }

        QString err = QString::fromUtf8(p.readAllStandardError());
        if (p.exitCode() || !err.isEmpty()) {
            estr = "crontab update error\n\n" + err;
            return false;
        }
        //		QFile::remove(fname);
    }

    return true;
}

QString Crontab::cronText()
{
    QString ret;

    if (!comment.isEmpty()) {
        QString s = comment;
        ret += "# " + s.replace('\n',"\n# ") + "\n\n";
    }

    for (Variable *v : variables) {
        if (!v->comment.isEmpty())
            ret += "# " + v->comment.replace('\n', "\n# ") + '\n';

        ret += v->name + "=" + v->value + '\n';
    }

    ret += "\n";
    for (TCommand *c : tCommands) {
        if (!c->comment.isEmpty())
            ret += "# " + c->comment.replace('\n', "\n# ") + '\n';

        if (cronOwner == "/etc/crontab")
            ret += c->time + " " + c->user + " " + c->command + '\n';
        else
            ret += c->time + " " + c->command + '\n';

    }

    return ret;
}

void Crontab::setup(const QString &str)
{
    QStringList slist = str.split('\n');

    if (cronOwner != "/etc/crontab") {
        if ( slist.at(0).contains("# DO NOT EDIT THIS FILE",
                                  Qt::CaseInsensitive)) {
            slist.removeFirst();
            slist.removeFirst();
            slist.removeFirst();
        }
    }

    QStringList cmnt;
    QStringList head;
    int headflag = 0;
    for (QString s : slist) {
        s = s.simplified();
        if (s.isEmpty()) {
            if (headflag == 0) {
                if (head.count() > 0) head << s;
                head << cmnt;
                cmnt.clear();
            } else {
                cmnt << s;
            }
        } else if (s.at(0) == '#') {
            if (s.size() > 1 && s.at(1) == ' ')
                cmnt << s.mid(2);
            else
                cmnt << s.mid(1);

        } else {
            if (headflag == 0) {
                headflag = 1;
                //				if (head.count() == 0)
                //					head << cmnt;

                comment = list2String(head);
            }
            if (s.contains(QRegularExpression("^\\S+\\s*=\\s*\\S*$"))) {
                // Variable
                QRegularExpression sep("\\s*=\\s*");
                QString name = s.section( sep, 0, 0 );
                QString val = s.section( sep, 1, 1 );
                variables << new Variable(name, val, list2String(cmnt));
            } else {
                // Command
                QRegularExpression sep("\\s+");
                int n;
                if (s.at(0) == '@')
                    n = 0;
                else
                    n = 4;

                QString time = s.section(sep, 0, n);
                QString user = cronOwner;
                n++;
                if (cronOwner == "/etc/crontab") {
                    user = s.section(sep, n, n);
                    n++;
                }

                QString cmnd = s.section(sep, n);
                tCommands << new TCommand(time, user, cmnd, list2String(cmnt), this);
            }
            cmnt.clear();
        }
    }

}

QString Crontab::list2String(QStringList list) const
{
    QString ret("");
    bool flag = false;

    for (const QString &s : list) {
        if (flag) ret += '\n';
        ret += s;
        flag = true;
    }

    return ret.replace(QRegularExpression("^\\n\\n"),"\n");
}
