(function() {
  var CompositeDisposable, Directory, DirectoryView, FileView, repoForPath;

  CompositeDisposable = require('event-kit').CompositeDisposable;

  Directory = require('./directory');

  FileView = require('./file-view');

  repoForPath = require('./helpers').repoForPath;

  module.exports = DirectoryView = (function() {
    function DirectoryView(directory) {
      var iconClass, ref, squashedDirectoryNameNode;
      this.directory = directory;
      this.subscriptions = new CompositeDisposable();
      this.subscriptions.add(this.directory.onDidDestroy((function(_this) {
        return function() {
          return _this.subscriptions.dispose();
        };
      })(this)));
      this.subscribeToDirectory();
      this.element = document.createElement('li');
      this.element.setAttribute('is', 'tree-view-directory');
      this.element.classList.add('directory', 'entry', 'list-nested-item', 'collapsed');
      this.header = document.createElement('div');
      this.header.classList.add('header', 'list-item');
      this.directoryName = document.createElement('span');
      this.directoryName.classList.add('name', 'icon');
      this.entries = document.createElement('ol');
      this.entries.classList.add('entries', 'list-tree');
      if (this.directory.symlink) {
        iconClass = 'icon-file-symlink-directory';
      } else {
        iconClass = 'icon-file-directory';
        if (this.directory.isRoot) {
          if ((ref = repoForPath(this.directory.path)) != null ? ref.isProjectAtRoot() : void 0) {
            iconClass = 'icon-repo';
          }
        } else {
          if (this.directory.submodule) {
            iconClass = 'icon-file-submodule';
          }
        }
      }
      this.directoryName.classList.add(iconClass);
      this.directoryName.dataset.path = this.directory.path;
      if (this.directory.squashedNames != null) {
        this.directoryName.dataset.name = this.directory.squashedNames.join('');
        this.directoryName.title = this.directory.squashedNames.join('');
        squashedDirectoryNameNode = document.createElement('span');
        squashedDirectoryNameNode.classList.add('squashed-dir');
        squashedDirectoryNameNode.textContent = this.directory.squashedNames[0];
        this.directoryName.appendChild(squashedDirectoryNameNode);
        this.directoryName.appendChild(document.createTextNode(this.directory.squashedNames[1]));
      } else {
        this.directoryName.dataset.name = this.directory.name;
        this.directoryName.title = this.directory.name;
        this.directoryName.textContent = this.directory.name;
      }
      this.element.appendChild(this.header);
      this.header.appendChild(this.directoryName);
      this.element.appendChild(this.entries);
      if (this.directory.isRoot) {
        this.element.classList.add('project-root');
        this.header.classList.add('project-root-header');
      } else {
        this.element.draggable = true;
        this.subscriptions.add(this.directory.onDidStatusChange((function(_this) {
          return function() {
            return _this.updateStatus();
          };
        })(this)));
        this.updateStatus();
      }
      if (this.directory.expansionState.isExpanded) {
        this.expand();
      }
      this.element.collapse = this.collapse.bind(this);
      this.element.expand = this.expand.bind(this);
      this.element.toggleExpansion = this.toggleExpansion.bind(this);
      this.element.reload = this.reload.bind(this);
      this.element.isExpanded = this.isExpanded;
      this.element.updateStatus = this.updateStatus.bind(this);
      this.element.isPathEqual = this.isPathEqual.bind(this);
      this.element.getPath = this.getPath.bind(this);
      this.element.directory = this.directory;
      this.element.header = this.header;
      this.element.entries = this.entries;
      this.element.directoryName = this.directoryName;
    }

    DirectoryView.prototype.updateStatus = function() {
      this.element.classList.remove('status-ignored', 'status-modified', 'status-added');
      if (this.directory.status != null) {
        return this.element.classList.add("status-" + this.directory.status);
      }
    };

    DirectoryView.prototype.subscribeToDirectory = function() {
      return this.subscriptions.add(this.directory.onDidAddEntries((function(_this) {
        return function(addedEntries) {
          var entry, i, insertionIndex, len, numberOfEntries, results, view;
          if (!_this.isExpanded) {
            return;
          }
          numberOfEntries = _this.entries.children.length;
          results = [];
          for (i = 0, len = addedEntries.length; i < len; i++) {
            entry = addedEntries[i];
            view = _this.createViewForEntry(entry);
            insertionIndex = entry.indexInParentDirectory;
            if (insertionIndex < numberOfEntries) {
              _this.entries.insertBefore(view.element, _this.entries.children[insertionIndex]);
            } else {
              _this.entries.appendChild(view.element);
            }
            results.push(numberOfEntries++);
          }
          return results;
        };
      })(this)));
    };

    DirectoryView.prototype.getPath = function() {
      return this.directory.path;
    };

    DirectoryView.prototype.isPathEqual = function(pathToCompare) {
      return this.directory.isPathEqual(pathToCompare);
    };

    DirectoryView.prototype.createViewForEntry = function(entry) {
      var subscription, view;
      if (entry instanceof Directory) {
        view = new DirectoryView(entry);
      } else {
        view = new FileView(entry);
      }
      subscription = this.directory.onDidRemoveEntries(function(removedEntries) {
        var removedEntry, removedName, results;
        results = [];
        for (removedName in removedEntries) {
          removedEntry = removedEntries[removedName];
          if (!(entry === removedEntry)) {
            continue;
          }
          view.element.remove();
          subscription.dispose();
          break;
        }
        return results;
      });
      this.subscriptions.add(subscription);
      return view;
    };

    DirectoryView.prototype.reload = function() {
      if (this.isExpanded) {
        return this.directory.reload();
      }
    };

    DirectoryView.prototype.toggleExpansion = function(isRecursive) {
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (this.isExpanded) {
        return this.collapse(isRecursive);
      } else {
        return this.expand(isRecursive);
      }
    };

    DirectoryView.prototype.expand = function(isRecursive) {
      var entry, i, len, ref;
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (!this.isExpanded) {
        this.isExpanded = true;
        this.element.isExpanded = this.isExpanded;
        this.element.classList.add('expanded');
        this.element.classList.remove('collapsed');
        this.directory.expand();
      }
      if (isRecursive) {
        ref = this.entries.children;
        for (i = 0, len = ref.length; i < len; i++) {
          entry = ref[i];
          if (entry.classList.contains('directory')) {
            entry.expand(true);
          }
        }
      }
      return false;
    };

    DirectoryView.prototype.collapse = function(isRecursive) {
      var entry, i, len, ref;
      if (isRecursive == null) {
        isRecursive = false;
      }
      this.isExpanded = false;
      this.element.isExpanded = false;
      if (isRecursive) {
        ref = this.entries.children;
        for (i = 0, len = ref.length; i < len; i++) {
          entry = ref[i];
          if (entry.isExpanded) {
            entry.collapse(true);
          }
        }
      }
      this.element.classList.remove('expanded');
      this.element.classList.add('collapsed');
      this.directory.collapse();
      return this.entries.innerHTML = '';
    };

    return DirectoryView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
